/*
 * Copyright (c) 2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __aarch64__

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>

namespace arm_gemm {

void a64_hybrid_u8s8s32_dot_6x16 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<uint8_t> A_arg,
    size_t M, size_t N, const int8_t *B_ptr, IndirectOutputArg<int32_t> output_arg,
    const int32_t *, Activation, bool accumulate
)
{
    struct KernelArgs {
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const int8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 171f\n"
      "cmp %x[M], #0x4\n"
      "bgt 137f\n"
      "beq 103f\n"
      "cmp %x[M], #0x2\n"
      "bgt 69f\n"
      "beq 35f\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "tbz %x[flags], #0, 12f\n"
      "cmp x11, #0x10\n"
      "bge 11f\n"
      "tbz x11, #3, 6f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "tbz x11, #2, 4f\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "tbz x11, #1, 3f\n"
      "ldr d11, [x9], #0x8\n"
      "mov x25, #0x38\n"
      "tbz x11, #0, 10f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "b 10f\n"
      "3:"  // Height 1: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x11, #0, 10f\n"
      "ldr s11, [x9, #0x0]\n"
      "b 10f\n"
      "4:"  // Height 1: Partial accumulate: partial_2_8
      "tbz x11, #1, 5f\n"
      "ldr d10, [x9], #0x8\n"
      "mov x25, #0x28\n"
      "tbz x11, #0, 10f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "b 10f\n"
      "5:"  // Height 1: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x11, #0, 10f\n"
      "ldr s10, [x9, #0x0]\n"
      "b 10f\n"
      "6:"  // Height 1: Partial accumulate: partial_4_0
      "tbz x11, #2, 8f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "tbz x11, #1, 7f\n"
      "ldr d9, [x9], #0x8\n"
      "mov x25, #0x18\n"
      "tbz x11, #0, 10f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "b 10f\n"
      "7:"  // Height 1: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x11, #0, 10f\n"
      "ldr s9, [x9, #0x0]\n"
      "b 10f\n"
      "8:"  // Height 1: Partial accumulate: partial_2_0
      "tbz x11, #1, 9f\n"
      "ldr d8, [x9], #0x8\n"
      "mov x25, #0x8\n"
      "tbz x11, #0, 10f\n"
      "ld1 { v8.s }[2], [x9]\n"
      "b 10f\n"
      "9:"  // Height 1: Partial accumulate: partial_1_0
      "ldr s8, [x9, #0x0]\n"
      "mov x25, #0x0\n"
      "10:"  // Height 1: Partial accumulate: Done
      "sub x9, x9, x25\n"
      "b 13f\n"
      "11:"  // Height 1: full accumulate
      "ldr q8, [x9, #0x0]\n"
      "ldr q9, [x9, #0x10]\n"
      "ldr q10, [x9, #0x20]\n"
      "ldr q11, [x9, #0x30]\n"
      "b 13f\n"
      "12:"  // Height 1: no accumulate
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "13:"  // Height 1: setup done
      "mov x28, #0x0\n"
      "14:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 15f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "cbnz x28, 16f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "b 16f\n"
      "15:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "16:"  // Height 1: input setup done
      "cmp x27, #0x10\n"
      "blt 19f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q6, [x10, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q7, [x10, #0x10]\n"
      "blt 18f\n"
      "17:"  // Height 1: Multiply loop: Main loop head
      ".inst 0x4f00f0c8  // sudot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr q17, [x10, #0x20]\n"
      ".inst 0x4f00f0e9  // sudot v9.4s, v7.16b, v0.4b[0]\n"
      "ldr q16, [x10, #0x30]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f00f22a  // sudot v10.4s, v17.16b, v0.4b[0]\n"
      "ldr q17, [x10, #0x40]\n"
      ".inst 0x4f00f20b  // sudot v11.4s, v16.16b, v0.4b[0]\n"
      "ldr q16, [x10, #0x50]\n"
      ".inst 0x4f20f228  // sudot v8.4s, v17.16b, v0.4b[1]\n"
      "ldr q17, [x10, #0x60]\n"
      ".inst 0x4f20f209  // sudot v9.4s, v16.16b, v0.4b[1]\n"
      "ldr q16, [x10, #0x70]\n"
      ".inst 0x4f20f22a  // sudot v10.4s, v17.16b, v0.4b[1]\n"
      "ldr q17, [x10, #0x80]\n"
      ".inst 0x4f20f20b  // sudot v11.4s, v16.16b, v0.4b[1]\n"
      "ldr q16, [x10, #0x90]\n"
      ".inst 0x4f00fa28  // sudot v8.4s, v17.16b, v0.4b[2]\n"
      "ldr q17, [x10, #0xa0]\n"
      ".inst 0x4f00fa09  // sudot v9.4s, v16.16b, v0.4b[2]\n"
      "ldr q16, [x10, #0xb0]\n"
      ".inst 0x4f00fa2a  // sudot v10.4s, v17.16b, v0.4b[2]\n"
      "ldr q17, [x10, #0xc0]\n"
      ".inst 0x4f00fa0b  // sudot v11.4s, v16.16b, v0.4b[2]\n"
      "ldr q16, [x10, #0xd0]\n"
      ".inst 0x4f20fa28  // sudot v8.4s, v17.16b, v0.4b[3]\n"
      "ldr q17, [x10, #0xe0]\n"
      ".inst 0x4f20fa09  // sudot v9.4s, v16.16b, v0.4b[3]\n"
      "ldr q16, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4f20fa2a  // sudot v10.4s, v17.16b, v0.4b[3]\n"
      "ldr q6, [x10, #0x0]\n"
      ".inst 0x4f20fa0b  // sudot v11.4s, v16.16b, v0.4b[3]\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "bge 17b\n"
      "18:"  // Height 1: Multiply loop: Single iteration only
      ".inst 0x4f00f0c8  // sudot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr q17, [x10, #0x20]\n"
      ".inst 0x4f00f0e9  // sudot v9.4s, v7.16b, v0.4b[0]\n"
      "ldr q16, [x10, #0x30]\n"
      "add x26, x26, #0x10\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f00f22a  // sudot v10.4s, v17.16b, v0.4b[0]\n"
      "ldr q17, [x10, #0x40]\n"
      ".inst 0x4f00f20b  // sudot v11.4s, v16.16b, v0.4b[0]\n"
      "ldr q16, [x10, #0x50]\n"
      ".inst 0x4f20f228  // sudot v8.4s, v17.16b, v0.4b[1]\n"
      "ldr q17, [x10, #0x60]\n"
      ".inst 0x4f20f209  // sudot v9.4s, v16.16b, v0.4b[1]\n"
      "ldr q16, [x10, #0x70]\n"
      ".inst 0x4f20f22a  // sudot v10.4s, v17.16b, v0.4b[1]\n"
      "ldr q17, [x10, #0x80]\n"
      ".inst 0x4f20f20b  // sudot v11.4s, v16.16b, v0.4b[1]\n"
      "ldr q16, [x10, #0x90]\n"
      ".inst 0x4f00fa28  // sudot v8.4s, v17.16b, v0.4b[2]\n"
      "ldr q17, [x10, #0xa0]\n"
      ".inst 0x4f00fa09  // sudot v9.4s, v16.16b, v0.4b[2]\n"
      "ldr q16, [x10, #0xb0]\n"
      ".inst 0x4f00fa2a  // sudot v10.4s, v17.16b, v0.4b[2]\n"
      "ldr q17, [x10, #0xc0]\n"
      ".inst 0x4f00fa0b  // sudot v11.4s, v16.16b, v0.4b[2]\n"
      "ldr q16, [x10, #0xd0]\n"
      ".inst 0x4f20fa28  // sudot v8.4s, v17.16b, v0.4b[3]\n"
      "ldr q17, [x10, #0xe0]\n"
      ".inst 0x4f20fa09  // sudot v9.4s, v16.16b, v0.4b[3]\n"
      "ldr q16, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4f20fa2a  // sudot v10.4s, v17.16b, v0.4b[3]\n"
      ".inst 0x4f20fa0b  // sudot v11.4s, v16.16b, v0.4b[3]\n"
      "19:"  // Height 1: Multiply loop: Main loop skip
      "cbz x27, 24f\n"
      "cmp x27, #0x4\n"
      "blt 21f\n"
      "20:"  // Height 1: Multiply loop: Odd block loop
      "ldr s18, [x26], #0x4\n"
      "ldr q17, [x10, #0x0]\n"
      "sub x27, x27, #0x4\n"
      "ldr q16, [x10, #0x10]\n"
      "cmp x27, #0x4\n"
      ".inst 0x4f12f228  // sudot v8.4s, v17.16b, v18.4b[0]\n"
      "ldr q17, [x10, #0x20]\n"
      ".inst 0x4f12f209  // sudot v9.4s, v16.16b, v18.4b[0]\n"
      "ldr q16, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f12f22a  // sudot v10.4s, v17.16b, v18.4b[0]\n"
      ".inst 0x4f12f20b  // sudot v11.4s, v16.16b, v18.4b[0]\n"
      "bge 20b\n"
      "21:"  // Height 1: Multiply loop: Skip odd blocks
      "cbz x27, 24f\n"
      "tbz x27, #1, 22f\n"
      "ldr h0, [x26], #0x2\n"
      "tbz x27, #0, 23f\n"
      "ld1 { v0.b }[2], [x26]\n"
      "b 23f\n"
      "22:"  // Height 1: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x26, #0x0]\n"
      "23:"  // Height 1: Multiply loop: Ragged operand read: Done
      "ldr q17, [x10, #0x0]\n"
      "ldr q16, [x10, #0x10]\n"
      ".inst 0x4f00f228  // sudot v8.4s, v17.16b, v0.4b[0]\n"
      "ldr q17, [x10, #0x20]\n"
      ".inst 0x4f00f209  // sudot v9.4s, v16.16b, v0.4b[0]\n"
      "ldr q16, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f00f22a  // sudot v10.4s, v17.16b, v0.4b[0]\n"
      ".inst 0x4f00f20b  // sudot v11.4s, v16.16b, v0.4b[0]\n"
      "24:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 14b\n"
      "cmp x11, #0x10\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "bge 33f\n"
      "tbz x11, #3, 28f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v9.4s }, [x9], #0x10\n"
      "tbz x11, #2, 26f\n"
      "st1 { v10.4s }, [x9], #0x10\n"
      "tbz x11, #1, 25f\n"
      "str d11, [x9], #0x8\n"
      "tbz x11, #0, 32f\n"
      "st1 { v11.s }[2], [x9]\n"
      "b 32f\n"
      "25:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x11, #0, 32f\n"
      "str s11, [x9, #0x0]\n"
      "b 32f\n"
      "26:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x11, #1, 27f\n"
      "str d10, [x9], #0x8\n"
      "tbz x11, #0, 32f\n"
      "st1 { v10.s }[2], [x9]\n"
      "b 32f\n"
      "27:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x11, #0, 32f\n"
      "str s10, [x9, #0x0]\n"
      "b 32f\n"
      "28:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x11, #2, 30f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "tbz x11, #1, 29f\n"
      "str d9, [x9], #0x8\n"
      "tbz x11, #0, 32f\n"
      "st1 { v9.s }[2], [x9]\n"
      "b 32f\n"
      "29:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x11, #0, 32f\n"
      "str s9, [x9, #0x0]\n"
      "b 32f\n"
      "30:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x11, #1, 31f\n"
      "str d8, [x9], #0x8\n"
      "tbz x11, #0, 32f\n"
      "st1 { v8.s }[2], [x9]\n"
      "b 32f\n"
      "31:"  // Height 1: Partial direct writeback: partial_1_0
      "str s8, [x9, #0x0]\n"
      "32:"  // Height 1: Partial direct writeback: Done
      "b 34f\n"
      "33:"  // Height 1: Full writeback
      "str q8, [x9, #0x0]\n"
      "str q9, [x9, #0x10]\n"
      "str q10, [x9, #0x20]\n"
      "str q11, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "34:"  // Height 1: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 2b\n"
      "b 206f\n"
      "35:"  // Height 2
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "36:"  // Height 2: Column loop
      "tbz %x[flags], #0, 46f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x11, #0x10\n"
      "add x24, x9, x20, LSL #2\n"
      "bge 45f\n"
      "tbz x11, #3, 40f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "tbz x11, #2, 38f\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "tbz x11, #1, 37f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d15, [x24], #0x8\n"
      "mov x25, #0x38\n"
      "tbz x11, #0, 44f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "b 44f\n"
      "37:"  // Height 2: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x11, #0, 44f\n"
      "ldr s11, [x9, #0x0]\n"
      "ldr s15, [x24, #0x0]\n"
      "b 44f\n"
      "38:"  // Height 2: Partial accumulate: partial_2_8
      "tbz x11, #1, 39f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d14, [x24], #0x8\n"
      "mov x25, #0x28\n"
      "tbz x11, #0, 44f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "b 44f\n"
      "39:"  // Height 2: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x11, #0, 44f\n"
      "ldr s10, [x9, #0x0]\n"
      "ldr s14, [x24, #0x0]\n"
      "b 44f\n"
      "40:"  // Height 2: Partial accumulate: partial_4_0
      "tbz x11, #2, 42f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "tbz x11, #1, 41f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d13, [x24], #0x8\n"
      "mov x25, #0x18\n"
      "tbz x11, #0, 44f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "b 44f\n"
      "41:"  // Height 2: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x11, #0, 44f\n"
      "ldr s9, [x9, #0x0]\n"
      "ldr s13, [x24, #0x0]\n"
      "b 44f\n"
      "42:"  // Height 2: Partial accumulate: partial_2_0
      "tbz x11, #1, 43f\n"
      "ldr d8, [x9], #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "mov x25, #0x8\n"
      "tbz x11, #0, 44f\n"
      "ld1 { v8.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "b 44f\n"
      "43:"  // Height 2: Partial accumulate: partial_1_0
      "ldr s8, [x9, #0x0]\n"
      "ldr s12, [x24, #0x0]\n"
      "mov x25, #0x0\n"
      "44:"  // Height 2: Partial accumulate: Done
      "sub x9, x9, x25\n"
      "b 47f\n"
      "45:"  // Height 2: full accumulate
      "ldr q8, [x9, #0x0]\n"
      "ldr q9, [x9, #0x10]\n"
      "ldr q10, [x9, #0x20]\n"
      "ldr q11, [x9, #0x30]\n"
      "ldr q12, [x24, #0x0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "b 47f\n"
      "46:"  // Height 2: no accumulate
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "47:"  // Height 2: setup done
      "mov x28, #0x0\n"
      "48:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 49f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 50f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "b 50f\n"
      "49:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "50:"  // Height 2: input setup done
      "cmp x27, #0x10\n"
      "blt 53f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "blt 52f\n"
      "51:"  // Height 2: Multiply loop: Main loop head
      ".inst 0x4f00f0c8  // sudot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f01f0cc  // sudot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr q17, [x10, #0x20]\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4f00f0e9  // sudot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f01f0ed  // sudot v13.4s, v7.16b, v1.4b[0]\n"
      "ldr q16, [x10, #0x30]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f00f22a  // sudot v10.4s, v17.16b, v0.4b[0]\n"
      ".inst 0x4f01f22e  // sudot v14.4s, v17.16b, v1.4b[0]\n"
      "ldr q17, [x10, #0x40]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f00f20b  // sudot v11.4s, v16.16b, v0.4b[0]\n"
      ".inst 0x4f01f20f  // sudot v15.4s, v16.16b, v1.4b[0]\n"
      "ldr q16, [x10, #0x50]\n"
      ".inst 0x4f20f228  // sudot v8.4s, v17.16b, v0.4b[1]\n"
      ".inst 0x4f21f22c  // sudot v12.4s, v17.16b, v1.4b[1]\n"
      "ldr q17, [x10, #0x60]\n"
      ".inst 0x4f20f209  // sudot v9.4s, v16.16b, v0.4b[1]\n"
      ".inst 0x4f21f20d  // sudot v13.4s, v16.16b, v1.4b[1]\n"
      "ldr q16, [x10, #0x70]\n"
      ".inst 0x4f20f22a  // sudot v10.4s, v17.16b, v0.4b[1]\n"
      ".inst 0x4f21f22e  // sudot v14.4s, v17.16b, v1.4b[1]\n"
      "ldr q17, [x10, #0x80]\n"
      ".inst 0x4f20f20b  // sudot v11.4s, v16.16b, v0.4b[1]\n"
      ".inst 0x4f21f20f  // sudot v15.4s, v16.16b, v1.4b[1]\n"
      "ldr q16, [x10, #0x90]\n"
      ".inst 0x4f00fa28  // sudot v8.4s, v17.16b, v0.4b[2]\n"
      ".inst 0x4f01fa2c  // sudot v12.4s, v17.16b, v1.4b[2]\n"
      "ldr q17, [x10, #0xa0]\n"
      ".inst 0x4f00fa09  // sudot v9.4s, v16.16b, v0.4b[2]\n"
      ".inst 0x4f01fa0d  // sudot v13.4s, v16.16b, v1.4b[2]\n"
      "ldr q16, [x10, #0xb0]\n"
      ".inst 0x4f00fa2a  // sudot v10.4s, v17.16b, v0.4b[2]\n"
      ".inst 0x4f01fa2e  // sudot v14.4s, v17.16b, v1.4b[2]\n"
      "ldr q17, [x10, #0xc0]\n"
      ".inst 0x4f00fa0b  // sudot v11.4s, v16.16b, v0.4b[2]\n"
      ".inst 0x4f01fa0f  // sudot v15.4s, v16.16b, v1.4b[2]\n"
      "ldr q16, [x10, #0xd0]\n"
      ".inst 0x4f20fa28  // sudot v8.4s, v17.16b, v0.4b[3]\n"
      ".inst 0x4f21fa2c  // sudot v12.4s, v17.16b, v1.4b[3]\n"
      "ldr q17, [x10, #0xe0]\n"
      ".inst 0x4f20fa09  // sudot v9.4s, v16.16b, v0.4b[3]\n"
      ".inst 0x4f21fa0d  // sudot v13.4s, v16.16b, v1.4b[3]\n"
      "ldr q16, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4f20fa2a  // sudot v10.4s, v17.16b, v0.4b[3]\n"
      ".inst 0x4f21fa2e  // sudot v14.4s, v17.16b, v1.4b[3]\n"
      "ldr q6, [x10, #0x0]\n"
      ".inst 0x4f20fa0b  // sudot v11.4s, v16.16b, v0.4b[3]\n"
      "ldr q0, [x26, #0x0]\n"
      ".inst 0x4f21fa0f  // sudot v15.4s, v16.16b, v1.4b[3]\n"
      "ldr q1, [x25, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "bge 51b\n"
      "52:"  // Height 2: Multiply loop: Single iteration only
      ".inst 0x4f00f0c8  // sudot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f01f0cc  // sudot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr q17, [x10, #0x20]\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f00f0e9  // sudot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f01f0ed  // sudot v13.4s, v7.16b, v1.4b[0]\n"
      "ldr q16, [x10, #0x30]\n"
      "add x25, x25, #0x10\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f00f22a  // sudot v10.4s, v17.16b, v0.4b[0]\n"
      ".inst 0x4f01f22e  // sudot v14.4s, v17.16b, v1.4b[0]\n"
      "ldr q17, [x10, #0x40]\n"
      ".inst 0x4f00f20b  // sudot v11.4s, v16.16b, v0.4b[0]\n"
      ".inst 0x4f01f20f  // sudot v15.4s, v16.16b, v1.4b[0]\n"
      "ldr q16, [x10, #0x50]\n"
      ".inst 0x4f20f228  // sudot v8.4s, v17.16b, v0.4b[1]\n"
      ".inst 0x4f21f22c  // sudot v12.4s, v17.16b, v1.4b[1]\n"
      "ldr q17, [x10, #0x60]\n"
      ".inst 0x4f20f209  // sudot v9.4s, v16.16b, v0.4b[1]\n"
      ".inst 0x4f21f20d  // sudot v13.4s, v16.16b, v1.4b[1]\n"
      "ldr q16, [x10, #0x70]\n"
      ".inst 0x4f20f22a  // sudot v10.4s, v17.16b, v0.4b[1]\n"
      ".inst 0x4f21f22e  // sudot v14.4s, v17.16b, v1.4b[1]\n"
      "ldr q17, [x10, #0x80]\n"
      ".inst 0x4f20f20b  // sudot v11.4s, v16.16b, v0.4b[1]\n"
      ".inst 0x4f21f20f  // sudot v15.4s, v16.16b, v1.4b[1]\n"
      "ldr q16, [x10, #0x90]\n"
      ".inst 0x4f00fa28  // sudot v8.4s, v17.16b, v0.4b[2]\n"
      ".inst 0x4f01fa2c  // sudot v12.4s, v17.16b, v1.4b[2]\n"
      "ldr q17, [x10, #0xa0]\n"
      ".inst 0x4f00fa09  // sudot v9.4s, v16.16b, v0.4b[2]\n"
      ".inst 0x4f01fa0d  // sudot v13.4s, v16.16b, v1.4b[2]\n"
      "ldr q16, [x10, #0xb0]\n"
      ".inst 0x4f00fa2a  // sudot v10.4s, v17.16b, v0.4b[2]\n"
      ".inst 0x4f01fa2e  // sudot v14.4s, v17.16b, v1.4b[2]\n"
      "ldr q17, [x10, #0xc0]\n"
      ".inst 0x4f00fa0b  // sudot v11.4s, v16.16b, v0.4b[2]\n"
      ".inst 0x4f01fa0f  // sudot v15.4s, v16.16b, v1.4b[2]\n"
      "ldr q16, [x10, #0xd0]\n"
      ".inst 0x4f20fa28  // sudot v8.4s, v17.16b, v0.4b[3]\n"
      ".inst 0x4f21fa2c  // sudot v12.4s, v17.16b, v1.4b[3]\n"
      "ldr q17, [x10, #0xe0]\n"
      ".inst 0x4f20fa09  // sudot v9.4s, v16.16b, v0.4b[3]\n"
      ".inst 0x4f21fa0d  // sudot v13.4s, v16.16b, v1.4b[3]\n"
      "ldr q16, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4f20fa2a  // sudot v10.4s, v17.16b, v0.4b[3]\n"
      ".inst 0x4f21fa2e  // sudot v14.4s, v17.16b, v1.4b[3]\n"
      ".inst 0x4f20fa0b  // sudot v11.4s, v16.16b, v0.4b[3]\n"
      ".inst 0x4f21fa0f  // sudot v15.4s, v16.16b, v1.4b[3]\n"
      "53:"  // Height 2: Multiply loop: Main loop skip
      "cbz x27, 58f\n"
      "cmp x27, #0x4\n"
      "blt 55f\n"
      "54:"  // Height 2: Multiply loop: Odd block loop
      "ldr s19, [x26], #0x4\n"
      "ldr s18, [x25], #0x4\n"
      "sub x27, x27, #0x4\n"
      "ldr q17, [x10, #0x0]\n"
      "ldr q16, [x10, #0x10]\n"
      "cmp x27, #0x4\n"
      ".inst 0x4f13f228  // sudot v8.4s, v17.16b, v19.4b[0]\n"
      ".inst 0x4f12f22c  // sudot v12.4s, v17.16b, v18.4b[0]\n"
      "ldr q17, [x10, #0x20]\n"
      ".inst 0x4f13f209  // sudot v9.4s, v16.16b, v19.4b[0]\n"
      ".inst 0x4f12f20d  // sudot v13.4s, v16.16b, v18.4b[0]\n"
      "ldr q16, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f13f22a  // sudot v10.4s, v17.16b, v19.4b[0]\n"
      ".inst 0x4f12f22e  // sudot v14.4s, v17.16b, v18.4b[0]\n"
      ".inst 0x4f13f20b  // sudot v11.4s, v16.16b, v19.4b[0]\n"
      ".inst 0x4f12f20f  // sudot v15.4s, v16.16b, v18.4b[0]\n"
      "bge 54b\n"
      "55:"  // Height 2: Multiply loop: Skip odd blocks
      "cbz x27, 58f\n"
      "tbz x27, #1, 56f\n"
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "tbz x27, #0, 57f\n"
      "ld1 { v0.b }[2], [x26]\n"
      "ld1 { v1.b }[2], [x25]\n"
      "b 57f\n"
      "56:"  // Height 2: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x26, #0x0]\n"
      "ldr b1, [x25, #0x0]\n"
      "57:"  // Height 2: Multiply loop: Ragged operand read: Done
      "ldr q17, [x10, #0x0]\n"
      "ldr q16, [x10, #0x10]\n"
      ".inst 0x4f00f228  // sudot v8.4s, v17.16b, v0.4b[0]\n"
      ".inst 0x4f01f22c  // sudot v12.4s, v17.16b, v1.4b[0]\n"
      "ldr q17, [x10, #0x20]\n"
      ".inst 0x4f00f209  // sudot v9.4s, v16.16b, v0.4b[0]\n"
      ".inst 0x4f01f20d  // sudot v13.4s, v16.16b, v1.4b[0]\n"
      "ldr q16, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f00f22a  // sudot v10.4s, v17.16b, v0.4b[0]\n"
      ".inst 0x4f01f22e  // sudot v14.4s, v17.16b, v1.4b[0]\n"
      ".inst 0x4f00f20b  // sudot v11.4s, v16.16b, v0.4b[0]\n"
      ".inst 0x4f01f20f  // sudot v15.4s, v16.16b, v1.4b[0]\n"
      "58:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 48b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x11, #0x10\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "add x24, x9, x20, LSL #2\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "bge 67f\n"
      "tbz x11, #3, 62f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v9.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v13.4s }, [x24], #0x10\n"
      "tbz x11, #2, 60f\n"
      "st1 { v10.4s }, [x9], #0x10\n"
      "st1 { v14.4s }, [x24], #0x10\n"
      "tbz x11, #1, 59f\n"
      "str d11, [x9], #0x8\n"
      "str d15, [x24], #0x8\n"
      "tbz x11, #0, 66f\n"
      "st1 { v11.s }[2], [x9]\n"
      "st1 { v15.s }[2], [x24]\n"
      "b 66f\n"
      "59:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x11, #0, 66f\n"
      "str s11, [x9, #0x0]\n"
      "str s15, [x24, #0x0]\n"
      "b 66f\n"
      "60:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x11, #1, 61f\n"
      "str d10, [x9], #0x8\n"
      "str d14, [x24], #0x8\n"
      "tbz x11, #0, 66f\n"
      "st1 { v10.s }[2], [x9]\n"
      "st1 { v14.s }[2], [x24]\n"
      "b 66f\n"
      "61:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x11, #0, 66f\n"
      "str s10, [x9, #0x0]\n"
      "str s14, [x24, #0x0]\n"
      "b 66f\n"
      "62:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x11, #2, 64f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "tbz x11, #1, 63f\n"
      "str d9, [x9], #0x8\n"
      "str d13, [x24], #0x8\n"
      "tbz x11, #0, 66f\n"
      "st1 { v9.s }[2], [x9]\n"
      "st1 { v13.s }[2], [x24]\n"
      "b 66f\n"
      "63:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x11, #0, 66f\n"
      "str s9, [x9, #0x0]\n"
      "str s13, [x24, #0x0]\n"
      "b 66f\n"
      "64:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x11, #1, 65f\n"
      "str d8, [x9], #0x8\n"
      "str d12, [x24], #0x8\n"
      "tbz x11, #0, 66f\n"
      "st1 { v8.s }[2], [x9]\n"
      "st1 { v12.s }[2], [x24]\n"
      "b 66f\n"
      "65:"  // Height 2: Partial direct writeback: partial_1_0
      "str s8, [x9, #0x0]\n"
      "str s12, [x24, #0x0]\n"
      "66:"  // Height 2: Partial direct writeback: Done
      "b 68f\n"
      "67:"  // Height 2: Full writeback
      "str q8, [x9, #0x0]\n"
      "str q9, [x9, #0x10]\n"
      "str q10, [x9, #0x20]\n"
      "str q11, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q12, [x24, #0x0]\n"
      "str q13, [x24, #0x10]\n"
      "str q14, [x24, #0x20]\n"
      "str q15, [x24, #0x30]\n"
      "68:"  // Height 2: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 36b\n"
      "b 206f\n"
      "69:"  // Height 3
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "70:"  // Height 3: Column loop
      "tbz %x[flags], #0, 80f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x11, #0x10\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "bge 79f\n"
      "tbz x11, #3, 74f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "tbz x11, #2, 72f\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "tbz x11, #1, 71f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d15, [x24], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d19, [x23], #0x8\n"
      "tbz x11, #0, 78f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "b 78f\n"
      "71:"  // Height 3: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x11, #0, 78f\n"
      "ldr s11, [x9, #0x0]\n"
      "ldr s15, [x24, #0x0]\n"
      "ldr s19, [x23, #0x0]\n"
      "b 78f\n"
      "72:"  // Height 3: Partial accumulate: partial_2_8
      "tbz x11, #1, 73f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d14, [x24], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d18, [x23], #0x8\n"
      "tbz x11, #0, 78f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "b 78f\n"
      "73:"  // Height 3: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x11, #0, 78f\n"
      "ldr s10, [x9, #0x0]\n"
      "ldr s14, [x24, #0x0]\n"
      "ldr s18, [x23, #0x0]\n"
      "b 78f\n"
      "74:"  // Height 3: Partial accumulate: partial_4_0
      "tbz x11, #2, 76f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "tbz x11, #1, 75f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d13, [x24], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d17, [x23], #0x8\n"
      "tbz x11, #0, 78f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "b 78f\n"
      "75:"  // Height 3: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x11, #0, 78f\n"
      "ldr s9, [x9, #0x0]\n"
      "ldr s13, [x24, #0x0]\n"
      "ldr s17, [x23, #0x0]\n"
      "b 78f\n"
      "76:"  // Height 3: Partial accumulate: partial_2_0
      "tbz x11, #1, 77f\n"
      "ldr d8, [x9], #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d16, [x23], #0x8\n"
      "tbz x11, #0, 78f\n"
      "ld1 { v8.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v16.s }[2], [x23]\n"
      "b 78f\n"
      "77:"  // Height 3: Partial accumulate: partial_1_0
      "ldr s8, [x9, #0x0]\n"
      "ldr s12, [x24, #0x0]\n"
      "mov x25, #0x0\n"
      "ldr s16, [x23, #0x0]\n"
      "78:"  // Height 3: Partial accumulate: Done
      "sub x9, x9, x25\n"
      "b 81f\n"
      "79:"  // Height 3: full accumulate
      "ldr q8, [x9, #0x0]\n"
      "ldr q9, [x9, #0x10]\n"
      "ldr q10, [x9, #0x20]\n"
      "ldr q11, [x9, #0x30]\n"
      "ldr q12, [x24, #0x0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q16, [x23, #0x0]\n"
      "ldr q17, [x23, #0x10]\n"
      "ldr q18, [x23, #0x20]\n"
      "ldr q19, [x23, #0x30]\n"
      "b 81f\n"
      "80:"  // Height 3: no accumulate
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "81:"  // Height 3: setup done
      "mov x28, #0x0\n"
      "82:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 83f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 84f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "b 84f\n"
      "83:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "84:"  // Height 3: input setup done
      "cmp x27, #0x10\n"
      "blt 87f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "blt 86f\n"
      "85:"  // Height 3: Multiply loop: Main loop head
      ".inst 0x4f00f0c8  // sudot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f01f0cc  // sudot v12.4s, v6.16b, v1.4b[0]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f02f0d0  // sudot v16.4s, v6.16b, v2.4b[0]\n"
      "ldr q21, [x10, #0x20]\n"
      ".inst 0x4f00f0e9  // sudot v9.4s, v7.16b, v0.4b[0]\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f01f0ed  // sudot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f02f0f1  // sudot v17.4s, v7.16b, v2.4b[0]\n"
      "ldr q20, [x10, #0x30]\n"
      "add x24, x24, #0x10\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f00f2aa  // sudot v10.4s, v21.16b, v0.4b[0]\n"
      ".inst 0x4f01f2ae  // sudot v14.4s, v21.16b, v1.4b[0]\n"
      ".inst 0x4f02f2b2  // sudot v18.4s, v21.16b, v2.4b[0]\n"
      "ldr q21, [x10, #0x40]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f00f28b  // sudot v11.4s, v20.16b, v0.4b[0]\n"
      ".inst 0x4f01f28f  // sudot v15.4s, v20.16b, v1.4b[0]\n"
      ".inst 0x4f02f293  // sudot v19.4s, v20.16b, v2.4b[0]\n"
      "ldr q20, [x10, #0x50]\n"
      ".inst 0x4f20f2a8  // sudot v8.4s, v21.16b, v0.4b[1]\n"
      ".inst 0x4f21f2ac  // sudot v12.4s, v21.16b, v1.4b[1]\n"
      ".inst 0x4f22f2b0  // sudot v16.4s, v21.16b, v2.4b[1]\n"
      "ldr q21, [x10, #0x60]\n"
      ".inst 0x4f20f289  // sudot v9.4s, v20.16b, v0.4b[1]\n"
      ".inst 0x4f21f28d  // sudot v13.4s, v20.16b, v1.4b[1]\n"
      ".inst 0x4f22f291  // sudot v17.4s, v20.16b, v2.4b[1]\n"
      "ldr q20, [x10, #0x70]\n"
      ".inst 0x4f20f2aa  // sudot v10.4s, v21.16b, v0.4b[1]\n"
      ".inst 0x4f21f2ae  // sudot v14.4s, v21.16b, v1.4b[1]\n"
      ".inst 0x4f22f2b2  // sudot v18.4s, v21.16b, v2.4b[1]\n"
      "ldr q21, [x10, #0x80]\n"
      ".inst 0x4f20f28b  // sudot v11.4s, v20.16b, v0.4b[1]\n"
      ".inst 0x4f21f28f  // sudot v15.4s, v20.16b, v1.4b[1]\n"
      ".inst 0x4f22f293  // sudot v19.4s, v20.16b, v2.4b[1]\n"
      "ldr q20, [x10, #0x90]\n"
      ".inst 0x4f00faa8  // sudot v8.4s, v21.16b, v0.4b[2]\n"
      ".inst 0x4f01faac  // sudot v12.4s, v21.16b, v1.4b[2]\n"
      ".inst 0x4f02fab0  // sudot v16.4s, v21.16b, v2.4b[2]\n"
      "ldr q21, [x10, #0xa0]\n"
      ".inst 0x4f00fa89  // sudot v9.4s, v20.16b, v0.4b[2]\n"
      ".inst 0x4f01fa8d  // sudot v13.4s, v20.16b, v1.4b[2]\n"
      ".inst 0x4f02fa91  // sudot v17.4s, v20.16b, v2.4b[2]\n"
      "ldr q20, [x10, #0xb0]\n"
      ".inst 0x4f00faaa  // sudot v10.4s, v21.16b, v0.4b[2]\n"
      ".inst 0x4f01faae  // sudot v14.4s, v21.16b, v1.4b[2]\n"
      ".inst 0x4f02fab2  // sudot v18.4s, v21.16b, v2.4b[2]\n"
      "ldr q21, [x10, #0xc0]\n"
      ".inst 0x4f00fa8b  // sudot v11.4s, v20.16b, v0.4b[2]\n"
      ".inst 0x4f01fa8f  // sudot v15.4s, v20.16b, v1.4b[2]\n"
      ".inst 0x4f02fa93  // sudot v19.4s, v20.16b, v2.4b[2]\n"
      "ldr q20, [x10, #0xd0]\n"
      ".inst 0x4f20faa8  // sudot v8.4s, v21.16b, v0.4b[3]\n"
      ".inst 0x4f21faac  // sudot v12.4s, v21.16b, v1.4b[3]\n"
      ".inst 0x4f22fab0  // sudot v16.4s, v21.16b, v2.4b[3]\n"
      "ldr q21, [x10, #0xe0]\n"
      ".inst 0x4f20fa89  // sudot v9.4s, v20.16b, v0.4b[3]\n"
      ".inst 0x4f21fa8d  // sudot v13.4s, v20.16b, v1.4b[3]\n"
      ".inst 0x4f22fa91  // sudot v17.4s, v20.16b, v2.4b[3]\n"
      "ldr q20, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4f20faaa  // sudot v10.4s, v21.16b, v0.4b[3]\n"
      ".inst 0x4f21faae  // sudot v14.4s, v21.16b, v1.4b[3]\n"
      ".inst 0x4f22fab2  // sudot v18.4s, v21.16b, v2.4b[3]\n"
      "ldr q6, [x10, #0x0]\n"
      ".inst 0x4f20fa8b  // sudot v11.4s, v20.16b, v0.4b[3]\n"
      "ldr q0, [x26, #0x0]\n"
      ".inst 0x4f21fa8f  // sudot v15.4s, v20.16b, v1.4b[3]\n"
      "ldr q1, [x25, #0x0]\n"
      ".inst 0x4f22fa93  // sudot v19.4s, v20.16b, v2.4b[3]\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "bge 85b\n"
      "86:"  // Height 3: Multiply loop: Single iteration only
      ".inst 0x4f00f0c8  // sudot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f01f0cc  // sudot v12.4s, v6.16b, v1.4b[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f02f0d0  // sudot v16.4s, v6.16b, v2.4b[0]\n"
      "ldr q21, [x10, #0x20]\n"
      ".inst 0x4f00f0e9  // sudot v9.4s, v7.16b, v0.4b[0]\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f01f0ed  // sudot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f02f0f1  // sudot v17.4s, v7.16b, v2.4b[0]\n"
      "ldr q20, [x10, #0x30]\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f00f2aa  // sudot v10.4s, v21.16b, v0.4b[0]\n"
      ".inst 0x4f01f2ae  // sudot v14.4s, v21.16b, v1.4b[0]\n"
      ".inst 0x4f02f2b2  // sudot v18.4s, v21.16b, v2.4b[0]\n"
      "ldr q21, [x10, #0x40]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f00f28b  // sudot v11.4s, v20.16b, v0.4b[0]\n"
      ".inst 0x4f01f28f  // sudot v15.4s, v20.16b, v1.4b[0]\n"
      ".inst 0x4f02f293  // sudot v19.4s, v20.16b, v2.4b[0]\n"
      "ldr q20, [x10, #0x50]\n"
      ".inst 0x4f20f2a8  // sudot v8.4s, v21.16b, v0.4b[1]\n"
      ".inst 0x4f21f2ac  // sudot v12.4s, v21.16b, v1.4b[1]\n"
      ".inst 0x4f22f2b0  // sudot v16.4s, v21.16b, v2.4b[1]\n"
      "ldr q21, [x10, #0x60]\n"
      ".inst 0x4f20f289  // sudot v9.4s, v20.16b, v0.4b[1]\n"
      ".inst 0x4f21f28d  // sudot v13.4s, v20.16b, v1.4b[1]\n"
      ".inst 0x4f22f291  // sudot v17.4s, v20.16b, v2.4b[1]\n"
      "ldr q20, [x10, #0x70]\n"
      ".inst 0x4f20f2aa  // sudot v10.4s, v21.16b, v0.4b[1]\n"
      ".inst 0x4f21f2ae  // sudot v14.4s, v21.16b, v1.4b[1]\n"
      ".inst 0x4f22f2b2  // sudot v18.4s, v21.16b, v2.4b[1]\n"
      "ldr q21, [x10, #0x80]\n"
      ".inst 0x4f20f28b  // sudot v11.4s, v20.16b, v0.4b[1]\n"
      ".inst 0x4f21f28f  // sudot v15.4s, v20.16b, v1.4b[1]\n"
      ".inst 0x4f22f293  // sudot v19.4s, v20.16b, v2.4b[1]\n"
      "ldr q20, [x10, #0x90]\n"
      ".inst 0x4f00faa8  // sudot v8.4s, v21.16b, v0.4b[2]\n"
      ".inst 0x4f01faac  // sudot v12.4s, v21.16b, v1.4b[2]\n"
      ".inst 0x4f02fab0  // sudot v16.4s, v21.16b, v2.4b[2]\n"
      "ldr q21, [x10, #0xa0]\n"
      ".inst 0x4f00fa89  // sudot v9.4s, v20.16b, v0.4b[2]\n"
      ".inst 0x4f01fa8d  // sudot v13.4s, v20.16b, v1.4b[2]\n"
      ".inst 0x4f02fa91  // sudot v17.4s, v20.16b, v2.4b[2]\n"
      "ldr q20, [x10, #0xb0]\n"
      ".inst 0x4f00faaa  // sudot v10.4s, v21.16b, v0.4b[2]\n"
      ".inst 0x4f01faae  // sudot v14.4s, v21.16b, v1.4b[2]\n"
      ".inst 0x4f02fab2  // sudot v18.4s, v21.16b, v2.4b[2]\n"
      "ldr q21, [x10, #0xc0]\n"
      ".inst 0x4f00fa8b  // sudot v11.4s, v20.16b, v0.4b[2]\n"
      ".inst 0x4f01fa8f  // sudot v15.4s, v20.16b, v1.4b[2]\n"
      ".inst 0x4f02fa93  // sudot v19.4s, v20.16b, v2.4b[2]\n"
      "ldr q20, [x10, #0xd0]\n"
      ".inst 0x4f20faa8  // sudot v8.4s, v21.16b, v0.4b[3]\n"
      ".inst 0x4f21faac  // sudot v12.4s, v21.16b, v1.4b[3]\n"
      ".inst 0x4f22fab0  // sudot v16.4s, v21.16b, v2.4b[3]\n"
      "ldr q21, [x10, #0xe0]\n"
      ".inst 0x4f20fa89  // sudot v9.4s, v20.16b, v0.4b[3]\n"
      ".inst 0x4f21fa8d  // sudot v13.4s, v20.16b, v1.4b[3]\n"
      ".inst 0x4f22fa91  // sudot v17.4s, v20.16b, v2.4b[3]\n"
      "ldr q20, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4f20faaa  // sudot v10.4s, v21.16b, v0.4b[3]\n"
      ".inst 0x4f21faae  // sudot v14.4s, v21.16b, v1.4b[3]\n"
      ".inst 0x4f22fab2  // sudot v18.4s, v21.16b, v2.4b[3]\n"
      ".inst 0x4f20fa8b  // sudot v11.4s, v20.16b, v0.4b[3]\n"
      ".inst 0x4f21fa8f  // sudot v15.4s, v20.16b, v1.4b[3]\n"
      ".inst 0x4f22fa93  // sudot v19.4s, v20.16b, v2.4b[3]\n"
      "87:"  // Height 3: Multiply loop: Main loop skip
      "cbz x27, 92f\n"
      "cmp x27, #0x4\n"
      "blt 89f\n"
      "88:"  // Height 3: Multiply loop: Odd block loop
      "ldr s24, [x26], #0x4\n"
      "ldr s23, [x25], #0x4\n"
      "sub x27, x27, #0x4\n"
      "ldr s22, [x24], #0x4\n"
      "ldr q21, [x10, #0x0]\n"
      "cmp x27, #0x4\n"
      "ldr q20, [x10, #0x10]\n"
      ".inst 0x4f18f2a8  // sudot v8.4s, v21.16b, v24.4b[0]\n"
      ".inst 0x4f17f2ac  // sudot v12.4s, v21.16b, v23.4b[0]\n"
      ".inst 0x4f16f2b0  // sudot v16.4s, v21.16b, v22.4b[0]\n"
      "ldr q21, [x10, #0x20]\n"
      ".inst 0x4f18f289  // sudot v9.4s, v20.16b, v24.4b[0]\n"
      ".inst 0x4f17f28d  // sudot v13.4s, v20.16b, v23.4b[0]\n"
      ".inst 0x4f16f291  // sudot v17.4s, v20.16b, v22.4b[0]\n"
      "ldr q20, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f18f2aa  // sudot v10.4s, v21.16b, v24.4b[0]\n"
      ".inst 0x4f17f2ae  // sudot v14.4s, v21.16b, v23.4b[0]\n"
      ".inst 0x4f16f2b2  // sudot v18.4s, v21.16b, v22.4b[0]\n"
      ".inst 0x4f18f28b  // sudot v11.4s, v20.16b, v24.4b[0]\n"
      ".inst 0x4f17f28f  // sudot v15.4s, v20.16b, v23.4b[0]\n"
      ".inst 0x4f16f293  // sudot v19.4s, v20.16b, v22.4b[0]\n"
      "bge 88b\n"
      "89:"  // Height 3: Multiply loop: Skip odd blocks
      "cbz x27, 92f\n"
      "tbz x27, #1, 90f\n"
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "ldr h2, [x24], #0x2\n"
      "tbz x27, #0, 91f\n"
      "ld1 { v0.b }[2], [x26]\n"
      "ld1 { v1.b }[2], [x25]\n"
      "ld1 { v2.b }[2], [x24]\n"
      "b 91f\n"
      "90:"  // Height 3: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x26, #0x0]\n"
      "ldr b1, [x25, #0x0]\n"
      "ldr b2, [x24, #0x0]\n"
      "91:"  // Height 3: Multiply loop: Ragged operand read: Done
      "ldr q21, [x10, #0x0]\n"
      "ldr q20, [x10, #0x10]\n"
      ".inst 0x4f00f2a8  // sudot v8.4s, v21.16b, v0.4b[0]\n"
      ".inst 0x4f01f2ac  // sudot v12.4s, v21.16b, v1.4b[0]\n"
      ".inst 0x4f02f2b0  // sudot v16.4s, v21.16b, v2.4b[0]\n"
      "ldr q21, [x10, #0x20]\n"
      ".inst 0x4f00f289  // sudot v9.4s, v20.16b, v0.4b[0]\n"
      ".inst 0x4f01f28d  // sudot v13.4s, v20.16b, v1.4b[0]\n"
      ".inst 0x4f02f291  // sudot v17.4s, v20.16b, v2.4b[0]\n"
      "ldr q20, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f00f2aa  // sudot v10.4s, v21.16b, v0.4b[0]\n"
      ".inst 0x4f01f2ae  // sudot v14.4s, v21.16b, v1.4b[0]\n"
      ".inst 0x4f02f2b2  // sudot v18.4s, v21.16b, v2.4b[0]\n"
      ".inst 0x4f00f28b  // sudot v11.4s, v20.16b, v0.4b[0]\n"
      ".inst 0x4f01f28f  // sudot v15.4s, v20.16b, v1.4b[0]\n"
      ".inst 0x4f02f293  // sudot v19.4s, v20.16b, v2.4b[0]\n"
      "92:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 82b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x11, #0x10\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "add x24, x9, x20, LSL #2\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "add x23, x24, x20, LSL #2\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "bge 101f\n"
      "tbz x11, #3, 96f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v9.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v13.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v17.4s }, [x23], #0x10\n"
      "tbz x11, #2, 94f\n"
      "st1 { v10.4s }, [x9], #0x10\n"
      "st1 { v14.4s }, [x24], #0x10\n"
      "st1 { v18.4s }, [x23], #0x10\n"
      "tbz x11, #1, 93f\n"
      "str d11, [x9], #0x8\n"
      "str d15, [x24], #0x8\n"
      "str d19, [x23], #0x8\n"
      "tbz x11, #0, 100f\n"
      "st1 { v11.s }[2], [x9]\n"
      "st1 { v15.s }[2], [x24]\n"
      "st1 { v19.s }[2], [x23]\n"
      "b 100f\n"
      "93:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x11, #0, 100f\n"
      "str s11, [x9, #0x0]\n"
      "str s15, [x24, #0x0]\n"
      "str s19, [x23, #0x0]\n"
      "b 100f\n"
      "94:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x11, #1, 95f\n"
      "str d10, [x9], #0x8\n"
      "str d14, [x24], #0x8\n"
      "str d18, [x23], #0x8\n"
      "tbz x11, #0, 100f\n"
      "st1 { v10.s }[2], [x9]\n"
      "st1 { v14.s }[2], [x24]\n"
      "st1 { v18.s }[2], [x23]\n"
      "b 100f\n"
      "95:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x11, #0, 100f\n"
      "str s10, [x9, #0x0]\n"
      "str s14, [x24, #0x0]\n"
      "str s18, [x23, #0x0]\n"
      "b 100f\n"
      "96:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x11, #2, 98f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "tbz x11, #1, 97f\n"
      "str d9, [x9], #0x8\n"
      "str d13, [x24], #0x8\n"
      "str d17, [x23], #0x8\n"
      "tbz x11, #0, 100f\n"
      "st1 { v9.s }[2], [x9]\n"
      "st1 { v13.s }[2], [x24]\n"
      "st1 { v17.s }[2], [x23]\n"
      "b 100f\n"
      "97:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x11, #0, 100f\n"
      "str s9, [x9, #0x0]\n"
      "str s13, [x24, #0x0]\n"
      "str s17, [x23, #0x0]\n"
      "b 100f\n"
      "98:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x11, #1, 99f\n"
      "str d8, [x9], #0x8\n"
      "str d12, [x24], #0x8\n"
      "str d16, [x23], #0x8\n"
      "tbz x11, #0, 100f\n"
      "st1 { v8.s }[2], [x9]\n"
      "st1 { v12.s }[2], [x24]\n"
      "st1 { v16.s }[2], [x23]\n"
      "b 100f\n"
      "99:"  // Height 3: Partial direct writeback: partial_1_0
      "str s8, [x9, #0x0]\n"
      "str s12, [x24, #0x0]\n"
      "str s16, [x23, #0x0]\n"
      "100:"  // Height 3: Partial direct writeback: Done
      "b 102f\n"
      "101:"  // Height 3: Full writeback
      "str q8, [x9, #0x0]\n"
      "str q9, [x9, #0x10]\n"
      "str q10, [x9, #0x20]\n"
      "str q11, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q12, [x24, #0x0]\n"
      "str q13, [x24, #0x10]\n"
      "str q14, [x24, #0x20]\n"
      "str q15, [x24, #0x30]\n"
      "str q16, [x23, #0x0]\n"
      "str q17, [x23, #0x10]\n"
      "str q18, [x23, #0x20]\n"
      "str q19, [x23, #0x30]\n"
      "102:"  // Height 3: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 70b\n"
      "b 206f\n"
      "103:"  // Height 4
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "104:"  // Height 4: Column loop
      "tbz %x[flags], #0, 114f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x11, #0x10\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "bge 113f\n"
      "tbz x11, #3, 108f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "tbz x11, #2, 106f\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "tbz x11, #1, 105f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d15, [x24], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d19, [x23], #0x8\n"
      "ldr d23, [x22], #0x8\n"
      "tbz x11, #0, 112f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "b 112f\n"
      "105:"  // Height 4: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x11, #0, 112f\n"
      "ldr s11, [x9, #0x0]\n"
      "ldr s15, [x24, #0x0]\n"
      "ldr s19, [x23, #0x0]\n"
      "ldr s23, [x22, #0x0]\n"
      "b 112f\n"
      "106:"  // Height 4: Partial accumulate: partial_2_8
      "tbz x11, #1, 107f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d14, [x24], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d18, [x23], #0x8\n"
      "ldr d22, [x22], #0x8\n"
      "tbz x11, #0, 112f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "b 112f\n"
      "107:"  // Height 4: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x11, #0, 112f\n"
      "ldr s10, [x9, #0x0]\n"
      "ldr s14, [x24, #0x0]\n"
      "ldr s18, [x23, #0x0]\n"
      "ldr s22, [x22, #0x0]\n"
      "b 112f\n"
      "108:"  // Height 4: Partial accumulate: partial_4_0
      "tbz x11, #2, 110f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "tbz x11, #1, 109f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d13, [x24], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d17, [x23], #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "tbz x11, #0, 112f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "b 112f\n"
      "109:"  // Height 4: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x11, #0, 112f\n"
      "ldr s9, [x9, #0x0]\n"
      "ldr s13, [x24, #0x0]\n"
      "ldr s17, [x23, #0x0]\n"
      "ldr s21, [x22, #0x0]\n"
      "b 112f\n"
      "110:"  // Height 4: Partial accumulate: partial_2_0
      "tbz x11, #1, 111f\n"
      "ldr d8, [x9], #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d16, [x23], #0x8\n"
      "ldr d20, [x22], #0x8\n"
      "tbz x11, #0, 112f\n"
      "ld1 { v8.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v16.s }[2], [x23]\n"
      "ld1 { v20.s }[2], [x22]\n"
      "b 112f\n"
      "111:"  // Height 4: Partial accumulate: partial_1_0
      "ldr s8, [x9, #0x0]\n"
      "ldr s12, [x24, #0x0]\n"
      "mov x25, #0x0\n"
      "ldr s16, [x23, #0x0]\n"
      "ldr s20, [x22, #0x0]\n"
      "112:"  // Height 4: Partial accumulate: Done
      "sub x9, x9, x25\n"
      "b 115f\n"
      "113:"  // Height 4: full accumulate
      "ldr q8, [x9, #0x0]\n"
      "ldr q9, [x9, #0x10]\n"
      "ldr q10, [x9, #0x20]\n"
      "ldr q11, [x9, #0x30]\n"
      "ldr q12, [x24, #0x0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q16, [x23, #0x0]\n"
      "ldr q17, [x23, #0x10]\n"
      "ldr q18, [x23, #0x20]\n"
      "ldr q19, [x23, #0x30]\n"
      "ldr q20, [x22, #0x0]\n"
      "ldr q21, [x22, #0x10]\n"
      "ldr q22, [x22, #0x20]\n"
      "ldr q23, [x22, #0x30]\n"
      "b 115f\n"
      "114:"  // Height 4: no accumulate
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "115:"  // Height 4: setup done
      "mov x28, #0x0\n"
      "116:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 117f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 118f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "b 118f\n"
      "117:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "118:"  // Height 4: input setup done
      "cmp x27, #0x10\n"
      "blt 121f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q3, [x23, #0x0]\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "blt 120f\n"
      "119:"  // Height 4: Multiply loop: Main loop head
      ".inst 0x4f00f0c8  // sudot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f01f0cc  // sudot v12.4s, v6.16b, v1.4b[0]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f02f0d0  // sudot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f03f0d4  // sudot v20.4s, v6.16b, v3.4b[0]\n"
      "ldr q25, [x10, #0x20]\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f00f0e9  // sudot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f01f0ed  // sudot v13.4s, v7.16b, v1.4b[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f02f0f1  // sudot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f03f0f5  // sudot v21.4s, v7.16b, v3.4b[0]\n"
      "ldr q24, [x10, #0x30]\n"
      "cmp x27, #0x20\n"
      ".inst 0x4f00f32a  // sudot v10.4s, v25.16b, v0.4b[0]\n"
      ".inst 0x4f01f32e  // sudot v14.4s, v25.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f02f332  // sudot v18.4s, v25.16b, v2.4b[0]\n"
      ".inst 0x4f03f336  // sudot v22.4s, v25.16b, v3.4b[0]\n"
      "ldr q25, [x10, #0x40]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f00f30b  // sudot v11.4s, v24.16b, v0.4b[0]\n"
      ".inst 0x4f01f30f  // sudot v15.4s, v24.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4f02f313  // sudot v19.4s, v24.16b, v2.4b[0]\n"
      ".inst 0x4f03f317  // sudot v23.4s, v24.16b, v3.4b[0]\n"
      "ldr q24, [x10, #0x50]\n"
      ".inst 0x4f20f328  // sudot v8.4s, v25.16b, v0.4b[1]\n"
      ".inst 0x4f21f32c  // sudot v12.4s, v25.16b, v1.4b[1]\n"
      ".inst 0x4f22f330  // sudot v16.4s, v25.16b, v2.4b[1]\n"
      ".inst 0x4f23f334  // sudot v20.4s, v25.16b, v3.4b[1]\n"
      "ldr q25, [x10, #0x60]\n"
      ".inst 0x4f20f309  // sudot v9.4s, v24.16b, v0.4b[1]\n"
      ".inst 0x4f21f30d  // sudot v13.4s, v24.16b, v1.4b[1]\n"
      ".inst 0x4f22f311  // sudot v17.4s, v24.16b, v2.4b[1]\n"
      ".inst 0x4f23f315  // sudot v21.4s, v24.16b, v3.4b[1]\n"
      "ldr q24, [x10, #0x70]\n"
      ".inst 0x4f20f32a  // sudot v10.4s, v25.16b, v0.4b[1]\n"
      ".inst 0x4f21f32e  // sudot v14.4s, v25.16b, v1.4b[1]\n"
      ".inst 0x4f22f332  // sudot v18.4s, v25.16b, v2.4b[1]\n"
      ".inst 0x4f23f336  // sudot v22.4s, v25.16b, v3.4b[1]\n"
      "ldr q25, [x10, #0x80]\n"
      ".inst 0x4f20f30b  // sudot v11.4s, v24.16b, v0.4b[1]\n"
      ".inst 0x4f21f30f  // sudot v15.4s, v24.16b, v1.4b[1]\n"
      ".inst 0x4f22f313  // sudot v19.4s, v24.16b, v2.4b[1]\n"
      ".inst 0x4f23f317  // sudot v23.4s, v24.16b, v3.4b[1]\n"
      "ldr q24, [x10, #0x90]\n"
      ".inst 0x4f00fb28  // sudot v8.4s, v25.16b, v0.4b[2]\n"
      ".inst 0x4f01fb2c  // sudot v12.4s, v25.16b, v1.4b[2]\n"
      ".inst 0x4f02fb30  // sudot v16.4s, v25.16b, v2.4b[2]\n"
      ".inst 0x4f03fb34  // sudot v20.4s, v25.16b, v3.4b[2]\n"
      "ldr q25, [x10, #0xa0]\n"
      ".inst 0x4f00fb09  // sudot v9.4s, v24.16b, v0.4b[2]\n"
      ".inst 0x4f01fb0d  // sudot v13.4s, v24.16b, v1.4b[2]\n"
      ".inst 0x4f02fb11  // sudot v17.4s, v24.16b, v2.4b[2]\n"
      ".inst 0x4f03fb15  // sudot v21.4s, v24.16b, v3.4b[2]\n"
      "ldr q24, [x10, #0xb0]\n"
      ".inst 0x4f00fb2a  // sudot v10.4s, v25.16b, v0.4b[2]\n"
      ".inst 0x4f01fb2e  // sudot v14.4s, v25.16b, v1.4b[2]\n"
      ".inst 0x4f02fb32  // sudot v18.4s, v25.16b, v2.4b[2]\n"
      ".inst 0x4f03fb36  // sudot v22.4s, v25.16b, v3.4b[2]\n"
      "ldr q25, [x10, #0xc0]\n"
      ".inst 0x4f00fb0b  // sudot v11.4s, v24.16b, v0.4b[2]\n"
      ".inst 0x4f01fb0f  // sudot v15.4s, v24.16b, v1.4b[2]\n"
      ".inst 0x4f02fb13  // sudot v19.4s, v24.16b, v2.4b[2]\n"
      ".inst 0x4f03fb17  // sudot v23.4s, v24.16b, v3.4b[2]\n"
      "ldr q24, [x10, #0xd0]\n"
      ".inst 0x4f20fb28  // sudot v8.4s, v25.16b, v0.4b[3]\n"
      ".inst 0x4f21fb2c  // sudot v12.4s, v25.16b, v1.4b[3]\n"
      ".inst 0x4f22fb30  // sudot v16.4s, v25.16b, v2.4b[3]\n"
      ".inst 0x4f23fb34  // sudot v20.4s, v25.16b, v3.4b[3]\n"
      "ldr q25, [x10, #0xe0]\n"
      ".inst 0x4f20fb09  // sudot v9.4s, v24.16b, v0.4b[3]\n"
      ".inst 0x4f21fb0d  // sudot v13.4s, v24.16b, v1.4b[3]\n"
      ".inst 0x4f22fb11  // sudot v17.4s, v24.16b, v2.4b[3]\n"
      ".inst 0x4f23fb15  // sudot v21.4s, v24.16b, v3.4b[3]\n"
      "ldr q24, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4f20fb2a  // sudot v10.4s, v25.16b, v0.4b[3]\n"
      ".inst 0x4f21fb2e  // sudot v14.4s, v25.16b, v1.4b[3]\n"
      ".inst 0x4f22fb32  // sudot v18.4s, v25.16b, v2.4b[3]\n"
      ".inst 0x4f23fb36  // sudot v22.4s, v25.16b, v3.4b[3]\n"
      "ldr q6, [x10, #0x0]\n"
      ".inst 0x4f20fb0b  // sudot v11.4s, v24.16b, v0.4b[3]\n"
      "ldr q0, [x26, #0x0]\n"
      ".inst 0x4f21fb0f  // sudot v15.4s, v24.16b, v1.4b[3]\n"
      "ldr q1, [x25, #0x0]\n"
      ".inst 0x4f22fb13  // sudot v19.4s, v24.16b, v2.4b[3]\n"
      "ldr q2, [x24, #0x0]\n"
      ".inst 0x4f23fb17  // sudot v23.4s, v24.16b, v3.4b[3]\n"
      "ldr q3, [x23, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "bge 119b\n"
      "120:"  // Height 4: Multiply loop: Single iteration only
      ".inst 0x4f00f0c8  // sudot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f01f0cc  // sudot v12.4s, v6.16b, v1.4b[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f02f0d0  // sudot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f03f0d4  // sudot v20.4s, v6.16b, v3.4b[0]\n"
      "ldr q25, [x10, #0x20]\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f00f0e9  // sudot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f01f0ed  // sudot v13.4s, v7.16b, v1.4b[0]\n"
      "add x23, x23, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f02f0f1  // sudot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f03f0f5  // sudot v21.4s, v7.16b, v3.4b[0]\n"
      "ldr q24, [x10, #0x30]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f00f32a  // sudot v10.4s, v25.16b, v0.4b[0]\n"
      ".inst 0x4f01f32e  // sudot v14.4s, v25.16b, v1.4b[0]\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f02f332  // sudot v18.4s, v25.16b, v2.4b[0]\n"
      ".inst 0x4f03f336  // sudot v22.4s, v25.16b, v3.4b[0]\n"
      "ldr q25, [x10, #0x40]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4f00f30b  // sudot v11.4s, v24.16b, v0.4b[0]\n"
      ".inst 0x4f01f30f  // sudot v15.4s, v24.16b, v1.4b[0]\n"
      ".inst 0x4f02f313  // sudot v19.4s, v24.16b, v2.4b[0]\n"
      ".inst 0x4f03f317  // sudot v23.4s, v24.16b, v3.4b[0]\n"
      "ldr q24, [x10, #0x50]\n"
      ".inst 0x4f20f328  // sudot v8.4s, v25.16b, v0.4b[1]\n"
      ".inst 0x4f21f32c  // sudot v12.4s, v25.16b, v1.4b[1]\n"
      ".inst 0x4f22f330  // sudot v16.4s, v25.16b, v2.4b[1]\n"
      ".inst 0x4f23f334  // sudot v20.4s, v25.16b, v3.4b[1]\n"
      "ldr q25, [x10, #0x60]\n"
      ".inst 0x4f20f309  // sudot v9.4s, v24.16b, v0.4b[1]\n"
      ".inst 0x4f21f30d  // sudot v13.4s, v24.16b, v1.4b[1]\n"
      ".inst 0x4f22f311  // sudot v17.4s, v24.16b, v2.4b[1]\n"
      ".inst 0x4f23f315  // sudot v21.4s, v24.16b, v3.4b[1]\n"
      "ldr q24, [x10, #0x70]\n"
      ".inst 0x4f20f32a  // sudot v10.4s, v25.16b, v0.4b[1]\n"
      ".inst 0x4f21f32e  // sudot v14.4s, v25.16b, v1.4b[1]\n"
      ".inst 0x4f22f332  // sudot v18.4s, v25.16b, v2.4b[1]\n"
      ".inst 0x4f23f336  // sudot v22.4s, v25.16b, v3.4b[1]\n"
      "ldr q25, [x10, #0x80]\n"
      ".inst 0x4f20f30b  // sudot v11.4s, v24.16b, v0.4b[1]\n"
      ".inst 0x4f21f30f  // sudot v15.4s, v24.16b, v1.4b[1]\n"
      ".inst 0x4f22f313  // sudot v19.4s, v24.16b, v2.4b[1]\n"
      ".inst 0x4f23f317  // sudot v23.4s, v24.16b, v3.4b[1]\n"
      "ldr q24, [x10, #0x90]\n"
      ".inst 0x4f00fb28  // sudot v8.4s, v25.16b, v0.4b[2]\n"
      ".inst 0x4f01fb2c  // sudot v12.4s, v25.16b, v1.4b[2]\n"
      ".inst 0x4f02fb30  // sudot v16.4s, v25.16b, v2.4b[2]\n"
      ".inst 0x4f03fb34  // sudot v20.4s, v25.16b, v3.4b[2]\n"
      "ldr q25, [x10, #0xa0]\n"
      ".inst 0x4f00fb09  // sudot v9.4s, v24.16b, v0.4b[2]\n"
      ".inst 0x4f01fb0d  // sudot v13.4s, v24.16b, v1.4b[2]\n"
      ".inst 0x4f02fb11  // sudot v17.4s, v24.16b, v2.4b[2]\n"
      ".inst 0x4f03fb15  // sudot v21.4s, v24.16b, v3.4b[2]\n"
      "ldr q24, [x10, #0xb0]\n"
      ".inst 0x4f00fb2a  // sudot v10.4s, v25.16b, v0.4b[2]\n"
      ".inst 0x4f01fb2e  // sudot v14.4s, v25.16b, v1.4b[2]\n"
      ".inst 0x4f02fb32  // sudot v18.4s, v25.16b, v2.4b[2]\n"
      ".inst 0x4f03fb36  // sudot v22.4s, v25.16b, v3.4b[2]\n"
      "ldr q25, [x10, #0xc0]\n"
      ".inst 0x4f00fb0b  // sudot v11.4s, v24.16b, v0.4b[2]\n"
      ".inst 0x4f01fb0f  // sudot v15.4s, v24.16b, v1.4b[2]\n"
      ".inst 0x4f02fb13  // sudot v19.4s, v24.16b, v2.4b[2]\n"
      ".inst 0x4f03fb17  // sudot v23.4s, v24.16b, v3.4b[2]\n"
      "ldr q24, [x10, #0xd0]\n"
      ".inst 0x4f20fb28  // sudot v8.4s, v25.16b, v0.4b[3]\n"
      ".inst 0x4f21fb2c  // sudot v12.4s, v25.16b, v1.4b[3]\n"
      ".inst 0x4f22fb30  // sudot v16.4s, v25.16b, v2.4b[3]\n"
      ".inst 0x4f23fb34  // sudot v20.4s, v25.16b, v3.4b[3]\n"
      "ldr q25, [x10, #0xe0]\n"
      ".inst 0x4f20fb09  // sudot v9.4s, v24.16b, v0.4b[3]\n"
      ".inst 0x4f21fb0d  // sudot v13.4s, v24.16b, v1.4b[3]\n"
      ".inst 0x4f22fb11  // sudot v17.4s, v24.16b, v2.4b[3]\n"
      ".inst 0x4f23fb15  // sudot v21.4s, v24.16b, v3.4b[3]\n"
      "ldr q24, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4f20fb2a  // sudot v10.4s, v25.16b, v0.4b[3]\n"
      ".inst 0x4f21fb2e  // sudot v14.4s, v25.16b, v1.4b[3]\n"
      ".inst 0x4f22fb32  // sudot v18.4s, v25.16b, v2.4b[3]\n"
      ".inst 0x4f23fb36  // sudot v22.4s, v25.16b, v3.4b[3]\n"
      ".inst 0x4f20fb0b  // sudot v11.4s, v24.16b, v0.4b[3]\n"
      ".inst 0x4f21fb0f  // sudot v15.4s, v24.16b, v1.4b[3]\n"
      ".inst 0x4f22fb13  // sudot v19.4s, v24.16b, v2.4b[3]\n"
      ".inst 0x4f23fb17  // sudot v23.4s, v24.16b, v3.4b[3]\n"
      "121:"  // Height 4: Multiply loop: Main loop skip
      "cbz x27, 126f\n"
      "cmp x27, #0x4\n"
      "blt 123f\n"
      "122:"  // Height 4: Multiply loop: Odd block loop
      "ldr s29, [x26], #0x4\n"
      "ldr s28, [x25], #0x4\n"
      "sub x27, x27, #0x4\n"
      "ldr s27, [x24], #0x4\n"
      "ldr s26, [x23], #0x4\n"
      "cmp x27, #0x4\n"
      "ldr q25, [x10, #0x0]\n"
      "ldr q24, [x10, #0x10]\n"
      ".inst 0x4f1df328  // sudot v8.4s, v25.16b, v29.4b[0]\n"
      ".inst 0x4f1cf32c  // sudot v12.4s, v25.16b, v28.4b[0]\n"
      ".inst 0x4f1bf330  // sudot v16.4s, v25.16b, v27.4b[0]\n"
      ".inst 0x4f1af334  // sudot v20.4s, v25.16b, v26.4b[0]\n"
      "ldr q25, [x10, #0x20]\n"
      ".inst 0x4f1df309  // sudot v9.4s, v24.16b, v29.4b[0]\n"
      ".inst 0x4f1cf30d  // sudot v13.4s, v24.16b, v28.4b[0]\n"
      ".inst 0x4f1bf311  // sudot v17.4s, v24.16b, v27.4b[0]\n"
      ".inst 0x4f1af315  // sudot v21.4s, v24.16b, v26.4b[0]\n"
      "ldr q24, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f1df32a  // sudot v10.4s, v25.16b, v29.4b[0]\n"
      ".inst 0x4f1cf32e  // sudot v14.4s, v25.16b, v28.4b[0]\n"
      ".inst 0x4f1bf332  // sudot v18.4s, v25.16b, v27.4b[0]\n"
      ".inst 0x4f1af336  // sudot v22.4s, v25.16b, v26.4b[0]\n"
      ".inst 0x4f1df30b  // sudot v11.4s, v24.16b, v29.4b[0]\n"
      ".inst 0x4f1cf30f  // sudot v15.4s, v24.16b, v28.4b[0]\n"
      ".inst 0x4f1bf313  // sudot v19.4s, v24.16b, v27.4b[0]\n"
      ".inst 0x4f1af317  // sudot v23.4s, v24.16b, v26.4b[0]\n"
      "bge 122b\n"
      "123:"  // Height 4: Multiply loop: Skip odd blocks
      "cbz x27, 126f\n"
      "tbz x27, #1, 124f\n"
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "ldr h2, [x24], #0x2\n"
      "ldr h3, [x23], #0x2\n"
      "tbz x27, #0, 125f\n"
      "ld1 { v0.b }[2], [x26]\n"
      "ld1 { v1.b }[2], [x25]\n"
      "ld1 { v2.b }[2], [x24]\n"
      "ld1 { v3.b }[2], [x23]\n"
      "b 125f\n"
      "124:"  // Height 4: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x26, #0x0]\n"
      "ldr b1, [x25, #0x0]\n"
      "ldr b2, [x24, #0x0]\n"
      "ldr b3, [x23, #0x0]\n"
      "125:"  // Height 4: Multiply loop: Ragged operand read: Done
      "ldr q25, [x10, #0x0]\n"
      "ldr q24, [x10, #0x10]\n"
      ".inst 0x4f00f328  // sudot v8.4s, v25.16b, v0.4b[0]\n"
      ".inst 0x4f01f32c  // sudot v12.4s, v25.16b, v1.4b[0]\n"
      ".inst 0x4f02f330  // sudot v16.4s, v25.16b, v2.4b[0]\n"
      ".inst 0x4f03f334  // sudot v20.4s, v25.16b, v3.4b[0]\n"
      "ldr q25, [x10, #0x20]\n"
      ".inst 0x4f00f309  // sudot v9.4s, v24.16b, v0.4b[0]\n"
      ".inst 0x4f01f30d  // sudot v13.4s, v24.16b, v1.4b[0]\n"
      ".inst 0x4f02f311  // sudot v17.4s, v24.16b, v2.4b[0]\n"
      ".inst 0x4f03f315  // sudot v21.4s, v24.16b, v3.4b[0]\n"
      "ldr q24, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f00f32a  // sudot v10.4s, v25.16b, v0.4b[0]\n"
      ".inst 0x4f01f32e  // sudot v14.4s, v25.16b, v1.4b[0]\n"
      ".inst 0x4f02f332  // sudot v18.4s, v25.16b, v2.4b[0]\n"
      ".inst 0x4f03f336  // sudot v22.4s, v25.16b, v3.4b[0]\n"
      ".inst 0x4f00f30b  // sudot v11.4s, v24.16b, v0.4b[0]\n"
      ".inst 0x4f01f30f  // sudot v15.4s, v24.16b, v1.4b[0]\n"
      ".inst 0x4f02f313  // sudot v19.4s, v24.16b, v2.4b[0]\n"
      ".inst 0x4f03f317  // sudot v23.4s, v24.16b, v3.4b[0]\n"
      "126:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 116b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x11, #0x10\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "add x24, x9, x20, LSL #2\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "add x23, x24, x20, LSL #2\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "add x22, x23, x20, LSL #2\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "bge 135f\n"
      "tbz x11, #3, 130f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v9.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v13.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v17.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v21.4s }, [x22], #0x10\n"
      "tbz x11, #2, 128f\n"
      "st1 { v10.4s }, [x9], #0x10\n"
      "st1 { v14.4s }, [x24], #0x10\n"
      "st1 { v18.4s }, [x23], #0x10\n"
      "st1 { v22.4s }, [x22], #0x10\n"
      "tbz x11, #1, 127f\n"
      "str d11, [x9], #0x8\n"
      "str d15, [x24], #0x8\n"
      "str d19, [x23], #0x8\n"
      "str d23, [x22], #0x8\n"
      "tbz x11, #0, 134f\n"
      "st1 { v11.s }[2], [x9]\n"
      "st1 { v15.s }[2], [x24]\n"
      "st1 { v19.s }[2], [x23]\n"
      "st1 { v23.s }[2], [x22]\n"
      "b 134f\n"
      "127:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x11, #0, 134f\n"
      "str s11, [x9, #0x0]\n"
      "str s15, [x24, #0x0]\n"
      "str s19, [x23, #0x0]\n"
      "str s23, [x22, #0x0]\n"
      "b 134f\n"
      "128:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x11, #1, 129f\n"
      "str d10, [x9], #0x8\n"
      "str d14, [x24], #0x8\n"
      "str d18, [x23], #0x8\n"
      "str d22, [x22], #0x8\n"
      "tbz x11, #0, 134f\n"
      "st1 { v10.s }[2], [x9]\n"
      "st1 { v14.s }[2], [x24]\n"
      "st1 { v18.s }[2], [x23]\n"
      "st1 { v22.s }[2], [x22]\n"
      "b 134f\n"
      "129:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x11, #0, 134f\n"
      "str s10, [x9, #0x0]\n"
      "str s14, [x24, #0x0]\n"
      "str s18, [x23, #0x0]\n"
      "str s22, [x22, #0x0]\n"
      "b 134f\n"
      "130:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x11, #2, 132f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "tbz x11, #1, 131f\n"
      "str d9, [x9], #0x8\n"
      "str d13, [x24], #0x8\n"
      "str d17, [x23], #0x8\n"
      "str d21, [x22], #0x8\n"
      "tbz x11, #0, 134f\n"
      "st1 { v9.s }[2], [x9]\n"
      "st1 { v13.s }[2], [x24]\n"
      "st1 { v17.s }[2], [x23]\n"
      "st1 { v21.s }[2], [x22]\n"
      "b 134f\n"
      "131:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x11, #0, 134f\n"
      "str s9, [x9, #0x0]\n"
      "str s13, [x24, #0x0]\n"
      "str s17, [x23, #0x0]\n"
      "str s21, [x22, #0x0]\n"
      "b 134f\n"
      "132:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x11, #1, 133f\n"
      "str d8, [x9], #0x8\n"
      "str d12, [x24], #0x8\n"
      "str d16, [x23], #0x8\n"
      "str d20, [x22], #0x8\n"
      "tbz x11, #0, 134f\n"
      "st1 { v8.s }[2], [x9]\n"
      "st1 { v12.s }[2], [x24]\n"
      "st1 { v16.s }[2], [x23]\n"
      "st1 { v20.s }[2], [x22]\n"
      "b 134f\n"
      "133:"  // Height 4: Partial direct writeback: partial_1_0
      "str s8, [x9, #0x0]\n"
      "str s12, [x24, #0x0]\n"
      "str s16, [x23, #0x0]\n"
      "str s20, [x22, #0x0]\n"
      "134:"  // Height 4: Partial direct writeback: Done
      "b 136f\n"
      "135:"  // Height 4: Full writeback
      "str q8, [x9, #0x0]\n"
      "str q9, [x9, #0x10]\n"
      "str q10, [x9, #0x20]\n"
      "str q11, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q12, [x24, #0x0]\n"
      "str q13, [x24, #0x10]\n"
      "str q14, [x24, #0x20]\n"
      "str q15, [x24, #0x30]\n"
      "str q16, [x23, #0x0]\n"
      "str q17, [x23, #0x10]\n"
      "str q18, [x23, #0x20]\n"
      "str q19, [x23, #0x30]\n"
      "str q20, [x22, #0x0]\n"
      "str q21, [x22, #0x10]\n"
      "str q22, [x22, #0x20]\n"
      "str q23, [x22, #0x30]\n"
      "136:"  // Height 4: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 104b\n"
      "b 206f\n"
      "137:"  // Height 5
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "138:"  // Height 5: Column loop
      "tbz %x[flags], #0, 148f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x11, #0x10\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "bge 147f\n"
      "tbz x11, #3, 142f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v24.4s }, [x21], #0x10\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "ld1 { v25.4s }, [x21], #0x10\n"
      "tbz x11, #2, 140f\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "ld1 { v26.4s }, [x21], #0x10\n"
      "tbz x11, #1, 139f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d15, [x24], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d19, [x23], #0x8\n"
      "ldr d23, [x22], #0x8\n"
      "ldr d27, [x21], #0x8\n"
      "tbz x11, #0, 146f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "ld1 { v27.s }[2], [x21]\n"
      "b 146f\n"
      "139:"  // Height 5: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x11, #0, 146f\n"
      "ldr s11, [x9, #0x0]\n"
      "ldr s15, [x24, #0x0]\n"
      "ldr s19, [x23, #0x0]\n"
      "ldr s23, [x22, #0x0]\n"
      "ldr s27, [x21, #0x0]\n"
      "b 146f\n"
      "140:"  // Height 5: Partial accumulate: partial_2_8
      "tbz x11, #1, 141f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d14, [x24], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d18, [x23], #0x8\n"
      "ldr d22, [x22], #0x8\n"
      "ldr d26, [x21], #0x8\n"
      "tbz x11, #0, 146f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "ld1 { v26.s }[2], [x21]\n"
      "b 146f\n"
      "141:"  // Height 5: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x11, #0, 146f\n"
      "ldr s10, [x9, #0x0]\n"
      "ldr s14, [x24, #0x0]\n"
      "ldr s18, [x23, #0x0]\n"
      "ldr s22, [x22, #0x0]\n"
      "ldr s26, [x21, #0x0]\n"
      "b 146f\n"
      "142:"  // Height 5: Partial accumulate: partial_4_0
      "tbz x11, #2, 144f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v24.4s }, [x21], #0x10\n"
      "tbz x11, #1, 143f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d13, [x24], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d17, [x23], #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "ldr d25, [x21], #0x8\n"
      "tbz x11, #0, 146f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "ld1 { v25.s }[2], [x21]\n"
      "b 146f\n"
      "143:"  // Height 5: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x11, #0, 146f\n"
      "ldr s9, [x9, #0x0]\n"
      "ldr s13, [x24, #0x0]\n"
      "ldr s17, [x23, #0x0]\n"
      "ldr s21, [x22, #0x0]\n"
      "ldr s25, [x21, #0x0]\n"
      "b 146f\n"
      "144:"  // Height 5: Partial accumulate: partial_2_0
      "tbz x11, #1, 145f\n"
      "ldr d8, [x9], #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d16, [x23], #0x8\n"
      "ldr d20, [x22], #0x8\n"
      "ldr d24, [x21], #0x8\n"
      "tbz x11, #0, 146f\n"
      "ld1 { v8.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v16.s }[2], [x23]\n"
      "ld1 { v20.s }[2], [x22]\n"
      "ld1 { v24.s }[2], [x21]\n"
      "b 146f\n"
      "145:"  // Height 5: Partial accumulate: partial_1_0
      "ldr s8, [x9, #0x0]\n"
      "ldr s12, [x24, #0x0]\n"
      "mov x25, #0x0\n"
      "ldr s16, [x23, #0x0]\n"
      "ldr s20, [x22, #0x0]\n"
      "ldr s24, [x21, #0x0]\n"
      "146:"  // Height 5: Partial accumulate: Done
      "sub x9, x9, x25\n"
      "b 149f\n"
      "147:"  // Height 5: full accumulate
      "ldr q8, [x9, #0x0]\n"
      "ldr q9, [x9, #0x10]\n"
      "ldr q10, [x9, #0x20]\n"
      "ldr q11, [x9, #0x30]\n"
      "ldr q12, [x24, #0x0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q16, [x23, #0x0]\n"
      "ldr q17, [x23, #0x10]\n"
      "ldr q18, [x23, #0x20]\n"
      "ldr q19, [x23, #0x30]\n"
      "ldr q20, [x22, #0x0]\n"
      "ldr q21, [x22, #0x10]\n"
      "ldr q22, [x22, #0x20]\n"
      "ldr q23, [x22, #0x30]\n"
      "ldr q24, [x21, #0x0]\n"
      "ldr q25, [x21, #0x10]\n"
      "ldr q26, [x21, #0x20]\n"
      "ldr q27, [x21, #0x30]\n"
      "b 149f\n"
      "148:"  // Height 5: no accumulate
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "149:"  // Height 5: setup done
      "mov x28, #0x0\n"
      "150:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 151f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 152f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "b 152f\n"
      "151:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "152:"  // Height 5: input setup done
      "cmp x27, #0x10\n"
      "blt 155f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q3, [x23, #0x0]\n"
      "ldr q4, [x22, #0x0]\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "blt 154f\n"
      "153:"  // Height 5: Multiply loop: Main loop head
      ".inst 0x4f00f0c8  // sudot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f01f0cc  // sudot v12.4s, v6.16b, v1.4b[0]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f02f0d0  // sudot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f03f0d4  // sudot v20.4s, v6.16b, v3.4b[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f04f0d8  // sudot v24.4s, v6.16b, v4.4b[0]\n"
      "ldr q29, [x10, #0x20]\n"
      ".inst 0x4f00f0e9  // sudot v9.4s, v7.16b, v0.4b[0]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f01f0ed  // sudot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f02f0f1  // sudot v17.4s, v7.16b, v2.4b[0]\n"
      "add x22, x22, #0x10\n"
      "cmp x27, #0x20\n"
      ".inst 0x4f03f0f5  // sudot v21.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f04f0f9  // sudot v25.4s, v7.16b, v4.4b[0]\n"
      "ldr q28, [x10, #0x30]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f00f3aa  // sudot v10.4s, v29.16b, v0.4b[0]\n"
      ".inst 0x4f01f3ae  // sudot v14.4s, v29.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f02f3b2  // sudot v18.4s, v29.16b, v2.4b[0]\n"
      ".inst 0x4f03f3b6  // sudot v22.4s, v29.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4f04f3ba  // sudot v26.4s, v29.16b, v4.4b[0]\n"
      "ldr q29, [x10, #0x40]\n"
      ".inst 0x4f00f38b  // sudot v11.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x4f01f38f  // sudot v15.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x4f02f393  // sudot v19.4s, v28.16b, v2.4b[0]\n"
      ".inst 0x4f03f397  // sudot v23.4s, v28.16b, v3.4b[0]\n"
      ".inst 0x4f04f39b  // sudot v27.4s, v28.16b, v4.4b[0]\n"
      "ldr q28, [x10, #0x50]\n"
      ".inst 0x4f20f3a8  // sudot v8.4s, v29.16b, v0.4b[1]\n"
      ".inst 0x4f21f3ac  // sudot v12.4s, v29.16b, v1.4b[1]\n"
      ".inst 0x4f22f3b0  // sudot v16.4s, v29.16b, v2.4b[1]\n"
      ".inst 0x4f23f3b4  // sudot v20.4s, v29.16b, v3.4b[1]\n"
      ".inst 0x4f24f3b8  // sudot v24.4s, v29.16b, v4.4b[1]\n"
      "ldr q29, [x10, #0x60]\n"
      ".inst 0x4f20f389  // sudot v9.4s, v28.16b, v0.4b[1]\n"
      ".inst 0x4f21f38d  // sudot v13.4s, v28.16b, v1.4b[1]\n"
      ".inst 0x4f22f391  // sudot v17.4s, v28.16b, v2.4b[1]\n"
      ".inst 0x4f23f395  // sudot v21.4s, v28.16b, v3.4b[1]\n"
      ".inst 0x4f24f399  // sudot v25.4s, v28.16b, v4.4b[1]\n"
      "ldr q28, [x10, #0x70]\n"
      ".inst 0x4f20f3aa  // sudot v10.4s, v29.16b, v0.4b[1]\n"
      ".inst 0x4f21f3ae  // sudot v14.4s, v29.16b, v1.4b[1]\n"
      ".inst 0x4f22f3b2  // sudot v18.4s, v29.16b, v2.4b[1]\n"
      ".inst 0x4f23f3b6  // sudot v22.4s, v29.16b, v3.4b[1]\n"
      ".inst 0x4f24f3ba  // sudot v26.4s, v29.16b, v4.4b[1]\n"
      "ldr q29, [x10, #0x80]\n"
      ".inst 0x4f20f38b  // sudot v11.4s, v28.16b, v0.4b[1]\n"
      ".inst 0x4f21f38f  // sudot v15.4s, v28.16b, v1.4b[1]\n"
      ".inst 0x4f22f393  // sudot v19.4s, v28.16b, v2.4b[1]\n"
      ".inst 0x4f23f397  // sudot v23.4s, v28.16b, v3.4b[1]\n"
      ".inst 0x4f24f39b  // sudot v27.4s, v28.16b, v4.4b[1]\n"
      "ldr q28, [x10, #0x90]\n"
      ".inst 0x4f00fba8  // sudot v8.4s, v29.16b, v0.4b[2]\n"
      ".inst 0x4f01fbac  // sudot v12.4s, v29.16b, v1.4b[2]\n"
      ".inst 0x4f02fbb0  // sudot v16.4s, v29.16b, v2.4b[2]\n"
      ".inst 0x4f03fbb4  // sudot v20.4s, v29.16b, v3.4b[2]\n"
      ".inst 0x4f04fbb8  // sudot v24.4s, v29.16b, v4.4b[2]\n"
      "ldr q29, [x10, #0xa0]\n"
      ".inst 0x4f00fb89  // sudot v9.4s, v28.16b, v0.4b[2]\n"
      ".inst 0x4f01fb8d  // sudot v13.4s, v28.16b, v1.4b[2]\n"
      ".inst 0x4f02fb91  // sudot v17.4s, v28.16b, v2.4b[2]\n"
      ".inst 0x4f03fb95  // sudot v21.4s, v28.16b, v3.4b[2]\n"
      ".inst 0x4f04fb99  // sudot v25.4s, v28.16b, v4.4b[2]\n"
      "ldr q28, [x10, #0xb0]\n"
      ".inst 0x4f00fbaa  // sudot v10.4s, v29.16b, v0.4b[2]\n"
      ".inst 0x4f01fbae  // sudot v14.4s, v29.16b, v1.4b[2]\n"
      ".inst 0x4f02fbb2  // sudot v18.4s, v29.16b, v2.4b[2]\n"
      ".inst 0x4f03fbb6  // sudot v22.4s, v29.16b, v3.4b[2]\n"
      ".inst 0x4f04fbba  // sudot v26.4s, v29.16b, v4.4b[2]\n"
      "ldr q29, [x10, #0xc0]\n"
      ".inst 0x4f00fb8b  // sudot v11.4s, v28.16b, v0.4b[2]\n"
      ".inst 0x4f01fb8f  // sudot v15.4s, v28.16b, v1.4b[2]\n"
      ".inst 0x4f02fb93  // sudot v19.4s, v28.16b, v2.4b[2]\n"
      ".inst 0x4f03fb97  // sudot v23.4s, v28.16b, v3.4b[2]\n"
      ".inst 0x4f04fb9b  // sudot v27.4s, v28.16b, v4.4b[2]\n"
      "ldr q28, [x10, #0xd0]\n"
      ".inst 0x4f20fba8  // sudot v8.4s, v29.16b, v0.4b[3]\n"
      ".inst 0x4f21fbac  // sudot v12.4s, v29.16b, v1.4b[3]\n"
      ".inst 0x4f22fbb0  // sudot v16.4s, v29.16b, v2.4b[3]\n"
      ".inst 0x4f23fbb4  // sudot v20.4s, v29.16b, v3.4b[3]\n"
      ".inst 0x4f24fbb8  // sudot v24.4s, v29.16b, v4.4b[3]\n"
      "ldr q29, [x10, #0xe0]\n"
      ".inst 0x4f20fb89  // sudot v9.4s, v28.16b, v0.4b[3]\n"
      ".inst 0x4f21fb8d  // sudot v13.4s, v28.16b, v1.4b[3]\n"
      ".inst 0x4f22fb91  // sudot v17.4s, v28.16b, v2.4b[3]\n"
      ".inst 0x4f23fb95  // sudot v21.4s, v28.16b, v3.4b[3]\n"
      ".inst 0x4f24fb99  // sudot v25.4s, v28.16b, v4.4b[3]\n"
      "ldr q28, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4f20fbaa  // sudot v10.4s, v29.16b, v0.4b[3]\n"
      ".inst 0x4f21fbae  // sudot v14.4s, v29.16b, v1.4b[3]\n"
      ".inst 0x4f22fbb2  // sudot v18.4s, v29.16b, v2.4b[3]\n"
      ".inst 0x4f23fbb6  // sudot v22.4s, v29.16b, v3.4b[3]\n"
      ".inst 0x4f24fbba  // sudot v26.4s, v29.16b, v4.4b[3]\n"
      "ldr q6, [x10, #0x0]\n"
      ".inst 0x4f20fb8b  // sudot v11.4s, v28.16b, v0.4b[3]\n"
      "ldr q0, [x26, #0x0]\n"
      ".inst 0x4f21fb8f  // sudot v15.4s, v28.16b, v1.4b[3]\n"
      "ldr q1, [x25, #0x0]\n"
      ".inst 0x4f22fb93  // sudot v19.4s, v28.16b, v2.4b[3]\n"
      "ldr q2, [x24, #0x0]\n"
      ".inst 0x4f23fb97  // sudot v23.4s, v28.16b, v3.4b[3]\n"
      "ldr q3, [x23, #0x0]\n"
      ".inst 0x4f24fb9b  // sudot v27.4s, v28.16b, v4.4b[3]\n"
      "ldr q4, [x22, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "bge 153b\n"
      "154:"  // Height 5: Multiply loop: Single iteration only
      ".inst 0x4f00f0c8  // sudot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f01f0cc  // sudot v12.4s, v6.16b, v1.4b[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f02f0d0  // sudot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f03f0d4  // sudot v20.4s, v6.16b, v3.4b[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f04f0d8  // sudot v24.4s, v6.16b, v4.4b[0]\n"
      "ldr q29, [x10, #0x20]\n"
      ".inst 0x4f00f0e9  // sudot v9.4s, v7.16b, v0.4b[0]\n"
      "add x22, x22, #0x10\n"
      ".inst 0x4f01f0ed  // sudot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f02f0f1  // sudot v17.4s, v7.16b, v2.4b[0]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f03f0f5  // sudot v21.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f04f0f9  // sudot v25.4s, v7.16b, v4.4b[0]\n"
      "ldr q28, [x10, #0x30]\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4f00f3aa  // sudot v10.4s, v29.16b, v0.4b[0]\n"
      ".inst 0x4f01f3ae  // sudot v14.4s, v29.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4f02f3b2  // sudot v18.4s, v29.16b, v2.4b[0]\n"
      ".inst 0x4f03f3b6  // sudot v22.4s, v29.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4f04f3ba  // sudot v26.4s, v29.16b, v4.4b[0]\n"
      "ldr q29, [x10, #0x40]\n"
      ".inst 0x4f00f38b  // sudot v11.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x4f01f38f  // sudot v15.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x4f02f393  // sudot v19.4s, v28.16b, v2.4b[0]\n"
      ".inst 0x4f03f397  // sudot v23.4s, v28.16b, v3.4b[0]\n"
      ".inst 0x4f04f39b  // sudot v27.4s, v28.16b, v4.4b[0]\n"
      "ldr q28, [x10, #0x50]\n"
      ".inst 0x4f20f3a8  // sudot v8.4s, v29.16b, v0.4b[1]\n"
      ".inst 0x4f21f3ac  // sudot v12.4s, v29.16b, v1.4b[1]\n"
      ".inst 0x4f22f3b0  // sudot v16.4s, v29.16b, v2.4b[1]\n"
      ".inst 0x4f23f3b4  // sudot v20.4s, v29.16b, v3.4b[1]\n"
      ".inst 0x4f24f3b8  // sudot v24.4s, v29.16b, v4.4b[1]\n"
      "ldr q29, [x10, #0x60]\n"
      ".inst 0x4f20f389  // sudot v9.4s, v28.16b, v0.4b[1]\n"
      ".inst 0x4f21f38d  // sudot v13.4s, v28.16b, v1.4b[1]\n"
      ".inst 0x4f22f391  // sudot v17.4s, v28.16b, v2.4b[1]\n"
      ".inst 0x4f23f395  // sudot v21.4s, v28.16b, v3.4b[1]\n"
      ".inst 0x4f24f399  // sudot v25.4s, v28.16b, v4.4b[1]\n"
      "ldr q28, [x10, #0x70]\n"
      ".inst 0x4f20f3aa  // sudot v10.4s, v29.16b, v0.4b[1]\n"
      ".inst 0x4f21f3ae  // sudot v14.4s, v29.16b, v1.4b[1]\n"
      ".inst 0x4f22f3b2  // sudot v18.4s, v29.16b, v2.4b[1]\n"
      ".inst 0x4f23f3b6  // sudot v22.4s, v29.16b, v3.4b[1]\n"
      ".inst 0x4f24f3ba  // sudot v26.4s, v29.16b, v4.4b[1]\n"
      "ldr q29, [x10, #0x80]\n"
      ".inst 0x4f20f38b  // sudot v11.4s, v28.16b, v0.4b[1]\n"
      ".inst 0x4f21f38f  // sudot v15.4s, v28.16b, v1.4b[1]\n"
      ".inst 0x4f22f393  // sudot v19.4s, v28.16b, v2.4b[1]\n"
      ".inst 0x4f23f397  // sudot v23.4s, v28.16b, v3.4b[1]\n"
      ".inst 0x4f24f39b  // sudot v27.4s, v28.16b, v4.4b[1]\n"
      "ldr q28, [x10, #0x90]\n"
      ".inst 0x4f00fba8  // sudot v8.4s, v29.16b, v0.4b[2]\n"
      ".inst 0x4f01fbac  // sudot v12.4s, v29.16b, v1.4b[2]\n"
      ".inst 0x4f02fbb0  // sudot v16.4s, v29.16b, v2.4b[2]\n"
      ".inst 0x4f03fbb4  // sudot v20.4s, v29.16b, v3.4b[2]\n"
      ".inst 0x4f04fbb8  // sudot v24.4s, v29.16b, v4.4b[2]\n"
      "ldr q29, [x10, #0xa0]\n"
      ".inst 0x4f00fb89  // sudot v9.4s, v28.16b, v0.4b[2]\n"
      ".inst 0x4f01fb8d  // sudot v13.4s, v28.16b, v1.4b[2]\n"
      ".inst 0x4f02fb91  // sudot v17.4s, v28.16b, v2.4b[2]\n"
      ".inst 0x4f03fb95  // sudot v21.4s, v28.16b, v3.4b[2]\n"
      ".inst 0x4f04fb99  // sudot v25.4s, v28.16b, v4.4b[2]\n"
      "ldr q28, [x10, #0xb0]\n"
      ".inst 0x4f00fbaa  // sudot v10.4s, v29.16b, v0.4b[2]\n"
      ".inst 0x4f01fbae  // sudot v14.4s, v29.16b, v1.4b[2]\n"
      ".inst 0x4f02fbb2  // sudot v18.4s, v29.16b, v2.4b[2]\n"
      ".inst 0x4f03fbb6  // sudot v22.4s, v29.16b, v3.4b[2]\n"
      ".inst 0x4f04fbba  // sudot v26.4s, v29.16b, v4.4b[2]\n"
      "ldr q29, [x10, #0xc0]\n"
      ".inst 0x4f00fb8b  // sudot v11.4s, v28.16b, v0.4b[2]\n"
      ".inst 0x4f01fb8f  // sudot v15.4s, v28.16b, v1.4b[2]\n"
      ".inst 0x4f02fb93  // sudot v19.4s, v28.16b, v2.4b[2]\n"
      ".inst 0x4f03fb97  // sudot v23.4s, v28.16b, v3.4b[2]\n"
      ".inst 0x4f04fb9b  // sudot v27.4s, v28.16b, v4.4b[2]\n"
      "ldr q28, [x10, #0xd0]\n"
      ".inst 0x4f20fba8  // sudot v8.4s, v29.16b, v0.4b[3]\n"
      ".inst 0x4f21fbac  // sudot v12.4s, v29.16b, v1.4b[3]\n"
      ".inst 0x4f22fbb0  // sudot v16.4s, v29.16b, v2.4b[3]\n"
      ".inst 0x4f23fbb4  // sudot v20.4s, v29.16b, v3.4b[3]\n"
      ".inst 0x4f24fbb8  // sudot v24.4s, v29.16b, v4.4b[3]\n"
      "ldr q29, [x10, #0xe0]\n"
      ".inst 0x4f20fb89  // sudot v9.4s, v28.16b, v0.4b[3]\n"
      ".inst 0x4f21fb8d  // sudot v13.4s, v28.16b, v1.4b[3]\n"
      ".inst 0x4f22fb91  // sudot v17.4s, v28.16b, v2.4b[3]\n"
      ".inst 0x4f23fb95  // sudot v21.4s, v28.16b, v3.4b[3]\n"
      ".inst 0x4f24fb99  // sudot v25.4s, v28.16b, v4.4b[3]\n"
      "ldr q28, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4f20fbaa  // sudot v10.4s, v29.16b, v0.4b[3]\n"
      ".inst 0x4f21fbae  // sudot v14.4s, v29.16b, v1.4b[3]\n"
      ".inst 0x4f22fbb2  // sudot v18.4s, v29.16b, v2.4b[3]\n"
      ".inst 0x4f23fbb6  // sudot v22.4s, v29.16b, v3.4b[3]\n"
      ".inst 0x4f24fbba  // sudot v26.4s, v29.16b, v4.4b[3]\n"
      ".inst 0x4f20fb8b  // sudot v11.4s, v28.16b, v0.4b[3]\n"
      ".inst 0x4f21fb8f  // sudot v15.4s, v28.16b, v1.4b[3]\n"
      ".inst 0x4f22fb93  // sudot v19.4s, v28.16b, v2.4b[3]\n"
      ".inst 0x4f23fb97  // sudot v23.4s, v28.16b, v3.4b[3]\n"
      ".inst 0x4f24fb9b  // sudot v27.4s, v28.16b, v4.4b[3]\n"
      "155:"  // Height 5: Multiply loop: Main loop skip
      "cbz x27, 160f\n"
      "cmp x27, #0x4\n"
      "blt 157f\n"
      "156:"  // Height 5: Multiply loop: Odd block loop
      "ldr s2, [x26], #0x4\n"
      "ldr s1, [x25], #0x4\n"
      "sub x27, x27, #0x4\n"
      "ldr s0, [x24], #0x4\n"
      "ldr s31, [x23], #0x4\n"
      "cmp x27, #0x4\n"
      "ldr s30, [x22], #0x4\n"
      "ldr q29, [x10, #0x0]\n"
      "ldr q28, [x10, #0x10]\n"
      ".inst 0x4f02f3a8  // sudot v8.4s, v29.16b, v2.4b[0]\n"
      ".inst 0x4f01f3ac  // sudot v12.4s, v29.16b, v1.4b[0]\n"
      ".inst 0x4f00f3b0  // sudot v16.4s, v29.16b, v0.4b[0]\n"
      ".inst 0x4f1ff3b4  // sudot v20.4s, v29.16b, v31.4b[0]\n"
      ".inst 0x4f1ef3b8  // sudot v24.4s, v29.16b, v30.4b[0]\n"
      "ldr q29, [x10, #0x20]\n"
      ".inst 0x4f02f389  // sudot v9.4s, v28.16b, v2.4b[0]\n"
      ".inst 0x4f01f38d  // sudot v13.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x4f00f391  // sudot v17.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x4f1ff395  // sudot v21.4s, v28.16b, v31.4b[0]\n"
      ".inst 0x4f1ef399  // sudot v25.4s, v28.16b, v30.4b[0]\n"
      "ldr q28, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f02f3aa  // sudot v10.4s, v29.16b, v2.4b[0]\n"
      ".inst 0x4f01f3ae  // sudot v14.4s, v29.16b, v1.4b[0]\n"
      ".inst 0x4f00f3b2  // sudot v18.4s, v29.16b, v0.4b[0]\n"
      ".inst 0x4f1ff3b6  // sudot v22.4s, v29.16b, v31.4b[0]\n"
      ".inst 0x4f1ef3ba  // sudot v26.4s, v29.16b, v30.4b[0]\n"
      ".inst 0x4f02f38b  // sudot v11.4s, v28.16b, v2.4b[0]\n"
      ".inst 0x4f01f38f  // sudot v15.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x4f00f393  // sudot v19.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x4f1ff397  // sudot v23.4s, v28.16b, v31.4b[0]\n"
      ".inst 0x4f1ef39b  // sudot v27.4s, v28.16b, v30.4b[0]\n"
      "bge 156b\n"
      "157:"  // Height 5: Multiply loop: Skip odd blocks
      "cbz x27, 160f\n"
      "tbz x27, #1, 158f\n"
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "ldr h2, [x24], #0x2\n"
      "ldr h3, [x23], #0x2\n"
      "ldr h4, [x22], #0x2\n"
      "tbz x27, #0, 159f\n"
      "ld1 { v0.b }[2], [x26]\n"
      "ld1 { v1.b }[2], [x25]\n"
      "ld1 { v2.b }[2], [x24]\n"
      "ld1 { v3.b }[2], [x23]\n"
      "ld1 { v4.b }[2], [x22]\n"
      "b 159f\n"
      "158:"  // Height 5: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x26, #0x0]\n"
      "ldr b1, [x25, #0x0]\n"
      "ldr b2, [x24, #0x0]\n"
      "ldr b3, [x23, #0x0]\n"
      "ldr b4, [x22, #0x0]\n"
      "159:"  // Height 5: Multiply loop: Ragged operand read: Done
      "ldr q29, [x10, #0x0]\n"
      "ldr q28, [x10, #0x10]\n"
      ".inst 0x4f00f3a8  // sudot v8.4s, v29.16b, v0.4b[0]\n"
      ".inst 0x4f01f3ac  // sudot v12.4s, v29.16b, v1.4b[0]\n"
      ".inst 0x4f02f3b0  // sudot v16.4s, v29.16b, v2.4b[0]\n"
      ".inst 0x4f03f3b4  // sudot v20.4s, v29.16b, v3.4b[0]\n"
      ".inst 0x4f04f3b8  // sudot v24.4s, v29.16b, v4.4b[0]\n"
      "ldr q29, [x10, #0x20]\n"
      ".inst 0x4f00f389  // sudot v9.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x4f01f38d  // sudot v13.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x4f02f391  // sudot v17.4s, v28.16b, v2.4b[0]\n"
      ".inst 0x4f03f395  // sudot v21.4s, v28.16b, v3.4b[0]\n"
      ".inst 0x4f04f399  // sudot v25.4s, v28.16b, v4.4b[0]\n"
      "ldr q28, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f00f3aa  // sudot v10.4s, v29.16b, v0.4b[0]\n"
      ".inst 0x4f01f3ae  // sudot v14.4s, v29.16b, v1.4b[0]\n"
      ".inst 0x4f02f3b2  // sudot v18.4s, v29.16b, v2.4b[0]\n"
      ".inst 0x4f03f3b6  // sudot v22.4s, v29.16b, v3.4b[0]\n"
      ".inst 0x4f04f3ba  // sudot v26.4s, v29.16b, v4.4b[0]\n"
      ".inst 0x4f00f38b  // sudot v11.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x4f01f38f  // sudot v15.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x4f02f393  // sudot v19.4s, v28.16b, v2.4b[0]\n"
      ".inst 0x4f03f397  // sudot v23.4s, v28.16b, v3.4b[0]\n"
      ".inst 0x4f04f39b  // sudot v27.4s, v28.16b, v4.4b[0]\n"
      "160:"  // Height 5: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 150b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x11, #0x10\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "add x24, x9, x20, LSL #2\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "add x23, x24, x20, LSL #2\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "add x22, x23, x20, LSL #2\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "add x21, x22, x20, LSL #2\n"
      "prfm pstl1keep, [x21, #0x0]\n"
      "bge 169f\n"
      "tbz x11, #3, 164f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v9.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v13.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v17.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v21.4s }, [x22], #0x10\n"
      "st1 { v24.4s }, [x21], #0x10\n"
      "st1 { v25.4s }, [x21], #0x10\n"
      "tbz x11, #2, 162f\n"
      "st1 { v10.4s }, [x9], #0x10\n"
      "st1 { v14.4s }, [x24], #0x10\n"
      "st1 { v18.4s }, [x23], #0x10\n"
      "st1 { v22.4s }, [x22], #0x10\n"
      "st1 { v26.4s }, [x21], #0x10\n"
      "tbz x11, #1, 161f\n"
      "str d11, [x9], #0x8\n"
      "str d15, [x24], #0x8\n"
      "str d19, [x23], #0x8\n"
      "str d23, [x22], #0x8\n"
      "str d27, [x21], #0x8\n"
      "tbz x11, #0, 168f\n"
      "st1 { v11.s }[2], [x9]\n"
      "st1 { v15.s }[2], [x24]\n"
      "st1 { v19.s }[2], [x23]\n"
      "st1 { v23.s }[2], [x22]\n"
      "st1 { v27.s }[2], [x21]\n"
      "b 168f\n"
      "161:"  // Height 5: Partial direct writeback: partial_1_12
      "tbz x11, #0, 168f\n"
      "str s11, [x9, #0x0]\n"
      "str s15, [x24, #0x0]\n"
      "str s19, [x23, #0x0]\n"
      "str s23, [x22, #0x0]\n"
      "str s27, [x21, #0x0]\n"
      "b 168f\n"
      "162:"  // Height 5: Partial direct writeback: partial_2_8
      "tbz x11, #1, 163f\n"
      "str d10, [x9], #0x8\n"
      "str d14, [x24], #0x8\n"
      "str d18, [x23], #0x8\n"
      "str d22, [x22], #0x8\n"
      "str d26, [x21], #0x8\n"
      "tbz x11, #0, 168f\n"
      "st1 { v10.s }[2], [x9]\n"
      "st1 { v14.s }[2], [x24]\n"
      "st1 { v18.s }[2], [x23]\n"
      "st1 { v22.s }[2], [x22]\n"
      "st1 { v26.s }[2], [x21]\n"
      "b 168f\n"
      "163:"  // Height 5: Partial direct writeback: partial_1_8
      "tbz x11, #0, 168f\n"
      "str s10, [x9, #0x0]\n"
      "str s14, [x24, #0x0]\n"
      "str s18, [x23, #0x0]\n"
      "str s22, [x22, #0x0]\n"
      "str s26, [x21, #0x0]\n"
      "b 168f\n"
      "164:"  // Height 5: Partial direct writeback: partial_4_0
      "tbz x11, #2, 166f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v24.4s }, [x21], #0x10\n"
      "tbz x11, #1, 165f\n"
      "str d9, [x9], #0x8\n"
      "str d13, [x24], #0x8\n"
      "str d17, [x23], #0x8\n"
      "str d21, [x22], #0x8\n"
      "str d25, [x21], #0x8\n"
      "tbz x11, #0, 168f\n"
      "st1 { v9.s }[2], [x9]\n"
      "st1 { v13.s }[2], [x24]\n"
      "st1 { v17.s }[2], [x23]\n"
      "st1 { v21.s }[2], [x22]\n"
      "st1 { v25.s }[2], [x21]\n"
      "b 168f\n"
      "165:"  // Height 5: Partial direct writeback: partial_1_4
      "tbz x11, #0, 168f\n"
      "str s9, [x9, #0x0]\n"
      "str s13, [x24, #0x0]\n"
      "str s17, [x23, #0x0]\n"
      "str s21, [x22, #0x0]\n"
      "str s25, [x21, #0x0]\n"
      "b 168f\n"
      "166:"  // Height 5: Partial direct writeback: partial_2_0
      "tbz x11, #1, 167f\n"
      "str d8, [x9], #0x8\n"
      "str d12, [x24], #0x8\n"
      "str d16, [x23], #0x8\n"
      "str d20, [x22], #0x8\n"
      "str d24, [x21], #0x8\n"
      "tbz x11, #0, 168f\n"
      "st1 { v8.s }[2], [x9]\n"
      "st1 { v12.s }[2], [x24]\n"
      "st1 { v16.s }[2], [x23]\n"
      "st1 { v20.s }[2], [x22]\n"
      "st1 { v24.s }[2], [x21]\n"
      "b 168f\n"
      "167:"  // Height 5: Partial direct writeback: partial_1_0
      "str s8, [x9, #0x0]\n"
      "str s12, [x24, #0x0]\n"
      "str s16, [x23, #0x0]\n"
      "str s20, [x22, #0x0]\n"
      "str s24, [x21, #0x0]\n"
      "168:"  // Height 5: Partial direct writeback: Done
      "b 170f\n"
      "169:"  // Height 5: Full writeback
      "str q8, [x9, #0x0]\n"
      "str q9, [x9, #0x10]\n"
      "str q10, [x9, #0x20]\n"
      "str q11, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q12, [x24, #0x0]\n"
      "str q13, [x24, #0x10]\n"
      "str q14, [x24, #0x20]\n"
      "str q15, [x24, #0x30]\n"
      "str q16, [x23, #0x0]\n"
      "str q17, [x23, #0x10]\n"
      "str q18, [x23, #0x20]\n"
      "str q19, [x23, #0x30]\n"
      "str q20, [x22, #0x0]\n"
      "str q21, [x22, #0x10]\n"
      "str q22, [x22, #0x20]\n"
      "str q23, [x22, #0x30]\n"
      "str q24, [x21, #0x0]\n"
      "str q25, [x21, #0x10]\n"
      "str q26, [x21, #0x20]\n"
      "str q27, [x21, #0x30]\n"
      "170:"  // Height 5: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 138b\n"
      "b 206f\n"
      "171:"  // Height 6
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "mov x20, #0x18\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "madd x20, x21, x20, x9\n"
      "str x20, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "172:"  // Height 6: Column loop
      "tbz %x[flags], #0, 182f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x11, #0x10\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "bge 181f\n"
      "tbz x11, #3, 176f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v24.4s }, [x21], #0x10\n"
      "ld1 { v28.4s }, [x20], #0x10\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "ld1 { v25.4s }, [x21], #0x10\n"
      "ld1 { v29.4s }, [x20], #0x10\n"
      "tbz x11, #2, 174f\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "ld1 { v26.4s }, [x21], #0x10\n"
      "ld1 { v30.4s }, [x20], #0x10\n"
      "tbz x11, #1, 173f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d15, [x24], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d19, [x23], #0x8\n"
      "ldr d23, [x22], #0x8\n"
      "ldr d27, [x21], #0x8\n"
      "ldr d31, [x20], #0x8\n"
      "tbz x11, #0, 180f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "ld1 { v27.s }[2], [x21]\n"
      "ld1 { v31.s }[2], [x20]\n"
      "b 180f\n"
      "173:"  // Height 6: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x11, #0, 180f\n"
      "ldr s11, [x9, #0x0]\n"
      "ldr s15, [x24, #0x0]\n"
      "ldr s19, [x23, #0x0]\n"
      "ldr s23, [x22, #0x0]\n"
      "ldr s27, [x21, #0x0]\n"
      "ldr s31, [x20, #0x0]\n"
      "b 180f\n"
      "174:"  // Height 6: Partial accumulate: partial_2_8
      "tbz x11, #1, 175f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d14, [x24], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d18, [x23], #0x8\n"
      "ldr d22, [x22], #0x8\n"
      "ldr d26, [x21], #0x8\n"
      "ldr d30, [x20], #0x8\n"
      "tbz x11, #0, 180f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "ld1 { v26.s }[2], [x21]\n"
      "ld1 { v30.s }[2], [x20]\n"
      "b 180f\n"
      "175:"  // Height 6: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x11, #0, 180f\n"
      "ldr s10, [x9, #0x0]\n"
      "ldr s14, [x24, #0x0]\n"
      "ldr s18, [x23, #0x0]\n"
      "ldr s22, [x22, #0x0]\n"
      "ldr s26, [x21, #0x0]\n"
      "ldr s30, [x20, #0x0]\n"
      "b 180f\n"
      "176:"  // Height 6: Partial accumulate: partial_4_0
      "tbz x11, #2, 178f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v24.4s }, [x21], #0x10\n"
      "ld1 { v28.4s }, [x20], #0x10\n"
      "tbz x11, #1, 177f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d13, [x24], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d17, [x23], #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "ldr d25, [x21], #0x8\n"
      "ldr d29, [x20], #0x8\n"
      "tbz x11, #0, 180f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "ld1 { v25.s }[2], [x21]\n"
      "ld1 { v29.s }[2], [x20]\n"
      "b 180f\n"
      "177:"  // Height 6: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x11, #0, 180f\n"
      "ldr s9, [x9, #0x0]\n"
      "ldr s13, [x24, #0x0]\n"
      "ldr s17, [x23, #0x0]\n"
      "ldr s21, [x22, #0x0]\n"
      "ldr s25, [x21, #0x0]\n"
      "ldr s29, [x20, #0x0]\n"
      "b 180f\n"
      "178:"  // Height 6: Partial accumulate: partial_2_0
      "tbz x11, #1, 179f\n"
      "ldr d8, [x9], #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d16, [x23], #0x8\n"
      "ldr d20, [x22], #0x8\n"
      "ldr d24, [x21], #0x8\n"
      "ldr d28, [x20], #0x8\n"
      "tbz x11, #0, 180f\n"
      "ld1 { v8.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v16.s }[2], [x23]\n"
      "ld1 { v20.s }[2], [x22]\n"
      "ld1 { v24.s }[2], [x21]\n"
      "ld1 { v28.s }[2], [x20]\n"
      "b 180f\n"
      "179:"  // Height 6: Partial accumulate: partial_1_0
      "ldr s8, [x9, #0x0]\n"
      "ldr s12, [x24, #0x0]\n"
      "mov x25, #0x0\n"
      "ldr s16, [x23, #0x0]\n"
      "ldr s20, [x22, #0x0]\n"
      "ldr s24, [x21, #0x0]\n"
      "ldr s28, [x20, #0x0]\n"
      "180:"  // Height 6: Partial accumulate: Done
      "sub x9, x9, x25\n"
      "b 183f\n"
      "181:"  // Height 6: full accumulate
      "ldr q8, [x9, #0x0]\n"
      "ldr q9, [x9, #0x10]\n"
      "ldr q10, [x9, #0x20]\n"
      "ldr q11, [x9, #0x30]\n"
      "ldr q12, [x24, #0x0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q16, [x23, #0x0]\n"
      "ldr q17, [x23, #0x10]\n"
      "ldr q18, [x23, #0x20]\n"
      "ldr q19, [x23, #0x30]\n"
      "ldr q20, [x22, #0x0]\n"
      "ldr q21, [x22, #0x10]\n"
      "ldr q22, [x22, #0x20]\n"
      "ldr q23, [x22, #0x30]\n"
      "ldr q24, [x21, #0x0]\n"
      "ldr q25, [x21, #0x10]\n"
      "ldr q26, [x21, #0x20]\n"
      "ldr q27, [x21, #0x30]\n"
      "ldr q28, [x20, #0x0]\n"
      "ldr q29, [x20, #0x10]\n"
      "ldr q30, [x20, #0x20]\n"
      "ldr q31, [x20, #0x30]\n"
      "b 183f\n"
      "182:"  // Height 6: no accumulate
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "movi v28.4s, #0x0\n"
      "movi v29.4s, #0x0\n"
      "movi v30.4s, #0x0\n"
      "movi v31.4s, #0x0\n"
      "183:"  // Height 6: setup done
      "mov x28, #0x0\n"
      "184:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 185f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 186f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "add x21, x21, x20\n"
      "b 186f\n"
      "185:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "add x21, x22, x21\n"
      "186:"  // Height 6: input setup done
      "cmp x27, #0x10\n"
      "blt 189f\n"
      "ldr q0, [x26, #0x0]\n"
      "ldr q1, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q2, [x24, #0x0]\n"
      "ldr q3, [x23, #0x0]\n"
      "ldr q4, [x22, #0x0]\n"
      "ldr q5, [x21, #0x0]\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "blt 188f\n"
      "187:"  // Height 6: Multiply loop: Main loop head
      ".inst 0x4f00f0c8  // sudot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f01f0cc  // sudot v12.4s, v6.16b, v1.4b[0]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f02f0d0  // sudot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f03f0d4  // sudot v20.4s, v6.16b, v3.4b[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f04f0d8  // sudot v24.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f05f0dc  // sudot v28.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f00f0e9  // sudot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f01f0ed  // sudot v13.4s, v7.16b, v1.4b[0]\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      ".inst 0x4f02f0f1  // sudot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f03f0f5  // sudot v21.4s, v7.16b, v3.4b[0]\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f04f0f9  // sudot v25.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f05f0fd  // sudot v29.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f00f0ca  // sudot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f01f0ce  // sudot v14.4s, v6.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4f02f0d2  // sudot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f03f0d6  // sudot v22.4s, v6.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      ".inst 0x4f04f0da  // sudot v26.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f05f0de  // sudot v30.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4f00f0eb  // sudot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f01f0ef  // sudot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f02f0f3  // sudot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f03f0f7  // sudot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f04f0fb  // sudot v27.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f05f0ff  // sudot v31.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4f20f0c8  // sudot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4f21f0cc  // sudot v12.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4f22f0d0  // sudot v16.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4f23f0d4  // sudot v20.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4f24f0d8  // sudot v24.4s, v6.16b, v4.4b[1]\n"
      ".inst 0x4f25f0dc  // sudot v28.4s, v6.16b, v5.4b[1]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4f20f0e9  // sudot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4f21f0ed  // sudot v13.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4f22f0f1  // sudot v17.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4f23f0f5  // sudot v21.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4f24f0f9  // sudot v25.4s, v7.16b, v4.4b[1]\n"
      ".inst 0x4f25f0fd  // sudot v29.4s, v7.16b, v5.4b[1]\n"
      "ldr q7, [x10, #0x70]\n"
      ".inst 0x4f20f0ca  // sudot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4f21f0ce  // sudot v14.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4f22f0d2  // sudot v18.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4f23f0d6  // sudot v22.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4f24f0da  // sudot v26.4s, v6.16b, v4.4b[1]\n"
      ".inst 0x4f25f0de  // sudot v30.4s, v6.16b, v5.4b[1]\n"
      "ldr q6, [x10, #0x80]\n"
      ".inst 0x4f20f0eb  // sudot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4f21f0ef  // sudot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4f22f0f3  // sudot v19.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4f23f0f7  // sudot v23.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4f24f0fb  // sudot v27.4s, v7.16b, v4.4b[1]\n"
      ".inst 0x4f25f0ff  // sudot v31.4s, v7.16b, v5.4b[1]\n"
      "ldr q7, [x10, #0x90]\n"
      ".inst 0x4f00f8c8  // sudot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f01f8cc  // sudot v12.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f02f8d0  // sudot v16.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f03f8d4  // sudot v20.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f04f8d8  // sudot v24.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x4f05f8dc  // sudot v28.4s, v6.16b, v5.4b[2]\n"
      "ldr q6, [x10, #0xa0]\n"
      ".inst 0x4f00f8e9  // sudot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f01f8ed  // sudot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f02f8f1  // sudot v17.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f03f8f5  // sudot v21.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f04f8f9  // sudot v25.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x4f05f8fd  // sudot v29.4s, v7.16b, v5.4b[2]\n"
      "ldr q7, [x10, #0xb0]\n"
      ".inst 0x4f00f8ca  // sudot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f01f8ce  // sudot v14.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f02f8d2  // sudot v18.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f03f8d6  // sudot v22.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f04f8da  // sudot v26.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x4f05f8de  // sudot v30.4s, v6.16b, v5.4b[2]\n"
      "ldr q6, [x10, #0xc0]\n"
      ".inst 0x4f00f8eb  // sudot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f01f8ef  // sudot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f02f8f3  // sudot v19.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f03f8f7  // sudot v23.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f04f8fb  // sudot v27.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x4f05f8ff  // sudot v31.4s, v7.16b, v5.4b[2]\n"
      "ldr q7, [x10, #0xd0]\n"
      ".inst 0x4f20f8c8  // sudot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4f21f8cc  // sudot v12.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4f22f8d0  // sudot v16.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4f23f8d4  // sudot v20.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4f24f8d8  // sudot v24.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x4f25f8dc  // sudot v28.4s, v6.16b, v5.4b[3]\n"
      "ldr q6, [x10, #0xe0]\n"
      ".inst 0x4f20f8e9  // sudot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4f21f8ed  // sudot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4f22f8f1  // sudot v17.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4f23f8f5  // sudot v21.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x4f24f8f9  // sudot v25.4s, v7.16b, v4.4b[3]\n"
      ".inst 0x4f25f8fd  // sudot v29.4s, v7.16b, v5.4b[3]\n"
      "ldr q7, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4f20f8ca  // sudot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4f21f8ce  // sudot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4f22f8d2  // sudot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4f23f8d6  // sudot v22.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4f24f8da  // sudot v26.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x4f25f8de  // sudot v30.4s, v6.16b, v5.4b[3]\n"
      "ldr q6, [x10, #0x0]\n"
      ".inst 0x4f20f8eb  // sudot v11.4s, v7.16b, v0.4b[3]\n"
      "ldr q0, [x26, #0x0]\n"
      ".inst 0x4f21f8ef  // sudot v15.4s, v7.16b, v1.4b[3]\n"
      "ldr q1, [x25, #0x0]\n"
      ".inst 0x4f22f8f3  // sudot v19.4s, v7.16b, v2.4b[3]\n"
      "ldr q2, [x24, #0x0]\n"
      ".inst 0x4f23f8f7  // sudot v23.4s, v7.16b, v3.4b[3]\n"
      "ldr q3, [x23, #0x0]\n"
      ".inst 0x4f24f8fb  // sudot v27.4s, v7.16b, v4.4b[3]\n"
      "ldr q4, [x22, #0x0]\n"
      ".inst 0x4f25f8ff  // sudot v31.4s, v7.16b, v5.4b[3]\n"
      "ldr q5, [x21, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "bge 187b\n"
      "188:"  // Height 6: Multiply loop: Single iteration only
      ".inst 0x4f00f0c8  // sudot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f01f0cc  // sudot v12.4s, v6.16b, v1.4b[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f02f0d0  // sudot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f03f0d4  // sudot v20.4s, v6.16b, v3.4b[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f04f0d8  // sudot v24.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f05f0dc  // sudot v28.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x10, #0x20]\n"
      "add x22, x22, #0x10\n"
      ".inst 0x4f00f0e9  // sudot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f01f0ed  // sudot v13.4s, v7.16b, v1.4b[0]\n"
      "add x21, x21, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f02f0f1  // sudot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f03f0f5  // sudot v21.4s, v7.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f04f0f9  // sudot v25.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f05f0fd  // sudot v29.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x10, #0x30]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4f00f0ca  // sudot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f01f0ce  // sudot v14.4s, v6.16b, v1.4b[0]\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4f02f0d2  // sudot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f03f0d6  // sudot v22.4s, v6.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      ".inst 0x4f04f0da  // sudot v26.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f05f0de  // sudot v30.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4f00f0eb  // sudot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f01f0ef  // sudot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f02f0f3  // sudot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f03f0f7  // sudot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f04f0fb  // sudot v27.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f05f0ff  // sudot v31.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4f20f0c8  // sudot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4f21f0cc  // sudot v12.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4f22f0d0  // sudot v16.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4f23f0d4  // sudot v20.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4f24f0d8  // sudot v24.4s, v6.16b, v4.4b[1]\n"
      ".inst 0x4f25f0dc  // sudot v28.4s, v6.16b, v5.4b[1]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4f20f0e9  // sudot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4f21f0ed  // sudot v13.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4f22f0f1  // sudot v17.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4f23f0f5  // sudot v21.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4f24f0f9  // sudot v25.4s, v7.16b, v4.4b[1]\n"
      ".inst 0x4f25f0fd  // sudot v29.4s, v7.16b, v5.4b[1]\n"
      "ldr q7, [x10, #0x70]\n"
      ".inst 0x4f20f0ca  // sudot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4f21f0ce  // sudot v14.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4f22f0d2  // sudot v18.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4f23f0d6  // sudot v22.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4f24f0da  // sudot v26.4s, v6.16b, v4.4b[1]\n"
      ".inst 0x4f25f0de  // sudot v30.4s, v6.16b, v5.4b[1]\n"
      "ldr q6, [x10, #0x80]\n"
      ".inst 0x4f20f0eb  // sudot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4f21f0ef  // sudot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4f22f0f3  // sudot v19.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4f23f0f7  // sudot v23.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4f24f0fb  // sudot v27.4s, v7.16b, v4.4b[1]\n"
      ".inst 0x4f25f0ff  // sudot v31.4s, v7.16b, v5.4b[1]\n"
      "ldr q7, [x10, #0x90]\n"
      ".inst 0x4f00f8c8  // sudot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f01f8cc  // sudot v12.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f02f8d0  // sudot v16.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f03f8d4  // sudot v20.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f04f8d8  // sudot v24.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x4f05f8dc  // sudot v28.4s, v6.16b, v5.4b[2]\n"
      "ldr q6, [x10, #0xa0]\n"
      ".inst 0x4f00f8e9  // sudot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f01f8ed  // sudot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f02f8f1  // sudot v17.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f03f8f5  // sudot v21.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f04f8f9  // sudot v25.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x4f05f8fd  // sudot v29.4s, v7.16b, v5.4b[2]\n"
      "ldr q7, [x10, #0xb0]\n"
      ".inst 0x4f00f8ca  // sudot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f01f8ce  // sudot v14.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f02f8d2  // sudot v18.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f03f8d6  // sudot v22.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f04f8da  // sudot v26.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x4f05f8de  // sudot v30.4s, v6.16b, v5.4b[2]\n"
      "ldr q6, [x10, #0xc0]\n"
      ".inst 0x4f00f8eb  // sudot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f01f8ef  // sudot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f02f8f3  // sudot v19.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f03f8f7  // sudot v23.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f04f8fb  // sudot v27.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x4f05f8ff  // sudot v31.4s, v7.16b, v5.4b[2]\n"
      "ldr q7, [x10, #0xd0]\n"
      ".inst 0x4f20f8c8  // sudot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4f21f8cc  // sudot v12.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4f22f8d0  // sudot v16.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4f23f8d4  // sudot v20.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4f24f8d8  // sudot v24.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x4f25f8dc  // sudot v28.4s, v6.16b, v5.4b[3]\n"
      "ldr q6, [x10, #0xe0]\n"
      ".inst 0x4f20f8e9  // sudot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4f21f8ed  // sudot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4f22f8f1  // sudot v17.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4f23f8f5  // sudot v21.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x4f24f8f9  // sudot v25.4s, v7.16b, v4.4b[3]\n"
      ".inst 0x4f25f8fd  // sudot v29.4s, v7.16b, v5.4b[3]\n"
      "ldr q7, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4f20f8ca  // sudot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4f21f8ce  // sudot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4f22f8d2  // sudot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4f23f8d6  // sudot v22.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4f24f8da  // sudot v26.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x4f25f8de  // sudot v30.4s, v6.16b, v5.4b[3]\n"
      ".inst 0x4f20f8eb  // sudot v11.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4f21f8ef  // sudot v15.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4f22f8f3  // sudot v19.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4f23f8f7  // sudot v23.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x4f24f8fb  // sudot v27.4s, v7.16b, v4.4b[3]\n"
      ".inst 0x4f25f8ff  // sudot v31.4s, v7.16b, v5.4b[3]\n"
      "189:"  // Height 6: Multiply loop: Main loop skip
      "cbz x27, 194f\n"
      "cmp x27, #0x4\n"
      "blt 191f\n"
      "190:"  // Height 6: Multiply loop: Odd block loop
      "ldr s7, [x26], #0x4\n"
      "ldr s6, [x25], #0x4\n"
      "sub x27, x27, #0x4\n"
      "ldr s5, [x24], #0x4\n"
      "ldr s4, [x23], #0x4\n"
      "cmp x27, #0x4\n"
      "ldr s3, [x22], #0x4\n"
      "ldr s2, [x21], #0x4\n"
      "ldr q1, [x10, #0x0]\n"
      "ldr q0, [x10, #0x10]\n"
      ".inst 0x4f07f028  // sudot v8.4s, v1.16b, v7.4b[0]\n"
      ".inst 0x4f06f02c  // sudot v12.4s, v1.16b, v6.4b[0]\n"
      ".inst 0x4f05f030  // sudot v16.4s, v1.16b, v5.4b[0]\n"
      ".inst 0x4f04f034  // sudot v20.4s, v1.16b, v4.4b[0]\n"
      ".inst 0x4f03f038  // sudot v24.4s, v1.16b, v3.4b[0]\n"
      ".inst 0x4f02f03c  // sudot v28.4s, v1.16b, v2.4b[0]\n"
      "ldr q1, [x10, #0x20]\n"
      ".inst 0x4f07f009  // sudot v9.4s, v0.16b, v7.4b[0]\n"
      ".inst 0x4f06f00d  // sudot v13.4s, v0.16b, v6.4b[0]\n"
      ".inst 0x4f05f011  // sudot v17.4s, v0.16b, v5.4b[0]\n"
      ".inst 0x4f04f015  // sudot v21.4s, v0.16b, v4.4b[0]\n"
      ".inst 0x4f03f019  // sudot v25.4s, v0.16b, v3.4b[0]\n"
      ".inst 0x4f02f01d  // sudot v29.4s, v0.16b, v2.4b[0]\n"
      "ldr q0, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f07f02a  // sudot v10.4s, v1.16b, v7.4b[0]\n"
      ".inst 0x4f06f02e  // sudot v14.4s, v1.16b, v6.4b[0]\n"
      ".inst 0x4f05f032  // sudot v18.4s, v1.16b, v5.4b[0]\n"
      ".inst 0x4f04f036  // sudot v22.4s, v1.16b, v4.4b[0]\n"
      ".inst 0x4f03f03a  // sudot v26.4s, v1.16b, v3.4b[0]\n"
      ".inst 0x4f02f03e  // sudot v30.4s, v1.16b, v2.4b[0]\n"
      ".inst 0x4f07f00b  // sudot v11.4s, v0.16b, v7.4b[0]\n"
      ".inst 0x4f06f00f  // sudot v15.4s, v0.16b, v6.4b[0]\n"
      ".inst 0x4f05f013  // sudot v19.4s, v0.16b, v5.4b[0]\n"
      ".inst 0x4f04f017  // sudot v23.4s, v0.16b, v4.4b[0]\n"
      ".inst 0x4f03f01b  // sudot v27.4s, v0.16b, v3.4b[0]\n"
      ".inst 0x4f02f01f  // sudot v31.4s, v0.16b, v2.4b[0]\n"
      "bge 190b\n"
      "191:"  // Height 6: Multiply loop: Skip odd blocks
      "cbz x27, 194f\n"
      "tbz x27, #1, 192f\n"
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "ldr h2, [x24], #0x2\n"
      "ldr h3, [x23], #0x2\n"
      "ldr h4, [x22], #0x2\n"
      "ldr h5, [x21], #0x2\n"
      "tbz x27, #0, 193f\n"
      "ld1 { v0.b }[2], [x26]\n"
      "ld1 { v1.b }[2], [x25]\n"
      "ld1 { v2.b }[2], [x24]\n"
      "ld1 { v3.b }[2], [x23]\n"
      "ld1 { v4.b }[2], [x22]\n"
      "ld1 { v5.b }[2], [x21]\n"
      "b 193f\n"
      "192:"  // Height 6: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x26, #0x0]\n"
      "ldr b1, [x25, #0x0]\n"
      "ldr b2, [x24, #0x0]\n"
      "ldr b3, [x23, #0x0]\n"
      "ldr b4, [x22, #0x0]\n"
      "ldr b5, [x21, #0x0]\n"
      "193:"  // Height 6: Multiply loop: Ragged operand read: Done
      "ldr q7, [x10, #0x0]\n"
      "ldr q6, [x10, #0x10]\n"
      ".inst 0x4f00f0e8  // sudot v8.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f01f0ec  // sudot v12.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f02f0f0  // sudot v16.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f03f0f4  // sudot v20.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f04f0f8  // sudot v24.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f05f0fc  // sudot v28.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4f00f0c9  // sudot v9.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f01f0cd  // sudot v13.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f02f0d1  // sudot v17.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f03f0d5  // sudot v21.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f04f0d9  // sudot v25.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f05f0dd  // sudot v29.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      ".inst 0x4f00f0ea  // sudot v10.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f01f0ee  // sudot v14.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f02f0f2  // sudot v18.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f03f0f6  // sudot v22.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f04f0fa  // sudot v26.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f05f0fe  // sudot v30.4s, v7.16b, v5.4b[0]\n"
      ".inst 0x4f00f0cb  // sudot v11.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f01f0cf  // sudot v15.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f02f0d3  // sudot v19.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f03f0d7  // sudot v23.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f04f0db  // sudot v27.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f05f0df  // sudot v31.4s, v6.16b, v5.4b[0]\n"
      "194:"  // Height 6: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 184b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x11, #0x10\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "add x24, x9, x20, LSL #2\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "add x23, x24, x20, LSL #2\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "add x22, x23, x20, LSL #2\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "add x21, x22, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "prfm pstl1keep, [x21, #0x0]\n"
      "prfm pstl1keep, [x20, #0x0]\n"
      "bge 203f\n"
      "tbz x11, #3, 198f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v9.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v13.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v17.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v21.4s }, [x22], #0x10\n"
      "st1 { v24.4s }, [x21], #0x10\n"
      "st1 { v25.4s }, [x21], #0x10\n"
      "st1 { v28.4s }, [x20], #0x10\n"
      "st1 { v29.4s }, [x20], #0x10\n"
      "tbz x11, #2, 196f\n"
      "st1 { v10.4s }, [x9], #0x10\n"
      "st1 { v14.4s }, [x24], #0x10\n"
      "st1 { v18.4s }, [x23], #0x10\n"
      "st1 { v22.4s }, [x22], #0x10\n"
      "st1 { v26.4s }, [x21], #0x10\n"
      "st1 { v30.4s }, [x20], #0x10\n"
      "tbz x11, #1, 195f\n"
      "str d11, [x9], #0x8\n"
      "str d15, [x24], #0x8\n"
      "str d19, [x23], #0x8\n"
      "str d23, [x22], #0x8\n"
      "str d27, [x21], #0x8\n"
      "str d31, [x20], #0x8\n"
      "tbz x11, #0, 202f\n"
      "st1 { v11.s }[2], [x9]\n"
      "st1 { v15.s }[2], [x24]\n"
      "st1 { v19.s }[2], [x23]\n"
      "st1 { v23.s }[2], [x22]\n"
      "st1 { v27.s }[2], [x21]\n"
      "st1 { v31.s }[2], [x20]\n"
      "b 202f\n"
      "195:"  // Height 6: Partial direct writeback: partial_1_12
      "tbz x11, #0, 202f\n"
      "str s11, [x9, #0x0]\n"
      "str s15, [x24, #0x0]\n"
      "str s19, [x23, #0x0]\n"
      "str s23, [x22, #0x0]\n"
      "str s27, [x21, #0x0]\n"
      "str s31, [x20, #0x0]\n"
      "b 202f\n"
      "196:"  // Height 6: Partial direct writeback: partial_2_8
      "tbz x11, #1, 197f\n"
      "str d10, [x9], #0x8\n"
      "str d14, [x24], #0x8\n"
      "str d18, [x23], #0x8\n"
      "str d22, [x22], #0x8\n"
      "str d26, [x21], #0x8\n"
      "str d30, [x20], #0x8\n"
      "tbz x11, #0, 202f\n"
      "st1 { v10.s }[2], [x9]\n"
      "st1 { v14.s }[2], [x24]\n"
      "st1 { v18.s }[2], [x23]\n"
      "st1 { v22.s }[2], [x22]\n"
      "st1 { v26.s }[2], [x21]\n"
      "st1 { v30.s }[2], [x20]\n"
      "b 202f\n"
      "197:"  // Height 6: Partial direct writeback: partial_1_8
      "tbz x11, #0, 202f\n"
      "str s10, [x9, #0x0]\n"
      "str s14, [x24, #0x0]\n"
      "str s18, [x23, #0x0]\n"
      "str s22, [x22, #0x0]\n"
      "str s26, [x21, #0x0]\n"
      "str s30, [x20, #0x0]\n"
      "b 202f\n"
      "198:"  // Height 6: Partial direct writeback: partial_4_0
      "tbz x11, #2, 200f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v24.4s }, [x21], #0x10\n"
      "st1 { v28.4s }, [x20], #0x10\n"
      "tbz x11, #1, 199f\n"
      "str d9, [x9], #0x8\n"
      "str d13, [x24], #0x8\n"
      "str d17, [x23], #0x8\n"
      "str d21, [x22], #0x8\n"
      "str d25, [x21], #0x8\n"
      "str d29, [x20], #0x8\n"
      "tbz x11, #0, 202f\n"
      "st1 { v9.s }[2], [x9]\n"
      "st1 { v13.s }[2], [x24]\n"
      "st1 { v17.s }[2], [x23]\n"
      "st1 { v21.s }[2], [x22]\n"
      "st1 { v25.s }[2], [x21]\n"
      "st1 { v29.s }[2], [x20]\n"
      "b 202f\n"
      "199:"  // Height 6: Partial direct writeback: partial_1_4
      "tbz x11, #0, 202f\n"
      "str s9, [x9, #0x0]\n"
      "str s13, [x24, #0x0]\n"
      "str s17, [x23, #0x0]\n"
      "str s21, [x22, #0x0]\n"
      "str s25, [x21, #0x0]\n"
      "str s29, [x20, #0x0]\n"
      "b 202f\n"
      "200:"  // Height 6: Partial direct writeback: partial_2_0
      "tbz x11, #1, 201f\n"
      "str d8, [x9], #0x8\n"
      "str d12, [x24], #0x8\n"
      "str d16, [x23], #0x8\n"
      "str d20, [x22], #0x8\n"
      "str d24, [x21], #0x8\n"
      "str d28, [x20], #0x8\n"
      "tbz x11, #0, 202f\n"
      "st1 { v8.s }[2], [x9]\n"
      "st1 { v12.s }[2], [x24]\n"
      "st1 { v16.s }[2], [x23]\n"
      "st1 { v20.s }[2], [x22]\n"
      "st1 { v24.s }[2], [x21]\n"
      "st1 { v28.s }[2], [x20]\n"
      "b 202f\n"
      "201:"  // Height 6: Partial direct writeback: partial_1_0
      "str s8, [x9, #0x0]\n"
      "str s12, [x24, #0x0]\n"
      "str s16, [x23, #0x0]\n"
      "str s20, [x22, #0x0]\n"
      "str s24, [x21, #0x0]\n"
      "str s28, [x20, #0x0]\n"
      "202:"  // Height 6: Partial direct writeback: Done
      "b 204f\n"
      "203:"  // Height 6: Full writeback
      "str q8, [x9, #0x0]\n"
      "str q9, [x9, #0x10]\n"
      "str q10, [x9, #0x20]\n"
      "str q11, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q12, [x24, #0x0]\n"
      "str q13, [x24, #0x10]\n"
      "str q14, [x24, #0x20]\n"
      "str q15, [x24, #0x30]\n"
      "str q16, [x23, #0x0]\n"
      "str q17, [x23, #0x10]\n"
      "str q18, [x23, #0x20]\n"
      "str q19, [x23, #0x30]\n"
      "str q20, [x22, #0x0]\n"
      "str q21, [x22, #0x10]\n"
      "str q22, [x22, #0x20]\n"
      "str q23, [x22, #0x30]\n"
      "str q24, [x21, #0x0]\n"
      "str q25, [x21, #0x10]\n"
      "str q26, [x21, #0x20]\n"
      "str q27, [x21, #0x30]\n"
      "str q28, [x20, #0x0]\n"
      "str q29, [x20, #0x10]\n"
      "str q30, [x20, #0x20]\n"
      "str q31, [x20, #0x30]\n"
      "204:"  // Height 6: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 172b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 206f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 205f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "205:"  // Update direct input
      "mov x20, #0x6\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "206:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [flags] "r" (flags), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x11", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm
#endif // __aarch64__
