/*
 * Copyright 2021 Google LLC
 *
 * Use of this source code is governed by a BSD-style license that can be
 * found in the LICENSE file.
 */

#include "bench/Benchmark.h"
#include "include/core/SkPath.h"
#include "src/core/SkArenaAlloc.h"
#include "src/gpu/ganesh/GrEagerVertexAllocator.h"
#include "src/gpu/ganesh/geometry/GrInnerFanTriangulator.h"
#include "src/gpu/ganesh/geometry/GrTriangulator.h"
#include <vector>

struct TigerPath {
    const char* fVerbs;
    std::vector<SkPoint> fPoints;
};

// Paths in desk_tigersvg.skp that take the "inner triangulation" path through the tessellator.
extern TigerPath kTigerPaths[];
extern int kNumTigerPaths;
constexpr float kTigerTolerance = 0.728769f;

class TriangulatorBenchmark : public Benchmark, public GrEagerVertexAllocator {
public:
    TriangulatorBenchmark(const char* name) {
        fName.printf("triangulator_%s", name);
    }

    const char* onGetName() override { return fName.c_str(); }
    bool isSuitableFor(Backend backend) final { return backend == kNonRendering_Backend; }

protected:
    void onDelayedSetup() override {
        for (int i = 0; i < kNumTigerPaths; ++i) {
            SkPath& path = fPaths.push_back();
            const std::vector<SkPoint>& pts = kTigerPaths[i].fPoints;
            int ptsIdx = 0;
            for (const char* v = kTigerPaths[i].fVerbs; *v; ++v) {
                switch (*v) {
                    case 'm':
                        path.moveTo(pts[ptsIdx]);
                        ++ptsIdx;
                        break;
                    case 'l':
                        path.lineTo(pts[ptsIdx]);
                        ++ptsIdx;
                        break;
                    case 'q':
                        path.quadTo(pts[ptsIdx], pts[ptsIdx + 1]);
                        ptsIdx += 2;
                        break;
                    case 'c':
                        path.cubicTo(pts[ptsIdx], pts[ptsIdx + 1], pts[ptsIdx + 2]);
                        ptsIdx += 3;
                        break;
                    case 'z':
                        path.close();
                        break;
                }
            }
            SkASSERT(ptsIdx == (int)pts.size());
        }
    }

    void onDraw(int loops, SkCanvas*) final {
        for (int i = 0; i < loops; ++i) {
            this->doLoop();
        }
    }

    // GrEagerVertexAllocator.
    void* lock(size_t stride, int eagerCount) override {
        size_t allocSize = eagerCount * stride;
        if (allocSize > fVertexAllocSize) {
            fVertexData.reset(allocSize);
        }
        return fVertexData;
    }

    void unlock(int) override {}

    virtual void doLoop() = 0;

    SkString fName;
    SkTArray<SkPath> fPaths;
    SkAutoTMalloc<char> fVertexData;
    size_t fVertexAllocSize = 0;
    SkArenaAllocWithReset fArena{GrTriangulator::kArenaDefaultChunkSize};
};

class PathToTrianglesBench : public TriangulatorBenchmark {
public:
    PathToTrianglesBench() : TriangulatorBenchmark("PathToTriangles") {}

    void doLoop() override {
        for (const SkPath& path : fPaths) {
            bool isLinear;
            GrTriangulator::PathToTriangles(path, kTigerTolerance, SkRect::MakeEmpty(), this,
                                            &isLinear);
        }
    }
};

DEF_BENCH( return new PathToTrianglesBench(); );

class TriangulateInnerFanBench : public TriangulatorBenchmark {
public:
    TriangulateInnerFanBench() : TriangulatorBenchmark("TriangulateInnerFan") {}

    void doLoop() override {
        bool isLinear;
        for (const SkPath& path : fPaths) {
            GrInnerFanTriangulator::BreadcrumbTriangleList breadcrumbList;
            GrInnerFanTriangulator(path, &fArena).pathToTriangles(this, &breadcrumbList, &isLinear);
        }
        fArena.reset();
    }
};

DEF_BENCH( return new TriangulateInnerFanBench(); );

#if 0
#include "src/gpu/tessellate/GrMiddleOutPolygonTriangulator.h"

class MiddleOutComparisonBench : public TriangulatorBenchmark {
public:
    MiddleOutComparisonBench() : TriangulatorBenchmark("MiddleOutComparison") {}
    void doLoop() override {
        for (const SkPath& path : fPaths) {
            int maxInnerTriangles = path.countVerbs() - 1;
            auto* data = this->GrEagerVertexAllocator::lock<SkPoint>(maxInnerTriangles * 3);
            int vertexCount = GrMiddleOutPolygonTriangulator::WritePathInnerFan(
                    data, 3/*perTriangleVertexAdvance*/, path) * 3;
            this->unlock(vertexCount);
        }
    }
};

DEF_BENCH( return new MiddleOutComparisonBench(); );
#endif

TigerPath kTigerPaths[] = {
    {"mcccccccclclclccccclcclccccccclclcccccccccccccccccccccccccccccllcclcccclccccccccccccccclz",
     {{-129.83f,103.06f}, {-129.33f,109.11f}, {-128.34f,115.68f}, {-126.6f,118.8f},
      {-126.6f,118.8f}, {-130.2f,131.2f}, {-121.4f,144.4f}, {-121.4f,144.4f}, {-121.8f,151.6f},
      {-120.2f,154.8f}, {-120.2f,154.8f}, {-116.2f,163.2f}, {-111.4f,164.f}, {-107.52f,164.65f},
      {-98.793f,167.72f}, {-88.932f,169.12f}, {-88.932f,169.12f}, {-71.8f,183.2f}, {-75.f,196.f},
      {-75.f,196.f}, {-75.4f,212.4f}, {-79.f,214.f}, {-79.f,214.f}, {-67.4f,202.8f}, {-77.f,219.6f},
      {-81.4f,238.4f}, {-81.4f,238.4f}, {-55.8f,216.8f}, {-71.4f,235.2f}, {-81.4f,261.2f},
      {-81.4f,261.2f}, {-61.8f,242.8f}, {-69.f,251.2f}, {-72.2f,260.f}, {-72.2f,260.f},
      {-29.f,232.8f}, {-59.8f,262.4f}, {-59.8f,262.4f}, {-51.8f,258.8f}, {-47.4f,261.6f},
      {-47.4f,261.6f}, {-40.6f,260.4f}, {-41.4f,262.f}, {-41.4f,262.f}, {-62.2f,272.4f},
      {-65.8f,290.8f}, {-65.8f,290.8f}, {-57.4f,280.8f}, {-60.6f,291.6f}, {-60.2f,303.2f},
      {-60.2f,303.2f}, {-56.2f,281.6f}, {-56.6f,319.2f}, {-56.6f,319.2f}, {-37.4f,301.2f},
      {-49.f,322.f}, {-49.f,338.8f}, {-49.f,338.8f}, {-33.8f,322.4f}, {-40.2f,335.2f},
      {-40.2f,335.2f}, {-30.2f,326.4f}, {-34.2f,341.6f}, {-34.2f,341.6f}, {-35.f,352.f},
      {-30.6f,340.8f}, {-30.6f,340.8f}, {-14.6f,310.2f}, {-20.6f,336.4f}, {-20.6f,336.4f},
      {-21.4f,355.6f}, {-16.6f,340.8f}, {-16.6f,340.8f}, {-16.2f,351.2f}, {-7.f,358.4f},
      {-7.f,358.4f}, {-8.2f,307.6f}, {4.6f,343.6f}, {8.6f,360.f}, {8.6f,360.f}, {11.4f,350.8f},
      {11.f,345.6f}, {19.f,353.6f}, {19.f,353.6f}, {34.2f,330.8f}, {31.f,344.f}, {31.f,344.f},
      {23.4f,360.f}, {25.f,364.8f}, {25.f,364.8f}, {41.8f,330.f}, {43.f,328.4f}, {43.f,328.4f},
      {41.f,370.8f}, {51.8f,334.8f}, {51.8f,334.8f}, {57.4f,346.8f}, {54.6f,351.2f}, {54.6f,351.2f},
      {62.6f,343.2f}, {61.8f,340.f}, {61.8f,340.f}, {66.4f,331.8f}, {69.2f,345.4f}, {69.2f,345.4f},
      {71.f,354.8f}, {72.6f,351.6f}, {72.6f,351.6f}, {76.6f,375.6f}, {77.8f,352.8f}, {77.8f,352.8f},
      {79.4f,339.2f}, {72.2f,327.6f}, {72.2f,327.6f}, {73.f,324.4f}, {70.2f,320.4f}, {70.2f,320.4f},
      {83.8f,342.f}, {76.6f,313.2f}, {76.6f,313.2f}, {87.801f,321.2f}, {89.001f,321.2f},
      {89.001f,321.2f}, {75.4f,298.f}, {84.2f,302.8f}, {84.2f,302.8f}, {79.f,292.4f},
      {97.001f,304.4f}, {97.001f,304.4f}, {81.f,288.4f}, {98.601f,298.f}, {98.601f,298.f},
      {106.6f,304.4f}, {99.001f,294.4f}, {99.001f,294.4f}, {84.6f,278.4f}, {106.6f,296.4f},
      {106.6f,296.4f}, {118.2f,312.8f}, {119.f,315.6f}, {119.f,315.6f}, {109.f,286.4f},
      {104.6f,283.6f}, {104.6f,283.6f}, {113.f,247.2f}, {154.2f,262.8f}, {154.2f,262.8f},
      {161.f,280.f}, {165.4f,261.6f}, {165.4f,261.6f}, {178.2f,255.2f}, {189.4f,282.8f},
      {189.4f,282.8f}, {193.4f,269.2f}, {192.6f,266.4f}, {192.6f,266.4f}, {199.4f,267.6f},
      {198.6f,266.4f}, {198.6f,266.4f}, {211.8f,270.8f}, {213.f,270.f}, {213.f,270.f},
      {219.8f,276.8f}, {220.2f,273.2f}, {220.2f,273.2f}, {229.4f,276.f}, {227.4f,272.4f},
      {227.4f,272.4f}, {236.2f,288.f}, {236.6f,291.6f}, {239.f,277.6f}, {241.f,280.4f},
      {241.f,280.4f}, {242.6f,272.8f}, {241.8f,271.6f}, {241.f,270.4f}, {261.8f,278.4f},
      {266.6f,299.2f}, {268.6f,307.6f}, {268.6f,307.6f}, {274.6f,292.8f}, {273.f,288.8f},
      {273.f,288.8f}, {278.2f,289.6f}, {278.6f,294.f}, {278.6f,294.f}, {282.6f,270.8f},
      {277.8f,264.8f}, {277.8f,264.8f}, {282.2f,264.f}, {283.4f,267.6f}, {283.4f,260.4f},
      {283.4f,260.4f}, {290.6f,261.2f}, {290.6f,258.8f}, {290.6f,258.8f}, {295.f,254.8f},
      {297.f,259.6f}, {297.f,259.6f}, {284.6f,224.4f}, {303.f,243.6f}, {303.f,243.6f},
      {310.2f,254.4f}, {306.6f,235.6f}, {303.f,216.8f}, {299.f,215.2f}, {303.8f,214.8f},
      {303.8f,214.8f}, {304.6f,211.2f}, {302.6f,209.6f}, {300.6f,208.f}, {303.8f,209.6f},
      {303.8f,209.6f}, {303.8f,209.6f}, {308.6f,213.6f}, {303.4f,191.6f}, {303.4f,191.6f},
      {309.8f,193.2f}, {297.8f,164.f}, {297.8f,164.f}, {300.6f,161.6f}, {296.6f,153.2f},
      {296.6f,153.2f}, {304.6f,157.6f}, {307.4f,156.f}, {307.4f,156.f}, {307.f,154.4f},
      {303.8f,150.4f}, {303.8f,150.4f}, {282.2f,95.6f}, {302.6f,117.6f}, {302.6f,117.6f},
      {314.45f,131.15f}, {308.05f,108.35f}, {308.05f,108.35f}, {298.94f,84.341f}, {299.72f,80.045f},
      {-129.83f,103.06f}}},
    {"mcccccclclcccccccccccclccccccccccccccccccccz",
     {{299.72f,80.245f}, {300.34f,80.426f}, {302.55f,81.55f}, {303.8f,83.2f}, {303.8f,83.2f},
      {310.6f,94.f}, {305.4f,75.6f}, {305.4f,75.6f}, {296.2f,46.8f}, {305.f,58.f}, {305.f,58.f},
      {311.f,65.2f}, {307.8f,51.6f}, {303.94f,35.173f}, {301.4f,28.8f}, {301.4f,28.8f},
      {301.4f,28.8f}, {313.f,33.6f}, {286.2f,-5.99999f}, {295.f,-2.39999f}, {295.f,-2.39999f},
      {275.4f,-42.f}, {253.8f,-47.2f}, {245.8f,-53.2f}, {245.8f,-53.2f}, {284.2f,-91.2f},
      {271.4f,-128.f}, {271.4f,-128.f}, {264.6f,-133.2f}, {255.f,-124.f}, {255.f,-124.f},
      {248.6f,-119.2f}, {242.6f,-120.8f}, {242.6f,-120.8f}, {211.8f,-119.6f}, {209.8f,-119.6f},
      {207.8f,-119.6f}, {173.f,-156.8f}, {107.4f,-139.2f}, {107.4f,-139.2f}, {102.2f,-137.2f},
      {97.801f,-138.4f}, {97.801f,-138.4f}, {79.4f,-154.4f}, {30.6f,-131.6f}, {30.6f,-131.6f},
      {20.6f,-129.6f}, {19.f,-129.6f}, {17.4f,-129.6f}, {14.6f,-129.6f}, {6.59998f,-123.2f},
      {-1.40002f,-116.8f}, {-1.80002f,-116.f}, {-3.80002f,-114.4f}, {-3.80002f,-114.4f},
      {-20.2f,-103.2f}, {-25.f,-102.4f}, {-25.f,-102.4f}, {-36.6f,-96.f}, {-41.f,-86.f},
      {-44.6f,-84.8f}, {-44.6f,-84.8f}, {-46.2f,-77.6f}, {-46.6f,-76.4f}, {-46.6f,-76.4f},
      {-51.4f,-72.8f}, {-52.2f,-67.2f}, {-52.2f,-67.2f}, {-61.f,-61.2f}, {-60.6f,-56.8f},
      {-60.6f,-56.8f}, {-62.2f,-51.6f}, {-63.f,-46.8f}, {-63.f,-46.8f}, {-70.2f,-42.f},
      {-69.4f,-39.2f}, {-69.4f,-39.2f}, {-77.f,-25.2f}, {-75.8f,-18.4f}, {-75.8f,-18.4f},
      {-82.2f,-18.8f}, {-85.f,-16.4f}, {-85.f,-16.4f}, {-85.8f,-11.6f}, {-87.4f,-11.2f},
      {-87.4f,-11.2f}, {-90.2f,-10.f}, {-87.8f,-6.00001f}, {-87.8f,-6.00001f}, {-89.4f,-3.20001f},
      {-89.8f,-1.60001f}, {-89.8f,-1.60001f}, {-89.f,1.19999f}, {-93.4f,6.79999f},
      {-93.4f,6.79999f}, {-99.8f,25.6f}, {-97.8f,30.8f}, {-97.8f,30.8f}, {-97.4f,35.6f},
      {-100.2f,37.2f}, {-100.2f,37.2f}, {-103.8f,36.8f}, {-95.4f,48.8f}, {-95.4f,48.8f},
      {-94.6f,50.f}, {-97.8f,52.4f}, {-97.8f,52.4f}, {-115.f,56.f}, {-117.4f,72.4f},
      {-117.4f,72.4f}, {-131.f,87.2f}, {-131.f,92.4f}, {-131.f,94.705f}, {-130.73f,97.852f},
      {-130.03f,102.46f}, {-130.03f,102.46f}, {-130.6f,110.8f}, {-103.f,111.6f}, {-75.4f,112.4f},
      {299.72f,80.245f}, {299.72f,80.245f}}},
    {"mcccclccccccccccclz",
     {{-115.6f,102.6f}, {-140.6f,63.2f}, {-126.2f,119.6f}, {-126.2f,119.6f}, {-117.4f,154.f},
      {12.2f,116.4f}, {12.2f,116.4f}, {12.2f,116.4f}, {181.f,86.f}, {192.2f,82.f}, {203.4f,78.f},
      {298.6f,84.4f}, {298.6f,84.4f}, {293.f,67.6f}, {228.2f,21.2f}, {209.f,44.4f}, {195.4f,40.4f},
      {181.8f,36.4f}, {184.2f,46.f}, {181.f,46.8f}, {177.8f,47.6f}, {138.6f,22.8f}, {132.2f,23.6f},
      {125.8f,24.4f}, {100.46f,0.649006f}, {115.4f,32.4f}, {131.4f,66.4f}, {57.f,71.6f},
      {40.2f,60.4f}, {23.4f,49.2f}, {47.4f,78.8f}, {47.4f,78.8f}, {65.8f,98.8f}, {31.4f,82.f},
      {31.4f,82.f}, {-3.f,69.2f}, {-27.f,94.8f}, {-30.2f,95.6f}, {-33.4f,96.4f}, {-38.2f,99.6f},
      {-39.f,93.2f}, {-39.8f,86.8f}, {-47.31f,70.099f}, {-79.f,96.4f}, {-99.f,113.f},
      {-112.8f,91.f}, {-112.8f,91.f}, {-115.6f,102.6f}}},
    {"mcccccccclcccclcccz",
     {{133.51f,25.346f}, {127.11f,26.146f}, {101.74f,2.407f}, {116.71f,34.146f}, {133.31f,69.346f},
      {58.31f,73.346f}, {41.51f,62.146f}, {24.709f,50.946f}, {48.71f,80.546f}, {48.71f,80.546f},
      {67.11f,100.55f}, {32.709f,83.746f}, {32.709f,83.746f}, {-1.69101f,70.946f},
      {-25.691f,96.546f}, {-28.891f,97.346f}, {-32.091f,98.146f}, {-36.891f,101.35f},
      {-37.691f,94.946f}, {-38.491f,88.546f}, {-45.87f,72.012f}, {-77.691f,98.146f},
      {-98.927f,115.49f}, {-112.42f,94.037f}, {-112.42f,94.037f}, {-115.62f,104.15f},
      {-140.62f,64.346f}, {-125.55f,122.66f}, {-125.55f,122.66f}, {-116.74f,157.06f},
      {13.51f,118.15f}, {13.51f,118.15f}, {13.51f,118.15f}, {182.31f,87.746f}, {193.51f,83.746f},
      {204.71f,79.746f}, {299.04f,86.073f}, {299.04f,86.073f}, {293.51f,68.764f}, {228.71f,22.364f},
      {210.31f,46.146f}, {196.71f,42.146f}, {183.11f,38.146f}, {185.51f,47.746f}, {182.31f,48.546f},
      {179.11f,49.346f}, {139.91f,24.546f}, {133.51f,25.346f}}},
    {"mcccccccclcccclcccz",
     {{134.82f,27.091f}, {128.42f,27.891f}, {103.68f,3.862f}, {118.02f,35.891f}, {134.22f,72.092f},
      {59.619f,75.092f}, {42.819f,63.892f}, {26.019f,52.692f}, {50.019f,82.292f}, {50.019f,82.292f},
      {68.419f,102.29f}, {34.019f,85.492f}, {34.019f,85.492f}, {-0.380993f,72.692f},
      {-24.382f,98.292f}, {-27.582f,99.092f}, {-30.782f,99.892f}, {-35.582f,103.09f},
      {-36.382f,96.692f}, {-37.182f,90.292f}, {-44.43f,73.925f}, {-76.382f,99.892f},
      {-98.855f,117.98f}, {-112.04f,97.074f}, {-112.04f,97.074f}, {-115.64f,105.69f},
      {-139.44f,66.692f}, {-124.89f,125.71f}, {-124.89f,125.71f}, {-116.09f,160.11f},
      {14.82f,119.89f}, {14.82f,119.89f}, {14.82f,119.89f}, {183.62f,89.492f}, {194.82f,85.492f},
      {206.02f,81.492f}, {299.47f,87.746f}, {299.47f,87.746f}, {294.02f,69.928f}, {229.22f,23.528f},
      {211.62f,47.891f}, {198.02f,43.891f}, {184.42f,39.891f}, {186.82f,49.491f}, {183.62f,50.292f},
      {180.42f,51.092f}, {141.22f,26.291f}, {134.82f,27.091f}}},
    {"mcccccccclcccclcccz",
     {{136.13f,28.837f}, {129.73f,29.637f}, {105.f,5.605f}, {119.33f,37.637f}, {136.13f,75.193f},
      {60.394f,76.482f}, {44.128f,65.637f}, {27.328f,54.437f}, {51.328f,84.037f}, {51.328f,84.037f},
      {69.728f,104.04f}, {35.328f,87.237f}, {35.328f,87.237f}, {0.927998f,74.437f},
      {-23.072f,100.04f}, {-26.272f,100.84f}, {-29.472f,101.64f}, {-34.272f,104.84f},
      {-35.072f,98.437f}, {-35.872f,92.037f}, {-42.989f,75.839f}, {-75.073f,101.64f},
      {-98.782f,120.47f}, {-111.66f,100.11f}, {-111.66f,100.11f}, {-115.66f,107.24f},
      {-137.46f,70.437f}, {-124.24f,128.76f}, {-124.24f,128.76f}, {-115.44f,163.16f},
      {16.13f,121.64f}, {16.13f,121.64f}, {16.13f,121.64f}, {184.93f,91.237f}, {196.13f,87.237f},
      {207.33f,83.237f}, {299.91f,89.419f}, {299.91f,89.419f}, {294.53f,71.092f}, {229.73f,24.691f},
      {212.93f,49.637f}, {199.33f,45.637f}, {185.73f,41.637f}, {188.13f,51.237f}, {184.93f,52.037f},
      {181.73f,52.837f}, {142.53f,28.037f}, {136.13f,28.837f}}},
    {"mcccccccclcccclcccz",
     {{137.44f,30.583f}, {131.04f,31.383f}, {106.81f,7.129f}, {120.64f,39.383f}, {137.44f,78.583f},
      {62.237f,78.583f}, {45.437f,67.383f}, {28.637f,56.183f}, {52.637f,85.783f}, {52.637f,85.783f},
      {71.037f,105.78f}, {36.637f,88.983f}, {36.637f,88.983f}, {2.237f,76.183f}, {-21.763f,101.78f},
      {-24.963f,102.58f}, {-28.163f,103.38f}, {-32.963f,106.58f}, {-33.763f,100.18f},
      {-34.563f,93.783f}, {-41.548f,77.752f}, {-73.763f,103.38f}, {-98.709f,122.96f},
      {-111.27f,103.15f}, {-111.27f,103.15f}, {-115.67f,108.78f}, {-135.47f,73.982f},
      {-123.58f,131.82f}, {-123.58f,131.82f}, {-114.78f,166.22f}, {17.44f,123.38f},
      {17.44f,123.38f}, {17.44f,123.38f}, {186.24f,92.983f}, {197.44f,88.983f}, {208.64f,84.983f},
      {300.35f,91.092f}, {300.35f,91.092f}, {295.04f,72.255f}, {230.24f,25.855f}, {214.24f,51.383f},
      {200.64f,47.383f}, {187.04f,43.383f}, {189.44f,52.983f}, {186.24f,53.783f}, {183.04f,54.583f},
      {143.84f,29.783f}, {137.44f,30.583f}}},
    {"mcccccccclcccclcccz",
     {{138.75f,32.328f}, {132.35f,33.128f}, {106.38f,9.677f}, {121.95f,41.128f}, {141.15f,79.928f},
      {63.546f,80.328f}, {46.746f,69.128f}, {29.946f,57.928f}, {53.946f,87.528f}, {53.946f,87.528f},
      {72.346f,107.53f}, {37.946f,90.728f}, {37.946f,90.728f}, {3.54599f,77.928f},
      {-20.454f,103.53f}, {-23.654f,104.33f}, {-26.854f,105.13f}, {-31.654f,108.33f},
      {-32.454f,101.93f}, {-33.254f,95.528f}, {-40.108f,79.665f}, {-72.454f,105.13f},
      {-98.636f,125.46f}, {-110.89f,106.18f}, {-110.89f,106.18f}, {-115.69f,110.33f},
      {-133.69f,77.128f}, {-122.93f,134.87f}, {-122.93f,134.87f}, {-114.13f,169.27f},
      {18.75f,125.13f}, {18.75f,125.13f}, {18.75f,125.13f}, {187.55f,94.728f}, {198.75f,90.728f},
      {209.95f,86.728f}, {300.78f,92.764f}, {300.78f,92.764f}, {295.55f,73.419f}, {230.75f,27.019f},
      {215.55f,53.128f}, {201.95f,49.128f}, {188.35f,45.128f}, {190.75f,54.728f}, {187.55f,55.528f},
      {184.35f,56.328f}, {145.15f,31.528f}, {138.75f,32.328f}}},
    {"mcccccccclcccclcccz",
     {{140.06f,34.073f}, {133.66f,34.873f}, {107.31f,11.613f}, {123.26f,42.873f}, {143.66f,82.874f},
      {64.855f,82.074f}, {48.055f,70.874f}, {31.255f,59.674f}, {55.255f,89.274f}, {55.255f,89.274f},
      {73.655f,109.27f}, {39.255f,92.474f}, {39.255f,92.474f}, {4.85499f,79.674f},
      {-19.145f,105.27f}, {-22.345f,106.07f}, {-25.545f,106.87f}, {-30.345f,110.07f},
      {-31.145f,103.67f}, {-31.945f,97.274f}, {-38.668f,81.578f}, {-71.145f,106.87f},
      {-98.564f,127.95f}, {-110.51f,109.22f}, {-110.51f,109.22f}, {-115.71f,111.87f},
      {-131.71f,81.674f}, {-122.27f,137.93f}, {-122.27f,137.93f}, {-113.47f,172.33f},
      {20.05f,126.87f}, {20.05f,126.87f}, {20.05f,126.87f}, {188.85f,96.474f}, {200.05f,92.474f},
      {211.25f,88.474f}, {301.21f,94.437f}, {301.21f,94.437f}, {296.05f,74.583f}, {231.25f,28.183f},
      {216.85f,54.874f}, {203.25f,50.874f}, {189.65f,46.873f}, {192.05f,56.474f}, {188.85f,57.274f},
      {185.65f,58.074f}, {146.45f,33.273f}, {140.05f,34.073f}}},
    {"mcccccccclcccclcccz",
     {{141.36f,35.819f}, {134.96f,36.619f}, {107.52f,13.944f}, {124.56f,44.619f}, {146.56f,84.219f},
      {66.164f,83.819f}, {49.364f,72.619f}, {32.564f,61.419f}, {56.564f,91.019f}, {56.564f,91.019f},
      {74.964f,111.02f}, {40.564f,94.219f}, {40.564f,94.219f}, {6.164f,81.419f}, {-17.836f,107.02f},
      {-21.036f,107.82f}, {-24.236f,108.62f}, {-29.036f,111.82f}, {-29.836f,105.42f},
      {-30.636f,99.019f}, {-37.227f,83.492f}, {-69.836f,108.62f}, {-98.491f,130.44f},
      {-110.13f,112.26f}, {-110.13f,112.26f}, {-115.73f,113.42f}, {-130.13f,85.019f},
      {-121.62f,140.98f}, {-121.62f,140.98f}, {-112.82f,175.38f}, {21.36f,128.62f},
      {21.36f,128.62f}, {21.36f,128.62f}, {190.16f,98.219f}, {201.36f,94.219f}, {212.56f,90.219f},
      {301.66f,96.11f}, {301.66f,96.11f}, {296.56f,75.746f}, {231.76f,29.346f}, {218.16f,56.619f},
      {204.56f,52.619f}, {190.96f,48.619f}, {193.36f,58.219f}, {190.16f,59.019f}, {186.96f,59.819f},
      {147.76f,35.019f}, {141.36f,35.819f}}},
    {"mcccccccclcccclcccz",
     {{142.67f,37.565f}, {136.27f,38.365f}, {108.83f,15.689f}, {125.87f,46.365f}, {147.87f,85.965f},
      {67.474f,85.565f}, {50.674f,74.365f}, {33.874f,63.165f}, {57.874f,92.765f}, {57.874f,92.765f},
      {76.274f,112.76f}, {41.874f,95.965f}, {41.874f,95.965f}, {7.473f,83.165f}, {-16.527f,108.76f},
      {-19.727f,109.56f}, {-22.927f,110.36f}, {-27.727f,113.56f}, {-28.527f,107.16f},
      {-29.327f,100.76f}, {-35.786f,85.405f}, {-68.527f,110.36f}, {-98.418f,132.93f},
      {-109.74f,115.29f}, {-109.74f,115.29f}, {-115.74f,114.96f}, {-129.35f,88.564f},
      {-120.96f,144.04f}, {-120.96f,144.04f}, {-112.16f,178.44f}, {22.67f,130.36f},
      {22.67f,130.36f}, {22.67f,130.36f}, {191.47f,99.965f}, {202.67f,95.965f}, {213.87f,91.965f},
      {302.09f,97.783f}, {302.09f,97.783f}, {297.08f,76.91f}, {232.27f,30.51f}, {219.47f,58.365f},
      {205.87f,54.365f}, {192.27f,50.365f}, {194.67f,59.965f}, {191.47f,60.765f}, {188.27f,61.565f},
      {149.07f,36.765f}, {142.67f,37.565f}}},
    {"mcccccccclcccclcccz",
     {{143.98f,39.31f}, {137.58f,40.11f}, {110.53f,17.223f}, {127.18f,48.11f}, {149.18f,88.91f},
      {68.783f,87.31f}, {51.983f,76.11f}, {35.183f,64.91f}, {59.183f,94.51f}, {59.183f,94.51f},
      {77.583f,114.51f}, {43.183f,97.71f}, {43.183f,97.71f}, {8.78299f,84.91f}, {-15.217f,110.51f},
      {-18.417f,111.31f}, {-21.618f,112.11f}, {-26.418f,115.31f}, {-27.218f,108.91f},
      {-28.018f,102.51f}, {-34.346f,87.318f}, {-67.218f,112.11f}, {-98.345f,135.42f},
      {-109.36f,118.33f}, {-109.36f,118.33f}, {-115.76f,116.51f}, {-128.76f,92.51f},
      {-120.31f,147.09f}, {-120.31f,147.09f}, {-111.51f,181.49f}, {23.98f,132.11f},
      {23.98f,132.11f}, {23.98f,132.11f}, {192.78f,101.71f}, {203.98f,97.71f}, {215.18f,93.71f},
      {302.53f,99.456f}, {302.53f,99.456f}, {297.58f,78.074f}, {232.78f,31.673f}, {220.78f,60.11f},
      {207.18f,56.11f}, {193.58f,52.11f}, {195.98f,61.71f}, {192.78f,62.51f}, {189.58f,63.31f},
      {150.38f,38.51f}, {143.98f,39.31f}}},
    {"mcccccccclcccclcccz",
     {{145.29f,41.055f}, {138.89f,41.855f}, {112.92f,18.411f}, {128.49f,49.855f}, {149.69f,92.656f},
      {70.092f,89.056f}, {53.292f,77.856f}, {36.492f,66.656f}, {60.492f,96.256f}, {60.492f,96.256f},
      {78.892f,116.26f}, {44.492f,99.456f}, {44.492f,99.456f}, {10.092f,86.656f},
      {-13.908f,112.26f}, {-17.108f,113.06f}, {-20.308f,113.86f}, {-25.108f,117.06f},
      {-25.908f,110.66f}, {-26.708f,104.26f}, {-32.905f,89.232f}, {-65.908f,113.86f},
      {-98.273f,137.91f}, {-108.98f,121.36f}, {-108.98f,121.36f}, {-115.78f,118.06f},
      {-128.58f,94.856f}, {-119.65f,150.15f}, {-119.65f,150.15f}, {-110.85f,184.55f},
      {25.29f,133.86f}, {25.29f,133.86f}, {25.29f,133.86f}, {194.09f,103.46f}, {205.29f,99.456f},
      {216.49f,95.456f}, {302.96f,101.13f}, {302.96f,101.13f}, {298.09f,79.237f}, {233.29f,32.837f},
      {222.09f,61.856f}, {208.49f,57.856f}, {194.89f,53.855f}, {197.29f,63.456f}, {194.09f,64.256f},
      {190.89f,65.056f}, {151.69f,40.255f}, {145.29f,41.055f}}},
    {"mcccclccccccccccclz",
     {{-115.8f,119.6f}, {-128.6f,97.6f}, {-119.f,153.2f}, {-119.f,153.2f}, {-110.2f,187.6f},
      {26.6f,135.6f}, {26.6f,135.6f}, {26.6f,135.6f}, {195.4f,105.2f}, {206.6f,101.2f},
      {217.8f,97.2f}, {303.4f,102.8f}, {303.4f,102.8f}, {298.6f,80.4f}, {233.8f,34.f},
      {223.4f,63.6f}, {209.8f,59.6f}, {196.2f,55.6f}, {198.6f,65.2f}, {195.4f,66.f}, {192.2f,66.8f},
      {153.f,42.f}, {146.6f,42.8f}, {140.2f,43.6f}, {114.98f,19.793f}, {129.8f,51.6f},
      {152.03f,99.307f}, {69.041f,89.227f}, {54.6f,79.6f}, {37.8f,68.4f}, {61.8f,98.f},
      {61.8f,98.f}, {80.2f,118.f}, {45.8f,101.2f}, {45.8f,101.2f}, {11.4f,88.4f}, {-12.6f,114.f},
      {-15.8f,114.8f}, {-19.f,115.6f}, {-23.8f,118.8f}, {-24.6f,112.4f}, {-25.4f,106.f},
      {-31.465f,91.144f}, {-64.6f,115.6f}, {-98.2f,140.4f}, {-108.6f,124.4f}, {-108.6f,124.4f},
      {-115.8f,119.6f}}},
    {"mlcccccccccccccclcccccclz",
     {{-34.6f,266.4f}, {-44.6f,274.f}, {-44.6f,274.f}, {-34.2f,266.4f}, {-30.6f,267.6f},
      {-30.6f,267.6f}, {-37.4f,278.8f}, {-38.2f,284.f}, {-38.2f,284.f}, {-27.8f,271.2f},
      {-22.2f,271.6f}, {-22.2f,271.6f}, {-14.6f,272.f}, {-14.6f,282.8f}, {-14.6f,282.8f},
      {-9.f,272.4f}, {-5.8f,272.8f}, {-5.8f,272.8f}, {-4.6f,279.2f}, {-5.8f,286.f}, {-5.8f,286.f},
      {-1.8f,278.4f}, {2.2f,280.f}, {2.2f,280.f}, {8.6f,278.f}, {7.8f,289.6f}, {7.8f,289.6f},
      {7.8f,300.f}, {7.f,302.8f}, {7.f,302.8f}, {12.6f,276.4f}, {15.f,276.f}, {15.f,276.f},
      {23.f,274.8f}, {27.8f,283.6f}, {27.8f,283.6f}, {23.8f,276.f}, {28.6f,278.f}, {28.6f,278.f},
      {39.4f,279.6f}, {42.6f,286.4f}, {42.6f,286.4f}, {35.8f,274.4f}, {41.4f,277.6f}, {49.4f,284.f},
      {49.4f,284.f}, {57.8f,305.2f}, {59.8f,306.8f}, {59.8f,306.8f}, {52.2f,285.2f}, {53.8f,285.2f},
      {53.8f,285.2f}, {51.8f,273.2f}, {57.f,288.f}, {57.f,288.f}, {53.8f,274.f}, {59.4f,274.8f},
      {65.f,275.6f}, {69.4f,285.6f}, {77.8f,283.2f}, {77.8f,283.2f}, {87.401f,288.8f},
      {89.401f,219.6f}, {-34.599f,266.4f}}},
    {"mcccllccccccccz",
     {{-29.8f,173.6f}, {-29.8f,173.6f}, {-15.f,167.6f}, {25.f,173.6f}, {25.f,173.6f}, {32.2f,174.f},
      {39.f,165.2f}, {45.8f,156.4f}, {72.6f,149.2f}, {79.f,151.2f}, {88.601f,157.6f},
      {89.401f,158.8f}, {89.401f,158.8f}, {101.8f,169.2f}, {102.2f,176.8f}, {102.6f,184.4f},
      {87.801f,232.4f}, {78.2f,248.4f}, {68.6f,264.4f}, {59.f,276.8f}, {39.8f,274.4f},
      {39.8f,274.4f}, {19.f,270.4f}, {-6.60001f,274.4f}, {-6.60001f,274.4f}, {-35.8f,272.8f},
      {-38.6f,264.8f}, {-41.4f,256.8f}, {-27.4f,241.6f}, {-27.4f,241.6f}, {-27.4f,241.6f},
      {-23.f,233.2f}, {-24.2f,218.8f}, {-25.4f,204.4f}, {-25.f,176.4f}, {-29.8f,173.6f}}},
    {"mccccccz",
     {{-7.8f,175.6f}, {0.599999f,194.f}, {-29.f,259.2f}, {-29.f,259.2f}, {-31.f,260.8f},
      {-16.34f,266.85f}, {-6.2f,264.4f}, {4.746f,261.76f}, {45.f,266.f}, {45.f,266.f},
      {68.6f,250.4f}, {81.4f,206.f}, {81.4f,206.f}, {81.4f,206.f}, {91.801f,182.f}, {74.2f,178.8f},
      {56.6f,175.6f}, {-7.8f,175.6f}, {-7.8f,175.6f}}},
    {"mccclccccccz",
     {{-9.8f,174.4f}, {-9.8f,174.4f}, {-12.6f,196.8f}, {-9.4f,205.2f}, {-6.2f,213.6f},
      {-7.f,215.6f}, {-7.8f,219.6f}, {-8.6f,223.6f}, {-4.2f,233.6f}, {1.4f,239.6f}, {13.4f,241.2f},
      {13.4f,241.2f}, {28.6f,237.6f}, {37.8f,240.4f}, {37.8f,240.4f}, {46.794f,241.74f},
      {50.2f,226.8f}, {50.2f,226.8f}, {55.f,220.4f}, {62.2f,217.6f}, {69.4f,214.8f}, {76.6f,173.2f},
      {72.6f,165.2f}, {68.6f,157.2f}, {54.2f,152.8f}, {38.2f,168.4f}, {22.2f,184.f}, {20.2f,167.2f},
      {-9.8f,174.4f}}},
    {"mccccccccz",
     {{43.88f,40.321f}, {71.601f,44.281f}, {97.121f,8.641f}, {98.881f,-1.04f}, {100.64f,-10.72f},
      {90.521f,-22.6f}, {90.521f,-22.6f}, {91.841f,-25.68f}, {87.001f,-39.76f}, {81.721f,-49.f},
      {76.441f,-58.24f}, {60.54f,-57.266f}, {43.f,-58.24f}, {27.16f,-59.12f}, {8.67999f,-35.8f},
      {7.35999f,-34.04f}, {6.03999f,-32.28f}, {12.2f,6.001f}, {13.52f,11.721f}, {14.84f,17.441f},
      {12.2f,43.841f}, {12.2f,43.841f}, {46.44f,34.741f}, {16.16f,36.361f}, {43.88f,40.321f}}},
    {"mccccccccz",
     {{8.088f,-33.392f}, {6.792f,-31.664f}, {12.84f,5.921f}, {14.136f,11.537f}, {15.432f,17.153f},
      {12.84f,43.073f}, {12.84f,43.073f}, {45.512f,34.193f}, {16.728f,35.729f}, {43.944f,39.617f},
      {71.161f,43.505f}, {96.217f,8.513f}, {97.945f,-0.992001f}, {99.673f,-10.496f},
      {89.737f,-22.16f}, {89.737f,-22.16f}, {91.033f,-25.184f}, {86.281f,-39.008f},
      {81.097f,-48.08f}, {75.913f,-57.152f}, {60.302f,-56.195f}, {43.08f,-57.152f},
      {27.528f,-58.016f}, {9.384f,-35.12f}, {8.088f,-33.392f}}},
    {"mccccccccz",
     {{8.816f,-32.744f}, {7.544f,-31.048f}, {13.48f,5.841f}, {14.752f,11.353f}, {16.024f,16.865f},
      {13.48f,42.305f}, {13.48f,42.305f}, {44.884f,33.145f}, {17.296f,35.097f}, {44.008f,38.913f},
      {70.721f,42.729f}, {95.313f,8.385f}, {97.009f,-0.943996f}, {98.705f,-10.272f},
      {88.953f,-21.72f}, {88.953f,-21.72f}, {90.225f,-24.688f}, {85.561f,-38.256f},
      {80.473f,-47.16f}, {75.385f,-56.064f}, {60.063f,-55.125f}, {43.16f,-56.064f},
      {27.896f,-56.912f}, {10.088f,-34.44f}, {8.81601f,-32.744f}}},
    {"mccccccccz",
     {{9.544f,-32.096f}, {8.296f,-30.432f}, {14.12f,5.761f}, {15.368f,11.169f}, {16.616f,16.577f},
      {14.12f,41.537f}, {14.12f,41.537f}, {43.556f,32.497f}, {17.864f,34.465f}, {44.072f,38.209f},
      {70.281f,41.953f}, {94.409f,8.257f}, {96.073f,-0.895f}, {97.737f,-10.048f}, {88.169f,-21.28f},
      {88.169f,-21.28f}, {89.417f,-24.192f}, {84.841f,-37.504f}, {79.849f,-46.24f},
      {74.857f,-54.976f}, {59.824f,-54.055f}, {43.24f,-54.976f}, {28.264f,-55.808f},
      {10.792f,-33.76f}, {9.544f,-32.096f}}},
    {"mccccccccz",
     {{10.272f,-31.448f}, {9.048f,-29.816f}, {14.76f,5.681f}, {15.984f,10.985f}, {17.208f,16.289f},
      {14.76f,40.769f}, {14.76f,40.769f}, {42.628f,31.849f}, {18.432f,33.833f}, {44.136f,37.505f},
      {69.841f,41.177f}, {93.505f,8.129f}, {95.137f,-0.848003f}, {96.769f,-9.824f},
      {87.385f,-20.84f}, {87.385f,-20.84f}, {88.609f,-23.696f}, {84.121f,-36.752f},
      {79.225f,-45.32f}, {74.329f,-53.888f}, {59.585f,-52.985f}, {43.32f,-53.888f},
      {28.632f,-54.704f}, {11.496f,-33.08f}, {10.272f,-31.448f}}},
    {"mccccccccz",
     {{44.2f,36.8f}, {69.4f,40.4f}, {92.601f,8.f}, {94.201f,-0.799999f}, {95.801f,-9.6f},
      {86.601f,-20.4f}, {86.601f,-20.4f}, {87.801f,-23.2f}, {83.4f,-36.f}, {78.6f,-44.4f},
      {73.8f,-52.8f}, {59.346f,-51.914f}, {43.4f,-52.8f}, {29.f,-53.6f}, {12.2f,-32.4f},
      {11.f,-30.8f}, {9.8f,-29.2f}, {15.4f,5.6f}, {16.6f,10.8f}, {17.8f,16.f}, {15.4f,40.f},
      {15.4f,40.f}, {40.9f,31.4f}, {19.f,33.2f}, {44.2f,36.8f}}},
    {"mclllccccccccccccccclccccclcccccccclcccllclcccccccccccccccclccz",
     {{143.8f,259.6f}, {143.8f,259.6f}, {164.2f,257.6f}, {171.f,250.8f}, {175.4f,254.4f},
      {193.f,216.f}, {196.6f,221.2f}, {196.6f,221.2f}, {211.f,206.4f}, {210.2f,198.4f},
      {209.4f,190.4f}, {223.f,204.4f}, {223.f,204.4f}, {223.f,204.4f}, {222.2f,192.8f},
      {229.4f,199.6f}, {229.4f,199.6f}, {227.f,184.f}, {235.4f,192.f}, {235.4f,192.f},
      {224.86f,161.84f}, {247.4f,187.6f}, {253.f,194.f}, {248.6f,187.2f}, {248.6f,187.2f},
      {248.6f,187.2f}, {222.6f,139.2f}, {244.2f,153.6f}, {244.2f,153.6f}, {246.2f,130.8f},
      {245.f,126.4f}, {243.8f,122.f}, {241.8f,99.6f}, {237.f,94.4f}, {232.2f,89.2f}, {237.4f,87.6f},
      {243.f,92.8f}, {243.f,92.8f}, {231.8f,68.8f}, {245.f,80.8f}, {245.f,80.8f}, {241.4f,65.6f},
      {237.f,62.8f}, {237.f,62.8f}, {231.4f,45.6f}, {246.6f,56.4f}, {246.6f,56.4f}, {242.2f,44.f},
      {239.f,40.8f}, {239.f,40.8f}, {227.4f,13.2f}, {234.6f,18.f}, {239.f,21.6f}, {239.f,21.6f},
      {232.2f,7.60001f}, {238.6f,12.f}, {245.f,16.4f}, {245.f,16.f}, {245.f,16.f}, {245.f,16.f},
      {223.8f,-17.2f}, {244.2f,0.400011f}, {244.2f,0.400011f}, {236.04f,-13.518f}, {232.6f,-20.4f},
      {232.6f,-20.4f}, {213.8f,-40.8f}, {228.2f,-34.4f}, {233.f,-32.8f}, {233.f,-32.8f},
      {224.2f,-42.8f}, {216.2f,-44.4f}, {208.2f,-46.f}, {218.6f,-52.4f}, {225.f,-50.4f},
      {231.4f,-48.4f}, {247.f,-40.8f}, {247.f,-40.8f}, {247.f,-40.8f}, {259.8f,-22.f},
      {263.8f,-21.6f}, {263.8f,-21.6f}, {243.8f,-29.2f}, {249.8f,-21.2f}, {249.8f,-21.2f},
      {264.2f,-7.19999f}, {257.f,-7.59999f}, {257.f,-7.59999f}, {251.f,-0.399987f},
      {255.8f,8.40001f}, {255.8f,8.40001f}, {237.34f,-9.99099f}, {252.2f,15.6f}, {259.f,32.f},
      {259.f,32.f}, {234.6f,7.20001f}, {245.8f,29.2f}, {245.8f,29.2f}, {263.f,52.8f}, {265.f,53.2f},
      {267.f,53.6f}, {271.4f,62.4f}, {271.4f,62.4f}, {267.f,60.4f}, {272.2f,69.2f}, {272.2f,69.2f},
      {261.f,57.2f}, {267.f,70.4f}, {272.6f,84.8f}, {272.6f,84.8f}, {252.2f,62.8f}, {265.8f,92.4f},
      {265.8f,92.4f}, {249.4f,87.2f}, {258.2f,104.4f}, {258.2f,104.4f}, {256.6f,120.4f},
      {257.f,125.6f}, {257.4f,130.8f}, {258.6f,159.2f}, {254.2f,167.2f}, {249.8f,175.2f},
      {260.2f,194.4f}, {262.2f,198.4f}, {264.2f,202.4f}, {267.8f,213.2f}, {259.f,204.f},
      {250.2f,194.8f}, {254.6f,200.4f}, {256.6f,209.2f}, {258.6f,218.f}, {264.6f,233.6f},
      {263.8f,239.2f}, {263.8f,239.2f}, {262.6f,240.4f}, {259.4f,236.8f}, {259.4f,236.8f},
      {244.6f,214.f}, {246.2f,228.4f}, {246.2f,228.4f}, {245.f,236.4f}, {241.8f,245.2f},
      {241.8f,245.2f}, {238.6f,256.f}, {238.6f,247.2f}, {238.6f,247.2f}, {235.4f,230.4f},
      {232.6f,238.f}, {229.8f,245.6f}, {226.2f,251.6f}, {223.4f,254.f}, {220.6f,256.4f},
      {215.4f,233.6f}, {214.2f,244.f}, {214.2f,244.f}, {202.2f,231.6f}, {197.4f,248.f},
      {185.8f,264.4f}, {185.8f,264.4f}, {185.4f,252.f}, {184.2f,258.f}, {184.2f,258.f},
      {154.2f,264.f}, {143.8f,259.6f}}},
    {"mcccccccccccccccccccccccccccccccccccccccccclccccccccccccccccccccccccccccccclccccccccccccccccl"
     "cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccclccccccccccccz",
     {{20.895f,54.407f}, {22.437f,55.87f}, {49.4f,84.8f}, {49.4f,84.8f}, {84.6f,121.4f},
      {56.6f,87.2f}, {56.6f,87.2f}, {49.f,82.4f}, {39.8f,63.6f}, {39.8f,63.6f}, {38.6f,60.8f},
      {53.8f,70.8f}, {53.8f,70.8f}, {57.8f,71.6f}, {71.4f,90.8f}, {71.4f,90.8f}, {64.6f,88.4f},
      {69.4f,95.6f}, {69.4f,95.6f}, {72.2f,97.6f}, {92.601f,113.2f}, {92.601f,113.2f},
      {96.201f,117.2f}, {100.2f,118.8f}, {100.2f,118.8f}, {114.2f,113.6f}, {107.8f,126.8f},
      {107.8f,126.8f}, {110.2f,133.6f}, {115.8f,122.f}, {115.8f,122.f}, {127.f,105.2f},
      {110.6f,107.6f}, {110.6f,107.6f}, {80.6f,110.4f}, {73.8f,94.4f}, {73.8f,94.4f}, {71.4f,92.f},
      {80.2f,94.4f}, {80.2f,94.4f}, {88.601f,96.4f}, {73.f,82.f}, {73.f,82.f}, {75.4f,82.f},
      {84.6f,88.8f}, {84.6f,88.8f}, {95.001f,98.f}, {97.001f,96.f}, {97.001f,96.f}, {115.f,87.2f},
      {125.4f,94.8f}, {125.4f,94.8f}, {127.4f,96.4f}, {121.8f,103.2f}, {123.4f,108.4f},
      {125.f,113.6f}, {129.8f,126.f}, {129.8f,126.f}, {127.4f,127.6f}, {127.8f,138.4f},
      {127.8f,138.4f}, {144.6f,161.6f}, {135.f,159.6f}, {135.f,159.6f}, {119.4f,159.2f},
      {134.2f,166.8f}, {134.2f,166.8f}, {137.4f,168.8f}, {146.2f,176.f}, {146.2f,176.f},
      {143.4f,174.8f}, {141.8f,180.f}, {141.8f,180.f}, {146.6f,184.f}, {143.8f,188.8f},
      {143.8f,188.8f}, {137.8f,190.f}, {136.6f,194.f}, {136.6f,194.f}, {143.4f,202.f},
      {133.4f,202.4f}, {133.4f,202.4f}, {137.f,206.8f}, {132.2f,218.8f}, {132.2f,218.8f},
      {127.4f,218.8f}, {121.f,224.4f}, {121.f,224.4f}, {123.4f,229.2f}, {113.f,234.8f},
      {113.f,234.8f}, {104.6f,236.4f}, {107.4f,243.2f}, {107.4f,243.2f}, {99.401f,249.2f},
      {97.001f,265.2f}, {97.001f,265.2f}, {96.201f,275.6f}, {93.801f,278.8f}, {99.001f,276.8f},
      {104.2f,274.8f}, {103.4f,262.4f}, {103.4f,262.4f}, {98.601f,246.8f}, {141.4f,230.8f},
      {141.4f,230.8f}, {145.4f,229.2f}, {146.2f,224.f}, {146.2f,224.f}, {148.2f,224.4f},
      {157.f,232.f}, {157.f,232.f}, {164.6f,243.2f}, {165.f,234.f}, {165.f,234.f}, {166.2f,230.4f},
      {164.6f,224.4f}, {164.6f,224.4f}, {170.6f,202.8f}, {156.6f,196.4f}, {156.6f,196.4f},
      {146.6f,162.8f}, {160.6f,171.2f}, {160.6f,171.2f}, {163.4f,176.8f}, {174.2f,182.f},
      {174.2f,182.f}, {177.8f,179.6f}, {176.2f,174.8f}, {184.6f,168.8f}, {184.6f,168.8f},
      {187.4f,175.2f}, {193.4f,167.2f}, {193.4f,167.2f}, {197.f,142.8f}, {209.4f,157.2f},
      {209.4f,157.2f}, {213.4f,158.4f}, {214.6f,151.6f}, {214.6f,151.6f}, {218.2f,141.2f},
      {214.6f,127.6f}, {214.6f,127.6f}, {218.2f,127.2f}, {227.8f,133.2f}, {227.8f,133.2f},
      {230.6f,129.6f}, {221.4f,112.8f}, {225.4f,115.2f}, {229.4f,117.6f}, {233.8f,119.2f},
      {233.8f,119.2f}, {234.6f,117.2f}, {224.6f,104.8f}, {224.6f,104.8f}, {220.2f,102.f},
      {215.f,81.5999f}, {215.f,81.5999f}, {222.2f,85.1999f}, {212.2f,69.9999f}, {212.2f,69.9999f},
      {212.2f,66.7999f}, {218.2f,55.5999f}, {218.2f,55.5999f}, {217.4f,48.7999f}, {218.2f,49.1999f},
      {218.2f,49.1999f}, {221.f,50.3999f}, {229.f,51.9999f}, {222.2f,45.5999f}, {215.4f,39.1999f},
      {223.f,34.3999f}, {223.f,34.3999f}, {227.4f,31.5999f}, {213.8f,31.9999f}, {213.8f,31.9999f},
      {208.6f,27.5999f}, {209.f,23.5999f}, {209.f,23.5999f}, {217.f,25.5999f}, {202.6f,11.1999f},
      {200.2f,7.59995f}, {197.8f,3.99994f}, {207.4f,-1.20006f}, {207.4f,-1.20006f},
      {220.6f,-4.80006f}, {209.f,-8.00006f}, {209.f,-8.00006f}, {189.4f,-7.60006f},
      {200.2f,-18.4001f}, {200.2f,-18.4001f}, {206.2f,-18.0001f}, {204.6f,-20.4001f},
      {204.6f,-20.4001f}, {199.4f,-21.6001f}, {189.8f,-28.0001f}, {189.8f,-28.0001f},
      {185.8f,-31.6001f}, {189.4f,-30.8001f}, {189.4f,-30.8001f}, {206.2f,-29.6001f},
      {177.4f,-40.8001f}, {177.4f,-40.8001f}, {185.4f,-40.8001f}, {167.4f,-51.2001f},
      {167.4f,-51.2001f}, {165.4f,-52.8001f}, {162.2f,-60.4001f}, {162.2f,-60.4001f},
      {156.2f,-65.6001f}, {151.4f,-72.4001f}, {151.4f,-72.4001f}, {151.f,-76.8001f},
      {146.2f,-81.6001f}, {146.2f,-81.6001f}, {134.6f,-95.2001f}, {129.f,-94.8f}, {129.f,-94.8f},
      {114.2f,-98.4f}, {109.f,-97.6001f}, {109.f,-97.6001f}, {56.2f,-93.2001f}, {29.8f,-80.4f},
      {37.6f,-59.4001f}, {37.6f,-59.4001f}, {44.f,-51.0001f}, {53.2f,-54.8001f}, {53.2f,-54.8001f},
      {57.8f,-61.0001f}, {69.4f,-58.8001f}, {69.4f,-58.8001f}, {89.801f,-55.6001f},
      {87.201f,-59.2001f}, {87.201f,-59.2001f}, {84.801f,-63.8001f}, {68.6f,-70.0001f},
      {68.4f,-70.6001f}, {68.2f,-71.2001f}, {59.4f,-74.6001f}, {59.4f,-74.6001f}, {56.4f,-75.8f},
      {52.f,-85.0001f}, {52.f,-85.0001f}, {48.8f,-88.4001f}, {64.6f,-82.6001f}, {64.6f,-82.6001f},
      {63.4f,-81.6001f}, {70.8f,-77.6001f}, {70.8f,-77.6001f}, {88.201f,-78.6001f},
      {98.801f,-67.8f}, {98.801f,-67.8f}, {109.6f,-51.2001f}, {109.8f,-59.4f}, {109.8f,-59.4f},
      {112.6f,-68.8f}, {100.8f,-90.f}, {100.8f,-90.f}, {101.2f,-92.f}, {109.4f,-85.4f},
      {109.4f,-85.4f}, {110.8f,-87.4f}, {111.6f,-81.6f}, {111.6f,-81.6f}, {111.8f,-79.2f},
      {115.6f,-71.2f}, {115.6f,-71.2f}, {118.4f,-58.2f}, {122.f,-65.6f}, {122.f,-65.6f},
      {126.6f,-56.2f}, {128.f,-53.6f}, {122.f,-46.f}, {122.f,-46.f}, {121.8f,-43.2f},
      {122.6f,-43.4f}, {117.f,-35.8f}, {111.4f,-28.2f}, {114.8f,-23.8f}, {114.8f,-23.8f},
      {113.4f,-17.2f}, {122.2f,-17.6f}, {122.2f,-17.6f}, {124.8f,-15.4f}, {128.2f,-15.4f},
      {128.2f,-15.4f}, {130.f,-13.4f}, {132.4f,-14.f}, {132.4f,-14.f}, {134.f,-17.8f},
      {140.2f,-15.8f}, {140.2f,-15.8f}, {141.6f,-18.2f}, {149.8f,-18.6f}, {149.8f,-18.6f},
      {150.8f,-21.2f}, {151.2f,-22.8f}, {154.6f,-23.4f}, {158.f,-24.f}, {133.4f,-67.f},
      {133.4f,-67.f}, {139.8f,-67.8f}, {131.6f,-80.2f}, {131.6f,-80.2f}, {129.4f,-86.8f},
      {140.8f,-72.2f}, {143.f,-70.8f}, {145.2f,-69.4f}, {146.2f,-67.2f}, {144.6f,-67.4f},
      {143.f,-67.6f}, {141.2f,-65.4f}, {142.6f,-65.2f}, {144.f,-65.f}, {157.f,-50.f},
      {160.4f,-39.8f}, {163.8f,-29.6f}, {169.8f,-25.6f}, {176.f,-19.6f}, {182.2f,-13.6f},
      {181.4f,10.6f}, {181.4f,10.6f}, {181.f,19.4f}, {187.f,30.f}, {187.f,30.f}, {189.f,33.8f},
      {184.8f,52.f}, {184.8f,52.f}, {182.8f,54.2f}, {184.2f,55.f}, {184.2f,55.f}, {185.2f,56.2f},
      {192.f,69.4f}, {192.f,69.4f}, {190.2f,69.2f}, {193.8f,72.8f}, {193.8f,72.8f}, {199.f,78.8f},
      {192.6f,75.8f}, {192.6f,75.8f}, {186.6f,74.2f}, {193.6f,84.f}, {193.6f,84.f}, {194.8f,85.8f},
      {185.8f,81.2f}, {185.8f,81.2f}, {176.6f,80.6f}, {188.2f,87.8f}, {188.2f,87.8f}, {196.8f,95.f},
      {185.4f,90.6f}, {185.4f,90.6f}, {180.8f,88.8f}, {184.f,95.6f}, {184.f,95.6f}, {187.2f,97.2f},
      {204.4f,104.2f}, {204.4f,104.2f}, {204.8f,108.f}, {201.8f,113.f}, {201.8f,113.f},
      {202.2f,117.f}, {200.f,120.4f}, {200.f,120.4f}, {198.8f,128.6f}, {198.2f,129.4f},
      {198.2f,129.4f}, {194.f,129.6f}, {186.6f,143.4f}, {186.6f,143.4f}, {184.8f,146.f},
      {174.6f,158.f}, {174.6f,158.f}, {172.6f,165.f}, {154.6f,157.8f}, {154.6f,157.8f},
      {148.f,161.2f}, {150.f,157.8f}, {150.f,157.8f}, {149.6f,155.6f}, {154.4f,149.6f},
      {154.4f,149.6f}, {161.4f,147.f}, {158.8f,136.2f}, {158.8f,136.2f}, {162.8f,134.8f},
      {151.6f,132.f}, {151.8f,130.8f}, {152.f,129.6f}, {157.8f,128.2f}, {157.8f,128.2f},
      {165.8f,126.2f}, {161.4f,123.8f}, {161.4f,123.8f}, {160.8f,119.8f}, {163.8f,114.2f},
      {163.8f,114.2f}, {175.4f,113.4f}, {163.8f,97.2f}, {163.8f,97.2f}, {153.f,89.6f},
      {152.f,83.8f}, {152.f,83.8f}, {164.6f,75.6f}, {156.4f,63.2f}, {156.6f,59.6f}, {156.8f,56.f},
      {158.f,34.4f}, {158.f,34.4f}, {156.f,28.2f}, {153.f,14.6f}, {153.f,14.6f}, {155.2f,9.39998f},
      {162.6f,-3.20002f}, {162.6f,-3.20002f}, {165.4f,-7.40002f}, {174.2f,-12.2f}, {172.f,-15.2f},
      {169.8f,-18.2f}, {162.f,-16.4f}, {162.f,-16.4f}, {154.2f,-17.8f}, {154.8f,-12.6f},
      {154.8f,-12.6f}, {153.2f,-11.6f}, {152.4f,-6.60002f}, {152.4f,-6.60002f}, {151.68f,1.33298f},
      {142.8f,7.59998f}, {142.8f,7.59998f}, {131.6f,13.8f}, {140.8f,17.8f}, {140.8f,17.8f},
      {146.8f,24.4f}, {137.f,24.6f}, {137.f,24.6f}, {126.f,22.8f}, {134.2f,33.f}, {134.2f,33.f},
      {145.f,45.8f}, {142.f,48.6f}, {142.f,48.6f}, {131.8f,49.6f}, {144.4f,58.8f}, {144.4f,58.8f},
      {144.4f,58.8f}, {143.6f,56.8f}, {143.8f,58.6f}, {144.f,60.4f}, {147.f,64.6f}, {147.8f,66.6f},
      {148.6f,68.6f}, {144.6f,68.8f}, {144.6f,68.8f}, {145.2f,78.4f}, {129.8f,74.2f},
      {129.8f,74.2f}, {128.2f,74.4f}, {126.6f,74.6f}, {115.4f,73.8f}, {109.6f,71.6f},
      {103.8f,69.4f}, {97.001f,69.4f}, {97.001f,69.4f}, {97.001f,69.4f}, {93.001f,71.2f},
      {85.4f,71.f}, {77.8f,70.8f}, {69.8f,73.6f}, {69.8f,73.6f}, {65.4f,73.2f}, {74.f,68.8f},
      {74.2f,69.f}, {74.4f,69.2f}, {80.f,63.6f}, {72.f,64.2f}, {50.203f,65.835f}, {39.4f,55.6f},
      {39.4f,55.6f}, {37.4f,54.2f}, {34.8f,51.4f}, {34.8f,51.4f}, {24.8f,49.4f}, {36.2f,63.8f},
      {36.2f,63.8f}, {37.4f,65.2f}, {36.f,66.2f}, {36.f,66.2f}, {35.2f,64.6f}, {27.4f,59.2f},
      {27.4f,59.2f}, {24.589f,58.227f}, {23.226f,56.893f}, {20.895f,54.407f}}},
    {"mcccz",
     {{29.f,128.8f}, {29.f,128.8f}, {28.8f,127.f}, {30.6f,127.8f}, {32.4f,128.6f}, {205.8f,115.6f},
      {229.4f,148.f}, {229.4f,148.f}, {219.8f,122.4f}, {29.f,128.8f}}},
    {"mcccz",
     {{11.526f,104.46f}, {11.526f,104.46f}, {11.082f,106.46f}, {12.631f,105.25f}, {28.699f,92.622f},
      {61.141f,33.72f}, {116.831f,28.086f}, {116.831f,28.086f}, {78.519f,15.976f},
      {11.531f,104.46f}}},
    {"mcccz",
     {{22.726f,102.66f}, {22.726f,102.66f}, {21.363f,101.47f}, {23.231f,100.85f}, {25.099f,100.22f},
      {137.541f,27.72f}, {176.831f,35.686f}, {176.831f,35.686f}, {149.721f,28.176f},
      {22.731f,102.66f}}},
    {"mcccz",
     {{36.855f,98.898f}, {36.855f,98.898f}, {35.654f,97.543f}, {37.586f,97.158f}, {39.518f,96.774f},
      {160.216f,39.061f}, {198.176f,51.927f}, {198.176f,51.927f}, {172.236f,41.053f},
      {36.856f,98.898f}}},
};

int kNumTigerPaths = (int)SK_ARRAY_COUNT(kTigerPaths);
