/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/t/cdf' );
import T = require( './../../../../../base/dists/t/ctor' );
import entropy = require( './../../../../../base/dists/t/entropy' );
import kurtosis = require( './../../../../../base/dists/t/kurtosis' );
import logcdf = require( './../../../../../base/dists/t/logcdf' );
import logpdf = require( './../../../../../base/dists/t/logpdf' );
import mean = require( './../../../../../base/dists/t/mean' );
import median = require( './../../../../../base/dists/t/median' );
import mode = require( './../../../../../base/dists/t/mode' );
import pdf = require( './../../../../../base/dists/t/pdf' );
import quantile = require( './../../../../../base/dists/t/quantile' );
import skewness = require( './../../../../../base/dists/t/skewness' );
import stdev = require( './../../../../../base/dists/t/stdev' );
import variance = require( './../../../../../base/dists/t/variance' );

/**
* Interface describing the `t` namespace.
*/
interface Namespace {
	/**
	* Student's t distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param v - degrees of freedom
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 2.0, 0.1 );
	* // returns ~0.611
	*
	* y = ns.cdf( 1.0, 2.0 );
	* // returns ~0.789
	*
	* y = ns.cdf( -1.0, 4.0 );
	* // returns ~0.187
	*
	* var mycdf = ns.cdf.factory( 0.5 );
	* y = mycdf( 3.0 );
	* // returns ~0.816
	*
	* y = mycdf( 1.0 );
	* // returns ~0.699
	*/
	cdf: typeof cdf;

	/**
	* Student's t distribution.
	*/
	T: typeof T;

	/**
	* Returns the differential entropy of a Student's t distribution.
	*
	* ## Notes
	*
	* -   If provided `v < 0`, the function returns `NaN`.
	*
	* @param v - degrees of freedom
	* @returns entropy
	*
	* @example
	* var v = ns.entropy( 9.0 );
	* // returns ~1.533
	*
	* @example
	* var v = ns.entropy( 2.0 );
	* // returns ~1.96
	*
	* @example
	* var v = ns.entropy( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( NaN );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis of a Student's t distribution.
	*
	* ## Notes
	*
	* -   If provided `v <= 2`, the function returns `NaN`.
	* -   If provided `2 < v <= 4`, the function returns positive infinity.
	*
	* @param v - degrees of freedom
	* @returns excess kurtosis
	*
	* @example
	* var v = ns.kurtosis( 9.0 );
	* // returns 1.2
	*
	* @example
	* var v = ns.kurtosis( 5.0 );
	* // returns 6.0
	*
	* @example
	* var v = ns.kurtosis( 3.0 );
	* // returns Infinity
	*
	* @example
	* var v = ns.kurtosis( 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Student's t distribution logarithm of cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param v - degrees of freedom
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 2.0, 0.1 );
	* // returns ~-0.493
	*
	* y = ns.logcdf( 1.0, 2.0 );
	* // returns ~-0.237
	*
	* y = ns.logcdf( -1.0, 4.0 );
	* // returns ~-1.677
	*
	* var mylogcdf = ns.logcdf.factory( 0.5 );
	* y = mylogcdf( 3.0 );
	* // returns ~-0.203
	*/
	logcdf: typeof logcdf;

	/**
	* Student's t distribution natural logarithm of probability density function (logPDF).
	*
	* @param x - input value
	* @param v - degrees of freedom
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 3.0, 1.0 );
	* // returns ~-3.442
	*
	* var mylogpdf = ns.logpdf.factory( 3.0 );
	* y = mylogpdf( 1.0 );
	* // returns ~-1.575
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value of a Student's t distribution.
	*
	* ## Notes
	*
	* -   If provided `v <= 1`, the function returns `NaN`.
	*
	* @param v - degrees of freedom
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 9.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.mean( 2.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.mean( 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median of a Student's t distribution.
	*
	* ## Notes
	*
	* -   If provided `v < 0`, the function returns `NaN`.
	*
	* @param v - degrees of freedom
	* @returns median
	*
	* @example
	* var v = ns.median( 9.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.median( 2.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.median( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( NaN );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Returns the mode of a Student's t distribution.
	*
	* ## Notes
	*
	* -   If provided `v < 0`, the function returns `NaN`.
	*
	* @param v - degrees of freedom
	* @returns mode
	*
	* @example
	* var v = ns.mode( 9.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.mode( 2.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.mode( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Student's t distribution probability density function (PDF).
	*
	* @param x - input value
	* @param v - degrees of freedom
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 3.0, 1.0 );
	* // returns ~0.032
	*
	* var myPDF = ns.pdf.factory( 3.0 );
	* y = myPDF( 1.0 );
	* // returns ~0.207
	*/
	pdf: typeof pdf;

	/**
	* Student's t distribution quantile function.
	*
	* @param p - input value
	* @param v - degrees of freedom
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 1.0 );
	* // returns ~1.376
	*
	* y = ns.quantile( 0.1, 1.0 );
	* // returns ~-3.078
	*
	* y = ns.quantile( 0.5, 0.1 );
	* // returns 0.0
	*
	* var myquantile = ns.quantile.factory( 4.0 );
	*
	* y = myquantile( 0.2 );
	* // returns ~-0.941
	*
	* y = myquantile( 0.9 );
	* // returns ~1.533
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of a Student's t distribution.
	*
	* ## Notes
	*
	* -   If provided `v <= 3`, the function returns `NaN`.
	*
	* @param v - degrees of freedom
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 9.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.skewness( 4.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.skewness( 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of a Student's t distribution.
	*
	* ## Notes
	*
	* -   If provided `1 < v <= 2`, the function returns positive infinity.
	* -   If provided `v <= 1`, the function returns `NaN`.
	*
	* @param v - degrees of freedom
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 9.0 );
	* // returns ~1.134
	*
	* @example
	* var v = ns.stdev( 2.0 );
	* // returns Infinity
	*
	* @example
	* var v = ns.stdev( 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of a Student's t distribution.
	*
	* ## Notes
	*
	* -   If provided `1 < v <= 2`, the function returns positive infinity.
	* -   If provided `v <= 1`, the function returns `NaN`.
	*
	* @param v - degrees of freedom
	* @returns variance
	*
	* @example
	* var v = ns.variance( 9.0 );
	* // returns ~1.286
	*
	* @example
	* var v = ns.variance( 2.0 );
	* // returns Infinity
	*
	* @example
	* var v = ns.variance( 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Student's t distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
