#ifndef _MODP_HTML_NAMED_ENTITIES_H
#define _MODP_HTML_NAMED_ENTITIES_H
#include <string.h>

typedef struct trienode {
    char value; /* 1 byte -- char value - we deal only in pure ASCII */
    char more; /* 1 byte -- should be read next value in array (boundary marker) */
    short next; /* 2 byte -- position of next leaf */
    int codepoint; /* 4 byte -- codepoint or 0 if none */
} trienode_t;

static const trienode_t entities[];
static int html_named_entity_decode(const char* s, size_t len, size_t* consumed);

static int html_named_entity_decode(const char* s, size_t len, size_t* consumed)
{
    int codepoint = 0;
    size_t pos = 0;
    int i = 0;

    *consumed = 0;

    while (pos < len) {
        if (s[pos] == entities[i].value) {
            if (entities[i].codepoint) {
                codepoint = entities[i].codepoint;
                *consumed = pos + 1;
            }
            if (entities[i].next != 0) {
                i = entities[i].next;
                pos += 1;
                continue;
            } else {
                break;
            }
        }
        if (entities[i].more == 0) {
            break;
        }
        ++i;
    }
    return codepoint;
}
/*
#include <stdio.h>
int main(int argc, char* argv[]) {
    size_t consumed;
    int codepoint;
    codepoint = html_named_entity_decode(argv[1], strlen(argv[1]), &consumed);
    printf("cp = %d, consumed = %d\n", codepoint, consumed);
    return 0;
}
*/
static const trienode_t entities[] = {

    { 83, 1, 52, 0 },
    { 99, 1, 74, 0 },
    { 110, 1, 109, 0 },
    { 115, 1, 144, 0 },
    { 76, 1, 169, 0 },
    { 105, 1, 211, 0 },
    { 113, 1, 232, 0 },
    { 82, 1, 241, 0 },
    { 97, 1, 274, 0 },
    { 98, 1, 295, 0 },
    { 78, 1, 326, 0 },
    { 86, 1, 385, 0 },
    { 108, 1, 409, 0 },
    { 69, 1, 464, 0 },
    { 118, 1, 499, 0 },
    { 84, 1, 529, 0 },
    { 104, 1, 569, 0 },
    { 114, 1, 604, 0 },
    { 120, 1, 638, 0 },
    { 101, 1, 664, 0 },
    { 100, 1, 684, 0 },
    { 71, 1, 708, 0 },
    { 116, 1, 740, 0 },
    { 109, 1, 779, 0 },
    { 111, 1, 807, 0 },
    { 103, 1, 854, 0 },
    { 79, 1, 876, 0 },
    { 77, 1, 910, 0 },
    { 72, 1, 940, 0 },
    { 112, 1, 998, 0 },
    { 67, 1, 1024, 0 },
    { 102, 1, 1040, 0 },
    { 122, 1, 1066, 0 },
    { 73, 1, 1132, 0 },
    { 68, 1, 1241, 0 },
    { 85, 1, 1417, 0 },
    { 90, 1, 1435, 0 },
    { 66, 1, 1470, 0 },
    { 81, 1, 1555, 0 },
    { 70, 1, 1680, 0 },
    { 107, 1, 1731, 0 },
    { 65, 1, 1746, 0 },
    { 117, 1, 1863, 0 },
    { 89, 1, 1909, 0 },
    { 80, 1, 2161, 0 },
    { 121, 1, 2453, 0 },
    { 75, 1, 2482, 0 },
    { 74, 1, 2648, 0 },
    { 119, 1, 3035, 0 },
    { 88, 1, 3162, 0 },
    { 87, 1, 4502, 0 },
    { 106, 0, 4774, 0 },
    { 113, 1, 65, 0 },
    { 117, 1, 550, 0 },
    { 116, 1, 2498, 0 },
    { 99, 1, 4340, 0 },
    { 79, 1, 5732, 0 },
    { 104, 1, 5834, 0 },
    { 97, 1, 7003, 0 },
    { 109, 1, 7428, 0 },
    { 115, 1, 7590, 0 },
    { 72, 1, 7618, 0 },
    { 105, 1, 8727, 0 },
    { 111, 1, 9572, 0 },
    { 102, 0, 9740, 0 },
    { 117, 1, 67, 0 },
    { 114, 0, 6264, 0 },
    { 97, 0, 68, 0 },
    { 114, 0, 69, 0 },
    { 101, 0, 70, 0 },
    { 59, 1, 0, 9633 }, /* &Square; */
    { 83, 1, 2438, 0 },
    { 85, 1, 5026, 0 },
    { 73, 0, 8698, 0 },
    { 105, 1, 89, 0 },
    { 111, 1, 317, 0 },
    { 104, 1, 657, 0 },
    { 97, 1, 964, 0 },
    { 117, 1, 1505, 0 },
    { 108, 1, 2957, 0 },
    { 101, 1, 3529, 0 },
    { 115, 1, 3593, 0 },
    { 99, 1, 4388, 0 },
    { 116, 1, 5395, 0 },
    { 102, 1, 5940, 0 },
    { 119, 1, 7231, 0 },
    { 100, 1, 7481, 0 },
    { 114, 1, 8309, 0 },
    { 121, 0, 8870, 0 },
    { 114, 0, 90, 0 },
    { 99, 1, 97, 0 },
    { 115, 1, 2221, 0 },
    { 102, 1, 3724, 0 },
    { 101, 1, 3729, 0 },
    { 109, 1, 3785, 0 },
    { 69, 1, 6234, 0 },
    { 59, 0, 0, 9675 }, /* &cir; */
    { 108, 1, 100, 0 },
    { 101, 1, 6995, 0 },
    { 59, 0, 0, 710 }, /* &circ; */
    { 101, 0, 101, 0 },
    { 100, 1, 103, 0 },
    { 97, 0, 6363, 0 },
    { 83, 1, 108, 0 },
    { 99, 1, 4579, 0 },
    { 100, 1, 6582, 0 },
    { 97, 1, 7950, 0 },
    { 82, 0, 9467, 0 },
    { 59, 0, 0, 9416 }, /* &circledS; */
    { 118, 1, 133, 0 },
    { 103, 1, 1099, 0 },
    { 104, 1, 1106, 0 },
    { 112, 1, 1112, 0 },
    { 116, 1, 1215, 0 },
    { 119, 1, 1234, 0 },
    { 71, 1, 1365, 0 },
    { 114, 1, 1586, 0 },
    { 97, 1, 1902, 0 },
    { 82, 1, 2132, 0 },
    { 100, 1, 2492, 0 },
    { 105, 1, 2558, 0 },
    { 101, 1, 2593, 0 },
    { 115, 1, 2634, 0 },
    { 106, 1, 3407, 0 },
    { 86, 1, 3624, 0 },
    { 111, 1, 3714, 0 },
    { 99, 1, 3853, 0 },
    { 98, 1, 4087, 0 },
    { 108, 1, 4165, 0 },
    { 117, 1, 5412, 0 },
    { 76, 1, 5485, 0 },
    { 102, 1, 8642, 0 },
    { 109, 0, 9337, 0 },
    { 97, 1, 142, 0 },
    { 103, 1, 526, 0 },
    { 108, 1, 2906, 0 },
    { 105, 1, 3075, 0 },
    { 72, 1, 3604, 0 },
    { 68, 1, 3708, 0 },
    { 114, 1, 4870, 0 },
    { 100, 1, 6552, 0 },
    { 115, 0, 9846, 0 },
    { 112, 0, 143, 0 },
    { 59, 0, 0, 8781 }, /* &nvap; */
    { 117, 1, 163, 0 },
    { 99, 1, 371, 0 },
    { 100, 1, 828, 0 },
    { 101, 1, 898, 0 },
    { 104, 1, 1258, 0 },
    { 98, 1, 1353, 0 },
    { 105, 1, 1836, 0 },
    { 109, 1, 1998, 0 },
    { 113, 1, 2826, 0 },
    { 115, 1, 2993, 0 },
    { 119, 1, 3128, 0 },
    { 116, 1, 3211, 0 },
    { 111, 1, 3569, 0 },
    { 122, 1, 3720, 0 },
    { 112, 1, 3946, 0 },
    { 102, 1, 5511, 0 },
    { 108, 1, 5550, 0 },
    { 114, 1, 8020, 0 },
    { 97, 0, 9794, 0 },
    { 109, 1, 168, 0 },
    { 98, 1, 485, 0 },
    { 99, 1, 1928, 0 },
    { 112, 1, 2070, 0 },
    { 110, 0, 6934, 0 },
    { 59, 0, 0, 8721 }, /* &sum; */
    { 101, 1, 180, 0 },
    { 102, 1, 497, 0 },
    { 97, 1, 1122, 0 },
    { 111, 1, 1302, 0 },
    { 108, 1, 1340, 0 },
    { 84, 1, 1409, 60 }, /* &LT */
    { 115, 1, 1921, 0 },
    { 99, 1, 2349, 0 },
    { 74, 1, 7135, 0 },
    { 116, 1, 7609, 0 },
    { 109, 0, 9404, 0 },
    { 102, 1, 182, 0 },
    { 115, 0, 3101, 0 },
    { 116, 0, 183, 0 },
    { 65, 1, 193, 0 },
    { 85, 1, 766, 0 },
    { 68, 1, 1192, 0 },
    { 84, 1, 2243, 0 },
    { 114, 1, 3583, 0 },
    { 97, 1, 3818, 0 },
    { 86, 1, 4136, 0 },
    { 70, 1, 6259, 0 },
    { 82, 1, 6278, 0 },
    { 67, 0, 6525, 0 },
    { 114, 1, 195, 0 },
    { 110, 0, 7673, 0 },
    { 114, 0, 196, 0 },
    { 111, 0, 197, 0 },
    { 119, 0, 198, 0 },
    { 82, 1, 201, 0 },
    { 59, 1, 0, 8592 }, /* &LeftArrow; */
    { 66, 0, 7793, 0 },
    { 105, 0, 202, 0 },
    { 103, 0, 203, 0 },
    { 104, 0, 204, 0 },
    { 116, 0, 205, 0 },
    { 65, 0, 206, 0 },
    { 114, 0, 207, 0 },
    { 114, 0, 208, 0 },
    { 111, 0, 209, 0 },
    { 119, 0, 210, 0 },
    { 59, 0, 0, 8646 }, /* &LeftArrowRightArrow; */
    { 111, 1, 226, 0 },
    { 101, 1, 953, 0 },
    { 115, 1, 1151, 0 },
    { 105, 1, 1168, 0 },
    { 99, 1, 1405, 0 },
    { 103, 1, 3018, 0 },
    { 109, 1, 3661, 0 },
    { 110, 1, 3730, 0 },
    { 113, 1, 3808, 0 },
    { 117, 1, 5383, 0 },
    { 97, 1, 6423, 0 },
    { 106, 1, 6715, 0 },
    { 102, 1, 6760, 0 },
    { 116, 1, 9367, 0 },
    { 112, 0, 9736, 0 },
    { 116, 1, 230, 0 },
    { 99, 1, 2514, 0 },
    { 112, 1, 5060, 0 },
    { 103, 0, 6508, 0 },
    { 97, 0, 231, 0 },
    { 59, 0, 0, 953 }, /* &iota; */
    { 111, 1, 238, 0 },
    { 117, 1, 4457, 0 },
    { 105, 1, 6235, 0 },
    { 102, 1, 7449, 0 },
    { 112, 1, 7872, 0 },
    { 115, 0, 9312, 0 },
    { 112, 0, 239, 0 },
    { 102, 0, 240, 0 },
    { 59, 0, 0, 120162 }, /* &qopf; */
    { 105, 1, 253, 0 },
    { 66, 1, 3953, 0 },
    { 101, 1, 4233, 0 },
    { 97, 1, 4451, 0 },
    { 99, 1, 6097, 0 },
    { 111, 1, 6980, 0 },
    { 115, 1, 7894, 0 },
    { 102, 1, 8079, 0 },
    { 104, 1, 8330, 0 },
    { 69, 1, 8938, 0 },
    { 117, 1, 9720, 0 },
    { 114, 0, 9832, 0 },
    { 103, 0, 254, 0 },
    { 104, 0, 255, 0 },
    { 116, 0, 256, 0 },
    { 86, 1, 264, 0 },
    { 84, 1, 3181, 0 },
    { 85, 1, 3274, 0 },
    { 68, 1, 3354, 0 },
    { 70, 1, 3635, 0 },
    { 65, 1, 3922, 0 },
    { 67, 1, 6892, 0 },
    { 97, 0, 7185, 0 },
    { 101, 0, 265, 0 },
    { 99, 0, 266, 0 },
    { 116, 0, 267, 0 },
    { 111, 0, 268, 0 },
    { 114, 0, 269, 0 },
    { 66, 1, 271, 0 },
    { 59, 0, 0, 8640 }, /* &RightVector; */
    { 97, 0, 272, 0 },
    { 114, 0, 273, 0 },
    { 59, 0, 0, 10579 }, /* &RightVectorBar; */
    { 116, 1, 290, 0 },
    { 114, 1, 1189, 0 },
    { 110, 1, 1480, 0 },
    { 117, 1, 2063, 0 },
    { 103, 1, 2209, 0 },
    { 99, 1, 2882, 0 },
    { 109, 1, 3812, 0 },
    { 112, 1, 3833, 0 },
    { 115, 1, 4632, 0 },
    { 101, 1, 4831, 0 },
    { 97, 1, 4834, 0 },
    { 102, 1, 4886, 0 },
    { 98, 1, 5498, 0 },
    { 108, 1, 5629, 0 },
    { 111, 1, 5916, 0 },
    { 119, 0, 5962, 0 },
    { 105, 0, 291, 0 },
    { 108, 0, 292, 0 },
    { 100, 0, 293, 0 },
    { 101, 0, 294, 227 }, /* &atilde */
    { 59, 0, 0, 227 }, /* &atilde; */
    { 108, 1, 311, 0 },
    { 111, 1, 440, 0 },
    { 97, 1, 519, 0 },
    { 105, 1, 587, 0 },
    { 115, 1, 1279, 0 },
    { 100, 1, 1773, 0 },
    { 101, 1, 1785, 0 },
    { 117, 1, 2365, 0 },
    { 114, 1, 3080, 0 },
    { 99, 1, 3375, 0 },
    { 110, 1, 4443, 0 },
    { 98, 1, 4793, 0 },
    { 102, 1, 7971, 0 },
    { 78, 1, 8898, 0 },
    { 107, 1, 9218, 0 },
    { 112, 0, 9819, 0 },
    { 111, 1, 314, 0 },
    { 97, 1, 3511, 0 },
    { 107, 0, 3848, 0 },
    { 99, 0, 315, 0 },
    { 107, 0, 316, 0 },
    { 59, 0, 0, 9608 }, /* &block; */
    { 109, 1, 321, 0 },
    { 110, 1, 1335, 0 },
    { 112, 1, 2673, 0 },
    { 108, 0, 4693, 0 },
    { 112, 1, 323, 0 },
    { 109, 0, 5182, 0 },
    { 59, 1, 0, 8705 }, /* &comp; */
    { 108, 1, 3551, 0 },
    { 102, 0, 6600, 0 },
    { 111, 1, 335, 0 },
    { 101, 1, 2260, 0 },
    { 97, 1, 2344, 0 },
    { 102, 1, 3230, 0 },
    { 99, 1, 3544, 0 },
    { 116, 1, 5249, 0 },
    { 117, 1, 6024, 0 },
    { 115, 1, 8950, 0 },
    { 74, 0, 9677, 0 },
    { 116, 1, 339, 0 },
    { 66, 1, 2065, 0 },
    { 112, 1, 6748, 0 },
    { 110, 0, 8826, 0 },
    { 83, 1, 352, 0 },
    { 69, 1, 1286, 0 },
    { 67, 1, 1410, 0 },
    { 86, 1, 1815, 0 },
    { 76, 1, 1948, 0 },
    { 80, 1, 2519, 0 },
    { 78, 1, 3978, 0 },
    { 71, 1, 4287, 0 },
    { 82, 1, 4304, 0 },
    { 59, 1, 0, 10988 }, /* &Not; */
    { 84, 1, 5770, 0 },
    { 72, 1, 7512, 0 },
    { 68, 0, 8759, 0 },
    { 117, 1, 354, 0 },
    { 113, 0, 2050, 0 },
    { 99, 1, 357, 0 },
    { 112, 1, 3420, 0 },
    { 98, 0, 4527, 0 },
    { 99, 0, 358, 0 },
    { 101, 0, 359, 0 },
    { 101, 0, 360, 0 },
    { 100, 0, 361, 0 },
    { 115, 0, 362, 0 },
    { 69, 1, 366, 0 },
    { 59, 1, 0, 8833 }, /* &NotSucceeds; */
    { 84, 1, 3800, 0 },
    { 83, 0, 5714, 0 },
    { 113, 0, 367, 0 },
    { 117, 0, 368, 0 },
    { 97, 0, 369, 0 },
    { 108, 0, 370, 0 },
    { 59, 0, 0, 10928 }, /* &NotSucceedsEqual; */
    { 110, 1, 381, 0 },
    { 105, 1, 2858, 0 },
    { 101, 1, 4100, 0 },
    { 69, 1, 5129, 0 },
    { 112, 1, 6476, 0 },
    { 121, 1, 6516, 0 },
    { 115, 1, 6573, 0 },
    { 59, 1, 0, 8827 }, /* &sc; */
    { 97, 1, 7644, 0 },
    { 99, 0, 9048, 0 },
    { 69, 1, 384, 0 },
    { 97, 1, 2125, 0 },
    { 115, 0, 9206, 0 },
    { 59, 0, 0, 10934 }, /* &scnE; */
    { 101, 1, 394, 0 },
    { 68, 1, 1970, 0 },
    { 115, 1, 3830, 0 },
    { 98, 1, 3879, 0 },
    { 100, 1, 5597, 0 },
    { 111, 1, 5654, 0 },
    { 118, 1, 7829, 0 },
    { 99, 1, 8229, 0 },
    { 102, 0, 9613, 0 },
    { 114, 1, 396, 0 },
    { 101, 0, 6075, 0 },
    { 121, 1, 399, 0 },
    { 116, 1, 3053, 0 },
    { 98, 0, 5304, 0 },
    { 84, 0, 400, 0 },
    { 104, 0, 401, 0 },
    { 105, 0, 402, 0 },
    { 110, 0, 403, 0 },
    { 83, 0, 404, 0 },
    { 112, 0, 405, 0 },
    { 97, 0, 406, 0 },
    { 99, 0, 407, 0 },
    { 101, 0, 408, 0 },
    { 59, 0, 0, 8202 }, /* &VeryThinSpace; */
    { 108, 1, 432, 0 },
    { 97, 1, 834, 0 },
    { 101, 1, 919, 0 },
    { 115, 1, 977, 0 },
    { 111, 1, 1600, 0 },
    { 102, 1, 1742, 0 },
    { 100, 1, 2086, 0 },
    { 98, 1, 2313, 0 },
    { 116, 1, 2618, 60 }, /* &lt */
    { 66, 1, 2678, 0 },
    { 104, 1, 3261, 0 },
    { 110, 1, 3446, 0 },
    { 112, 1, 3781, 0 },
    { 109, 1, 4037, 0 },
    { 65, 1, 4143, 0 },
    { 117, 1, 4178, 0 },
    { 99, 1, 4395, 0 },
    { 114, 1, 4900, 0 },
    { 118, 1, 5125, 0 },
    { 69, 1, 7126, 0 },
    { 103, 1, 8410, 0 },
    { 106, 1, 9246, 0 },
    { 72, 0, 9382, 0 },
    { 97, 1, 437, 0 },
    { 116, 1, 1357, 0 },
    { 104, 1, 5665, 0 },
    { 59, 1, 0, 8810 }, /* &ll; */
    { 99, 0, 8297, 0 },
    { 114, 0, 438, 0 },
    { 114, 0, 439, 0 },
    { 59, 0, 0, 8647 }, /* &llarr; */
    { 120, 1, 444, 0 },
    { 119, 1, 6723, 0 },
    { 116, 1, 6851, 0 },
    { 112, 0, 8718, 0 },
    { 118, 1, 456, 0 },
    { 86, 1, 1086, 0 },
    { 72, 1, 1974, 0 },
    { 109, 1, 2891, 0 },
    { 117, 1, 3001, 0 },
    { 100, 1, 3011, 0 },
    { 104, 1, 3156, 0 },
    { 85, 1, 3674, 0 },
    { 68, 1, 3770, 0 },
    { 116, 1, 3935, 0 },
    { 98, 1, 7226, 0 },
    { 112, 0, 7708, 0 },
    { 82, 1, 463, 0 },
    { 59, 1, 0, 9474 }, /* &boxv; */
    { 114, 1, 4365, 0 },
    { 72, 1, 6069, 0 },
    { 104, 1, 8009, 0 },
    { 76, 1, 8121, 0 },
    { 108, 0, 9771, 0 },
    { 59, 0, 0, 9566 }, /* &boxvR; */
    { 103, 1, 480, 0 },
    { 115, 1, 988, 0 },
    { 113, 1, 992, 0 },
    { 102, 1, 1992, 0 },
    { 116, 1, 2790, 0 },
    { 99, 1, 2869, 0 },
    { 97, 1, 3301, 0 },
    { 84, 1, 3945, 0 },
    { 117, 1, 4163, 0 },
    { 111, 1, 5374, 0 },
    { 112, 1, 6989, 0 },
    { 109, 1, 7855, 0 },
    { 100, 1, 7968, 0 },
    { 120, 1, 8631, 0 },
    { 78, 1, 8736, 0 },
    { 108, 0, 9555, 0 },
    { 114, 0, 481, 0 },
    { 97, 0, 482, 0 },
    { 118, 0, 483, 0 },
    { 101, 0, 484, 200 }, /* &Egrave */
    { 59, 0, 0, 200 }, /* &Egrave; */
    { 100, 1, 494, 0 },
    { 109, 1, 1162, 0 },
    { 110, 1, 1320, 0 },
    { 115, 1, 3250, 0 },
    { 69, 1, 3339, 0 },
    { 114, 1, 4207, 0 },
    { 101, 1, 5859, 0 },
    { 59, 1, 0, 8834 }, /* &sub; */
    { 112, 0, 9278, 0 },
    { 111, 0, 495, 0 },
    { 116, 0, 496, 0 },
    { 59, 0, 0, 10941 }, /* &subdot; */
    { 114, 0, 498, 0 },
    { 59, 0, 0, 120079 }, /* &Lfr; */
    { 66, 1, 514, 0 },
    { 101, 1, 1616, 0 },
    { 97, 1, 1649, 0 },
    { 115, 1, 1777, 0 },
    { 99, 1, 2451, 0 },
    { 112, 1, 2850, 0 },
    { 68, 1, 3115, 0 },
    { 65, 1, 4124, 0 },
    { 110, 1, 5299, 0 },
    { 122, 1, 5607, 0 },
    { 111, 1, 8100, 0 },
    { 102, 1, 8429, 0 },
    { 100, 1, 8480, 0 },
    { 114, 1, 8963, 0 },
    { 108, 0, 9742, 0 },
    { 97, 0, 515, 0 },
    { 114, 0, 516, 0 },
    { 118, 1, 518, 0 },
    { 59, 0, 0, 10984 }, /* &vBar; */
    { 59, 0, 0, 10985 }, /* &vBarv; */
    { 114, 1, 521, 0 },
    { 99, 0, 3432, 0 },
    { 118, 1, 523, 0 },
    { 119, 0, 5399, 0 },
    { 101, 0, 524, 0 },
    { 101, 0, 525, 0 },
    { 59, 0, 0, 8893 }, /* &barvee; */
    { 116, 1, 528, 0 },
    { 101, 0, 8678, 0 },
    { 59, 0, 0, 62 }, /* &nvgt; */
    { 104, 1, 540, 0 },
    { 99, 1, 957, 0 },
    { 105, 1, 2145, 0 },
    { 82, 1, 2630, 0 },
    { 102, 1, 3643, 0 },
    { 72, 1, 4127, 0 },
    { 97, 1, 4738, 0 },
    { 115, 1, 5853, 0 },
    { 83, 1, 7378, 0 },
    { 114, 1, 8455, 0 },
    { 111, 0, 9657, 0 },
    { 101, 1, 542, 0 },
    { 105, 0, 9099, 0 },
    { 114, 1, 544, 0 },
    { 116, 0, 5967, 0 },
    { 101, 0, 545, 0 },
    { 102, 0, 546, 0 },
    { 111, 0, 547, 0 },
    { 114, 0, 548, 0 },
    { 101, 0, 549, 0 },
    { 59, 0, 0, 8756 }, /* &Therefore; */
    { 99, 1, 554, 0 },
    { 112, 1, 601, 0 },
    { 98, 1, 4257, 0 },
    { 109, 0, 6119, 0 },
    { 99, 1, 556, 0 },
    { 104, 0, 9327, 0 },
    { 101, 0, 557, 0 },
    { 101, 0, 558, 0 },
    { 100, 0, 559, 0 },
    { 115, 0, 560, 0 },
    { 84, 1, 564, 0 },
    { 59, 1, 0, 8827 }, /* &Succeeds; */
    { 83, 1, 2896, 0 },
    { 69, 0, 3524, 0 },
    { 105, 0, 565, 0 },
    { 108, 0, 566, 0 },
    { 100, 0, 567, 0 },
    { 101, 0, 568, 0 },
    { 59, 0, 0, 8831 }, /* &SucceedsTilde; */
    { 101, 1, 579, 0 },
    { 102, 1, 896, 0 },
    { 115, 1, 1094, 0 },
    { 98, 1, 1294, 0 },
    { 107, 1, 2795, 0 },
    { 97, 1, 2873, 0 },
    { 111, 1, 3745, 0 },
    { 121, 1, 6061, 0 },
    { 65, 1, 6116, 0 },
    { 99, 0, 7937, 0 },
    { 97, 1, 582, 0 },
    { 108, 1, 2193, 0 },
    { 114, 0, 6643, 0 },
    { 114, 0, 583, 0 },
    { 116, 0, 584, 0 },
    { 115, 0, 585, 0 },
    { 59, 1, 0, 9829 }, /* &hearts; */
    { 117, 0, 5997, 0 },
    { 103, 0, 588, 0 },
    { 111, 1, 595, 0 },
    { 115, 1, 2501, 0 },
    { 116, 1, 4763, 0 },
    { 118, 1, 6078, 0 },
    { 99, 1, 6266, 0 },
    { 119, 1, 7820, 0 },
    { 117, 0, 8303, 0 },
    { 100, 1, 598, 0 },
    { 116, 1, 4148, 0 },
    { 112, 0, 7487, 0 },
    { 111, 0, 599, 0 },
    { 116, 0, 600, 0 },
    { 59, 0, 0, 10752 }, /* &bigodot; */
    { 59, 1, 0, 8913 }, /* &Sup; */
    { 101, 1, 4484, 0 },
    { 115, 0, 8788, 0 },
    { 97, 1, 625, 0 },
    { 101, 1, 1369, 0 },
    { 116, 1, 1463, 0 },
    { 109, 1, 1565, 0 },
    { 111, 1, 1828, 0 },
    { 105, 1, 2005, 0 },
    { 104, 1, 2585, 0 },
    { 112, 1, 2608, 0 },
    { 98, 1, 2754, 0 },
    { 117, 1, 2844, 0 },
    { 108, 1, 3540, 0 },
    { 65, 1, 3616, 0 },
    { 115, 1, 3679, 0 },
    { 114, 1, 3940, 0 },
    { 102, 1, 5165, 0 },
    { 100, 1, 5257, 0 },
    { 99, 1, 5386, 0 },
    { 66, 1, 7122, 0 },
    { 120, 1, 7376, 0 },
    { 72, 1, 8562, 0 },
    { 110, 0, 8851, 0 },
    { 101, 1, 632, 0 },
    { 114, 1, 2290, 0 },
    { 116, 1, 2772, 0 },
    { 113, 1, 5409, 0 },
    { 110, 1, 5702, 0 },
    { 100, 1, 8616, 0 },
    { 99, 0, 9171, 0 },
    { 109, 0, 633, 0 },
    { 112, 0, 634, 0 },
    { 116, 0, 635, 0 },
    { 121, 0, 636, 0 },
    { 118, 0, 637, 0 },
    { 59, 0, 0, 10675 }, /* &raemptyv; */
    { 108, 1, 652, 0 },
    { 104, 1, 2767, 0 },
    { 110, 1, 3140, 0 },
    { 117, 1, 4807, 0 },
    { 99, 1, 5083, 0 },
    { 114, 1, 5709, 0 },
    { 115, 1, 5738, 0 },
    { 109, 1, 6290, 0 },
    { 111, 1, 6556, 0 },
    { 118, 1, 6977, 0 },
    { 105, 1, 7625, 0 },
    { 119, 1, 8015, 0 },
    { 100, 1, 9505, 0 },
    { 102, 0, 9844, 0 },
    { 65, 1, 654, 0 },
    { 97, 0, 1502, 0 },
    { 114, 0, 655, 0 },
    { 114, 0, 656, 0 },
    { 59, 0, 0, 10232 }, /* &xlArr; */
    { 101, 1, 660, 0 },
    { 105, 1, 4820, 0 },
    { 99, 0, 9611, 0 },
    { 99, 0, 661, 0 },
    { 107, 0, 662, 0 },
    { 59, 1, 0, 10003 }, /* &check; */
    { 109, 0, 1383, 0 },
    { 116, 1, 682, 0 },
    { 110, 1, 1082, 0 },
    { 113, 1, 1722, 0 },
    { 117, 1, 1893, 0 },
    { 114, 1, 3269, 0 },
    { 112, 1, 3685, 0 },
    { 103, 1, 3759, 0 },
    { 99, 1, 4069, 0 },
    { 115, 1, 4119, 0 },
    { 68, 1, 4383, 0 },
    { 108, 1, 4989, 0 },
    { 97, 1, 5189, 0 },
    { 109, 1, 5415, 0 },
    { 120, 1, 6303, 0 },
    { 111, 1, 8384, 0 },
    { 101, 1, 8647, 0 },
    { 102, 1, 8865, 0 },
    { 100, 0, 9364, 0 },
    { 104, 1, 7898, 240 }, /* &eth */
    { 97, 0, 5273, 0 },
    { 104, 1, 703, 0 },
    { 115, 1, 1325, 0 },
    { 116, 1, 1330, 0 },
    { 102, 1, 1712, 0 },
    { 98, 1, 1715, 0 },
    { 122, 1, 2097, 0 },
    { 111, 1, 2375, 0 },
    { 105, 1, 3187, 0 },
    { 101, 1, 3306, 0 },
    { 114, 1, 4464, 0 },
    { 97, 1, 5678, 0 },
    { 106, 1, 5696, 0 },
    { 99, 1, 5724, 0 },
    { 100, 1, 5870, 0 },
    { 117, 1, 6353, 0 },
    { 65, 1, 6823, 0 },
    { 108, 1, 8474, 0 },
    { 72, 1, 8953, 0 },
    { 119, 0, 9813, 0 },
    { 97, 0, 704, 0 },
    { 114, 0, 705, 0 },
    { 114, 1, 707, 0 },
    { 108, 0, 9296, 0 },
    { 59, 0, 0, 8642 }, /* &dharr; */
    { 114, 1, 720, 0 },
    { 100, 1, 2201, 0 },
    { 84, 1, 2740, 62 }, /* &GT */
    { 99, 1, 3368, 0 },
    { 98, 1, 4418, 0 },
    { 103, 1, 4737, 0 },
    { 115, 1, 5380, 0 },
    { 116, 1, 5541, 0 },
    { 74, 1, 5693, 0 },
    { 97, 1, 6244, 0 },
    { 102, 1, 8232, 0 },
    { 111, 0, 9301, 0 },
    { 101, 0, 721, 0 },
    { 97, 0, 722, 0 },
    { 116, 0, 723, 0 },
    { 101, 0, 724, 0 },
    { 114, 0, 725, 0 },
    { 70, 1, 731, 0 },
    { 83, 1, 3143, 0 },
    { 76, 1, 3630, 0 },
    { 71, 1, 5172, 0 },
    { 69, 1, 5978, 0 },
    { 84, 0, 7111, 0 },
    { 117, 0, 732, 0 },
    { 108, 0, 733, 0 },
    { 108, 0, 734, 0 },
    { 69, 0, 735, 0 },
    { 113, 0, 736, 0 },
    { 117, 0, 737, 0 },
    { 97, 0, 738, 0 },
    { 108, 0, 739, 0 },
    { 59, 0, 0, 8807 }, /* &GreaterFullEqual; */
    { 104, 1, 753, 0 },
    { 115, 1, 1183, 0 },
    { 102, 1, 1323, 0 },
    { 101, 1, 1360, 0 },
    { 99, 1, 1446, 0 },
    { 114, 1, 1528, 0 },
    { 111, 1, 2204, 0 },
    { 119, 1, 4348, 0 },
    { 105, 1, 5077, 0 },
    { 112, 1, 5119, 0 },
    { 97, 1, 5657, 0 },
    { 100, 1, 7657, 0 },
    { 98, 0, 9293, 0 },
    { 101, 1, 757, 0 },
    { 105, 1, 2782, 0 },
    { 111, 1, 3441, 0 },
    { 107, 0, 4842, 0 },
    { 116, 1, 759, 0 },
    { 114, 0, 7087, 0 },
    { 97, 0, 760, 0 },
    { 115, 1, 763, 0 },
    { 118, 1, 7209, 0 },
    { 59, 0, 0, 952 }, /* &theta; */
    { 121, 0, 764, 0 },
    { 109, 0, 765, 0 },
    { 59, 0, 0, 977 }, /* &thetasym; */
    { 112, 0, 767, 0 },
    { 84, 1, 770, 0 },
    { 86, 1, 4646, 0 },
    { 68, 0, 8346, 0 },
    { 101, 0, 771, 0 },
    { 101, 0, 772, 0 },
    { 86, 0, 773, 0 },
    { 101, 0, 774, 0 },
    { 99, 0, 775, 0 },
    { 116, 0, 776, 0 },
    { 111, 0, 777, 0 },
    { 114, 0, 778, 0 },
    { 59, 0, 0, 10592 }, /* &LeftUpTeeVector; */
    { 97, 1, 793, 0 },
    { 117, 1, 1453, 0 },
    { 115, 1, 1561, 0 },
    { 105, 1, 2706, 0 },
    { 108, 1, 3297, 0 },
    { 111, 1, 3350, 0 },
    { 112, 1, 5748, 0 },
    { 68, 1, 6274, 0 },
    { 101, 1, 6839, 0 },
    { 100, 1, 6930, 0 },
    { 99, 1, 8652, 0 },
    { 104, 1, 8814, 0 },
    { 110, 1, 9061, 0 },
    { 102, 0, 9352, 0 },
    { 112, 1, 797, 0 },
    { 114, 1, 3875, 0 },
    { 108, 1, 4965, 0 },
    { 99, 0, 5319, 0 },
    { 115, 1, 799, 0 },
    { 59, 0, 0, 8614 }, /* &map; */
    { 116, 0, 800, 0 },
    { 111, 0, 801, 0 },
    { 117, 1, 805, 0 },
    { 100, 1, 3136, 0 },
    { 59, 1, 0, 8614 }, /* &mapsto; */
    { 108, 0, 8060, 0 },
    { 112, 0, 806, 0 },
    { 59, 0, 0, 8613 }, /* &mapstoup; */
    { 99, 1, 825, 0 },
    { 109, 1, 1576, 0 },
    { 100, 1, 1670, 0 },
    { 111, 1, 1967, 0 },
    { 83, 1, 2312, 0 },
    { 118, 1, 2322, 0 },
    { 103, 1, 2465, 0 },
    { 114, 1, 2941, 0 },
    { 102, 1, 3086, 0 },
    { 117, 1, 3178, 0 },
    { 116, 1, 4238, 0 },
    { 108, 1, 4331, 0 },
    { 115, 1, 5091, 0 },
    { 104, 1, 5635, 0 },
    { 97, 1, 6801, 0 },
    { 112, 1, 7585, 0 },
    { 101, 1, 8686, 0 },
    { 105, 0, 9009, 0 },
    { 121, 1, 827, 0 },
    { 105, 0, 4922, 0 },
    { 59, 0, 0, 1086 }, /* &ocy; */
    { 111, 0, 829, 0 },
    { 116, 0, 830, 0 },
    { 98, 1, 833, 0 },
    { 59, 1, 0, 8901 }, /* &sdot; */
    { 101, 0, 9477, 0 },
    { 59, 0, 0, 8865 }, /* &sdotb; */
    { 114, 1, 843, 0 },
    { 110, 1, 2356, 0 },
    { 116, 1, 4055, 0 },
    { 101, 1, 4934, 0 },
    { 113, 1, 5931, 0 },
    { 109, 1, 6569, 0 },
    { 99, 1, 7801, 0 },
    { 103, 1, 7881, 0 },
    { 112, 0, 9616, 0 },
    { 114, 0, 844, 0 },
    { 116, 1, 852, 0 },
    { 112, 1, 1166, 0 },
    { 115, 1, 2814, 0 },
    { 59, 1, 0, 8592 }, /* &larr; */
    { 98, 1, 6517, 0 },
    { 104, 1, 7193, 0 },
    { 102, 1, 8107, 0 },
    { 108, 0, 9465, 0 },
    { 108, 0, 853, 0 },
    { 59, 0, 0, 8610 }, /* &larrtl; */
    { 110, 1, 871, 0 },
    { 103, 1, 1129, 0 },
    { 116, 1, 1387, 62 }, /* &gt */
    { 115, 1, 1706, 0 },
    { 97, 1, 3232, 0 },
    { 101, 1, 3410, 0 },
    { 105, 1, 3534, 0 },
    { 106, 1, 4076, 0 },
    { 111, 1, 4524, 0 },
    { 114, 1, 5422, 0 },
    { 98, 1, 5660, 0 },
    { 102, 1, 6339, 0 },
    { 118, 1, 7141, 0 },
    { 69, 1, 7148, 0 },
    { 99, 1, 8125, 0 },
    { 100, 1, 8158, 0 },
    { 108, 0, 9043, 0 },
    { 69, 1, 875, 0 },
    { 97, 1, 3962, 0 },
    { 101, 1, 5195, 0 },
    { 115, 0, 7248, 0 },
    { 59, 0, 0, 8809 }, /* &gnE; */
    { 109, 1, 890, 0 },
    { 117, 1, 2288, 0 },
    { 118, 1, 2656, 0 },
    { 100, 1, 2695, 0 },
    { 103, 1, 2760, 0 },
    { 97, 1, 2777, 0 },
    { 115, 1, 3239, 0 },
    { 99, 1, 3601, 0 },
    { 112, 1, 5210, 0 },
    { 114, 1, 5533, 0 },
    { 116, 1, 6341, 0 },
    { 69, 1, 7626, 0 },
    { 102, 1, 8134, 0 },
    { 111, 0, 8895, 0 },
    { 101, 1, 893, 0 },
    { 97, 1, 6169, 0 },
    { 105, 0, 6382, 0 },
    { 103, 0, 894, 0 },
    { 97, 0, 895, 0 },
    { 59, 0, 0, 937 }, /* &Omega; */
    { 114, 0, 897, 0 },
    { 59, 0, 0, 120101 }, /* &hfr; */
    { 97, 1, 905, 0 },
    { 65, 1, 1380, 0 },
    { 116, 1, 2253, 0 },
    { 115, 1, 2399, 0 },
    { 99, 1, 3539, 0 },
    { 120, 1, 7162, 0 },
    { 109, 0, 8295, 0 },
    { 114, 0, 906, 0 },
    { 104, 1, 908, 0 },
    { 114, 0, 7116, 0 },
    { 107, 0, 909, 0 },
    { 59, 0, 0, 10533 }, /* &searhk; */
    { 117, 1, 918, 0 },
    { 102, 1, 1994, 0 },
    { 105, 1, 2304, 0 },
    { 99, 1, 3176, 0 },
    { 101, 1, 6107, 0 },
    { 97, 1, 8679, 0 },
    { 115, 1, 8781, 0 },
    { 111, 0, 9627, 0 },
    { 59, 0, 0, 924 }, /* &Mu; */
    { 102, 1, 924, 0 },
    { 115, 1, 2975, 0 },
    { 103, 1, 6929, 0 },
    { 113, 1, 6968, 0 },
    { 59, 0, 0, 8804 }, /* &le; */
    { 116, 0, 925, 0 },
    { 104, 1, 930, 0 },
    { 114, 1, 2539, 0 },
    { 108, 1, 4195, 0 },
    { 116, 1, 5200, 0 },
    { 97, 0, 5887, 0 },
    { 97, 0, 931, 0 },
    { 114, 0, 932, 0 },
    { 112, 0, 933, 0 },
    { 111, 0, 934, 0 },
    { 111, 0, 935, 0 },
    { 110, 0, 936, 0 },
    { 117, 1, 938, 0 },
    { 100, 0, 9545, 0 },
    { 112, 0, 939, 0 },
    { 59, 0, 0, 8636 }, /* &leftharpoonup; */
    { 65, 1, 948, 0 },
    { 115, 1, 1177, 0 },
    { 97, 1, 1943, 0 },
    { 117, 1, 3453, 0 },
    { 111, 1, 5897, 0 },
    { 102, 1, 6120, 0 },
    { 99, 1, 8694, 0 },
    { 105, 0, 9023, 0 },
    { 82, 0, 949, 0 },
    { 68, 0, 950, 0 },
    { 99, 0, 951, 0 },
    { 121, 0, 952, 0 },
    { 59, 0, 0, 1066 }, /* &HARDcy; */
    { 99, 1, 955, 0 },
    { 120, 0, 1826, 0 },
    { 121, 0, 956, 0 },
    { 59, 0, 0, 1077 }, /* &iecy; */
    { 101, 1, 960, 0 },
    { 97, 1, 7300, 0 },
    { 121, 0, 8171, 0 },
    { 100, 0, 961, 0 },
    { 105, 0, 962, 0 },
    { 108, 0, 963, 0 },
    { 59, 0, 0, 354 }, /* &Tcedil; */
    { 112, 1, 967, 0 },
    { 99, 1, 3766, 0 },
    { 114, 0, 7083, 0 },
    { 99, 1, 973, 0 },
    { 100, 1, 4426, 0 },
    { 98, 1, 5528, 0 },
    { 97, 1, 8816, 0 },
    { 59, 1, 0, 8745 }, /* &cap; */
    { 115, 0, 9070, 0 },
    { 117, 1, 975, 0 },
    { 97, 0, 7611, 0 },
    { 112, 0, 976, 0 },
    { 59, 0, 0, 10823 }, /* &capcup; */
    { 113, 1, 983, 0 },
    { 116, 1, 6178, 0 },
    { 105, 1, 6374, 0 },
    { 99, 1, 6535, 0 },
    { 97, 1, 9318, 0 },
    { 104, 0, 9527, 0 },
    { 117, 1, 985, 0 },
    { 98, 0, 2792, 0 },
    { 111, 0, 986, 0 },
    { 59, 1, 0, 8216 }, /* &lsquo; */
    { 114, 0, 6694, 0 },
    { 105, 1, 990, 0 },
    { 99, 0, 6598, 0 },
    { 109, 0, 991, 0 },
    { 59, 0, 0, 10867 }, /* &Esim; */
    { 117, 0, 993, 0 },
    { 97, 1, 995, 0 },
    { 105, 0, 5354, 0 },
    { 108, 0, 996, 0 },
    { 59, 1, 0, 10869 }, /* &Equal; */
    { 84, 0, 8224, 0 },
    { 114, 1, 1010, 0 },
    { 108, 1, 1263, 0 },
    { 104, 1, 1297, 0 },
    { 111, 1, 2105, 0 },
    { 115, 1, 2854, 0 },
    { 109, 1, 2868, 0 },
    { 97, 1, 4130, 0 },
    { 101, 1, 5105, 0 },
    { 105, 1, 5745, 0 },
    { 99, 1, 7416, 0 },
    { 102, 1, 8784, 0 },
    { 117, 0, 9528, 0 },
    { 111, 1, 1020, 0 },
    { 69, 1, 2881, 0 },
    { 101, 1, 2924, 0 },
    { 110, 1, 3716, 0 },
    { 105, 1, 3912, 0 },
    { 117, 1, 5948, 0 },
    { 99, 1, 7995, 0 },
    { 115, 1, 8136, 0 },
    { 59, 1, 0, 8826 }, /* &pr; */
    { 97, 0, 8819, 0 },
    { 100, 1, 1023, 0 },
    { 102, 1, 2214, 0 },
    { 112, 0, 3698, 0 },
    { 59, 0, 0, 8719 }, /* &prod; */
    { 104, 1, 1038, 0 },
    { 79, 1, 1812, 0 },
    { 99, 1, 2042, 0 },
    { 72, 1, 2764, 0 },
    { 105, 1, 2803, 0 },
    { 111, 1, 3788, 0 },
    { 101, 1, 3993, 0 },
    { 97, 1, 4372, 0 },
    { 115, 1, 6675, 0 },
    { 117, 1, 7641, 0 },
    { 108, 1, 8362, 0 },
    { 114, 1, 8967, 0 },
    { 102, 1, 9223, 0 },
    { 100, 0, 9696, 0 },
    { 105, 0, 1039, 0 },
    { 59, 0, 0, 935 }, /* &Chi; */
    { 114, 1, 1052, 0 },
    { 111, 1, 1456, 0 },
    { 102, 1, 2422, 0 },
    { 97, 1, 2562, 0 },
    { 108, 1, 4003, 0 },
    { 101, 1, 4079, 0 },
    { 110, 1, 4490, 0 },
    { 106, 1, 7372, 0 },
    { 115, 1, 7491, 0 },
    { 99, 1, 8109, 0 },
    { 112, 1, 9182, 0 },
    { 105, 0, 9805, 0 },
    { 97, 1, 1054, 0 },
    { 111, 0, 2516, 0 },
    { 99, 1, 1056, 0 },
    { 115, 0, 4301, 0 },
    { 51, 1, 1062, 0 },
    { 49, 1, 1208, 0 },
    { 53, 1, 3443, 0 },
    { 55, 1, 6238, 0 },
    { 52, 1, 7291, 0 },
    { 50, 0, 8084, 0 },
    { 56, 1, 1065, 0 },
    { 53, 1, 4099, 0 },
    { 52, 0, 9340, 190 }, /* &frac34 */
    { 59, 0, 0, 8540 }, /* &frac38; */
    { 99, 1, 1076, 0 },
    { 100, 1, 1677, 0 },
    { 104, 1, 2129, 0 },
    { 101, 1, 3384, 0 },
    { 119, 1, 3860, 0 },
    { 115, 1, 4423, 0 },
    { 97, 1, 4616, 0 },
    { 105, 1, 5431, 0 },
    { 111, 1, 5686, 0 },
    { 102, 0, 8082, 0 },
    { 97, 1, 1078, 0 },
    { 121, 0, 7484, 0 },
    { 114, 0, 1079, 0 },
    { 111, 0, 1080, 0 },
    { 110, 0, 1081, 0 },
    { 59, 0, 0, 382 }, /* &zcaron; */
    { 115, 1, 1084, 0 },
    { 103, 0, 9533, 0 },
    { 112, 0, 1085, 0 },
    { 59, 0, 0, 8194 }, /* &ensp; */
    { 72, 1, 1093, 0 },
    { 108, 1, 1575, 0 },
    { 76, 1, 4815, 0 },
    { 114, 1, 4964, 0 },
    { 59, 1, 0, 9553 }, /* &boxV; */
    { 82, 1, 7091, 0 },
    { 104, 0, 8361, 0 },
    { 59, 0, 0, 9580 }, /* &boxVH; */
    { 99, 1, 1097, 0 },
    { 108, 1, 4784, 0 },
    { 116, 0, 5245, 0 },
    { 114, 0, 1098, 0 },
    { 59, 0, 0, 119997 }, /* &hscr; */
    { 116, 1, 1103, 0 },
    { 115, 1, 2326, 0 },
    { 101, 1, 3153, 0 },
    { 69, 0, 9300, 0 },
    { 114, 1, 1105, 0 },
    { 59, 0, 0, 8815 }, /* &ngt; */
    { 59, 0, 0, 8815 }, /* &ngtr; */
    { 112, 1, 1109, 0 },
    { 65, 1, 5325, 0 },
    { 97, 0, 9315, 0 },
    { 97, 0, 1110, 0 },
    { 114, 0, 1111, 0 },
    { 59, 0, 0, 10994 }, /* &nhpar; */
    { 114, 1, 1115, 0 },
    { 111, 1, 1768, 0 },
    { 97, 0, 2986, 0 },
    { 101, 1, 1118, 0 },
    { 59, 1, 0, 8832 }, /* &npr; */
    { 99, 0, 7960, 0 },
    { 99, 1, 1120, 0 },
    { 59, 0, 0, 10927 }, /* &npre; */
    { 59, 1, 0, 8832 }, /* &nprec; */
    { 101, 0, 4735, 0 },
    { 114, 1, 1127, 0 },
    { 109, 1, 2023, 0 },
    { 99, 1, 2284, 0 },
    { 110, 1, 5089, 0 },
    { 112, 0, 7278, 0 },
    { 114, 0, 1128, 0 },
    { 59, 0, 0, 8606 }, /* &Larr; */
    { 103, 1, 1131, 0 },
    { 59, 0, 0, 8811 }, /* &gg; */
    { 59, 0, 0, 8921 }, /* &ggg; */
    { 111, 1, 1146, 0 },
    { 79, 1, 1350, 0 },
    { 110, 1, 2027, 0 },
    { 102, 1, 2606, 0 },
    { 103, 1, 3070, 0 },
    { 116, 1, 3334, 0 },
    { 115, 1, 3404, 0 },
    { 99, 1, 3742, 0 },
    { 117, 1, 4186, 0 },
    { 69, 1, 4235, 0 },
    { 109, 1, 5542, 0 },
    { 97, 1, 8422, 0 },
    { 74, 1, 8732, 0 },
    { 100, 0, 9211, 0 },
    { 112, 1, 1149, 0 },
    { 116, 1, 8569, 0 },
    { 103, 0, 8791, 0 },
    { 102, 0, 1150, 0 },
    { 59, 0, 0, 120128 }, /* &Iopf; */
    { 105, 1, 1153, 0 },
    { 99, 0, 6321, 0 },
    { 110, 0, 1154, 0 },
    { 115, 1, 1159, 0 },
    { 100, 1, 4028, 0 },
    { 118, 1, 6177, 0 },
    { 59, 1, 0, 8712 }, /* &isin; */
    { 69, 0, 8088, 0 },
    { 118, 1, 1161, 0 },
    { 59, 0, 0, 8948 }, /* &isins; */
    { 59, 0, 0, 8947 }, /* &isinsv; */
    { 117, 0, 1163, 0 },
    { 108, 0, 1164, 0 },
    { 116, 0, 1165, 0 },
    { 59, 0, 0, 10945 }, /* &submult; */
    { 108, 0, 1167, 0 },
    { 59, 0, 0, 10553 }, /* &larrpl; */
    { 105, 1, 1172, 0 },
    { 110, 1, 4926, 0 },
    { 59, 1, 0, 8520 }, /* &ii; */
    { 111, 0, 9143, 0 },
    { 105, 1, 1174, 0 },
    { 110, 0, 4370, 0 },
    { 110, 0, 1175, 0 },
    { 116, 0, 1176, 0 },
    { 59, 0, 0, 10764 }, /* &iiiint; */
    { 116, 1, 1179, 0 },
    { 99, 0, 7598, 0 },
    { 114, 0, 1180, 0 },
    { 111, 0, 1181, 0 },
    { 107, 0, 1182, 0 },
    { 59, 0, 0, 294 }, /* &Hstrok; */
    { 104, 1, 1186, 0 },
    { 116, 1, 2574, 0 },
    { 99, 0, 4839, 0 },
    { 99, 0, 1187, 0 },
    { 121, 0, 1188, 0 },
    { 59, 0, 0, 1115 }, /* &tshcy; */
    { 105, 0, 1190, 0 },
    { 110, 0, 1191, 0 },
    { 103, 0, 6068, 229 }, /* &aring */
    { 111, 0, 1193, 0 },
    { 119, 1, 1195, 0 },
    { 117, 0, 4058, 0 },
    { 110, 0, 1196, 0 },
    { 86, 1, 1198, 0 },
    { 84, 0, 4594, 0 },
    { 101, 0, 1199, 0 },
    { 99, 0, 1200, 0 },
    { 116, 0, 1201, 0 },
    { 111, 0, 1202, 0 },
    { 114, 0, 1203, 0 },
    { 66, 1, 1205, 0 },
    { 59, 0, 0, 8643 }, /* &LeftDownVector; */
    { 97, 0, 1206, 0 },
    { 114, 0, 1207, 0 },
    { 59, 0, 0, 10585 }, /* &LeftDownVectorBar; */
    { 54, 1, 1214, 0 },
    { 51, 1, 4300, 0 },
    { 50, 1, 4963, 189 }, /* &frac12 */
    { 52, 1, 9226, 188 }, /* &frac14 */
    { 56, 1, 8213, 0 },
    { 53, 0, 8776, 0 },
    { 59, 0, 0, 8537 }, /* &frac16; */
    { 114, 1, 1219, 0 },
    { 103, 1, 3098, 0 },
    { 105, 1, 7653, 0 },
    { 108, 0, 9575, 0 },
    { 105, 0, 1220, 0 },
    { 97, 0, 1221, 0 },
    { 110, 0, 1222, 0 },
    { 103, 0, 1223, 0 },
    { 108, 0, 1224, 0 },
    { 101, 0, 1225, 0 },
    { 108, 1, 1227, 0 },
    { 114, 0, 3608, 0 },
    { 101, 0, 1228, 0 },
    { 102, 0, 1229, 0 },
    { 116, 0, 1230, 0 },
    { 101, 1, 1232, 0 },
    { 59, 0, 0, 8938 }, /* &ntriangleleft; */
    { 113, 0, 1233, 0 },
    { 59, 0, 0, 8940 }, /* &ntrianglelefteq; */
    { 110, 1, 1237, 0 },
    { 97, 1, 4977, 0 },
    { 65, 0, 8565, 0 },
    { 101, 0, 1238, 0 },
    { 97, 0, 1239, 0 },
    { 114, 0, 1240, 0 },
    { 59, 0, 0, 10535 }, /* &nwnear; */
    { 111, 1, 1252, 0 },
    { 105, 1, 2469, 0 },
    { 97, 1, 2578, 0 },
    { 101, 1, 4718, 0 },
    { 68, 1, 5130, 0 },
    { 74, 1, 5473, 0 },
    { 102, 1, 6449, 0 },
    { 90, 1, 6672, 0 },
    { 83, 1, 7727, 0 },
    { 99, 1, 8089, 0 },
    { 115, 0, 8188, 0 },
    { 112, 1, 1256, 0 },
    { 117, 1, 1625, 0 },
    { 119, 1, 3882, 0 },
    { 116, 0, 5041, 0 },
    { 102, 0, 1257, 0 },
    { 59, 0, 0, 120123 }, /* &Dopf; */
    { 121, 1, 1262, 173 }, /* &shy */
    { 111, 1, 2114, 0 },
    { 99, 1, 7196, 0 },
    { 97, 0, 8332, 0 },
    { 59, 0, 0, 173 }, /* &shy; */
    { 117, 1, 1265, 0 },
    { 97, 0, 4667, 0 },
    { 115, 0, 1266, 0 },
    { 97, 1, 1275, 0 },
    { 59, 1, 0, 43 }, /* &plus; */
    { 98, 1, 2781, 0 },
    { 109, 1, 3704, 0 },
    { 100, 1, 5453, 0 },
    { 116, 1, 6519, 0 },
    { 101, 1, 8013, 0 },
    { 99, 1, 8531, 0 },
    { 115, 0, 8878, 0 },
    { 99, 0, 1276, 0 },
    { 105, 0, 1277, 0 },
    { 114, 0, 1278, 0 },
    { 59, 0, 0, 10787 }, /* &plusacir; */
    { 105, 1, 1283, 0 },
    { 99, 1, 1996, 0 },
    { 111, 1, 6499, 0 },
    { 101, 0, 8890, 0 },
    { 109, 0, 1284, 0 },
    { 59, 1, 0, 8765 }, /* &bsim; */
    { 101, 0, 8314, 0 },
    { 120, 1, 1289, 0 },
    { 113, 1, 5437, 0 },
    { 108, 0, 7031, 0 },
    { 105, 0, 1290, 0 },
    { 115, 0, 1291, 0 },
    { 116, 0, 1292, 0 },
    { 115, 0, 1293, 0 },
    { 59, 0, 0, 8708 }, /* &NotExists; */
    { 97, 0, 1295, 0 },
    { 114, 0, 1296, 0 },
    { 59, 0, 0, 8463 }, /* &hbar; */
    { 105, 1, 1300, 0 },
    { 109, 1, 2018, 0 },
    { 111, 0, 4974, 0 },
    { 59, 1, 0, 966 }, /* &phi; */
    { 118, 0, 2022, 0 },
    { 110, 1, 1305, 0 },
    { 119, 1, 5514, 0 },
    { 112, 0, 5866, 0 },
    { 103, 0, 1306, 0 },
    { 82, 1, 1310, 0 },
    { 76, 1, 6542, 0 },
    { 108, 1, 7464, 0 },
    { 114, 0, 9252, 0 },
    { 105, 0, 1311, 0 },
    { 103, 0, 1312, 0 },
    { 104, 0, 1313, 0 },
    { 116, 0, 1314, 0 },
    { 65, 0, 1315, 0 },
    { 114, 0, 1316, 0 },
    { 114, 0, 1317, 0 },
    { 111, 0, 1318, 0 },
    { 119, 0, 1319, 0 },
    { 59, 0, 0, 10230 }, /* &LongRightArrow; */
    { 101, 1, 1322, 0 },
    { 69, 0, 8319, 0 },
    { 59, 0, 0, 8842 }, /* &subne; */
    { 114, 0, 1324, 0 },
    { 59, 0, 0, 120113 }, /* &tfr; */
    { 111, 1, 1328, 0 },
    { 99, 1, 3754, 0 },
    { 116, 0, 6942, 0 },
    { 108, 0, 1329, 0 },
    { 59, 0, 0, 10742 }, /* &dsol; */
    { 114, 1, 1332, 0 },
    { 100, 0, 7045, 0 },
    { 105, 0, 1333, 0 },
    { 59, 1, 0, 9663 }, /* &dtri; */
    { 102, 0, 8671, 0 },
    { 105, 1, 1337, 0 },
    { 103, 0, 4641, 0 },
    { 110, 0, 1338, 0 },
    { 116, 0, 1339, 0 },
    { 59, 0, 0, 8750 }, /* &conint; */
    { 101, 1, 1342, 0 },
    { 59, 0, 0, 8920 }, /* &Ll; */
    { 102, 0, 1343, 0 },
    { 116, 0, 1344, 0 },
    { 97, 0, 1345, 0 },
    { 114, 0, 1346, 0 },
    { 114, 0, 1347, 0 },
    { 111, 0, 1348, 0 },
    { 119, 0, 1349, 0 },
    { 59, 0, 0, 8666 }, /* &Lleftarrow; */
    { 99, 0, 1351, 0 },
    { 121, 0, 1352, 0 },
    { 59, 0, 0, 1025 }, /* &IOcy; */
    { 113, 0, 1354, 0 },
    { 117, 0, 1355, 0 },
    { 111, 0, 1356, 0 },
    { 59, 0, 0, 8218 }, /* &sbquo; */
    { 114, 0, 1358, 0 },
    { 105, 0, 1359, 0 },
    { 59, 0, 0, 9722 }, /* &lltri; */
    { 108, 0, 1361, 0 },
    { 114, 0, 1362, 0 },
    { 101, 0, 1363, 0 },
    { 99, 0, 1364, 0 },
    { 59, 0, 0, 8981 }, /* &telrec; */
    { 116, 1, 1367, 0 },
    { 103, 0, 4219, 0 },
    { 59, 1, 0, 8811 }, /* &nGt; */
    { 118, 0, 4692, 0 },
    { 97, 1, 1372, 0 },
    { 103, 1, 5328, 174 }, /* &reg */
    { 99, 0, 7001, 0 },
    { 108, 0, 1373, 0 },
    { 105, 1, 1377, 0 },
    { 115, 1, 3399, 0 },
    { 59, 1, 0, 8476 }, /* &real; */
    { 112, 0, 6753, 0 },
    { 110, 0, 1378, 0 },
    { 101, 0, 1379, 0 },
    { 59, 0, 0, 8475 }, /* &realine; */
    { 114, 0, 1381, 0 },
    { 114, 0, 1382, 0 },
    { 59, 0, 0, 8664 }, /* &seArr; */
    { 97, 0, 1384, 0 },
    { 114, 0, 1385, 0 },
    { 107, 0, 1386, 0 },
    { 59, 0, 0, 10003 }, /* &checkmark; */
    { 114, 1, 1393, 0 },
    { 99, 1, 3903, 0 },
    { 113, 1, 3907, 0 },
    { 59, 1, 0, 62 }, /* &gt; */
    { 108, 1, 5625, 0 },
    { 100, 0, 7877, 0 },
    { 101, 1, 1398, 0 },
    { 97, 1, 3049, 0 },
    { 108, 1, 8672, 0 },
    { 115, 1, 8690, 0 },
    { 100, 0, 9754, 0 },
    { 113, 0, 1399, 0 },
    { 108, 1, 1401, 0 },
    { 113, 0, 6658, 0 },
    { 101, 0, 1402, 0 },
    { 115, 0, 1403, 0 },
    { 115, 0, 1404, 0 },
    { 59, 0, 0, 8923 }, /* &gtreqless; */
    { 121, 1, 1408, 0 },
    { 105, 1, 2373, 0 },
    { 59, 0, 0, 8291 }, /* &ic; */
    { 59, 0, 0, 1080 }, /* &icy; */
    { 59, 0, 0, 60 }, /* &LT; */
    { 117, 1, 1412, 0 },
    { 111, 0, 4999, 0 },
    { 112, 0, 1413, 0 },
    { 67, 0, 1414, 0 },
    { 97, 0, 1415, 0 },
    { 112, 0, 1416, 0 },
    { 59, 0, 0, 8813 }, /* &NotCupCap; */
    { 99, 1, 1431, 0 },
    { 102, 1, 1553, 0 },
    { 98, 1, 1595, 0 },
    { 112, 1, 1794, 0 },
    { 114, 1, 2329, 0 },
    { 110, 1, 4251, 0 },
    { 109, 1, 4439, 0 },
    { 103, 1, 4566, 0 },
    { 111, 1, 5669, 0 },
    { 100, 1, 6070, 0 },
    { 116, 1, 6464, 0 },
    { 97, 1, 6590, 0 },
    { 115, 1, 7531, 0 },
    { 117, 0, 8072, 0 },
    { 105, 1, 1433, 0 },
    { 121, 0, 2342, 0 },
    { 114, 0, 1434, 0 },
    { 99, 0, 8039, 219 }, /* &Ucirc */
    { 111, 1, 1443, 0 },
    { 101, 1, 3031, 0 },
    { 115, 1, 4680, 0 },
    { 102, 1, 6388, 0 },
    { 100, 1, 8221, 0 },
    { 99, 1, 8628, 0 },
    { 97, 1, 9132, 0 },
    { 72, 0, 9157, 0 },
    { 112, 0, 1444, 0 },
    { 102, 0, 1445, 0 },
    { 59, 0, 0, 8484 }, /* &Zopf; */
    { 97, 1, 1449, 0 },
    { 121, 1, 7610, 0 },
    { 101, 0, 9482, 0 },
    { 114, 0, 1450, 0 },
    { 111, 0, 1451, 0 },
    { 110, 0, 1452, 0 },
    { 59, 0, 0, 357 }, /* &tcaron; */
    { 59, 1, 0, 956 }, /* &mu; */
    { 109, 1, 7724, 0 },
    { 108, 0, 9373, 0 },
    { 114, 1, 1458, 0 },
    { 112, 0, 7210, 0 },
    { 97, 1, 1460, 0 },
    { 107, 0, 1612, 0 },
    { 108, 0, 1461, 0 },
    { 108, 0, 1462, 0 },
    { 59, 0, 0, 8704 }, /* &forall; */
    { 105, 1, 1466, 0 },
    { 114, 1, 2700, 0 },
    { 104, 0, 5350, 0 },
    { 109, 0, 1467, 0 },
    { 101, 0, 1468, 0 },
    { 115, 0, 1469, 0 },
    { 59, 0, 0, 8906 }, /* &rtimes; */
    { 102, 1, 1478, 0 },
    { 114, 1, 1963, 0 },
    { 117, 1, 3480, 0 },
    { 101, 1, 4271, 0 },
    { 97, 1, 5310, 0 },
    { 115, 1, 5827, 0 },
    { 111, 1, 6025, 0 },
    { 99, 0, 7109, 0 },
    { 114, 0, 1479, 0 },
    { 59, 0, 0, 120069 }, /* &Bfr; */
    { 103, 1, 1482, 0 },
    { 100, 0, 5018, 0 },
    { 109, 1, 1489, 0 },
    { 59, 1, 0, 8736 }, /* &ang; */
    { 122, 1, 3597, 0 },
    { 101, 1, 4303, 0 },
    { 115, 1, 4324, 0 },
    { 108, 1, 5362, 0 },
    { 114, 0, 9176, 0 },
    { 115, 0, 1490, 0 },
    { 100, 0, 1491, 0 },
    { 97, 1, 1493, 0 },
    { 59, 0, 0, 8737 }, /* &angmsd; */
    { 100, 1, 1501, 0 },
    { 103, 1, 2735, 0 },
    { 102, 1, 3558, 0 },
    { 97, 1, 4369, 0 },
    { 99, 1, 5556, 0 },
    { 98, 1, 8717, 0 },
    { 104, 1, 9306, 0 },
    { 101, 0, 9751, 0 },
    { 59, 0, 0, 10667 }, /* &angmsdad; */
    { 114, 0, 1503, 0 },
    { 114, 0, 1504, 0 },
    { 59, 0, 0, 10229 }, /* &xlarr; */
    { 114, 1, 1512, 0 },
    { 100, 1, 1887, 0 },
    { 101, 1, 2741, 0 },
    { 112, 1, 3390, 0 },
    { 108, 1, 3576, 0 },
    { 119, 1, 4025, 0 },
    { 118, 0, 4812, 0 },
    { 118, 1, 1516, 0 },
    { 97, 1, 1852, 0 },
    { 114, 1, 2127, 0 },
    { 108, 0, 4911, 0 },
    { 101, 0, 1517, 0 },
    { 97, 0, 1518, 0 },
    { 114, 0, 1519, 0 },
    { 114, 0, 1520, 0 },
    { 111, 0, 1521, 0 },
    { 119, 0, 1522, 0 },
    { 108, 1, 1524, 0 },
    { 114, 0, 7717, 0 },
    { 101, 0, 1525, 0 },
    { 102, 0, 1526, 0 },
    { 116, 0, 1527, 0 },
    { 59, 0, 0, 8630 }, /* &curvearrowleft; */
    { 105, 1, 1531, 0 },
    { 112, 1, 1857, 0 },
    { 97, 0, 4971, 0 },
    { 97, 1, 1538, 0 },
    { 101, 1, 1615, 0 },
    { 116, 1, 4575, 0 },
    { 112, 1, 6347, 0 },
    { 109, 1, 6899, 0 },
    { 100, 1, 7312, 0 },
    { 115, 0, 9089, 0 },
    { 110, 0, 1539, 0 },
    { 103, 0, 1540, 0 },
    { 108, 0, 1541, 0 },
    { 101, 0, 1542, 0 },
    { 114, 1, 1547, 0 },
    { 59, 1, 0, 9653 }, /* &triangle; */
    { 108, 1, 6489, 0 },
    { 113, 1, 6997, 0 },
    { 100, 0, 7118, 0 },
    { 105, 0, 1548, 0 },
    { 103, 0, 1549, 0 },
    { 104, 0, 1550, 0 },
    { 116, 0, 1551, 0 },
    { 59, 1, 0, 9657 }, /* &triangleright; */
    { 101, 0, 9304, 0 },
    { 114, 0, 1554, 0 },
    { 59, 0, 0, 120088 }, /* &Ufr; */
    { 85, 1, 1559, 0 },
    { 115, 1, 6744, 0 },
    { 111, 1, 6763, 0 },
    { 102, 0, 6780, 0 },
    { 79, 0, 1560, 0 },
    { 84, 0, 8231, 34 }, /* &QUOT */
    { 99, 1, 1563, 0 },
    { 116, 0, 7825, 0 },
    { 114, 0, 1564, 0 },
    { 59, 0, 0, 120002 }, /* &mscr; */
    { 111, 0, 1566, 0 },
    { 117, 0, 1567, 0 },
    { 115, 0, 1568, 0 },
    { 116, 0, 1569, 0 },
    { 97, 1, 1571, 0 },
    { 59, 0, 0, 9137 }, /* &rmoust; */
    { 99, 0, 1572, 0 },
    { 104, 0, 1573, 0 },
    { 101, 0, 1574, 0 },
    { 59, 0, 0, 9137 }, /* &rmoustache; */
    { 59, 0, 0, 9570 }, /* &boxVl; */
    { 105, 1, 1579, 0 },
    { 97, 1, 7542, 0 },
    { 101, 0, 9825, 0 },
    { 99, 1, 1582, 0 },
    { 110, 1, 5975, 0 },
    { 100, 0, 9509, 0 },
    { 114, 0, 1583, 0 },
    { 111, 0, 1584, 0 },
    { 110, 0, 1585, 0 },
    { 59, 0, 0, 959 }, /* &omicron; */
    { 116, 1, 1590, 0 },
    { 65, 1, 6054, 0 },
    { 105, 1, 7770, 0 },
    { 97, 0, 8619, 0 },
    { 114, 0, 1591, 0 },
    { 105, 0, 1592, 0 },
    { 101, 1, 1594, 0 },
    { 59, 0, 0, 8939 }, /* &nrtri; */
    { 59, 0, 0, 8941 }, /* &nrtrie; */
    { 114, 0, 1596, 0 },
    { 99, 1, 1598, 0 },
    { 101, 0, 2972, 0 },
    { 121, 0, 1599, 0 },
    { 59, 0, 0, 1038 }, /* &Ubrcy; */
    { 97, 1, 1608, 0 },
    { 112, 1, 1896, 0 },
    { 111, 1, 2912, 0 },
    { 122, 1, 3919, 0 },
    { 119, 1, 4020, 0 },
    { 110, 1, 4653, 0 },
    { 98, 1, 8104, 0 },
    { 116, 0, 8883, 0 },
    { 114, 1, 1610, 0 },
    { 110, 0, 3917, 0 },
    { 114, 0, 1611, 0 },
    { 59, 0, 0, 8701 }, /* &loarr; */
    { 118, 1, 1614, 0 },
    { 59, 0, 0, 8916 }, /* &fork; */
    { 59, 0, 0, 10969 }, /* &forkv; */
    { 59, 0, 0, 8796 }, /* &trie; */
    { 101, 1, 1619, 0 },
    { 114, 1, 5933, 0 },
    { 108, 0, 9391, 0 },
    { 98, 1, 1622, 0 },
    { 59, 1, 0, 8744 }, /* &vee; */
    { 101, 0, 7899, 0 },
    { 97, 0, 1623, 0 },
    { 114, 0, 1624, 0 },
    { 59, 0, 0, 8891 }, /* &veebar; */
    { 98, 0, 1626, 0 },
    { 108, 0, 1627, 0 },
    { 101, 0, 1628, 0 },
    { 67, 1, 1634, 0 },
    { 85, 1, 2430, 0 },
    { 86, 1, 3864, 0 },
    { 76, 1, 5066, 0 },
    { 82, 1, 5749, 0 },
    { 68, 0, 8488, 0 },
    { 111, 0, 1635, 0 },
    { 110, 0, 1636, 0 },
    { 116, 0, 1637, 0 },
    { 111, 0, 1638, 0 },
    { 117, 0, 1639, 0 },
    { 114, 0, 1640, 0 },
    { 73, 0, 1641, 0 },
    { 110, 0, 1642, 0 },
    { 116, 0, 1643, 0 },
    { 101, 0, 1644, 0 },
    { 103, 0, 1645, 0 },
    { 114, 0, 1646, 0 },
    { 97, 0, 1647, 0 },
    { 108, 0, 1648, 0 },
    { 59, 0, 0, 8751 }, /* &DoubleContourIntegral; */
    { 114, 1, 1651, 0 },
    { 110, 0, 8720, 0 },
    { 115, 1, 1658, 0 },
    { 101, 1, 3654, 0 },
    { 112, 1, 4282, 0 },
    { 110, 1, 4754, 0 },
    { 116, 1, 5143, 0 },
    { 114, 1, 5158, 0 },
    { 107, 0, 7474, 0 },
    { 117, 1, 1660, 0 },
    { 105, 0, 5365, 0 },
    { 112, 1, 1662, 0 },
    { 98, 0, 2414, 0 },
    { 115, 0, 1663, 0 },
    { 101, 0, 1664, 0 },
    { 116, 0, 1665, 0 },
    { 110, 0, 1666, 0 },
    { 101, 0, 1667, 0 },
    { 113, 0, 1668, 0 },
    { 59, 1, 0, 8843 }, /* &varsupsetneq; */
    { 113, 0, 8028, 0 },
    { 105, 1, 1675, 0 },
    { 111, 1, 3712, 0 },
    { 97, 1, 6104, 0 },
    { 98, 1, 6985, 0 },
    { 115, 0, 7129, 0 },
    { 118, 0, 1676, 0 },
    { 59, 0, 0, 10808 }, /* &odiv; */
    { 111, 0, 1678, 0 },
    { 116, 0, 1679, 0 },
    { 59, 0, 0, 380 }, /* &zdot; */
    { 105, 1, 1685, 0 },
    { 111, 1, 2403, 0 },
    { 99, 1, 3506, 0 },
    { 115, 1, 4279, 0 },
    { 102, 0, 4555, 0 },
    { 108, 0, 1686, 0 },
    { 108, 0, 1687, 0 },
    { 101, 0, 1688, 0 },
    { 100, 0, 1689, 0 },
    { 86, 1, 1691, 0 },
    { 83, 0, 5007, 0 },
    { 101, 0, 1692, 0 },
    { 114, 0, 1693, 0 },
    { 121, 0, 1694, 0 },
    { 83, 0, 1695, 0 },
    { 109, 0, 1696, 0 },
    { 97, 0, 1697, 0 },
    { 108, 0, 1698, 0 },
    { 108, 0, 1699, 0 },
    { 83, 0, 1700, 0 },
    { 113, 0, 1701, 0 },
    { 117, 0, 1702, 0 },
    { 97, 0, 1703, 0 },
    { 114, 0, 1704, 0 },
    { 101, 0, 1705, 0 },
    { 59, 0, 0, 9642 }, /* &FilledVerySmallSquare; */
    { 105, 1, 1708, 0 },
    { 99, 0, 8888, 0 },
    { 109, 0, 1709, 0 },
    { 59, 1, 0, 8819 }, /* &gsim; */
    { 101, 1, 4819, 0 },
    { 108, 0, 7768, 0 },
    { 114, 1, 1714, 0 },
    { 105, 0, 9774, 0 },
    { 59, 0, 0, 120097 }, /* &dfr; */
    { 107, 1, 1717, 0 },
    { 108, 0, 7986, 0 },
    { 97, 0, 1718, 0 },
    { 114, 0, 1719, 0 },
    { 111, 0, 1720, 0 },
    { 119, 0, 1721, 0 },
    { 59, 0, 0, 10511 }, /* &dbkarow; */
    { 99, 1, 1726, 0 },
    { 118, 1, 5329, 0 },
    { 115, 1, 5443, 0 },
    { 117, 0, 6469, 0 },
    { 105, 1, 1728, 0 },
    { 111, 0, 7581, 0 },
    { 114, 0, 1729, 0 },
    { 99, 0, 1730, 0 },
    { 59, 0, 0, 8790 }, /* &eqcirc; */
    { 111, 1, 1739, 0 },
    { 106, 1, 2142, 0 },
    { 104, 1, 3465, 0 },
    { 97, 1, 3693, 0 },
    { 103, 1, 4094, 0 },
    { 102, 1, 7901, 0 },
    { 99, 1, 8194, 0 },
    { 115, 0, 8571, 0 },
    { 112, 0, 1740, 0 },
    { 102, 0, 1741, 0 },
    { 59, 0, 0, 120156 }, /* &kopf; */
    { 114, 1, 1745, 0 },
    { 105, 1, 4211, 0 },
    { 108, 0, 9699, 0 },
    { 59, 0, 0, 120105 }, /* &lfr; */
    { 115, 1, 1762, 0 },
    { 111, 1, 2121, 0 },
    { 116, 1, 2188, 0 },
    { 117, 1, 2861, 0 },
    { 102, 1, 2939, 0 },
    { 99, 1, 3705, 0 },
    { 112, 1, 4043, 0 },
    { 109, 1, 4115, 0 },
    { 114, 1, 4725, 0 },
    { 77, 1, 5082, 0 },
    { 110, 1, 6175, 0 },
    { 69, 1, 6194, 0 },
    { 103, 1, 6402, 0 },
    { 98, 1, 6808, 0 },
    { 97, 1, 7494, 0 },
    { 108, 0, 9113, 0 },
    { 115, 1, 1764, 0 },
    { 99, 0, 3096, 0 },
    { 105, 0, 1765, 0 },
    { 103, 0, 1766, 0 },
    { 110, 0, 1767, 0 },
    { 59, 0, 0, 8788 }, /* &Assign; */
    { 108, 0, 1769, 0 },
    { 105, 0, 1770, 0 },
    { 110, 0, 1771, 0 },
    { 116, 0, 1772, 0 },
    { 59, 0, 0, 10772 }, /* &npolint; */
    { 113, 0, 1774, 0 },
    { 117, 0, 1775, 0 },
    { 111, 0, 1776, 0 },
    { 59, 0, 0, 8222 }, /* &bdquo; */
    { 117, 1, 1779, 0 },
    { 99, 0, 8936, 0 },
    { 98, 1, 1781, 0 },
    { 112, 0, 2361, 0 },
    { 110, 0, 1782, 0 },
    { 69, 1, 1784, 0 },
    { 101, 0, 4882, 0 },
    { 59, 0, 0, 10955 }, /* &vsubnE; */
    { 116, 1, 1790, 0 },
    { 112, 1, 3400, 0 },
    { 99, 1, 4857, 0 },
    { 114, 1, 5969, 0 },
    { 109, 0, 8926, 0 },
    { 97, 1, 1793, 0 },
    { 104, 1, 7916, 0 },
    { 119, 0, 9081, 0 },
    { 59, 0, 0, 946 }, /* &beta; */
    { 65, 1, 1802, 0 },
    { 115, 1, 4109, 0 },
    { 68, 1, 5878, 0 },
    { 97, 1, 6189, 0 },
    { 100, 1, 6617, 0 },
    { 112, 1, 6695, 0 },
    { 84, 1, 6733, 0 },
    { 69, 0, 9578, 0 },
    { 114, 0, 1803, 0 },
    { 114, 0, 1804, 0 },
    { 111, 0, 1805, 0 },
    { 119, 0, 1806, 0 },
    { 66, 1, 1809, 0 },
    { 59, 1, 0, 8593 }, /* &UpArrow; */
    { 68, 0, 8509, 0 },
    { 97, 0, 1810, 0 },
    { 114, 0, 1811, 0 },
    { 59, 0, 0, 10514 }, /* &UpArrowBar; */
    { 80, 0, 1813, 0 },
    { 89, 0, 1814, 169 }, /* &COPY */
    { 59, 0, 0, 169 }, /* &COPY; */
    { 101, 0, 1816, 0 },
    { 114, 0, 1817, 0 },
    { 116, 0, 1818, 0 },
    { 105, 0, 1819, 0 },
    { 99, 0, 1820, 0 },
    { 97, 0, 1821, 0 },
    { 108, 0, 1822, 0 },
    { 66, 0, 1823, 0 },
    { 97, 0, 1824, 0 },
    { 114, 0, 1825, 0 },
    { 59, 0, 0, 8740 }, /* &NotVerticalBar; */
    { 99, 0, 1827, 0 },
    { 108, 0, 4403, 161 }, /* &iexcl */
    { 112, 1, 1832, 0 },
    { 116, 1, 4627, 0 },
    { 97, 1, 4684, 0 },
    { 98, 0, 9760, 0 },
    { 102, 1, 1835, 0 },
    { 108, 1, 1849, 0 },
    { 97, 0, 3497, 0 },
    { 59, 0, 0, 120163 }, /* &ropf; */
    { 109, 1, 1838, 0 },
    { 103, 0, 7037, 0 },
    { 101, 1, 1846, 0 },
    { 59, 1, 0, 8764 }, /* &sim; */
    { 110, 1, 3757, 0 },
    { 108, 1, 4706, 0 },
    { 103, 1, 5622, 0 },
    { 100, 1, 6057, 0 },
    { 114, 1, 7637, 0 },
    { 112, 0, 9092, 0 },
    { 113, 1, 1848, 0 },
    { 59, 0, 0, 8771 }, /* &sime; */
    { 59, 0, 0, 8771 }, /* &simeq; */
    { 117, 0, 1850, 0 },
    { 115, 0, 1851, 0 },
    { 59, 0, 0, 10798 }, /* &roplus; */
    { 114, 0, 1853, 0 },
    { 114, 0, 1854, 0 },
    { 109, 1, 1856, 0 },
    { 59, 0, 0, 8631 }, /* &curarr; */
    { 59, 0, 0, 10556 }, /* &curarrm; */
    { 101, 0, 1858, 0 },
    { 122, 0, 1859, 0 },
    { 105, 0, 1860, 0 },
    { 117, 0, 1861, 0 },
    { 109, 0, 1862, 0 },
    { 59, 0, 0, 9186 }, /* &trpezium; */
    { 104, 1, 1881, 0 },
    { 111, 1, 1938, 0 },
    { 112, 1, 1979, 0 },
    { 117, 1, 2249, 0 },
    { 114, 1, 2333, 0 },
    { 116, 1, 2507, 0 },
    { 115, 1, 3508, 0 },
    { 98, 1, 3564, 0 },
    { 65, 1, 3805, 0 },
    { 97, 1, 4220, 0 },
    { 119, 1, 4729, 0 },
    { 100, 1, 4825, 0 },
    { 109, 1, 4968, 0 },
    { 99, 1, 5793, 0 },
    { 102, 1, 7251, 0 },
    { 108, 1, 7399, 0 },
    { 103, 1, 9128, 0 },
    { 72, 0, 9151, 0 },
    { 97, 1, 1883, 0 },
    { 98, 0, 6647, 0 },
    { 114, 0, 1884, 0 },
    { 114, 1, 1886, 0 },
    { 108, 0, 9276, 0 },
    { 59, 0, 0, 8638 }, /* &uharr; */
    { 97, 0, 1888, 0 },
    { 114, 0, 1889, 0 },
    { 114, 0, 1890, 0 },
    { 108, 1, 1892, 0 },
    { 114, 0, 8008, 0 },
    { 59, 0, 0, 10552 }, /* &cudarrl; */
    { 109, 1, 1895, 0 },
    { 114, 0, 2496, 0 },
    { 108, 0, 2372, 235 }, /* &euml */
    { 108, 1, 1899, 0 },
    { 102, 1, 8693, 0 },
    { 97, 0, 9672, 0 },
    { 117, 0, 1900, 0 },
    { 115, 0, 1901, 0 },
    { 59, 0, 0, 10797 }, /* &loplus; */
    { 110, 1, 1907, 0 },
    { 116, 1, 3340, 0 },
    { 112, 1, 5647, 0 },
    { 98, 1, 5675, 0 },
    { 99, 0, 7977, 0 },
    { 103, 0, 1908, 0 },
    { 59, 0, 0, 8736 }, /* &nang; */
    { 99, 1, 1918, 0 },
    { 97, 1, 2840, 0 },
    { 65, 1, 3548, 0 },
    { 117, 1, 4603, 0 },
    { 115, 1, 5613, 0 },
    { 85, 1, 6407, 0 },
    { 111, 1, 8242, 0 },
    { 73, 1, 9608, 0 },
    { 102, 0, 9734, 0 },
    { 121, 1, 1920, 0 },
    { 105, 0, 8266, 0 },
    { 59, 0, 0, 1067 }, /* &Ycy; */
    { 116, 1, 1924, 0 },
    { 99, 1, 3016, 0 },
    { 104, 0, 5595, 0 },
    { 114, 0, 1925, 0 },
    { 111, 0, 1926, 0 },
    { 107, 0, 1927, 0 },
    { 59, 0, 0, 321 }, /* &Lstrok; */
    { 99, 0, 1929, 0 },
    { 115, 1, 1935, 0 },
    { 99, 1, 5335, 0 },
    { 110, 1, 6124, 0 },
    { 97, 1, 8518, 0 },
    { 59, 1, 0, 8827 }, /* &succ; */
    { 101, 0, 9749, 0 },
    { 105, 0, 1936, 0 },
    { 109, 0, 1937, 0 },
    { 59, 0, 0, 8831 }, /* &succsim; */
    { 103, 1, 1940, 0 },
    { 112, 0, 9109, 0 },
    { 111, 0, 1941, 0 },
    { 110, 0, 1942, 0 },
    { 59, 0, 0, 371 }, /* &uogon; */
    { 99, 1, 1945, 0 },
    { 116, 0, 4709, 0 },
    { 101, 0, 1946, 0 },
    { 107, 0, 1947, 0 },
    { 59, 0, 0, 711 }, /* &Hacek; */
    { 101, 0, 1949, 0 },
    { 115, 1, 1951, 0 },
    { 102, 0, 5557, 0 },
    { 115, 0, 1952, 0 },
    { 84, 1, 1958, 0 },
    { 76, 1, 2197, 0 },
    { 71, 1, 2392, 0 },
    { 69, 1, 3206, 0 },
    { 59, 1, 0, 8814 }, /* &NotLess; */
    { 83, 0, 6043, 0 },
    { 105, 0, 1959, 0 },
    { 108, 0, 1960, 0 },
    { 100, 0, 1961, 0 },
    { 101, 0, 1962, 0 },
    { 59, 0, 0, 8820 }, /* &NotLessTilde; */
    { 101, 0, 1964, 0 },
    { 118, 0, 1965, 0 },
    { 101, 0, 1966, 0 },
    { 59, 0, 0, 728 }, /* &Breve; */
    { 112, 0, 1968, 0 },
    { 102, 0, 1969, 0 },
    { 59, 0, 0, 120160 }, /* &oopf; */
    { 97, 0, 1971, 0 },
    { 115, 0, 1972, 0 },
    { 104, 0, 1973, 0 },
    { 59, 0, 0, 8875 }, /* &VDash; */
    { 59, 1, 0, 9552 }, /* &boxH; */
    { 117, 1, 3623, 0 },
    { 100, 1, 5503, 0 },
    { 85, 1, 6053, 0 },
    { 68, 0, 7597, 0 },
    { 115, 1, 1985, 0 },
    { 97, 1, 4228, 0 },
    { 108, 1, 6005, 0 },
    { 117, 1, 6323, 0 },
    { 104, 1, 6678, 0 },
    { 100, 0, 9518, 0 },
    { 105, 0, 1986, 0 },
    { 108, 1, 1989, 0 },
    { 104, 1, 3488, 0 },
    { 59, 0, 0, 965 }, /* &upsi; */
    { 111, 0, 1990, 0 },
    { 110, 0, 1991, 0 },
    { 59, 0, 0, 965 }, /* &upsilon; */
    { 114, 0, 1993, 0 },
    { 59, 0, 0, 120072 }, /* &Efr; */
    { 114, 0, 1995, 0 },
    { 59, 0, 0, 120080 }, /* &Mfr; */
    { 114, 0, 1997, 0 },
    { 59, 0, 0, 119991 }, /* &bscr; */
    { 105, 1, 2002, 0 },
    { 101, 1, 2667, 0 },
    { 116, 1, 3228, 0 },
    { 97, 0, 6431, 0 },
    { 100, 1, 2004, 0 },
    { 108, 0, 6122, 0 },
    { 59, 0, 0, 8739 }, /* &smid; */
    { 115, 1, 2008, 0 },
    { 103, 1, 2225, 0 },
    { 110, 0, 7568, 0 },
    { 105, 0, 2009, 0 },
    { 110, 0, 2010, 0 },
    { 103, 0, 2011, 0 },
    { 100, 0, 2012, 0 },
    { 111, 0, 2013, 0 },
    { 116, 0, 2014, 0 },
    { 115, 0, 2015, 0 },
    { 101, 0, 2016, 0 },
    { 113, 0, 2017, 0 },
    { 59, 0, 0, 8787 }, /* &risingdotseq; */
    { 109, 0, 2019, 0 },
    { 97, 0, 2020, 0 },
    { 116, 0, 2021, 0 },
    { 59, 0, 0, 8499 }, /* &phmmat; */
    { 59, 0, 0, 981 }, /* &phiv; */
    { 98, 0, 2024, 0 },
    { 100, 0, 2025, 0 },
    { 97, 0, 2026, 0 },
    { 59, 0, 0, 923 }, /* &Lambda; */
    { 118, 1, 2029, 0 },
    { 116, 0, 6947, 0 },
    { 105, 0, 2030, 0 },
    { 115, 0, 2031, 0 },
    { 105, 0, 2032, 0 },
    { 98, 0, 2033, 0 },
    { 108, 0, 2034, 0 },
    { 101, 0, 2035, 0 },
    { 67, 1, 2037, 0 },
    { 84, 0, 8821, 0 },
    { 111, 0, 2038, 0 },
    { 109, 0, 2039, 0 },
    { 109, 0, 2040, 0 },
    { 97, 0, 2041, 0 },
    { 59, 0, 0, 8291 }, /* &InvisibleComma; */
    { 97, 1, 2046, 0 },
    { 101, 1, 5035, 0 },
    { 111, 1, 6019, 0 },
    { 105, 0, 7451, 0 },
    { 114, 0, 2047, 0 },
    { 111, 0, 2048, 0 },
    { 110, 0, 2049, 0 },
    { 59, 0, 0, 268 }, /* &Ccaron; */
    { 117, 0, 2051, 0 },
    { 97, 0, 2052, 0 },
    { 114, 0, 2053, 0 },
    { 101, 0, 2054, 0 },
    { 83, 0, 2055, 0 },
    { 117, 0, 2056, 0 },
    { 98, 1, 2058, 0 },
    { 112, 0, 6143, 0 },
    { 115, 0, 2059, 0 },
    { 101, 0, 2060, 0 },
    { 116, 0, 2061, 0 },
    { 59, 1, 0, 8847 }, /* &NotSquareSubset; */
    { 69, 0, 6771, 0 },
    { 109, 0, 2064, 0 },
    { 108, 0, 7195, 228 }, /* &auml */
    { 114, 0, 2066, 0 },
    { 101, 0, 2067, 0 },
    { 97, 0, 2068, 0 },
    { 107, 0, 2069, 0 },
    { 59, 0, 0, 8288 }, /* &NoBreak; */
    { 110, 1, 2083, 0 },
    { 101, 1, 2745, 0 },
    { 112, 1, 2750, 0 },
    { 100, 1, 3288, 0 },
    { 51, 1, 6378, 179 }, /* &sup3 */
    { 50, 1, 5815, 178 }, /* &sup2 */
    { 49, 1, 7438, 185 }, /* &sup1 */
    { 59, 1, 0, 8835 }, /* &sup; */
    { 69, 1, 3847, 0 },
    { 115, 1, 4031, 0 },
    { 104, 1, 4865, 0 },
    { 108, 1, 6000, 0 },
    { 109, 0, 8029, 0 },
    { 101, 1, 2085, 0 },
    { 69, 0, 8681, 0 },
    { 59, 0, 0, 8843 }, /* &supne; */
    { 114, 1, 2090, 0 },
    { 115, 1, 3581, 0 },
    { 99, 1, 5620, 0 },
    { 113, 0, 5784, 0 },
    { 117, 1, 2092, 0 },
    { 100, 0, 7942, 0 },
    { 115, 0, 2093, 0 },
    { 104, 0, 2094, 0 },
    { 97, 0, 2095, 0 },
    { 114, 0, 2096, 0 },
    { 59, 0, 0, 10571 }, /* &ldrushar; */
    { 105, 1, 2099, 0 },
    { 99, 0, 5187, 0 },
    { 103, 0, 2100, 0 },
    { 114, 0, 2101, 0 },
    { 97, 0, 2102, 0 },
    { 114, 0, 2103, 0 },
    { 114, 0, 2104, 0 },
    { 59, 0, 0, 10239 }, /* &dzigrarr; */
    { 105, 1, 2108, 0 },
    { 117, 1, 4000, 0 },
    { 112, 0, 6564, 0 },
    { 110, 0, 2109, 0 },
    { 116, 0, 2110, 0 },
    { 105, 0, 2111, 0 },
    { 110, 0, 2112, 0 },
    { 116, 0, 2113, 0 },
    { 59, 0, 0, 10773 }, /* &pointint; */
    { 114, 0, 2115, 0 },
    { 116, 0, 2116, 0 },
    { 109, 1, 2118, 0 },
    { 112, 0, 9635, 0 },
    { 105, 0, 2119, 0 },
    { 100, 0, 2120, 0 },
    { 59, 0, 0, 8739 }, /* &shortmid; */
    { 112, 1, 2123, 0 },
    { 103, 0, 6919, 0 },
    { 102, 0, 2124, 0 },
    { 59, 0, 0, 120120 }, /* &Aopf; */
    { 112, 0, 2126, 0 },
    { 59, 0, 0, 10938 }, /* &scnap; */
    { 101, 0, 2128, 0 },
    { 110, 0, 9091, 164 }, /* &curren */
    { 99, 0, 2130, 0 },
    { 121, 0, 2131, 0 },
    { 59, 0, 0, 1078 }, /* &zhcy; */
    { 105, 0, 2133, 0 },
    { 103, 0, 2134, 0 },
    { 104, 0, 2135, 0 },
    { 116, 0, 2136, 0 },
    { 97, 0, 2137, 0 },
    { 114, 0, 2138, 0 },
    { 114, 0, 2139, 0 },
    { 111, 0, 2140, 0 },
    { 119, 0, 2141, 0 },
    { 59, 0, 0, 8655 }, /* &nRightarrow; */
    { 99, 0, 2143, 0 },
    { 121, 0, 2144, 0 },
    { 59, 0, 0, 1116 }, /* &kjcy; */
    { 108, 0, 2146, 0 },
    { 100, 0, 2147, 0 },
    { 101, 0, 2148, 0 },
    { 70, 1, 2152, 0 },
    { 69, 1, 2553, 0 },
    { 84, 1, 3314, 0 },
    { 59, 0, 0, 8764 }, /* &Tilde; */
    { 117, 0, 2153, 0 },
    { 108, 0, 2154, 0 },
    { 108, 0, 2155, 0 },
    { 69, 0, 2156, 0 },
    { 113, 0, 2157, 0 },
    { 117, 0, 2158, 0 },
    { 97, 0, 2159, 0 },
    { 108, 0, 2160, 0 },
    { 59, 0, 0, 8773 }, /* &TildeFullEqual; */
    { 114, 1, 2170, 0 },
    { 105, 1, 3852, 0 },
    { 111, 1, 4586, 0 },
    { 108, 1, 6650, 0 },
    { 115, 1, 7364, 0 },
    { 104, 1, 7444, 0 },
    { 97, 1, 7630, 0 },
    { 99, 1, 8524, 0 },
    { 102, 0, 9830, 0 },
    { 101, 1, 2174, 0 },
    { 111, 1, 3244, 0 },
    { 105, 1, 6614, 0 },
    { 59, 0, 0, 10939 }, /* &Pr; */
    { 99, 0, 2175, 0 },
    { 101, 0, 2176, 0 },
    { 100, 0, 2177, 0 },
    { 101, 0, 2178, 0 },
    { 115, 0, 2179, 0 },
    { 69, 1, 2183, 0 },
    { 59, 1, 0, 8826 }, /* &Precedes; */
    { 84, 1, 7359, 0 },
    { 83, 0, 8599, 0 },
    { 113, 0, 2184, 0 },
    { 117, 0, 2185, 0 },
    { 97, 0, 2186, 0 },
    { 108, 0, 2187, 0 },
    { 59, 0, 0, 10927 }, /* &PrecedesEqual; */
    { 105, 0, 2189, 0 },
    { 108, 0, 2190, 0 },
    { 100, 0, 2191, 0 },
    { 101, 0, 2192, 195 }, /* &Atilde */
    { 59, 0, 0, 195 }, /* &Atilde; */
    { 108, 0, 2194, 0 },
    { 105, 0, 2195, 0 },
    { 112, 0, 2196, 0 },
    { 59, 0, 0, 8230 }, /* &hellip; */
    { 101, 0, 2198, 0 },
    { 115, 0, 2199, 0 },
    { 115, 0, 2200, 0 },
    { 59, 0, 0, 8810 }, /* &NotLessLess; */
    { 111, 0, 2202, 0 },
    { 116, 0, 2203, 0 },
    { 59, 0, 0, 288 }, /* &Gdot; */
    { 101, 1, 2207, 0 },
    { 112, 1, 4610, 0 },
    { 115, 0, 7853, 0 },
    { 97, 0, 2208, 0 },
    { 59, 0, 0, 10536 }, /* &toea; */
    { 114, 0, 2210, 0 },
    { 97, 0, 2211, 0 },
    { 118, 0, 2212, 0 },
    { 101, 0, 2213, 224 }, /* &agrave */
    { 59, 0, 0, 224 }, /* &agrave; */
    { 97, 1, 2217, 0 },
    { 115, 1, 5269, 0 },
    { 108, 0, 6936, 0 },
    { 108, 0, 2218, 0 },
    { 97, 0, 2219, 0 },
    { 114, 0, 2220, 0 },
    { 59, 0, 0, 9006 }, /* &profalar; */
    { 99, 0, 2222, 0 },
    { 105, 0, 2223, 0 },
    { 114, 0, 2224, 0 },
    { 59, 0, 0, 10690 }, /* &cirscir; */
    { 104, 0, 2226, 0 },
    { 116, 0, 2227, 0 },
    { 104, 1, 2233, 0 },
    { 108, 1, 2682, 0 },
    { 114, 1, 4009, 0 },
    { 97, 1, 5575, 0 },
    { 115, 1, 6214, 0 },
    { 116, 0, 6813, 0 },
    { 97, 0, 2234, 0 },
    { 114, 0, 2235, 0 },
    { 112, 0, 2236, 0 },
    { 111, 0, 2237, 0 },
    { 111, 0, 2238, 0 },
    { 110, 0, 2239, 0 },
    { 117, 1, 2241, 0 },
    { 100, 0, 9139, 0 },
    { 112, 0, 2242, 0 },
    { 59, 0, 0, 8640 }, /* &rightharpoonup; */
    { 101, 1, 2245, 0 },
    { 114, 0, 2817, 0 },
    { 101, 0, 2246, 0 },
    { 59, 1, 0, 8867 }, /* &LeftTee; */
    { 86, 1, 6297, 0 },
    { 65, 0, 7963, 0 },
    { 109, 1, 2251, 0 },
    { 97, 0, 5699, 0 },
    { 108, 0, 2252, 252 }, /* &uuml */
    { 59, 0, 0, 252 }, /* &uuml; */
    { 109, 0, 2254, 0 },
    { 105, 1, 2256, 0 },
    { 110, 0, 6955, 0 },
    { 110, 0, 2257, 0 },
    { 117, 0, 2258, 0 },
    { 115, 0, 2259, 0 },
    { 59, 0, 0, 8726 }, /* &setminus; */
    { 103, 1, 2263, 0 },
    { 119, 1, 3006, 0 },
    { 115, 0, 8391, 0 },
    { 97, 0, 2264, 0 },
    { 116, 0, 2265, 0 },
    { 105, 0, 2266, 0 },
    { 118, 0, 2267, 0 },
    { 101, 0, 2268, 0 },
    { 86, 1, 2271, 0 },
    { 77, 1, 6158, 0 },
    { 84, 0, 8464, 0 },
    { 101, 0, 2272, 0 },
    { 114, 0, 2273, 0 },
    { 121, 0, 2274, 0 },
    { 84, 0, 2275, 0 },
    { 104, 0, 2276, 0 },
    { 105, 0, 2277, 0 },
    { 110, 0, 2278, 0 },
    { 83, 0, 2279, 0 },
    { 112, 0, 2280, 0 },
    { 97, 0, 2281, 0 },
    { 99, 0, 2282, 0 },
    { 101, 0, 2283, 0 },
    { 59, 0, 0, 8203 }, /* &NegativeVeryThinSpace; */
    { 117, 0, 2285, 0 },
    { 116, 0, 2286, 0 },
    { 101, 0, 2287, 0 },
    { 59, 0, 0, 313 }, /* &Lacute; */
    { 109, 0, 2289, 0 },
    { 108, 0, 9589, 214 }, /* &Ouml */
    { 114, 0, 2291, 0 },
    { 97, 1, 2302, 0 },
    { 119, 1, 2839, 0 },
    { 115, 1, 3485, 0 },
    { 104, 1, 5781, 0 },
    { 98, 1, 6358, 0 },
    { 59, 1, 0, 8594 }, /* &rarr; */
    { 108, 1, 7218, 0 },
    { 116, 1, 7397, 0 },
    { 102, 1, 7917, 0 },
    { 99, 1, 8050, 0 },
    { 112, 0, 9209, 0 },
    { 112, 0, 2303, 0 },
    { 59, 0, 0, 10613 }, /* &rarrap; */
    { 110, 0, 2305, 0 },
    { 117, 0, 2306, 0 },
    { 115, 0, 2307, 0 },
    { 80, 0, 2308, 0 },
    { 108, 0, 2309, 0 },
    { 117, 0, 2310, 0 },
    { 115, 0, 2311, 0 },
    { 59, 0, 0, 8723 }, /* &MinusPlus; */
    { 59, 0, 0, 9416 }, /* &oS; */
    { 114, 1, 2316, 0 },
    { 98, 1, 6750, 0 },
    { 97, 0, 7594, 0 },
    { 97, 1, 2318, 0 },
    { 107, 0, 3119, 0 },
    { 99, 0, 2319, 0 },
    { 101, 1, 2321, 0 },
    { 107, 0, 2343, 0 },
    { 59, 0, 0, 123 }, /* &lbrace; */
    { 98, 0, 2323, 0 },
    { 97, 0, 2324, 0 },
    { 114, 0, 2325, 0 },
    { 59, 0, 0, 9021 }, /* &ovbar; */
    { 105, 0, 2327, 0 },
    { 109, 0, 2328, 0 },
    { 59, 0, 0, 8821 }, /* &ngsim; */
    { 105, 0, 2330, 0 },
    { 110, 0, 2331, 0 },
    { 103, 0, 2332, 0 },
    { 59, 0, 0, 366 }, /* &Uring; */
    { 99, 1, 2336, 0 },
    { 116, 1, 3640, 0 },
    { 105, 0, 4225, 0 },
    { 111, 1, 2338, 0 },
    { 114, 0, 5057, 0 },
    { 114, 0, 2339, 0 },
    { 110, 0, 2340, 0 },
    { 59, 1, 0, 8989 }, /* &urcorn; */
    { 101, 0, 7722, 0 },
    { 59, 0, 0, 1059 }, /* &Ucy; */
    { 59, 0, 0, 91 }, /* &lbrack; */
    { 99, 0, 2345, 0 },
    { 117, 0, 2346, 0 },
    { 116, 0, 2347, 0 },
    { 101, 0, 2348, 0 },
    { 59, 0, 0, 323 }, /* &Nacute; */
    { 97, 1, 2352, 0 },
    { 121, 1, 3403, 0 },
    { 101, 0, 3974, 0 },
    { 114, 0, 2353, 0 },
    { 111, 0, 2354, 0 },
    { 110, 0, 2355, 0 },
    { 59, 0, 0, 317 }, /* &Lcaron; */
    { 103, 0, 2357, 0 },
    { 100, 1, 2360, 0 },
    { 108, 1, 7133, 0 },
    { 59, 0, 0, 10216 }, /* &lang; */
    { 59, 0, 0, 10641 }, /* &langd; */
    { 110, 0, 2362, 0 },
    { 101, 1, 2364, 0 },
    { 69, 0, 9511, 0 },
    { 59, 0, 0, 8843 }, /* &vsupne; */
    { 109, 1, 2367, 0 },
    { 108, 0, 5179, 0 },
    { 112, 0, 2368, 0 },
    { 69, 1, 2371, 0 },
    { 101, 1, 4897, 0 },
    { 59, 0, 0, 8782 }, /* &bump; */
    { 59, 0, 0, 10926 }, /* &bumpE; */
    { 59, 0, 0, 235 }, /* &euml; */
    { 114, 0, 2374, 0 },
    { 99, 0, 6732, 238 }, /* &icirc */
    { 117, 1, 2380, 0 },
    { 116, 1, 3471, 0 },
    { 119, 1, 5638, 0 },
    { 112, 1, 5819, 0 },
    { 108, 0, 7306, 0 },
    { 98, 0, 2381, 0 },
    { 108, 0, 2382, 0 },
    { 101, 0, 2383, 0 },
    { 98, 0, 2384, 0 },
    { 97, 0, 2385, 0 },
    { 114, 0, 2386, 0 },
    { 119, 0, 2387, 0 },
    { 101, 0, 2388, 0 },
    { 100, 0, 2389, 0 },
    { 103, 0, 2390, 0 },
    { 101, 0, 2391, 0 },
    { 59, 0, 0, 8966 }, /* &doublebarwedge; */
    { 114, 0, 2393, 0 },
    { 101, 0, 2394, 0 },
    { 97, 0, 2395, 0 },
    { 116, 0, 2396, 0 },
    { 101, 0, 2397, 0 },
    { 114, 0, 2398, 0 },
    { 59, 0, 0, 8824 }, /* &NotLessGreater; */
    { 119, 0, 2400, 0 },
    { 97, 0, 2401, 0 },
    { 114, 0, 2402, 0 },
    { 59, 0, 0, 10537 }, /* &seswar; */
    { 117, 1, 2406, 0 },
    { 114, 1, 4590, 0 },
    { 112, 0, 8658, 0 },
    { 114, 0, 2407, 0 },
    { 105, 0, 2408, 0 },
    { 101, 0, 2409, 0 },
    { 114, 0, 2410, 0 },
    { 116, 0, 2411, 0 },
    { 114, 0, 2412, 0 },
    { 102, 0, 2413, 0 },
    { 59, 0, 0, 8497 }, /* &Fouriertrf; */
    { 115, 0, 2415, 0 },
    { 101, 0, 2416, 0 },
    { 116, 0, 2417, 0 },
    { 110, 0, 2418, 0 },
    { 101, 0, 2419, 0 },
    { 113, 0, 2420, 0 },
    { 59, 1, 0, 8842 }, /* &varsubsetneq; */
    { 113, 0, 5730, 0 },
    { 108, 1, 2425, 0 },
    { 105, 1, 7538, 0 },
    { 114, 0, 9354, 0 },
    { 108, 1, 2427, 0 },
    { 105, 0, 7688, 0 },
    { 105, 0, 2428, 0 },
    { 103, 0, 2429, 0 },
    { 59, 0, 0, 64260 }, /* &ffllig; */
    { 112, 0, 2431, 0 },
    { 65, 1, 2433, 0 },
    { 68, 0, 3645, 0 },
    { 114, 0, 2434, 0 },
    { 114, 0, 2435, 0 },
    { 111, 0, 2436, 0 },
    { 119, 0, 2437, 0 },
    { 59, 0, 0, 8657 }, /* &DoubleUpArrow; */
    { 117, 0, 2439, 0 },
    { 98, 1, 2441, 0 },
    { 112, 0, 5588, 0 },
    { 115, 0, 2442, 0 },
    { 101, 0, 2443, 0 },
    { 116, 0, 2444, 0 },
    { 69, 1, 2446, 0 },
    { 59, 0, 0, 8847 }, /* &SquareSubset; */
    { 113, 0, 2447, 0 },
    { 117, 0, 2448, 0 },
    { 97, 0, 2449, 0 },
    { 108, 0, 2450, 0 },
    { 59, 0, 0, 8849 }, /* &SquareSubsetEqual; */
    { 121, 0, 2452, 0 },
    { 59, 0, 0, 1074 }, /* &vcy; */
    { 97, 1, 2461, 0 },
    { 117, 1, 4366, 0 },
    { 111, 1, 4883, 0 },
    { 101, 1, 6337, 0 },
    { 102, 1, 6596, 0 },
    { 99, 1, 7989, 0 },
    { 115, 1, 8327, 0 },
    { 105, 0, 9385, 0 },
    { 99, 0, 2462, 0 },
    { 121, 1, 2464, 0 },
    { 117, 0, 8933, 0 },
    { 59, 0, 0, 1103 }, /* &yacy; */
    { 116, 1, 2468, 0 },
    { 111, 1, 5039, 0 },
    { 114, 0, 6522, 0 },
    { 59, 0, 0, 10689 }, /* &ogt; */
    { 102, 1, 2471, 0 },
    { 97, 0, 2717, 0 },
    { 102, 0, 2472, 0 },
    { 101, 0, 2473, 0 },
    { 114, 0, 2474, 0 },
    { 101, 0, 2475, 0 },
    { 110, 0, 2476, 0 },
    { 116, 0, 2477, 0 },
    { 105, 0, 2478, 0 },
    { 97, 0, 2479, 0 },
    { 108, 0, 2480, 0 },
    { 68, 0, 2481, 0 },
    { 59, 0, 0, 8518 }, /* &DifferentialD; */
    { 99, 1, 2489, 0 },
    { 74, 1, 3827, 0 },
    { 72, 1, 4875, 0 },
    { 115, 1, 7028, 0 },
    { 111, 1, 8150, 0 },
    { 102, 1, 8263, 0 },
    { 97, 0, 8592, 0 },
    { 121, 1, 2491, 0 },
    { 101, 0, 8214, 0 },
    { 59, 0, 0, 1050 }, /* &Kcy; */
    { 97, 0, 2493, 0 },
    { 115, 0, 2494, 0 },
    { 104, 0, 2495, 0 },
    { 59, 0, 0, 8211 }, /* &ndash; */
    { 111, 0, 2497, 0 },
    { 59, 0, 0, 8364 }, /* &euro; */
    { 97, 0, 2499, 0 },
    { 114, 0, 2500, 0 },
    { 59, 0, 0, 8902 }, /* &Star; */
    { 113, 1, 2503, 0 },
    { 116, 0, 5821, 0 },
    { 99, 0, 2504, 0 },
    { 117, 0, 2505, 0 },
    { 112, 0, 2506, 0 },
    { 59, 0, 0, 10758 }, /* &bigsqcup; */
    { 105, 1, 2510, 0 },
    { 100, 1, 7974, 0 },
    { 114, 0, 8484, 0 },
    { 108, 0, 2511, 0 },
    { 100, 0, 2512, 0 },
    { 101, 0, 2513, 0 },
    { 59, 0, 0, 361 }, /* &utilde; */
    { 121, 0, 2515, 0 },
    { 59, 0, 0, 1105 }, /* &iocy; */
    { 119, 0, 2517, 0 },
    { 110, 0, 2518, 0 },
    { 59, 0, 0, 8994 }, /* &frown; */
    { 114, 0, 2520, 0 },
    { 101, 0, 2521, 0 },
    { 99, 0, 2522, 0 },
    { 101, 0, 2523, 0 },
    { 100, 0, 2524, 0 },
    { 101, 0, 2525, 0 },
    { 115, 0, 2526, 0 },
    { 83, 1, 2529, 0 },
    { 59, 1, 0, 8832 }, /* &NotPrecedes; */
    { 69, 0, 6484, 0 },
    { 108, 0, 2530, 0 },
    { 97, 0, 2531, 0 },
    { 110, 0, 2532, 0 },
    { 116, 0, 2533, 0 },
    { 69, 0, 2534, 0 },
    { 113, 0, 2535, 0 },
    { 117, 0, 2536, 0 },
    { 97, 0, 2537, 0 },
    { 108, 0, 2538, 0 },
    { 59, 0, 0, 8928 }, /* &NotPrecedesSlantEqual; */
    { 105, 0, 2540, 0 },
    { 103, 0, 2541, 0 },
    { 104, 0, 2542, 0 },
    { 116, 0, 2543, 0 },
    { 97, 1, 2546, 0 },
    { 104, 1, 6602, 0 },
    { 115, 0, 7048, 0 },
    { 114, 0, 2547, 0 },
    { 114, 0, 2548, 0 },
    { 111, 0, 2549, 0 },
    { 119, 0, 2550, 0 },
    { 115, 1, 2552, 0 },
    { 59, 0, 0, 8596 }, /* &leftrightarrow; */
    { 59, 0, 0, 8646 }, /* &leftrightarrows; */
    { 113, 0, 2554, 0 },
    { 117, 0, 2555, 0 },
    { 97, 0, 2556, 0 },
    { 108, 0, 2557, 0 },
    { 59, 0, 0, 8771 }, /* &TildeEqual; */
    { 118, 1, 2561, 0 },
    { 115, 1, 4583, 0 },
    { 59, 0, 0, 8715 }, /* &ni; */
    { 59, 0, 0, 8715 }, /* &niv; */
    { 108, 0, 2563, 0 },
    { 108, 0, 2564, 0 },
    { 105, 0, 2565, 0 },
    { 110, 0, 2566, 0 },
    { 103, 0, 2567, 0 },
    { 100, 0, 2568, 0 },
    { 111, 0, 2569, 0 },
    { 116, 0, 2570, 0 },
    { 115, 0, 2571, 0 },
    { 101, 0, 2572, 0 },
    { 113, 0, 2573, 0 },
    { 59, 0, 0, 8786 }, /* &fallingdotseq; */
    { 114, 0, 2575, 0 },
    { 111, 0, 2576, 0 },
    { 107, 0, 2577, 0 },
    { 59, 0, 0, 359 }, /* &tstrok; */
    { 103, 1, 2581, 0 },
    { 115, 1, 4268, 0 },
    { 114, 0, 7310, 0 },
    { 103, 0, 2582, 0 },
    { 101, 0, 2583, 0 },
    { 114, 0, 2584, 0 },
    { 59, 0, 0, 8225 }, /* &Dagger; */
    { 97, 1, 2587, 0 },
    { 111, 0, 5508, 0 },
    { 114, 0, 2588, 0 },
    { 117, 1, 2590, 0 },
    { 100, 0, 8038, 0 },
    { 108, 1, 2592, 0 },
    { 59, 0, 0, 8640 }, /* &rharu; */
    { 59, 0, 0, 10604 }, /* &rharul; */
    { 120, 1, 2600, 0 },
    { 97, 1, 2613, 0 },
    { 113, 1, 4930, 0 },
    { 115, 1, 5991, 0 },
    { 59, 1, 0, 8800 }, /* &ne; */
    { 65, 1, 7615, 0 },
    { 100, 0, 9693, 0 },
    { 105, 0, 2601, 0 },
    { 115, 0, 2602, 0 },
    { 116, 0, 2603, 0 },
    { 115, 1, 2605, 0 },
    { 59, 0, 0, 8708 }, /* &nexist; */
    { 59, 0, 0, 8708 }, /* &nexists; */
    { 114, 0, 2607, 0 },
    { 59, 0, 0, 8465 }, /* &Ifr; */
    { 97, 1, 2610, 0 },
    { 112, 0, 3344, 0 },
    { 114, 0, 2611, 0 },
    { 59, 1, 0, 41 }, /* &rpar; */
    { 103, 0, 4982, 0 },
    { 114, 0, 2614, 0 },
    { 114, 1, 2616, 0 },
    { 104, 0, 6853, 0 },
    { 59, 1, 0, 8599 }, /* &nearr; */
    { 111, 0, 2793, 0 },
    { 105, 1, 2626, 0 },
    { 108, 1, 2864, 0 },
    { 113, 1, 3969, 0 },
    { 114, 1, 5047, 0 },
    { 100, 1, 5307, 0 },
    { 99, 1, 5538, 0 },
    { 104, 1, 6293, 0 },
    { 59, 0, 0, 60 }, /* &lt; */
    { 109, 0, 2627, 0 },
    { 101, 0, 2628, 0 },
    { 115, 0, 2629, 0 },
    { 59, 0, 0, 8905 }, /* &ltimes; */
    { 65, 0, 2631, 0 },
    { 68, 0, 2632, 0 },
    { 69, 0, 2633, 0 },
    { 59, 0, 0, 8482 }, /* &TRADE; */
    { 117, 1, 2641, 0 },
    { 105, 1, 3292, 0 },
    { 99, 1, 4953, 0 },
    { 109, 1, 6566, 0 },
    { 112, 1, 7525, 0 },
    { 113, 1, 7663, 0 },
    { 104, 0, 8273, 0 },
    { 112, 1, 2644, 0 },
    { 98, 1, 2736, 0 },
    { 99, 0, 6511, 0 },
    { 59, 1, 0, 8837 }, /* &nsup; */
    { 115, 1, 3489, 0 },
    { 69, 1, 4970, 0 },
    { 101, 0, 7199, 0 },
    { 99, 1, 2653, 0 },
    { 111, 1, 3125, 0 },
    { 102, 1, 3702, 0 },
    { 115, 1, 4493, 0 },
    { 117, 0, 9617, 0 },
    { 121, 1, 2655, 0 },
    { 105, 0, 7439, 0 },
    { 59, 0, 0, 1049 }, /* &Jcy; */
    { 101, 0, 2657, 0 },
    { 114, 0, 2658, 0 },
    { 66, 1, 2660, 0 },
    { 80, 0, 5456, 0 },
    { 114, 1, 2662, 0 },
    { 97, 0, 4557, 0 },
    { 97, 0, 2663, 0 },
    { 99, 0, 2664, 0 },
    { 101, 1, 2666, 0 },
    { 107, 0, 9644, 0 },
    { 59, 0, 0, 9182 }, /* &OverBrace; */
    { 112, 0, 2668, 0 },
    { 97, 0, 2669, 0 },
    { 114, 0, 2670, 0 },
    { 115, 0, 2671, 0 },
    { 108, 0, 2672, 0 },
    { 59, 0, 0, 10724 }, /* &smeparsl; */
    { 121, 1, 2676, 169 }, /* &copy */
    { 102, 1, 5162, 0 },
    { 114, 0, 7998, 0 },
    { 59, 1, 0, 169 }, /* &copy; */
    { 115, 0, 7207, 0 },
    { 97, 0, 2679, 0 },
    { 114, 0, 2680, 0 },
    { 114, 0, 2681, 0 },
    { 59, 0, 0, 10510 }, /* &lBarr; */
    { 101, 0, 2683, 0 },
    { 102, 0, 2684, 0 },
    { 116, 0, 2685, 0 },
    { 104, 1, 2687, 0 },
    { 97, 0, 4801, 0 },
    { 97, 0, 2688, 0 },
    { 114, 0, 2689, 0 },
    { 112, 0, 2690, 0 },
    { 111, 0, 2691, 0 },
    { 111, 0, 2692, 0 },
    { 110, 0, 2693, 0 },
    { 115, 0, 2694, 0 },
    { 59, 0, 0, 8652 }, /* &rightleftharpoons; */
    { 98, 0, 2696, 0 },
    { 108, 0, 2697, 0 },
    { 97, 0, 2698, 0 },
    { 99, 0, 2699, 0 },
    { 59, 0, 0, 336 }, /* &Odblac; */
    { 105, 0, 2701, 0 },
    { 102, 1, 2705, 0 },
    { 108, 1, 4191, 0 },
    { 59, 1, 0, 9657 }, /* &rtri; */
    { 101, 0, 9332, 0 },
    { 59, 0, 0, 9656 }, /* &rtrif; */
    { 110, 1, 2709, 0 },
    { 99, 1, 5405, 0 },
    { 100, 0, 6198, 0 },
    { 117, 0, 2710, 0 },
    { 115, 0, 2711, 0 },
    { 100, 1, 2714, 0 },
    { 59, 1, 0, 8722 }, /* &minus; */
    { 98, 0, 9577, 0 },
    { 117, 1, 2716, 0 },
    { 59, 0, 0, 8760 }, /* &minusd; */
    { 59, 0, 0, 10794 }, /* &minusdu; */
    { 99, 1, 2719, 0 },
    { 109, 0, 7649, 0 },
    { 114, 0, 2720, 0 },
    { 105, 0, 2721, 0 },
    { 116, 0, 2722, 0 },
    { 105, 0, 2723, 0 },
    { 99, 0, 2724, 0 },
    { 97, 0, 2725, 0 },
    { 108, 0, 2726, 0 },
    { 84, 1, 2730, 0 },
    { 71, 1, 5138, 0 },
    { 65, 1, 7981, 0 },
    { 68, 0, 8338, 0 },
    { 105, 0, 2731, 0 },
    { 108, 0, 2732, 0 },
    { 100, 0, 2733, 0 },
    { 101, 0, 2734, 0 },
    { 59, 0, 0, 732 }, /* &DiacriticalTilde; */
    { 59, 0, 0, 10670 }, /* &angmsdag; */
    { 59, 1, 0, 8836 }, /* &nsub; */
    { 115, 1, 4673, 0 },
    { 101, 1, 6826, 0 },
    { 69, 0, 8644, 0 },
    { 59, 0, 0, 62 }, /* &GT; */
    { 115, 1, 2743, 0 },
    { 112, 0, 6212, 0 },
    { 99, 0, 2744, 0 },
    { 59, 0, 0, 8927 }, /* &cuesc; */
    { 100, 1, 2747, 0 },
    { 59, 0, 0, 8839 }, /* &supe; */
    { 111, 0, 2748, 0 },
    { 116, 0, 2749, 0 },
    { 59, 0, 0, 10948 }, /* &supedot; */
    { 108, 0, 2751, 0 },
    { 117, 0, 2752, 0 },
    { 115, 0, 2753, 0 },
    { 59, 0, 0, 10944 }, /* &supplus; */
    { 97, 1, 2757, 0 },
    { 114, 1, 3023, 0 },
    { 98, 0, 8001, 0 },
    { 114, 0, 2758, 0 },
    { 114, 0, 2759, 0 },
    { 59, 0, 0, 10509 }, /* &rbarr; */
    { 114, 0, 2761, 0 },
    { 97, 0, 2762, 0 },
    { 118, 0, 2763, 0 },
    { 101, 0, 9688, 210 }, /* &Ograve */
    { 99, 0, 2765, 0 },
    { 121, 0, 2766, 0 },
    { 59, 0, 0, 1063 }, /* &CHcy; */
    { 97, 1, 2769, 0 },
    { 65, 0, 6965, 0 },
    { 114, 0, 2770, 0 },
    { 114, 0, 2771, 0 },
    { 59, 0, 0, 10231 }, /* &xharr; */
    { 105, 1, 2774, 0 },
    { 97, 0, 8588, 0 },
    { 111, 0, 2775, 0 },
    { 59, 1, 0, 8758 }, /* &ratio; */
    { 110, 0, 7274, 0 },
    { 99, 0, 2778, 0 },
    { 117, 0, 2779, 0 },
    { 116, 0, 2780, 0 },
    { 101, 0, 9127, 211 }, /* &Oacute */
    { 59, 0, 0, 8862 }, /* &plusb; */
    { 99, 1, 2784, 0 },
    { 110, 0, 6532, 0 },
    { 107, 0, 2785, 0 },
    { 115, 1, 2787, 0 },
    { 97, 0, 8841, 0 },
    { 105, 0, 2788, 0 },
    { 109, 0, 2789, 0 },
    { 59, 0, 0, 8764 }, /* &thicksim; */
    { 97, 0, 2791, 0 },
    { 59, 0, 0, 919 }, /* &Eta; */
    { 59, 0, 0, 91 }, /* &lsqb; */
    { 119, 0, 2794, 0 },
    { 59, 0, 0, 8599 }, /* &nearrow; */
    { 115, 0, 2796, 0 },
    { 101, 1, 2798, 0 },
    { 119, 0, 7243, 0 },
    { 97, 0, 2799, 0 },
    { 114, 0, 2800, 0 },
    { 111, 0, 2801, 0 },
    { 119, 0, 2802, 0 },
    { 59, 0, 0, 10533 }, /* &hksearow; */
    { 114, 0, 2804, 0 },
    { 99, 0, 2805, 0 },
    { 108, 0, 2806, 0 },
    { 101, 0, 2807, 0 },
    { 68, 1, 2811, 0 },
    { 84, 1, 3957, 0 },
    { 80, 1, 4821, 0 },
    { 77, 0, 4984, 0 },
    { 111, 0, 2812, 0 },
    { 116, 0, 2813, 0 },
    { 59, 0, 0, 8857 }, /* &CircleDot; */
    { 105, 0, 2815, 0 },
    { 109, 0, 2816, 0 },
    { 59, 0, 0, 10611 }, /* &larrsim; */
    { 105, 0, 2818, 0 },
    { 97, 0, 2819, 0 },
    { 110, 0, 2820, 0 },
    { 103, 0, 2821, 0 },
    { 108, 0, 2822, 0 },
    { 101, 0, 2823, 0 },
    { 59, 1, 0, 8882 }, /* &LeftTriangle; */
    { 69, 1, 3091, 0 },
    { 66, 0, 7145, 0 },
    { 115, 1, 2829, 0 },
    { 117, 1, 2965, 0 },
    { 99, 0, 5114, 0 },
    { 117, 0, 2830, 0 },
    { 112, 1, 2832, 0 },
    { 98, 0, 7368, 0 },
    { 115, 1, 2835, 0 },
    { 59, 1, 0, 8848 }, /* &sqsup; */
    { 101, 0, 4565, 0 },
    { 101, 0, 2836, 0 },
    { 116, 0, 2837, 0 },
    { 59, 1, 0, 8848 }, /* &sqsupset; */
    { 101, 0, 3469, 0 },
    { 59, 0, 0, 8605 }, /* &rarrw; */
    { 99, 0, 2841, 0 },
    { 117, 0, 2842, 0 },
    { 116, 0, 2843, 0 },
    { 101, 0, 9433, 221 }, /* &Yacute */
    { 108, 0, 2845, 0 },
    { 117, 0, 2846, 0 },
    { 104, 0, 2847, 0 },
    { 97, 0, 2848, 0 },
    { 114, 0, 2849, 0 },
    { 59, 0, 0, 10600 }, /* &ruluhar; */
    { 114, 0, 2851, 0 },
    { 111, 0, 2852, 0 },
    { 112, 0, 2853, 0 },
    { 59, 0, 0, 8733 }, /* &vprop; */
    { 99, 1, 2856, 0 },
    { 105, 0, 8801, 0 },
    { 114, 0, 2857, 0 },
    { 59, 0, 0, 120005 }, /* &pscr; */
    { 114, 0, 2859, 0 },
    { 99, 0, 2860, 0 },
    { 59, 0, 0, 349 }, /* &scirc; */
    { 109, 0, 2862, 0 },
    { 108, 0, 2863, 196 }, /* &Auml */
    { 59, 0, 0, 196 }, /* &Auml; */
    { 97, 0, 2865, 0 },
    { 114, 0, 2866, 0 },
    { 114, 0, 2867, 0 },
    { 59, 0, 0, 10614 }, /* &ltlarr; */
    { 59, 0, 0, 177 }, /* &pm; */
    { 121, 1, 2872, 0 },
    { 97, 1, 6133, 0 },
    { 105, 0, 8738, 0 },
    { 59, 0, 0, 1069 }, /* &Ecy; */
    { 109, 1, 2877, 0 },
    { 105, 1, 4105, 0 },
    { 114, 1, 4410, 0 },
    { 108, 0, 4788, 0 },
    { 105, 0, 2878, 0 },
    { 108, 0, 2879, 0 },
    { 116, 0, 2880, 0 },
    { 59, 0, 0, 8459 }, /* &hamilt; */
    { 59, 0, 0, 10931 }, /* &prE; */
    { 117, 1, 2888, 0 },
    { 121, 1, 6233, 0 },
    { 69, 1, 7480, 0 },
    { 59, 1, 0, 8766 }, /* &ac; */
    { 105, 1, 9018, 0 },
    { 100, 0, 9414, 0 },
    { 116, 0, 2889, 0 },
    { 101, 0, 2890, 180 }, /* &acute */
    { 59, 0, 0, 180 }, /* &acute; */
    { 105, 0, 2892, 0 },
    { 110, 0, 2893, 0 },
    { 117, 0, 2894, 0 },
    { 115, 0, 2895, 0 },
    { 59, 0, 0, 8863 }, /* &boxminus; */
    { 108, 0, 2897, 0 },
    { 97, 0, 2898, 0 },
    { 110, 0, 2899, 0 },
    { 116, 0, 2900, 0 },
    { 69, 0, 2901, 0 },
    { 113, 0, 2902, 0 },
    { 117, 0, 2903, 0 },
    { 97, 0, 2904, 0 },
    { 108, 0, 2905, 0 },
    { 59, 0, 0, 8829 }, /* &SucceedsSlantEqual; */
    { 65, 1, 2909, 0 },
    { 116, 1, 6014, 0 },
    { 101, 0, 8153, 0 },
    { 114, 0, 2910, 0 },
    { 114, 0, 2911, 0 },
    { 59, 0, 0, 10498 }, /* &nvlArr; */
    { 112, 0, 2913, 0 },
    { 97, 0, 2914, 0 },
    { 114, 0, 2915, 0 },
    { 114, 0, 2916, 0 },
    { 111, 0, 2917, 0 },
    { 119, 0, 2918, 0 },
    { 108, 1, 2920, 0 },
    { 114, 0, 8412, 0 },
    { 101, 0, 2921, 0 },
    { 102, 0, 2922, 0 },
    { 116, 0, 2923, 0 },
    { 59, 0, 0, 8619 }, /* &looparrowleft; */
    { 99, 1, 2926, 0 },
    { 59, 0, 0, 10927 }, /* &pre; */
    { 99, 1, 2932, 0 },
    { 59, 1, 0, 8826 }, /* &prec; */
    { 97, 1, 6576, 0 },
    { 101, 1, 7686, 0 },
    { 110, 1, 7730, 0 },
    { 115, 0, 8638, 0 },
    { 117, 0, 2933, 0 },
    { 114, 0, 2934, 0 },
    { 108, 0, 2935, 0 },
    { 121, 0, 2936, 0 },
    { 101, 0, 2937, 0 },
    { 113, 0, 2938, 0 },
    { 59, 0, 0, 8828 }, /* &preccurlyeq; */
    { 114, 0, 2940, 0 },
    { 59, 0, 0, 120068 }, /* &Afr; */
    { 100, 1, 2948, 0 },
    { 115, 1, 3065, 0 },
    { 118, 1, 4323, 0 },
    { 105, 1, 4940, 0 },
    { 59, 1, 0, 8744 }, /* &or; */
    { 97, 1, 7409, 0 },
    { 111, 0, 8064, 0 },
    { 101, 1, 2952, 0 },
    { 102, 1, 6060, 170 }, /* &ordf */
    { 109, 1, 8646, 186 }, /* &ordm */
    { 59, 0, 0, 10845 }, /* &ord; */
    { 114, 0, 2953, 0 },
    { 111, 1, 2955, 0 },
    { 59, 0, 0, 8500 }, /* &order; */
    { 102, 0, 2956, 0 },
    { 59, 0, 0, 8500 }, /* &orderof; */
    { 117, 0, 2958, 0 },
    { 98, 0, 2959, 0 },
    { 115, 0, 2960, 0 },
    { 117, 1, 2962, 0 },
    { 59, 0, 0, 9827 }, /* &clubs; */
    { 105, 0, 2963, 0 },
    { 116, 0, 2964, 0 },
    { 59, 0, 0, 9827 }, /* &clubsuit; */
    { 97, 1, 2968, 0 },
    { 59, 1, 0, 9633 }, /* &squ; */
    { 102, 0, 8203, 0 },
    { 114, 0, 2969, 0 },
    { 101, 1, 2971, 0 },
    { 102, 0, 9452, 0 },
    { 59, 0, 0, 9633 }, /* &square; */
    { 118, 0, 2973, 0 },
    { 101, 0, 2974, 0 },
    { 59, 0, 0, 364 }, /* &Ubreve; */
    { 100, 1, 2980, 0 },
    { 99, 1, 4536, 0 },
    { 115, 1, 6828, 0 },
    { 103, 1, 6859, 0 },
    { 59, 0, 0, 10877 }, /* &les; */
    { 111, 0, 2981, 0 },
    { 116, 0, 2982, 0 },
    { 111, 1, 2984, 0 },
    { 59, 0, 0, 10879 }, /* &lesdot; */
    { 59, 1, 0, 10881 }, /* &lesdoto; */
    { 114, 0, 3634, 0 },
    { 114, 0, 2987, 0 },
    { 115, 1, 2991, 0 },
    { 59, 1, 0, 8742 }, /* &npar; */
    { 97, 1, 5952, 0 },
    { 116, 0, 9607, 0 },
    { 108, 0, 2992, 0 },
    { 59, 0, 0, 11005 }, /* &nparsl; */
    { 116, 1, 2997, 0 },
    { 101, 1, 4688, 0 },
    { 99, 1, 5768, 0 },
    { 109, 0, 8805, 0 },
    { 97, 0, 2998, 0 },
    { 114, 0, 2999, 0 },
    { 102, 0, 3000, 0 },
    { 59, 0, 0, 8902 }, /* &sstarf; */
    { 76, 1, 3005, 0 },
    { 114, 1, 4008, 0 },
    { 108, 1, 5574, 0 },
    { 82, 0, 9733, 0 },
    { 59, 0, 0, 9563 }, /* &boxuL; */
    { 76, 0, 3007, 0 },
    { 105, 0, 3008, 0 },
    { 110, 0, 3009, 0 },
    { 101, 0, 3010, 0 },
    { 59, 0, 0, 10 }, /* &NewLine; */
    { 114, 1, 3015, 0 },
    { 82, 1, 5737, 0 },
    { 108, 1, 8078, 0 },
    { 76, 0, 9809, 0 },
    { 59, 0, 0, 9484 }, /* &boxdr; */
    { 114, 0, 3017, 0 },
    { 59, 0, 0, 8466 }, /* &Lscr; */
    { 114, 0, 3019, 0 },
    { 97, 0, 3020, 0 },
    { 118, 0, 3021, 0 },
    { 101, 0, 3022, 236 }, /* &igrave */
    { 59, 0, 0, 236 }, /* &igrave; */
    { 107, 1, 3025, 0 },
    { 97, 0, 8797, 0 },
    { 115, 1, 3027, 0 },
    { 101, 0, 7530, 0 },
    { 108, 0, 3028, 0 },
    { 117, 1, 3030, 0 },
    { 100, 0, 3100, 0 },
    { 59, 0, 0, 10640 }, /* &rbrkslu; */
    { 116, 1, 3033, 0 },
    { 114, 0, 6782, 0 },
    { 97, 0, 3034, 0 },
    { 59, 0, 0, 918 }, /* &Zeta; */
    { 101, 1, 3042, 0 },
    { 111, 1, 4328, 0 },
    { 102, 1, 5163, 0 },
    { 99, 1, 6254, 0 },
    { 114, 1, 7068, 0 },
    { 112, 1, 7344, 0 },
    { 115, 0, 8993, 0 },
    { 100, 1, 3044, 0 },
    { 105, 0, 8648, 0 },
    { 103, 1, 3046, 0 },
    { 98, 0, 9249, 0 },
    { 101, 0, 3047, 0 },
    { 59, 1, 0, 8743 }, /* &wedge; */
    { 113, 0, 9571, 0 },
    { 114, 1, 3051, 0 },
    { 112, 0, 7563, 0 },
    { 114, 0, 3052, 0 },
    { 59, 0, 0, 10616 }, /* &gtrarr; */
    { 105, 1, 3055, 0 },
    { 59, 0, 0, 8214 }, /* &Vert; */
    { 99, 0, 3056, 0 },
    { 97, 0, 3057, 0 },
    { 108, 0, 3058, 0 },
    { 66, 1, 3062, 0 },
    { 76, 1, 5265, 0 },
    { 84, 1, 6922, 0 },
    { 83, 0, 7388, 0 },
    { 97, 0, 3063, 0 },
    { 114, 0, 3064, 0 },
    { 59, 0, 0, 8739 }, /* &VerticalBar; */
    { 108, 0, 3066, 0 },
    { 111, 0, 3067, 0 },
    { 112, 0, 3068, 0 },
    { 101, 0, 3069, 0 },
    { 59, 0, 0, 10839 }, /* &orslope; */
    { 114, 0, 3071, 0 },
    { 97, 0, 3072, 0 },
    { 118, 0, 3073, 0 },
    { 101, 0, 3074, 204 }, /* &Igrave */
    { 59, 0, 0, 204 }, /* &Igrave; */
    { 110, 0, 3076, 0 },
    { 102, 0, 3077, 0 },
    { 105, 0, 3078, 0 },
    { 110, 0, 3079, 0 },
    { 59, 0, 0, 10718 }, /* &nvinfin; */
    { 118, 1, 3082, 0 },
    { 101, 0, 9421, 0 },
    { 98, 0, 3083, 0 },
    { 97, 0, 3084, 0 },
    { 114, 0, 3085, 166 }, /* &brvbar */
    { 59, 0, 0, 166 }, /* &brvbar; */
    { 99, 1, 3088, 0 },
    { 114, 0, 6858, 0 },
    { 105, 0, 3089, 0 },
    { 114, 0, 3090, 0 },
    { 59, 0, 0, 10687 }, /* &ofcir; */
    { 113, 0, 3092, 0 },
    { 117, 0, 3093, 0 },
    { 97, 0, 3094, 0 },
    { 108, 0, 3095, 0 },
    { 59, 0, 0, 8884 }, /* &LeftTriangleEqual; */
    { 114, 0, 3097, 0 },
    { 59, 0, 0, 119964 }, /* &Ascr; */
    { 108, 0, 3099, 0 },
    { 59, 0, 0, 8825 }, /* &ntgl; */
    { 59, 0, 0, 10638 }, /* &rbrksld; */
    { 115, 0, 3102, 0 },
    { 71, 1, 3108, 0 },
    { 70, 1, 4546, 0 },
    { 76, 1, 5504, 0 },
    { 69, 1, 8534, 0 },
    { 84, 1, 8754, 0 },
    { 83, 0, 9561, 0 },
    { 114, 0, 3109, 0 },
    { 101, 0, 3110, 0 },
    { 97, 0, 3111, 0 },
    { 116, 0, 3112, 0 },
    { 101, 0, 3113, 0 },
    { 114, 0, 3114, 0 },
    { 59, 0, 0, 8822 }, /* &LessGreater; */
    { 97, 0, 3116, 0 },
    { 115, 0, 3117, 0 },
    { 104, 0, 3118, 0 },
    { 59, 0, 0, 8872 }, /* &vDash; */
    { 115, 1, 3121, 0 },
    { 101, 0, 8014, 0 },
    { 108, 0, 3122, 0 },
    { 117, 1, 3124, 0 },
    { 100, 0, 9597, 0 },
    { 59, 0, 0, 10637 }, /* &lbrkslu; */
    { 112, 0, 3126, 0 },
    { 102, 0, 3127, 0 },
    { 59, 0, 0, 120129 }, /* &Jopf; */
    { 97, 1, 3131, 0 },
    { 65, 1, 6155, 0 },
    { 110, 0, 9085, 0 },
    { 114, 0, 3132, 0 },
    { 104, 1, 3134, 0 },
    { 114, 0, 8498, 0 },
    { 107, 0, 3135, 0 },
    { 59, 0, 0, 10534 }, /* &swarhk; */
    { 111, 0, 3137, 0 },
    { 119, 0, 3138, 0 },
    { 110, 0, 3139, 0 },
    { 59, 0, 0, 8615 }, /* &mapstodown; */
    { 105, 0, 3141, 0 },
    { 115, 0, 3142, 0 },
    { 59, 0, 0, 8955 }, /* &xnis; */
    { 108, 0, 3144, 0 },
    { 97, 0, 3145, 0 },
    { 110, 0, 3146, 0 },
    { 116, 0, 3147, 0 },
    { 69, 0, 3148, 0 },
    { 113, 0, 3149, 0 },
    { 117, 0, 3150, 0 },
    { 97, 0, 3151, 0 },
    { 108, 0, 3152, 0 },
    { 59, 0, 0, 10878 }, /* &GreaterSlantEqual; */
    { 59, 1, 0, 8817 }, /* &nge; */
    { 113, 1, 3168, 0 },
    { 115, 0, 6362, 0 },
    { 68, 1, 3161, 0 },
    { 117, 1, 3944, 0 },
    { 100, 1, 6971, 0 },
    { 59, 1, 0, 9472 }, /* &boxh; */
    { 85, 0, 9689, 0 },
    { 59, 0, 0, 9573 }, /* &boxhD; */
    { 102, 1, 3166, 0 },
    { 105, 1, 7448, 0 },
    { 111, 1, 8172, 0 },
    { 115, 0, 9273, 0 },
    { 114, 0, 3167, 0 },
    { 59, 0, 0, 120091 }, /* &Xfr; */
    { 115, 1, 3171, 0 },
    { 59, 1, 0, 8817 }, /* &ngeq; */
    { 113, 0, 7094, 0 },
    { 108, 0, 3172, 0 },
    { 97, 0, 3173, 0 },
    { 110, 0, 3174, 0 },
    { 116, 0, 3175, 0 },
    { 59, 0, 0, 10878 }, /* &ngeqslant; */
    { 121, 0, 3177, 0 },
    { 59, 0, 0, 1052 }, /* &Mcy; */
    { 109, 0, 3179, 0 },
    { 108, 0, 3180, 246 }, /* &ouml */
    { 59, 0, 0, 246 }, /* &ouml; */
    { 101, 1, 3183, 0 },
    { 114, 0, 8051, 0 },
    { 101, 0, 3184, 0 },
    { 59, 1, 0, 8866 }, /* &RightTee; */
    { 65, 1, 4958, 0 },
    { 86, 0, 8128, 0 },
    { 118, 1, 3192, 0 },
    { 97, 1, 3322, 0 },
    { 103, 1, 4906, 0 },
    { 115, 1, 6318, 0 },
    { 101, 0, 9752, 0 },
    { 105, 1, 3195, 0 },
    { 111, 1, 4816, 0 },
    { 59, 0, 0, 247 }, /* &div; */
    { 100, 0, 3196, 0 },
    { 101, 0, 3197, 247 }, /* &divide */
    { 111, 1, 3199, 0 },
    { 59, 0, 0, 247 }, /* &divide; */
    { 110, 0, 3200, 0 },
    { 116, 0, 3201, 0 },
    { 105, 0, 3202, 0 },
    { 109, 0, 3203, 0 },
    { 101, 0, 3204, 0 },
    { 115, 0, 3205, 0 },
    { 59, 0, 0, 8903 }, /* &divideontimes; */
    { 113, 0, 3207, 0 },
    { 117, 0, 3208, 0 },
    { 97, 0, 3209, 0 },
    { 108, 0, 3210, 0 },
    { 59, 0, 0, 8816 }, /* &NotLessEqual; */
    { 114, 1, 3213, 0 },
    { 97, 0, 6998, 0 },
    { 97, 1, 3215, 0 },
    { 110, 0, 9016, 0 },
    { 105, 0, 3216, 0 },
    { 103, 0, 3217, 0 },
    { 104, 0, 3218, 0 },
    { 116, 0, 3219, 0 },
    { 101, 1, 3221, 0 },
    { 112, 0, 4499, 0 },
    { 112, 0, 3222, 0 },
    { 115, 0, 3223, 0 },
    { 105, 0, 3224, 0 },
    { 108, 0, 3225, 0 },
    { 111, 0, 3226, 0 },
    { 110, 0, 3227, 0 },
    { 59, 0, 0, 1013 }, /* &straightepsilon; */
    { 59, 1, 0, 10922 }, /* &smt; */
    { 101, 0, 5618, 0 },
    { 114, 0, 3231, 0 },
    { 59, 0, 0, 120081 }, /* &Nfr; */
    { 99, 1, 3235, 0 },
    { 109, 1, 5602, 0 },
    { 112, 0, 9453, 0 },
    { 117, 0, 3236, 0 },
    { 116, 0, 3237, 0 },
    { 101, 0, 3238, 0 },
    { 59, 0, 0, 501 }, /* &gacute; */
    { 108, 1, 3241, 0 },
    { 99, 0, 8586, 0 },
    { 97, 0, 3242, 0 },
    { 115, 0, 3243, 0 },
    { 104, 0, 5865, 216 }, /* &Oslash */
    { 100, 1, 3246, 0 },
    { 112, 0, 5289, 0 },
    { 117, 0, 3247, 0 },
    { 99, 0, 3248, 0 },
    { 116, 0, 3249, 0 },
    { 59, 0, 0, 8719 }, /* &Product; */
    { 101, 1, 3253, 0 },
    { 117, 1, 3503, 0 },
    { 105, 0, 7613, 0 },
    { 116, 0, 3254, 0 },
    { 110, 1, 3257, 0 },
    { 101, 1, 3499, 0 },
    { 59, 0, 0, 8834 }, /* &subset; */
    { 101, 0, 3258, 0 },
    { 113, 0, 3259, 0 },
    { 59, 1, 0, 8842 }, /* &subsetneq; */
    { 113, 0, 8840, 0 },
    { 97, 1, 3263, 0 },
    { 98, 0, 6855, 0 },
    { 114, 0, 3264, 0 },
    { 117, 1, 3266, 0 },
    { 100, 0, 5731, 0 },
    { 108, 1, 3268, 0 },
    { 59, 0, 0, 8636 }, /* &lharu; */
    { 59, 0, 0, 10602 }, /* &lharul; */
    { 68, 1, 3271, 0 },
    { 97, 0, 8452, 0 },
    { 111, 0, 3272, 0 },
    { 116, 0, 3273, 0 },
    { 59, 0, 0, 8787 }, /* &erDot; */
    { 112, 0, 3275, 0 },
    { 68, 1, 3278, 0 },
    { 84, 1, 4741, 0 },
    { 86, 0, 7546, 0 },
    { 111, 0, 3279, 0 },
    { 119, 0, 3280, 0 },
    { 110, 0, 3281, 0 },
    { 86, 0, 3282, 0 },
    { 101, 0, 3283, 0 },
    { 99, 0, 3284, 0 },
    { 116, 0, 3285, 0 },
    { 111, 0, 3286, 0 },
    { 114, 0, 3287, 0 },
    { 59, 0, 0, 10575 }, /* &RightUpDownVector; */
    { 111, 1, 3290, 0 },
    { 115, 0, 3380, 0 },
    { 116, 0, 3291, 0 },
    { 59, 0, 0, 10942 }, /* &supdot; */
    { 109, 0, 3293, 0 },
    { 101, 1, 3295, 0 },
    { 59, 0, 0, 8769 }, /* &nsim; */
    { 59, 1, 0, 8772 }, /* &nsime; */
    { 113, 0, 8119, 0 },
    { 99, 1, 3299, 0 },
    { 100, 0, 6940, 0 },
    { 112, 0, 3300, 0 },
    { 59, 0, 0, 10971 }, /* &mlcp; */
    { 99, 0, 3302, 0 },
    { 117, 0, 3303, 0 },
    { 116, 0, 3304, 0 },
    { 101, 0, 3305, 201 }, /* &Eacute */
    { 59, 0, 0, 201 }, /* &Eacute; */
    { 109, 1, 3309, 0 },
    { 108, 1, 3319, 0 },
    { 103, 0, 5497, 176 }, /* &deg */
    { 112, 0, 3310, 0 },
    { 116, 0, 3311, 0 },
    { 121, 0, 3312, 0 },
    { 118, 0, 3313, 0 },
    { 59, 0, 0, 10673 }, /* &demptyv; */
    { 105, 0, 3315, 0 },
    { 108, 0, 3316, 0 },
    { 100, 0, 3317, 0 },
    { 101, 0, 3318, 0 },
    { 59, 0, 0, 8776 }, /* &TildeTilde; */
    { 116, 0, 3320, 0 },
    { 97, 0, 3321, 0 },
    { 59, 0, 0, 948 }, /* &delta; */
    { 109, 0, 3323, 0 },
    { 111, 1, 3326, 0 },
    { 115, 1, 4864, 0 },
    { 59, 0, 0, 8900 }, /* &diam; */
    { 110, 0, 3327, 0 },
    { 100, 0, 3328, 0 },
    { 115, 1, 3330, 0 },
    { 59, 0, 0, 8900 }, /* &diamond; */
    { 117, 0, 3331, 0 },
    { 105, 0, 3332, 0 },
    { 116, 0, 3333, 0 },
    { 59, 0, 0, 9830 }, /* &diamondsuit; */
    { 105, 0, 3335, 0 },
    { 108, 0, 3336, 0 },
    { 100, 0, 3337, 0 },
    { 101, 0, 3338, 0 },
    { 59, 0, 0, 296 }, /* &Itilde; */
    { 59, 0, 0, 10949 }, /* &subE; */
    { 117, 0, 3341, 0 },
    { 114, 0, 3342, 0 },
    { 59, 1, 0, 9838 }, /* &natur; */
    { 97, 0, 7847, 0 },
    { 111, 0, 3345, 0 },
    { 108, 0, 3346, 0 },
    { 105, 0, 3347, 0 },
    { 110, 0, 3348, 0 },
    { 116, 0, 3349, 0 },
    { 59, 0, 0, 10770 }, /* &rppolint; */
    { 112, 1, 3352, 0 },
    { 100, 0, 9468, 0 },
    { 102, 0, 3353, 0 },
    { 59, 0, 0, 120158 }, /* &mopf; */
    { 111, 0, 3355, 0 },
    { 117, 1, 3357, 0 },
    { 119, 0, 6410, 0 },
    { 98, 0, 3358, 0 },
    { 108, 0, 3359, 0 },
    { 101, 0, 3360, 0 },
    { 66, 0, 3361, 0 },
    { 114, 0, 3362, 0 },
    { 97, 0, 3363, 0 },
    { 99, 0, 3364, 0 },
    { 107, 0, 3365, 0 },
    { 101, 0, 3366, 0 },
    { 116, 0, 3367, 0 },
    { 59, 0, 0, 10215 }, /* &RightDoubleBracket; */
    { 101, 1, 3371, 0 },
    { 121, 1, 3383, 0 },
    { 105, 0, 9118, 0 },
    { 100, 0, 3372, 0 },
    { 105, 0, 3373, 0 },
    { 108, 0, 3374, 0 },
    { 59, 0, 0, 290 }, /* &Gcedil; */
    { 111, 1, 3377, 0 },
    { 121, 0, 9512, 0 },
    { 110, 0, 3378, 0 },
    { 103, 0, 3379, 0 },
    { 59, 0, 0, 8780 }, /* &bcong; */
    { 117, 0, 3381, 0 },
    { 98, 0, 3382, 0 },
    { 59, 0, 0, 10968 }, /* &supdsub; */
    { 59, 0, 0, 1043 }, /* &Gcy; */
    { 101, 1, 3386, 0 },
    { 116, 0, 6076, 0 },
    { 116, 0, 3387, 0 },
    { 114, 0, 3388, 0 },
    { 102, 0, 3389, 0 },
    { 59, 0, 0, 8488 }, /* &zeetrf; */
    { 100, 1, 3396, 0 },
    { 98, 1, 3559, 0 },
    { 99, 1, 4606, 0 },
    { 59, 1, 0, 8746 }, /* &cup; */
    { 111, 1, 6906, 0 },
    { 115, 0, 9227, 0 },
    { 111, 0, 3397, 0 },
    { 116, 0, 3398, 0 },
    { 59, 0, 0, 8845 }, /* &cupdot; */
    { 59, 0, 0, 8477 }, /* &reals; */
    { 115, 0, 3401, 0 },
    { 105, 0, 3402, 0 },
    { 59, 0, 0, 1014 }, /* &bepsi; */
    { 59, 0, 0, 1051 }, /* &Lcy; */
    { 99, 0, 3405, 0 },
    { 114, 0, 3406, 0 },
    { 59, 0, 0, 8464 }, /* &Iscr; */
    { 99, 0, 3408, 0 },
    { 121, 0, 3409, 0 },
    { 59, 0, 0, 1114 }, /* &njcy; */
    { 115, 1, 3414, 0 },
    { 59, 1, 0, 8805 }, /* &ge; */
    { 108, 1, 6693, 0 },
    { 113, 0, 6737, 0 },
    { 108, 1, 3418, 0 },
    { 59, 1, 0, 10878 }, /* &ges; */
    { 99, 1, 8117, 0 },
    { 100, 0, 8502, 0 },
    { 59, 1, 0, 8923 }, /* &gesl; */
    { 101, 0, 5960, 0 },
    { 101, 0, 3421, 0 },
    { 114, 0, 3422, 0 },
    { 115, 0, 3423, 0 },
    { 101, 0, 3424, 0 },
    { 116, 0, 3425, 0 },
    { 69, 1, 3427, 0 },
    { 59, 0, 0, 8835 }, /* &NotSuperset; */
    { 113, 0, 3428, 0 },
    { 117, 0, 3429, 0 },
    { 97, 0, 3430, 0 },
    { 108, 0, 3431, 0 },
    { 59, 0, 0, 8841 }, /* &NotSupersetEqual; */
    { 107, 0, 3433, 0 },
    { 99, 1, 3437, 0 },
    { 115, 1, 3775, 0 },
    { 112, 1, 6038, 0 },
    { 101, 0, 8320, 0 },
    { 111, 0, 3438, 0 },
    { 110, 0, 3439, 0 },
    { 103, 0, 3440, 0 },
    { 59, 0, 0, 8780 }, /* &backcong; */
    { 114, 0, 3442, 0 },
    { 110, 0, 7573, 254 }, /* &thorn */
    { 54, 1, 3445, 0 },
    { 56, 0, 3468, 0 },
    { 59, 0, 0, 8538 }, /* &frac56; */
    { 115, 1, 3450, 0 },
    { 101, 1, 4480, 0 },
    { 97, 1, 5098, 0 },
    { 69, 0, 8067, 0 },
    { 105, 0, 3451, 0 },
    { 109, 0, 3452, 0 },
    { 59, 0, 0, 8934 }, /* &lnsim; */
    { 109, 0, 3454, 0 },
    { 112, 0, 3455, 0 },
    { 68, 1, 3457, 0 },
    { 69, 0, 7712, 0 },
    { 111, 0, 3458, 0 },
    { 119, 0, 3459, 0 },
    { 110, 0, 3460, 0 },
    { 72, 0, 3461, 0 },
    { 117, 0, 3462, 0 },
    { 109, 0, 3463, 0 },
    { 112, 0, 3464, 0 },
    { 59, 0, 0, 8782 }, /* &HumpDownHump; */
    { 99, 0, 3466, 0 },
    { 121, 0, 3467, 0 },
    { 59, 0, 0, 1093 }, /* &khcy; */
    { 59, 0, 0, 8541 }, /* &frac58; */
    { 113, 0, 3470, 0 },
    { 59, 0, 0, 8850 }, /* &sqsupseteq; */
    { 112, 1, 3476, 0 },
    { 101, 1, 6081, 0 },
    { 59, 1, 0, 729 }, /* &dot; */
    { 115, 1, 7811, 0 },
    { 109, 0, 8356, 0 },
    { 108, 0, 3477, 0 },
    { 117, 0, 3478, 0 },
    { 115, 0, 3479, 0 },
    { 59, 0, 0, 8724 }, /* &dotplus; */
    { 109, 0, 3481, 0 },
    { 112, 0, 3482, 0 },
    { 101, 0, 3483, 0 },
    { 113, 0, 3484, 0 },
    { 59, 0, 0, 8782 }, /* &Bumpeq; */
    { 105, 0, 3486, 0 },
    { 109, 0, 3487, 0 },
    { 59, 0, 0, 10612 }, /* &rarrsim; */
    { 59, 0, 0, 978 }, /* &upsih; */
    { 101, 0, 3490, 0 },
    { 116, 0, 3491, 0 },
    { 101, 1, 3493, 0 },
    { 59, 0, 0, 8835 }, /* &nsupset; */
    { 113, 0, 3494, 0 },
    { 113, 1, 3496, 0 },
    { 59, 0, 0, 8841 }, /* &nsupseteq; */
    { 59, 0, 0, 10950 }, /* &nsupseteqq; */
    { 114, 0, 3498, 0 },
    { 59, 0, 0, 10630 }, /* &ropar; */
    { 113, 0, 3500, 0 },
    { 113, 1, 3502, 0 },
    { 59, 0, 0, 8838 }, /* &subseteq; */
    { 59, 0, 0, 10949 }, /* &subseteqq; */
    { 112, 1, 3505, 0 },
    { 98, 0, 3538, 0 },
    { 59, 0, 0, 10963 }, /* &subsup; */
    { 121, 0, 3507, 0 },
    { 59, 0, 0, 1060 }, /* &Fcy; */
    { 99, 0, 3509, 0 },
    { 114, 0, 3510, 0 },
    { 59, 0, 0, 120010 }, /* &uscr; */
    { 99, 1, 3513, 0 },
    { 110, 0, 4205, 0 },
    { 107, 0, 3514, 0 },
    { 108, 1, 3517, 0 },
    { 115, 1, 4699, 0 },
    { 116, 0, 5274, 0 },
    { 111, 0, 3518, 0 },
    { 122, 0, 3519, 0 },
    { 101, 0, 3520, 0 },
    { 110, 0, 3521, 0 },
    { 103, 0, 3522, 0 },
    { 101, 0, 3523, 0 },
    { 59, 0, 0, 10731 }, /* &blacklozenge; */
    { 113, 0, 3525, 0 },
    { 117, 0, 3526, 0 },
    { 97, 0, 3527, 0 },
    { 108, 0, 3528, 0 },
    { 59, 0, 0, 10928 }, /* &SucceedsEqual; */
    { 100, 1, 3532, 0 },
    { 109, 1, 4570, 0 },
    { 110, 0, 6428, 0 },
    { 105, 0, 3533, 0 },
    { 108, 0, 5124, 184 }, /* &cedil */
    { 109, 0, 3535, 0 },
    { 101, 0, 3536, 0 },
    { 108, 0, 3537, 0 },
    { 59, 0, 0, 8503 }, /* &gimel; */
    { 59, 0, 0, 10965 }, /* &subsub; */
    { 116, 0, 7454, 167 }, /* &sect */
    { 109, 1, 3543, 0 },
    { 97, 1, 6094, 0 },
    { 104, 0, 6446, 0 },
    { 59, 0, 0, 8207 }, /* &rlm; */
    { 121, 1, 3547, 0 },
    { 101, 1, 7239, 0 },
    { 97, 0, 8113, 0 },
    { 59, 0, 0, 1053 }, /* &Ncy; */
    { 99, 0, 3549, 0 },
    { 121, 0, 3550, 0 },
    { 59, 0, 0, 1071 }, /* &YAcy; */
    { 101, 0, 3552, 0 },
    { 109, 1, 3554, 0 },
    { 120, 0, 5742, 0 },
    { 101, 0, 3555, 0 },
    { 110, 0, 3556, 0 },
    { 116, 0, 3557, 0 },
    { 59, 0, 0, 8705 }, /* &complement; */
    { 59, 0, 0, 10669 }, /* &angmsdaf; */
    { 114, 0, 3560, 0 },
    { 99, 0, 3561, 0 },
    { 97, 0, 3562, 0 },
    { 112, 0, 3563, 0 },
    { 59, 0, 0, 10824 }, /* &cupbrcap; */
    { 114, 0, 3565, 0 },
    { 99, 1, 3567, 0 },
    { 101, 0, 9780, 0 },
    { 121, 0, 3568, 0 },
    { 59, 0, 0, 1118 }, /* &ubrcy; */
    { 102, 1, 3572, 0 },
    { 108, 1, 5031, 0 },
    { 112, 0, 7043, 0 },
    { 116, 0, 3573, 0 },
    { 99, 0, 3574, 0 },
    { 121, 0, 3575, 0 },
    { 59, 0, 0, 1100 }, /* &softcy; */
    { 97, 0, 3577, 0 },
    { 114, 0, 3578, 0 },
    { 114, 0, 3579, 0 },
    { 59, 1, 0, 8630 }, /* &cularr; */
    { 112, 0, 5783, 0 },
    { 104, 0, 3582, 0 },
    { 59, 0, 0, 8626 }, /* &ldsh; */
    { 105, 0, 3584, 0 },
    { 103, 0, 3585, 0 },
    { 104, 0, 3586, 0 },
    { 116, 0, 3587, 0 },
    { 97, 0, 3588, 0 },
    { 114, 0, 3589, 0 },
    { 114, 0, 3590, 0 },
    { 111, 0, 3591, 0 },
    { 119, 0, 3592, 0 },
    { 59, 0, 0, 8660 }, /* &Leftrightarrow; */
    { 99, 1, 3595, 0 },
    { 117, 0, 4246, 0 },
    { 114, 0, 3596, 0 },
    { 59, 0, 0, 119992 }, /* &cscr; */
    { 97, 0, 3598, 0 },
    { 114, 0, 3599, 0 },
    { 114, 0, 3600, 0 },
    { 59, 0, 0, 9084 }, /* &angzarr; */
    { 121, 1, 3603, 0 },
    { 105, 0, 5471, 0 },
    { 59, 0, 0, 1054 }, /* &Ocy; */
    { 97, 0, 3605, 0 },
    { 114, 0, 3606, 0 },
    { 114, 0, 3607, 0 },
    { 59, 0, 0, 10500 }, /* &nvHarr; */
    { 105, 0, 3609, 0 },
    { 103, 0, 3610, 0 },
    { 104, 0, 3611, 0 },
    { 116, 0, 3612, 0 },
    { 101, 1, 3614, 0 },
    { 59, 0, 0, 8939 }, /* &ntriangleright; */
    { 113, 0, 3615, 0 },
    { 59, 0, 0, 8941 }, /* &ntrianglerighteq; */
    { 116, 1, 3619, 0 },
    { 114, 1, 5797, 0 },
    { 97, 0, 9154, 0 },
    { 97, 0, 3620, 0 },
    { 105, 0, 3621, 0 },
    { 108, 0, 3622, 0 },
    { 59, 0, 0, 10524 }, /* &rAtail; */
    { 59, 0, 0, 9575 }, /* &boxHu; */
    { 100, 1, 3626, 0 },
    { 68, 0, 6972, 0 },
    { 97, 0, 3627, 0 },
    { 115, 0, 3628, 0 },
    { 104, 0, 3629, 0 },
    { 59, 0, 0, 8878 }, /* &nVdash; */
    { 101, 0, 3631, 0 },
    { 115, 0, 3632, 0 },
    { 115, 0, 3633, 0 },
    { 59, 0, 0, 8823 }, /* &GreaterLess; */
    { 59, 0, 0, 10883 }, /* &lesdotor; */
    { 108, 0, 3636, 0 },
    { 111, 0, 3637, 0 },
    { 111, 0, 3638, 0 },
    { 114, 0, 3639, 0 },
    { 59, 0, 0, 8971 }, /* &RightFloor; */
    { 114, 0, 3641, 0 },
    { 105, 0, 3642, 0 },
    { 59, 0, 0, 9721 }, /* &urtri; */
    { 114, 0, 3644, 0 },
    { 59, 0, 0, 120087 }, /* &Tfr; */
    { 111, 0, 3646, 0 },
    { 119, 0, 3647, 0 },
    { 110, 0, 3648, 0 },
    { 65, 0, 3649, 0 },
    { 114, 0, 3650, 0 },
    { 114, 0, 3651, 0 },
    { 111, 0, 3652, 0 },
    { 119, 0, 3653, 0 },
    { 59, 0, 0, 8661 }, /* &DoubleUpDownArrow; */
    { 112, 0, 3655, 0 },
    { 115, 0, 3656, 0 },
    { 105, 0, 3657, 0 },
    { 108, 0, 3658, 0 },
    { 111, 0, 3659, 0 },
    { 110, 0, 3660, 0 },
    { 59, 0, 0, 1013 }, /* &varepsilon; */
    { 97, 1, 3664, 0 },
    { 112, 1, 5585, 0 },
    { 111, 0, 7289, 0 },
    { 103, 1, 3667, 0 },
    { 99, 1, 4338, 0 },
    { 116, 0, 7229, 0 },
    { 108, 1, 3670, 0 },
    { 112, 1, 4215, 0 },
    { 101, 0, 9409, 0 },
    { 105, 0, 3671, 0 },
    { 110, 0, 3672, 0 },
    { 101, 0, 3673, 0 },
    { 59, 0, 0, 8464 }, /* &imagline; */
    { 114, 1, 3678, 0 },
    { 108, 1, 5349, 0 },
    { 76, 1, 7377, 0 },
    { 82, 0, 8813, 0 },
    { 59, 0, 0, 9561 }, /* &boxUr; */
    { 99, 1, 3683, 0 },
    { 97, 1, 3823, 0 },
    { 113, 1, 7690, 0 },
    { 104, 0, 8546, 0 },
    { 114, 0, 3684, 0 },
    { 59, 0, 0, 120007 }, /* &rscr; */
    { 115, 1, 3688, 0 },
    { 108, 1, 6271, 0 },
    { 97, 0, 8095, 0 },
    { 105, 0, 3689, 0 },
    { 118, 1, 3692, 0 },
    { 59, 1, 0, 949 }, /* &epsi; */
    { 108, 0, 8010, 0 },
    { 59, 0, 0, 1013 }, /* &epsiv; */
    { 112, 0, 3694, 0 },
    { 112, 0, 3695, 0 },
    { 97, 0, 3696, 0 },
    { 59, 1, 0, 954 }, /* &kappa; */
    { 118, 0, 8103, 0 },
    { 116, 1, 3700, 0 },
    { 59, 0, 0, 8733 }, /* &prop; */
    { 111, 0, 3701, 0 },
    { 59, 0, 0, 8733 }, /* &propto; */
    { 114, 0, 3703, 0 },
    { 59, 0, 0, 120077 }, /* &Jfr; */
    { 110, 0, 6728, 177 }, /* &plusmn */
    { 121, 1, 3707, 0 },
    { 105, 0, 6090, 0 },
    { 59, 0, 0, 1040 }, /* &Acy; */
    { 97, 0, 3709, 0 },
    { 115, 0, 3710, 0 },
    { 104, 0, 3711, 0 },
    { 59, 0, 0, 8877 }, /* &nvDash; */
    { 116, 0, 3713, 0 },
    { 59, 0, 0, 8857 }, /* &odot; */
    { 116, 1, 4153, 172 }, /* &not */
    { 112, 0, 7426, 0 },
    { 69, 1, 3719, 0 },
    { 97, 1, 4244, 0 },
    { 115, 0, 6087, 0 },
    { 59, 0, 0, 10933 }, /* &prnE; */
    { 108, 0, 3721, 0 },
    { 105, 0, 3722, 0 },
    { 103, 0, 3723, 223 }, /* &szlig */
    { 59, 0, 0, 223 }, /* &szlig; */
    { 110, 0, 3725, 0 },
    { 105, 0, 3726, 0 },
    { 110, 0, 3727, 0 },
    { 116, 0, 3728, 0 },
    { 59, 0, 0, 10768 }, /* &cirfnint; */
    { 59, 0, 0, 8791 }, /* &cire; */
    { 102, 1, 3735, 0 },
    { 116, 1, 4538, 0 },
    { 99, 1, 5689, 0 },
    { 59, 1, 0, 8712 }, /* &in; */
    { 111, 0, 8997, 0 },
    { 105, 0, 3736, 0 },
    { 110, 0, 3737, 0 },
    { 116, 1, 3739, 0 },
    { 59, 0, 0, 8734 }, /* &infin; */
    { 105, 0, 3740, 0 },
    { 101, 0, 3741, 0 },
    { 59, 0, 0, 10717 }, /* &infinite; */
    { 121, 1, 3744, 0 },
    { 105, 0, 4761, 0 },
    { 59, 0, 0, 1048 }, /* &Icy; */
    { 109, 1, 3750, 0 },
    { 97, 1, 5957, 0 },
    { 111, 1, 6451, 0 },
    { 114, 1, 8034, 0 },
    { 112, 0, 9125, 0 },
    { 116, 0, 3751, 0 },
    { 104, 0, 3752, 0 },
    { 116, 0, 3753, 0 },
    { 59, 0, 0, 8763 }, /* &homtht; */
    { 114, 1, 3756, 0 },
    { 121, 0, 7953, 0 },
    { 59, 0, 0, 119993 }, /* &dscr; */
    { 101, 0, 3758, 0 },
    { 59, 0, 0, 8774 }, /* &simne; */
    { 114, 1, 3762, 0 },
    { 115, 1, 5369, 0 },
    { 59, 0, 0, 10906 }, /* &eg; */
    { 97, 0, 3763, 0 },
    { 118, 0, 3764, 0 },
    { 101, 0, 3765, 232 }, /* &egrave */
    { 59, 0, 0, 232 }, /* &egrave; */
    { 117, 0, 3767, 0 },
    { 116, 0, 3768, 0 },
    { 101, 0, 3769, 0 },
    { 59, 0, 0, 263 }, /* &cacute; */
    { 114, 1, 3774, 0 },
    { 76, 1, 6610, 0 },
    { 108, 1, 8568, 0 },
    { 82, 0, 9517, 0 },
    { 59, 0, 0, 9555 }, /* &boxDr; */
    { 105, 0, 3776, 0 },
    { 109, 0, 3777, 0 },
    { 101, 1, 3779, 0 },
    { 59, 0, 0, 8765 }, /* &backsim; */
    { 113, 0, 3780, 0 },
    { 59, 0, 0, 8909 }, /* &backsimeq; */
    { 97, 0, 3782, 0 },
    { 114, 0, 3783, 0 },
    { 59, 1, 0, 40 }, /* &lpar; */
    { 108, 0, 9803, 0 },
    { 105, 0, 3786, 0 },
    { 100, 0, 3787, 0 },
    { 59, 0, 0, 10991 }, /* &cirmid; */
    { 112, 1, 3792, 0 },
    { 110, 1, 5942, 0 },
    { 108, 1, 6504, 0 },
    { 117, 0, 7315, 0 },
    { 114, 1, 3794, 0 },
    { 102, 0, 9656, 0 },
    { 111, 0, 3795, 0 },
    { 100, 0, 3796, 0 },
    { 117, 0, 3797, 0 },
    { 99, 0, 3798, 0 },
    { 116, 0, 3799, 0 },
    { 59, 0, 0, 8720 }, /* &Coproduct; */
    { 105, 0, 3801, 0 },
    { 108, 0, 3802, 0 },
    { 100, 0, 3803, 0 },
    { 101, 0, 3804, 0 },
    { 59, 0, 0, 8831 }, /* &NotSucceedsTilde; */
    { 114, 0, 3806, 0 },
    { 114, 0, 3807, 0 },
    { 59, 0, 0, 8657 }, /* &uArr; */
    { 117, 0, 3809, 0 },
    { 101, 0, 3810, 0 },
    { 115, 0, 3811, 0 },
    { 116, 0, 5186, 191 }, /* &iquest */
    { 97, 1, 3814, 0 },
    { 112, 0, 9668, 38 }, /* &amp */
    { 99, 1, 3816, 0 },
    { 108, 0, 7413, 0 },
    { 114, 0, 3817, 0 },
    { 59, 0, 0, 257 }, /* &amacr; */
    { 114, 0, 3819, 0 },
    { 114, 0, 3820, 0 },
    { 111, 0, 3821, 0 },
    { 119, 0, 3822, 0 },
    { 59, 0, 0, 8656 }, /* &Leftarrow; */
    { 113, 0, 3824, 0 },
    { 117, 0, 3825, 0 },
    { 111, 0, 3826, 0 },
    { 59, 0, 0, 8250 }, /* &rsaquo; */
    { 99, 0, 3828, 0 },
    { 121, 0, 3829, 0 },
    { 59, 0, 0, 1036 }, /* &KJcy; */
    { 99, 0, 3831, 0 },
    { 114, 0, 3832, 0 },
    { 59, 0, 0, 119985 }, /* &Vscr; */
    { 112, 1, 3840, 0 },
    { 69, 1, 5379, 0 },
    { 105, 1, 5673, 0 },
    { 97, 1, 5830, 0 },
    { 111, 1, 7304, 0 },
    { 101, 1, 7418, 0 },
    { 59, 0, 0, 8776 }, /* &ap; */
    { 114, 0, 3841, 0 },
    { 111, 0, 3842, 0 },
    { 120, 0, 3843, 0 },
    { 101, 1, 3845, 0 },
    { 59, 0, 0, 8776 }, /* &approx; */
    { 113, 0, 3846, 0 },
    { 59, 0, 0, 8778 }, /* &approxeq; */
    { 59, 0, 0, 10950 }, /* &supE; */
    { 51, 1, 3850, 0 },
    { 49, 0, 4084, 0 },
    { 52, 0, 3851, 0 },
    { 59, 0, 0, 9619 }, /* &blk34; */
    { 59, 0, 0, 928 }, /* &Pi; */
    { 117, 1, 3858, 0 },
    { 97, 1, 4710, 0 },
    { 111, 1, 8206, 0 },
    { 121, 1, 8218, 0 },
    { 101, 0, 9005, 0 },
    { 112, 0, 3859, 0 },
    { 59, 0, 0, 10818 }, /* &ncup; */
    { 110, 1, 3862, 0 },
    { 106, 0, 4683, 0 },
    { 106, 0, 3863, 0 },
    { 59, 0, 0, 8204 }, /* &zwnj; */
    { 101, 0, 3865, 0 },
    { 114, 0, 3866, 0 },
    { 116, 0, 3867, 0 },
    { 105, 0, 3868, 0 },
    { 99, 0, 3869, 0 },
    { 97, 0, 3870, 0 },
    { 108, 0, 3871, 0 },
    { 66, 0, 3872, 0 },
    { 97, 0, 3873, 0 },
    { 114, 0, 3874, 0 },
    { 59, 0, 0, 8741 }, /* &DoubleVerticalBar; */
    { 107, 0, 3876, 0 },
    { 101, 0, 3877, 0 },
    { 114, 0, 3878, 0 },
    { 59, 0, 0, 9646 }, /* &marker; */
    { 97, 0, 3880, 0 },
    { 114, 0, 3881, 0 },
    { 59, 0, 0, 10987 }, /* &Vbar; */
    { 110, 0, 3883, 0 },
    { 65, 1, 3889, 0 },
    { 82, 1, 4511, 0 },
    { 84, 1, 4949, 0 },
    { 66, 1, 5052, 0 },
    { 76, 1, 5802, 0 },
    { 97, 0, 9662, 0 },
    { 114, 0, 3890, 0 },
    { 114, 0, 3891, 0 },
    { 111, 0, 3892, 0 },
    { 119, 0, 3893, 0 },
    { 85, 1, 3896, 0 },
    { 59, 1, 0, 8595 }, /* &DownArrow; */
    { 66, 0, 8315, 0 },
    { 112, 0, 3897, 0 },
    { 65, 0, 3898, 0 },
    { 114, 0, 3899, 0 },
    { 114, 0, 3900, 0 },
    { 111, 0, 3901, 0 },
    { 119, 0, 3902, 0 },
    { 59, 0, 0, 8693 }, /* &DownArrowUpArrow; */
    { 105, 1, 3905, 0 },
    { 99, 0, 5364, 0 },
    { 114, 0, 3906, 0 },
    { 59, 0, 0, 10874 }, /* &gtcir; */
    { 117, 0, 3908, 0 },
    { 101, 0, 3909, 0 },
    { 115, 0, 3910, 0 },
    { 116, 0, 3911, 0 },
    { 59, 0, 0, 10876 }, /* &gtquest; */
    { 109, 0, 3913, 0 },
    { 101, 0, 3914, 0 },
    { 115, 1, 3916, 0 },
    { 59, 0, 0, 8242 }, /* &prime; */
    { 59, 0, 0, 8473 }, /* &primes; */
    { 103, 0, 3918, 0 },
    { 59, 0, 0, 10220 }, /* &loang; */
    { 59, 1, 0, 9674 }, /* &loz; */
    { 102, 1, 8731, 0 },
    { 101, 0, 9214, 0 },
    { 110, 1, 3924, 0 },
    { 114, 0, 5342, 0 },
    { 103, 0, 3925, 0 },
    { 108, 0, 3926, 0 },
    { 101, 0, 3927, 0 },
    { 66, 0, 3928, 0 },
    { 114, 0, 3929, 0 },
    { 97, 0, 3930, 0 },
    { 99, 0, 3931, 0 },
    { 107, 0, 3932, 0 },
    { 101, 0, 3933, 0 },
    { 116, 0, 3934, 0 },
    { 59, 0, 0, 10217 }, /* &RightAngleBracket; */
    { 105, 0, 3936, 0 },
    { 109, 0, 3937, 0 },
    { 101, 0, 3938, 0 },
    { 115, 0, 3939, 0 },
    { 59, 0, 0, 8864 }, /* &boxtimes; */
    { 97, 0, 3941, 0 },
    { 114, 0, 3942, 0 },
    { 114, 0, 3943, 0 },
    { 59, 0, 0, 8649 }, /* &rrarr; */
    { 59, 0, 0, 9524 }, /* &boxhu; */
    { 72, 0, 6984, 208 }, /* &ETH */
    { 97, 0, 3947, 0 },
    { 100, 1, 3949, 0 },
    { 114, 0, 6642, 0 },
    { 101, 0, 3950, 0 },
    { 115, 0, 3951, 0 },
    { 59, 1, 0, 9824 }, /* &spades; */
    { 117, 0, 6443, 0 },
    { 97, 0, 3954, 0 },
    { 114, 0, 3955, 0 },
    { 114, 0, 3956, 0 },
    { 59, 0, 0, 10512 }, /* &RBarr; */
    { 105, 0, 3958, 0 },
    { 109, 0, 3959, 0 },
    { 101, 0, 3960, 0 },
    { 115, 0, 3961, 0 },
    { 59, 0, 0, 8855 }, /* &CircleTimes; */
    { 112, 0, 3963, 0 },
    { 112, 1, 3965, 0 },
    { 59, 0, 0, 10890 }, /* &gnap; */
    { 114, 0, 3966, 0 },
    { 111, 0, 3967, 0 },
    { 120, 0, 3968, 0 },
    { 59, 0, 0, 10890 }, /* &gnapprox; */
    { 117, 0, 3970, 0 },
    { 101, 0, 3971, 0 },
    { 115, 0, 3972, 0 },
    { 116, 0, 3973, 0 },
    { 59, 0, 0, 10875 }, /* &ltquest; */
    { 100, 0, 3975, 0 },
    { 105, 0, 3976, 0 },
    { 108, 0, 3977, 0 },
    { 59, 0, 0, 315 }, /* &Lcedil; */
    { 101, 0, 3979, 0 },
    { 115, 0, 3980, 0 },
    { 116, 0, 3981, 0 },
    { 101, 0, 3982, 0 },
    { 100, 0, 3983, 0 },
    { 76, 1, 3985, 0 },
    { 71, 0, 7779, 0 },
    { 101, 0, 3986, 0 },
    { 115, 0, 3987, 0 },
    { 115, 0, 3988, 0 },
    { 76, 0, 3989, 0 },
    { 101, 0, 3990, 0 },
    { 115, 0, 3991, 0 },
    { 115, 0, 3992, 0 },
    { 59, 0, 0, 10913 }, /* &NotNestedLessLess; */
    { 100, 1, 3995, 0 },
    { 110, 0, 7419, 0 },
    { 105, 0, 3996, 0 },
    { 108, 0, 3997, 0 },
    { 108, 0, 3998, 0 },
    { 97, 0, 3999, 0 },
    { 59, 0, 0, 184 }, /* &Cedilla; */
    { 110, 0, 4001, 0 },
    { 100, 0, 4002, 163 }, /* &pound */
    { 59, 0, 0, 163 }, /* &pound; */
    { 97, 1, 4006, 0 },
    { 116, 1, 4715, 0 },
    { 108, 0, 5778, 0 },
    { 116, 0, 4007, 0 },
    { 59, 0, 0, 9837 }, /* &flat; */
    { 59, 0, 0, 9492 }, /* &boxur; */
    { 105, 0, 4010, 0 },
    { 103, 0, 4011, 0 },
    { 104, 0, 4012, 0 },
    { 116, 0, 4013, 0 },
    { 97, 0, 4014, 0 },
    { 114, 0, 4015, 0 },
    { 114, 0, 4016, 0 },
    { 111, 0, 4017, 0 },
    { 119, 0, 4018, 0 },
    { 115, 0, 4019, 0 },
    { 59, 0, 0, 8649 }, /* &rightrightarrows; */
    { 98, 1, 4022, 0 },
    { 97, 0, 7697, 0 },
    { 97, 0, 4023, 0 },
    { 114, 0, 4024, 0 },
    { 59, 0, 0, 95 }, /* &lowbar; */
    { 101, 0, 4026, 0 },
    { 100, 0, 4027, 0 },
    { 59, 0, 0, 8911 }, /* &cuwed; */
    { 111, 0, 4029, 0 },
    { 116, 0, 4030, 0 },
    { 59, 0, 0, 8949 }, /* &isindot; */
    { 117, 1, 4034, 0 },
    { 101, 1, 5760, 0 },
    { 105, 0, 8076, 0 },
    { 98, 1, 4036, 0 },
    { 112, 0, 6503, 0 },
    { 59, 0, 0, 10964 }, /* &supsub; */
    { 105, 1, 4039, 0 },
    { 111, 0, 5476, 0 },
    { 100, 0, 4040, 0 },
    { 111, 0, 4041, 0 },
    { 116, 0, 4042, 0 },
    { 59, 0, 0, 320 }, /* &lmidot; */
    { 112, 0, 4044, 0 },
    { 108, 0, 4045, 0 },
    { 121, 0, 4046, 0 },
    { 70, 0, 4047, 0 },
    { 117, 0, 4048, 0 },
    { 110, 0, 4049, 0 },
    { 99, 0, 4050, 0 },
    { 116, 0, 4051, 0 },
    { 105, 0, 4052, 0 },
    { 111, 0, 4053, 0 },
    { 110, 0, 4054, 0 },
    { 59, 0, 0, 8289 }, /* &ApplyFunction; */
    { 59, 1, 0, 10923 }, /* &lat; */
    { 97, 1, 6172, 0 },
    { 101, 0, 7345, 0 },
    { 98, 0, 4059, 0 },
    { 108, 0, 4060, 0 },
    { 101, 0, 4061, 0 },
    { 66, 0, 4062, 0 },
    { 114, 0, 4063, 0 },
    { 97, 0, 4064, 0 },
    { 99, 0, 4065, 0 },
    { 107, 0, 4066, 0 },
    { 101, 0, 4067, 0 },
    { 116, 0, 4068, 0 },
    { 59, 0, 0, 10214 }, /* &LeftDoubleBracket; */
    { 105, 1, 4073, 0 },
    { 111, 1, 4888, 0 },
    { 97, 1, 5467, 0 },
    { 121, 0, 9626, 0 },
    { 114, 0, 4074, 0 },
    { 59, 1, 0, 8790 }, /* &ecir; */
    { 99, 0, 5076, 234 }, /* &ecirc */
    { 99, 0, 4077, 0 },
    { 121, 0, 4078, 0 },
    { 59, 0, 0, 1107 }, /* &gjcy; */
    { 109, 0, 4080, 0 },
    { 97, 0, 4081, 0 },
    { 108, 0, 4082, 0 },
    { 101, 0, 4083, 0 },
    { 59, 0, 0, 9792 }, /* &female; */
    { 50, 1, 4086, 0 },
    { 52, 0, 8637, 0 },
    { 59, 0, 0, 9618 }, /* &blk12; */
    { 117, 1, 4089, 0 },
    { 115, 0, 7442, 0 },
    { 109, 0, 4090, 0 },
    { 112, 0, 4091, 0 },
    { 101, 1, 4093, 0 },
    { 59, 0, 0, 8782 }, /* &nbump; */
    { 59, 0, 0, 8783 }, /* &nbumpe; */
    { 114, 0, 4095, 0 },
    { 101, 0, 4096, 0 },
    { 101, 0, 4097, 0 },
    { 110, 0, 4098, 0 },
    { 59, 0, 0, 312 }, /* &kgreen; */
    { 59, 0, 0, 8535 }, /* &frac35; */
    { 100, 1, 4102, 0 },
    { 59, 0, 0, 10928 }, /* &sce; */
    { 105, 0, 4103, 0 },
    { 108, 0, 4104, 0 },
    { 59, 0, 0, 351 }, /* &scedil; */
    { 114, 0, 4106, 0 },
    { 115, 0, 4107, 0 },
    { 112, 0, 4108, 0 },
    { 59, 0, 0, 8202 }, /* &hairsp; */
    { 105, 0, 4110, 0 },
    { 108, 1, 4112, 0 },
    { 59, 0, 0, 978 }, /* &Upsi; */
    { 111, 0, 4113, 0 },
    { 110, 0, 4114, 0 },
    { 59, 0, 0, 933 }, /* &Upsilon; */
    { 97, 0, 4116, 0 },
    { 99, 0, 4117, 0 },
    { 114, 0, 4118, 0 },
    { 59, 0, 0, 256 }, /* &Amacr; */
    { 105, 1, 4122, 0 },
    { 100, 1, 4790, 0 },
    { 99, 0, 8219, 0 },
    { 109, 0, 4123, 0 },
    { 59, 0, 0, 8770 }, /* &esim; */
    { 114, 0, 4125, 0 },
    { 114, 0, 4126, 0 },
    { 59, 0, 0, 8661 }, /* &vArr; */
    { 79, 0, 4128, 0 },
    { 82, 0, 4129, 0 },
    { 78, 0, 8256, 222 }, /* &THORN */
    { 114, 0, 4131, 0 },
    { 116, 1, 4135, 0 },
    { 97, 1, 4559, 182 }, /* &para */
    { 115, 1, 7806, 0 },
    { 59, 0, 0, 8741 }, /* &par; */
    { 59, 0, 0, 8706 }, /* &part; */
    { 101, 0, 4137, 0 },
    { 99, 0, 4138, 0 },
    { 116, 0, 4139, 0 },
    { 111, 0, 4140, 0 },
    { 114, 0, 4141, 0 },
    { 59, 1, 0, 8636 }, /* &LeftVector; */
    { 66, 0, 9265, 0 },
    { 114, 1, 4146, 0 },
    { 97, 1, 9034, 0 },
    { 116, 0, 9603, 0 },
    { 114, 0, 4147, 0 },
    { 59, 0, 0, 8656 }, /* &lArr; */
    { 105, 0, 4149, 0 },
    { 109, 0, 4150, 0 },
    { 101, 0, 4151, 0 },
    { 115, 0, 4152, 0 },
    { 59, 0, 0, 10754 }, /* &bigotimes; */
    { 110, 1, 4156, 0 },
    { 105, 1, 6663, 0 },
    { 59, 0, 0, 172 }, /* &not; */
    { 105, 0, 4157, 0 },
    { 118, 1, 4159, 0 },
    { 59, 0, 0, 8716 }, /* &notni; */
    { 97, 1, 4162, 0 },
    { 98, 1, 4863, 0 },
    { 99, 0, 5421, 0 },
    { 59, 0, 0, 8716 }, /* &notniva; */
    { 109, 0, 4164, 0 },
    { 108, 0, 9111, 203 }, /* &Euml */
    { 101, 1, 4172, 0 },
    { 97, 1, 4380, 0 },
    { 116, 1, 4471, 0 },
    { 65, 1, 5850, 0 },
    { 115, 1, 6757, 0 },
    { 100, 1, 7498, 0 },
    { 69, 0, 8033, 0 },
    { 115, 1, 4176, 0 },
    { 113, 1, 4750, 0 },
    { 59, 1, 0, 8816 }, /* &nle; */
    { 102, 0, 7015, 0 },
    { 59, 1, 0, 10877 }, /* &nles; */
    { 115, 0, 8272, 0 },
    { 114, 0, 4179, 0 },
    { 100, 1, 4181, 0 },
    { 117, 0, 4878, 0 },
    { 115, 0, 4182, 0 },
    { 104, 0, 4183, 0 },
    { 97, 0, 4184, 0 },
    { 114, 0, 4185, 0 },
    { 59, 0, 0, 10570 }, /* &lurdshar; */
    { 107, 1, 4188, 0 },
    { 109, 0, 6595, 0 },
    { 99, 0, 4189, 0 },
    { 121, 0, 4190, 0 },
    { 59, 0, 0, 1030 }, /* &Iukcy; */
    { 116, 0, 4192, 0 },
    { 114, 0, 4193, 0 },
    { 105, 0, 4194, 0 },
    { 59, 0, 0, 10702 }, /* &rtriltri; */
    { 101, 0, 4196, 0 },
    { 102, 0, 4197, 0 },
    { 116, 0, 4198, 0 },
    { 97, 0, 4199, 0 },
    { 114, 0, 4200, 0 },
    { 114, 0, 4201, 0 },
    { 111, 0, 4202, 0 },
    { 119, 0, 4203, 0 },
    { 115, 0, 4204, 0 },
    { 59, 0, 0, 8647 }, /* &leftleftarrows; */
    { 107, 0, 4206, 0 },
    { 59, 0, 0, 9251 }, /* &blank; */
    { 97, 0, 4208, 0 },
    { 114, 0, 4209, 0 },
    { 114, 0, 4210, 0 },
    { 59, 0, 0, 10617 }, /* &subrarr; */
    { 115, 0, 4212, 0 },
    { 104, 0, 4213, 0 },
    { 116, 0, 4214, 0 },
    { 59, 0, 0, 10620 }, /* &lfisht; */
    { 97, 0, 4216, 0 },
    { 114, 0, 4217, 0 },
    { 116, 0, 4218, 0 },
    { 59, 0, 0, 8465 }, /* &imagpart; */
    { 59, 0, 0, 8921 }, /* &nGg; */
    { 99, 1, 4222, 0 },
    { 114, 0, 5995, 0 },
    { 117, 0, 4223, 0 },
    { 116, 0, 4224, 0 },
    { 101, 0, 6915, 250 }, /* &uacute */
    { 110, 0, 4226, 0 },
    { 103, 0, 4227, 0 },
    { 59, 0, 0, 367 }, /* &uring; */
    { 114, 0, 4229, 0 },
    { 114, 0, 4230, 0 },
    { 111, 0, 4231, 0 },
    { 119, 0, 4232, 0 },
    { 59, 0, 0, 8593 }, /* &uparrow; */
    { 59, 1, 0, 8476 }, /* &Re; */
    { 118, 0, 7741, 0 },
    { 99, 0, 4236, 0 },
    { 121, 0, 4237, 0 },
    { 59, 0, 0, 1045 }, /* &IEcy; */
    { 105, 0, 4239, 0 },
    { 108, 1, 4241, 0 },
    { 109, 0, 7265, 0 },
    { 100, 0, 4242, 0 },
    { 101, 0, 4243, 245 }, /* &otilde */
    { 59, 0, 0, 245 }, /* &otilde; */
    { 112, 0, 4245, 0 },
    { 59, 0, 0, 10937 }, /* &prnap; */
    { 112, 1, 4248, 0 },
    { 98, 0, 7570, 0 },
    { 101, 1, 4250, 0 },
    { 59, 0, 0, 10960 }, /* &csup; */
    { 59, 0, 0, 10962 }, /* &csupe; */
    { 105, 1, 4253, 0 },
    { 100, 0, 4846, 0 },
    { 111, 0, 4254, 0 },
    { 110, 0, 4255, 0 },
    { 59, 1, 0, 8899 }, /* &Union; */
    { 80, 0, 7796, 0 },
    { 115, 1, 4259, 0 },
    { 59, 0, 0, 8912 }, /* &Sub; */
    { 101, 0, 4260, 0 },
    { 116, 0, 4261, 0 },
    { 69, 1, 4263, 0 },
    { 59, 0, 0, 8912 }, /* &Subset; */
    { 113, 0, 4264, 0 },
    { 117, 0, 4265, 0 },
    { 97, 0, 4266, 0 },
    { 108, 0, 4267, 0 },
    { 59, 0, 0, 8838 }, /* &SubsetEqual; */
    { 104, 0, 4269, 0 },
    { 118, 0, 4270, 0 },
    { 59, 0, 0, 10980 }, /* &Dashv; */
    { 99, 1, 4274, 0 },
    { 114, 1, 6204, 0 },
    { 116, 0, 7736, 0 },
    { 97, 0, 4275, 0 },
    { 117, 0, 4276, 0 },
    { 115, 0, 4277, 0 },
    { 101, 0, 4278, 0 },
    { 59, 0, 0, 8757 }, /* &Because; */
    { 99, 0, 4280, 0 },
    { 114, 0, 4281, 0 },
    { 59, 0, 0, 8497 }, /* &Fscr; */
    { 104, 1, 4285, 0 },
    { 105, 1, 4705, 0 },
    { 114, 0, 7701, 0 },
    { 105, 0, 4286, 0 },
    { 59, 0, 0, 981 }, /* &varphi; */
    { 114, 0, 4288, 0 },
    { 101, 0, 4289, 0 },
    { 97, 0, 4290, 0 },
    { 116, 0, 4291, 0 },
    { 101, 0, 4292, 0 },
    { 114, 0, 4293, 0 },
    { 59, 1, 0, 8815 }, /* &NotGreater; */
    { 69, 1, 4892, 0 },
    { 83, 1, 5224, 0 },
    { 84, 1, 7221, 0 },
    { 71, 1, 8609, 0 },
    { 70, 1, 9443, 0 },
    { 76, 0, 9799, 0 },
    { 59, 0, 0, 8531 }, /* &frac13; */
    { 108, 0, 4302, 0 },
    { 59, 0, 0, 8260 }, /* &frasl; */
    { 59, 0, 0, 10660 }, /* &ange; */
    { 105, 1, 4306, 0 },
    { 101, 0, 7903, 0 },
    { 103, 0, 4307, 0 },
    { 104, 0, 4308, 0 },
    { 116, 0, 4309, 0 },
    { 84, 0, 4310, 0 },
    { 114, 0, 4311, 0 },
    { 105, 0, 4312, 0 },
    { 97, 0, 4313, 0 },
    { 110, 0, 4314, 0 },
    { 103, 0, 4315, 0 },
    { 108, 0, 4316, 0 },
    { 101, 0, 4317, 0 },
    { 66, 1, 4320, 0 },
    { 59, 1, 0, 8939 }, /* &NotRightTriangle; */
    { 69, 0, 9146, 0 },
    { 97, 0, 4321, 0 },
    { 114, 0, 4322, 0 },
    { 59, 0, 0, 10704 }, /* &NotRightTriangleBar; */
    { 59, 0, 0, 10843 }, /* &orv; */
    { 112, 1, 4326, 0 },
    { 116, 0, 8958, 0 },
    { 104, 0, 4327, 0 },
    { 59, 0, 0, 8738 }, /* &angsph; */
    { 112, 0, 4329, 0 },
    { 102, 0, 4330, 0 },
    { 59, 0, 0, 120168 }, /* &wopf; */
    { 97, 1, 4335, 0 },
    { 99, 1, 7200, 0 },
    { 116, 1, 8560, 0 },
    { 105, 0, 8714, 0 },
    { 114, 0, 4336, 0 },
    { 114, 0, 4337, 0 },
    { 59, 0, 0, 8634 }, /* &olarr; */
    { 114, 0, 4339, 0 },
    { 59, 0, 0, 299 }, /* &imacr; */
    { 105, 1, 4345, 0 },
    { 97, 1, 5062, 0 },
    { 101, 1, 9437, 0 },
    { 59, 1, 0, 10940 }, /* &Sc; */
    { 121, 0, 9596, 0 },
    { 114, 0, 4346, 0 },
    { 99, 0, 4347, 0 },
    { 59, 0, 0, 348 }, /* &Scirc; */
    { 111, 1, 4350, 0 },
    { 105, 0, 8802, 0 },
    { 104, 0, 4351, 0 },
    { 101, 0, 4352, 0 },
    { 97, 0, 4353, 0 },
    { 100, 0, 4354, 0 },
    { 108, 1, 4356, 0 },
    { 114, 0, 4429, 0 },
    { 101, 0, 4357, 0 },
    { 102, 0, 4358, 0 },
    { 116, 0, 4359, 0 },
    { 97, 0, 4360, 0 },
    { 114, 0, 4361, 0 },
    { 114, 0, 4362, 0 },
    { 111, 0, 4363, 0 },
    { 119, 0, 4364, 0 },
    { 59, 0, 0, 8606 }, /* &twoheadleftarrow; */
    { 59, 0, 0, 9500 }, /* &boxvr; */
    { 109, 1, 4368, 0 },
    { 99, 0, 8192, 0 },
    { 108, 0, 6137, 255 }, /* &yuml */
    { 59, 0, 0, 10664 }, /* &angmsdaa; */
    { 116, 0, 4371, 0 },
    { 59, 0, 0, 8749 }, /* &iiint; */
    { 121, 1, 4375, 0 },
    { 112, 1, 4723, 0 },
    { 99, 0, 9196, 0 },
    { 108, 0, 4376, 0 },
    { 101, 0, 4377, 0 },
    { 121, 0, 4378, 0 },
    { 115, 0, 4379, 0 },
    { 59, 0, 0, 8493 }, /* &Cayleys; */
    { 114, 0, 4381, 0 },
    { 114, 0, 4382, 0 },
    { 59, 0, 0, 8602 }, /* &nlarr; */
    { 68, 1, 4385, 0 },
    { 111, 0, 8257, 0 },
    { 111, 0, 4386, 0 },
    { 116, 0, 4387, 0 },
    { 59, 0, 0, 10871 }, /* &eDDot; */
    { 101, 1, 4392, 0 },
    { 117, 1, 4404, 0 },
    { 105, 1, 6496, 0 },
    { 97, 0, 7574, 0 },
    { 100, 0, 4393, 0 },
    { 105, 0, 4394, 0 },
    { 108, 0, 6827, 231 }, /* &ccedil */
    { 97, 1, 4399, 0 },
    { 101, 1, 4532, 0 },
    { 117, 1, 7528, 0 },
    { 121, 0, 8075, 0 },
    { 114, 0, 4400, 0 },
    { 111, 0, 4401, 0 },
    { 110, 0, 4402, 0 },
    { 59, 0, 0, 318 }, /* &lcaron; */
    { 59, 0, 0, 161 }, /* &iexcl; */
    { 112, 0, 4405, 0 },
    { 115, 0, 4406, 0 },
    { 115, 1, 4408, 0 },
    { 59, 0, 0, 10828 }, /* &ccups; */
    { 109, 0, 4409, 0 },
    { 59, 0, 0, 10832 }, /* &ccupssm; */
    { 114, 1, 4412, 0 },
    { 100, 0, 4477, 0 },
    { 99, 1, 4415, 0 },
    { 119, 1, 5596, 0 },
    { 59, 0, 0, 8596 }, /* &harr; */
    { 105, 0, 4416, 0 },
    { 114, 0, 4417, 0 },
    { 59, 0, 0, 10568 }, /* &harrcir; */
    { 114, 0, 4419, 0 },
    { 101, 0, 4420, 0 },
    { 118, 0, 4421, 0 },
    { 101, 0, 4422, 0 },
    { 59, 0, 0, 286 }, /* &Gbreve; */
    { 99, 0, 4424, 0 },
    { 114, 0, 4425, 0 },
    { 59, 0, 0, 120015 }, /* &zscr; */
    { 111, 0, 4427, 0 },
    { 116, 0, 4428, 0 },
    { 59, 0, 0, 10816 }, /* &capdot; */
    { 105, 0, 4430, 0 },
    { 103, 0, 4431, 0 },
    { 104, 0, 4432, 0 },
    { 116, 0, 4433, 0 },
    { 97, 0, 4434, 0 },
    { 114, 0, 4435, 0 },
    { 114, 0, 4436, 0 },
    { 111, 0, 4437, 0 },
    { 119, 0, 4438, 0 },
    { 59, 0, 0, 8608 }, /* &twoheadrightarrow; */
    { 97, 0, 4440, 0 },
    { 99, 0, 4441, 0 },
    { 114, 0, 4442, 0 },
    { 59, 0, 0, 362 }, /* &Umacr; */
    { 101, 1, 4445, 0 },
    { 111, 0, 8389, 0 },
    { 113, 1, 4447, 0 },
    { 59, 0, 0, 61 }, /* &bne; */
    { 117, 0, 4448, 0 },
    { 105, 0, 4449, 0 },
    { 118, 0, 4450, 0 },
    { 59, 0, 0, 8801 }, /* &bnequiv; */
    { 114, 1, 4454, 0 },
    { 99, 1, 8259, 0 },
    { 110, 0, 9594, 0 },
    { 114, 0, 4455, 0 },
    { 59, 1, 0, 8608 }, /* &Rarr; */
    { 116, 0, 7993, 0 },
    { 101, 1, 4460, 0 },
    { 111, 1, 5414, 0 },
    { 97, 0, 6008, 0 },
    { 115, 0, 4461, 0 },
    { 116, 0, 4462, 0 },
    { 59, 1, 0, 63 }, /* &quest; */
    { 101, 0, 5616, 0 },
    { 99, 1, 4466, 0 },
    { 98, 0, 9424, 0 },
    { 111, 1, 4468, 0 },
    { 114, 0, 8234, 0 },
    { 114, 0, 4469, 0 },
    { 110, 0, 4470, 0 },
    { 59, 0, 0, 8991 }, /* &drcorn; */
    { 114, 1, 4473, 0 },
    { 59, 0, 0, 8814 }, /* &nlt; */
    { 105, 0, 4474, 0 },
    { 101, 1, 4476, 0 },
    { 59, 0, 0, 8938 }, /* &nltri; */
    { 59, 0, 0, 8940 }, /* &nltrie; */
    { 99, 0, 4478, 0 },
    { 121, 0, 4479, 0 },
    { 59, 0, 0, 1098 }, /* &hardcy; */
    { 113, 1, 4482, 0 },
    { 59, 0, 0, 10887 }, /* &lne; */
    { 59, 1, 0, 10887 }, /* &lneq; */
    { 113, 0, 8318, 0 },
    { 114, 0, 4485, 0 },
    { 115, 0, 4486, 0 },
    { 101, 0, 4487, 0 },
    { 116, 0, 4488, 0 },
    { 59, 1, 0, 8835 }, /* &Superset; */
    { 69, 0, 4944, 0 },
    { 111, 0, 4491, 0 },
    { 102, 0, 4492, 0 },
    { 59, 0, 0, 402 }, /* &fnof; */
    { 101, 1, 4495, 0 },
    { 99, 0, 6092, 0 },
    { 114, 0, 4496, 0 },
    { 99, 0, 4497, 0 },
    { 121, 0, 4498, 0 },
    { 59, 0, 0, 1032 }, /* &Jsercy; */
    { 104, 0, 4500, 0 },
    { 105, 0, 4501, 0 },
    { 59, 0, 0, 981 }, /* &straightphi; */
    { 101, 1, 4507, 0 },
    { 99, 1, 5321, 0 },
    { 115, 1, 6916, 0 },
    { 102, 1, 8574, 0 },
    { 111, 0, 9122, 0 },
    { 100, 0, 4508, 0 },
    { 103, 0, 4509, 0 },
    { 101, 0, 4510, 0 },
    { 59, 0, 0, 8896 }, /* &Wedge; */
    { 105, 0, 4512, 0 },
    { 103, 0, 4513, 0 },
    { 104, 0, 4514, 0 },
    { 116, 0, 4515, 0 },
    { 86, 1, 4517, 0 },
    { 84, 0, 7885, 0 },
    { 101, 0, 4518, 0 },
    { 99, 0, 4519, 0 },
    { 116, 0, 4520, 0 },
    { 111, 0, 4521, 0 },
    { 114, 0, 4522, 0 },
    { 59, 1, 0, 8641 }, /* &DownRightVector; */
    { 66, 0, 9013, 0 },
    { 112, 0, 4525, 0 },
    { 102, 0, 4526, 0 },
    { 59, 0, 0, 120152 }, /* &gopf; */
    { 115, 0, 4528, 0 },
    { 101, 0, 4529, 0 },
    { 116, 0, 4530, 0 },
    { 59, 1, 0, 8834 }, /* &NotSubset; */
    { 69, 0, 8139, 0 },
    { 105, 1, 4534, 0 },
    { 100, 0, 6741, 0 },
    { 108, 0, 4535, 0 },
    { 59, 0, 0, 8968 }, /* &lceil; */
    { 99, 0, 4537, 0 },
    { 59, 0, 0, 10920 }, /* &lescc; */
    { 99, 1, 4543, 0 },
    { 101, 1, 4621, 0 },
    { 112, 1, 7296, 0 },
    { 59, 1, 0, 8747 }, /* &int; */
    { 108, 0, 9071, 0 },
    { 97, 0, 4544, 0 },
    { 108, 0, 4545, 0 },
    { 59, 0, 0, 8890 }, /* &intcal; */
    { 117, 0, 4547, 0 },
    { 108, 0, 4548, 0 },
    { 108, 0, 4549, 0 },
    { 69, 0, 4550, 0 },
    { 113, 0, 4551, 0 },
    { 117, 0, 4552, 0 },
    { 97, 0, 4553, 0 },
    { 108, 0, 4554, 0 },
    { 59, 0, 0, 8806 }, /* &LessFullEqual; */
    { 114, 0, 4556, 0 },
    { 59, 0, 0, 120073 }, /* &Ffr; */
    { 114, 0, 4558, 0 },
    { 59, 0, 0, 8254 }, /* &OverBar; */
    { 108, 1, 4561, 0 },
    { 59, 0, 0, 182 }, /* &para; */
    { 108, 0, 4562, 0 },
    { 101, 0, 4563, 0 },
    { 108, 0, 4564, 0 },
    { 59, 0, 0, 8741 }, /* &parallel; */
    { 59, 0, 0, 8850 }, /* &sqsupe; */
    { 114, 0, 4567, 0 },
    { 97, 0, 4568, 0 },
    { 118, 0, 4569, 0 },
    { 101, 0, 8685, 217 }, /* &Ugrave */
    { 112, 0, 4571, 0 },
    { 116, 0, 4572, 0 },
    { 121, 0, 4573, 0 },
    { 118, 0, 4574, 0 },
    { 59, 0, 0, 10674 }, /* &cemptyv; */
    { 105, 0, 4576, 0 },
    { 109, 0, 4577, 0 },
    { 101, 0, 4578, 0 },
    { 59, 0, 0, 10811 }, /* &tritime; */
    { 105, 0, 4580, 0 },
    { 114, 0, 4581, 0 },
    { 99, 0, 4582, 0 },
    { 59, 0, 0, 8858 }, /* &circledcirc; */
    { 100, 1, 4585, 0 },
    { 59, 0, 0, 8956 }, /* &nis; */
    { 59, 0, 0, 8954 }, /* &nisd; */
    { 112, 1, 4588, 0 },
    { 105, 0, 5234, 0 },
    { 102, 0, 4589, 0 },
    { 59, 0, 0, 8473 }, /* &Popf; */
    { 65, 0, 4591, 0 },
    { 108, 0, 4592, 0 },
    { 108, 0, 4593, 0 },
    { 59, 0, 0, 8704 }, /* &ForAll; */
    { 101, 0, 4595, 0 },
    { 101, 0, 4596, 0 },
    { 86, 0, 4597, 0 },
    { 101, 0, 4598, 0 },
    { 99, 0, 4599, 0 },
    { 116, 0, 4600, 0 },
    { 111, 0, 4601, 0 },
    { 114, 0, 4602, 0 },
    { 59, 0, 0, 10593 }, /* &LeftDownTeeVector; */
    { 109, 0, 4604, 0 },
    { 108, 0, 4605, 0 },
    { 59, 0, 0, 376 }, /* &Yuml; */
    { 97, 1, 4608, 0 },
    { 117, 0, 8901, 0 },
    { 112, 0, 4609, 0 },
    { 59, 0, 0, 10822 }, /* &cupcap; */
    { 102, 1, 4614, 0 },
    { 98, 1, 7817, 0 },
    { 59, 1, 0, 8868 }, /* &top; */
    { 99, 0, 8179, 0 },
    { 59, 1, 0, 120165 }, /* &topf; */
    { 111, 0, 5254, 0 },
    { 99, 0, 4617, 0 },
    { 117, 0, 4618, 0 },
    { 116, 0, 4619, 0 },
    { 101, 0, 4620, 0 },
    { 59, 0, 0, 378 }, /* &zacute; */
    { 103, 1, 4623, 0 },
    { 114, 0, 6797, 0 },
    { 101, 0, 4624, 0 },
    { 114, 0, 4625, 0 },
    { 115, 0, 4626, 0 },
    { 59, 0, 0, 8484 }, /* &integers; */
    { 105, 0, 4628, 0 },
    { 109, 0, 4629, 0 },
    { 101, 0, 4630, 0 },
    { 115, 0, 4631, 0 },
    { 59, 0, 0, 10805 }, /* &rotimes; */
    { 121, 1, 4635, 0 },
    { 99, 1, 8740, 0 },
    { 116, 0, 9692, 0 },
    { 109, 0, 4636, 0 },
    { 112, 0, 4637, 0 },
    { 101, 1, 4639, 0 },
    { 59, 0, 0, 8776 }, /* &asymp; */
    { 113, 0, 4640, 0 },
    { 59, 0, 0, 8781 }, /* &asympeq; */
    { 100, 1, 4643, 0 },
    { 59, 0, 0, 8773 }, /* &cong; */
    { 111, 0, 4644, 0 },
    { 116, 0, 4645, 0 },
    { 59, 0, 0, 10861 }, /* &congdot; */
    { 101, 0, 4647, 0 },
    { 99, 0, 4648, 0 },
    { 116, 0, 4649, 0 },
    { 111, 0, 4650, 0 },
    { 114, 0, 4651, 0 },
    { 59, 1, 0, 8639 }, /* &LeftUpVector; */
    { 66, 0, 9849, 0 },
    { 103, 0, 4654, 0 },
    { 108, 1, 4657, 0 },
    { 109, 1, 6626, 0 },
    { 114, 0, 9051, 0 },
    { 101, 0, 4658, 0 },
    { 102, 0, 4659, 0 },
    { 116, 0, 4660, 0 },
    { 97, 1, 4662, 0 },
    { 114, 0, 9704, 0 },
    { 114, 0, 4663, 0 },
    { 114, 0, 4664, 0 },
    { 111, 0, 4665, 0 },
    { 119, 0, 4666, 0 },
    { 59, 0, 0, 10229 }, /* &longleftarrow; */
    { 110, 0, 4668, 0 },
    { 99, 1, 4670, 0 },
    { 107, 0, 6482, 0 },
    { 107, 0, 4671, 0 },
    { 59, 1, 0, 8463 }, /* &planck; */
    { 104, 0, 8027, 0 },
    { 101, 0, 4674, 0 },
    { 116, 0, 4675, 0 },
    { 101, 1, 4677, 0 },
    { 59, 0, 0, 8834 }, /* &nsubset; */
    { 113, 0, 4678, 0 },
    { 59, 1, 0, 8840 }, /* &nsubseteq; */
    { 113, 0, 9403, 0 },
    { 99, 0, 4681, 0 },
    { 114, 0, 4682, 0 },
    { 59, 0, 0, 119989 }, /* &Zscr; */
    { 59, 0, 0, 8205 }, /* &zwj; */
    { 114, 1, 4686, 0 },
    { 110, 0, 8881, 0 },
    { 114, 0, 4687, 0 },
    { 59, 0, 0, 8702 }, /* &roarr; */
    { 116, 0, 4689, 0 },
    { 109, 0, 4690, 0 },
    { 110, 0, 4691, 0 },
    { 59, 0, 0, 8726 }, /* &ssetmn; */
    { 59, 0, 0, 8811 }, /* &nGtv; */
    { 111, 0, 4694, 0 },
    { 110, 0, 4695, 0 },
    { 101, 1, 4697, 0 },
    { 59, 0, 0, 58 }, /* &colon; */
    { 59, 1, 0, 8788 }, /* &colone; */
    { 113, 0, 8290, 0 },
    { 113, 0, 4700, 0 },
    { 117, 0, 4701, 0 },
    { 97, 0, 4702, 0 },
    { 114, 0, 4703, 0 },
    { 101, 0, 4704, 0 },
    { 59, 0, 0, 9642 }, /* &blacksquare; */
    { 59, 0, 0, 982 }, /* &varpi; */
    { 69, 1, 4708, 0 },
    { 59, 0, 0, 10909 }, /* &siml; */
    { 59, 0, 0, 10911 }, /* &simlE; */
    { 59, 0, 0, 94 }, /* &Hat; */
    { 114, 1, 4712, 0 },
    { 112, 0, 9277, 0 },
    { 111, 0, 4713, 0 },
    { 110, 0, 4714, 0 },
    { 59, 0, 0, 328 }, /* &ncaron; */
    { 110, 0, 4716, 0 },
    { 115, 0, 4717, 0 },
    { 59, 0, 0, 9649 }, /* &fltns; */
    { 108, 0, 4719, 0 },
    { 116, 1, 4721, 0 },
    { 59, 0, 0, 8711 }, /* &Del; */
    { 97, 0, 4722, 0 },
    { 59, 0, 0, 916 }, /* &Delta; */
    { 59, 1, 0, 8914 }, /* &Cap; */
    { 105, 0, 7919, 0 },
    { 105, 0, 4726, 0 },
    { 110, 0, 4727, 0 },
    { 103, 0, 4728, 197 }, /* &Aring */
    { 59, 0, 0, 197 }, /* &Aring; */
    { 97, 0, 4730, 0 },
    { 110, 0, 4731, 0 },
    { 103, 0, 4732, 0 },
    { 108, 0, 4733, 0 },
    { 101, 0, 4734, 0 },
    { 59, 0, 0, 10663 }, /* &uwangle; */
    { 113, 0, 4736, 0 },
    { 59, 0, 0, 10927 }, /* &npreceq; */
    { 59, 0, 0, 8921 }, /* &Gg; */
    { 117, 1, 4740, 0 },
    { 98, 0, 5348, 0 },
    { 59, 0, 0, 932 }, /* &Tau; */
    { 101, 0, 4742, 0 },
    { 101, 0, 4743, 0 },
    { 86, 0, 4744, 0 },
    { 101, 0, 4745, 0 },
    { 99, 0, 4746, 0 },
    { 116, 0, 4747, 0 },
    { 111, 0, 4748, 0 },
    { 114, 0, 4749, 0 },
    { 59, 0, 0, 10588 }, /* &RightUpTeeVector; */
    { 113, 1, 4753, 0 },
    { 59, 1, 0, 8816 }, /* &nleq; */
    { 115, 0, 9764, 0 },
    { 59, 0, 0, 8806 }, /* &nleqq; */
    { 111, 0, 4755, 0 },
    { 116, 0, 4756, 0 },
    { 104, 0, 4757, 0 },
    { 105, 0, 4758, 0 },
    { 110, 0, 4759, 0 },
    { 103, 0, 4760, 0 },
    { 59, 0, 0, 8709 }, /* &varnothing; */
    { 114, 0, 4762, 0 },
    { 99, 0, 9204, 206 }, /* &Icirc */
    { 114, 0, 4764, 0 },
    { 105, 0, 4765, 0 },
    { 97, 0, 4766, 0 },
    { 110, 0, 4767, 0 },
    { 103, 0, 4768, 0 },
    { 108, 0, 4769, 0 },
    { 101, 0, 4770, 0 },
    { 117, 1, 4772, 0 },
    { 100, 0, 8847, 0 },
    { 112, 0, 4773, 0 },
    { 59, 0, 0, 9651 }, /* &bigtriangleup; */
    { 109, 1, 4780, 0 },
    { 99, 1, 7063, 0 },
    { 111, 1, 7356, 0 },
    { 117, 1, 8809, 0 },
    { 102, 1, 8893, 0 },
    { 115, 0, 8907, 0 },
    { 97, 0, 4781, 0 },
    { 116, 0, 4782, 0 },
    { 104, 0, 4783, 0 },
    { 59, 0, 0, 567 }, /* &jmath; */
    { 97, 0, 4785, 0 },
    { 115, 0, 4786, 0 },
    { 104, 0, 4787, 0 },
    { 59, 0, 0, 8463 }, /* &hslash; */
    { 102, 0, 4789, 0 },
    { 59, 0, 0, 189 }, /* &half; */
    { 111, 0, 4791, 0 },
    { 116, 0, 4792, 0 },
    { 59, 0, 0, 8784 }, /* &esdot; */
    { 114, 0, 4794, 0 },
    { 107, 0, 4795, 0 },
    { 116, 1, 4797, 0 },
    { 59, 0, 0, 9141 }, /* &bbrk; */
    { 98, 0, 4798, 0 },
    { 114, 0, 4799, 0 },
    { 107, 0, 4800, 0 },
    { 59, 0, 0, 9142 }, /* &bbrktbrk; */
    { 114, 0, 4802, 0 },
    { 114, 0, 4803, 0 },
    { 111, 0, 4804, 0 },
    { 119, 0, 4805, 0 },
    { 115, 0, 4806, 0 },
    { 59, 0, 0, 8644 }, /* &rightleftarrows; */
    { 116, 1, 4809, 0 },
    { 112, 0, 5534, 0 },
    { 114, 0, 4810, 0 },
    { 105, 0, 4811, 0 },
    { 59, 0, 0, 9651 }, /* &xutri; */
    { 101, 0, 4813, 0 },
    { 101, 0, 4814, 0 },
    { 59, 0, 0, 8910 }, /* &cuvee; */
    { 59, 0, 0, 9571 }, /* &boxVL; */
    { 110, 0, 4817, 0 },
    { 120, 0, 4818, 0 },
    { 59, 0, 0, 8903 }, /* &divonx; */
    { 59, 0, 0, 10894 }, /* &gsime; */
    { 59, 0, 0, 967 }, /* &chi; */
    { 108, 0, 4822, 0 },
    { 117, 0, 4823, 0 },
    { 115, 0, 4824, 0 },
    { 59, 0, 0, 8853 }, /* &CirclePlus; */
    { 104, 1, 4828, 0 },
    { 98, 1, 8291, 0 },
    { 97, 0, 9179, 0 },
    { 97, 0, 4829, 0 },
    { 114, 0, 4830, 0 },
    { 59, 0, 0, 10606 }, /* &udhar; */
    { 108, 0, 4832, 0 },
    { 105, 0, 4833, 0 },
    { 103, 0, 9621, 230 }, /* &aelig */
    { 99, 0, 4835, 0 },
    { 117, 0, 4836, 0 },
    { 116, 0, 4837, 0 },
    { 101, 0, 4838, 225 }, /* &aacute */
    { 59, 0, 0, 225 }, /* &aacute; */
    { 114, 1, 4841, 0 },
    { 121, 0, 8645, 0 },
    { 59, 0, 0, 120009 }, /* &tscr; */
    { 97, 1, 4844, 0 },
    { 115, 0, 5988, 0 },
    { 112, 0, 4845, 0 },
    { 59, 0, 0, 8776 }, /* &thkap; */
    { 101, 0, 4847, 0 },
    { 114, 0, 4848, 0 },
    { 66, 1, 4850, 0 },
    { 80, 0, 9282, 0 },
    { 114, 1, 4852, 0 },
    { 97, 0, 5868, 0 },
    { 97, 0, 4853, 0 },
    { 99, 0, 4854, 0 },
    { 101, 1, 4856, 0 },
    { 107, 0, 6889, 0 },
    { 59, 0, 0, 9183 }, /* &UnderBrace; */
    { 97, 0, 4858, 0 },
    { 117, 0, 4859, 0 },
    { 115, 0, 4860, 0 },
    { 101, 1, 4862, 0 },
    { 59, 0, 0, 8757 }, /* &becaus; */
    { 59, 0, 0, 8757 }, /* &because; */
    { 59, 0, 0, 8958 }, /* &notnivb; */
    { 59, 0, 0, 9830 }, /* &diams; */
    { 115, 0, 4866, 0 },
    { 111, 1, 4868, 0 },
    { 117, 0, 9514, 0 },
    { 108, 0, 4869, 0 },
    { 59, 0, 0, 10185 }, /* &suphsol; */
    { 65, 1, 4872, 0 },
    { 116, 0, 6961, 0 },
    { 114, 0, 4873, 0 },
    { 114, 0, 4874, 0 },
    { 59, 0, 0, 10499 }, /* &nvrArr; */
    { 99, 0, 4876, 0 },
    { 121, 0, 4877, 0 },
    { 59, 0, 0, 1061 }, /* &KHcy; */
    { 104, 0, 4879, 0 },
    { 97, 0, 4880, 0 },
    { 114, 0, 4881, 0 },
    { 59, 0, 0, 10598 }, /* &luruhar; */
    { 59, 0, 0, 8842 }, /* &vsubne; */
    { 112, 0, 4884, 0 },
    { 102, 0, 4885, 0 },
    { 59, 0, 0, 120170 }, /* &yopf; */
    { 59, 1, 0, 8289 }, /* &af; */
    { 114, 0, 8561, 0 },
    { 108, 0, 4889, 0 },
    { 111, 0, 4890, 0 },
    { 110, 0, 4891, 0 },
    { 59, 0, 0, 8789 }, /* &ecolon; */
    { 113, 0, 4893, 0 },
    { 117, 0, 4894, 0 },
    { 97, 0, 4895, 0 },
    { 108, 0, 4896, 0 },
    { 59, 0, 0, 8817 }, /* &NotGreaterEqual; */
    { 113, 1, 4899, 0 },
    { 59, 0, 0, 8783 }, /* &bumpe; */
    { 59, 0, 0, 8783 }, /* &bumpeq; */
    { 109, 1, 4905, 0 },
    { 97, 1, 5824, 0 },
    { 104, 1, 6250, 0 },
    { 99, 1, 8144, 0 },
    { 116, 0, 9810, 0 },
    { 59, 0, 0, 8206 }, /* &lrm; */
    { 97, 0, 4907, 0 },
    { 109, 0, 4908, 0 },
    { 109, 0, 4909, 0 },
    { 97, 0, 4910, 0 },
    { 59, 0, 0, 989 }, /* &digamma; */
    { 121, 0, 4912, 0 },
    { 101, 1, 4915, 0 },
    { 118, 1, 7286, 0 },
    { 119, 0, 9630, 0 },
    { 113, 0, 4916, 0 },
    { 112, 1, 4918, 0 },
    { 115, 0, 7008, 0 },
    { 114, 0, 4919, 0 },
    { 101, 0, 4920, 0 },
    { 99, 0, 4921, 0 },
    { 59, 0, 0, 8926 }, /* &curlyeqprec; */
    { 114, 0, 4923, 0 },
    { 99, 1, 4925, 244 }, /* &ocirc */
    { 59, 0, 0, 8858 }, /* &ocir; */
    { 59, 0, 0, 244 }, /* &ocirc; */
    { 102, 0, 4927, 0 },
    { 105, 0, 4928, 0 },
    { 110, 0, 4929, 0 },
    { 59, 0, 0, 10716 }, /* &iinfin; */
    { 117, 0, 4931, 0 },
    { 105, 0, 4932, 0 },
    { 118, 0, 4933, 0 },
    { 59, 0, 0, 8802 }, /* &nequiv; */
    { 109, 0, 4935, 0 },
    { 112, 0, 4936, 0 },
    { 116, 0, 4937, 0 },
    { 121, 0, 4938, 0 },
    { 118, 0, 4939, 0 },
    { 59, 0, 0, 10676 }, /* &laemptyv; */
    { 103, 0, 4941, 0 },
    { 111, 0, 4942, 0 },
    { 102, 0, 4943, 0 },
    { 59, 0, 0, 8886 }, /* &origof; */
    { 113, 0, 4945, 0 },
    { 117, 0, 4946, 0 },
    { 97, 0, 4947, 0 },
    { 108, 0, 4948, 0 },
    { 59, 0, 0, 8839 }, /* &SupersetEqual; */
    { 101, 0, 4950, 0 },
    { 101, 0, 4951, 0 },
    { 59, 1, 0, 8868 }, /* &DownTee; */
    { 65, 0, 8237, 0 },
    { 114, 1, 4957, 0 },
    { 59, 1, 0, 8833 }, /* &nsc; */
    { 99, 1, 7509, 0 },
    { 101, 0, 7810, 0 },
    { 59, 0, 0, 120003 }, /* &nscr; */
    { 114, 0, 4959, 0 },
    { 114, 0, 4960, 0 },
    { 111, 0, 4961, 0 },
    { 119, 0, 4962, 0 },
    { 59, 0, 0, 8614 }, /* &RightTeeArrow; */
    { 59, 0, 0, 189 }, /* &frac12; */
    { 59, 0, 0, 9567 }, /* &boxVr; */
    { 101, 1, 4967, 0 },
    { 116, 0, 5407, 0 },
    { 59, 0, 0, 9794 }, /* &male; */
    { 108, 1, 6067, 168 }, /* &uml */
    { 97, 0, 5816, 0 },
    { 59, 0, 0, 10950 }, /* &nsupE; */
    { 100, 0, 4972, 0 },
    { 101, 0, 4973, 0 },
    { 59, 0, 0, 8482 }, /* &trade; */
    { 110, 0, 4975, 0 },
    { 101, 0, 4976, 0 },
    { 59, 0, 0, 9742 }, /* &phone; */
    { 114, 0, 4978, 0 },
    { 114, 1, 4980, 0 },
    { 104, 0, 9769, 0 },
    { 59, 1, 0, 8598 }, /* &nwarr; */
    { 111, 0, 9441, 0 },
    { 116, 0, 4983, 0 },
    { 59, 0, 0, 10644 }, /* &rpargt; */
    { 105, 0, 4985, 0 },
    { 110, 0, 4986, 0 },
    { 117, 0, 4987, 0 },
    { 115, 0, 4988, 0 },
    { 59, 0, 0, 8854 }, /* &CircleMinus; */
    { 105, 1, 4993, 0 },
    { 59, 1, 0, 10905 }, /* &el; */
    { 115, 1, 8875, 0 },
    { 108, 0, 9264, 0 },
    { 110, 0, 4994, 0 },
    { 116, 0, 4995, 0 },
    { 101, 0, 4996, 0 },
    { 114, 0, 4997, 0 },
    { 115, 0, 4998, 0 },
    { 59, 0, 0, 9191 }, /* &elinters; */
    { 110, 0, 5000, 0 },
    { 103, 0, 5001, 0 },
    { 114, 0, 5002, 0 },
    { 117, 0, 5003, 0 },
    { 101, 0, 5004, 0 },
    { 110, 0, 5005, 0 },
    { 116, 0, 5006, 0 },
    { 59, 0, 0, 8802 }, /* &NotCongruent; */
    { 109, 0, 5008, 0 },
    { 97, 0, 5009, 0 },
    { 108, 0, 5010, 0 },
    { 108, 0, 5011, 0 },
    { 83, 0, 5012, 0 },
    { 113, 0, 5013, 0 },
    { 117, 0, 5014, 0 },
    { 97, 0, 5015, 0 },
    { 114, 0, 5016, 0 },
    { 101, 0, 5017, 0 },
    { 59, 0, 0, 9724 }, /* &FilledSmallSquare; */
    { 97, 1, 5023, 0 },
    { 115, 1, 7058, 0 },
    { 59, 1, 0, 8743 }, /* &and; */
    { 118, 1, 8265, 0 },
    { 100, 0, 8463, 0 },
    { 110, 0, 5024, 0 },
    { 100, 0, 5025, 0 },
    { 59, 0, 0, 10837 }, /* &andand; */
    { 110, 0, 5027, 0 },
    { 105, 0, 5028, 0 },
    { 111, 0, 5029, 0 },
    { 110, 0, 5030, 0 },
    { 59, 0, 0, 8852 }, /* &SquareUnion; */
    { 98, 1, 5033, 0 },
    { 59, 0, 0, 47 }, /* &sol; */
    { 59, 1, 0, 10692 }, /* &solb; */
    { 97, 0, 8111, 0 },
    { 100, 0, 5036, 0 },
    { 105, 0, 5037, 0 },
    { 108, 0, 5038, 199 }, /* &Ccedil */
    { 59, 0, 0, 199 }, /* &Ccedil; */
    { 110, 0, 5040, 0 },
    { 59, 0, 0, 731 }, /* &ogon; */
    { 68, 1, 5044, 0 },
    { 69, 1, 5911, 0 },
    { 59, 0, 0, 168 }, /* &Dot; */
    { 111, 0, 5045, 0 },
    { 116, 0, 5046, 0 },
    { 59, 0, 0, 8412 }, /* &DotDot; */
    { 80, 1, 5049, 0 },
    { 105, 0, 5861, 0 },
    { 97, 0, 5050, 0 },
    { 114, 0, 5051, 0 },
    { 59, 0, 0, 10646 }, /* &ltrPar; */
    { 114, 0, 5053, 0 },
    { 101, 0, 5054, 0 },
    { 118, 0, 5055, 0 },
    { 101, 0, 5056, 0 },
    { 59, 0, 0, 785 }, /* &DownBreve; */
    { 111, 0, 5058, 0 },
    { 112, 0, 5059, 0 },
    { 59, 0, 0, 8974 }, /* &urcrop; */
    { 102, 0, 5061, 0 },
    { 59, 0, 0, 120154 }, /* &iopf; */
    { 114, 0, 5063, 0 },
    { 111, 0, 5064, 0 },
    { 110, 0, 5065, 0 },
    { 59, 0, 0, 352 }, /* &Scaron; */
    { 101, 1, 5068, 0 },
    { 111, 0, 6861, 0 },
    { 102, 0, 5069, 0 },
    { 116, 0, 5070, 0 },
    { 84, 1, 5073, 0 },
    { 65, 1, 6138, 0 },
    { 82, 0, 8855, 0 },
    { 101, 0, 5074, 0 },
    { 101, 0, 5075, 0 },
    { 59, 0, 0, 10980 }, /* &DoubleLeftTee; */
    { 59, 0, 0, 234 }, /* &ecirc; */
    { 109, 1, 5080, 0 },
    { 110, 1, 5554, 0 },
    { 108, 0, 7254, 0 },
    { 101, 0, 5081, 0 },
    { 115, 0, 7600, 215 }, /* &times */
    { 80, 0, 7415, 38 }, /* &AMP */
    { 105, 1, 5086, 0 },
    { 117, 1, 6351, 0 },
    { 97, 0, 9842, 0 },
    { 114, 0, 5087, 0 },
    { 99, 0, 5088, 0 },
    { 59, 0, 0, 9711 }, /* &xcirc; */
    { 103, 0, 5090, 0 },
    { 59, 0, 0, 10218 }, /* &Lang; */
    { 108, 1, 5094, 0 },
    { 111, 1, 5938, 0 },
    { 99, 0, 6691, 0 },
    { 97, 0, 5095, 0 },
    { 115, 0, 5096, 0 },
    { 104, 0, 5097, 248 }, /* &oslash */
    { 59, 0, 0, 248 }, /* &oslash; */
    { 112, 0, 5099, 0 },
    { 112, 1, 5101, 0 },
    { 59, 0, 0, 10889 }, /* &lnap; */
    { 114, 0, 5102, 0 },
    { 111, 0, 5103, 0 },
    { 120, 0, 5104, 0 },
    { 59, 0, 0, 10889 }, /* &lnapprox; */
    { 114, 0, 5106, 0 },
    { 109, 1, 5111, 0 },
    { 112, 1, 6390, 0 },
    { 99, 1, 6881, 0 },
    { 116, 1, 9200, 0 },
    { 105, 0, 9297, 0 },
    { 105, 0, 5112, 0 },
    { 108, 0, 5113, 0 },
    { 59, 0, 0, 8240 }, /* &permil; */
    { 117, 1, 5116, 0 },
    { 97, 0, 7946, 0 },
    { 112, 0, 5117, 0 },
    { 59, 1, 0, 8852 }, /* &sqcup; */
    { 115, 0, 6258, 0 },
    { 114, 0, 5120, 0 },
    { 105, 0, 5121, 0 },
    { 109, 0, 5122, 0 },
    { 101, 0, 5123, 0 },
    { 59, 0, 0, 8244 }, /* &tprime; */
    { 59, 0, 0, 184 }, /* &cedil; */
    { 110, 1, 5127, 0 },
    { 101, 0, 6635, 0 },
    { 69, 0, 5128, 0 },
    { 59, 0, 0, 8808 }, /* &lvnE; */
    { 59, 0, 0, 10932 }, /* &scE; */
    { 111, 1, 5132, 0 },
    { 59, 0, 0, 8517 }, /* &DD; */
    { 116, 0, 5133, 0 },
    { 114, 0, 5134, 0 },
    { 97, 0, 5135, 0 },
    { 104, 0, 5136, 0 },
    { 100, 0, 5137, 0 },
    { 59, 0, 0, 10513 }, /* &DDotrahd; */
    { 114, 0, 5139, 0 },
    { 97, 0, 5140, 0 },
    { 118, 0, 5141, 0 },
    { 101, 0, 5142, 0 },
    { 59, 0, 0, 96 }, /* &DiacriticalGrave; */
    { 114, 1, 5145, 0 },
    { 104, 0, 6240, 0 },
    { 105, 0, 5146, 0 },
    { 97, 0, 5147, 0 },
    { 110, 0, 5148, 0 },
    { 103, 0, 5149, 0 },
    { 108, 0, 5150, 0 },
    { 101, 0, 5151, 0 },
    { 114, 1, 5153, 0 },
    { 108, 0, 6333, 0 },
    { 105, 0, 5154, 0 },
    { 103, 0, 5155, 0 },
    { 104, 0, 5156, 0 },
    { 116, 0, 5157, 0 },
    { 59, 0, 0, 8883 }, /* &vartriangleright; */
    { 104, 1, 5160, 0 },
    { 59, 0, 0, 8597 }, /* &varr; */
    { 111, 0, 5161, 0 },
    { 59, 0, 0, 1009 }, /* &varrho; */
    { 59, 0, 0, 120148 }, /* &copf; */
    { 114, 0, 5164, 0 },
    { 59, 0, 0, 120116 }, /* &wfr; */
    { 105, 1, 5168, 0 },
    { 114, 1, 6946, 0 },
    { 108, 0, 9308, 0 },
    { 115, 0, 5169, 0 },
    { 104, 0, 5170, 0 },
    { 116, 0, 5171, 0 },
    { 59, 0, 0, 10621 }, /* &rfisht; */
    { 114, 0, 5173, 0 },
    { 101, 0, 5174, 0 },
    { 97, 0, 5175, 0 },
    { 116, 0, 5176, 0 },
    { 101, 0, 5177, 0 },
    { 114, 0, 5178, 0 },
    { 59, 0, 0, 10914 }, /* &GreaterGreater; */
    { 108, 0, 5180, 0 },
    { 59, 1, 0, 8226 }, /* &bull; */
    { 101, 0, 9189, 0 },
    { 97, 0, 5183, 0 },
    { 116, 1, 5185, 0 },
    { 59, 0, 0, 44 }, /* &comma; */
    { 59, 0, 0, 64 }, /* &commat; */
    { 59, 0, 0, 191 }, /* &iquest; */
    { 121, 0, 5188, 0 },
    { 59, 0, 0, 1119 }, /* &dzcy; */
    { 99, 1, 5191, 0 },
    { 115, 0, 7174, 0 },
    { 117, 0, 5192, 0 },
    { 116, 0, 5193, 0 },
    { 101, 0, 5194, 233 }, /* &eacute */
    { 59, 0, 0, 233 }, /* &eacute; */
    { 113, 1, 5197, 0 },
    { 59, 0, 0, 10888 }, /* &gne; */
    { 113, 1, 5199, 0 },
    { 59, 0, 0, 10888 }, /* &gneq; */
    { 59, 0, 0, 8809 }, /* &gneqq; */
    { 104, 0, 5201, 0 },
    { 114, 0, 5202, 0 },
    { 101, 0, 5203, 0 },
    { 101, 0, 5204, 0 },
    { 116, 0, 5205, 0 },
    { 105, 0, 5206, 0 },
    { 109, 0, 5207, 0 },
    { 101, 0, 5208, 0 },
    { 115, 0, 5209, 0 },
    { 59, 0, 0, 8907 }, /* &leftthreetimes; */
    { 101, 0, 5211, 0 },
    { 110, 0, 5212, 0 },
    { 67, 0, 5213, 0 },
    { 117, 0, 5214, 0 },
    { 114, 0, 5215, 0 },
    { 108, 0, 5216, 0 },
    { 121, 0, 5217, 0 },
    { 81, 1, 5219, 0 },
    { 68, 0, 6391, 0 },
    { 117, 0, 5220, 0 },
    { 111, 0, 5221, 0 },
    { 116, 0, 5222, 0 },
    { 101, 0, 5223, 0 },
    { 59, 0, 0, 8216 }, /* &OpenCurlyQuote; */
    { 108, 0, 5225, 0 },
    { 97, 0, 5226, 0 },
    { 110, 0, 5227, 0 },
    { 116, 0, 5228, 0 },
    { 69, 0, 5229, 0 },
    { 113, 0, 5230, 0 },
    { 117, 0, 5231, 0 },
    { 97, 0, 5232, 0 },
    { 108, 0, 5233, 0 },
    { 59, 0, 0, 10878 }, /* &NotGreaterSlantEqual; */
    { 110, 0, 5235, 0 },
    { 99, 0, 5236, 0 },
    { 97, 0, 5237, 0 },
    { 114, 0, 5238, 0 },
    { 101, 0, 5239, 0 },
    { 112, 0, 5240, 0 },
    { 108, 0, 5241, 0 },
    { 97, 0, 5242, 0 },
    { 110, 0, 5243, 0 },
    { 101, 0, 5244, 0 },
    { 59, 0, 0, 8460 }, /* &Poincareplane; */
    { 114, 0, 5246, 0 },
    { 111, 0, 5247, 0 },
    { 107, 0, 5248, 0 },
    { 59, 0, 0, 295 }, /* &hstrok; */
    { 105, 0, 5250, 0 },
    { 108, 0, 5251, 0 },
    { 100, 0, 5252, 0 },
    { 101, 0, 5253, 209 }, /* &Ntilde */
    { 59, 0, 0, 209 }, /* &Ntilde; */
    { 114, 0, 5255, 0 },
    { 107, 0, 5256, 0 },
    { 59, 0, 0, 10970 }, /* &topfork; */
    { 113, 1, 5261, 0 },
    { 108, 1, 5426, 0 },
    { 115, 1, 7837, 0 },
    { 99, 0, 9475, 0 },
    { 117, 0, 5262, 0 },
    { 111, 0, 5263, 0 },
    { 59, 1, 0, 8221 }, /* &rdquo; */
    { 114, 0, 9415, 0 },
    { 105, 0, 5266, 0 },
    { 110, 0, 5267, 0 },
    { 101, 0, 5268, 0 },
    { 59, 0, 0, 124 }, /* &VerticalLine; */
    { 117, 0, 5270, 0 },
    { 114, 0, 5271, 0 },
    { 102, 0, 5272, 0 },
    { 59, 0, 0, 8979 }, /* &profsurf; */
    { 59, 0, 0, 951 }, /* &eta; */
    { 114, 0, 5275, 0 },
    { 105, 0, 5276, 0 },
    { 97, 0, 5277, 0 },
    { 110, 0, 5278, 0 },
    { 103, 0, 5279, 0 },
    { 108, 0, 5280, 0 },
    { 101, 0, 5281, 0 },
    { 100, 1, 5285, 0 },
    { 59, 1, 0, 9652 }, /* &blacktriangle; */
    { 108, 1, 6586, 0 },
    { 114, 0, 7383, 0 },
    { 111, 0, 5286, 0 },
    { 119, 0, 5287, 0 },
    { 110, 0, 5288, 0 },
    { 59, 0, 0, 9662 }, /* &blacktriangledown; */
    { 111, 0, 5290, 0 },
    { 114, 0, 5291, 0 },
    { 116, 0, 5292, 0 },
    { 105, 0, 5293, 0 },
    { 111, 0, 5294, 0 },
    { 110, 0, 5295, 0 },
    { 97, 1, 5297, 0 },
    { 59, 0, 0, 8759 }, /* &Proportion; */
    { 108, 0, 5298, 0 },
    { 59, 0, 0, 8733 }, /* &Proportional; */
    { 115, 0, 5300, 0 },
    { 117, 0, 5301, 0 },
    { 98, 1, 5303, 0 },
    { 112, 0, 5442, 0 },
    { 59, 0, 0, 8834 }, /* &vnsub; */
    { 97, 0, 5305, 0 },
    { 114, 0, 5306, 0 },
    { 59, 0, 0, 8214 }, /* &Verbar; */
    { 111, 0, 5308, 0 },
    { 116, 0, 5309, 0 },
    { 59, 0, 0, 8918 }, /* &ltdot; */
    { 99, 1, 5312, 0 },
    { 114, 0, 7604, 0 },
    { 107, 0, 5313, 0 },
    { 115, 0, 5314, 0 },
    { 108, 0, 5315, 0 },
    { 97, 0, 5316, 0 },
    { 115, 0, 5317, 0 },
    { 104, 0, 5318, 0 },
    { 59, 0, 0, 8726 }, /* &Backslash; */
    { 114, 0, 5320, 175 }, /* &macr */
    { 59, 0, 0, 175 }, /* &macr; */
    { 105, 0, 5322, 0 },
    { 114, 0, 5323, 0 },
    { 99, 0, 5324, 0 },
    { 59, 0, 0, 372 }, /* &Wcirc; */
    { 114, 0, 5326, 0 },
    { 114, 0, 5327, 0 },
    { 59, 0, 0, 8654 }, /* &nhArr; */
    { 59, 0, 0, 174 }, /* &reg; */
    { 112, 0, 5330, 0 },
    { 97, 0, 5331, 0 },
    { 114, 0, 5332, 0 },
    { 115, 0, 5333, 0 },
    { 108, 0, 5334, 0 },
    { 59, 0, 0, 10725 }, /* &eqvparsl; */
    { 117, 0, 5336, 0 },
    { 114, 0, 5337, 0 },
    { 108, 0, 5338, 0 },
    { 121, 0, 5339, 0 },
    { 101, 0, 5340, 0 },
    { 113, 0, 5341, 0 },
    { 59, 0, 0, 8829 }, /* &succcurlyeq; */
    { 114, 0, 5343, 0 },
    { 111, 0, 5344, 0 },
    { 119, 0, 5345, 0 },
    { 59, 1, 0, 8594 }, /* &RightArrow; */
    { 66, 1, 7190, 0 },
    { 76, 0, 9355, 0 },
    { 59, 0, 0, 9 }, /* &Tab; */
    { 59, 0, 0, 9564 }, /* &boxUl; */
    { 114, 0, 5351, 0 },
    { 101, 0, 5352, 0 },
    { 101, 0, 5353, 0 },
    { 59, 0, 0, 8908 }, /* &rthree; */
    { 108, 0, 5355, 0 },
    { 105, 0, 5356, 0 },
    { 98, 0, 5357, 0 },
    { 114, 0, 5358, 0 },
    { 105, 0, 5359, 0 },
    { 117, 0, 5360, 0 },
    { 109, 0, 5361, 0 },
    { 59, 0, 0, 8652 }, /* &Equilibrium; */
    { 101, 0, 5363, 0 },
    { 59, 0, 0, 8736 }, /* &angle; */
    { 59, 0, 0, 10919 }, /* &gtcc; */
    { 103, 0, 5366, 0 },
    { 109, 0, 5367, 0 },
    { 97, 0, 5368, 0 },
    { 59, 0, 0, 962 }, /* &varsigma; */
    { 100, 1, 5371, 0 },
    { 59, 0, 0, 10902 }, /* &egs; */
    { 111, 0, 5372, 0 },
    { 116, 0, 5373, 0 },
    { 59, 0, 0, 10904 }, /* &egsdot; */
    { 103, 1, 5376, 0 },
    { 112, 0, 7446, 0 },
    { 111, 0, 5377, 0 },
    { 110, 0, 5378, 0 },
    { 59, 0, 0, 280 }, /* &Eogon; */
    { 59, 0, 0, 10864 }, /* &apE; */
    { 99, 0, 5381, 0 },
    { 114, 0, 5382, 0 },
    { 59, 0, 0, 119970 }, /* &Gscr; */
    { 109, 1, 5385, 0 },
    { 107, 0, 9434, 0 },
    { 108, 0, 6976, 239 }, /* &iuml */
    { 101, 1, 5390, 0 },
    { 97, 1, 5789, 0 },
    { 121, 1, 6475, 0 },
    { 117, 0, 8959, 0 },
    { 100, 1, 5392, 0 },
    { 105, 0, 7092, 0 },
    { 105, 0, 5393, 0 },
    { 108, 0, 5394, 0 },
    { 59, 0, 0, 343 }, /* &rcedil; */
    { 100, 0, 5396, 0 },
    { 111, 0, 5397, 0 },
    { 116, 0, 5398, 0 },
    { 59, 0, 0, 8943 }, /* &ctdot; */
    { 101, 0, 5400, 0 },
    { 100, 0, 5401, 0 },
    { 103, 1, 5403, 0 },
    { 59, 0, 0, 8965 }, /* &barwed; */
    { 101, 0, 5404, 0 },
    { 59, 0, 0, 8965 }, /* &barwedge; */
    { 114, 0, 5406, 0 },
    { 111, 0, 7479, 181 }, /* &micro */
    { 59, 1, 0, 10016 }, /* &malt; */
    { 101, 0, 6632, 0 },
    { 117, 0, 5410, 0 },
    { 111, 0, 5411, 187 }, /* &raquo */
    { 59, 0, 0, 187 }, /* &raquo; */
    { 59, 1, 0, 957 }, /* &nu; */
    { 109, 0, 7269, 0 },
    { 116, 0, 8157, 34 }, /* &quot */
    { 97, 1, 5418, 0 },
    { 115, 1, 6875, 0 },
    { 112, 0, 6956, 0 },
    { 99, 0, 5419, 0 },
    { 114, 0, 5420, 0 },
    { 59, 0, 0, 275 }, /* &emacr; */
    { 59, 0, 0, 8957 }, /* &notnivc; */
    { 97, 0, 5423, 0 },
    { 118, 0, 5424, 0 },
    { 101, 0, 5425, 0 },
    { 59, 0, 0, 96 }, /* &grave; */
    { 100, 0, 5427, 0 },
    { 104, 0, 5428, 0 },
    { 97, 0, 5429, 0 },
    { 114, 0, 5430, 0 },
    { 59, 0, 0, 10601 }, /* &rdldhar; */
    { 103, 0, 5432, 0 },
    { 114, 0, 5433, 0 },
    { 97, 0, 5434, 0 },
    { 114, 0, 5435, 0 },
    { 114, 0, 5436, 0 },
    { 59, 0, 0, 8669 }, /* &zigrarr; */
    { 117, 0, 5438, 0 },
    { 97, 0, 5439, 0 },
    { 108, 0, 5440, 0 },
    { 59, 1, 0, 8800 }, /* &NotEqual; */
    { 84, 0, 8526, 0 },
    { 59, 0, 0, 8835 }, /* &vnsup; */
    { 108, 1, 5445, 0 },
    { 105, 0, 9690, 0 },
    { 97, 0, 5446, 0 },
    { 110, 0, 5447, 0 },
    { 116, 0, 5448, 0 },
    { 103, 1, 5450, 0 },
    { 108, 0, 8710, 0 },
    { 116, 0, 5451, 0 },
    { 114, 0, 5452, 0 },
    { 59, 0, 0, 10902 }, /* &eqslantgtr; */
    { 117, 1, 5455, 0 },
    { 111, 0, 9513, 0 },
    { 59, 0, 0, 10789 }, /* &plusdu; */
    { 97, 0, 5457, 0 },
    { 114, 0, 5458, 0 },
    { 101, 0, 5459, 0 },
    { 110, 0, 5460, 0 },
    { 116, 0, 5461, 0 },
    { 104, 0, 5462, 0 },
    { 101, 0, 5463, 0 },
    { 115, 0, 5464, 0 },
    { 105, 0, 5465, 0 },
    { 115, 0, 5466, 0 },
    { 59, 0, 0, 9180 }, /* &OverParenthesis; */
    { 114, 0, 5468, 0 },
    { 111, 0, 5469, 0 },
    { 110, 0, 5470, 0 },
    { 59, 0, 0, 283 }, /* &ecaron; */
    { 114, 0, 5472, 0 },
    { 99, 0, 9510, 212 }, /* &Ocirc */
    { 99, 0, 5474, 0 },
    { 121, 0, 5475, 0 },
    { 59, 0, 0, 1026 }, /* &DJcy; */
    { 117, 0, 5477, 0 },
    { 115, 0, 5478, 0 },
    { 116, 0, 5479, 0 },
    { 97, 1, 5481, 0 },
    { 59, 0, 0, 9136 }, /* &lmoust; */
    { 99, 0, 5482, 0 },
    { 104, 0, 5483, 0 },
    { 101, 0, 5484, 0 },
    { 59, 0, 0, 9136 }, /* &lmoustache; */
    { 101, 1, 5488, 0 },
    { 116, 1, 5973, 0 },
    { 108, 0, 8204, 0 },
    { 102, 0, 5489, 0 },
    { 116, 0, 5490, 0 },
    { 97, 1, 5492, 0 },
    { 114, 0, 5921, 0 },
    { 114, 0, 5493, 0 },
    { 114, 0, 5494, 0 },
    { 111, 0, 5495, 0 },
    { 119, 0, 5496, 0 },
    { 59, 0, 0, 8653 }, /* &nLeftarrow; */
    { 59, 0, 0, 176 }, /* &deg; */
    { 114, 0, 5499, 0 },
    { 101, 0, 5500, 0 },
    { 118, 0, 5501, 0 },
    { 101, 0, 5502, 0 },
    { 59, 0, 0, 259 }, /* &abreve; */
    { 59, 0, 0, 9572 }, /* &boxHd; */
    { 101, 0, 5505, 0 },
    { 115, 0, 5506, 0 },
    { 115, 0, 5507, 0 },
    { 59, 0, 0, 10913 }, /* &LessLess; */
    { 118, 1, 5510, 0 },
    { 59, 0, 0, 961 }, /* &rho; */
    { 59, 0, 0, 1009 }, /* &rhov; */
    { 114, 0, 5512, 0 },
    { 59, 1, 0, 120112 }, /* &sfr; */
    { 111, 0, 5799, 0 },
    { 101, 0, 5515, 0 },
    { 114, 0, 5516, 0 },
    { 82, 1, 5518, 0 },
    { 76, 0, 7074, 0 },
    { 105, 0, 5519, 0 },
    { 103, 0, 5520, 0 },
    { 104, 0, 5521, 0 },
    { 116, 0, 5522, 0 },
    { 65, 0, 5523, 0 },
    { 114, 0, 5524, 0 },
    { 114, 0, 5525, 0 },
    { 111, 0, 5526, 0 },
    { 119, 0, 5527, 0 },
    { 59, 0, 0, 8600 }, /* &LowerRightArrow; */
    { 114, 0, 5529, 0 },
    { 99, 0, 5530, 0 },
    { 117, 0, 5531, 0 },
    { 112, 0, 5532, 0 },
    { 59, 0, 0, 10825 }, /* &capbrcup; */
    { 59, 0, 0, 10836 }, /* &Or; */
    { 108, 0, 5535, 0 },
    { 117, 0, 5536, 0 },
    { 115, 0, 5537, 0 },
    { 59, 0, 0, 10756 }, /* &xuplus; */
    { 99, 1, 5540, 0 },
    { 105, 0, 7706, 0 },
    { 59, 0, 0, 10918 }, /* &ltcc; */
    { 59, 0, 0, 8811 }, /* &Gt; */
    { 112, 1, 5545, 0 },
    { 59, 1, 0, 8465 }, /* &Im; */
    { 97, 0, 8577, 0 },
    { 108, 0, 5546, 0 },
    { 105, 0, 5547, 0 },
    { 101, 0, 5548, 0 },
    { 115, 0, 5549, 0 },
    { 59, 0, 0, 8658 }, /* &Implies; */
    { 97, 0, 5551, 0 },
    { 114, 0, 5552, 0 },
    { 114, 0, 5553, 0 },
    { 59, 0, 0, 8592 }, /* &slarr; */
    { 116, 0, 5555, 0 },
    { 59, 0, 0, 8749 }, /* &tint; */
    { 59, 0, 0, 10666 }, /* &angmsdac; */
    { 116, 0, 5558, 0 },
    { 84, 0, 5559, 0 },
    { 114, 0, 5560, 0 },
    { 105, 0, 5561, 0 },
    { 97, 0, 5562, 0 },
    { 110, 0, 5563, 0 },
    { 103, 0, 5564, 0 },
    { 108, 0, 5565, 0 },
    { 101, 0, 5566, 0 },
    { 69, 1, 5569, 0 },
    { 59, 1, 0, 8938 }, /* &NotLeftTriangle; */
    { 66, 0, 9388, 0 },
    { 113, 0, 5570, 0 },
    { 117, 0, 5571, 0 },
    { 97, 0, 5572, 0 },
    { 108, 0, 5573, 0 },
    { 59, 0, 0, 8940 }, /* &NotLeftTriangleEqual; */
    { 59, 0, 0, 9496 }, /* &boxul; */
    { 114, 0, 5576, 0 },
    { 114, 0, 5577, 0 },
    { 111, 0, 5578, 0 },
    { 119, 0, 5579, 0 },
    { 116, 1, 5581, 0 },
    { 59, 0, 0, 8594 }, /* &rightarrow; */
    { 97, 0, 5582, 0 },
    { 105, 0, 5583, 0 },
    { 108, 0, 5584, 0 },
    { 59, 0, 0, 8611 }, /* &rightarrowtail; */
    { 101, 0, 5586, 0 },
    { 100, 0, 5587, 0 },
    { 59, 0, 0, 437 }, /* &imped; */
    { 101, 0, 5589, 0 },
    { 114, 0, 5590, 0 },
    { 115, 0, 5591, 0 },
    { 101, 0, 5592, 0 },
    { 116, 0, 5593, 0 },
    { 59, 1, 0, 8848 }, /* &SquareSuperset; */
    { 69, 0, 8417, 0 },
    { 59, 0, 0, 8624 }, /* &Lsh; */
    { 59, 0, 0, 8621 }, /* &harrw; */
    { 97, 0, 5598, 0 },
    { 115, 0, 5599, 0 },
    { 104, 0, 5600, 0 },
    { 59, 1, 0, 8873 }, /* &Vdash; */
    { 108, 0, 9703, 0 },
    { 109, 0, 5603, 0 },
    { 97, 0, 5604, 0 },
    { 100, 1, 5606, 0 },
    { 59, 0, 0, 947 }, /* &gamma; */
    { 59, 0, 0, 989 }, /* &gammad; */
    { 105, 0, 5608, 0 },
    { 103, 0, 5609, 0 },
    { 122, 0, 5610, 0 },
    { 97, 0, 5611, 0 },
    { 103, 0, 5612, 0 },
    { 59, 0, 0, 10650 }, /* &vzigzag; */
    { 99, 0, 5614, 0 },
    { 114, 0, 5615, 0 },
    { 59, 0, 0, 119988 }, /* &Yscr; */
    { 113, 0, 5617, 0 },
    { 59, 0, 0, 8799 }, /* &questeq; */
    { 59, 1, 0, 10924 }, /* &smte; */
    { 115, 0, 9687, 0 },
    { 97, 0, 5621, 0 },
    { 59, 0, 0, 10550 }, /* &ldca; */
    { 69, 1, 5624, 0 },
    { 59, 0, 0, 10910 }, /* &simg; */
    { 59, 0, 0, 10912 }, /* &simgE; */
    { 80, 0, 5626, 0 },
    { 97, 0, 5627, 0 },
    { 114, 0, 5628, 0 },
    { 59, 0, 0, 10645 }, /* &gtlPar; */
    { 101, 1, 5631, 0 },
    { 112, 0, 8182, 0 },
    { 112, 1, 5633, 0 },
    { 102, 0, 9066, 0 },
    { 104, 0, 5634, 0 },
    { 59, 0, 0, 8501 }, /* &aleph; */
    { 109, 1, 5637, 0 },
    { 98, 0, 8122, 0 },
    { 59, 0, 0, 937 }, /* &ohm; */
    { 110, 0, 5639, 0 },
    { 97, 1, 5642, 0 },
    { 100, 1, 6028, 0 },
    { 104, 0, 7755, 0 },
    { 114, 0, 5643, 0 },
    { 114, 0, 5644, 0 },
    { 111, 0, 5645, 0 },
    { 119, 0, 5646, 0 },
    { 59, 0, 0, 8595 }, /* &downarrow; */
    { 105, 1, 5652, 0 },
    { 69, 1, 6747, 0 },
    { 111, 1, 6904, 0 },
    { 59, 1, 0, 8777 }, /* &nap; */
    { 112, 0, 8068, 0 },
    { 100, 0, 5653, 0 },
    { 59, 0, 0, 8779 }, /* &napid; */
    { 112, 0, 5655, 0 },
    { 102, 0, 5656, 0 },
    { 59, 0, 0, 120141 }, /* &Vopf; */
    { 117, 1, 5659, 0 },
    { 114, 0, 9478, 0 },
    { 59, 0, 0, 964 }, /* &tau; */
    { 114, 0, 5661, 0 },
    { 101, 0, 5662, 0 },
    { 118, 0, 5663, 0 },
    { 101, 0, 5664, 0 },
    { 59, 0, 0, 287 }, /* &gbreve; */
    { 97, 0, 5666, 0 },
    { 114, 0, 5667, 0 },
    { 100, 0, 5668, 0 },
    { 59, 0, 0, 10603 }, /* &llhard; */
    { 112, 1, 5671, 0 },
    { 103, 0, 6379, 0 },
    { 102, 0, 5672, 0 },
    { 59, 0, 0, 120140 }, /* &Uopf; */
    { 100, 0, 5674, 0 },
    { 59, 0, 0, 8779 }, /* &apid; */
    { 108, 0, 5676, 0 },
    { 97, 0, 5677, 0 },
    { 59, 0, 0, 8711 }, /* &nabla; */
    { 115, 1, 5682, 0 },
    { 103, 1, 6719, 0 },
    { 108, 1, 8004, 0 },
    { 114, 0, 9778, 0 },
    { 104, 0, 5683, 0 },
    { 118, 1, 5685, 0 },
    { 59, 0, 0, 8208 }, /* &dash; */
    { 59, 0, 0, 8867 }, /* &dashv; */
    { 112, 0, 5687, 0 },
    { 102, 0, 5688, 0 },
    { 59, 0, 0, 120171 }, /* &zopf; */
    { 97, 0, 5690, 0 },
    { 114, 0, 5691, 0 },
    { 101, 0, 5692, 0 },
    { 59, 0, 0, 8453 }, /* &incare; */
    { 99, 0, 5694, 0 },
    { 121, 0, 5695, 0 },
    { 59, 0, 0, 1027 }, /* &GJcy; */
    { 99, 0, 5697, 0 },
    { 121, 0, 5698, 0 },
    { 59, 0, 0, 1106 }, /* &djcy; */
    { 114, 0, 5700, 0 },
    { 114, 0, 5701, 0 },
    { 59, 0, 0, 8648 }, /* &uuarr; */
    { 103, 0, 5703, 0 },
    { 108, 1, 5707, 0 },
    { 59, 1, 0, 10217 }, /* &rang; */
    { 100, 1, 8641, 0 },
    { 101, 0, 9112, 0 },
    { 101, 0, 5708, 0 },
    { 59, 0, 0, 10217 }, /* &rangle; */
    { 97, 1, 5711, 0 },
    { 65, 0, 9680, 0 },
    { 114, 0, 5712, 0 },
    { 114, 0, 5713, 0 },
    { 59, 0, 0, 10230 }, /* &xrarr; */
    { 108, 0, 5715, 0 },
    { 97, 0, 5716, 0 },
    { 110, 0, 5717, 0 },
    { 116, 0, 5718, 0 },
    { 69, 0, 5719, 0 },
    { 113, 0, 5720, 0 },
    { 117, 0, 5721, 0 },
    { 97, 0, 5722, 0 },
    { 108, 0, 5723, 0 },
    { 59, 0, 0, 8929 }, /* &NotSucceedsSlantEqual; */
    { 97, 1, 5726, 0 },
    { 121, 0, 7941, 0 },
    { 114, 0, 5727, 0 },
    { 111, 0, 5728, 0 },
    { 110, 0, 5729, 0 },
    { 59, 0, 0, 271 }, /* &dcaron; */
    { 59, 0, 0, 10955 }, /* &varsubsetneqq; */
    { 59, 0, 0, 8637 }, /* &lhard; */
    { 70, 0, 5733, 0 },
    { 84, 0, 5734, 0 },
    { 99, 0, 5735, 0 },
    { 121, 0, 5736, 0 },
    { 59, 0, 0, 1068 }, /* &SOFTcy; */
    { 59, 0, 0, 9554 }, /* &boxdR; */
    { 99, 1, 5740, 0 },
    { 113, 0, 8980, 0 },
    { 114, 0, 5741, 0 },
    { 59, 0, 0, 120013 }, /* &xscr; */
    { 101, 0, 5743, 0 },
    { 115, 0, 5744, 0 },
    { 59, 0, 0, 8450 }, /* &complexes; */
    { 59, 1, 0, 960 }, /* &pi; */
    { 116, 1, 7178, 0 },
    { 118, 0, 9323, 0 },
    { 59, 0, 0, 8723 }, /* &mp; */
    { 105, 0, 5750, 0 },
    { 103, 0, 5751, 0 },
    { 104, 0, 5752, 0 },
    { 116, 0, 5753, 0 },
    { 65, 1, 5755, 0 },
    { 84, 0, 7204, 0 },
    { 114, 0, 5756, 0 },
    { 114, 0, 5757, 0 },
    { 111, 0, 5758, 0 },
    { 119, 0, 5759, 0 },
    { 59, 0, 0, 8658 }, /* &DoubleRightArrow; */
    { 116, 0, 5761, 0 },
    { 110, 1, 5764, 0 },
    { 101, 1, 9020, 0 },
    { 59, 0, 0, 8835 }, /* &supset; */
    { 101, 0, 5765, 0 },
    { 113, 0, 5766, 0 },
    { 59, 1, 0, 8843 }, /* &supsetneq; */
    { 113, 0, 9322, 0 },
    { 114, 0, 5769, 0 },
    { 59, 0, 0, 120008 }, /* &sscr; */
    { 105, 0, 5771, 0 },
    { 108, 0, 5772, 0 },
    { 100, 0, 5773, 0 },
    { 101, 0, 5774, 0 },
    { 59, 1, 0, 8769 }, /* &NotTilde; */
    { 69, 1, 6537, 0 },
    { 84, 1, 9395, 0 },
    { 70, 0, 9647, 0 },
    { 105, 0, 5779, 0 },
    { 103, 0, 5780, 0 },
    { 59, 0, 0, 64258 }, /* &fllig; */
    { 107, 0, 5782, 0 },
    { 59, 0, 0, 8618 }, /* &rarrhk; */
    { 59, 0, 0, 10557 }, /* &cularrp; */
    { 117, 0, 5785, 0 },
    { 111, 0, 5786, 0 },
    { 114, 1, 5788, 0 },
    { 59, 0, 0, 8220 }, /* &ldquo; */
    { 59, 0, 0, 8222 }, /* &ldquor; */
    { 114, 0, 5790, 0 },
    { 111, 0, 5791, 0 },
    { 110, 0, 5792, 0 },
    { 59, 0, 0, 345 }, /* &rcaron; */
    { 105, 1, 5795, 0 },
    { 121, 0, 6004, 0 },
    { 114, 0, 5796, 0 },
    { 99, 0, 7107, 251 }, /* &ucirc */
    { 114, 0, 5798, 0 },
    { 59, 0, 0, 8658 }, /* &rArr; */
    { 119, 0, 5800, 0 },
    { 110, 0, 5801, 0 },
    { 59, 0, 0, 8994 }, /* &sfrown; */
    { 101, 0, 5803, 0 },
    { 102, 0, 5804, 0 },
    { 116, 0, 5805, 0 },
    { 86, 1, 5808, 0 },
    { 84, 1, 7095, 0 },
    { 82, 0, 9160, 0 },
    { 101, 0, 5809, 0 },
    { 99, 0, 5810, 0 },
    { 116, 0, 5811, 0 },
    { 111, 0, 5812, 0 },
    { 114, 0, 5813, 0 },
    { 59, 1, 0, 8637 }, /* &DownLeftVector; */
    { 66, 0, 8024, 0 },
    { 59, 0, 0, 178 }, /* &sup2; */
    { 99, 0, 5817, 0 },
    { 114, 0, 5818, 0 },
    { 59, 0, 0, 363 }, /* &umacr; */
    { 102, 0, 5820, 0 },
    { 59, 0, 0, 120149 }, /* &dopf; */
    { 97, 0, 5822, 0 },
    { 114, 0, 5823, 0 },
    { 59, 0, 0, 9733 }, /* &bigstar; */
    { 114, 0, 5825, 0 },
    { 114, 0, 5826, 0 },
    { 59, 0, 0, 8646 }, /* &lrarr; */
    { 99, 0, 5828, 0 },
    { 114, 0, 5829, 0 },
    { 59, 0, 0, 8492 }, /* &Bscr; */
    { 99, 0, 5831, 0 },
    { 105, 0, 5832, 0 },
    { 114, 0, 5833, 0 },
    { 59, 0, 0, 10863 }, /* &apacir; */
    { 111, 0, 5835, 0 },
    { 114, 0, 5836, 0 },
    { 116, 0, 5837, 0 },
    { 76, 1, 5841, 0 },
    { 85, 1, 6182, 0 },
    { 68, 1, 6224, 0 },
    { 82, 0, 7164, 0 },
    { 101, 0, 5842, 0 },
    { 102, 0, 5843, 0 },
    { 116, 0, 5844, 0 },
    { 65, 0, 5845, 0 },
    { 114, 0, 5846, 0 },
    { 114, 0, 5847, 0 },
    { 111, 0, 5848, 0 },
    { 119, 0, 5849, 0 },
    { 59, 0, 0, 8592 }, /* &ShortLeftArrow; */
    { 114, 0, 5851, 0 },
    { 114, 0, 5852, 0 },
    { 59, 0, 0, 8653 }, /* &nlArr; */
    { 116, 1, 5855, 0 },
    { 99, 0, 9772, 0 },
    { 114, 0, 5856, 0 },
    { 111, 0, 5857, 0 },
    { 107, 0, 5858, 0 },
    { 59, 0, 0, 358 }, /* &Tstrok; */
    { 59, 1, 0, 8838 }, /* &sube; */
    { 100, 0, 7839, 0 },
    { 102, 1, 5864, 0 },
    { 101, 1, 6387, 0 },
    { 59, 0, 0, 9667 }, /* &ltri; */
    { 59, 0, 0, 9666 }, /* &ltrif; */
    { 59, 0, 0, 216 }, /* &Oslash; */
    { 102, 0, 5867, 0 },
    { 59, 0, 0, 120131 }, /* &Lopf; */
    { 114, 0, 5869, 0 },
    { 59, 0, 0, 95 }, /* &UnderBar; */
    { 111, 1, 5873, 0 },
    { 59, 1, 0, 8518 }, /* &dd; */
    { 97, 0, 8777, 0 },
    { 116, 0, 5874, 0 },
    { 115, 0, 5875, 0 },
    { 101, 0, 5876, 0 },
    { 113, 0, 5877, 0 },
    { 59, 0, 0, 10871 }, /* &ddotseq; */
    { 111, 0, 5879, 0 },
    { 119, 0, 5880, 0 },
    { 110, 0, 5881, 0 },
    { 65, 0, 5882, 0 },
    { 114, 0, 5883, 0 },
    { 114, 0, 5884, 0 },
    { 111, 0, 5885, 0 },
    { 119, 0, 5886, 0 },
    { 59, 0, 0, 8597 }, /* &UpDownArrow; */
    { 114, 0, 5888, 0 },
    { 114, 0, 5889, 0 },
    { 111, 0, 5890, 0 },
    { 119, 0, 5891, 0 },
    { 116, 1, 5893, 0 },
    { 59, 0, 0, 8592 }, /* &leftarrow; */
    { 97, 0, 5894, 0 },
    { 105, 0, 5895, 0 },
    { 108, 0, 5896, 0 },
    { 59, 0, 0, 8610 }, /* &leftarrowtail; */
    { 114, 1, 5899, 0 },
    { 112, 0, 6316, 0 },
    { 105, 0, 5900, 0 },
    { 122, 0, 5901, 0 },
    { 111, 0, 5902, 0 },
    { 110, 0, 5903, 0 },
    { 116, 0, 5904, 0 },
    { 97, 0, 5905, 0 },
    { 108, 0, 5906, 0 },
    { 76, 0, 5907, 0 },
    { 105, 0, 5908, 0 },
    { 110, 0, 5909, 0 },
    { 101, 0, 5910, 0 },
    { 59, 0, 0, 9472 }, /* &HorizontalLine; */
    { 113, 0, 5912, 0 },
    { 117, 0, 5913, 0 },
    { 97, 0, 5914, 0 },
    { 108, 0, 5915, 0 },
    { 59, 0, 0, 8784 }, /* &DotEqual; */
    { 103, 1, 5918, 0 },
    { 112, 0, 6927, 0 },
    { 111, 0, 5919, 0 },
    { 110, 0, 5920, 0 },
    { 59, 0, 0, 261 }, /* &aogon; */
    { 105, 0, 5922, 0 },
    { 103, 0, 5923, 0 },
    { 104, 0, 5924, 0 },
    { 116, 0, 5925, 0 },
    { 97, 0, 5926, 0 },
    { 114, 0, 5927, 0 },
    { 114, 0, 5928, 0 },
    { 111, 0, 5929, 0 },
    { 119, 0, 5930, 0 },
    { 59, 0, 0, 8654 }, /* &nLeftrightarrow; */
    { 117, 0, 5932, 0 },
    { 111, 0, 7138, 171 }, /* &laquo */
    { 98, 1, 5935, 0 },
    { 116, 0, 8066, 0 },
    { 97, 0, 5936, 0 },
    { 114, 0, 5937, 0 },
    { 59, 0, 0, 124 }, /* &verbar; */
    { 108, 0, 5939, 0 },
    { 59, 0, 0, 8856 }, /* &osol; */
    { 114, 0, 5941, 0 },
    { 59, 0, 0, 120096 }, /* &cfr; */
    { 105, 1, 5945, 0 },
    { 103, 1, 8248, 0 },
    { 116, 0, 8742, 0 },
    { 110, 0, 5946, 0 },
    { 116, 0, 5947, 0 },
    { 59, 0, 0, 8751 }, /* &Conint; */
    { 114, 0, 5949, 0 },
    { 101, 0, 5950, 0 },
    { 108, 0, 5951, 0 },
    { 59, 0, 0, 8880 }, /* &prurel; */
    { 108, 0, 5953, 0 },
    { 108, 0, 5954, 0 },
    { 101, 0, 5955, 0 },
    { 108, 0, 5956, 0 },
    { 59, 0, 0, 8742 }, /* &nparallel; */
    { 114, 0, 5958, 0 },
    { 114, 0, 5959, 0 },
    { 59, 0, 0, 8703 }, /* &hoarr; */
    { 115, 0, 5961, 0 },
    { 59, 0, 0, 10900 }, /* &gesles; */
    { 105, 1, 5964, 0 },
    { 99, 0, 9714, 0 },
    { 110, 0, 5965, 0 },
    { 116, 0, 5966, 0 },
    { 59, 0, 0, 10769 }, /* &awint; */
    { 97, 0, 5968, 0 },
    { 59, 0, 0, 920 }, /* &Theta; */
    { 110, 0, 5970, 0 },
    { 111, 0, 5971, 0 },
    { 117, 0, 5972, 0 },
    { 59, 0, 0, 8492 }, /* &bernou; */
    { 59, 1, 0, 8810 }, /* &nLt; */
    { 118, 0, 9763, 0 },
    { 117, 0, 5976, 0 },
    { 115, 0, 5977, 0 },
    { 59, 0, 0, 8854 }, /* &ominus; */
    { 113, 0, 5979, 0 },
    { 117, 0, 5980, 0 },
    { 97, 0, 5981, 0 },
    { 108, 0, 5982, 0 },
    { 76, 1, 5984, 0 },
    { 59, 0, 0, 8805 }, /* &GreaterEqual; */
    { 101, 0, 5985, 0 },
    { 115, 0, 5986, 0 },
    { 115, 0, 5987, 0 },
    { 59, 0, 0, 8923 }, /* &GreaterEqualLess; */
    { 105, 0, 5989, 0 },
    { 109, 0, 5990, 0 },
    { 59, 0, 0, 8764 }, /* &thksim; */
    { 105, 1, 5993, 0 },
    { 101, 0, 7012, 0 },
    { 109, 0, 5994, 0 },
    { 59, 0, 0, 8770 }, /* &nesim; */
    { 114, 0, 5996, 0 },
    { 59, 0, 0, 8593 }, /* &uarr; */
    { 105, 0, 5998, 0 },
    { 116, 0, 5999, 0 },
    { 59, 0, 0, 9829 }, /* &heartsuit; */
    { 97, 0, 6001, 0 },
    { 114, 0, 6002, 0 },
    { 114, 0, 6003, 0 },
    { 59, 0, 0, 10619 }, /* &suplarr; */
    { 59, 0, 0, 1091 }, /* &ucy; */
    { 117, 0, 6006, 0 },
    { 115, 0, 6007, 0 },
    { 59, 0, 0, 8846 }, /* &uplus; */
    { 116, 0, 6009, 0 },
    { 105, 1, 6011, 0 },
    { 101, 0, 6908, 0 },
    { 110, 0, 6012, 0 },
    { 116, 0, 6013, 0 },
    { 59, 0, 0, 10774 }, /* &quatint; */
    { 114, 1, 6016, 0 },
    { 59, 0, 0, 60 }, /* &nvlt; */
    { 105, 0, 6017, 0 },
    { 101, 0, 6018, 0 },
    { 59, 0, 0, 8884 }, /* &nvltrie; */
    { 110, 0, 6020, 0 },
    { 105, 0, 6021, 0 },
    { 110, 0, 6022, 0 },
    { 116, 0, 6023, 0 },
    { 59, 0, 0, 8752 }, /* &Cconint; */
    { 59, 0, 0, 925 }, /* &Nu; */
    { 112, 0, 6026, 0 },
    { 102, 0, 6027, 0 },
    { 59, 0, 0, 120121 }, /* &Bopf; */
    { 111, 0, 6029, 0 },
    { 119, 0, 6030, 0 },
    { 110, 0, 6031, 0 },
    { 97, 0, 6032, 0 },
    { 114, 0, 6033, 0 },
    { 114, 0, 6034, 0 },
    { 111, 0, 6035, 0 },
    { 119, 0, 6036, 0 },
    { 115, 0, 6037, 0 },
    { 59, 0, 0, 8650 }, /* &downdownarrows; */
    { 114, 0, 6039, 0 },
    { 105, 0, 6040, 0 },
    { 109, 0, 6041, 0 },
    { 101, 0, 6042, 0 },
    { 59, 0, 0, 8245 }, /* &backprime; */
    { 108, 0, 6044, 0 },
    { 97, 0, 6045, 0 },
    { 110, 0, 6046, 0 },
    { 116, 0, 6047, 0 },
    { 69, 0, 6048, 0 },
    { 113, 0, 6049, 0 },
    { 117, 0, 6050, 0 },
    { 97, 0, 6051, 0 },
    { 108, 0, 6052, 0 },
    { 59, 0, 0, 10877 }, /* &NotLessSlantEqual; */
    { 59, 0, 0, 9577 }, /* &boxHU; */
    { 114, 0, 6055, 0 },
    { 114, 0, 6056, 0 },
    { 59, 0, 0, 8655 }, /* &nrArr; */
    { 111, 0, 6058, 0 },
    { 116, 0, 6059, 0 },
    { 59, 0, 0, 10858 }, /* &simdot; */
    { 59, 0, 0, 170 }, /* &ordf; */
    { 98, 1, 6063, 0 },
    { 112, 0, 7460, 0 },
    { 117, 0, 6064, 0 },
    { 108, 0, 6065, 0 },
    { 108, 0, 6066, 0 },
    { 59, 0, 0, 8259 }, /* &hybull; */
    { 59, 0, 0, 168 }, /* &uml; */
    { 59, 0, 0, 229 }, /* &aring; */
    { 59, 0, 0, 9578 }, /* &boxvH; */
    { 98, 0, 6071, 0 },
    { 108, 0, 6072, 0 },
    { 97, 0, 6073, 0 },
    { 99, 0, 6074, 0 },
    { 59, 0, 0, 368 }, /* &Udblac; */
    { 59, 0, 0, 8897 }, /* &Vee; */
    { 97, 0, 6077, 0 },
    { 59, 0, 0, 950 }, /* &zeta; */
    { 101, 0, 6079, 0 },
    { 101, 0, 6080, 0 },
    { 59, 0, 0, 8897 }, /* &bigvee; */
    { 113, 0, 6082, 0 },
    { 100, 1, 6084, 0 },
    { 59, 0, 0, 8784 }, /* &doteq; */
    { 111, 0, 6085, 0 },
    { 116, 0, 6086, 0 },
    { 59, 0, 0, 8785 }, /* &doteqdot; */
    { 105, 0, 6088, 0 },
    { 109, 0, 6089, 0 },
    { 59, 0, 0, 8936 }, /* &prnsim; */
    { 114, 0, 6091, 0 },
    { 99, 0, 6727, 194 }, /* &Acirc */
    { 114, 0, 6093, 0 },
    { 59, 0, 0, 119973 }, /* &Jscr; */
    { 114, 0, 6095, 0 },
    { 114, 0, 6096, 0 },
    { 59, 0, 0, 8644 }, /* &rlarr; */
    { 101, 1, 6100, 0 },
    { 97, 1, 7505, 0 },
    { 121, 0, 7769, 0 },
    { 100, 0, 6101, 0 },
    { 105, 0, 6102, 0 },
    { 108, 0, 6103, 0 },
    { 59, 0, 0, 342 }, /* &Rcedil; */
    { 115, 0, 6105, 0 },
    { 104, 0, 6106, 0 },
    { 59, 0, 0, 8861 }, /* &odash; */
    { 108, 1, 6109, 0 },
    { 100, 0, 8984, 0 },
    { 108, 0, 6110, 0 },
    { 105, 0, 6111, 0 },
    { 110, 0, 6112, 0 },
    { 116, 0, 6113, 0 },
    { 114, 0, 6114, 0 },
    { 102, 0, 6115, 0 },
    { 59, 0, 0, 8499 }, /* &Mellintrf; */
    { 114, 0, 6117, 0 },
    { 114, 0, 6118, 0 },
    { 59, 0, 0, 8660 }, /* &hArr; */
    { 59, 0, 0, 8721 }, /* &Sum; */
    { 114, 0, 6121, 0 },
    { 59, 0, 0, 8460 }, /* &Hfr; */
    { 101, 0, 6123, 0 },
    { 59, 0, 0, 8995 }, /* &smile; */
    { 97, 1, 6127, 0 },
    { 101, 1, 6611, 0 },
    { 115, 0, 6729, 0 },
    { 112, 0, 6128, 0 },
    { 112, 0, 6129, 0 },
    { 114, 0, 6130, 0 },
    { 111, 0, 6131, 0 },
    { 120, 0, 6132, 0 },
    { 59, 0, 0, 10938 }, /* &succnapprox; */
    { 114, 0, 6134, 0 },
    { 111, 0, 6135, 0 },
    { 110, 0, 6136, 0 },
    { 59, 0, 0, 282 }, /* &Ecaron; */
    { 59, 0, 0, 255 }, /* &yuml; */
    { 114, 0, 6139, 0 },
    { 114, 0, 6140, 0 },
    { 111, 0, 6141, 0 },
    { 119, 0, 6142, 0 },
    { 59, 0, 0, 8656 }, /* &DoubleLeftArrow; */
    { 101, 0, 6144, 0 },
    { 114, 0, 6145, 0 },
    { 115, 0, 6146, 0 },
    { 101, 0, 6147, 0 },
    { 116, 0, 6148, 0 },
    { 69, 1, 6150, 0 },
    { 59, 0, 0, 8848 }, /* &NotSquareSuperset; */
    { 113, 0, 6151, 0 },
    { 117, 0, 6152, 0 },
    { 97, 0, 6153, 0 },
    { 108, 0, 6154, 0 },
    { 59, 0, 0, 8931 }, /* &NotSquareSupersetEqual; */
    { 114, 0, 6156, 0 },
    { 114, 0, 6157, 0 },
    { 59, 0, 0, 8665 }, /* &swArr; */
    { 101, 0, 6159, 0 },
    { 100, 0, 6160, 0 },
    { 105, 0, 6161, 0 },
    { 117, 0, 6162, 0 },
    { 109, 0, 6163, 0 },
    { 83, 0, 6164, 0 },
    { 112, 0, 6165, 0 },
    { 97, 0, 6166, 0 },
    { 99, 0, 6167, 0 },
    { 101, 0, 6168, 0 },
    { 59, 0, 0, 8203 }, /* &NegativeMediumSpace; */
    { 99, 0, 6170, 0 },
    { 114, 0, 6171, 0 },
    { 59, 0, 0, 332 }, /* &Omacr; */
    { 105, 0, 6173, 0 },
    { 108, 0, 6174, 0 },
    { 59, 0, 0, 10521 }, /* &latail; */
    { 100, 0, 6176, 0 },
    { 59, 0, 0, 10835 }, /* &And; */
    { 59, 0, 0, 8712 }, /* &isinv; */
    { 114, 0, 6179, 0 },
    { 111, 0, 6180, 0 },
    { 107, 0, 6181, 0 },
    { 59, 0, 0, 322 }, /* &lstrok; */
    { 112, 0, 6183, 0 },
    { 65, 0, 6184, 0 },
    { 114, 0, 6185, 0 },
    { 114, 0, 6186, 0 },
    { 111, 0, 6187, 0 },
    { 119, 0, 6188, 0 },
    { 59, 0, 0, 8593 }, /* &ShortUpArrow; */
    { 114, 0, 6190, 0 },
    { 114, 0, 6191, 0 },
    { 111, 0, 6192, 0 },
    { 119, 0, 6193, 0 },
    { 59, 0, 0, 8657 }, /* &Uparrow; */
    { 108, 0, 6195, 0 },
    { 105, 0, 6196, 0 },
    { 103, 0, 6197, 198 }, /* &AElig */
    { 59, 0, 0, 198 }, /* &AElig; */
    { 100, 1, 6202, 0 },
    { 59, 1, 0, 8739 }, /* &mid; */
    { 97, 1, 7834, 0 },
    { 99, 0, 8200, 0 },
    { 111, 0, 6203, 0 },
    { 116, 0, 9205, 183 }, /* &middot */
    { 110, 0, 6205, 0 },
    { 111, 0, 6206, 0 },
    { 117, 0, 6207, 0 },
    { 108, 0, 6208, 0 },
    { 108, 0, 6209, 0 },
    { 105, 0, 6210, 0 },
    { 115, 0, 6211, 0 },
    { 59, 0, 0, 8492 }, /* &Bernoullis; */
    { 114, 0, 6213, 0 },
    { 59, 0, 0, 8926 }, /* &cuepr; */
    { 113, 0, 6215, 0 },
    { 117, 0, 6216, 0 },
    { 105, 0, 6217, 0 },
    { 103, 0, 6218, 0 },
    { 97, 0, 6219, 0 },
    { 114, 0, 6220, 0 },
    { 114, 0, 6221, 0 },
    { 111, 0, 6222, 0 },
    { 119, 0, 6223, 0 },
    { 59, 0, 0, 8605 }, /* &rightsquigarrow; */
    { 111, 0, 6225, 0 },
    { 119, 0, 6226, 0 },
    { 110, 0, 6227, 0 },
    { 65, 0, 6228, 0 },
    { 114, 0, 6229, 0 },
    { 114, 0, 6230, 0 },
    { 111, 0, 6231, 0 },
    { 119, 0, 6232, 0 },
    { 59, 0, 0, 8595 }, /* &ShortDownArrow; */
    { 59, 0, 0, 1072 }, /* &acy; */
    { 59, 0, 0, 10691 }, /* &cirE; */
    { 110, 0, 6236, 0 },
    { 116, 0, 6237, 0 },
    { 59, 0, 0, 10764 }, /* &qint; */
    { 56, 0, 6239, 0 },
    { 59, 0, 0, 8542 }, /* &frac78; */
    { 101, 0, 6241, 0 },
    { 116, 0, 6242, 0 },
    { 97, 0, 6243, 0 },
    { 59, 0, 0, 977 }, /* &vartheta; */
    { 109, 0, 6245, 0 },
    { 109, 0, 6246, 0 },
    { 97, 0, 6247, 0 },
    { 100, 1, 6249, 0 },
    { 59, 0, 0, 915 }, /* &Gamma; */
    { 59, 0, 0, 988 }, /* &Gammad; */
    { 97, 0, 6251, 0 },
    { 114, 0, 6252, 0 },
    { 59, 1, 0, 8651 }, /* &lrhar; */
    { 100, 0, 7805, 0 },
    { 105, 0, 6255, 0 },
    { 114, 0, 6256, 0 },
    { 99, 0, 6257, 0 },
    { 59, 0, 0, 373 }, /* &wcirc; */
    { 59, 0, 0, 8852 }, /* &sqcups; */
    { 108, 0, 6260, 0 },
    { 111, 0, 6261, 0 },
    { 111, 0, 6262, 0 },
    { 114, 0, 6263, 0 },
    { 59, 0, 0, 8970 }, /* &LeftFloor; */
    { 116, 0, 6265, 0 },
    { 59, 0, 0, 8730 }, /* &Sqrt; */
    { 97, 1, 6269, 0 },
    { 105, 1, 7212, 0 },
    { 117, 0, 8683, 0 },
    { 112, 0, 6270, 0 },
    { 59, 0, 0, 8898 }, /* &bigcap; */
    { 117, 0, 6272, 0 },
    { 115, 0, 6273, 0 },
    { 59, 0, 0, 10865 }, /* &eplus; */
    { 68, 0, 6275, 0 },
    { 111, 0, 6276, 0 },
    { 116, 0, 6277, 0 },
    { 59, 0, 0, 8762 }, /* &mDDot; */
    { 105, 0, 6279, 0 },
    { 103, 0, 6280, 0 },
    { 104, 0, 6281, 0 },
    { 116, 0, 6282, 0 },
    { 86, 1, 6284, 0 },
    { 65, 0, 7842, 0 },
    { 101, 0, 6285, 0 },
    { 99, 0, 6286, 0 },
    { 116, 0, 6287, 0 },
    { 111, 0, 6288, 0 },
    { 114, 0, 6289, 0 },
    { 59, 0, 0, 10574 }, /* &LeftRightVector; */
    { 97, 0, 6291, 0 },
    { 112, 0, 6292, 0 },
    { 59, 0, 0, 10236 }, /* &xmap; */
    { 114, 0, 6294, 0 },
    { 101, 0, 6295, 0 },
    { 101, 0, 6296, 0 },
    { 59, 0, 0, 8907 }, /* &lthree; */
    { 101, 0, 6298, 0 },
    { 99, 0, 6299, 0 },
    { 116, 0, 6300, 0 },
    { 111, 0, 6301, 0 },
    { 114, 0, 6302, 0 },
    { 59, 0, 0, 10586 }, /* &LeftTeeVector; */
    { 112, 1, 6306, 0 },
    { 99, 1, 6331, 0 },
    { 105, 0, 8794, 0 },
    { 101, 1, 6308, 0 },
    { 111, 0, 9237, 0 },
    { 99, 0, 6309, 0 },
    { 116, 0, 6310, 0 },
    { 97, 0, 6311, 0 },
    { 116, 0, 6312, 0 },
    { 105, 0, 6313, 0 },
    { 111, 0, 6314, 0 },
    { 110, 0, 6315, 0 },
    { 59, 0, 0, 8496 }, /* &expectation; */
    { 102, 0, 6317, 0 },
    { 59, 0, 0, 8461 }, /* &Hopf; */
    { 105, 0, 6319, 0 },
    { 110, 0, 6320, 0 },
    { 59, 0, 0, 8946 }, /* &disin; */
    { 114, 0, 6322, 0 },
    { 59, 0, 0, 119998 }, /* &iscr; */
    { 112, 0, 6324, 0 },
    { 97, 0, 6325, 0 },
    { 114, 0, 6326, 0 },
    { 114, 0, 6327, 0 },
    { 111, 0, 6328, 0 },
    { 119, 0, 6329, 0 },
    { 115, 0, 6330, 0 },
    { 59, 0, 0, 8648 }, /* &upuparrows; */
    { 108, 0, 6332, 0 },
    { 59, 0, 0, 33 }, /* &excl; */
    { 101, 0, 6334, 0 },
    { 102, 0, 6335, 0 },
    { 116, 0, 6336, 0 },
    { 59, 0, 0, 8882 }, /* &vartriangleleft; */
    { 110, 0, 6338, 165 }, /* &yen */
    { 59, 0, 0, 165 }, /* &yen; */
    { 114, 0, 6340, 0 },
    { 59, 0, 0, 120100 }, /* &gfr; */
    { 105, 0, 6342, 0 },
    { 108, 1, 6344, 0 },
    { 109, 0, 7104, 0 },
    { 100, 0, 6345, 0 },
    { 101, 0, 6346, 213 }, /* &Otilde */
    { 59, 0, 0, 213 }, /* &Otilde; */
    { 108, 0, 6348, 0 },
    { 117, 0, 6349, 0 },
    { 115, 0, 6350, 0 },
    { 59, 0, 0, 10809 }, /* &triplus; */
    { 112, 0, 6352, 0 },
    { 59, 0, 0, 8899 }, /* &xcup; */
    { 97, 1, 6355, 0 },
    { 104, 0, 9430, 0 },
    { 114, 0, 6356, 0 },
    { 114, 0, 6357, 0 },
    { 59, 0, 0, 8693 }, /* &duarr; */
    { 102, 1, 6360, 0 },
    { 59, 0, 0, 8677 }, /* &rarrb; */
    { 115, 0, 6361, 0 },
    { 59, 0, 0, 10528 }, /* &rarrbfs; */
    { 59, 0, 0, 10878 }, /* &nges; */
    { 114, 0, 6364, 0 },
    { 114, 0, 6365, 0 },
    { 111, 0, 6366, 0 },
    { 119, 0, 6367, 0 },
    { 114, 1, 6369, 0 },
    { 108, 0, 8286, 0 },
    { 105, 0, 6370, 0 },
    { 103, 0, 6371, 0 },
    { 104, 0, 6372, 0 },
    { 116, 0, 6373, 0 },
    { 59, 0, 0, 8635 }, /* &circlearrowright; */
    { 109, 0, 6375, 0 },
    { 59, 1, 0, 8818 }, /* &lsim; */
    { 101, 1, 7593, 0 },
    { 103, 0, 8591, 0 },
    { 59, 0, 0, 179 }, /* &sup3; */
    { 111, 0, 6380, 0 },
    { 110, 0, 6381, 0 },
    { 59, 0, 0, 370 }, /* &Uogon; */
    { 99, 0, 6383, 0 },
    { 114, 0, 6384, 0 },
    { 111, 0, 6385, 0 },
    { 110, 0, 6386, 0 },
    { 59, 0, 0, 927 }, /* &Omicron; */
    { 59, 0, 0, 8884 }, /* &ltrie; */
    { 114, 0, 6389, 0 },
    { 59, 0, 0, 8488 }, /* &Zfr; */
    { 59, 0, 0, 8869 }, /* &perp; */
    { 111, 0, 6392, 0 },
    { 117, 0, 6393, 0 },
    { 98, 0, 6394, 0 },
    { 108, 0, 6395, 0 },
    { 101, 0, 6396, 0 },
    { 81, 0, 6397, 0 },
    { 117, 0, 6398, 0 },
    { 111, 0, 6399, 0 },
    { 116, 0, 6400, 0 },
    { 101, 0, 6401, 0 },
    { 59, 0, 0, 8220 }, /* &OpenCurlyDoubleQuote; */
    { 114, 0, 6403, 0 },
    { 97, 0, 6404, 0 },
    { 118, 0, 6405, 0 },
    { 101, 0, 6406, 192 }, /* &Agrave */
    { 59, 0, 0, 192 }, /* &Agrave; */
    { 99, 0, 6408, 0 },
    { 121, 0, 6409, 0 },
    { 59, 0, 0, 1070 }, /* &YUcy; */
    { 110, 0, 6411, 0 },
    { 86, 1, 6413, 0 },
    { 84, 0, 8971, 0 },
    { 101, 0, 6414, 0 },
    { 99, 0, 6415, 0 },
    { 116, 0, 6416, 0 },
    { 111, 0, 6417, 0 },
    { 114, 0, 6418, 0 },
    { 66, 1, 6420, 0 },
    { 59, 0, 0, 8642 }, /* &RightDownVector; */
    { 97, 0, 6421, 0 },
    { 114, 0, 6422, 0 },
    { 59, 0, 0, 10581 }, /* &RightDownVectorBar; */
    { 99, 0, 6424, 0 },
    { 117, 0, 6425, 0 },
    { 116, 0, 6426, 0 },
    { 101, 0, 6427, 237 }, /* &iacute */
    { 59, 0, 0, 237 }, /* &iacute; */
    { 116, 0, 6429, 162 }, /* &cent */
    { 59, 1, 0, 162 }, /* &cent; */
    { 101, 0, 6766, 0 },
    { 108, 1, 6433, 0 },
    { 115, 0, 8596, 0 },
    { 108, 0, 6434, 0 },
    { 115, 0, 6435, 0 },
    { 101, 0, 6436, 0 },
    { 116, 0, 6437, 0 },
    { 109, 0, 6438, 0 },
    { 105, 0, 6439, 0 },
    { 110, 0, 6440, 0 },
    { 117, 0, 6441, 0 },
    { 115, 0, 6442, 0 },
    { 59, 0, 0, 8726 }, /* &smallsetminus; */
    { 105, 0, 6444, 0 },
    { 116, 0, 6445, 0 },
    { 59, 0, 0, 9824 }, /* &spadesuit; */
    { 97, 0, 6447, 0 },
    { 114, 0, 6448, 0 },
    { 59, 0, 0, 8652 }, /* &rlhar; */
    { 114, 0, 6450, 0 },
    { 59, 0, 0, 120071 }, /* &Dfr; */
    { 107, 0, 6452, 0 },
    { 114, 1, 6454, 0 },
    { 108, 0, 7347, 0 },
    { 105, 0, 6455, 0 },
    { 103, 0, 6456, 0 },
    { 104, 0, 6457, 0 },
    { 116, 0, 6458, 0 },
    { 97, 0, 6459, 0 },
    { 114, 0, 6460, 0 },
    { 114, 0, 6461, 0 },
    { 111, 0, 6462, 0 },
    { 119, 0, 6463, 0 },
    { 59, 0, 0, 8618 }, /* &hookrightarrow; */
    { 105, 0, 6465, 0 },
    { 108, 0, 6466, 0 },
    { 100, 0, 6467, 0 },
    { 101, 0, 6468, 0 },
    { 59, 0, 0, 360 }, /* &Utilde; */
    { 97, 1, 6472, 0 },
    { 101, 1, 6794, 0 },
    { 105, 0, 6884, 0 },
    { 108, 0, 6473, 0 },
    { 115, 0, 6474, 0 },
    { 59, 0, 0, 61 }, /* &equals; */
    { 59, 0, 0, 1088 }, /* &rcy; */
    { 111, 0, 6477, 0 },
    { 108, 0, 6478, 0 },
    { 105, 0, 6479, 0 },
    { 110, 0, 6480, 0 },
    { 116, 0, 6481, 0 },
    { 59, 0, 0, 10771 }, /* &scpolint; */
    { 118, 0, 6483, 0 },
    { 59, 0, 0, 8463 }, /* &plankv; */
    { 113, 0, 6485, 0 },
    { 117, 0, 6486, 0 },
    { 97, 0, 6487, 0 },
    { 108, 0, 6488, 0 },
    { 59, 0, 0, 10927 }, /* &NotPrecedesEqual; */
    { 101, 0, 6490, 0 },
    { 102, 0, 6491, 0 },
    { 116, 0, 6492, 0 },
    { 101, 1, 6494, 0 },
    { 59, 0, 0, 9667 }, /* &triangleleft; */
    { 113, 0, 6495, 0 },
    { 59, 0, 0, 8884 }, /* &trianglelefteq; */
    { 114, 0, 6497, 0 },
    { 99, 0, 6498, 0 },
    { 59, 0, 0, 265 }, /* &ccirc; */
    { 108, 0, 6500, 0 },
    { 59, 1, 0, 92 }, /* &bsol; */
    { 104, 1, 9683, 0 },
    { 98, 0, 9753, 0 },
    { 59, 0, 0, 10966 }, /* &supsup; */
    { 111, 0, 6505, 0 },
    { 110, 0, 6506, 0 },
    { 59, 1, 0, 8759 }, /* &Colon; */
    { 101, 0, 7973, 0 },
    { 111, 0, 6509, 0 },
    { 110, 0, 6510, 0 },
    { 59, 0, 0, 303 }, /* &iogon; */
    { 99, 0, 6512, 0 },
    { 101, 1, 6514, 0 },
    { 59, 0, 0, 8833 }, /* &nsucc; */
    { 113, 0, 6515, 0 },
    { 59, 0, 0, 10928 }, /* &nsucceq; */
    { 59, 0, 0, 1089 }, /* &scy; */
    { 59, 1, 0, 8676 }, /* &larrb; */
    { 102, 0, 8931, 0 },
    { 119, 0, 6520, 0 },
    { 111, 0, 6521, 0 },
    { 59, 0, 0, 10791 }, /* &plustwo; */
    { 97, 0, 6523, 0 },
    { 118, 0, 6524, 0 },
    { 101, 0, 9117, 242 }, /* &ograve */
    { 101, 0, 6526, 0 },
    { 105, 0, 6527, 0 },
    { 108, 0, 6528, 0 },
    { 105, 0, 6529, 0 },
    { 110, 0, 6530, 0 },
    { 103, 0, 6531, 0 },
    { 59, 0, 0, 8968 }, /* &LeftCeiling; */
    { 115, 0, 6533, 0 },
    { 112, 0, 6534, 0 },
    { 59, 0, 0, 8201 }, /* &thinsp; */
    { 114, 0, 6536, 0 },
    { 59, 0, 0, 120001 }, /* &lscr; */
    { 113, 0, 6538, 0 },
    { 117, 0, 6539, 0 },
    { 97, 0, 6540, 0 },
    { 108, 0, 6541, 0 },
    { 59, 0, 0, 8772 }, /* &NotTildeEqual; */
    { 101, 0, 6543, 0 },
    { 102, 0, 6544, 0 },
    { 116, 0, 6545, 0 },
    { 65, 1, 6547, 0 },
    { 82, 0, 9490, 0 },
    { 114, 0, 6548, 0 },
    { 114, 0, 6549, 0 },
    { 111, 0, 6550, 0 },
    { 119, 0, 6551, 0 },
    { 59, 0, 0, 10229 }, /* &LongLeftArrow; */
    { 97, 0, 6553, 0 },
    { 115, 0, 6554, 0 },
    { 104, 0, 6555, 0 },
    { 59, 0, 0, 8876 }, /* &nvdash; */
    { 112, 1, 6559, 0 },
    { 100, 1, 7215, 0 },
    { 116, 0, 9333, 0 },
    { 108, 1, 6561, 0 },
    { 102, 0, 7108, 0 },
    { 117, 0, 6562, 0 },
    { 115, 0, 6563, 0 },
    { 59, 0, 0, 10753 }, /* &xoplus; */
    { 102, 0, 6565, 0 },
    { 59, 0, 0, 120161 }, /* &popf; */
    { 105, 0, 6567, 0 },
    { 100, 0, 6568, 0 },
    { 59, 0, 0, 8740 }, /* &nsmid; */
    { 98, 0, 6570, 0 },
    { 100, 0, 6571, 0 },
    { 97, 0, 6572, 0 },
    { 59, 0, 0, 955 }, /* &lambda; */
    { 105, 0, 6574, 0 },
    { 109, 0, 6575, 0 },
    { 59, 0, 0, 8831 }, /* &scsim; */
    { 112, 0, 6577, 0 },
    { 112, 0, 6578, 0 },
    { 114, 0, 6579, 0 },
    { 111, 0, 6580, 0 },
    { 120, 0, 6581, 0 },
    { 59, 0, 0, 10935 }, /* &precapprox; */
    { 97, 0, 6583, 0 },
    { 115, 0, 6584, 0 },
    { 104, 0, 6585, 0 },
    { 59, 0, 0, 8861 }, /* &circleddash; */
    { 101, 0, 6587, 0 },
    { 102, 0, 6588, 0 },
    { 116, 0, 6589, 0 },
    { 59, 0, 0, 9666 }, /* &blacktriangleleft; */
    { 99, 1, 6592, 0 },
    { 114, 0, 6708, 0 },
    { 117, 0, 6593, 0 },
    { 116, 0, 6594, 0 },
    { 101, 0, 7412, 218 }, /* &Uacute */
    { 108, 0, 7220, 207 }, /* &Iuml */
    { 114, 0, 6597, 0 },
    { 59, 0, 0, 120118 }, /* &yfr; */
    { 114, 0, 6599, 0 },
    { 59, 0, 0, 8496 }, /* &Escr; */
    { 110, 0, 6601, 0 },
    { 59, 0, 0, 8728 }, /* &compfn; */
    { 97, 0, 6603, 0 },
    { 114, 0, 6604, 0 },
    { 112, 0, 6605, 0 },
    { 111, 0, 6606, 0 },
    { 111, 0, 6607, 0 },
    { 110, 0, 6608, 0 },
    { 115, 0, 6609, 0 },
    { 59, 0, 0, 8651 }, /* &leftrightharpoons; */
    { 59, 0, 0, 9559 }, /* &boxDL; */
    { 113, 0, 6612, 0 },
    { 113, 0, 6613, 0 },
    { 59, 0, 0, 10934 }, /* &succneqq; */
    { 109, 0, 6615, 0 },
    { 101, 0, 6616, 0 },
    { 59, 0, 0, 8243 }, /* &Prime; */
    { 111, 0, 6618, 0 },
    { 119, 0, 6619, 0 },
    { 110, 0, 6620, 0 },
    { 97, 0, 6621, 0 },
    { 114, 0, 6622, 0 },
    { 114, 0, 6623, 0 },
    { 111, 0, 6624, 0 },
    { 119, 0, 6625, 0 },
    { 59, 0, 0, 8661 }, /* &Updownarrow; */
    { 97, 0, 6627, 0 },
    { 112, 0, 6628, 0 },
    { 115, 0, 6629, 0 },
    { 116, 0, 6630, 0 },
    { 111, 0, 6631, 0 },
    { 59, 0, 0, 10236 }, /* &longmapsto; */
    { 115, 0, 6633, 0 },
    { 101, 0, 6634, 0 },
    { 59, 0, 0, 10016 }, /* &maltese; */
    { 114, 0, 6636, 0 },
    { 116, 0, 6637, 0 },
    { 110, 0, 6638, 0 },
    { 101, 0, 6639, 0 },
    { 113, 0, 6640, 0 },
    { 113, 0, 6641, 0 },
    { 59, 0, 0, 8808 }, /* &lvertneqq; */
    { 59, 0, 0, 8741 }, /* &spar; */
    { 99, 0, 6644, 0 },
    { 111, 0, 6645, 0 },
    { 110, 0, 6646, 0 },
    { 59, 0, 0, 8889 }, /* &hercon; */
    { 108, 0, 6648, 0 },
    { 107, 0, 6649, 0 },
    { 59, 0, 0, 9600 }, /* &uhblk; */
    { 117, 0, 6651, 0 },
    { 115, 0, 6652, 0 },
    { 77, 0, 6653, 0 },
    { 105, 0, 6654, 0 },
    { 110, 0, 6655, 0 },
    { 117, 0, 6656, 0 },
    { 115, 0, 6657, 0 },
    { 59, 0, 0, 177 }, /* &PlusMinus; */
    { 108, 0, 6659, 0 },
    { 101, 0, 6660, 0 },
    { 115, 0, 6661, 0 },
    { 115, 0, 6662, 0 },
    { 59, 0, 0, 10892 }, /* &gtreqqless; */
    { 110, 0, 6664, 0 },
    { 118, 1, 6668, 0 },
    { 59, 1, 0, 8713 }, /* &notin; */
    { 100, 1, 8335, 0 },
    { 69, 0, 9195, 0 },
    { 98, 1, 6671, 0 },
    { 99, 1, 7151, 0 },
    { 97, 0, 8120, 0 },
    { 59, 0, 0, 8951 }, /* &notinvb; */
    { 99, 0, 6673, 0 },
    { 121, 0, 6674, 0 },
    { 59, 0, 0, 1039 }, /* &DZcy; */
    { 99, 0, 6676, 0 },
    { 114, 0, 6677, 0 },
    { 59, 0, 0, 119966 }, /* &Cscr; */
    { 97, 0, 6679, 0 },
    { 114, 0, 6680, 0 },
    { 112, 0, 6681, 0 },
    { 111, 0, 6682, 0 },
    { 111, 0, 6683, 0 },
    { 110, 0, 6684, 0 },
    { 114, 1, 6686, 0 },
    { 108, 0, 6776, 0 },
    { 105, 0, 6687, 0 },
    { 103, 0, 6688, 0 },
    { 104, 0, 6689, 0 },
    { 116, 0, 6690, 0 },
    { 59, 0, 0, 8638 }, /* &upharpoonright; */
    { 114, 0, 6692, 0 },
    { 59, 0, 0, 8500 }, /* &oscr; */
    { 59, 0, 0, 8923 }, /* &gel; */
    { 59, 0, 0, 8218 }, /* &lsquor; */
    { 101, 0, 6696, 0 },
    { 114, 0, 6697, 0 },
    { 76, 1, 6699, 0 },
    { 82, 0, 7553, 0 },
    { 101, 0, 6700, 0 },
    { 102, 0, 6701, 0 },
    { 116, 0, 6702, 0 },
    { 65, 0, 6703, 0 },
    { 114, 0, 6704, 0 },
    { 114, 0, 6705, 0 },
    { 111, 0, 6706, 0 },
    { 119, 0, 6707, 0 },
    { 59, 0, 0, 8598 }, /* &UpperLeftArrow; */
    { 114, 0, 6709, 0 },
    { 111, 1, 6711, 0 },
    { 59, 0, 0, 8607 }, /* &Uarr; */
    { 99, 0, 6712, 0 },
    { 105, 0, 6713, 0 },
    { 114, 0, 6714, 0 },
    { 59, 0, 0, 10569 }, /* &Uarrocir; */
    { 108, 0, 6716, 0 },
    { 105, 0, 6717, 0 },
    { 103, 0, 6718, 0 },
    { 59, 0, 0, 307 }, /* &ijlig; */
    { 103, 0, 6720, 0 },
    { 101, 0, 6721, 0 },
    { 114, 0, 6722, 0 },
    { 59, 0, 0, 8224 }, /* &dagger; */
    { 116, 0, 6724, 0 },
    { 105, 0, 6725, 0 },
    { 101, 0, 6726, 0 },
    { 59, 0, 0, 8904 }, /* &bowtie; */
    { 59, 0, 0, 194 }, /* &Acirc; */
    { 59, 0, 0, 177 }, /* &plusmn; */
    { 105, 0, 6730, 0 },
    { 109, 0, 6731, 0 },
    { 59, 0, 0, 8937 }, /* &succnsim; */
    { 59, 0, 0, 238 }, /* &icirc; */
    { 101, 0, 6734, 0 },
    { 101, 0, 6735, 0 },
    { 59, 1, 0, 8869 }, /* &UpTee; */
    { 65, 0, 9076, 0 },
    { 113, 1, 6740, 0 },
    { 59, 1, 0, 8805 }, /* &geq; */
    { 115, 0, 9268, 0 },
    { 59, 0, 0, 8807 }, /* &geqq; */
    { 105, 0, 6742, 0 },
    { 108, 0, 6743, 0 },
    { 59, 0, 0, 316 }, /* &lcedil; */
    { 99, 0, 6745, 0 },
    { 114, 0, 6746, 0 },
    { 59, 0, 0, 119980 }, /* &Qscr; */
    { 59, 0, 0, 10864 }, /* &napE; */
    { 102, 0, 6749, 0 },
    { 59, 0, 0, 8469 }, /* &Nopf; */
    { 114, 0, 6751, 0 },
    { 107, 0, 6752, 0 },
    { 59, 0, 0, 10098 }, /* &lbbrk; */
    { 97, 0, 6754, 0 },
    { 114, 0, 6755, 0 },
    { 116, 0, 6756, 0 },
    { 59, 0, 0, 8476 }, /* &realpart; */
    { 105, 0, 6758, 0 },
    { 109, 0, 6759, 0 },
    { 59, 0, 0, 8820 }, /* &nlsim; */
    { 102, 1, 6762, 0 },
    { 114, 0, 9615, 0 },
    { 59, 0, 0, 8660 }, /* &iff; */
    { 112, 0, 6764, 0 },
    { 102, 0, 6765, 0 },
    { 59, 0, 0, 8474 }, /* &Qopf; */
    { 114, 0, 6767, 0 },
    { 100, 0, 6768, 0 },
    { 111, 0, 6769, 0 },
    { 116, 0, 6770, 0 },
    { 59, 0, 0, 183 }, /* &centerdot; */
    { 113, 0, 6772, 0 },
    { 117, 0, 6773, 0 },
    { 97, 0, 6774, 0 },
    { 108, 0, 6775, 0 },
    { 59, 0, 0, 8930 }, /* &NotSquareSubsetEqual; */
    { 101, 0, 6777, 0 },
    { 102, 0, 6778, 0 },
    { 116, 0, 6779, 0 },
    { 59, 0, 0, 8639 }, /* &upharpoonleft; */
    { 114, 0, 6781, 0 },
    { 59, 0, 0, 120084 }, /* &Qfr; */
    { 111, 0, 6783, 0 },
    { 87, 0, 6784, 0 },
    { 105, 0, 6785, 0 },
    { 100, 0, 6786, 0 },
    { 116, 0, 6787, 0 },
    { 104, 0, 6788, 0 },
    { 83, 0, 6789, 0 },
    { 112, 0, 6790, 0 },
    { 97, 0, 6791, 0 },
    { 99, 0, 6792, 0 },
    { 101, 0, 6793, 0 },
    { 59, 0, 0, 8203 }, /* &ZeroWidthSpace; */
    { 115, 0, 6795, 0 },
    { 116, 0, 6796, 0 },
    { 59, 0, 0, 8799 }, /* &equest; */
    { 99, 0, 6798, 0 },
    { 97, 0, 6799, 0 },
    { 108, 0, 6800, 0 },
    { 59, 0, 0, 8890 }, /* &intercal; */
    { 99, 1, 6803, 0 },
    { 115, 0, 6806, 0 },
    { 117, 0, 6804, 0 },
    { 116, 0, 6805, 0 },
    { 101, 0, 9108, 243 }, /* &oacute */
    { 116, 0, 6807, 0 },
    { 59, 0, 0, 8859 }, /* &oast; */
    { 114, 0, 6809, 0 },
    { 101, 0, 6810, 0 },
    { 118, 0, 6811, 0 },
    { 101, 0, 6812, 0 },
    { 59, 0, 0, 258 }, /* &Abreve; */
    { 104, 0, 6814, 0 },
    { 114, 0, 6815, 0 },
    { 101, 0, 6816, 0 },
    { 101, 0, 6817, 0 },
    { 116, 0, 6818, 0 },
    { 105, 0, 6819, 0 },
    { 109, 0, 6820, 0 },
    { 101, 0, 6821, 0 },
    { 115, 0, 6822, 0 },
    { 59, 0, 0, 8908 }, /* &rightthreetimes; */
    { 114, 0, 6824, 0 },
    { 114, 0, 6825, 0 },
    { 59, 0, 0, 8659 }, /* &dArr; */
    { 59, 0, 0, 8840 }, /* &nsube; */
    { 59, 0, 0, 231 }, /* &ccedil; */
    { 101, 1, 6833, 0 },
    { 103, 1, 7578, 0 },
    { 115, 1, 7850, 0 },
    { 97, 1, 7954, 0 },
    { 100, 0, 8625, 0 },
    { 113, 0, 6834, 0 },
    { 103, 1, 6836, 0 },
    { 113, 0, 7534, 0 },
    { 116, 0, 6837, 0 },
    { 114, 0, 6838, 0 },
    { 59, 0, 0, 8922 }, /* &lesseqgtr; */
    { 97, 0, 6840, 0 },
    { 115, 0, 6841, 0 },
    { 117, 0, 6842, 0 },
    { 114, 0, 6843, 0 },
    { 101, 0, 6844, 0 },
    { 100, 0, 6845, 0 },
    { 97, 0, 6846, 0 },
    { 110, 0, 6847, 0 },
    { 103, 0, 6848, 0 },
    { 108, 0, 6849, 0 },
    { 101, 0, 6850, 0 },
    { 59, 0, 0, 8737 }, /* &measuredangle; */
    { 59, 1, 0, 8869 }, /* &bot; */
    { 116, 0, 9757, 0 },
    { 107, 0, 6854, 0 },
    { 59, 0, 0, 10532 }, /* &nearhk; */
    { 108, 0, 6856, 0 },
    { 107, 0, 6857, 0 },
    { 59, 0, 0, 9604 }, /* &lhblk; */
    { 59, 0, 0, 120108 }, /* &ofr; */
    { 59, 1, 0, 8922 }, /* &lesg; */
    { 101, 0, 7485, 0 },
    { 110, 0, 6862, 0 },
    { 103, 0, 6863, 0 },
    { 76, 1, 6865, 0 },
    { 82, 0, 8940, 0 },
    { 101, 0, 6866, 0 },
    { 102, 0, 6867, 0 },
    { 116, 0, 6868, 0 },
    { 65, 1, 6870, 0 },
    { 82, 0, 7152, 0 },
    { 114, 0, 6871, 0 },
    { 114, 0, 6872, 0 },
    { 111, 0, 6873, 0 },
    { 119, 0, 6874, 0 },
    { 59, 0, 0, 10232 }, /* &DoubleLongLeftArrow; */
    { 112, 0, 6876, 0 },
    { 49, 1, 6878, 0 },
    { 59, 0, 0, 8195 }, /* &emsp; */
    { 52, 1, 6880, 0 },
    { 51, 0, 8308, 0 },
    { 59, 0, 0, 8197 }, /* &emsp14; */
    { 110, 0, 6882, 0 },
    { 116, 0, 6883, 0 },
    { 59, 0, 0, 37 }, /* &percnt; */
    { 118, 0, 6885, 0 },
    { 68, 1, 6887, 0 },
    { 59, 0, 0, 8801 }, /* &equiv; */
    { 68, 0, 6888, 0 },
    { 59, 0, 0, 10872 }, /* &equivDD; */
    { 101, 0, 6890, 0 },
    { 116, 0, 6891, 0 },
    { 59, 0, 0, 9141 }, /* &UnderBracket; */
    { 101, 0, 6893, 0 },
    { 105, 0, 6894, 0 },
    { 108, 0, 6895, 0 },
    { 105, 0, 6896, 0 },
    { 110, 0, 6897, 0 },
    { 103, 0, 6898, 0 },
    { 59, 0, 0, 8969 }, /* &RightCeiling; */
    { 105, 0, 6900, 0 },
    { 110, 0, 6901, 0 },
    { 117, 0, 6902, 0 },
    { 115, 0, 6903, 0 },
    { 59, 0, 0, 10810 }, /* &triminus; */
    { 115, 0, 6905, 0 },
    { 59, 0, 0, 329 }, /* &napos; */
    { 114, 0, 6907, 0 },
    { 59, 0, 0, 10821 }, /* &cupor; */
    { 114, 0, 6909, 0 },
    { 110, 0, 6910, 0 },
    { 105, 0, 6911, 0 },
    { 111, 0, 6912, 0 },
    { 110, 0, 6913, 0 },
    { 115, 0, 6914, 0 },
    { 59, 0, 0, 8461 }, /* &quaternions; */
    { 59, 0, 0, 250 }, /* &uacute; */
    { 99, 0, 6917, 0 },
    { 114, 0, 6918, 0 },
    { 59, 0, 0, 119986 }, /* &Wscr; */
    { 111, 0, 6920, 0 },
    { 110, 0, 6921, 0 },
    { 59, 0, 0, 260 }, /* &Aogon; */
    { 105, 0, 6923, 0 },
    { 108, 0, 6924, 0 },
    { 100, 0, 6925, 0 },
    { 101, 0, 6926, 0 },
    { 59, 0, 0, 8768 }, /* &VerticalTilde; */
    { 102, 0, 6928, 0 },
    { 59, 0, 0, 120146 }, /* &aopf; */
    { 59, 0, 0, 8922 }, /* &leg; */
    { 97, 0, 6931, 0 },
    { 115, 0, 6932, 0 },
    { 104, 0, 6933, 0 },
    { 59, 0, 0, 8212 }, /* &mdash; */
    { 103, 0, 6935, 0 },
    { 59, 0, 0, 9834 }, /* &sung; */
    { 105, 0, 6937, 0 },
    { 110, 0, 6938, 0 },
    { 101, 0, 6939, 0 },
    { 59, 0, 0, 8978 }, /* &profline; */
    { 114, 0, 6941, 0 },
    { 59, 0, 0, 8230 }, /* &mldr; */
    { 114, 0, 6943, 0 },
    { 111, 0, 6944, 0 },
    { 107, 0, 6945, 0 },
    { 59, 0, 0, 273 }, /* &dstrok; */
    { 59, 0, 0, 120111 }, /* &rfr; */
    { 101, 1, 6949, 0 },
    { 59, 0, 0, 8748 }, /* &Int; */
    { 103, 1, 6951, 0 },
    { 114, 0, 7257, 0 },
    { 114, 0, 6952, 0 },
    { 97, 0, 6953, 0 },
    { 108, 0, 6954, 0 },
    { 59, 0, 0, 8747 }, /* &Integral; */
    { 59, 0, 0, 8726 }, /* &setmn; */
    { 116, 0, 6957, 0 },
    { 121, 0, 6958, 0 },
    { 59, 1, 0, 8709 }, /* &empty; */
    { 115, 1, 7293, 0 },
    { 118, 0, 8081, 0 },
    { 114, 0, 6962, 0 },
    { 105, 0, 6963, 0 },
    { 101, 0, 6964, 0 },
    { 59, 0, 0, 8885 }, /* &nvrtrie; */
    { 114, 0, 6966, 0 },
    { 114, 0, 6967, 0 },
    { 59, 0, 0, 10234 }, /* &xhArr; */
    { 59, 1, 0, 8804 }, /* &leq; */
    { 115, 1, 7455, 0 },
    { 113, 0, 7880, 0 },
    { 59, 0, 0, 9516 }, /* &boxhd; */
    { 97, 0, 6973, 0 },
    { 115, 0, 6974, 0 },
    { 104, 0, 6975, 0 },
    { 59, 0, 0, 8879 }, /* &nVDash; */
    { 59, 0, 0, 239 }, /* &iuml; */
    { 101, 0, 6978, 0 },
    { 101, 0, 6979, 0 },
    { 59, 0, 0, 8897 }, /* &xvee; */
    { 112, 1, 6982, 0 },
    { 117, 0, 8161, 0 },
    { 102, 0, 6983, 0 },
    { 59, 0, 0, 8477 }, /* &Ropf; */
    { 59, 0, 0, 208 }, /* &ETH; */
    { 108, 0, 6986, 0 },
    { 97, 0, 6987, 0 },
    { 99, 0, 6988, 0 },
    { 59, 0, 0, 337 }, /* &odblac; */
    { 115, 0, 6990, 0 },
    { 105, 0, 6991, 0 },
    { 108, 0, 6992, 0 },
    { 111, 0, 6993, 0 },
    { 110, 0, 6994, 0 },
    { 59, 0, 0, 917 }, /* &Epsilon; */
    { 113, 0, 6996, 0 },
    { 59, 0, 0, 8791 }, /* &circeq; */
    { 59, 0, 0, 8796 }, /* &triangleq; */
    { 114, 0, 6999, 0 },
    { 59, 1, 0, 9734 }, /* &star; */
    { 102, 0, 7700, 0 },
    { 116, 0, 7002, 0 },
    { 59, 0, 0, 9645 }, /* &rect; */
    { 99, 0, 7004, 0 },
    { 117, 0, 7005, 0 },
    { 116, 0, 7006, 0 },
    { 101, 0, 7007, 0 },
    { 59, 0, 0, 346 }, /* &Sacute; */
    { 117, 0, 7009, 0 },
    { 99, 0, 7010, 0 },
    { 99, 0, 7011, 0 },
    { 59, 0, 0, 8927 }, /* &curlyeqsucc; */
    { 97, 0, 7013, 0 },
    { 114, 0, 7014, 0 },
    { 59, 0, 0, 10536 }, /* &nesear; */
    { 116, 0, 7016, 0 },
    { 114, 1, 7018, 0 },
    { 97, 0, 7500, 0 },
    { 105, 0, 7019, 0 },
    { 103, 0, 7020, 0 },
    { 104, 0, 7021, 0 },
    { 116, 0, 7022, 0 },
    { 97, 0, 7023, 0 },
    { 114, 0, 7024, 0 },
    { 114, 0, 7025, 0 },
    { 111, 0, 7026, 0 },
    { 119, 0, 7027, 0 },
    { 59, 0, 0, 8622 }, /* &nleftrightarrow; */
    { 99, 0, 7029, 0 },
    { 114, 0, 7030, 0 },
    { 59, 0, 0, 119974 }, /* &Kscr; */
    { 101, 0, 7032, 0 },
    { 109, 0, 7033, 0 },
    { 101, 0, 7034, 0 },
    { 110, 0, 7035, 0 },
    { 116, 0, 7036, 0 },
    { 59, 0, 0, 8713 }, /* &NotElement; */
    { 109, 0, 7038, 0 },
    { 97, 0, 7039, 0 },
    { 102, 1, 7042, 0 },
    { 118, 1, 7545, 0 },
    { 59, 0, 0, 963 }, /* &sigma; */
    { 59, 0, 0, 962 }, /* &sigmaf; */
    { 102, 0, 7044, 0 },
    { 59, 0, 0, 120164 }, /* &sopf; */
    { 111, 0, 7046, 0 },
    { 116, 0, 7047, 0 },
    { 59, 0, 0, 8945 }, /* &dtdot; */
    { 113, 0, 7049, 0 },
    { 117, 0, 7050, 0 },
    { 105, 0, 7051, 0 },
    { 103, 0, 7052, 0 },
    { 97, 0, 7053, 0 },
    { 114, 0, 7054, 0 },
    { 114, 0, 7055, 0 },
    { 111, 0, 7056, 0 },
    { 119, 0, 7057, 0 },
    { 59, 0, 0, 8621 }, /* &leftrightsquigarrow; */
    { 108, 0, 7059, 0 },
    { 111, 0, 7060, 0 },
    { 112, 0, 7061, 0 },
    { 101, 0, 7062, 0 },
    { 59, 0, 0, 10840 }, /* &andslope; */
    { 105, 1, 7065, 0 },
    { 121, 0, 7624, 0 },
    { 114, 0, 7066, 0 },
    { 99, 0, 7067, 0 },
    { 59, 0, 0, 309 }, /* &jcirc; */
    { 101, 1, 7070, 0 },
    { 59, 0, 0, 8768 }, /* &wr; */
    { 97, 0, 7071, 0 },
    { 116, 0, 7072, 0 },
    { 104, 0, 7073, 0 },
    { 59, 0, 0, 8768 }, /* &wreath; */
    { 101, 0, 7075, 0 },
    { 102, 0, 7076, 0 },
    { 116, 0, 7077, 0 },
    { 65, 0, 7078, 0 },
    { 114, 0, 7079, 0 },
    { 114, 0, 7080, 0 },
    { 111, 0, 7081, 0 },
    { 119, 0, 7082, 0 },
    { 59, 0, 0, 8601 }, /* &LowerLeftArrow; */
    { 101, 1, 7085, 0 },
    { 111, 0, 8254, 0 },
    { 116, 0, 7086, 0 },
    { 59, 0, 0, 8257 }, /* &caret; */
    { 101, 0, 7088, 0 },
    { 52, 1, 7090, 0 },
    { 102, 0, 9191, 0 },
    { 59, 0, 0, 8756 }, /* &there4; */
    { 59, 0, 0, 9568 }, /* &boxVR; */
    { 108, 0, 7093, 0 },
    { 59, 0, 0, 8969 }, /* &rceil; */
    { 59, 0, 0, 8807 }, /* &ngeqq; */
    { 101, 0, 7096, 0 },
    { 101, 0, 7097, 0 },
    { 86, 0, 7098, 0 },
    { 101, 0, 7099, 0 },
    { 99, 0, 7100, 0 },
    { 116, 0, 7101, 0 },
    { 111, 0, 7102, 0 },
    { 114, 0, 7103, 0 },
    { 59, 0, 0, 10590 }, /* &DownLeftTeeVector; */
    { 101, 0, 7105, 0 },
    { 115, 0, 7106, 0 },
    { 59, 0, 0, 10807 }, /* &Otimes; */
    { 59, 0, 0, 251 }, /* &ucirc; */
    { 59, 0, 0, 120169 }, /* &xopf; */
    { 121, 0, 7110, 0 },
    { 59, 0, 0, 1041 }, /* &Bcy; */
    { 105, 0, 7112, 0 },
    { 108, 0, 7113, 0 },
    { 100, 0, 7114, 0 },
    { 101, 0, 7115, 0 },
    { 59, 0, 0, 8819 }, /* &GreaterTilde; */
    { 59, 1, 0, 8600 }, /* &searr; */
    { 111, 0, 7139, 0 },
    { 111, 0, 7119, 0 },
    { 119, 0, 7120, 0 },
    { 110, 0, 7121, 0 },
    { 59, 0, 0, 9663 }, /* &triangledown; */
    { 97, 0, 7123, 0 },
    { 114, 0, 7124, 0 },
    { 114, 0, 7125, 0 },
    { 59, 0, 0, 10511 }, /* &rBarr; */
    { 103, 1, 7128, 0 },
    { 59, 0, 0, 8806 }, /* &lE; */
    { 59, 0, 0, 10891 }, /* &lEg; */
    { 111, 0, 7130, 0 },
    { 108, 0, 7131, 0 },
    { 100, 0, 7132, 0 },
    { 59, 0, 0, 10684 }, /* &odsold; */
    { 101, 0, 7134, 0 },
    { 59, 0, 0, 10216 }, /* &langle; */
    { 99, 0, 7136, 0 },
    { 121, 0, 7137, 0 },
    { 59, 0, 0, 1033 }, /* &LJcy; */
    { 59, 0, 0, 171 }, /* &laquo; */
    { 119, 0, 7140, 0 },
    { 59, 0, 0, 8600 }, /* &searrow; */
    { 110, 1, 7143, 0 },
    { 101, 0, 9454, 0 },
    { 69, 0, 7144, 0 },
    { 59, 0, 0, 8809 }, /* &gvnE; */
    { 97, 0, 7146, 0 },
    { 114, 0, 7147, 0 },
    { 59, 0, 0, 10703 }, /* &LeftTriangleBar; */
    { 108, 1, 7150, 0 },
    { 59, 0, 0, 8807 }, /* &gE; */
    { 59, 0, 0, 10892 }, /* &gEl; */
    { 59, 0, 0, 8950 }, /* &notinvc; */
    { 105, 0, 7153, 0 },
    { 103, 0, 7154, 0 },
    { 104, 0, 7155, 0 },
    { 116, 0, 7156, 0 },
    { 65, 0, 7157, 0 },
    { 114, 0, 7158, 0 },
    { 114, 0, 7159, 0 },
    { 111, 0, 7160, 0 },
    { 119, 0, 7161, 0 },
    { 59, 0, 0, 10234 }, /* &DoubleLongLeftRightArrow; */
    { 116, 0, 7163, 0 },
    { 59, 0, 0, 10038 }, /* &sext; */
    { 105, 0, 7165, 0 },
    { 103, 0, 7166, 0 },
    { 104, 0, 7167, 0 },
    { 116, 0, 7168, 0 },
    { 65, 0, 7169, 0 },
    { 114, 0, 7170, 0 },
    { 114, 0, 7171, 0 },
    { 111, 0, 7172, 0 },
    { 119, 0, 7173, 0 },
    { 59, 0, 0, 8594 }, /* &ShortRightArrow; */
    { 116, 0, 7175, 0 },
    { 101, 0, 7176, 0 },
    { 114, 0, 7177, 0 },
    { 59, 0, 0, 10862 }, /* &easter; */
    { 99, 0, 7179, 0 },
    { 104, 0, 7180, 0 },
    { 102, 0, 7181, 0 },
    { 111, 0, 7182, 0 },
    { 114, 0, 7183, 0 },
    { 107, 0, 7184, 0 },
    { 59, 0, 0, 8916 }, /* &pitchfork; */
    { 114, 0, 7186, 0 },
    { 114, 0, 7187, 0 },
    { 111, 0, 7188, 0 },
    { 119, 0, 7189, 0 },
    { 59, 0, 0, 8658 }, /* &Rightarrow; */
    { 97, 0, 7191, 0 },
    { 114, 0, 7192, 0 },
    { 59, 0, 0, 8677 }, /* &RightArrowBar; */
    { 107, 0, 7194, 0 },
    { 59, 0, 0, 8617 }, /* &larrhk; */
    { 59, 0, 0, 228 }, /* &auml; */
    { 121, 1, 7198, 0 },
    { 104, 0, 9324, 0 },
    { 59, 0, 0, 1096 }, /* &shcy; */
    { 59, 0, 0, 8841 }, /* &nsupe; */
    { 105, 1, 7202, 0 },
    { 114, 0, 9461, 0 },
    { 114, 0, 7203, 0 },
    { 59, 0, 0, 10686 }, /* &olcir; */
    { 101, 0, 7205, 0 },
    { 101, 0, 7206, 0 },
    { 59, 0, 0, 8872 }, /* &DoubleRightTee; */
    { 114, 0, 7208, 0 },
    { 59, 0, 0, 8471 }, /* &copysr; */
    { 59, 0, 0, 977 }, /* &thetav; */
    { 102, 0, 7211, 0 },
    { 59, 0, 0, 120151 }, /* &fopf; */
    { 114, 0, 7213, 0 },
    { 99, 0, 7214, 0 },
    { 59, 0, 0, 9711 }, /* &bigcirc; */
    { 111, 0, 7216, 0 },
    { 116, 0, 7217, 0 },
    { 59, 0, 0, 10752 }, /* &xodot; */
    { 112, 0, 7219, 0 },
    { 59, 0, 0, 8620 }, /* &rarrlp; */
    { 59, 0, 0, 207 }, /* &Iuml; */
    { 105, 0, 7222, 0 },
    { 108, 0, 7223, 0 },
    { 100, 0, 7224, 0 },
    { 101, 0, 7225, 0 },
    { 59, 0, 0, 8821 }, /* &NotGreaterTilde; */
    { 111, 0, 7227, 0 },
    { 120, 0, 7228, 0 },
    { 59, 0, 0, 10697 }, /* &boxbox; */
    { 104, 0, 7230, 0 },
    { 59, 0, 0, 305 }, /* &imath; */
    { 99, 1, 7233, 0 },
    { 105, 0, 8426, 0 },
    { 111, 0, 7234, 0 },
    { 110, 0, 7235, 0 },
    { 105, 0, 7236, 0 },
    { 110, 0, 7237, 0 },
    { 116, 0, 7238, 0 },
    { 59, 0, 0, 8754 }, /* &cwconint; */
    { 100, 0, 7240, 0 },
    { 105, 0, 7241, 0 },
    { 108, 0, 7242, 0 },
    { 59, 0, 0, 325 }, /* &Ncedil; */
    { 97, 0, 7244, 0 },
    { 114, 0, 7245, 0 },
    { 111, 0, 7246, 0 },
    { 119, 0, 7247, 0 },
    { 59, 0, 0, 10534 }, /* &hkswarow; */
    { 105, 0, 7249, 0 },
    { 109, 0, 7250, 0 },
    { 59, 0, 0, 8935 }, /* &gnsim; */
    { 114, 1, 7253, 0 },
    { 105, 0, 7669, 0 },
    { 59, 0, 0, 120114 }, /* &ufr; */
    { 100, 0, 7255, 0 },
    { 101, 0, 7256, 0 },
    { 59, 0, 0, 732 }, /* &tilde; */
    { 115, 0, 7258, 0 },
    { 101, 0, 7259, 0 },
    { 99, 0, 7260, 0 },
    { 116, 0, 7261, 0 },
    { 105, 0, 7262, 0 },
    { 111, 0, 7263, 0 },
    { 110, 0, 7264, 0 },
    { 59, 0, 0, 8898 }, /* &Intersection; */
    { 101, 0, 7266, 0 },
    { 115, 0, 7267, 0 },
    { 59, 1, 0, 8855 }, /* &otimes; */
    { 97, 0, 8786, 0 },
    { 115, 1, 7272, 0 },
    { 59, 1, 0, 35 }, /* &num; */
    { 101, 0, 9400, 0 },
    { 112, 0, 7273, 0 },
    { 59, 0, 0, 8199 }, /* &numsp; */
    { 97, 0, 7275, 0 },
    { 108, 0, 7276, 0 },
    { 115, 0, 7277, 0 },
    { 59, 0, 0, 8474 }, /* &rationals; */
    { 108, 0, 7279, 0 },
    { 97, 0, 7280, 0 },
    { 99, 0, 7281, 0 },
    { 101, 0, 7282, 0 },
    { 116, 0, 7283, 0 },
    { 114, 0, 7284, 0 },
    { 102, 0, 7285, 0 },
    { 59, 0, 0, 8466 }, /* &Laplacetrf; */
    { 101, 0, 7287, 0 },
    { 101, 0, 7288, 0 },
    { 59, 0, 0, 8910 }, /* &curlyvee; */
    { 102, 0, 7290, 0 },
    { 59, 0, 0, 8887 }, /* &imof; */
    { 53, 0, 7292, 0 },
    { 59, 0, 0, 8536 }, /* &frac45; */
    { 101, 0, 7294, 0 },
    { 116, 0, 7295, 0 },
    { 59, 0, 0, 8709 }, /* &emptyset; */
    { 114, 0, 7297, 0 },
    { 111, 0, 7298, 0 },
    { 100, 0, 7299, 0 },
    { 59, 0, 0, 10812 }, /* &intprod; */
    { 114, 0, 7301, 0 },
    { 111, 0, 7302, 0 },
    { 110, 0, 7303, 0 },
    { 59, 0, 0, 356 }, /* &Tcaron; */
    { 115, 0, 7305, 0 },
    { 59, 0, 0, 39 }, /* &apos; */
    { 108, 0, 7307, 0 },
    { 97, 0, 7308, 0 },
    { 114, 0, 7309, 0 },
    { 59, 0, 0, 36 }, /* &dollar; */
    { 114, 0, 7311, 0 },
    { 59, 0, 0, 8609 }, /* &Darr; */
    { 111, 0, 7313, 0 },
    { 116, 0, 7314, 0 },
    { 59, 0, 0, 9708 }, /* &tridot; */
    { 110, 0, 7316, 0 },
    { 116, 0, 7317, 0 },
    { 101, 0, 7318, 0 },
    { 114, 0, 7319, 0 },
    { 67, 0, 7320, 0 },
    { 108, 0, 7321, 0 },
    { 111, 0, 7322, 0 },
    { 99, 0, 7323, 0 },
    { 107, 0, 7324, 0 },
    { 119, 0, 7325, 0 },
    { 105, 0, 7326, 0 },
    { 115, 0, 7327, 0 },
    { 101, 0, 7328, 0 },
    { 67, 0, 7329, 0 },
    { 111, 0, 7330, 0 },
    { 110, 0, 7331, 0 },
    { 116, 0, 7332, 0 },
    { 111, 0, 7333, 0 },
    { 117, 0, 7334, 0 },
    { 114, 0, 7335, 0 },
    { 73, 0, 7336, 0 },
    { 110, 0, 7337, 0 },
    { 116, 0, 7338, 0 },
    { 101, 0, 7339, 0 },
    { 103, 0, 7340, 0 },
    { 114, 0, 7341, 0 },
    { 97, 0, 7342, 0 },
    { 108, 0, 7343, 0 },
    { 59, 0, 0, 8755 }, /* &CounterClockwiseContourIntegral; */
    { 59, 0, 0, 8472 }, /* &wp; */
    { 59, 1, 0, 10925 }, /* &late; */
    { 115, 0, 8205, 0 },
    { 101, 0, 7348, 0 },
    { 102, 0, 7349, 0 },
    { 116, 0, 7350, 0 },
    { 97, 0, 7351, 0 },
    { 114, 0, 7352, 0 },
    { 114, 0, 7353, 0 },
    { 111, 0, 7354, 0 },
    { 119, 0, 7355, 0 },
    { 59, 0, 0, 8617 }, /* &hookleftarrow; */
    { 112, 0, 7357, 0 },
    { 102, 0, 7358, 0 },
    { 59, 0, 0, 120155 }, /* &jopf; */
    { 105, 0, 7360, 0 },
    { 108, 0, 7361, 0 },
    { 100, 0, 7362, 0 },
    { 101, 0, 7363, 0 },
    { 59, 0, 0, 8830 }, /* &PrecedesTilde; */
    { 99, 1, 7366, 0 },
    { 105, 0, 9307, 0 },
    { 114, 0, 7367, 0 },
    { 59, 0, 0, 119979 }, /* &Pscr; */
    { 101, 1, 7371, 0 },
    { 115, 1, 8342, 0 },
    { 59, 0, 0, 8847 }, /* &sqsub; */
    { 59, 0, 0, 8849 }, /* &sqsube; */
    { 108, 0, 7373, 0 },
    { 105, 0, 7374, 0 },
    { 103, 0, 7375, 0 },
    { 59, 0, 0, 102 }, /* &fjlig; */
    { 59, 0, 0, 8478 }, /* &rx; */
    { 59, 0, 0, 9565 }, /* &boxUL; */
    { 72, 1, 7380, 0 },
    { 99, 0, 9473, 0 },
    { 99, 0, 7381, 0 },
    { 121, 0, 7382, 0 },
    { 59, 0, 0, 1035 }, /* &TSHcy; */
    { 105, 0, 7384, 0 },
    { 103, 0, 7385, 0 },
    { 104, 0, 7386, 0 },
    { 116, 0, 7387, 0 },
    { 59, 0, 0, 9656 }, /* &blacktriangleright; */
    { 101, 0, 7389, 0 },
    { 112, 0, 7390, 0 },
    { 97, 0, 7391, 0 },
    { 114, 0, 7392, 0 },
    { 97, 0, 7393, 0 },
    { 116, 0, 7394, 0 },
    { 111, 0, 7395, 0 },
    { 114, 0, 7396, 0 },
    { 59, 0, 0, 10072 }, /* &VerticalSeparator; */
    { 108, 0, 7398, 0 },
    { 59, 0, 0, 8611 }, /* &rarrtl; */
    { 99, 1, 7401, 0 },
    { 116, 0, 8154, 0 },
    { 111, 1, 7403, 0 },
    { 114, 0, 8668, 0 },
    { 114, 0, 7404, 0 },
    { 110, 0, 7405, 0 },
    { 101, 1, 7407, 0 },
    { 59, 0, 0, 8988 }, /* &ulcorn; */
    { 114, 0, 7408, 0 },
    { 59, 0, 0, 8988 }, /* &ulcorner; */
    { 114, 0, 7410, 0 },
    { 114, 0, 7411, 0 },
    { 59, 0, 0, 8635 }, /* &orarr; */
    { 59, 0, 0, 218 }, /* &Uacute; */
    { 103, 0, 7414, 0 },
    { 59, 0, 0, 10815 }, /* &amalg; */
    { 59, 0, 0, 38 }, /* &AMP; */
    { 121, 0, 7417, 0 },
    { 59, 0, 0, 1087 }, /* &pcy; */
    { 59, 0, 0, 8778 }, /* &ape; */
    { 116, 0, 7420, 0 },
    { 101, 0, 7421, 0 },
    { 114, 0, 7422, 0 },
    { 68, 0, 7423, 0 },
    { 111, 0, 7424, 0 },
    { 116, 0, 7425, 0 },
    { 59, 0, 0, 183 }, /* &CenterDot; */
    { 102, 0, 7427, 0 },
    { 59, 0, 0, 120159 }, /* &nopf; */
    { 97, 0, 7429, 0 },
    { 108, 0, 7430, 0 },
    { 108, 0, 7431, 0 },
    { 67, 0, 7432, 0 },
    { 105, 0, 7433, 0 },
    { 114, 0, 7434, 0 },
    { 99, 0, 7435, 0 },
    { 108, 0, 7436, 0 },
    { 101, 0, 7437, 0 },
    { 59, 0, 0, 8728 }, /* &SmallCircle; */
    { 59, 0, 0, 185 }, /* &sup1; */
    { 114, 0, 7440, 0 },
    { 99, 0, 7441, 0 },
    { 59, 0, 0, 308 }, /* &Jcirc; */
    { 112, 0, 7443, 160 }, /* &nbsp */
    { 59, 0, 0, 160 }, /* &nbsp; */
    { 105, 0, 7445, 0 },
    { 59, 0, 0, 934 }, /* &Phi; */
    { 102, 0, 7447, 0 },
    { 59, 0, 0, 120124 }, /* &Eopf; */
    { 59, 0, 0, 926 }, /* &Xi; */
    { 114, 0, 7450, 0 },
    { 59, 0, 0, 120110 }, /* &qfr; */
    { 114, 0, 7452, 0 },
    { 99, 0, 7453, 0 },
    { 59, 0, 0, 264 }, /* &Ccirc; */
    { 59, 0, 0, 167 }, /* &sect; */
    { 108, 0, 7456, 0 },
    { 97, 0, 7457, 0 },
    { 110, 0, 7458, 0 },
    { 116, 0, 7459, 0 },
    { 59, 0, 0, 10877 }, /* &leqslant; */
    { 104, 0, 7461, 0 },
    { 101, 0, 7462, 0 },
    { 110, 0, 7463, 0 },
    { 59, 0, 0, 8208 }, /* &hyphen; */
    { 101, 0, 7465, 0 },
    { 102, 0, 7466, 0 },
    { 116, 0, 7467, 0 },
    { 97, 1, 7469, 0 },
    { 114, 0, 9342, 0 },
    { 114, 0, 7470, 0 },
    { 114, 0, 7471, 0 },
    { 111, 0, 7472, 0 },
    { 119, 0, 7473, 0 },
    { 59, 0, 0, 10232 }, /* &Longleftarrow; */
    { 97, 0, 7475, 0 },
    { 112, 0, 7476, 0 },
    { 112, 0, 7477, 0 },
    { 97, 0, 7478, 0 },
    { 59, 0, 0, 1008 }, /* &varkappa; */
    { 59, 0, 0, 181 }, /* &micro; */
    { 59, 0, 0, 8766 }, /* &acE; */
    { 111, 0, 7482, 0 },
    { 116, 0, 7483, 0 },
    { 59, 0, 0, 267 }, /* &cdot; */
    { 59, 0, 0, 1079 }, /* &zcy; */
    { 115, 0, 7486, 0 },
    { 59, 0, 0, 10899 }, /* &lesges; */
    { 108, 0, 7488, 0 },
    { 117, 0, 7489, 0 },
    { 115, 0, 7490, 0 },
    { 59, 0, 0, 10753 }, /* &bigoplus; */
    { 99, 0, 7492, 0 },
    { 114, 0, 7493, 0 },
    { 59, 0, 0, 119995 }, /* &fscr; */
    { 99, 0, 7495, 0 },
    { 117, 0, 7496, 0 },
    { 116, 0, 7497, 0 },
    { 101, 0, 7992, 193 }, /* &Aacute */
    { 114, 0, 7499, 0 },
    { 59, 0, 0, 8229 }, /* &nldr; */
    { 114, 0, 7501, 0 },
    { 114, 0, 7502, 0 },
    { 111, 0, 7503, 0 },
    { 119, 0, 7504, 0 },
    { 59, 0, 0, 8602 }, /* &nleftarrow; */
    { 114, 0, 7506, 0 },
    { 111, 0, 7507, 0 },
    { 110, 0, 7508, 0 },
    { 59, 0, 0, 344 }, /* &Rcaron; */
    { 117, 0, 7510, 0 },
    { 101, 0, 7511, 0 },
    { 59, 0, 0, 8929 }, /* &nsccue; */
    { 117, 0, 7513, 0 },
    { 109, 0, 7514, 0 },
    { 112, 0, 7515, 0 },
    { 68, 1, 7517, 0 },
    { 69, 0, 9500, 0 },
    { 111, 0, 7518, 0 },
    { 119, 0, 7519, 0 },
    { 110, 0, 7520, 0 },
    { 72, 0, 7521, 0 },
    { 117, 0, 7522, 0 },
    { 109, 0, 7523, 0 },
    { 112, 0, 7524, 0 },
    { 59, 0, 0, 8782 }, /* &NotHumpDownHump; */
    { 97, 0, 7526, 0 },
    { 114, 0, 7527, 0 },
    { 59, 0, 0, 8742 }, /* &nspar; */
    { 98, 0, 7529, 0 },
    { 59, 0, 0, 123 }, /* &lcub; */
    { 59, 0, 0, 10636 }, /* &rbrke; */
    { 99, 0, 7532, 0 },
    { 114, 0, 7533, 0 },
    { 59, 0, 0, 119984 }, /* &Uscr; */
    { 103, 0, 7535, 0 },
    { 116, 0, 7536, 0 },
    { 114, 0, 7537, 0 },
    { 59, 0, 0, 10891 }, /* &lesseqqgtr; */
    { 108, 0, 7539, 0 },
    { 105, 0, 7540, 0 },
    { 103, 0, 7541, 0 },
    { 59, 0, 0, 64259 }, /* &ffilig; */
    { 99, 0, 7543, 0 },
    { 114, 0, 7544, 0 },
    { 59, 0, 0, 333 }, /* &omacr; */
    { 59, 0, 0, 962 }, /* &sigmav; */
    { 101, 0, 7547, 0 },
    { 99, 0, 7548, 0 },
    { 116, 0, 7549, 0 },
    { 111, 0, 7550, 0 },
    { 114, 0, 7551, 0 },
    { 59, 1, 0, 8638 }, /* &RightUpVector; */
    { 66, 0, 8724, 0 },
    { 105, 0, 7554, 0 },
    { 103, 0, 7555, 0 },
    { 104, 0, 7556, 0 },
    { 116, 0, 7557, 0 },
    { 65, 0, 7558, 0 },
    { 114, 0, 7559, 0 },
    { 114, 0, 7560, 0 },
    { 111, 0, 7561, 0 },
    { 119, 0, 7562, 0 },
    { 59, 0, 0, 8599 }, /* &UpperRightArrow; */
    { 112, 0, 7564, 0 },
    { 114, 0, 7565, 0 },
    { 111, 0, 7566, 0 },
    { 120, 0, 7567, 0 },
    { 59, 0, 0, 10886 }, /* &gtrapprox; */
    { 103, 0, 7569, 0 },
    { 59, 0, 0, 730 }, /* &ring; */
    { 101, 1, 7572, 0 },
    { 59, 0, 0, 10959 }, /* &csub; */
    { 59, 0, 0, 10961 }, /* &csube; */
    { 59, 0, 0, 254 }, /* &thorn; */
    { 112, 1, 7576, 0 },
    { 114, 0, 7660, 0 },
    { 115, 0, 7577, 0 },
    { 59, 0, 0, 10829 }, /* &ccaps; */
    { 116, 0, 7579, 0 },
    { 114, 0, 7580, 0 },
    { 59, 0, 0, 8822 }, /* &lessgtr; */
    { 108, 0, 7582, 0 },
    { 111, 0, 7583, 0 },
    { 110, 0, 7584, 0 },
    { 59, 0, 0, 8789 }, /* &eqcolon; */
    { 97, 1, 7588, 0 },
    { 101, 1, 7738, 0 },
    { 108, 0, 9674, 0 },
    { 114, 0, 7589, 0 },
    { 59, 0, 0, 10679 }, /* &opar; */
    { 99, 0, 7591, 0 },
    { 114, 0, 7592, 0 },
    { 59, 0, 0, 119982 }, /* &Sscr; */
    { 59, 0, 0, 10893 }, /* &lsime; */
    { 114, 0, 7595, 0 },
    { 114, 0, 7596, 0 },
    { 59, 0, 0, 10508 }, /* &lbarr; */
    { 59, 0, 0, 9574 }, /* &boxHD; */
    { 114, 0, 7599, 0 },
    { 59, 0, 0, 8459 }, /* &Hscr; */
    { 100, 1, 7603, 0 },
    { 59, 1, 0, 215 }, /* &times; */
    { 98, 0, 8676, 0 },
    { 59, 0, 0, 10800 }, /* &timesd; */
    { 119, 1, 7606, 0 },
    { 118, 0, 7800, 0 },
    { 101, 0, 7607, 0 },
    { 100, 0, 7608, 0 },
    { 59, 0, 0, 8966 }, /* &Barwed; */
    { 59, 0, 0, 8810 }, /* &Lt; */
    { 59, 0, 0, 1090 }, /* &tcy; */
    { 112, 0, 7612, 0 },
    { 59, 0, 0, 10827 }, /* &capcap; */
    { 109, 0, 7614, 0 },
    { 59, 0, 0, 10951 }, /* &subsim; */
    { 114, 0, 7616, 0 },
    { 114, 0, 7617, 0 },
    { 59, 0, 0, 8663 }, /* &neArr; */
    { 67, 1, 7620, 0 },
    { 99, 0, 7684, 0 },
    { 72, 0, 7621, 0 },
    { 99, 0, 7622, 0 },
    { 121, 0, 7623, 0 },
    { 59, 0, 0, 1065 }, /* &SHCHcy; */
    { 59, 0, 0, 1081 }, /* &jcy; */
    { 59, 0, 0, 958 }, /* &xi; */
    { 108, 0, 7627, 0 },
    { 105, 0, 7628, 0 },
    { 103, 0, 7629, 0 },
    { 59, 0, 0, 338 }, /* &OElig; */
    { 114, 0, 7631, 0 },
    { 116, 0, 7632, 0 },
    { 105, 0, 7633, 0 },
    { 97, 0, 7634, 0 },
    { 108, 0, 7635, 0 },
    { 68, 0, 7636, 0 },
    { 59, 0, 0, 8706 }, /* &PartialD; */
    { 97, 0, 7638, 0 },
    { 114, 0, 7639, 0 },
    { 114, 0, 7640, 0 },
    { 59, 0, 0, 10610 }, /* &simrarr; */
    { 112, 0, 7642, 0 },
    { 59, 1, 0, 8915 }, /* &Cup; */
    { 67, 0, 9783, 0 },
    { 114, 1, 7646, 0 },
    { 112, 0, 9829, 0 },
    { 111, 0, 7647, 0 },
    { 110, 0, 7648, 0 },
    { 59, 0, 0, 353 }, /* &scaron; */
    { 111, 0, 7650, 0 },
    { 110, 0, 7651, 0 },
    { 100, 0, 7652, 0 },
    { 59, 0, 0, 8900 }, /* &Diamond; */
    { 108, 0, 7654, 0 },
    { 100, 0, 7655, 0 },
    { 101, 0, 7656, 241 }, /* &ntilde */
    { 59, 0, 0, 241 }, /* &ntilde; */
    { 111, 0, 7658, 0 },
    { 116, 0, 7659, 0 },
    { 59, 0, 0, 8411 }, /* &tdot; */
    { 111, 0, 7661, 0 },
    { 110, 0, 7662, 0 },
    { 59, 0, 0, 269 }, /* &ccaron; */
    { 115, 0, 7664, 0 },
    { 117, 0, 7665, 0 },
    { 98, 1, 7667, 0 },
    { 112, 0, 7695, 0 },
    { 101, 0, 7668, 0 },
    { 59, 0, 0, 8930 }, /* &nsqsube; */
    { 115, 0, 7670, 0 },
    { 104, 0, 7671, 0 },
    { 116, 0, 7672, 0 },
    { 59, 0, 0, 10622 }, /* &ufisht; */
    { 103, 0, 7674, 0 },
    { 108, 0, 7675, 0 },
    { 101, 0, 7676, 0 },
    { 66, 0, 7677, 0 },
    { 114, 0, 7678, 0 },
    { 97, 0, 7679, 0 },
    { 99, 0, 7680, 0 },
    { 107, 0, 7681, 0 },
    { 101, 0, 7682, 0 },
    { 116, 0, 7683, 0 },
    { 59, 0, 0, 10216 }, /* &LeftAngleBracket; */
    { 121, 0, 7685, 0 },
    { 59, 0, 0, 1064 }, /* &SHcy; */
    { 113, 0, 7687, 0 },
    { 59, 0, 0, 10927 }, /* &preceq; */
    { 103, 0, 7689, 0 },
    { 59, 0, 0, 64256 }, /* &fflig; */
    { 117, 1, 7692, 0 },
    { 98, 0, 8087, 0 },
    { 111, 0, 7693, 0 },
    { 59, 1, 0, 8217 }, /* &rsquo; */
    { 114, 0, 9472, 0 },
    { 101, 0, 7696, 0 },
    { 59, 0, 0, 8931 }, /* &nsqsupe; */
    { 115, 0, 7698, 0 },
    { 116, 0, 7699, 0 },
    { 59, 0, 0, 8727 }, /* &lowast; */
    { 59, 0, 0, 9733 }, /* &starf; */
    { 111, 0, 7702, 0 },
    { 112, 0, 7703, 0 },
    { 116, 0, 7704, 0 },
    { 111, 0, 7705, 0 },
    { 59, 0, 0, 8733 }, /* &varpropto; */
    { 114, 0, 7707, 0 },
    { 59, 0, 0, 10873 }, /* &ltcir; */
    { 108, 0, 7709, 0 },
    { 117, 0, 7710, 0 },
    { 115, 0, 7711, 0 },
    { 59, 0, 0, 8862 }, /* &boxplus; */
    { 113, 0, 7713, 0 },
    { 117, 0, 7714, 0 },
    { 97, 0, 7715, 0 },
    { 108, 0, 7716, 0 },
    { 59, 0, 0, 8783 }, /* &HumpEqual; */
    { 105, 0, 7718, 0 },
    { 103, 0, 7719, 0 },
    { 104, 0, 7720, 0 },
    { 116, 0, 7721, 0 },
    { 59, 0, 0, 8631 }, /* &curvearrowright; */
    { 114, 0, 7723, 0 },
    { 59, 0, 0, 8989 }, /* &urcorner; */
    { 97, 0, 7725, 0 },
    { 112, 0, 7726, 0 },
    { 59, 0, 0, 8888 }, /* &mumap; */
    { 99, 0, 7728, 0 },
    { 121, 0, 7729, 0 },
    { 59, 0, 0, 1029 }, /* &DScy; */
    { 115, 1, 7733, 0 },
    { 97, 1, 9037, 0 },
    { 101, 0, 9730, 0 },
    { 105, 0, 7734, 0 },
    { 109, 0, 7735, 0 },
    { 59, 0, 0, 8936 }, /* &precnsim; */
    { 97, 0, 7737, 0 },
    { 59, 0, 0, 914 }, /* &Beta; */
    { 114, 0, 7739, 0 },
    { 112, 0, 7740, 0 },
    { 59, 0, 0, 10681 }, /* &operp; */
    { 101, 0, 7742, 0 },
    { 114, 0, 7743, 0 },
    { 115, 0, 7744, 0 },
    { 101, 0, 7745, 0 },
    { 69, 1, 7747, 0 },
    { 85, 0, 8547, 0 },
    { 108, 1, 7749, 0 },
    { 113, 0, 8040, 0 },
    { 101, 0, 7750, 0 },
    { 109, 0, 7751, 0 },
    { 101, 0, 7752, 0 },
    { 110, 0, 7753, 0 },
    { 116, 0, 7754, 0 },
    { 59, 0, 0, 8715 }, /* &ReverseElement; */
    { 97, 0, 7756, 0 },
    { 114, 0, 7757, 0 },
    { 112, 0, 7758, 0 },
    { 111, 0, 7759, 0 },
    { 111, 0, 7760, 0 },
    { 110, 0, 7761, 0 },
    { 114, 1, 7763, 0 },
    { 108, 0, 8175, 0 },
    { 105, 0, 7764, 0 },
    { 103, 0, 7765, 0 },
    { 104, 0, 7766, 0 },
    { 116, 0, 7767, 0 },
    { 59, 0, 0, 8642 }, /* &downharpoonright; */
    { 59, 0, 0, 10896 }, /* &gsiml; */
    { 59, 0, 0, 1056 }, /* &Rcy; */
    { 103, 0, 7771, 0 },
    { 104, 0, 7772, 0 },
    { 116, 0, 7773, 0 },
    { 97, 0, 7774, 0 },
    { 114, 0, 7775, 0 },
    { 114, 0, 7776, 0 },
    { 111, 0, 7777, 0 },
    { 119, 0, 7778, 0 },
    { 59, 0, 0, 8603 }, /* &nrightarrow; */
    { 114, 0, 7780, 0 },
    { 101, 0, 7781, 0 },
    { 97, 0, 7782, 0 },
    { 116, 0, 7783, 0 },
    { 101, 0, 7784, 0 },
    { 114, 0, 7785, 0 },
    { 71, 0, 7786, 0 },
    { 114, 0, 7787, 0 },
    { 101, 0, 7788, 0 },
    { 97, 0, 7789, 0 },
    { 116, 0, 7790, 0 },
    { 101, 0, 7791, 0 },
    { 114, 0, 7792, 0 },
    { 59, 0, 0, 10914 }, /* &NotNestedGreaterGreater; */
    { 97, 0, 7794, 0 },
    { 114, 0, 7795, 0 },
    { 59, 0, 0, 8676 }, /* &LeftArrowBar; */
    { 108, 0, 7797, 0 },
    { 117, 0, 7798, 0 },
    { 115, 0, 7799, 0 },
    { 59, 0, 0, 8846 }, /* &UnionPlus; */
    { 59, 0, 0, 10983 }, /* &Barv; */
    { 117, 0, 7802, 0 },
    { 116, 0, 7803, 0 },
    { 101, 0, 7804, 0 },
    { 59, 0, 0, 314 }, /* &lacute; */
    { 59, 0, 0, 10605 }, /* &lrhard; */
    { 105, 1, 7808, 0 },
    { 108, 0, 9012, 0 },
    { 109, 0, 7809, 0 },
    { 59, 0, 0, 10995 }, /* &parsim; */
    { 59, 0, 0, 10928 }, /* &nsce; */
    { 113, 0, 7812, 0 },
    { 117, 0, 7813, 0 },
    { 97, 0, 7814, 0 },
    { 114, 0, 7815, 0 },
    { 101, 0, 7816, 0 },
    { 59, 0, 0, 8865 }, /* &dotsquare; */
    { 111, 0, 7818, 0 },
    { 116, 0, 7819, 0 },
    { 59, 0, 0, 9014 }, /* &topbot; */
    { 101, 0, 7821, 0 },
    { 100, 0, 7822, 0 },
    { 103, 0, 7823, 0 },
    { 101, 0, 7824, 0 },
    { 59, 0, 0, 8896 }, /* &bigwedge; */
    { 112, 0, 7826, 0 },
    { 111, 0, 7827, 0 },
    { 115, 0, 7828, 0 },
    { 59, 0, 0, 8766 }, /* &mstpos; */
    { 100, 0, 7830, 0 },
    { 97, 0, 7831, 0 },
    { 115, 0, 7832, 0 },
    { 104, 0, 7833, 0 },
    { 59, 0, 0, 8874 }, /* &Vvdash; */
    { 115, 0, 7835, 0 },
    { 116, 0, 7836, 0 },
    { 59, 0, 0, 42 }, /* &midast; */
    { 104, 0, 7838, 0 },
    { 59, 0, 0, 8627 }, /* &rdsh; */
    { 111, 0, 7840, 0 },
    { 116, 0, 7841, 0 },
    { 59, 0, 0, 10947 }, /* &subedot; */
    { 114, 0, 7843, 0 },
    { 114, 0, 7844, 0 },
    { 111, 0, 7845, 0 },
    { 119, 0, 7846, 0 },
    { 59, 0, 0, 8596 }, /* &LeftRightArrow; */
    { 108, 0, 7848, 0 },
    { 59, 1, 0, 9838 }, /* &natural; */
    { 115, 0, 7936, 0 },
    { 105, 0, 7851, 0 },
    { 109, 0, 7852, 0 },
    { 59, 0, 0, 8818 }, /* &lesssim; */
    { 97, 0, 7854, 0 },
    { 59, 0, 0, 10537 }, /* &tosa; */
    { 112, 1, 7857, 0 },
    { 97, 0, 9746, 0 },
    { 116, 0, 7858, 0 },
    { 121, 0, 7859, 0 },
    { 83, 1, 7861, 0 },
    { 86, 0, 8911, 0 },
    { 109, 0, 7862, 0 },
    { 97, 0, 7863, 0 },
    { 108, 0, 7864, 0 },
    { 108, 0, 7865, 0 },
    { 83, 0, 7866, 0 },
    { 113, 0, 7867, 0 },
    { 117, 0, 7868, 0 },
    { 97, 0, 7869, 0 },
    { 114, 0, 7870, 0 },
    { 101, 0, 7871, 0 },
    { 59, 0, 0, 9723 }, /* &EmptySmallSquare; */
    { 114, 0, 7873, 0 },
    { 105, 0, 7874, 0 },
    { 109, 0, 7875, 0 },
    { 101, 0, 7876, 0 },
    { 59, 0, 0, 8279 }, /* &qprime; */
    { 111, 0, 7878, 0 },
    { 116, 0, 7879, 0 },
    { 59, 0, 0, 8919 }, /* &gtdot; */
    { 59, 0, 0, 8806 }, /* &leqq; */
    { 114, 0, 7882, 0 },
    { 97, 0, 7883, 0 },
    { 110, 0, 7884, 0 },
    { 59, 0, 0, 8466 }, /* &lagran; */
    { 101, 0, 7886, 0 },
    { 101, 0, 7887, 0 },
    { 86, 0, 7888, 0 },
    { 101, 0, 7889, 0 },
    { 99, 0, 7890, 0 },
    { 116, 0, 7891, 0 },
    { 111, 0, 7892, 0 },
    { 114, 0, 7893, 0 },
    { 59, 0, 0, 10591 }, /* &DownRightTeeVector; */
    { 99, 1, 7896, 0 },
    { 104, 0, 8996, 0 },
    { 114, 0, 7897, 0 },
    { 59, 0, 0, 8475 }, /* &Rscr; */
    { 59, 0, 0, 240 }, /* &eth; */
    { 113, 0, 7900, 0 },
    { 59, 0, 0, 8794 }, /* &veeeq; */
    { 114, 0, 7902, 0 },
    { 59, 0, 0, 120104 }, /* &kfr; */
    { 118, 0, 7904, 0 },
    { 101, 0, 7905, 0 },
    { 114, 0, 7906, 0 },
    { 115, 0, 7907, 0 },
    { 101, 0, 7908, 0 },
    { 69, 0, 7909, 0 },
    { 108, 0, 7910, 0 },
    { 101, 0, 7911, 0 },
    { 109, 0, 7912, 0 },
    { 101, 0, 7913, 0 },
    { 110, 0, 7914, 0 },
    { 116, 0, 7915, 0 },
    { 59, 0, 0, 8716 }, /* &NotReverseElement; */
    { 59, 0, 0, 8502 }, /* &beth; */
    { 115, 0, 7918, 0 },
    { 59, 0, 0, 10526 }, /* &rarrfs; */
    { 116, 0, 7920, 0 },
    { 97, 0, 7921, 0 },
    { 108, 0, 7922, 0 },
    { 68, 0, 7923, 0 },
    { 105, 0, 7924, 0 },
    { 102, 0, 7925, 0 },
    { 102, 0, 7926, 0 },
    { 101, 0, 7927, 0 },
    { 114, 0, 7928, 0 },
    { 101, 0, 7929, 0 },
    { 110, 0, 7930, 0 },
    { 116, 0, 7931, 0 },
    { 105, 0, 7932, 0 },
    { 97, 0, 7933, 0 },
    { 108, 0, 7934, 0 },
    { 68, 0, 7935, 0 },
    { 59, 0, 0, 8517 }, /* &CapitalDifferentialD; */
    { 59, 0, 0, 8469 }, /* &naturals; */
    { 105, 0, 7938, 0 },
    { 114, 0, 7939, 0 },
    { 99, 0, 7940, 0 },
    { 59, 0, 0, 293 }, /* &hcirc; */
    { 59, 0, 0, 1076 }, /* &dcy; */
    { 104, 0, 7943, 0 },
    { 97, 0, 7944, 0 },
    { 114, 0, 7945, 0 },
    { 59, 0, 0, 10599 }, /* &ldrdhar; */
    { 112, 0, 7947, 0 },
    { 115, 1, 7949, 0 },
    { 59, 0, 0, 8851 }, /* &sqcap; */
    { 59, 0, 0, 8851 }, /* &sqcaps; */
    { 115, 0, 7951, 0 },
    { 116, 0, 7952, 0 },
    { 59, 0, 0, 8859 }, /* &circledast; */
    { 59, 0, 0, 1109 }, /* &dscy; */
    { 112, 0, 7955, 0 },
    { 112, 0, 7956, 0 },
    { 114, 0, 7957, 0 },
    { 111, 0, 7958, 0 },
    { 120, 0, 7959, 0 },
    { 59, 0, 0, 10885 }, /* &lessapprox; */
    { 117, 0, 7961, 0 },
    { 101, 0, 7962, 0 },
    { 59, 0, 0, 8928 }, /* &nprcue; */
    { 114, 0, 7964, 0 },
    { 114, 0, 7965, 0 },
    { 111, 0, 7966, 0 },
    { 119, 0, 7967, 0 },
    { 59, 0, 0, 8612 }, /* &LeftTeeArrow; */
    { 111, 0, 7969, 0 },
    { 116, 0, 7970, 0 },
    { 59, 0, 0, 278 }, /* &Edot; */
    { 114, 0, 7972, 0 },
    { 59, 0, 0, 120095 }, /* &bfr; */
    { 59, 0, 0, 10868 }, /* &Colone; */
    { 111, 0, 7975, 0 },
    { 116, 0, 7976, 0 },
    { 59, 0, 0, 8944 }, /* &utdot; */
    { 117, 0, 7978, 0 },
    { 116, 0, 7979, 0 },
    { 101, 0, 7980, 0 },
    { 59, 0, 0, 324 }, /* &nacute; */
    { 99, 0, 7982, 0 },
    { 117, 0, 7983, 0 },
    { 116, 0, 7984, 0 },
    { 101, 0, 7985, 0 },
    { 59, 0, 0, 180 }, /* &DiacriticalAcute; */
    { 97, 0, 7987, 0 },
    { 99, 0, 7988, 0 },
    { 59, 0, 0, 733 }, /* &dblac; */
    { 121, 1, 7991, 0 },
    { 105, 0, 8245, 0 },
    { 59, 0, 0, 1099 }, /* &ycy; */
    { 59, 0, 0, 193 }, /* &Aacute; */
    { 108, 0, 7994, 0 },
    { 59, 0, 0, 10518 }, /* &Rarrtl; */
    { 117, 0, 7996, 0 },
    { 101, 0, 7997, 0 },
    { 59, 0, 0, 8828 }, /* &prcue; */
    { 111, 0, 7999, 0 },
    { 100, 0, 8000, 0 },
    { 59, 0, 0, 8720 }, /* &coprod; */
    { 114, 0, 8002, 0 },
    { 107, 0, 8003, 0 },
    { 59, 0, 0, 10099 }, /* &rbbrk; */
    { 101, 0, 8005, 0 },
    { 116, 0, 8006, 0 },
    { 104, 0, 8007, 0 },
    { 59, 0, 0, 8504 }, /* &daleth; */
    { 59, 0, 0, 10549 }, /* &cudarrr; */
    { 59, 0, 0, 9532 }, /* &boxvh; */
    { 111, 0, 8011, 0 },
    { 110, 0, 8012, 0 },
    { 59, 0, 0, 949 }, /* &epsilon; */
    { 59, 0, 0, 10866 }, /* &pluse; */
    { 59, 0, 0, 10635 }, /* &lbrke; */
    { 101, 0, 8016, 0 },
    { 100, 0, 8017, 0 },
    { 103, 0, 8018, 0 },
    { 101, 0, 8019, 0 },
    { 59, 0, 0, 8896 }, /* &xwedge; */
    { 97, 0, 8021, 0 },
    { 114, 0, 8022, 0 },
    { 114, 0, 8023, 0 },
    { 59, 0, 0, 8594 }, /* &srarr; */
    { 97, 0, 8025, 0 },
    { 114, 0, 8026, 0 },
    { 59, 0, 0, 10582 }, /* &DownLeftVectorBar; */
    { 59, 0, 0, 8462 }, /* &planckh; */
    { 59, 0, 0, 10956 }, /* &varsupsetneqq; */
    { 117, 0, 8030, 0 },
    { 108, 0, 8031, 0 },
    { 116, 0, 8032, 0 },
    { 59, 0, 0, 10946 }, /* &supmult; */
    { 59, 0, 0, 8806 }, /* &nlE; */
    { 98, 0, 8035, 0 },
    { 97, 0, 8036, 0 },
    { 114, 0, 8037, 0 },
    { 59, 0, 0, 8213 }, /* &horbar; */
    { 59, 0, 0, 8641 }, /* &rhard; */
    { 59, 0, 0, 219 }, /* &Ucirc; */
    { 117, 0, 8041, 0 },
    { 105, 0, 8042, 0 },
    { 108, 0, 8043, 0 },
    { 105, 0, 8044, 0 },
    { 98, 0, 8045, 0 },
    { 114, 0, 8046, 0 },
    { 105, 0, 8047, 0 },
    { 117, 0, 8048, 0 },
    { 109, 0, 8049, 0 },
    { 59, 0, 0, 8651 }, /* &ReverseEquilibrium; */
    { 59, 0, 0, 10547 }, /* &rarrc; */
    { 105, 0, 8052, 0 },
    { 97, 0, 8053, 0 },
    { 110, 0, 8054, 0 },
    { 103, 0, 8055, 0 },
    { 108, 0, 8056, 0 },
    { 101, 0, 8057, 0 },
    { 59, 1, 0, 8883 }, /* &RightTriangle; */
    { 66, 1, 8269, 0 },
    { 69, 0, 9416, 0 },
    { 101, 0, 8061, 0 },
    { 102, 0, 8062, 0 },
    { 116, 0, 8063, 0 },
    { 59, 0, 0, 8612 }, /* &mapstoleft; */
    { 114, 0, 8065, 0 },
    { 59, 0, 0, 10838 }, /* &oror; */
    { 59, 0, 0, 124 }, /* &vert; */
    { 59, 0, 0, 8808 }, /* &lnE; */
    { 114, 0, 8069, 0 },
    { 111, 0, 8070, 0 },
    { 120, 0, 8071, 0 },
    { 59, 0, 0, 8777 }, /* &napprox; */
    { 109, 0, 8073, 0 },
    { 108, 0, 8074, 220 }, /* &Uuml */
    { 59, 0, 0, 220 }, /* &Uuml; */
    { 59, 0, 0, 1083 }, /* &lcy; */
    { 109, 0, 8077, 0 },
    { 59, 0, 0, 10952 }, /* &supsim; */
    { 59, 0, 0, 9488 }, /* &boxdl; */
    { 114, 0, 8080, 0 },
    { 59, 0, 0, 8476 }, /* &Rfr; */
    { 59, 0, 0, 8709 }, /* &emptyv; */
    { 114, 0, 8083, 0 },
    { 59, 0, 0, 120119 }, /* &zfr; */
    { 53, 1, 8086, 0 },
    { 51, 0, 9225, 0 },
    { 59, 0, 0, 8534 }, /* &frac25; */
    { 59, 0, 0, 93 }, /* &rsqb; */
    { 59, 0, 0, 8953 }, /* &isinE; */
    { 97, 1, 8091, 0 },
    { 121, 0, 8682, 0 },
    { 114, 0, 8092, 0 },
    { 111, 0, 8093, 0 },
    { 110, 0, 8094, 0 },
    { 59, 0, 0, 270 }, /* &Dcaron; */
    { 114, 0, 8096, 0 },
    { 115, 1, 8098, 0 },
    { 59, 0, 0, 8917 }, /* &epar; */
    { 108, 0, 8099, 0 },
    { 59, 0, 0, 10723 }, /* &eparsl; */
    { 112, 0, 8101, 0 },
    { 102, 0, 8102, 0 },
    { 59, 0, 0, 120167 }, /* &vopf; */
    { 59, 0, 0, 1008 }, /* &kappav; */
    { 114, 0, 8105, 0 },
    { 107, 0, 8106, 0 },
    { 59, 0, 0, 10214 }, /* &lobrk; */
    { 115, 0, 8108, 0 },
    { 59, 0, 0, 10525 }, /* &larrfs; */
    { 121, 0, 8110, 0 },
    { 59, 0, 0, 1092 }, /* &fcy; */
    { 114, 0, 8112, 0 },
    { 59, 0, 0, 9023 }, /* &solbar; */
    { 114, 0, 8114, 0 },
    { 111, 0, 8115, 0 },
    { 110, 0, 8116, 0 },
    { 59, 0, 0, 327 }, /* &Ncaron; */
    { 99, 0, 8118, 0 },
    { 59, 0, 0, 10921 }, /* &gescc; */
    { 59, 0, 0, 8772 }, /* &nsimeq; */
    { 59, 0, 0, 8713 }, /* &notinva; */
    { 59, 0, 0, 9569 }, /* &boxvL; */
    { 97, 0, 8123, 0 },
    { 114, 0, 8124, 0 },
    { 59, 0, 0, 10677 }, /* &ohbar; */
    { 121, 1, 8127, 0 },
    { 105, 0, 8185, 0 },
    { 59, 0, 0, 1075 }, /* &gcy; */
    { 101, 0, 8129, 0 },
    { 99, 0, 8130, 0 },
    { 116, 0, 8131, 0 },
    { 111, 0, 8132, 0 },
    { 114, 0, 8133, 0 },
    { 59, 0, 0, 10587 }, /* &RightTeeVector; */
    { 114, 0, 8135, 0 },
    { 59, 0, 0, 120082 }, /* &Ofr; */
    { 105, 0, 8137, 0 },
    { 109, 0, 8138, 0 },
    { 59, 0, 0, 8830 }, /* &prsim; */
    { 113, 0, 8140, 0 },
    { 117, 0, 8141, 0 },
    { 97, 0, 8142, 0 },
    { 108, 0, 8143, 0 },
    { 59, 0, 0, 8840 }, /* &NotSubsetEqual; */
    { 111, 0, 8145, 0 },
    { 114, 0, 8146, 0 },
    { 110, 0, 8147, 0 },
    { 101, 0, 8148, 0 },
    { 114, 0, 8149, 0 },
    { 59, 0, 0, 8991 }, /* &lrcorner; */
    { 112, 0, 8151, 0 },
    { 102, 0, 8152, 0 },
    { 59, 0, 0, 120130 }, /* &Kopf; */
    { 59, 0, 0, 8804 }, /* &nvle; */
    { 114, 0, 8155, 0 },
    { 105, 0, 8156, 0 },
    { 59, 0, 0, 9720 }, /* &ultri; */
    { 59, 0, 0, 34 }, /* &quot; */
    { 111, 0, 8159, 0 },
    { 116, 0, 8160, 0 },
    { 59, 0, 0, 289 }, /* &gdot; */
    { 110, 0, 8162, 0 },
    { 100, 0, 8163, 0 },
    { 73, 0, 8164, 0 },
    { 109, 0, 8165, 0 },
    { 112, 0, 8166, 0 },
    { 108, 0, 8167, 0 },
    { 105, 0, 8168, 0 },
    { 101, 0, 8169, 0 },
    { 115, 0, 8170, 0 },
    { 59, 0, 0, 10608 }, /* &RoundImplies; */
    { 59, 0, 0, 1058 }, /* &Tcy; */
    { 112, 0, 8173, 0 },
    { 102, 0, 8174, 0 },
    { 59, 0, 0, 120143 }, /* &Xopf; */
    { 101, 0, 8176, 0 },
    { 102, 0, 8177, 0 },
    { 116, 0, 8178, 0 },
    { 59, 0, 0, 8643 }, /* &downharpoonleft; */
    { 105, 0, 8180, 0 },
    { 114, 0, 8181, 0 },
    { 59, 0, 0, 10993 }, /* &topcir; */
    { 104, 0, 8183, 0 },
    { 97, 0, 8184, 0 },
    { 59, 0, 0, 945 }, /* &alpha; */
    { 114, 0, 8186, 0 },
    { 99, 0, 8187, 0 },
    { 59, 0, 0, 285 }, /* &gcirc; */
    { 99, 1, 8190, 0 },
    { 116, 0, 9622, 0 },
    { 114, 0, 8191, 0 },
    { 59, 0, 0, 119967 }, /* &Dscr; */
    { 121, 0, 8193, 0 },
    { 59, 0, 0, 1102 }, /* &yucy; */
    { 101, 1, 8196, 0 },
    { 121, 0, 9590, 0 },
    { 100, 0, 8197, 0 },
    { 105, 0, 8198, 0 },
    { 108, 0, 8199, 0 },
    { 59, 0, 0, 311 }, /* &kcedil; */
    { 105, 0, 8201, 0 },
    { 114, 0, 8202, 0 },
    { 59, 0, 0, 10992 }, /* &midcir; */
    { 59, 0, 0, 9642 }, /* &squf; */
    { 59, 0, 0, 8920 }, /* &nLl; */
    { 59, 0, 0, 10925 }, /* &lates; */
    { 110, 0, 8207, 0 },
    { 103, 0, 8208, 0 },
    { 100, 1, 8210, 0 },
    { 59, 0, 0, 8775 }, /* &ncong; */
    { 111, 0, 8211, 0 },
    { 116, 0, 8212, 0 },
    { 59, 0, 0, 10861 }, /* &ncongdot; */
    { 59, 0, 0, 8539 }, /* &frac18; */
    { 100, 0, 8215, 0 },
    { 105, 0, 8216, 0 },
    { 108, 0, 8217, 0 },
    { 59, 0, 0, 310 }, /* &Kcedil; */
    { 59, 0, 0, 1085 }, /* &ncy; */
    { 114, 0, 8220, 0 },
    { 59, 0, 0, 8495 }, /* &escr; */
    { 111, 0, 8222, 0 },
    { 116, 0, 8223, 0 },
    { 59, 0, 0, 379 }, /* &Zdot; */
    { 105, 0, 8225, 0 },
    { 108, 0, 8226, 0 },
    { 100, 0, 8227, 0 },
    { 101, 0, 8228, 0 },
    { 59, 0, 0, 8770 }, /* &EqualTilde; */
    { 121, 0, 8230, 0 },
    { 59, 0, 0, 1042 }, /* &Vcy; */
    { 59, 0, 0, 34 }, /* &QUOT; */
    { 114, 0, 8233, 0 },
    { 59, 0, 0, 120074 }, /* &Gfr; */
    { 111, 0, 8235, 0 },
    { 112, 0, 8236, 0 },
    { 59, 0, 0, 8972 }, /* &drcrop; */
    { 114, 0, 8238, 0 },
    { 114, 0, 8239, 0 },
    { 111, 0, 8240, 0 },
    { 119, 0, 8241, 0 },
    { 59, 0, 0, 8615 }, /* &DownTeeArrow; */
    { 112, 0, 8243, 0 },
    { 102, 0, 8244, 0 },
    { 59, 0, 0, 120144 }, /* &Yopf; */
    { 114, 0, 8246, 0 },
    { 99, 0, 8247, 0 },
    { 59, 0, 0, 375 }, /* &ycirc; */
    { 114, 0, 8249, 0 },
    { 117, 0, 8250, 0 },
    { 101, 0, 8251, 0 },
    { 110, 0, 8252, 0 },
    { 116, 0, 8253, 0 },
    { 59, 0, 0, 8801 }, /* &Congruent; */
    { 110, 0, 8255, 0 },
    { 59, 0, 0, 711 }, /* &caron; */
    { 59, 0, 0, 222 }, /* &THORN; */
    { 116, 0, 8258, 0 },
    { 59, 0, 0, 8785 }, /* &eDot; */
    { 117, 0, 8260, 0 },
    { 116, 0, 8261, 0 },
    { 101, 0, 8262, 0 },
    { 59, 0, 0, 340 }, /* &Racute; */
    { 114, 0, 8264, 0 },
    { 59, 0, 0, 120078 }, /* &Kfr; */
    { 59, 0, 0, 10842 }, /* &andv; */
    { 114, 0, 8267, 0 },
    { 99, 0, 8268, 0 },
    { 59, 0, 0, 374 }, /* &Ycirc; */
    { 97, 0, 8270, 0 },
    { 114, 0, 8271, 0 },
    { 59, 0, 0, 10704 }, /* &RightTriangleBar; */
    { 59, 0, 0, 8814 }, /* &nless; */
    { 111, 0, 8274, 0 },
    { 114, 0, 8275, 0 },
    { 116, 0, 8276, 0 },
    { 112, 1, 8278, 0 },
    { 109, 0, 9096, 0 },
    { 97, 0, 8279, 0 },
    { 114, 0, 8280, 0 },
    { 97, 0, 8281, 0 },
    { 108, 0, 8282, 0 },
    { 108, 0, 8283, 0 },
    { 101, 0, 8284, 0 },
    { 108, 0, 8285, 0 },
    { 59, 0, 0, 8742 }, /* &nshortparallel; */
    { 101, 0, 8287, 0 },
    { 102, 0, 8288, 0 },
    { 116, 0, 8289, 0 },
    { 59, 0, 0, 8634 }, /* &circlearrowleft; */
    { 59, 0, 0, 8788 }, /* &coloneq; */
    { 108, 0, 8292, 0 },
    { 97, 0, 8293, 0 },
    { 99, 0, 8294, 0 },
    { 59, 0, 0, 369 }, /* &udblac; */
    { 105, 0, 8296, 0 },
    { 59, 0, 0, 59 }, /* &semi; */
    { 111, 0, 8298, 0 },
    { 114, 0, 8299, 0 },
    { 110, 0, 8300, 0 },
    { 101, 0, 8301, 0 },
    { 114, 0, 8302, 0 },
    { 59, 0, 0, 8990 }, /* &llcorner; */
    { 112, 0, 8304, 0 },
    { 108, 0, 8305, 0 },
    { 117, 0, 8306, 0 },
    { 115, 0, 8307, 0 },
    { 59, 0, 0, 10756 }, /* &biguplus; */
    { 59, 0, 0, 8196 }, /* &emsp13; */
    { 111, 1, 8311, 0 },
    { 97, 0, 9379, 0 },
    { 115, 0, 8312, 0 },
    { 115, 0, 8313, 0 },
    { 59, 0, 0, 10007 }, /* &cross; */
    { 59, 0, 0, 8909 }, /* &bsime; */
    { 97, 0, 8316, 0 },
    { 114, 0, 8317, 0 },
    { 59, 0, 0, 10515 }, /* &DownArrowBar; */
    { 59, 0, 0, 8808 }, /* &lneqq; */
    { 59, 0, 0, 10955 }, /* &subnE; */
    { 112, 0, 8321, 0 },
    { 115, 0, 8322, 0 },
    { 105, 0, 8323, 0 },
    { 108, 0, 8324, 0 },
    { 111, 0, 8325, 0 },
    { 110, 0, 8326, 0 },
    { 59, 0, 0, 1014 }, /* &backepsilon; */
    { 99, 0, 8328, 0 },
    { 114, 0, 8329, 0 },
    { 59, 0, 0, 120014 }, /* &yscr; */
    { 111, 0, 8331, 0 },
    { 59, 0, 0, 929 }, /* &Rho; */
    { 114, 0, 8333, 0 },
    { 112, 0, 8334, 0 },
    { 59, 0, 0, 9839 }, /* &sharp; */
    { 111, 0, 8336, 0 },
    { 116, 0, 8337, 0 },
    { 59, 0, 0, 8949 }, /* &notindot; */
    { 111, 0, 8339, 0 },
    { 116, 1, 8341, 0 },
    { 117, 0, 9228, 0 },
    { 59, 0, 0, 729 }, /* &DiacriticalDot; */
    { 101, 0, 8343, 0 },
    { 116, 0, 8344, 0 },
    { 59, 1, 0, 8847 }, /* &sqsubset; */
    { 101, 0, 9262, 0 },
    { 111, 0, 8347, 0 },
    { 119, 0, 8348, 0 },
    { 110, 0, 8349, 0 },
    { 86, 0, 8350, 0 },
    { 101, 0, 8351, 0 },
    { 99, 0, 8352, 0 },
    { 116, 0, 8353, 0 },
    { 111, 0, 8354, 0 },
    { 114, 0, 8355, 0 },
    { 59, 0, 0, 10577 }, /* &LeftUpDownVector; */
    { 105, 0, 8357, 0 },
    { 110, 0, 8358, 0 },
    { 117, 0, 8359, 0 },
    { 115, 0, 8360, 0 },
    { 59, 0, 0, 8760 }, /* &dotminus; */
    { 59, 0, 0, 9579 }, /* &boxVh; */
    { 111, 0, 8363, 0 },
    { 115, 1, 8365, 0 },
    { 99, 0, 8431, 0 },
    { 101, 0, 8366, 0 },
    { 67, 0, 8367, 0 },
    { 117, 0, 8368, 0 },
    { 114, 0, 8369, 0 },
    { 108, 0, 8370, 0 },
    { 121, 0, 8371, 0 },
    { 68, 1, 8373, 0 },
    { 81, 0, 9598, 0 },
    { 111, 0, 8374, 0 },
    { 117, 0, 8375, 0 },
    { 98, 0, 8376, 0 },
    { 108, 0, 8377, 0 },
    { 101, 0, 8378, 0 },
    { 81, 0, 8379, 0 },
    { 117, 0, 8380, 0 },
    { 111, 0, 8381, 0 },
    { 116, 0, 8382, 0 },
    { 101, 0, 8383, 0 },
    { 59, 0, 0, 8221 }, /* &CloseCurlyDoubleQuote; */
    { 103, 1, 8386, 0 },
    { 112, 0, 8956, 0 },
    { 111, 0, 8387, 0 },
    { 110, 0, 8388, 0 },
    { 59, 0, 0, 281 }, /* &eogon; */
    { 116, 0, 8390, 0 },
    { 59, 0, 0, 8976 }, /* &bnot; */
    { 116, 0, 8392, 0 },
    { 101, 0, 8393, 0 },
    { 100, 0, 8394, 0 },
    { 71, 1, 8396, 0 },
    { 76, 0, 8660, 0 },
    { 114, 0, 8397, 0 },
    { 101, 0, 8398, 0 },
    { 97, 0, 8399, 0 },
    { 116, 0, 8400, 0 },
    { 101, 0, 8401, 0 },
    { 114, 0, 8402, 0 },
    { 71, 0, 8403, 0 },
    { 114, 0, 8404, 0 },
    { 101, 0, 8405, 0 },
    { 97, 0, 8406, 0 },
    { 116, 0, 8407, 0 },
    { 101, 0, 8408, 0 },
    { 114, 0, 8409, 0 },
    { 59, 0, 0, 8811 }, /* &NestedGreaterGreater; */
    { 59, 1, 0, 8822 }, /* &lg; */
    { 69, 0, 8877, 0 },
    { 105, 0, 8413, 0 },
    { 103, 0, 8414, 0 },
    { 104, 0, 8415, 0 },
    { 116, 0, 8416, 0 },
    { 59, 0, 0, 8620 }, /* &looparrowright; */
    { 113, 0, 8418, 0 },
    { 117, 0, 8419, 0 },
    { 97, 0, 8420, 0 },
    { 108, 0, 8421, 0 },
    { 59, 0, 0, 8850 }, /* &SquareSupersetEqual; */
    { 99, 0, 8423, 0 },
    { 117, 0, 8424, 0 },
    { 116, 0, 8425, 0 },
    { 101, 0, 8576, 205 }, /* &Iacute */
    { 110, 0, 8427, 0 },
    { 116, 0, 8428, 0 },
    { 59, 0, 0, 8753 }, /* &cwint; */
    { 114, 0, 8430, 0 },
    { 59, 0, 0, 120115 }, /* &vfr; */
    { 107, 0, 8432, 0 },
    { 119, 0, 8433, 0 },
    { 105, 0, 8434, 0 },
    { 115, 0, 8435, 0 },
    { 101, 0, 8436, 0 },
    { 67, 0, 8437, 0 },
    { 111, 0, 8438, 0 },
    { 110, 0, 8439, 0 },
    { 116, 0, 8440, 0 },
    { 111, 0, 8441, 0 },
    { 117, 0, 8442, 0 },
    { 114, 0, 8443, 0 },
    { 73, 0, 8444, 0 },
    { 110, 0, 8445, 0 },
    { 116, 0, 8446, 0 },
    { 101, 0, 8447, 0 },
    { 103, 0, 8448, 0 },
    { 114, 0, 8449, 0 },
    { 97, 0, 8450, 0 },
    { 108, 0, 8451, 0 },
    { 59, 0, 0, 8754 }, /* &ClockwiseContourIntegral; */
    { 114, 0, 8453, 0 },
    { 114, 0, 8454, 0 },
    { 59, 0, 0, 10609 }, /* &erarr; */
    { 105, 0, 8456, 0 },
    { 112, 0, 8457, 0 },
    { 108, 0, 8458, 0 },
    { 101, 0, 8459, 0 },
    { 68, 0, 8460, 0 },
    { 111, 0, 8461, 0 },
    { 116, 0, 8462, 0 },
    { 59, 0, 0, 8411 }, /* &TripleDot; */
    { 59, 0, 0, 10844 }, /* &and; */
    { 104, 0, 8465, 0 },
    { 105, 0, 8466, 0 },
    { 110, 1, 8468, 0 },
    { 99, 0, 9786, 0 },
    { 83, 0, 8469, 0 },
    { 112, 0, 8470, 0 },
    { 97, 0, 8471, 0 },
    { 99, 0, 8472, 0 },
    { 101, 0, 8473, 0 },
    { 59, 0, 0, 8203 }, /* &NegativeThinSpace; */
    { 99, 0, 8475, 0 },
    { 114, 1, 8477, 0 },
    { 111, 0, 9591, 0 },
    { 111, 0, 8478, 0 },
    { 112, 0, 8479, 0 },
    { 59, 0, 0, 8973 }, /* &dlcrop; */
    { 97, 0, 8481, 0 },
    { 115, 0, 8482, 0 },
    { 104, 0, 8483, 0 },
    { 59, 0, 0, 8866 }, /* &vdash; */
    { 105, 0, 8485, 0 },
    { 102, 1, 8487, 0 },
    { 59, 0, 0, 9653 }, /* &utri; */
    { 59, 0, 0, 9652 }, /* &utrif; */
    { 111, 0, 8489, 0 },
    { 119, 1, 8491, 0 },
    { 116, 0, 9643, 0 },
    { 110, 0, 8492, 0 },
    { 65, 0, 8493, 0 },
    { 114, 0, 8494, 0 },
    { 114, 0, 8495, 0 },
    { 111, 0, 8496, 0 },
    { 119, 0, 8497, 0 },
    { 59, 0, 0, 8659 }, /* &DoubleDownArrow; */
    { 111, 1, 8500, 0 },
    { 59, 0, 0, 8601 }, /* &swarr; */
    { 119, 0, 8501, 0 },
    { 59, 0, 0, 8601 }, /* &swarrow; */
    { 111, 0, 8503, 0 },
    { 116, 0, 8504, 0 },
    { 111, 1, 8506, 0 },
    { 59, 0, 0, 10880 }, /* &gesdot; */
    { 108, 1, 8508, 0 },
    { 59, 0, 0, 10882 }, /* &gesdoto; */
    { 59, 0, 0, 10884 }, /* &gesdotol; */
    { 111, 0, 8510, 0 },
    { 119, 0, 8511, 0 },
    { 110, 0, 8512, 0 },
    { 65, 0, 8513, 0 },
    { 114, 0, 8514, 0 },
    { 114, 0, 8515, 0 },
    { 111, 0, 8516, 0 },
    { 119, 0, 8517, 0 },
    { 59, 0, 0, 8645 }, /* &UpArrowDownArrow; */
    { 112, 0, 8519, 0 },
    { 112, 0, 8520, 0 },
    { 114, 0, 8521, 0 },
    { 111, 0, 8522, 0 },
    { 120, 0, 8523, 0 },
    { 59, 0, 0, 10936 }, /* &succapprox; */
    { 121, 0, 8525, 0 },
    { 59, 0, 0, 1055 }, /* &Pcy; */
    { 105, 0, 8527, 0 },
    { 108, 0, 8528, 0 },
    { 100, 0, 8529, 0 },
    { 101, 0, 8530, 0 },
    { 59, 0, 0, 8770 }, /* &NotEqualTilde; */
    { 105, 0, 8532, 0 },
    { 114, 0, 8533, 0 },
    { 59, 0, 0, 10786 }, /* &pluscir; */
    { 113, 0, 8535, 0 },
    { 117, 0, 8536, 0 },
    { 97, 0, 8537, 0 },
    { 108, 0, 8538, 0 },
    { 71, 0, 8539, 0 },
    { 114, 0, 8540, 0 },
    { 101, 0, 8541, 0 },
    { 97, 0, 8542, 0 },
    { 116, 0, 8543, 0 },
    { 101, 0, 8544, 0 },
    { 114, 0, 8545, 0 },
    { 59, 0, 0, 8922 }, /* &LessEqualGreater; */
    { 59, 0, 0, 8625 }, /* &rsh; */
    { 112, 0, 8548, 0 },
    { 69, 0, 8549, 0 },
    { 113, 0, 8550, 0 },
    { 117, 0, 8551, 0 },
    { 105, 0, 8552, 0 },
    { 108, 0, 8553, 0 },
    { 105, 0, 8554, 0 },
    { 98, 0, 8555, 0 },
    { 114, 0, 8556, 0 },
    { 105, 0, 8557, 0 },
    { 117, 0, 8558, 0 },
    { 109, 0, 8559, 0 },
    { 59, 0, 0, 10607 }, /* &ReverseUpEquilibrium; */
    { 59, 0, 0, 10688 }, /* &olt; */
    { 59, 0, 0, 120094 }, /* &afr; */
    { 97, 0, 8563, 0 },
    { 114, 0, 8564, 0 },
    { 59, 0, 0, 10596 }, /* &rHar; */
    { 114, 0, 8566, 0 },
    { 114, 0, 8567, 0 },
    { 59, 0, 0, 8662 }, /* &nwArr; */
    { 59, 0, 0, 9558 }, /* &boxDl; */
    { 97, 0, 8570, 0 },
    { 59, 0, 0, 921 }, /* &Iota; */
    { 99, 0, 8572, 0 },
    { 114, 0, 8573, 0 },
    { 59, 0, 0, 120000 }, /* &kscr; */
    { 114, 0, 8575, 0 },
    { 59, 0, 0, 120090 }, /* &Wfr; */
    { 59, 0, 0, 205 }, /* &Iacute; */
    { 103, 1, 8579, 0 },
    { 99, 0, 9137, 0 },
    { 105, 0, 8580, 0 },
    { 110, 0, 8581, 0 },
    { 97, 0, 8582, 0 },
    { 114, 0, 8583, 0 },
    { 121, 0, 8584, 0 },
    { 73, 0, 8585, 0 },
    { 59, 0, 0, 8520 }, /* &ImaginaryI; */
    { 114, 0, 8587, 0 },
    { 59, 0, 0, 119978 }, /* &Oscr; */
    { 105, 0, 8589, 0 },
    { 108, 0, 8590, 0 },
    { 59, 0, 0, 10522 }, /* &ratail; */
    { 59, 0, 0, 10895 }, /* &lsimg; */
    { 112, 0, 8593, 0 },
    { 112, 0, 8594, 0 },
    { 97, 0, 8595, 0 },
    { 59, 0, 0, 922 }, /* &Kappa; */
    { 104, 0, 8597, 0 },
    { 112, 0, 8598, 0 },
    { 59, 0, 0, 10803 }, /* &smashp; */
    { 108, 0, 8600, 0 },
    { 97, 0, 8601, 0 },
    { 110, 0, 8602, 0 },
    { 116, 0, 8603, 0 },
    { 69, 0, 8604, 0 },
    { 113, 0, 8605, 0 },
    { 117, 0, 8606, 0 },
    { 97, 0, 8607, 0 },
    { 108, 0, 8608, 0 },
    { 59, 0, 0, 8828 }, /* &PrecedesSlantEqual; */
    { 114, 0, 8610, 0 },
    { 101, 0, 8611, 0 },
    { 97, 0, 8612, 0 },
    { 116, 0, 8613, 0 },
    { 101, 0, 8614, 0 },
    { 114, 0, 8615, 0 },
    { 59, 0, 0, 8811 }, /* &NotGreaterGreater; */
    { 105, 0, 8617, 0 },
    { 99, 0, 8618, 0 },
    { 59, 0, 0, 8730 }, /* &radic; */
    { 114, 0, 8620, 0 },
    { 114, 0, 8621, 0 },
    { 99, 1, 8624, 0 },
    { 119, 1, 9544, 0 },
    { 59, 0, 0, 8603 }, /* &nrarr; */
    { 59, 0, 0, 10547 }, /* &nrarrc; */
    { 111, 0, 8626, 0 },
    { 116, 0, 8627, 0 },
    { 59, 0, 0, 8918 }, /* &lessdot; */
    { 121, 1, 8630, 0 },
    { 97, 0, 8903, 0 },
    { 59, 0, 0, 1047 }, /* &Zcy; */
    { 105, 1, 8633, 0 },
    { 112, 0, 9534, 0 },
    { 115, 0, 8634, 0 },
    { 116, 0, 8635, 0 },
    { 115, 0, 8636, 0 },
    { 59, 0, 0, 8707 }, /* &Exists; */
    { 59, 0, 0, 9617 }, /* &blk14; */
    { 105, 0, 8639, 0 },
    { 109, 0, 8640, 0 },
    { 59, 0, 0, 8830 }, /* &precsim; */
    { 59, 0, 0, 10642 }, /* &rangd; */
    { 114, 0, 8643, 0 },
    { 59, 0, 0, 120107 }, /* &nfr; */
    { 59, 0, 0, 10949 }, /* &nsubE; */
    { 59, 0, 0, 1094 }, /* &tscy; */
    { 59, 0, 0, 186 }, /* &ordm; */
    { 59, 0, 0, 8519 }, /* &ee; */
    { 101, 0, 8649, 0 },
    { 114, 0, 8650, 0 },
    { 112, 0, 8651, 0 },
    { 59, 0, 0, 8472 }, /* &weierp; */
    { 111, 1, 8654, 0 },
    { 121, 0, 9852, 0 },
    { 109, 0, 8655, 0 },
    { 109, 0, 8656, 0 },
    { 97, 0, 8657, 0 },
    { 59, 0, 0, 10793 }, /* &mcomma; */
    { 102, 0, 8659, 0 },
    { 59, 0, 0, 120125 }, /* &Fopf; */
    { 101, 0, 8661, 0 },
    { 115, 0, 8662, 0 },
    { 115, 0, 8663, 0 },
    { 76, 0, 8664, 0 },
    { 101, 0, 8665, 0 },
    { 115, 0, 8666, 0 },
    { 115, 0, 8667, 0 },
    { 59, 0, 0, 8810 }, /* &NestedLessLess; */
    { 111, 0, 8669, 0 },
    { 112, 0, 8670, 0 },
    { 59, 0, 0, 8975 }, /* &ulcrop; */
    { 59, 0, 0, 9662 }, /* &dtrif; */
    { 101, 0, 8673, 0 },
    { 115, 0, 8674, 0 },
    { 115, 0, 8675, 0 },
    { 59, 0, 0, 8823 }, /* &gtrless; */
    { 59, 1, 0, 8864 }, /* &timesb; */
    { 97, 0, 8961, 0 },
    { 59, 0, 0, 8805 }, /* &nvge; */
    { 112, 0, 8680, 0 },
    { 59, 0, 0, 10501 }, /* &Map; */
    { 59, 0, 0, 10956 }, /* &supnE; */
    { 59, 0, 0, 1044 }, /* &Dcy; */
    { 112, 0, 8684, 0 },
    { 59, 0, 0, 8899 }, /* &bigcup; */
    { 59, 0, 0, 217 }, /* &Ugrave; */
    { 108, 0, 8687, 0 },
    { 105, 0, 8688, 0 },
    { 103, 0, 8689, 0 },
    { 59, 0, 0, 339 }, /* &oelig; */
    { 105, 0, 8691, 0 },
    { 109, 0, 8692, 0 },
    { 59, 0, 0, 8819 }, /* &gtrsim; */
    { 59, 0, 0, 120157 }, /* &lopf; */
    { 105, 0, 8695, 0 },
    { 114, 0, 8696, 0 },
    { 99, 0, 8697, 0 },
    { 59, 0, 0, 292 }, /* &Hcirc; */
    { 110, 0, 8699, 0 },
    { 116, 0, 8700, 0 },
    { 101, 0, 8701, 0 },
    { 114, 0, 8702, 0 },
    { 115, 0, 8703, 0 },
    { 101, 0, 8704, 0 },
    { 99, 0, 8705, 0 },
    { 116, 0, 8706, 0 },
    { 105, 0, 8707, 0 },
    { 111, 0, 8708, 0 },
    { 110, 0, 8709, 0 },
    { 59, 0, 0, 8851 }, /* &SquareIntersection; */
    { 101, 0, 8711, 0 },
    { 115, 0, 8712, 0 },
    { 115, 0, 8713, 0 },
    { 59, 0, 0, 10901 }, /* &eqslantless; */
    { 110, 0, 8715, 0 },
    { 101, 0, 8716, 0 },
    { 59, 0, 0, 8254 }, /* &oline; */
    { 59, 0, 0, 10665 }, /* &angmsdab; */
    { 102, 0, 8719, 0 },
    { 59, 0, 0, 120147 }, /* &bopf; */
    { 103, 0, 8721, 0 },
    { 114, 0, 8722, 0 },
    { 116, 0, 8723, 0 },
    { 59, 0, 0, 10652 }, /* &vangrt; */
    { 97, 0, 8725, 0 },
    { 114, 0, 8726, 0 },
    { 59, 0, 0, 10580 }, /* &RightUpVectorBar; */
    { 103, 0, 8728, 0 },
    { 109, 0, 8729, 0 },
    { 97, 0, 8730, 0 },
    { 59, 0, 0, 931 }, /* &Sigma; */
    { 59, 0, 0, 10731 }, /* &lozf; */
    { 108, 0, 8733, 0 },
    { 105, 0, 8734, 0 },
    { 103, 0, 8735, 0 },
    { 59, 0, 0, 306 }, /* &IJlig; */
    { 71, 0, 8737, 0 },
    { 59, 0, 0, 330 }, /* &ENG; */
    { 114, 0, 8739, 0 },
    { 99, 0, 9516, 202 }, /* &Ecirc */
    { 114, 0, 8741, 0 },
    { 59, 0, 0, 119990 }, /* &ascr; */
    { 111, 0, 8743, 0 },
    { 117, 0, 8744, 0 },
    { 114, 0, 8745, 0 },
    { 73, 0, 8746, 0 },
    { 110, 0, 8747, 0 },
    { 116, 0, 8748, 0 },
    { 101, 0, 8749, 0 },
    { 103, 0, 8750, 0 },
    { 114, 0, 8751, 0 },
    { 97, 0, 8752, 0 },
    { 108, 0, 8753, 0 },
    { 59, 0, 0, 8750 }, /* &ContourIntegral; */
    { 105, 0, 8755, 0 },
    { 108, 0, 8756, 0 },
    { 100, 0, 8757, 0 },
    { 101, 0, 8758, 0 },
    { 59, 0, 0, 8818 }, /* &LessTilde; */
    { 111, 0, 8760, 0 },
    { 117, 0, 8761, 0 },
    { 98, 0, 8762, 0 },
    { 108, 0, 8763, 0 },
    { 101, 0, 8764, 0 },
    { 86, 0, 8765, 0 },
    { 101, 0, 8766, 0 },
    { 114, 0, 8767, 0 },
    { 116, 0, 8768, 0 },
    { 105, 0, 8769, 0 },
    { 99, 0, 8770, 0 },
    { 97, 0, 8771, 0 },
    { 108, 0, 8772, 0 },
    { 66, 0, 8773, 0 },
    { 97, 0, 8774, 0 },
    { 114, 0, 8775, 0 },
    { 59, 0, 0, 8742 }, /* &NotDoubleVerticalBar; */
    { 59, 0, 0, 8533 }, /* &frac15; */
    { 114, 1, 8779, 0 },
    { 103, 0, 9410, 0 },
    { 114, 0, 8780, 0 },
    { 59, 0, 0, 8650 }, /* &ddarr; */
    { 99, 0, 8782, 0 },
    { 114, 0, 8783, 0 },
    { 59, 0, 0, 8499 }, /* &Mscr; */
    { 114, 0, 8785, 0 },
    { 59, 0, 0, 120109 }, /* &pfr; */
    { 115, 0, 8787, 0 },
    { 59, 0, 0, 10806 }, /* &otimesas; */
    { 101, 0, 8789, 0 },
    { 116, 0, 8790, 0 },
    { 59, 0, 0, 8913 }, /* &Supset; */
    { 111, 0, 8792, 0 },
    { 110, 0, 8793, 0 },
    { 59, 0, 0, 302 }, /* &Iogon; */
    { 115, 0, 8795, 0 },
    { 116, 0, 8796, 0 },
    { 59, 0, 0, 8707 }, /* &exist; */
    { 99, 0, 8798, 0 },
    { 107, 1, 8800, 0 },
    { 101, 0, 9667, 0 },
    { 59, 0, 0, 93 }, /* &rbrack; */
    { 59, 0, 0, 968 }, /* &psi; */
    { 120, 0, 8803, 0 },
    { 116, 0, 8804, 0 },
    { 59, 0, 0, 8812 }, /* &twixt; */
    { 105, 0, 8806, 0 },
    { 108, 0, 8807, 0 },
    { 101, 0, 8808, 0 },
    { 59, 0, 0, 8995 }, /* &ssmile; */
    { 107, 0, 8810, 0 },
    { 99, 0, 8811, 0 },
    { 121, 0, 8812, 0 },
    { 59, 0, 0, 1108 }, /* &jukcy; */
    { 59, 0, 0, 9562 }, /* &boxUR; */
    { 111, 0, 8815, 0 },
    { 59, 0, 0, 8487 }, /* &mho; */
    { 110, 0, 8817, 0 },
    { 100, 0, 8818, 0 },
    { 59, 0, 0, 10820 }, /* &capand; */
    { 112, 0, 8820, 0 },
    { 59, 0, 0, 10935 }, /* &prap; */
    { 105, 0, 8822, 0 },
    { 109, 0, 8823, 0 },
    { 101, 0, 8824, 0 },
    { 115, 0, 8825, 0 },
    { 59, 0, 0, 8290 }, /* &InvisibleTimes; */
    { 66, 0, 8827, 0 },
    { 114, 0, 8828, 0 },
    { 101, 0, 8829, 0 },
    { 97, 0, 8830, 0 },
    { 107, 0, 8831, 0 },
    { 105, 0, 8832, 0 },
    { 110, 0, 8833, 0 },
    { 103, 0, 8834, 0 },
    { 83, 0, 8835, 0 },
    { 112, 0, 8836, 0 },
    { 97, 0, 8837, 0 },
    { 99, 0, 8838, 0 },
    { 101, 0, 8839, 0 },
    { 59, 0, 0, 160 }, /* &NonBreakingSpace; */
    { 59, 0, 0, 10955 }, /* &subsetneqq; */
    { 112, 0, 8842, 0 },
    { 112, 0, 8843, 0 },
    { 114, 0, 8844, 0 },
    { 111, 0, 8845, 0 },
    { 120, 0, 8846, 0 },
    { 59, 0, 0, 8776 }, /* &thickapprox; */
    { 111, 0, 8848, 0 },
    { 119, 0, 8849, 0 },
    { 110, 0, 8850, 0 },
    { 59, 0, 0, 9661 }, /* &bigtriangledown; */
    { 109, 0, 8852, 0 },
    { 105, 0, 8853, 0 },
    { 100, 0, 8854, 0 },
    { 59, 0, 0, 10990 }, /* &rnmid; */
    { 105, 0, 8856, 0 },
    { 103, 0, 8857, 0 },
    { 104, 0, 8858, 0 },
    { 116, 0, 8859, 0 },
    { 65, 0, 8860, 0 },
    { 114, 0, 8861, 0 },
    { 114, 0, 8862, 0 },
    { 111, 0, 8863, 0 },
    { 119, 0, 8864, 0 },
    { 59, 0, 0, 8660 }, /* &DoubleLeftRightArrow; */
    { 68, 1, 8867, 0 },
    { 114, 0, 9341, 0 },
    { 111, 0, 8868, 0 },
    { 116, 0, 8869, 0 },
    { 59, 0, 0, 8786 }, /* &efDot; */
    { 108, 0, 8871, 0 },
    { 99, 0, 8872, 0 },
    { 116, 0, 8873, 0 },
    { 121, 0, 8874, 0 },
    { 59, 0, 0, 9005 }, /* &cylcty; */
    { 59, 1, 0, 10901 }, /* &els; */
    { 100, 0, 9669, 0 },
    { 59, 0, 0, 10897 }, /* &lgE; */
    { 105, 0, 8879, 0 },
    { 109, 0, 8880, 0 },
    { 59, 0, 0, 10790 }, /* &plussim; */
    { 103, 0, 8882, 0 },
    { 59, 0, 0, 10221 }, /* &roang; */
    { 105, 0, 8884, 0 },
    { 109, 0, 8885, 0 },
    { 101, 0, 8886, 0 },
    { 115, 0, 8887, 0 },
    { 59, 0, 0, 10804 }, /* &lotimes; */
    { 114, 0, 8889, 0 },
    { 59, 0, 0, 8458 }, /* &gscr; */
    { 109, 0, 8891, 0 },
    { 105, 0, 8892, 0 },
    { 59, 0, 0, 8271 }, /* &bsemi; */
    { 114, 0, 8894, 0 },
    { 59, 0, 0, 120103 }, /* &jfr; */
    { 112, 0, 8896, 0 },
    { 102, 0, 8897, 0 },
    { 59, 0, 0, 120134 }, /* &Oopf; */
    { 111, 0, 8899, 0 },
    { 116, 0, 8900, 0 },
    { 59, 0, 0, 10989 }, /* &bNot; */
    { 112, 0, 8902, 0 },
    { 59, 0, 0, 10826 }, /* &cupcup; */
    { 114, 0, 8904, 0 },
    { 111, 0, 8905, 0 },
    { 110, 0, 8906, 0 },
    { 59, 0, 0, 381 }, /* &Zcaron; */
    { 99, 1, 8909, 0 },
    { 101, 0, 9001, 0 },
    { 114, 0, 8910, 0 },
    { 59, 0, 0, 119999 }, /* &jscr; */
    { 101, 0, 8912, 0 },
    { 114, 0, 8913, 0 },
    { 121, 0, 8914, 0 },
    { 83, 0, 8915, 0 },
    { 109, 0, 8916, 0 },
    { 97, 0, 8917, 0 },
    { 108, 0, 8918, 0 },
    { 108, 0, 8919, 0 },
    { 83, 0, 8920, 0 },
    { 113, 0, 8921, 0 },
    { 117, 0, 8922, 0 },
    { 97, 0, 8923, 0 },
    { 114, 0, 8924, 0 },
    { 101, 0, 8925, 0 },
    { 59, 0, 0, 9643 }, /* &EmptyVerySmallSquare; */
    { 112, 0, 8927, 0 },
    { 116, 0, 8928, 0 },
    { 121, 0, 8929, 0 },
    { 118, 0, 8930, 0 },
    { 59, 0, 0, 10672 }, /* &bemptyv; */
    { 115, 0, 8932, 0 },
    { 59, 0, 0, 10527 }, /* &larrbfs; */
    { 116, 0, 8934, 0 },
    { 101, 0, 8935, 253 }, /* &yacute */
    { 59, 0, 0, 253 }, /* &yacute; */
    { 114, 0, 8937, 0 },
    { 59, 0, 0, 120011 }, /* &vscr; */
    { 71, 0, 8939, 174 }, /* &REG */
    { 59, 0, 0, 174 }, /* &REG; */
    { 105, 0, 8941, 0 },
    { 103, 0, 8942, 0 },
    { 104, 0, 8943, 0 },
    { 116, 0, 8944, 0 },
    { 65, 0, 8945, 0 },
    { 114, 0, 8946, 0 },
    { 114, 0, 8947, 0 },
    { 111, 0, 8948, 0 },
    { 119, 0, 8949, 0 },
    { 59, 0, 0, 10233 }, /* &DoubleLongRightArrow; */
    { 99, 0, 8951, 0 },
    { 114, 0, 8952, 0 },
    { 59, 0, 0, 119977 }, /* &Nscr; */
    { 97, 0, 8954, 0 },
    { 114, 0, 8955, 0 },
    { 59, 0, 0, 10597 }, /* &dHar; */
    { 102, 0, 8957, 0 },
    { 59, 0, 0, 120150 }, /* &eopf; */
    { 59, 0, 0, 197 }, /* &angst; */
    { 98, 0, 8960, 0 },
    { 59, 0, 0, 125 }, /* &rcub; */
    { 114, 0, 8962, 0 },
    { 59, 0, 0, 10801 }, /* &timesbar; */
    { 116, 0, 8964, 0 },
    { 114, 0, 8965, 0 },
    { 105, 0, 8966, 0 },
    { 59, 0, 0, 8883 }, /* &vrtri; */
    { 111, 0, 8968, 0 },
    { 115, 0, 8969, 0 },
    { 115, 0, 8970, 0 },
    { 59, 0, 0, 10799 }, /* &Cross; */
    { 101, 0, 8972, 0 },
    { 101, 0, 8973, 0 },
    { 86, 0, 8974, 0 },
    { 101, 0, 8975, 0 },
    { 99, 0, 8976, 0 },
    { 116, 0, 8977, 0 },
    { 111, 0, 8978, 0 },
    { 114, 0, 8979, 0 },
    { 59, 0, 0, 10589 }, /* &RightDownTeeVector; */
    { 99, 0, 8981, 0 },
    { 117, 0, 8982, 0 },
    { 112, 0, 8983, 0 },
    { 59, 0, 0, 10758 }, /* &xsqcup; */
    { 105, 0, 8985, 0 },
    { 117, 0, 8986, 0 },
    { 109, 0, 8987, 0 },
    { 83, 0, 8988, 0 },
    { 112, 0, 8989, 0 },
    { 97, 0, 8990, 0 },
    { 99, 0, 8991, 0 },
    { 101, 0, 8992, 0 },
    { 59, 0, 0, 8287 }, /* &MediumSpace; */
    { 99, 0, 8994, 0 },
    { 114, 0, 8995, 0 },
    { 59, 0, 0, 120012 }, /* &wscr; */
    { 59, 0, 0, 8625 }, /* &Rsh; */
    { 100, 0, 8998, 0 },
    { 111, 0, 8999, 0 },
    { 116, 0, 9000, 0 },
    { 59, 0, 0, 305 }, /* &inodot; */
    { 114, 0, 9002, 0 },
    { 99, 0, 9003, 0 },
    { 121, 0, 9004, 0 },
    { 59, 0, 0, 1112 }, /* &jsercy; */
    { 100, 0, 9006, 0 },
    { 105, 0, 9007, 0 },
    { 108, 0, 9008, 0 },
    { 59, 0, 0, 326 }, /* &ncedil; */
    { 110, 0, 9010, 0 },
    { 116, 0, 9011, 0 },
    { 59, 0, 0, 8750 }, /* &oint; */
    { 59, 0, 0, 11005 }, /* &parsl; */
    { 97, 0, 9014, 0 },
    { 114, 0, 9015, 0 },
    { 59, 0, 0, 10583 }, /* &DownRightVectorBar; */
    { 115, 0, 9017, 0 },
    { 59, 0, 0, 175 }, /* &strns; */
    { 114, 0, 9019, 0 },
    { 99, 0, 9121, 226 }, /* &acirc */
    { 113, 0, 9021, 0 },
    { 59, 1, 0, 8839 }, /* &supseteq; */
    { 113, 0, 9828, 0 },
    { 108, 0, 9024, 0 },
    { 98, 0, 9025, 0 },
    { 101, 0, 9026, 0 },
    { 114, 0, 9027, 0 },
    { 116, 0, 9028, 0 },
    { 83, 0, 9029, 0 },
    { 112, 0, 9030, 0 },
    { 97, 0, 9031, 0 },
    { 99, 0, 9032, 0 },
    { 101, 0, 9033, 0 },
    { 59, 0, 0, 8459 }, /* &HilbertSpace; */
    { 114, 0, 9035, 0 },
    { 114, 0, 9036, 0 },
    { 59, 0, 0, 8666 }, /* &lAarr; */
    { 112, 0, 9038, 0 },
    { 112, 0, 9039, 0 },
    { 114, 0, 9040, 0 },
    { 111, 0, 9041, 0 },
    { 120, 0, 9042, 0 },
    { 59, 0, 0, 10937 }, /* &precnapprox; */
    { 69, 1, 9047, 0 },
    { 59, 1, 0, 8823 }, /* &gl; */
    { 97, 1, 9660, 0 },
    { 106, 0, 9661, 0 },
    { 59, 0, 0, 10898 }, /* &glE; */
    { 117, 0, 9049, 0 },
    { 101, 0, 9050, 0 },
    { 59, 0, 0, 8829 }, /* &sccue; */
    { 105, 0, 9052, 0 },
    { 103, 0, 9053, 0 },
    { 104, 0, 9054, 0 },
    { 116, 0, 9055, 0 },
    { 97, 0, 9056, 0 },
    { 114, 0, 9057, 0 },
    { 114, 0, 9058, 0 },
    { 111, 0, 9059, 0 },
    { 119, 0, 9060, 0 },
    { 59, 0, 0, 10230 }, /* &longrightarrow; */
    { 112, 0, 9062, 0 },
    { 108, 0, 9063, 0 },
    { 117, 0, 9064, 0 },
    { 115, 0, 9065, 0 },
    { 59, 0, 0, 8723 }, /* &mnplus; */
    { 115, 0, 9067, 0 },
    { 121, 0, 9068, 0 },
    { 109, 0, 9069, 0 },
    { 59, 0, 0, 8501 }, /* &alefsym; */
    { 59, 0, 0, 8745 }, /* &caps; */
    { 97, 0, 9072, 0 },
    { 114, 0, 9073, 0 },
    { 104, 0, 9074, 0 },
    { 107, 0, 9075, 0 },
    { 59, 0, 0, 10775 }, /* &intlarhk; */
    { 114, 0, 9077, 0 },
    { 114, 0, 9078, 0 },
    { 111, 0, 9079, 0 },
    { 119, 0, 9080, 0 },
    { 59, 0, 0, 8613 }, /* &UpTeeArrow; */
    { 101, 0, 9082, 0 },
    { 101, 0, 9083, 0 },
    { 110, 0, 9084, 0 },
    { 59, 0, 0, 8812 }, /* &between; */
    { 119, 0, 9086, 0 },
    { 97, 0, 9087, 0 },
    { 114, 0, 9088, 0 },
    { 59, 0, 0, 10538 }, /* &swnwar; */
    { 98, 0, 9090, 0 },
    { 59, 0, 0, 10701 }, /* &trisb; */
    { 59, 0, 0, 164 }, /* &curren; */
    { 108, 0, 9093, 0 },
    { 117, 0, 9094, 0 },
    { 115, 0, 9095, 0 },
    { 59, 0, 0, 10788 }, /* &simplus; */
    { 105, 0, 9097, 0 },
    { 100, 0, 9098, 0 },
    { 59, 0, 0, 8740 }, /* &nshortmid; */
    { 99, 1, 9101, 0 },
    { 110, 0, 9549, 0 },
    { 107, 0, 9102, 0 },
    { 83, 0, 9103, 0 },
    { 112, 0, 9104, 0 },
    { 97, 0, 9105, 0 },
    { 99, 0, 9106, 0 },
    { 101, 0, 9107, 0 },
    { 59, 0, 0, 8287 }, /* &ThickSpace; */
    { 59, 0, 0, 243 }, /* &oacute; */
    { 102, 0, 9110, 0 },
    { 59, 0, 0, 120166 }, /* &uopf; */
    { 59, 0, 0, 203 }, /* &Euml; */
    { 59, 0, 0, 10661 }, /* &range; */
    { 112, 0, 9114, 0 },
    { 104, 0, 9115, 0 },
    { 97, 0, 9116, 0 },
    { 59, 0, 0, 913 }, /* &Alpha; */
    { 59, 0, 0, 242 }, /* &ograve; */
    { 114, 0, 9119, 0 },
    { 99, 0, 9120, 0 },
    { 59, 0, 0, 284 }, /* &Gcirc; */
    { 59, 0, 0, 226 }, /* &acirc; */
    { 112, 0, 9123, 0 },
    { 102, 0, 9124, 0 },
    { 59, 0, 0, 120142 }, /* &Wopf; */
    { 102, 0, 9126, 0 },
    { 59, 0, 0, 120153 }, /* &hopf; */
    { 59, 0, 0, 211 }, /* &Oacute; */
    { 114, 0, 9129, 0 },
    { 97, 0, 9130, 0 },
    { 118, 0, 9131, 0 },
    { 101, 0, 9793, 249 }, /* &ugrave */
    { 99, 0, 9133, 0 },
    { 117, 0, 9134, 0 },
    { 116, 0, 9135, 0 },
    { 101, 0, 9136, 0 },
    { 59, 0, 0, 377 }, /* &Zacute; */
    { 114, 0, 9138, 0 },
    { 59, 0, 0, 298 }, /* &Imacr; */
    { 111, 0, 9140, 0 },
    { 119, 0, 9141, 0 },
    { 110, 0, 9142, 0 },
    { 59, 0, 0, 8641 }, /* &rightharpoondown; */
    { 116, 0, 9144, 0 },
    { 97, 0, 9145, 0 },
    { 59, 0, 0, 8489 }, /* &iiota; */
    { 113, 0, 9147, 0 },
    { 117, 0, 9148, 0 },
    { 97, 0, 9149, 0 },
    { 108, 0, 9150, 0 },
    { 59, 0, 0, 8941 }, /* &NotRightTriangleEqual; */
    { 97, 0, 9152, 0 },
    { 114, 0, 9153, 0 },
    { 59, 0, 0, 10595 }, /* &uHar; */
    { 114, 0, 9155, 0 },
    { 114, 0, 9156, 0 },
    { 59, 0, 0, 8667 }, /* &rAarr; */
    { 99, 0, 9158, 0 },
    { 121, 0, 9159, 0 },
    { 59, 0, 0, 1046 }, /* &ZHcy; */
    { 105, 0, 9161, 0 },
    { 103, 0, 9162, 0 },
    { 104, 0, 9163, 0 },
    { 116, 0, 9164, 0 },
    { 86, 0, 9165, 0 },
    { 101, 0, 9166, 0 },
    { 99, 0, 9167, 0 },
    { 116, 0, 9168, 0 },
    { 111, 0, 9169, 0 },
    { 114, 0, 9170, 0 },
    { 59, 0, 0, 10576 }, /* &DownLeftRightVector; */
    { 117, 1, 9173, 0 },
    { 101, 0, 9824, 0 },
    { 116, 0, 9174, 0 },
    { 101, 0, 9175, 0 },
    { 59, 0, 0, 341 }, /* &racute; */
    { 116, 0, 9177, 0 },
    { 59, 1, 0, 8735 }, /* &angrt; */
    { 118, 0, 9486, 0 },
    { 114, 0, 9180, 0 },
    { 114, 0, 9181, 0 },
    { 59, 0, 0, 8645 }, /* &udarr; */
    { 97, 0, 9183, 0 },
    { 114, 0, 9184, 0 },
    { 116, 0, 9185, 0 },
    { 105, 0, 9186, 0 },
    { 110, 0, 9187, 0 },
    { 116, 0, 9188, 0 },
    { 59, 0, 0, 10765 }, /* &fpartint; */
    { 116, 0, 9190, 0 },
    { 59, 0, 0, 8226 }, /* &bullet; */
    { 111, 0, 9192, 0 },
    { 114, 0, 9193, 0 },
    { 101, 0, 9194, 0 },
    { 59, 0, 0, 8756 }, /* &therefore; */
    { 59, 0, 0, 8953 }, /* &notinE; */
    { 117, 0, 9197, 0 },
    { 116, 0, 9198, 0 },
    { 101, 0, 9199, 0 },
    { 59, 0, 0, 262 }, /* &Cacute; */
    { 101, 0, 9201, 0 },
    { 110, 0, 9202, 0 },
    { 107, 0, 9203, 0 },
    { 59, 0, 0, 8241 }, /* &pertenk; */
    { 59, 0, 0, 206 }, /* &Icirc; */
    { 59, 0, 0, 183 }, /* &middot; */
    { 105, 0, 9207, 0 },
    { 109, 0, 9208, 0 },
    { 59, 0, 0, 8937 }, /* &scnsim; */
    { 108, 0, 9210, 0 },
    { 59, 0, 0, 10565 }, /* &rarrpl; */
    { 111, 0, 9212, 0 },
    { 116, 0, 9213, 0 },
    { 59, 0, 0, 304 }, /* &Idot; */
    { 110, 0, 9215, 0 },
    { 103, 0, 9216, 0 },
    { 101, 0, 9217, 0 },
    { 59, 0, 0, 9674 }, /* &lozenge; */
    { 97, 0, 9219, 0 },
    { 114, 0, 9220, 0 },
    { 111, 0, 9221, 0 },
    { 119, 0, 9222, 0 },
    { 59, 0, 0, 10509 }, /* &bkarow; */
    { 114, 0, 9224, 0 },
    { 59, 0, 0, 8493 }, /* &Cfr; */
    { 59, 0, 0, 8532 }, /* &frac23; */
    { 59, 0, 0, 188 }, /* &frac14; */
    { 59, 0, 0, 8746 }, /* &cups; */
    { 98, 0, 9229, 0 },
    { 108, 0, 9230, 0 },
    { 101, 0, 9231, 0 },
    { 65, 0, 9232, 0 },
    { 99, 0, 9233, 0 },
    { 117, 0, 9234, 0 },
    { 116, 0, 9235, 0 },
    { 101, 0, 9236, 0 },
    { 59, 0, 0, 733 }, /* &DiacriticalDoubleAcute; */
    { 110, 0, 9238, 0 },
    { 101, 0, 9239, 0 },
    { 110, 0, 9240, 0 },
    { 116, 0, 9241, 0 },
    { 105, 0, 9242, 0 },
    { 97, 0, 9243, 0 },
    { 108, 0, 9244, 0 },
    { 101, 0, 9245, 0 },
    { 59, 0, 0, 8519 }, /* &exponentiale; */
    { 99, 0, 9247, 0 },
    { 121, 0, 9248, 0 },
    { 59, 0, 0, 1113 }, /* &ljcy; */
    { 97, 0, 9250, 0 },
    { 114, 0, 9251, 0 },
    { 59, 0, 0, 10847 }, /* &wedbar; */
    { 105, 0, 9253, 0 },
    { 103, 0, 9254, 0 },
    { 104, 0, 9255, 0 },
    { 116, 0, 9256, 0 },
    { 97, 0, 9257, 0 },
    { 114, 0, 9258, 0 },
    { 114, 0, 9259, 0 },
    { 111, 0, 9260, 0 },
    { 119, 0, 9261, 0 },
    { 59, 0, 0, 10233 }, /* &Longrightarrow; */
    { 113, 0, 9263, 0 },
    { 59, 0, 0, 8849 }, /* &sqsubseteq; */
    { 59, 0, 0, 8467 }, /* &ell; */
    { 97, 0, 9266, 0 },
    { 114, 0, 9267, 0 },
    { 59, 0, 0, 10578 }, /* &LeftVectorBar; */
    { 108, 0, 9269, 0 },
    { 97, 0, 9270, 0 },
    { 110, 0, 9271, 0 },
    { 116, 0, 9272, 0 },
    { 59, 0, 0, 10878 }, /* &geqslant; */
    { 99, 0, 9274, 0 },
    { 114, 0, 9275, 0 },
    { 59, 0, 0, 119987 }, /* &Xscr; */
    { 59, 0, 0, 8639 }, /* &uharl; */
    { 59, 0, 0, 10819 }, /* &ncap; */
    { 108, 0, 9279, 0 },
    { 117, 0, 9280, 0 },
    { 115, 0, 9281, 0 },
    { 59, 0, 0, 10943 }, /* &subplus; */
    { 97, 0, 9283, 0 },
    { 114, 0, 9284, 0 },
    { 101, 0, 9285, 0 },
    { 110, 0, 9286, 0 },
    { 116, 0, 9287, 0 },
    { 104, 0, 9288, 0 },
    { 101, 0, 9289, 0 },
    { 115, 0, 9290, 0 },
    { 105, 0, 9291, 0 },
    { 115, 0, 9292, 0 },
    { 59, 0, 0, 9181 }, /* &UnderParenthesis; */
    { 114, 0, 9294, 0 },
    { 107, 0, 9295, 0 },
    { 59, 0, 0, 9140 }, /* &tbrk; */
    { 59, 0, 0, 8643 }, /* &dharl; */
    { 111, 0, 9298, 0 },
    { 100, 0, 9299, 0 },
    { 59, 0, 0, 46 }, /* &period; */
    { 59, 0, 0, 8807 }, /* &ngE; */
    { 112, 0, 9302, 0 },
    { 102, 0, 9303, 0 },
    { 59, 0, 0, 120126 }, /* &Gopf; */
    { 113, 0, 9305, 0 },
    { 59, 0, 0, 8885 }, /* &trianglerighteq; */
    { 59, 0, 0, 10671 }, /* &angmsdah; */
    { 59, 0, 0, 936 }, /* &Psi; */
    { 111, 0, 9309, 0 },
    { 111, 0, 9310, 0 },
    { 114, 0, 9311, 0 },
    { 59, 0, 0, 8971 }, /* &rfloor; */
    { 99, 0, 9313, 0 },
    { 114, 0, 9314, 0 },
    { 59, 0, 0, 120006 }, /* &qscr; */
    { 114, 0, 9316, 0 },
    { 114, 0, 9317, 0 },
    { 59, 0, 0, 8622 }, /* &nharr; */
    { 113, 0, 9319, 0 },
    { 117, 0, 9320, 0 },
    { 111, 0, 9321, 0 },
    { 59, 0, 0, 8249 }, /* &lsaquo; */
    { 59, 0, 0, 10956 }, /* &supsetneqq; */
    { 59, 0, 0, 982 }, /* &piv; */
    { 99, 0, 9325, 0 },
    { 121, 0, 9326, 0 },
    { 59, 0, 0, 1097 }, /* &shchcy; */
    { 84, 0, 9328, 0 },
    { 104, 0, 9329, 0 },
    { 97, 0, 9330, 0 },
    { 116, 0, 9331, 0 },
    { 59, 0, 0, 8715 }, /* &SuchThat; */
    { 59, 0, 0, 8885 }, /* &rtrie; */
    { 105, 0, 9334, 0 },
    { 109, 0, 9335, 0 },
    { 101, 0, 9336, 0 },
    { 59, 0, 0, 10754 }, /* &xotime; */
    { 105, 0, 9338, 0 },
    { 100, 0, 9339, 0 },
    { 59, 0, 0, 8740 }, /* &nmid; */
    { 59, 0, 0, 190 }, /* &frac34; */
    { 59, 0, 0, 120098 }, /* &efr; */
    { 105, 0, 9343, 0 },
    { 103, 0, 9344, 0 },
    { 104, 0, 9345, 0 },
    { 116, 0, 9346, 0 },
    { 97, 0, 9347, 0 },
    { 114, 0, 9348, 0 },
    { 114, 0, 9349, 0 },
    { 111, 0, 9350, 0 },
    { 119, 0, 9351, 0 },
    { 59, 0, 0, 10234 }, /* &Longleftrightarrow; */
    { 114, 0, 9353, 0 },
    { 59, 0, 0, 120106 }, /* &mfr; */
    { 59, 0, 0, 120099 }, /* &ffr; */
    { 101, 0, 9356, 0 },
    { 102, 0, 9357, 0 },
    { 116, 0, 9358, 0 },
    { 65, 0, 9359, 0 },
    { 114, 0, 9360, 0 },
    { 114, 0, 9361, 0 },
    { 111, 0, 9362, 0 },
    { 119, 0, 9363, 0 },
    { 59, 0, 0, 8644 }, /* &RightArrowLeftArrow; */
    { 111, 0, 9365, 0 },
    { 116, 0, 9366, 0 },
    { 59, 0, 0, 279 }, /* &edot; */
    { 105, 1, 9369, 0 },
    { 59, 0, 0, 8290 }, /* &it; */
    { 108, 0, 9370, 0 },
    { 100, 0, 9371, 0 },
    { 101, 0, 9372, 0 },
    { 59, 0, 0, 297 }, /* &itilde; */
    { 116, 0, 9374, 0 },
    { 105, 0, 9375, 0 },
    { 109, 0, 9376, 0 },
    { 97, 0, 9377, 0 },
    { 112, 0, 9378, 0 },
    { 59, 0, 0, 8888 }, /* &multimap; */
    { 114, 0, 9380, 0 },
    { 114, 0, 9381, 0 },
    { 59, 0, 0, 8629 }, /* &crarr; */
    { 97, 0, 9383, 0 },
    { 114, 0, 9384, 0 },
    { 59, 0, 0, 10594 }, /* &lHar; */
    { 99, 0, 9386, 0 },
    { 121, 0, 9387, 0 },
    { 59, 0, 0, 1111 }, /* &yicy; */
    { 97, 0, 9389, 0 },
    { 114, 0, 9390, 0 },
    { 59, 0, 0, 10703 }, /* &NotLeftTriangleBar; */
    { 108, 0, 9392, 0 },
    { 105, 0, 9393, 0 },
    { 112, 0, 9394, 0 },
    { 59, 0, 0, 8942 }, /* &vellip; */
    { 105, 0, 9396, 0 },
    { 108, 0, 9397, 0 },
    { 100, 0, 9398, 0 },
    { 101, 0, 9399, 0 },
    { 59, 0, 0, 8777 }, /* &NotTildeTilde; */
    { 114, 0, 9401, 0 },
    { 111, 0, 9402, 0 },
    { 59, 0, 0, 8470 }, /* &numero; */
    { 59, 0, 0, 10949 }, /* &nsubseteqq; */
    { 105, 0, 9405, 0 },
    { 100, 0, 9406, 0 },
    { 111, 0, 9407, 0 },
    { 116, 0, 9408, 0 },
    { 59, 0, 0, 319 }, /* &Lmidot; */
    { 59, 0, 0, 8465 }, /* &image; */
    { 103, 0, 9411, 0 },
    { 101, 0, 9412, 0 },
    { 114, 0, 9413, 0 },
    { 59, 0, 0, 8225 }, /* &ddagger; */
    { 59, 0, 0, 8767 }, /* &acd; */
    { 59, 0, 0, 8221 }, /* &rdquor; */
    { 113, 0, 9417, 0 },
    { 117, 0, 9418, 0 },
    { 97, 0, 9419, 0 },
    { 108, 0, 9420, 0 },
    { 59, 0, 0, 8885 }, /* &RightTriangleEqual; */
    { 118, 0, 9422, 0 },
    { 101, 0, 9423, 0 },
    { 59, 0, 0, 728 }, /* &breve; */
    { 107, 0, 9425, 0 },
    { 97, 0, 9426, 0 },
    { 114, 0, 9427, 0 },
    { 111, 0, 9428, 0 },
    { 119, 0, 9429, 0 },
    { 59, 0, 0, 10512 }, /* &drbkarow; */
    { 97, 0, 9431, 0 },
    { 114, 0, 9432, 0 },
    { 59, 0, 0, 10607 }, /* &duhar; */
    { 59, 0, 0, 221 }, /* &Yacute; */
    { 99, 0, 9435, 0 },
    { 121, 0, 9436, 0 },
    { 59, 0, 0, 1110 }, /* &iukcy; */
    { 100, 0, 9438, 0 },
    { 105, 0, 9439, 0 },
    { 108, 0, 9440, 0 },
    { 59, 0, 0, 350 }, /* &Scedil; */
    { 119, 0, 9442, 0 },
    { 59, 0, 0, 8598 }, /* &nwarrow; */
    { 117, 0, 9444, 0 },
    { 108, 0, 9445, 0 },
    { 108, 0, 9446, 0 },
    { 69, 0, 9447, 0 },
    { 113, 0, 9448, 0 },
    { 117, 0, 9449, 0 },
    { 97, 0, 9450, 0 },
    { 108, 0, 9451, 0 },
    { 59, 0, 0, 8807 }, /* &NotGreaterFullEqual; */
    { 59, 0, 0, 9642 }, /* &squarf; */
    { 59, 0, 0, 10886 }, /* &gap; */
    { 114, 0, 9455, 0 },
    { 116, 0, 9456, 0 },
    { 110, 0, 9457, 0 },
    { 101, 0, 9458, 0 },
    { 113, 0, 9459, 0 },
    { 113, 0, 9460, 0 },
    { 59, 0, 0, 8809 }, /* &gvertneqq; */
    { 111, 0, 9462, 0 },
    { 115, 0, 9463, 0 },
    { 115, 0, 9464, 0 },
    { 59, 0, 0, 10683 }, /* &olcross; */
    { 112, 0, 9466, 0 },
    { 59, 0, 0, 8619 }, /* &larrlp; */
    { 59, 0, 0, 174 }, /* &circledR; */
    { 101, 0, 9469, 0 },
    { 108, 0, 9470, 0 },
    { 115, 0, 9471, 0 },
    { 59, 0, 0, 8871 }, /* &models; */
    { 59, 0, 0, 8217 }, /* &rsquor; */
    { 121, 0, 9474, 0 },
    { 59, 0, 0, 1062 }, /* &TScy; */
    { 97, 0, 9476, 0 },
    { 59, 0, 0, 10551 }, /* &rdca; */
    { 59, 0, 0, 10854 }, /* &sdote; */
    { 103, 0, 9479, 0 },
    { 101, 0, 9480, 0 },
    { 116, 0, 9481, 0 },
    { 59, 0, 0, 8982 }, /* &target; */
    { 100, 0, 9483, 0 },
    { 105, 0, 9484, 0 },
    { 108, 0, 9485, 0 },
    { 59, 0, 0, 355 }, /* &tcedil; */
    { 98, 0, 9487, 0 },
    { 100, 1, 9489, 0 },
    { 59, 0, 0, 8894 }, /* &angrtvb; */
    { 59, 0, 0, 10653 }, /* &angrtvbd; */
    { 105, 0, 9491, 0 },
    { 103, 0, 9492, 0 },
    { 104, 0, 9493, 0 },
    { 116, 0, 9494, 0 },
    { 65, 0, 9495, 0 },
    { 114, 0, 9496, 0 },
    { 114, 0, 9497, 0 },
    { 111, 0, 9498, 0 },
    { 119, 0, 9499, 0 },
    { 59, 0, 0, 10231 }, /* &LongLeftRightArrow; */
    { 113, 0, 9501, 0 },
    { 117, 0, 9502, 0 },
    { 97, 0, 9503, 0 },
    { 108, 0, 9504, 0 },
    { 59, 0, 0, 8783 }, /* &NotHumpEqual; */
    { 116, 0, 9506, 0 },
    { 114, 0, 9507, 0 },
    { 105, 0, 9508, 0 },
    { 59, 0, 0, 9661 }, /* &xdtri; */
    { 59, 0, 0, 10678 }, /* &omid; */
    { 59, 0, 0, 212 }, /* &Ocirc; */
    { 59, 0, 0, 10956 }, /* &vsupnE; */
    { 59, 0, 0, 1073 }, /* &bcy; */
    { 59, 0, 0, 8724 }, /* &plusdo; */
    { 98, 0, 9515, 0 },
    { 59, 0, 0, 10967 }, /* &suphsub; */
    { 59, 0, 0, 202 }, /* &Ecirc; */
    { 59, 0, 0, 9556 }, /* &boxDR; */
    { 111, 0, 9519, 0 },
    { 119, 0, 9520, 0 },
    { 110, 0, 9521, 0 },
    { 97, 0, 9522, 0 },
    { 114, 0, 9523, 0 },
    { 114, 0, 9524, 0 },
    { 111, 0, 9525, 0 },
    { 119, 0, 9526, 0 },
    { 59, 0, 0, 8597 }, /* &updownarrow; */
    { 59, 0, 0, 8624 }, /* &lsh; */
    { 110, 0, 9529, 0 },
    { 99, 0, 9530, 0 },
    { 115, 0, 9531, 0 },
    { 112, 0, 9532, 0 },
    { 59, 0, 0, 8200 }, /* &puncsp; */
    { 59, 0, 0, 331 }, /* &eng; */
    { 111, 0, 9535, 0 },
    { 110, 0, 9536, 0 },
    { 101, 0, 9537, 0 },
    { 110, 0, 9538, 0 },
    { 116, 0, 9539, 0 },
    { 105, 0, 9540, 0 },
    { 97, 0, 9541, 0 },
    { 108, 0, 9542, 0 },
    { 69, 0, 9543, 0 },
    { 59, 0, 0, 8519 }, /* &ExponentialE; */
    { 59, 0, 0, 8605 }, /* &nrarrw; */
    { 111, 0, 9546, 0 },
    { 119, 0, 9547, 0 },
    { 110, 0, 9548, 0 },
    { 59, 0, 0, 8637 }, /* &leftharpoondown; */
    { 83, 0, 9550, 0 },
    { 112, 0, 9551, 0 },
    { 97, 0, 9552, 0 },
    { 99, 0, 9553, 0 },
    { 101, 0, 9554, 0 },
    { 59, 0, 0, 8201 }, /* &ThinSpace; */
    { 101, 0, 9556, 0 },
    { 109, 0, 9557, 0 },
    { 101, 0, 9558, 0 },
    { 110, 0, 9559, 0 },
    { 116, 0, 9560, 0 },
    { 59, 0, 0, 8712 }, /* &Element; */
    { 108, 0, 9562, 0 },
    { 97, 0, 9563, 0 },
    { 110, 0, 9564, 0 },
    { 116, 0, 9565, 0 },
    { 69, 0, 9566, 0 },
    { 113, 0, 9567, 0 },
    { 117, 0, 9568, 0 },
    { 97, 0, 9569, 0 },
    { 108, 0, 9570, 0 },
    { 59, 0, 0, 10877 }, /* &LessSlantEqual; */
    { 59, 0, 0, 8793 }, /* &wedgeq; */
    { 112, 0, 9573, 0 },
    { 102, 0, 9574, 0 },
    { 59, 0, 0, 120138 }, /* &Sopf; */
    { 103, 0, 9576, 0 },
    { 59, 0, 0, 8824 }, /* &ntlg; */
    { 59, 0, 0, 8863 }, /* &minusb; */
    { 113, 0, 9579, 0 },
    { 117, 0, 9580, 0 },
    { 105, 0, 9581, 0 },
    { 108, 0, 9582, 0 },
    { 105, 0, 9583, 0 },
    { 98, 0, 9584, 0 },
    { 114, 0, 9585, 0 },
    { 105, 0, 9586, 0 },
    { 117, 0, 9587, 0 },
    { 109, 0, 9588, 0 },
    { 59, 0, 0, 10606 }, /* &UpEquilibrium; */
    { 59, 0, 0, 214 }, /* &Ouml; */
    { 59, 0, 0, 1082 }, /* &kcy; */
    { 114, 0, 9592, 0 },
    { 110, 0, 9593, 0 },
    { 59, 0, 0, 8990 }, /* &dlcorn; */
    { 103, 0, 9595, 0 },
    { 59, 0, 0, 10219 }, /* &Rang; */
    { 59, 0, 0, 1057 }, /* &Scy; */
    { 59, 0, 0, 10639 }, /* &lbrksld; */
    { 117, 0, 9599, 0 },
    { 111, 0, 9600, 0 },
    { 116, 0, 9601, 0 },
    { 101, 0, 9602, 0 },
    { 59, 0, 0, 8217 }, /* &CloseCurlyQuote; */
    { 97, 0, 9604, 0 },
    { 105, 0, 9605, 0 },
    { 108, 0, 9606, 0 },
    { 59, 0, 0, 10523 }, /* &lAtail; */
    { 59, 0, 0, 8706 }, /* &npart; */
    { 99, 0, 9609, 0 },
    { 121, 0, 9610, 0 },
    { 59, 0, 0, 1031 }, /* &YIcy; */
    { 121, 0, 9612, 0 },
    { 59, 0, 0, 1095 }, /* &chcy; */
    { 114, 0, 9614, 0 },
    { 59, 0, 0, 120089 }, /* &Vfr; */
    { 59, 0, 0, 120102 }, /* &ifr; */
    { 59, 0, 0, 10885 }, /* &lap; */
    { 107, 0, 9618, 0 },
    { 99, 0, 9619, 0 },
    { 121, 0, 9620, 0 },
    { 59, 0, 0, 1028 }, /* &Jukcy; */
    { 59, 0, 0, 230 }, /* &aelig; */
    { 114, 0, 9623, 0 },
    { 111, 0, 9624, 0 },
    { 107, 0, 9625, 0 },
    { 59, 0, 0, 272 }, /* &Dstrok; */
    { 59, 0, 0, 1101 }, /* &ecy; */
    { 112, 0, 9628, 0 },
    { 102, 0, 9629, 0 },
    { 59, 0, 0, 120132 }, /* &Mopf; */
    { 101, 0, 9631, 0 },
    { 100, 0, 9632, 0 },
    { 103, 0, 9633, 0 },
    { 101, 0, 9634, 0 },
    { 59, 0, 0, 8911 }, /* &curlywedge; */
    { 97, 0, 9636, 0 },
    { 114, 0, 9637, 0 },
    { 97, 0, 9638, 0 },
    { 108, 0, 9639, 0 },
    { 108, 0, 9640, 0 },
    { 101, 0, 9641, 0 },
    { 108, 0, 9642, 0 },
    { 59, 0, 0, 8741 }, /* &shortparallel; */
    { 59, 0, 0, 168 }, /* &DoubleDot; */
    { 101, 0, 9645, 0 },
    { 116, 0, 9646, 0 },
    { 59, 0, 0, 9140 }, /* &OverBracket; */
    { 117, 0, 9648, 0 },
    { 108, 0, 9649, 0 },
    { 108, 0, 9650, 0 },
    { 69, 0, 9651, 0 },
    { 113, 0, 9652, 0 },
    { 117, 0, 9653, 0 },
    { 97, 0, 9654, 0 },
    { 108, 0, 9655, 0 },
    { 59, 0, 0, 8775 }, /* &NotTildeFullEqual; */
    { 59, 0, 0, 8450 }, /* &Copf; */
    { 112, 0, 9658, 0 },
    { 102, 0, 9659, 0 },
    { 59, 0, 0, 120139 }, /* &Topf; */
    { 59, 0, 0, 10917 }, /* &gla; */
    { 59, 0, 0, 10916 }, /* &glj; */
    { 114, 0, 9663, 0 },
    { 114, 0, 9664, 0 },
    { 111, 0, 9665, 0 },
    { 119, 0, 9666, 0 },
    { 59, 0, 0, 8659 }, /* &Downarrow; */
    { 59, 0, 0, 125 }, /* &rbrace; */
    { 59, 0, 0, 38 }, /* &amp; */
    { 111, 0, 9670, 0 },
    { 116, 0, 9671, 0 },
    { 59, 0, 0, 10903 }, /* &elsdot; */
    { 114, 0, 9673, 0 },
    { 59, 0, 0, 10629 }, /* &lopar; */
    { 117, 0, 9675, 0 },
    { 115, 0, 9676, 0 },
    { 59, 0, 0, 8853 }, /* &oplus; */
    { 99, 0, 9678, 0 },
    { 121, 0, 9679, 0 },
    { 59, 0, 0, 1034 }, /* &NJcy; */
    { 114, 0, 9681, 0 },
    { 114, 0, 9682, 0 },
    { 59, 0, 0, 10233 }, /* &xrArr; */
    { 115, 0, 9684, 0 },
    { 117, 0, 9685, 0 },
    { 98, 0, 9686, 0 },
    { 59, 0, 0, 10184 }, /* &bsolhsub; */
    { 59, 0, 0, 10924 }, /* &smtes; */
    { 59, 0, 0, 210 }, /* &Ograve; */
    { 59, 0, 0, 9576 }, /* &boxhU; */
    { 109, 0, 9691, 0 },
    { 59, 0, 0, 8770 }, /* &eqsim; */
    { 59, 0, 0, 42 }, /* &ast; */
    { 111, 0, 9694, 0 },
    { 116, 0, 9695, 0 },
    { 59, 0, 0, 8784 }, /* &nedot; */
    { 111, 0, 9697, 0 },
    { 116, 0, 9698, 0 },
    { 59, 0, 0, 266 }, /* &Cdot; */
    { 111, 0, 9700, 0 },
    { 111, 0, 9701, 0 },
    { 114, 0, 9702, 0 },
    { 59, 0, 0, 8970 }, /* &lfloor; */
    { 59, 0, 0, 10982 }, /* &Vdashl; */
    { 105, 0, 9705, 0 },
    { 103, 0, 9706, 0 },
    { 104, 0, 9707, 0 },
    { 116, 0, 9708, 0 },
    { 97, 0, 9709, 0 },
    { 114, 0, 9710, 0 },
    { 114, 0, 9711, 0 },
    { 111, 0, 9712, 0 },
    { 119, 0, 9713, 0 },
    { 59, 0, 0, 10231 }, /* &longleftrightarrow; */
    { 111, 0, 9715, 0 },
    { 110, 0, 9716, 0 },
    { 105, 0, 9717, 0 },
    { 110, 0, 9718, 0 },
    { 116, 0, 9719, 0 },
    { 59, 0, 0, 8755 }, /* &awconint; */
    { 108, 0, 9721, 0 },
    { 101, 0, 9722, 0 },
    { 68, 0, 9723, 0 },
    { 101, 0, 9724, 0 },
    { 108, 0, 9725, 0 },
    { 97, 0, 9726, 0 },
    { 121, 0, 9727, 0 },
    { 101, 0, 9728, 0 },
    { 100, 0, 9729, 0 },
    { 59, 0, 0, 10740 }, /* &RuleDelayed; */
    { 113, 0, 9731, 0 },
    { 113, 0, 9732, 0 },
    { 59, 0, 0, 10933 }, /* &precneqq; */
    { 59, 0, 0, 9560 }, /* &boxuR; */
    { 114, 0, 9735, 0 },
    { 59, 0, 0, 120092 }, /* &Yfr; */
    { 114, 0, 9737, 0 },
    { 111, 0, 9738, 0 },
    { 100, 0, 9739, 0 },
    { 59, 0, 0, 10812 }, /* &iprod; */
    { 114, 0, 9741, 0 },
    { 59, 0, 0, 120086 }, /* &Sfr; */
    { 116, 0, 9743, 0 },
    { 114, 0, 9744, 0 },
    { 105, 0, 9745, 0 },
    { 59, 0, 0, 8882 }, /* &vltri; */
    { 99, 0, 9747, 0 },
    { 114, 0, 9748, 0 },
    { 59, 0, 0, 274 }, /* &Emacr; */
    { 113, 0, 9750, 0 },
    { 59, 0, 0, 10928 }, /* &succeq; */
    { 59, 0, 0, 10668 }, /* &angmsdae; */
    { 59, 0, 0, 168 }, /* &die; */
    { 59, 0, 0, 10693 }, /* &bsolb; */
    { 111, 0, 9755, 0 },
    { 116, 0, 9756, 0 },
    { 59, 0, 0, 8919 }, /* &gtrdot; */
    { 111, 0, 9758, 0 },
    { 109, 0, 9759, 0 },
    { 59, 0, 0, 8869 }, /* &bottom; */
    { 114, 0, 9761, 0 },
    { 107, 0, 9762, 0 },
    { 59, 0, 0, 10215 }, /* &robrk; */
    { 59, 0, 0, 8810 }, /* &nLtv; */
    { 108, 0, 9765, 0 },
    { 97, 0, 9766, 0 },
    { 110, 0, 9767, 0 },
    { 116, 0, 9768, 0 },
    { 59, 0, 0, 10877 }, /* &nleqslant; */
    { 107, 0, 9770, 0 },
    { 59, 0, 0, 10531 }, /* &nwarhk; */
    { 59, 0, 0, 9508 }, /* &boxvl; */
    { 114, 0, 9773, 0 },
    { 59, 0, 0, 119983 }, /* &Tscr; */
    { 115, 0, 9775, 0 },
    { 104, 0, 9776, 0 },
    { 116, 0, 9777, 0 },
    { 59, 0, 0, 10623 }, /* &dfisht; */
    { 114, 0, 9779, 0 },
    { 59, 0, 0, 8595 }, /* &darr; */
    { 118, 0, 9781, 0 },
    { 101, 0, 9782, 0 },
    { 59, 0, 0, 365 }, /* &ubreve; */
    { 97, 0, 9784, 0 },
    { 112, 0, 9785, 0 },
    { 59, 0, 0, 8781 }, /* &CupCap; */
    { 107, 0, 9787, 0 },
    { 83, 0, 9788, 0 },
    { 112, 0, 9789, 0 },
    { 97, 0, 9790, 0 },
    { 99, 0, 9791, 0 },
    { 101, 0, 9792, 0 },
    { 59, 0, 0, 8203 }, /* &NegativeThickSpace; */
    { 59, 0, 0, 249 }, /* &ugrave; */
    { 99, 0, 9795, 0 },
    { 117, 0, 9796, 0 },
    { 116, 0, 9797, 0 },
    { 101, 0, 9798, 0 },
    { 59, 0, 0, 347 }, /* &sacute; */
    { 101, 0, 9800, 0 },
    { 115, 0, 9801, 0 },
    { 115, 0, 9802, 0 },
    { 59, 0, 0, 8825 }, /* &NotGreaterLess; */
    { 116, 0, 9804, 0 },
    { 59, 0, 0, 10643 }, /* &lparlt; */
    { 108, 0, 9806, 0 },
    { 105, 0, 9807, 0 },
    { 103, 0, 9808, 0 },
    { 59, 0, 0, 64257 }, /* &filig; */
    { 59, 0, 0, 9557 }, /* &boxdL; */
    { 114, 0, 9811, 0 },
    { 105, 0, 9812, 0 },
    { 59, 0, 0, 8895 }, /* &lrtri; */
    { 97, 0, 9814, 0 },
    { 110, 0, 9815, 0 },
    { 103, 0, 9816, 0 },
    { 108, 0, 9817, 0 },
    { 101, 0, 9818, 0 },
    { 59, 0, 0, 10662 }, /* &dwangle; */
    { 114, 0, 9820, 0 },
    { 105, 0, 9821, 0 },
    { 109, 0, 9822, 0 },
    { 101, 0, 9823, 0 },
    { 59, 0, 0, 8245 }, /* &bprime; */
    { 59, 0, 0, 8765 }, /* &race; */
    { 103, 0, 9826, 0 },
    { 97, 0, 9827, 0 },
    { 59, 0, 0, 969 }, /* &omega; */
    { 59, 0, 0, 10950 }, /* &supseteqq; */
    { 59, 0, 0, 10936 }, /* &scap; */
    { 114, 0, 9831, 0 },
    { 59, 0, 0, 120083 }, /* &Pfr; */
    { 105, 0, 9833, 0 },
    { 103, 0, 9834, 0 },
    { 104, 0, 9835, 0 },
    { 116, 0, 9836, 0 },
    { 97, 0, 9837, 0 },
    { 114, 0, 9838, 0 },
    { 114, 0, 9839, 0 },
    { 111, 0, 9840, 0 },
    { 119, 0, 9841, 0 },
    { 59, 0, 0, 8667 }, /* &Rrightarrow; */
    { 112, 0, 9843, 0 },
    { 59, 0, 0, 8898 }, /* &xcap; */
    { 114, 0, 9845, 0 },
    { 59, 0, 0, 120117 }, /* &xfr; */
    { 105, 0, 9847, 0 },
    { 109, 0, 9848, 0 },
    { 59, 0, 0, 8764 }, /* &nvsim; */
    { 97, 0, 9850, 0 },
    { 114, 0, 9851, 0 },
    { 59, 0, 0, 10584 }, /* &LeftUpVectorBar; */
    { 59, 0, 0, 1084 }, /* &mcy; */
};
#endif
