/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* libe-book
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libebook.sourceforge.net
 */

#include <boost/shared_ptr.hpp>

#include <libwpd/WPXDocumentInterface.h>
#include <libwpd/WPXPropertyList.h>
#include <libwpd/WPXPropertyListVector.h>
#include <libwpd/WPXString.h>

#include <libwpd-stream/libwpd-stream.h>

#include <libe-book/QIOODocument.h>

#include "libebook_utils.h"

#include "WPXZipStream.h"

using boost::shared_ptr;

namespace libebook
{

namespace
{

bool parseText(WPXInputStream *const input, WPXDocumentInterface *const document)
{
  WPXPropertyList dummy;

  if (input->atEOS()) // empty document, nothing will be produced
    return false;

  document->startDocument();
  document->setDocumentMetaData(dummy);
  document->openPageSpan(dummy);

  // output paragraphs
  WPXString text;
  bool ignoreNextLineBreak = false;
  while (!input->atEOS())
  {
    const uint8_t c = readU8(input);
    if (('\n' == c) || ('\r' == c))
    {
      if (ignoreNextLineBreak)
        ignoreNextLineBreak = false;
      else
      {
        document->openParagraph(dummy, WPXPropertyListVector());
        document->openSpan(dummy);
        document->insertText(text);
        document->closeSpan();
        document->closeParagraph();
        text.clear();
        ignoreNextLineBreak = true;
      }
    }
    else
    {
      text.append(c);
      ignoreNextLineBreak = false;
    }
  }

  document->closePageSpan();
  document->endDocument();

  return true;
}

}

bool QIOODocument::isSupported(WPXInputStream *const input) try
{
  input->seek(0, WPX_SEEK_SET);
  if (!WPXZipStream::isZipFile(input))
    return false;

  input->seek(0, WPX_SEEK_SET);
  shared_ptr<WPXInputStream> reader(WPXZipStream::getSubstream(input, "reader/MobileLibrary.class"));
  shared_ptr<WPXInputStream> data(WPXZipStream::getSubstream(input, "data"));

  return reader.get() && data.get();
}
catch (...)
{
  return false;
}

bool QIOODocument::parse(WPXInputStream *const input, WPXDocumentInterface *const document) try
{
  if (!QIOODocument::isSupported(input))
    return false;

  input->seek(0, WPX_SEEK_SET);
  shared_ptr<WPXInputStream> data(WPXZipStream::getSubstream(input, "data"));
  if (!data.get())
    return false;

  return parseText(data.get(), document);
}
catch (...)
{
  return false;
}

} // namespace libebook

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
