/*
 * Copyright (C) 2021 Andreas Steffen, strongSec GmbH
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include "test_suite.h"

#include <plugins/plugin_feature.h>

typedef struct rsa_oaep_test_t rsa_oaep_test_t;

struct rsa_oaep_test_t {
	chunk_t msg;
	chunk_t label;
	chunk_t ct;
};

/**
 * Private RSA key to load
 */
static chunk_t rsa_2048_key = chunk_from_chars( /* PKCS#8 format */
		0x30,0x82,0x04,0xbc,0x02,0x01,0x00,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,
		0x0d,0x01,0x01,0x01,0x05,0x00,0x04,0x82,0x04,0xa6,0x30,0x82,0x04,0xa2,0x02,0x01,
		0x00,0x02,0x82,0x01,0x01,0x00,0xbd,0x31,0xc7,0xa0,0x26,0x91,0xd2,0xd9,0x58,0x7e,
		0xf6,0xa9,0x46,0xff,0x78,0x85,0x44,0xcc,0xad,0xd4,0xb2,0x98,0x8a,0xd6,0x20,0x86,
		0x79,0x2a,0x6b,0xf9,0x6c,0x86,0x16,0xb4,0xad,0x13,0x31,0x7d,0x22,0x70,0xb9,0x01,
		0xd0,0xfc,0xd1,0xd8,0x80,0xcb,0x8f,0x52,0xfb,0x87,0x30,0x4a,0x52,0x58,0xc1,0x1b,
		0x38,0xdf,0xea,0xe8,0xdf,0x67,0x0a,0xee,0xe7,0xea,0x1d,0x0d,0x9d,0xf8,0xe0,0x0e,
		0x80,0x84,0x7e,0x41,0xe5,0x98,0x9e,0xd4,0x02,0xd4,0x4e,0x78,0xb3,0x0f,0xef,0x17,
		0xb5,0x67,0x1d,0x3a,0xdb,0xf8,0x68,0x5e,0x4d,0xc2,0x04,0x49,0x9e,0xcd,0x18,0x63,
		0xe1,0xd5,0xaf,0xf2,0x8a,0x7c,0xf6,0x6e,0xad,0xf3,0x1f,0xec,0x92,0x36,0xc1,0x20,
		0xad,0xd1,0x34,0x51,0x52,0x2c,0x64,0x7c,0x98,0x32,0xa6,0x72,0xcd,0x64,0xd3,0x28,
		0xc1,0xc3,0x22,0x18,0x3f,0x46,0x61,0xd0,0x9b,0xda,0x60,0xb8,0xdd,0x5f,0x03,0x28,
		0xda,0x54,0x20,0x82,0x14,0x24,0xaf,0xda,0xbb,0x1a,0x80,0xc5,0xd1,0x27,0x63,0xa1,
		0xb0,0x23,0x8c,0xd8,0x9d,0x07,0x42,0xbf,0xc5,0x0b,0x6a,0x2f,0xcb,0x70,0x1d,0x82,
		0x42,0x18,0xf9,0x82,0x6f,0x4f,0x78,0xa2,0x3a,0x2b,0x5a,0xa4,0x2a,0xce,0x7f,0x17,
		0x53,0x76,0xfb,0x6c,0xbd,0xb2,0xba,0xd2,0x93,0xba,0x58,0x3d,0x4d,0x31,0xc6,0xb8,
		0xf9,0x02,0x9e,0x46,0xb1,0x36,0x89,0x24,0x98,0x55,0xf5,0x05,0x75,0x6e,0x00,0xe2,
		0x25,0xa6,0xa4,0x5a,0x18,0x76,0x9b,0xd8,0xd2,0xb3,0xa4,0xac,0xb9,0xf1,0xc2,0x3d,
		0x3e,0x51,0x88,0x25,0x61,0xe5,0x02,0x03,0x01,0x00,0x01,0x02,0x82,0x01,0x00,0x07,
		0x47,0xd5,0x20,0xca,0x9b,0x2d,0xfc,0x03,0x35,0xcf,0x94,0x30,0x11,0x40,0xb8,0x10,
		0x2f,0xb8,0x38,0xc9,0x2e,0xa9,0x99,0x17,0xe3,0x54,0xe2,0x57,0x09,0x60,0x2d,0xc0,
		0xcd,0xef,0x23,0x1f,0xf2,0x6a,0xd2,0xdf,0xab,0xbc,0x39,0x1e,0x72,0x30,0x72,0xa4,
		0xef,0x52,0x92,0x9f,0x3f,0x1c,0xf0,0x21,0x6e,0x44,0xc0,0x09,0x30,0x30,0xf8,0xac,
		0x29,0x09,0x6f,0xae,0xf2,0x89,0x33,0xbc,0x9b,0x92,0x42,0x87,0x01,0x4e,0x2e,0x93,
		0x5e,0x21,0x45,0xbe,0x47,0x52,0xcc,0xb8,0x2c,0xb7,0x64,0x62,0x71,0x93,0x69,0x79,
		0xca,0xb7,0x3f,0x52,0x1e,0xeb,0x27,0x85,0x5a,0xc2,0xa4,0x10,0x83,0x4a,0xe5,0x08,
		0x2e,0xb6,0x2e,0xf6,0xf9,0x89,0x9e,0xe5,0xcb,0xb6,0x24,0xa8,0xd8,0xc0,0x4a,0xb7,
		0xed,0xcf,0x78,0xfe,0x13,0x5d,0x0f,0x80,0x8f,0x63,0xb5,0xde,0x6e,0x07,0x17,0x79,
		0xbf,0x1d,0x9e,0xaa,0xf8,0xd3,0xef,0x96,0x54,0x86,0xf2,0xc5,0x26,0x25,0xe6,0xe5,
		0x7a,0x65,0x1c,0x7b,0xbd,0x01,0x97,0xc9,0x9c,0x05,0x7b,0x38,0x60,0xa3,0x26,0x0c,
		0x86,0x4a,0x11,0xb3,0xcf,0x22,0xba,0x44,0xde,0x3a,0x3b,0x3f,0x31,0x17,0xe6,0x4e,
		0xbf,0x9a,0x91,0xf6,0x26,0xeb,0xb3,0xf3,0xc2,0x6d,0x1b,0x0b,0xb8,0x07,0x53,0x46,
		0x8d,0x4b,0xf6,0xa9,0x97,0xb2,0x83,0x35,0xf2,0x0e,0xd8,0xc3,0x84,0xd7,0x55,0x83,
		0x60,0xa9,0xae,0x97,0x55,0x66,0x1f,0xa3,0x74,0x98,0x46,0xb1,0x74,0xe6,0xbf,0x33,
		0x0e,0x52,0xd0,0x99,0x41,0xf6,0xcc,0xd7,0xde,0x50,0x04,0x77,0x9f,0x9b,0xb9,0x02,
		0x81,0x81,0x00,0xfe,0x3d,0x23,0x8d,0x37,0x6a,0x38,0x56,0xd9,0x59,0x2f,0xd4,0x96,
		0x1c,0x17,0xa7,0xfb,0x0f,0xa3,0x9d,0x44,0x2e,0x9d,0xf2,0x2e,0x7e,0xda,0x11,0x5c,
		0x64,0x8c,0x5c,0xb7,0x38,0xdf,0x89,0xea,0x92,0xd8,0x37,0x85,0x58,0xd0,0xd9,0xc1,
		0xcf,0x3d,0x8b,0xff,0x74,0x0d,0x37,0x43,0xa0,0x14,0x0a,0x6a,0x75,0x23,0xe9,0x17,
		0x6c,0x29,0x53,0xb2,0xc1,0x1d,0x7f,0x44,0x69,0x1c,0xf8,0xbe,0xc2,0xb3,0x4f,0xb5,
		0x9c,0x64,0x95,0x35,0x98,0x33,0x19,0x48,0x81,0x18,0xc0,0x31,0x18,0xae,0x74,0x45,
		0xe7,0x0d,0xf4,0x66,0xce,0x98,0x67,0x12,0x60,0xf0,0x5d,0x50,0x68,0xce,0x77,0x7d,
		0xa9,0x7e,0x2b,0xa8,0x47,0x79,0x49,0x05,0x00,0xce,0xb4,0xff,0x7f,0x62,0xce,0xca,
		0x9a,0x14,0x7f,0x02,0x81,0x81,0x00,0xbe,0x81,0x4a,0xf2,0x6b,0xf3,0xc2,0x71,0x15,
		0xbc,0x75,0x95,0xd0,0xd1,0x2a,0x1a,0xf7,0x71,0x3f,0xec,0xa8,0x77,0xae,0x6e,0x6d,
		0x85,0x88,0x25,0xed,0xa3,0xd1,0x92,0x70,0x47,0x74,0xf4,0x2e,0x1a,0xe1,0xb6,0x7f,
		0x17,0x39,0xe8,0xc4,0x5a,0x04,0x76,0x38,0x80,0x70,0xfa,0x54,0x5b,0x7c,0xc9,0x6b,
		0x8a,0x44,0x00,0x3b,0xc8,0xd0,0x7f,0xde,0x73,0xf5,0xbc,0xdd,0x18,0x26,0x83,0x95,
		0xdc,0xc2,0x84,0x2b,0x50,0x23,0xff,0x14,0xb3,0x52,0x38,0x26,0x07,0xb7,0x17,0x9a,
		0x5e,0xa3,0x30,0x84,0x0f,0xc4,0x50,0x45,0xd8,0x6e,0x0a,0xcc,0x4f,0xeb,0x13,0x5f,
		0x0a,0xb9,0xb7,0xce,0x6b,0x8e,0xb5,0xf5,0xa5,0x75,0xf2,0xa2,0xa8,0xde,0x29,0x9c,
		0x0d,0x3f,0x50,0x6d,0xf4,0x87,0x9b,0x02,0x81,0x80,0x61,0xbb,0x4c,0xf4,0x57,0x2e,
		0x0c,0xf4,0x4d,0xde,0xfc,0x05,0x03,0xa3,0x4e,0xf0,0xb3,0x11,0x42,0xfb,0x0c,0x9f,
		0xc2,0xf5,0xbb,0x2b,0xe6,0x56,0x30,0x27,0x32,0xd1,0xdd,0x59,0x0e,0x8d,0x6c,0xd0,
		0x00,0x2f,0xef,0xa1,0xc1,0xbf,0x43,0xc2,0x8f,0xcd,0x46,0x68,0xed,0x3f,0xa6,0x4e,
		0x5e,0x05,0xd7,0x00,0xaa,0xca,0x4a,0xc7,0xb8,0x67,0x45,0x5a,0xf9,0x24,0xd8,0x3e,
		0x73,0xf5,0xa7,0x17,0x1f,0x68,0xef,0xce,0x6b,0xd7,0xf3,0xdf,0x4b,0x1d,0x58,0x02,
		0xd0,0x69,0xe9,0x35,0xac,0xb2,0x63,0xc0,0xbb,0x5b,0x8d,0x3d,0xb3,0xd6,0x43,0x79,
		0x62,0x4c,0xec,0xc1,0x2d,0x45,0x3f,0xd8,0x2c,0xf9,0x49,0x5a,0x18,0x64,0x9a,0x78,
		0xc1,0xfd,0x50,0x0e,0x3f,0x04,0x71,0x5b,0xa3,0xaf,0x02,0x81,0x80,0x66,0x49,0xdb,
		0xe9,0x20,0xa7,0xa8,0xce,0xb9,0x82,0x10,0x8c,0xd9,0xab,0x85,0x75,0x27,0x49,0x8a,
		0xae,0x52,0xa2,0xb8,0x69,0x18,0xab,0x6b,0x38,0x1a,0x4a,0x2c,0xbb,0xf9,0x47,0x94,
		0xf7,0x8b,0x4d,0xe9,0xc5,0xe8,0xa5,0x9a,0xf5,0x6d,0x80,0x7c,0x06,0xd2,0x3d,0xbc,
		0xeb,0xdc,0xa1,0xa6,0xf6,0x2b,0xa4,0xf8,0xf0,0x3b,0x29,0x8f,0x32,0x29,0x4a,0x90,
		0x36,0xd1,0x6c,0x27,0x39,0xde,0x1c,0xc3,0xe0,0xdd,0xb4,0xd1,0x2f,0x24,0x79,0x48,
		0x7b,0x57,0x88,0x2a,0xfa,0x24,0x6c,0x0e,0x29,0x7d,0xaf,0x4d,0x94,0x60,0x7c,0x5e,
		0x51,0xb3,0xf4,0x71,0x5f,0x5c,0x57,0xd0,0xf2,0x6d,0x86,0x79,0x01,0x20,0x9d,0xb0,
		0x20,0x10,0xd0,0xa5,0x41,0x28,0x46,0x40,0xa9,0xb9,0xf0,0x63,0x63,0x02,0x81,0x80,
		0x39,0xec,0x60,0x6f,0x5c,0x9f,0x1f,0x33,0xd9,0x64,0x60,0x6e,0x12,0x8e,0x9e,0x90,
		0xcf,0xf9,0x32,0x77,0x2c,0x57,0x15,0xaa,0x75,0x98,0x16,0xd5,0x04,0xef,0xe2,0xc4,
		0x62,0x06,0xf3,0x37,0x9a,0xa6,0x1d,0x31,0xf8,0x8e,0x72,0x6b,0x50,0xdd,0x80,0xb8,
		0xce,0x85,0x58,0xa0,0xd7,0x40,0x85,0x25,0xf5,0x8c,0x33,0x99,0xce,0x32,0xe2,0x39,
		0xd7,0x09,0x41,0xf1,0x2d,0x21,0xe7,0x76,0x1c,0x04,0x3e,0x0b,0x02,0x22,0x0d,0x25,
		0xff,0xd0,0x35,0x71,0xc7,0xae,0x5b,0xbd,0x59,0xc3,0x1f,0x48,0x12,0x06,0x76,0xf9,
		0xd7,0xc3,0x70,0xa7,0x49,0xd8,0x49,0x1b,0xdf,0xce,0xa0,0x16,0x62,0xa1,0x36,0xc8,
		0x5d,0x34,0x9b,0x6c,0x71,0x0f,0xc6,0xf6,0xeb,0xf8,0xa7,0xa7,0x9b,0x62,0xad,0xc3);

/**
 * RSA OAEP SHA1 decryption test vectors from https://boringssl.googlesource.com/
 */
static rsa_oaep_test_t rsa_2048_oaep_sha1_tests[] = {
	{ chunk_from_chars(                                                          /* 2 */
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00),
	  chunk_from_chars(),
	  chunk_from_chars(
	  	0x87,0x73,0xfd,0xde,0x3f,0x14,0x8e,0xc3,0x3d,0x4f,0xac,0xdf,0xba,0x81,0x2e,0x44,
	  	0x00,0x1d,0x90,0x86,0x1c,0x04,0x7a,0x9e,0xef,0xe6,0xc6,0x54,0x02,0x12,0x79,0x5b,
	  	0x50,0xdd,0x18,0x6b,0xb7,0x71,0x2b,0x84,0x48,0x83,0x0f,0xc1,0x79,0x9e,0x05,0x43,
	  	0xb7,0x09,0x3e,0x47,0x70,0xf3,0x01,0xe9,0x21,0x69,0x22,0xe8,0x02,0x80,0xe9,0xb5,
	  	0x05,0x82,0xc6,0x69,0xe3,0xa9,0xa4,0xfe,0xb0,0x8e,0xc6,0xf5,0xb2,0x42,0x8d,0xc7,
	  	0xef,0xbc,0xf6,0xfe,0x49,0xfe,0xb6,0x1f,0xd2,0x79,0x5a,0xf1,0xc8,0xb2,0x3a,0xcb,
	  	0x60,0x58,0x12,0xc9,0xd7,0x58,0x6e,0x4f,0x48,0xe4,0xd7,0x58,0xf0,0x69,0x5b,0x68,
	  	0x8d,0x1b,0x4b,0x56,0x30,0x67,0x85,0x2b,0x1c,0xe9,0xdb,0x5d,0xc8,0x2a,0xba,0x29,
	  	0xab,0x41,0x61,0x78,0xf6,0x16,0x42,0x6e,0x5d,0x0c,0xaf,0x3f,0x99,0x03,0xde,0x2a,
	  	0x5b,0xee,0xb9,0x70,0xaf,0x72,0x81,0xe7,0xcd,0xac,0xaa,0xa0,0x06,0x62,0xae,0x36,
	  	0xfb,0x03,0x65,0xc2,0x0c,0x65,0x3b,0x67,0x0b,0x3e,0x7b,0xe4,0xb0,0x68,0x09,0x32,
	  	0xf3,0x25,0x4d,0x48,0xec,0x35,0x05,0xca,0x15,0x47,0xd1,0xbe,0x27,0xd2,0x41,0x4a,
	  	0x35,0x39,0xf4,0x05,0xdf,0x91,0x05,0x7f,0x4f,0x97,0xaa,0x77,0x96,0xd7,0xcc,0xb6,
	  	0x9e,0xd5,0x53,0x44,0x71,0x31,0xfb,0xe7,0x96,0x11,0x75,0x6b,0x7e,0x56,0x4b,0xba,
	  	0x7d,0x76,0xce,0x92,0x56,0x16,0x84,0x46,0xb3,0x44,0x98,0xa2,0xe9,0x1d,0x9a,0x95,
	  	0x5a,0x8a,0xa9,0x8a,0x71,0xa4,0x3d,0x65,0xee,0x60,0xfc,0xea,0x39,0xd8,0x11,0xfe)
	},
	{ chunk_from_chars(                                                          /* 3 */
		0x54,0x65,0x73,0x74),
	  chunk_from_chars(),
	  chunk_from_chars(
	  	0x50,0x4b,0x4e,0x0d,0x94,0xd8,0x00,0xa9,0xda,0x92,0xbe,0x8f,0xa8,0x4f,0xc3,0x45,
	  	0xf7,0xab,0x49,0x21,0xec,0x53,0xc0,0x73,0xc7,0x95,0x92,0xb3,0x47,0xa6,0x92,0xcc,
	  	0xe2,0x45,0x01,0x35,0xe9,0x98,0xe0,0x7b,0xb7,0x5e,0xd7,0xee,0x44,0xf0,0x47,0xf7,
	  	0x5a,0x07,0xe3,0x7b,0x9c,0x91,0x2d,0x3d,0x52,0x9b,0x62,0xbe,0xf1,0x44,0x95,0x2d,
	  	0x5b,0x20,0x22,0x46,0xad,0x32,0xc5,0xd3,0x4e,0xc8,0x55,0x9c,0x41,0x63,0xe2,0x14,
	  	0x5f,0xa5,0xfd,0x4d,0x56,0xde,0x28,0xcb,0xfe,0x7e,0x53,0x7d,0x57,0x32,0x30,0x59,
	  	0x45,0xc9,0x37,0x0b,0xbc,0x6d,0xcd,0x72,0xb3,0x29,0x6f,0xd5,0x4a,0x26,0x60,0x36,
	  	0x57,0x15,0xf7,0xd9,0x11,0x09,0x9c,0x96,0xdf,0xe6,0xd1,0x14,0xee,0xc7,0xb4,0x42,
	  	0x5c,0xb6,0x57,0x01,0xaa,0x9e,0x08,0xd9,0x9c,0x7a,0xc2,0x01,0x79,0x65,0x6b,0xb6,
	  	0x78,0x55,0x4d,0x74,0xba,0xca,0x45,0x01,0xb5,0xd0,0xb3,0x66,0xc9,0x72,0x24,0xbf,
	  	0x0c,0x87,0x34,0xe0,0x0f,0xf2,0xaf,0x8b,0x91,0x6a,0xef,0x8b,0x83,0x14,0x2d,0x5e,
	  	0xc1,0x42,0x49,0x3e,0x0c,0xc4,0xc5,0x73,0x26,0x51,0x5a,0x50,0xa3,0x1e,0xc2,0xc6,
	  	0x70,0x35,0x12,0x03,0x46,0x42,0xe5,0x78,0x96,0x49,0x77,0x3d,0x2f,0x83,0x12,0x56,
	  	0x1b,0xf2,0xf2,0xf6,0x00,0x0c,0xeb,0xdb,0xf7,0xc7,0xa9,0x5f,0xbf,0xd0,0xf7,0x21,
	  	0xc3,0x1b,0xa3,0x92,0xac,0xea,0x37,0x56,0x8a,0x24,0xf1,0x76,0x76,0x7f,0x71,0xb3,
	  	0x1f,0x40,0x75,0x73,0xde,0xef,0x95,0xc3,0x05,0xe0,0xe0,0x63,0xdb,0x0e,0x91,0x68)
	},
	{ chunk_from_chars(                                                          /* 4 */
		0x31,0x32,0x33,0x34,0x30,0x30),
	  chunk_from_chars(),
	  chunk_from_chars(
	  	0x25,0xa9,0x18,0xa0,0xc7,0x98,0x5a,0x72,0xf1,0x8f,0x6b,0x62,0x16,0xd0,0xe2,0xd1,
	  	0x9a,0x9b,0x42,0xa4,0xd5,0xb6,0x80,0xe9,0x95,0xc3,0xc5,0xc5,0x84,0x96,0x6e,0x29,
	  	0xc6,0x17,0x98,0x77,0xcf,0xd0,0x31,0x7f,0x9e,0x4d,0xcd,0xc8,0x7f,0x76,0x71,0x49,
	  	0x94,0xe4,0x40,0x7c,0xd8,0x68,0x84,0x23,0x50,0xe2,0x8f,0x32,0x8b,0x3f,0x33,0x74,
	  	0x6a,0x89,0x7e,0xa9,0x99,0x55,0x75,0xf3,0x28,0xbe,0x7f,0x99,0x12,0x36,0x94,0x82,
	  	0xae,0x1a,0x66,0xff,0x5b,0x7a,0x1c,0x8a,0xc8,0xfd,0x4e,0x57,0x71,0xe5,0x80,0x77,
	  	0xb5,0x4a,0x55,0xdb,0x6c,0x68,0x49,0xff,0x28,0x8d,0x50,0xde,0x0e,0xe1,0xe4,0x5b,
	  	0xfd,0xf8,0x1f,0xfa,0xed,0x16,0x97,0x1f,0x19,0xab,0xae,0x66,0x03,0x38,0xa3,0xa9,
	  	0x24,0x0a,0x0b,0x73,0x29,0x84,0xfa,0xdc,0xd2,0x8a,0x85,0xc6,0x80,0xf0,0xda,0xc8,
	  	0xb7,0x33,0x06,0x48,0x1a,0x09,0x6f,0x70,0x07,0x06,0xe9,0x1f,0x7c,0x10,0x0c,0xe6,
	  	0x4e,0x1c,0x6b,0xba,0xbd,0x41,0x98,0x07,0xdc,0x16,0x7f,0x1f,0x92,0x4e,0xdc,0x6f,
	  	0x44,0xa6,0x9b,0x75,0xa7,0xc7,0x29,0x89,0xb2,0x5c,0xe3,0x54,0x80,0x43,0x3b,0xc4,
	  	0x56,0xc6,0xaf,0xa9,0x3a,0x0e,0x9b,0x41,0x56,0xd7,0x58,0x63,0xf9,0x44,0x0a,0xe4,
	  	0x42,0xee,0xfd,0x21,0x3b,0x8a,0x3f,0x53,0xa0,0x65,0xda,0xa8,0xcd,0x02,0x06,0x88,
	  	0x6d,0x15,0x44,0xc3,0xbf,0xec,0xaf,0x62,0x65,0x9c,0xfa,0x59,0x1d,0xa5,0xe9,0x20,
	  	0xd7,0xd4,0x37,0x0a,0x9a,0xab,0xf7,0xf9,0x98,0x3f,0xad,0xe1,0x01,0xda,0x2d,0xba)
	},
	{ chunk_from_chars(                                                          /* 8 */
		0x31,0x32,0x33,0x34,0x30,0x30),
	  chunk_from_chars(
	  	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00),
	  chunk_from_chars(
	  	0x4d,0x0c,0xb9,0xe2,0x3e,0xa2,0xf2,0x5c,0xd9,0x3f,0x6a,0x4a,0x5b,0x91,0x89,0x15,
	  	0x61,0x78,0x57,0x22,0x14,0xc2,0xbb,0xc7,0x73,0x15,0x8e,0xd8,0x2b,0x83,0xcd,0xfe,
	  	0x7a,0xc1,0xe9,0x6d,0x66,0x2a,0xfd,0x46,0xa9,0x22,0x5b,0x9e,0x8d,0x1d,0xed,0x27,
	  	0x4e,0x00,0x04,0x8c,0x05,0x8d,0x4c,0xb6,0x6c,0xcc,0xf4,0xeb,0x6c,0x4e,0xba,0xf3,
	  	0x0b,0x97,0xe3,0xff,0x17,0xa2,0x12,0x23,0xcd,0xed,0x87,0x08,0x52,0x4b,0xf7,0x5e,
	  	0x3d,0xec,0x07,0x58,0x33,0xdc,0x95,0x3d,0xfd,0x9a,0x13,0x41,0x07,0x2f,0x14,0x85,
	  	0xc6,0x00,0x18,0xf9,0x2b,0xd4,0x1c,0xed,0xd0,0x51,0x00,0x85,0xd8,0x18,0xa5,0xcb,
	  	0x4d,0xf3,0x2c,0x44,0x72,0x94,0xf4,0x24,0x53,0x85,0xe1,0xe5,0xc4,0x62,0x7c,0x41,
	  	0x45,0x0c,0x1d,0xfd,0x22,0xf3,0x4f,0x96,0x51,0x4a,0xed,0x15,0x65,0x97,0x67,0x35,
	  	0xe3,0x21,0xc2,0x66,0x30,0x0d,0xe6,0xb9,0x3d,0xd9,0x3e,0x4d,0xb4,0xe5,0x21,0xb1,
	  	0xa2,0x69,0x94,0x7c,0xb7,0x9b,0xcc,0xd4,0x72,0xcd,0x40,0xe3,0xfd,0x43,0x07,0x26,
	  	0x72,0x3c,0xb9,0x7b,0xca,0x50,0xe6,0x10,0xbd,0x81,0xda,0xed,0x31,0xfc,0x16,0x4c,
	  	0x18,0xef,0x37,0x8f,0x68,0xe4,0x05,0xee,0x6d,0x97,0x8f,0x97,0xcc,0x04,0x2d,0xd2,
	  	0x66,0xc9,0x0b,0x90,0x9b,0xa5,0x21,0x28,0x82,0x4f,0xea,0x7f,0x44,0x96,0xad,0x56,
	  	0xe0,0x3f,0xa4,0xae,0xb5,0x28,0xd2,0xfd,0x3d,0x9b,0xfa,0xe8,0x29,0x5e,0x1a,0x53,
	  	0x06,0xb3,0x89,0x65,0x4b,0x1d,0x70,0x58,0x89,0x0b,0x5d,0xbd,0x67,0x70,0xff,0x5b)
	},
	{ chunk_from_chars(                                                          /* 9 */
		0x31,0x32,0x33,0x34,0x30,0x30),
	  chunk_from_chars(
		0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,
		0x10,0x11,0x12,0x13),
	  chunk_from_chars(
		0x87,0x3c,0x4e,0xcb,0x48,0x58,0xc6,0x02,0x18,0xb4,0xff,0x5f,0xe7,0x5e,0xc2,0x16,
		0x50,0x23,0x67,0xe0,0xa8,0x44,0x6f,0xef,0x23,0x02,0xda,0x89,0x28,0x16,0xf1,0x27,
		0xa8,0xbe,0xdf,0x7e,0x2f,0xff,0x34,0x9b,0xe4,0x93,0x85,0x7c,0x4e,0x36,0x34,0x93,
		0xb2,0x8f,0x90,0x36,0x5e,0xae,0x44,0x32,0x3a,0x93,0x3e,0xae,0x25,0x70,0x1d,0x84,
		0x02,0x4c,0x84,0xe0,0x9b,0x91,0x63,0xde,0x0d,0xc7,0xba,0x2d,0x44,0xf9,0x7c,0x29,
		0xaf,0x5b,0x94,0x22,0xc4,0x17,0xbb,0x24,0x1a,0x2f,0xd3,0x26,0xef,0xd9,0xae,0xd2,
		0xdc,0xd7,0xce,0xa5,0x5b,0xb1,0x6b,0x95,0x71,0x64,0x7e,0xae,0xce,0x0a,0x0c,0xa7,
		0x1f,0x7d,0xa7,0xde,0xcf,0xbc,0x58,0x8c,0xaf,0xa8,0xbe,0xac,0xf0,0xb2,0x86,0xe1,
		0x43,0xff,0x77,0xf0,0x54,0x07,0x38,0x8b,0x72,0xc8,0x9c,0x6a,0x4b,0xe3,0x5d,0x07,
		0x40,0x83,0xe2,0xa1,0x28,0xfd,0xa5,0x3b,0xa7,0xd3,0x36,0x3c,0x0d,0x26,0x14,0x30,
		0x66,0x77,0x70,0xfd,0x7b,0x42,0x03,0x2a,0x4d,0x97,0x64,0xfa,0x86,0xdd,0x48,0x34,
		0xeb,0xa3,0x86,0x4a,0xcf,0x3e,0x0c,0x23,0x0f,0xd0,0x35,0x81,0x29,0x25,0x62,0x15,
		0x1f,0xaa,0x1e,0xd3,0xf8,0x43,0x3b,0xc6,0x34,0x72,0xc4,0x25,0xc3,0x82,0xd7,0x42,
		0x5b,0x11,0xc9,0x79,0x69,0xb1,0xfd,0x0f,0xa0,0xb0,0x14,0xd4,0x7d,0x87,0xaf,0x91,
		0x9d,0xe6,0xdf,0x72,0xba,0x52,0xbe,0x88,0x8e,0x44,0xde,0x4f,0xa6,0x3c,0x21,0x5e,
		0xa8,0x0f,0xc6,0x10,0x72,0xa3,0x20,0xa2,0x4e,0x99,0xc6,0xa2,0x47,0x88,0x0c,0x9a)
	},
	{ chunk_from_chars(                                                         /* 10 */
		0x31,0x32,0x33,0x34,0x30,0x30),
	  chunk_from_chars(
		0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,
		0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1a,0x1b,0x1c,0x1d,0x1e,0x1f),
	  chunk_from_chars(
		0x91,0xde,0x22,0xe7,0x67,0x14,0xef,0xf6,0x12,0xa4,0x13,0x18,0x30,0x3d,0xc5,0xaa,
		0xf1,0x54,0xdc,0x5e,0x4b,0x79,0xdf,0x84,0x95,0xb5,0xac,0xba,0x18,0xa2,0x7b,0xe6,
		0xdd,0x45,0x4f,0x2c,0x39,0x49,0x4a,0xa1,0x8d,0x19,0x11,0xaa,0xbf,0x38,0xbb,0xad,
		0x04,0x60,0x92,0x24,0x04,0xc6,0xe2,0xc0,0x8a,0xc3,0xab,0xfd,0xcd,0xb9,0x63,0x62,
		0x3f,0x1c,0x1c,0x07,0xeb,0x20,0x96,0xdf,0xf7,0x06,0x4d,0x3c,0x55,0xc6,0x22,0x17,
		0x73,0xe4,0xb2,0xa8,0x0a,0xda,0xeb,0x21,0x05,0x32,0xec,0x76,0xa3,0xc9,0x28,0x32,
		0xb0,0xaa,0xf7,0xda,0x03,0xcf,0xb6,0x03,0x03,0xe0,0x11,0x6c,0xb6,0xcd,0xea,0xd2,
		0xb7,0x37,0x1c,0xbf,0xae,0xbb,0xf3,0xe5,0x11,0x50,0x3b,0x9b,0x8b,0xcd,0x54,0xc1,
		0x3c,0x03,0x5a,0x44,0x36,0x8f,0xba,0x19,0xdf,0xb8,0x73,0x5c,0x61,0x17,0xa3,0xed,
		0xcb,0xe9,0x93,0x9a,0xfa,0xbd,0x72,0x04,0xb6,0x54,0x2c,0x77,0xd3,0x1a,0x6a,0x24,
		0x4b,0x10,0x48,0x12,0x11,0xec,0x99,0xe4,0x52,0x31,0x20,0x37,0x20,0x04,0x3f,0xe8,
		0x9a,0x57,0xc7,0xa6,0x12,0xb3,0x58,0x8b,0x12,0x16,0x6c,0x55,0xed,0xec,0x13,0xb8,
		0x26,0x5a,0x2c,0x09,0x1d,0xd8,0x59,0xd4,0xf3,0x4e,0xb6,0x6e,0xad,0x8b,0x58,0x3f,
		0xd1,0xf0,0xf0,0xec,0x73,0x9b,0x92,0xe7,0x61,0x8c,0x1a,0xbe,0x1e,0x3e,0x92,0x19,
		0xfc,0x43,0xf6,0x28,0xfe,0x89,0xcc,0xdf,0x2c,0xab,0xc0,0x7f,0xfc,0xc5,0xc4,0x85,
		0xc4,0xca,0xdb,0xbb,0xeb,0x02,0xef,0xd8,0x39,0x4f,0x4d,0xb0,0xb7,0x2d,0x81,0xd8)
	},
	{ chunk_from_chars(                                                         /* 11 */
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78),
	  chunk_from_chars(),
	  chunk_from_chars(
	  	0x86,0x54,0x64,0x30,0x90,0xce,0x42,0x61,0x18,0xdc,0xce,0x2b,0xb0,0xfc,0xa3,0xb2,
	  	0xa4,0xed,0x45,0x58,0x9d,0x73,0xae,0x2f,0x56,0x83,0x33,0x4f,0x17,0xca,0x5b,0x9f,
	  	0x41,0xeb,0x97,0xff,0x56,0x6c,0xbb,0xc0,0x17,0x1d,0x89,0x62,0xfc,0xca,0x46,0x5f,
	  	0x66,0xf4,0xcf,0x14,0xd0,0x11,0x4d,0x77,0x68,0xbb,0xac,0x1d,0xfb,0x72,0x6d,0x91,
	  	0x33,0x16,0x70,0xe0,0xe8,0x8c,0xe7,0x28,0x6e,0x1f,0x0e,0xe7,0xac,0xd5,0xcd,0x75,
	  	0xe0,0xd3,0x25,0xad,0xc9,0xf4,0x7f,0xf0,0x28,0x20,0x54,0x47,0x70,0xec,0xf4,0x6f,
	  	0x38,0x1f,0xf3,0x30,0xa5,0xc7,0x59,0x78,0xe8,0xd2,0x48,0xcc,0x12,0xf0,0x75,0x2c,
	  	0xc7,0xc4,0x52,0x26,0x5a,0xe9,0x19,0xf5,0xa4,0xdd,0x7c,0x20,0x02,0x3f,0xbf,0x51,
	  	0xd0,0xb4,0x13,0xaf,0xe0,0x9f,0x74,0x61,0x14,0xd1,0x6d,0x1b,0xe4,0x99,0xc9,0x8d,
	  	0x27,0x0e,0x92,0x13,0xa3,0x7a,0x2a,0x74,0x5a,0x3b,0xb2,0x16,0xbd,0x30,0xa0,0x7d,
	  	0x11,0x05,0x58,0x09,0x0f,0x4f,0xe7,0x3d,0xeb,0xea,0x00,0x9e,0x18,0xeb,0x37,0x70,
	  	0xee,0xdd,0xd7,0x15,0x0b,0x4c,0x72,0x31,0x53,0x9f,0x72,0x32,0xd8,0x37,0x83,0x64,
	  	0x90,0xc9,0x54,0x7e,0x27,0xda,0xc0,0x0e,0xfb,0x5a,0x8c,0xcd,0xf9,0xa7,0xc3,0xc1,
	  	0xd2,0x9c,0x03,0x51,0x70,0xa5,0x17,0xdb,0x89,0x9e,0xe6,0x19,0x25,0xcc,0x7a,0x2c,
	  	0xd2,0x78,0xcf,0x57,0x6e,0xbe,0x82,0x81,0xbd,0x3a,0x4f,0x26,0x65,0xf1,0x5d,0xd0,
	  	0xb2,0xe6,0x1b,0xcd,0x65,0x17,0x2c,0x2f,0xff,0x50,0xa8,0x6a,0x2a,0xdb,0xf2,0xf1)
	}
};

static void test_rsa_oaep(chunk_t key, encryption_scheme_t encryption_scheme,
						  rsa_oaep_test_t *test)
{
	private_key_t *privkey;
	public_key_t *pubkey;
	chunk_t msg = chunk_empty, ct = chunk_empty;
	bool success;

	privkey = lib->creds->create(lib->creds, CRED_PRIVATE_KEY, KEY_ANY,
								 BUILD_BLOB_ASN1_DER, key, BUILD_END);
	ck_assert(privkey != NULL);
	ck_assert(privkey->get_type(privkey) == KEY_RSA);

	pubkey = privkey->get_public_key(privkey);
	ck_assert(pubkey != NULL);

	/* decrypt */
	success = privkey->decrypt(privkey, encryption_scheme, &test->label,
							   test->ct, &msg);
	if (test->label.len > 0 && !success)
	{
		warn("RSA OAEP labels might not be supported");
		goto end;
	}
	ck_assert(success);
	ck_assert_chunk_eq(msg, test->msg);
	chunk_free(&msg);

	/* encrypt and decrypt again */
	ck_assert(pubkey->encrypt(pubkey, encryption_scheme, &test->label,
							  test->msg, &ct));
	ck_assert(privkey->decrypt(privkey, encryption_scheme, &test->label,
							   ct, &msg));
	ck_assert_chunk_eq(msg, test->msg);
	chunk_free(&ct);
	chunk_free(&msg);

end:
	pubkey->destroy(pubkey);
	privkey->destroy(privkey);
}

START_TEST(test_rsa_2048_oaep_sha1)
{
	test_rsa_oaep(rsa_2048_key, ENCRYPT_RSA_OAEP_SHA1,
				 &rsa_2048_oaep_sha1_tests[_i]);
}
END_TEST

Suite *rsa_oaep_sha1_suite_create()
{
	Suite *s;
	TCase *tc;

	s = suite_create("rsa_oaep_sha1");

	tc = tcase_create("rsa_2048_oaep_sha1");
	tcase_add_loop_test(tc, test_rsa_2048_oaep_sha1, 0, countof(rsa_2048_oaep_sha1_tests));
	suite_add_tcase(s, tc);

	return s;
}
