/*
 * Copyright (c) 2019-2021, 2023 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef ARM_COMPUTE_ENABLE_SVE

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>
#include <cstdint>

namespace arm_gemm {

void sve_hybrid_u8qa_dot_4x4VL (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<uint8_t> A_arg,
    size_t M, size_t N, const uint8_t *B_ptr, IndirectOutputArg<uint8_t> output_arg,
    const Requantize32 *qp, const int32_t *col_bias, unsigned int
)
{
    struct KernelArgs {
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const uint8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
    } ka;

    unsigned long flags=0;
    void *output_ptr;
    void *input_ptr;

    if (output_arg.is_indirect) {
        output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    if (qp->c_offset > qp->minval) {
        flags |= 0x20;
    }
    __asm__ __volatile__(
      "ptrue p2.b\n"
      "1:"  // Row loop
      "cmp %x[M], #0x4\n"
      "bge 43f\n"
      "cmp %x[M], #0x2\n"
      "bgt 29f\n"
      "beq 15f\n"
      "mov x10, %x[col_bias]\n"
      "mov z11.s, #0x0\n"
      "mov z15.b, #0x1\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "ldr x9, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x28, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x27, %x[output_ptr]\n"
      "2:"  // Height 1: Column loop
      "mov x20, #0x0\n"
      "whilelt p1.b, x20, x9\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "3:"  // Height 1: setup done
      "mov x26, #0x0\n"
      "4:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 5f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "cbnz x26, 6f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "b 6f\n"
      "5:"  // Height 1: setup direct input
      "mov x24, %x[input_ptr]\n"
      "6:"  // Height 1: input setup done
      "cmp x25, #0x10\n"
      "ble 9f\n"
      "7:"  // Height 1: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x25\n"
      "ld1rqb { z0.b }, p0/Z, [x24]\n"
      "ld1b { z20.b }, p2/Z, [x28]\n"
      "udot z16.s, z20.b, z0.b[0]\n"
      "ld1b { z21.b }, p2/Z, [x28, #1, MUL VL]\n"
      "ld1b { z20.b }, p2/Z, [x28, #2, MUL VL]\n"
      "udot z17.s, z21.b, z0.b[0]\n"
      "udot z18.s, z20.b, z0.b[0]\n"
      "ld1b { z20.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z19.s, z20.b, z0.b[0]\n"
      "ld1b { z20.b }, p2/Z, [x28, #4, MUL VL]\n"
      "udot z16.s, z20.b, z0.b[1]\n"
      "ld1b { z21.b }, p2/Z, [x28, #5, MUL VL]\n"
      "ld1b { z20.b }, p2/Z, [x28, #6, MUL VL]\n"
      "udot z17.s, z21.b, z0.b[1]\n"
      "udot z18.s, z20.b, z0.b[1]\n"
      "ld1b { z20.b }, p2/Z, [x28, #7, MUL VL]\n"
      "addvl x28, x28, #16\n"
      "udot z19.s, z20.b, z0.b[1]\n"
      "ld1b { z22.b }, p2/Z, [x28, #-8, MUL VL]\n"
      "ld1b { z20.b }, p2/Z, [x28, #-7, MUL VL]\n"
      "ld1b { z21.b }, p2/Z, [x28, #-6, MUL VL]\n"
      "udot z16.s, z22.b, z0.b[2]\n"
      "udot z17.s, z20.b, z0.b[2]\n"
      "ld1b { z20.b }, p2/Z, [x28, #-5, MUL VL]\n"
      "udot z18.s, z21.b, z0.b[2]\n"
      "udot z19.s, z20.b, z0.b[2]\n"
      "ld1b { z22.b }, p2/Z, [x28, #-4, MUL VL]\n"
      "ld1b { z20.b }, p2/Z, [x28, #-3, MUL VL]\n"
      "ld1b { z21.b }, p2/Z, [x28, #-2, MUL VL]\n"
      "udot z16.s, z22.b, z0.b[3]\n"
      "udot z17.s, z20.b, z0.b[3]\n"
      "ld1b { z20.b }, p2/Z, [x28, #-1, MUL VL]\n"
      "udot z18.s, z21.b, z0.b[3]\n"
      "udot z19.s, z20.b, z0.b[3]\n"
      "add x24, x24, #0x10\n"
      "tbnz %x[flags], #31, 8f\n"
      "udot z11.s, z0.b, z15.b\n"
      "8:"  // Height 1: Multiply loop: unique 1: skip row sum
      "sub x25, x25, #0x10\n"
      "cmp x25, #0x10\n"
      "bgt 7b\n"
      "9:"  // Height 1: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x25\n"
      "ld1rqb { z0.b }, p0/Z, [x24]\n"
      "ld1b { z22.b }, p2/Z, [x28]\n"
      "subs x25, x25, #0x4\n"
      "ld1b { z20.b }, p2/Z, [x28, #1, MUL VL]\n"
      "ld1b { z21.b }, p2/Z, [x28, #2, MUL VL]\n"
      "udot z16.s, z22.b, z0.b[0]\n"
      "udot z17.s, z20.b, z0.b[0]\n"
      "ld1b { z20.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z18.s, z21.b, z0.b[0]\n"
      "udot z19.s, z20.b, z0.b[0]\n"
      "addvl x28, x28, #4\n"
      "ble 10f\n"
      "ld1b { z20.b }, p2/Z, [x28]\n"
      "ld1b { z22.b }, p2/Z, [x28, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "udot z16.s, z20.b, z0.b[1]\n"
      "ld1b { z21.b }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1b { z20.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z17.s, z22.b, z0.b[1]\n"
      "udot z18.s, z21.b, z0.b[1]\n"
      "udot z19.s, z20.b, z0.b[1]\n"
      "addvl x28, x28, #4\n"
      "ble 10f\n"
      "ld1b { z20.b }, p2/Z, [x28]\n"
      "ld1b { z22.b }, p2/Z, [x28, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "udot z16.s, z20.b, z0.b[2]\n"
      "ld1b { z21.b }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1b { z20.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z17.s, z22.b, z0.b[2]\n"
      "udot z18.s, z21.b, z0.b[2]\n"
      "udot z19.s, z20.b, z0.b[2]\n"
      "addvl x28, x28, #4\n"
      "ble 10f\n"
      "ld1b { z21.b }, p2/Z, [x28]\n"
      "ld1b { z20.b }, p2/Z, [x28, #1, MUL VL]\n"
      "udot z16.s, z21.b, z0.b[3]\n"
      "udot z17.s, z20.b, z0.b[3]\n"
      "ld1b { z21.b }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1b { z20.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z18.s, z21.b, z0.b[3]\n"
      "udot z19.s, z20.b, z0.b[3]\n"
      "addvl x28, x28, #4\n"
      "10:"  // Height 1: Multiply loop: multiply skip
      "tbnz %x[flags], #31, 11f\n"
      "udot z11.s, z0.b, z15.b\n"
      "11:"  // Height 1: Multiply loop: unique 2: skip row sum
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 4b\n"
      "tbnz %x[flags], #31, 12f\n"
      "mov x20, #0x4\n"
      "whilelt p0.s, XZR, x20\n"
      "add x20, %x[qp], %[b_offset]\n"
      "ld1rw { z20.s }, p2/Z, [x20]\n"
      "uaddv d11, p0, z11.s\n"
      "mov z11.s, z11.s[0]\n"
      "neg z20.s, p2/M, z20.s\n"
      "mul z11.s, p2/M, z11.s, z20.s\n"
      "12:"  // Height 1: skip row sum fixup
      "add z16.s, z16.s, z11.s\n"
      "add z17.s, z17.s, z11.s\n"
      "ld1w { z23.s }, p2/Z, [x10]\n"
      "ld1w { z22.s }, p2/Z, [x10, #1, MUL VL]\n"
      "add z18.s, z18.s, z11.s\n"
      "add z19.s, z19.s, z11.s\n"
      "ld1w { z21.s }, p2/Z, [x10, #2, MUL VL]\n"
      "ld1w { z20.s }, p2/Z, [x10, #3, MUL VL]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "add z16.s, z16.s, z23.s\n"
      "add z17.s, z17.s, z22.s\n"
      "add z18.s, z18.s, z21.s\n"
      "add z19.s, z19.s, z20.s\n"
      "ld1rw { z20.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "ld1rw { z0.s }, p2/Z, [x20]\n"
      ".inst 0x04b47610  // sqrdmulh z16.s, z16.s, z20.s\n"
      ".inst 0x04b47631  // sqrdmulh z17.s, z17.s, z20.s\n"
      "addvl x10, x10, #4\n"
      ".inst 0x04b47652  // sqrdmulh z18.s, z18.s, z20.s\n"
      ".inst 0x04b47673  // sqrdmulh z19.s, z19.s, z20.s\n"
      "tbz %x[flags], #5, 13f\n"
      "and z23.d, z16.d, z0.d\n"
      "and z22.d, z17.d, z0.d\n"
      "and z21.d, z18.d, z0.d\n"
      "and z20.d, z19.d, z0.d\n"
      "asr z23.s, z23.s, #0x1f\n"
      "asr z22.s, z22.s, #0x1f\n"
      "asr z21.s, z21.s, #0x1f\n"
      "asr z20.s, z20.s, #0x1f\n"
      "sqadd z16.s, z16.s, z23.s\n"
      "sqadd z17.s, z17.s, z22.s\n"
      "sqadd z18.s, z18.s, z21.s\n"
      "sqadd z19.s, z19.s, z20.s\n"
      "13:"  // Height 1: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      "ld1rw { z20.s }, p2/Z, [x20]\n"
      ".inst 0x44828810  // srshl z16.s, p2/M, z16.s, z0.s\n"
      "add z16.s, z16.s, z20.s\n"
      ".inst 0x44828811  // srshl z17.s, p2/M, z17.s, z0.s\n"
      ".inst 0x44828812  // srshl z18.s, p2/M, z18.s, z0.s\n"
      "add z17.s, z17.s, z20.s\n"
      "add z18.s, z18.s, z20.s\n"
      ".inst 0x44828813  // srshl z19.s, p2/M, z19.s, z0.s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1rw { z21.s }, p2/Z, [x20]\n"
      "add z19.s, z19.s, z20.s\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1rw { z20.s }, p2/Z, [x20]\n"
      "smin z16.s, p2/M, z16.s, z21.s\n"
      "smin z17.s, p2/M, z17.s, z21.s\n"
      "smin z18.s, p2/M, z18.s, z21.s\n"
      "smin z19.s, p2/M, z19.s, z21.s\n"
      "smax z16.s, p2/M, z16.s, z20.s\n"
      "smax z17.s, p2/M, z17.s, z20.s\n"
      "smax z18.s, p2/M, z18.s, z20.s\n"
      "uzp1 z16.h, z16.h, z17.h\n"
      "smax z19.s, p2/M, z19.s, z20.s\n"
      "uzp1 z17.h, z18.h, z19.h\n"
      "uzp1 z16.b, z16.b, z17.b\n"
      "st1b { z16.b }, p1, [x27]\n"
      "addvl x27, x27, #1\n"
      "14:"  // Height 1: Writeback done
      "decw x9, ALL, MUL #4\n"
      "cmp x9, XZR\n"
      "bgt 2b\n"
      "b 58f\n"
      "15:"  // Height 2
      "mov x10, %x[col_bias]\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "mov z15.b, #0x1\n"
      "ldr x9, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x28, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x27, %x[output_ptr]\n"
      "16:"  // Height 2: Column loop
      "mov x20, #0x0\n"
      "whilelt p1.b, x20, x9\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "mov z20.s, #0x0\n"
      "mov z21.s, #0x0\n"
      "mov z22.s, #0x0\n"
      "mov z23.s, #0x0\n"
      "17:"  // Height 2: setup done
      "mov x26, #0x0\n"
      "18:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 19f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "cbnz x26, 20f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "b 20f\n"
      "19:"  // Height 2: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21\n"
      "20:"  // Height 2: input setup done
      "cmp x25, #0x10\n"
      "ble 23f\n"
      "21:"  // Height 2: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x25\n"
      "ld1rqb { z0.b }, p0/Z, [x24]\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "add x24, x24, #0x10\n"
      "ld1b { z24.b }, p2/Z, [x28]\n"
      "ld1b { z26.b }, p2/Z, [x28, #1, MUL VL]\n"
      "udot z16.s, z24.b, z0.b[0]\n"
      "udot z20.s, z24.b, z1.b[0]\n"
      "ld1b { z24.b }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1b { z25.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z17.s, z26.b, z0.b[0]\n"
      "udot z21.s, z26.b, z1.b[0]\n"
      "udot z18.s, z24.b, z0.b[0]\n"
      "udot z22.s, z24.b, z1.b[0]\n"
      "ld1b { z24.b }, p2/Z, [x28, #4, MUL VL]\n"
      "ld1b { z27.b }, p2/Z, [x28, #5, MUL VL]\n"
      "udot z19.s, z25.b, z0.b[0]\n"
      "udot z23.s, z25.b, z1.b[0]\n"
      "ld1b { z26.b }, p2/Z, [x28, #6, MUL VL]\n"
      "ld1b { z25.b }, p2/Z, [x28, #7, MUL VL]\n"
      "addvl x28, x28, #16\n"
      "udot z16.s, z24.b, z0.b[1]\n"
      "udot z20.s, z24.b, z1.b[1]\n"
      "ld1b { z24.b }, p2/Z, [x28, #-8, MUL VL]\n"
      "udot z17.s, z27.b, z0.b[1]\n"
      "udot z21.s, z27.b, z1.b[1]\n"
      "ld1b { z30.b }, p2/Z, [x28, #-7, MUL VL]\n"
      "ld1b { z29.b }, p2/Z, [x28, #-6, MUL VL]\n"
      "udot z18.s, z26.b, z0.b[1]\n"
      "udot z22.s, z26.b, z1.b[1]\n"
      "ld1b { z28.b }, p2/Z, [x28, #-5, MUL VL]\n"
      "ld1b { z27.b }, p2/Z, [x28, #-4, MUL VL]\n"
      "udot z19.s, z25.b, z0.b[1]\n"
      "udot z23.s, z25.b, z1.b[1]\n"
      "ld1b { z26.b }, p2/Z, [x28, #-3, MUL VL]\n"
      "ld1b { z25.b }, p2/Z, [x28, #-2, MUL VL]\n"
      "udot z16.s, z24.b, z0.b[2]\n"
      "udot z20.s, z24.b, z1.b[2]\n"
      "ld1b { z24.b }, p2/Z, [x28, #-1, MUL VL]\n"
      "add x23, x23, #0x10\n"
      "udot z17.s, z30.b, z0.b[2]\n"
      "udot z21.s, z30.b, z1.b[2]\n"
      "udot z18.s, z29.b, z0.b[2]\n"
      "udot z22.s, z29.b, z1.b[2]\n"
      "udot z19.s, z28.b, z0.b[2]\n"
      "udot z23.s, z28.b, z1.b[2]\n"
      "udot z16.s, z27.b, z0.b[3]\n"
      "udot z20.s, z27.b, z1.b[3]\n"
      "udot z17.s, z26.b, z0.b[3]\n"
      "udot z21.s, z26.b, z1.b[3]\n"
      "udot z18.s, z25.b, z0.b[3]\n"
      "udot z22.s, z25.b, z1.b[3]\n"
      "udot z19.s, z24.b, z0.b[3]\n"
      "udot z23.s, z24.b, z1.b[3]\n"
      "tbnz %x[flags], #31, 22f\n"
      "udot z11.s, z0.b, z15.b\n"
      "udot z12.s, z1.b, z15.b\n"
      "22:"  // Height 2: Multiply loop: unique 3: skip row sum
      "sub x25, x25, #0x10\n"
      "cmp x25, #0x10\n"
      "bgt 21b\n"
      "23:"  // Height 2: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x25\n"
      "ld1rqb { z0.b }, p0/Z, [x24]\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "subs x25, x25, #0x4\n"
      "ld1b { z24.b }, p2/Z, [x28]\n"
      "ld1b { z26.b }, p2/Z, [x28, #1, MUL VL]\n"
      "udot z16.s, z24.b, z0.b[0]\n"
      "udot z20.s, z24.b, z1.b[0]\n"
      "ld1b { z25.b }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1b { z24.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z17.s, z26.b, z0.b[0]\n"
      "udot z21.s, z26.b, z1.b[0]\n"
      "udot z18.s, z25.b, z0.b[0]\n"
      "udot z22.s, z25.b, z1.b[0]\n"
      "addvl x28, x28, #4\n"
      "udot z19.s, z24.b, z0.b[0]\n"
      "udot z23.s, z24.b, z1.b[0]\n"
      "ble 24f\n"
      "ld1b { z27.b }, p2/Z, [x28]\n"
      "ld1b { z26.b }, p2/Z, [x28, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "udot z16.s, z27.b, z0.b[1]\n"
      "ld1b { z25.b }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1b { z24.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z20.s, z27.b, z1.b[1]\n"
      "udot z17.s, z26.b, z0.b[1]\n"
      "udot z21.s, z26.b, z1.b[1]\n"
      "udot z18.s, z25.b, z0.b[1]\n"
      "addvl x28, x28, #4\n"
      "udot z22.s, z25.b, z1.b[1]\n"
      "udot z19.s, z24.b, z0.b[1]\n"
      "udot z23.s, z24.b, z1.b[1]\n"
      "ble 24f\n"
      "ld1b { z27.b }, p2/Z, [x28]\n"
      "ld1b { z26.b }, p2/Z, [x28, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "udot z16.s, z27.b, z0.b[2]\n"
      "ld1b { z25.b }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1b { z24.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z20.s, z27.b, z1.b[2]\n"
      "udot z17.s, z26.b, z0.b[2]\n"
      "udot z21.s, z26.b, z1.b[2]\n"
      "udot z18.s, z25.b, z0.b[2]\n"
      "addvl x28, x28, #4\n"
      "udot z22.s, z25.b, z1.b[2]\n"
      "udot z19.s, z24.b, z0.b[2]\n"
      "udot z23.s, z24.b, z1.b[2]\n"
      "ble 24f\n"
      "ld1b { z24.b }, p2/Z, [x28]\n"
      "ld1b { z26.b }, p2/Z, [x28, #1, MUL VL]\n"
      "udot z16.s, z24.b, z0.b[3]\n"
      "udot z20.s, z24.b, z1.b[3]\n"
      "ld1b { z25.b }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1b { z24.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z17.s, z26.b, z0.b[3]\n"
      "udot z21.s, z26.b, z1.b[3]\n"
      "udot z18.s, z25.b, z0.b[3]\n"
      "udot z22.s, z25.b, z1.b[3]\n"
      "addvl x28, x28, #4\n"
      "udot z19.s, z24.b, z0.b[3]\n"
      "udot z23.s, z24.b, z1.b[3]\n"
      "24:"  // Height 2: Multiply loop: multiply skip
      "tbnz %x[flags], #31, 25f\n"
      "udot z11.s, z0.b, z15.b\n"
      "udot z12.s, z1.b, z15.b\n"
      "25:"  // Height 2: Multiply loop: unique 4: skip row sum
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 18b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x23, x27, x20\n"
      "tbnz %x[flags], #31, 26f\n"
      "mov x20, #0x4\n"
      "whilelt p0.s, XZR, x20\n"
      "add x20, %x[qp], %[b_offset]\n"
      "ld1rw { z24.s }, p2/Z, [x20]\n"
      "uaddv d11, p0, z11.s\n"
      "mov z11.s, z11.s[0]\n"
      "uaddv d12, p0, z12.s\n"
      "neg z24.s, p2/M, z24.s\n"
      "mov z12.s, z12.s[0]\n"
      "mul z11.s, p2/M, z11.s, z24.s\n"
      "mul z12.s, p2/M, z12.s, z24.s\n"
      "26:"  // Height 2: skip row sum fixup
      "add z16.s, z16.s, z11.s\n"
      "add z17.s, z17.s, z11.s\n"
      "ld1w { z28.s }, p2/Z, [x10]\n"
      "ld1w { z27.s }, p2/Z, [x10, #1, MUL VL]\n"
      "add z18.s, z18.s, z11.s\n"
      "add z19.s, z19.s, z11.s\n"
      "ld1w { z26.s }, p2/Z, [x10, #2, MUL VL]\n"
      "ld1w { z25.s }, p2/Z, [x10, #3, MUL VL]\n"
      "add z20.s, z20.s, z12.s\n"
      "add z21.s, z21.s, z12.s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "add z22.s, z22.s, z12.s\n"
      "add z23.s, z23.s, z12.s\n"
      "ld1rw { z24.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "add z16.s, z16.s, z28.s\n"
      "add z17.s, z17.s, z27.s\n"
      "addvl x10, x10, #4\n"
      "add z18.s, z18.s, z26.s\n"
      "add z19.s, z19.s, z25.s\n"
      "add z20.s, z20.s, z28.s\n"
      "add z21.s, z21.s, z27.s\n"
      "ld1rw { z0.s }, p2/Z, [x20]\n"
      "add z22.s, z22.s, z26.s\n"
      "add z23.s, z23.s, z25.s\n"
      ".inst 0x04b87610  // sqrdmulh z16.s, z16.s, z24.s\n"
      ".inst 0x04b87631  // sqrdmulh z17.s, z17.s, z24.s\n"
      ".inst 0x04b87652  // sqrdmulh z18.s, z18.s, z24.s\n"
      ".inst 0x04b87673  // sqrdmulh z19.s, z19.s, z24.s\n"
      ".inst 0x04b87694  // sqrdmulh z20.s, z20.s, z24.s\n"
      ".inst 0x04b876b5  // sqrdmulh z21.s, z21.s, z24.s\n"
      ".inst 0x04b876d6  // sqrdmulh z22.s, z22.s, z24.s\n"
      ".inst 0x04b876f7  // sqrdmulh z23.s, z23.s, z24.s\n"
      "tbz %x[flags], #5, 27f\n"
      "and z24.d, z16.d, z0.d\n"
      "asr z24.s, z24.s, #0x1f\n"
      "sqadd z16.s, z16.s, z24.s\n"
      "and z30.d, z17.d, z0.d\n"
      "and z29.d, z18.d, z0.d\n"
      "and z28.d, z19.d, z0.d\n"
      "and z27.d, z20.d, z0.d\n"
      "and z26.d, z21.d, z0.d\n"
      "and z25.d, z22.d, z0.d\n"
      "and z24.d, z23.d, z0.d\n"
      "asr z30.s, z30.s, #0x1f\n"
      "asr z29.s, z29.s, #0x1f\n"
      "asr z28.s, z28.s, #0x1f\n"
      "asr z27.s, z27.s, #0x1f\n"
      "asr z26.s, z26.s, #0x1f\n"
      "asr z25.s, z25.s, #0x1f\n"
      "asr z24.s, z24.s, #0x1f\n"
      "sqadd z17.s, z17.s, z30.s\n"
      "sqadd z18.s, z18.s, z29.s\n"
      "sqadd z19.s, z19.s, z28.s\n"
      "sqadd z20.s, z20.s, z27.s\n"
      "sqadd z21.s, z21.s, z26.s\n"
      "sqadd z22.s, z22.s, z25.s\n"
      "sqadd z23.s, z23.s, z24.s\n"
      "27:"  // Height 2: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      "ld1rw { z24.s }, p2/Z, [x20]\n"
      ".inst 0x44828810  // srshl z16.s, p2/M, z16.s, z0.s\n"
      "add z16.s, z16.s, z24.s\n"
      ".inst 0x44828811  // srshl z17.s, p2/M, z17.s, z0.s\n"
      ".inst 0x44828812  // srshl z18.s, p2/M, z18.s, z0.s\n"
      "add z17.s, z17.s, z24.s\n"
      "add z18.s, z18.s, z24.s\n"
      ".inst 0x44828813  // srshl z19.s, p2/M, z19.s, z0.s\n"
      ".inst 0x44828814  // srshl z20.s, p2/M, z20.s, z0.s\n"
      "add z19.s, z19.s, z24.s\n"
      "add z20.s, z20.s, z24.s\n"
      ".inst 0x44828815  // srshl z21.s, p2/M, z21.s, z0.s\n"
      ".inst 0x44828816  // srshl z22.s, p2/M, z22.s, z0.s\n"
      "add z21.s, z21.s, z24.s\n"
      "add z22.s, z22.s, z24.s\n"
      ".inst 0x44828817  // srshl z23.s, p2/M, z23.s, z0.s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1rw { z25.s }, p2/Z, [x20]\n"
      "add z23.s, z23.s, z24.s\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1rw { z24.s }, p2/Z, [x20]\n"
      "smin z16.s, p2/M, z16.s, z25.s\n"
      "smin z17.s, p2/M, z17.s, z25.s\n"
      "smin z18.s, p2/M, z18.s, z25.s\n"
      "smin z19.s, p2/M, z19.s, z25.s\n"
      "smin z20.s, p2/M, z20.s, z25.s\n"
      "smin z21.s, p2/M, z21.s, z25.s\n"
      "smin z22.s, p2/M, z22.s, z25.s\n"
      "smin z23.s, p2/M, z23.s, z25.s\n"
      "smax z16.s, p2/M, z16.s, z24.s\n"
      "smax z17.s, p2/M, z17.s, z24.s\n"
      "smax z18.s, p2/M, z18.s, z24.s\n"
      "uzp1 z16.h, z16.h, z17.h\n"
      "smax z19.s, p2/M, z19.s, z24.s\n"
      "smax z20.s, p2/M, z20.s, z24.s\n"
      "uzp1 z17.h, z18.h, z19.h\n"
      "uzp1 z16.b, z16.b, z17.b\n"
      "smax z21.s, p2/M, z21.s, z24.s\n"
      "smax z22.s, p2/M, z22.s, z24.s\n"
      "uzp1 z20.h, z20.h, z21.h\n"
      "st1b { z16.b }, p1, [x27]\n"
      "smax z23.s, p2/M, z23.s, z24.s\n"
      "uzp1 z16.h, z22.h, z23.h\n"
      "uzp1 z20.b, z20.b, z16.b\n"
      "st1b { z20.b }, p1, [x23]\n"
      "addvl x27, x27, #1\n"
      "28:"  // Height 2: Writeback done
      "decw x9, ALL, MUL #4\n"
      "cmp x9, XZR\n"
      "bgt 16b\n"
      "b 58f\n"
      "29:"  // Height 3
      "mov x10, %x[col_bias]\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "mov z13.s, #0x0\n"
      "mov z15.b, #0x1\n"
      "ldr x9, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x28, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x27, %x[output_ptr]\n"
      "30:"  // Height 3: Column loop
      "mov x20, #0x0\n"
      "whilelt p1.b, x20, x9\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "mov z20.s, #0x0\n"
      "mov z21.s, #0x0\n"
      "mov z22.s, #0x0\n"
      "mov z23.s, #0x0\n"
      "mov z24.s, #0x0\n"
      "mov z25.s, #0x0\n"
      "mov z26.s, #0x0\n"
      "mov z27.s, #0x0\n"
      "31:"  // Height 3: setup done
      "mov x26, #0x0\n"
      "32:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 33f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "ldr x22, [x20, #0x10]\n"
      "cbnz x26, 34f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "b 34f\n"
      "33:"  // Height 3: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "34:"  // Height 3: input setup done
      "cmp x25, #0x10\n"
      "ble 37f\n"
      "35:"  // Height 3: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x25\n"
      "ld1rqb { z0.b }, p0/Z, [x24]\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "add x24, x24, #0x10\n"
      "ld1rqb { z2.b }, p0/Z, [x22]\n"
      "ld1b { z28.b }, p2/Z, [x28]\n"
      "udot z16.s, z28.b, z0.b[0]\n"
      "udot z20.s, z28.b, z1.b[0]\n"
      "ld1b { z30.b }, p2/Z, [x28, #1, MUL VL]\n"
      "ld1b { z29.b }, p2/Z, [x28, #2, MUL VL]\n"
      "udot z24.s, z28.b, z2.b[0]\n"
      "udot z17.s, z30.b, z0.b[0]\n"
      "ld1b { z28.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z21.s, z30.b, z1.b[0]\n"
      "udot z25.s, z30.b, z2.b[0]\n"
      "ld1b { z3.b }, p2/Z, [x28, #4, MUL VL]\n"
      "udot z18.s, z29.b, z0.b[0]\n"
      "udot z22.s, z29.b, z1.b[0]\n"
      "ld1b { z31.b }, p2/Z, [x28, #5, MUL VL]\n"
      "ld1b { z30.b }, p2/Z, [x28, #6, MUL VL]\n"
      "udot z26.s, z29.b, z2.b[0]\n"
      "udot z19.s, z28.b, z0.b[0]\n"
      "ld1b { z29.b }, p2/Z, [x28, #7, MUL VL]\n"
      "addvl x28, x28, #16\n"
      "udot z23.s, z28.b, z1.b[0]\n"
      "udot z27.s, z28.b, z2.b[0]\n"
      "ld1b { z28.b }, p2/Z, [x28, #-8, MUL VL]\n"
      "ld1b { z5.b }, p2/Z, [x28, #-7, MUL VL]\n"
      "udot z16.s, z3.b, z0.b[1]\n"
      "udot z20.s, z3.b, z1.b[1]\n"
      "ld1b { z4.b }, p2/Z, [x28, #-6, MUL VL]\n"
      "add x23, x23, #0x10\n"
      "udot z24.s, z3.b, z2.b[1]\n"
      "udot z17.s, z31.b, z0.b[1]\n"
      "ld1b { z3.b }, p2/Z, [x28, #-5, MUL VL]\n"
      "add x22, x22, #0x10\n"
      "udot z21.s, z31.b, z1.b[1]\n"
      "udot z25.s, z31.b, z2.b[1]\n"
      "ld1b { z31.b }, p2/Z, [x28, #-4, MUL VL]\n"
      "udot z18.s, z30.b, z0.b[1]\n"
      "udot z22.s, z30.b, z1.b[1]\n"
      "udot z26.s, z30.b, z2.b[1]\n"
      "udot z19.s, z29.b, z0.b[1]\n"
      "ld1b { z30.b }, p2/Z, [x28, #-3, MUL VL]\n"
      "udot z23.s, z29.b, z1.b[1]\n"
      "udot z27.s, z29.b, z2.b[1]\n"
      "ld1b { z29.b }, p2/Z, [x28, #-2, MUL VL]\n"
      "udot z16.s, z28.b, z0.b[2]\n"
      "udot z20.s, z28.b, z1.b[2]\n"
      "udot z24.s, z28.b, z2.b[2]\n"
      "udot z17.s, z5.b, z0.b[2]\n"
      "ld1b { z28.b }, p2/Z, [x28, #-1, MUL VL]\n"
      "udot z21.s, z5.b, z1.b[2]\n"
      "udot z25.s, z5.b, z2.b[2]\n"
      "udot z18.s, z4.b, z0.b[2]\n"
      "udot z22.s, z4.b, z1.b[2]\n"
      "udot z26.s, z4.b, z2.b[2]\n"
      "udot z19.s, z3.b, z0.b[2]\n"
      "udot z23.s, z3.b, z1.b[2]\n"
      "udot z27.s, z3.b, z2.b[2]\n"
      "udot z16.s, z31.b, z0.b[3]\n"
      "udot z20.s, z31.b, z1.b[3]\n"
      "udot z24.s, z31.b, z2.b[3]\n"
      "udot z17.s, z30.b, z0.b[3]\n"
      "udot z21.s, z30.b, z1.b[3]\n"
      "udot z25.s, z30.b, z2.b[3]\n"
      "udot z18.s, z29.b, z0.b[3]\n"
      "udot z22.s, z29.b, z1.b[3]\n"
      "udot z26.s, z29.b, z2.b[3]\n"
      "udot z19.s, z28.b, z0.b[3]\n"
      "udot z23.s, z28.b, z1.b[3]\n"
      "udot z27.s, z28.b, z2.b[3]\n"
      "tbnz %x[flags], #31, 36f\n"
      "udot z11.s, z0.b, z15.b\n"
      "udot z12.s, z1.b, z15.b\n"
      "udot z13.s, z2.b, z15.b\n"
      "36:"  // Height 3: Multiply loop: unique 5: skip row sum
      "sub x25, x25, #0x10\n"
      "cmp x25, #0x10\n"
      "bgt 35b\n"
      "37:"  // Height 3: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x25\n"
      "ld1rqb { z0.b }, p0/Z, [x24]\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "subs x25, x25, #0x4\n"
      "ld1rqb { z2.b }, p0/Z, [x22]\n"
      "ld1b { z28.b }, p2/Z, [x28]\n"
      "udot z16.s, z28.b, z0.b[0]\n"
      "udot z20.s, z28.b, z1.b[0]\n"
      "ld1b { z30.b }, p2/Z, [x28, #1, MUL VL]\n"
      "ld1b { z29.b }, p2/Z, [x28, #2, MUL VL]\n"
      "udot z24.s, z28.b, z2.b[0]\n"
      "udot z17.s, z30.b, z0.b[0]\n"
      "ld1b { z28.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z21.s, z30.b, z1.b[0]\n"
      "udot z25.s, z30.b, z2.b[0]\n"
      "addvl x28, x28, #4\n"
      "udot z18.s, z29.b, z0.b[0]\n"
      "udot z22.s, z29.b, z1.b[0]\n"
      "udot z26.s, z29.b, z2.b[0]\n"
      "udot z19.s, z28.b, z0.b[0]\n"
      "udot z23.s, z28.b, z1.b[0]\n"
      "udot z27.s, z28.b, z2.b[0]\n"
      "ble 38f\n"
      "ld1b { z31.b }, p2/Z, [x28]\n"
      "ld1b { z30.b }, p2/Z, [x28, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "udot z16.s, z31.b, z0.b[1]\n"
      "ld1b { z29.b }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1b { z28.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z20.s, z31.b, z1.b[1]\n"
      "udot z24.s, z31.b, z2.b[1]\n"
      "udot z17.s, z30.b, z0.b[1]\n"
      "udot z21.s, z30.b, z1.b[1]\n"
      "addvl x28, x28, #4\n"
      "udot z25.s, z30.b, z2.b[1]\n"
      "udot z18.s, z29.b, z0.b[1]\n"
      "udot z22.s, z29.b, z1.b[1]\n"
      "udot z26.s, z29.b, z2.b[1]\n"
      "udot z19.s, z28.b, z0.b[1]\n"
      "udot z23.s, z28.b, z1.b[1]\n"
      "udot z27.s, z28.b, z2.b[1]\n"
      "ble 38f\n"
      "ld1b { z31.b }, p2/Z, [x28]\n"
      "ld1b { z30.b }, p2/Z, [x28, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "udot z16.s, z31.b, z0.b[2]\n"
      "ld1b { z29.b }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1b { z28.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z20.s, z31.b, z1.b[2]\n"
      "udot z24.s, z31.b, z2.b[2]\n"
      "udot z17.s, z30.b, z0.b[2]\n"
      "udot z21.s, z30.b, z1.b[2]\n"
      "addvl x28, x28, #4\n"
      "udot z25.s, z30.b, z2.b[2]\n"
      "udot z18.s, z29.b, z0.b[2]\n"
      "udot z22.s, z29.b, z1.b[2]\n"
      "udot z26.s, z29.b, z2.b[2]\n"
      "udot z19.s, z28.b, z0.b[2]\n"
      "udot z23.s, z28.b, z1.b[2]\n"
      "udot z27.s, z28.b, z2.b[2]\n"
      "ble 38f\n"
      "ld1b { z31.b }, p2/Z, [x28]\n"
      "ld1b { z30.b }, p2/Z, [x28, #1, MUL VL]\n"
      "udot z16.s, z31.b, z0.b[3]\n"
      "udot z20.s, z31.b, z1.b[3]\n"
      "ld1b { z29.b }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1b { z28.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z24.s, z31.b, z2.b[3]\n"
      "udot z17.s, z30.b, z0.b[3]\n"
      "udot z21.s, z30.b, z1.b[3]\n"
      "udot z25.s, z30.b, z2.b[3]\n"
      "addvl x28, x28, #4\n"
      "udot z18.s, z29.b, z0.b[3]\n"
      "udot z22.s, z29.b, z1.b[3]\n"
      "udot z26.s, z29.b, z2.b[3]\n"
      "udot z19.s, z28.b, z0.b[3]\n"
      "udot z23.s, z28.b, z1.b[3]\n"
      "udot z27.s, z28.b, z2.b[3]\n"
      "38:"  // Height 3: Multiply loop: multiply skip
      "tbnz %x[flags], #31, 39f\n"
      "udot z11.s, z0.b, z15.b\n"
      "udot z12.s, z1.b, z15.b\n"
      "udot z13.s, z2.b, z15.b\n"
      "39:"  // Height 3: Multiply loop: unique 6: skip row sum
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 32b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x23, x27, x20\n"
      "add x22, x23, x20\n"
      "tbnz %x[flags], #31, 40f\n"
      "mov x20, #0x4\n"
      "whilelt p0.s, XZR, x20\n"
      "add x20, %x[qp], %[b_offset]\n"
      "ld1rw { z28.s }, p2/Z, [x20]\n"
      "uaddv d11, p0, z11.s\n"
      "mov z11.s, z11.s[0]\n"
      "uaddv d12, p0, z12.s\n"
      "uaddv d13, p0, z13.s\n"
      "mov z12.s, z12.s[0]\n"
      "mov z13.s, z13.s[0]\n"
      "neg z28.s, p2/M, z28.s\n"
      "mul z11.s, p2/M, z11.s, z28.s\n"
      "mul z12.s, p2/M, z12.s, z28.s\n"
      "mul z13.s, p2/M, z13.s, z28.s\n"
      "40:"  // Height 3: skip row sum fixup
      "add z16.s, z16.s, z11.s\n"
      "add z17.s, z17.s, z11.s\n"
      "ld1w { z0.s }, p2/Z, [x10]\n"
      "ld1w { z31.s }, p2/Z, [x10, #1, MUL VL]\n"
      "add z18.s, z18.s, z11.s\n"
      "add z19.s, z19.s, z11.s\n"
      "ld1w { z30.s }, p2/Z, [x10, #2, MUL VL]\n"
      "ld1w { z29.s }, p2/Z, [x10, #3, MUL VL]\n"
      "add z20.s, z20.s, z12.s\n"
      "add z21.s, z21.s, z12.s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "add z22.s, z22.s, z12.s\n"
      "add z23.s, z23.s, z12.s\n"
      "ld1rw { z28.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "add z24.s, z24.s, z13.s\n"
      "add z25.s, z25.s, z13.s\n"
      "addvl x10, x10, #4\n"
      "add z26.s, z26.s, z13.s\n"
      "add z27.s, z27.s, z13.s\n"
      "add z16.s, z16.s, z0.s\n"
      "add z17.s, z17.s, z31.s\n"
      "add z18.s, z18.s, z30.s\n"
      "add z19.s, z19.s, z29.s\n"
      "add z20.s, z20.s, z0.s\n"
      "add z21.s, z21.s, z31.s\n"
      "add z22.s, z22.s, z30.s\n"
      "add z23.s, z23.s, z29.s\n"
      "add z24.s, z24.s, z0.s\n"
      "add z25.s, z25.s, z31.s\n"
      "ld1rw { z0.s }, p2/Z, [x20]\n"
      "add z26.s, z26.s, z30.s\n"
      "add z27.s, z27.s, z29.s\n"
      ".inst 0x04bc7610  // sqrdmulh z16.s, z16.s, z28.s\n"
      ".inst 0x04bc7631  // sqrdmulh z17.s, z17.s, z28.s\n"
      ".inst 0x04bc7652  // sqrdmulh z18.s, z18.s, z28.s\n"
      ".inst 0x04bc7673  // sqrdmulh z19.s, z19.s, z28.s\n"
      ".inst 0x04bc7694  // sqrdmulh z20.s, z20.s, z28.s\n"
      ".inst 0x04bc76b5  // sqrdmulh z21.s, z21.s, z28.s\n"
      ".inst 0x04bc76d6  // sqrdmulh z22.s, z22.s, z28.s\n"
      ".inst 0x04bc76f7  // sqrdmulh z23.s, z23.s, z28.s\n"
      ".inst 0x04bc7718  // sqrdmulh z24.s, z24.s, z28.s\n"
      ".inst 0x04bc7739  // sqrdmulh z25.s, z25.s, z28.s\n"
      ".inst 0x04bc775a  // sqrdmulh z26.s, z26.s, z28.s\n"
      ".inst 0x04bc777b  // sqrdmulh z27.s, z27.s, z28.s\n"
      "tbz %x[flags], #5, 41f\n"
      "and z1.d, z16.d, z0.d\n"
      "and z31.d, z17.d, z0.d\n"
      "and z30.d, z18.d, z0.d\n"
      "and z29.d, z19.d, z0.d\n"
      "and z28.d, z20.d, z0.d\n"
      "asr z1.s, z1.s, #0x1f\n"
      "asr z31.s, z31.s, #0x1f\n"
      "asr z30.s, z30.s, #0x1f\n"
      "asr z29.s, z29.s, #0x1f\n"
      "asr z28.s, z28.s, #0x1f\n"
      "sqadd z16.s, z16.s, z1.s\n"
      "sqadd z17.s, z17.s, z31.s\n"
      "sqadd z18.s, z18.s, z30.s\n"
      "sqadd z19.s, z19.s, z29.s\n"
      "sqadd z20.s, z20.s, z28.s\n"
      "and z3.d, z21.d, z0.d\n"
      "and z2.d, z22.d, z0.d\n"
      "and z1.d, z23.d, z0.d\n"
      "and z31.d, z24.d, z0.d\n"
      "and z30.d, z25.d, z0.d\n"
      "and z29.d, z26.d, z0.d\n"
      "and z28.d, z27.d, z0.d\n"
      "asr z3.s, z3.s, #0x1f\n"
      "asr z2.s, z2.s, #0x1f\n"
      "asr z1.s, z1.s, #0x1f\n"
      "asr z31.s, z31.s, #0x1f\n"
      "asr z30.s, z30.s, #0x1f\n"
      "asr z29.s, z29.s, #0x1f\n"
      "asr z28.s, z28.s, #0x1f\n"
      "sqadd z21.s, z21.s, z3.s\n"
      "sqadd z22.s, z22.s, z2.s\n"
      "sqadd z23.s, z23.s, z1.s\n"
      "sqadd z24.s, z24.s, z31.s\n"
      "sqadd z25.s, z25.s, z30.s\n"
      "sqadd z26.s, z26.s, z29.s\n"
      "sqadd z27.s, z27.s, z28.s\n"
      "41:"  // Height 3: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      "ld1rw { z28.s }, p2/Z, [x20]\n"
      ".inst 0x44828810  // srshl z16.s, p2/M, z16.s, z0.s\n"
      "add z16.s, z16.s, z28.s\n"
      ".inst 0x44828811  // srshl z17.s, p2/M, z17.s, z0.s\n"
      ".inst 0x44828812  // srshl z18.s, p2/M, z18.s, z0.s\n"
      "add z17.s, z17.s, z28.s\n"
      "add z18.s, z18.s, z28.s\n"
      ".inst 0x44828813  // srshl z19.s, p2/M, z19.s, z0.s\n"
      ".inst 0x44828814  // srshl z20.s, p2/M, z20.s, z0.s\n"
      "add z19.s, z19.s, z28.s\n"
      "add z20.s, z20.s, z28.s\n"
      ".inst 0x44828815  // srshl z21.s, p2/M, z21.s, z0.s\n"
      ".inst 0x44828816  // srshl z22.s, p2/M, z22.s, z0.s\n"
      "add z21.s, z21.s, z28.s\n"
      "add z22.s, z22.s, z28.s\n"
      ".inst 0x44828817  // srshl z23.s, p2/M, z23.s, z0.s\n"
      ".inst 0x44828818  // srshl z24.s, p2/M, z24.s, z0.s\n"
      "add z23.s, z23.s, z28.s\n"
      "add z24.s, z24.s, z28.s\n"
      ".inst 0x44828819  // srshl z25.s, p2/M, z25.s, z0.s\n"
      ".inst 0x4482881a  // srshl z26.s, p2/M, z26.s, z0.s\n"
      "add z25.s, z25.s, z28.s\n"
      "add z26.s, z26.s, z28.s\n"
      ".inst 0x4482881b  // srshl z27.s, p2/M, z27.s, z0.s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1rw { z29.s }, p2/Z, [x20]\n"
      "add z27.s, z27.s, z28.s\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1rw { z28.s }, p2/Z, [x20]\n"
      "smin z16.s, p2/M, z16.s, z29.s\n"
      "smin z17.s, p2/M, z17.s, z29.s\n"
      "smin z18.s, p2/M, z18.s, z29.s\n"
      "smin z19.s, p2/M, z19.s, z29.s\n"
      "smin z20.s, p2/M, z20.s, z29.s\n"
      "smin z21.s, p2/M, z21.s, z29.s\n"
      "smin z22.s, p2/M, z22.s, z29.s\n"
      "smin z23.s, p2/M, z23.s, z29.s\n"
      "smin z24.s, p2/M, z24.s, z29.s\n"
      "smin z25.s, p2/M, z25.s, z29.s\n"
      "smin z26.s, p2/M, z26.s, z29.s\n"
      "smin z27.s, p2/M, z27.s, z29.s\n"
      "smax z16.s, p2/M, z16.s, z28.s\n"
      "smax z17.s, p2/M, z17.s, z28.s\n"
      "smax z18.s, p2/M, z18.s, z28.s\n"
      "uzp1 z16.h, z16.h, z17.h\n"
      "smax z19.s, p2/M, z19.s, z28.s\n"
      "smax z20.s, p2/M, z20.s, z28.s\n"
      "uzp1 z17.h, z18.h, z19.h\n"
      "uzp1 z16.b, z16.b, z17.b\n"
      "smax z21.s, p2/M, z21.s, z28.s\n"
      "smax z22.s, p2/M, z22.s, z28.s\n"
      "uzp1 z20.h, z20.h, z21.h\n"
      "st1b { z16.b }, p1, [x27]\n"
      "smax z23.s, p2/M, z23.s, z28.s\n"
      "smax z24.s, p2/M, z24.s, z28.s\n"
      "uzp1 z16.h, z22.h, z23.h\n"
      "uzp1 z20.b, z20.b, z16.b\n"
      "smax z25.s, p2/M, z25.s, z28.s\n"
      "smax z26.s, p2/M, z26.s, z28.s\n"
      "uzp1 z24.h, z24.h, z25.h\n"
      "st1b { z20.b }, p1, [x23]\n"
      "smax z27.s, p2/M, z27.s, z28.s\n"
      "uzp1 z16.h, z26.h, z27.h\n"
      "uzp1 z24.b, z24.b, z16.b\n"
      "st1b { z24.b }, p1, [x22]\n"
      "addvl x27, x27, #1\n"
      "42:"  // Height 3: Writeback done
      "decw x9, ALL, MUL #4\n"
      "cmp x9, XZR\n"
      "bgt 30b\n"
      "b 58f\n"
      "43:"  // Height 4
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "mov x20, #0x4\n"
      "mov x10, %x[col_bias]\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "mov z13.s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "ldr x9, [%x[args_ptr], %[offsetof_N]]\n"
      "mov z14.s, #0x0\n"
      "mov z15.b, #0x1\n"
      "ldr x28, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x27, %x[output_ptr]\n"
      "madd %x[output_ptr], x21, x20, %x[output_ptr]\n"
      "44:"  // Height 4: Column loop
      "mov x20, #0x0\n"
      "whilelt p1.b, x20, x9\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "mov z20.s, #0x0\n"
      "mov z21.s, #0x0\n"
      "mov z22.s, #0x0\n"
      "mov z23.s, #0x0\n"
      "mov z24.s, #0x0\n"
      "mov z25.s, #0x0\n"
      "mov z26.s, #0x0\n"
      "mov z27.s, #0x0\n"
      "mov z28.s, #0x0\n"
      "mov z29.s, #0x0\n"
      "mov z30.s, #0x0\n"
      "mov z31.s, #0x0\n"
      "45:"  // Height 4: setup done
      "mov x26, #0x0\n"
      "46:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 47f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "ldr x22, [x20, #0x10]\n"
      "ldr x21, [x20, #0x18]\n"
      "cbnz x26, 48f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "add x21, x21, x20\n"
      "b 48f\n"
      "47:"  // Height 4: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "add x21, x22, x21\n"
      "48:"  // Height 4: input setup done
      "cmp x25, #0x10\n"
      "ble 51f\n"
      "49:"  // Height 4: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x25\n"
      "ld1rqb { z0.b }, p0/Z, [x24]\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "add x24, x24, #0x10\n"
      "ld1rqb { z2.b }, p0/Z, [x22]\n"
      "ld1rqb { z3.b }, p0/Z, [x21]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "ld1b { z5.b }, p2/Z, [x28]\n"
      "ld1b { z4.b }, p2/Z, [x28, #1, MUL VL]\n"
      "udot z16.s, z5.b, z0.b[0]\n"
      "udot z20.s, z5.b, z1.b[0]\n"
      "ld1b { z10.b }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1b { z9.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z24.s, z5.b, z2.b[0]\n"
      "udot z28.s, z5.b, z3.b[0]\n"
      "udot z17.s, z4.b, z0.b[0]\n"
      "udot z21.s, z4.b, z1.b[0]\n"
      "ld1b { z8.b }, p2/Z, [x28, #4, MUL VL]\n"
      "ld1b { z7.b }, p2/Z, [x28, #5, MUL VL]\n"
      "udot z25.s, z4.b, z2.b[0]\n"
      "udot z29.s, z4.b, z3.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x28, #6, MUL VL]\n"
      "ld1b { z5.b }, p2/Z, [x28, #7, MUL VL]\n"
      "udot z18.s, z10.b, z0.b[0]\n"
      "udot z22.s, z10.b, z1.b[0]\n"
      "addvl x28, x28, #16\n"
      "ld1b { z4.b }, p2/Z, [x28, #-8, MUL VL]\n"
      "udot z26.s, z10.b, z2.b[0]\n"
      "udot z30.s, z10.b, z3.b[0]\n"
      "ld1b { z10.b }, p2/Z, [x28, #-7, MUL VL]\n"
      "add x21, x21, #0x10\n"
      "udot z19.s, z9.b, z0.b[0]\n"
      "udot z23.s, z9.b, z1.b[0]\n"
      "udot z27.s, z9.b, z2.b[0]\n"
      "udot z31.s, z9.b, z3.b[0]\n"
      "ld1b { z9.b }, p2/Z, [x28, #-6, MUL VL]\n"
      "udot z16.s, z8.b, z0.b[1]\n"
      "udot z20.s, z8.b, z1.b[1]\n"
      "udot z24.s, z8.b, z2.b[1]\n"
      "udot z28.s, z8.b, z3.b[1]\n"
      "ld1b { z8.b }, p2/Z, [x28, #-5, MUL VL]\n"
      "udot z17.s, z7.b, z0.b[1]\n"
      "udot z21.s, z7.b, z1.b[1]\n"
      "udot z25.s, z7.b, z2.b[1]\n"
      "udot z29.s, z7.b, z3.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x28, #-4, MUL VL]\n"
      "udot z18.s, z6.b, z0.b[1]\n"
      "udot z22.s, z6.b, z1.b[1]\n"
      "udot z26.s, z6.b, z2.b[1]\n"
      "udot z30.s, z6.b, z3.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x28, #-3, MUL VL]\n"
      "udot z19.s, z5.b, z0.b[1]\n"
      "udot z23.s, z5.b, z1.b[1]\n"
      "udot z27.s, z5.b, z2.b[1]\n"
      "udot z31.s, z5.b, z3.b[1]\n"
      "ld1b { z5.b }, p2/Z, [x28, #-2, MUL VL]\n"
      "udot z16.s, z4.b, z0.b[2]\n"
      "udot z20.s, z4.b, z1.b[2]\n"
      "udot z24.s, z4.b, z2.b[2]\n"
      "udot z28.s, z4.b, z3.b[2]\n"
      "ld1b { z4.b }, p2/Z, [x28, #-1, MUL VL]\n"
      "udot z17.s, z10.b, z0.b[2]\n"
      "udot z21.s, z10.b, z1.b[2]\n"
      "udot z25.s, z10.b, z2.b[2]\n"
      "udot z29.s, z10.b, z3.b[2]\n"
      "udot z18.s, z9.b, z0.b[2]\n"
      "udot z22.s, z9.b, z1.b[2]\n"
      "udot z26.s, z9.b, z2.b[2]\n"
      "udot z30.s, z9.b, z3.b[2]\n"
      "udot z19.s, z8.b, z0.b[2]\n"
      "udot z23.s, z8.b, z1.b[2]\n"
      "udot z27.s, z8.b, z2.b[2]\n"
      "udot z31.s, z8.b, z3.b[2]\n"
      "udot z16.s, z7.b, z0.b[3]\n"
      "udot z20.s, z7.b, z1.b[3]\n"
      "udot z24.s, z7.b, z2.b[3]\n"
      "udot z28.s, z7.b, z3.b[3]\n"
      "udot z17.s, z6.b, z0.b[3]\n"
      "udot z21.s, z6.b, z1.b[3]\n"
      "udot z25.s, z6.b, z2.b[3]\n"
      "udot z29.s, z6.b, z3.b[3]\n"
      "udot z18.s, z5.b, z0.b[3]\n"
      "udot z22.s, z5.b, z1.b[3]\n"
      "udot z26.s, z5.b, z2.b[3]\n"
      "udot z30.s, z5.b, z3.b[3]\n"
      "udot z19.s, z4.b, z0.b[3]\n"
      "udot z23.s, z4.b, z1.b[3]\n"
      "udot z27.s, z4.b, z2.b[3]\n"
      "udot z31.s, z4.b, z3.b[3]\n"
      "tbnz %x[flags], #31, 50f\n"
      "udot z11.s, z0.b, z15.b\n"
      "udot z12.s, z1.b, z15.b\n"
      "udot z13.s, z2.b, z15.b\n"
      "udot z14.s, z3.b, z15.b\n"
      "50:"  // Height 4: Multiply loop: unique 7: skip row sum
      "sub x25, x25, #0x10\n"
      "cmp x25, #0x10\n"
      "bgt 49b\n"
      "51:"  // Height 4: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x25\n"
      "ld1rqb { z0.b }, p0/Z, [x24]\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "subs x25, x25, #0x4\n"
      "ld1rqb { z2.b }, p0/Z, [x22]\n"
      "ld1rqb { z3.b }, p0/Z, [x21]\n"
      "ld1b { z7.b }, p2/Z, [x28]\n"
      "ld1b { z6.b }, p2/Z, [x28, #1, MUL VL]\n"
      "udot z16.s, z7.b, z0.b[0]\n"
      "udot z20.s, z7.b, z1.b[0]\n"
      "ld1b { z5.b }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1b { z4.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z24.s, z7.b, z2.b[0]\n"
      "udot z28.s, z7.b, z3.b[0]\n"
      "udot z17.s, z6.b, z0.b[0]\n"
      "udot z21.s, z6.b, z1.b[0]\n"
      "addvl x28, x28, #4\n"
      "udot z25.s, z6.b, z2.b[0]\n"
      "udot z29.s, z6.b, z3.b[0]\n"
      "udot z18.s, z5.b, z0.b[0]\n"
      "udot z22.s, z5.b, z1.b[0]\n"
      "udot z26.s, z5.b, z2.b[0]\n"
      "udot z30.s, z5.b, z3.b[0]\n"
      "udot z19.s, z4.b, z0.b[0]\n"
      "udot z23.s, z4.b, z1.b[0]\n"
      "udot z27.s, z4.b, z2.b[0]\n"
      "udot z31.s, z4.b, z3.b[0]\n"
      "ble 52f\n"
      "ld1b { z7.b }, p2/Z, [x28]\n"
      "ld1b { z6.b }, p2/Z, [x28, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "udot z16.s, z7.b, z0.b[1]\n"
      "ld1b { z5.b }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1b { z4.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z20.s, z7.b, z1.b[1]\n"
      "udot z24.s, z7.b, z2.b[1]\n"
      "udot z28.s, z7.b, z3.b[1]\n"
      "udot z17.s, z6.b, z0.b[1]\n"
      "addvl x28, x28, #4\n"
      "udot z21.s, z6.b, z1.b[1]\n"
      "udot z25.s, z6.b, z2.b[1]\n"
      "udot z29.s, z6.b, z3.b[1]\n"
      "udot z18.s, z5.b, z0.b[1]\n"
      "udot z22.s, z5.b, z1.b[1]\n"
      "udot z26.s, z5.b, z2.b[1]\n"
      "udot z30.s, z5.b, z3.b[1]\n"
      "udot z19.s, z4.b, z0.b[1]\n"
      "udot z23.s, z4.b, z1.b[1]\n"
      "udot z27.s, z4.b, z2.b[1]\n"
      "udot z31.s, z4.b, z3.b[1]\n"
      "ble 52f\n"
      "ld1b { z7.b }, p2/Z, [x28]\n"
      "ld1b { z6.b }, p2/Z, [x28, #1, MUL VL]\n"
      "subs x25, x25, #0x4\n"
      "udot z16.s, z7.b, z0.b[2]\n"
      "ld1b { z5.b }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1b { z4.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z20.s, z7.b, z1.b[2]\n"
      "udot z24.s, z7.b, z2.b[2]\n"
      "udot z28.s, z7.b, z3.b[2]\n"
      "udot z17.s, z6.b, z0.b[2]\n"
      "addvl x28, x28, #4\n"
      "udot z21.s, z6.b, z1.b[2]\n"
      "udot z25.s, z6.b, z2.b[2]\n"
      "udot z29.s, z6.b, z3.b[2]\n"
      "udot z18.s, z5.b, z0.b[2]\n"
      "udot z22.s, z5.b, z1.b[2]\n"
      "udot z26.s, z5.b, z2.b[2]\n"
      "udot z30.s, z5.b, z3.b[2]\n"
      "udot z19.s, z4.b, z0.b[2]\n"
      "udot z23.s, z4.b, z1.b[2]\n"
      "udot z27.s, z4.b, z2.b[2]\n"
      "udot z31.s, z4.b, z3.b[2]\n"
      "ble 52f\n"
      "ld1b { z7.b }, p2/Z, [x28]\n"
      "ld1b { z6.b }, p2/Z, [x28, #1, MUL VL]\n"
      "udot z16.s, z7.b, z0.b[3]\n"
      "udot z20.s, z7.b, z1.b[3]\n"
      "ld1b { z5.b }, p2/Z, [x28, #2, MUL VL]\n"
      "ld1b { z4.b }, p2/Z, [x28, #3, MUL VL]\n"
      "udot z24.s, z7.b, z2.b[3]\n"
      "udot z28.s, z7.b, z3.b[3]\n"
      "udot z17.s, z6.b, z0.b[3]\n"
      "udot z21.s, z6.b, z1.b[3]\n"
      "addvl x28, x28, #4\n"
      "udot z25.s, z6.b, z2.b[3]\n"
      "udot z29.s, z6.b, z3.b[3]\n"
      "udot z18.s, z5.b, z0.b[3]\n"
      "udot z22.s, z5.b, z1.b[3]\n"
      "udot z26.s, z5.b, z2.b[3]\n"
      "udot z30.s, z5.b, z3.b[3]\n"
      "udot z19.s, z4.b, z0.b[3]\n"
      "udot z23.s, z4.b, z1.b[3]\n"
      "udot z27.s, z4.b, z2.b[3]\n"
      "udot z31.s, z4.b, z3.b[3]\n"
      "52:"  // Height 4: Multiply loop: multiply skip
      "tbnz %x[flags], #31, 53f\n"
      "udot z11.s, z0.b, z15.b\n"
      "udot z12.s, z1.b, z15.b\n"
      "udot z13.s, z2.b, z15.b\n"
      "udot z14.s, z3.b, z15.b\n"
      "53:"  // Height 4: Multiply loop: unique 8: skip row sum
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 46b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x23, x27, x20\n"
      "add x22, x23, x20\n"
      "add x21, x22, x20\n"
      "tbnz %x[flags], #31, 54f\n"
      "mov x20, #0x4\n"
      "whilelt p0.s, XZR, x20\n"
      "add x20, %x[qp], %[b_offset]\n"
      "ld1rw { z0.s }, p2/Z, [x20]\n"
      "uaddv d11, p0, z11.s\n"
      "mov z11.s, z11.s[0]\n"
      "uaddv d12, p0, z12.s\n"
      "uaddv d13, p0, z13.s\n"
      "mov z12.s, z12.s[0]\n"
      "mov z13.s, z13.s[0]\n"
      "uaddv d14, p0, z14.s\n"
      "neg z0.s, p2/M, z0.s\n"
      "mov z14.s, z14.s[0]\n"
      "mul z11.s, p2/M, z11.s, z0.s\n"
      "mul z12.s, p2/M, z12.s, z0.s\n"
      "mul z13.s, p2/M, z13.s, z0.s\n"
      "mul z14.s, p2/M, z14.s, z0.s\n"
      "54:"  // Height 4: skip row sum fixup
      "add z16.s, z16.s, z11.s\n"
      "add z17.s, z17.s, z11.s\n"
      "ld1w { z4.s }, p2/Z, [x10]\n"
      "ld1w { z0.s }, p2/Z, [x10, #1, MUL VL]\n"
      "add z18.s, z18.s, z11.s\n"
      "add z19.s, z19.s, z11.s\n"
      "ld1w { z3.s }, p2/Z, [x10, #2, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x10, #3, MUL VL]\n"
      "add z20.s, z20.s, z12.s\n"
      "add z21.s, z21.s, z12.s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "add z22.s, z22.s, z12.s\n"
      "add z23.s, z23.s, z12.s\n"
      "ld1rw { z1.s }, p2/Z, [x20]\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "add z24.s, z24.s, z13.s\n"
      "add z25.s, z25.s, z13.s\n"
      "addvl x10, x10, #4\n"
      "add z26.s, z26.s, z13.s\n"
      "add z27.s, z27.s, z13.s\n"
      "add z28.s, z28.s, z14.s\n"
      "add z29.s, z29.s, z14.s\n"
      "add z30.s, z30.s, z14.s\n"
      "add z31.s, z31.s, z14.s\n"
      "add z16.s, z16.s, z4.s\n"
      "add z17.s, z17.s, z0.s\n"
      "add z18.s, z18.s, z3.s\n"
      "add z19.s, z19.s, z2.s\n"
      "add z20.s, z20.s, z4.s\n"
      "add z21.s, z21.s, z0.s\n"
      "add z22.s, z22.s, z3.s\n"
      "add z23.s, z23.s, z2.s\n"
      "add z24.s, z24.s, z4.s\n"
      "add z25.s, z25.s, z0.s\n"
      "add z26.s, z26.s, z3.s\n"
      "add z27.s, z27.s, z2.s\n"
      "add z28.s, z28.s, z4.s\n"
      "add z29.s, z29.s, z0.s\n"
      "ld1rw { z0.s }, p2/Z, [x20]\n"
      "add z30.s, z30.s, z3.s\n"
      "add z31.s, z31.s, z2.s\n"
      ".inst 0x04a17610  // sqrdmulh z16.s, z16.s, z1.s\n"
      ".inst 0x04a17631  // sqrdmulh z17.s, z17.s, z1.s\n"
      ".inst 0x04a17652  // sqrdmulh z18.s, z18.s, z1.s\n"
      ".inst 0x04a17673  // sqrdmulh z19.s, z19.s, z1.s\n"
      ".inst 0x04a17694  // sqrdmulh z20.s, z20.s, z1.s\n"
      ".inst 0x04a176b5  // sqrdmulh z21.s, z21.s, z1.s\n"
      ".inst 0x04a176d6  // sqrdmulh z22.s, z22.s, z1.s\n"
      ".inst 0x04a176f7  // sqrdmulh z23.s, z23.s, z1.s\n"
      ".inst 0x04a17718  // sqrdmulh z24.s, z24.s, z1.s\n"
      ".inst 0x04a17739  // sqrdmulh z25.s, z25.s, z1.s\n"
      ".inst 0x04a1775a  // sqrdmulh z26.s, z26.s, z1.s\n"
      ".inst 0x04a1777b  // sqrdmulh z27.s, z27.s, z1.s\n"
      ".inst 0x04a1779c  // sqrdmulh z28.s, z28.s, z1.s\n"
      ".inst 0x04a177bd  // sqrdmulh z29.s, z29.s, z1.s\n"
      ".inst 0x04a177de  // sqrdmulh z30.s, z30.s, z1.s\n"
      ".inst 0x04a177ff  // sqrdmulh z31.s, z31.s, z1.s\n"
      "tbz %x[flags], #5, 55f\n"
      "and z2.d, z16.d, z0.d\n"
      "and z1.d, z17.d, z0.d\n"
      "asr z2.s, z2.s, #0x1f\n"
      "asr z1.s, z1.s, #0x1f\n"
      "sqadd z16.s, z16.s, z2.s\n"
      "sqadd z17.s, z17.s, z1.s\n"
      "and z7.d, z18.d, z0.d\n"
      "and z6.d, z19.d, z0.d\n"
      "and z5.d, z20.d, z0.d\n"
      "and z4.d, z21.d, z0.d\n"
      "and z3.d, z22.d, z0.d\n"
      "and z2.d, z23.d, z0.d\n"
      "and z1.d, z24.d, z0.d\n"
      "asr z7.s, z7.s, #0x1f\n"
      "asr z6.s, z6.s, #0x1f\n"
      "asr z5.s, z5.s, #0x1f\n"
      "asr z4.s, z4.s, #0x1f\n"
      "asr z3.s, z3.s, #0x1f\n"
      "asr z2.s, z2.s, #0x1f\n"
      "asr z1.s, z1.s, #0x1f\n"
      "sqadd z18.s, z18.s, z7.s\n"
      "sqadd z19.s, z19.s, z6.s\n"
      "sqadd z20.s, z20.s, z5.s\n"
      "sqadd z21.s, z21.s, z4.s\n"
      "sqadd z22.s, z22.s, z3.s\n"
      "sqadd z23.s, z23.s, z2.s\n"
      "sqadd z24.s, z24.s, z1.s\n"
      "and z7.d, z25.d, z0.d\n"
      "and z6.d, z26.d, z0.d\n"
      "and z5.d, z27.d, z0.d\n"
      "and z4.d, z28.d, z0.d\n"
      "and z3.d, z29.d, z0.d\n"
      "and z2.d, z30.d, z0.d\n"
      "and z1.d, z31.d, z0.d\n"
      "asr z7.s, z7.s, #0x1f\n"
      "asr z6.s, z6.s, #0x1f\n"
      "asr z5.s, z5.s, #0x1f\n"
      "asr z4.s, z4.s, #0x1f\n"
      "asr z3.s, z3.s, #0x1f\n"
      "asr z2.s, z2.s, #0x1f\n"
      "asr z1.s, z1.s, #0x1f\n"
      "sqadd z25.s, z25.s, z7.s\n"
      "sqadd z26.s, z26.s, z6.s\n"
      "sqadd z27.s, z27.s, z5.s\n"
      "sqadd z28.s, z28.s, z4.s\n"
      "sqadd z29.s, z29.s, z3.s\n"
      "sqadd z30.s, z30.s, z2.s\n"
      "sqadd z31.s, z31.s, z1.s\n"
      "55:"  // Height 4: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      "ld1rw { z2.s }, p2/Z, [x20]\n"
      ".inst 0x44828810  // srshl z16.s, p2/M, z16.s, z0.s\n"
      "add z16.s, z16.s, z2.s\n"
      ".inst 0x44828811  // srshl z17.s, p2/M, z17.s, z0.s\n"
      ".inst 0x44828812  // srshl z18.s, p2/M, z18.s, z0.s\n"
      "add z17.s, z17.s, z2.s\n"
      "add z18.s, z18.s, z2.s\n"
      ".inst 0x44828813  // srshl z19.s, p2/M, z19.s, z0.s\n"
      ".inst 0x44828814  // srshl z20.s, p2/M, z20.s, z0.s\n"
      "add z19.s, z19.s, z2.s\n"
      "add z20.s, z20.s, z2.s\n"
      ".inst 0x44828815  // srshl z21.s, p2/M, z21.s, z0.s\n"
      ".inst 0x44828816  // srshl z22.s, p2/M, z22.s, z0.s\n"
      "add z21.s, z21.s, z2.s\n"
      "add z22.s, z22.s, z2.s\n"
      ".inst 0x44828817  // srshl z23.s, p2/M, z23.s, z0.s\n"
      ".inst 0x44828818  // srshl z24.s, p2/M, z24.s, z0.s\n"
      "add z23.s, z23.s, z2.s\n"
      "add z24.s, z24.s, z2.s\n"
      ".inst 0x44828819  // srshl z25.s, p2/M, z25.s, z0.s\n"
      ".inst 0x4482881a  // srshl z26.s, p2/M, z26.s, z0.s\n"
      "add z25.s, z25.s, z2.s\n"
      "add z26.s, z26.s, z2.s\n"
      ".inst 0x4482881b  // srshl z27.s, p2/M, z27.s, z0.s\n"
      ".inst 0x4482881c  // srshl z28.s, p2/M, z28.s, z0.s\n"
      "add z27.s, z27.s, z2.s\n"
      "add z28.s, z28.s, z2.s\n"
      ".inst 0x4482881d  // srshl z29.s, p2/M, z29.s, z0.s\n"
      ".inst 0x4482881e  // srshl z30.s, p2/M, z30.s, z0.s\n"
      "add z29.s, z29.s, z2.s\n"
      "add z30.s, z30.s, z2.s\n"
      ".inst 0x4482881f  // srshl z31.s, p2/M, z31.s, z0.s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1rw { z1.s }, p2/Z, [x20]\n"
      "add z31.s, z31.s, z2.s\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1rw { z0.s }, p2/Z, [x20]\n"
      "smin z16.s, p2/M, z16.s, z1.s\n"
      "smin z17.s, p2/M, z17.s, z1.s\n"
      "smin z18.s, p2/M, z18.s, z1.s\n"
      "smin z19.s, p2/M, z19.s, z1.s\n"
      "smin z20.s, p2/M, z20.s, z1.s\n"
      "smin z21.s, p2/M, z21.s, z1.s\n"
      "smin z22.s, p2/M, z22.s, z1.s\n"
      "smin z23.s, p2/M, z23.s, z1.s\n"
      "smin z24.s, p2/M, z24.s, z1.s\n"
      "smin z25.s, p2/M, z25.s, z1.s\n"
      "smin z26.s, p2/M, z26.s, z1.s\n"
      "smin z27.s, p2/M, z27.s, z1.s\n"
      "smin z28.s, p2/M, z28.s, z1.s\n"
      "smin z29.s, p2/M, z29.s, z1.s\n"
      "smin z30.s, p2/M, z30.s, z1.s\n"
      "smin z31.s, p2/M, z31.s, z1.s\n"
      "smax z16.s, p2/M, z16.s, z0.s\n"
      "smax z17.s, p2/M, z17.s, z0.s\n"
      "smax z18.s, p2/M, z18.s, z0.s\n"
      "uzp1 z16.h, z16.h, z17.h\n"
      "smax z19.s, p2/M, z19.s, z0.s\n"
      "smax z20.s, p2/M, z20.s, z0.s\n"
      "uzp1 z17.h, z18.h, z19.h\n"
      "uzp1 z16.b, z16.b, z17.b\n"
      "smax z21.s, p2/M, z21.s, z0.s\n"
      "smax z22.s, p2/M, z22.s, z0.s\n"
      "uzp1 z20.h, z20.h, z21.h\n"
      "st1b { z16.b }, p1, [x27]\n"
      "smax z23.s, p2/M, z23.s, z0.s\n"
      "smax z24.s, p2/M, z24.s, z0.s\n"
      "uzp1 z16.h, z22.h, z23.h\n"
      "uzp1 z20.b, z20.b, z16.b\n"
      "smax z25.s, p2/M, z25.s, z0.s\n"
      "smax z26.s, p2/M, z26.s, z0.s\n"
      "uzp1 z24.h, z24.h, z25.h\n"
      "st1b { z20.b }, p1, [x23]\n"
      "smax z27.s, p2/M, z27.s, z0.s\n"
      "smax z28.s, p2/M, z28.s, z0.s\n"
      "uzp1 z16.h, z26.h, z27.h\n"
      "uzp1 z24.b, z24.b, z16.b\n"
      "smax z29.s, p2/M, z29.s, z0.s\n"
      "smax z30.s, p2/M, z30.s, z0.s\n"
      "uzp1 z28.h, z28.h, z29.h\n"
      "st1b { z24.b }, p1, [x22]\n"
      "smax z31.s, p2/M, z31.s, z0.s\n"
      "uzp1 z16.h, z30.h, z31.h\n"
      "uzp1 z28.b, z28.b, z16.b\n"
      "st1b { z28.b }, p1, [x21]\n"
      "addvl x27, x27, #1\n"
      "56:"  // Height 4: Writeback done
      "decw x9, ALL, MUL #4\n"
      "cmp x9, XZR\n"
      "bgt 44b\n"
      "subs %x[M], %x[M], #0x4\n"
      "beq 58f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 57f\n"
      "add x21, x21, #0x4\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "57:"  // Update direct input
      "mov x20, #0x4\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "58:"  // Exit
      : [M] "+&r" (M), [flags] "+&r" (flags), [input_ptr] "+&r" (input_ptr), [output_ptr] "+&r" (output_ptr)
      : [args_ptr] "r" (&ka), [b_offset] "I" (offsetof(Requantize32, b_offset)), [c_offset] "I" (offsetof(Requantize32, c_offset)), [col_bias] "r" (col_bias), [maxval] "I" (offsetof(Requantize32, maxval)), [minval] "I" (offsetof(Requantize32, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths)), [per_layer_mul] "I" (offsetof(Requantize32, per_layer_mul)), [per_layer_right_shift] "I" (offsetof(Requantize32, per_layer_right_shift)), [qp] "r" (qp)
      : "cc", "memory", "p0", "p1", "p2", "x9", "x10", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm
#endif  // ARM_COMPUTE_ENABLE_SVE
