//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Device/Detector/SphericalDetector.cpp
//! @brief     Implements class SphericalDetector.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "Device/Detector/SphericalDetector.h"
#include "Base/Axis/MakeScale.h"
#include "Base/Axis/Scale.h"
#include "Base/Const/Units.h"
#include "Base/Pixel/SphericalPixel.h"
#include "Base/Util/Assert.h"
#include "Device/Beam/Beam.h"
#include "Device/Coord/CoordSystem2D.h"
#include "Device/Resolution/IDetectorResolution.h"
#include <numbers>

using std::numbers::pi;

SphericalDetector::SphericalDetector(std::array<std::shared_ptr<Scale>, 2> axes)
    : IDetector(axes)
{
}

SphericalDetector::SphericalDetector(size_t n_phi, double phi_min, double phi_max, size_t n_alpha,
                                     double alpha_min, double alpha_max)
    : SphericalDetector(std::array<std::shared_ptr<Scale>, 2>{
        sharedEquiDivision("phi_f", n_phi, phi_min, phi_max),
        sharedEquiDivision("alpha_f", n_alpha, alpha_min, alpha_max)})
{
}

SphericalDetector::SphericalDetector(size_t n_bin, double width, double phi, double alpha)
    : SphericalDetector(n_bin, phi - width / 2, phi + width / 2, n_bin, alpha - width / 2,
                        alpha + width / 2)
{
}

SphericalDetector::SphericalDetector(const SphericalDetector& other) = default;

SphericalDetector* SphericalDetector::clone() const
{
    return new SphericalDetector(*this);
}

IPixel* SphericalDetector::createPixel(size_t index) const
{
    const Scale& phi_axis = axis(0);
    const Scale& alpha_axis = axis(1);
    const size_t phi_index = axisBinIndex(index, 0);
    const size_t alpha_index = axisBinIndex(index, 1);

    const Bin1D alpha_bin = alpha_axis.bin(alpha_index);
    const Bin1D phi_bin = phi_axis.bin(phi_index);
    return new SphericalPixel(alpha_bin, phi_bin);
}

size_t SphericalDetector::indexOfSpecular(const Beam& beam) const
{
    double alpha = beam.alpha_i();
    double phi = beam.phi_i();
    const Scale& phi_axis = axis(0);
    const Scale& alpha_axis = axis(1);
    if (phi_axis.rangeComprises(phi) && alpha_axis.rangeComprises(alpha))
        return getGlobalIndex(phi_axis.closestIndex(phi), alpha_axis.closestIndex(alpha));
    return totalSize();
}

const CoordSystem2D* SphericalDetector::scatteringCoords(const Beam& beam) const
{
    return new SphericalCoords(axesClippedToRegionOfInterest(), beam.ki());
}
