//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Beam/DistributionItemCatalog.cpp
//! @brief     Implements class DistributionItemCatalog
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Beam/DistributionItemCatalog.h"
#include "Base/Util/Assert.h"
#include "GUI/Model/Descriptor/DistributionItems.h"

DistributionItem* DistributionItemCatalog::create(Type type)
{
    switch (type) {
    case Type::None:
        return new DistributionNoneItem();
    case Type::Gate:
        return new DistributionGateItem();
    case Type::Lorentz:
        return new DistributionLorentzItem();
    case Type::Gaussian:
        return new DistributionGaussianItem();
    case Type::LogNormal:
        return new DistributionLogNormalItem();
    case Type::Cosine:
        return new DistributionCosineItem();
    case Type::Trapezoid:
        return new DistributionTrapezoidItem();
    }
    ASSERT(false);
}

QVector<DistributionItemCatalog::Type> DistributionItemCatalog::types()
{
    return {Type::None,      Type::Gate,   Type::Lorentz,  Type::Gaussian,
            Type::LogNormal, Type::Cosine, Type::Trapezoid};
}

QVector<DistributionItemCatalog::Type> DistributionItemCatalog::symmetricTypes()
{
    return {Type::None, Type::Lorentz, Type::Gaussian, Type::Cosine};
}

UiInfo DistributionItemCatalog::uiInfo(Type type)
{
    switch (type) {
    case Type::None:
        return {"None", "", ""};
    case Type::Gate:
        return {"Gate", "", ""};
    case Type::Lorentz:
        return {"Lorentz", "", ""};
    case Type::Gaussian:
        return {"Gaussian", "", ""};
    case Type::LogNormal:
        return {"Log Normal", "", ""};
    case Type::Cosine:
        return {"Cosine", "", ""};
    case Type::Trapezoid:
        return {"Trapezoid", "", ""};
    }
    ASSERT(false);
}

DistributionItemCatalog::Type DistributionItemCatalog::type(const DistributionItem* item)
{
    if (dynamic_cast<const DistributionNoneItem*>(item))
        return Type::None;
    if (dynamic_cast<const DistributionGateItem*>(item))
        return Type::Gate;
    if (dynamic_cast<const DistributionLorentzItem*>(item))
        return Type::Lorentz;
    if (dynamic_cast<const DistributionGaussianItem*>(item))
        return Type::Gaussian;
    if (dynamic_cast<const DistributionLogNormalItem*>(item))
        return Type::LogNormal;
    if (dynamic_cast<const DistributionCosineItem*>(item))
        return Type::Cosine;
    if (dynamic_cast<const DistributionTrapezoidItem*>(item))
        return Type::Trapezoid;

    ASSERT(false);
}
