//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/InterferenceItems.h
//! @brief     Defines InterferenceItems's classes
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_MODEL_SAMPLE_INTERFERENCEITEMS_H
#define BORNAGAIN_GUI_MODEL_SAMPLE_INTERFERENCEITEMS_H

#include "GUI/Model/Descriptor/DoubleProperty.h"
#include "GUI/Model/Descriptor/SelectionProperty.h"
#include "GUI/Model/Sample/Lattice2DItemCatalog.h"
#include "GUI/Model/Sample/Lattice2DItems.h"
#include "GUI/Model/Sample/ProfileItemCatalogs.h"
#include "GUI/Model/Sample/ProfileItems.h"
#include <memory>

class IInterference;

class InterferenceItem {
public:
    virtual ~InterferenceItem() = default;
    virtual std::unique_ptr<IInterference> createInterference() const = 0;

    virtual void writeTo(QXmlStreamWriter* w) const;
    virtual void readFrom(QXmlStreamReader* r);

    DoubleProperty& positionVariance() { return m_positionVariance; }
    const DoubleProperty& positionVariance() const { return m_positionVariance; }
    void setPositionVariance(double v) { m_positionVariance.setValue(v); }

protected:
    InterferenceItem();

    DoubleProperty m_positionVariance;
};

// ------------------------------------------------------------------------------------------------

class Interference1DLatticeItem : public InterferenceItem {
public:
    Interference1DLatticeItem();

    std::unique_ptr<IInterference> createInterference() const override;

    void writeTo(QXmlStreamWriter* w) const override;
    void readFrom(QXmlStreamReader* r) override;

    DoubleProperty& length() { return m_length; }
    const DoubleProperty& length() const { return m_length; }
    void setLength(double v) { m_length.setValue(v); }

    DoubleProperty& rotationAngle() { return m_rotationAngle; }
    const DoubleProperty& rotationAngle() const { return m_rotationAngle; }
    void setRotationAngle(double v) { m_rotationAngle.setValue(v); }

    SelectionProperty<Profile1DItemCatalog>& decayFunctionSelection() { return m_decayFunction; }
    void setDecayFunctionType(Profile1DItem* p) { m_decayFunction.setCurrentItem(p); }

private:
    DoubleProperty m_length;
    DoubleProperty m_rotationAngle;
    SelectionProperty<Profile1DItemCatalog> m_decayFunction;
};

// ------------------------------------------------------------------------------------------------

class Interference2DAbstractLatticeItem : public InterferenceItem {
public:
    Lattice2DItem* latticeTypeItem() const { return m_latticeType.currentItem(); }
    SelectionProperty<Lattice2DItemCatalog>& latticeTypeSelection() { return m_latticeType; }
    void setLatticeType(Lattice2DItem* p) { m_latticeType.setCurrentItem(p); }

    bool xiIntegration() const { return m_xiIntegration; }
    void setXiIntegration(bool b) { m_xiIntegration = b; }

    void writeTo(QXmlStreamWriter* w) const override;
    void readFrom(QXmlStreamReader* r) override;

protected:
    explicit Interference2DAbstractLatticeItem(bool xiIntegration);

    bool m_xiIntegration;
    SelectionProperty<Lattice2DItemCatalog> m_latticeType;
};

// ------------------------------------------------------------------------------------------------

class Interference2DLatticeItem : public Interference2DAbstractLatticeItem {
public:
    Interference2DLatticeItem();
    std::unique_ptr<IInterference> createInterference() const override;

    void writeTo(QXmlStreamWriter* w) const override;
    void readFrom(QXmlStreamReader* r) override;

    SelectionProperty<Profile2DItemCatalog>& decayFunctionSelection() { return m_decayFunction; }
    void setDecayFunctionType(Profile2DItem* p) { m_decayFunction.setCurrentItem(p); }

protected:
    SelectionProperty<Profile2DItemCatalog> m_decayFunction;
};

// ------------------------------------------------------------------------------------------------

class Interference2DParacrystalItem : public Interference2DAbstractLatticeItem {
public:
    Interference2DParacrystalItem();
    std::unique_ptr<IInterference> createInterference() const override;

    void writeTo(QXmlStreamWriter* w) const override;
    void readFrom(QXmlStreamReader* r) override;

    DoubleProperty& dampingLength() { return m_dampingLength; }
    const DoubleProperty& dampingLength() const { return m_dampingLength; }
    void setDampingLength(double dampingLength) { m_dampingLength.setValue(dampingLength); }

    DoubleProperty& domainSize1() { return m_domainSize1; }
    const DoubleProperty& domainSize1() const { return m_domainSize1; }
    void setDomainSize1(double size) { m_domainSize1.setValue(size); }

    DoubleProperty& domainSize2() { return m_domainSize2; }
    const DoubleProperty& domainSize2() const { return m_domainSize2; }
    void setDomainSize2(double size) { m_domainSize2.setValue(size); }

    SelectionProperty<Profile2DItemCatalog>& probabilityDistributionSelection1() { return m_pdf1; }
    void setPDF1Type(Profile2DItem* p) { m_pdf1.setCurrentItem(p); }

    SelectionProperty<Profile2DItemCatalog>& probabilityDistributionSelection2() { return m_pdf2; }
    void setPDF2Type(Profile2DItem* p) { m_pdf2.setCurrentItem(p); }

private:
    DoubleProperty m_dampingLength;
    DoubleProperty m_domainSize1;
    DoubleProperty m_domainSize2;
    SelectionProperty<Profile2DItemCatalog> m_pdf1;
    SelectionProperty<Profile2DItemCatalog> m_pdf2;
};

// ------------------------------------------------------------------------------------------------

class InterferenceFinite2DLatticeItem : public Interference2DAbstractLatticeItem {
public:
    InterferenceFinite2DLatticeItem();
    std::unique_ptr<IInterference> createInterference() const override;

    void writeTo(QXmlStreamWriter* w) const override;
    void readFrom(QXmlStreamReader* r) override;

    uint domainSize1() const { return m_domainSize1; }
    void setDomainSize1(uint v) { m_domainSize1 = v; }

    uint domainSize2() const { return m_domainSize2; }
    void setDomainSize2(uint v) { m_domainSize2 = v; }

private:
    uint m_domainSize1 = 100;
    uint m_domainSize2 = 100;
};

// ------------------------------------------------------------------------------------------------

class InterferenceHardDiskItem : public InterferenceItem {
public:
    InterferenceHardDiskItem();
    std::unique_ptr<IInterference> createInterference() const override;

    void writeTo(QXmlStreamWriter* w) const override;
    void readFrom(QXmlStreamReader* r) override;

    DoubleProperty& radius() { return m_radius; }
    const DoubleProperty& radius() const { return m_radius; }
    void setRadius(double v) { m_radius.setValue(v); }

    DoubleProperty& density() { return m_density; }
    const DoubleProperty& density() const { return m_density; }
    void setDensity(double v) { m_density.setValue(v); }

private:
    DoubleProperty m_radius;
    DoubleProperty m_density;
};

// ------------------------------------------------------------------------------------------------

class InterferenceRadialParacrystalItem : public InterferenceItem {
public:
    InterferenceRadialParacrystalItem();
    std::unique_ptr<IInterference> createInterference() const override;

    void writeTo(QXmlStreamWriter* w) const override;
    void readFrom(QXmlStreamReader* r) override;

    DoubleProperty& peakDistance() { return m_peakDistance; }
    const DoubleProperty& peakDistance() const { return m_peakDistance; }
    void setPeakDistance(double v) { m_peakDistance.setValue(v); }

    DoubleProperty& dampingLength() { return m_dampingLength; }
    const DoubleProperty& dampingLength() const { return m_dampingLength; }
    void setDampingLength(double v) { m_dampingLength.setValue(v); }

    DoubleProperty& domainSize() { return m_domainSize; }
    const DoubleProperty& domainSize() const { return m_domainSize; }
    void setDomainSize(double v) { m_domainSize.setValue(v); }

    DoubleProperty& kappa() { return m_kappa; }
    const DoubleProperty& kappa() const { return m_kappa; }
    void setKappa(double v) { m_kappa.setValue(v); }

    SelectionProperty<Profile1DItemCatalog>& probabilityDistributionSelection() { return m_pdf; }
    void setPDFType(Profile1DItem* p) { m_pdf.setCurrentItem(p); }

private:
    DoubleProperty m_peakDistance;
    DoubleProperty m_dampingLength;
    DoubleProperty m_domainSize;
    DoubleProperty m_kappa;
    SelectionProperty<Profile1DItemCatalog> m_pdf;
};

#endif // BORNAGAIN_GUI_MODEL_SAMPLE_INTERFERENCEITEMS_H
