//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Tool/GroupBoxCollapser.cpp
//! @brief     Implements class GroupBoxCollapser
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Tool/GroupBoxCollapser.h"
#include "GUI/Application/ApplicationSettings.h"
#include <QAction>
#include <QBoxLayout>
#include <QGroupBox>
#include <QMenu>
#include <QToolButton>
#include <QVariant>

GroupBoxCollapser* GroupBoxCollapser::installIntoGroupBox(QGroupBox* groupBox, bool expanded)
{
    auto* p = new GroupBoxCollapser(groupBox);

    if (!expanded) {
        QSignalBlocker b(p->m_toggleButton);
        p->m_toggleButton->setChecked(false);
        p->m_toggleButton->setArrowType(Qt::ArrowType::RightArrow);
        p->m_contentArea->hide();
    }

    return p;
}

GroupBoxCollapser* GroupBoxCollapser::findInstalledCollapser(QGroupBox* groupBox)
{
    if (groupBox == nullptr)
        return nullptr;
    return groupBox->findChild<GroupBoxCollapser*>();
}

void GroupBoxCollapser::setTitle(const QString& title)
{
    m_toggleButton->setText(title);
}

void GroupBoxCollapser::addAction(QAction* action)
{
    auto* btn = new QToolButton(m_titleWidget);
    btn->setToolButtonStyle(Qt::ToolButtonIconOnly);
    btn->setDefaultAction(action);
    if (action->menu() != nullptr)
        btn->setPopupMode(QToolButton::InstantPopup);

    m_titleLayout->addWidget(btn);

    connect(action, &QAction::changed, [=]() { btn->setVisible(action->isVisible()); });
}

void GroupBoxCollapser::addWidget(QWidget* widget)
{
    m_titleLayout->addWidget(widget);
}

QWidget* GroupBoxCollapser::contentArea() const
{
    return m_contentArea;
}

void GroupBoxCollapser::setExpanded(bool expanded)
{
    if (m_toggleButton->isChecked() != expanded)
        m_toggleButton->click();
}

GroupBoxCollapser::GroupBoxCollapser(QGroupBox* groupBox)
    : QObject(groupBox)
{
    auto* mainLayout = new QVBoxLayout;
    mainLayout->setSpacing(0);
    mainLayout->setContentsMargins(0, 0, 0, 0);

    m_contentArea = new QWidget(groupBox);
    m_contentArea->setObjectName("ContentArea");
    m_contentArea->setLayout(groupBox->layout());

    mainLayout->addWidget(m_contentArea);
    groupBox->setLayout(mainLayout);

    m_toggleButton = new QToolButton(groupBox);
    m_toggleButton->setObjectName("GroupBoxToggler");
    if (appSettings->currentStyle() == ApplicationSettings::Style::native)
        m_toggleButton->setStyleSheet(
            "QToolButton { border: none; text-align: left; font: bold; padding: 5px}");
    m_toggleButton->setToolButtonStyle(Qt::ToolButtonTextBesideIcon);
    m_toggleButton->setCheckable(true);
    m_toggleButton->setText(groupBox->title());
    m_toggleButton->setArrowType(Qt::ArrowType::DownArrow);
    m_toggleButton->setChecked(true);
    m_toggleButton->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::MinimumExpanding);

    m_titleWidget = new QWidget(groupBox);
    m_titleWidget->setObjectName("GroupBoxTogglerTitleWidget");
    m_titleWidget->setAttribute(Qt::WA_StyledBackground, true);

    m_titleLayout = new QHBoxLayout;
    m_titleLayout->setContentsMargins(0, 0, 3, 0);
    m_titleLayout->setSpacing(3);
    m_titleLayout->setAlignment(Qt::AlignVCenter);
    m_titleWidget->setLayout(m_titleLayout);
    m_titleLayout->addWidget(m_toggleButton);

    groupBox->layout()->setMenuBar(m_titleWidget);
    groupBox->setTitle("");

    connect(m_toggleButton, &QAbstractButton::clicked, this, &GroupBoxCollapser::toggle,
            Qt::UniqueConnection);
}

void GroupBoxCollapser::toggle(bool checked)
{
    m_toggleButton->setArrowType(checked ? Qt::ArrowType::DownArrow : Qt::ArrowType::RightArrow);

    if (m_toggleButton->isChecked())
        m_contentArea->show();
    else
        m_contentArea->hide();
    emit toggled(m_toggleButton->isChecked());
}
