//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Resample/Element/SpecularElement.h
//! @brief     Declares the class SpecularElement.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifdef SWIG
#error no need to expose this header to Swig
#endif // SWIG
#ifndef BORNAGAIN_RESAMPLE_ELEMENT_SPECULARELEMENT_H
#define BORNAGAIN_RESAMPLE_ELEMENT_SPECULARELEMENT_H

#include "Resample/Element/IElement.h"
#include <functional>
#include <heinz/Complex.h>
#include <heinz/Vectors3D.h>
#include <memory>
#include <vector>

class SliceStack;

//! Data stucture containing both input and output of a single image pixel for specular simulation.

class SpecularElement : public IElement {
public:
    static SpecularElement FromQzScan(size_t i_out, double weight, double kz,
                                      const SpinMatrix& polarizer, const SpinMatrix& analyzer,
                                      bool computable);
    static SpecularElement FromAlphaScan(size_t i_out, double weight, double wavelength,
                                         double alpha, double footprint,
                                         const SpinMatrix& polarizer, const SpinMatrix& analyzer,
                                         bool computable);

    SpecularElement(const SpecularElement& other) = delete;
    SpecularElement(SpecularElement&& other) noexcept = default;

    ~SpecularElement();

    size_t i_out() const { return m_i_out; }
    double weight() const { return m_weight; }
    double footprint() const { return m_footprint; }

    //! Returns calculation flag (if it's false, zero intensity is assigned to the element)
    bool isCalculated() const { return m_computable; }

    //! Returns kz values for Abeles computation of reflection/transition coefficients
    std::vector<complex_t> produceKz(const SliceStack& slices);

private:
    SpecularElement(size_t i_out, bool computable, double weight, double footprint,
                    const SpinMatrix& polarizer, const SpinMatrix& analyzer,
                    std::function<std::vector<complex_t>(const SliceStack&)> kz_comp);

    size_t m_i_out; //!< index in scan, designates the channel to which this element contributes
    double m_weight;
    double m_footprint;
    const bool m_computable;
    const std::function<std::vector<complex_t>(const SliceStack&)> m_kz_computation;
};

#endif // BORNAGAIN_RESAMPLE_ELEMENT_SPECULARELEMENT_H
