//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Resample/Particle/ReCompound.cpp
//! @brief     Implements class ReCompound.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "Resample/Particle/ReCompound.h"
#include "Base/Types/Span.h"
#include "Base/Util/Assert.h"
#include "Base/Vector/WavevectorInfo.h" // debug

ReCompound::ReCompound(const std::optional<size_t>& i_layer)
    : IReParticle(i_layer)
{
}

ReCompound::~ReCompound() = default;

ReCompound* ReCompound::clone() const
{
    auto* result = new ReCompound(i_layer());
    for (size_t index = 0; index < m_components.size(); ++index)
        result->addFormFactor(*m_components[index]);
    return result;
}

double ReCompound::radialExtension() const
{
    double result{0.0};
    for (size_t index = 0; index < m_components.size(); ++index)
        result += m_components[index]->radialExtension();
    return result;
}

Span ReCompound::zSpan() const
{
    ASSERT(!m_components.empty());
    Span result = m_components[0]->zSpan();
    for (size_t i = 1; i < m_components.size(); ++i)
        result = Span::unite(result, m_components[i]->zSpan());
    return result;
}

void ReCompound::addFormFactor(const IReParticle& formfactor)
{
    m_components.emplace_back(formfactor.clone());
}

void ReCompound::setAmbientMaterial(const Material& material)
{
    for (size_t index = 0; index < m_components.size(); ++index)
        m_components[index]->setAmbientMaterial(material);
}

complex_t ReCompound::theFF(const WavevectorInfo& wavevectors) const
{
    complex_t result(0.0, 0.0);
    for (size_t index = 0; index < m_components.size(); ++index)
        result += m_components[index]->theFF(wavevectors);
    return result;
}

SpinMatrix ReCompound::thePolFF(const WavevectorInfo& wavevectors) const
{
    SpinMatrix result;
    for (size_t index = 0; index < m_components.size(); ++index)
        result += m_components[index]->thePolFF(wavevectors);
    return result;
}
