/*
 * Author: Andrei Zavada <johnhommer@gmail.com>
 *         building on original work by Thomas Nowotny <tnowotny@ucsd.edu>
 *
 * License: GPL-2+
 *
 * Initial version: 2009-04-03
 *
 */


#include <iostream>

#include "hosted-synapses.hh"
#include "param-unit-literals.hh"

#include "types.hh"
#include "model.hh"

#include "config.h"

using namespace std;


// the base synapse here
CNRun::C_HostedSynapse::
C_HostedSynapse( TUnitType intype,
		 C_BaseNeuron *insource, C_BaseNeuron *intarget,
		 double ing, CModel *inM, int s_mask,
		 bool do_allocations_immediately)
      : C_BaseSynapse( intype, insource, intarget, ing, inM, s_mask),
	C_HostedAttributes()
{
	if ( M )
		M->include_unit( this, do_allocations_immediately);
	else
		idx = (unsigned long)-1;
}



CNRun::C_HostedSynapse::
~C_HostedSynapse()
{
	if ( __cn_verbosely > 5 )
		fprintf( stderr, " deleting hosted synapse \"%s\"\n", _label);
}






// -- parameters

const char* const CNRun::__CN_ParamNames_SynapseAB_dd[] = {
//	"Synaptic strength g, " __CN_PU_CONDUCTANCE,
	"Reversal potential Esyn, " __CN_PU_POTENTIAL,
	"Presyn threshold potential Epre, " __CN_PU_POTENTIAL,
	"Rise rate \316\261, " __CN_PU_RATE,
	"Decay rate \316\262, " __CN_PU_RATE,
	"Time of transmitter release, " __CN_PU_TIME,
//	"Noise level \317\203",
};
const char* const CNRun::__CN_ParamSyms_SynapseAB_dd[] = {
//	"gsyn",
	"Esyn",
	"Epre",
	"alpha",
	"beta",
	"trel",
//	"sigma",
};

const double CNRun::__CN_Params_SynapseAB_dd[] = {
//	0.12,
	0,
      -20,
	0.5,
	0.05,
	5.0,
//	0.
};

const double CNRun::__CN_Params_SynapseABMinus_dd[] = {
//	0.12,
	0,
      -20,
	0.27785150819749,
	0.05,
	5.0,
//	0.
};

const double CNRun::__CN_Params_SynapseMxAB_dd[] = {
//	0.12,
	0,
      -20,
	0.27785150819749,  // the only parameter differing from its AB namesake,
			   // which is also by principle the same as in the ABMinus variation
	0.05,
	5.0,
//	0.
};


const char* const CNRun::__CN_ParamNames_SynapseAB_dr[] = {
//	"Synaptic strength g, " __CN_PU_CONDUCTANCE,
	"Assumed (target->E - Esyn), " __CN_PU_POTENTIAL,
	"Presyn threshold potential Epre, " __CN_PU_POTENTIAL,
	"Rise rate \316\261, " __CN_PU_RATE,
	"Decay rate \316\262, " __CN_PU_RATE,
	"Time of transmitter release, " __CN_PU_TIME,
//	"Noise level \317\203",
};
const char* const CNRun::__CN_ParamSyms_SynapseAB_dr[] = {
//	"gsyn",
	"Ediff",
	"Epre",
	"alpha",
	"beta",
	"trel",
//	"sigma",
};


const double CNRun::__CN_Params_SynapseMxAB_dr[] = {
//	0.12,
      -60 - 0,  // Ediff: a reasonable Esyn - target->E, the latter being -60 mV at rest
      -20,
	0.27785150819749,
	0.05,
	5.0,
//	0.
};







const char* const CNRun::__CN_ParamNames_SynapseAB_rr[] = {
//	"Synaptic strength g, " __CN_PU_CONDUCTANCE,
	"Assumed (target->E - Esyn), " __CN_PU_VOLTAGE,
	"Rise rate \316\261, " __CN_PU_RATE,
	"Decay rate \316\262, " __CN_PU_RATE,
	"Refractory period T, " __CN_PU_TIME,
//	"Noise level \317\203",
};
const char* const CNRun::__CN_ParamSyms_SynapseAB_rr[] = {
//	"gsyn",
	"Ediff",
	"alpha",
	"beta",
	"T",
//	"sigma",
};
const double CNRun::__CN_Params_SynapseAB_rr[] = {
//	0.12,
      -60 - 0,
	0.27785150819749,
	0.05,
	5,
//	0.
};



const char* const CNRun::__CN_ParamNames_SynapseRall_dd[] = {
//	"Synaptic strength g, " __CN_PU_CONDUCTANCE,
	"Reversal potential, " __CN_PU_POTENTIAL,
	"Presynaptic threshold potential, " __CN_PU_POTENTIAL,
	"\317\204, " __CN_PU_RATE,
//	"Noise level \317\203",
};
const char* const CNRun::__CN_ParamSyms_SynapseRall_dd[] = {
//	"gsyn",
	"Esyn",
	"Epre",
	"tau",
//	"sigma",
};
const double CNRun::__CN_Params_SynapseRall_dd[] = {
//	0.12,
	0,
      -20,
	2,
//	0.
};




// -- variables

const char* const CNRun::__CN_VarNames_SynapseAB[] = {
	"Amount of neurotransmitter released S"
};
const char* const CNRun::__CN_VarSyms_SynapseAB[] = {
	"S"
};
const double CNRun::__CN_Vars_SynapseAB[] = {
	0.
};


const char* const CNRun::__CN_VarNames_SynapseRall[] = {
	"Amount of neurotransmitter released S",
	"Amount of neurotransmitter absorbed R",
};
const char* const CNRun::__CN_VarSyms_SynapseRall[] = {
	"S",
	"R",
};
const double CNRun::__CN_Vars_SynapseRall[] = {
	0.,
	0.
};







void
CNRun::CSynapseAB_dd::
derivative( vector<double>& x, vector<double>& dx)
{
	if ( x[0] - t_last_release_started <= P[_rtime_] ) {
	      // continue release from an old spike
		dS(dx) = P[_alpha_] * (1 - S(x)) - P[_beta_] * S(x);
	} else
		if ( _source->E(x) > P[_Epre_] ) {
		      // new spike ... start releasing
			t_last_release_started = x[0];
			dS(dx) = P[_alpha_] * (1 - S(x)) - P[_beta_] * S(x);
		} else {
		      // no release
			dS(dx) = -P[_beta_] * S(x);
		}
}




void
CNRun::CSynapseABMinus_dd::
derivative( vector<double>& x, vector<double>& dx)
{
	if ( x[0] - t_last_release_started <= P[_rtime_] ) {
	      // continue release from an old spike
		dS(dx) = P[_alpha_] * 1 - P[_beta_] * S(x);
	} else
		if ( _source->E(x) > P[_Epre_] ) {
		      // new spike ... start releasing
			t_last_release_started = x[0];
			dS(dx) = P[_alpha_] * 1 - P[_beta_] * S(x);
		} else {
		      // no release
			dS(dx) = -P[_beta_] * S(x);
		}
}




// -------- Multiplexing AB

void
CNRun::CSynapseMxAB_dd::
derivative( vector<double>& x, vector<double>& dx)
{
//	printf( "%s %lu %d %g\n", _source->label, _source->serial_id, _source->idx, _source->E(x));

	if ( q() > 0 ) {
		unsigned effective_q = q();
	      // as we nudge along a little within RK's operational
	      // dt, some spikes can expire in that brief while:
	      // decrement q then, just for this while
		while ( effective_q  &&  M->model_time(x) - _kq[q()-effective_q] > P[_rtime_] )
			--effective_q;
#ifdef __CN_MORECODE__
		if ( effective_q < q() && M->verbosely > 6 )
			printf( "YMxAB %s smacks %u spike(s) of %u at %g(+%g)\n", label,
				(unsigned)q() - effective_q, (unsigned)q(),
				M->model_time(),
				M->model_time(x) - M->model_time());
#endif
		dS(dx) = P[_alpha_] * effective_q - P[_beta_] * S(x);
	} else
	      // no release, decay
		dS(dx) = -P[_beta_] * S(x);
}



void
CNRun::CSynapseMxAB_dd::
update_queue()
{
	unsigned k = _source -> n_spikes_in_last_dt();
	while ( k-- )
		_kq.push_back( model_time());

      // see if the oldest spike has gone past synapse release time
      // disregard spike duration, measure time from saved spike_start
      // (which is == spike_end)
	while ( true ) {
		if ( q() > 0 && model_time() - _kq.front() > P[_rtime_] )
			_kq.erase( _kq.begin());
		else
			break;
//		cout << "q--\n";
	}
}










void
CNRun::CSynapseAB_rr::
derivative( vector<double>& x, vector<double>& dx)
{
	// if ( source()->F(x) > 0 )
	// 	printf( "%s->F(x) = %g\n", _source->label, source()->F(x));
	dS(dx) = -P[_beta_] * S(x)
		+ P[_alpha_] * _numerator / (exp( P[_beta_] / source()->F(x)) + 1);
}







inline int Heaviside( double val)  { return (val >= 0) ? 1 : 0; }

void
CNRun::CSynapseRall_dd::
derivative( vector<double>& x, vector<double>& dx)
{
	dR(dx) = 1 / P[_tau_] * (-R(x) + Heaviside( _source->E(x) - P[_Epre_]));
	dS(dx) = 1 / P[_tau_] * (-S(x) + R(x));
}



// eof
