/*
 * This file is part of the KMOS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 *                              Includes
 *----------------------------------------------------------------------------*/

#include <string.h>
#include <math.h>

#include <cpl.h>

#include "kmo_utils.h"
#include "kmos_oscan.h"

/*-----------------------------------------------------------------------------
 *                          Functions prototypes
 *----------------------------------------------------------------------------*/

static int kmos_test_create(cpl_plugin *);
static int kmos_test_exec(cpl_plugin *);
static int kmos_test_destroy(cpl_plugin *);
static int kmos_test(cpl_parameterlist *, cpl_frameset *);

/*-----------------------------------------------------------------------------
 *                          Static variables
 *----------------------------------------------------------------------------*/

static char kmos_test_description[] = "Testing recipe\n" ;

int cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
    cpl_plugin *plugin = &recipe->interface;

    cpl_plugin_init(plugin,
            CPL_PLUGIN_API,
            KMOS_BINARY_VERSION,
            CPL_PLUGIN_TYPE_RECIPE,
            "kmos_test",
            "test",
            kmos_test_description,
            "Yves Jung",
            "yjung@eso.org",
            kmos_get_license(),
            kmos_test_create,
            kmos_test_exec,
            kmos_test_destroy);

    cpl_pluginlist_append(list, plugin);

    return 0;
}
static int kmos_test_create(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    cpl_parameter *p;

    /* Check that the plugin is part of a valid recipe */
    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) 
        recipe = (cpl_recipe *)plugin;
    else
        return -1;

    /* Create the parameters list in the cpl_recipe object */
    recipe->parameters = cpl_parameterlist_new();

    return 0 ;
}

static int kmos_test_exec(cpl_plugin *plugin)
{
    cpl_recipe  *recipe;

    /* Get the recipe out of the plugin */
    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) 
        recipe = (cpl_recipe *)plugin;
    else return -1;

    return kmos_test(recipe->parameters, recipe->frames);
}

static int kmos_test_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    /* Get the recipe out of the plugin */
    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) 
        recipe = (cpl_recipe *)plugin;
    else return -1 ;

    cpl_parameterlist_delete(recipe->parameters);
    return 0 ;
}

static int kmos_test(cpl_parameterlist * parlist, cpl_frameset * frameset)
{
    cpl_frame       *   fr ;
    cpl_image       *   img_in, * img_in_oscan ;
    char            *   name ;
    int                 i ;

    fr = cpl_frameset_get_position(frameset, 0);

    /* Loop on detectors */
    for (i = 1; i <= 3 ; i++) {
        cpl_msg_info(cpl_func, "Processing detector No. %d", i);
        cpl_msg_indent_more() ;

        /* Load current detector DARK frames into an imagelist */
        img_in = cpl_image_load(cpl_frame_get_filename(fr),CPL_TYPE_FLOAT,0,i);
       
        /* Overscan Correction */
        img_in_oscan = kmos_oscan_correct(img_in) ;
        if (img_in_oscan == NULL) {
            cpl_msg_info(__func__, "Failed on detector %d", i) ;
            cpl_image_delete(img_in) ;
            return -1 ;
        }
        cpl_image_delete(img_in) ;

        /* Save */
        name = cpl_sprintf("out_%d.fits", i) ;
        cpl_image_save(img_in_oscan, name, CPL_TYPE_FLOAT, NULL,
                CPL_IO_CREATE) ;

        cpl_free(name) ;
        cpl_image_delete(img_in_oscan) ;
        cpl_msg_indent_less() ;
    } 
    /* Free and Return */
    return 0;
}

/**@}*/



