/* @(#)flames_preordpos    */
/*===========================================================================
  Copyright (C) 2001 European Southern Observatory (ESO)

  This program is free software; you can redistribute it and/or 
  modify it under the terms of the GNU General Public License as 
  published by the Free Software Foundation; either version 2 of 
  the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public 
  License along with this program; if not, write to the Free 
  Software Foundation, Inc., 675 Massachusetss Ave, Cambridge, 
  MA 02139, USA.

  Corresponding concerning ESO-MIDAS should be addressed as follows:
    Internet e-mail: midas@eso.org
    Postal address: European Southern Observatory
            Data Management Division 
            Karl-Schwarzschild-Strasse 2
            D 85748 Garching bei Muenchen 
            GERMANY
===========================================================================*/
/* Program  : flames_preordpos.c                                           */
/* Author   : I. Porceddu  -  ITAL_FLAMES Consortium                       */
/* Date     :                                                              */
/*                                                                         */
/* Purpose  :                                                              */
/* Compute the y values at the middle x coordinate. First guess values are 
   being taken from the UVES physical model output 
   (table o[setup][lightpath][u/l] --> e.g. o580REDL                       */

/* Input:                                                                   */
/*      ORDTAB     Name of input table generated by uves_calmap (see above) */
/*      DEFPOL     Degree in x and m domain as computed during format check 
               loop                                                     */
/*      COEFFD     A descriptor containing the coeffs of the bivariate poly 
               as extracted from the format check application           */
/*      ECHORD     Number of the first guess detected orders                */

/* Output:                                                                  */
/*      LTAB       middummd.tbl                                            */ 
/*                                                                         */
/* DRS Functions called:                                                   */
/* none                                                                    */ 
/*                                                                         */ 
/* Pseudocode:                                                             */
/* Missing                                                                 */ 
/*                                                                         */ 
/* Version  :                                                              */
/* Last modification date: 2002/08/05                                      */
/* Who     When        Why                Where                            */
/* AMo     02-08-05   Add header         header                            */
/*-------------------------------------------------------------------------*/


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <flames_preordpos.h>
#include <flames_getordpos.h>

#include <flames_midas_tblsys.h>
#include <flames_midas_tblerr.h>
#include <flames_midas_tbldef.h>
#include <flames_midas_macrogen.h>
#include <flames_midas_atype.h>
#include <flames_midas_def.h>
#include <flames_uves.h>
#include <flames_newmatrix.h>
#include <uves_msg.h>

#include <cpl.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

/* Prototypes of "local" functions here */

static
flames_err initorderpos(orderpos *orderstruct, int32_t xdeg, 
                        int32_t mdeg, int maxfibres);

static
flames_err free_orderpos(orderpos **orderstruct);

int flames_preordpos(const char *GTAB,
                     const int *MAXFIBRES,
                     const cpl_frameset *CATNAME,
                     const int *MAXEXTEND,
                     const char *LTAB,
                     double *TAB_IO_YSHIFT)
{
    flames_err status;
    char ordtab[60],outtab[60],templatename[CATREC_LEN+4];
    const cpl_frameset *catname;
    char output[200];
    int orderpixel_deg[2] = {0,0};
    int xdeg = 0;
    int mdeg = 0;
    int upointer = 0;
    int null = 0;
    int no_orders=0;
    int read_elements;
    int i=0, j=0, maxextend=0, maxrows=0;
    int actvals=0;
    int TINid=0;
    int TOUTid=0;
    int colnum[4]={0,0,0,0};
    int noent=0;
    int last=0;
    int k=0;
    double **pcoeffd = 0;
    double yvalue = 0;
    double **central_y;
    double half_x[3] = {0,0,0};
    int templateid = 0;
    int naxis = 0;
    int unit = 0;
    double start[2] = {0,0};
    double step[2] = {0,0};
    int npix[2] = {0,0};
    double start2[2] = {0,0};
    double step2[2] = {0,0};
    int npix2[2] = {0,0};
    float fbuf[2] = {0,0};
    int firstorder = 0;
    int lastorder = 0;
    char chipchoice=0;
    char chipchoice2=0;
    char desctype=0;
    int noelem=0;
    int bytelem=0;
    int minorder=0;
    int maxfibres=0;
    double dbuf=0;
    int ibuf=0;
    double startyshift=0;
    orderpos *orderstruct=0;

    memset(ordtab, 0, 60);
    memset(outtab, 0, 60);
    memset(templatename, 0, CATREC_LEN+4);
    memset(output, 0, 200);

    orderstruct=calloc(1, sizeof(orderpos));

    /* === interface to MIDAS ================================================== */

    /*Let's initialize the MIDAS environment */
    SCSPRO(__func__);


    /* The ORDTAB keyword contains the name of the actual order table
     as written by uves_calmap */
    SCKGETC(GTAB,1,60,&actvals,ordtab);

    uves_msg_debug("GTAB = %s", GTAB);

    /* initialise MAXFIBRES from keyword */
    if (SCKRDI(MAXFIBRES, 1, 1, &actvals, &maxfibres, &unit, &null) != 0) {
        /* problems reading MAXFIBRES */
        SCTPUT("Error reading the MAXFIBRES keyword");
        free(orderstruct);
        return flames_midas_fail();
    }

    /* Let's open the order table file */
    TCTOPN(ordtab,F_IO_MODE,&TINid);

    /* the 2nd element of the COEFFI descriptor of the table contains the
     number of independent variables in the polynomial fit made on the 
     "first guess" order table, and it must be 2 */
    if(SCDRDI(TINid, "COEFFI", 2, 1, &actvals, orderpixel_deg, &unit, &null)!=0){
        sprintf(output, "Error: the guess polynomial fit has %d independent \
variables", orderpixel_deg[0]);
        SCTPUT(output);
        free(orderstruct);
        return flames_midas_fail();
    }
    /* The 6th and 7th elements of the COEFFI descriptor of the table contain
     the degrees of the x (pixel) and m (order) polynomial degrees. */
    if(SCDRDI(TINid, "COEFFI", 6, 2, &actvals, orderpixel_deg, &unit, &null)!=0){
        SCTPUT("Error reading the degrees of the guess polynomial fit");
        free(orderstruct);
        return flames_midas_fail();
    }
    xdeg = orderpixel_deg[0];
    mdeg = orderpixel_deg[1];

    /* Read the name of the FF frames catalog */
    SCKGETC_fs(CATNAME,1,30,&actvals,&catname);

    /* read the number of entries in the catalog */
    SCCSHO(catname,&noent,&last);
    if (noent<1) {
        /* less than 1 entry in the catalog, bail out */
        SCTPUT("Error: no entries in supplied image catalog");
        free(orderstruct);
        return flames_midas_fail();
    }
    /* open the first frame in the catalog */
    SCCFND(catname, 1, templatename);
    if (SCFINF(templatename,99,0)!=0) {
        /* the first file in the catalog does not exist, report and exit */
        SCTPUT("Error: the first file in the supplied catalog does not exist");
        free(orderstruct);
        return flames_midas_fail();
    }
    if (SCFOPN(templatename, FLAMESDATATYPE, 0, F_IMA_TYPE, &templateid)!=0) {
        /* error opening template file */
        SCTPUT("Error opening first file in the supplied catalog");
        free(orderstruct);
        return flames_midas_fail();
    }
    if (SCDRDI(templateid, "NAXIS", 1, 1, &actvals, &naxis, &unit, &null)!=0) {
        /* error reading NAXIS from template file */
        SCTPUT("Error reading NAXIS from first file in the supplied catalog");
        free(orderstruct);
        return flames_midas_fail();
    }
    if (naxis!=2) {
        SCTPUT("Error: NAXIS!=2 in first file in the supplied catalog");
        free(orderstruct);
        return flames_midas_fail();
    }
    if (SCDRDD(templateid, "START", 1, 2, &actvals, start, &unit, &null)!=0) {
        SCTPUT("Error reading START from first file in the supplied catalog");
        free(orderstruct);
        return flames_midas_fail();
    }
    if (SCDRDD(templateid, "STEP", 1, 2, &actvals, step, &unit, &null)!=0) {
        SCTPUT("Error reading STEP from first file in the supplied catalog");
        free(orderstruct);
        return flames_midas_fail();
    }
    if (SCDRDI(templateid, "NPIX", 1, 2, &actvals, npix, &unit, &null)!=0) {
        SCTPUT("Error reading NPIX from first file in the supplied catalog");
        free(orderstruct);
        return flames_midas_fail();
    }
    if (SCDRDC(templateid, "CHIPCHOICE", 1, 1, 1, &actvals, &chipchoice,
                    &unit, &null)!=0) {
        SCTPUT("Error reading CHIPCHOICE from first file in the supplied catalog");
        free(orderstruct);
        return flames_midas_fail();
    }
    SCFCLO(templateid);


    for (i=2; i<=noent; i++) {
        SCCFND(catname,i,templatename);
        if (SCFINF(templatename,99,0)!=0) {
            /* the file in the catalog does not exist, report and exit */
            sprintf(output,
                            "Error: file %d in the supplied catalog does not exist",
                            i);
            SCTPUT(output);
            free(orderstruct);
            return flames_midas_fail();
        }
        if (SCFOPN(templatename, FLAMESDATATYPE, 0, F_IMA_TYPE, &templateid)!=0) {
            /* error opening template file */
            sprintf(output,
                            "Error opening file %d in the supplied catalog",
                            i);
            SCTPUT(output);
            free(orderstruct);
            return flames_midas_fail();
        }
        if (SCDRDI(templateid, "NAXIS", 1, 1, &actvals, &naxis, &unit, &null)!=0) {
            /* error reading NAXIS from template file */
            sprintf(output,
                            "Error reading NAXIS from file %d in the supplied catalog",
                            i);
            SCTPUT(output);
            free(orderstruct);
            return flames_midas_fail();
        }
        if (naxis!=2) {
            sprintf(output,"Error: NAXIS!=2 in file %d in the supplied catalog",i);
            SCTPUT(output);
            free(orderstruct);
            return flames_midas_fail();
        }
        if (SCDRDD(templateid, "START", 1, 2, &actvals, start2, &unit, &null)!=0) {
            sprintf(output,
                            "Error reading START from file %d in the supplied catalog",i);
            SCTPUT(output);
            free(orderstruct);
            return flames_midas_fail();
        }
        if (SCDRDD(templateid, "STEP", 1, 2, &actvals, step2, &unit, &null)!=0) {
            sprintf(output,
                            "Error reading STEP from file %d in the supplied catalog",i);
            SCTPUT(output);
            free(orderstruct);
            return flames_midas_fail();
        }
        if (SCDRDI(templateid, "NPIX", 1, 2, &actvals, npix2, &unit, &null)!=0) {
            sprintf(output,
                            "Error reading NPIX from file %d in the supplied catalog",i);
            SCTPUT(output);
            free(orderstruct);
            return flames_midas_fail();
        }
        if (SCDRDC(templateid, "CHIPCHOICE", 1, 1, 1, &actvals, &chipchoice2,
                        &unit, &null)!=0) {
            sprintf(output, "Error reading CHIPCHOICE from file %d in the \
supplied catalog",i);
            SCTPUT(output);
            free(orderstruct);
            return flames_midas_fail();
        }
        SCFCLO(templateid);


        /* check that sizes actually match among frames in the catalog */
        if ((start2[0] != start[0]) || (start2[1] != start[1])) {
            sprintf(output, "START mismatch between file 1 (START=%f,%f) and file %d (START=%f,%f) in the \
supplied catalog",
start[0], start[1], i,
start2[0], start2[1]);
            SCTPUT(output);
            free_orderpos(&orderstruct);
            return flames_midas_fail();
        }
        if ((step2[0] != step[0]) || (step2[1] != step[1])) {
            sprintf(output, "STEP mismatch between file 1 and file %d in the \
supplied catalog",i);
            SCTPUT(output);
            free_orderpos(&orderstruct);
            return flames_midas_fail();
        }
        if ((npix2[0] != npix[0]) || (npix2[1] != npix[1])) {
            sprintf(output, "NPIX mismatch between file 1 and file %d in the \
supplied catalog",i);
            SCTPUT(output);
            free_orderpos(&orderstruct);
            return flames_midas_fail();
        }
        if (chipchoice2 != chipchoice) {
            sprintf(output, "CHIPCHOICE mismatch between file 1 (%c) and file %d (%c) in the \
supplied catalog", chipchoice, i, chipchoice2);
            SCTPUT(output);
            free_orderpos(&orderstruct);
            return flames_midas_fail();
        }
    }

    /* now we read sizes from the frames, we do not need to read them from
     guess order table descriptors, just set them */

    /* We want to calculate for each order the y values corresponding
     to the CCD x central values */
    half_x[0] = start[0]+((double)(npix[0]-1)*step[0]*.4);
    half_x[1] = start[0]+((double)(npix[0]-1)*step[0]*.5);
    half_x[2] = start[0]+((double)(npix[0]-1)*step[0]*.6);
    /* printf("start[0]=%f step[0]=%f\n",start[0],step[0]); */
    /* printf("npix[0]=%f (npix[0]-1)=%f\n",npix[0], (float)(npix[0]-1));
  printf("(double)(npix[0]-1)*step[0]*.4=%g
          start[0]+((double)(npix[0]-1)*step[0]*.4=%g)\n",
     (double)(npix[0]-1)*step[0]*.4,
           start[0]+((double)(npix[0]-1)*step[0]*.4));

  printf("half_x: 0=%f, 1=%f, 2=%f\n",half_x[0],half_x[1],half_x[2]);
     */
    /* The MAXEXTEND keyword provides the maximum no. of orders to be
     considered above and below the ones detected in the format check */
    SCKRDI(MAXEXTEND,1,1,&actvals,&maxextend,&upointer,&null);

    if (maxextend<0) maxextend=0;

    /* The LTAB keyword does provide the actual name for output table */
    SCKGETC(LTAB,1,14,&actvals,outtab);

    /* We compute the number of coeffs we should extract from COEFFD */
    read_elements = (orderpixel_deg[0]+1)*(orderpixel_deg[1]+1);

    /* We allocate the memory for a data frame which will host the poly coeff */
    pcoeffd=dmatrix(0,orderpixel_deg[1],0,orderpixel_deg[0]);
    for (i=0; i<=orderpixel_deg[1]; i++) {
        for (k=0; k<=orderpixel_deg[0]; k++) {
            pcoeffd[i][k] = 0;
        }
    }

    /* We extract the coefficients from the COEFFD descriptor -
     RUNNING VERSION */
    if(SCDRDD(TINid,"COEFFD",1,read_elements,&actvals,pcoeffd[0],&upointer,&null) != 0) {
        sprintf(output,
                        "Error reading COEFFD from %s", GTAB);
        SCTPUT(output);
        return flames_midas_fail();
    }


    /* read the first and last found order from COEFFR */
    if (SCDRDR(TINid,"COEFFR",3,2,&actvals,fbuf,&upointer,&null) != 0) {
        sprintf(output,
                        "Error reading COEFFR from %s", GTAB);
        SCTPUT(output);
        return flames_midas_fail();
    }

    firstorder = (int) fbuf[0];
    lastorder = (int) fbuf[1];

    no_orders = lastorder-firstorder+1;


    SCDFND(TINid, "TAB_IN_OUT_OSHIFT", &desctype, &noelem, &bytelem);
    switch (desctype) {
    case ' ':
        /* the TAB_IN_OUT_OSHIFT descriptor is undefined, set minorder to zero */
        minorder = 0;
        break;
    case 'I':
        /* the TAB_IN_OUT_OSHIFT descriptor is defined and integer, use it */
        if (noelem > 0) {
            /* read the value anyway */
            SCDRDI(TINid, "TAB_IN_OUT_OSHIFT", 1, 1, &actvals, &minorder, &unit,
                            &null);
        }
        else {
            minorder = 0;
        }
        break;
    case 'R':
        /* the TAB_IN_OUT_OSHIFT descriptor is defined but of the wrong numerical
       type */
        SCTPUT("Warning: TAB_IN_OUT_OSHIFT should be an integer");
        SCTPUT("taking the value anyway, but deleting the old descriptor");
        if (noelem > 0) {
            /* read the value anyway */
            SCDRDR(TINid, "TAB_IN_OUT_OSHIFT", 1, 1, &actvals, fbuf, &unit, &null);
            minorder = (int) (*fbuf);
        }
        else {
            minorder = 0;
        }
        /* delete the existing descriptor anyway */
        SCDDEL(TINid, "TAB_IN_OUT_OSHIFT");
        break;
    case 'D':
        /* the TAB_IN_OUT_OSHIFT descriptor is defined but of the wrong numerical
       type */
        SCTPUT("Warning: TAB_IN_OUT_OSHIFT should be an integer");
        SCTPUT("taking the value anyway, but deleting the old descriptor");
        if (noelem > 0) {
            /* read the value anyway */
            SCDRDD(TINid, "TAB_IN_OUT_OSHIFT", 1, 1, &actvals, &dbuf, &unit, &null);
            minorder = (int) dbuf;
        }
        else {
            minorder = 0;
        }
        /* delete the existing descriptor anyway */
        SCDDEL(TINid, "TAB_IN_OUT_OSHIFT");
        break;
    default:
        /* the TAB_IN_OUT_OSHIFT descriptor is defined but not numeric */
        SCTPUT("Warning: TAB_IN_OUT_OSHIFT not found, should be an integer");
        SCTPUT("resetting to zero and deleting the non-numeric descriptor");
        minorder = 0;
        SCDDEL(TINid, "TAB_IN_OUT_OSHIFT");
        break;
    }

    SCDFND(TINid, "TAB_IN_OUT_YSHIFT", &desctype, &noelem, &bytelem);

    switch (desctype) {
    case ' ':
        /* the TAB_IO_YSHIFT descriptor is undefined, set startyshift to
       zero */
        startyshift = 0;
        break;
    case 'I':
        /* the TAB_IO_YSHIFT descriptor is defined but of the wrong
       numerical type */
        SCTPUT("Warning: TAB_IN_OUT_YSHIFT should be a double");
        SCTPUT("taking the value anyway, but deleting the old descriptor");
        if (noelem > 0) {
            /* read the value anyway */
            SCDRDI(TINid, "TAB_IN_OUT_YSHIFT", 1, 1, &actvals, &ibuf, &unit,
                            &null);
            startyshift = (double) ibuf;
        }
        else {
            startyshift = 0;
        }
        /* delete the existing descriptor anyway */
        SCDDEL(TINid, "TAB_IN_OUT_YSHIFT");
        break;
    case 'R':
        /* the TAB_IO_YSHIFT descriptor is defined but of the wrong numerical
       type */
        SCTPUT("Warning: TAB_IN_OUT_YSHIFT should be a double");
        SCTPUT("taking the value anyway, but deleting the old descriptor");
        if (noelem > 0) {
            /* read the value anyway */
            SCDRDR(TINid, "TAB_IN_OUT_YSHIFT", 1, 1, &actvals, fbuf, &unit, &null);
            startyshift = (int) (*fbuf);
        }
        else {
            startyshift = 0;
        }
        /* delete the existing descriptor anyway */
        SCDDEL(TINid, "TAB_IN_OUT_YSHIFT");
        break;
    case 'D':
        /* the TAB_IN_OUT_OSHIFT descriptor is defined and double, use it */
        if (noelem > 0) {
            /* read the value anyway */
            SCDRDD(TINid, "TAB_IN_OUT_YSHIFT", 1, 1, &actvals, &startyshift,
                            &unit, &null);
        }
        else {
            startyshift = 0;
        }
        break;
    default:
        /* the TAB_IO_YSHIFT descriptor is defined but not numeric */
        SCTPUT("Warning: TAB_IN_OUT_YSHIFT not found, should be an integer");
        SCTPUT("resetting to zero and deleting the non-numeric descriptor");
        startyshift = 0;
        SCDDEL(TINid, "TAB_IN_OUT_YSHIFT");
        break;
    }

    /* write the original TAB_IO_YSHIFT in a (local) keyword */
    SCKWRD(TAB_IO_YSHIFT, &startyshift, 1, 1, &unit);

    /* close the ORDPOS table */
    TCTCLO(TINid);

    /* Initializing the orderstruct structure */
    status = initorderpos(orderstruct, xdeg, mdeg,maxfibres);

    /* Filling up the orderpol element with the actual poly coefficients */
    orderstruct->orderpol = pcoeffd;

    /* Creating a new table */
    if ((maxrows=no_orders+2*maxextend)<40) maxrows=40;
    central_y = dmatrix(0,2,0,maxrows);
    /* jmlarsen: Use F_O_MODE for new table
     old code: TCTINI(outtab,F_IO_MODE,maxrows,&TOUTid); */
    TCTINI(outtab,F_O_MODE,maxrows,&TOUTid);

    /* Creating new columns */
    TCCINI(TOUTid, D_I4_FORMAT, 1, "I6", " ", "ORDER", &colnum[0]);
    TCCINI(TOUTid, D_R8_FORMAT, 1, "F8.2", " ", "YQ1FTCK", &colnum[1]);
    TCCINI(TOUTid, D_R8_FORMAT, 1, "F8.2", " ", "YMIDFTCK", &colnum[2]);
    TCCINI(TOUTid, D_R8_FORMAT, 1, "F8.2", " ", "YQ3FTCK", &colnum[3]);

    for (j=0; j<=2; j++) {
        for (i=firstorder-maxextend;i<=(lastorder+maxextend);i++) {
            /* if (j==0) printf("i=%d half_x=%f\n",i,half_x[j]); */
            if ((status=get_ordpos(orderstruct,(double)i,half_x[j],&yvalue))
                            != NOERR) return status;
            central_y[j][i+maxextend-firstorder] = yvalue;
        }
    }
    /* Writing table values */
    for (i=0; i<=(no_orders+2*maxextend-1);i++) {
        j = i+firstorder-maxextend+minorder;
        TCEWRI(TOUTid, i+1, colnum[0], &j);
        TCEWRD(TOUTid, i+1, colnum[1], &central_y[0][i]);
        /* printf("central_y[0][i]=%f\n",central_y[0][i]); */
        TCEWRD(TOUTid, i+1, colnum[2], &central_y[1][i]);
        /* printf("central_y[1][i]=%f\n",central_y[1][i]); */
        TCEWRD(TOUTid, i+1, colnum[3], &central_y[2][i]);
        /* printf("central_y[2][i]=%f\n",central_y[2][i]); */
    }

    /* Write also the START, STEP, NPIX and CHIPCHOICE variables as
     descriptors, so that we do not have to look them up any more later */
    SCDWRD(TOUTid, "REFSTART", start, 1, 2, &unit);
    SCDWRD(TOUTid, "REFSTEP", step, 1, 2, &unit);
    SCDWRI(TOUTid, "REFNPIX", npix, 1, 2, &unit);
    SCDWRC(TOUTid, "CHIPCHOICE", 1, &chipchoice, 1, 1, &unit);

    TCTCLO(TOUTid);

    //free(orderstruct);
    free_orderpos(&orderstruct);
    free_dmatrix(central_y,0,2,0,maxrows);
    free_dmatrix(pcoeffd,0,orderpixel_deg[1],0,orderpixel_deg[0]);
    return SCSEPI();

}

static
flames_err initorderpos(orderpos *orderstruct, int32_t xdeg, 
                        int32_t mdeg, int maxfibres)
{
    /*  orderstruct->orderpol = dmatrix(0, mdeg, 0, xdeg);*/
    orderstruct->xdegree = xdeg;
    orderstruct->mdegree = mdeg;

    /*
     TO BE CHECKED 
     We do not the name of the keyword which will contain the number of used 
     fibres. 
     So we suppose to have 9 (nine) fibres, and 0 is the first one:
     Be warned about it: firsfibreindex must be in agreement with this last 
     value!!!
     */
    orderstruct->maxfibres = maxfibres;
    orderstruct->fibrepos = dvector(0,maxfibres-1);
    orderstruct->fibremask = ivector(0,maxfibres-1);
    return(NOERR);
}


static
flames_err free_orderpos(orderpos **orderstruct)
{

    free_dvector((*orderstruct)->fibrepos,0,(*orderstruct)->maxfibres);
    free_ivector((*orderstruct)->fibremask,0,(*orderstruct)->maxfibres);
    free(*orderstruct);
    *orderstruct=NULL;

    return(NOERR);
}




