// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using Microsoft.DotNet.XUnitExtensions;
using Test.Cryptography;
using Xunit;

namespace System.Security.Cryptography.Tests
{
    [ConditionalClass(typeof(MLKem), nameof(MLKem.IsSupported))]
    public static class MLKemKeyTests
    {
        [Theory]
        [MemberData(nameof(MLKemTestData.MLKemAlgorithms), MemberType = typeof(MLKemTestData))]
        public static void Generate_Roundtrip(MLKemAlgorithm algorithm)
        {
            using MLKem kem = MLKem.GenerateKey(algorithm);
            Assert.Equal(algorithm, kem.Algorithm);

            Span<byte> seed = new byte[algorithm.PrivateSeedSizeInBytes];

            kem.ExportPrivateSeed(seed);
            byte[] allocatedSeed1 = kem.ExportPrivateSeed();
            Assert.True(seed.ContainsAnyExcept((byte)0));
            AssertExtensions.SequenceEqual(seed, allocatedSeed1.AsSpan());

            using MLKem kem2 = MLKem.ImportPrivateSeed(algorithm, seed);
            Span<byte> seed2 = new byte[algorithm.PrivateSeedSizeInBytes];
            kem2.ExportPrivateSeed(seed2);
            byte[] allocatedSeed2 = kem2.ExportPrivateSeed();
            AssertExtensions.SequenceEqual(seed, seed2);
            AssertExtensions.SequenceEqual(seed2, allocatedSeed2.AsSpan());
        }

        [Fact]
        public static void Generate_NistVectors_Span()
        {
            foreach (MLKemGenerateTestVector vector in MLKemGenerateTestVectors)
            {
                ReadOnlySpan<byte> seed = vector.Seed.HexToByteArray();
                using MLKem kem = MLKem.ImportPrivateSeed(vector.Algorithm, seed);
                Assert.Equal(vector.Algorithm, kem.Algorithm);

                Span<byte> decapsKey = new byte[vector.Algorithm.DecapsulationKeySizeInBytes];
                kem.ExportDecapsulationKey(decapsKey);
                AssertExtensions.SequenceEqual(vector.DecapsulationKey.HexToByteArray().AsSpan(), decapsKey);

                Span<byte> encapsKey = new byte[vector.Algorithm.EncapsulationKeySizeInBytes];
                kem.ExportEncapsulationKey(encapsKey);
                AssertExtensions.SequenceEqual(vector.EncapsulationKey.HexToByteArray().AsSpan(), encapsKey);
            }
        }

        [Fact]
        public static void Generate_NistVectors_Array()
        {
            foreach (MLKemGenerateTestVector vector in MLKemGenerateTestVectors)
            {
                byte[] seed = vector.Seed.HexToByteArray();
                using MLKem kem = MLKem.ImportPrivateSeed(vector.Algorithm, seed);
                Assert.Equal(vector.Algorithm, kem.Algorithm);

                byte[] decapsKey = kem.ExportDecapsulationKey();
                AssertExtensions.SequenceEqual(vector.DecapsulationKey.HexToByteArray(), decapsKey);

                byte[] encapsKey = kem.ExportEncapsulationKey();
                AssertExtensions.SequenceEqual(vector.EncapsulationKey.HexToByteArray(), encapsKey);
            }
        }

        [Fact]
        public static void ImportEncapsulationKey_Array_Roundtrip()
        {
            foreach (MLKemGenerateTestVector vector in MLKemGenerateTestVectors)
            {
                byte[] encapsulationKeyBytes = vector.EncapsulationKey.HexToByteArray();
                using MLKem kem = MLKem.ImportEncapsulationKey(vector.Algorithm, encapsulationKeyBytes);
                Assert.Equal(vector.Algorithm, kem.Algorithm);

                byte[] exportedEncapsulationKey = kem.ExportEncapsulationKey();
                AssertExtensions.SequenceEqual(encapsulationKeyBytes, exportedEncapsulationKey);
            }
        }

        [Fact]
        public static void ImportEncapsulationKey_Span_Roundtrip()
        {
            foreach (MLKemGenerateTestVector vector in MLKemGenerateTestVectors)
            {
                ReadOnlySpan<byte> encapsulationKeyBytes = vector.EncapsulationKey.HexToByteArray();
                using MLKem kem = MLKem.ImportEncapsulationKey(vector.Algorithm, encapsulationKeyBytes);
                Assert.Equal(vector.Algorithm, kem.Algorithm);

                Span<byte> exportedEncapsulationKey = new byte[vector.Algorithm.EncapsulationKeySizeInBytes];
                kem.ExportEncapsulationKey(exportedEncapsulationKey);
                AssertExtensions.SequenceEqual(encapsulationKeyBytes, exportedEncapsulationKey);
            }
        }

        [Fact]
        public static void ImportDecapsulationKey_Span_Roundtrip()
        {
            foreach (MLKemGenerateTestVector vector in MLKemGenerateTestVectors)
            {
                ReadOnlySpan<byte> decapsulationKeyBytes = vector.DecapsulationKey.HexToByteArray();
                ReadOnlySpan<byte> encapsulationKeyBytes = vector.EncapsulationKey.HexToByteArray();
                using MLKem kem = MLKem.ImportDecapsulationKey(vector.Algorithm, decapsulationKeyBytes);
                Assert.Equal(vector.Algorithm, kem.Algorithm);

                Span<byte> exportedDecapsulationKey = new byte[vector.Algorithm.DecapsulationKeySizeInBytes];
                kem.ExportDecapsulationKey(exportedDecapsulationKey);
                AssertExtensions.SequenceEqual(decapsulationKeyBytes, exportedDecapsulationKey);

                Span<byte> exportedEncapsulationKey = new byte[vector.Algorithm.EncapsulationKeySizeInBytes];
                kem.ExportEncapsulationKey(exportedEncapsulationKey);
                AssertExtensions.SequenceEqual(encapsulationKeyBytes, exportedEncapsulationKey);
            }
        }

        [Fact]
        public static void ImportDecapsulationKey_Array_Roundtrip()
        {
            foreach (MLKemGenerateTestVector vector in MLKemGenerateTestVectors)
            {
                byte[] decapsulationKeyBytes = vector.DecapsulationKey.HexToByteArray();
                byte[] encapsulationKeyBytes = vector.EncapsulationKey.HexToByteArray();
                using MLKem kem = MLKem.ImportDecapsulationKey(vector.Algorithm, decapsulationKeyBytes);
                Assert.Equal(vector.Algorithm, kem.Algorithm);

                byte[] exportedDecapsulationKey = kem.ExportDecapsulationKey();
                AssertExtensions.SequenceEqual(decapsulationKeyBytes, exportedDecapsulationKey);

                byte[] exportedEncapsulationKey = kem.ExportEncapsulationKey();
                AssertExtensions.SequenceEqual(encapsulationKeyBytes, exportedEncapsulationKey);
            }
        }

        public record class MLKemGenerateTestVector(MLKemAlgorithm Algorithm, string Seed, string EncapsulationKey, string DecapsulationKey);

        public static IEnumerable<MLKemGenerateTestVector> MLKemGenerateTestVectors
        {
            get
            {
                // Vectors that have zeros to ensure they round-trip correctly.
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "00F8418B8BE63D8433058E3AFEBEFA43F434B2393896C2E75750FAFA5E9AE9A4226FD2493AF33A6A92282E3FF0E5BA8AB9F09529C53671CBA1D98981DA8FE800",
                    "002645126709F87B5C6DF9116BA175020895C5C3031AFBBFCDF95AF2839095396A21F23BB1232091EB8F983BCBC95400D4A335C555187FC2311B79402A261AEB276F1DE4CA62458D5AA772F0C25C3A4824DC095AB63584CF863CD4DCA9736CAAFADC1CA2249475946B4BF4B208334EC419C916157E84C9056DE713CD055D69009D295C1C1A6A07E008297BD1B167D4641ED946E0432CDD285A5E9794E6D3CF7422012999AC50C42CACC68E659C37CDD93A70166192719CC0E22D1317C1A3F00715004D505A2B606055626421611BCC64425D2F1452E32B8D68B2584A66180D648B8925467DB306DFC39859EAC383AB60B1F9793051C8AE2151621166B3F862B52C44CD796602C80F45D3C095749EA9D65EB6024752761B364891C7669441E3AE30E43178D13B2D1C3B3AFC999FD4B444E3591320A766ACCA6C4059B8F34289211F82C5A333EA029154C716B640AAC1BC7D516DFB00881EFA9472725A9BC57DD1781E7A0689C5E399CB1B1F41A031D33C3528D421047A48A29C91222074FCC538814242C78520D201171615518D36CC98BC885F2127CE182BA15041B07737A8ABA7ACC4C0A22624D49C4854674A54F11AFFB03D7AE6B90E7413B854A131A6496D02744A66BA7E4B8DF973577EA902E7D1B7CA40CA7AA63F2C04B5D6DA87A735148F003F78395010B11AC9C87E1AB7BDCD8B18CEBA7226E6899DD3A7F2937686D52F7BA21D1A439A7DA06B367C240F644058C434337880FFE79DE9F2A277B13B0B82907E22104F0134423A3F06167FF8254276922361C0368CBC73BFB8C35CD1C0A79076C420B053E604C7098C74344F9A5094983B17ADD46C34984CA317CDED2B0A1ECC12C52124655389FDB813DBBA8B04310259F0A9EE57B03AB04B18079ABFA336FFF854DA70BD82760A39FB3D9BB2837DF052E925449C88680242B73A39AB9D332A9C0371CD06663FC6A65968CA27663E44984B91C8B51A2CB0B6364CCE2097DB286E01FB8D2C871472C68117EA6497196F5F56A3CE778D48B687DAF440BB483748B7C2F0889F02DAB06EC64F33E59E49931A20084C7E78563A766B5223909DC385C5BC4BD8AFF51B5CC52F60FD181D8AC43537254ABC2F29E8FCB8698CD4",
                    "00E5CDA3A6960283140B49775D0111591493BD0A8C002C1701B53442FAB2A6C7C60B7AB64F1B90104460986803C8C464FA4AA334513887F28A7A4810073301ED40BACD4B1123D73679F63735D17F2EF39B40B92A0A65A170A7A14721C8D23350FE1B90C913C7955A5E3DDB701417C6C505667377A611AA26F5AC981C4953A058BC51E1519B47780702A65407722B9C32BC4819181B675C05AE55116C96B084E58CAFAA5BA1FB9AA893C2B4C399B6C1503AD7782FF6568EEE402451C874EBB8A4417A2AF04A102ABC7D4064BF7E451B81508ECFF32529491126536D64A65142F9AD804881B802433F89C3C38701DCC670628A7E8DCB77F5086C972397D777982B7C420ACA22492720D218B69AAC36A5509FEEAC0324BB9E3263B38251C8BEF95BCADC953749C3F4B886EFC8A880366177D8A45E3C433809C7D5889893C588B5216E3C47683D79318BB90E20EC09F481B0AD77B6433773458C24FDE361AE050BE76959A069530771AC66D4BEA1F4285C462260C48EC894AC444582DA3C1F4DBC03FB6046AEF9BE8DE01A479156DC200FED2518E36A7EC4915AF7E59096037C35B5B3F1C4C8CA96C30A372B8D72B54E91B57E8050B464B27AB671B2EC44D6244020D1098709AB5A79A2B5CC47CCF3480DB0BE27509C162CCB842671E3D5BB571986BA02548A81668F41BC5F6A7C4F66639262CD37C69FAC6BA024183B1D4902773C82F9156EADF17E7057B8FCCA21D8C895A92B080879B49A001DF8E88FB4F667D069CDCDC237324A51297224DEF542F65904F7FCBC4ED66A1BA90321172C3570BCDA36415DE93522984C0CAB3EF1819A31808B294ACFF1D517E045A0D2D59BD1E1374DA3AE97F09D6720C0516B9CC4352A2CD6479309335C300E4596AF5424B5E7D4A5FF06685481407FAB8AB32B0CA336886F5892135B6D49906DB9497244C7983AD6A474194329081AB26920724B317DDB59FD43108F97BDDBEA49E6C7A681A11ACF374AD5AC01443203A53281C143114A19CE2CE427A78C1B24A603372BC1A3E16ED826639D294047F89CFEE17CC1E00CB69A0EA71C20B37A4C208C5FB9CA44002645126709F87B5C6DF9116BA175020895C5C3031AFBBFCDF95AF2839095396A21F23BB1232091EB8F983BCBC95400D4A335C555187FC2311B79402A261AEB276F1DE4CA62458D5AA772F0C25C3A4824DC095AB63584CF863CD4DCA9736CAAFADC1CA2249475946B4BF4B208334EC419C916157E84C9056DE713CD055D69009D295C1C1A6A07E008297BD1B167D4641ED946E0432CDD285A5E9794E6D3CF7422012999AC50C42CACC68E659C37CDD93A70166192719CC0E22D1317C1A3F00715004D505A2B606055626421611BCC64425D2F1452E32B8D68B2584A66180D648B8925467DB306DFC39859EAC383AB60B1F9793051C8AE2151621166B3F862B52C44CD796602C80F45D3C095749EA9D65EB6024752761B364891C7669441E3AE30E43178D13B2D1C3B3AFC999FD4B444E3591320A766ACCA6C4059B8F34289211F82C5A333EA029154C716B640AAC1BC7D516DFB00881EFA9472725A9BC57DD1781E7A0689C5E399CB1B1F41A031D33C3528D421047A48A29C91222074FCC538814242C78520D201171615518D36CC98BC885F2127CE182BA15041B07737A8ABA7ACC4C0A22624D49C4854674A54F11AFFB03D7AE6B90E7413B854A131A6496D02744A66BA7E4B8DF973577EA902E7D1B7CA40CA7AA63F2C04B5D6DA87A735148F003F78395010B11AC9C87E1AB7BDCD8B18CEBA7226E6899DD3A7F2937686D52F7BA21D1A439A7DA06B367C240F644058C434337880FFE79DE9F2A277B13B0B82907E22104F0134423A3F06167FF8254276922361C0368CBC73BFB8C35CD1C0A79076C420B053E604C7098C74344F9A5094983B17ADD46C34984CA317CDED2B0A1ECC12C52124655389FDB813DBBA8B04310259F0A9EE57B03AB04B18079ABFA336FFF854DA70BD82760A39FB3D9BB2837DF052E925449C88680242B73A39AB9D332A9C0371CD06663FC6A65968CA27663E44984B91C8B51A2CB0B6364CCE2097DB286E01FB8D2C871472C68117EA6497196F5F56A3CE778D48B687DAF440BB483748B7C2F0889F02DAB06EC64F33E59E49931A20084C7E78563A766B5223909DC385C5BC4BD8AFF51B5CC52F60FD181D8AC43537254ABC2F29E8FCB8698CD4C74CD655ECCF259367D8B329389410E01AFB493B6A1D936E1F3D2F8E8CC43B72226FD2493AF33A6A92282E3FF0E5BA8AB9F09529C53671CBA1D98981DA8FE800"
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "002BF35FBD830602325D2FBA9E36F2C527DA7D017DB31474D6908749790DF1D504150BD6B5A2E7E237795334EA683215868A5ACED024E77889C143B2AC209A00",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "00EF362328DF9AF7EEACBA98DFDB6908CAC76CB15C790EAD244930CA00BDD9D163B1AD5C313C2F4D6F7E6AFB3576F0668792FE722DF04041FAAB9E97907B1500",
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
                );

                // The vectors below are generated from the NIST ACVP projections.
                // https://github.com/usnistgov/ACVP-Server/blob/85f8742965b2691862079172982683757d8d91db/gen-val/json-files/ML-KEM-keyGen-FIPS203/internalProjection.json
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "1EB4400A01629D517974E2CD85B9DEF59082DE508E6F9C2B0E341E12965955CA1A394111163803FE2E8519C335A6867556338EADAFA22B5FC557430560CCD693",
                    "5B318622F73E6FC6CBA5571D0537894AA890426B835640489AA218972180BB2534BCC477C62CC839135934F3B14CD0808A11557D331103B30F9A8C0CB0FA8F0A2A152E802E48E408087510D5114D4D2399A51530616C7E310528308176D0042710BC8344EC3D4CA810A92978BFABB516D81CAB0753CDF325AC2377A1F96EFC73C15F5AA367A1582A13651B0337C7943C1D54637669686BEBBD392511FFFC9E3A68CBEEC0CE2CF59A8D51C4DE288EB4641DF6610C82D09CDDA418ACD83F0DCA2859B27117E87981AAA8EBA47515812DA2C27ADF9C682E373D5AF294BE3104474B8D14173788965ECCD80322B6CA04240E7D150F2CD4B04066C1924039B9E4A9E06C2B55DBA2FDDABB4065CFE7EBC5AE01CD45C76374683CB1820C34A841836391B9D8C2AA22B29E7436CFCAB789B3CE8AE2700351C1165B7B4F72CC53E913E5668AE75170352A0DE68A5E3819443DB4113161A2019C4930C97011F31540B833E9A890503A7EC3F38C0D94BE3C7501C6161F39099E3CAC0139ACC7271B70D1664A36A89FA4D22857C6C15AD4C52D5C26E23B81DCDA9FD7A49980C5818888AB2538AD91F54E691B7558C63FAE433A7FAB51485989F4335E6187B65041401238AA0A5A932356207796AF2C70363034546F4615499245E1228BFF2C76674634A60C9A04E00FB276C6C00A114BF1B2C8961E740A082940CEEAAB464370BBBB3919C7421BC81C732415A711AA935A4C2C02CB5D0BCBB99CE830EDDBAE4C228E4F095E29FBC27EA2B881697A1D309D28C480C3E9691FB63480BC5C6239B6CCAA41CD52A6209038C2C887BC71C1BD514A0FAA21721A2A5B30ACB168227833A8260422C1F4815EC2ADB207389FB1B817D78FC96063434B6728E18469475DB5D712BC403D8231CF9C8926D0A94B6830881FA5678AD04499F40D5CA83479BA85A70B1196C32A68A6B7FFB40EA6FC3FF020768B91B27F653746546C5E256B14069E827C1616FC7647F8B70F8A32DB551CF715BBB315B7B9BC20FF76847CFC4AEAC23DDC1302EC928CFE40447C761143194DA1415D3D8389F61BAB41EB605729123A320BB54B3B3FBCBC787C46F354C7D7D60F8DFE3729375AEF1891C08A79DE237E39E860061D",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "47877987835EF38823A23B37CA4CE2844BC017F9A1EFADF8C5D73CB5562C81A461846347FDE6757082E3D56EA060CEB773D5F38A84B8D8AACF776EF9CF766CF8",
                    "B67136147474B5A56BA8663BF0157518A3C806C20259F4CBC59314710336328BA6945983838ABC65A91820536791048ECD17B7BBE4650C0A32EAA77FC799CF2797C27F35566E630C2BA6B05F608B12818231D67D4886CFF09B286443C7972376EBBA08ECD71C04E945A9A75F82F08E726057DF829D96CB83E4848B92E4263F9596007A1F6B224D794244EA58052EB574896870828C8053658F051BBF1699B657591826348E520707EF8666AB138DB4C3B0E401361A5BABADE2059932356734BCFDFC69146250DF2A78E5A530661446089C030B224F27B3748D2A8239C13D20F10E70F473B82BC3B0CB7C2C82200121478EB60E120C6993A3961AC5850BB25577688EACAABD7F031345AAA692754C3EC48484583D13AC7B11565D9B08A356A1A902067676B20C80DC67D20246B47C199EEC3A934C3D24634ACB119B76BCB3A8A823AFD0CA2CE9277CD9BF03A5565AEA2CDFB811548237D1644F8FBC233EDA44C195241AA0C3C9C94317913A7A8A69566B5166022540B96CBD7781CC5C30529CBA9FEC5E5038BA66A7491F8A0C5CA7B662123B1C5C7F90D0320D617C4CE04F31B466C26A56E1E1CB4721B8C0AC75CDD815C6D37A0D6462D31B264D88A7303A857C333C173C8F09C34F38740E5C87A494003B9347418973462D78B7D57694941B062B06C762F70DF64323DAB70411932349A986C40795187696A672108B1C11357C0D5EB3502E5406DE2721D6C677594B39D9480A7B3B67F3041B6B82C53399335C821940604DD8D03C7D3198167BC83F8C830C785773EC8221986DCCF0C19B462EC5475E878B16C3E2A85C3099C6107A30483CB6E27965DA0D40EACEB7567553396CCD5A1BA6941BA57A33A3A0CAB7B35A31057E24130822D9C78FE9C5AC98268E36BB3D210313A60C3A3359030C6CD1D9C49A277B5EA4825133285B9019CDE943F18CA1FCFC3A33036E88416E22C61B0A6731B395159355009F753B6158B834896D8779A237B783631CB1FEC799F2A25D6A7C0E3B14AEA275CBF87530333869DB5A9CCCA513063C717911275864B2E732484A044E0E73C970F69E15B1263C69A2D0811531DBC5DBB0E759A103FECB7101BA8797658CCA85C45C3CA938FE44B75F",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "DAAA521D7780E6C72D3A6626B8D5DF8ABBC5AE44A9ABC647576035A9A6A9DB4D2056FB9B4E06A790F1D1538BA6D81DE1953902D9D50EA8D646A0231F11D89981",
                    "D0BC1226CCBAA1558CB1D97BE2239CF6D889B3B64C5D22516C26A230C38291923828A889E44B1C11E495F5D0C0E889057AF35A96B8B10F159A2DD81A518B89017C8019042232C59396C1CE8103747BD981B739CF24CC732F01CC27952E69467566140D4ED27B7B016352E73C4EEC6512E592C18368F8664ACF50C4A82637A58A2A76915BFC8CC5F2D577243A1D5F5646414844921C79896031C772CFF75772CAC91B28795DC5DB2AC558094B3301BC83684D8BB8578660E9C392000807BFAB7E37152DB7E0B89B5A237F5B99CD7757D15006CA617694370C8F09A669467BEFE4BC444359FC105986A592C2AC756812B5DDB6437B038B51AB09AC3A68B9C95103C751EFA07B1AA51A2E579E545503B97A0E595102FF578336E8A19C9616ABB38B4A05CA054023100939910159C2278DA72811CA342E15D0AF20D17923E3AA8C5A9A7CD1BF525009FD51B2AC91B0BB3A2BDCC2AF9AE057FE4219A5994CD8B52FFB14C6C0493AB37C7561DCA143667F92FC8CC1A70EA3024E1A2880F6209A39996F8A6A59ADD159C20577CDEBA9259733A9B7838BFA2D8C213CEDAB87939A4E2AA2634EA3621F789D05C74AA867ADE527B2A0973195812425FB9D7B7866428B8B9A533405707AE3804AC336527FB5A45869C6B5596E293A0981E30038C489C63147E9E412581410FAB669B3253B914571F396232527328EBA8661E8B5554B6077AA8CE6286E376380A99BBC100CC024326870282EDD3363899C8EA1783B5B784FF66333950A146B52B9F7047983443DF0655E0FACBE4D16AE7F629586C3AB4FAA25D7135ED8128BFA1227FB916D4E297F02B7A4CDB9873FA4B1B479C8F9B311A2A2A96128126B031836CB908F79BAB12158464CB731D9331F4CAEC2C96CF484A8E6752899299C13D261FCF6649672CD55432AD12B6FCDB9BBA52319A982A31EA1231CD14EE52122DC0B38D82C853F77189B650894D09E65F924CDC3BB9EB01EBC513E4964A1E4E59B984BB1F4170BF6C68271BA3D18D270ABD56F659C854B252B3F2C006EAA4472C0409FFC902B589A937992094419D7758FA9861E9E89B46810BA280F36FC0C276F0E5E97A0F7D9F06E216FF1BD0A6E0533CF1172A771D31F222F",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "01026FDD554810181EF771310C94168CC0925D7F62032554049873927DDB99FE0AA3C89691B236AC7FE67013D397FAC23E8107587856A5C99900908CF48FFF8F",
                    "8B772D7EB64F387418C76C872019AB8C4114B3DA51AD132D545B4BDB3473D88712CE832CE4284B4C824365000D0826516DD19AB8210D7CB73A344074026505C3C6096F624ECBEA67D26B1481F48A72025426296C0120C368B0B2B263A3E8D48060648F0CAA4EBEEC81C995CE55E81197B906783A0B230C5200082739E4175245280D728ABAD725EABB5F58870DC3E944331AB7B78C231ECA88E5F99F085C6021668C2307434147B1A1532FA5F98CF86A069D434B50B0C7CD6C7B2CF03DBF7A48650C49A797BFB5498B96531E5D3C3CAFB6B36FB89AC2A04402D8B7C652C3A4D0CC7CB984B0B001D737053B82CE71D83622A931D7CB87F8461FB5831A98424BE4B70370A8471C2C529FBB175885725F3698E32039FA3B1125C8BA3525377829A5DBA2255D40BDA49469D6CCAC37B626311A8DC96145EDD1B65B354B0C51BFCFB0C01E5BB7A7E02BB1A04221443A7551547972570FA92DD0CC390C965EA8207893E306BFACCA055A8C88EC57E296436182493CC44BC369CE62E0ABBEF01130D920CBE61612F14786722251D331C3806F0BF676B867A85CA1602058CDA887CB3DD78CB4BA76D2671541A42CC24A96E1126B5BC70DB58C2A8269AC6338506DBC31132239127451E551CF82F903EB58783ED177E3950CBCC72E4C1869F176102966AE25269A5CE2606978C762659030775D05178181E37352C88C394233A8968AB23B029F89B41F6B1729CB2CCBDCA05F2B20C44229D0DA8EDA66B16B4A39B8A96526CAAAA92C03615A02DED4AD15918F82ACA02277CCA7A7AD660A56096552F8ECC9D1CACD9D87764FDB6083A765EFB839AE6CB93C9886B9493018A336B418A8DC6A272CDC208618610EB4B24597C3488C0FDBD73C821BB6F521191AD775B0E5BBEE6C88EE627DAEAA6A8484A02E925E3B7C4E9DB3CFEF88CAD5FB17ED3598C1C634592650AE6A61BBC812921564BFA94379A0523085B55075B40703CC113647EFBBA2A593BB22D23DC85175B61388E98398E90B9DDA18528B874326E36C6AFC813B79515D662ACED3B697699E917C4F0B8BC6DD385C1B58CF8DD1A676877C9A2511797E34415FFF7A77CAC769C91D2DF95D16704A933B539936DE505A4DC914E757",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "BB340F80FAE62CAAC952C11D038CA6B48B481E32C3D23256A8B3734AF66D4C491CF9C6E284F7A0274D54A824029FA75B614DC0C0284B33811793789A0E7AFE72",
                    "6BC02A993B7AE4CA512AB647F8E37EB6F175B7D59B8016CD35D41FA51A7DF8878AC69B53018395FED12F152A9074F47652C21ECB00B73F84048C754EDBC3912626A812061CB5532A9B14BAFF5739284AAC55B51599FB6BB6E9B61679CA82B951BC76B5C648904185502972022C6546536A9FD5D45A79E154A1F0A7A3056CB306AC6FE10E7AA45EA3C6BF0D0840264B2DDD979CB3945DD3887C316B70D572945A249A42914754377947A12990DA44C75A1047988FEF87CC63360AD58A5B2CB694664BA42CE7BB20660470362BE585A3A55AC93C11887A366DD4ABBA8A8C812D418C6857904F080A010356F8C49DCC44257C9C4DACA950F9D137A367AD3788AD46FA92B2F27F9D72B1B5712B6F70AB5BD6C45A780904713F838A206FB37E174C5F7E9708DCB7BEBECACC6D5A57EF3A35C2E71404B3AB100A04FBAAC056048798C35678D456C8E5B010712DB4F79637347CD2C8AFEA4AA7B14083842B4F682352C8331250821D9EF3B7ED97275816B5C2966C3E66A707185EA97B84BFEA09A7437A83C18C1BBA892B1C8574C21A19A0BFA8537322B90B96610CE3A9524BD8BF19285473D3A2B8314FC7AB9D267042480A879252C280F76561879428A650BC0B3757062C16F61D131AB1DA83443EB547A4A641BBB09D6E3B183910656EF8A7DE93108EC3CFAF8A1FCD2214A1056A2F83C6F38A4867A00C9CE31532C141872257B32C500735CAD72842A6A65200696C2D723C0F3147ECF0AD864B219D6616D36C7389B03014065A9AC69155B08CBDB4BAED38CC06F2C121636087783D5299B3B07494DE79B80A1AB59B11B5302C5EA0B18208EB96574A2DACCBB48E4645058203829B62C709A9F74AA0B9C890B2FA06329595FEA79D291864724186F1D98E010B73A380B830958B6DFC811B4143A5A4A570D17659C0A448E40097FBB11B35BB66EA47EFECA10DA14D0B7560DA5023BB2B76B37C24725C6B934B8CFBFB32A7CC6AE556C22114BFC7C6C5E3B6CA2EB43244A13E7A2C362686A54AE66368559A9A33AAD1FA459F786249135DCBA89163DBAC0414C7E769CA8FA8975EEB9ED2960C63CC526DBDAC994C54BAD10A310D76190D645207F683CE9DD28519E6BBED8D42787371",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "B9D7B2FB98BD6A3251534E088C92B8E5DD7148AA3A66F9B8269F9B68C8544C310DE000E8490D449A103FD744767006AE2662A8993BDE5DBAC0FB09DD577B7128",
                    "A2C7B45E3CC718B5071FA09041A84C1476B4D7081DE0890EB1A76BC0454D60A87CE7E299BDDC8DA4D15C128357BAC549A756B7F1D8596879A293C749AB44A6296385BED93271893790E36C9B1A4C390363CFBA9F1BA27EA546AE15B99249B7B90ED83589F7CC4DEC65F1E896F9D18A90A81C774301DF221C6674A3F7EB861B1CC70E57699C6652AE8510474A53E236BEB3AC54F34C64D0F0588771C4312760BAA99034C251FB69952BE644EABC042BBB5D68B7B30CB35DD772BF63A248083B97DCAA395F607809A14F1FB791F023240B98A9593352F8DC8C976CAF1C63B4B7ECB161632C35ABACD67387A14BABC4044F9526B29BC35B4B923D91F8649D60539A0939645040E2257F570C32E989CD8E9CB1ECC288E91A389268A2549BB1D38C7859086B7448A91A5B8107C25B2E053E9445780EC35F5D753B9205ADF2B3C0593C9EAB2A217212AFABA7A6BDF38E59AA7727BA90DB59A9AD85AB163662C48AB9F5E4B907993B2D66C1B9CC5D58681818F9974FB5801E8255BFF41436C9813E8660F278780D807A498892DCE51CE43452FB0C1BE84C74BCF313578272BBB5127577C2A3E3490000B7CD2349CB49625519B124649AF331CC7AD06B28315507462A7375AD22239C6DD326BCFC5F1C02529BC35715051973EAC90DC4C3D1227AE6BA8E150143E76C5E83ACB6EAEC6BB3CB5FA573C94D46C39EF019D85A4C7A6614794BA310A0763B4A9C358434274626C6162C94D44867DB00D9886AA3AC8D0260209C4A241AB3444D513A86829B8446913DDA4FB0307124172993B9BF80B067FB2B9EBF082FE1821A5FB04131C0BF8AF0702B443251771296C047FA6A1498DA8E5618426841A22A2A4C25277ED341C830197B716A96B380CCD36481F33CBA46F828667CB1B5AB5F07C51D6A3B863F404D93E548D5E3905509131D842FA2247043A9C05B4838B6B03E11CC6E085115C8F408EBF05D600A4A68339892012666257B6F0816C6648B422A38068AB045DB34C41BBB34AA270407363010AFE89CCC372A25D2EBA178292E33837B399271EA4237B6C580C41750EFE4ADED3361B2185A1C73648C5B5E04E760DF705268122C916EA5DA43C89235801D975812BC1521C4C7A9D3",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "8832AF93DA473A24913CED5C41630E344B89915AD753190F1C60F5F8E3B3C4E06EBD1DD09EF84345BE9DFC145D865D60B4E43598DA1AE4A5034D8C1D8CF9C18D",
                    "9236226CC39D8B9B08E2A986B8534BE09400A1182D61B699F7C44FBC9218417A5AD20207B79A77F70A62358900D827C0BD605535E18566952956AB917D28815A7C629DEBCDE809B44F9C54F1CAADC162998A0CB239767585F15F2A45575B497DBF805BA8D000F469A9E2BA1697D41280AB54EA0A2D681C1F0F34B703C94C37DBBF9B912A7EAB13DDCA070572A8AAE7C095679C9A15813962B53FFBCBED171A6EBB2E669A14048B97876279E90C8B31004C1F0510BAA797D4E07E47A2C1B9B9593446200BF54B98A9B19D2C04616904CAA1BECCEA231EA448A5C0057F52ADBDFCB4822A2A0EB898E8F6994039140F2BA1F11CBD8240B427793449F2CBBB054028F49121F509FCA849F9036BC7646163EAB7FB041EA80CB780EB7E69E56B49F9B9C2F5011FAA8CBD21BE144321F4D015695591FF2CB9C44A3C9506740721BE0A783D836CBC2AD7886251854745845089920AE59C07D2940B310F9FF45AF179B48B03C483D562AAE4310168884F0893A47263F9DB6228C292F1BC6B5F83147276A36A32A488ABAD1CF90666356CBCEB9487E5951BAB61F91BA7501A7D55B89A15319277D71DE43A96160421AE605B91871F290B4E746B0C3C90C22279367AF76AC64C762FC6725C7B4BAB99A300F29783EA96D9C6410D103D56813BB974A7C9C563D9CC6F22D620536406BF0C4B2ED2185F612B8B685DCD2200171869847628DECC9C436122C5A42DAE482A6EC3778C71926260AE4094471C448E6CC839B1DBC26BA57870C856F1DC5EF36C23470892CCEB95BDE336035806F936A015267E3AE87E81994B5D220FB04B03C971A60ACBAC9152769D56868B98808AEACDBBA6291946CA27D7CA48D7717547690FD9CE9A0278B1F56F76988102B5613F865DC77021878397CA27A79F3A8EE0034A042885A9E188B2E30542AA0405245B269BAA3C55277CA6A289531208AC9C6BAC93F7190209FB58488C7A5BE3C537C8AA118C8C3238B9CD68294784C81DD54949A7293C2B6B174A3CF2AC5DB36543AA02598195BF5A4AC1756A3E4481127B111EB9343594F8CF7367199363A0E292939B951853216C27C423399B123084BCCBCB8C9A6649493E775BAE4533F795CB8E7FCDC0448496",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "12ACE472A49609E131BD7627EBBFD9925CC0FBDABBDD5E5DA7D8B43A5694D73CE6AAF266FA60B4FE09D01C8EFD61B1F7308BB1462379740512DFD3599413B14E",
                    "F82C8FC5A972DF3CCECB329B91C9A9ADFC355B0650D2B66120CAADC1D394A9C77243048AC861445AD553F5CA87532B3E82CA909BA6672017AF5C3236A463CA40AB83FAF05074B810C09725F7F901AB5209D0B53D4F290711B0052E924F93B61810EA8C4C3A38F17B5C1E982B8DFCA3D084992865395535C41E3C533B1C8415A13BFC16ACE1711A6BE6A82D0A396F957A71831E9764BFA6396AB5699E94651BE32CAFD577C96017AC9B12258241A9185A81E4C72C62B638B893B9F91787F7B65F620BC2332C2908F08FC276B4A6B6448C52C613A357511C45F130298D97CE8465851D9B9535D063E1A4C6FFCC4C9557A54D039CD0E9A31660B60F8B0F7F758E59EC6B45C395046A24567BB0ADC488D420BECC5C91D2D091712474FA25A1B06349F93344625857B2E820FB6C1FDC3CAB90C69EB093780B9CC23879B23CC54804571A3CB25BA5CB707FC319D1DA433F502019BAB8362C58A5F4120AF6B65E0316F1BAA619164CF9B358BFD361C70C6809A683C86C870E92379AA77B15667B9A3284D17B0025109BF1374A2887923C184A82A76669B259CDFB33C9ACCD5E0CB6F57BCEAC45B29C009411298370A9AABAB0AFFD4863E9786694F2251373BB58F965F3A76E252A3FB4D50E5F705355A53A92BA080482BA5D5A9660135B96869B1413464FB06E1AA46B2B30BFAC6618DF852030BA043BC721F3651288900C38BB3E15CC8B6A86750F24B2A498C785772E86931F54CC89D0D96964D5748CB9CEE9C4594D7116E590BC4D8105D7B1AAD729542F568CD64300705A6F62348FC4656C21FB280FC41B2A8125607A1F400915D7BA52357656312A2579C8A8792B296FDC6DC98322734C7DD7801BFD6306EB620009272485CCB8C3315507254CB7B11C27758B4C1C465BF9A38D668B2A3B97A6B564BD022208646BC92B5182BB834F397BD4C48FD36C3DD742BA631BCE9D184260125F7E0847D294C3AF430207AA1FE7CB7840F00CAAB33AE0C6AE0BA97E4AEB9BBC34C82D0A3345F53969268FFD77066D8C400BEB40BD4444BDF60EE5B357ECC99B83CA546E01C3E4295D42A034BC2A13F54CC699E4DD47D7BCEE5C54356A407D12C7E267D4BF17605738973228DBC836E0B80D",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "77BDFD425F4C0699AACD7045E62093D55EE7419FF244F75233C9B597AEB0CBE75F8985ADBDBF2287F3127CBAC269ADB9B364C5A9FA4E69FD6BBB0182D4F9AB26",
                    "40A1765A99C6CA801DB608C6DB79ACC2615B690991826BBC61501900E170ACE025C0D8771F79524DD5A6BC563BE69C16FA0A2161866A42F1955B7405DB52B52385C2D5E0B98DE180309666B2B53D4CF62772CC56515C155CCB458179382FC82240ECA71423499590BAD96337F6904DC75AB63E8C9596630966E8310378638C8C200FBB2921E98B21D97FD8066C57CB657C57B5674791C83C57C53930A7C92C78B44C31916D0A06CBDDCA63549A4097243538623DFC4593FF73A0DC170B1F013883BB9232CA3D6B6C3F59925AC6DC63C168668198C6A78C44DCA311B9B25F5AC168EAB2CE0CDB9B2A0C1776B5A75EB51242C414E74C49C5CC4CCA6A656A95233C79692153051B115DB3B2BA02D1B0DA6478C3B106753237704459FA04AEEF5916F756428B1B7C003C10EAEC1965A41E03F63411757E46D6BBD908D0621A42643B2B7A186A68695D9CB20417233A79D433F5498DE3A548FE66738A579CE9F2BCBE322B6E2B14FE11A571ACA89C80596AB76A645117FF4C9AB9057ED1CB45B9F4568D26A05A4A250282C112FC041D170F2E627182CB8A93E3C2F1171BA32606BD375EB624376BCC022F0B3C43AC89C1298B0DC4C691D8667F26C9FD37A671D56736C92B46913346FA0E2459CDC2F893A330683D6B5EFCDB162AB871A7767B945C50C0D57CE30124CFF31C4008AD7D12111F751810C981CB370B0D84BAB3D165409C0C499AB764487C69A6411288C3983A748DBA2C21977C911A18569B5A9691B95EA5C1BA8B3998294738570E029505306865B172B25D36996E734676253D72C07F0D285562E3893068BC9496783CCA8283B8BE719953E301C7DDB1638EC5B1F473B2B7A8CF24501AE167B46F2101B2C83E18FB4C63658A24535454413D69692BE57B4073B41F63491ACF03B2786CB2294208CCD689F6FA2E11908A8DFB201759B8F546236D63306102BD4562B4F4F02E32610542DC8914806AC0F14CAA67985CE458DD0746780B5AC6B2A825C6700D9AC119A5CB2E7B8D7D5B74B40B4242B28FE9F6B8846613616741E80460DB19848673BAFA6A2F40E0B9DF38C304219AD654C721BEC2D45FCF567AFC43F48C0EE3185A745399D3562034E732A82180C21E4C4A",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "A524C3BEBA5320D30ADF207C6EB093DACCF28DD76C127CE45E3E521447F1CAB17934341E3175621AC6BFBF328C6704F4022F143F2F9FBAD942F5341D6EE792C9",
                    "F71347DB4300F8645F8BD4A98130A9C49CC838F198DDF1306ECA75EC59A813B731F2F554E4C11AE46B1E8A6B313EF8486B2A408B572DE7553B36B136DAF2209C0852273B596B5C2CED878226621650961D8CC804A142B7A2D78D027A7792F7850CEA24B8747176060C8063585F83B8BEC7BDCC1C1350D2CBEBBC716D404D7C1BA40EA1AB453B8EAC79070F45465D412F58D12A80D46B1DF99B84FA51CB7340BC3C94FE67276AD15F723966B06CBF1F8CCF8E039FFF21483D56CDA1B86AAD3B96B143664C8B4494A0B841F7C7E7317984316CF12530697A4B1B75250D032CC35C5412D77031854200E41C56490F0FD27A0C59019538B9C61C937374CF156611DFF4CA218A45B62814DC49A090A42E10BAA303C17E1C123D8C898C4D9385C6878B865CA3FFC61CBB75BEB1C918EA2A5C4679A4A161AECEB1242C39802DF0A9717255518430A3F828CA41576618CF0D456741E3574EE0C36F45088EA3A22FC59A5F0176B4578267FCB85AA97147925E4968220163020618611C3C3DFACCAB6F3007A1C1A6774047999B4160D3C614594B228222D2B20634C066AB30AC35A6895305503905C14796BEEF8A3DDD47A8A4194C2A601224099895322D43E86633BB7DB3DA030674CF04F960E3ECAF49A3A287136EF9473922B1CBE711257C76B09F303E63912F7CF0C5C0830197F25583C50DBB491A2CE01E25BB101C5576D9E72DF36B05E1900D8F385A7F754B27508215EC6ED8F52DA47660441A784E99589ED7AE4B197E5E726DF41599C9F6BEBE34022C0781EA42BC64980201CC743A227DF759265921B7AEF22CDF460EA04C37BC91A16402981407CF1C6A3EEF165BCB091CF13779F267BA582C431E33700FDCB14DA57A191700F18236C2C054A5690BB05BA83EBCCC01F5014160C3EE2BB08998978C0531D90C416C95A80D1BCB97054423572E1728C704175B6CDB91938BB42AA2C49E4237AEF174F6F61FCBFC50EFE362A6718FE823AE174853640B2662C5C467458E62A6A53B2B42178087F6506BCFB162EB26C30B0279BC5893D784B6A14747B7E132119C14871197045499BC66358F95A4A6FFD534996B1692277B792964F489319ACD20A3B1C472A9085CADFFEF500BBC",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "34A702EF41FD50C390A4729EE6D7725E9B4B65B22BEF1962FF1E16622E5ACA45452BEE93E45A01BD819BFC9274A4E7059BA4E491DE2E3E233AE64942BAD25E8D",
                    "437827E1D842BDD37E772A52A4F0601303027F32A4FCA965F57967385657FF981034D39C73BA5EC59115E3C8CB3375164F5B1062394836C4448461C2512B8C96205B24B3BF906167A7D7630DFB9D67632368973CDDA76141399055E6912A8B8374EC9DE265B877F91930D12E2E827FE6CA4E665B0557E21A82069685930BAA442EDFD926572A0825F11A4DFB1B328A6A62D09FDACBB22451289A261416A87916B700546BA5D61A67DFE45CD6C9208C545E5B422A897463BD827E5E542CF28C30C008AEE3F8472A38066247297E1A4880CB4B98CA9C3CE8B019C049441831E2757CD2A08B44600B2C353F1F0B0A7270CF59B6C6AAD0351486018B74AC21444EA9C1451342A3215624AE72653F75C9738B7A27C7349C8AAB77F403742C8CFACC626E5640FA517C53C57D3DE900C18816DA364E7D8A396C35AA268AA998040FBB9B076E75208CE22D87FA16ADBA2EA47769DB1560FED51C34B1C182868A47BC04B1BA632274650AE443AF42A09012BD985C9EC5467227698F6EEA68B0ABA9ECCB979148C6D7E90121CB19042851F56B8F92078D48E6B17C82595F9292ADDB727FCCC194EB224156A18A73C65F86C3BF27C0BEA33BAB0B3493D61C75F5933A383AA6E3220F557D3BACAAC7011870591D03F4AC94E52704BBAA1B697025D619ED5A7D1B251B93B237023C6078D124CA532F1DA64DDA759B6036CDC4A606F3079E6469AB0AF701B986BA4B5B26A5B84E3BC0045A21C8B2B4560CB081745B00DAEB764AEC840176B0A25C397D05A77AE1C7ADC15D92D41FD03CBE9F97CD47B27A52F18C7579C2FB025341D696E0CB80C3C1798426BD22B0A0D616CF1E1809CC5C08E675AD76B623E7D894A306300C5CB0DCCC91093CAA93D49576306278A99808EBCE2F582C7E2AB02094481BA951463A8746AB441644B68388BD0C704043E1AF94E25EDD6B3D45A676FF02458895850C4938C531BEC7B62488B513CDD0878EE766503761955298FFB0152EC31DBA428F3DD51A89F914C103B5BAA983B7B93476C4A2F4F81FCA3372E1F8020E52AEE0F64624108AE73691172521D6C52F1C0101011186476FDA16B16D602FA89ADE65C1573353C7E43D488E165E821A7595DE64244452",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "9CFE35504744F6969872C77278049482A0D213C4FED821880221B44004F35A13E68535A032D34A9FA916F17828244ABD2C747E59CCDD20D29E682492E39FAC8A",
                    "202A8FD3FA108B9C0398C8522C016492219F5BE71ED003CC254917521A0CF42034A45BCE92044F32D048FDCA144D045BACD4649202A7899036D5E57DDAD53A43E675E61815E162029D9C39836B6DF771294F5623A7424D9B0A0C2CEC40AE7B7A94E92063BCA462047185B0836E85CEA6050ADC616610D49BED50CE87EB6FF803CBD0E8A2B4C967F0E20B221B1F919A4B5DB6A60123C601793FE6DA4334FC582A459AF7381582C4057D91BFC243B9BD88A1C1829FD9AC9E3681CF67A4B8AB69A948E030EDD48EE843C43DCB656F46587E4A715DB47D7DFC30BC353341752DA204AA0C44531A749E4E163CB6915F0FE0A8A6D05AE3FA2637D893BE3AC624865750F8A3D134B1999A81E7884AD555CA0D429AC9AA9BE69C990B5191382089F40661912B4C10F49885612C065A55401B97AD195892A3800FD435C31B0ADAB331EC3884AA77A4B8E17F6CD22885BA0C2F90768E983C67D8990118C9D69457DAC4057E0C48F2A81517D9549689196320BB10E87878596275F2112C4460F7848C83242F9D47CC7B937D12524C1FA681C230B1FBC0A54B231583648ED7B16EB7F297F4C901B87849CD3150226CBDCE44355FA218A79519FD7A881AA21981C6A956D47DE688A874769F8AB40810A25E4694073D7579180CAB1E02C95ED0A6F72A42C46CA668FC016B8979D3957525CB632DCC1D84E67E990C52917B5C6B911F58F7B4859B30BEB6859A72502E890327426D79ACC1B654C5EFFA98717B1715F9410489A9EB89A03009C5B70333DDD14C3CD37E3502CA60856EF2552583AACD369A25DB855EB30009379A84E5927308695A80BB200C821F904420920A40FF2B47DE4A0E3FBAA8CF3CCEB21C11F59359305A91E95B69F1E755E3807E2A52B9D6B0753D58A69806562F7285773832E46A21CD8287766BCFC9EB1AE5046966BC2D1C6955954CB92C6B82E9F8C3D5900DA0707779EC1C8A8403CF6C06B3896A48999601EA5DBBBA1051C78C5C25B4404A1DCE37098E5CAF1E6420D7AA87A312B52FD89D5059133FD957AD5508A2956973F69CF4E35DF6269956832C5F4A464325B28B92AB841741E24219E52C7E722CA7F35DC065DCB1D15C0FF7ECAA99ADC8BB06024F0C8015D5",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "DDE35BC33D452011114F5271799C21CA740F0FBFD34CD6637CE8836025A3CBFB8FB1127BEBA4B6667EDE28F67D08514443F4EAF93E8E46F05F4767F4AB9EEF39",
                    "88E9CECAC580BBD381649276FC4872B11281F178074450B7CCA2B3901946FE865B430067458082189050A8B07B5EB823F4E82FF8B7490D4A8D9B639E3464201D0C443066358577489E0919DAD250A3032B37898F36995118C70AF358051B7AA2A1F9C7FD57BC22F4595C4610EDD040E69229459C91DC7C39E240A5B11BBAE4DB46DEA84A1D9A2C3F9595634019BDA2B8DC814ED6C9404DF4BEA4C5C70BA4287F08A07EA25058B0186E74484E8741FDF50BFC3B5D37E34F9DCA9374B77E056454B1EBC7E92235037C6A9C5A44D293C722491CBE7C54185CB0FBF817232CC25ECA5739C386F314530E6911D3B9B604197989560F5B3720B8D1BA7F18031A735A45116FD5932EC947BDD8A8026CC781CA32C92B082A226B2D7664BACAB7A175CCB553016E30382A18810AAE139ABAB40E0F749BAAFB693BC4AA69BBAD00BB9C092B5A8FCC55657C6C01199971A61D5A13CACC76A31E3787BD909563D4036653A23441ABC5BB54EE7BB743153F6A875B75A33703B29EE7B874CF545D1E1418425975B4B0540EE65A78E8373B7B4CD4282C88CC3710F8288CF078BF654F8F18C7838C06A7B234CF8B74C5B930741C1CD1B7698C217D8CC91ABEC21C07B8C60A2826A7D9834BC881F179831DF9014C444DE1B37381D93B26DC90F8D1C0C0793C8F2A19E66953EEA336115558E8A51E4E1A851898B09440A686D37E0CE62A7D2450A747C8C880B3BBE3C4895AAD5AF1144D342BCE122AB45A235677482C26A69FCA137820CA5E1A29B5E2687714C9BB16662A5B47BB99965973478F22AD408191A6180AFA57C2AE97B7FBA081B0CA6337C46011B64AF187C573157897C586E1C22C8D8B54D621AE879674D31C294EE226D0F12432E18959B66250C405C6632384B09EDFE44B3E6B2DC053912DA55364F41F7D2926B2C38926F88032343EF5160828CA3307B2380A52AAC2E98F98555E7447002269C9CD8B1C9744AF37D43EBF433DCFA9C6C5705FB58AC07C252A1B02CE20B6280E3686503B13CE915D9BA8516DABAAF00757F172060443734A7C0CBA770D6C758A23148114C98F7208BE0242BB9664CB86E122E05F76BDC74E0747413A82CB261BA52FF80C9C2F1B498F0A2D88D19DC8",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "CA530DA877D570D5BDC677702D9858735E63A7ACF6079FD27EA695F3253787CAAEEDA4F351997FA6552ACC9D38C4C6936208A47CD832A86E475F42F67BC6FF8A",
                    "2C70AF1E561D8647C50111429F9218A0E699B048962E89ADB1F00DFEB6BEDFB8520AA02E98362DCD8184ED211A65321864B04373ACCC02F0B33ADC9491937BF341C8BF736DD31ABA15630E8C7CC95BF42C1BA3BB22A1B87D5CAA4BF476C5078FC4D579799349D32952FA6A65F11376EA4520F31815C10040632A370AB65926A81F14D72286983F2D15868EBB753DC278181B8027D5541D427A354960067C43CB404695316B7B694EC319AA77289E3B89556CB598A91789A566AF00E40DE7DB3D1E854000ED7D13F2ABFAF1AE59D935D912A7BAC8102F496427870BAFBB949A9A51A8370B267C93D92C80F5F3BE18A92A49722138129C98F26A0E3494A7AC3E45465756F1CB329964D78062A5209C6FFACFCB248E780381A43B9D71649AE44BB6264B46C0F68C957405CB34CF27323C11A243D50C9907D897FE07C9D0A77261C42F68B964B2CC8EBB3446D7D99711022718D758D687C44EA45B556CCC2D172F9F0A634CD3410CD3C09BA49A2339917C8022EC163640365B9CB1358E914659211136CA49E6F6B00D738F07EC9C7787A931408A141B9D1364200C1C736CE7C048E52644B2C3A6FA1B7F814C253074E77A3CC9829BCA7C4D3B6C1BF559180FC7699AA1C6D8D761BEB8BAB7B4846E91A038653E78E6026692CBADDC3CEB8B37BC5123C7C50553A944328268EB602EE39556B60BAC91C05DC250991B8A08370B4982C126E52AA6643511BF9309040A3166C03D8816A00D9B3673F830C8A62873A3572C0A3DE62079653178AF56A26F429E2A224FBAFB6142555B99809B82B9064BE5BF01F182F2B5A612283B1A86420C2B04A90AA031D85CFFC61E13D6727AFA9A18E5B1C5859186FA62C1684024E56C1F68B571ACA4270185AA6A83B6CA662BB156541B157A1190A5A9B3E0555FED0BBABA476B984B95529331FDB157B379A6716980B94556075119453964C57BAE0E33C2262989B2465C74C32337E30442C121A80634B6B12811762B5A59231E98592665305AD62746EC5985391E73EB56A1253C63B16AD05C2AE0BB0304A679786928BE0A7164A7AB9529C341A5497284C3BAF867D9A83DF9BEC7A5D38C721C1C2326116039BB6F21281B86ED75F9039D7FD14F60",
                    "FF135CC3D863C199424F066600E732B2744ADAA4C0D0B077CAFBA7572CBCD79B3EF68B6E636C155C142187C1AE094C81507BB4B94167A5218C03C82BB1A1C1BCDC1EE5E8768F5855186BB86BD2BA55408EC482499691C5A8B654DFB8C2C40241843B42D7110C853A06610A6155BA7CAF99CE11E8C3B4D2402D47C53C2AA01A8CCA2F6531BEB5896FB30EA2FCC63DA5A4FBDB793EA940CA4479776201997C9025F6BD691CB21E1B42F296700F27336F2C1E56A468F1A940C2D8A3E108CB6B863F2C0056100431424362CF5AC0D31984DA1C9BB04C5E6F5C36551857B2AB4F4DDC7F68046C45D93593A208B7342E4F00A1AB978018124A9C49BFA0E95F02706CF5B9CA09A11BEF417AAF549B3486642FDC1E022597244374F3DBBC745513C03A7D6173A0AE526563723BD24227A09C81E6513A2D8B4CD4D21B5299438FFB242D7364DEE7412EE259DE56AFEBBC8A1887BB2BF1679E5A148C54293A68B036732E6A59C77EDC8481F59C52B2932532B89F17C5B0A679C6BA984C039BB675717DB326A9704AF0C7A7E91B2F49DC33867A38AC583025108EAC06353FE7A7D9524CC36CCA718391438669610B6FD522230E44AA0D541B84CB1A261C687AEA637AD1BE22BBBB6789271CE925E0ACA48EB93322F5014F147BC9542F15B9C2156A37F92B99FB90118B5A1AE73A75F35626ABF2BDCF0C0A803549FFFA7A19004F8435A7E8B12D6F88ABAEE3CC41543B42890CC0CA9DEFF16957640EBDC84065BA52D92750C1A395DD8597D2A3A68C69140E472F7373497FA939F56C67A5A935BF5616B577B67DF4591A653B1BF543805C7F0643C18E59274978810F36740CF1815822B3855AB34A87C5D9DB8A0E92B713422E0A53BB41978C1ABCAD6053568299C7F66A9621679B8B6C95B7993C4AD57F43C27B0CB8482D6A2288B2C6A2D6BE328426CB344079C587F17A0F3868AD5F4347729CA8677C12427A4A31372F83818A73C53660E9399C0A69AF0432AF382DC83C5F1B181AFFD523BD05CBED92C785EB45DC1C8478391709987AFEFC2BA43B5CE96382F2951D71DA3F888A063AF02E7B453D753B552C70AF1E561D8647C50111429F9218A0E699B048962E89ADB1F00DFEB6BEDFB8520AA02E98362DCD8184ED211A65321864B04373ACCC02F0B33ADC9491937BF341C8BF736DD31ABA15630E8C7CC95BF42C1BA3BB22A1B87D5CAA4BF476C5078FC4D579799349D32952FA6A65F11376EA4520F31815C10040632A370AB65926A81F14D72286983F2D15868EBB753DC278181B8027D5541D427A354960067C43CB404695316B7B694EC319AA77289E3B89556CB598A91789A566AF00E40DE7DB3D1E854000ED7D13F2ABFAF1AE59D935D912A7BAC8102F496427870BAFBB949A9A51A8370B267C93D92C80F5F3BE18A92A49722138129C98F26A0E3494A7AC3E45465756F1CB329964D78062A5209C6FFACFCB248E780381A43B9D71649AE44BB6264B46C0F68C957405CB34CF27323C11A243D50C9907D897FE07C9D0A77261C42F68B964B2CC8EBB3446D7D99711022718D758D687C44EA45B556CCC2D172F9F0A634CD3410CD3C09BA49A2339917C8022EC163640365B9CB1358E914659211136CA49E6F6B00D738F07EC9C7787A931408A141B9D1364200C1C736CE7C048E52644B2C3A6FA1B7F814C253074E77A3CC9829BCA7C4D3B6C1BF559180FC7699AA1C6D8D761BEB8BAB7B4846E91A038653E78E6026692CBADDC3CEB8B37BC5123C7C50553A944328268EB602EE39556B60BAC91C05DC250991B8A08370B4982C126E52AA6643511BF9309040A3166C03D8816A00D9B3673F830C8A62873A3572C0A3DE62079653178AF56A26F429E2A224FBAFB6142555B99809B82B9064BE5BF01F182F2B5A612283B1A86420C2B04A90AA031D85CFFC61E13D6727AFA9A18E5B1C5859186FA62C1684024E56C1F68B571ACA4270185AA6A83B6CA662BB156541B157A1190A5A9B3E0555FED0BBABA476B984B95529331FDB157B379A6716980B94556075119453964C57BAE0E33C2262989B2465C74C32337E30442C121A80634B6B12811762B5A59231E98592665305AD62746EC5985391E73EB56A1253C63B16AD05C2AE0BB0304A679786928BE0A7164A7AB9529C341A5497284C3BAF867D9A83DF9BEC7A5D38C721C1C2326116039BB6F21281B86ED75F9039D7FD14F60CB3C2AADA4973514F817346DC7415C0DFE62978860046CDA86E5E9828A2CEA84AEEDA4F351997FA6552ACC9D38C4C6936208A47CD832A86E475F42F67BC6FF8A"
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "57E4DC20779D50918E6AFB7C83C46E75468A722BD30AB0D485A4C53DC102AE05448ECF05372ACB6596886A1BBB2416BDA3BA5557CD6BA226EA134B4302E71AF8",
                    "A150BF17218357CCB38D267020D9113D22B61FD02EEA861642F89B1273C979D1C1F6E53BD3D87DEA8804C77CC06AC40144BA4FD00791C38036F63496AED2713809798F44C365991A36E81DB5E6759FA440B2C78BBF04C4AED9A46AF369B493627824A73C477539459494C12B5B378C659B5835547CEF9B20B7E07429F7CB923B2FBBA778936200A56B9A4E483B847C413A797941716B51655045E331312415DFC60C3B4713A79B062868049AB2B1EB120B3D4392D4E915E28A6085BBC656EA2023039D5A651EA2CC275267005561411A76B9B7752AC9814B34076FB96AB9F238C951D9400C544CB845863851473FF29F13D542EA8B0533F552AB457CE9C43C9C202C4FE0A09194B321306CA96C599E110E3057C7580718D9E4C0E694C3267C628BA69F0AA55F3B8BAC007584832C3C611A33EEEA1D309B260A6989C0FC202F1490E9AA4397D960E9D1C934417639025161ACC25E3AA7857536F2B6B3930CA1F48A7482E90F074B226A990038A65657C53D96640427D0C87ED8CF0A1A9D06F740A8942E31AA5E0D0B25F79C7D3C6296AC1C79C1CC197BF1A528C179B2C611C114C6C2E07C0030C55B5895DC8C36DCD625BCE0A00E5CA75C530454D80D42AC2FFAB7288A6175A161CA841160B5A5970C34017E42792AF59DBFA5912BEA3FE3BC0D0D5392733B08E030805EA61894637F3F988F9D8A80BB0100ED357C773481F1C981BF5A47C6F76916DB22BDBAC5A0BBB5C8D92AD51359992B99CDC071EB02B11ED6A1B8AC11CC4A34EB5811D652608B388D5CA7651718966390AEC6B38DF5F3BB670018CF5209C819BA63D400480CBE99DC07299985E22994D220589C3C6105779275A699D9A4C6F79CB8DAC45AC5AC615D7B48B1B0327442AA105C8B8CA443A9A78F69315FE8F55BCEF15F96196A531CA89D73B61DA52A6351A334281C7B8695790287B540C4494A263AA381CEE47632A00B11488DE0E97AB2863522556F382A5EE7EB372589C20CC118DCF2C938CBA9944B5CF18C818C8B7A8E98600D6C26A33369D06A1D8697BDCFCA3313E86732756F4DFC9B131AA356A018B41C25DC84A0D0F6F65975B3106AF8BDE7C173B18335125EB6DB5290BB24B7F432691A567034",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "A0B7E77AF241CA39AD959BB5955BF7DC8BD03CD17DE97E735EC1D59D834790B2F6E414BA470D63920B930E3DCAC99BCB238658BE8328D8B4722E8855D3D6CDBC",
                    "462BAA997A8C7DA79A6C96A216E73089A3CDD1F38B49E4A5D67BABD64C867378ACE9BA902D511DAE2604524C54FF3775B5A9BE870488672C60C8E8973FA15F13FC6412E830C96939E6490FB0656B2A352C56561D608264A996672BEA5CCFA7482460C4F8A0972A46A10661C8F03188AA9A3A4F763A402B393BA18EE2934C0B0A5127110C69E67C3E121420A523547B45EC3291E7441A428895C7739B895268DEB15E6CA31D8E14670343AC191070E5958308D1B4C6E4557497C7417B4487A9BEBB628E682A2037BB4A8777B32E509DEB44940A7007B3E5ACFC28AE0247B66B13C3B76931F1698C6F6C3F7A97C813C6B30B7698484486896289F9B31DC929398E376509CAAF11F950CA31C6BAF94BBCD43DC3F12181320057F7B54DE0711A7581FE860FCBA00E2A5B537005153D9C719F549A3C3294C90A52107883C45585F1745FDE397C8188C60A747839BAA84640091E6747EB08893956A42E621604D57C5C623893E729B3E340925710DD24C78DF87E4E73691365644C2C7831136232520C2BA75E425493B02A3403B041F14937C85034EA5837BC989D174BB577D090475936FFFB0F7528C20C236785BBBD6658AD599B960744998827B45BD5C2C638558D587721365226F24381089065C19166266864631FFA478F9C6C992130736DC808073456DA231A45C3557D3326A0526AE680553CE3B329347507938E7E764FC006B1CECA9B09F37DA7866ECE5702173B652AB4BF06F84A431AC0DF5AB77E8C4846941B8AB064A8738AF2455ACFF481E370C72C255BFA357C64E92ED7F30C9160B635AB2C3E103CB0B72772932F47A216DA638F53E44250A76C52A70AF271531C845A7E696DE1D04BA9EA448D3A396E7383C92811F4564DDCD084E3C67AACE09C39A73484FA0D6C6A03771C053E45ABCED48116188540270B0D32329CB3C0DC16047F63B30B87BA67E3B5BC5B7EEA558939E25C13213DABDB3BDE1637498AB6B04C9666434FD5822D630528C470C462F92806A86F81F9CE1D7760C66B8BB9346A16037891EB3E40173FE47076F9728D83546E11E753D8F5C5A90A7172A58EEE79A6471695141F80A35AAC292AD3B45C42EAF9153948F4951EA33F7BF5753CBFBCB1",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "B280280A639A63E76C9DFE81FDF471A0919E4EB062C7C44CDFFB28B78494B478AE78239F751AF50099C9EDE00B3658C30AF77341B59A93D61F9D25599AFCB059",
                    "A243B0EE496A23743D2D0753A47B6A312B200CAA9528ECBCDF23CFF19906D689B088E2400FB79DDED6CD8F4912F5D5B4CE1194A8FA3032D3A53CA15F5F1060D36276D26920BC90CFD117A8747A2996115D5AB54739F85F632787238884C011400CFA9CB2557E37D04B813B5F83C8CDE487A89D39AEE713771F5067FE0A60F6E160C0F8A1F85C9B40A6942A231AFCC6C0161A6931416C80EB9C09792A03E38EAE1327CF665914A026224258361C14C7F468186B528222A508199A4451C5ABD02162E106F4664F56F5215FC086A881938432636DE2BAEEA4151D2C84880101EB562213892CF7F161B45C63FE86B5086BC8EFD26FB322C8DBF237D7C1BF9024471F414B51404605D1532EA563BBA2A04C43659034B2F55323C3F7524B66102AD04AD836B1C24739ED724DDAD021F36954B2A94412E192D3A97B9BF52F62D91371179AE0A539A61332DDE254A3D40FF4724F1AC486B45667ADEA5D6F054E4A13840E1977C3168B9099031C6B908468048929B317836B0F9A9A163831D75C1825209731011BE3159F480BB8D97480110181AD33CDF1E7AAE952231C30386EA16166EBCF56A207633220DBC9B80FEB05C2978C8DCB9683A31FFDCC1F272AC4AA387E58CCCBDAFB3536A840BBA60AE7901835E8636FA697A3B48D30E3B16724C5DC70C4E939AB86852510B65E4B8634E587AB3C80BFEA71CDA5F3296E555D75B9AFDDC91B3BA1B445FCB4F14B28374038E20A2F3ECB0E85604ECD4188CB23AF26CA89A4810FAEA9CED8822C520BC6C6219BBF92B7996384846C8EE9D9806E37BA88165ED17216C836B25F589FBAC77FA9D516A46A4981249919299C69CCA1AC91BEB700A26F408CF7B705770BBC54FC3A44657ABF0211786B7D0AB9600B36452E24AA3332549E1BB2989236B9CAB7B8E4C8B4660E68B45FC438483612701762CCA6C61F70A8B8A222497489CC28B64BB39726FCF30A7AF103302CC6EADC17F948AEE8FAC4539C4F317435B9205D2AD4363647012C3B4B42D66995B437FEFCA0B1D8952CA7C04C2908F9C15499C19ED23862F94101C75167CBC79EA0E4559BA10902E479BE89D92ADDB45A5B1A369BEF80450A4575CAAD582004AECB03A9EC5FCEDCD7B5",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "3717730B2F35291F2F8DF2B9CF628351397FDE6CED8DE18E4570464057338B213FE925B8F7C0F435A07B3B913170D00C989B39ECEE157C5DCF6BCB246AC99A6B",
                    "1449378BD2C64BD10D8BBC3834A4CF2B640F3D9154708B7B156AAC85D3189FE76D10F35CE00C329A487D9A9868562606796427D158238B0B91BE01B98E4C812EB1602C4A42D6180454797FC75728D4E0017CBC830E05604FF9536AD64A46DCB203E6526B009D51604567EB6E3E735260A6B485650924C0741C4652D8592296892DF3F23AF782A1D57ACFC1D429B430C5EEA369D6E314D3CBAEFA9A4C667C39607B278F6953FA84C18008588EB56CB46B61B9B15EEE6B7ABC91308F037EBF3798D7F72724EA3E6CD273A0269A17296EC74AB078A618173A562BC175FC8A5A522ACA8DCAA87CB36EC0B46E3776B9203696604046D4168A02E73A351A6C0DB127E6359D93712FAB89B855F74FDCF87581760854045FAA3BC9FB440D007B935250715B527F7FA47FA3125EA1860907259E43802F1CB4CBC41025C119835BF796F9F96779ABB5D0DB6308685CD01A6A60EB0887A52EECC8B2D10ABEA5C470928BB33F067F29A2A21E52659171243CB215A6C6C96171236CD31F80D930C1EA8C2AE908579333B55AC450FC4896C79394A7CF8C9894D9CC029D687403E545D7E68F3F22A35D6378CA29C3A11456672626C1C9BA4FD86FB77A41BEFB3D45FB6C4172B7D79B51511415C0B012C1338854CB5CB640197F048B7763502FF64090D589180498D497347A7494FC8B864557A7A5BB8B998448C155B1022257BAA0B56FA9974E099739ECC9CA75A587399E6B3B3440048357DB1E840988B2B6CAA67874710B44AECA293BF0935A320C63FC10BC04B186787CA376281935CDD7AA379F5717CE59A3C13C737553B31AF7BFE903CAF2529FB696B08D9A755E3438D9924E77F324B4CC8D87DC63521C54EED49F8E1708E0D249F9373E82C3625FFA95C420B737B665DF54824286868C797BBC056FDEA7286FFB4451E9B434D23316D758F6B92133807B7F1637358C2346377778C7C88E72B792CC28DF32C7AC3407DEAB6B86B72DA428243F457113965552B7A59B6A02F2FC5A96AC99FF60A6B569223601CA73041F79F35E127785FF50001F6650B8A44D62DC35D1B938443A80515B7E9B17CD75712E59A34A1634AA80C508F54CEC691377708DC99B9FFC53174487BF329A8E852472",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "0E381FCE010F6D34612D663374A9A6C99B8EA2A3621D61401551D7D2BA22A622197571D21C352BCAF54A5BE0FCC5C68DC16296B7E033937F0A3B042DD9637EBD",
                    "D5CB9D93A5C06BEA4081B1C0292241208112540A1380F2CB70792217E69516B89596AAA541425E0C5549F9D34779BA06967925AA2271AA0B5C8257873134B0815355DFD7AFE1BC7269F44DF0E49AE96B78F443ACDD82C3B899175ED946025C1F7607C7EF3A390BF025B48C9F2C6AAA38D4AB1D26C2A6916F5E355D0630846AF143FC13B253113DBC096B8BF459FAA3C662695F26D63D6A331B76864C7844769A91595C086D35B9BD34F70B95824799386800703C00EDB9E0B89B211B16B6BB5EE309331C0B0AF3986D2CC3280875B6E6990BEE4370067A093B94C4E0B898D6998806B48028A985C140CC4E89025331928FD8AFDBBC62E020859E746F35C7ABA0AB2F088CCCB18894D78CC7A48128138BA15923458293C824D728D1A47AC15991151BC088599B64E3C819D7277E88415DCA7774B7B1277C89CAB436220239E8A7ADA54299E4C957CA88A06134BF309B31DF6715882C4140FA12833B2A524738A8C870E249ABB635B17A0462287842D4B96197663DF8CA404505170E163B2A67CECE1A8C25F886E1C15A2DA4C5FAD25C6FF7ACFB14C4CFA06CF306959B63621F108D7BB31F3EBB9931B0C2D0C4A973F496249703C206530B4C65A548813C669EFD838C35D65AD42721525A08AFC2CCDE6B9366F9B526FC769C79281A365B3059A2437C797D7008A6F5C044505228729EA4261D04F3C618665309215E088357734015F634A26AE67F01DA9A9D738759118C9F8B994EC751D9E2661C639A0720CA70688DEFC516C905588458AD15EB144A91065CC0423BA8BE7BC7656CCBB766282FDC95BB6A9272C597864204C3BC914FEBD9C3ADACA5D9D13156714DB38C94D751B2BB07A07DA207322169F9E63E95A3CDC6B0BE8D7B8C6C463008B020F247B349305C202054B26858F2EC94BA683BFA1A9CEEB55495C9A00CF194EF90916B08C2F2F4B3537C100AC4B4C47474386B73CF2C1DADC304A66735466563D790023A5C5BBA156B68604829438A20DABFCDF3CBEE0A50B9D37E35634ACA31B7F8B3AF81BC9C4EE1702E089DCA5CA1F28BBD86EC657F73B2FAC9B6C0B5BEC33336F55C9D0DB5175344ECC591A12B2170887B2C3B39E361ABB1E691C3C161C981AC0E7419",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "DEA119A81ABA617774ED177E180975BF6281D2894E381388ADD21317A33F1A5EADD51118D30AC5D0BB27CDB127893741CF6782D18A4BF6F0BAEDB3DF8B1D9E41",
                    "143C00A212CB50B783B71C787B09746585B606BC7BA2400D4983C10FC9333FA44B7B4C9FB6A26866419D38C948D865A6818179E5B1962750189CACA59113379AC2901BB69C65E7025DA1C34025746B383968F845F112841B5A31F4279078D9964CFB74B727A29C7A97C4F7773ADB42A22A19E832AA2DDACECD0AC8E781AE7ADB008F8459CD59ACAEC1211B128435617C0CAA290C535940AA3F8E21124C0A275FA8B017A87419A8230398A61B92170BBA5351055C526C2FAD285281959BB38384C95C0B65738122A799D30999EA734A5272BF015594EEB0CBF195B6A3594F30875F424B3B86516459260851DC7E83ECCE1ED03FD573034344562E069A82F76CCB7647F1B612E9953BD0FA744C092BDED99955060D04EC2395323F14583C4BF4441C1B82793441798523B77BA17655767EB643127B690372BF6E01CBD908AB5BFB58628A4CD30BB233AC9024A51BA8C9327298CCED4571FF34653641038D92292446B772921C83A58F15452D0BD6959CEBBED9715B4414A4FB6641B19BC051F802A41B819E2C3F483BCC5DACCCF08A063C86867C873BAF5743612506FF7B4D25667064D46C4045807E22B1A50444CBD146B7D5A67181A4F7D97A575AA3E9CA46A851B7B5084403AA6A02E38ED9186C1AB36E117110B59A9E742BCAEC7B23BDD92F05695A53A3CEB6D5BE40816EDC38095B9211726AC5C572B60A27AB83541F73362F0C9239749BC665A9CFAEE83BDF79A58B334FC2F1B0516C1E2C23969FD47FF252781955CE83315B62308B57466FC19798E6F48A70125A6206AA8E418FB2796D7070951D293537C003DCCC84D5991DB195799E5AA6E1F64224A8C17DF6660E086791C9B01E8B8D4A524BB32198CED727C2FA67462C042B7856115303A740125261B7B87A4271465837C449A341C2CF47223736116A93069A37208420539EC9CE09E6ABBB56CCC140C5E937BAF2029635F4B69A1BBFF66715ED5CB8DCFABF71F26FB362AF8AB48BCB717CF625325A94698A47A64DD2862BBC87A6F0A29BD098DCCC20D9B898271CCFBD336EFA059E02C58C4C4CCAD3759E4AA08C4C6451B593B388F9A97F5107D775ADBD3C9E76DECD390584A8652A1809775871012518676F81",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "9E3E07504B4426E4525CD54529CE7660B19E344180B99DB5255D5DFAEF172C916CF6809EE9E9DF03A9028A27001C4D17E8A2F7B7AE0B371EDCF1392B48CFE7DD",
                    "21F49DD8A02EA3F2A647358CCA91B46BD82E65B30296CB2F8D8077EC879199D99B8FDB9A09B16EBB21CA7970CFBE93A9D7C548F2E9105B80C3C4415AAA67342BA0646C614223B70651883FD8C5A8917602E1C38054592EF20B0EE1D064DD938878FC729824A617C9B0661B66F441BB54D0BC230459F260B06E32B464687B95F7BCB5F53A77CB2258A4AAD338247EC13624B9813FE50F59358DD8CA86364B98B6F327A6C0C825ACA64536A94317C90BA75232A803D8EA9F19EBB9758915A53A7DACB287F8E12DC8D91A0B59264E64BFE1E377DF401E4CCB229806C5A7402AABF3931471B0C313C34CD76CA80ABDD097B54CA3AB67656C0DDBAB254CA4EC9A23DBB40F2AE3639716A30952C085E37FFB593B784879A4C07271924AEB850487664A1337BD1FB5C86FE727B5B611B15CAACC2822446728B3D36C9A961F3DE08660F5BFA196BF360B3E0638BD33340A5C7402A46725AEB3CB842267949BA1F9643A7F0C27EBE55738C0A08781B9FDA49F1384479E374FBBA6C15BE17AD2E137554224AC861C4EFC408A85453088A9C2A830F0C6128E0641BD08264D2667B10C799DD579F53895C3A84A52F43526564B41C8251163CA33B83837F1AEF619A7879406DEF7498AB2B9FF98AF91518D026AB59B47B55FB6A3B35B787DEA1048216103EB140597689A74BD1E426273B10770B11B97337223628FF4D926F990C0F7B255F36051E518AFDC68CC52FA57B302C73C50C8C6E91C92F0B84F1AAA42E06D46706E94954062E2858CC15B60D4190698B84A13284D8245E9334D39D47A0203C4712749FC82135C360B387407E9A39D9860BED3A49AC5FB9E08DA48D99A535878BAE07B19FEB86D480BB5ECCCA644C2AD8A2846023C4B51FC65E1D60AF586B273D2492C5ACFF4BA46CE3BA17A6269FB2BCB43213F1659B9D599307A338CD33387F0206902E198E7EC6FCA79425ABB3336DA8A49C18AE4CB0C04C75287543B334A7227821D88D98C47A08A6AC251B6D09CFC6CA5ACB83BE37A0DC1D1A930877C5A80889D719D2CD3B93A3A164BF7CE67F598B339A97DE5C4DD36C793EC7CE35511CDF59714DE4604C98B78305ABD93321161CA2661C819AD1DFFAA4D2D34F88F08DE653B",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "7BA3CF854B04DC90FC0B3E4FA9BE56447418CA2E35E96F1A82B9F19D84DA3180EF2DF47A1CA519E4E97865B564F29FA21A249F5EC4DABA6F1DEB2A65E9C60862",
                    "D28A54AA8BB1899A29FD82B6DF8AC8D1F57B26A766DEEC1F6143B8E6E2581A10A397194F30C7C3BA8B122BA54B5E293C3C5054F394C3670284D94CBE6EA2276ADA2D0E5767B527A060B770047CAE4671A4018AC3196C0D66300541DA9BDAE88244686E6686BF36154DF78319D6483A2D9181BB7ABA6926CFCF8C56D2E635511B6D13326825B2BDCEC62051258412E4723AAC542F70A7F381407AB69423FA2023804BFFB2C892CA8371D00BB9AA28593619F97CCF77DAB94E1129D1B7C02B7B6FCB147C3314ACB3DAC2D83B5FD811CD9839A99C9ACA4A8BC931C4ABB9878F504537C8C4938E9015AF8290BA4AAC15CAC4945B3D7EAC59F48987DE36B95320263B62308046BBA557738197634E4C74EE09964CB7BBFDB75844D6B6F0BCCDACD47A9D5AAA6A51CF61264435DA8AB872525B091C4A1A6746F41B5ABC0B59F4899AD6322D78178148708DB80A20621F41168F489704FDF208B39850F2FC6206BA02D369AAB1199FA727AB8CCCC6F213B1F8EB9328DA811735C2973119B2D69A1278A4BAC584E2DB17E8344BFDD6A84BF00217324A48811BADFB2FA273A604B74EFFF05B77919C7DC12E97940DF3C6A8BE0B0B9A8BBF72D8CC0694574BF5A9C1E47C1AB2A43BF05593D4012D7ABF5149929A819891727456E7A8CF082CE9365CC01B7EA9E429B1052B93D61D490237BA383CAEFB32E9C251B1835FDDF399D7108BBDEA689269AAE0C6A182B02F99141C8A5209769C80614404152C3322BBB7A54429596455A1CCBBA5D181F13C41C97770578B334AAA82F8C8C51F93A12947540DF055EB941B587B4BD54795BC212F0DEB86546A4D65812BF108122729AAA8DA6386B0621EB25E790B1A491B04A020A2DC2738F4C28FF52C9F99581E763275FA9A25EDE52FFEA969B4AA5F2A1056BAD8B6B2A3571C6C48C55C21A23988E2BCBD4CC9642F0912000043C7E93F1CE04FF721CB982A8BB4FB9F295C6B60453158F2C6D98BA9FC94BFD2A205DFB5665167CFF1805ADB736113D7C1FE8C002B169DCA1698EE8867B6E2B690B27C7BA016FFD9971447CE1A28B624D0242144A2D31B7EB0165E1F581459AC361660EC89F0EE4380E776E11446F9D1692E158E4DFB399EBCFC14",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "C50CFED6B775C3BE9F845252810CE87B7F4681402AA0591F2CB21111C447144424D5978E539D2717792D7A2E2DD58857B2E867251CBE565C9B5273DE8C3CEC1A",
                    "9D956DCFAAA1FEAC063B721CB3D92B0DD350A2236067A0052C144610DB19499C48B3951A3C4710D3C494B64B10BF33B081B51FC84B0F742745FBB28DB25A20D3B0B7D1082E6BA50337F8245FC87BA37A4BC131B18FE8C1C4017CEA747341A9C2564A4781F43278615FDFBB108DB4676FC943ABC036E16B423BEA6082DC643890404570026F7468487CC31BC5A3ED2A69FB7869C80C5AEA109AFF9722A6B51025E101F868CD5607C1BB63CD1263AE22334257645FE70CA94C756A25591EB600C98E9C906E163F9102BB4B62BBD0FB4277C3188D2A827D063A6D5760138B27C6B5413558A004A458F999C61391A2E9164240D54841B4848C109E64D94659150D60876DC49C55776390F1986F57744BD19601B59C44336B3EFD74C18EF46735C52FC7E7A382027A53842C90D158FFB9C64EDB43CCA9C168AA831BBBA72C421BDB9BB7D6CC2890F88F98791F91BCC7271972C6087C54E06BCDA60C3B97AAC69A5776A82F7DA5751F5097F8D913D9F24522C4B81AA6AF2FC41F12C82FAB9B71B0B827F7068221BB8A93935E851820E7FC84E55C272117251F3432F26C656DBB54D8287594A525B4CA91A3A749D93ACFC29578A54078F4DAC339C48BFD067D94D73C8ADAB969A21CA322AE91A9562F5B3DFF4A7EF7989399743B696CBB1C85B7378CC00755B1944950585BA540D8C9E09C73DE7A538B7C5E1FF23AD82392B522520AA01B2012CF099208DC95974C144846B3BE8C1434131567697A58212B6C0A6AB59B090CE30A0F4A746894B4288F922B36A8683DA555575B01DEE961C6F51CC83491C5F4CA6553B2B4815EFA94AAE0D98100408137903A946C360D2C0030216897A228A60C04454191DED57CA93C297C5A10A95377E3172817A8A7BA5C867405040D71ABB4B805AE37BE069CCCCF78C6F42A85286A6E6E449112D2C2619241DA3CAF8AB202E9398EFB9571BE3780944994CD30BF8CE13BEA33040CFA82F3D21A3F1963CFD4B5BC807807028EBBA944BBD104C270269A902679F6219C8B170749B5D4AC04E7851712E69B9386B60D187951C9230D64135A10374143CF6119159DF23660261B6785A9B62C25BF46E62ECD45DFF487255970F7E5A8F1D84C3BF025263F",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "24EE8A22B29702351C470CCD3463E71C4C23ADAA83FFF4B621277DE488440E28214750C76485C079BBE501C72B529A9D2751E73E7FBCAF74B5EFA87310BE1A1E",
                    "80DB3438E4BAB645C334D7098D9178979146B30A6CDB82B7F0F177D780CD2A93C92E490DB90263D13C7C9123B9DE83715E09030CB958E9D895E4E1688628235A47BB8154CC1831BC6DF150E2F4001BC935CA66955CB77F275C4B65B1875C3576CFA2BF1E23689FE0128B48588CDCBC0F77799FB909949422BC600131725D29D742C03980227081E702BBDDAB94026209909324AC3064B4333569B427F2C636E31234274AC3A3F3C7F664046B1AAFC7613EC8497E5EB59426074AF6811333543B6B39AF6C5C9569D6A520152108AA60C45C9811B0529A287ECC6B3E936111FF64150B206798EC936D556B5F82589A3493DC204404B04032A83DBA0A30E65A4139DB8C817675B1F39D37B7090BE98F8E61CEC1641A22AB76A388C56BBA4920F44B29A8A559694D86BBA1A0633E7D5475A768A96BA5A1A1AB141C2871689919FD900410F6B8D614050B9B5FEF4490A990545E23B9190C2BD0E967BED3469E9848691366D9CBC68BB00087A160BA98BEEC785DAFEA59C6C68C75D554B60ACFB1A17821A52EBCD6483891A9A15A687C82A2614C7AEE9164C5245376E82E37811283A4C81B8AAAAEE22677FA9F98C63C66A2C5723CCD1026B79532C84E5CA52EB7552E2633980A8465A82990B37C39A1AFD6C7844A655759B941A9BCC629CA18CEE2A110FC3DA367153F87872CB12B6C26A01379B89163B653664F785857D9DB2CE813AC27C4943038B7F524A4834A930B01CA0C6901CF428F6590B88B2B2766B2960057B615C11ACBE9A89849CF74F7BB993A154662BFA0FA34CF509BBE533098B493DC1C6CB061545C94B0D5F596A151B1216681EEAB7FD9384FE322360F4805594498C68A58339B686079C453171CB5C19EEAF4962E4B6B62EB6050F887CAD593AADC2D06712828E386A44534570A73DA201619E7A4D7A37B1AB5399BB72C737506F16032CC3AB6237C7532A65DE68B06037B6A0886603B6233ED7822BD175EEB7A2D79BBCA059AA9439AADBEB001B18854D2533060EC75C4487C41480468A4228F2586E346A3F77724DE59A5513363DEBA2A8D9B3527757FAD099A3B3549338A083FFE722A02FC7638C74D63F34648879E37C8DD615C23EDD5881131054121674B",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem512,
                    "D4F529B4DD5DA1F2B3B4EE8ED3959E389F857507A5FB2AF3F762C7265658D11BF43AE8B5C3DE7BA1E7FA9E25BCC44AE2E1AD27A7C45E80BDABEC9259FE88ED51",
                    "4E64CBAE347CD46994A20B99B6812F6ED39D80352E8A6524DBF35E3EB1BC7F814A8FA2001D3608850C391FE5424FE4AAAE12B6E4459BE0B4A9EDC31328DC37DCF7872474AF884181324A7BCE7527A5A8C0BF1BC2A996BF12648ADC4468A366B9FC1C97F564A46AE40669436BCAD3BA292CB052C2721CCCA58437CAD0132781DC6C2F32AA7123510A811C7D8C7EA42768D2561F0D211F4953309692087EA1C70B6451E12A3B64B8110497AEB9225E4E124A77EA5F1C789A3E023587247F588B201143857894595C6A5383C0041CA5A228F400A236441AB463069C0E769C7024320C0BF10D73DC77F0E15F7BF0AB3AE9BD599485B9A4A9F7AC6E5EF54D5292CD2C1B5AA865219941A0BA2530FBF30516D0075D2C93EF1AB951F1975E818147A3342798141C086FFD336D44E7B658E3368A1A354255CD6295CDEE171CB86CBCED468EC98B5A247CA30C313B415C3BA89C59D31609CE137E66FBA2713475140C24D9368D7D464DFF2C020197759AA92A22A093C6B7373E282E7FC78701289B55416F13E258195CA7B7B19F558040B662239475988685415B02A90B6C697CFB5CFFB4508BB49CF90148F9B13106E31889051AC85193DB051E46C7C90CA88E76980414C60A8B619A16555F92B58B98C702296464032BBB9E2A2AE19B7740E3A61BA1CC85B465ECBA5FBC20C4AE8C251C4A08D8BCC7A572A1BAF55F333CC1168633BC651923430C97D75347D7BAD4981A174624F0251431341093F44842F695A8B96545308269B34017E7C1AD889CEB596C04178412940B27299DAFA140B6C7650312A976C1579FA5AD04F482B0E88F3BF6B7EEAA24639B773D554FCF685E346C974EEB250D3A8BFC45C604E86CCE5B68B65A07BB5B3FCCE19085E21A10E0304371C719E04742D95EB80B3D606C453455BEE13A5127A6398656561C93B744042D74E88D49A74630E253471672E319246E2BBBA0CCC2CD28421DD5621C51B1BDAB0E1C9862E0E1BE6841398D65C668788B0D6B241E5640C96C750622B421CB260B70CEBA10CBA274203AF3829E0593AB8917D2676B701A032A2A350620AE4649946C5631BA466C507CEE64C991E234D629BE2203650FCB8DC9669E5D304E0170FE38E214",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "97C4789340DF47850A84889820B9B4307DF3E43A7DE3F5857340C791934F1E99D7F7FE2AA083A10A9C267D3163D0FB888C9D2D7614482095EF1817E1E9E4AC88",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "A930919E0BD10943E56B216C31D7A9F7949E9A1225FE03C58AC42C401176E9ECC70DEA33B6BC2813484E8D9B60CBB7D2D02E26FC01A74B9363485B34E5529173",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "7E09B756B87B9805E0034BF69DE4EA0555C3B435EE338266CFDB1B0D1FD503E4C08AC5F53BAD87BE2D6175370063B23BAB442F9EE170E443FFF3033F3AC56655",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "45D079CAE065BE5E094D4D1A3D36AAE4DD2FFF14DD11CC087BF3E1AA03BFB3CCF8652C28EC2F734309C80A0A97C2E341F26BB45F6AE79E72F866F953A9C82714",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "7E300B66BB2C16A9516CC44CE83929C98A1168444BDAC419E0845E8CEE2D023F0D4553633EE269A829D6941FCE0A36AE3BFABA9941EC79D2944A4924D5407FE5",
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
                    "517733D08B62D563B01D42C46DDC155D6B17C7B9C169E8420FF2B1C9395100983CD4CC3FA7811FF889B102339BB404436F102BE55276EB36CA82F958F2E777961928C2855A448737D2FB1C5669653690766377C2F037360DDC3A6CD9A4AB86CE8BC4548D27CB60C58AD935852F0938149344195B631AD163216AA02E64824481239B227FB67626581A9E75D587352C74BE47646AE83CABF9194514799C42A835E72E41B43AE6D0C57BA94A8F802660E2A89F44C74CB8BC3E703DA1672A3D43C655FB2D9908BF33983AE7EA8A39CA40FB51245EE150AB387B40D32AF3E55D5AF20A6CE8469B36438D378E41831E30733CD1CB6083EBA36A4016ED9ABC3B100C0B2851805A13FE99019B2A335811AF34103EC415C83A803E1197CC5888625C2907E5D54727BAC32BF2A7211777C758A4067933DC0CBE35809F08385B48B1C26881CC9B414DF3F26C36FC30DBF64768D7C908F5A5ED933842FABF3F1C8E80E37179E80F98B1769AD5626485A005AB02FC8B34BEA2A555D605A22A1F1C4C5F65EC000356A165A4BCB7060C312852C516842E42AFB1B69BDABC484A5002D0B70B7C3998123588F5DA434969CD01D89FD806CF678169DEB784CE09948919757D9B8E49F2428199B91A300E3F96CA21462B31278D590A6DF883933C7724AF7949519C248560BFA5D080E64C288FEB388DB2890EC45BC7334240740551585C57AB7F561724EF4616407C4EFE7A8A1BE369614C5267C6658088A58B01961B98A45CE59222795D2BD231B40B7783A7981A72064DA5320D43B92A244C474058E656ACB85AA7861459D286862C6184BAB6569DA35ABC597B1922AEA48A2C6AE273954AAB4E0B6E5049104AA035CB456518A04B3E56C29BAA7DA39B523C18B2E23BA978179D9CD82C4E7C4FC73C651A61B10BC62EFD11A83180C069398634F341EA724E669815474A8F69301A3FB03A04B25E05D3B9C2A7A0456519C51CC6EFDC61035A749CA6B8E2C79D3FCB238F16638E487BE625AC91521E395928716924D2D3A4C7341897BA1A0DD661D9DC81686CB25FA4293BD937506C444C0C1FAC04BE19CB89F966932A5176DF3910B008220DEB69CED7282B9000E11B6A1C7B0731984FB0A67FB5D9CFA8EB48BE7B1610B4B3DAA07F05519E771316F09B998A1A3B7D463D5C423B24D12C91D6177E35668C1C2179258D9C9C290E23190A140DF04BC8EB26CAE830204D973A677401857AB71863931F1517E6B80CEDBB26AA127012724DC1F98A22856E6724C6A9F36E69894FE7A54D08D6A642056BFBAB498FB252F5044148694727DA205F1824AE8C09D2585823B896AE72694E694B487B008FF99072C27E60C64EEEB9336904C8EED55EF0B4479B49654460C4A15752D6B80B13362E3BAC6C5BF773D141BCE38435DA735C5C590310716C03209A20957D34454B90E8A55737B0FF120E0764CF9E0185364524F5F68F7F741C501621C13AC5ADC254CCB60453C03C9268057C33728A3BAE2D639C39EC447BC82E1079C9D9BC756665285AD9A605F17D0E28A73516C303E082ECBC0DB6339E70B10C84A2ADE45ACDD8770BC5D88D46184EC16CB306A2592E994617BA004B9BB60446527054068926C64242A8096303738BC5C23138CCF99BEE78BD874ABA515A2936285678B60DD533A21F3CCC82B36F81B744E3E3325F915DDE071E8C813C07CAA1987033049569B35C72D0F34540D9803BA1A0AB723FC0945ABDA375BC58463E07696327C9CFD9C189916870DBB125C5A7E940789069367E468578650F520B4D1BC693D9465AC53010B9ABABC2385C41309F7C3CB6C39B02938740A4CB29D47B7639F46B99EA2DE8E512F9547EF1CC5221C219C5FA291C4669DEE32C58C06C2CF979E649700DE0C4814CC1AE39ACB6B33B76BAC4BD70AFC102649EC20DD6226FFDE06B2BE92500C8610491C523401E6C029D6F32CC8CB323163094B2780C63074E50F5AA0E2120F3FC6129D304A51310A8B55CA8DB3ABA19719165698694B900DA9E54A0705D6A35B29231E9125B3B218677C6C8A705BE451B6D11A93F04919F59839DE051CE602740BC34CB16F02B8895C66B138B19EA9F8546217B6AC2493A0541F2BD8207CEEEFB023C5373E23AA19DE38FB0A79E9B0067D3354A3632C7A32855F98B99721C5A23E49634582BAA030DCE19A08C77C714B35255A0A472E6474983310D756503F2275CF8C910C910399C335F021DB4EB93CD7C110FEABE1A5418E3C816750B017997B2ED042819E12923917D0AE789730C0447255444A171C597121AD9114630C5FCCAADDC31C652C76A8FF53D98D9A8C959486F7473A9E61378C294EA30638D72836D30659A716FDF91365FFC64CD225353A1C992F8116C869C6CE7C8B29CB145AB08CDE5755D610310D88170F421FB320E50169047C56AB9973AAD1BAE810684DA7BB886284B859BB966CBAF7F0C62E7659CB6C1740096AC061533537C1FE99879C323298C51A3D5D9221BF00C55D4441A77CF6EB646BA45562619BC6437C2C44B25AEEC7E8727BDFA903230F69BC3709D5F1809AB42688BFC55C9D08043A90E3AAC84CD120D84C83D8B013D1E5C02724B8B4FB08B98607BFF64AFFDB06D8FC676A586CBAFA7268DD11BBD2812144BCEEDC538994967D781658250B6F4322963B9A7D46B5AEA411BF3F4AA400B42BF58282D163EBE988A455BB7947A995A27C45F67425813AC75643F1C7C4BFF06289ED853B8887F9C331A7E88C3388790D090829BB6B36B129860C9A10F0610E9F3CDBDD4390AB9879A43BA50A293EEC07E78A4C0F0BCBFA2D140CBB42578081B01A82740179248AA2692C196F5AC55B86933348A39DB3226A45608425343F3C8410BC037BA55AE488831C548510BD570025A1192675A8CEABAFBF9745C480D2596C31E462757E7A4052040C81ABE1BCA03CE731D9DA1060784AE3E34C4D4489346C88154D660B0D2C903888E50EB5371A48603A1548D7891E105C56E8988ED190D950B5BF4EBA7CD9A4FE4357E2E527C9BE28EC05B2DA6681A7D116EBA563EFBB025697868CF2166DDC65BA91062FF1C1C3B9B3E8F3518F7087682B618A499622F7C4E124450A4DA3BB089B99473CC0F9A4C006C17C0D5B44E68B628E250A7FB839D71CBDD6CC87DA36FB28B686C6998A35272CC002FB76A1429FB1166A50626B658163C182685A318834185530474A2556F60C22BA055C6A2B83F9A10F6025DFAF94F8C665593591118B91934E959F51CB850B09880B33F416B6A81CF23AA079E99013B50454AF2E94F6B1B022214C307F8BDE80C55863BFB07F16368F11AB0BF848FBBFCE5F9CF3E1145E71D6D6A7CAC01CC78E03770AA0D4553633EE269A829D6941FCE0A36AE3BFABA9941EC79D2944A4924D5407FE5"
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "CA73AB6AE164B26FA824ACDDCADA823321832E11DBBD83B524D84B9F33AD32F832B452097F559FAE9BA877887A037AE6A8BA3A35F69B111D7C2BBF80124D1720",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "CC365FFA8E50DE9C82952DDDC59A62EFB5E2BB17F84BC28348A6767025694C296EC3011D0FEC9D7EEADB23251709DAC2160AFDAEDEA1BECD546EE72BCFC878A6",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "8C92561EE865C444AAA930C5AC586329A834CC6E138ED263633BC8022613525A9B40773133DFE20768CD271350DD0D5C4D6F38289A1B2603C48AFECB6D2C12D1",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "77C4BCABE807024B08467EF75449EA214D81C4F4DFC30DD8A4AEEA8643A86DCC4BFD6EEDC0F4DD25419F843524D730011ABB427B46496FE5B17DA0C52D44EB8C",
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
                    "75F74B4F338B06B4036512B9A2E60408652F8DB34D280A332ED42044A6917121687E6457D07405EDA12817D66537F3B9FD92520AD52DD564CFD57C8C26CC29E226B7642C8D4679A9F37079219660A8C4C642B783055873F60BCB6018834A1BBC86B55DF61555B1B83935406A0B0692B0958E0E8BAC5F24B6591C7F045052E12C0F268B03B27707785A8B24A7A94C058032D481E6C63725FAC96D0755E165A01E0146195465407319581C70EFBC448AB510844B5763DB35314995FD41199223759CAA79F28447D9DC1F15A2165E5C3FFEC323B9388C6C6551F5A50B7A33BA5472BEB4D5CC6B232BB89918664AAA70D6557AB7B0A14A7BA15509A5BC7CEFF073C26ACD1D649272ABC1AE796C337B08C3F021F5770555E593D7F65FD1C87F7D76B5B7C799BA95538B76378C81091E56BDC1FC03332B30D8AA1CA72410AE6827C809A12E9646881A4FA709228F2C3AAA05B988A3C2CEE603F5C44C25A5032A9C20014A9C5BC3CC99D83C42E1B79994C27E0B4E074867EBD41CB97630A0342AC962A291A40A77B82E590C791824BE28931848463AC9A0427F0810A2D7517CB97EA0992ABD954DE6878B4EC03EA6482F0C3A0CD4E584DCB79D8271197F254979EBB3B609A789F2B77BD503BEFB9EE996861F82B18BB957FF734A36E206575B6377927A9AF6A07623A341C1944CE392D0B3321959C1D6E6B4D5E93451D1122C98B92903B81C892B2C84A0098B775848A56E27CF59A49B67D706AF8B09B82B867BB155F46A635960CEEEE4432E8526D20751542CB1F4E5AF364755CB7A804B43CB39B0A1F2A60BF52ABA8A91B90E8A19710577E6B9268ED71C94B29338F894B084CBA555C1DAA2B7D082A9D7EBB0A624013D722095839BDBA38BE4A4ACD2B7886CD88C33C393085A43F7046C55151EEF203ACBA47E21E9BB65D09FE8328A234C56203A9AB3486CD66A4257B453EA8C7561895397F16C92710FDDC06297DAA3F563C87B8B623624C9C2A468BE54312535ADEB891A82869D5CDA6892A8670A3504ABE1729635CD2C73A2379CABA8A1A3CE076A3A0481E0F3456D37B1AE7BB29F715309FA92EF43390B63A113E855D9F815DD5C40B8A0AFAEE1935024B14105A2B6019137BA99751189261132D16B2BC2F767DAD847F1098C9E0940F4D5124BE27E1836BE70F258E54BB26AF060C4D6059AAA621504A45C18B9F9578F2DF3671EC78E0FC9924BE790FC72CC9DB49ACEC2246DD10F1E1A9590C52580D0C33A2A47A65B903EE7B4B0B385248899F15834F7C09CDD05AA495B5839E23BEF2770A15B02FAB83BA88283C73C70E2433C86D2B88355C4932201838106F83B3CA103709CB636F3047C75E7728E4049E0E09A2CDB315D51713F24A3033B9A5CAB679996490B946EEEBCCF6900A43786B6F00CB627375EE1A96E7B0BB1B878A357C0CBA585626323CB48255B3E5B4B6F561B842831D1A700A880CD12D1B49ECAB8A9E6CDE0F768B86112056CA1299B7A9E7C0D3FF000BBD68BACF3A20D3973701BA1479B49369944B4FC991A21A003405D98C87D37C88EF9A10231D52FBE0A012D18C3824B6DF5C82959C9C3F97169979547B1019980A4B8CF706CC9C274242C637958278900C608BB811EE81E4FD69B51973A232C174ABAA6CC51CFB82C0B90483717646E2C5A8BC4B444FCFA05989A65CE2A58D53C9D8152052678293FE68CB757A4D3CC859F283A8D320FBCE38C53495FF5635690D94287527385961F77E71D2B011E44F966718111EB4B7C05608F35B14D0E310980F705C273BF3C720C796864D88821F4940CE6C47D5FE717AEC2499BB98E163858181A3703D01DCB4A55DB238F7D36B374F03937699D1F7A6B562B573456518DE18846A7A1191BBB53A3059CB369837A4051840C21EA82EDA81E927CB0C9594468D1998E610D7DD411CA709CE5DB0D0EC199702C924D8ACAFEBC68FA0811FFD41573594A889B53215524254434E0F50E54B190147A29230A9438F3B3B4470A0993B82B742F09C47DCDFC213519A7A2581A21CB72AC65A04204AC4753576EE3C1F9507AEB34A1ED2C1CA9D275B69CB9D1F584A0E02E53C861DE3590856AC0C7907243903A0D6B5EC85037EC33250CCAA906298FDA1A3A44D67908EB40F31773F307202A36B214D69A7A47AB9EC42746B8C009950FFA1949CADA0C1E77B363E075FB13C79E784AB45804775A04C400A29CA47EC384398FF704F5737303084C5A4A804920846ED0922F0ABFA7B88C276BBCBFE99524CACF6453948BA0563963A69F9ACC4ADA2235F66AFBA3A08DB10133D00AE74B89AA929011108CE9203E0B1B60E9E8A877EC0DBE6A24CD636606DC500D05054184CBE298805B4C74F3402EFD35127157884477B9301527A65927A6474B150C0F3B9B17E864B5D890BCCFF967D2F8256BE764852054C02983810A81779BC08B47A29E6A024C842B5D0C16ADC95273C9B61F3A08C62C3BC2C3CD24411ECF510C46466A5DD8705AF80869BBB1D0D583AB29B48923AD24844D39E23F2750080C272847F1C24896628B4A2C73D6005FF8A19F817DF8C3068654CCB61C4AC57427C1BB4F51E5BD0B552928197E6F977303448BA4627C56696428B54A57CAA60C9A42A0A544EEA50C4FE21989AA16317222E435B80D9B3FB11016F92B8E988A77D6D4712B05CBEF15153DB4B687B17B21E50377D43D771249E02A4450A481D1F194484B80341CBE51B85E01F6022E244A6D984472952F8945A3DEFB59D5450651884B3A6C041B272F8D8A45AAD56D7B33574A77BB6E30250DA691E4638E68923995763EEF05CA33A9285B364797EA04A1487D15C309A78337EE321C0BB6CC73117FA50A627F877348E5CFA4107CEFD0AE762966170B160801033E119837C474E7CA06D99749940322FDE4038C0BBF65F462C2D08906367FCEC14CAD467B0567520A57521213C30FB5001650661FFAACF683B561054663814136B49DCC627E29A65CF1320857A7B3C07687ABA2749ED7B9B6488402D0A4829A45CA614D23D712591B7B457119DB2B6184F442B44830DDC537FD7376C00B96C18698C108CC58154D0A239C53C94416146660F0CFE4273D7D783F9F39076EDB9D34E0835958902C03C76976A1B6A73A20A97652D5281F925C923A553E79A3C6087183A42138EC521B209207261E7DBB5057A4B9B589456DA59708144AB8042717890D652B74BF1C38CB178E3CF3499B018E8AFA877EA31A4E9BA54825207133AF99EA2E432CC04293655059AF6BB45C33D6F7539D3A48B51362884040FEA99E8FFCBD5AE96635216B8660278F83A0115118DAE444C8769834D86B4E66138CBB79B4AF4D7B62C4DBAF2394CF414BFD6EEDC0F4DD25419F843524D730011ABB427B46496FE5B17DA0C52D44EB8C"
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "3A4C594F1F9549846196A5D2768159D31B55D2862B3E3119DAE48A8532E855D334EBB85A36CD907ECB5BDC128ABE278A7836288F989A41F3A68F1DB01315DF3D",
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
                    "79A2CEE385BADC93399056946A9B7211B0283E355091345861E9886660997A8A72B8BB231861539F543A627092FA8403CBDCBB91A07CB391909B39A4306999C2023405AB78BD24B654884DF4EA114E408A87D77F45B7AB3136C7705992A0FC288F0284E59454917B2474934344EA9FB0670159211751F46656CB93C81A68F58476723986AE1686037A3ECBF0542E369205AAA90A7CC4DE45C79106D05B75816A602BC56A0994E3935473C0BBB0B2B7117ACAC039E2947D47848C4AA81D39AA5840239C8042A8280C80873ABDA6802EC755C114B6C828040B0277B52AD7AB5E6C849D2C9063B27A26E45C3A5C08A751472718A6E16C91E03C907D9499C61BB55FA7409BA2084603065A1248BD814DAD30AAED3702D7B3A75A3276DEF14F93E859D1320F3F861CDA089D166BA69D3C79FFA9CA73742B498C072F3B79148CB66B9A2018916BBB2014977BAC0782AAD4508D53228872657BA4D2AE7D08C22B524A3513C7D5881A58A98256F629C286750FE81F8C476AD59AAF8C43415B0C0C6EDC6DF3A909E2A53B822C033D89AF83B48A2AEBAB5F4C72072C169FD824927831F3922E0300309224470DC62C5153537420C2C86A0F0AD505C2CBA47180AA7CB7A47D354155063D5A346AB5224BE032B3E7E6BBC4C8A7E1917410B53850AB1949A71CC994590A92229E82641E740242C48A9525191AA2777BCA0ED286B70D59CF629B24A5015719C1799D8953ECEB70A5FCA962CB2C25229DB2909EFCDB2B7E3CA0A9848872F4AF1E447B5FF60AB7BA8255A30C7A24B28D2C9CC4F6A59BC42B920462186C642C710E9EC4BFE289C45FB1759B06109AB13A354831AE28A5ED7AC592714337FB688C5816E17400AB65CE83C868BCB49151045365566418919D822B98D4FC96B9D97FADDB313008AB12A40F2F7A59547B596F7156FE0205242971E9BA8898D96A6EFC62F236396083592B04C3FDA990B05597BAE09742850484755338D40DDC0C5159E883CDE927329174F3F2B6B5E0A3EF83A3C722303A371A62AB2D25784A7CC52231885AEF16199406BA4024AA45E9A74018773E34C597D5CD0B172D6E90BAFEDAC91B85330D3075F3B79342626B1B13450982534AC135BCA656A0C520BDFA2DADA82E87A142C7FC4825B04CD69C57080248C09639138136E80185B798B0A464CE4E6621E7747A3EA1B444E1746EA87CA95790C7BAA876A2B937DB3963E28143862D6C58085BA86FA1736127226F9B3201CD7840AA4B8142665630E64234A0A6E883707094C23AD09D7901AF4B42A01FC4B1A730575690BC6E390E688C3342A258D09C2818D7AD4D0CA7B3242078E2CAB8AA4E90D919E9A8496D7240D4FB2BDCB4B545F17C1BE1B9177A1C451A024A730ED03489EEBB73845B3D3AA473DA1B3E02D5779C97221D50870F914C2D2B86BF2A946216C144249E0FC0CB408B942032447867A56591196BF84DEFD7A1FD43C816C2843F01A6E66BBAC9A80505C8581C0A3AD807CFBA173635135CA76666F49ACCBCE15A2312BC1F038BA90B8491DB27A3A8B25454C15FC2A6AF8C8993440909C14D02A995E917CC7449AD726705EF866372C3B503249F8E0A3E252541DCD4635258B55161519EE138AF44C0E3781E62BCBF542ACACE64C2F966B0F7D7016FB4347B79C51AD78147F612AB709026B3B08181628F781A27C1B040C5862131012895CCAED976D8BBA57584441325BE014768AD5104A4130B594A1FF7F9701F60B255251F754BB7EB66CE0D9C4E0E5BB9FC6C1B881865212578E13C423FD7056399A28FD3557ADA9D9EE5BAEE53A9291106A6E00468600E50CB2FDE146268B5713DEA993B985965F713828C072C324F6A97267B39241C0245572C51D9E8AEE13B1BC3542B06707C0674B369833B854A820987B32E679F4BBB47789734D9D3CAC6B976919CA17512CC41449FACC4205AE1635902B9C6D35C1740C052B6592735AB8F09A227BB3F67640E13E39A877490A0B4535F01A551FC0E4A909B40C5ADC4C948B3444430765842F51212FBB086B70F56463EDC250360998032E41A7B6570A526615C438B7E4CCB9BEA153E486BCE50A347B1759BC9654B74C92A10A06DC69518F995A796659AF9945BEACF62A9860392B224815202389B772A07C74823269C1350970203F5439AC9CC19D016BA5C29B9E59E4BD2C63285B4F8E8BC584B3C014B2A45D819A2157017E50D33142F1D319BE7467B5E69B7D4595DD66488E2BC7C76C2AC7C4097F4826A621A6149F103D8B05C6A585BEFD07940528F91249774F870992B793A2B804F417FFA6BA4035AC2E4A751D4AA68DC19B0A701A2597244645A7ED7556C9D7373E1B916C6515F3A8C83E700112133CB1648836B17A69F335CC8599129707CF77232655ABE78413448B1C260097D2E37540AA19D744537010CC580C8CF859818EC5294F769250817366670C7D4F9C64475960D1CB5DAC066E7C054073490D95737250B1F6E405936BB08C07CAB94828DB6BA6182C5AC85FAC9FCF5CB6D19CD3C590136AB7055C1313C63BB24414106C0BB56C31D8CA4BE6A2096538187443B0FF085502AEBB45DBC371AB0A31E5B51DDA47FF1239AA5B88EB27912C2E18DD28A8F0EF281E5A97A8B954123B95EBFC799E5609CCE50C7B1E6475346CB17E81112F60336E589563918DCCA32C7E087F6F688EF8934825C0CB5BA8149C2A0601671DF42152237089B528C1546036E5CB2A0659723CB5601529A225B61F77729E9E664BD4A5AF99873F000827618C996AC6048835F65512817D075CB6529601563760CB4B39500A8E5CB01D8BD1CB251409A2AAEEC5243F1664275423E3C6D30B18F79599DD8E8868726306D14929303AF5BF87A6566152C24982866C9EBA549B4A094A3C7983CC491EC12823C7B4F7706C8F1A12FCD548F33753E76683704889834FC2835423A2F1C9A3BFC5D9CFAB507484A5E608EC3AC646D36AA2F5119CD501B841ABDDC1B7F646BABE2BB11070960BEF82BD8282087568CB6DB42735359CC40650B363C23714F9F06C8D7607DB7D7750649ACCBA853AAD93DB8B9460203A14814BEF49A8B6829A7B831B59BAC9F15112104DAA07E03ABA02531E7306516A87802CB664A462D5A7AA0DBA86EEDBBC42AF302D11275CD4454D49C5A82E318CC54CA481B303CC94BE98CA712903013D4B89CC869DBE68BCF8BBF7C3085E1E22AB941A8FC90424C035495264BF9C63A588A4CDC3C8EBAC9066D0C720739960B325CD1C99C74F399511494A2DF45AE38B715CFAB3293CDFAABD0AF1278174EF92BEDE18AE32D113B72A6DC72A2F0F70F1885231F9925E4E4318D4E6190DEA0DDAB595E929C2F045C34EBB85A36CD907ECB5BDC128ABE278A7836288F989A41F3A68F1DB01315DF3D"
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "AC27E145C12AED5CB925ACEDDAFFE0E4585EEC4BCDDF1F13FA14336DB158D3489551885BE8D2459405071416AF0B31B09871D8E695AD723482C91706BFF363CE",
                    "9DC6C6E0C029A2C9B5BD92339A38470D81C86C006FC5C44D37A4B6A57869BF007AD4BB926FA299DD529CC7879B21D75552D714FDEC6BFB7A6C0C0AC90F93AE7623CC8AC18345823786FBC9BB630AB3B0A27E196F33D07CA253035C352C90F03A6710AF339CB5562C89EF4038AD43384B2052D4559812C08867991EEF8B507B2939698402B83B0CA0BB1DE9BA0EC7C26699E82CF17768A67C595FF6C54CA6AA135322342C39ADA321753CC35EC6433AECB904D8B1D06890E8C2BFF6500D4C217B82737C04521E3F41BD8B168730788B1A215A33A50BAEAA664C7C9E9B3C7A9231A6BE3196129A0512A89151731FF7333B7A3B29BAEB35D6C09A57B65C3024A32A4589FD090D2E45B7A09457AEF116D82B6E839AB8763402909358665A9333C95673014E55B40AF06A57F7E20CC8DC8661E7A3ED1971890A95535940E9F02D1CEB3F53224F4CB696A937C701077F2D6A6F958364568494BF183A28630FFCA9277B693A093193270511AB76ACDC1A4BFD4747CE1A791FC29951745DD67233AAD5796FB3B6DD6BAA0379292F9B9CBCE0847B898B55335C16A433CCEB5BE2D209EB43879A1BAB3A5A9BA77A4499514176E91798EC19E32468F6F4C36A239762E5AAAA72CD6BB49A7D3767DCE3659BEA142AE04A6D1C14E6D29EE6DC4FFDEB0EABC69712048D581719F65C60FAF47FC10BA10574C0A2296A14E633EC962D00E8CCF9D20F703372C2325367029A588C1928045573C3B64B8405A3969F88E3CA0EF1AE478818279145B8B38CFEC805A7801CDB319039F6CCA94B84555454384571CDD7CB747300A07C5DD9C102B81B051B6B2032E63D8A8137DD9526FE996FC679050E3B9C99E1888259CC65B1649678272D571036625D7AB79CD69A135ED839261673B1DC1FCA93B393011077EA718946766FC1329B203BB2D102928514D8F0142CF56B03B98F6CD8ABB901968E8351BD646AA8566184B0B0F7E8970774B571C67C320C0DBBE5B7153278CC474D457155ACE19922449290DCA02DA1A54820730CB28CA37456E70ACD5B00A45467455A01A545C56A231B1D8499400918A061CAC3248BC0CEBA03715C154056339CE25B0BD989C7B1BCE6C85528837AF40C9B87A39F772894D4AAC512BA600ED76F0AF8627BB3AE3CC33365502BA54B1831159416390BF78B9C2990CEC8A19C3D1C76E79881BC3CB07B6350EE813F4747C1FFB1408ACAA711A02152611F4286ABF3A54ECA5021231330E2F1CB44179BEE442C2D612E08D60A7E51B97A910EA2F04168744581A7251D9692376C6C1DA71F769293C97B5219FB3B2F53390DF85039993496A55A2ED89D5192BBC3411254BA0428A88097EC51F9EABDCBAA5D500712A2C5C5B57A3F8D9106C03863961B1D84B685FF270C87D5216F0112741817C488B8E59236FBBC9D2D2370AA60B502388F02574F05941F2F34BB1FF4499C06B10D65C6E880876FABA8636C107D21A4ED7231B55762C314BF6D41914671586FC8B8C884CF5FC0C5DB6B1E2259A93F222FB96635A5E4517AE7CF81FC1B262B9EB827749B51381CBB7591E3A075B0535455111149B1125C4303C1659FA4BD32128B00B12168F1AA0AA04FCD1791362CC89037064035A2C1C0BECE5DB437A7E8F4D54784F5B13D31897BC1D153B094252EC51B6E2629CA446A6A",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "B5EB681B4241CDAA69F9839D79E49E8A22B1753FD8D4844012406C6E2E0A0420C3F615025B1D3A02DD054A998933D55B8745ECED046F7FCBEDC4397C44ABB0B4",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "0B62B2B97733B9EFA5E6F47BF85FEB902288D5434ABA6472D879A26BA08849C68BE4CF93D24A99FB17414EB9C36B8292553A1624BF576BDAF68503112ED031F4",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "4B73899DB7091D0D061E532FE2703D7A31494E08E223AF7047A96F186EBA1D5DFFFFFE3ADCE3407E939469D2E9BBAF82F74113B170B04D49AD6680F788B1A26F",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "5070646F41C499AB0A1BC5A22AF22DA39EB3EAE28CD4CFE2845101456A3F31332953C78CBE76B378027E5286D85887CBFCD48DE645407417FD22E5F53C9C295D",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "7A9C8AA10EE09CD43C11A315E12506B279720B000F021B771F490A2EB012CD3A54F16968911506FD7719A3651AE8603A45542038C26F36E3041CB614DD13EEE0",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "DB0686C838AEF0DC36363AAB0579ED25CA9BC6014CCAF5EE0E6B0783E4C9877EEC7D3B2991DBD5BC973E0F4362DF9B94F2760AA14AA2C8173017A30D6A871B45",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "36E643CEDC0D0280DD44E16A1353494861D1358DE8012B88FACBA1E6DADF01630510E6BDED49D6D031CC930AB638EEDBCD2422BC35649D1CA33017B3DD344D71",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "268D0BAB19A0A48FF67B114997413679B98D65D0CEF327763950FED9CC6B845F7E140F3CACBE34156D8729D55F1AD8008D20F17DB6F5D436CAB76A4EB5111620",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "514AE29EC3C21981216B74BCF91D5CA87D81F1A541162B515CE193B77A64B03359F631BA2FDAE1A68A77AE066C4186960506823D2D1B8383041178A568A3F8D0",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "631D57398086D55F1D9E2173F46E16B78F0390D24D586C58F08CF2A74F69C30941DD2D8B1CC8C5A1F7E9155F19EF800D5283B852A83BF590A874014628CD4832",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "F13DEA91F234BC45FABEB82F6D5CD6C7A1F8C9102A0E9AEA90F16E98283B3D4CC4D62EC083507AC0523EBF0B9BDE3B1BFEA833C5FD66451FFD6EE26DF40E83BA",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "71E0FD80B462AFFFBE1EC0C5993063D586FC5B733522145776E51CDED9A0C44B3C4836CF160D936179B75F4173FA6921F052452DEF3418A8B0138D47C429B71E",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "6034BF02515427CC2A4AA74ECAC46B032DE00890BF257FFB4B88233DD9BDAC21062A06DEFE9602F9E304536E306430D3516604704ABD496712BD99544D0AD07D",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem768,
                    "FD3A62757B683A4755380E42E915CCF2D31A74979BC143A453A8F4570F1BDA062CFE75E5E15327C71B5313ECDF48399CCB106102D37431722043C8A32F569B68",
                    "4CA22FC2745AB5A1B51D36A99E6C0AA6B9851A49B2BB6688A6C107A4E02A2AA612DE156BCFCC4C5D2C3862C5295E2B0280104FE7D8BCA7AB3AC8210AA3D64CED435F9083AFA142A7C072007E8454DB2339474625EB03AE8F9A2F4F5215FA8534146CC512503C3F33A744898C68DB0882E05E08656C897B313DD713C81C26D38B300F4628C7C7AF551280C5BA65F7B310CDC279622117AFE16C66686C4B825BA1E331E1B25FCF878F07F9B809607B1C918E18308232EC086F7581D03921D806781C5819D198931CAC74202B66022ABFD1641FC6B9827634A0093A96C6B4B9EBB13D7F13B5C0354588163B05EAB9FCD17DEEB9C06FD176603BC1FA34B34EDA908361169A95C48125CE131B1FB6F6C53CDB2BB1568571A3B25875C21AC274B78C795D95C01D99A0F6C740415519FE9871F0A781FAB79540F05A95800512C644359566CF029C41B5C5DC082605A2096F403C47E344C0C99635C882CE116B70DC111D1688E6D43333D3111EB96CF1E103146B7E7DBB3BC83B53EE976192A872AEE21E5D5A7D9EB4AC644C177898414917244BB80F88359CD5FA26F45C7FF8D2BA1DA114D53C4057C821DEF0AA81099B4580B6C3EB839A9820DFCA6E94800E9DF565C42A30CBE5B10D24311CC736790212F8DC0E3C5348AD402BFA0A7106292C16467C97CB54002BB3F5EC889799C53D2CBC97074DF3AC8163BA940DB07698E5ACBB7A4C74D53D2A046DF3895C5312729DB939AFC93730142DAD533E34E59211866E8F5A8C16133EA8FBAB89355FA5148F62401C03217B8FE37984DB7347F6B12344BCD95A3163EC9FB1E6B15AA4312D6B6F67E0417CC64C60CA7D9906541AF81FE4F66DBA4123C34B29C53616A1C80D074A086D585E119739A61823EECB3F5572A3D0269134CC682F535F7D1349364109E63345623521D01C97CDF53D8ACB0AECD659D8671E31669586E8486E1B23284C61A3C1ADEC8235CDDBCEA9EA5786592B99DB8A8E104F47D60EC4489F26D744547A8EEA30A40C52C820C91120C07BC3C6112929884BE2BC8BF127B05690E5D0583FB7644C594A0F9511D678547089801F43B553E32C1804448107B9B9A8227776531EE587D41B31E9C04B1DD3B47DD259288299A6A42771D8707D493655559B406A4EA302367E249E2067B5A52C0B4D50426BFA8840C3653D34A74617B6155CAE1784A45E356351A72535242803F984F36A876E227DA3EC7B165095C1F954907B71BD5238464385B9053C2DE953FDBA4B924AB54A452F2530024A2AC7198CAD628B7EB8B18FAED975C59B29A7C982DE419814CB4C69B8009D6894CDC995EFC86A8B190BD591B28CE469875B0A040AA77EB391D9D64307F189F9630D37914CDEEA9958AA4B6A0B3C2E0162E091BEF89766D27283EFB592C06C1AE4E02136A4200F4A02F7661854A753B9FC0A03328C84B953C6D826E28ABFEEE8C77DCA98258A6ED8912CC1EA94645BA753EB5A44D043CEF621F4344E7302CEFF09314079B7624502BFA4C6CA6416B1808E53081549478211988FDEC39388D49D2855413D91AE581C47116C304A1959DCCBB3B8BA1A873CAD94195E653B9B85E628B743AEC592833A44A1B940483164CB50FBCCA3C4029929249D5B3054255C7BB7BE5DE283E42A2EB1F8F12B6961ACB5BFA4178F3EB3238814",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "EBD6260BBFA0C542CD1C857C318BA68153832A60F81DE839E71284CCC919BB71600FDD18126AC926D3A745172186BEB557CE6F0D7911A5C9EE387689C799E912",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "E8AFD845036AB72B2DA4A7F200BD75B0C360AC607400526B824074F1D836F44F00B70B6BD2EF4B24E7CEE9BE756538C211259BBDF2C0FEF31BAB344E8E9FFA17",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "BB45E6A05C79FAB39099B8B61917F230D245389D0F086F55635E97099F2D18A3923D3D0939AC084FA80DB64EF6222CEB788759E86C0AE3F84E204C65C8A2C047",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "0A45C90756550DC7EAD5D608CDBE54952568217C6AA444D3AB1EEC9F7A208255EF3ED2282C1F2A60FE72FCBC2A51E885A00AAC76E4EECDDC6DF52780B60B3AE6",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "AEFC8873C0813CBDFACA93A56A0B5F0A0D2E440D432D48BD10F99A87BC50D6B9A10ADABA73D62FD5234D8F33A0371EA2697D66A3D5BDD789B6FB4AF056600A8A",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "11DC979904555A3E4B9BB27A9A7C7033D00B2BC3259281F91D679D5725C10C355D70EA10939FC39F99181D05B1F6126CCF93BADCABCA69FBED2B38C7F5F35768",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "A16470ACE83FC78AE8D78D8E8E6E24562F924A7C2AAD99CE1CD2505A2AAB7626A7AC74D64D41FE1A10E48A449986CE3D8562736B19FE15153ABDEEE659D76F72",
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
                    "B5323D19064D5E2492EB09A909A5215724444E1584712C10EE6611477A43DF6B1A6F31929166C250D960CE452B0486816DC629E1029A3FEB625721533C842638B98AFE89CA4EBC45CA73714AEB84FD7B4DACC82DCC76734B3C645E8AAD5A9A714CA934592C80030A77E939748B838B073A530A09C5A654A5C4F4A2DB307F70A96FC3EB915F01034E60B4F7650286FC06E69AB2BCA2AE9BDA699CA27A10381661C22D639A8C18494FA0D40388660DFBDA1CB44486EF338B35313709D3CED0F2BBCE0A28FAA26AFF43C21B38CF5D0947E781B27E12825AA594C1C69DAB34C563FBCA006A623A85A6A2296D0AB9CC36AA0899408919D93118B22EEB59BDFA4524CA13862B61C01AF10E89F29DD15679C86261AEA6920466B1E09012F30368CF761694EBCED58640D4E96A1FC2334B300FEBA1557A3C7B569B1CC3A30FE53633B4F95AD7D0CF78FA15EE06C03C586AE6F93545269FA58A7B42624012326440D860D695BA098845E018190BD1B43344B2D54871AB50C3AF0CBA93160DD13BC3F63C3B0B37131FA5C25AF130C17988DE274C3802001FA2B7323A73C2D01A1D253653C84F9F31C277F5CBA3537DFD2CB9ED430AC0463D7C1AC814FA45C25792F74797CE0463A73CBFEC2534F43224CA00B143EB0789516427DC4084BCA50503852DEC69C4E04E67896AB6149876E6692678991A6523CB5A026566682D0C11A3DC0E142601DC1182F15B0770A064DCB473B85B2D5E77097DC4AEB7328A5FC0A7050B7B51CB4F607567ACA0153EC694CE7219B818518B26AE401982F0E1B526990A7BD1553F4AABC7488026DB5B2E02431ABC6DF4AA7A3FF01CA3C85B30C2462E25832700C45F09563F969FE7E3CADA285C3D7B27080A591CFC142D37C24FB89C16836871B918F6D03427072D46F9604F0425C3220524AA7717C12AE57761A87B980A2CB378AC9026E15C90842868B239857308F775C6D4E90CFEFA96882B617878B54B831278A111C1480A7BD74ECA3B7695323D780236799BB0D367A2F380BF21D0CC9474615DE1737F91968640C5AD2A8101851255D00E1336227392CA0F69B4FD1617ED1B0B59D17CF8E7572AE239BF630658A60FFC54BA42F0C0C3A05F756444017815CFD2BEBFE6860C7CB4023091C577A834A1310F3B069ACCA76B8AB326011900C526B5FC6DF2212D52BCBE03F73133297C4DD26E0C0826CEEA76F376538A3530BB3830329002004981C6812592A3693CD28B1EF630815AC23395864E490E57E876895BB6689A1D3D1BAC49D9381FE71BEE181E478907A694B4DB35586D22A2BD51A90DD3B7F2F29F0D51305499167A217395FC3FD54208FF62C33804B17F5C6DD3655DD4724BAD2C315CBC0121D84E5B28B07F1A2FA9FA9640EA103A8072D2D6613144A72E6B371974A272060C3A19A2F412485CA94623FC1E7B4437EEC496F1D0C8DBF5BAF56AA16139B3D83AAE7FEB4A707B8D3727CEFB999EBFA353DAFC788AB614D7B56F86AB788D1924B5896AFA7431A315B4A0584DC5577CEB62629F665CD4C7283C571F91D58611680A302838D7017095C0A7E81627D631C911FB38760B1C754596D0A9AA9F861C22CA684AC06D1B950ED4B03FC0A37C87F22D20EB00D6245F71595C6BE817B4248DB5D9A4877BA2F7FA1642B45087114CF4A597A68CB006E11869B222F61A8066901BF56B42A157279037A81CD39BD5DAC4BA87632A450043022102299A92779FA2204A28B0538C62183F0A5D969127D84737AC0A6EA40491E7F3693A86CEABF7C586A92EA9265358D2650A1C0530D7B43C688C967AB3B7EBACAEFCBC1AE07112542F7FE13875156724E7BB766A44A6968D1D8ABDC1376BD597CFFFEA7384B9022D470051605EC75C496A146E3C301CA6EC05EC4A2810A04098AC564A3109C463C4D57BA8FE559C4019490F692210DB121DE43929D46B0DD978960AA668D04D8DB5741760859C7B7256987138BC2AB2FAA58EA5025BA515FBE47EE70807BA308D0E2AB777E30DAB31797548CB27D3CD93512CCABCC210D88FF4C90F3C7C38E643314E7280BEE861D640B8C82C5C633A083BBA6BB41856638C2517E3621E321949B4AE32F017171B6D6B7182A01CBB7C06A061F82EFC602CC1D0826AC78B6F052C552AACBF4BBAD171AE3BCBAEB339CDCC4A9A3013206EBA14D47657033C06BC3560176966CBC01FF415CB6A689D4B0430B4B541A9B70C0908B16E75660BA37DD1880F864227926A7224B43A7867A5ADEB51D7931BF1F8946F51277375C43BD63C4C081851E65ACA74A4F6445EE8158338560BFA3335CE54363BC80B5AF87C6097CFF02C646B9C96693438CF1255163C4114398A239C7DAAFC59E48534DA3C615D2BC26ADB733979551AA6951FE2B1DB20176779097C006F98218BDA686E13514ED438086822AE53C471CE429AF5469D75215E1529314905B064E204C9C177AB034FAC46566A98B1C3D95749108CDADB45976103E557B12EA29034AA6C9BF93E9F037EB2EB3B496C40765B6FF9DC58E1425B6086AE8246A94F844A9AE471F30A52F50444323C020F6A74E0787BAB1AC29C2B7F21D335CA286BDA349CD91258673A35D3AB62A3AA0E9DC083E7C76478BC956BEAB25E959B715C539FC02D8AD091E3F43D907B484ABC628679227B078F62987C19CAC108A98EE566751F6A4AA679B0C0273742D910C2F65DD476831897CBF88B1F1B79207EE3401C42CEFDF097461695CE3B3BA903CABEDC387E19784568BD5F234910B3A864F40D7CBA425A344900E4781DBA9A0838487923B2F751AA878256FFD9186ACB226F49054B4732D301AD143A9CF5561C3ED0A834738B92AB974E89CC17461DFDC4BAF6A11E65145E9294CC55C1249A46711A852E4EF140A9564574B4B8D1483E3203A6CE8545BFD6BE317252FF3CC344EA9E5C936AD861B293A3A7ADD6CD999134A07C1430CCBD24A5389552AB9D393C39062B8EB7AD7B18A65AF13FDE426C06E1C01EF5A0F9264477697C601280469CB437C47A44706460F3C7D9F05CB7160F6BAC8937F32BC19A610C883C602765F1041977FB904648951C267287B8AD655A784F76A8AB1C49A0C92CB6E75D162353896AB2742C17142B4692380DC6B00A679B21044378A13436DC342E3D88ACDDB7450CFA6BF6E38EA4DA66EE321500AD2FC3D85CE319A07DC3AAFB2563507580D287B03E704D7A842648A55775E020F28107F3059CB62AB6658A2F51BA2476B68DFC7C8AD5F7C767A1C8470245CFD8777658830A4A4D4C26C66552ACC5B3BD92E83C736AC5524570355452FC6C2196C0C900C47926DA33BBCC8EA14978C6B4021129B6C5825FE3893BB3F2951CA60E81EA244DE9832AC800BE427B20BBCB84598B24978ECAF951561836EC172740BC881CD0972CBCCBEAC84A0592AD469A5673A5A5919A406C2591E82315779759D3A67C9B7C005AD9678FD58558566B1CF5406CE1630690CF5FB79FBFAC56A9DA94A73B64A32AC9E4805F0C402107E68768B7034D488B10B587E396C86227BBA5882CDB8445845603BA87CAC253751EF7043A1530CAE89A73603299024124C389F92A894D2A2D7F01267BE754F34015325760A6A18C4184639CE11ECA4086B1EA6330C1C65C57A0E284069C82B54B151D001213899A2BD939B7A3E409E0E24C01B097DE470DF8D0A9C89519FE3180FA3A83119943034209103854794BCB4CD565F08C06A2881225967AAC8148DD61171C6159F9AA6B539C41DAE5C38F255FB76093BB6B4BAFC690F3DCB41C623888D534BAA3B91B4B347D7350818C2C4209825EFB3DB269179905A3D26C944530070167A2A94518C6B8A8FAD095271279F438C6C7A4BEAAA42943306D70942F0CC97F9B010D95E29FDE851B5C0C0507785629CCA9B5E994E05691600A9B293529AA33159AA7C57BB38AB1DC8E3B5185415B9534AB544B3450037295CA077AAED2B77977B7CFF1C2AD5B6ED2249D073C611ADC2275F7814053764762AF6B11CA02B034F0266E1AA04B403B2E15597EEF804587381B6DC19C4CAABD6B2A62616906607A0CCC8C9004A443FEECB9E8E218E84B1C25F97B5886C6F2EB112519335354ACC403961B59309118AA27C16B81AAA7165CC3B20A8E502652372A7D39482F7664825AD6ADA8DA60B9780EEC345925B29E356714C05BCB335A0E4F4A5DAB4526344A7905E7CD5E23919F910A81785E532C93A32932F95C6DBFB7C7DA514BD0146040023BC0F652D5487E57266937BA83BF36A34DE43576F62FB251960F2AAE6E8C7F94BCB45B99C2436731764C4201856E4426BEF275A270D0AAA6E98DC429838390CB6EE19A692567CF2C8770E50411D1A765EA671E90BE7CE784978D8402FBE9156B23E5CB488173EDC0B1A2761F7E7EF75AE5C50B0512399CB6735EADAB41261B37BD495B6C1D594DEF2A40DD2E3F428AA7BBC5A7AC74D64D41FE1A10E48A449986CE3D8562736B19FE15153ABDEEE659D76F72"
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "1E232A9F6E1BEC736BCD8269EBB1F91194B10AF29FAA456A8894D4E333600223E527422A8164BDA28DD896337F05F2A44EFF846DED7F92614EAC032BBE794C02",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "EBEF97C3C1C0F61E2CDFA21FAAC0F026B498DFB17E46027D48F76128F2777E4B359C39184AFE550C25FCA0CE5A68C90090160962EAE2272408E1C116CC1B1BCC",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "4E98A2022F166D848D6BF1851ECFB9117E7F1B275050FBA71215D35ACD643B2352B1D970496D5AE84B730C310CAF32778BE41719579F222808EC6F8AFEB8F79A",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "A29DFBB1BD2A782C99302200EBCD056A463DE1939CD3D1240584B736A34DA8257E3D54DDC2CD09DE682ABA1664799BB5598942E5C1A89AF67BBD7FC609E05654",
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
                    "DFC76145C694C069B088192383A811B848A897CAAEADD9950E731AF85219C151AFDD386317A03C78805F2AE9992B9243A1C674BDE2B0CD1CBF256B60D7C5C3726758974325710C3D0ACA7FA0E05A4F612A2E1A3399D6331B09A0D1E709300A25D3644724041B0568B08D356A268C6413D431F175A002857476D85BB8B00ECDE824A277A39729A60F87B4D4D518C6992D2B51B22DC4A1D4295B18D43F4A4A50FC886693F00F34A8BD03D4A1841A207BD65221252A16C5C713DC01378A1F8B4272D071844CD9B2DD0A98D5FA628FF1CF42FC4FE5E327801360094218609C3C83263D3259700A91C8E646A54DC63DDEC913C7E653B6C4ABC0D565EF584BE1021D7B2C4CBA66115DC46644FB5C4139C528D763D9DC3D90BB84131983D938C946D83B9BE61BC2EB14FB106A0F490BC8F6C9DAD4A328532CD8CC22626BADA8D94D690B6682FB4D31DB81DF5B2DC50C49CAEB9E43CCAE963918C0940525C5878F31B1C87B1FAD4B4336FAB326120AAC5835DAC99A76E8AAFE98B9C5E9BAA1AA5FA9E7AFF86640F46803447885A1862DAEF74957D7B3C2E044CEF3210A9B8B036586E0B4052C946636A26E9E0A064C7666B364654BA1BD962459CCC91E38D93CCB2BABF2C22972C2536B319A7F25017DF0208ADC518DCA07009DAD91A146C4793CB7A64600C4574FE919CBA3C2AB4CA33C9770D4E027E0CA8D7EE68DBAB98F1597C345432417FA19CB53282B7799CA6B56BC411BBFCA04621543FD300CEAC04FF8D2604F281FF516CF35D07F605B2A623B4EF8940D963670F6B54799A3091A907FA3C927498128505019496AA14A4C9BC9F804DA85C4D2295D4440BDBF9083B206CFE535832CC04150BC5C2F552DC8070437E3C33E103D5D6417A9329A83835A2E91C927278AB0CA5AFB1C440697B8C756192CE3B61FD2A11E3621408A99A82CC58C5A4F95C70F8AF50FD3109E56317A5E398719E8CD8CB7C9A1782236157934ECA45D90555C5A4228D0859B17BDB7812551A407C72C381EE12C8D300549978EB9915FC1B7CC881C3C06276F524078827144BA79B0E5F8AC26E774FB647A25209D319AC1448040D28B0E2A8631489283F47521BB564D057202D230766329C050E87D743376F9DA439E260044F248CCF6049FE65349FC5C56101EF5E97091D12618FBBDE46897C1964593D42FC301BA2C0220A4575040D54645C9272D76B7308ABA9AE348968162BECAA4EF937F5FDB74C6F84ACC287CD8259D8B3C7E344B2F90C2A5D73274DAF32ACD570522B3CF6FD3589D0013AC3A9EB69AA6A2E726EB758E2E505EAE51613BD56E3A06C44699BE1626456F7B0DEF1B7A62D6BFC1851E1B750BC224AF339C594849673443B1E8282BCF9BB6AD830C3F292219BAB11A3B20DFD0C10E4309C41CC4B692C2FB556C2D421926C98A84368561E01097C6C3FE0430F04B656EE7A2F233C3CF90238D32BE5F4A198F20A4892853E344B9E6C59819E798FF9634B3F236BD4C062EFB6F70419067A08FE63C33B1584932223A3790C76E6CBD64493275567B93DC9A09693BDA3A58269959B7A14ADCE535B6199B56D617C5375459DB602DE8B89BC59D5E1835724A4A421431CD11465E866E6B3660FDC386060614951C09EED1630C6752E40C963DFA1D94A1525F779DB942AEC4DA44A457275787931B0AB1B750915D74597902519C91258B26C0E8C1BEF5F281B06871E945C1836ACA7920216294414CE07784AB792FF94B3A0811906055AF860F74B4A6BD2A72C135C91F79A11ED138867ACDE172B86F41965CC9CB22A63CB9F60A846A58C1087D0F3AC1473A9FDC44A71AA194869B1017147E2EE7B44AA2241B848C0EB95C80569540983187A19ABBE346C92A659CC2429945299A018B269C87B3D290EA823D8C1BA06F8A8FCAC3829058140B92C482D9801C189C89E1111B351C1F499F66B889DE8403D0BC1DD5012C9FBAA4FAF06BF6270F6171CE3C627A57BB6379CC8559983D259C971CE97FE78A6CA418B97CABA98BF24D7CB542ABD5AD337588D7C6C8F733B86A689EFB212DDB2BCABA764D451427EA0C4E13680985E12ADCF7BBC517A0D67A99C3E62156B3840AC79CD091ADD3174472A3A1E286175A422A1E359CBEAAAAE56603A11338D1364B93D665CDABAA09075C9EB73D99436CF678CD33767E4D16AEF4E20DEA770B07C36F82C77D23E16659F85EABE65603E53CBE3BB622B64FA4FB9902C6681537ACD151BBD0B69EFDF38C911722269550633620B3881FC1B673BFE7C0D958A24045A2E35596FDC319292988E88471152C7D39D58156B388EAC55DCF98BA10D9428135BBFF04C59F096212C17DB2CA27F607B37E41CC679B47EF648602B36076470C28828CFD560FC0D1788FC57860B3449D8A03DAF76581802D1F7769D89954C9A817807A2E496461E9CA2BF0F96E05772CAA3CBAA808AE73E3A105EA69A97CBDA06C35252817F7067F79848E9C3ACC84943070B30CFBF4BA54787B1BDA450BB05CD20571B295331AB1C17C8A99EA178A730716DBC49AD1A17CB5C7A965C30AB4102E1356A2E29A98E729598DB8907914AF86CC91A567A718E94502C82F75867C73A73EEEF99332F47283753FED944ABEF36006FB1B802495F89741C073270775B0DD612BAAD8C9D526CE36F90A2ED035B731A00A17B087B690A90019E1011456D00B04707750902089A11C6F29422FB52144B55E97F98ED765924B87718269BEE41C276E1C4655C18C51AC5C3DCCACF253B22AD75139677255AA1D182C65058562D0E684494A7D2DC6311DBAB6C931C98E1376E91A4221E129C2F12A20C491D8E17392F9951F526CEE94B416112B7C7B0B37F03EC58BB70609BCADD3C8366323BD48C57B491479CAB0CE2979CB8414A8D74E978968EDF53D13BCA653F42D31889FAEDBC5AEDB5B4F9C7C42425128FB1C91C49DD2AC7A64951FF38AC85986AFD77362FD778C15B58467D56DE473B79B8C1CDC5C2D63C763733B59E16C4244A4B44D281E544A005659C00EE17401AA57FF14B38A7476E455978F1015F3385BCC4956EE93748A0712074587AC4043B9982B89CCA03C63728C16086A2978A7366F95EB0CB02C1AD09AB95331C6D8DC6F8CF820A91B6D0F392A58CA1F11117B42A50869D781EA2B4811E175306059C60A9ECD12809CD553F6C51E47D62A68DC35EA71790D3BCBCD40B5116105066AA5316227AD1335DE3425FC627A70A0A4BFC4799BE3897E57779606763A69494445AEA93970E15B2E10858905C460E3E441D2D4BAF1C59C9FD04D5B44CFBF9970B04562231AC3785C6BC46B3BB9F51E7B107996F703B06043A3BCAAE735C1AF6A753AC8BBAC457057F794C355817B713290326E80754F4A4529B8253DFF763FE5F09D4234A9BD01BA0DC7C1A1ACCF6BBC4F5E1B5FFE566D779B7024E4187B488860F1BF2BF5B49817C7B4F07359B0C3C2809B487A61A213C3780C7742F7AA847C1492E7B31423B2772089A502655108C19FF3226EFA64CC1469420352C506268D82C327CB444408471FCB54DB415391404BE2F30EFF287D9FF13B636666E45564C95098B356339A955EB82313FF544AB1BBAD9E327237A9AB51648C9671B0A2677C7162A0E87C3901E2BAD4EC8B34C864DB329CF0087DA6D5003FC19217B7375D455EB91052C90392BC177A2BB21028D9A275F2920546696BBB9075826AF46A2357184AB42769B62B0B320B848B35003AB1A88710176764AD4E433C00ECC3C7616112FC9402E505488C8325C72EFFC3169D3C83F621C750CA4320E36FEF44CF57E979DEF5CD7B49C56791A5A1B530E992116C2A456D2737078439F5499639C8315D977FD7E0ACBF701CA9E85E301410966769B4B59E3DA1CAB1C844681A350C68760B5627074B3477A8C1BDE452114878A4B36832E82F9069C53E95639682AB7A94B16B253B42E4091844220375A1A3D421F310A55FA2A2EBC0CD65A224C27BB320173D40B59525F789ED4839F6FC2831555AF0F03FA1372ACBE0AB9B5B858EDBB579D1CA2CD529C49BB67FD30654D902957C3B563107E3D135AB64A821563353A48217436977A5472C88C6A8A30B4B2C4A92F62EA3C1910151A472E7AD81B2042A416629BBBCF4B7C64C739E9E54C612E41BAC517164B8258A1497B7C85E2FD28C714C224F5C1A9BEA8DBBA17A45735592F8CA7992975821A7D4B680D68A6A724AC439F8687BB4B59D549A73D4796DE10B21EB0F99395D4E809268C1AFF7E4110178C210D1A6759969338C21D7F85B846939BC599B6F99CC50B333DB805DAA0C49A219B15263BE09E6406FF66618E74A9D632B69717BE25656CCA251839C0A5A376AB416A57C933E0C2C464F86168D72476ADBF85AD10583F2FBCB0A2017ED8BB87797995EEAD8089EC7A257489778A6C55BF9AB90BE5934E39F80FA5DFBFAEE68AEFA4AD35CF089419A1D7E3D54DDC2CD09DE682ABA1664799BB5598942E5C1A89AF67BBD7FC609E05654"
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "1B26145DACAAB8A2FD647B3FC3D38C0BB9A931991D914DBC0A218005E4FC7F8B244D18259188E7B7FC90FD67BB1F37BAC3D2E3D86393C0334CA6E59461B533FB",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "920059B543F49FD2DE066B3881B1C8D29FA2ABB407145549F4151CCBF108676AAB7E6DF32388274BBEB48B3A37C1F28AC119891D75E56500F37D3D6B405D8669",
                    "86D222E8F62742720C95350BDB82874A33BCB8A9A6F0B12E2A3089FDBA7526A566B351617F0C5F1611989AB49B246B18E3D28093679E4E0A00C7967980B302F679BBA2425D1CA4C83E088F2A44BDCA760210D3097CD7583264663661BE8CE927E5911E4706408AFA58C4DC9F9ED5856CAAB7DDB9B2FB39477BA48D50FB6A16F90DFF933CA14CBF0C74A01A501036E47477EC9B43B23C594AB5236A386D31BA69739067D389B7D2310A3451F400A0D65BC318E16F4C2925BB0624936BC0A9665F0D21A106586A3A103C6B55734926BED1E7333697256C770F58104FFD94234346B637E7122306CAE6EC8D2CC89D26CA7D6BDB62E5413143DC9E61E87D4914457D67717FA206774C5CA00CA2A3FB65B75CA32FD7AC8A41505597117D7B7FBFFC4E107B5385B09288AC7003815FB95677EA6A9708D24493981CDCE2826CC70820C930F31754B38BC0CC60A45F86511A7761C3BCB50A44BA450797EDA3B873D927D4C92C0FFB0F28910D6E95A1F42AC106D5A046E45202B4623FF1BEEA076BF321C7BE2C0ED0D723443696C9DA5952128FF83A41302714204487FDAC3262B04491DBBA70922F3D9086E205284BE047D2B61E4073B87E2ACB36E11E91E46BFF443CE8EB1C1FA71B6DD0378C9AC1694B4F2B2A4071075BD0B0871F3C7425B950F22405974CB843CCA540A74DF5108B04D638522C50C374146A336253096EB2B40879E99472E916859543B9199A124B5216887FDB75BAE131C02C0BC31A034191F1216F5960AEB1AEAC2335AE198881712159B20E7E343976256842EA9CDC7284B3205A074148B48815AE826F102082E1013BD271781C80C5D50B753C94A8953667941B88E0BB463F11A30E7784549392EC6C624FE0B4F3E06365361C84C4AEBD18559D2535D267A0B6F08F865764F2021B62504AF0845A2917B20AA51DC2F7644BD417A5C1A8C58786BD7C1870B64849E50EDEA0B9F86C58233A75A55BB1B7D03D4EF464936B828055A5F06270ABBB19E6B36441A4B25F7A82EC3B1DA6725568B8B98B7A819D763E60EC5F04DB2D79B409BBD170ECFC3A8771356BC71F9C042ABCA45C6BB56ABA654EB50866B515668DC1C63EA8B88A816D35939CEE464A3011BBE7E5C326AB84C1E14892F755A329B063014E0B4607E62967AE6BCB1C9102942810DFA557E5479F5870C4D44023C8E5BAD7EC6ADB5B6C8858932749707ABBB81D01880CD1704F19217D6348CE4472F7229164FA2003B243AFFBAAF3EB2707886EF4980820F779A5117FC026C679C2B83539115D4AAA7593B6D14A5468541B44D77CDD96AFA40A2F59F0B08DAA59D72A31ED740ECBDA2166E65376DA16EF342C10CC79A8699D85A12082F8CBF8B73F5735B3A65C19A20B72EAD860D502255A24AA9251AA926750269A045346579070B98E8AA7DA79A4613655F787A1B7F1C7AB144A6A56474ED94BD1333CE9402CB2E7C8EA965CAB168DF71519483026D364BBFBA6441266B90C578B6760143CBC1DB2C834AC9C34DFE0BA52C84147826EAAD1BB5A0BAA01DBA5B2C7B804E21E6F8C87EB143C4F44A3BD79260BF736A1A82E38C7748D241EC114C1DAE6141B6994617774E2B41D8B397EA0264607E03570DA46B6FA4D03C100FD366B33C0659D40C9892B2258D5A0B85A33DFC391707B1A47E6783BA7127E124C1A920BA4A25AC6B727E6771339A892C1D8C2E9CABB8A8527BC3605F57A4D4E7979E3ECACD311A6EDE5A910877ED6AB31CAD8A3B9B7B1FC4174ED1162C3B5B05BA6AC585C38A3910E334128FB4C40E9B476A860B4F13124829A0332A73625C16F46DC3626691026C2AF0D2A5D469151B8F79044494FA843BA10E369890C8556768CCB349681FB42EBC16F4A2C74FA074C487C7FD0F680119A5198009ACF8478ACF900900758150352A14208776260B4A19B13319504CC56261738D5CAC0B3561088AC4091383D9CBABAFF23249462A897C37B1639A6ACB590FF067C1CD9902E748D9408305C4BA029CBCEB1E7672AAB3CA0B45080736E96ABB81A956F40767D0D0C0BBF453C8D873EEE41A5F5F81769AA9016FABE3E99A92311868FE72A1F062081F65C08096ECB1B880AF5B92CF0788FF3CC1AEC3CC639AE0E83C162480AF08048969C190D0B2C7B8B601BC16676D56E4A6775530AC1C98C950A30A69B39ABE281C8F9F321B2F62E74ECD92136C2D2928670B6F2E70D21D4EF270824",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "FD33DC89AEC88774D219CA1BB6FFB85AEAF0E7C7DAD4EC7E948882FA4B01EEEE6846325992089351444178C13B6B1139F596F2454E73D0308C6F914119CBB54A",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "63B035F32BBA7B03DF80DF9F75E731C19D11B925C1EA7AE3E1CFAECC9D5E2734C942846A8563A41C2D666753168A3EF62BDF55A2FAC8CCE81ABA57CFD6A86946",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "2099CF585476FC3A3D96811F628042BF5ACE4DD83533629A575C2AF9C89C51E8EA0332BC7444E0E1E84DB4873221A7811FDEF6AA2609985256449D3457E487F1",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "920CD04F4025CF0E9F7AAF57A072044BBDB019DEBF962C944556272B2CD97D8C5DEB0C8D7F930CA124AB843003C42115EF6556C1380D9F9F994FE24CC6F4BA24",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "3B5CF90BAF7956304482F83F13A2DD111AF36115260FA48F5D38D53F038916FE4728B88C1D4BA3F987FEF5393CCF6C57091A90D6F1ABF6F4E817575373DE2F3A",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "BD81A2BAB4B099E15C6689399297EE6206352F39E510DC3EFB2CE2CCC03F6EF207B9AA3E4014927CA81E3876592130A657BE0ADB1F198B68C0E3CF472D42FF94",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "068238E9AA7926BB9A9BB97091660EE5F81192A3E9EF6E00F694ADDD78C3AF02BF1F026EB1B0E362D210EC6FA841271025D3F79454519208F0EA998DDBF8CB86",
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
                    "D174AFB1F07902EB8BBB3414C5D3131A28C8435C7DA1179EEF9B6A1EFA380DC77218F46A0407248D422E3CC8ACD56C4918996CFEE59F8A1046B46456229B3198725C5C23C245B047DD4135A075509827377E4782B8D5BFCF204D8ECA2E8A633D91B804CE8CCF2BC658A9FABC43A6B08C9613342117C45ABAAD5790D52191D9BBB3E1FA3897D13D9005C14B959918748C19731DA0CB65CFD552AEB78E4F663CDA22050A197BBB394A29E4A13E146F8DDCCD36E47A826537D6DA0E58A8A3CAC3BB605216DDD93D30A0CDCF92C7960B2F54AA0AC942B36E5996DCA19D6119568559ADC0912072B42706D11FC8D8B540F765634C1DB4EBC1B856C241AB66B0A22EAEBB734758C5D4824CDAF3A6C473AC80D7581C5B89D8D7A26096B7DAE9CA664920ED6AC7011ACAB6B5000D7671B0B778A133172B03BA9A30B6615A3D217572CC82138744B30B7A48BCD7A3F9F8BCFB943318C641D8AB2EB93C02A6327AB2888A6C4485272A8A65DCA333602E9DB35671A70F6A8C8B90542B51227F1E83596E206605E028C4D5B1BC1CBFB67282A5380423E95787D4593BF4A12568455F24AA1968B51108801E8ACA8B18690E92B0C35385176C4F93A02009EC601F1985F5DA145A4720A8F2BA47D89F9A443F09A77ECDF44393D260292718902C9DFB0AB3B5242009234D70D46DDFA90FB2402A3384C94A620FF9986D5AAA5E354403A08A1A3FAB4BD1DC7E5FBB500F64921DEB60FF1A52FFC741EEE78BBB421AAAF1747D620D5F082D1CD9229F307DC959B98D3B0FE2D827FD0857931B5F786845979883923404285AAE4C9A313C1C915E528AE1850B53D00570F006BD135E3B4178DC7675DCF1989AAC7BABC567E00C803CC0CACC447F6320089D34A28AF446BE1CB5A434B7BD661E0D5B75FD4BBBF1D43B57741433E3A854208B9A9C9206EC932C3809A931CD961983F2C8499D20C6EF724CEBA89FA0A5C8BA2CCB441AC91E82C1DD3BBECDE5079CB322C15CA4135B4A9C73C1D1B34DB372AF9BDA1800B3779D51C56D02226CCA8E6B9C0AFBE7B6D49C6205660451725871F602F75C19E86B1BE69493ED404D170311ADC8C9E2901CAAD188F3387BC8A70D7FCC23E5F915A534138712412C61624638388C69C94A45A24B041CA459AB14B887A2CB84C65022304CAD20C55F72D460E438AB0BA338DF17713B858F8FA38C0BFAA769272970130DB1D60E117B59BFB6AD3B004BDAE07A0AF102DE5606CD5925A6741C3DD2846EE96454E8A57477471237127AE4133403666BB0400EE1528BF528B2FA391606947873A63D3082681C90EC1B5793554767076299B1401ED59A1A257F3127A2F3841D10F3C9172BC291832223E7BDD1501D5D41C04B6B9DE5EA5C7025406296A199168730218AA7470E43C1A7C6E65542737A4155AD279C037DA21B2527241138BEE3959C7A18777E3B7B425047577A56052B7B5F95993E642ED99634AE3314C815155C91B5C6E09CA324C8D5496ECE56578AA302CD2CB99E9905BB4AC68FCA6658913B4AC40AF3B308224357E7E144F3607D16B57F899C724EA15561F83905EA594F924931711DB02C7A813C8D8E553231915FA677B0A348983D4851279ABAD8D3548EE805F46890956260704016DA6B23E1974E4C674D570B4E5E88B924920EB9878830C61BF0097293668F46E851704771A4875170673EAC3341274B0CAD949234475B2F40428670578A47898D025B4B54AE9AF5A756D3C35B590E4A34ADEF000814D59893F88F280B3FAA228CF0ABCC6066068D636924D455B649AAD1E32450C72EC686CE5B91CC9202496F073F6056450EA757AF4465988125CCF0B2FC0877A9836B54952039114F6952AF17D800C2390F1BC2C62447C6F80A3683858A192693E7F30332177F5579BEBB854901D11937399C12825F25CA18FAE60AB2964A89EB2099FA579DC04577C3304F5B4DD4B43EE1D4A31A5809DE807E4D16AA20B5A6D0FC93AB4330F2758A774341798C3E79632540C2C9CE5C461519C874E5AB113482133263A3D32A4E1BA3DC49C76C8B3D742558CD9C3CE982753F65CDC1935C99B89F160741E3FA1771B2C7F38BCE3D57A26E433E7378A19CE66D7380A2707C72F55A08A7419E2A986BE5262F1398978C6B2470B67C911AAC2BD5846A389F02CAAEB301201A70A65611B13D4B6BF1E36B048086611133DEF6CB5D733F86E88942B33E579A547CC08510A528E626A403144328E6BF2B829744A321DA6241999B41BD87A77004B874B22327BB66C53715F7EC1996D298691A68A920CDEB0808D763991F37BBCC7A962BC235AF210359250DC496309ECC66E7350F0EA0365419324C2B2C934CB992F1AE0B444979A69E2D262B7B4C7A1F6395C88C60035A2F8EB3C086442307E984844607D7299A5F63BF9A181EE3E14301898F50A8671B38BC53F9007E289718F4217AEC842DF8843745130911403AC8067167895EA39EF7852ECD6149DBF0C9F144AE3D4B1ACD2C8A5B0208DD348089B898585B9F5AD27AC4C8088F8620C539924E8A463A923A59BC0C38E2635BD35B5FE029134C3FC811237B4B0BA181CA413619A5B01A21596EB0D6B526796C36490E0D86765A5C8350D03968CC57A5D07D0AD09C0399BE777271EBFB426A98076ED169DF6035C9B251A2E1BC5E31447134A07E2BBF0115635FA05BF8C25C888906EE005675372538E5103D8B4C43439896D0238BD94FB4016C8721965BDC4485E7509F63A15FB65532E8CABD4405572370FA7844F388721E6A91FF83345A6C0D126321EBB272FD784EF8CB580FA1C2F67039EB65389D822E03B067FE069F49FBA16CE73EA88B5FAAA96530B27BC5483F1F245BFCD80D3A7A4E377B795344A0E82609E5C4C29CB245D8F6A112452CFF192B1B310A33A72CE86B91CA1BB6384A7F39D360F2164E8BB340C03B0020F53CDC178BC09C9196A2C4767A0C82A20DA4A5AC8AC67C48294D69D4C1899BA3C328BB4A733D3F46152011264F259189E71E30B909CFAA7264879A62E2B1EDB065641BCC99E9890275AF9113A3704BA719286516DB110629280D5124837878ECC52BAA066EEEC4C21F007F3D7101DEB0AC78F35F3A88012CA90EB0965A8CB6081D639725BA5489297086D8B7A8C835FF5C07E0518DCB6BBF1EB46D232A22D6E550CBD5BE45D0759F8C98A3312A921209A5932806F0B136F37CC14A9F9F51C3F18805EC43834D33AFD6158ECF680AA75A0C41121C4E9969305359E51B9CFDF16A5AE097C0919593824D6264BC00C53D847CC33851B254C94956E321CDC33086D8742F30599B284B95D8CDA85C7B4AC97DAA02566C8C189E26A47C240C21C53FFA2AAE3E405FD4F840DC3BAC25B3BDC3229A54AAB7885734E7E35ED8E544E1227CAF92174B21177319C9E5F6C04FCAA136E063B628A6418B49EE4B441EF411483ACF7D44C96EF46A85BA3A2D5B740CAC5F3673A4A2039F3F36AE39630086B07395C658C962C8CC885B04C156B8DAAD96B93D7D3CCAA404BE8B75191C584AD321268F3511AE30806E599DA218537B230F631745A41B59738B9F71D6B850B2BF7AA62BD87179493B5A977C2529618B65F58BA182404F720134F00F9478298BF30833CC167655870B96B4979B94754B60B57BA74DFA8CCDFAC65341C47DB36210246EC5F1C300742381A12069F04A060457A1A596DBB1AF5ADA683D18223AD28DD267AB4679479CFBACFCA7162419844056BAFA1B115BE18796101A536789D7B8AE72114C74597CF7A2373C9608878C14EEF22AA9C336ACC27D3FD7072DEC5288EAACD8D8592A9910E6107AE9C25832DA9932A95157C55F6C7418BB21030F00187B3C1AC221484D421C0A8C80EFD11DF130B76473C8337240FE818C3F01863F3663DAB699D6211185353879045BA5F572677B58B658B21D5B0CF8250485B288FD63832FB37F733556423249827836A24686520C8795C6444F960B28C49E54D776FD1B80241945D8158407AC3272AC25466ACE681BA971982B7647BE5CFB02B2858F1B0023224328AD77B0A1954209468F58D88AF98652181682E8D6BDFA7167074A969386B532C28264956AC031201D0A61FD857CA85BA803B329FC532981C321B920B8E87267B5359BF9CA0BAF746B94981D54E92591F848E8BC0692C585D2F79D9990C0E4E030DD4265A7E946200B397DA61F9607A22E48157B6261DDAB835C171C9B57688AB52D6E31866D22833AF425E55634E0733F5464B38F68303A4003AB374F4EA8B3AC34449B7A2218028360C1CD642370E429BFCAB7BE93B949EDA56C66881E3CDC59CBF9BA79BB9D3BE8951AE095A96B9A18041DFB842973273218B30982736D5A597BA0015575416A851A6ABC1550A269343E65E0C5946F328CB3AEACBD87EB6D8BF20D9B9B2C4006DA49A608EBCD81CAAD324E8CAEC0AC3974A54278A60F53C18588BF1F026EB1B0E362D210EC6FA841271025D3F79454519208F0EA998DDBF8CB86"
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "AED60297778AE47064D6FF0500D7BC2F9EDF4F78A3B3DBBDFB20CAA6EA2D5A726F04B758B52CCB2D472EBC07A214A743FE31D8949D9EDB652EB7C442B72B9E21",
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
                    "B2039FFF7802F49A794AD71F49106E42C3B338C964BCD7695CA2C358841CDF8B1D402058E12178A3F6023FF9C80FA3C45A058E3DF44F81841AAAEB2019FC4AA625120833936C026731261C7E4550E715BFBA18C5685058F681175190B49517A69650787DA88EB06B6797787253656AD1B922218AA1C1C8006A66CD907642C0839131A36EBECC7F8E4BA96DB49DB6611D16A368F2000072860D7DC79CCD4B342F983593732DE6E133FC521E325869CC4A94E3F41E03724918B01485729F5E5BC451422639A0328E740EEAF49526752CF2C83AAEB87B36D65824184C3E1329D382491F07A85A80242DC3B99D91C60F7C07C61899AAF5342B5B274B3433A316BEF3AB290A8830BDC88FC0F3C535412B1EC78997B6004AC20C8ECC37BA0C67CCE6420958C5A2594970D2BF514134F69435005B351BA09BD7793688E5522AE274EA53BCF7028DC6961BF9ACB935A80282886C66EC426A707A8045B05B284A25C636E01C47E8958453F40B492B6F106C6EF267A66AB0CC4317192CC2C88F45CAD41352E2835F3A26868B2843E4F16E6975AA4C3357503A70E2615BA9B14A272040C7354FE4946A3543030781CAC0DA548D2C47440C4923FA4E6197B70AA8B62E0A74CA20C6FAA987EB420E5B3C1B34CAC49F790B155A5AB7BA08627647DEC928F01309690BB065033D35B0897BC5C77B33CCDBFAC20764796FA59DD721BC4891A30687CD348670A5D596AAD7AF6B1884DD33AB8569B3A132772DE4B862C9472E010922E691320A9F11450114628A00155FFA880F90AA1185891E0B04B51DF51880D09B289473A15B4A45EA74950885E2021B7444205C2C7D691C02BD34110CD1CAF7089F02A5044076583EC013A0D427A1A799040297ED6CC37FA52F0FD76B5A87012DD9248CF892E822C40340AA541108415CC74159C83A7A1F2E809138745778997D89D11ADCB375246701A98B3AE0CC0E75A1C6E12A514BF65220B9CD4A25BB12E54EBCA815B6037F1E000A10478755375021D52535681877BA5AAC34693FBA7968C35F8AC265F93233F445166EBC8CC5F7C7D19AA457B37F41B3A60C793A15208F92A4A54B0922100A01D8F2BA11B08F3EA3792CC9CE989269D774645638206D4C9733267B054B5C998254B7B3AAD29452F8E3710D0274147643A7E83E9C6897EBF85C0464AEEBD2059A95637E8173DEC125436597167BC7AF78125F428C9FCC3A250A5C9E1CC527E98FBFF44355CB3A75E49DF7D68829760152D1C37A265F62A361719CBD4C459C41402D0943A362B1CAAEE5A1D72837FA6183F33818E4105BB5172F9583B0A4618EEF41AA8AFC73B26B4D6718B1966083F25332DA28BA9CE3B4A3A02264661BD7A9B927F60CBEB540EED7A2D658CD1CE3613BF14E8A128EB3009E6089C8A33BBBBB44B66786C694C785C29B6D309451F78C526B67577CE01B31166DFF4320B04C37877298E9A63D0124291ED630AA7AAC94A078EE2A99A125513961A20E0881FFB6439F371BF068956C320889BC5796528B5BAB5BC78832CDE75263354D04D26BFDE5C94E02AE3E63B8CA1C41650A55FD565D6461871B07CE859C78BC6648A93B1586E793889C9271A954DD0A6B31C2A16B92CE350753E1B6357B14C49A8589C82242073C4E276AC1F290880DB87B12C27911B909817A8DC2EC06F347A0144A065C5A6784135CEB6A473A1A11DA66CF5EAA690F8675032818401B001891CC8D320F33C094FB812CFA94524DCC2F6099AEDD59BE56782DD6097D127CBE4E1B2035377DB7723D6F8968EA2413DA2A04705BB233F03879FBC6ED70A4BED67C9B47569A959BA18ABFB6C45452BC9A6DD957B2F28DCA3A8675424E0402650400CB8722B81E1149A5D55A724B0170B00F8A61AB641C8833CC61E1E5A76855B071E5B8797102CB339E6566B058237167A86490192E5D592BC85AA1D3734117B51B0B3B9259DB8E82DB15A504B0BFD5858C0C2B5D59202C004000C2BAD21593C755B87A421856F02200723CCACA17A8809D64D95139EC55EFC795C276B9FE47B4280C67DF685F806243178BB4B7462CB03A5AF46398EDB1C61D2671410AA3843B9D48ECC136BC7AA5DCA9F6B87268A38D59A0BCA9A7602D8CB4706731F5F3B0657796BF3C3DC1266D54714929DB3ECB684C9C132F025914539629A324A098481CD37A32F9A74E11B390CB51BC7C48B643234E617A9C17683C2987B1AF30A4EEE73029A67C49EC8853C81B0DD1B5399171D6A5B026275AF8542255583F36869751608B67B19E86181AE306040D573D0556534A8C390F84A14BF36B2FC3877A3362E869C70194492EA409DB43AD762A5BB5297700E838CD81A02211318FDA13231538A82B98BFBC2300205E7A7351DCF3A3660C9943204202F1CA3DE22BD78C2BF6C861B3257C66B95216EA6BA60A7949748CECD430FA426E79F0945B824D90E964F2514C9A75C251546B26D12E8A4116620B16540AADD37A48D2BA29455585A76A9A2EB02C1919ACBCE03FD56043CC17AD4060646AA065DFFB08AD3624D6C31A25DC99B9951B1C484DA39430A6C1BE0CD8A3385A2FE6A602F15370966392A57C7F97817BAE4832B51907EF937C1BD6030E072F680A0D81D7949EF025D52C3CB9115A6B65B42101852D2C7BBCE56C0EA9B9EA4283F524263BB6CBF710476BC235F3164DFAAAAD5C3200308581D6584B02A5555069CB9715C6EF9238DF381598941589C06BD613A7DC45513A198432705B17B278826415204B5D0675C9C75B68792942831859028AA54EBB111A038557416FBE6AAF91B18CC4F6092C0BB26C902F32E81AF94B101733A572BB14BAC3CC7EB6333C662647A49F3318B531C9C5F5A0A924B501250218196BA66E6C9717190A6F94A76AAAB478C69EFD3C4760E694C234563EFB2BB9743BAA466B3CC54017F8A482428D8F1BC094B9AC2369557E0CC2712C1877F718BA712FBE76CA1BE3C01D6C44D435192113A5C362C18BA10AADA28B7B98564022163542C35FCC6EB59B0A7C3621B94471CCF691CE363EB0F486DEB91D3DA11E2C031D2C331C32E88D42E6BF8FF3C025EA44732B1A0A2B49803A4969DA9F868398B5E88F1C92745A8CB9D64C65187104DA24B3D1C3A9571C75868117AB5ACE698B50E329B30F3245CE02B032802D7695CCF9778BC0B622100C35AF184DDAEC743628AB053405C115937BBA2D383293D4E93E7726AA70C89DCBAC6ADCC32DF0F6CDD88CB8F1C9513B230928052843DC9388BA593154621D4B6F2C042D1762B9E28B6B605AAA2267A441F775720C7D5DD96900EB5C326CA4AF452A637717D41558BF59475AC34069B87B2A2086CB34945EB21A5E79CA47B18ADB7043CC7482BCB52CFCB6C44719962E045ABAF88BC6DB92B09219AAE7C74453B86D576AFA1571E86B1BBF28B4C3D82ABEEA9FBB3B8836A97DEF5B0141794A9036B205C87D3812753DE74C071C3C3F6B1A59AA19552C2F42342FECE63CA27997A9B98F220C20F8F47C5E3021E9295983C757E24088D5D5197F5C912087C35D0A114D26647E2CBB1C16A0AF9C471F90720CBC118C0AA2D4A0638F3B02C33849FBA0A94542492260005F959AE3907B27F2A7F2602CDAA2109654078C2A90379A5E9D49894B007E6F854AE93520BA142B32660A7FDCAB877B2E0C68756B1324D3B9A606B4291C15B5429060833BB6575CA0E5C7689A4A131E26166E984A9CB06AC91A9991F978D5F6310AA91A1F4B14BB802B4C008CDB70ABB8068AF3C40DB979884FB61767C260B6B72B3A6BBA2ED2968B83689EC80D387ABAE88C184CE166DC0A2D096ACC36DA5F925430FB048BC1FA4246B8C0FD103745376A4BF858A3310A7E611B29F830F9C5C57166AA0FB3AAD45B6C55110562ECC50625384DEC579E5B37303C8C115801FD0617BC5133E0A3A1A1BA5B1CF51D87009CD5F617A0B1AC9970722784407624451DD8725BF4874321936B0C333892CD2E3097517604A1657B24220558AA766EE44314C780A2D5BB271A7E8A8149C08C8F65B3A293431748CC03E0F3B1C985096CA181FD41C50C775C667CB67B75CB91F972E5AA6CECBC0175676A18971E2A0B682EF16D66D87815F8B662AC427F2B4B48DA88E407495154322F1943F331B0F7B1A04E9712BAE9B672440FA4729253D09473BBB023B91281357C88679C1897AF4F096CE3A38F9C88A3AB847E15371E6C8B590DA6899ED640462BB440883CCE06CADC8CA113676296DC5BD1FC57CF829216CB3E32F5C9DE384189C5688D164A4E480A0424973A476170C63CBDB094AF629570C35EF97C341DE3279DDA717FF823EB0A34AE153E07427949AB07B015A1CF32C531D952F139A8453464F7360F68B04EDBCD136DFBF32AE6F93BDF366E855FD0422FF874F2B65947791ED43084F92B2B0B485987B6F0ABBC825E85FC66DB25478D52D1C387AB92FADC4B6F04B758B52CCB2D472EBC07A214A743FE31D8949D9EDB652EB7C442B72B9E21"
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "ABB31CC188F3B1147252EC5E36C5E328A3EE1FE7FE283F79AC948130A77FF3100ABAF6E7ECC4040E4C55EA78864C341B808ACCBA04C5DFA29A09868C02FD695C",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "0D63F93A37672C314582D7F69D6C7C1A7FB13538E0B38512B0A8394867714DCC40F84C79E0463EF69B766B6646442844846B9F262FBEB65ADDD980755EC8A7B0",
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
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "41A3FA743B8AEB29BF21996C1449C631815D297872A80DA5F2A97D0FB1E76C58219F84FEF373E7EA91FCC6838A57A9708DEF52ADB6A709C5A2EEE822747D2585",
                    "848C3A88783DF0A82AE0BC9679AC85DBC60E90A1BCAB784DF42B1C20A29A30269DD0F4C7873AB9DB1271C5158264B28C5616B7CBF92AFCB83FA40B8C461A5A62198F80F5AF00FC304EC693553748A0A82DF43B96A73131FF59A7BBFA05AE5A9714115B4648CC11E4A0A23625C338BFB7604FA5836A7EB05332578179AA25BBB022F94C326CE52525B23B148995F39C575309398DF25161428479C0322E54A5AC699D151148CBC86A2B5C9514BBC235305B52284A4868944364BF8FA119746190057C17380ACC5C509FCBD1B84E610909951ABDDA1676CC62F3E45AA9E841F4FC41C3CBC83B541AD0102949D017C57963E1F094F39BCB8FE3B5714221D7F977637943EC51526CF01BD30344B5E247DD1498ABEB25A505551F832616E12E343674D81059DFF22EAC24CF921A33CD11C911FA590151B88FBB08F56A905C4610060A14316B29703826C5995141A2BFA04896BCD641C0FA1CFAFB245129488F01421C611C4360C96333A45CA59400C1CB94C85F0F162BD2A11BFA4783AEF2721D9B0A08A7451738CE3E057993D8399B90B25CD14B23216164FC31F4A39E43E62E772ACCE28AA28A19604D5607734B475CE5928EB98BC79A0ACC5C976AA6A421C4895E88C669231D1C1B8924F643E51CCF312982023A4E2879365312CF9EA862BC4C926F523DDA22A80F075CCB49294CD43DF50238E73C7FCD9692BB432E77EB6CA260CCC50C3AEFF43995BA7DBE25507CE7956AA266DBE477F30BAF7A0B3174D7C2391B08AC997761CC4A069549AC9178FDA481B434241E21CEA4178A9F4030C71C5D93459273F27DBFAA88F428B3CBB1C4F94209B49BB950C27BE37C25F4A523C616995D56C7EEC4222C7519EA4661ACC5C0B441CEFE61425F441843D162F6B0B3B7DB9D37934524D7C753BC66B5451518B5297672B4683A8E84FB6FBC46528D8618FB2B6E780211AEB8108929B0896A6E714AC0E07C2DDF66BB1682B794DB3A1FD8CC027A59EC605DEA106937852CB22C7D10014B36F6016C795F540B79B282A2E43433AFE1A039512A5B711D34BC74F9FA2B69F74EA02325946C3C1CBBC915B96E85BA1AFBBC7060B29006A537FC0AB648D17DF5993ED8CB24A143CFA9215D4CB7185F66674D2404A3F682C400635AA65BB0C4538D886434706488D01600AA9230245EE7303936FA04D656297212B6C0F4479498AE78BA1911D1A179E20BBCE28CA0E001FA3524477CB181968B03934E20809958E556DDCB742D7A817ACC64CC4950C1438FE7144DD9A499AB6B5469D0C1443CA03E19BBFA57A3E6F5C962C0915F3C97B1584DF8767ACFD21AB4E8A69B82A2017857A0490A87692B057994F5527354114BDB1806E08CC7DF876BEBFA9FB955B5FEDA192F190890911D8D6C5F15036FD43C3C4B09112C08B690FB7509F994870B35E5DB3281CC2020E3B6D532798BC6C01874265F0284330ACEE94B1A88F353ADAC6004734EA18039D5329EF185AB655A363FA85771F6CBDD963185E97B84134D4945A38E5A89B289735AEBAD7C3CC55E51675B8621D6B385DB636EFC630068698C6E701C9BCB2975A46FA48B9D2908CA86162626C26EAF588018D5C2ECA92C1F468BCE98A82BFC16336900AECAA720BBCFC920BA584B5F987C3ED9E55FC3768BF6D78D385C0DA3A649D60C82D8C784A45C2949858D5B5133C2A23143A1B7535898E3582F0D945D13493690D8884C377A6145A772C710D7E02D65B9134BA130EC320873341D0536721782722ECBCF288311F86CA2A0298753F78E6AE5300CD30D4C61A8095C06361C7A19D84B53B47F24BCCD598197AD0036157164D1BAB96A348511851C97E77103531C2E617E8B049D3D57570B9951CFD83D3691C57218AF498A5E7CCA948A2542EEF48038FC88512A9F836560E4A36C6B974528572D32428E0C71A35185472B66578BB6A316A043EFCB1BA9B4A375233325A235DCC24F6C68839DE2C6B46279F99215DE82AC40CB1310696504DBC88B598F161B0FD1F6374C590B89A44E7C3532D739708114AA634567A030B503E1A58DF81F4D5057F5990C6AF64B98B89A39818F5ED48B4E70BA5331666B46A78FAC2157D174300677F191B834956695B83154A85D32C1C10343B78A8A32EA023C34E00F70805C10394EB8749C40CB56F3793FF9D14E287B323DC7CD5E8B10AB2173AC13B3F316CC551C4CF5629CB7725FB79FFD1FF1FF0347",
                    "FB82C889A42F580A49459BA4A855B93EF529AC787F2DC68374D4C9DA1100C9E3BBFBC60E2DDBC59BF77808C69177737EE59685EF6B41318C7CE2933DA466A5DDD82489B14286937A22A7864F4A049FD31643F7CD67389B1B9C64C1135861207E34B237DC30058D1754EF0533606674305BCAE6C737E6585BBAB9B1CA49B36995ACA37549CABB630363C322636AA4063FE1F50EB7FCA6226AC6A0D0116DFB3491DB9C291B14FB4A4F55E19383B03206964492D1968218B94D37B9B73A355CE73C779C3DEA2C0EFF802560ACC86D7052A4634AC401128285529DD008F013C0E1FB44578249307399D088B0DF79679516CFA126B8E3397CBBB97139A3144C52A2CF103C68081CC4B8812D9A08EBC0577972AB60360767D4C8B032831F7869519A190B04984E22A2F6B590A2A035B79A82D8780BAEC5A82B985E545AC17B57A2AB24AF80B801A85A55A9C149B39B51AD873A85D23A00C105CEC7B5D5A894F87B5E7AC21F79D529F3367284B70EB00781E8EA4EAACA5C32E7B7E876BAB694A81AE90797A0877FD363503B468E01C3208B4B8A754914410E9BBA2EF6B75326DA77C2A7A287E27515C9C8F4942DB88A93189869E613A8DF5B966A245038243BCD01820D2007EDF046F9343177258CE282CB87D79F176AB753BAB4E18914D714A8C447192CB3AA85666C2B6779B7E986A21089ACC36A70A13529032D00E902E4FB7682C3351B71201F3086C4311D4A440A31287EF8BB1AB7F36D9546B04F61A67917AA1A1568A2A0337109A7A9E01DF09A2DF8A390F9DA6520C1611E946F0DF03D3F67AB9FF2308178BD300013274957BB541E718375EC280611F2B696148098069B90FC6A34C074963C7D245199E0B5279D555EF4FA438F178A7E84BA4DA895DD57BA0B3CC565711D613944A52836597B41449394D2441ABB489776C566CFB93CDD703CC4610E511B18F6E73E69EA3D74356168900B239CB897EC1609D118C4E814D0A34FA123596395865A7A2610983D03207C98CC8ACE445280E2A80DB917234C9CB8FB2469C466DE9687D4D3514565AD72B54CD9BBB3B02C0C12D4710DF7418841C13023B7BCC67936643BDAA5B4A43A757B7B625ABB01268C2995F399D6DB9F599C8CBBA94E152A6C9E02368B7ACAA915BC5C6A4BA8F5A52D6AA2CC3A4A117649C196CC3F924F89115B818B099EB109461289DB90717CC7C56D48035D1A4BF2F0BCCC4CC0E055BA765791E03B1D1622B5883718AE886CEBE9B97EF13DBB1C5866988FD7811E5C693A718141A7E98986C820CB5264AD3C22E14CACB300457E3A926129701DC6945F1B3F28712A9B830507D50D959444AF48106A17B99C4607ED389333D003EF7440832459D5826A95429BE6122F5ADA2A36755F066C5F5A63AFEFAA36C4C17795C68BFC0A91D8459EBCC7BC24E1354E56863A65333600B5F24834A8B3670BDC837ED977C4D18BF5A76252270D87D796F75C69DFA814B84AC1888A58BF3268A347611AD0C851D426D656468C627ABF335401F7906C5872983AB6DAC3362520BEEAB1CF1D90C798C881D7F908EE8530365060D97C2E1B704C589488DCB60B1622BDC74304DA05958E6468A6565A58600EA6E8CA78A39A73B87BADFA337031184BE559FE42B321C42452B94B6E57321794BF574407A2953799079D0BACBEA1DA141C723066F3BC8839446C33095518A109FAA70FE0011542AC07FB0CB0AB3F5E05B6412967DD1C376EFA6BABFB499C2665973AACA0A5A70436AA707810D8F508AF2914312257A374B40C8962C8E4267276CB3513BA96675CA86860A1F72F4F11530C07B41EF89BCC92CB9FA564C5283D8D1989E40A4AA87022DEC24367D1B6A4DA7CB995108FA1462226961A255857B2264009A74F66344F34408FE609654073382869805B26656B21DC6039E0831F79244F6F173985FBBD8E33CD987617701948A7132879AB69FF5324D35A59D8E275FB6AC3FD55271A3A9A60FAA8AFA75798E68851A2A65A3028EB55CA57000C3559CEF03872A2D7A3378B4808D7AD2DE01BCC1797E1283C87548D6BC8A13348A5BF5359D28C8F578A35A52ACF8D2A3C6AE0A55EC1A11A8382252A19BC464F4DB2CB696B1C5B98774A194EFE89813A950AFF05099353371420A226C971027BCDBDE3A745C25D9E49C8D022CE3A538C848C3A88783DF0A82AE0BC9679AC85DBC60E90A1BCAB784DF42B1C20A29A30269DD0F4C7873AB9DB1271C5158264B28C5616B7CBF92AFCB83FA40B8C461A5A62198F80F5AF00FC304EC693553748A0A82DF43B96A73131FF59A7BBFA05AE5A9714115B4648CC11E4A0A23625C338BFB7604FA5836A7EB05332578179AA25BBB022F94C326CE52525B23B148995F39C575309398DF25161428479C0322E54A5AC699D151148CBC86A2B5C9514BBC235305B52284A4868944364BF8FA119746190057C17380ACC5C509FCBD1B84E610909951ABDDA1676CC62F3E45AA9E841F4FC41C3CBC83B541AD0102949D017C57963E1F094F39BCB8FE3B5714221D7F977637943EC51526CF01BD30344B5E247DD1498ABEB25A505551F832616E12E343674D81059DFF22EAC24CF921A33CD11C911FA590151B88FBB08F56A905C4610060A14316B29703826C5995141A2BFA04896BCD641C0FA1CFAFB245129488F01421C611C4360C96333A45CA59400C1CB94C85F0F162BD2A11BFA4783AEF2721D9B0A08A7451738CE3E057993D8399B90B25CD14B23216164FC31F4A39E43E62E772ACCE28AA28A19604D5607734B475CE5928EB98BC79A0ACC5C976AA6A421C4895E88C669231D1C1B8924F643E51CCF312982023A4E2879365312CF9EA862BC4C926F523DDA22A80F075CCB49294CD43DF50238E73C7FCD9692BB432E77EB6CA260CCC50C3AEFF43995BA7DBE25507CE7956AA266DBE477F30BAF7A0B3174D7C2391B08AC997761CC4A069549AC9178FDA481B434241E21CEA4178A9F4030C71C5D93459273F27DBFAA88F428B3CBB1C4F94209B49BB950C27BE37C25F4A523C616995D56C7EEC4222C7519EA4661ACC5C0B441CEFE61425F441843D162F6B0B3B7DB9D37934524D7C753BC66B5451518B5297672B4683A8E84FB6FBC46528D8618FB2B6E780211AEB8108929B0896A6E714AC0E07C2DDF66BB1682B794DB3A1FD8CC027A59EC605DEA106937852CB22C7D10014B36F6016C795F540B79B282A2E43433AFE1A039512A5B711D34BC74F9FA2B69F74EA02325946C3C1CBBC915B96E85BA1AFBBC7060B29006A537FC0AB648D17DF5993ED8CB24A143CFA9215D4CB7185F66674D2404A3F682C400635AA65BB0C4538D886434706488D01600AA9230245EE7303936FA04D656297212B6C0F4479498AE78BA1911D1A179E20BBCE28CA0E001FA3524477CB181968B03934E20809958E556DDCB742D7A817ACC64CC4950C1438FE7144DD9A499AB6B5469D0C1443CA03E19BBFA57A3E6F5C962C0915F3C97B1584DF8767ACFD21AB4E8A69B82A2017857A0490A87692B057994F5527354114BDB1806E08CC7DF876BEBFA9FB955B5FEDA192F190890911D8D6C5F15036FD43C3C4B09112C08B690FB7509F994870B35E5DB3281CC2020E3B6D532798BC6C01874265F0284330ACEE94B1A88F353ADAC6004734EA18039D5329EF185AB655A363FA85771F6CBDD963185E97B84134D4945A38E5A89B289735AEBAD7C3CC55E51675B8621D6B385DB636EFC630068698C6E701C9BCB2975A46FA48B9D2908CA86162626C26EAF588018D5C2ECA92C1F468BCE98A82BFC16336900AECAA720BBCFC920BA584B5F987C3ED9E55FC3768BF6D78D385C0DA3A649D60C82D8C784A45C2949858D5B5133C2A23143A1B7535898E3582F0D945D13493690D8884C377A6145A772C710D7E02D65B9134BA130EC320873341D0536721782722ECBCF288311F86CA2A0298753F78E6AE5300CD30D4C61A8095C06361C7A19D84B53B47F24BCCD598197AD0036157164D1BAB96A348511851C97E77103531C2E617E8B049D3D57570B9951CFD83D3691C57218AF498A5E7CCA948A2542EEF48038FC88512A9F836560E4A36C6B974528572D32428E0C71A35185472B66578BB6A316A043EFCB1BA9B4A375233325A235DCC24F6C68839DE2C6B46279F99215DE82AC40CB1310696504DBC88B598F161B0FD1F6374C590B89A44E7C3532D739708114AA634567A030B503E1A58DF81F4D5057F5990C6AF64B98B89A39818F5ED48B4E70BA5331666B46A78FAC2157D174300677F191B834956695B83154A85D32C1C10343B78A8A32EA023C34E00F70805C10394EB8749C40CB56F3793FF9D14E287B323DC7CD5E8B10AB2173AC13B3F316CC551C4CF5629CB7725FB79FFD1FF1FF034722DAE6A75688BD9C1E7F50A40E002CE2E10B19A9C469EACAADA696F2D56111BA219F84FEF373E7EA91FCC6838A57A9708DEF52ADB6A709C5A2EEE822747D2585"
                );
                yield return new
                (
                    MLKemAlgorithm.MLKem1024,
                    "E81D3AC5B001861ED765F7960E63EC311D97110F2135E97ABEFD7A6664165B930D79C9327545DC957BC15C193974585740E58BB95BB52AB24FAF0B8C9601DF34",
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
                );
            }
        }


    }
}
