// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using Test.Cryptography;

namespace System.Security.Cryptography.Pkcs.Tests
{
    internal static class SignedDocuments
    {
        internal static readonly byte[] RsaPssDocument = (
            "308204EC06092A864886F70D010702A08204DD308204D9020103310D300B0609" +
            "608648016503040201301F06092A864886F70D010701A0120410546869732069" +
            "73206120746573740D0AA08202CA308202C63082022EA003020102020900F399" +
            "4D1706DEC3F8300D06092A864886F70D01010B0500307B310B30090603550406" +
            "130255533113301106035504080C0A57617368696E67746F6E3110300E060355" +
            "04070C075265646D6F6E6431183016060355040A0C0F4D6963726F736F667420" +
            "436F72702E31173015060355040B0C0E2E4E4554204672616D65776F726B3112" +
            "301006035504030C096C6F63616C686F7374301E170D31363033303230323337" +
            "35345A170D3137303330323032333735345A307B310B30090603550406130255" +
            "533113301106035504080C0A57617368696E67746F6E3110300E06035504070C" +
            "075265646D6F6E6431183016060355040A0C0F4D6963726F736F667420436F72" +
            "702E31173015060355040B0C0E2E4E4554204672616D65776F726B3112301006" +
            "035504030C096C6F63616C686F73743081A0300D06092A864886F70D01010105" +
            "0003818E0030818A02818200BCACB1A5349D7B35A580AC3B3998EB15EBF900EC" +
            "B329BF1F75717A00B2199C8A18D791B592B7EC52BD5AF2DB0D3B635F0595753D" +
            "FF7BA7C9872DBF7E3226DEF44A07CA568D1017992C2B41BFE5EC3570824CF1F4" +
            "B15919FED513FDA56204AF2034A2D08FF04C2CCA49D168FA03FA2FA32FCCD348" +
            "4C15F0A2E5467C76FC760B55090203010001A350304E301D0603551D0E041604" +
            "141063CAB14FB14C47DC211C0E0285F3EE5946BF2D301F0603551D2304183016" +
            "80141063CAB14FB14C47DC211C0E0285F3EE5946BF2D300C0603551D13040530" +
            "030101FF300D06092A864886F70D01010B050003818200435774FB66802AB3CE" +
            "2F1392C079483B48CC8913E0BF3B7AD88351E4C15B55CAD3061AA5875900C56B" +
            "2E7E84BB49CA2A0C1895BD60149C6A0AE983E48370E2144052943B066BD85F70" +
            "543CF6F2F255C028AE1DC8FB898AD3DCA97BF1D607370287077A4C147268C911" +
            "8CF9CAD318D2830D3468727E0A3247B3FEB8D87A7DE4F1E2318201D4308201D0" +
            "02010380141063CAB14FB14C47DC211C0E0285F3EE5946BF2D300B0609608648" +
            "016503040201A081E4301806092A864886F70D010903310B06092A864886F70D" +
            "010701301C06092A864886F70D010905310F170D313731303236303130363235" +
            "5A302F06092A864886F70D0109043122042007849DC26FCBB2F3BD5F57BDF214" +
            "BAE374575F1BD4E6816482324799417CB379307906092A864886F70D01090F31" +
            "6C306A300B060960864801650304012A300B0609608648016503040116300B06" +
            "09608648016503040102300A06082A864886F70D0307300E06082A864886F70D" +
            "030202020080300D06082A864886F70D0302020140300706052B0E030207300D" +
            "06082A864886F70D0302020128303D06092A864886F70D01010A3030A00D300B" +
            "0609608648016503040201A11A301806092A864886F70D010108300B06096086" +
            "48016503040201A20302015F048181B93E81D141B3C9F159AB0021910635DC72" +
            "E8E860BE43C28E5D53243D6DC247B7D4F18C20195E80DEDCC75B29C43CE5047A" +
            "D775B65BFC93589BD748B950C68BADDF1A4673130302BBDA8667D5DDE5EA91EC" +
            "CB13A9B4C04F1C4842FEB1697B7669C7692DD3BDAE13B5AA8EE3EB5679F3729D" +
            "1DC4F2EB9DC89B7E8773F2F8C6108C05").HexToByteArray();

        public static byte[] RsaPkcs1OneSignerIssuerAndSerialNumber = (
            "3082033706092A864886F70D010702A082032830820324020101310B30090605" +
            "2B0E03021A0500302406092A864886F70D010701A01704154D6963726F736F66" +
            "7420436F72706F726174696F6EA08202103082020C30820179A0030201020210" +
            "5D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D0500301E311C30" +
            "1A060355040313135253414B65795472616E736665724361706931301E170D31" +
            "35303431353037303030305A170D3235303431353037303030305A301E311C30" +
            "1A060355040313135253414B65795472616E73666572436170693130819F300D" +
            "06092A864886F70D010101050003818D0030818902818100AA272700586C0CC4" +
            "1B05C65C7D846F5A2BC27B03E301C37D9BFF6D75B6EB6671BA9596C5C63BA2B1" +
            "AF5C318D9CA39E7400D10C238AC72630579211B86570D1A1D44EC86AA8F6C9D2" +
            "B4E283EA3535923F398A312A23EAEACD8D34FAACA965CD910B37DA4093EF76C1" +
            "3B337C1AFAB7D1D07E317B41A336BAA4111299F99424408D0203010001A35330" +
            "51304F0603551D0104483046801015432DB116B35D07E4BA89EDB2469D7AA120" +
            "301E311C301A060355040313135253414B65795472616E736665724361706931" +
            "82105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D0500038181" +
            "0081E5535D8ECEEF265ACBC82F6C5F8BC9D84319265F3CCF23369FA533C8DC19" +
            "38952C5931662D9ECD8B1E7B81749E48468167E2FCE3D019FA70D54646975B6D" +
            "C2A3BA72D5A5274C1866DA6D7A5DF47938E034A075D11957D653B5C78E5291E4" +
            "401045576F6D4EDA81BEF3C369AF56121E49A083C8D1ADB09F291822E99A4296" +
            "463181D73081D40201013032301E311C301A060355040313135253414B657954" +
            "72616E73666572436170693102105D2FFFF863BABC9B4D3C80AB178A4CCA3009" +
            "06052B0E03021A0500300D06092A864886F70D01010105000481805A1717621D" +
            "450130B3463662160EEC06F7AE77E017DD95F294E97A0BDD433FE6B2CCB34FAA" +
            "C33AEA50BFD7D9E78DC7174836284619F744278AE77B8495091E096EEF682D9C" +
            "A95F6E81C7DDCEDDA6A12316B453C894B5000701EB09DF57A53B733A4E80DA27" +
            "FA710870BD88C86E2FDB9DCA14D18BEB2F0C87E9632ABF02BE2FE3").HexToByteArray();

        public static byte[] CounterSignedRsaPkcs1OneSigner = (
            "3082044906092A864886F70D010702A082043A30820436020101310B30090605" +
            "2B0E03021A0500302406092A864886F70D010701A01704154D6963726F736F66" +
            "7420436F72706F726174696F6EA08202103082020C30820179A0030201020210" +
            "5D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D0500301E311C30" +
            "1A060355040313135253414B65795472616E736665724361706931301E170D31" +
            "35303431353037303030305A170D3235303431353037303030305A301E311C30" +
            "1A060355040313135253414B65795472616E73666572436170693130819F300D" +
            "06092A864886F70D010101050003818D0030818902818100AA272700586C0CC4" +
            "1B05C65C7D846F5A2BC27B03E301C37D9BFF6D75B6EB6671BA9596C5C63BA2B1" +
            "AF5C318D9CA39E7400D10C238AC72630579211B86570D1A1D44EC86AA8F6C9D2" +
            "B4E283EA3535923F398A312A23EAEACD8D34FAACA965CD910B37DA4093EF76C1" +
            "3B337C1AFAB7D1D07E317B41A336BAA4111299F99424408D0203010001A35330" +
            "51304F0603551D0104483046801015432DB116B35D07E4BA89EDB2469D7AA120" +
            "301E311C301A060355040313135253414B65795472616E736665724361706931" +
            "82105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D0500038181" +
            "0081E5535D8ECEEF265ACBC82F6C5F8BC9D84319265F3CCF23369FA533C8DC19" +
            "38952C5931662D9ECD8B1E7B81749E48468167E2FCE3D019FA70D54646975B6D" +
            "C2A3BA72D5A5274C1866DA6D7A5DF47938E034A075D11957D653B5C78E5291E4" +
            "401045576F6D4EDA81BEF3C369AF56121E49A083C8D1ADB09F291822E99A4296" +
            "46318201E8308201E40201013032301E311C301A060355040313135253414B65" +
            "795472616E73666572436170693102105D2FFFF863BABC9B4D3C80AB178A4CCA" +
            "300906052B0E03021A0500300D06092A864886F70D01010105000481805A1717" +
            "621D450130B3463662160EEC06F7AE77E017DD95F294E97A0BDD433FE6B2CCB3" +
            "4FAAC33AEA50BFD7D9E78DC7174836284619F744278AE77B8495091E096EEF68" +
            "2D9CA95F6E81C7DDCEDDA6A12316B453C894B5000701EB09DF57A53B733A4E80" +
            "DA27FA710870BD88C86E2FDB9DCA14D18BEB2F0C87E9632ABF02BE2FE3A18201" +
            "0C3082010806092A864886F70D0109063181FA3081F702010380146B4A6B92FD" +
            "ED07EE0119F3674A96D1A70D2A588D300906052B0E03021A0500A03F30180609" +
            "2A864886F70D010903310B06092A864886F70D010701302306092A864886F70D" +
            "010904311604148C054D6DF2B08E69A86D8DB23C1A509123F9DBA4300D06092A" +
            "864886F70D0101010500048180962518DEF789B0886C7E6295754ECDBDC4CB9D" +
            "153ECE5EBBE7A82142B92C30DDBBDFC22B5B954F5D844CBAEDCA9C4A068B2483" +
            "0E2A96141A5D0320B69EA5DFCFEA441E162D04506F8FFA79D7312524F111A9B9" +
            "B0184007139F94E46C816E0E33F010AEB949F5D884DC8987765002F7A643F34B" +
            "7654E3B2FD5FB34A420279B1EA").HexToByteArray();

        public static byte[] NoSignatureSignedWithAttributesAndCounterSignature = (
            "3082042406092A864886F70D010702A082041530820411020101310B30090605" +
            "2B0E03021A0500302406092A864886F70D010701A01704154D6963726F736F66" +
            "7420436F72706F726174696F6EA08202103082020C30820179A0030201020210" +
            "5D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D0500301E311C30" +
            "1A060355040313135253414B65795472616E736665724361706931301E170D31" +
            "35303431353037303030305A170D3235303431353037303030305A301E311C30" +
            "1A060355040313135253414B65795472616E73666572436170693130819F300D" +
            "06092A864886F70D010101050003818D0030818902818100AA272700586C0CC4" +
            "1B05C65C7D846F5A2BC27B03E301C37D9BFF6D75B6EB6671BA9596C5C63BA2B1" +
            "AF5C318D9CA39E7400D10C238AC72630579211B86570D1A1D44EC86AA8F6C9D2" +
            "B4E283EA3535923F398A312A23EAEACD8D34FAACA965CD910B37DA4093EF76C1" +
            "3B337C1AFAB7D1D07E317B41A336BAA4111299F99424408D0203010001A35330" +
            "51304F0603551D0104483046801015432DB116B35D07E4BA89EDB2469D7AA120" +
            "301E311C301A060355040313135253414B65795472616E736665724361706931" +
            "82105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D0500038181" +
            "0081E5535D8ECEEF265ACBC82F6C5F8BC9D84319265F3CCF23369FA533C8DC19" +
            "38952C5931662D9ECD8B1E7B81749E48468167E2FCE3D019FA70D54646975B6D" +
            "C2A3BA72D5A5274C1866DA6D7A5DF47938E034A075D11957D653B5C78E5291E4" +
            "401045576F6D4EDA81BEF3C369AF56121E49A083C8D1ADB09F291822E99A4296" +
            "46318201C3308201BF020101301C3017311530130603550403130C44756D6D79" +
            "205369676E6572020100300906052B0E03021A0500A05D301806092A864886F7" +
            "0D010903310B06092A864886F70D010701301C06092A864886F70D010905310F" +
            "170D3137313130313137313731375A302306092A864886F70D01090431160414" +
            "A5F085E7F326F3D6CA3BFD6280A3DE8EBC2EA60E300C06082B06010505070602" +
            "050004148B70D20D0477A35CD84AB962C10DC52FBA6FAD6BA182010C30820108" +
            "06092A864886F70D0109063181FA3081F702010380146B4A6B92FDED07EE0119" +
            "F3674A96D1A70D2A588D300906052B0E03021A0500A03F301806092A864886F7" +
            "0D010903310B06092A864886F70D010701302306092A864886F70D0109043116" +
            "0414833378066BDCCBA7047EF6919843D181A57D6479300D06092A864886F70D" +
            "01010105000481802155D226DD744166E582D040E60535210195050EA00F2C17" +
            "9897198521DABD0E6B27750FD8BA5F9AAF58B4863B6226456F38553A22453CAF" +
            "0A0F106766C7AB6F3D6AFD106753DC50F8A6E4F9E5508426D236C2DBB4BCB816" +
            "2FA42E995CBA16A340FD7C793569DF1B71368E68253299BC74E38312B40B8F52" +
            "EAEDE10DF414A522").HexToByteArray();

        public static byte[] NoSignatureWithNoAttributes = (
            "30819B06092A864886F70D010702A0818D30818A020101310B300906052B0E03" +
            "021A0500302406092A864886F70D010701A01704154D6963726F736F66742043" +
            "6F72706F726174696F6E31523050020101301C3017311530130603550403130C" +
            "44756D6D79205369676E6572020100300906052B0E03021A0500300C06082B06" +
            "01050507060205000414A5F085E7F326F3D6CA3BFD6280A3DE8EBC2EA60E").HexToByteArray();

        public static byte[] RsaCapiTransfer1_NoEmbeddedCert = (
            "3082016606092A864886F70D010702A082015730820153020103310B30090605" +
            "2B0E03021A0500302406092A864886F70D010701A01704154D6963726F736F66" +
            "7420436F72706F726174696F6E318201193082011502010380146B4A6B92FDED" +
            "07EE0119F3674A96D1A70D2A588D300906052B0E03021A0500A05D301806092A" +
            "864886F70D010903310B06092A864886F70D010701301C06092A864886F70D01" +
            "0905310F170D3137313130323135333430345A302306092A864886F70D010904" +
            "31160414A5F085E7F326F3D6CA3BFD6280A3DE8EBC2EA60E300D06092A864886" +
            "F70D01010105000481800EDE3870B8A80B45A21BAEC4681D059B46502E1B1AA6" +
            "B8920CF50D4D837646A55559B4C05849126C655D95FF3C6C1B420E07DC42629F" +
            "294EE69822FEA56F32D41B824CBB6BF809B7583C27E77B7AC58DFC925B1C60EA" +
            "4A67AA84D73FC9E9191D33B36645F17FD6748A2D8B12C6C384C3C734D2727338" +
            "6211E4518FE2B4ED0147").HexToByteArray();

        public static byte[] OneRsaSignerTwoRsaCounterSigners = (
            "3082075106092A864886F70D010702A08207423082073E020101310B30090605" +
            "2B0E03021A0500302406092A864886F70D010701A01704154D6963726F736F66" +
            "7420436F72706F726174696F6EA08203F9308201E530820152A0030201020210" +
            "D5B5BC1C458A558845BFF51CB4DFF31C300906052B0E03021D05003011310F30" +
            "0D060355040313064D794E616D65301E170D3130303430313038303030305A17" +
            "0D3131303430313038303030305A3011310F300D060355040313064D794E616D" +
            "6530819F300D06092A864886F70D010101050003818D0030818902818100B11E" +
            "30EA87424A371E30227E933CE6BE0E65FF1C189D0D888EC8FF13AA7B42B68056" +
            "128322B21F2B6976609B62B6BC4CF2E55FF5AE64E9B68C78A3C2DACC916A1BC7" +
            "322DD353B32898675CFB5B298B176D978B1F12313E3D865BC53465A11CCA1068" +
            "70A4B5D50A2C410938240E92B64902BAEA23EB093D9599E9E372E48336730203" +
            "010001A346304430420603551D01043B3039801024859EBF125E76AF3F0D7979" +
            "B4AC7A96A1133011310F300D060355040313064D794E616D658210D5B5BC1C45" +
            "8A558845BFF51CB4DFF31C300906052B0E03021D0500038181009BF6E2CF830E" +
            "D485B86D6B9E8DFFDCD65EFC7EC145CB9348923710666791FCFA3AB59D689FFD" +
            "7234B7872611C5C23E5E0714531ABADB5DE492D2C736E1C929E648A65CC9EB63" +
            "CD84E57B5909DD5DDF5DBBBA4A6498B9CA225B6E368B94913BFC24DE6B2BD9A2" +
            "6B192B957304B89531E902FFC91B54B237BB228BE8AFCDA264763082020C3082" +
            "0179A00302010202105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03" +
            "021D0500301E311C301A060355040313135253414B65795472616E7366657243" +
            "61706931301E170D3135303431353037303030305A170D323530343135303730" +
            "3030305A301E311C301A060355040313135253414B65795472616E7366657243" +
            "6170693130819F300D06092A864886F70D010101050003818D00308189028181" +
            "00AA272700586C0CC41B05C65C7D846F5A2BC27B03E301C37D9BFF6D75B6EB66" +
            "71BA9596C5C63BA2B1AF5C318D9CA39E7400D10C238AC72630579211B86570D1" +
            "A1D44EC86AA8F6C9D2B4E283EA3535923F398A312A23EAEACD8D34FAACA965CD" +
            "910B37DA4093EF76C13B337C1AFAB7D1D07E317B41A336BAA4111299F9942440" +
            "8D0203010001A3533051304F0603551D0104483046801015432DB116B35D07E4" +
            "BA89EDB2469D7AA120301E311C301A060355040313135253414B65795472616E" +
            "73666572436170693182105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B" +
            "0E03021D05000381810081E5535D8ECEEF265ACBC82F6C5F8BC9D84319265F3C" +
            "CF23369FA533C8DC1938952C5931662D9ECD8B1E7B81749E48468167E2FCE3D0" +
            "19FA70D54646975B6DC2A3BA72D5A5274C1866DA6D7A5DF47938E034A075D119" +
            "57D653B5C78E5291E4401045576F6D4EDA81BEF3C369AF56121E49A083C8D1AD" +
            "B09F291822E99A42964631820307308203030201013032301E311C301A060355" +
            "040313135253414B65795472616E73666572436170693102105D2FFFF863BABC" +
            "9B4D3C80AB178A4CCA300906052B0E03021A0500300D06092A864886F70D0101" +
            "0105000481805A1717621D450130B3463662160EEC06F7AE77E017DD95F294E9" +
            "7A0BDD433FE6B2CCB34FAAC33AEA50BFD7D9E78DC7174836284619F744278AE7" +
            "7B8495091E096EEF682D9CA95F6E81C7DDCEDDA6A12316B453C894B5000701EB" +
            "09DF57A53B733A4E80DA27FA710870BD88C86E2FDB9DCA14D18BEB2F0C87E963" +
            "2ABF02BE2FE3A182022B3082010806092A864886F70D0109063181FA3081F702" +
            "010380146B4A6B92FDED07EE0119F3674A96D1A70D2A588D300906052B0E0302" +
            "1A0500A03F301806092A864886F70D010903310B06092A864886F70D01070130" +
            "2306092A864886F70D010904311604148C054D6DF2B08E69A86D8DB23C1A5091" +
            "23F9DBA4300D06092A864886F70D0101010500048180962518DEF789B0886C7E" +
            "6295754ECDBDC4CB9D153ECE5EBBE7A82142B92C30DDBBDFC22B5B954F5D844C" +
            "BAEDCA9C4A068B24830E2A96141A5D0320B69EA5DFCFEA441E162D04506F8FFA" +
            "79D7312524F111A9B9B0184007139F94E46C816E0E33F010AEB949F5D884DC89" +
            "87765002F7A643F34B7654E3B2FD5FB34A420279B1EA3082011B06092A864886" +
            "F70D0109063182010C3082010802010130253011310F300D060355040313064D" +
            "794E616D650210D5B5BC1C458A558845BFF51CB4DFF31C300906052B0E03021A" +
            "0500A03F301806092A864886F70D010903310B06092A864886F70D0107013023" +
            "06092A864886F70D010904311604148C054D6DF2B08E69A86D8DB23C1A509123" +
            "F9DBA4300D06092A864886F70D01010105000481801AA282DBED4D862D7CEA30" +
            "F803E790BDB0C97EE852778CEEDDCD94BB9304A1552E60A8D36052AC8C2D2875" +
            "5F3B2F473824100AB3A6ABD4C15ABD77E0FFE13D0DF253BCD99C718FA673B6CB" +
            "0CBBC68CE5A4AC671298C0A07C7223522E0E7FFF15CEDBAB55AAA99588517674" +
            "671691065EB083FB729D1E9C04B2BF99A9953DAA5E").HexToByteArray();

        public static readonly byte[] RsaPkcs1CounterSignedWithNoSignature = (
            "308203E106092A864886F70D010702A08203D2308203CE020101310B30090605" +
            "2B0E03021A0500302406092A864886F70D010701A01704154D6963726F736F66" +
            "7420436F72706F726174696F6EA08202103082020C30820179A0030201020210" +
            "5D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D0500301E311C30" +
            "1A060355040313135253414B65795472616E736665724361706931301E170D31" +
            "35303431353037303030305A170D3235303431353037303030305A301E311C30" +
            "1A060355040313135253414B65795472616E73666572436170693130819F300D" +
            "06092A864886F70D010101050003818D0030818902818100AA272700586C0CC4" +
            "1B05C65C7D846F5A2BC27B03E301C37D9BFF6D75B6EB6671BA9596C5C63BA2B1" +
            "AF5C318D9CA39E7400D10C238AC72630579211B86570D1A1D44EC86AA8F6C9D2" +
            "B4E283EA3535923F398A312A23EAEACD8D34FAACA965CD910B37DA4093EF76C1" +
            "3B337C1AFAB7D1D07E317B41A336BAA4111299F99424408D0203010001A35330" +
            "51304F0603551D0104483046801015432DB116B35D07E4BA89EDB2469D7AA120" +
            "301E311C301A060355040313135253414B65795472616E736665724361706931" +
            "82105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D0500038181" +
            "0081E5535D8ECEEF265ACBC82F6C5F8BC9D84319265F3CCF23369FA533C8DC19" +
            "38952C5931662D9ECD8B1E7B81749E48468167E2FCE3D019FA70D54646975B6D" +
            "C2A3BA72D5A5274C1866DA6D7A5DF47938E034A075D11957D653B5C78E5291E4" +
            "401045576F6D4EDA81BEF3C369AF56121E49A083C8D1ADB09F291822E99A4296" +
            "46318201803082017C0201013032301E311C301A060355040313135253414B65" +
            "795472616E73666572436170693102105D2FFFF863BABC9B4D3C80AB178A4CCA" +
            "300906052B0E03021A0500300D06092A864886F70D01010105000481805A1717" +
            "621D450130B3463662160EEC06F7AE77E017DD95F294E97A0BDD433FE6B2CCB3" +
            "4FAAC33AEA50BFD7D9E78DC7174836284619F744278AE77B8495091E096EEF68" +
            "2D9CA95F6E81C7DDCEDDA6A12316B453C894B5000701EB09DF57A53B733A4E80" +
            "DA27FA710870BD88C86E2FDB9DCA14D18BEB2F0C87E9632ABF02BE2FE3A181A5" +
            "3081A206092A864886F70D010906318194308191020101301C30173115301306" +
            "03550403130C44756D6D79205369676E6572020100300906052B0E03021A0500" +
            "A03F301806092A864886F70D010903310B06092A864886F70D01070130230609" +
            "2A864886F70D010904311604148C054D6DF2B08E69A86D8DB23C1A509123F9DB" +
            "A4300C06082B060105050706020500041466124B3D99FE06A19BBD3C83C593AB" +
            "55D875E28B").HexToByteArray();

        public static readonly byte[] UnsortedSignerInfos = (
            "30820B1E06092A864886F70D010702A0820B0F30820B0B020103310B30090605" +
            "2B0E03021A0500301006092A864886F70D010701A003040107A0820540308202" +
            "0C30820179A00302010202105D2FFFF863BABC9B4D3C80AB178A4CCA30090605" +
            "2B0E03021D0500301E311C301A060355040313135253414B65795472616E7366" +
            "65724361706931301E170D3135303431353037303030305A170D323530343135" +
            "3037303030305A301E311C301A060355040313135253414B65795472616E7366" +
            "6572436170693130819F300D06092A864886F70D010101050003818D00308189" +
            "02818100AA272700586C0CC41B05C65C7D846F5A2BC27B03E301C37D9BFF6D75" +
            "B6EB6671BA9596C5C63BA2B1AF5C318D9CA39E7400D10C238AC72630579211B8" +
            "6570D1A1D44EC86AA8F6C9D2B4E283EA3535923F398A312A23EAEACD8D34FAAC" +
            "A965CD910B37DA4093EF76C13B337C1AFAB7D1D07E317B41A336BAA4111299F9" +
            "9424408D0203010001A3533051304F0603551D0104483046801015432DB116B3" +
            "5D07E4BA89EDB2469D7AA120301E311C301A060355040313135253414B657954" +
            "72616E73666572436170693182105D2FFFF863BABC9B4D3C80AB178A4CCA3009" +
            "06052B0E03021D05000381810081E5535D8ECEEF265ACBC82F6C5F8BC9D84319" +
            "265F3CCF23369FA533C8DC1938952C5931662D9ECD8B1E7B81749E48468167E2" +
            "FCE3D019FA70D54646975B6DC2A3BA72D5A5274C1866DA6D7A5DF47938E034A0" +
            "75D11957D653B5C78E5291E4401045576F6D4EDA81BEF3C369AF56121E49A083" +
            "C8D1ADB09F291822E99A4296463082032C30820214A003020102020900E0D8AB" +
            "6819D7306E300D06092A864886F70D01010B0500303831363034060355040313" +
            "2D54776F2074686F7573616E6420666F7274792065696768742062697473206F" +
            "662052534120676F6F646E657373301E170D3137313130333233353131355A17" +
            "0D3138313130333233353131355A3038313630340603550403132D54776F2074" +
            "686F7573616E6420666F7274792065696768742062697473206F662052534120" +
            "676F6F646E65737330820122300D06092A864886F70D01010105000382010F00" +
            "3082010A028201010096C114A5898D09133EF859F89C1D848BA8CB5258793E05" +
            "B92D499C55EEFACE274BBBC26803FB813B9C11C6898153CC1745DED2C4D2672F" +
            "807F0B2D957BC4B65EBC9DDE26E2EA7B2A6FE9A7C4D8BD1EF6032B8F0BB6AA33" +
            "C8B57248B3D5E3901D8A38A283D7E25FF8E6F522381EE5484234CFF7B30C1746" +
            "35418FA89E14C468AD89DCFCBBB535E5AF53510F9EA7F9DA8C1B53375B6DAB95" +
            "A291439A5648726EE1012E41388E100691642CF6917F5569D8351F2782F435A5" +
            "79014E8448EEA0C4AECAFF2F476799D88457E2C8BCB56E5E128782B4FE26AFF0" +
            "720D91D52CCAFE344255808F5271D09F784F787E8323182080915BE0AE15A71D" +
            "66476D0F264DD084F30203010001A3393037301D0603551D0E04160414745B5F" +
            "12EF962E84B897E246D399A2BADEA9C5AC30090603551D1304023000300B0603" +
            "551D0F040403020780300D06092A864886F70D01010B0500038201010087A15D" +
            "F37FBD6E9DED7A8FFF25E60B731F635469BA01DD14BC03B2A24D99EFD8B894E9" +
            "493D63EC88C496CB04B33DF25222544F23D43F4023612C4D97B719C1F9431E4D" +
            "B7A580CDF66A3E5F0DAF89A267DD187ABFFB08361B1F79232376AA5FC5AD384C" +
            "C2F98FE36C1CEA0B943E1E3961190648889C8ABE8397A5A338843CBFB1D8B212" +
            "BE46685ACE7B80475CC7C97FC0377936ABD5F664E9C09C463897726650711A11" +
            "10FA9866BC1C278D95E5636AB96FAE95CCD67FD572A8C727E2C03E7B24245731" +
            "8BEC1BE52CA5BD9454A0A41140AE96ED1C56D220D1FD5DD3B1B4FB2AA0E04FC9" +
            "4F7E3C7D476F298962245563953AD7225EDCEAC8B8509E49292E62D8BF318205" +
            "A1308202FB0201038014745B5F12EF962E84B897E246D399A2BADEA9C5AC3009" +
            "06052B0E03021A0500300D06092A864886F70D0101010500048201005E03C5E2" +
            "E736792EFB1C8632C3A864AA6F0E930717FE02C755C0F94DC671244A371926F6" +
            "09878DC8CBFCBA6F83A841B24F48952DA5344F2210BFE9B744E3367B1F8399C8" +
            "96F675923A57E084EBD7DC76A24A1530CD513F0DF6A7703246BF335CC3D09776" +
            "442942150F1C31B9B212AF48850B44B95EB5BD64105F09723EF6AD4711FD81CD" +
            "1FC0418E68EA4428CED9E184126761BF2B25756B6D9BC1A0530E56D38F2A0B78" +
            "3F21D6A5C0703C38F29A2B701B13CAFFCA1DC21C39059E4388E54AEA2519C4E8" +
            "83C7A6BD78200DCB931CA6AB3D18DBBF46A5444C89B6DFE2F48F32C44BA9C030" +
            "F399AC677AA323203137D33CEBFBF1BBF9A506309953B23C4100CA7CA18201C0" +
            "308201BC06092A864886F70D010906318201AD308201A9020101304530383136" +
            "30340603550403132D54776F2074686F7573616E6420666F7274792065696768" +
            "742062697473206F662052534120676F6F646E657373020900E0D8AB6819D730" +
            "6E300906052B0E03021A0500A03F301806092A864886F70D010903310B06092A" +
            "864886F70D010701302306092A864886F70D0109043116041481BF56A6550A60" +
            "A649B0D97971C49897635953D0300D06092A864886F70D010101050004820100" +
            "6E41B7585FEB419005362FEAAAAFB2059E98F8905221A7564F7B0B5510CB221D" +
            "F3DD914A4CD441EAC1C6746A6EC4FC8399C12A61C6B0F50DDA090F564F3D65B2" +
            "6D4BDBC1CE3D39CF47CF33B0D269D15A9FAF2169C60887C3E2CC9828B5E16D45" +
            "DC27A94BAF8D6650EE63D2DBB7DA319B3F61DD18E28AF6FE6DF2CC15C2910BD6" +
            "0B7E038F2C6E8BAEC35CBBBF9484D4C76ECE041DF534B8713B6537854EFE6D58" +
            "41768CCBB9A3B729FDDAE07780CB143A3EE5972DCDDF60A38C65CD3FFF35D1B6" +
            "B76227C1B53831773DA441603F4FB5764D33AADE102F9B85D2CDAEC0E3D6C6E8" +
            "C24C434BFAA3E12E02202142784ED0EB2D9CDCC276D21474747DCD3E4F4D54FC" +
            "3081D40201013032301E311C301A060355040313135253414B65795472616E73" +
            "666572436170693102105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E" +
            "03021A0500300D06092A864886F70D01010105000481805EB33C6A9ED5B62240" +
            "90C431E79F51D70B4F2A7D31ED4ED8C3465F6E01281C3FFA44116238B2D168D8" +
            "9154136DDB8B4EB31EA685FB719B7384510F5EF077A10DE6A5CA86F4F6D28B58" +
            "79AFD6CFF0BDA005C2D7CFF53620D28988CBAA44F18AA2D50229FA930B0A7262" +
            "D780DFDEC0334A97DF872F1D95087DC11A881568AF5B88308201C70201013045" +
            "3038313630340603550403132D54776F2074686F7573616E6420666F72747920" +
            "65696768742062697473206F662052534120676F6F646E657373020900E0D8AB" +
            "6819D7306E300906052B0E03021A0500A05D301806092A864886F70D01090331" +
            "0B06092A864886F70D010701301C06092A864886F70D010905310F170D313731" +
            "3130393136303934315A302306092A864886F70D010904311604145D1BE7E9DD" +
            "A1EE8896BE5B7E34A85EE16452A7B4300D06092A864886F70D01010105000482" +
            "01000BB9410F23CFD9C1FCB16179612DB871224F5B88A8E2C012DCDBB3699780" +
            "A3311FD330FFDD6DF1434C52DADD6E07D81FEF145B806E71AF471223914B98CD" +
            "588CCCDFB50ABE3D991B11D62BD83DE158A9001BAED3549BC49B8C204D25C17B" +
            "D042756B026692959E321ACC1AFE6BF52C9356FD49936116D2B3D1F6569F8A8B" +
            "F0FBB2E403AD5788681F3AD131E57390ACB9B8C2EA0BE717F22EFE577EFB1063" +
            "6AC465469191B7E4B3F03CF8DC6C310A20D2B0891BC27350C7231BC2EAABF129" +
            "83755B4C0EDF8A0EE99A615D4E8B381C67A7CDB1405D98C2A6285FEDCED5A65F" +
            "C45C31CD33E3CEB96223DB45E9156B9BD7C8E442C40ED1BB6866C03548616061" +
            "3DAF").HexToByteArray();

        public static byte[] OneDsa1024 = (
            "3082044206092A864886F70D010702A08204333082042F020103310B30090605" +
            "2B0E03021A0500302406092A864886F70D010701A01704154D6963726F736F66" +
            "7420436F72706F726174696F6EA08203913082038D3082034AA0030201020209" +
            "00AB740A714AA83C92300B060960864801650304030230818D310B3009060355" +
            "040613025553311330110603550408130A57617368696E67746F6E3110300E06" +
            "0355040713075265646D6F6E64311E301C060355040A13154D6963726F736F66" +
            "7420436F72706F726174696F6E3120301E060355040B13172E4E455420467261" +
            "6D65776F726B2028436F7265465829311530130603550403130C313032342D62" +
            "697420445341301E170D3135313132353134343030335A170D31353132323531" +
            "34343030335A30818D310B300906035504061302555331133011060355040813" +
            "0A57617368696E67746F6E3110300E060355040713075265646D6F6E64311E30" +
            "1C060355040A13154D6963726F736F667420436F72706F726174696F6E312030" +
            "1E060355040B13172E4E4554204672616D65776F726B2028436F726546582931" +
            "1530130603550403130C313032342D62697420445341308201B73082012C0607" +
            "2A8648CE3804013082011F02818100AEE3309FC7C9DB750D4C3797D333B3B9B2" +
            "34B462868DB6FFBDED790B7FC8DDD574C2BD6F5E749622507AB2C09DF5EAAD84" +
            "859FC0706A70BB8C9C8BE22B4890EF2325280E3A7F9A3CE341DBABEF6058D063" +
            "EA6783478FF8B3B7A45E0CA3F7BAC9995DCFDDD56DF168E91349130F719A4E71" +
            "7351FAAD1A77EAC043611DC5CC5A7F021500D23428A76743EA3B49C62EF0AA17" +
            "314A85415F0902818100853F830BDAA738465300CFEE02418E6B07965658EAFD" +
            "A7E338A2EB1531C0E0CA5EF1A12D9DDC7B550A5A205D1FF87F69500A4E4AF575" +
            "9F3F6E7F0C48C55396B738164D9E35FB506BD50E090F6A497C70E7E868C61BD4" +
            "477C1D62922B3DBB40B688DE7C175447E2E826901A109FAD624F1481B276BF63" +
            "A665D99C87CEE9FD06330381840002818025B8E7078E149BAC35266762362002" +
            "9F5E4A5D4126E336D56F1189F9FF71EA671B844EBD351514F27B69685DDF716B" +
            "32F102D60EA520D56F544D19B2F08F5D9BDDA3CBA3A73287E21E559E6A075861" +
            "94AFAC4F6E721EDCE49DE0029627626D7BD30EEB337311DB4FF62D7608997B6C" +
            "C32E9C42859820CA7EF399590D5A388C48A330302E302C0603551D1104253023" +
            "87047F00000187100000000000000000000000000000000182096C6F63616C68" +
            "6F7374300B0609608648016503040302033000302D021500B9316CC7E05C9F79" +
            "197E0B41F6FD4E3FCEB72A8A0214075505CCAECB18B7EF4C00F9C069FA3BC780" +
            "14DE31623060020103801428A2CB1D204C2656A79C931EFAE351AB548248D030" +
            "0906052B0E03021A0500300906072A8648CE380403042F302D021476DCB780CE" +
            "D5B308A3630726A85DB97FBC50DFD1021500CDF2649B50500BB7428B9DCA6BEF" +
            "2C7E7EF1B79C").HexToByteArray();

        public static byte[] RsaPkcs1TwoCounterSignaturesInSingleAttribute = (
            "30820BBA06092A864886F70D010702A0820BAB30820BA7020101310D300B0609" +
            "608648016503040201301406092A864886F70D010701A00704050102030405A0" +
            "82081D308201583081FFA003020102021035428F3B3C5107AD49E776D6E74C4D" +
            "C8300A06082A8648CE3D04030230153113301106035504030C0A454344534120" +
            "54657374301E170D3135303530313030333730335A170D313630353031303035" +
            "3730335A30153113301106035504030C0A454344534120546573743059301306" +
            "072A8648CE3D020106082A8648CE3D030107034200047590F69CA114E92927E0" +
            "34C997B7C882A8C992AC00CEFB4EB831901536F291E1B515263BCD20E1EA3249" +
            "6FDAC84E2D8D1B703266A9088F6EAF652549D9BB63D5A331302F300E0603551D" +
            "0F0101FF040403020388301D0603551D0E0416041411218A92C5EB12273B3C5C" +
            "CFB8220CCCFDF387DB300A06082A8648CE3D040302034800304502201AFE595E" +
            "19F1AE4B6A4B231E8851926438C55B5DDE632E6ADF13C1023A65898E022100CB" +
            "DF434FDD197D8B594E8026E44263BADE773C2BEBD060CC4109484A498E7C7E30" +
            "82032C30820214A003020102020900E0D8AB6819D7306E300D06092A864886F7" +
            "0D01010B05003038313630340603550403132D54776F2074686F7573616E6420" +
            "666F7274792065696768742062697473206F662052534120676F6F646E657373" +
            "301E170D3137313130333233353131355A170D3138313130333233353131355A" +
            "3038313630340603550403132D54776F2074686F7573616E6420666F72747920" +
            "65696768742062697473206F662052534120676F6F646E65737330820122300D" +
            "06092A864886F70D01010105000382010F003082010A028201010096C114A589" +
            "8D09133EF859F89C1D848BA8CB5258793E05B92D499C55EEFACE274BBBC26803" +
            "FB813B9C11C6898153CC1745DED2C4D2672F807F0B2D957BC4B65EBC9DDE26E2" +
            "EA7B2A6FE9A7C4D8BD1EF6032B8F0BB6AA33C8B57248B3D5E3901D8A38A283D7" +
            "E25FF8E6F522381EE5484234CFF7B30C174635418FA89E14C468AD89DCFCBBB5" +
            "35E5AF53510F9EA7F9DA8C1B53375B6DAB95A291439A5648726EE1012E41388E" +
            "100691642CF6917F5569D8351F2782F435A579014E8448EEA0C4AECAFF2F4767" +
            "99D88457E2C8BCB56E5E128782B4FE26AFF0720D91D52CCAFE344255808F5271" +
            "D09F784F787E8323182080915BE0AE15A71D66476D0F264DD084F30203010001" +
            "A3393037301D0603551D0E04160414745B5F12EF962E84B897E246D399A2BADE" +
            "A9C5AC30090603551D1304023000300B0603551D0F040403020780300D06092A" +
            "864886F70D01010B0500038201010087A15DF37FBD6E9DED7A8FFF25E60B731F" +
            "635469BA01DD14BC03B2A24D99EFD8B894E9493D63EC88C496CB04B33DF25222" +
            "544F23D43F4023612C4D97B719C1F9431E4DB7A580CDF66A3E5F0DAF89A267DD" +
            "187ABFFB08361B1F79232376AA5FC5AD384CC2F98FE36C1CEA0B943E1E396119" +
            "0648889C8ABE8397A5A338843CBFB1D8B212BE46685ACE7B80475CC7C97FC037" +
            "7936ABD5F664E9C09C463897726650711A1110FA9866BC1C278D95E5636AB96F" +
            "AE95CCD67FD572A8C727E2C03E7B242457318BEC1BE52CA5BD9454A0A41140AE" +
            "96ED1C56D220D1FD5DD3B1B4FB2AA0E04FC94F7E3C7D476F298962245563953A" +
            "D7225EDCEAC8B8509E49292E62D8BF3082038D3082034AA003020102020900AB" +
            "740A714AA83C92300B060960864801650304030230818D310B30090603550406" +
            "13025553311330110603550408130A57617368696E67746F6E3110300E060355" +
            "040713075265646D6F6E64311E301C060355040A13154D6963726F736F667420" +
            "436F72706F726174696F6E3120301E060355040B13172E4E4554204672616D65" +
            "776F726B2028436F7265465829311530130603550403130C313032342D626974" +
            "20445341301E170D3135313132353134343030335A170D313531323235313434" +
            "3030335A30818D310B3009060355040613025553311330110603550408130A57" +
            "617368696E67746F6E3110300E060355040713075265646D6F6E64311E301C06" +
            "0355040A13154D6963726F736F667420436F72706F726174696F6E3120301E06" +
            "0355040B13172E4E4554204672616D65776F726B2028436F7265465829311530" +
            "130603550403130C313032342D62697420445341308201B73082012C06072A86" +
            "48CE3804013082011F02818100AEE3309FC7C9DB750D4C3797D333B3B9B234B4" +
            "62868DB6FFBDED790B7FC8DDD574C2BD6F5E749622507AB2C09DF5EAAD84859F" +
            "C0706A70BB8C9C8BE22B4890EF2325280E3A7F9A3CE341DBABEF6058D063EA67" +
            "83478FF8B3B7A45E0CA3F7BAC9995DCFDDD56DF168E91349130F719A4E717351" +
            "FAAD1A77EAC043611DC5CC5A7F021500D23428A76743EA3B49C62EF0AA17314A" +
            "85415F0902818100853F830BDAA738465300CFEE02418E6B07965658EAFDA7E3" +
            "38A2EB1531C0E0CA5EF1A12D9DDC7B550A5A205D1FF87F69500A4E4AF5759F3F" +
            "6E7F0C48C55396B738164D9E35FB506BD50E090F6A497C70E7E868C61BD4477C" +
            "1D62922B3DBB40B688DE7C175447E2E826901A109FAD624F1481B276BF63A665" +
            "D99C87CEE9FD06330381840002818025B8E7078E149BAC352667623620029F5E" +
            "4A5D4126E336D56F1189F9FF71EA671B844EBD351514F27B69685DDF716B32F1" +
            "02D60EA520D56F544D19B2F08F5D9BDDA3CBA3A73287E21E559E6A07586194AF" +
            "AC4F6E721EDCE49DE0029627626D7BD30EEB337311DB4FF62D7608997B6CC32E" +
            "9C42859820CA7EF399590D5A388C48A330302E302C0603551D11042530238704" +
            "7F00000187100000000000000000000000000000000182096C6F63616C686F73" +
            "74300B0609608648016503040302033000302D021500B9316CC7E05C9F79197E" +
            "0B41F6FD4E3FCEB72A8A0214075505CCAECB18B7EF4C00F9C069FA3BC78014DE" +
            "3182035A3082035602010130453038313630340603550403132D54776F207468" +
            "6F7573616E6420666F7274792065696768742062697473206F66205253412067" +
            "6F6F646E657373020900E0D8AB6819D7306E300B060960864801650304020130" +
            "0B06092A864886F70D01010104820100457E2996B3A1AE5C7DC2F4EF4D9010F4" +
            "8B62B72DFB43F2EDC503FD32408A1058EE7BBCF4750CB4B4242B11A599C40792" +
            "70D32D15A57FF791FF59836A027E634B9B97E1764173597A9A6155D5ED5365F6" +
            "5DF14FDD15928ABD63E1409DBF2D1A713D20D80E09EE76BC63775F3FA8638A26" +
            "ED3816FF87C7CDC8A9299485055BFC38AE158BB6577812AA98436FB54844544A" +
            "C92CD449690B8107447044580FAE590D8A7326A8D139886C8A4AC8CEEACB0458" +
            "1666D8447D267F1A9E9CAB20F155E05D5EC055AC863C047B5E1E3A98528EA766" +
            "7C19B33AD98B2D33ABBD7E607C1DA18BCDB87C626554C277E069CE9EC489BC87" +
            "2E7DEAED4C642DE5AB10BD2D558EAFB3A18201EA308201E606092A864886F70D" +
            "010906318201D73082010D02010130819B30818D310B30090603550406130255" +
            "53311330110603550408130A57617368696E67746F6E3110300E060355040713" +
            "075265646D6F6E64311E301C060355040A13154D6963726F736F667420436F72" +
            "706F726174696F6E3120301E060355040B13172E4E4554204672616D65776F72" +
            "6B2028436F7265465829311530130603550403130C313032342D626974204453" +
            "41020900AB740A714AA83C92300706052B0E03021AA025302306092A864886F7" +
            "0D0109043116041409200943E2EDD3DD3B186C5839BDC9B1051903FF30090607" +
            "2A8648CE380403042F302D0215009FDBE95176B1EC0697155ADDF335E5126A9F" +
            "59D60214736F650C74E73BEA577151BCFD226FEDC06832E53081C30201013029" +
            "30153113301106035504030C0A45434453412054657374021035428F3B3C5107" +
            "AD49E776D6E74C4DC8300B0609608648016503040201A031302F06092A864886" +
            "F70D01090431220420DF5D49DB775A8F94CAB3129038B200EDE9FCD2AE8F039D" +
            "B1AB96D9B827D299D2300A06082A8648CE3D0403020447304502202327A60E1A" +
            "5A798CD29B72C7C7991F968D29DB15C4865BEE83A7E2FD73326CA4022100899F" +
            "000179F77BFE296783548EAE56BA7F53C0DB0563A27A36A149BAEC9C23AC").HexToByteArray();

        internal static readonly byte[] SignedCmsOverEnvelopedCms_IssuerSerial_NetFx = (
            "3082047C06092A864886F70D010702A082046D30820469020101310B30090605" +
            "2B0E03021A05003082012406092A864886F70D010703A0820115308201110609" +
            "2A864886F70D010703A08201023081FF0201003181CC3081C90201003032301E" +
            "311C301A060355040313135253414B65795472616E7366657243617069310210" +
            "5D2FFFF863BABC9B4D3C80AB178A4CCA300D06092A864886F70D010101050004" +
            "81800BB53BF3BD028A6B54703899B241CB358CACBF9018A4497A733C27EA223E" +
            "05BD31099EB80AE04ADBB23A5E397C181A14476668402EFE3BCA08BCA615C743" +
            "41FA06D56671AA940BF09B6B7B4C6905AD2927DE94960ED03DF141360589979F" +
            "9944DB48B91AA1B139EB652D6A1BAC48DF33AF14006CD9DB4C09E7DA270733D0" +
            "DF90302B06092A864886F70D010701301406082A864886F70D03070408E4972B" +
            "4188B1B4FE80084CBF0A9D37B094EBA08202103082020C30820179A003020102" +
            "02105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D0500301E31" +
            "1C301A060355040313135253414B65795472616E736665724361706931301E17" +
            "0D3135303431353037303030305A170D3235303431353037303030305A301E31" +
            "1C301A060355040313135253414B65795472616E73666572436170693130819F" +
            "300D06092A864886F70D010101050003818D0030818902818100AA272700586C" +
            "0CC41B05C65C7D846F5A2BC27B03E301C37D9BFF6D75B6EB6671BA9596C5C63B" +
            "A2B1AF5C318D9CA39E7400D10C238AC72630579211B86570D1A1D44EC86AA8F6" +
            "C9D2B4E283EA3535923F398A312A23EAEACD8D34FAACA965CD910B37DA4093EF" +
            "76C13B337C1AFAB7D1D07E317B41A336BAA4111299F99424408D0203010001A3" +
            "533051304F0603551D0104483046801015432DB116B35D07E4BA89EDB2469D7A" +
            "A120301E311C301A060355040313135253414B65795472616E73666572436170" +
            "693182105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D050003" +
            "81810081E5535D8ECEEF265ACBC82F6C5F8BC9D84319265F3CCF23369FA533C8" +
            "DC1938952C5931662D9ECD8B1E7B81749E48468167E2FCE3D019FA70D5464697" +
            "5B6DC2A3BA72D5A5274C1866DA6D7A5DF47938E034A075D11957D653B5C78E52" +
            "91E4401045576F6D4EDA81BEF3C369AF56121E49A083C8D1ADB09F291822E99A" +
            "42964631820119308201150201013032301E311C301A06035504031313525341" +
            "4B65795472616E73666572436170693102105D2FFFF863BABC9B4D3C80AB178A" +
            "4CCA300906052B0E03021A0500A03F301806092A864886F70D010903310B0609" +
            "2A864886F70D010703302306092A864886F70D01090431160414FE46C861E86B" +
            "719D0F665AFAE48165B56CDFBFD4300D06092A864886F70D0101010500048180" +
            "32CEE36532673C2734C908A48B6E017FD695BE69FAC21028B6627466B72688D8" +
            "60FC65F2F18E5C19FED2301351F247DF90217087C5F88D76CA052287E6A2F47F" +
            "7DA5AC226B4FC202AB0B5B73A24B5C138247F54466621288F2DA941320C4CE89" +
            "A503ED3E6F63112798A841E55344BEE84E1366E4CF3788C9788C5E86D1879029").HexToByteArray();

        internal static readonly byte[] SignedCmsOverEnvelopedCms_SKID_NetFx = (
            "3082046006092A864886F70D010702A08204513082044D020103310B30090605" +
            "2B0E03021A05003082012806092A864886F70D010703A0820119048201153082" +
            "011106092A864886F70D010703A08201023081FF0201003181CC3081C9020100" +
            "3032301E311C301A060355040313135253414B65795472616E73666572436170" +
            "693102105D2FFFF863BABC9B4D3C80AB178A4CCA300D06092A864886F70D0101" +
            "0105000481803ECF128C059F49199D3344979BD0EBAC2A5443D4F27775B8CFAC" +
            "7B1F28AFDDAD86097FF34DFB3ED2D514C325B78074D6D17CA14952EA954E860B" +
            "D5980F2C629C70AE402D3E9E867246E532E345712DFA33C37EF141E2EBFD10F7" +
            "249CFD193B313825CB7B297FB204DA755F02384659F51D97AB31F867C7E973C6" +
            "28B9F6E43018302B06092A864886F70D010701301406082A864886F70D030704" +
            "089FC5129D8AB0CDDE80086D7E35774EFA334AA08202103082020C30820179A0" +
            "0302010202105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D05" +
            "00301E311C301A060355040313135253414B65795472616E7366657243617069" +
            "31301E170D3135303431353037303030305A170D323530343135303730303030" +
            "5A301E311C301A060355040313135253414B65795472616E7366657243617069" +
            "3130819F300D06092A864886F70D010101050003818D0030818902818100AA27" +
            "2700586C0CC41B05C65C7D846F5A2BC27B03E301C37D9BFF6D75B6EB6671BA95" +
            "96C5C63BA2B1AF5C318D9CA39E7400D10C238AC72630579211B86570D1A1D44E" +
            "C86AA8F6C9D2B4E283EA3535923F398A312A23EAEACD8D34FAACA965CD910B37" +
            "DA4093EF76C13B337C1AFAB7D1D07E317B41A336BAA4111299F99424408D0203" +
            "010001A3533051304F0603551D0104483046801015432DB116B35D07E4BA89ED" +
            "B2469D7AA120301E311C301A060355040313135253414B65795472616E736665" +
            "72436170693182105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E0302" +
            "1D05000381810081E5535D8ECEEF265ACBC82F6C5F8BC9D84319265F3CCF2336" +
            "9FA533C8DC1938952C5931662D9ECD8B1E7B81749E48468167E2FCE3D019FA70" +
            "D54646975B6DC2A3BA72D5A5274C1866DA6D7A5DF47938E034A075D11957D653" +
            "B5C78E5291E4401045576F6D4EDA81BEF3C369AF56121E49A083C8D1ADB09F29" +
            "1822E99A4296463181FA3081F702010380146B4A6B92FDED07EE0119F3674A96" +
            "D1A70D2A588D300906052B0E03021A0500A03F301806092A864886F70D010903" +
            "310B06092A864886F70D010703302306092A864886F70D0109043116041435DE" +
            "A4AE3B383A023271BA27D2D50EC021D40800300D06092A864886F70D01010105" +
            "00048180386A2EB06AB0ED0111EB37214480CD782243C66105948AD8EAB3236A" +
            "7ECF135F22B6558F3C601140F6BBDF313F7DB98B3E6277ED5C2407D57323348D" +
            "A97F6A9653C7C219EE1B0E3F85A970FA6CFC00B53E72484F732916E6067E2F0D" +
            "4D31EFF51CECD46F3EF245FEF8729C4E1F16C0A3054054477D6C787FC7C94D79" +
            "A24AC54B").HexToByteArray();

        internal static readonly byte[] SignedCmsOverEnvelopedCms_IssuerSerial_CoreFx = (
            "3082048E06092A864886F70D010702A082047F3082047B020103310D300B0609" +
            "6086480165030402013082012806092A864886F70D010703A082011904820115" +
            "3082011106092A864886F70D010703A08201023081FF0201003181CC3081C902" +
            "01003032301E311C301A060355040313135253414B65795472616E7366657243" +
            "6170693102105D2FFFF863BABC9B4D3C80AB178A4CCA300D06092A864886F70D" +
            "01010105000481801B7806566B26A92076D5C9F5A06FBC9AB1D53BD63D3B7F97" +
            "569B683219C4BA0B285F2F3EF533387EDD7E6BE38DFDD1F33EBA8E5001238BD0" +
            "E75B9A5C5E2504FD78954B372A2E8B183F4CBD2D239CB72D129E112D0476D9A9" +
            "A00AF0EC700776F4719BC4838DBAC7F06C671F67B977ABDF449B42C98D28035A" +
            "194CE2B786E8C8A2302B06092A864886F70D010701301406082A864886F70D03" +
            "070408B4B41A525B6E8F628008767424A015173966A08202103082020C308201" +
            "79A00302010202105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E0302" +
            "1D0500301E311C301A060355040313135253414B65795472616E736665724361" +
            "706931301E170D3135303431353037303030305A170D32353034313530373030" +
            "30305A301E311C301A060355040313135253414B65795472616E736665724361" +
            "70693130819F300D06092A864886F70D010101050003818D0030818902818100" +
            "AA272700586C0CC41B05C65C7D846F5A2BC27B03E301C37D9BFF6D75B6EB6671" +
            "BA9596C5C63BA2B1AF5C318D9CA39E7400D10C238AC72630579211B86570D1A1" +
            "D44EC86AA8F6C9D2B4E283EA3535923F398A312A23EAEACD8D34FAACA965CD91" +
            "0B37DA4093EF76C13B337C1AFAB7D1D07E317B41A336BAA4111299F99424408D" +
            "0203010001A3533051304F0603551D0104483046801015432DB116B35D07E4BA" +
            "89EDB2469D7AA120301E311C301A060355040313135253414B65795472616E73" +
            "666572436170693182105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E" +
            "03021D05000381810081E5535D8ECEEF265ACBC82F6C5F8BC9D84319265F3CCF" +
            "23369FA533C8DC1938952C5931662D9ECD8B1E7B81749E48468167E2FCE3D019" +
            "FA70D54646975B6DC2A3BA72D5A5274C1866DA6D7A5DF47938E034A075D11957" +
            "D653B5C78E5291E4401045576F6D4EDA81BEF3C369AF56121E49A083C8D1ADB0" +
            "9F291822E99A42964631820125308201210201013032301E311C301A06035504" +
            "0313135253414B65795472616E73666572436170693102105D2FFFF863BABC9B" +
            "4D3C80AB178A4CCA300B0609608648016503040201A04B301806092A864886F7" +
            "0D010903310B06092A864886F70D010703302F06092A864886F70D0109043122" +
            "042018BEF3F24109B4BCD5BF3D5372EA7A0D16AF6DF46DE9BE5C2373DF065381" +
            "5E13300B06092A864886F70D01010104818016A02798B3CEC42BE258C85A4BED" +
            "06099339C9E716B8C72A3330923BE4B6A0538A5DCE031CD710589E8281E24074" +
            "F26AB6B86CEACF78449B82FF1512F511B5A97ABA4403029E2BA1D837D3F9D230" +
            "45E0EB3CE59E3AF7E52B814EFCBBCFD7A442327C5C408D166D4302AEFF807ECB" +
            "D107C811DC66EC35FE167408B58FB03B7F84").HexToByteArray();

        internal static readonly byte[] SignedCmsOverEnvelopedCms_SKID_CoreFx = (
            "3082047006092A864886F70D010702A08204613082045D020103310D300B0609" +
            "6086480165030402013082012806092A864886F70D010703A082011904820115" +
            "3082011106092A864886F70D010703A08201023081FF0201003181CC3081C902" +
            "01003032301E311C301A060355040313135253414B65795472616E7366657243" +
            "6170693102105D2FFFF863BABC9B4D3C80AB178A4CCA300D06092A864886F70D" +
            "0101010500048180724D9D5E0D2110B8147589120524B1D1E7019A3F436AD459" +
            "3DF555413423AE28FCBA01548B20FDCA21901ECF6B54331542CECD4326C7E292" +
            "54AA563D7F38C2287C146B648E6779FA3843FB0F11A3726265266DF87BAAF04B" +
            "AA1DD4825B9FFFEBD1DC47414EA4978580A03484B9159E57045018DAA3054704" +
            "84046F89465169A0302B06092A864886F70D010701301406082A864886F70D03" +
            "0704087E74D74C2652F5198008930CBA811F9E9E15A08202103082020C308201" +
            "79A00302010202105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E0302" +
            "1D0500301E311C301A060355040313135253414B65795472616E736665724361" +
            "706931301E170D3135303431353037303030305A170D32353034313530373030" +
            "30305A301E311C301A060355040313135253414B65795472616E736665724361" +
            "70693130819F300D06092A864886F70D010101050003818D0030818902818100" +
            "AA272700586C0CC41B05C65C7D846F5A2BC27B03E301C37D9BFF6D75B6EB6671" +
            "BA9596C5C63BA2B1AF5C318D9CA39E7400D10C238AC72630579211B86570D1A1" +
            "D44EC86AA8F6C9D2B4E283EA3535923F398A312A23EAEACD8D34FAACA965CD91" +
            "0B37DA4093EF76C13B337C1AFAB7D1D07E317B41A336BAA4111299F99424408D" +
            "0203010001A3533051304F0603551D0104483046801015432DB116B35D07E4BA" +
            "89EDB2469D7AA120301E311C301A060355040313135253414B65795472616E73" +
            "666572436170693182105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E" +
            "03021D05000381810081E5535D8ECEEF265ACBC82F6C5F8BC9D84319265F3CCF" +
            "23369FA533C8DC1938952C5931662D9ECD8B1E7B81749E48468167E2FCE3D019" +
            "FA70D54646975B6DC2A3BA72D5A5274C1866DA6D7A5DF47938E034A075D11957" +
            "D653B5C78E5291E4401045576F6D4EDA81BEF3C369AF56121E49A083C8D1ADB0" +
            "9F291822E99A429646318201073082010302010380146B4A6B92FDED07EE0119" +
            "F3674A96D1A70D2A588D300B0609608648016503040201A04B301806092A8648" +
            "86F70D010903310B06092A864886F70D010703302F06092A864886F70D010904" +
            "31220420873B6A3B7CE192922129761C3EDD8D68C4A6B0369F3BF5B3D30B0A9E" +
            "2336A8F4300B06092A864886F70D0101010481807D31B3260AE00DE3992DDD1E" +
            "B01FDECA28053F2B87AA723CCD27B92896E3199F7C4B3B4A391C181899E5CBD1" +
            "4A4BCDDFF6DC6CD10CA118DAA62E32589F066D1669D2948E51B5363B7BEE2BA9" +
            "351CDE1791D118E552F0C8A4FB58EC7C34F5BAB2D562B415C4B3F673179B8410" +
            "86A9B0F03ED56DBD4FA9CBB775307C9BB3045F72").HexToByteArray();

        // This document signature was generated without sorting the attributes,
        // it ensures a compatibility fallback in CheckSignature.
        internal static readonly byte[] DigiCertTimeStampToken = (
            "30820EC406092A864886F70D010702A0820EB530820EB1020103310F300D0609" +
            "608648016503040201050030818A060B2A864886F70D0109100104A07B047930" +
            "7702010106096086480186FD6C07013031300D06096086480165030402010500" +
            "04205A3B99F86CE06F5639C3ED7D0E16AE44A5CD58ABA0F6853A2AF389F91B6C" +
            "7D7F02104E640CDE5C761BFF32FA0ADEA58C5F56180F32303138303731373136" +
            "313732315A0211457A3006B9483730D09772076C4B74BDF8A0820BBB30820682" +
            "3082056AA003020102021009C0FC46C8044213B5598BAF284F4E41300D06092A" +
            "864886F70D01010B05003072310B300906035504061302555331153013060355" +
            "040A130C446967694365727420496E6331193017060355040B13107777772E64" +
            "696769636572742E636F6D3131302F0603550403132844696769436572742053" +
            "48413220417373757265642049442054696D657374616D70696E67204341301E" +
            "170D3137303130343030303030305A170D3238303131383030303030305A304C" +
            "310B30090603550406130255533111300F060355040A13084469676943657274" +
            "312A302806035504031321446967694365727420534841322054696D65737461" +
            "6D7020526573706F6E64657230820122300D06092A864886F70D010101050003" +
            "82010F003082010A02820101009E95986A343B731BA87EFCC7BE296989C76826" +
            "465F3D8D62738781A3A19CF0B75B24375A92D4F459D77689E4DCD527F0D566BC" +
            "0AEEB42B3167AC58C54A91592B451E0901D664B359EE8D664DFB235ECC100D0B" +
            "8A67EF52AEA00890C252F7F5A8B56E9B2C7B9DE7B53EFB78CD325018BF40B54C" +
            "8CBB57F4A04F11456C4242B9E5AFD6DFF4A77C0A68960FD25F2957CEFB1D32FF" +
            "F411A11322FB12CBEFD753D2EB97CBA2AC1B1D9D58215182C2C2DEEA2B3F2C22" +
            "84D043EC3B3B3F47C4F656DC453798B46B74B559AF785769C80F090278DDD853" +
            "C199DB60C49DEAAEAFE07E864A5CA95861A85E748A012868724EA7869DB50252" +
            "87706648D38EEF8124CCDCD8650203010001A382033830820334300E0603551D" +
            "0F0101FF040403020780300C0603551D130101FF0402300030160603551D2501" +
            "01FF040C300A06082B06010505070308308201BF0603551D20048201B6308201" +
            "B2308201A106096086480186FD6C070130820192302806082B06010505070201" +
            "161C68747470733A2F2F7777772E64696769636572742E636F6D2F4350533082" +
            "016406082B06010505070202308201561E8201520041006E0079002000750073" +
            "00650020006F0066002000740068006900730020004300650072007400690066" +
            "0069006300610074006500200063006F006E0073007400690074007500740065" +
            "007300200061006300630065007000740061006E006300650020006F00660020" +
            "007400680065002000440069006700690043006500720074002000430050002F" +
            "00430050005300200061006E00640020007400680065002000520065006C0079" +
            "0069006E0067002000500061007200740079002000410067007200650065006D" +
            "0065006E00740020007700680069006300680020006C0069006D006900740020" +
            "006C0069006100620069006C00690074007900200061006E0064002000610072" +
            "006500200069006E0063006F00720070006F0072006100740065006400200068" +
            "0065007200650069006E0020006200790020007200650066006500720065006E" +
            "00630065002E300B06096086480186FD6C0315301F0603551D23041830168014" +
            "F4B6E1201DFE29AED2E461A5B2A225B2C817356E301D0603551D0E04160414E1" +
            "A7324AEE0121287D54D5F207926EB4070F3D8730710603551D1F046A30683032" +
            "A030A02E862C687474703A2F2F63726C332E64696769636572742E636F6D2F73" +
            "6861322D617373757265642D74732E63726C3032A030A02E862C687474703A2F" +
            "2F63726C342E64696769636572742E636F6D2F736861322D617373757265642D" +
            "74732E63726C30818506082B0601050507010104793077302406082B06010505" +
            "0730018618687474703A2F2F6F6373702E64696769636572742E636F6D304F06" +
            "082B060105050730028643687474703A2F2F636163657274732E646967696365" +
            "72742E636F6D2F44696769436572745348413241737375726564494454696D65" +
            "7374616D70696E6743412E637274300D06092A864886F70D01010B0500038201" +
            "01001EF0418232AEEDF1B43513DC50C2D597AE22229D0E0EAF33D34CFD7CBF6F" +
            "0111A79465225CC622A1C889526B9A8C735CD95E3F32DE16604C8B36FD31990A" +
            "BDC184B78D1DEF8926130556F347CD475BAD84B238AF6A23B545E31E88324680" +
            "D2B7A69922FDC178CFF58BD80C8C0509EE44E680D56D70CC9F531E27DD2A48DE" +
            "DA9365AD6E65A399A7C2400E73CC584F8F4528E5BC9C88E628CE605D2D255D8B" +
            "732EA50D5B51DA9A4EFF50058928DAF278BBD258788D44A7AC3A009178698964" +
            "04D35D96DF2ABFF9A54C2C93FFE68ADD82ACF1D2B3A2869AC15589566A473FFA" +
            "D6339543358905785A3A69DA22B80443D36F6835367A143E45E99864860F130C" +
            "264A3082053130820419A00302010202100AA125D6D6321B7E41E405DA3697C2" +
            "15300D06092A864886F70D01010B05003065310B300906035504061302555331" +
            "153013060355040A130C446967694365727420496E6331193017060355040B13" +
            "107777772E64696769636572742E636F6D312430220603550403131B44696769" +
            "43657274204173737572656420494420526F6F74204341301E170D3136303130" +
            "373132303030305A170D3331303130373132303030305A3072310B3009060355" +
            "04061302555331153013060355040A130C446967694365727420496E63311930" +
            "17060355040B13107777772E64696769636572742E636F6D3131302F06035504" +
            "0313284469676943657274205348413220417373757265642049442054696D65" +
            "7374616D70696E6720434130820122300D06092A864886F70D01010105000382" +
            "010F003082010A0282010100BDD032EE4BCD8F7FDDA9BA8299C539542857B623" +
            "4AC40E07453351107DD0F97D4D687EE7B6A0F48DB388E497BF63219098BF13BC" +
            "57D3C3E17E08D66A140038F72E1E3BEECCA6F63259FE5F653FE09BEBE3464706" +
            "1A557E0B277EC0A2F5A0E0DE223F0EFF7E95FBF3A3BA223E18AC11E4F099036D" +
            "3B857C09D3EE5DC89A0B54E3A809716BE0CF22100F75CF71724E0AADDF403A5C" +
            "B751E1A17914C64D2423305DBCEC3C606AAC2F07CCFDF0EA47D988505EFD666E" +
            "56612729898451E682E74650FD942A2CA7E4753EBA980F847F9F3114D6ADD5F2" +
            "64CB7B1E05D084197217F11706EF3DCDD64DEF0642FDA2532A4F851DC41D3CAF" +
            "CFDAAC10F5DDACACE956FF930203010001A38201CE308201CA301D0603551D0E" +
            "04160414F4B6E1201DFE29AED2E461A5B2A225B2C817356E301F0603551D2304" +
            "183016801445EBA2AFF492CB82312D518BA7A7219DF36DC80F30120603551D13" +
            "0101FF040830060101FF020100300E0603551D0F0101FF040403020186301306" +
            "03551D25040C300A06082B06010505070308307906082B06010505070101046D" +
            "306B302406082B060105050730018618687474703A2F2F6F6373702E64696769" +
            "636572742E636F6D304306082B060105050730028637687474703A2F2F636163" +
            "657274732E64696769636572742E636F6D2F4469676943657274417373757265" +
            "644944526F6F7443412E6372743081810603551D1F047A3078303AA038A03686" +
            "34687474703A2F2F63726C342E64696769636572742E636F6D2F446967694365" +
            "7274417373757265644944526F6F7443412E63726C303AA038A0368634687474" +
            "703A2F2F63726C332E64696769636572742E636F6D2F44696769436572744173" +
            "73757265644944526F6F7443412E63726C30500603551D20044930473038060A" +
            "6086480186FD6C000204302A302806082B06010505070201161C68747470733A" +
            "2F2F7777772E64696769636572742E636F6D2F435053300B06096086480186FD" +
            "6C0701300D06092A864886F70D01010B05000382010100719512E951875669CD" +
            "EFDDDA7CAA637AB378CF06374084EF4B84BFCACF0302FDC5A7C30E20422CAF77" +
            "F32B1F0C215A2AB705341D6AAE99F827A266BF09AA60DF76A43A930FF8B2D1D8" +
            "7C1962E85E82251EC4BA1C7B2C21E2D65B2C1435430468B2DB7502E072C798D6" +
            "3C64E51F4810185F8938614D62462487638C91522CAF2989E5781FD60B14A580" +
            "D7124770B375D59385937EB69267FB536189A8F56B96C0F458690D7CC801B1B9" +
            "2875B7996385228C61CA79947E59FC8C0FE36FB50126B66CA5EE875121E45860" +
            "9BBA0C2D2B6DA2C47EBBC4252B4702087C49AE13B6E17C424228C61856CF4134" +
            "B6665DB6747BF55633222F2236B24BA24A95D8F5A68E523182024D3082024902" +
            "01013081863072310B300906035504061302555331153013060355040A130C44" +
            "6967694365727420496E6331193017060355040B13107777772E646967696365" +
            "72742E636F6D3131302F06035504031328446967694365727420534841322041" +
            "7373757265642049442054696D657374616D70696E67204341021009C0FC46C8" +
            "044213B5598BAF284F4E41300D06096086480165030402010500A08198301A06" +
            "092A864886F70D010903310D060B2A864886F70D0109100104301C06092A8648" +
            "86F70D010905310F170D3138303731373136313732315A302F06092A864886F7" +
            "0D01090431220420E66606A88254749C2E5575722F93AA67174FDDDB7703B7E6" +
            "FAD6B3FE000F3DE2302B060B2A864886F70D010910020C311C301A3018301604" +
            "14400191475C98891DEBA104AF47091B5EB6D4CBCB300D06092A864886F70D01" +
            "01010500048201005AF349DE87550378C702ED31AE6DD6D50E6B24298DB2DFD6" +
            "1396C6FA3E465FE7323ACD65AE157C06BCB993551F33702C6D1F6951AECDE74A" +
            "969E41A8F0F95188780F990EAF6B129633CDE42FF149501BFAC05C516B6DA9EF" +
            "E492488013928BA801D66C32EFE7EEDFF22DC96DDA4783674EEE8231E7A3AD8A" +
            "98A506DABB68D6337D7FFDBD2F7112AF2FEE718CF6E7E5544DB7B4BDCD8191EB" +
            "C73D568EE4D2A30B8478D676910E3B4EB868010AAF22400198D0593F987C86A9" +
            "101711B9C6AC5C5776923C699E772B07864755C1AC50F387655C4E67DB356207" +
            "76252A2F4605B97BD3C299D1CD79929273BB86E7DF9E113C92802380ED6D4041" +
            "9DA4C01214D4FA24").HexToByteArray();

        internal static readonly byte[] RsaPkcs1Sha256WithRsa = (
            "3082071B06092A864886F70D010702A082070C30820708020101310F300D0609" +
            "6086480165030402010500301606092A864886F70D010701A009040700010203" +
            "040506A08205223082051E30820406A00302010202100D85090F3FACFF0A9008" +
            "A12A9FB00A54300D06092A864886F70D01010B05003072310B30090603550406" +
            "1302555331153013060355040A130C446967694365727420496E633119301706" +
            "0355040B13107777772E64696769636572742E636F6D3131302F060355040313" +
            "2844696769436572742053484132204173737572656420494420436F64652053" +
            "69676E696E67204341301E170D3138303832393030303030305A170D31393039" +
            "30333132303030305A305B310B3009060355040613025553310B300906035504" +
            "0813025641311330110603550407130A416C6578616E64726961311430120603" +
            "55040A130B4B6576696E204A6F6E6573311430120603550403130B4B6576696E" +
            "204A6F6E657330820122300D06092A864886F70D01010105000382010F003082" +
            "010A0282010100F1F4542FF6CA57FBC44986EC816F07D1FD50BFD477C412D299" +
            "1C962D0A22194A4296BCD0751F47CE4932F73871277CE3CDD2C78157599C7A35" +
            "80CC96A11F7031E3A798F4BAA93988F0E4077D30316252B24337DB26914E1F77" +
            "9CB4979544514B0234E5388E936B195B91863B258F0C8951454D3668F0C4D456" +
            "A8497758D21C433626E46F2CFF5A7CC7945F788948998E5F8786E1E990E240BB" +
            "0780CD258F57761AFB5D42AD8E3D703C3126861E83F191ECE9F0B83221F96214" +
            "533B2A47977F43715FE501FBC4A4040839DD3EBCA8B67259A7DD0EA9EFAE2200" +
            "943EFB7D0404B8978B49A445849B5F6898B06269F427F30DBC8DB2FD7963943A" +
            "8C461760E6A4F30203010001A38201C5308201C1301F0603551D230418301680" +
            "145AC4B97B2A0AA3A5EA7103C060F92DF665750E58301D0603551D0E04160414" +
            "33795EB2D84BFAA3F96E5930F64EC6A94C6FD36A300E0603551D0F0101FF0404" +
            "0302078030130603551D25040C300A06082B0601050507030330770603551D1F" +
            "0470306E3035A033A031862F687474703A2F2F63726C332E6469676963657274" +
            "2E636F6D2F736861322D617373757265642D63732D67312E63726C3035A033A0" +
            "31862F687474703A2F2F63726C342E64696769636572742E636F6D2F73686132" +
            "2D617373757265642D63732D67312E63726C304C0603551D2004453043303706" +
            "096086480186FD6C0301302A302806082B06010505070201161C68747470733A" +
            "2F2F7777772E64696769636572742E636F6D2F4350533008060667810C010401" +
            "30818406082B0601050507010104783076302406082B06010505073001861868" +
            "7474703A2F2F6F6373702E64696769636572742E636F6D304E06082B06010505" +
            "0730028642687474703A2F2F636163657274732E64696769636572742E636F6D" +
            "2F446967694365727453484132417373757265644944436F64655369676E696E" +
            "6743412E637274300C0603551D130101FF04023000300D06092A864886F70D01" +
            "010B0500038201010045B9D9868E494BD02635D0E42DDE80B37A865C389CFDD9" +
            "9BFC9B62E2C169A73B5EABF282607439EFF5C61630886DEB63415B53683446A7" +
            "3041686C326BA35FF0029FEF603D7C80FA0177A4DE35013529B01F759FD50414" +
            "79BDBB6B93B18144CB14E431BC144146848EF8ADB0E28952EAD1BB49E8547FFE" +
            "9934817036338B20C4E0B9D7C6A4E5BE3D57157F21904A5C864946313EA6B7D9" +
            "50EE0235B5D2CD01490AD2B2A1AB5F66EC8986D64A1D9D239C131E09E5CA1C02" +
            "A75F2D7EC07E4C858856A6A58AB94DEAC8B3D3A5BBF492EE2463B156E6A0660B" +
            "B452E35922D00456F0DEE0ED15A8BF8FFF31008756B14EEE0AC14BCF19A3CD16" +
            "819DC990F5F45CDE21318201B2308201AE0201013081863072310B3009060355" +
            "04061302555331153013060355040A130C446967694365727420496E63311930" +
            "17060355040B13107777772E64696769636572742E636F6D3131302F06035504" +
            "03132844696769436572742053484132204173737572656420494420436F6465" +
            "205369676E696E6720434102100D85090F3FACFF0A9008A12A9FB00A54300D06" +
            "096086480165030402010500300D06092A864886F70D01010B050004820100E2" +
            "980C5A30EC00729D1CFA826D7A65B43FF6806B5E0ABA23A78E4F1CAA3F6436EF" +
            "00941C6947A9B8F20D0757B5346CF640AA217F7361BEEFF2BC997FB1D3597BF3" +
            "D7457BD4A94062FB03660F9D86710BE2FC99876A848251F4965E1B16192714C8" +
            "F9788C09CCDE83603ADC919297BA496E921B95F3BD9554A873E09912640FCFAA" +
            "D9DD1441D1851E637031D390C038223AE64B048E806462DDBAC98C156BE2EE47" +
            "2B78166BDB1612848B535ADC3F0E7BE52991A17F48AFDCCC1698A236BA338930" +
            "50EBAAC4460DAA35185C16670F597E0E6E0CB0AA83F51AAEF452F3367DD9350A" +
            "8A49A5A8F79DF8E921303AB5D6646A482F0F59D9980310E1AE3EE8D77CB857").HexToByteArray();

        internal static readonly byte[] RsaPkcs1SignedSha1DeclaredSha256WithRsa = (
            "3082071306092A864886F70D010702A082070430820700020101310B30090605" +
            "2B0E03021A0500301606092A864886F70D010701A009040700010203040506A0" +
            "8205223082051E30820406A00302010202100D85090F3FACFF0A9008A12A9FB0" +
            "0A54300D06092A864886F70D01010B05003072310B3009060355040613025553" +
            "31153013060355040A130C446967694365727420496E6331193017060355040B" +
            "13107777772E64696769636572742E636F6D3131302F06035504031328446967" +
            "69436572742053484132204173737572656420494420436F6465205369676E69" +
            "6E67204341301E170D3138303832393030303030305A170D3139303930333132" +
            "303030305A305B310B3009060355040613025553310B30090603550408130256" +
            "41311330110603550407130A416C6578616E6472696131143012060355040A13" +
            "0B4B6576696E204A6F6E6573311430120603550403130B4B6576696E204A6F6E" +
            "657330820122300D06092A864886F70D01010105000382010F003082010A0282" +
            "010100F1F4542FF6CA57FBC44986EC816F07D1FD50BFD477C412D2991C962D0A" +
            "22194A4296BCD0751F47CE4932F73871277CE3CDD2C78157599C7A3580CC96A1" +
            "1F7031E3A798F4BAA93988F0E4077D30316252B24337DB26914E1F779CB49795" +
            "44514B0234E5388E936B195B91863B258F0C8951454D3668F0C4D456A8497758" +
            "D21C433626E46F2CFF5A7CC7945F788948998E5F8786E1E990E240BB0780CD25" +
            "8F57761AFB5D42AD8E3D703C3126861E83F191ECE9F0B83221F96214533B2A47" +
            "977F43715FE501FBC4A4040839DD3EBCA8B67259A7DD0EA9EFAE2200943EFB7D" +
            "0404B8978B49A445849B5F6898B06269F427F30DBC8DB2FD7963943A8C461760" +
            "E6A4F30203010001A38201C5308201C1301F0603551D230418301680145AC4B9" +
            "7B2A0AA3A5EA7103C060F92DF665750E58301D0603551D0E0416041433795EB2" +
            "D84BFAA3F96E5930F64EC6A94C6FD36A300E0603551D0F0101FF040403020780" +
            "30130603551D25040C300A06082B0601050507030330770603551D1F0470306E" +
            "3035A033A031862F687474703A2F2F63726C332E64696769636572742E636F6D" +
            "2F736861322D617373757265642D63732D67312E63726C3035A033A031862F68" +
            "7474703A2F2F63726C342E64696769636572742E636F6D2F736861322D617373" +
            "757265642D63732D67312E63726C304C0603551D200445304330370609608648" +
            "0186FD6C0301302A302806082B06010505070201161C68747470733A2F2F7777" +
            "772E64696769636572742E636F6D2F4350533008060667810C01040130818406" +
            "082B0601050507010104783076302406082B060105050730018618687474703A" +
            "2F2F6F6373702E64696769636572742E636F6D304E06082B0601050507300286" +
            "42687474703A2F2F636163657274732E64696769636572742E636F6D2F446967" +
            "694365727453484132417373757265644944436F64655369676E696E6743412E" +
            "637274300C0603551D130101FF04023000300D06092A864886F70D01010B0500" +
            "038201010045B9D9868E494BD02635D0E42DDE80B37A865C389CFDD99BFC9B62" +
            "E2C169A73B5EABF282607439EFF5C61630886DEB63415B53683446A73041686C" +
            "326BA35FF0029FEF603D7C80FA0177A4DE35013529B01F759FD5041479BDBB6B" +
            "93B18144CB14E431BC144146848EF8ADB0E28952EAD1BB49E8547FFE99348170" +
            "36338B20C4E0B9D7C6A4E5BE3D57157F21904A5C864946313EA6B7D950EE0235" +
            "B5D2CD01490AD2B2A1AB5F66EC8986D64A1D9D239C131E09E5CA1C02A75F2D7E" +
            "C07E4C858856A6A58AB94DEAC8B3D3A5BBF492EE2463B156E6A0660BB452E359" +
            "22D00456F0DEE0ED15A8BF8FFF31008756B14EEE0AC14BCF19A3CD16819DC990" +
            "F5F45CDE21318201AE308201AA0201013081863072310B300906035504061302" +
            "555331153013060355040A130C446967694365727420496E6331193017060355" +
            "040B13107777772E64696769636572742E636F6D3131302F0603550403132844" +
            "696769436572742053484132204173737572656420494420436F646520536967" +
            "6E696E6720434102100D85090F3FACFF0A9008A12A9FB00A54300906052B0E03" +
            "021A0500300D06092A864886F70D01010B050004820100EAEEB9E1D4BFB979F1" +
            "A1C00EE1EC45069366CDD7489A0671F6DC9E3353F7FAEDCE7B87BD467ADFC850" +
            "877414966E7EB39C33367ABB03B3AA8BB1438BD952484CB807451499CAE8FDC9" +
            "527304D459D82CA039087560B5D3D0EA03DEA1B9472EFC44CBB55DD9A3C6A5C8" +
            "DFFD0786D5523F22604B412D6FC5A15E2D6285D7AB76EC216DE859391D129D51" +
            "6C27348EDAE7DC43335D12242D939CAF05385A118235F5B1E342EC034E70F655" +
            "793FF2FE037558EC2F45BD2683704F8FFD49B910131F4F2804B4282C5C36E41C" +
            "9E4E4F93446D44E3106760D265C5C7A849CF03426ACCB294712E51313D5414A7" +
            "8227AB79F6B18E2A2054E3FA781DAA2998EB33EDDCDA80").HexToByteArray();

        internal static readonly byte[] IndefiniteLengthContentDocument = (
            "3082265206092A864886F70D010702A08226433082263F020103310F300D0609" +
            "6086480165030402010500306806092A864886F70D010701A05B248004555665" +
            "7273696F6E3A310A0A322E31362E3834302E312E3130312E332E342E322E312D" +
            "486173683A49413134467074344A365A504F5A47506275364A676A524B696930" +
            "664E584A365A6B78746F744B665733383D0A0A0000A082106D308203C5308202" +
            "ADA003020102021002AC5C266A0B409B8F0B79F2AE462577300D06092A864886" +
            "F70D0101050500306C310B300906035504061302555331153013060355040A13" +
            "0C446967694365727420496E6331193017060355040B13107777772E64696769" +
            "636572742E636F6D312B30290603550403132244696769436572742048696768" +
            "204173737572616E636520455620526F6F74204341301E170D30363131313030" +
            "30303030305A170D3331313131303030303030305A306C310B30090603550406" +
            "1302555331153013060355040A130C446967694365727420496E633119301706" +
            "0355040B13107777772E64696769636572742E636F6D312B3029060355040313" +
            "2244696769436572742048696768204173737572616E636520455620526F6F74" +
            "20434130820122300D06092A864886F70D01010105000382010F003082010A02" +
            "82010100C6CCE573E6FBD4BBE52D2D32A6DFE5813FC9CD2549B6712AC3D59434" +
            "67A20A1CB05F69A640B1C4B7B28FD098A4A941593AD3DC94D63CDB7438A44ACC" +
            "4D2582F74AA5531238EEF3496D71917E63B6ABA65FC3A484F84F6251BEF8C5EC" +
            "DB3892E306E508910CC4284155FBCB5A89157E71E835BF4D72093DBE3A38505B" +
            "77311B8DB3C724459AA7AC6D00145A04B7BA13EB510A984141224E6561878141" +
            "50A6795C89DE194A57D52EE65D1C532C7E98CD1A0616A46873D03404135CA171" +
            "D35A7C55DB5E64E13787305604E511B4298012F1793988A202117C2766B788B7" +
            "78F2CA0AA838AB0A64C2BF665D9584C1A1251E875D1A500B2012CC41BB6E0B51" +
            "38B84BCB0203010001A3633061300E0603551D0F0101FF040403020186300F06" +
            "03551D130101FF040530030101FF301D0603551D0E04160414B13EC36903F8BF" +
            "4701D498261A0802EF63642BC3301F0603551D23041830168014B13EC36903F8" +
            "BF4701D498261A0802EF63642BC3300D06092A864886F70D0101050500038201" +
            "01001C1A0697DCD79C9F3C886606085721DB2147F82A67AABF183276401057C1" +
            "8AF37AD911658E35FA9EFC45B59ED94C314BB891E8432C8EB378CEDBE3537971" +
            "D6E5219401DA55879A2464F68A66CCDE9C37CDA834B1699B23C89E78222B7043" +
            "E35547316119EF58C5852F4E30F6A0311623C8E7E2651633CBBF1A1BA03DF8CA" +
            "5E8B318B6008892D0C065C52B7C4F90A98D1155F9F12BE7C366338BD44A47FE4" +
            "262B0AC497690DE98CE2C01057B8C876129155F24869D8BC2A025B0F44D42031" +
            "DBF4BA70265D90609EBC4B17092FB4CB1E4368C90727C1D25CF7EA21B968129C" +
            "3C9CBF9EFC805C9B63CDEC47AA252767A037F300827D54D7A9F8E92E13A377E8" +
            "1F4A308205E0308204C8A0030201020210070C57D60A8AB12F9B5F2D5CD2EDA5" +
            "04300D06092A864886F70D01010B0500306C310B300906035504061302555331" +
            "153013060355040A130C446967694365727420496E6331193017060355040B13" +
            "107777772E64696769636572742E636F6D312B30290603550403132244696769" +
            "4365727420455620436F6465205369676E696E6720434120285348413229301E" +
            "170D3137303531363030303030305A170D3230303532303132303030305A3081" +
            "F6311D301B060355040F0C1450726976617465204F7267616E697A6174696F6E" +
            "31133011060B2B0601040182373C0201031302555331193017060B2B06010401" +
            "82373C02010213084E657720596F726B3110300E060355040513073531303136" +
            "3333311E301C06035504091315353435205720456E6420417665204170742031" +
            "3645310E300C060355041113053130303234310B300906035504061302555331" +
            "0B3009060355040813024E593111300F060355040713084E657720596F726B31" +
            "1A3018060355040A13114F72656E204E6F766F746E792C204C4C43311A301806" +
            "0355040313114F72656E204E6F766F746E792C204C4C4330820122300D06092A" +
            "864886F70D01010105000382010F003082010A0282010100E253870D0B9B81C7" +
            "B2449CBD7BA9837EDFE23C79F54CE2C1BFA0F5C92C261AAC4E3D40CD890F8046" +
            "289C7BCF0525BC0D2E193C28E7AD3ABF87FD35CAB9CBC869CEF8964B80DD77CE" +
            "E209EFCC42A465FDC1338A478DAA80254D17679548ED78528A13632B66FD0531" +
            "555E04BCAF14A67AD266CECB03346FAB4757C1AC120FE34C15E4B5BBBDE3E601" +
            "EF9A8BBE6EE1802EF18E852233130B9DE0995FAACC869D414380B478E5E343AB" +
            "B2C18DDEC8170EB601301EC178B6CD66FF1E10D81637CB609FECCFBF91666090" +
            "A30F0D95B3BC204FDFC6314913C20091DBE60DEC88488C6B48E66E71247E11D2" +
            "7662A2A7106806BE8B95041F46D1375688F0B3C0BE805C750203010001A38201" +
            "F1308201ED301F0603551D230418301680148FE87EF06D326A000523C770976A" +
            "3A90FF6BEAD4301D0603551D0E0416041451B49AE070DEBDAE12B71C87F99F4E" +
            "0E495B30A1302E0603551D1104273025A02306082B06010505070803A0173015" +
            "0C1355532D4E455720594F524B2D35313031363333300E0603551D0F0101FF04" +
            "040302078030130603551D25040C300A06082B06010505070303307B0603551D" +
            "1F047430723037A035A0338631687474703A2F2F63726C332E64696769636572" +
            "742E636F6D2F4556436F64655369676E696E67534841322D67312E63726C3037" +
            "A035A0338631687474703A2F2F63726C342E64696769636572742E636F6D2F45" +
            "56436F64655369676E696E67534841322D67312E63726C304B0603551D200444" +
            "3042303706096086480186FD6C0302302A302806082B06010505070201161C68" +
            "747470733A2F2F7777772E64696769636572742E636F6D2F4350533007060567" +
            "810C0103307E06082B0601050507010104723070302406082B06010505073001" +
            "8618687474703A2F2F6F6373702E64696769636572742E636F6D304806082B06" +
            "010505073002863C687474703A2F2F636163657274732E64696769636572742E" +
            "636F6D2F44696769436572744556436F64655369676E696E6743412D53484132" +
            "2E637274300C0603551D130101FF04023000300D06092A864886F70D01010B05" +
            "0003820101000481103296C95A10AE7055ECB980F0A5910F756DB48704BB078C" +
            "B53E8859EDDF906E4B6207301C8BAE9CE39BE70D6D897EA2F32BD05270B8DC1E" +
            "44ADB0ECDCCAC7A858295E81E030B9A5CF571CD81536EBE36E4D87A9DD6E0C42" +
            "3B5E6F3A249C90AFB1418925F806C6B91346EBFDC61C1D392FA0861AE7B5ADF8" +
            "E0E213BBCCA53F826DA3B7514A80A7148968DC4855F584DBB2AB6E99569C6C89" +
            "333CFEEA0914F3C77A04E5EE44D7A27370F34FDFF4753762B62D55809C43FD63" +
            "F37BBB0239CB1B25F1F7C1932D0885F526CC87F589ED40BCEFBBDDD76620AA2D" +
            "5D373B9EE51F9FD02905EEC71DD5F5419010E5D061643A6A04211FA3165943AE" +
            "D0FF1FB58D23308206BC308205A4A003020102021003F1B4E15F3A82F1149678" +
            "B3D7D8475C300D06092A864886F70D01010B0500306C310B3009060355040613" +
            "02555331153013060355040A130C446967694365727420496E63311930170603" +
            "55040B13107777772E64696769636572742E636F6D312B302906035504031322" +
            "44696769436572742048696768204173737572616E636520455620526F6F7420" +
            "4341301E170D3132303431383132303030305A170D3237303431383132303030" +
            "305A306C310B300906035504061302555331153013060355040A130C44696769" +
            "4365727420496E6331193017060355040B13107777772E64696769636572742E" +
            "636F6D312B302906035504031322446967694365727420455620436F64652053" +
            "69676E696E672043412028534841322930820122300D06092A864886F70D0101" +
            "0105000382010F003082010A0282010100A753FA0FB2B513F164CF8480FCAE80" +
            "35D1B6D7C7A32CAC1A2CACF184AC3A35123A9291BA57E4C4C9F32FA8483CB7D6" +
            "6EDC9722BA517961AF432F0DB79BB44931AE44583EA4A196A7874F237EC36C65" +
            "2490553EA1CA237CC542E9C47A62459B7DDE6374CB9E6325F8849A9AAD454FAE" +
            "7D1FC813CB759BC9E1E18AF80B0C98F4CA3ED045AA7A1EA558933634BE2B2E2B" +
            "315866B432109F9DF052A1EFE83ED376F2405ADCFA6A3D1B4BAD76B08C5CEE36" +
            "BA83EA30A84CDEF10B2A584188AE0089AB03D11682202276EB5E54381262E1D2" +
            "7024DBED1F70D26409802DE2B69DCE1FF2BB21F36CDBD8B3197B8A509FEFEC36" +
            "0A5C9AB74AD308A03979FDDDBF3D3A09250203010001A3820358308203543012" +
            "0603551D130101FF040830060101FF020100300E0603551D0F0101FF04040302" +
            "018630130603551D25040C300A06082B06010505070303307F06082B06010505" +
            "07010104733071302406082B060105050730018618687474703A2F2F6F637370" +
            "2E64696769636572742E636F6D304906082B06010505073002863D687474703A" +
            "2F2F636163657274732E64696769636572742E636F6D2F446967694365727448" +
            "6967684173737572616E63654556526F6F7443412E63727430818F0603551D1F" +
            "0481873081843040A03EA03C863A687474703A2F2F63726C332E646967696365" +
            "72742E636F6D2F4469676943657274486967684173737572616E63654556526F" +
            "6F7443412E63726C3040A03EA03C863A687474703A2F2F63726C342E64696769" +
            "636572742E636F6D2F4469676943657274486967684173737572616E63654556" +
            "526F6F7443412E63726C308201C40603551D20048201BB308201B7308201B306" +
            "096086480186FD6C0302308201A4303A06082B06010505070201162E68747470" +
            "3A2F2F7777772E64696769636572742E636F6D2F73736C2D6370732D7265706F" +
            "7369746F72792E68746D3082016406082B06010505070202308201561E820152" +
            "0041006E007900200075007300650020006F0066002000740068006900730020" +
            "0043006500720074006900660069006300610074006500200063006F006E0073" +
            "007400690074007500740065007300200061006300630065007000740061006E" +
            "006300650020006F006600200074006800650020004400690067006900430065" +
            "00720074002000430050002F00430050005300200061006E0064002000740068" +
            "0065002000520065006C00790069006E00670020005000610072007400790020" +
            "00410067007200650065006D0065006E00740020007700680069006300680020" +
            "006C0069006D006900740020006C0069006100620069006C0069007400790020" +
            "0061006E0064002000610072006500200069006E0063006F00720070006F0072" +
            "0061007400650064002000680065007200650069006E00200062007900200072" +
            "00650066006500720065006E00630065002E301D0603551D0E041604148FE87E" +
            "F06D326A000523C770976A3A90FF6BEAD4301F0603551D23041830168014B13E" +
            "C36903F8BF4701D498261A0802EF63642BC3300D06092A864886F70D01010B05" +
            "00038201010019334A0C813337DBAD36C9E4C93ABBB51B2E7AA2E2F44342179E" +
            "BF4EA14DE1B1DBE981DD9F01F2E488D5E9FE09FD21C1EC5D80D2F0D6C143C2FE" +
            "772BDBF9D79133CE6CD5B2193BE62ED6C9934F88408ECDE1F57EF10FC6595672" +
            "E8EB6A41BD1CD546D57C49CA663815C1BFE091707787DCC98D31C90C29A233ED" +
            "8DE287CD898D3F1BFFD5E01A978B7CDA6DFBA8C6B23A666B7B01B3CDD8A634EC" +
            "1201AB9558A5C45357A860E6E70212A0B92364A24DBB7C81256421BECFEE4218" +
            "4397BBA53706AF4DFF26A54D614BEC4641B865CEB8799E08960B818C8A3B8FC7" +
            "998CA32A6E986D5E61C696B78AB9612D93B8EB0E0443D7F5FEA6F062D4996AA5" +
            "C1C1F06494803182154C30821548020103801451B49AE070DEBDAE12B71C87F9" +
            "9F4E0E495B30A1300D06096086480165030402010500A082015B301806092A86" +
            "4886F70D010903310B06092A864886F70D010701301C06092A864886F70D0109" +
            "05310F170D3138303632333139303131385A301E060B2A864886F70D01091002" +
            "10310F300D060B2A864886F70D0109100601302F06092A864886F70D01090431" +
            "220420F67BA3F4A22E1F7D27A0FF8940FD931FF3D5AAA6BA99B92B6ABA6B63B3" +
            "93E5DB3081CF060B2A864886F70D010910022F3181BF3081BC3081B93081B630" +
            "0B0609608648016503040201042017B677DFF2A3A70C935DCABCEAF6595183D6" +
            "2F8972E77AC688369EDE50D2B2563081843070A46E306C310B30090603550406" +
            "1302555331153013060355040A130C446967694365727420496E633119301706" +
            "0355040B13107777772E64696769636572742E636F6D312B3029060355040313" +
            "22446967694365727420455620436F6465205369676E696E6720434120285348" +
            "4132290210070C57D60A8AB12F9B5F2D5CD2EDA504300D06092A864886F70D01" +
            "010B050004820100AA697F9C5F4517BB139B8A7FAEC1BBBBAB53381B4F649EE1" +
            "9A35E9F5D47ABE0C2FCF881C6DC176E52F8191A43486630939264F42464B865D" +
            "B46979EF4996CD081CF615F194BBC9F539609000F2A2A6CBD9F88E497F4B907F" +
            "AEE95B55E73921CCAF89287FDC3F18FA1DB5D13BA7932A0559A9C4194E937611" +
            "003228D161B2EE40FD5C8BDB0463F90DFC5B0D0ABBBDFE7918BDDA2F5658F8E2" +
            "BA47362DC677CD8785D0AD76B73CC544411B7B168E471A449FDF8BD356719D56" +
            "C67E69F81F9534A2DE9D1CDD22717D3D3F84D6CB10C7FC2651945A000A2B7696" +
            "6A69CBE5D37B513E67E945B29CE2CC09CB2E4C8CDBAF105BCFE0927D14B8CEEF" +
            "92851AF4476053DCA18212AA308212A6060B2A864886F70D010910020E318212" +
            "953082129106092A864886F70D010702A08212823082127E020103310F300D06" +
            "09608648016503040201050030819C060B2A864886F70D0109100104A0818C04" +
            "818930818602010106096086480186FD6C07013031300D060960864801650304" +
            "0201050004208A42931DB9A7486014564D4922BE742D01B6ADF152B1421227B7" +
            "D6FBB943EEC202105B0A3E465D2092EB05B245401BC2594E180F323031383036" +
            "32333139303132305A02205718795810783FECCC7143D821251F44AEE16A57A1" +
            "CA24C1B2EF1796CB777AD3A0820F76308203B73082029FA00302010202100CE7" +
            "E0E517D846FE8FE560FC1BF03039300D06092A864886F70D0101050500306531" +
            "0B300906035504061302555331153013060355040A130C446967694365727420" +
            "496E6331193017060355040B13107777772E64696769636572742E636F6D3124" +
            "30220603550403131B4469676943657274204173737572656420494420526F6F" +
            "74204341301E170D3036313131303030303030305A170D333131313130303030" +
            "3030305A3065310B300906035504061302555331153013060355040A130C4469" +
            "67694365727420496E6331193017060355040B13107777772E64696769636572" +
            "742E636F6D312430220603550403131B44696769436572742041737375726564" +
            "20494420526F6F7420434130820122300D06092A864886F70D01010105000382" +
            "010F003082010A0282010100AD0E15CEE443805CB187F3B760F97112A5AEDC26" +
            "9488AAF4CEF520392858600CF880DAA9159532613CB5B128848A8ADC9F0A0C83" +
            "177A8F90AC8AE779535C31842AF60F98323676CCDEDD3CA8A2EF6AFB21F25261" +
            "DF9F20D71FE2B1D9FE1864D2125B5FF9581835BC47CDA136F96B7FD4B0383EC1" +
            "1BC38C33D9D82F18FE280FB3A783D6C36E44C061359616FE599C8B766DD7F1A2" +
            "4B0D2BFF0B72DA9E60D08E9035C678558720A1CFE56D0AC8497C3198336C22E9" +
            "87D0325AA2BA138211ED39179D993A72A1E6FAA4D9D5173175AE857D22AE3F01" +
            "4686F62879C8B1DAE45717C47E1C0EB0B492A656B3BDB297EDAAA7F0B7C5A83F" +
            "9516D0FFA196EB085F18774F0203010001A3633061300E0603551D0F0101FF04" +
            "0403020186300F0603551D130101FF040530030101FF301D0603551D0E041604" +
            "1445EBA2AFF492CB82312D518BA7A7219DF36DC80F301F0603551D2304183016" +
            "801445EBA2AFF492CB82312D518BA7A7219DF36DC80F300D06092A864886F70D" +
            "01010505000382010100A20EBCDFE2EDF0E372737A6494BFF77266D832E44275" +
            "62AE87EBF2D5D9DE56B39FCCCE1428B90D97605C124C58E4D33D834945589735" +
            "691AA847EA56C679AB12D8678184DF7F093C94E6B8262C20BD3DB32889F75FFF" +
            "22E297841FE965EF87E0DFC16749B35DEBB2092AEB26ED78BE7D3F2BF3B72635" +
            "6D5F8901B6495B9F01059BAB3D25C1CCB67FC2F16F86C6FA6468EB812D94EB42" +
            "B7FA8C1EDD62F1BE5067B76CBDF3F11F6B0C3607167F377CA95B6D7AF1124660" +
            "83D72704BE4BCE97BEC3672A6811DF80E70C3366BF130D146EF37F1F63101EFA" +
            "8D1B256D6C8FA5B76101B1D2A326A110719DADE2C3F9C39951B72B0708CE2EE6" +
            "50B2A7FA0A452FA2F0F23082053130820419A00302010202100AA125D6D6321B" +
            "7E41E405DA3697C215300D06092A864886F70D01010B05003065310B30090603" +
            "5504061302555331153013060355040A130C446967694365727420496E633119" +
            "3017060355040B13107777772E64696769636572742E636F6D31243022060355" +
            "0403131B4469676943657274204173737572656420494420526F6F7420434130" +
            "1E170D3136303130373132303030305A170D3331303130373132303030305A30" +
            "72310B300906035504061302555331153013060355040A130C44696769436572" +
            "7420496E6331193017060355040B13107777772E64696769636572742E636F6D" +
            "3131302F06035504031328446967694365727420534841322041737375726564" +
            "2049442054696D657374616D70696E6720434130820122300D06092A864886F7" +
            "0D01010105000382010F003082010A0282010100BDD032EE4BCD8F7FDDA9BA82" +
            "99C539542857B6234AC40E07453351107DD0F97D4D687EE7B6A0F48DB388E497" +
            "BF63219098BF13BC57D3C3E17E08D66A140038F72E1E3BEECCA6F63259FE5F65" +
            "3FE09BEBE34647061A557E0B277EC0A2F5A0E0DE223F0EFF7E95FBF3A3BA223E" +
            "18AC11E4F099036D3B857C09D3EE5DC89A0B54E3A809716BE0CF22100F75CF71" +
            "724E0AADDF403A5CB751E1A17914C64D2423305DBCEC3C606AAC2F07CCFDF0EA" +
            "47D988505EFD666E56612729898451E682E74650FD942A2CA7E4753EBA980F84" +
            "7F9F3114D6ADD5F264CB7B1E05D084197217F11706EF3DCDD64DEF0642FDA253" +
            "2A4F851DC41D3CAFCFDAAC10F5DDACACE956FF930203010001A38201CE308201" +
            "CA301D0603551D0E04160414F4B6E1201DFE29AED2E461A5B2A225B2C817356E" +
            "301F0603551D2304183016801445EBA2AFF492CB82312D518BA7A7219DF36DC8" +
            "0F30120603551D130101FF040830060101FF020100300E0603551D0F0101FF04" +
            "040302018630130603551D25040C300A06082B06010505070308307906082B06" +
            "010505070101046D306B302406082B060105050730018618687474703A2F2F6F" +
            "6373702E64696769636572742E636F6D304306082B0601050507300286376874" +
            "74703A2F2F636163657274732E64696769636572742E636F6D2F446967694365" +
            "7274417373757265644944526F6F7443412E6372743081810603551D1F047A30" +
            "78303AA038A0368634687474703A2F2F63726C342E64696769636572742E636F" +
            "6D2F4469676943657274417373757265644944526F6F7443412E63726C303AA0" +
            "38A0368634687474703A2F2F63726C332E64696769636572742E636F6D2F4469" +
            "676943657274417373757265644944526F6F7443412E63726C30500603551D20" +
            "044930473038060A6086480186FD6C000204302A302806082B06010505070201" +
            "161C68747470733A2F2F7777772E64696769636572742E636F6D2F435053300B" +
            "06096086480186FD6C0701300D06092A864886F70D01010B0500038201010071" +
            "9512E951875669CDEFDDDA7CAA637AB378CF06374084EF4B84BFCACF0302FDC5" +
            "A7C30E20422CAF77F32B1F0C215A2AB705341D6AAE99F827A266BF09AA60DF76" +
            "A43A930FF8B2D1D87C1962E85E82251EC4BA1C7B2C21E2D65B2C1435430468B2" +
            "DB7502E072C798D63C64E51F4810185F8938614D62462487638C91522CAF2989" +
            "E5781FD60B14A580D7124770B375D59385937EB69267FB536189A8F56B96C0F4" +
            "58690D7CC801B1B92875B7996385228C61CA79947E59FC8C0FE36FB50126B66C" +
            "A5EE875121E458609BBA0C2D2B6DA2C47EBBC4252B4702087C49AE13B6E17C42" +
            "4228C61856CF4134B6665DB6747BF55633222F2236B24BA24A95D8F5A68E5230" +
            "8206823082056AA003020102021009C0FC46C8044213B5598BAF284F4E41300D" +
            "06092A864886F70D01010B05003072310B300906035504061302555331153013" +
            "060355040A130C446967694365727420496E6331193017060355040B13107777" +
            "772E64696769636572742E636F6D3131302F0603550403132844696769436572" +
            "74205348413220417373757265642049442054696D657374616D70696E672043" +
            "41301E170D3137303130343030303030305A170D323830313138303030303030" +
            "5A304C310B30090603550406130255533111300F060355040A13084469676943" +
            "657274312A302806035504031321446967694365727420534841322054696D65" +
            "7374616D7020526573706F6E64657230820122300D06092A864886F70D010101" +
            "05000382010F003082010A02820101009E95986A343B731BA87EFCC7BE296989" +
            "C76826465F3D8D62738781A3A19CF0B75B24375A92D4F459D77689E4DCD527F0" +
            "D566BC0AEEB42B3167AC58C54A91592B451E0901D664B359EE8D664DFB235ECC" +
            "100D0B8A67EF52AEA00890C252F7F5A8B56E9B2C7B9DE7B53EFB78CD325018BF" +
            "40B54C8CBB57F4A04F11456C4242B9E5AFD6DFF4A77C0A68960FD25F2957CEFB" +
            "1D32FFF411A11322FB12CBEFD753D2EB97CBA2AC1B1D9D58215182C2C2DEEA2B" +
            "3F2C2284D043EC3B3B3F47C4F656DC453798B46B74B559AF785769C80F090278" +
            "DDD853C199DB60C49DEAAEAFE07E864A5CA95861A85E748A012868724EA7869D" +
            "B5025287706648D38EEF8124CCDCD8650203010001A382033830820334300E06" +
            "03551D0F0101FF040403020780300C0603551D130101FF040230003016060355" +
            "1D250101FF040C300A06082B06010505070308308201BF0603551D20048201B6" +
            "308201B2308201A106096086480186FD6C070130820192302806082B06010505" +
            "070201161C68747470733A2F2F7777772E64696769636572742E636F6D2F4350" +
            "533082016406082B06010505070202308201561E8201520041006E0079002000" +
            "75007300650020006F0066002000740068006900730020004300650072007400" +
            "6900660069006300610074006500200063006F006E0073007400690074007500" +
            "740065007300200061006300630065007000740061006E006300650020006F00" +
            "6600200074006800650020004400690067006900430065007200740020004300" +
            "50002F00430050005300200061006E0064002000740068006500200052006500" +
            "6C00790069006E00670020005000610072007400790020004100670072006500" +
            "65006D0065006E00740020007700680069006300680020006C0069006D006900" +
            "740020006C0069006100620069006C00690074007900200061006E0064002000" +
            "610072006500200069006E0063006F00720070006F0072006100740065006400" +
            "2000680065007200650069006E00200062007900200072006500660065007200" +
            "65006E00630065002E300B06096086480186FD6C0315301F0603551D23041830" +
            "168014F4B6E1201DFE29AED2E461A5B2A225B2C817356E301D0603551D0E0416" +
            "0414E1A7324AEE0121287D54D5F207926EB4070F3D8730710603551D1F046A30" +
            "683032A030A02E862C687474703A2F2F63726C332E64696769636572742E636F" +
            "6D2F736861322D617373757265642D74732E63726C3032A030A02E862C687474" +
            "703A2F2F63726C342E64696769636572742E636F6D2F736861322D6173737572" +
            "65642D74732E63726C30818506082B0601050507010104793077302406082B06" +
            "0105050730018618687474703A2F2F6F6373702E64696769636572742E636F6D" +
            "304F06082B060105050730028643687474703A2F2F636163657274732E646967" +
            "69636572742E636F6D2F44696769436572745348413241737375726564494454" +
            "696D657374616D70696E6743412E637274300D06092A864886F70D01010B0500" +
            "03820101001EF0418232AEEDF1B43513DC50C2D597AE22229D0E0EAF33D34CFD" +
            "7CBF6F0111A79465225CC622A1C889526B9A8C735CD95E3F32DE16604C8B36FD" +
            "31990ABDC184B78D1DEF8926130556F347CD475BAD84B238AF6A23B545E31E88" +
            "324680D2B7A69922FDC178CFF58BD80C8C0509EE44E680D56D70CC9F531E27DD" +
            "2A48DEDA9365AD6E65A399A7C2400E73CC584F8F4528E5BC9C88E628CE605D2D" +
            "255D8B732EA50D5B51DA9A4EFF50058928DAF278BBD258788D44A7AC3A009178" +
            "69896404D35D96DF2ABFF9A54C2C93FFE68ADD82ACF1D2B3A2869AC15589566A" +
            "473FFAD6339543358905785A3A69DA22B80443D36F6835367A143E45E9986486" +
            "0F130C264A3182024D308202490201013081863072310B300906035504061302" +
            "555331153013060355040A130C446967694365727420496E6331193017060355" +
            "040B13107777772E64696769636572742E636F6D3131302F0603550403132844" +
            "69676943657274205348413220417373757265642049442054696D657374616D" +
            "70696E67204341021009C0FC46C8044213B5598BAF284F4E41300D0609608648" +
            "0165030402010500A08198301A06092A864886F70D010903310D060B2A864886" +
            "F70D0109100104301C06092A864886F70D010905310F170D3138303632333139" +
            "303132305A302F06092A864886F70D01090431220420DAE824F466CED9AD07AC" +
            "66897D7489E25E678911AA3E85BBBAB76F93DF77FFCA302B060B2A864886F70D" +
            "010910020C311C301A301830160414400191475C98891DEBA104AF47091B5EB6" +
            "D4CBCB300D06092A864886F70D010101050004820100451DCB791B5E31865B9F" +
            "697B8A790C4DB1867261B10EDCEB65F8AD03350D8D25D89012E8A9C46D3DA436" +
            "04F222C645A8C4C7658D0C5C31A473A09FE314D049A886001CAC3C2C5B106BD7" +
            "1D02785A3BD02E4171E70A9AA1C7EC5BDB7D1B5A2F3A5F6EF92EE3296CB8FC52" +
            "D7ABF3ADB91F2BF30A2D0DE9713CFE5CF203B6D3F0F51897DCB075E8AAB8CB96" +
            "1F9505656CA73C8935E29A71C09BE3AA0BA1326FA295B6686771D6431BB2DAB3" +
            "8DECE164DE7487945C09DC0BF525086256C4D22CC71C70DD886BDD2EEF609631" +
            "771EF1E49426819679F9C080815B27AD93090D56DBF566D585693B27B5FDA70B" +
            "18401534B87B522B7B0AA98E3F31DFDD64FDA0DE9471").HexToByteArray();

        internal static readonly byte[] MD5WithRSADigestAlgorithm = (
            "308204BB06092A864886F70D010702A08204AC308204A8020101310D300B0609" +
            "2A864886F70D010104301206092A864886F70D010701A0050403010203A08203" +
            "223082031E30820206A003020102020900DCE4D72A7F3CA58E300D06092A8648" +
            "86F70D01010B050030243122302006035504030C195253413230343853686132" +
            "35364B65795472616E7366657231301E170D3139303130333230343133345A17" +
            "0D3238313233313230343133345A30243122302006035504030C195253413230" +
            "34385368613235364B65795472616E736665723130820122300D06092A864886" +
            "F70D01010105000382010F003082010A0282010100BF8495EF84C91EB42AE919" +
            "6C52AA2C6E22A258C4D76E09E93980D81FAFE6977FE1A3C8B4E9CA414405A1A4" +
            "DB6D549A4849189D568AA639F10FA556F55377C8FDE8EC33CCDC833C4C5B8B36" +
            "275E8F1899443871B7363F204A28268BD9E890FDD488B17BC49890F78FAB5D8A" +
            "5927B69655760D400A30E0EE87DD38B4168DF6814596E0EFAAFB2F9CC134664B" +
            "B8629E65AA107847B4416BD57F0B701C30E9A160AA589646E981815D9823201A" +
            "E0EE601208C51C08D3E50B1FAB1560802916E678310C03E3CFF3BCC039F98D5B" +
            "254E2BAF027AE9F78FAEF72C2D93075ECA64B5AD3E92348068C4DCB8CAD0BAD9" +
            "54A7D04FCE3AFF94EF8CF282095DB75A25A4114F8F0203010001A3533051301D" +
            "0603551D0E04160414FEDFD2558B95E705584A50008ED0D1683AEE840E301F06" +
            "03551D23041830168014FEDFD2558B95E705584A50008ED0D1683AEE840E300F" +
            "0603551D130101FF040530030101FF300D06092A864886F70D01010B05000382" +
            "01010031C46689DB724A1A8D268EAA7139EFCF8A31496F15F325A19349547114" +
            "D7992EAD5FA0BD6FFD5539B1639F0CF8F975C5E9EBED8A048289C782EC7E0342" +
            "F339F53E811CC6A6864173502388875964F0C50D84ABA9228DD2E3DDD3339374" +
            "B76A6826F4F848A403ADDA2E095974232B084C8672B12C18D675D81ED8FA4B15" +
            "A1C7ABC73F9CCB5BC25716EEB9D2C97F8EBCB9D4824FB23C7FBC78E1D3B9B1B3" +
            "749F6C0981B2B6CC81DC723AA5CC589346DCECA222C78F68FE80563E12282F7D" +
            "F936A161163F86A9EADBA2E180A906DE73AEBEEA0775F6AB9AE97C71F26E187E" +
            "3D8CB50C8137663C99D440877D5A5E17DF9785976E9031DD96134A07899281B6" +
            "B64E3F3182015830820154020101303130243122302006035504030C19525341" +
            "323034385368613235364B65795472616E7366657231020900DCE4D72A7F3CA5" +
            "8E300B06092A864886F70D010104300B06092A864886F70D010101048201005C" +
            "5174F8B1C6EF0A555CDBA272DF76BB798CB4170D361E4238C3A260748598B7C9" +
            "561605640BCAFC23C66C45A99E54A25854C5379CE2D26DECABDB6C6AE1FDC79E" +
            "1BBFA5833640BD8C96AF66CD3379B83B8A53660FB2FA6C1BD8ACE986741FF73A" +
            "14C437B7AF8617854A8F9484D84E684255EB7C37CBCA51BBC81D8D7E9A6287A0" +
            "409AC5AE0FA7D6411F7B25BDFC49EC91BD664630741ECCB0C6F02FFC5B9256A3" +
            "8AC30FDB4A7C2253E372858DF5D2333BBE29CEBCE24A0FABB9E62B0E631CAB2F" +
            "7A9137FE16A0448BE17ACF55F630CD6A522D94A42D6AB79C82C9AAB0DE7284C5" +
            "F8F16BF845A7619AE6DEE8016D42EA13189B51E50440F18F52475453B6F899").HexToByteArray();

        internal static readonly byte[] SHA1WithRSADigestAlgorithm = (
            "308204BB06092A864886F70D010702A08204AC308204A8020101310D300B0609" +
            "2A864886F70D010105301206092A864886F70D010701A0050403010203A08203" +
            "223082031E30820206A003020102020900DCE4D72A7F3CA58E300D06092A8648" +
            "86F70D01010B050030243122302006035504030C195253413230343853686132" +
            "35364B65795472616E7366657231301E170D3139303130333230343133345A17" +
            "0D3238313233313230343133345A30243122302006035504030C195253413230" +
            "34385368613235364B65795472616E736665723130820122300D06092A864886" +
            "F70D01010105000382010F003082010A0282010100BF8495EF84C91EB42AE919" +
            "6C52AA2C6E22A258C4D76E09E93980D81FAFE6977FE1A3C8B4E9CA414405A1A4" +
            "DB6D549A4849189D568AA639F10FA556F55377C8FDE8EC33CCDC833C4C5B8B36" +
            "275E8F1899443871B7363F204A28268BD9E890FDD488B17BC49890F78FAB5D8A" +
            "5927B69655760D400A30E0EE87DD38B4168DF6814596E0EFAAFB2F9CC134664B" +
            "B8629E65AA107847B4416BD57F0B701C30E9A160AA589646E981815D9823201A" +
            "E0EE601208C51C08D3E50B1FAB1560802916E678310C03E3CFF3BCC039F98D5B" +
            "254E2BAF027AE9F78FAEF72C2D93075ECA64B5AD3E92348068C4DCB8CAD0BAD9" +
            "54A7D04FCE3AFF94EF8CF282095DB75A25A4114F8F0203010001A3533051301D" +
            "0603551D0E04160414FEDFD2558B95E705584A50008ED0D1683AEE840E301F06" +
            "03551D23041830168014FEDFD2558B95E705584A50008ED0D1683AEE840E300F" +
            "0603551D130101FF040530030101FF300D06092A864886F70D01010B05000382" +
            "01010031C46689DB724A1A8D268EAA7139EFCF8A31496F15F325A19349547114" +
            "D7992EAD5FA0BD6FFD5539B1639F0CF8F975C5E9EBED8A048289C782EC7E0342" +
            "F339F53E811CC6A6864173502388875964F0C50D84ABA9228DD2E3DDD3339374" +
            "B76A6826F4F848A403ADDA2E095974232B084C8672B12C18D675D81ED8FA4B15" +
            "A1C7ABC73F9CCB5BC25716EEB9D2C97F8EBCB9D4824FB23C7FBC78E1D3B9B1B3" +
            "749F6C0981B2B6CC81DC723AA5CC589346DCECA222C78F68FE80563E12282F7D" +
            "F936A161163F86A9EADBA2E180A906DE73AEBEEA0775F6AB9AE97C71F26E187E" +
            "3D8CB50C8137663C99D440877D5A5E17DF9785976E9031DD96134A07899281B6" +
            "B64E3F3182015830820154020101303130243122302006035504030C19525341" +
            "323034385368613235364B65795472616E7366657231020900DCE4D72A7F3CA5" +
            "8E300B06092A864886F70D010105300B06092A864886F70D010101048201002A" +
            "1EAADFA508990F3395BCCBCB5ECB9CDFF236B94306481C3F7EB6C0C0189FEDC0" +
            "3AF8DA8BB0474F7D10DD87A81661D8A703FF7D695A45BCC25D8185CF9C5EF39B" +
            "C0D582ABD3392B19D9A29BC4629FC401A97B2135C73F2D356B369EA18D864C3E" +
            "2CE77EAF0B1B71244D3D5FE442E6C92E2A0FB24EFF7EC8FD5EBB248870FDBDC5" +
            "FA8F458DB3C2159234BCF1FCDBCCB232CDEF7A7208C51A7DD4AC5AD01613B603" +
            "B11E5E1D7509286B92C701C4538D44DC3487656F9E25BA8C5CF6FBD7F532B749" +
            "0FBBBC5F4FC9BA6FFCB3910E85295C9C7DD36CF5AC3B3A1C425FAD84EA99CBE0" +
            "962CE8F7AAED44AA202273A3B5A2A7D8D98978642EE321703F9AB8768BECD2").HexToByteArray();

        internal static readonly byte[] SHA256WithRSADigestAlgorithm = (
            "308204BB06092A864886F70D010702A08204AC308204A8020101310D300B0609" +
            "2A864886F70D01010B301206092A864886F70D010701A0050403010203A08203" +
            "223082031E30820206A003020102020900DCE4D72A7F3CA58E300D06092A8648" +
            "86F70D01010B050030243122302006035504030C195253413230343853686132" +
            "35364B65795472616E7366657231301E170D3139303130333230343133345A17" +
            "0D3238313233313230343133345A30243122302006035504030C195253413230" +
            "34385368613235364B65795472616E736665723130820122300D06092A864886" +
            "F70D01010105000382010F003082010A0282010100BF8495EF84C91EB42AE919" +
            "6C52AA2C6E22A258C4D76E09E93980D81FAFE6977FE1A3C8B4E9CA414405A1A4" +
            "DB6D549A4849189D568AA639F10FA556F55377C8FDE8EC33CCDC833C4C5B8B36" +
            "275E8F1899443871B7363F204A28268BD9E890FDD488B17BC49890F78FAB5D8A" +
            "5927B69655760D400A30E0EE87DD38B4168DF6814596E0EFAAFB2F9CC134664B" +
            "B8629E65AA107847B4416BD57F0B701C30E9A160AA589646E981815D9823201A" +
            "E0EE601208C51C08D3E50B1FAB1560802916E678310C03E3CFF3BCC039F98D5B" +
            "254E2BAF027AE9F78FAEF72C2D93075ECA64B5AD3E92348068C4DCB8CAD0BAD9" +
            "54A7D04FCE3AFF94EF8CF282095DB75A25A4114F8F0203010001A3533051301D" +
            "0603551D0E04160414FEDFD2558B95E705584A50008ED0D1683AEE840E301F06" +
            "03551D23041830168014FEDFD2558B95E705584A50008ED0D1683AEE840E300F" +
            "0603551D130101FF040530030101FF300D06092A864886F70D01010B05000382" +
            "01010031C46689DB724A1A8D268EAA7139EFCF8A31496F15F325A19349547114" +
            "D7992EAD5FA0BD6FFD5539B1639F0CF8F975C5E9EBED8A048289C782EC7E0342" +
            "F339F53E811CC6A6864173502388875964F0C50D84ABA9228DD2E3DDD3339374" +
            "B76A6826F4F848A403ADDA2E095974232B084C8672B12C18D675D81ED8FA4B15" +
            "A1C7ABC73F9CCB5BC25716EEB9D2C97F8EBCB9D4824FB23C7FBC78E1D3B9B1B3" +
            "749F6C0981B2B6CC81DC723AA5CC589346DCECA222C78F68FE80563E12282F7D" +
            "F936A161163F86A9EADBA2E180A906DE73AEBEEA0775F6AB9AE97C71F26E187E" +
            "3D8CB50C8137663C99D440877D5A5E17DF9785976E9031DD96134A07899281B6" +
            "B64E3F3182015830820154020101303130243122302006035504030C19525341" +
            "323034385368613235364B65795472616E7366657231020900DCE4D72A7F3CA5" +
            "8E300B06092A864886F70D01010B300B06092A864886F70D0101010482010054" +
            "5BAEC400A8F0EC436BDCE64B2FB22E8332C17538C9BBB72FCCF7C1F539A859A3" +
            "7E96754CA81AE81B24F937E1601681EB2315E632A76162B87C33403534DC64F0" +
            "E7DC8999BA05E7E030505B538124BB13FEC099F16372AE61DB75464DDE663ACE" +
            "10F136A6DA45CD988866B3630DEC1288AFA0B1864B93EE2B0713BA6C745A6775" +
            "061F19A1960218564478A814054EF844A6EF2C0ABFA5A854E60E1F275D7FB42A" +
            "0A4844DFB480AE88ABC83401DB59B333612AD2AAC6F5BB24FC09460D22499257" +
            "43F59476A8EC6E9E7E6C423F6574034B371F4562F347D10981141D9C7C36F47F" +
            "54882AD0FE5C78487C1F96785BD6BEBB958F5C140F86DFCE6F1E03B0AAE8C5").HexToByteArray();

        internal static readonly byte[] SHA384WithRSADigestAlgorithm = (
            "308204BB06092A864886F70D010702A08204AC308204A8020101310D300B0609" +
            "2A864886F70D01010C301206092A864886F70D010701A0050403010203A08203" +
            "223082031E30820206A003020102020900DCE4D72A7F3CA58E300D06092A8648" +
            "86F70D01010B050030243122302006035504030C195253413230343853686132" +
            "35364B65795472616E7366657231301E170D3139303130333230343133345A17" +
            "0D3238313233313230343133345A30243122302006035504030C195253413230" +
            "34385368613235364B65795472616E736665723130820122300D06092A864886" +
            "F70D01010105000382010F003082010A0282010100BF8495EF84C91EB42AE919" +
            "6C52AA2C6E22A258C4D76E09E93980D81FAFE6977FE1A3C8B4E9CA414405A1A4" +
            "DB6D549A4849189D568AA639F10FA556F55377C8FDE8EC33CCDC833C4C5B8B36" +
            "275E8F1899443871B7363F204A28268BD9E890FDD488B17BC49890F78FAB5D8A" +
            "5927B69655760D400A30E0EE87DD38B4168DF6814596E0EFAAFB2F9CC134664B" +
            "B8629E65AA107847B4416BD57F0B701C30E9A160AA589646E981815D9823201A" +
            "E0EE601208C51C08D3E50B1FAB1560802916E678310C03E3CFF3BCC039F98D5B" +
            "254E2BAF027AE9F78FAEF72C2D93075ECA64B5AD3E92348068C4DCB8CAD0BAD9" +
            "54A7D04FCE3AFF94EF8CF282095DB75A25A4114F8F0203010001A3533051301D" +
            "0603551D0E04160414FEDFD2558B95E705584A50008ED0D1683AEE840E301F06" +
            "03551D23041830168014FEDFD2558B95E705584A50008ED0D1683AEE840E300F" +
            "0603551D130101FF040530030101FF300D06092A864886F70D01010B05000382" +
            "01010031C46689DB724A1A8D268EAA7139EFCF8A31496F15F325A19349547114" +
            "D7992EAD5FA0BD6FFD5539B1639F0CF8F975C5E9EBED8A048289C782EC7E0342" +
            "F339F53E811CC6A6864173502388875964F0C50D84ABA9228DD2E3DDD3339374" +
            "B76A6826F4F848A403ADDA2E095974232B084C8672B12C18D675D81ED8FA4B15" +
            "A1C7ABC73F9CCB5BC25716EEB9D2C97F8EBCB9D4824FB23C7FBC78E1D3B9B1B3" +
            "749F6C0981B2B6CC81DC723AA5CC589346DCECA222C78F68FE80563E12282F7D" +
            "F936A161163F86A9EADBA2E180A906DE73AEBEEA0775F6AB9AE97C71F26E187E" +
            "3D8CB50C8137663C99D440877D5A5E17DF9785976E9031DD96134A07899281B6" +
            "B64E3F3182015830820154020101303130243122302006035504030C19525341" +
            "323034385368613235364B65795472616E7366657231020900DCE4D72A7F3CA5" +
            "8E300B06092A864886F70D01010C300B06092A864886F70D0101010482010066" +
            "7739542DAC383580DD47E88B092FA77F8C7C2AF3C71E1E532D0CE6CA91FA27CF" +
            "A30F89AE9ACB3BD2ED666D8C7BE83FADEBC22847A8D12465AA5009967365E089" +
            "C1177FC71F80EF8EFFC0EC109368517EFEE26C989AAD2C7DA4C9DB52E9349F84" +
            "2EB3CFEA3E66338565715124D0980E6B27619B9F1CB8B1FA84946E034CDBDE43" +
            "18241EFC89D5E90F9B4434E4452827C00D01803C9E40B1E98D97E822AE549B32" +
            "6E547F8E8176DC3BAB9C9B2B9A87AC7EC5E0B2E5135C6FC9EC2B8ACC8F9D152D" +
            "A729967FAE63AEA42730A49E2740C9064DAA8F981E27DAB42D142FACE3D0261D" +
            "45C46FA632B0D7FFFC6F9EAC26E5EFC765D01A9C739C08D10775536488BF0D").HexToByteArray();

        internal static readonly byte[] SHA512WithRSADigestAlgorithm = (
            "308204BB06092A864886F70D010702A08204AC308204A8020101310D300B0609" +
            "2A864886F70D01010D301206092A864886F70D010701A0050403010203A08203" +
            "223082031E30820206A003020102020900DCE4D72A7F3CA58E300D06092A8648" +
            "86F70D01010B050030243122302006035504030C195253413230343853686132" +
            "35364B65795472616E7366657231301E170D3139303130333230343133345A17" +
            "0D3238313233313230343133345A30243122302006035504030C195253413230" +
            "34385368613235364B65795472616E736665723130820122300D06092A864886" +
            "F70D01010105000382010F003082010A0282010100BF8495EF84C91EB42AE919" +
            "6C52AA2C6E22A258C4D76E09E93980D81FAFE6977FE1A3C8B4E9CA414405A1A4" +
            "DB6D549A4849189D568AA639F10FA556F55377C8FDE8EC33CCDC833C4C5B8B36" +
            "275E8F1899443871B7363F204A28268BD9E890FDD488B17BC49890F78FAB5D8A" +
            "5927B69655760D400A30E0EE87DD38B4168DF6814596E0EFAAFB2F9CC134664B" +
            "B8629E65AA107847B4416BD57F0B701C30E9A160AA589646E981815D9823201A" +
            "E0EE601208C51C08D3E50B1FAB1560802916E678310C03E3CFF3BCC039F98D5B" +
            "254E2BAF027AE9F78FAEF72C2D93075ECA64B5AD3E92348068C4DCB8CAD0BAD9" +
            "54A7D04FCE3AFF94EF8CF282095DB75A25A4114F8F0203010001A3533051301D" +
            "0603551D0E04160414FEDFD2558B95E705584A50008ED0D1683AEE840E301F06" +
            "03551D23041830168014FEDFD2558B95E705584A50008ED0D1683AEE840E300F" +
            "0603551D130101FF040530030101FF300D06092A864886F70D01010B05000382" +
            "01010031C46689DB724A1A8D268EAA7139EFCF8A31496F15F325A19349547114" +
            "D7992EAD5FA0BD6FFD5539B1639F0CF8F975C5E9EBED8A048289C782EC7E0342" +
            "F339F53E811CC6A6864173502388875964F0C50D84ABA9228DD2E3DDD3339374" +
            "B76A6826F4F848A403ADDA2E095974232B084C8672B12C18D675D81ED8FA4B15" +
            "A1C7ABC73F9CCB5BC25716EEB9D2C97F8EBCB9D4824FB23C7FBC78E1D3B9B1B3" +
            "749F6C0981B2B6CC81DC723AA5CC589346DCECA222C78F68FE80563E12282F7D" +
            "F936A161163F86A9EADBA2E180A906DE73AEBEEA0775F6AB9AE97C71F26E187E" +
            "3D8CB50C8137663C99D440877D5A5E17DF9785976E9031DD96134A07899281B6" +
            "B64E3F3182015830820154020101303130243122302006035504030C19525341" +
            "323034385368613235364B65795472616E7366657231020900DCE4D72A7F3CA5" +
            "8E300B06092A864886F70D01010D300B06092A864886F70D0101010482010045" +
            "B324E7BE9FD3B1279018B01CD177A55A47A5C811CA693ED6BB32C8A243B1404D" +
            "89DC4EAA348453CB07BF8B1A11DE1BAF6397502C41D384790C0A285D69C17006" +
            "4F49D5735045EAAFAE43998D2660C6D15AEA1433FDFB7C4A995BBA4231AEB66B" +
            "81E29A035973CD148906A2FA16D6E551174D232E9DD738FE18D09E21517753A3" +
            "DF2687B83E0F8B70912F68FCF12311593637A362A3525BE875172BB402D998AE" +
            "D54F085DA892DD967FFE8906E23C0EF7E1C2ED89BB3986871BA997E725876535" +
            "119CFE807690BC067BA5781231E68A61CDE4D66AA67CD31981CF92275FD3485C" +
            "01CBA2BA29AED480F7AD26B361B39EE3622D1A705238E5C1A8BE2BBE921804").HexToByteArray();

        internal static readonly byte[] SHA256ECDSAWithRsaSha256DigestIdentifier = (
            "3082023006092A864886F70D010702A08202213082021D020101310D300B0609" +
            "2A864886F70D01010B301206092A864886F70D010701A0050403010203A08201" +
            "5C308201583081FFA003020102021035428F3B3C5107AD49E776D6E74C4DC830" +
            "0A06082A8648CE3D04030230153113301106035504030C0A4543445341205465" +
            "7374301E170D3135303530313030333730335A170D3136303530313030353730" +
            "335A30153113301106035504030C0A454344534120546573743059301306072A" +
            "8648CE3D020106082A8648CE3D030107034200047590F69CA114E92927E034C9" +
            "97B7C882A8C992AC00CEFB4EB831901536F291E1B515263BCD20E1EA32496FDA" +
            "C84E2D8D1B703266A9088F6EAF652549D9BB63D5A331302F300E0603551D0F01" +
            "01FF040403020388301D0603551D0E0416041411218A92C5EB12273B3C5CCFB8" +
            "220CCCFDF387DB300A06082A8648CE3D040302034800304502201AFE595E19F1" +
            "AE4B6A4B231E8851926438C55B5DDE632E6ADF13C1023A65898E022100CBDF43" +
            "4FDD197D8B594E8026E44263BADE773C2BEBD060CC4109484A498E7C7E318194" +
            "308191020101302930153113301106035504030C0A4543445341205465737402" +
            "1035428F3B3C5107AD49E776D6E74C4DC8300B06092A864886F70D01010B300A" +
            "06082A8648CE3D0403020448304602210087FAF07C8B9259F453A8EF63A1F2E9" +
            "36D6845255FE1109D84592E23E8E384611022100C8BA4D440C8AD0753CC43966" +
            "0CBB1A3FA93E7C1F8427BAB27150EB15F92D2928").HexToByteArray();

        internal static readonly byte[] SHA256withRSADigestAndSHA256WithRSASignatureAlgorithm = (
            "3082071B06092A864886F70D010702A082070C30820708020101310F300D0609" +
            "2A864886F70D01010B0500301606092A864886F70D010701A009040700010203" +
            "040506A08205223082051E30820406A00302010202100D85090F3FACFF0A9008" +
            "A12A9FB00A54300D06092A864886F70D01010B05003072310B30090603550406" +
            "1302555331153013060355040A130C446967694365727420496E633119301706" +
            "0355040B13107777772E64696769636572742E636F6D3131302F060355040313" +
            "2844696769436572742053484132204173737572656420494420436F64652053" +
            "69676E696E67204341301E170D3138303832393030303030305A170D31393039" +
            "30333132303030305A305B310B3009060355040613025553310B300906035504" +
            "0813025641311330110603550407130A416C6578616E64726961311430120603" +
            "55040A130B4B6576696E204A6F6E6573311430120603550403130B4B6576696E" +
            "204A6F6E657330820122300D06092A864886F70D01010105000382010F003082" +
            "010A0282010100F1F4542FF6CA57FBC44986EC816F07D1FD50BFD477C412D299" +
            "1C962D0A22194A4296BCD0751F47CE4932F73871277CE3CDD2C78157599C7A35" +
            "80CC96A11F7031E3A798F4BAA93988F0E4077D30316252B24337DB26914E1F77" +
            "9CB4979544514B0234E5388E936B195B91863B258F0C8951454D3668F0C4D456" +
            "A8497758D21C433626E46F2CFF5A7CC7945F788948998E5F8786E1E990E240BB" +
            "0780CD258F57761AFB5D42AD8E3D703C3126861E83F191ECE9F0B83221F96214" +
            "533B2A47977F43715FE501FBC4A4040839DD3EBCA8B67259A7DD0EA9EFAE2200" +
            "943EFB7D0404B8978B49A445849B5F6898B06269F427F30DBC8DB2FD7963943A" +
            "8C461760E6A4F30203010001A38201C5308201C1301F0603551D230418301680" +
            "145AC4B97B2A0AA3A5EA7103C060F92DF665750E58301D0603551D0E04160414" +
            "33795EB2D84BFAA3F96E5930F64EC6A94C6FD36A300E0603551D0F0101FF0404" +
            "0302078030130603551D25040C300A06082B0601050507030330770603551D1F" +
            "0470306E3035A033A031862F687474703A2F2F63726C332E6469676963657274" +
            "2E636F6D2F736861322D617373757265642D63732D67312E63726C3035A033A0" +
            "31862F687474703A2F2F63726C342E64696769636572742E636F6D2F73686132" +
            "2D617373757265642D63732D67312E63726C304C0603551D2004453043303706" +
            "096086480186FD6C0301302A302806082B06010505070201161C68747470733A" +
            "2F2F7777772E64696769636572742E636F6D2F4350533008060667810C010401" +
            "30818406082B0601050507010104783076302406082B06010505073001861868" +
            "7474703A2F2F6F6373702E64696769636572742E636F6D304E06082B06010505" +
            "0730028642687474703A2F2F636163657274732E64696769636572742E636F6D" +
            "2F446967694365727453484132417373757265644944436F64655369676E696E" +
            "6743412E637274300C0603551D130101FF04023000300D06092A864886F70D01" +
            "010B0500038201010045B9D9868E494BD02635D0E42DDE80B37A865C389CFDD9" +
            "9BFC9B62E2C169A73B5EABF282607439EFF5C61630886DEB63415B53683446A7" +
            "3041686C326BA35FF0029FEF603D7C80FA0177A4DE35013529B01F759FD50414" +
            "79BDBB6B93B18144CB14E431BC144146848EF8ADB0E28952EAD1BB49E8547FFE" +
            "9934817036338B20C4E0B9D7C6A4E5BE3D57157F21904A5C864946313EA6B7D9" +
            "50EE0235B5D2CD01490AD2B2A1AB5F66EC8986D64A1D9D239C131E09E5CA1C02" +
            "A75F2D7EC07E4C858856A6A58AB94DEAC8B3D3A5BBF492EE2463B156E6A0660B" +
            "B452E35922D00456F0DEE0ED15A8BF8FFF31008756B14EEE0AC14BCF19A3CD16" +
            "819DC990F5F45CDE21318201B2308201AE0201013081863072310B3009060355" +
            "04061302555331153013060355040A130C446967694365727420496E63311930" +
            "17060355040B13107777772E64696769636572742E636F6D3131302F06035504" +
            "03132844696769436572742053484132204173737572656420494420436F6465" +
            "205369676E696E6720434102100D85090F3FACFF0A9008A12A9FB00A54300D06" +
            "092A864886F70D01010B0500300D06092A864886F70D01010B050004820100E2" +
            "980C5A30EC00729D1CFA826D7A65B43FF6806B5E0ABA23A78E4F1CAA3F6436EF" +
            "00941C6947A9B8F20D0757B5346CF640AA217F7361BEEFF2BC997FB1D3597BF3" +
            "D7457BD4A94062FB03660F9D86710BE2FC99876A848251F4965E1B16192714C8" +
            "F9788C09CCDE83603ADC919297BA496E921B95F3BD9554A873E09912640FCFAA" +
            "D9DD1441D1851E637031D390C038223AE64B048E806462DDBAC98C156BE2EE47" +
            "2B78166BDB1612848B535ADC3F0E7BE52991A17F48AFDCCC1698A236BA338930" +
            "50EBAAC4460DAA35185C16670F597E0E6E0CB0AA83F51AAEF452F3367DD9350A" +
            "8A49A5A8F79DF8E921303AB5D6646A482F0F59D9980310E1AE3EE8D77CB857").HexToByteArray();

        internal static readonly byte[] TstWithAttributeCertificate = (
            "308212E306092A864886F70D010702A08212D4308212D0020103310F300D0609" +
            "608648016503040201050030820159060B2A864886F70D0109100104A0820148" +
            "0482014430820140020101060A2B0601040184590A03013031300D0609608648" +
            "01650304020105000420C61245B435391DCDD1EFF021681FA2A46E69410A9E23" +
            "09F1B1736BB7C2BB504402066148B8B838B11813323032313130303730343230" +
            "30372E3037345A3004800201F4A081D8A481D53081D2310B3009060355040613" +
            "025553311330110603550408130A57617368696E67746F6E3110300E06035504" +
            "0713075265646D6F6E64311E301C060355040A13154D6963726F736F66742043" +
            "6F72706F726174696F6E312D302B060355040B13244D6963726F736F66742049" +
            "72656C616E64204F7065726174696F6E73204C696D6974656431263024060355" +
            "040B131D5468616C6573205453532045534E3A303834322D344245362D433239" +
            "41312530230603550403131C4D6963726F736F66742054696D652D5374616D70" +
            "2053657276696365A0820E4A308204F9308203E1A00302010202133300000139" +
            "CCE8E8438BF034E1000000000139300D06092A864886F70D01010B0500307C31" +
            "0B3009060355040613025553311330110603550408130A57617368696E67746F" +
            "6E3110300E060355040713075265646D6F6E64311E301C060355040A13154D69" +
            "63726F736F667420436F72706F726174696F6E312630240603550403131D4D69" +
            "63726F736F66742054696D652D5374616D70205043412032303130301E170D32" +
            "30313031353137323832315A170D3232303131323137323832315A3081D2310B" +
            "3009060355040613025553311330110603550408130A57617368696E67746F6E" +
            "3110300E060355040713075265646D6F6E64311E301C060355040A13154D6963" +
            "726F736F667420436F72706F726174696F6E312D302B060355040B13244D6963" +
            "726F736F6674204972656C616E64204F7065726174696F6E73204C696D697465" +
            "6431263024060355040B131D5468616C6573205453532045534E3A303834322D" +
            "344245362D43323941312530230603550403131C4D6963726F736F6674205469" +
            "6D652D5374616D70205365727669636530820122300D06092A864886F70D0101" +
            "0105000382010F003082010A0282010100DA13F98CE3259325A18EB32A06FCA2" +
            "78F85A1F98374F7F50C1DBBA1EA02759C2E5CFDFD92A80BF405CF8E606F469DD" +
            "7822FA856859B627AA3EBE185B7F8A1024E1C34DC8D95C12904DB413FF0E8B09" +
            "6BDD0979C5425D9AF2711DE3612613BAF145598ABA66C7A04295F90C1874673B" +
            "D2AE4EF0CD9892A27F4AD72B16116D9A117172F559FA08386B3CEF13CEFEB282" +
            "33615EB3A8CD8A58EFD34B2F597A88B95F84286D5E802AEC091F4F32499DA540" +
            "15F39DDF2BC03CF2EBE058A895E29BE6FE57EEC4DFBE356FA710F5F98E340A47" +
            "2E1906AA8D3BFC1D783641AAA8EF7BF210235ED5684292A32AEEB2C57CECB294" +
            "278F4AE57EC57F1F2496FE5FA2CF1ECA4F0203010001A382011B30820117301D" +
            "0603551D0E04160414999E9B867254C299DA928AB19F46F6502D060CE9301F06" +
            "03551D23041830168014D5633A5C8A3190F3437B7C461BC533685A856D553056" +
            "0603551D1F044F304D304BA049A0478645687474703A2F2F63726C2E6D696372" +
            "6F736F66742E636F6D2F706B692F63726C2F70726F64756374732F4D69635469" +
            "6D5374615043415F323031302D30372D30312E63726C305A06082B0601050507" +
            "0101044E304C304A06082B06010505073002863E687474703A2F2F7777772E6D" +
            "6963726F736F66742E636F6D2F706B692F63657274732F4D696354696D537461" +
            "5043415F323031302D30372D30312E637274300C0603551D130101FF04023000" +
            "30130603551D25040C300A06082B06010505070308300D06092A864886F70D01" +
            "010B05000382010100585C286AF3FF371A85CDC15AED438288BF100DB126FBBA" +
            "F6118893A16D20F1D758C8AE566B514077FCA7243D6AC9CFD9C71F473FDD32A9" +
            "7293FABCF8835C08B1049694A09BDD71B821586C584084B26DA28CDFDFC687E6" +
            "B63667158DF5BB831249C97E21A22BA43EF2490235699B5A3D83C51C0417C21F" +
            "0708C5EC43E160381D8A52B3E3CEAEC21828B28AF51AE0D68A56231A830DAB79" +
            "6BC0322463FF37294A49868AE5A6205D908344B159027F3C842E67BD0B7B5A99" +
            "9B4C497C6B519C17E96ADFDDD498100AB2DFA3C9649AE2C13C21BF38C87AA0FC" +
            "78ACD4521DEE7C4DBF1231006DA6C1842A36FAD528CA74FA4C026D8DD8AF6B88" +
            "3B3FBDFF550AD270C73082067130820459A003020102020A6109812A00000000" +
            "0002300D06092A864886F70D01010B0500308188310B30090603550406130255" +
            "53311330110603550408130A57617368696E67746F6E3110300E060355040713" +
            "075265646D6F6E64311E301C060355040A13154D6963726F736F667420436F72" +
            "706F726174696F6E31323030060355040313294D6963726F736F667420526F6F" +
            "7420436572746966696361746520417574686F726974792032303130301E170D" +
            "3130303730313231333635355A170D3235303730313231343635355A307C310B" +
            "3009060355040613025553311330110603550408130A57617368696E67746F6E" +
            "3110300E060355040713075265646D6F6E64311E301C060355040A13154D6963" +
            "726F736F667420436F72706F726174696F6E312630240603550403131D4D6963" +
            "726F736F66742054696D652D5374616D7020504341203230313030820122300D" +
            "06092A864886F70D01010105000382010F003082010A0282010100A91D0DBC77" +
            "118A3A20ECFC1397F5FA7F69946B745410D5A50A008285FBED7C684B2C5FC5C3" +
            "E561C276B73E662B5BF015532704311F411B1A951DCE09138E7C613059B13044" +
            "0FF160888454430CD74DB83808B342DD93ACD67330572682A3450DD0EAF54781" +
            "CDBF246032586046F258478632841E746167915F8154B1CF934C92C1C4A65DD1" +
            "61136E28C61AF98680BBDF61FC46C1271D246712721A218AAF4B64895062B15D" +
            "FD771F3DF05775ACBD8A424D4051D10F9C063E677FF566C00396447EEFD04BFD" +
            "6EE59ACAB1A8F27A2A0A31F0DA4E0691B6880835E8781CB0E999CD3CE72F44BA" +
            "A7F4DC64BDA401C120099378CDFCBCC0C9445D5E169C01054F224D0203010001" +
            "A38201E6308201E2301006092B06010401823715010403020100301D0603551D" +
            "0E04160414D5633A5C8A3190F3437B7C461BC533685A856D55301906092B0601" +
            "040182371402040C1E0A00530075006200430041300B0603551D0F0404030201" +
            "86300F0603551D130101FF040530030101FF301F0603551D23041830168014D5" +
            "F656CB8FE8A25C6268D13D94905BD7CE9A18C430560603551D1F044F304D304B" +
            "A049A0478645687474703A2F2F63726C2E6D6963726F736F66742E636F6D2F70" +
            "6B692F63726C2F70726F64756374732F4D6963526F6F4365724175745F323031" +
            "302D30362D32332E63726C305A06082B06010505070101044E304C304A06082B" +
            "06010505073002863E687474703A2F2F7777772E6D6963726F736F66742E636F" +
            "6D2F706B692F63657274732F4D6963526F6F4365724175745F323031302D3036" +
            "2D32332E6372743081A00603551D200101FF04819530819230818F06092B0601" +
            "040182372E03308181303D06082B060105050702011631687474703A2F2F7777" +
            "772E6D6963726F736F66742E636F6D2F504B492F646F63732F4350532F646566" +
            "61756C742E68746D304006082B0601050507020230341E32201D004C00650067" +
            "0061006C005F0050006F006C006900630079005F00530074006100740065006D" +
            "0065006E0074002E201D300D06092A864886F70D01010B0500038202010007E6" +
            "88510DE2C6E0983F8171033D9DA3A1216FB3EBA6CCF531BECF05E2A9FEFA576D" +
            "1930B3C2C566C96ADFF5E7F078BDC7A89E25E3F9BCED6B5457082B51824412FB" +
            "B9538CCCF460128A76CC4040419BDC5C17FF5CF95E17359824564B74EF4210C8" +
            "AFBF7FC67FF2377D5A3F1CF299794A915200AF380F17F52F798165D9A9B56BE4" +
            "C7CEF6CA7A006F4B304424223CCFED03A5968F5929BCB6FD04E1709F324A27FD" +
            "55AF2FFEB6E58E33BB625F9ADB5740E9F1CE9966908CFF6A627FDDC54A0B9126" +
            "E239EC194A71639D7B216DC39CA3A23CFA7F7D966A9078A66DD2E19CF91DFC38" +
            "D894F4C6A50A9686A4BD9E1AAE044283B8B5809B223820B525E564ECF7F4BF7E" +
            "6359250F7A2E395776A271AA068A0F8916BA61A711CB9AD80E479A80C5D0CDA7" +
            "D0EF7D83F0E13B7109DF5D7498220861DAB0501E6FBDF1E100DFE73107A4933A" +
            "F7654778E8F8A848ABF7DE727E616B6F77A981CBA709AC39BBECC6CBD882B472" +
            "CD1DF4B885011E80FB1B892A5439B25BDAC80D55997A87733B08E6982DEA8DE0" +
            "332E1229F5C02F542721F7C8AC4EDA28B8B1A9DB96B2A742A2C9CF19414DE086" +
            "F92A9AA3116630D3BB74324BDF637BF5998A2F1BC721AF59B5AEDC443C975071" +
            "D7A1D2C555E369DE57C1D1DE30C0FDCCE64DFB0DBF5D4FE99D1E19382FBCCF58" +
            "052EEF0DA05035DAEF09271CD5B37E351E08BADA36DBD35F8FDE74884912A182" +
            "02D43082023D02010130820100A181D8A481D53081D2310B3009060355040613" +
            "025553311330110603550408130A57617368696E67746F6E3110300E06035504" +
            "0713075265646D6F6E64311E301C060355040A13154D6963726F736F66742043" +
            "6F72706F726174696F6E312D302B060355040B13244D6963726F736F66742049" +
            "72656C616E64204F7065726174696F6E73204C696D6974656431263024060355" +
            "040B131D5468616C6573205453532045534E3A303834322D344245362D433239" +
            "41312530230603550403131C4D6963726F736F66742054696D652D5374616D70" +
            "2053657276696365A2230A0101300706052B0E03021A0315000D4D94FE1BDE96" +
            "848D743F1DEE4A04C6B3658C07A08183308180A47E307C310B30090603550406" +
            "13025553311330110603550408130A57617368696E67746F6E3110300E060355" +
            "040713075265646D6F6E64311E301C060355040A13154D6963726F736F667420" +
            "436F72706F726174696F6E312630240603550403131D4D6963726F736F667420" +
            "54696D652D5374616D70205043412032303130300D06092A864886F70D010105" +
            "0500020500E5084E973022180F32303231313030373030333433315A180F3230" +
            "3231313030383030333433315A3074303A060A2B0601040184590A0401312C30" +
            "2A300A020500E5084E97020100300702010002021DE630070201000202113630" +
            "0A020500E509A0170201003036060A2B0601040184590A040231283026300C06" +
            "0A2B0601040184590A0302A00A3008020100020307A120A10A30080201000203" +
            "0186A0300D06092A864886F70D0101050500038181002AE0DF2A01AAE13A86C0" +
            "2DD8ECA787327C8F04A7C13D47256E65C60676B8372EE46362CD35391B6B898E" +
            "DC1884082AAA7CF9B1CF40BE2C30146D0080ACB225C50B7C1FE94694732EDEEB" +
            "9EDE73DB7D8C0762CBDFABD3ACCC82DD3858AA16C3ED185A40A39E9676772099" +
            "346E3362621286651B06D9AD26D574F967F6C7EC335A3182030D308203090201" +
            "01308193307C310B3009060355040613025553311330110603550408130A5761" +
            "7368696E67746F6E3110300E060355040713075265646D6F6E64311E301C0603" +
            "55040A13154D6963726F736F667420436F72706F726174696F6E312630240603" +
            "550403131D4D6963726F736F66742054696D652D5374616D7020504341203230" +
            "313002133300000139CCE8E8438BF034E1000000000139300D06096086480165" +
            "030402010500A082014A301A06092A864886F70D010903310D060B2A864886F7" +
            "0D0109100104302F06092A864886F70D01090431220420EA37C01965AB5A8B08" +
            "C2A59F8EF2008C2B60F78E7F0385E2DD18A07C63433EF43081FA060B2A864886" +
            "F70D010910022F3181EA3081E73081E43081BD04203CA18EE438A3D7247B3142" +
            "B1E28105AE7C6A5527F39A7A9F26A6D4A0070FFC9F308198308180A47E307C31" +
            "0B3009060355040613025553311330110603550408130A57617368696E67746F" +
            "6E3110300E060355040713075265646D6F6E64311E301C060355040A13154D69" +
            "63726F736F667420436F72706F726174696F6E312630240603550403131D4D69" +
            "63726F736F66742054696D652D5374616D702050434120323031300213330000" +
            "0139CCE8E8438BF034E1000000000139302204204A6E54AC16FAF6985726E0CB" +
            "5480381C41888386AB1D1BECC241E6BCFCBDD45B300D06092A864886F70D0101" +
            "0B0500048201006957F018A5C13CC539EB266EC2725998F4AC0ED50CE1CAA696" +
            "CAE377977F45FB8030136A4038B49F7E5A6ADCF72FA3901C26B52815621F52A2" +
            "EB2ACFC423087ECD954F34C341342E31BDF30443CAC4F1297AB0181467B90604" +
            "DDCF11F9F8930290614E8AA9D868E56971E4F83427819F8A1A11ED87AD1D5469" +
            "7AFFE60A8D7B7877CBB1D09F46397769B9D0F8EB605DBFE8F05C47A3F8BC1F65" +
            "F9F63553F187194B35E971FEFFB00C9127BDFD0F7FCF2424BE2A108AC5C532EA" +
            "6AB46C76047604BA7DF071EE0137B73F4D9DF616095C8F1CA9EB925B6D4C6ABC" +
            "FFEB73A81903169C6B487B316AFC951F696831A4B29B9ABCEA7EBCD243A553D2" +
            "F8B44FDA1B0AC0").HexToByteArray();

        // produced with
        // echo -n hello | openssl cms -sign -signer cert.pfx -md sha3-256 -nodetach
        internal static readonly byte[] Rsa_Sha3_256_SignedDocument =(
            "308205C806092A864886F70D010702A08205B9308205B5020101310D300B" +
            "0609608648016503040208301406092A864886F70D010701A00704056865" +
            "6C6C6FA08203303082032C30820214A003020102020900E0D8AB6819D730" +
            "6E300D06092A864886F70D01010B05003038313630340603550403132D54" +
            "776F2074686F7573616E6420666F7274792065696768742062697473206F" +
            "662052534120676F6F646E657373301E170D313731313033323335313135" +
            "5A170D3138313130333233353131355A3038313630340603550403132D54" +
            "776F2074686F7573616E6420666F7274792065696768742062697473206F" +
            "662052534120676F6F646E65737330820122300D06092A864886F70D0101" +
            "0105000382010F003082010A028201010096C114A5898D09133EF859F89C" +
            "1D848BA8CB5258793E05B92D499C55EEFACE274BBBC26803FB813B9C11C6" +
            "898153CC1745DED2C4D2672F807F0B2D957BC4B65EBC9DDE26E2EA7B2A6F" +
            "E9A7C4D8BD1EF6032B8F0BB6AA33C8B57248B3D5E3901D8A38A283D7E25F" +
            "F8E6F522381EE5484234CFF7B30C174635418FA89E14C468AD89DCFCBBB5" +
            "35E5AF53510F9EA7F9DA8C1B53375B6DAB95A291439A5648726EE1012E41" +
            "388E100691642CF6917F5569D8351F2782F435A579014E8448EEA0C4AECA" +
            "FF2F476799D88457E2C8BCB56E5E128782B4FE26AFF0720D91D52CCAFE34" +
            "4255808F5271D09F784F787E8323182080915BE0AE15A71D66476D0F264D" +
            "D084F30203010001A3393037301D0603551D0E04160414745B5F12EF962E" +
            "84B897E246D399A2BADEA9C5AC30090603551D1304023000300B0603551D" +
            "0F040403020780300D06092A864886F70D01010B0500038201010087A15D" +
            "F37FBD6E9DED7A8FFF25E60B731F635469BA01DD14BC03B2A24D99EFD8B8" +
            "94E9493D63EC88C496CB04B33DF25222544F23D43F4023612C4D97B719C1" +
            "F9431E4DB7A580CDF66A3E5F0DAF89A267DD187ABFFB08361B1F79232376" +
            "AA5FC5AD384CC2F98FE36C1CEA0B943E1E3961190648889C8ABE8397A5A3" +
            "38843CBFB1D8B212BE46685ACE7B80475CC7C97FC0377936ABD5F664E9C0" +
            "9C463897726650711A1110FA9866BC1C278D95E5636AB96FAE95CCD67FD5" +
            "72A8C727E2C03E7B242457318BEC1BE52CA5BD9454A0A41140AE96ED1C56" +
            "D220D1FD5DD3B1B4FB2AA0E04FC94F7E3C7D476F298962245563953AD722" +
            "5EDCEAC8B8509E49292E62D8BF3182025530820251020101304530383136" +
            "30340603550403132D54776F2074686F7573616E6420666F727479206569" +
            "6768742062697473206F662052534120676F6F646E657373020900E0D8AB" +
            "6819D7306E300B0609608648016503040208A081E4301806092A864886F7" +
            "0D010903310B06092A864886F70D010701301C06092A864886F70D010905" +
            "310F170D3233303531373139333732315A302F06092A864886F70D010904" +
            "312204203338BE694F50C5F338814986CDF0686453A888B84F424D792AF4" +
            "B9202398F392307906092A864886F70D01090F316C306A300B0609608648" +
            "01650304012A300B0609608648016503040116300B060960864801650304" +
            "0102300A06082A864886F70D0307300E06082A864886F70D030202020080" +
            "300D06082A864886F70D0302020140300706052B0E030207300D06082A86" +
            "4886F70D0302020128300D06092A864886F70D0101010500048201002CDE" +
            "D12EA41BA5443CE742A804E754D00AA8A21B4B9022780DF59BD597EA44DD" +
            "85EEE42AA531A645CD11CAA58297190C06A8193CE7AF16B1FD8B9DF4E2FA" +
            "15024E8E845D0E3537577C6F27CEB3317D57781B618388FFB61926D24B1F" +
            "75E0CC460C52FFB43D8DDA1B7AFBCD76C00EECF99F7F7D2A02B19A055C9B" +
            "A5437ADE06537F4194F15AD0C5071F0275128B435B8883FE8E2E07F6B908" +
            "33B9FB56015FE57B3BEC08A5B9031AB8D372C662DCA6E3CDEA5D3F3FC76E" +
            "02F8E52045386B8D83D46C6300F5B54B2135F6392DD0B6A9C5531E7C9405" +
            "B48BF1DA5C5C7B9D6E00937673C7C97DB3B2DB6945D7C8F0F8E455AAF4DF" +
            "6F7D7A7726B09B6025ED6DE21864").HexToByteArray();

        // produced with
        // echo -n hello | openssl cms -sign -signer cert.pfx -md sha3-384 -nodetach
        internal static readonly byte[] Rsa_Sha3_384_SignedDocument = (
            "308205D806092A864886F70D010702A08205C9308205C5020101310D300B" +
            "0609608648016503040209301406092A864886F70D010701A00704056865" +
            "6C6C6FA08203303082032C30820214A003020102020900E0D8AB6819D730" +
            "6E300D06092A864886F70D01010B05003038313630340603550403132D54" +
            "776F2074686F7573616E6420666F7274792065696768742062697473206F" +
            "662052534120676F6F646E657373301E170D313731313033323335313135" +
            "5A170D3138313130333233353131355A3038313630340603550403132D54" +
            "776F2074686F7573616E6420666F7274792065696768742062697473206F" +
            "662052534120676F6F646E65737330820122300D06092A864886F70D0101" +
            "0105000382010F003082010A028201010096C114A5898D09133EF859F89C" +
            "1D848BA8CB5258793E05B92D499C55EEFACE274BBBC26803FB813B9C11C6" +
            "898153CC1745DED2C4D2672F807F0B2D957BC4B65EBC9DDE26E2EA7B2A6F" +
            "E9A7C4D8BD1EF6032B8F0BB6AA33C8B57248B3D5E3901D8A38A283D7E25F" +
            "F8E6F522381EE5484234CFF7B30C174635418FA89E14C468AD89DCFCBBB5" +
            "35E5AF53510F9EA7F9DA8C1B53375B6DAB95A291439A5648726EE1012E41" +
            "388E100691642CF6917F5569D8351F2782F435A579014E8448EEA0C4AECA" +
            "FF2F476799D88457E2C8BCB56E5E128782B4FE26AFF0720D91D52CCAFE34" +
            "4255808F5271D09F784F787E8323182080915BE0AE15A71D66476D0F264D" +
            "D084F30203010001A3393037301D0603551D0E04160414745B5F12EF962E" +
            "84B897E246D399A2BADEA9C5AC30090603551D1304023000300B0603551D" +
            "0F040403020780300D06092A864886F70D01010B0500038201010087A15D" +
            "F37FBD6E9DED7A8FFF25E60B731F635469BA01DD14BC03B2A24D99EFD8B8" +
            "94E9493D63EC88C496CB04B33DF25222544F23D43F4023612C4D97B719C1" +
            "F9431E4DB7A580CDF66A3E5F0DAF89A267DD187ABFFB08361B1F79232376" +
            "AA5FC5AD384CC2F98FE36C1CEA0B943E1E3961190648889C8ABE8397A5A3" +
            "38843CBFB1D8B212BE46685ACE7B80475CC7C97FC0377936ABD5F664E9C0" +
            "9C463897726650711A1110FA9866BC1C278D95E5636AB96FAE95CCD67FD5" +
            "72A8C727E2C03E7B242457318BEC1BE52CA5BD9454A0A41140AE96ED1C56" +
            "D220D1FD5DD3B1B4FB2AA0E04FC94F7E3C7D476F298962245563953AD722" +
            "5EDCEAC8B8509E49292E62D8BF3182026530820261020101304530383136" +
            "30340603550403132D54776F2074686F7573616E6420666F727479206569" +
            "6768742062697473206F662052534120676F6F646E657373020900E0D8AB" +
            "6819D7306E300B0609608648016503040209A081F4301806092A864886F7" +
            "0D010903310B06092A864886F70D010701301C06092A864886F70D010905" +
            "310F170D3233303531373139353133375A303F06092A864886F70D010904" +
            "31320430720AEA11019EF06440FBF05D87AA24680A2153DF3907B23631E7" +
            "177CE620FA1330FF07C0FDDEE54699A4C3EE0EE9D887307906092A864886" +
            "F70D01090F316C306A300B060960864801650304012A300B060960864801" +
            "6503040116300B0609608648016503040102300A06082A864886F70D0307" +
            "300E06082A864886F70D030202020080300D06082A864886F70D03020201" +
            "40300706052B0E030207300D06082A864886F70D0302020128300D06092A" +
            "864886F70D010101050004820100208BB078601C3AA70070F74E70D34C73" +
            "90CC21B06CB5566708C9287F191C08E71D476ABFBCE52AE21286D114746D" +
            "B1CD32F5D1738347305CEEC67332F4B9B2B272183719212562FADFE4AD81" +
            "1C257220EFB0E59C271D6006AA332175CAFEB295A53BC3148156D07312DB" +
            "B215307DE92BC00BD45B3E819CF15F50F05AD34EC855A1387BBE11AF6200" +
            "5E1AB04CE442BB9A14490D2BA96E8490E94BC10CFC0F32671443F9EB6E6C" +
            "4AAE7F036D1CD4FB93AEFAFCFA98F705298157921304D1066A723671FF9A" +
            "A63C6B06E2FA16763F1A8D5A9B048BBEF5C5BF6906615A9C279532FF6AAC" +
            "45361541683350FF343D3496C79EE503F1D1BF60F9CBBB562F14DCC4B80A").HexToByteArray();

        // produced with
        // echo -n hello | openssl cms -sign -signer cert.pfx -md sha3-512 -nodetach
        internal static readonly byte[] Rsa_Sha3_512_SignedDocument = (
            "308205E906092A864886F70D010702A08205DA308205D6020101310D300B" +
            "060960864801650304020A301406092A864886F70D010701A00704056865" +
            "6C6C6FA08203303082032C30820214A003020102020900E0D8AB6819D730" +
            "6E300D06092A864886F70D01010B05003038313630340603550403132D54" +
            "776F2074686F7573616E6420666F7274792065696768742062697473206F" +
            "662052534120676F6F646E657373301E170D313731313033323335313135" +
            "5A170D3138313130333233353131355A3038313630340603550403132D54" +
            "776F2074686F7573616E6420666F7274792065696768742062697473206F" +
            "662052534120676F6F646E65737330820122300D06092A864886F70D0101" +
            "0105000382010F003082010A028201010096C114A5898D09133EF859F89C" +
            "1D848BA8CB5258793E05B92D499C55EEFACE274BBBC26803FB813B9C11C6" +
            "898153CC1745DED2C4D2672F807F0B2D957BC4B65EBC9DDE26E2EA7B2A6F" +
            "E9A7C4D8BD1EF6032B8F0BB6AA33C8B57248B3D5E3901D8A38A283D7E25F" +
            "F8E6F522381EE5484234CFF7B30C174635418FA89E14C468AD89DCFCBBB5" +
            "35E5AF53510F9EA7F9DA8C1B53375B6DAB95A291439A5648726EE1012E41" +
            "388E100691642CF6917F5569D8351F2782F435A579014E8448EEA0C4AECA" +
            "FF2F476799D88457E2C8BCB56E5E128782B4FE26AFF0720D91D52CCAFE34" +
            "4255808F5271D09F784F787E8323182080915BE0AE15A71D66476D0F264D" +
            "D084F30203010001A3393037301D0603551D0E04160414745B5F12EF962E" +
            "84B897E246D399A2BADEA9C5AC30090603551D1304023000300B0603551D" +
            "0F040403020780300D06092A864886F70D01010B0500038201010087A15D" +
            "F37FBD6E9DED7A8FFF25E60B731F635469BA01DD14BC03B2A24D99EFD8B8" +
            "94E9493D63EC88C496CB04B33DF25222544F23D43F4023612C4D97B719C1" +
            "F9431E4DB7A580CDF66A3E5F0DAF89A267DD187ABFFB08361B1F79232376" +
            "AA5FC5AD384CC2F98FE36C1CEA0B943E1E3961190648889C8ABE8397A5A3" +
            "38843CBFB1D8B212BE46685ACE7B80475CC7C97FC0377936ABD5F664E9C0" +
            "9C463897726650711A1110FA9866BC1C278D95E5636AB96FAE95CCD67FD5" +
            "72A8C727E2C03E7B242457318BEC1BE52CA5BD9454A0A41140AE96ED1C56" +
            "D220D1FD5DD3B1B4FB2AA0E04FC94F7E3C7D476F298962245563953AD722" +
            "5EDCEAC8B8509E49292E62D8BF3182027630820272020101304530383136" +
            "30340603550403132D54776F2074686F7573616E6420666F727479206569" +
            "6768742062697473206F662052534120676F6F646E657373020900E0D8AB" +
            "6819D7306E300B060960864801650304020AA0820104301806092A864886" +
            "F70D010903310B06092A864886F70D010701301C06092A864886F70D0109" +
            "05310F170D3233303531373139353830395A304F06092A864886F70D0109" +
            "043142044075D527C368F2EFE848ECF6B073A36767800805E9EEF2B1857D" +
            "5F984F036EB6DF891D75F72D9B154518C1CD58835286D1DA9A38DEBA3DE9" +
            "8B5A53E5ED78A84976307906092A864886F70D01090F316C306A300B0609" +
            "60864801650304012A300B0609608648016503040116300B060960864801" +
            "6503040102300A06082A864886F70D0307300E06082A864886F70D030202" +
            "020080300D06082A864886F70D0302020140300706052B0E030207300D06" +
            "082A864886F70D0302020128300D06092A864886F70D0101010500048201" +
            "002E3AA896EB261EE1E8A552B4D5F7A466332EBAF90B74E2E563EC939E2C" +
            "C4494D8A6C27BE9D23FD009EA4CD3734E3E5BFED0845156FBC2D520A223D" +
            "17113658948955A0D95B1B8D7683D1A5E1AD6C41C8E4572C1D74D1B56B3E" +
            "5E17E2E05AF4AA34919F34B01EB93C6614D0781CCE28B3281C9E10FEB5CE" +
            "D0280E1B86761E06A315F027577DA15EDBBA19FC2095AC4D970FE718EAE8" +
            "7F4AD106E6F4C674FD1958EE16EE07C54DAB405EC7A7A55A6D3B665427D4" +
            "F11C632B4F605A41821A3F15B4F537FD5F0EE3426A7A03732AC946C3B435" +
            "776A873A3DAE93FB8312C681144CF51F05CE37A0DB4C1544E178F88E421C" +
            "0B5456D18C13B335DA808CE60C4E35F507").HexToByteArray();

        // produced with the below PowerShell code using the pfx Certificates.ECDsaP256Win
        // $cert = [System.Security.Cryptography.X509Certificates.X509Certificate2]::new(
        //     [System.Convert]::FromBase64String($Certificates_ECDsaP256Win),
        //     'Test',
        //     'EphemeralKeySet')
        // $signer = [System.Security.Cryptography.Pkcs.CmsSigner]::new('IssuerAndSerialNumber', $cert)
        // $signer.IncludeOption = 'ExcludeRoot'
        // $signer.DigestAlgorithm = '2.16.840.1.101.3.4.2.1'
        // $contentInfo = [System.Security.Cryptography.Pkcs.ContentInfo]::new([byte[]]@(0))
        // $signedCms = [System.Security.Cryptography.Pkcs.SignedCms]::new($contentInfo, $false)
        // $signedCms.ComputeSignature($signer, $true)
        // $signedCms.Encode()
        internal static readonly byte[] Ecdsa_Sha256_FromNetFX_SignedDocument = (
            "3082023106092A864886F70D010702A08202223082021E020101310F300D" +
            "06096086480165030402010500301006092A864886F70D010701A0030401" +
            "00A082015C308201583081FFA003020102021035428F3B3C5107AD49E776" +
            "D6E74C4DC8300A06082A8648CE3D04030230153113301106035504030C0A" +
            "45434453412054657374301E170D3135303530313030333730335A170D31" +
            "36303530313030353730335A30153113301106035504030C0A4543445341" +
            "20546573743059301306072A8648CE3D020106082A8648CE3D0301070342" +
            "00047590F69CA114E92927E034C997B7C882A8C992AC00CEFB4EB8319015" +
            "36F291E1B515263BCD20E1EA32496FDAC84E2D8D1B703266A9088F6EAF65" +
            "2549D9BB63D5A331302F300E0603551D0F0101FF040403020388301D0603" +
            "551D0E0416041411218A92C5EB12273B3C5CCFB8220CCCFDF387DB300A06" +
            "082A8648CE3D040302034800304502201AFE595E19F1AE4B6A4B231E8851" +
            "926438C55B5DDE632E6ADF13C1023A65898E022100CBDF434FDD197D8B59" +
            "4E8026E44263BADE773C2BEBD060CC4109484A498E7C7E31819530819202" +
            "0101302930153113301106035504030C0A45434453412054657374021035" +
            "428F3B3C5107AD49E776D6E74C4DC8300D06096086480165030402010500" +
            "300B06072A8648CE3D020105000446304402203557687B26E650E4F86F4B" +
            "77A5BF5851350C96F01142696CC1391632CB95C3370220017FD4D9329F00" +
            "1EC74210CD34CAEE3878B2302602DB7930347E104679734291").HexToByteArray();

        // produced with
        // echo -n "Hello World!" | openssl cms -sign -signer ietfcert.pfx -md sha256 -nodetach -outform DER
        internal static readonly byte[] MLDsa65_Sha256_SignedDocument = Convert.FromBase64String(
            """
            MIIkEAYJKoZIhvcNAQcCoIIkATCCI/0CAQExDTALBglghkgBZQMEAgEwGwYJKoZIhvcNAQcBoA4E
            DEhlbGxvIFdvcmxkIaCCFZEwghWNMIIIiqADAgECAhQVn/5vIv1cxCxSTfb9XijQ3jjzTjALBglg
            hkgBZQMEAxIwIjENMAsGA1UEChMESUVURjERMA8GA1UEAxMITEFNUFMgV0cwHhcNMjAwMjAzMDQz
            MjEwWhcNNDAwMTI5MDQzMjEwWjAiMQ0wCwYDVQQKEwRJRVRGMREwDwYDVQQDEwhMQU1QUyBXRzCC
            B7IwCwYJYIZIAWUDBAMSA4IHoQBIaD2Rl44x6z3duLBHNILSuIpfYllJ/Y9YpWHmlr1MJ9BbONuy
            7fAeZk79gb4eqJNojOaKotUcWVj4u8brTonuZ9LAMglU1XISyscin/HW6vA5KL1RUR+NiNhHc2x9
            4nMNWXjlQQcTFgl4hncRv1U5oL/Ew1DCvlcrrw7i4vsWzP6ggCjZmsSa67dZN93OERzati//POqL
            oiM9Hlb7xcWh5ybeY/rdKvAWsRkXf6PZcaLZJ3Fz/OVbZ3Ra8LfCHVl9vrk+ajLzQcSaWovp6CUI
            jR8qpFFV1siuFTZ+TrADuP33hRBxlJc5+f/wkCPq9FEE0qhKRZBu7UZxpE3CjSeYe7Vd9p6ehWH2
            GoCnJplQOGX+2bfucqjhehnECBRPSymv73Axw6bYVxYQtCyfQhJFqI8ZfhaBKwMRWbZblofls+k0
            xSJa6Yp5unPSs5nXNRDv+tGeU7hFDwuo/OEBL9mNJgp0qqoT+uJJoAaxw09boLiC8mN4Ii+zbyKD
            wkPw/+tfG7QUoKcNVePUCla2y8iK4fA7eyiC2Y3uoo4UXJ3t/Y6vHO8u2UqLBQ+JZPRtHqDQwqQ+
            Ddphgq2/T27RdbZ0IleFm/IvOkF+zx+diTF7XlOdWHrxa54TE+BFFP+mS6iz/yuDIfiBHLP7AiyP
            ZE5wpLgKL7/uYEq7c3kJHqjmxcdN/AKDZmtAwHk4cAKCBKE2v12pVo63mNNJA4vbDBHgNEXnhHy1
            Bpx1zyisYBx3mdlYIQ3byyJuUa/vnx3kewc4c9bT+XRWvt4IUILnSimLLNSPSzCTFV82bI+mAcav
            hY36MsCEkbKimIf5AzWUml1u2qZ5iCo6lda/bZcKIh9LnT2MvzhK+BqsleKzKU4EeJrINyel3ARV
            n5avQdigU1Fv7u68UnRutqsoGeCRCHENg18BH6YwZYcq0zTVzf+ysjEFB+kvyZOuMX2pf08wnNrw
            9n7ZnZAhVXYIOEn5U7JG1/7bP9tnZ5hQpa1ATmQUf7fPT2rt3QWvtLg0lo0f6IAUlg3OXZQiNlJu
            EqR41p5fvmlwMQswjAaEUBjPx7KrQwoTprGse7AszLs9kRrC8RBoYT++Apv9zgLPXNOJUO1yyDlE
            7fvHVhWvh/hkwFHzxVRWxUEoY6QMBtHatWK9/wVxuNPDkXu9MAiAu6XpmCOblfqRt9ZBbU85izrb
            zTCYPtNZK02e99Qjb9APUNmKpTojWsQXJyD3fZYXJnKYDP6P96WnAng+3CujGyJZAVoRL8f0aKnC
            +UZAOQAtMO9ni0y3mLwRYha/epp8GLoDt7WP0HUV0xFQSdNhS+egfnRDAHUN8dLFh1M4kFnq/D14
            XM3THAdki+3AOlw7itRtBk1ZwT1XN0cp/E4pU2LipRkSBFMEKLwVIq+ij/X+FlXjBMpbyMJ60ODG
            o53U3yiVbBSzjMk2gs7+QCu9XoLSnEZOROtdN7SPxWjf4Mxujha66gXlE1WQ8ZKU5z6DZ7AhbbuB
            UDC53lWRPwgDnEI1HFnlUV3Vr44ImhXmJej23uY5OGxGSX16JjKId03lgafelim0G0QkFB+Xj7gz
            Egjv3sPG4N45vFcGPz3NbEcDc8CIkeopy8fMbWSDuIiQg6zoaqe1Gxws/m4q0Y2Xzjb7xW6kL66X
            5qesEUhkR4w2bfHrseexGpCYUE/Vl1vfH0nccAArY8Fzmp0mP7rUBz9qn2wrivS0wzKhA6DP+l3u
            stBiyjwhX9NgAmvnxRZPSkQk73SUiATWb0ZIdzLIICx5VHhke06nHWJ8CGAkzKNUpB8Id7OPGbN3
            StIJXI2lOwaeIcdq4tIAfhZxntQAgNM099pS6fWlmQQ5yvCDqVuDPwKtEKCMGm0PJgwAcoW9Si9H
            cDpa70ZSh9JTsYrCJRQxYhD/VmgUsQ+HopPW8ZnTw5WZkNDBJotPUNX5/O+78je9DCi4AYLWZZdB
            8U8Qv7shu6EqtiCqI5b1bAaGtOqQF5kCJCFrL+itdsSpFI7vmoajY1pqp3vB3Ptvulmnff2pt1MN
            wMqGSMjZc3OOAbq48ItJBehKpGQb1gJBDNl1ICZfLyMfKzXhXrL6BNK9lNWnerrx4OFhAQqZAIf1
            tG6piLK8BRL9oPqSPa3WxFxTAdCUg2cyZbWrLhD0ulIPa7rVZKXD1eJ72wgPfSDhMpajGBlUw5xk
            nJQ+vhffXB96rgqP4SbEd1haXU1kig0Ai2r16M0xvmmpKW1PP9Je2G8iHkuT9l9ZKZZ1M2JLkjV1
            DDBwdVC1hTbRCacTHFpbvkpXFVZ8ElNK7HZgdh7rufriiRx3RYm4DlZq1Vfd73NnGWtyJ+qYcO8J
            3f7HnWuTGaaHm1IF12v3q6Ws8zr7WdF/xU5oOD1r5aCOm2baU9zeAIuylLhYK9EyzcxJlZ/bwh5S
            chiAyK0DUsefA6Q7vYTEzf3GxSkAXh582aNJpxaKNVabpd6oGJaNWpFGa9bmTiC/YkFxmK/E6Bwo
            3XftQCgjI5i1L73oa8hPR1uQFnEM4qq8EaBrTbrJAewWzzZco/LVOBOUimk6D5PnnEbKXVptyj0o
            ylCtGL0T/KVQWd2bGF95+cRxlqToGyEEvEYKBR4C8uhET6NCMEAwDgYDVR0PAQH/BAQDAgGGMA8G
            A1UdEwEB/wQFMAMBAf8wHQYDVR0OBBYEFBsFY+PNM0YUnIyevPI7Ck5akA7qMAsGCWCGSAFlAwQD
            EgOCDO4AEYFoaKkMZpNL27HRnFOmpdwwU1/23MhmmyQoSukxDppXx5GHArX6ScYzvkddVJLZ1+6j
            kUJoGakmmrWU9DSUK038u6WoPczTU7569u/dZglyHo7nORQAtEfleD00qmzd8Y2t/7DK4KPImf4P
            9jvmBakzblLuNeuy7i+7ZM2qXOTLNh6EnHrO7cHvRAnS98EIu2GN0F6mNDfdoDlsCAmggwN3dRQn
            C0N15g/5Y5/qpGrcJpgSKCr+F8FvmxiboxLqVjwgIOMe97TlvaKC4QISR5kZd/FIXTD+0iDKE+hH
            Ho+8i7K5UjXhrje21I/MBOoAuqgDqdSJ9yCv5GlYx41AExMmVzoScxfEH4fTCr9l9WbHuLGBOA6b
            yQjNwe+s7rJCRvKsPiE5exg3t5psm+5kikVM/LF8byTAFSX/xWunmGXzP0XV8lTcnxs5SFpeZOsd
            T6Azyg1WudQdi59wukmSGK59Va+5e0eybK9mb03qeQ4IMKQnBSin3V3Q917Z0Ec4pxvYvWVZ8rLg
            3jee32e2V2vizQuJ4hvVcMUz54mYym/r+ZvWejtb2Shd/6PMitt5j2cwa9Bb6ilXZcDwcRjInaJo
            WsCf1tzI78IUKzI9FyEnuw7oV9IUhITDpqQPeFMIypJo9iThMd6koGLvf890pzJVcRktcVNPXA+/
            49pxETiuA+V+lZ77qU62xlbwAb4WrMsTUCDzsePlbXGZM8LRH/juWexD4dVYR8wa23WQK5ap8D06
            RA3IrmiAuQp/UoFgXoh7kDLa/Dpwpk6UdKYDKfCySCrO484YCoRR1uiadJu6Usf9e63xh9HSA0fm
            ra/MDqWvsfDo2kIyIfdInf71JPXANLUpL0c3qA+HyrZWmzrsTODwGcSwnbDZC3VZ8VJeU23AZlIs
            tlVxluTohGy+DkTX16AztSjOZzrbWt1kQ5TnqfbhfJYM/ehUmsLL3+buYw96wijn5JrShnIb1Bh/
            yaOpyRRtLz7PXEVfiz5okHduT4AJskFLYSclSMw+La+9DrSdT5644vFEEGLMyRKGJZSgCwz/UaTs
            4zZZTSZ00+ytZU2KHgXQsuzGEGUMFr/v4qT1f1eq8xpM/5ElTZnqvReHP/RA4NIfVp5+bYBgKCa/
            99QiPxJCwwXGetuhxqsJVzQFsksw+tCbvVI+xXNkKofAwT5osMyVMToyqPi5Hx96KVOAryZyVxSA
            tAsKyMAz/FPYHdEZVRLHhbbWrQVg/Lgv00LIox8U6Dz7cKGYKDm7pdgqdxIeW3KgQTxdUOAQqDE4
            3LmG4Jg+kOjTzsp7UAaDoGvpZOYMbCdc4jPJA0wQmVo5loIIpSqK1+534O8APf5vOTpZPCyPs4xQ
            R6qjk0FTExH95ZMlEGZVP3t/7pKcaV35QGWJWLpX9LE55NZAw72WMp32Uxqvy5P9bLK8HJQJ68fM
            QB7liIsx7M9LyHDuW7hdGF77e4PkmxfznjNUo6uto6HW3SxkeXF0077rnqG0qjh07k5sZuqnW1aa
            9bDlURxfbffKpKc3S+P7/SX2VR9Ltg5isJAe1t2z6AuUvxmGkGF1xl38cvB+5KEF2+PBFGY2CUbm
            UoX0bXGoDEuv1ki6ig8oMb0hYPdFENK5dwEIlw+HUVDglVkODFza+etoL2KRLuIF2etwVswBKNO+
            gc26jmPjZqpxrx1K3FdbhZf2A13Ut34i0sDYZoGLUQr6Z/5/7fvELjrFn8GD+RbFP0dgcXlLDFnv
            OzNlktIrKQl3TOJQh15gyBnhijq/TxJTnX/ZlLMDFBbhNnqPzxgkGy5fxquKqIIo5MX6Udq9IrtG
            Oxo34xy8nQpHpFnD2IF472dqN6+VhPUfFk8p+9VBVZ/KGZ686jKjdmFA/TvM5G6hepc6J3KlBalj
            ZFL6DtZT4doXJ4XNkBtgUQszkSJuHXpqkS085eCUMpZOIs11TrCzRPnKYc2CsBku2rfTxGFCRMIV
            Hv/JieVtwyH0wCl7Rnz9o0URTeNuZMUFFkZhfu1+okweBgBpVboja7bsjH9tywsM34rWHArVCYcj
            OnFoAB5KHp86rsMNybVqES7Exlf+ZXtIDxrwtC9gyTQpN657EfDoH6FKKktRF4pswhhLBhLaowku
            Qj5MU/udjgq88ZoqIghsUMcxNEof2NLYH6tSJVoOoIxlbutnwN4QaMZhNvR2etPQi2ZVHKlDKLfo
            /JTJsCvf1HLtcN1A4nB1c6Gr+qsvwgxA6FeAYBVYPdUZVwkBpAceAQzydUQ3G24OXrZLvXOfTcQq
            jzbHcXswadh8NocVKFvVQJHEgLZI+sACfyIffWyZOXcIP7kd0QG/fHN5j539MovbPXf3lI4xA00T
            Bacna3JjCS55F1zg7USiYin2+baCNmJXqcCepppzzu93M6MbHa7SEtZ9H7vu18vY/2r0EB3bzIJM
            CJ6L45yCMEbSiiKwj2rEBdEUySA0m0Syh7QWTftrPcwE86j2NaI+Up8Yqf+fz9hwAeR/nPfQiLAr
            8F6dgsk8V9YJ1dbc7Fc4QKY/Mqa376I4eOkB42IYlTw4G+OEefygYHE41v5ea+A27M6RLpe1w+O3
            lvNtAEUVefGKptbkX2o9+CDbwjf3txAMjjKwbDjrT56F/uVKp8IVSEoQ2SSk9xQXZ5IHKeLdUsuZ
            3DlaxxSTcTLiSaKi68uTmilV18mu7JGVu5kxOpOFdFkyp3GJzASidllobNYEWbgwUVGbehcn01qn
            7VCAgzq6odVIHBHbus/umHmI/3cT0viWvvzFRc+VsSrywHa7/mW69L6Mi7TVGOTVT6EgEswvq46Y
            1MdfxFQCPJBqz5s5DeuG+KTZs9rJsvNYSFi4OkFqI7Kftpa1kd3mFGZmuhujdZxUGeda/6luOfds
            qW4yES2dcjg604/lNUAZQo/STZ5pvk9GEdG93ODtmfgNly0DfvD6nHCv9MKFcBrZ7JSfGGvG/we7
            0WsXyiYJHLJbyzgeZz5ZWDXpVdCXVsTRYorUK1FQpBqI0IEdrpRhFzNEQYgVjCRjVEwChDDBK6p4
            kbfJo9Pt7NhIOoRkEBWvqPdHOA2x/QwyT+pUM84DqD8P30MgHoP2CFQVKBqL/VG+jl5pWzsH1ftz
            T6orwfTO6xgYMV3l+v1v7tb2PRmjnhsa7i4Bb7u+HumryOo4b4vdsyS2Hxv/tmwt+SRgK2Wo/ZV8
            4vyrBYbM4UtfqT1qc0WB3kinnZ7kq9F40IdIq4envOvOT9bJXNTzVkslgcK/g/dEfFK1Acx1qqer
            G08ReBf05FYXZrDyZ/1dtdqseuQc6TKHCL60Xt0vFUM8xEQkqbXL+Op9Jok/Vo/R6zhBJ0XkIQGF
            ruiLHLTR6OobHnPcegbqZogA4ZRNv9pNCk9Hw6YLNxw/quoRHBLpKX5POcqLqdtuuNFzIM7/1Ocz
            wbUCPdPR+JT2HaFclCcPe+5HAjkfxOREMrjzTgIybMof+mcBaX1SY6X/Tro/PiDRcRf+ODzA8oe+
            czJ3pBtK53ljWzA6pyel6uGGRU36YmpORK6OMWvsQx2RSwbVIrQF5NVL2VJvqK7tstras6Fr2X/M
            8jUWntaIeoA2c5bGByXdAkdWS/GqKMMitKFiVLZtyfzFgvp7U83F8kKrER8MkeaRDBgOKpFmmS4w
            nZnn65g8K1fr7pukiuYsFBkXFH9iAUuUXm3mu0dp0vcMKM5bK9PS06G9B9qh46iOBJqtiubkbO8K
            UTRIXU8+yf+qwVjSzx+mOlU7C5T4TeqLn1XotVkHg0RifzHypOx3J5n5n53stuew+Ku9IVT6v9lQ
            JM/xHN3Z6MfxqQahR6l5eCpRcd2qLzqO+ChN2zy8dSDsTc4aMEfibL3hVo3Ho7wcgSG1jrOrZfXw
            iRG/BSE710shRmpg0CALmf/w1CvFGqKJjRHEn1Q5u15/liU2w02heYvi1e8n16ww1MwVHlscuUVb
            l2l8d3ZMLMBkrFE94au4pDXgoWZd+9ybDXy1Xthc1OK0+0Fv7miyIkhfwYEb3V6KnE3K3Er+qhWq
            +9+PK1J/gGLRcL8jUTjFDjpTBO8ZAotO1EfJ6De8b6BfmtpErY3LohcMmPSxxgTv2rzdMBy5Zcfq
            YnQPTgR1v+637afnEvlUN17+teXOe1sJhPDFrgFsmZiPBAKIpXMN6Sdd0yPGxXQS7Dc6SbcIhCnt
            uI+jvXTzLmR4mWXocSnHuxZPfuY06iPgAV2bv3DMQlCGXqko6jgGl+PV61MVABxToTfB7JUP1E6L
            ta1IEfgQWEU5+B+BrUIcPwMdvAN2xr52EiP2Y5EKnt0RQ+O3lCLmaKaWpDBgV8Icj5qnpViyuGnN
            aveA+LwQQ1GxtNQRZZ7Mz+f4EfhSeYWZBjtIZnJ4e4O8U3N00gAAAAAAAAAAAAAAAAAAAAAAAAAA
            AAAABg0PExwgMYIONTCCDjECAQEwOjAiMQ0wCwYDVQQKEwRJRVRGMREwDwYDVQQDEwhMQU1QUyBX
            RwIUFZ/+byL9XMQsUk32/V4o0N44804wCwYJYIZIAWUDBAIBoIHkMBgGCSqGSIb3DQEJAzELBgkq
            hkiG9w0BBwEwHAYJKoZIhvcNAQkFMQ8XDTI1MDcyNjIxNTIxM1owLwYJKoZIhvcNAQkEMSIEIH+D
            sWV/8fxTuS3BgUih1l38LUsfo9Z3KErd0gASbZBpMHkGCSqGSIb3DQEJDzFsMGowCwYJYIZIAWUD
            BAEqMAsGCWCGSAFlAwQBFjALBglghkgBZQMEAQIwCgYIKoZIhvcNAwcwDgYIKoZIhvcNAwICAgCA
            MA0GCCqGSIb3DQMCAgFAMAcGBSsOAwIHMA0GCCqGSIb3DQMCAgEoMAsGCWCGSAFlAwQDEgSCDO1B
            HRj/PxK/AyCcl3roqEPVygicxOS54X2WN6JlYAphs6Pd1rnTpNd5V8xhH6RqtW2M7/UwSPxsGVmb
            avr9pVt35pxWvS4PIdwMiBiuI4sktzkp1DXtMaOuJUgUAaeCR+cAGhoWSPF4JHh78oVX6lS5T4Cp
            vpNwdb/i6DMawWfhS2KFdlt9L5RSeDtFpxqo//uYgeDjh6rbYW75Fr9EAdrizorRxXE0Jp4J0mYR
            Ab5IukUQR3dG05YPPtBsxdqjOc1L+ZgrxS6XOKUJFAgyAwRogMOYaycdo7OMJQmOrPrHkPa9Glgu
            SA45G+7SpcyYEnejZ2rPfw/+Mea6gIfqg1kqGXrAu+EZgkf6pElX1j7IqD4ox4S33jVQP6AHLiqZ
            XzuBweftQpZUb0MxQSfSwkHIWyKL8QMJ5clrmCTz5ywWXrhFnOwaFA1c0T9LTpcrFpqcjiyW5Ztt
            ZtAbW671VjhN5A8HGIpC5g0dJBkqEq+qZn7G2G9iGF3h3XtbkRaafxvkHw0CfmOCr0WV60VyPwuJ
            5YUlBaaZryyJGWPhUo/ty7ab0bYEPdk8SDlNeX8Em/kYBYMCRQfT9yetsFZvhR3AVUzMbBIvBwxJ
            oFDZ5m3J3BbkJQ/kMvGx3n3nkWwW9SL3nQxZ/uWbKJsRmrdtJ5WYIs8pF9SwI0cHQ+16UdSin5hW
            GnmgU+RmOkiMx8H1PSfFeC1GJ+ZkBZXye+0s8li6Z2RXBbFkGnQqky1393EFoJsw2GsmJdHh88QS
            2TSy+6lBD8HHGDVtMme/nHQYyEjoGZmTFLCyb/OUEGGuBFssYgPiXABHqAfIcP18EAsM/pt87K0s
            qkGtA+0XOf4NdGEpw2m9zE8Xrum/mJD/p0UDhkH4txonj2X7TT3HJJIwx/gRSZmif/YrbcYpyf0W
            /V0hyEPUbAl4UxFyFS4lwHYf1d7RaCOwSW7tyRw81zYXU7RAyPA4X1W3yklrQv6dTqczos+7O6d8
            +mTPMf59RL1UAZT5yhunVsZFnWYPfY4MPvEykvzqU0TcXDxgjeCjb6FagUmasDIbPWSA0K4iBudT
            J8JsqR0sZ086gtOh1h323/iUYoq2fMqapMIIQpdpr6w72jSY4Sl9p5ctEJzqeF8VrvM1bDmMD9s9
            /zUpzK1IecXeLIQ4/9KVCuk55u6uK99+CdkfWbocXx3M95PIhz+7pQEDK/P74XSKWitigKLkZ48Z
            n0R4ao9ZQAkyMAAIbL4+b7E9PfGHaz4OISr1/09Czc0w8/IDqJNqvEkcKzzKymrjoEUUh1py0YBA
            r5ynL8J4tc38Ic+xHFEUUcOWVauLUOW/dV0iIbxwEfeQpGMyQ4KieZD2fvEigEDP0NyUbCfTsRmw
            92ttQD9yApKiaQe/n8HDjmJJ4qUjj8Q96VVJEAOTJ9NIZSmPNNW0FJ9WA0zxpnpIwNT+DpThu/Kv
            xeBZFR8Uf6f5F6Nly7CLYi7QwhElGp7eDLfZdvVeYHzD0hNPB3SVMHOzdvpbTAmXodya7cwwKJRU
            hno2DxDgldYDIhfbCrh34zaPPHR2afqOCUdR21Cfbh4jfOwxpheGk5vKND4luXlpwjOY5RNIE0kT
            ZVDOQyHJal99u4rb17m1qCgIi8g72+0iQU8xZEMeRTA+gnYlUCPy6rBTggynzb6sRIQt5NOpgxZF
            YYUOsiAr1NxX8RZH56RSVSk1jhIjrbQ4QKIscj7i1Csu0otU/bYm5znZnvGzleJqu4q/Fdo0EEoV
            eZpszXzPU7LarZX7gl5YGc5yvyf5CQq3DALWT/J83XsW7/94edGbjxOeXYtD2CUw0D0eOSjV/dR2
            K8Bwpbaf6H8Qus77CMt0zWAUZnaWtP+sNjbLQA6W/EBpxDlDY/IklnJK+AxHdMXMzBZZSnbOtt7m
            EOO3Sb65/Gy4HS4U3KuiOend+df5q4CbagtG7oNiLGlkIy+RJ0dm3GHoeAhFtCyx3e+V4ap9/Z89
            5N8C44830s8LRYusQgiBptAXq5XCfU0NtxPJXf0xnkN7dZAmPrX8PmP51PrKdoA66p/PEkvzWSSQ
            zGz+b2odJYyjfjIiurnPRigBN/o2I/om3tj70fdJuh1A6SbD/p0P5r+9dhc8L/SRfsuUKAS19XCj
            8lTAYGKs+w0yD1FQ0C+Z2W3Y3BkB7jtSE2mYBUCe0dccFBe60H5ByDg5xEvG+TjOJqBJ7WTBGPKq
            8z2wDAFbXumowE8c6uVVZiJLMhEjG6XPlPczFsXjuDBqhSri8FhSwUD+I/7w+zDKf688gzD+2QV2
            v5rOKGoHb1iwV9FaQ82ZSnbMG+YKM5s+4FKJ/2YRsDJDfhqaY3JRRViLMvj/NPFPdtqGrhVZP0rd
            1VQsw9Y2/mF6IXAZz1UbE9Du5Qb9thA+a6ujov6aqQTQuYLZbhjfc8NQUubB5W760UtvXVx1wm2f
            Xb+94ROVIRwKe5THx1jtPabIAgh/1qcaKpNvmzoZxowhSruDu4k6WVWkUCYh8UMt/byDvQpA5idS
            7VnxkeEqy95pH6txHYg77pBbUKGMWolwp4axQEbpr3YIXtP6/ONxqFfOg/b+R2jPfRS3Rn4iIumH
            RtAh2PbJeIH8NxLOQ7ZfmTf3+d4ctgawjQLXW/yPItj2Rp8rXocHpEfMBN5sAkc3AgKoGB18Fbrt
            rh5GYKABvB5aaoGudRj1nfYRDO3HmVxJywWuV2pLgr3Jo4yqrTXQJ3CkPQskXs/ebeUtTJD5vwxO
            oo/9keKjRaC8SzoG4PDnflgN/a7a0vwFvVWmOnfwshMFvpvbVL/GW37I47Nf+RGZ58jyTDjhF5nM
            2t38dBvjMVoEaFUHbil/anRVKF4fvtEs21jOUB+A3yZrAJIrQAM+J1qwzIT3wdmulFreTJ2EnbRw
            lwz0QoSlA6E3GR8jZgc8eGYFUVOGU174ZrbFjiUp7qKtQ7r2vDjOTn940Eza3jC5vGi9O1iV5Ybz
            C6TtwGfpodpf4sDsDH428rkgiap1dvZm1Q4mWw8NJXM93++gy0dnGXr4ODCR75HmQN2XVeHpkYjG
            L0EbtWyBxd/+IWNOyg849yKlBE3o06mmMChA9rmsrAGEPfq5LfI1cmA1KAzYqlGASQgAQOWTBJfZ
            9Z3JtbbBQ0zd/Q+eE+VDlqh0BY2bwfKmSjO9PzmMWbo5sp5JM4kSd0yh41e12XT08cQBYpSApNcH
            24cVapZ/k3bRG6e1dP8j+hbGIBHWhjn+ZFh1hRabu8zGhp5ULk9TLuYRmUq8S3YvPwIjoFShNkAC
            k2UQ5spzZWBual9o2Qg4RqWWbqRxNRK95FhlrTGPpizgcPKb3UKFfdFtef5bmeETY/gwP0l8yjJa
            72v9Lx2EhmnD/Y4Z/n0G4rkSDXigfGC8wI+YusijSkSkzqmygf275RndfRQfKGiR9guwC77QaiYZ
            vh6TU+F3wIeEpfiCjoVAbjS/xpaCxdoKrhEHdNr1s3z4W5Lu0dkBpA0Ccexs6krvTc59frf1sL4G
            OW/vdzgjJPSGuHcS1PffPBnD+ETZrlvvsiNH6OnwCu6XjD4j4HtML646Iz+2upTFOhgzd0jmLX4v
            eNjXgb+epelbUQdl4//sNxZUxS6lpK5xTpgVy35yhFe7UVWd4Uaq6EbMC9bcwS2lSZpWqPylw97K
            LDmMSK0VZulufoPnloKGJj+lll/uxeUJCaE5M6q8lTjp80stEL+umjcOH55TtL/EcE94x7Chb0mz
            zaJfxGcbLgCI6sxOlwrXgWVm8xaD69j2VjLtsUtSuUqKh5suaN5qMdcsUxc5r9bv42MmqclNF92H
            aSLzrEvrDmwJdQyjyCIi2BEDQ/kRV264Y7gskeUfSzwVsWA7OEle/udpqTMFStUyOnCnHDLLf9k1
            0hz3n7m9z6n/uP/0Y2ioYw/15A/+pxBC/cPVMVSs4EjtB68oYtCwVuG3OJGCQCGuJv2lf2uWJZ0b
            fs/qHsC+IGlvGvmv+b1aTTJMCIe7Xr1+HYziCHePDoOozzNGUOnRfkxzQYw3128lV/1wNV7tYneX
            mnegvdrEuBNtbo+njoSWSXyAS5NR5die6UlUqxVuv+0WMRXO2afLKDuNUeYRCapvFBGDtlEsPDfg
            WgLJrhGWLu/6Z1OVkNqCyMqlnNK6uofsfin2DMXw/pOFBiZcq/mgvyGFN79E1sM/N3GLNlF0tvsm
            WqVykwXnUxKA2gocjrFKLlWUluD2AFaYuwWOgq1EFVd+0xDZSCDZkd0AlnvRQZgayvvQb6C5NpZR
            VEGfn2+pkbR3DgFCW9zwbCwieHG/4FiTNoYkBXNqR+uXKrHCKeyXxIO7A23Uxn2rWcxL+xIKEQ8R
            NDpZdpalrcHS4Q0SG0RosLw8WW+XydjdAykvSKjpIk2UquIKE0BRVYaUuwAAAAAAAAAAAAAMExog
            JS0=
            """);

        // produced with
        // echo -n "Hello World!" | openssl cms -sign -signer ietfcert.pfx -md sha256 -nodetach -outform DER
        internal static readonly byte[] SlhDsa_Sha256_SignedDocument = Convert.FromBase64String(
            """
            MIJAkwYJKoZIhvcNAQcCoIJAhDCCQIACAQExDTALBglghkgBZQMEAgEwGwYJKoZIhvcNAQcBoA4E
            DEhlbGxvIFdvcmxkIaCCIDEwgiAtMIIBZ6ADAgECAhRDhWOiaQGZLDnPvEBXG1+jzMeIRTALBglg
            hkgBZQMEAxQwQjELMAkGA1UEBhMCRlIxDjAMBgNVBAcMBVBhcmlzMSMwIQYDVQQKDBpCb2d1cyBT
            TEgtRFNBLVNIQTItMTI4cyBDQTAeFw0yNDEwMTYxMzQyMTJaFw0zNDEwMTQxMzQyMTJaMEIxCzAJ
            BgNVBAYTAkZSMQ4wDAYDVQQHDAVQYXJpczEjMCEGA1UECgwaQm9ndXMgU0xILURTQS1TSEEyLTEy
            OHMgQ0EwMDALBglghkgBZQMEAxQDIQArgQnsd3yqTh8CTM/PlJfZkYBQkoD0JWrysHr4Aom0lKNj
            MGEwHQYDVR0OBBYEFM1ZNqr+xBHHpHJpPwvos4shexntMB8GA1UdIwQYMBaAFM1ZNqr+xBHHpHJp
            Pwvos4shexntMA8GA1UdEwEB/wQFMAMBAf8wDgYDVR0PAQH/BAQDAgEGMAsGCWCGSAFlAwQDFAOC
            HrEAqqBR3rDDFNDN+xJGojEgye2rP9xXpftF9vA7f+NajLWHHh8LFZ+qVmhDfuojBSHRM8uEYVV+
            OXQYPOqOAaSNmvs1dGnJYjV/DjQBHJBBlxP/xaRlrg+/mzLSKiyXhi1J67qumnDnNWc/Cn463Qtm
            TvhFsubYcKv7cmDrha5iPKS/PHrl3Uok4k7QtTvDrOkm+GzKO+FGFX8YxUFAkHO5GWOGIzqyfxI6
            X7vDEGxOsmLuO0vF4mkkdD5ugeJoSMgnJbyyrNqornVaXAkiHL6VCgteDAhJQjoNLfuJO7MV3u7n
            sl4fpvBK9mXBXV4Fem0q58LDIDfOqw9s6sk58yjRdYExfwHiCchWgVDPTvqCGmA+h79hyqBAJ5W/
            +E8Esf0ff84p+hVc75Sa9vAMfwl/7LY2JoNpqi1pnhd6FaqbUUPBkHzJaTpase53ySjnIdiTCoAZ
            nF63YV8UbJoAIqpNuIYDtYNK6fNadsyjO+QTlPdWllYz3RnZPY1Vq5nlACT3//TuCEeNQ7P04zrV
            Eu8EAJlioV7NX5+Q88KONZuKRuxUThMgWV9j2WGx4sQ20uUnVh9TWZwk7Gp5Kx1q8pM42Ot6zdeK
            yJjUh2G/eTwqZEIPWxW0vcDHxN4gTLvYD2Euqmfhp/8Nt90Fz1zLDEYm4NlIy0V2J4hRSd9MFmWM
            GoSCCfPU7sQqF6l7wHck/U8AmBLtEOdnw31UeA/IZ3/08oArGzQM+l/EEoUcX+aEjc4S56717+uW
            X2Jvhzo1Z8rYrbVVCw0GkdOdGpYuZ9ixDo8HP3vW/rV2YhmD9tIINTufHQr3FNJFUHBckcy1D0vv
            ee/Tx70CevqLg80xB7D3innEaBneAfhzGm2Kx1TIS5pAU+NL5L06UlDG3t4Z156oiHDxcKYRVbBG
            XkA3spBckXa9IB0k23EzgbhH7+x+eNIlK0vibgGB1BL/QP/g15AphYDmSvVbMmy3BRwgJ+CYV4Dn
            opfLkc7ZwaNf3CR/uPVc2pGD5a6MZXOEalvJP5dRfcw/1jnhcfFUjR9PM3DMB/gDcL6MgeFdc8Gc
            vnw9acDMcpDPZTg1cRauHeKmCMh73cAw9LQqRfwF5hzvr/NTAy92tXvxqX0WM7G1wk+bVXsNIvYI
            SziyZ07Z+PFlA9ZaHx+Ly9p4/HtSpdcbNbLNBn4eHYtgQJF0L5HJxsfEAfUvEMLqq4T29i78d8GF
            KJClEdztB3jCdJxghmlAPBebOuXoZSLCf9mIvkNqMZDVIzfrk3DkvDSUT6+kwW/zMBvG4fXx2Huk
            Tm5pvoLQgKiumUTh1vpF5QWlUgpdYBc6Hi7dLrSGkzGTD8pfBVKOMRXoizCIM9fakVJAPNcYvHKN
            iLJlxf4KfFBEfg+2UlOLKPxb+pNUNsrhwWt/RhPeBX2+M41nUrptr0vuAQvHViF9Fr0Zg5DIFFGL
            +4PBpcppWq7Z8afc91Of9qNDlPs4hh8qD1DPjbw2Uc6Or4D+tYD4Q3PqOteipLZzOlprSKcxo9NC
            OvwusCnSZ4qa0SaVCAthP3HusZb0SQzXO1BhbBXKMTHcDfzYX6Em0+JDzRM5SlAtZFe/AqhcVErU
            N0XyCf3PU2cZ6ZKkzRuCCSxNKTCAwSOLyhw4xhGPojwsf4Ylyf6jGvyCq2nptTexDpqZEM2ntlKf
            xuRuCPGQzRS4wuCpWC6KTFLf1e6KV86CV6aJD3QgTCIdAskEUmh481nJw2CFkgEwdaDrKStmVbdI
            St+Put+ovNlFXOsEqMOUtrsdBRlIm66NYy261tNe5XpAtgV0obB6t9e0Z9bWrPUFb1NFpu3gDLMM
            MsaJ+0J7EXSUJdwBfLtOT0+XVCiw+0hmhzrQ2hi/qhMMatPHPhEmQ+hAs1cpAHAAr1iwdYOeuUtb
            OfF/P4mNHQsaeE3ljOYHhnUjGxQfzQRNmNHN9U8dAFX7+MeS9e5exfMkhCLuEUiRS1H3h6icoJpI
            vJP1PBx+2awVHB+3+blmn/TlWEr5flw/o1ogVL5XdHRlgA30MKkNU+ZxUvl+9AIk5bQhC7wTLmcA
            vWRUi4K0ZPhSRrLyN10ySYq+GU4hp8yaGSnJV6r+20rv4KEGGl9YTJeu/qwWoOOnYO+2v4BnNchs
            /hEWGL0EkDK2dWQTVbIuxt8vtzXWPPGrTB7awk/8JPKSzmTd73B6riYHAWGf5i7+5DWM1e7ivv07
            j8TcXFBMWi6qFMQOtYETVdCFgRY9zgPwKyU5tvnO/8D1TXdghgMl/91Xy/0o/eKOu3z7SUacLA40
            dM/SuEW+/cEqa44wSMOnQWcEeGidgRw19JNaH0erOjReTi1DK/RSvFg0UhVTNhnJsLxXfJWzhu5+
            aJ9zsgkwT/iQrguN9PTRRxvo0QOFki2KYKsw8+omXjfpkLYt9ggfvP0TWv2pKXyrWBDZbTsndTH0
            dKjocACjY/GMtJciK9D44LJuT0qW1fA9/nPhyLr7qJa/AcJjcPrdl+XJjwAEXfrAOWi65dyqez29
            JapD4gKhVyt4dID41uqiRH8eNUbLfS+D3Holh+Anzt8SFYO2Jir5TiIYyml942iGCED6RRulPWOh
            qhnKgz0uSxNNWCZi8u88axPMmZUhwsf1rwjvoCEaS+n0HE1Gcogii6q13P475o25UY1F9HATaKIr
            CpyCFmT8OloqGab+kjRl4mqcpZMkIbS2ULgEMQIc30+4nLY7GWYmqsAz/Zv7Ai/IB4wfZor288UL
            dM51xJQ0gGBTwUIJLSH7JbT/wQAw8citzmLGHdeUzA97KgC+s/PIP+WIr20ZkDFxltaMWzS4hbVC
            8vsXoIO7amGG8O8f284AL5Cq7geXWVaFlhyXa8rUfZq93AFS3Ry8gl6BCJE2hX8+EmNZqgMQswMt
            rRd9YZHW4bkuOVQniqSRh7ozVChSDUbw52NAbRV2EVEoG1+U6jBvADSm2ELEMqA2G1UEkIeOLgRH
            8SXI+9RYeTZcuYEYxf8Wq/64AQr7SpM9m8WC1R+/leqqNu/F+Nir98rISdww+zSdgeJ8bAZ4NKmq
            RHSfQqXFkZ9BxPF5fg3NNtUhMl2CTbOADXIZqyoO3vQizki3skQC8Zmxv3ndSQu/Pvi5peMojY+J
            s9i8l8su+MCP8BDNAC/fvLur4Hfe2UQXjnDwB+GdxaX7ke497vSYnWcQBDqm8gP86AVT7gApPIT/
            NfTfk3SCFuxYJUOBAbJo0qdR7ZftwgYe6411zxEwsPcPwdLB8UNdQnD6wfkq66KvAAfLmcrLmlCF
            w2N2063179TwyXWkS4hLMoHDQ5e/qAvAWiO0KEZMBHA2iO7r9SaymQXMawoO+QZz/cO+N8cmKRFi
            1CDgBvJow1fbv4XmL8vxgZaIcJ6iakIC/HmQ9smw+7NupWjE7ruMh2yBIBWofxu69y6y91+jwANE
            zuIn8gTQwLJ9vrMRTul3fL6DlAMTdS/E1IrpvKP6bVxy+mKGF+Lbl4jKbEytaCtXz/W2ki4CLoLR
            XJ87junljXZ8ZZ1X5SvfycqxjOyG5wmV3nNXTuyvYkdFecb9CTLZW3PeZ0Q5KKP/HY8iYQRIhPvw
            RAQPARutv5//NCyDPdaFPJuC70fHq6Linqxx69Zep9jgeVM5KRUOprlWOZMWfwpIAG02CipKEe+A
            10PE8Abiokma5i3F/UaWqINFIrXHVdzPP4SOC2l83OAwGh+mFNZC0w+RS2w/L/lkJbvkg7lEgLNs
            x/I+WKNhehoEYdiijOdD1+v0kEiQMNzBVbPrS2gJr2J51/YJYYm3azc+CU7V1+MFsUvw5R9rPvBr
            6yqNHa72h8Zw8nT6kkYd1n7WqxrT3hFxvvCh4wWCTjqhLtIrxJIOo3AQP9/EzFKX90ymWnvM6HRa
            RxJCc9hbCX4xqWgzd/bRcnKjIuLZbsX88jDVhcXCUHkQpp8VUDGkh9fL2rlfN6v+fwkl5cMewNZ4
            IKAhIBBvPNC9Rv68rd8lJ430DQxNsjCxcI6qJZ+AuWC3ebIlvqXf7u2MrIfJaT/q5c9N0URzf6dO
            m2lk39qKV1MRDlT9r8pMbeCtVh9/xQcAi+SzCVOvpNvhocThwNZw1C3o1L04lMeTOWRxUG2lMH3+
            HmHQoSa7avgyYwU3ZbsjlwYTxtZGtYP905ujlOxnjpy7nq8L3+go7UX/pIzZ+eMw3SDyPa1P0Lkr
            F7/QSo4DjaIfFvr+h+s8V334ePktdNSC2FPgkbaDb3N5ytnKg+2EdRDgXvqnD6GbZyHQmrCQg2g8
            mZdpQhEsUblvXAMfLu54tzoU29idF2marZ6A1dfe/jsY7qZ9nztvMGd0ofT/+2it5OyPf1sCRmIm
            EGqIsaeJ0YcApJWElp60H7/xb2e2P9XCXB9BEM0Gpej+4h5S41xGucTpGKp44Et4gnisPVn9JEBE
            Adata4e9EaHBvfKpzL6uBVJ7vYZj1p69Ujwl3KS7c7wMBATBDOlu0SbDUKyY+0tJxWnt2DC7fNJu
            03ZaEwyCKM9AXA4WJOiCXSrwh4kjmS1+aoWh3at4G+bPdrz+JrImpafh1ESj/yCthHNbJrI6FcnE
            Ap37sivPtfKjfpne+dmT94sW4wRPxLxNZ5s/ui15ekfx6tg2z13r97OuDOBi+PYs0CmRivpovyBX
            73kNcWL3pyXHd/IDSC2Vc3u6wPVie7sNBraIdKS0fki5pm2SeD2HTmhE1kUjyXsEAn7HQH+gQfwk
            juVDGfRlsqXncycDtFIO3jMSYu22wysZzaBpC8tj64WDoRapK3LB58Zjf6RBbhlhO3i622oYXPSx
            XaVd3zj9X4DPz/CV4bG8ei4s/wQAXsd5HEfgp1feG+ZpE3o7z6DYaRbynkXmsX2f90cl2R9QCm7d
            2lPgTVKRM4eKPzfveusamKBV4Pnl8gMf4uvlMGwMS3Wkz0CH2jBJJeEl/TjORCDjdX8lK3vdsgLX
            4g+WpLvPDN8W51uRRjG8TRi2yjOhW+ZwlQNAeakSqR0J6DjX1H3DqCVswqoLeBlbFsuKJE+yesqH
            aIWbIhdQ6v0orkX3trp23knOn6RIsbvxuviIjhQeLy1Teb8yDvwZILG6EmhdjNg8PNZjii6L5Hx1
            BSeo6eBbvod31bOIdNvNX1kQXJxE4dR9vzbs+3CVv6cb2aju/deRTXKx0XKHCwJYIiPLsXI2BEcz
            pjmZNPpzauG5IRd6BFsjZGWfvxTmjU5wG54Zr5uYPm8TLjWlkKfGJIq20AqhYOtAz3vFA4fip3aK
            EFtOdcE+rTce/0ZZqLFuxP5lgWFnbYNRnyJYH6LhOd3UM3QikMuTv2WmWo2S256aYB6WX11mE7jz
            gvsTWuo86R9d17R/GJk40x5Jgyao7MATmK+izy0qSkp+MvwgtYTAL9YMQFqtNNv81fOMXs7NFfto
            1GDEDvqc8X4LwpXP4R9rS7SLfRsFRY5lYtgkT8kx9Z4bOtPNRwWT4JGJn36HUKkKSyjfAFUBf1j2
            1IoXwmAaVipJnI0RJX5C52CQIPc+EiV7ggVJ1S+Iz3PbCX4P8X3GpA/cPV8lpCvhdH1wWqW0Z2xm
            dMSGATCv1en6SXI4OwCV3vvGru7I0K+yFI+d2jJfnueFdqkafNNpiwJLPP9RO6CAafCVARCuupSp
            Wc6gkK+N9dtFYwtPivuW2yZm2rjiz34VR8gQA0aMO79GDCnmfYBCOsKNOLRILSyWoTdxE5xyAAL/
            pHn/dFoxuqY6JAi/jkG0SG+8Q4UxfbnKBmB2+6fRo6+t0KfLBwIIurfOqwZWKF0xeSzbEFJVTGVT
            EM4eXw7lFSXE4HgSPNIMifNg3fHvi+x+ipssWJsfe/DT3UfXSV8R+u16chyEbAYPdkSo5i8kGz9m
            Rjznxn/jBhtefObWZwg082Qs/TCd2OJ1FJWR0A9M2fCVQ0KyFdtPPRXLYGwi+PvgxEMc0HGdEJv2
            dsPU6PHYYrOzj/TiaaX94woj5k6bD6UsoQkBzicmlKeQwOgOgphDRIedNFdztbc1+qOvR88JSCd5
            08YbBHoI36Z4D2ouXOXGphasT01tBtZF3mg6LPIiMmGM5tDlYqlJ/rqGrcvGvilrC0vNTFlOvRds
            m8nW2c2fqgGMyaPdr2tf6fUYJG2Q4RSeVoYELjuiQiH4Cu4FcTFV91aZX3IYhyL/bU98wsIyhF1M
            HdpZEnFImDdoyGwUjLaM1Enl9isPBKxmG/fE0Bht410STZ00xkw2z5YrXa7XsXTJ8ES28MZFMk63
            QkLT+bXDUVQ+uEpwDoIuOQe8ZqmRk0Pyf+2kYfI1+uCfhgDJh1tpfjv40frneObQRifVgNQ0D4+/
            HCdHYD+ntcTts8IVNzezi9HBpxtHJHPOInTa+8g/oWVNeWfRittxedRdfaGuBZN4MZjT9syjQpPh
            EQZRLDxMt2tdB/qoCHJMmiYLrygccFWxHciCmD2ltGL/dwcThLAQfvMzNyFBLs072k7m+q0/7vMF
            OY1lINyUSZjk6aEmszo9yWkf5JwpfRuRAnAni3ffGH5QUFgGG/w3a0wAcerugkzii6SngfiHVwdQ
            2dC/9IXHT5vP5FHu0WsKo6d5qX/kauuDWYL45TLGa5NXGGHnibH/p/cxi1Qx3zDICy9+XE0dmeLN
            YZe1KBQ2PzYOtCc4yGFo4JWNJjzUg12Wn6Y3llnbEKRfkLZE8X5shkQlQAr879dcl7obTJWe456Q
            uQJYMB1gt5Qw9Xi1pOo3gnr1c2wN04HKcsyMzb9v+n/LOScaWZpxUdjzs0DT2maD9PKUpY+1oH9y
            wsjnG0E2/vttgdirijNBGL9CyRqKIvolnuC3RUbuqztXOo9kllF6H2aV+VKVQHdRafVuvTyXlVOQ
            CbD8X4zK1S1AqynCITGAdbkMyVdG+X7h/JVjwZGtEJCvLaKFAlXRoRB22ySsNx01v4oJKSG32tUm
            bQBudz9k4IhrCTfpgvjHrbwF6h11pLrD1PtDrpkoOhn9hFNLhIqzdq6m3am7/lbCfRQFYjqkr307
            zYDE3YdYVCGeIfJgo0Km3lUxjsl8Aa79h2dSQ7p6pO4jn28KUts4EkEYxC1KhYQ2WaYjnjiOUcKI
            I4U63GBSVnmZhLClqbMbrCfIXU2CjTzu54THDXKsgMiCVbsFex4z9KMMOVsr7aT2z6UVj1i+oLub
            NSfMe3iq7qsP+t6qu5WUN7ZE/yHhZEFzRiLZsIlhJLRTAZkXS3np3eA9Csk91QIcSU69JtmbsDIu
            aiK4cPXG7VFP7qA3KXXzF1010qY7cUOLbyKbGn2gxfd/fiR6k2e5C0yEYfLdbW9ge2NWR8bNHK4l
            GKnPIaq81XBIdTinEF68vKHgJ09sGLRA+IABdB/80oJYs8TzHPHlZmHAbGNMO7ZhehWdvnVLwwQ1
            o6cD+cxQYtA4dMHiyM5GG3ZCoDv/XDwEx3M9qza0HO9Hfpl5DIedVMlFSmEpQzRyTqbZJCwwdHU9
            FoeRA1g+eTvz0YtqEIcYkskN5apjRQpgg8KBETi2w834sHHY4FsExVcqVTzbP4Im69sJtwvyaJA0
            vnlBJZed0ZcOr0yuQCFhXvO+mdqjgjGYllschiBIa6+S3+ct9Q2XVQRLPW8QR5hp8waLoJqIfAqi
            hI1xSl8jdC7tuygy0jM0q3dA5/jUFv6wc+QUpfU8PqDw4EIdz8PD+LsHWlYgbU+OrGP2PP32ESuX
            LIZmZhEW61HCKQYwhLrkgZhWaHBDMV3C7+vm5YbLm+M3jqP6rUbNY53SoW1d32XPfDnNJK6GQLA/
            03cdWFRKEbl9JcCIedc2x6os2D/bhoL/+Q8i0FpxjFuyI+rKy+62US1eQ9r9GIRHIpUx4OVoLWVr
            D/mUQOhFTRbQa6xXJN7iweuZZZGeemxsbsc3qy5OgIAJYNUQC1GbJH8gsn13teEzoi7AemL7qryo
            ugfvJ8RpwEva/4mAE4IfJVk7QNwR9F3exaSg1UfAGe0e02dKsHbbhS3fT+tuF6yezGcNdAMQW4jT
            3sfgBVVIAby+eoIs+14998osQiDtUP88KwfEjdETV6omZ4MCG3mIBMXvCm7I+KTNk1e7SjlLnsEX
            Z1SfhV6LpBXzgbothWSomeoRDJuDUoADGMAdcp7SC9SM5VkIKKXPi0bv6YKbVPDiCXC0LfQx0fHq
            2lccG7ves4VH9BnkxAaFh1QjdmzhPSjBwCUAszTTUa/Z3w+LuLVtyFP+jVm68Q4ABU6/UZtZEFkH
            D18nmZ98a6MUQDLa5ImNtcbTP+3j+S0VrNGoEUEtLHKrpNX0nK7Xr3054hyPqP8+kn3kdjjU/qKZ
            bh1rEXDj3vJNH03lzERD+ELImRHGKSLu+RPVCBVx/A7KgpexEfu5jCc8vqTX2E88DTyCXc8YAQko
            yh3w97pxgOt2eljpkbiGcdBx0hM8t2XnxP8n9y/yPyTVxt9s0N0K7t5LFmZvaM6UsflpZwzEGSAs
            KXT4p+IABhPJLR1PdnQDKEZ5t4Cy2tI5ClZHX8OBmu4XkQ1J9CM/NttVSNgWQ/9sb/rKrBfKo2JN
            3mBc7fWjljM1UyQGmY8w1qS4Bz3h2coHm1RwUMYO0kuTnAcWt54e10KMxv1BzapO/CwRGm4A21sl
            bpbIKUOsaL7A0yw8G9S2nCqgn5sWoyrd7QAsuZ2TWWWB3qmpuJasxEMwkyFMPEIGjqv6N5ZyyOwi
            GRuLyiJzvgjfah3X7xMLQ679oNahEIr3XhPlXaGBwIEGP1/qs+F4mfUtHFYL38MdTh/26iKe2DMT
            K7vpP7EXzzMOgIVycnLArXC0gZvYV9akn/eSFeNy0O4ioUewkOPxFLaZ//zDyzQD+AB23X3ETR3C
            60hzTUFAneGAXDfMZadqirCaNdUszPOjzUP351xGeuFfsqCT1wDKnjoVTGGr/GLkOXnWIirZfo+k
            ZRrpHYkrnO/XPzb8k5zs5aaTzuwykUhGsAqy4zMZ36H7eCDjE1QT8/uKWvKeujTh/utY4sSvtmNW
            MkLP433F8NVv9mRTQBfAiPBUjZwFjVI5Y2gjhoaRNPKcpN0XuiZaf3N3GVuTWiyJB18nRSuqhhqY
            mFkqRsiOT3Uw3Drp9h/AM+8KEzBcMkWIGWdOTajx+omw7+RCPiZggJMhe0a59Gy+n8Z/xknJ4UnI
            LQc2k2kUGOP7O2t5NwC98uH2BnssB+qG4h5iZEhDWX0v/STIoU+UrI0efRWhMgEluj811hZXJCj2
            aDXTgCHMkXa9FX+hQmuOpZB7+l0Bei4CIbQx+SxAiDR1AcuDORs8OKLCXTPjg1V/+vDXz8lknwY5
            shjzQYFg/1BdUBI3DoLA2i9q+PwWX7siKYMURqQByvjYLHntz0A3RqhIf2Z9DqD/LwfAo1jsLDon
            M+M/UqyUmRArFYQR6XHANcN59yW/81tCRhdEXcHErPwBYGppXcxlCOAxwNsB7XhwGBuTr/exLAsf
            tWiWuPlpn+XmNcu8BmVkEdWr1ObTeTGhsOLTgHjC9od04zRIq4teMFLWOwJyzT6k+drKbdpsWQc5
            c9oI8NA8nflSg3dgZ1ifZxEkE/SGho0picVOhiIShhGUDvTGJj4PjgaNWmAw0Kmov3Y/iDR5qNp4
            G3GfjDNZjftrz5ZFT75U5RXG05t96tlhU3WRPcUQfaJdAM1Kd7qWbFFXpGh1QyfsC0lKTSXJOP3M
            MxvacL8bw9RZ3YoF/ofFjlkW7zNLiBT0jj9lQ+vqo5xc69yB1997pR5NhFzNMeICpjfPgU+1kUGH
            BJLzwV1iLlLxhq6NE7+2x1Y27+aXtgXMOdtJr7U+7Mo3LqRRxtcDLchpO1j3ke3UiA6cBX/+jF8M
            GDE5S608JU0mJEJFmRjfDqyTRwtHYFhTYw8LsGelBxLKoWTpo74W3vZwjiONYdeNSzFveUiMsL4B
            SPJOPSpP4FWQcj7TDFz3+BVF5BDfrZzQI8O7o1JwCOL6rrqwdDXdpkv7mrc8KBeHCHBHQl5YOqaE
            rJQ0QVw80awKtL+hxtrCWaMizKbj6dWSFYC7LiSR04oCE+VRBfVVSnhB1ediHbfVH+U097Guxg/s
            OMKoI47/XbaHik+/d9bBrqHIiNVm4XcGypEQ2xQgTKCP2IsbcWa4lgkIauzfwUvWkQOMZuLIHckO
            85k+CrRgg4q8PcoZALP9sF6EYbcjBNtkNQaaq0oDR6J5bNgLnsl3u0de22bk8zPrjOJJpNahyWGX
            SuY6qxZks98WWt7l+bpdfesE9fTw8H3kGnT8fQMWpMr24AWV4PqdgAdYtBJeNEMErZCfP74xyj3T
            ydC3kcdc0CuBczS/yqVuI0+z87S/A/S9r/3XCYtlowx23B58l9K+hdRlbfk9bq5sV/QQQCHWBC2b
            m+WVkJxSqK1hi82wEsETJsNNjiKCgpv+bQHnPGV5tHmfnrAQ3V5qV0OMa0HV5quUusdnpbRB2BAM
            /Sl34gvNKYAurl6lhaOiCTFRgpgLLHprlu+NwPUfmLT2IrYhbjbjuxjaHSRGDWUotmoxgiAYMIIg
            FAIBATBaMEIxCzAJBgNVBAYTAkZSMQ4wDAYDVQQHDAVQYXJpczEjMCEGA1UECgwaQm9ndXMgU0xI
            LURTQS1TSEEyLTEyOHMgQ0ECFEOFY6JpAZksOc+8QFcbX6PMx4hFMAsGCWCGSAFlAwQCAaCB5DAY
            BgkqhkiG9w0BCQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3DQEJBTEPFw0yNTA1MDgwNzU5MzVaMC8G
            CSqGSIb3DQEJBDEiBCB/g7Flf/H8U7ktwYFIodZd/C1LH6PWdyhK3dIAEm2QaTB5BgkqhkiG9w0B
            CQ8xbDBqMAsGCWCGSAFlAwQBKjALBglghkgBZQMEARYwCwYJYIZIAWUDBAECMAoGCCqGSIb3DQMH
            MA4GCCqGSIb3DQMCAgIAgDANBggqhkiG9w0DAgIBQDAHBgUrDgMCBzANBggqhkiG9w0DAgIBKDAL
            BglghkgBZQMEAxQEgh6wlt8ES3yBvh1qUcJ1UhbLrJrJoOv1ZkSGyCDVW8GWQ7yL7Ys3P9+CQIwg
            MN27werRpdhrmlJ246DUrRXUE/TL7g3eO5DWQPkGcNn65+J1F849SlI4jChlavgMQklltDbduaLF
            MaVuA+u19OFcCXI3N/7elti6nipuiHthEHvENOZKsZk46/yYqpMThOAdU2tF/UHIbcXX3bgtw9fQ
            zuSpHxG1PDHvajDz57hfV/Ql8eytGALnTK46alD8IPw3auGAcqO2fcfPEBLW7B9V1190FChGPHCI
            nT7StZMqGmDndQoc/wABrDV3WNHjGDzsPy28qzOVzSDqK5p9N4927+S0by/uAGvrsqTehbsSqQpn
            d1lDvW4oKNOwngeb+AdJwNBjDVkVB90y4WoO/Rq0LdoVpK/TJEgSego1C15JbAmy1boFMFd8D0K5
            lhDgd5DDPZPMWw9d/l99XHEb4oCJRCnewlZwIZJcFpX5KunE/E98KSxxTcdPKQXKZLwARu8bO34f
            vrSKJcUwIJlFQyDclRTMxC4reIOSRBUFBxa9dgw9tBcVm4BBX+P5DE5eG01kXP66SrkroEijweDS
            xuuSxAvanJHqPDeuYdL48fBYyB4IWXE60TtTtYqWm45zEtR7X/D8dTy0UgmBpMorZ7bisiHwIaXb
            70SzcUUm7NPb4YS5TXU2Qp8iXTaupEUEE8WclcLwzCHL0fLnaTHaNz+UNK/YaDAyBg1rNREEJHBd
            X6jBZ2qWaza6OuSPVTe7YBIxhOaQp5c1SrpgW9vVkCTCkk3hIsLaCf0V+50JnB2VsAV1+aDWc+pd
            yBJmBuYktU5nZslLJxVYLYRd425BNiHv9en7iUN4HmKX/9VLDAbhumLHIXaRFojO8IO2vv7B6/76
            7IcyOL5QHCKb7ROsVD/0Y8nLLT2P+/yvZOEXDmR5gw6OnNBCxDjiyUo+CIO/mYpNQw1LPNOBE6oi
            +yOopjd6iFieVd8rRzB0/BIi4MR7YamWV6/1i+pvn0qkmopSGW7thGOUL8bCf6p0i9jXXPfMijXi
            0DslGlEQKK0LkBYMnb9+XGtvy2DsZqcKhEseCFBgO6+GQczfbHe/FwVJqkYJ0IH0T4fLKJryhech
            Af6/qvkSWk+wquGYwzScYe1AcFs8dX7aTQqVZvaNxtQzKaivybntjWqZ0aEXJu+IeAFPacGtQcvg
            Rm6K8nwtcwcmDRrb9F14N5el7XWO+jwTyw9kqrzMzNeyHxmFb0OPGFqo7MuWNA+p7XJ8aG+gy7hc
            07evYgZiGv+0am1A0rfEJlQG41sE+RkUO2ozhP4Gn9WhrQpzl3fGS1E96VFBQbvSDTONtX/ArVX3
            oLlK6V9TgVU8G/sKl9zUN3nSSG354v7BwfE3hlM/2iH9DHDYox1XFMO5F8ZIqklds2dvlgMl/Jjp
            VWFHVvsuM5cVZ06IH/liFuEdzUqYPKg/wjM7dlouvFeYbG5vsJOv4wwQLsWBKOXjzF6riynDLTmC
            eMw+BPdN73X1lRuoieti67XVWDTlvvbE/1Skgl47wCKkehyLU9PawCTVuZR0IQ/CvKvmHnbn4vSY
            LCtM6inTuE86zGJ5daZGJl9qZ0Yz/3HRhsA7AVkOy6yC7SzJRyaDowqoNVwZaC5oMQUdvdKEeSQH
            QJE+xsaGsUeWaCvmw2wg32Ws4KaxQfae906zBc9zbXWbKj2ToXYoVsHsYSNenSdq+J8/wC05VuL0
            dTJfi3N4l484pNNARymgWnjBYGkVYHSN7EVesoSUm7MomZtkWEd5tKVleRo6lBW1QpBngGopHtrG
            PUWNXTLA6Y3+Hk+aQNguJCxnLTmKmtB0zwqrd991XqqRB8pFtPKATfe6hPc4NpEpExqaSTYeRjXQ
            o21vQAEyNi6EV0W3ZT+IIt6iuJutwGaOaAxUHqumb3rGoHlPPQJThY9HHeYXi/IXWc13/bXml/zW
            K3G2SY91ot398dEXSLmyY8lU9XfssC8dXZBqWD+U/DPC4XWh/FXun3tHZt/Hwrznp43R6FdN5QW7
            KYRhANaBrb/vAV9gkyu87pmrfem59tbRoOZP1GuaH8Rf7wz3t0LnUug+5KLEorqT+q0+uQAPHx05
            YZM9XCeBhcJmeiI9U/PRBpYhPT9W00BkTxYhICnWKolS9tPmG8IrXi6LLvoEe7nzTfYCf1DeZpsU
            ClMXxgteRN3xLxnkIMW5LIA+nOqwk0DL8E8Kdezjzscl3XCBl2KCqvgQ7PTjp3W4FWklAagSugLG
            19y4Y5C2sfaRWppYjnot5q/LRwclpex+8eCxped577fjAIgtDtSVPlotAo2xn2OeVi2iUo9dUAWr
            uHFuOFRqRT1NL78GdHQGop+chiFncTx8JV98NWBbTsYDlS+AvwBwYf1wSQTduRn+mLMJvb8RZViX
            vnbNJLt7+YhoU0B/EsJeX91pbDXcuNwxlCA8n8CayCT51DBjdvr/XL0HaXM9sXBFQXekxGtZGKvr
            5VKCRrgLtUKKZD5ovwbw0ixNB24BD/5gTGpww269W+ptoAUzqEZZSLGMJ/I92YcE2vCn7r1KMnqC
            EKq9Uw6V0YyOSSmFNA1XkIZFr5U8tQupskoGCStRIgWtDQLra5h7tU6IqLllXsk7wL8ujsw8ccMh
            0oQGTb7aukBMuuU2NZtDfAandKT0Xq1ytmUODq1zS8vY81YW1x6yYdS+gfOqynnEWKmFijbzNWNB
            Edv7Ihhd5qMytge0ay58yM2uBu5bwQmcUGl/7fZuREldC3iEOEi2LNnLtkKu6jrrE6M+7voDqvhR
            HLOtohMz1LpQyuTlw8Ewp3FVr7+JJ3fvnQTEramvELsB7Vfsfg2FUWeklmKboSRv+cBWClog5aGi
            ENWoVfLTzgqIG7oq4k/uU3mBgwXA0i09HQq2xyhQV18fJxK+xFUpEJqcm05zLodzFXKp3nbN2pBN
            Ilw45cw0yDrEzs/InpBv2AHkPDvVhEo6yYNDHl2tBovLskwsT6EULF6Af68v70l3l0QlmuEq5LR4
            lQa1dlc7lECnH3Cx/4+mdsLoVZBa5r8oz1kiKTKZoEMHnYfvHXEw4Li6yxA4heCBvuoMm12imed7
            60w6a436LtYlE78JHhSXFWZJ1Fv32BGLG1pTptQs4a2YLdVv99vnxCTz3yUn+7W0gyho2jOI3bcO
            3CiFtxjNIkw/huy52QBdC09xwZqO1PWI0KDXTmz4KpK4aR6pQEbogEcEjVrTFnGaSGEK1grU9FHH
            l3tVhHt0bO701Rd7AycGInmQOUqa80FyffyQfCc/rKkAtfCW+Glr8tTIRuavktk7kPcvkh9ab8t8
            3DK2cc7Fn7wJsk73b66qUjoaLvMyK0aUWpDVDx7UKctKVizf91Po/YHpnnmIkGG+yNTXeoc7kvee
            YbYfhr6KTx+wLEIQSAal9FUOxq/uUHs8nQjGLDVWlK7ygo6+tE09qSlqLylokPSTUycx6cayjysi
            s+Nheq07hMRhvD2luvFG6zdZnVHvQSwBBqhoQttqx8voLiPbhfjvc2aXJ/Tc1/uq+KRObObvEIZo
            SHmezWyTHqREoON5Opq9TyIlpQQMoiZCu2Ng0t6nOl0V410NVsqU/414XCbxRrZWxNQAqQIjrrsb
            42K0jo7FrjMhEVY9nazV2k1nAzaK75Y64Q6Kh1Bd0X2ktAcKigFhfhgIRbPYx9dB5yvMP6MRhneN
            Hs1gPP3HTkG1WaypwTZYw58AChgOFpELzwdI6c+0t5OrHVaIhDtuTLYi1yqXOaYLSj9GQ8tzfYKv
            sEfsb587oAGiHVltwIEddK148RN9Ta44oZNc28OsvfobAtmRgG26Yf/tORt9BK6CNAMEaOqg5BQV
            tFqvCsZqsXYek411rH3NvACuVJMWc83Q+Ht9YVh62cBuEmptW35qZrCN8xqdqxUeQI6NhfAVmE4i
            ze+Dk3t3I5CRdpB+CLU5ZSUE+eqq88V1hRWHMnhnCBvP2/Pdoj19Yo8lh/14rCIF4jSfICh+x0xk
            V9bpkr8GJPrrqP39Q0VtXpmIt8BX0EBHCo5yH93B/CtSiZ9sCVT6bNmLjTP1PUBxxRkmLUdiq5UI
            Z14wJ9eIi4o5iLM42S4sKicw1NE6+4BC1SLMJAwt8s72rwTEPjBUdAIOGXKFUH1qENsuKJLwbZ42
            rWTiG4Ael3yDgL3438wJzw7EXJ6QYSGVBp3dlcVt8D1XXc4PrCZDSok7yhjw0TxdzDnHWwrxUvQ0
            UARP42oMBl178IHxL4K7Z8rjjxniBIatDiKR6iKCyCc3qgDLdI8Ta9wsl+vqrqZgETbMbghN8w39
            dOmpVwQe4E80yUu2N4rm50JYMM4BuxmE6cQOdNAZQE7ARJ/a11NwDlH9ZEGK86ox/goZpLPJFB47
            S+N7bDgUXicnkqL2ZV5Ui7UYEwQHwsKTsdDtV9Yjqx12cEB+yBVyYTwSnByE+NA7JPB/n5hBCsIC
            JxhECoh7q824Pd3SO9GZemUrry3MfwPlj+qic1CbRMFMViAXI/PWh1EG8eVC4rAtdzQV7BIWHr72
            77i8+GJUGou+hEGTH1Hzq41/QHUYxVqGDRarUelw6eWiBYntVb0u/ONhvyHd2iA0dIqXwF5ZgfHa
            jqC14jrGUzMMCc3/mxKT33g1PXaRMh/wJJlx3ZfVE+hiGsZDV9KRYoziK3Aemz9JtxScAfELVkQB
            MGmzlldFEHwQIX/qnu/su6zC+Nwoxm7SNJlgsAtKuZ1wgSPQuml5ioMJ8fAJ6ZMM/4+YxHMu4IfO
            eJb00E5KI9CxoWq5pdNjaw7iB8EREpKocNrqOOwShZ3Wvb4L7Fsy2zQAa/AeRCXZPlwgwlS5QhWk
            qZL7ABTJBSgf4d/vAI3YIlN5AwURdfUm9Es1QMayYafiQpE1AOTQeF3b2OwrGhsbWlndZ6wPrHVo
            bVTeuHv2uySMFV+XBeJzU+HkmoqdH+pBo93Ze4vi7Ip0A0Uuh/4NX0Mzd6phKZsyPlcvvaTXdXz+
            D5cSA6ei9rmQryneHY7RgfJAymtjWgijwZUG1EV2OmIAlqnwy861Cwq3ZcR1AoIdGlmQVCr+FKWd
            SpHZmbcVxRuEdPYl0NuPDioD6vP2TvybWt1xoQ/Hs3165grb3hAwy2Qn/7aHsZUnehYRNRRQOrAe
            OhD0UtWynBaqbmn8MfphjUSLapj99X+zv/tt5bsEuyQ7Kf9oy8tLaGLL0pLEBAMtjAod5kKrWWfs
            QhtMItwIxJscGmJrkMkwuqxLChCDYRStVZoho63kp34ZAL9N2bRsG6af+yxZnAGfGGJiUtk54VeU
            14hX4C6xkL3wMnhfygE8cj6rfLJ0xc673A7Tp590r1oR4r/nexbjbZM7AxIE3ytJbAtB8D/Nq3cD
            EYc6TCK9M3cmmk4E8xd/iZrBbDonRiiwomesigU4kQUQRds4tXrf0X4okKEI7/HD/yPpATFEBD7c
            +jiSnJd7tUOjCwwKZSvG+VBU4KYirOdJBBayG57UwSqs+71cDutif12vb6t4uUzFqCEN9KIBUDCW
            jw1HwybXnyCblC1ts/rciBrMK225bnpRtWF0MxCAhSBm43447oEJZPsOUfDgXCzNq9V23UXiYcvb
            Yt5ppsod8plt9m5nt6dbWM02VSp0YQgDdOkISVrnTbzNNP7Lg7qz3IuMg3KSeBLZ7jOUh5OLH4de
            FZBzg3hU2/j+LjeOp08q18LtT10r4ZARYUACmK84BOv8UCVueCi5Haos8xgRj/JMHjF13VlkVJp5
            06jM0duZGD35LEYGnnKDsL8+4p7HkUsW2e+QlVUhjZQPoBl45hXWOmvIUMjQaBYcctiqT2ocv+fB
            /hwwGtH6CYZkG5mTYgpXKyqpUMkhV5hvlZ1jyQaT8KD4RyDf+PgHsW7ZsfXuCCE4sLgOEuwZvOHx
            tZYOL7SoD/A9gMyogeSfAN7f3hXjBy6O7sA/Xr2atgnhJsWRirW8JIeZSApu8JSnWGiiQjzNXHaX
            clUlBt8GpTHeyvbMLWaC8+HpXPT/imJgZS67B1lzsgYUN3TBRZFEZjdfLGW7C0Vqw9d/+0kbzmLn
            9vJZfAu6IUorHjg6m98/dlnoXvkPsIZzZuhoLu9XMk1HnT7MYdLRW9NvlQhFwRsTaL9LUF9A2xII
            9M+ntItAdAlmquVCylKd+/xS9pDCTz2hdA7Cgqh+QtjfEpIYYWaS3ysRGXmC8LsRmhDX+pj6Y7cD
            CsEPALrwhiQYfYNLWSwijQbE4J0/inX5DOADjHESasxm8XwzXrumQch28E8UJcGU9geniA4YOuVk
            K8fGPoaF8oBfdyOhAklWMGrZWwBrUpNDLJ63HLPwVl/T8PbQucm+5DQp/4TocqSYFa4FD3l7qrFc
            jjgZK1Pb4xRemmfyfm3pCwTHSLvaVdTejJRO29JtLX/sDl10qu1a9Xp3tfE3ZyynPWheHwgpyCdV
            s3pXZ7HPOiJPKNdwB+F7/7SMMkloV3Xw3qYHH0K7vgYyNONMwIs287N2QLV3UrU3Sg9NJ1ds3bJM
            Y9BcNQq+XuSkMRS1SEExQwC1g8dHd8+Imyu8peTEGRdyvH8Y+VSkrjvSFsgkkDmfa+eBfPTh3k1M
            YPw/I9RPmzvHmh7c3FuTuCbs7PyMtLM5pgmWlk7A76kA3lewfDRPy78zEm/jlvWDOSzrDJTr21ES
            00iRc+zzQJ3ptdwrXWkLyYe4VRRhwJWFlJ3zj+UuzdRCJH7FD/IaPVbZrRTzt6yL51LA7uL0jn/X
            GNPyfvqDq+uGdedUJjFHS7GzZKMxf2Zdfzfqg3FK0GMAVRG84M7a0usVSUVO0XQWPe3OYAP+ByV+
            D54i9qroov1gJ4Iz7KTN0RElWeQuo3+B0F6IWXXqmseK/QLKPbGXcDAYKWYFHqF1uuJyOWXUCNrT
            Lws977mZtNPW4M22T4HImJSWXDAASGF8PMScxltqLVX3+NFMealtxYM4AMtCHbZm25hiMlmNeNld
            M3YhxV/tFQuLHqSXDINNFZe1XwCQn2+ynfDdEyhSdza0QEp20I3epHNxSHvui/eNdesTiptAmy/H
            bR5fiXrXe8pJZqlqh84Daedga66np2hrM6kKEPn+40Bpl5qSKtKrdqATj4rXHvPOHcamXAH5A3/M
            ulJZMD3QaK0Xe5MbIP8j8Mim7YDQA6C+eKno+KGoR3t1s7pZAV/frNH7ouuYrh1dAXpDVSW89Am2
            wJDXyGoc42Uzglaxeu9Lzwr4R6DPJVe9jMMTS6n57J0oRvr87j28aJayMbh5f9MaUI7Dn8zZtBgO
            +eriGqbNlIh6N+Kc5rbEInRPIVzva/+voKh0GCb1EtCIxOrCTallQJmapl/dAoM5/JKe6b9QLgIl
            f+fXOYS88yJfLmx5R1ber+osJDUgeDPo46H3dfiqa8HSVKrHRL/N6tzif2GPSJl0vI7cnT1RAk3u
            pnOh5+cynWF7eENGcpk+T1A3DWbomq3OP7UbWfi6JoVJChn8o2s8I7IggorVhVBJLW5nN5tawn/8
            KUadUl5TuqCFFYj8CtIHf+O86q0qSL4yGSW9PHL14wna/H1Nyd+his26tbIZIfZeJkmGeqWUrUth
            CK4vX94FhurCNG6+Lxd57jIdhyb0bBVYfYGHcSAX5mmRYaalO7sLOYrapKS3QeNW39FnvbDLTH5m
            T6FhjCN+AtK4+x7/rby2FDnp5yu5QZBtXECRmsqwWbn5SOw2/J0QpU5IKaUcNd0qqTZLWmEI9IG8
            MSRQhmtYIjcm2E0gzQ7drFsolzbXrT5Y8yvRBclL3pPacT/d5XM1uu4CMV1P0MkJDWCJGTPdgo9Q
            kYeLcsd3WPL5IgnhKbFmLXtvtI5SeRLPNvqa1esD9rNEl1EUxLzitiS+0uj/oqku91oPF1iTMcxH
            GatNzhL34akdIkx/eNQ+mc1RfGo+Fsmwpz3PVEfGDKBAp68wgfNCDYzhpjIfDDXipm0yaka9dtJS
            28kDjwyy4iPWbGqLyDKaciteKyFoIDWROy/XQxL44r8vPwLLcb2ILphd8Kz4v+Ok/l/gEYiQuyrw
            odlLA89DK9DAKQxgrhAQMonaATL1ZvDqbKRBWN15BVOMgrDVSx307VYmhQtHaopMjDL7oMy2YcDN
            vCqbI/0NjFW9tauBLRowWD3Pu9Ghc/n8KjwewMogwSvW8MaVPX5iTaMEiQNuXYHCsJ1BqYQ++5eb
            zZoV7ZmUnRajyOI/HSPLu5AmFaOqDv89cSXbH9sgh9AO9x7ysZG6mbjJOtUrurDt8ZedUKKWyyIv
            nT0GM1Tx11D7MKc7K9PJw6OxUdGH0bhV5loCpGfl+BotzTFJUCVp9GI90vL6WQJM/rt+4AWdqy+w
            pHutuIJrJqsP3Dwyk9F56qJsWV/jXLFhqZueYWiuZKPjB9xipBksgj3yPLHRjIIH3UQ2ireh2dMZ
            uFt9p4aL9QwOhmy2nLYM09eRzn3flP3TtFAqAj0aR3YlLKBaJ5rlBVpaHshP0KqzHWkU7tqAleBm
            mCExxvx8L+2iWQ6mxIg4IelH4OVTSPQwGdiORXLcTGkpFtVJSPkAzfp99yV9o1LY5E2AF7AFUjL7
            plgiZBqt2NcEWpVKHlBcycTwSSPHT9W0NIybknC7Na9DXsOID4fbgDgBW2X0LrTHp6D8BlWleh1t
            4lywT5IFZOuqSjpgB3Eewfz3G4wKBVa3b2r5aBRBK2yFfOBAOeAj++HjOfVriHHOyXXosLjBjbP9
            MEpnBJGGpTaAn87gqJQQUuQa8IHTC8GoED4aya5HfNd687OizSw2+HTsVkkAXh1w9EJJsjoBBJqV
            d2m1UnKQXciUo2TZccd4GQHsNzonyosSdEVaIgVk11hZexiTx6tiyIB47SudAOXzErzVN2MeLN6G
            q8XGxKvLQxEQykqBQzXjDcS/Uk8neGmCtTKZD7FxbSQ+03x0OZgZdxkpP0WDIHdcymKR2BlT/US9
            zhXc/6cNWuTDvPbTqpoijASHW4s6YeDbWDgfqMT98nUVsBAEAGF8aBzMVIL6U0OSIlMpie3IU/2o
            xn80YuXw6NeYAbh4JKU+ERF/MZ12BwuhDahfYtIluBagrM0F0tsEOlFTxcwQ0OAd1YGMVZN2lGIl
            xb9TB49YMx1wX+Rao9JTl2vlTax8SRmUXWwJvEvhNZt00xbfqQldgkFJs+HgqAnPiIGTc0w2vgCk
            faz/3fnPHNz0jo/Q9zAe4LL4RrSgDfe8VxpjoXIzFV4vl8lkujWQzoz6T9AIfukMrU7khmFZDzy4
            EIcY6sqx9jbOu8Cp0WXQatCIKrRh4D6Hu0TjV3BV2O5ws/b1qtiVyTTMbKocrzhQkcNft6w+YeTV
            XsZG1yUKIyjmBJjc75ZeDuqvNkg/kuRonEAigar+5X/xRvR8ms5snA+XU36SQHhI6GLQseBepICV
            1HDhoCGkkyxH3d4kqinnc85x/EklcFr6I0Rccw8Tn7h5BRtWfgwFHjkW9pLpFFbWI/alET5zgxeE
            PlTKmWt97sSYULna2VzRAcpv+bDM1CODNyTEPY/mW0/eNSlw4GZRjCSA7+1WElk4Hl2oNHnRXWwQ
            Ci9gS+bONqMVob0vZkdTq9kzGH31xKezHbn+Dn+QLcWN53LrrvA1aVffcs1BlB2mKR3w3Rl00U1M
            1v0g6+1lGWOOIPkp/+fxC1tQUgnCGgWXmzDCSKHsd4Hvx0f5CE9e5xyrioWUf4t5BO6bxNC51i8l
            3eGGzK6SxMyv7KuE5jT35Xq9EoE+CFu8JawEI3NPjaTtWQfWS3ay9v/G3TPSf9+mzURtqnGxg9r4
            nqHXRkXRDeFwqlZuciqM9dmzP817Tv5hO9O8on8EMLoLPiKMkEbqVLYg4Ui3GXbQRLk36gVj6ETP
            6RSoKAP9XOHYQK1Q7GxyYe1CcWR6KPAZWxfnTiRlkSQ0jml58pMqoSK5X8r7V08IlSwH/eUtYyjX
            wLhbgCEtXG1uRj0b4sEqxZ1VPaSWSOr5hVQeJ6lHt6KrkVEBiBGts5vzjMzUt6qox6B5PCKKcQF3
            RkKklmZ0r46KWWiLuTMWZrqIg4QGtgqB+/DfKZ1c6HJKsTTMJAD/dhnhSj1/xJnTy1yNdgHHVz9X
            eoon4dI/lZgw+aNG/RllR4wzdV2SwQ0NYsFPpytAbruXAQK+GjafVen7M0qJzeHEt5X9xy+gFLTv
            zmzaoxsCS1ssui3oe+xxYDcFepWkZzPqkLdXxjpeKkoCTb7MYNsPWwTVRtPR180+rVJEyZy13AI4
            IIX8+kDHVPFCdylE6WtGT/FI6YSL6Unt8GLLYJPk4Oq6B0WovXU6b0+XyoLXlndmkO3NDjsrYJxS
            mYW8sLTRE2Bn5fBSWGFTyaq5cmwDVkNjjFb4iSbwEf/t7KqRAr2JnQRi7/kw+cmY2AqIC8U+ZEU3
            nh4eHRqklwlDFUa+QHqrBRjOkVfw80zBaX4KE5qBgpgLLHprlu+NwPUfmLT2IrYhbjbjuxjaHSRG
            DWUotmo=
            """);
    }
}
