// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System.Text.Json;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.Logging;
using Microsoft.Extensions.Logging.Abstractions;

namespace Microsoft.AspNetCore.Http.HttpResults;

public class ProblemResultTests
{
    [Fact]
    public async Task ExecuteAsync_UsesDefaults_ForProblemDetails()
    {
        // Arrange
        var details = new ProblemDetails();

        var result = new ProblemHttpResult(details);
        var stream = new MemoryStream();
        var httpContext = new DefaultHttpContext()
        {
            RequestServices = CreateServices(),
            Response =
                {
                    Body = stream,
                },
        };

        // Act
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.Equal(StatusCodes.Status500InternalServerError, httpContext.Response.StatusCode);
        stream.Position = 0;
        var responseDetails = JsonSerializer.Deserialize<ProblemDetails>(stream);
        Assert.Equal("https://tools.ietf.org/html/rfc7231#section-6.6.1", responseDetails.Type);
        Assert.Equal("An error occurred while processing your request.", responseDetails.Title);
        Assert.Equal(StatusCodes.Status500InternalServerError, responseDetails.Status);
    }

    [Fact]
    public async Task ExecuteAsync_UsesDefaults_ForValidationProblemDetails()
    {
        // Arrange
        var details = new HttpValidationProblemDetails();

        var result = new ProblemHttpResult(details);
        var stream = new MemoryStream();
        var httpContext = new DefaultHttpContext()
        {
            RequestServices = CreateServices(),
            Response =
                {
                    Body = stream,
                },
        };

        // Act
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.Equal(StatusCodes.Status400BadRequest, httpContext.Response.StatusCode);
        stream.Position = 0;
        var responseDetails = JsonSerializer.Deserialize<HttpValidationProblemDetails>(stream);
        Assert.Equal("https://tools.ietf.org/html/rfc7231#section-6.5.1", responseDetails.Type);
        Assert.Equal("One or more validation errors occurred.", responseDetails.Title);
        Assert.Equal(StatusCodes.Status400BadRequest, responseDetails.Status);
    }

    [Fact]
    public async Task ExecuteAsync_IncludeErrors_ForValidationProblemDetails()
    {
        // Arrange
        var details = new HttpValidationProblemDetails(new Dictionary<string, string[]>
        {
            { "testError", new string[] { "message" } }
        });

        var result = new ProblemHttpResult(details);
        var stream = new MemoryStream();
        var httpContext = new DefaultHttpContext()
        {
            RequestServices = CreateServices(),
            Response =
                {
                    Body = stream,
                },
        };

        // Act
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.Equal(StatusCodes.Status400BadRequest, httpContext.Response.StatusCode);
        stream.Position = 0;
        var responseDetails = JsonSerializer.Deserialize<HttpValidationProblemDetails>(stream);
        Assert.Equal(StatusCodes.Status400BadRequest, responseDetails.Status);
        var error = Assert.Single(responseDetails.Errors);
        Assert.Equal("testError", error.Key);
    }

    [Fact]
    public async Task ExecuteAsync_GetsStatusCodeFromProblemDetails()
    {
        // Arrange
        var details = new ProblemDetails { Status = StatusCodes.Status413RequestEntityTooLarge, };

        var result = new ProblemHttpResult(details);

        var httpContext = new DefaultHttpContext()
        {
            RequestServices = CreateServices(),
        };

        // Act
        await result.ExecuteAsync(httpContext);

        // Assert
        Assert.Equal(StatusCodes.Status413RequestEntityTooLarge, details.Status.Value);
        Assert.Equal(StatusCodes.Status413RequestEntityTooLarge, result.StatusCode);
        Assert.Equal(StatusCodes.Status413RequestEntityTooLarge, httpContext.Response.StatusCode);
    }

    [Fact]
    public void ExecuteAsync_ThrowsArgumentNullException_WhenHttpContextIsNull()
    {
        // Arrange
        var result = new ProblemHttpResult(new());
        HttpContext httpContext = null;

        // Act & Assert
        Assert.ThrowsAsync<ArgumentNullException>("httpContext", () => result.ExecuteAsync(httpContext));
    }

    [Fact]
    public void ProblemResult_Implements_IStatusCodeHttpResult_Correctly()
    {
        // Act & Assert
        var result = Assert.IsAssignableFrom<IStatusCodeHttpResult>(new ProblemHttpResult(new() { Status = StatusCodes.Status416RangeNotSatisfiable }));
        Assert.Equal(StatusCodes.Status416RangeNotSatisfiable, result.StatusCode);
    }

    [Fact]
    public void ProblemResult_Implements_IStatusCodeHttpResult_Correctly_WithDefaultStatusCode()
    {
        // Act & Assert
        var result = Assert.IsAssignableFrom<IStatusCodeHttpResult>(new ProblemHttpResult(new()));
        Assert.Equal(StatusCodes.Status500InternalServerError, result.StatusCode);
    }

    [Fact]
    public void ProblemResult_Implements_IValueHttpResult_Correctly()
    {
        // Arrange
        var value = new ProblemDetails();

        // Act & Assert
        var result = Assert.IsAssignableFrom<IValueHttpResult>(new ProblemHttpResult(value));
        Assert.IsType<ProblemDetails>(result.Value);
        Assert.Equal(value, result.Value);
    }

    [Fact]
    public void ProblemResult_Implements_IValueHttpResultOfT_Correctly()
    {
        // Arrange 
        var value = new ProblemDetails();

        // Act & Assert
        var result = Assert.IsAssignableFrom<IValueHttpResult<ProblemDetails>>(new ProblemHttpResult(value));
        Assert.IsType<ProblemDetails>(result.Value);
        Assert.Equal(value, result.Value);
    }

    [Fact]
    public void ProblemResult_Implements_IContentTypeHttpResult_Correctly()
    {
        // Act & Assert
        var result = Assert.IsAssignableFrom<IContentTypeHttpResult>(new ProblemHttpResult(new()));
        Assert.Equal("application/problem+json", result.ContentType);
    }

    private static IServiceProvider CreateServices()
    {
        var services = new ServiceCollection();
        services.AddTransient(typeof(ILogger<>), typeof(NullLogger<>));
        services.AddSingleton<ILoggerFactory>(NullLoggerFactory.Instance);

        return services.BuildServiceProvider();
    }
}
