/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file assistant.c
 * \brief Assistant dialog
 */

#include <ffgtk.h>

/** New profile, valid during assistant progress */
static struct sProfile *psNewProfile = NULL;

/**
 * \brief Get host name out of assistant structure
 * \param psAssistant assistant widget
 * \return host name
 */
static gchar *assistantGetHost( GtkAssistant *psAssistant ) {
	GtkWidget *psEntry = g_object_get_data( G_OBJECT( psAssistant ), "router_entry" );
	GtkWidget *psTreeView = g_object_get_data( G_OBJECT( psAssistant ), "detected_treeview" );
	GtkWidget *psRadioAuto = g_object_get_data( G_OBJECT( psAssistant ), "radio_detected" );
	GtkTreeSelection *psSelection;
	GtkTreeModel *psModel;
	GtkTreePath *psPath;
	GList *psSelRows;
	GtkTreeIter sIterator;
	GValue sName = { 0 };
	gchar *pnItem;

	if ( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psRadioAuto ) ) == TRUE ) {
		psSelection = gtk_tree_view_get_selection( GTK_TREE_VIEW( psTreeView ) );
		psSelRows = gtk_tree_selection_get_selected_rows( psSelection, &psModel );

		psPath = ( GtkTreePath * ) g_list_nth_data( psSelRows, 0 );
		gtk_tree_model_get_iter( GTK_TREE_MODEL( psModel ), &sIterator, psPath );
		gtk_tree_model_get_value( GTK_TREE_MODEL( psModel ), &sIterator, 0, &sName );
		pnItem = g_strdup( g_value_get_string( &sName ) );
		g_value_unset( &sName );
	} else {
		pnItem = g_strdup( gtk_entry_get_text( GTK_ENTRY( psEntry ) ) );
	}

	return pnItem;
}

/**
 * \brief Get password from assistant structure
 * \param psAssistant assistant widget
 * \return password
 */
static const gchar *assistantGetPassword( GtkAssistant *psAssistant ) {
	GtkWidget *psEntry = g_object_get_data( G_OBJECT( psAssistant ), "password_entry" );

	return gtk_entry_get_text( GTK_ENTRY( psEntry ) );
}

/**
 * \brief Get profile name from assistant structure
 * \param psAssistant assistant widget
 * \return profile name
 */
static const gchar *assistantGetProfileName( GtkAssistant *psAssistant ) {
	GtkWidget *psEntry = g_object_get_data( G_OBJECT( psAssistant ), "profile_entry" );

	return gtk_entry_get_text( GTK_ENTRY( psEntry ) );
}

/**
 * \brief Find UPNP devices through SSDP and add them to the treeview
 * \param psTreeView treeview containing detected devices
 */
static void findDevices( GtkWidget *psTreeView ) {
	GtkListStore *psStore = g_object_get_data( G_OBJECT( psTreeView ), "liststore" );
	GtkTreeIter sIter;
	char *pnName;
	unsigned short nPort;

	if ( ssdpDiscover( &pnName, &nPort ) ) {
		return;
	}

	gtk_list_store_clear( psStore );
	gtk_list_store_append( psStore, &sIter );
	gtk_list_store_set( psStore, &sIter, 0, g_locale_to_utf8( pnName, -1, 0, 0, 0 ), -1 );
}

/**
 * \brief Name entry input changed, if we have some unique entry set page to complete else false
 * \param psEntry editable entry widget
 * \param psAssistant assistant widget
 */
static void nameEntryChanged( GtkEditable *psEntry, GtkAssistant *psAssistant ) {
	const gchar *pnName = gtk_entry_get_text( GTK_ENTRY( psEntry ) );
	gint nNum = gtk_assistant_get_current_page( psAssistant );
	GtkWidget *psPage = gtk_assistant_get_nth_page( psAssistant, nNum );
	GtkWidget *psLabel = g_object_get_data( G_OBJECT( psAssistant ), "warning_label" );
	GList *psList = getProfiles();
	struct sProfile *psProfile;

	while ( psList != NULL ) {
		psProfile = psList -> data;
		if ( psProfile != NULL && !strcmp( psProfile -> pnName, pnName ) ) {
			gtk_assistant_set_page_complete( psAssistant, psPage, FALSE );
			gtk_label_set_text( GTK_LABEL( psLabel ), _( "Profile name already in use!" ) );
			return;
		}
		psList = psList -> next;
	}

	gtk_label_set_text( GTK_LABEL( psLabel ), "" );
	gtk_assistant_set_page_complete( psAssistant, psPage, TRUE );
}

/**
 * \brief Entry input changed, if we have some entry set page to complete else false
 * \param psEntry editable entry widget
 * \param psAssistant assistant widget
 */
static void entryChanged( GtkEditable *psEntry, GtkAssistant *psAssistant ) {
	const gchar *pnPassword = gtk_entry_get_text( GTK_ENTRY( psEntry ) );
	gint nNum = gtk_assistant_get_current_page( psAssistant );
	GtkWidget *psPage = gtk_assistant_get_nth_page( psAssistant, nNum );

	gtk_assistant_set_page_complete( psAssistant, psPage, ( strlen( pnPassword ) > 0 ) );
}

/**
 * \brief Column within upnp detected devices clicked, select router
 * \param psSelection tree selection
 * \param psAssistant assistant widget
 */
static void columnClicked( GtkTreeSelection *psSelection, GtkAssistant *psAssistant ) {
	GtkTreeModel *psModel;
	GtkTreePath *psPath;
	GList *psSelRows;
	GtkTreeIter sIterator;
	GValue sName = { 0 };
	gint nNum = gtk_assistant_get_current_page( psAssistant );
	GtkWidget *psPage = gtk_assistant_get_nth_page( psAssistant, nNum );

	psSelRows = gtk_tree_selection_get_selected_rows( psSelection, &psModel );

	if ( g_list_length( psSelRows ) == 1 ) {
		psPath = ( GtkTreePath * ) g_list_nth_data( psSelRows, 0 );
		gtk_tree_model_get_iter( GTK_TREE_MODEL( psModel ), &sIterator, psPath );
		gtk_tree_model_get_value( GTK_TREE_MODEL( psModel ), &sIterator, 0, &sName );
		g_value_unset( &sName );
		gtk_assistant_set_page_complete( psAssistant, psPage, TRUE );
	} else {
		gtk_assistant_set_page_complete( psAssistant, psPage, FALSE );
	}
}

/**
 * \brief Switch between "Manual" and "Auto" occurred, handle both cases
 * \param psRadio radio button which gets the event
 * \param pUserData depends on input button
 */
static void radioCallback( GtkRadioButton *psRadio, gpointer pUserData ) {
	if ( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psRadio ) ) == TRUE ) {
		gtk_widget_set_sensitive( GTK_WIDGET( pUserData ), TRUE );
		if ( G_OBJECT_TYPE( pUserData ) == GTK_TYPE_ENTRY ) {
			entryChanged( pUserData, g_object_get_data( pUserData, "assistant" ) );
		} else if ( G_OBJECT_TYPE( pUserData ) == GTK_TYPE_TREE_VIEW ) {
			GtkTreeSelection *psSel = gtk_tree_view_get_selection( GTK_TREE_VIEW( pUserData ) );
			columnClicked( psSel, g_object_get_data( pUserData, "assistant" ) );
		}
	} else {
		gtk_widget_set_sensitive( GTK_WIDGET( pUserData ), FALSE );
	}
}

/**
 * \brief Check for valid host name
 * \param pnHostName
 * \return 0=success, else error
 */
static gint checkHostName( gchar *pnHostName ) {
#ifndef G_OS_WIN32
	GList *psList;
	GResolver *psResolver;

	psResolver = g_resolver_get_default();
	psList = g_resolver_lookup_by_name( psResolver, pnHostName, NULL, NULL );
	g_object_unref( psResolver );

	if ( psList != NULL ) {
		g_resolver_free_addresses( psList );
		return 0;
	}

	return 1;
#else
	return 0;
#endif
}

/**
 * \brief Check for valid password
 * \param psAssistant assistant widget
 * \return 0=success, else error
 */
static int checkPassword( GtkAssistant *psAssistant ) {
	gint nError = 0;

	/* Initialize router structure */
	routerInit( psNewProfile );

	nError = routerLogin( psNewProfile );
	if ( nError == 0 ) {
		routerLogout( psNewProfile );
	}

	return nError;
}

/**
 * \brief Check for open monitor port
 * \param psProfile current profile
 * \return error code, 0=success else errror
 */
static gint checkMonitorPort( struct sProfile *psProfile ) {
#ifndef G_OS_WIN32
	GSocketClient *psClient = NULL;
	GSocketConnection *psConnection = NULL;
	const gchar *pnHostName = NULL;

	psClient = g_socket_client_new();
	if ( psClient == NULL ) {
		return 3;
	}

	pnHostName = routerGetHost( psProfile );

	psConnection = g_socket_client_connect_to_host( psClient, pnHostName, 1012, NULL, NULL );
	g_object_unref( psClient );
	if ( psConnection != NULL ) {
		g_object_unref( psConnection );
	} else {
		Debug( KERN_WARNING, _( "could not open socket, abort\n" ) );
		return 2;
	}
#endif

	return 0;
}

/**
 * \brief Activate monitor port through call number interface
 * \param psProfile current profile
 * \return 0 for success
 */
static int activateMonitorPort( struct sProfile *psProfile ) {
	return routerCallNumber( psProfile, "#96*5*", NULL );
}

/**
 * \brief Check for open capi port
 * \param psProfile current profile
 * \return error code, 0=success else error
 */
static int checkCapiPort( struct sProfile *psProfile ) {
#ifndef G_OS_WIN32
	GSocketClient *psClient = NULL;
	GSocketConnection *psConnection = NULL;
	const gchar *pnHostName = NULL;

	psClient = g_socket_client_new();
	if ( psClient == NULL ) {
		return 3;
	}

	pnHostName = routerGetHost( psProfile );

	psConnection = g_socket_client_connect_to_host( psClient, pnHostName, 5031, NULL, NULL );
	g_object_unref( psClient );
	if ( psConnection != NULL ) {
		g_object_unref( psConnection );
	} else {
		Debug( KERN_WARNING, _( "could not open socket, abort\n" ) );
		return 2;
	}
#endif

	return 0;
}

/**
 * \brief Activate capi port through call number interface
 * \param psProfile current profile
 * \return 0 for success
 */
static gint activateCapiPort( struct sProfile *psProfile ) {
	return routerCallNumber( psProfile, "#96*3*", NULL );
}

/**
 * \brief get valid router ports
 * \param psProfile current profile
 * \return 0 on success, else error
 */
static gint checkGetPorts( struct sProfile *psProfile ) {
	if ( routerGetActivePorts( psProfile ) != 0 ) {
		return 1;
	}

	return 0;
}

/**
 * \brief Start scan button clicked
 * \param psButton button widget
 * \param psAssistant assistant widget
 */
static void startScanClicked( GtkWidget *psButton, GtkAssistant *psAssistant ) {
	gint nNum = gtk_assistant_get_current_page( psAssistant );
	GtkWidget *psPage = gtk_assistant_get_nth_page( psAssistant, nNum );
	GtkWidget *psCheck = g_object_get_data( G_OBJECT( psAssistant ), "check" );
	gint nError;

	gtk_widget_set_sensitive( GTK_WIDGET( psButton ), FALSE );

	/* Check hostname */
	nError = checkHostName( assistantGetHost( psAssistant ) );
	if ( nError != 0 ) {
		if ( nError == 1 ) {
			checklistSetError( psCheck, _( "Error: Invalid host name" ) );
		} else {
			checklistSetError( psCheck, _( "Error: Unknown router" ) );
		}
		gtk_widget_set_sensitive( GTK_WIDGET( psButton ), TRUE );
		return;
	}
	checklistCompleteStage( psCheck );

	GTK_FLUSH;

	/* Check password */
	if ( checkPassword( psAssistant ) != 0 ) {
		checklistSetError( psCheck, _( "Error: Wrong password" ) );
		gtk_widget_set_sensitive( GTK_WIDGET( psButton ), TRUE );
		return;
	}
	checklistCompleteStage( psCheck );

	GTK_FLUSH;

	/* Get phone settings */
	if ( routerGetPhoneSettings( psNewProfile ) != 0 ) {
		checklistSetError( psCheck, _( "Error: Could not get phone settings" ) );
		gtk_widget_set_sensitive( GTK_WIDGET( psButton ), TRUE );
		return;
	}
	if ( psNewProfile -> sRouterInfo.pnCountryCode != NULL ) {
		routerSetCountryCode( psNewProfile, psNewProfile -> sRouterInfo.pnCountryCode );
	}
	if ( psNewProfile -> sRouterInfo.pnAreaCode != NULL ) {
		routerSetAreaCode( psNewProfile, psNewProfile -> sRouterInfo.pnAreaCode );
	}

	checklistCompleteStage( psCheck );

	GTK_FLUSH;

	/* Get firmware */
	if ( routerFirmwareDetect( psNewProfile ) != 0 ) {
		checklistSetError( psCheck, _( "Error: Could not detect firmware" ) );
		gtk_widget_set_sensitive( GTK_WIDGET( psButton ), TRUE );
		return;
	}
	checklistCompleteStage( psCheck );

	GTK_FLUSH;

	/* Get ports */
	if ( checkGetPorts( psNewProfile ) != 0 ) {
		checklistSetError( psCheck, _( "Error: Could not get active ports" ) );
		gtk_widget_set_sensitive( GTK_WIDGET( psButton ), TRUE );
		return;
	}
	checklistCompleteStage( psCheck );

	GTK_FLUSH;

	/* Check monitor port */
	if ( checkMonitorPort( psNewProfile ) != 0 ) {
		GtkWidget *psDialog = NULL;
		gint nRet = -1;

		if ( routerHasDial( psNewProfile ) == TRUE ) {
			psDialog = gtk_message_dialog_new( GTK_WINDOW( psAssistant ), GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO, _( "The port for the callmonitor is currently not active" ) );
			gtk_message_dialog_format_secondary_text( GTK_MESSAGE_DIALOG( psDialog ), _( "Do you want to enable this port?" ) );
		} else {
			psDialog = gtk_message_dialog_new( GTK_WINDOW( psAssistant ), GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION, GTK_BUTTONS_OK, _( "The port for the callmonitor is currently not active" ) );
			gtk_message_dialog_format_secondary_text( GTK_MESSAGE_DIALOG( psDialog ), _( "Please dial #96*5* with your phone to enable it" ) );
		}
		
		gtk_window_set_title( GTK_WINDOW( psDialog ), _( "Unlock port" ) );
		nRet = gtk_dialog_run( GTK_DIALOG( psDialog ) );
		gtk_widget_destroy( GTK_WIDGET( psDialog ) );

		if ( nRet == GTK_RESPONSE_YES && activateMonitorPort( psNewProfile ) != 0 ) {
			GtkWidget *psDialog = gtk_message_dialog_new( GTK_WINDOW( psAssistant ), GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION, GTK_BUTTONS_OK, _( "The port for the callmonitor could not be activated" ) );
			gtk_message_dialog_format_secondary_text( GTK_MESSAGE_DIALOG( psDialog ), _( "Please dial #96*5* with your phone to enable it" ) );
			gtk_window_set_title( GTK_WINDOW( psDialog ), _( "Unlock port" ) );
			nRet = gtk_dialog_run( GTK_DIALOG( psDialog ) );
			gtk_widget_destroy( GTK_WIDGET( psDialog ) );
		}
	}
	checklistCompleteStage( psCheck );

	GTK_FLUSH;

	/* Check capi port */
	if ( checkCapiPort( psNewProfile ) != 0 ) {
		GtkWidget *psDialog = NULL;
		gint nRet = -1;

		if ( routerHasDial( psNewProfile ) == TRUE ) {
			psDialog = gtk_message_dialog_new( GTK_WINDOW( psAssistant ), GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO, _( "The port for the capi fax is currently not active" ) );
			gtk_message_dialog_format_secondary_text( GTK_MESSAGE_DIALOG( psDialog ), _( "Do you want to enable this port?" ) );
		} else {
			psDialog = gtk_message_dialog_new( GTK_WINDOW( psAssistant ), GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION, GTK_BUTTONS_OK, _( "The port for the capi fax is currently not active" ) );
			gtk_message_dialog_format_secondary_text( GTK_MESSAGE_DIALOG( psDialog ), _( "Please dial #96*3* with your phone to enable it" ) );
		}

		gtk_window_set_title( GTK_WINDOW( psDialog ), _( "Unlock port" ) );
		nRet = gtk_dialog_run( GTK_DIALOG( psDialog ) );
		gtk_widget_destroy( GTK_WIDGET( psDialog ) );

		if ( nRet == GTK_RESPONSE_YES && activateCapiPort( psNewProfile ) != 0 ) {
			psDialog = gtk_message_dialog_new( GTK_WINDOW( psAssistant ), GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION, GTK_BUTTONS_OK, _( "The port for the capi fax could not be activated" ) );
			gtk_message_dialog_format_secondary_text( GTK_MESSAGE_DIALOG( psDialog ), _( "Please dial #96*3* with your phone to enable it" ) );
			gtk_window_set_title( GTK_WINDOW( psDialog ), _( "Unlock port" ) );
			nRet = gtk_dialog_run( GTK_DIALOG( psDialog ) );
			gtk_widget_destroy( GTK_WIDGET( psDialog ) );
		}
	}
	checklistCompleteStage( psCheck );

	GTK_FLUSH;

	gtk_widget_set_sensitive( GTK_WIDGET( psButton ), TRUE );
	gtk_assistant_set_page_complete( psAssistant, psPage, TRUE );
}

/**
 * \brief Assistant prepare - called before a assistant page is created/displayed
 * \param psAssistant assistant widget
 * \param psPage current page
 * \param pData user data
 */
static void assistantPrepare( GtkAssistant *psAssistant, GtkWidget *psPage, gpointer pData ) {
	gint nNum = gtk_assistant_get_current_page( psAssistant );
	
	switch ( nNum ) {
		case 1: {
			GtkWidget *psProfileNameEntry = g_object_get_data( G_OBJECT( psAssistant ), "profile_entry" );

			gtk_widget_grab_focus( psProfileNameEntry );
			break;
		}
		case 2: {
			GtkWidget *psTreeView = g_object_get_data( G_OBJECT( psAssistant ), "detected_treeview" );
			GtkWidget *psRouterEntry = g_object_get_data( G_OBJECT( psAssistant ), "router_entry" );
			gint nRet = 0;

			Debug( KERN_DEBUG, "findDevices()\n" );
			findDevices( psTreeView );
			Debug( KERN_DEBUG, "findDevices() done\n" );

			if ( strlen(gtk_entry_get_text( GTK_ENTRY( psRouterEntry ) ) ) == 0 ) {
				nRet = checkHostName( "speedport.ip" );
				if ( nRet != 0 ) {
					/* error, check "fritz.box" */
					nRet = checkHostName( "fritz.box" );

					if ( nRet != 0 ) {
						/* error, check "alice.box" */
						nRet = checkHostName( "alice.box" );

						if ( nRet == 0 ) {
							gtk_entry_set_text( GTK_ENTRY( psRouterEntry ), "alice.box" );
						}
					} else {
						gtk_entry_set_text( GTK_ENTRY( psRouterEntry ), "fritz.box" );
					}
				} else {
					gtk_entry_set_text( GTK_ENTRY( psRouterEntry ), "speedport.ip" );
				}
			}
			break;
		}
		case 3: {
			GtkWidget *psCheck = g_object_get_data( G_OBJECT( psAssistant ), "check" );
			if ( psNewProfile == NULL ) {
				psNewProfile = createProfile( assistantGetProfileName( psAssistant ), NULL );
			}
			routerSetHost( psNewProfile, assistantGetHost( psAssistant ) );
			routerSetPassword( psNewProfile, assistantGetPassword( psAssistant ) );

			checklistReset( psCheck );
			break;
		}
		case 4: {
			GtkWidget *psLabel = g_object_get_data( G_OBJECT( psAssistant ), "psSummaryLabel" );
			gchar *pnTmp;

			pnTmp = g_strdup_printf(
				_( "New box integrated:\n%s\nCountry-Code: %s\nArea-Code: %s\n \nNow please go to the preferences and customize ffgtk" ),
					routerGetName( psNewProfile ), routerGetCountryCode( psNewProfile ), routerGetAreaCode( psNewProfile ) );
			gtk_label_set_text( GTK_LABEL( psLabel ), pnTmp);
			g_free( pnTmp);
			break;
		}
	}
}

/**
 * \brief Called when apply button in assistant is pressed
 * \param psAssistant assistant widget
 * \param pUserData user data
 */
static void assistantApply( GtkAssistant *psAssistant, gpointer pUserData ) {
	char *pnHost = assistantGetHost( psAssistant );

	setActiveProfile( psNewProfile );
	SavePreferences( psNewProfile );
	saveProfiles();

	g_free( pnHost );
}

/**
 * \brief Called when cancel button is pressed in assistant
 * \param psAssistant assistant widget
 * \param pUserData user data
 */
static void assistantCancel( GtkAssistant *psAssistant, gpointer pUserData ) {
	if ( psNewProfile != NULL ) {
		removeProfile( psNewProfile -> pnName );
		psNewProfile = NULL;
	}
	gtk_widget_destroy( GTK_WIDGET( psAssistant ) );
}

/**
 * \brief Initialize and start assistant
 * \return error code, 0=success else error
 */
int assistant( void ) {
	GtkWidget *psAssistant;
	GtkWidget *psIntroLabel;
	GtkBuilder *psBuilder;
	GError *psError = NULL;
	gchar *pnUiFile;

	psBuilder = gtk_builder_new();
	pnUiFile = getUiFile( "assistant.ui" );
	if ( gtk_builder_add_from_file( psBuilder, pnUiFile, &psError ) == 0 ) {
		Debug( KERN_WARNING, "Error: %s\n", psError -> message );
	    g_error_free( psError );
		g_free( pnUiFile );
	    return -2;
	}
	g_free( pnUiFile );

	psAssistant = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psAssistant" ) );

	/* Page 1 - Introduction */
	psIntroLabel = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psIntroLabel" ) );
	gtk_assistant_set_page_complete( GTK_ASSISTANT( psAssistant ), psIntroLabel, TRUE );

	/* Page 2 - Enter profile name */
	GtkWidget *psProfileNameEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psProfileNameEntry" ) );
	g_signal_connect( G_OBJECT( psProfileNameEntry ), "changed", G_CALLBACK( nameEntryChanged ), psAssistant );
	g_object_set_data( G_OBJECT( psAssistant ), "profile_entry", psProfileNameEntry );
	GtkWidget *psNameWarningLabel = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psNameWarningLabel" ) );
	g_object_set_data( G_OBJECT( psAssistant ), "warning_label", psNameWarningLabel );

	/* Page 3 - Select router box */
	GtkWidget *psRouterEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psRouterEntry" ) );
	gtk_entry_set_text( GTK_ENTRY( psRouterEntry ), "" );
	GtkWidget *psPasswordEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPasswordEntry" ) );
	GtkWidget *psRadioDetected = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psRadioDetected" ) );
	GtkWidget *psDetectedTreeView = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psDetectedTreeView" ) );
	GtkWidget *psRadioManual = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psRadioManual" ) );
	GtkTreeViewColumn *psCol = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( psCol, "Devices" );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psDetectedTreeView ), psCol );

	GtkCellRenderer *psRenderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start( psCol, psRenderer, TRUE );
	gtk_tree_view_column_add_attribute( psCol, psRenderer, "text", 0 );	

	GtkListStore *psStore = gtk_list_store_new( 1, G_TYPE_STRING );
	GtkTreeModel *psListStore = GTK_TREE_MODEL( psStore );
	gtk_tree_view_set_model( GTK_TREE_VIEW( psDetectedTreeView ), psListStore );

	/* Add change on selection event to tree view */
	GtkTreeSelection *psSel = gtk_tree_view_get_selection( GTK_TREE_VIEW( psDetectedTreeView ) );
	g_signal_connect( G_OBJECT( psSel ), "changed", G_CALLBACK( columnClicked ), psAssistant );

	g_signal_connect( G_OBJECT( psRadioManual ), "toggled", G_CALLBACK( radioCallback ), psRouterEntry );
	g_signal_connect( G_OBJECT( psRadioDetected ), "toggled", G_CALLBACK( radioCallback ), psDetectedTreeView );
	g_object_set_data( G_OBJECT( psDetectedTreeView ), "liststore", psListStore );
	g_object_set_data( G_OBJECT( psAssistant ), "radio_detected", psRadioDetected );
	g_object_set_data( G_OBJECT( psDetectedTreeView ), "assistant", psAssistant );
	g_object_set_data( G_OBJECT( psAssistant ), "detected_treeview", psDetectedTreeView );
	g_object_set_data( G_OBJECT( psRouterEntry ), "assistant", psAssistant );
	g_object_set_data( G_OBJECT( psAssistant ), "router_entry", psRouterEntry );
	g_object_set_data( G_OBJECT( psAssistant ), "password_entry", psPasswordEntry );

	/* Page 4 - Acquire information */
	GtkWidget *psAcquireAlignment = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psAcquireAlignment" ) );
	GtkWidget *psCheck = checklistNew();
	checklistAddStage( psCheck, _( "Check hostname" ) );
	checklistAddStage( psCheck, _( "Check password" ) );
	checklistAddStage( psCheck, _( "Get phone settings" ) );
	checklistAddStage( psCheck, _( "Detect firmware" ) );
	checklistAddStage( psCheck, _( "Get active ports" ) );
	checklistAddStage( psCheck, _( "Check monitor port" ) );
	checklistAddStage( psCheck, _( "Check capi port" ) );
	checklistAddProgressbar( psCheck );
	g_object_set_data( G_OBJECT( psAssistant ), "check", psCheck );
	gtk_container_add( GTK_CONTAINER( psAcquireAlignment ), GTK_WIDGET( psCheck ) );
	GtkWidget *psStartButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psStartButton" ) );
	g_signal_connect( G_OBJECT( psStartButton ), "clicked", G_CALLBACK( startScanClicked ), psAssistant );

	/* Page 5 - Summary */
	GtkWidget *psSummaryLabel = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psSummaryLabel" ) );
	g_object_set_data( G_OBJECT( psAssistant ), "psSummaryLabel", psSummaryLabel );
	gtk_assistant_set_page_complete( GTK_ASSISTANT( psAssistant ), psSummaryLabel, TRUE );

	gtk_builder_connect_signals( psBuilder, NULL );

	g_object_unref( G_OBJECT( psBuilder ) );

	g_signal_connect( G_OBJECT( psAssistant ), "apply", G_CALLBACK( assistantApply ), NULL );
	g_signal_connect( G_OBJECT( psAssistant ), "cancel", G_CALLBACK( assistantCancel ), NULL );
	g_signal_connect( G_OBJECT( psAssistant ), "close", G_CALLBACK( gtk_widget_destroy ), psAssistant );
	g_signal_connect( G_OBJECT( psAssistant ), "prepare", G_CALLBACK( assistantPrepare ), NULL );

	psNewProfile = NULL;
	gtk_widget_show_all( psAssistant );

	return 0;
}
