#!/bin/bash

# gen-control.sh -- generates a Debian control file based in Sid
#
# This file is part of the forensics-all.
#
# Copyright 2015-2017 Giovani Augusto Ferreira <giovani@debian.org>
# Copyright 2015-2017 Joao Eriberto Mota Filho <eriberto@debian.org>
#
# You can use this program under the BSD-3-Clause conditions.

#################
### Variables ###
#################

# Debian Policy Version (Standards-Version)
STD_VER=4.1.2

# Here is the list of packages in main/Sid.
LIST='/var/lib/apt/lists/*debian.org_debian_dists_sid_main_binary-*_Packages'

# These packages will be excluded.
# Use spaces between names. Do not wrap the line.
# Ex: EXCLUDE='PACKAGE1 PACKAGE2 PACKAGE3'
#
# RATIONALE:
# forensics-all is this package.
# md5deep still being provided by hashdeep as dummy package.
EXCLUDE='forensics-all md5deep'

# The following packages will be included as 'logical or' in Depends field. Use
# one line per occurrence. Do not use spaces inside variables.
LOGICAL_OR[1]='crack|crack-md5' # See #852846
LOGICAL_OR[2]=''
LOGICAL_OR[3]=''
LOGICAL_OR[4]=''
LOGICAL_OR[5]=''

# These type of files are dependencies for other packages and will be excluded.
# Do not use spaces between names. You can use regex here.
NOTDEPEND='-common|-dbg|-dkms|-doc|^lib|^python'

# The following package will be put in Suggests field (autoexcluded from
# Depends field).
#
# Use spaces between names. Do not wrap the line.
# Ex: SUGGESTS='PACKAGE1 PACKAGE2 PACKAGE3 PACKAGE4|PACKAGE5'
#
# RATIONALE:
# grr-* - see #867860. grr-client-templates-installer not seems a good idea for
# a desktop forensics environment.
SUGGESTS='forensics-extra forensics-extra-gui forensics-full grr-server grr-client-templates-installer'

# The following package will be put in Recommend field (autoexcluded from
# Depends field).
#
# Use spaces between names. Do not wrap the line.
# Ex: RECOMMENDS='PACKAGE1 PACKAGE2 PACKAGE3 PACKAGE4|PACKAGE5'
RECOMMENDS='outguess' # See #882538



####################
### Main program ###
####################

### Initial check

[ -e /usr/bin/wrap-and-sort ] || { echo "ERROR: wrap-and-sort command not found"; exit 1; }

### Initial message

echo " "
echo "gen-control.sh -- generates a Debian control file based in Sid"
echo " "
echo "Remember to run 'apt-get update' before to use this script."
echo " "
echo "You can edit the gen-control.sh file to exclude packages."
echo " "
echo "Press ENTER to continue or Ctrl-C to abort."
echo " "
read NOTHING

### Formatting exclusions

# EXC1 = first exclusions

[ "$EXCLUDE" ] && EXC1=$(echo $EXCLUDE | tr -s " " | sed 's/ /\|/g')

# EXC2 = logical or

for i in 1 2 3 4 5
do
    [ "${LOGICAL_OR[$i]}" ] && EXC2="$(echo $EXC2 "${LOGICAL_OR[$i]}" | tr -s " " | sed 's/ /\|/g')"
done

[ "$EXC2" ] || EXC2="none"

# EXC3 = Suggests

[ "$SUGGESTS" ] && EXC3=$(echo $SUGGESTS | tr -s " " | sed 's/ /\|/g')

[ "$EXC3" ] || EXC3="none"

# EXC4 = Recommends

[ "$RECOMMENDS" ] && EXC4=$(echo $RECOMMENDS | tr -s " " | sed 's/ /\|/g')

[ "$EXC4" ] || EXC4="none"

### Initial list of packages

# Depends field
[ -d build ] || mkdir build

cat $LIST | grep "Maintainer: Debian Forensics" -B6 | grep Package: | \
cut -d" " -f2 | egrep -v "($NOTDEPEND)" | \
egrep -v "^($EXC1|$EXC2|$EXC3|$EXC4)" | sort | sed 's/$/,/g' > build/PACKAGES.tmp

for i in 1 2 3 4 5
do
    [ "${LOGICAL_OR[$i]}" ] && echo "${LOGICAL_OR[$i]}," >> build/PACKAGES.tmp
done

cat build/PACKAGES.tmp | sort > build/PACKAGES.txt

# Suggests and Recommends field
echo "$SUGGESTS" | tr -s " " | sed 's/ /,\n/g' > build/PACKAGES2.txt
echo "$RECOMMENDS" | tr -s " " | sed 's/ /,\n/g' > build/PACKAGES3.txt

# Showing...
echo " "
echo "The packages to be put in DEPENDS field (debian/control) are listed below."
echo " "
cat build/PACKAGES.txt | xargs | sed 's/,$//'
echo " "
echo "TOTAL: $(cat build/PACKAGES.txt | wc -l) packages."
echo " "
if [ "$SUGGESTS" ]
then
    echo "For SUGGESTS field, there are the following packages:"
    echo " "
    cat build/PACKAGES2.txt | xargs | sed 's/,$//'
    echo " "
fi
if [ "$RECOMMENDS" ]
then
    echo "For RECOMMENDS field, there are the following packages:"
    echo " "
    cat build/PACKAGES3.txt | xargs | sed 's/,$//'
    echo " "
fi
echo "Press ENTER to continue or Ctrl-C to abort."
echo " "
read NOTHING

### Build the final control

cat templates/control.part1 | sed "s/X.Y.Z/$STD_VER/" > control.NEW

if [ "$SUGGESTS" ]
then
    echo "Suggests: $(cat build/PACKAGES2.txt)" | sed 's/^/            /' | \
    sed 's/ \+Suggests:/Suggests:/' >> control.NEW
fi

if [ "$RECOMMENDS" ]
then
    echo "Recommends: $(cat build/PACKAGES3.txt)" | sed 's/^/            /' | \
    sed 's/ \+Recommends:/Recommends:/' >> control.NEW
fi

echo "Depends: $(cat build/PACKAGES.txt)" | sed 's/^/         /' | \
sed 's/ \+Depends:/Depends:/' >> control.NEW

cat templates/control.part2 >> control.NEW

cat build/PACKAGES.txt | xargs | sed 's/,$/./' | fold -sw 69 | sed 's/^/   /' | \
sed 's/|/ or /g' | sed 's/ $//' >> control.NEW

cat templates/control.part3 >> control.NEW

wrap-and-sort -f ./control.NEW

### Final message

echo " "
echo "The control.NEW file is done. Please, rename to"
echo "control and move to debian/ directory."
echo " "

### Remove the build directory

rm -rf build
