/**
 * Author......: See docs/credits.txt
 * License.....: MIT
 */

#include "common.h"
#include "types.h"
#include "modules.h"
#include "bitops.h"
#include "convert.h"
#include "shared.h"
#include "memory.h"

static const u32   ATTACK_EXEC    = ATTACK_EXEC_OUTSIDE_KERNEL;
static const u32   DGST_POS0      = 0;
static const u32   DGST_POS1      = 1;
static const u32   DGST_POS2      = 2;
static const u32   DGST_POS3      = 3;
static const u32   DGST_SIZE      = DGST_SIZE_4_16;
static const u32   HASH_CATEGORY  = HASH_CATEGORY_FDE;
static const char *HASH_NAME      = "LUKS v1 RIPEMD-160 + AES";
static const u64   KERN_TYPE      = 14641; // old kernel used here
static const u32   OPTI_TYPE      = OPTI_TYPE_ZERO_BYTE
                                  | OPTI_TYPE_SLOW_HASH_SIMD_LOOP;
static const u64   OPTS_TYPE      = OPTS_TYPE_STOCK_MODULE
                                  | OPTS_TYPE_PT_GENERATE_LE;
static const u32   SALT_TYPE      = SALT_TYPE_EMBEDDED;
static const char *ST_PASS        = "hashcat";
static const char *ST_HASH        = "$luks$1$ripemd160$aes$cbc-essiv:sha256$256$106666$0e4b2d61153a322af5defd708c98b8251b34b09d0f39d3dcee80f8114d7922b3$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$561ab9d16929150d3877c88a32899e358c1bc58a28a4f446f907a04d4f13198f3c98d1c19db2ce8eac0a2158c503dfcc02040b0554fc8c3628f36b5e6d2bf047eb66bc0cd96f66ae39767012e835ff1c22297653bfcbde81245d40b7e8f17e84309f70b1fec7a5041bf2d348335fca38435d00489f1337809e244889d30566070dd3cd68e5f6b9f7a439dcf5e96fb668cbd4b7d5b2f5fe9e6d4bb9f08cc419badbab50e3451e6aae1d242fa2e5a8340d63f73f480f15ed391920c760b8670b195ea6dca0436a5dd837fc4a4e55e64036f6f414c09d143fa2736d1d29fa1fa337d44dbf4f2f5985fcd22ecb877dbf65fdc48f554eaf0d2ebbb0333a730dc3a795067b11e636d318626c175ead464eaf2936b9997811719e74e3146c600d3c46614bff5d18b0060e469c12bfd84adfd00924d6b949fed312d216c7e75dcaa9b35fcf84bce26d4defec16be5ffcd5a20dbb123d32afbc35601078ce2fde24b3142e8ab8975888693d1c7510b9edc40cae3197d7f1add88c11de1bf4135b58fdb613d7f16c44d775079bf5e2a42bd17f953c36269cd5f3a6cb9b311685ab63e6f2eb7f4ca4365faab8dc506d72b40961b705c346e7c344e679a054318d1f9b0b2b939f852c9f54a6e3ca6cf557004f780e063bf060cb9c91f48be22632d03e4d21dd82a90fcebedface46dd256ed78dd4da0018a7e1838c66ac44b598bf9c3e0b582";

u32         module_attack_exec    (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ATTACK_EXEC;     }
u32         module_dgst_pos0      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS0;       }
u32         module_dgst_pos1      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS1;       }
u32         module_dgst_pos2      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS2;       }
u32         module_dgst_pos3      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS3;       }
u32         module_dgst_size      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_SIZE;       }
u32         module_hash_category  (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_CATEGORY;   }
const char *module_hash_name      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_NAME;       }
u64         module_kern_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return KERN_TYPE;       }
u32         module_opti_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTI_TYPE;       }
u64         module_opts_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTS_TYPE;       }
u32         module_salt_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return SALT_TYPE;       }
const char *module_st_hash        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_HASH;         }
const char *module_st_pass        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_PASS;         }

#define LUKS_STRIPES        (                                   4000)
#define LUKS_SALT_LEN       (                                     32)
#define LUKS_SALT_HEX_LEN   (                      LUKS_SALT_LEN * 2)
#define LUKS_CT_LEN         (                                    512)
#define LUKS_CT_HEX_LEN     (                        LUKS_CT_LEN * 2)
#define LUKS_AF_MIN_LEN     (HC_LUKS_KEY_SIZE_128 / 8 * LUKS_STRIPES)
#define LUKS_AF_MIN_HEX_LEN (                    LUKS_AF_MIN_LEN * 2)
#define LUKS_AF_MAX_LEN     (HC_LUKS_KEY_SIZE_512 / 8 * LUKS_STRIPES)
#define LUKS_AF_MAX_HEX_LEN (                    LUKS_AF_MAX_LEN * 2)

typedef enum hc_luks_hash_type
{
  HC_LUKS_HASH_TYPE_SHA1      = 1,
  HC_LUKS_HASH_TYPE_SHA256    = 2,
  HC_LUKS_HASH_TYPE_SHA512    = 3,
  HC_LUKS_HASH_TYPE_RIPEMD160 = 4,
  HC_LUKS_HASH_TYPE_WHIRLPOOL = 5,

} hc_luks_hash_type_t;

typedef enum hc_luks_key_size
{
  HC_LUKS_KEY_SIZE_128 = 128,
  HC_LUKS_KEY_SIZE_256 = 256,
  HC_LUKS_KEY_SIZE_512 = 512,

} hc_luks_key_size_t;

typedef enum hc_luks_cipher_type
{
  HC_LUKS_CIPHER_TYPE_AES     = 1,
  HC_LUKS_CIPHER_TYPE_SERPENT = 2,
  HC_LUKS_CIPHER_TYPE_TWOFISH = 3,

} hc_luks_cipher_type_t;

typedef enum hc_luks_cipher_mode
{
  HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256 = 1,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN        = 2,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN64      = 3,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN        = 4,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN64      = 5,

} hc_luks_cipher_mode_t;

typedef struct luks
{
  int hash_type;   // hc_luks_hash_type_t
  int key_size;    // hc_luks_key_size_t
  int cipher_type; // hc_luks_cipher_type_t
  int cipher_mode; // hc_luks_cipher_mode_t

  u32 ct_buf[LUKS_CT_LEN / 4];

  u32 af_buf[LUKS_AF_MAX_LEN / 4];
  u32 af_len;

} luks_t;

typedef struct luks_tmp
{
  u32 ipad32[8];
  u64 ipad64[8];

  u32 opad32[8];
  u64 opad64[8];

  u32 dgst32[32];
  u64 dgst64[16];

  u32 out32[32];
  u64 out64[16];

} luks_tmp_t;

static const char *SIGNATURE_LUKS = "$luks$1$ripemd160$aes$";

u64 module_esalt_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 esalt_size = (const u64) sizeof (luks_t);

  return esalt_size;
}

u64 module_tmp_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 tmp_size = (const u64) sizeof (luks_tmp_t);

  return tmp_size;
}

int module_hash_decode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED void *digest_buf, MAYBE_UNUSED salt_t *salt, MAYBE_UNUSED void *esalt_buf, MAYBE_UNUSED void *hook_salt_buf, MAYBE_UNUSED hashinfo_t *hash_info, const char *line_buf, MAYBE_UNUSED const int line_len)
{
  luks_t *luks = (luks_t *) esalt_buf;

  hc_token_t token;

  memset (&token, 0, sizeof (hc_token_t));

  token.token_cnt  = 7;

  token.signatures_cnt    = 1;
  token.signatures_buf[0] = SIGNATURE_LUKS;

  // signature with hash and cipher type
  token.len[0]     = 22;
  token.attr[0]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_SIGNATURE;

  // cipher mode
  token.sep[1]     = '$';
  token.len_min[1] = 9;
  token.len_max[1] = 16;
  token.attr[1]    = TOKEN_ATTR_VERIFY_LENGTH;

  // key size
  token.sep[2]     = '$';
  token.len[2]     = 3;
  token.attr[2]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // iter
  token.sep[3]     = '$';
  token.len_min[3] = 1;
  token.len_max[3] = 10;
  token.attr[3]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // salt
  token.sep[4]     = '$';
  token.len[4]     = LUKS_SALT_HEX_LEN;
  token.attr[4]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // af
  token.sep[5]     = '$';
  token.len_min[5] = LUKS_AF_MIN_HEX_LEN;
  token.len_max[5] = LUKS_AF_MAX_HEX_LEN;
  token.attr[5]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // ct
  token.sep[6]     = '$';
  token.len[6]     = LUKS_CT_HEX_LEN;
  token.attr[6]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  const int rc_tokenizer = input_tokenizer ((const u8 *) line_buf, line_len, &token);

  if (rc_tokenizer != PARSER_OK) return (rc_tokenizer);

  // hash type

  luks->hash_type = HC_LUKS_HASH_TYPE_RIPEMD160;

  // cipher type

  luks->cipher_type = HC_LUKS_CIPHER_TYPE_AES;

  // cipher mode

  const u8 *cipher_mode_pos = token.buf[1];
  const u32 cipher_mode_len = token.len[1];

  if ((strncmp ((const char *) cipher_mode_pos, "cbc-essiv:sha256", 16) == 0) && (cipher_mode_len == 16))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "cbc-plain",    9) == 0) && (cipher_mode_len ==  9))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "cbc-plain64", 11) == 0) && (cipher_mode_len == 11))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN64;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "xts-plain",    9) == 0) && (cipher_mode_len ==  9))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "xts-plain64", 11) == 0) && (cipher_mode_len == 11))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN64;
  }
  else
  {
    return (PARSER_LUKS_CIPHER_MODE);
  }

  // key size

  const u8 *key_size_pos = token.buf[2];

  const u32 key_size = hc_strtoul ((const char *) key_size_pos, NULL, 10);

  if (key_size == 128)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_128;
  }
  else if (key_size == 256)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_256;
  }
  else if (key_size == 512)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_512;
  }
  else
  {
    return (PARSER_LUKS_KEY_SIZE);
  }

  // salt

  const u8 *salt_pos = token.buf[4];

  salt->salt_len = hex_decode (salt_pos, LUKS_SALT_HEX_LEN, (u8 *) salt->salt_buf);

  // iter

  const u8 *iter_pos = token.buf[3];

  const u32 iter = hc_strtoul ((const char *) iter_pos, NULL, 10);

  salt->salt_iter = iter - 1;

  // af

  u32 af_len = token.len[5] / 2;

  if (af_len != (key_size / 8) * LUKS_STRIPES) return (PARSER_LUKS_FILE_SIZE);

  const u8 *af_pos = token.buf[5];

  luks->af_len = hex_decode (af_pos, token.len[5], (u8 *) luks->af_buf);

  // ct

  u32 ct_len = token.len[6] / 2;

  if (ct_len != LUKS_CT_LEN) return (PARSER_LUKS_FILE_SIZE);

  const u8 *ct_pos = token.buf[6];

  hex_decode (ct_pos, LUKS_CT_HEX_LEN, (u8 *) luks->ct_buf);

  return (PARSER_OK);
}

int module_hash_encode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const void *digest_buf, MAYBE_UNUSED const salt_t *salt, MAYBE_UNUSED const void *esalt_buf, MAYBE_UNUSED const void *hook_salt_buf, MAYBE_UNUSED const hashinfo_t *hash_info, char *line_buf, MAYBE_UNUSED const int line_size)
{
  const luks_t *luks = (const luks_t *) esalt_buf;

  // cipher mode

  const char *cipher_mode = "";

  switch (luks->cipher_mode)
  {
    case HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256: cipher_mode = "cbc-essiv:sha256"; break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN:        cipher_mode = "cbc-plain";        break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN64:      cipher_mode = "cbc-plain64";      break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN:        cipher_mode = "xts-plain";        break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN64:      cipher_mode = "xts-plain64";      break;
    default:                                                                     return 0;
  }

  // key size

  u32 key_size = 0;

  switch (luks->key_size)
  {
    case HC_LUKS_KEY_SIZE_128: key_size = 128; break;
    case HC_LUKS_KEY_SIZE_256: key_size = 256; break;
    case HC_LUKS_KEY_SIZE_512: key_size = 512; break;
    default:                                   return 0;
  }

  // iterations

  u32 iterations = salt->salt_iter + 1;

  // salt

  char salt_buf[LUKS_SALT_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) salt->salt_buf, LUKS_SALT_LEN, (u8 *) salt_buf);

  // af

  char af_buf[LUKS_AF_MAX_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) luks->af_buf, luks->af_len, (u8 *) af_buf);

  // ct

  char ct_buf[LUKS_CT_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) luks->ct_buf, LUKS_CT_LEN, (u8 *) ct_buf);

  // output

  int line_len = snprintf (line_buf, line_size, "%s%s$%u$%u$%s$%s$%s",
    SIGNATURE_LUKS,
    cipher_mode,
    key_size,
    iterations,
    salt_buf,
    af_buf,
    ct_buf);

  return line_len;
}

void module_init (module_ctx_t *module_ctx)
{
  module_ctx->module_context_size             = MODULE_CONTEXT_SIZE_CURRENT;
  module_ctx->module_interface_version        = MODULE_INTERFACE_VERSION_CURRENT;

  module_ctx->module_attack_exec              = module_attack_exec;
  module_ctx->module_benchmark_esalt          = MODULE_DEFAULT;
  module_ctx->module_benchmark_hook_salt      = MODULE_DEFAULT;
  module_ctx->module_benchmark_mask           = MODULE_DEFAULT;
  module_ctx->module_benchmark_charset        = MODULE_DEFAULT;
  module_ctx->module_benchmark_salt           = MODULE_DEFAULT;
  module_ctx->module_bridge_name              = MODULE_DEFAULT;
  module_ctx->module_bridge_type              = MODULE_DEFAULT;
  module_ctx->module_build_plain_postprocess  = MODULE_DEFAULT;
  module_ctx->module_deep_comp_kernel         = MODULE_DEFAULT;
  module_ctx->module_deprecated_notice        = MODULE_DEFAULT;
  module_ctx->module_dgst_pos0                = module_dgst_pos0;
  module_ctx->module_dgst_pos1                = module_dgst_pos1;
  module_ctx->module_dgst_pos2                = module_dgst_pos2;
  module_ctx->module_dgst_pos3                = module_dgst_pos3;
  module_ctx->module_dgst_size                = module_dgst_size;
  module_ctx->module_dictstat_disable         = MODULE_DEFAULT;
  module_ctx->module_esalt_size               = module_esalt_size;
  module_ctx->module_extra_buffer_size        = MODULE_DEFAULT;
  module_ctx->module_extra_tmp_size           = MODULE_DEFAULT;
  module_ctx->module_extra_tuningdb_block     = MODULE_DEFAULT;
  module_ctx->module_forced_outfile_format    = MODULE_DEFAULT;
  module_ctx->module_hash_binary_count        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_parse        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_save         = MODULE_DEFAULT;
  module_ctx->module_hash_decode_postprocess  = MODULE_DEFAULT;
  module_ctx->module_hash_decode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_decode_zero_hash    = MODULE_DEFAULT;
  module_ctx->module_hash_decode              = module_hash_decode;
  module_ctx->module_hash_encode_status       = MODULE_DEFAULT;
  module_ctx->module_hash_encode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_encode              = module_hash_encode;
  module_ctx->module_hash_init_selftest       = MODULE_DEFAULT;
  module_ctx->module_hash_mode                = MODULE_DEFAULT;
  module_ctx->module_hash_category            = module_hash_category;
  module_ctx->module_hash_name                = module_hash_name;
  module_ctx->module_hashes_count_min         = MODULE_DEFAULT;
  module_ctx->module_hashes_count_max         = MODULE_DEFAULT;
  module_ctx->module_hlfmt_disable            = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_size    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_init    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_term    = MODULE_DEFAULT;
  module_ctx->module_hook12                   = MODULE_DEFAULT;
  module_ctx->module_hook23                   = MODULE_DEFAULT;
  module_ctx->module_hook_salt_size           = MODULE_DEFAULT;
  module_ctx->module_hook_size                = MODULE_DEFAULT;
  module_ctx->module_jit_build_options        = MODULE_DEFAULT;
  module_ctx->module_jit_cache_disable        = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_max       = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_min       = MODULE_DEFAULT;
  module_ctx->module_kern_type                = module_kern_type;
  module_ctx->module_kern_type_dynamic        = MODULE_DEFAULT;
  module_ctx->module_opti_type                = module_opti_type;
  module_ctx->module_opts_type                = module_opts_type;
  module_ctx->module_outfile_check_disable    = MODULE_DEFAULT;
  module_ctx->module_outfile_check_nocomp     = MODULE_DEFAULT;
  module_ctx->module_potfile_custom_check     = MODULE_DEFAULT;
  module_ctx->module_potfile_disable          = MODULE_DEFAULT;
  module_ctx->module_potfile_keep_all_hashes  = MODULE_DEFAULT;
  module_ctx->module_pwdump_column            = MODULE_DEFAULT;
  module_ctx->module_pw_max                   = MODULE_DEFAULT;
  module_ctx->module_pw_min                   = MODULE_DEFAULT;
  module_ctx->module_salt_max                 = MODULE_DEFAULT;
  module_ctx->module_salt_min                 = MODULE_DEFAULT;
  module_ctx->module_salt_type                = module_salt_type;
  module_ctx->module_separator                = MODULE_DEFAULT;
  module_ctx->module_st_hash                  = module_st_hash;
  module_ctx->module_st_pass                  = module_st_pass;
  module_ctx->module_tmp_size                 = module_tmp_size;
  module_ctx->module_unstable_warning         = MODULE_DEFAULT;
  module_ctx->module_warmup_disable           = MODULE_DEFAULT;
}
