/***************************************************************************
                          contactaddeduserwidget.cpp - QWidget class with
                          QTabWidget
                             -------------------
    begin                : Friday Sept 24 2010
    copyright            : (C) 2010 by Timo Tambet
    email                : ttambet@gmail.com
    copyright            : (C) 2002 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "contactaddeduserwidget.h"
#include "../contact/contact.h"
#include "../model/contactlist.h"
#include "../contact/group.h"
#include "../currentaccount.h"
#include "../kmessdebug.h"
#include "../utils/kmessconfig.h"
#include <kstandardguiitem.h>



/**
 * Constructor
 *
 * This contains all the widgets for selecting what action to perform when a user adds you
 * to their contact list. Multiple instances of this widget are added to the QTabWidget in
 * ContactAddedUserDialog.
 *
 * @param contactHandle         Reference to the handle of the contact that has added you
 * @param contactFriendlyName   The friendly name of the contact
 */
ContactAddedUserWidget::ContactAddedUserWidget( const QString& contactHandle, const QString& contactFriendlyName )
  : QWidget()
  , Ui::ContactAddedUserWidget()
  , contactHandle_( contactHandle )
{
  setupUi( this );

  buttonbox->addButton( KStandardGuiItem::ok(),      QDialogButtonBox::AcceptRole, this, SLOT( accept()));
  buttonbox->addButton( KStandardGuiItem::discard(), QDialogButtonBox::RejectRole, this, SLOT( reject()));

  if( contactHandle == contactFriendlyName )
  {
    messageLabel_->setText( i18n( "%1\nhas added you to his or her contact list."
                                  , contactHandle ) );
  }
  else
  {
    messageLabel_->setText( i18n( "%1 (%2)\nhas added you to his or her contact list."
                                  , contactFriendlyName
                                  , contactHandle ) );
  }

   // if they're already a friend, ie in the contact list, don't allow the "add" option to
  // be selected. only options should be to allow or block.
  Contact *c = CurrentAccount::instance()->getContactList()->getContactByHandle( contactHandle );
  if ( c && c->isFriend() )
  {
    addContactOption_->setEnabled( false );
  }

  // Fill up the groups list from the contactlist data
  QList<Group*> groupList( CurrentAccount::instance()->getContactList()->getGroupList() );
  QListWidgetItem *item;

  foreach( const Group *group, groupList )
    {
      if( ! group->isSpecialGroup() )
      {
        item = new QListWidgetItem( group->getName(), groupList_ );

        // Set the groupId in the item's data, to retrieve it later
        item->setData( Qt::UserRole, group->getId() );
        item->setFlags( Qt::ItemIsUserCheckable | Qt::ItemIsEnabled );
        item->setCheckState( Qt::Unchecked );
      }
    }
}



/**
* @brief Destructor
*
*/
ContactAddedUserWidget::~ContactAddedUserWidget()
{
}



/**
* @brief Called when the user pressed the "ok" button
*
*/
void ContactAddedUserWidget::accept()
{
  QStringList groupIds;
  if( addContactOption_->isChecked() )
  {
    // Find all checked items in the list
    QListWidgetItem *item;
    for( int i = 0; i < groupList_->count(); i++ )
    {
      item = groupList_->item( i );
      if( item->checkState() == Qt::Checked )
      {
        // Retrieve the group id from the item's data
        groupIds.append( item->data( Qt::UserRole ).toString() );
      }
    }
    emit userChoice( contactHandle_, groupIds, ADD );
  }
  else if( allowContactOption_->isChecked() )
  {
    emit userChoice( contactHandle_, groupIds, ALLOW );
  }
  else if ( blockContactOption_->isChecked() )
  {
    emit userChoice( contactHandle_, groupIds, BLOCK );
  }

  // The widget is not necessary anymore, delete it
  deleteLater();
}



/**
* @brief Called when the user pressed the "cancel" button
*
*/
void ContactAddedUserWidget::reject()
{
  // Communicate the user choice
  emit userChoice( contactHandle_, QStringList(), IGNORE );

  // The widget is not necessary anymore, delete it
  deleteLater();
}



#include "contactaddeduserwidget.moc"
