#include "SimTKmolmodel.h"
#include "SimTKsimbody_aux.h"

#include <vector>
#include <string>
#include <sstream>

using namespace SimTK;

#define Deg2Rad (3.14159/180.0)

class CartesianAtom {
public:
	CartesianAtom(const Element& element, Vec3 location)
		: element(element), location(location)
	{}

	const Element& getElement() const {
		return element;
	}

	Real getMass() const {
		return getElement().getMass();
	}

	const Vec3& getLocation() const {
		return location;
	}

	CartesianAtom& setLocation(const Vec3& l) {
		location = l;
		return *this;
	}

private:
	Element element;
	Vec3 location;
};

class AtomClusterBody : public Body::Rigid {
public:
	typedef std::vector<CartesianAtom> AtomList;

	/// Move atoms so center of mass is at body origin
	void moveMassToOrigin() 
	{
		Vec3 centerOfMass = calcCenterOfMass();
		AtomList::iterator a;
		for (a = atoms.begin(); a != atoms.end(); ++a) 
			a->setLocation( a->getLocation() - centerOfMass );
	}

	Vec3 calcCenterOfMass() const 
	{
		Vec3 centerOfMass(0);
		Real mass(0);

		AtomList::const_iterator a;
		for (a = atoms.begin(); a != atoms.end(); ++a) {
			centerOfMass += a->getMass() * a->getLocation();
			mass += a->getMass();
		}
		
		if (mass > 0)
			centerOfMass /= mass;

		return centerOfMass;
	}

	Real calcMass() const 
	{
		Real mass(0);

		AtomList::const_iterator a;
		for (a = atoms.begin(); a != atoms.end(); ++a) {
			mass += a->getMass();
		}
		
		return mass;
	}

	Inertia calcInertia() const 
	{
		Inertia inertia(0);

		AtomList::const_iterator a;
		for (a = atoms.begin(); a != atoms.end(); ++a) {
			inertia += Inertia(a->getLocation(), a->getMass());
		}
		
		return inertia;
	}

	MassProperties calcMassProperties() const 
	{
		return MassProperties(calcMass(), calcCenterOfMass(), calcInertia());
	}

	void addAtom(const Element& element, Vec3 location) {
		atoms.push_back( CartesianAtom(element, location) );
	}

	void addCylinderDecoration() 
	{
		// cylinder about the size of DNA
		addDecoration(
			Transform(), 
			DecorativeCylinder(1.0, 1.7)
		);

		// plus a bump, to see when it twists
		addDecoration(
			Transform(Vec3(1.0, 0, 0)),
			DecorativeSphere(0.5)
		);
	}

private:
	AtomList atoms;
};

class DNADuplexTurnBody : public AtomClusterBody {
public:
	DNADuplexTurnBody() {
		// TODO put atoms on body

		// Microsoft compiler as of 2008 has retarded limit of 65k characters in a string
		// So split pdb specification into two parts

		// Created ideal B-DNA structure using NAB web server
		const char* strand1String = 
"ATOM      1  H5    A A   1      -0.423  -8.150  -2.094  1.00  0.00           \n"
"ATOM      2  O5*   A A   1       0.427  -7.826  -1.788  1.00  0.00           \n"
"ATOM      3  C5*   A A   1       1.443  -7.510  -2.756  1.00  0.00           \n"
"ATOM      4 1H5*   A A   1       1.100  -6.745  -3.453  1.00  0.00           \n"
"ATOM      5 2H5*   A A   1       1.674  -8.423  -3.305  1.00  0.00           \n"
"ATOM      6  C4*   A A   1       2.695  -7.020  -2.053  1.00  0.00           \n"
"ATOM      7  H4*   A A   1       3.574  -7.056  -2.696  1.00  0.00           \n"
"ATOM      8  O4*   A A   1       2.477  -5.630  -1.823  1.00  0.00           \n"
"ATOM      9  C1*   A A   1       2.308  -5.333  -0.452  1.00  0.00           \n"
"ATOM     10  H1*   A A   1       3.091  -4.700  -0.035  1.00  0.00           \n"
"ATOM     11  N9    A A   1       1.098  -4.468  -0.370  1.00  0.00           \n"
"ATOM     12  C8    A A   1      -0.229  -4.831  -0.385  1.00  0.00           \n"
"ATOM     13  H8    A A   1      -0.503  -5.875  -0.417  1.00  0.00           \n"
"ATOM     14  N7    A A   1      -1.050  -3.830  -0.296  1.00  0.00           \n"
"ATOM     15  C5    A A   1      -0.219  -2.721  -0.217  1.00  0.00           \n"
"ATOM     16  C6    A A   1      -0.482  -1.346  -0.105  1.00  0.00           \n"
"ATOM     17  N6    A A   1      -1.719  -0.832  -0.051  1.00  0.00           \n"
"ATOM     18 1H6    A A   1      -1.848   0.166   0.030  1.00  0.00           \n"
"ATOM     19 2H6    A A   1      -2.522  -1.443  -0.091  1.00  0.00           \n"
"ATOM     20  N1    A A   1       0.575  -0.516  -0.051  1.00  0.00           \n"
"ATOM     21  C2    A A   1       1.802  -1.033  -0.105  1.00  0.00           \n"
"ATOM     22  H2    A A   1       2.665  -0.385  -0.062  1.00  0.00           \n"
"ATOM     23  N3    A A   1       2.168  -2.292  -0.209  1.00  0.00           \n"
"ATOM     24  C4    A A   1       1.091  -3.102  -0.262  1.00  0.00           \n"
"ATOM     25  C3*   A A   1       2.950  -7.607  -0.665  1.00  0.00           \n"
"ATOM     26  H3*   A A   1       2.586  -8.629  -0.554  1.00  0.00           \n"
"ATOM     27  C2*   A A   1       2.148  -6.679   0.247  1.00  0.00           \n"
"ATOM     28 1H2*   A A   1       1.344  -7.241   0.722  1.00  0.00           \n"
"ATOM     29 2H2*   A A   1       2.804  -6.267   1.014  1.00  0.00           \n"
"ATOM     30  O3*   A A   1       4.324  -7.545  -0.307  1.00  0.00           \n"
"ATOM     31  P     T A   2       4.728  -7.626   1.239  1.00  0.00           \n"
"ATOM     32  O1P   T A   2       6.009  -8.356   1.380  1.00  0.00           \n"
"ATOM     33  O2P   T A   2       3.592  -8.164   2.021  1.00  0.00           \n"
"ATOM     34  O5*   T A   2       4.945  -6.081   1.592  1.00  0.00           \n"
"ATOM     35  C5*   T A   2       5.582  -5.227   0.624  1.00  0.00           \n"
"ATOM     36 1H5*   T A   2       4.855  -4.809  -0.072  1.00  0.00           \n"
"ATOM     37 2H5*   T A   2       6.306  -5.829   0.075  1.00  0.00           \n"
"ATOM     38  C4*   T A   2       6.306  -4.095   1.327  1.00  0.00           \n"
"ATOM     39  H4*   T A   2       7.038  -3.607   0.684  1.00  0.00           \n"
"ATOM     40  O4*   T A   2       5.313  -3.099   1.557  1.00  0.00           \n"
"ATOM     41  C1*   T A   2       5.002  -2.958   2.928  1.00  0.00           \n"
"ATOM     42  H1*   T A   2       5.263  -1.978   3.329  1.00  0.00           \n"
"ATOM     43  N1    T A   2       3.515  -2.969   3.010  1.00  0.00           \n"
"ATOM     44  C6    T A   2       2.818  -4.148   2.980  1.00  0.00           \n"
"ATOM     45  H6    T A   2       3.350  -5.087   2.941  1.00  0.00           \n"
"ATOM     46  C5    T A   2       1.477  -4.168   3.053  1.00  0.00           \n"
"ATOM     47  C7    T A   2       0.704  -5.454   3.022  1.00  0.00           \n"
"ATOM     48 1H7    T A   2       1.005  -6.080   3.862  1.00  0.00           \n"
"ATOM     49 2H7    T A   2      -0.362  -5.239   3.093  1.00  0.00           \n"
"ATOM     50 3H7    T A   2       0.907  -5.978   2.088  1.00  0.00           \n"
"ATOM     51  C4    T A   2       0.726  -2.941   3.166  1.00  0.00           \n"
"ATOM     52  O4    T A   2      -0.502  -2.859   3.239  1.00  0.00           \n"
"ATOM     53  N3    T A   2       1.506  -1.802   3.189  1.00  0.00           \n"
"ATOM     54  H3    T A   2       0.984  -0.849   3.275  1.00  0.00           \n"
"ATOM     55  C2    T A   2       2.884  -1.752   3.115  1.00  0.00           \n"
"ATOM     56  O2    T A   2       3.489  -0.694   3.143  1.00  0.00           \n"
"ATOM     57  C3*   T A   2       6.858  -4.420   2.715  1.00  0.00           \n"
"ATOM     58  H3*   T A   2       7.164  -5.460   2.826  1.00  0.00           \n"
"ATOM     59  C2*   T A   2       5.664  -4.141   3.627  1.00  0.00           \n"
"ATOM     60 1H2*   T A   2       5.344  -5.069   4.102  1.00  0.00           \n"
"ATOM     61 2H2*   T A   2       5.952  -3.422   4.394  1.00  0.00           \n"
"ATOM     62  O3*   T A   2       7.933  -3.562   3.073  1.00  0.00           \n"
"ATOM     63  P     A A   3       8.308  -3.390   4.619  1.00  0.00           \n"
"ATOM     64  O1P   A A   3       9.773  -3.228   4.760  1.00  0.00           \n"
"ATOM     65  O2P   A A   3       7.705  -4.493   5.401  1.00  0.00           \n"
"ATOM     66  O5*   A A   3       7.575  -2.013   4.972  1.00  0.00           \n"
"ATOM     67  C5*   A A   3       7.588  -0.948   4.004  1.00  0.00           \n"
"ATOM     68 1H5*   A A   3       6.754  -1.037   3.308  1.00  0.00           \n"
"ATOM     69 2H5*   A A   3       8.528  -1.010   3.455  1.00  0.00           \n"
"ATOM     70  C4*   A A   3       7.509   0.394   4.707  1.00  0.00           \n"
"ATOM     71  H4*   A A   3       7.815   1.219   4.064  1.00  0.00           \n"
"ATOM     72  O4*   A A   3       6.120   0.616   4.937  1.00  0.00           \n"
"ATOM     73  C1*   A A   3       5.785   0.547   6.308  1.00  0.00           \n"
"ATOM     74  H1*   A A   3       5.425   1.487   6.725  1.00  0.00           \n"
"ATOM     75  N9    A A   3       4.589  -0.336   6.390  1.00  0.00           \n"
"ATOM     76  C8    A A   3       4.523  -1.711   6.375  1.00  0.00           \n"
"ATOM     77  H8    A A   3       5.432  -2.294   6.343  1.00  0.00           \n"
"ATOM     78  N7    A A   3       3.318  -2.182   6.464  1.00  0.00           \n"
"ATOM     79  C5    A A   3       2.520  -1.050   6.543  1.00  0.00           \n"
"ATOM     80  C6    A A   3       1.131  -0.875   6.655  1.00  0.00           \n"
"ATOM     81  N6    A A   3       0.260  -1.892   6.709  1.00  0.00           \n"
"ATOM     82 1H6    A A   3      -0.729  -1.706   6.790  1.00  0.00           \n"
"ATOM     83 2H6    A A   3       0.593  -2.845   6.669  1.00  0.00           \n"
"ATOM     84  N1    A A   3       0.669   0.388   6.709  1.00  0.00           \n"
"ATOM     85  C2    A A   3       1.540   1.394   6.655  1.00  0.00           \n"
"ATOM     86  H2    A A   3       1.189   2.415   6.698  1.00  0.00           \n"
"ATOM     87  N3    A A   3       2.850   1.354   6.551  1.00  0.00           \n"
"ATOM     88  C4    A A   3       3.287   0.079   6.498  1.00  0.00           \n"
"ATOM     89  C3*   A A   3       8.146   0.455   6.095  1.00  0.00           \n"
"ATOM     90  H3*   A A   3       9.005  -0.207   6.205  1.00  0.00           \n"
"ATOM     91  C2*   A A   3       7.016  -0.021   7.007  1.00  0.00           \n"
"ATOM     92 1H2*   A A   3       7.302  -0.959   7.482  1.00  0.00           \n"
"ATOM     93 2H2*   A A   3       6.826   0.730   7.774  1.00  0.00           \n"
"ATOM     94  O3*   A A   3       8.512   1.781   6.453  1.00  0.00           \n"
"ATOM     95  P     C A   4       8.714   2.140   7.999  1.00  0.00           \n"
"ATOM     96  O1P   C A   4       9.804   3.133   8.140  1.00  0.00           \n"
"ATOM     97  O2P   C A   4       8.874   0.894   8.781  1.00  0.00           \n"
"ATOM     98  O5*   C A   4       7.312   2.824   8.352  1.00  0.00           \n"
"ATOM     99  C5*   C A   4       6.696   3.693   7.384  1.00  0.00           \n"
"ATOM    100 1H5*   C A   4       6.074   3.130   6.689  1.00  0.00           \n"
"ATOM    101 2H5*   C A   4       7.492   4.195   6.834  1.00  0.00           \n"
"ATOM    102  C4*   C A   4       5.843   4.732   8.087  1.00  0.00           \n"
"ATOM    103  H4*   C A   4       5.606   5.579   7.443  1.00  0.00           \n"
"ATOM    104  O4*   C A   4       4.589   4.096   8.317  1.00  0.00           \n"
"ATOM    105  C1*   C A   4       4.359   3.843   9.688  1.00  0.00           \n"
"ATOM    106  H1*   C A   4       3.495   4.407  10.038  1.00  0.00           \n"
"ATOM    107  N1    C A   4       3.910   2.425   9.770  1.00  0.00           \n"
"ATOM    108  C6    C A   4       4.806   1.403   9.741  1.00  0.00           \n"
"ATOM    109  H6    C A   4       5.862   1.615   9.658  1.00  0.00           \n"
"ATOM    110  C5    C A   4       4.398   0.111   9.815  1.00  0.00           \n"
"ATOM    111  H5    C A   4       5.132  -0.695   9.790  1.00  0.00           \n"
"ATOM    112  C4    C A   4       2.988  -0.118   9.925  1.00  0.00           \n"
"ATOM    113  N4    C A   4       2.518  -1.354  10.002  1.00  0.00           \n"
"ATOM    114 1H4    C A   4       1.518  -1.472  10.078  1.00  0.00           \n"
"ATOM    115 2H4    C A   4       3.150  -2.141   9.984  1.00  0.00           \n"
"ATOM    116  N3    C A   4       2.114   0.896   9.953  1.00  0.00           \n"
"ATOM    117  C2    C A   4       2.543   2.182   9.877  1.00  0.00           \n"
"ATOM    118  O2    C A   4       1.766   3.144   9.899  1.00  0.00           \n"
"ATOM    119  C3*   C A   4       6.323   5.157   9.475  1.00  0.00           \n"
"ATOM    120  H3*   C A   4       7.407   5.127   9.586  1.00  0.00           \n"
"ATOM    121  C2*   C A   4       5.688   4.107  10.387  1.00  0.00           \n"
"ATOM    122 1H2*   C A   4       6.471   3.516  10.862  1.00  0.00           \n"
"ATOM    123 2H2*   C A   4       5.093   4.603  11.154  1.00  0.00           \n"
"ATOM    124  O3*   C A   4       5.839   6.444   9.833  1.00  0.00           \n"
"ATOM    125  P     C A   5       5.792   6.854  11.379  1.00  0.00           \n"
"ATOM    126  O1P   C A   5       6.090   8.297  11.520  1.00  0.00           \n"
"ATOM    127  O2P   C A   5       6.654   5.939  12.161  1.00  0.00           \n"
"ATOM    128  O5*   C A   5       4.255   6.582  11.732  1.00  0.00           \n"
"ATOM    129  C5*   C A   5       3.246   6.924  10.764  1.00  0.00           \n"
"ATOM    130 1H5*   C A   5       3.073   6.103  10.068  1.00  0.00           \n"
"ATOM    131 2H5*   C A   5       3.595   7.798  10.215  1.00  0.00           \n"
"ATOM    132  C4*   C A   5       1.946   7.263  11.467  1.00  0.00           \n"
"ATOM    133  H4*   C A   5       1.256   7.809  10.823  1.00  0.00           \n"
"ATOM    134  O4*   C A   5       1.305   6.011  11.697  1.00  0.00           \n"
"ATOM    135  C1*   C A   5       1.268   5.671  13.068  1.00  0.00           \n"
"ATOM    136  H1*   C A   5       0.237   5.620  13.418  1.00  0.00           \n"
"ATOM    137  N1    C A   5       1.738   4.260  13.150  1.00  0.00           \n"
"ATOM    138  C6    C A   5       3.064   3.960  13.121  1.00  0.00           \n"
"ATOM    139  H6    C A   5       3.794   4.752  13.038  1.00  0.00           \n"
"ATOM    140  C5    C A   5       3.493   2.674  13.195  1.00  0.00           \n"
"ATOM    141  H5    C A   5       4.560   2.452  13.170  1.00  0.00           \n"
"ATOM    142  C4    C A   5       2.486   1.661  13.305  1.00  0.00           \n"
"ATOM    143  N4    C A   5       2.833   0.385  13.382  1.00  0.00           \n"
"ATOM    144 1H4    C A   5       2.094  -0.299  13.459  1.00  0.00           \n"
"ATOM    145 2H4    C A   5       3.808   0.120  13.364  1.00  0.00           \n"
"ATOM    146  N3    C A   5       1.183   1.968  13.333  1.00  0.00           \n"
"ATOM    147  C2    C A   5       0.775   3.260  13.257  1.00  0.00           \n"
"ATOM    148  O2    C A   5      -0.419   3.582  13.279  1.00  0.00           \n"
"ATOM    149  C3*   C A   5       2.084   7.888  12.855  1.00  0.00           \n"
"ATOM    150  H3*   C A   5       2.978   8.501  12.966  1.00  0.00           \n"
"ATOM    151  C2*   C A   5       2.188   6.666  13.767  1.00  0.00           \n"
"ATOM    152 1H2*   C A   5       3.169   6.649  14.242  1.00  0.00           \n"
"ATOM    153 2H2*   C A   5       1.415   6.717  14.533  1.00  0.00           \n"
"ATOM    154  O3*   C A   5       0.936   8.645  13.213  1.00  0.00           \n"
"ATOM    155  P     G A   6       0.657   8.949  14.759  1.00  0.00           \n"
"ATOM    156  O1P   G A   6       0.050  10.292  14.900  1.00  0.00           \n"
"ATOM    157  O2P   G A   6       1.892   8.716  15.541  1.00  0.00           \n"
"ATOM    158  O5*   G A   6      -0.427   7.826  15.112  1.00  0.00           \n"
"ATOM    159  C5*   G A   6      -1.443   7.510  14.144  1.00  0.00           \n"
"ATOM    160 1H5*   G A   6      -1.100   6.745  13.447  1.00  0.00           \n"
"ATOM    161 2H5*   G A   6      -1.674   8.423  13.595  1.00  0.00           \n"
"ATOM    162  C4*   G A   6      -2.695   7.020  14.847  1.00  0.00           \n"
"ATOM    163  H4*   G A   6      -3.574   7.056  14.204  1.00  0.00           \n"
"ATOM    164  O4*   G A   6      -2.477   5.630  15.077  1.00  0.00           \n"
"ATOM    165  C1*   G A   6      -2.308   5.333  16.448  1.00  0.00           \n"
"ATOM    166  H1*   G A   6      -3.112   4.686  16.799  1.00  0.00           \n"
"ATOM    167  N9    G A   6      -1.098   4.468  16.530  1.00  0.00           \n"
"ATOM    168  C8    G A   6       0.232   4.827  16.516  1.00  0.00           \n"
"ATOM    169  H8    G A   6       0.509   5.868  16.435  1.00  0.00           \n"
"ATOM    170  N7    G A   6       1.059   3.814  16.605  1.00  0.00           \n"
"ATOM    171  C5    G A   6       0.218   2.705  16.684  1.00  0.00           \n"
"ATOM    172  C6    G A   6       0.532   1.326  16.797  1.00  0.00           \n"
"ATOM    173  O6    G A   6       1.635   0.790  16.853  1.00  0.00           \n"
"ATOM    174  N1    G A   6      -0.628   0.540  16.847  1.00  0.00           \n"
"ATOM    175  H1    G A   6      -0.509  -0.450  16.929  1.00  0.00           \n"
"ATOM    176  C2    G A   6      -1.921   1.023  16.794  1.00  0.00           \n"
"ATOM    177  N2    G A   6      -2.891   0.108  16.857  1.00  0.00           \n"
"ATOM    178 1H2    G A   6      -2.778  -0.893  16.927  1.00  0.00           \n"
"ATOM    179 2H2    G A   6      -3.839   0.452  16.810  1.00  0.00           \n"
"ATOM    180  N3    G A   6      -2.214   2.317  16.688  1.00  0.00           \n"
"ATOM    181  C4    G A   6      -1.101   3.094  16.639  1.00  0.00           \n"
"ATOM    182  C3*   G A   6      -2.950   7.607  16.235  1.00  0.00           \n"
"ATOM    183  H3*   G A   6      -2.586   8.629  16.346  1.00  0.00           \n"
"ATOM    184  C2*   G A   6      -2.148   6.679  17.147  1.00  0.00           \n"
"ATOM    185 1H2*   G A   6      -1.344   7.241  17.622  1.00  0.00           \n"
"ATOM    186 2H2*   G A   6      -2.804   6.267  17.914  1.00  0.00           \n"
"ATOM    187  O3*   G A   6      -4.324   7.545  16.593  1.00  0.00           \n"
"ATOM    188  P     A A   7      -4.728   7.626  18.139  1.00  0.00           \n"
"ATOM    189  O1P   A A   7      -6.009   8.356  18.280  1.00  0.00           \n"
"ATOM    190  O2P   A A   7      -3.592   8.164  18.921  1.00  0.00           \n"
"ATOM    191  O5*   A A   7      -4.945   6.081  18.492  1.00  0.00           \n"
"ATOM    192  C5*   A A   7      -5.582   5.227  17.524  1.00  0.00           \n"
"ATOM    193 1H5*   A A   7      -4.855   4.809  16.828  1.00  0.00           \n"
"ATOM    194 2H5*   A A   7      -6.306   5.829  16.975  1.00  0.00           \n"
"ATOM    195  C4*   A A   7      -6.306   4.095  18.227  1.00  0.00           \n"
"ATOM    196  H4*   A A   7      -7.038   3.607  17.584  1.00  0.00           \n"
"ATOM    197  O4*   A A   7      -5.313   3.099  18.457  1.00  0.00           \n"
"ATOM    198  C1*   A A   7      -5.002   2.958  19.828  1.00  0.00           \n"
"ATOM    199  H1*   A A   7      -5.264   1.986  20.246  1.00  0.00           \n"
"ATOM    200  N9    A A   7      -3.515   2.969  19.910  1.00  0.00           \n"
"ATOM    201  C8    A A   7      -2.654   4.043  19.895  1.00  0.00           \n"
"ATOM    202  H8    A A   7      -3.046   5.049  19.863  1.00  0.00           \n"
"ATOM    203  N7    A A   7      -1.402   3.715  19.984  1.00  0.00           \n"
"ATOM    204  C5    A A   7      -1.422   2.330  20.063  1.00  0.00           \n"
"ATOM    205  C6    A A   7      -0.401   1.373  20.175  1.00  0.00           \n"
"ATOM    206  N6    A A   7       0.902   1.684  20.229  1.00  0.00           \n"
"ATOM    207 1H6    A A   7       1.593   0.952  20.310  1.00  0.00           \n"
"ATOM    208 2H6    A A   7       1.193   2.650  20.189  1.00  0.00           \n"
"ATOM    209  N1    A A   7      -0.769   0.079  20.229  1.00  0.00           \n"
"ATOM    210  C2    A A   7      -2.065  -0.223  20.175  1.00  0.00           \n"
"ATOM    211  H2    A A   7      -2.382  -1.255  20.218  1.00  0.00           \n"
"ATOM    212  N3    A A   7      -3.101   0.580  20.071  1.00  0.00           \n"
"ATOM    213  C4    A A   7      -2.705   1.868  20.018  1.00  0.00           \n"
"ATOM    214  C3*   A A   7      -6.858   4.420  19.615  1.00  0.00           \n"
"ATOM    215  H3*   A A   7      -7.164   5.460  19.726  1.00  0.00           \n"
"ATOM    216  C2*   A A   7      -5.664   4.141  20.527  1.00  0.00           \n"
"ATOM    217 1H2*   A A   7      -5.344   5.069  21.002  1.00  0.00           \n"
"ATOM    218 2H2*   A A   7      -5.952   3.422  21.294  1.00  0.00           \n"
"ATOM    219  O3*   A A   7      -7.933   3.562  19.973  1.00  0.00           \n"
"ATOM    220  P     T A   8      -8.308   3.390  21.519  1.00  0.00           \n"
"ATOM    221  O1P   T A   8      -9.773   3.228  21.660  1.00  0.00           \n"
"ATOM    222  O2P   T A   8      -7.705   4.493  22.301  1.00  0.00           \n"
"ATOM    223  O5*   T A   8      -7.575   2.013  21.872  1.00  0.00           \n"
"ATOM    224  C5*   T A   8      -7.588   0.948  20.904  1.00  0.00           \n"
"ATOM    225 1H5*   T A   8      -6.754   1.037  20.208  1.00  0.00           \n"
"ATOM    226 2H5*   T A   8      -8.528   1.010  20.355  1.00  0.00           \n"
"ATOM    227  C4*   T A   8      -7.509  -0.394  21.607  1.00  0.00           \n"
"ATOM    228  H4*   T A   8      -7.815  -1.219  20.964  1.00  0.00           \n"
"ATOM    229  O4*   T A   8      -6.120  -0.616  21.837  1.00  0.00           \n"
"ATOM    230  C1*   T A   8      -5.785  -0.547  23.208  1.00  0.00           \n"
"ATOM    231  H1*   T A   8      -5.420  -1.493  23.609  1.00  0.00           \n"
"ATOM    232  N1    T A   8      -4.589   0.336  23.290  1.00  0.00           \n"
"ATOM    233  C6    T A   8      -4.718   1.700  23.260  1.00  0.00           \n"
"ATOM    234  H6    T A   8      -5.700   2.147  23.221  1.00  0.00           \n"
"ATOM    235  C5    T A   8      -3.645   2.504  23.333  1.00  0.00           \n"
"ATOM    236  C7    T A   8      -3.776   3.998  23.302  1.00  0.00           \n"
"ATOM    237 1H7    T A   8      -4.388   4.327  24.142  1.00  0.00           \n"
"ATOM    238 2H7    T A   8      -2.787   4.451  23.373  1.00  0.00           \n"
"ATOM    239 3H7    T A   8      -4.248   4.303  22.368  1.00  0.00           \n"
"ATOM    240  C4    T A   8      -2.316   1.952  23.446  1.00  0.00           \n"
"ATOM    241  O4    T A   8      -1.274   2.608  23.519  1.00  0.00           \n"
"ATOM    242  N3    T A   8      -2.278   0.573  23.469  1.00  0.00           \n"
"ATOM    243  H3    T A   8      -1.295   0.109  23.555  1.00  0.00           \n"
"ATOM    244  C2    T A   8      -3.364  -0.278  23.395  1.00  0.00           \n"
"ATOM    245  O2    T A   8      -3.230  -1.489  23.423  1.00  0.00           \n"
"ATOM    246  C3*   T A   8      -8.146  -0.455  22.995  1.00  0.00           \n"
"ATOM    247  H3*   T A   8      -9.005   0.207  23.105  1.00  0.00           \n"
"ATOM    248  C2*   T A   8      -7.016   0.021  23.907  1.00  0.00           \n"
"ATOM    249 1H2*   T A   8      -7.302   0.959  24.382  1.00  0.00           \n"
"ATOM    250 2H2*   T A   8      -6.826  -0.730  24.674  1.00  0.00           \n"
"ATOM    251  O3*   T A   8      -8.512  -1.781  23.353  1.00  0.00           \n"
"ATOM    252  P     A A   9      -8.714  -2.140  24.899  1.00  0.00           \n"
"ATOM    253  O1P   A A   9      -9.804  -3.133  25.040  1.00  0.00           \n"
"ATOM    254  O2P   A A   9      -8.874  -0.894  25.681  1.00  0.00           \n"
"ATOM    255  O5*   A A   9      -7.312  -2.824  25.252  1.00  0.00           \n"
"ATOM    256  C5*   A A   9      -6.696  -3.693  24.284  1.00  0.00           \n"
"ATOM    257 1H5*   A A   9      -6.074  -3.130  23.589  1.00  0.00           \n"
"ATOM    258 2H5*   A A   9      -7.492  -4.195  23.734  1.00  0.00           \n"
"ATOM    259  C4*   A A   9      -5.843  -4.732  24.987  1.00  0.00           \n"
"ATOM    260  H4*   A A   9      -5.606  -5.579  24.343  1.00  0.00           \n"
"ATOM    261  O4*   A A   9      -4.589  -4.096  25.217  1.00  0.00           \n"
"ATOM    262  C1*   A A   9      -4.359  -3.843  26.588  1.00  0.00           \n"
"ATOM    263  H1*   A A   9      -3.515  -4.392  27.006  1.00  0.00           \n"
"ATOM    264  N9    A A   9      -3.910  -2.425  26.670  1.00  0.00           \n"
"ATOM    265  C8    A A   9      -4.665  -1.274  26.655  1.00  0.00           \n"
"ATOM    266  H8    A A   9      -5.743  -1.336  26.623  1.00  0.00           \n"
"ATOM    267  N7    A A   9      -3.967  -0.185  26.744  1.00  0.00           \n"
"ATOM    268  C5    A A   9      -2.656  -0.632  26.823  1.00  0.00           \n"
"ATOM    269  C6    A A   9      -1.429   0.043  26.935  1.00  0.00           \n"
"ATOM    270  N6    A A   9      -1.322   1.378  26.989  1.00  0.00           \n"
"ATOM    271 1H6    A A   9      -0.412   1.809  27.070  1.00  0.00           \n"
"ATOM    272 2H6    A A   9      -2.151   1.953  26.949  1.00  0.00           \n"
"ATOM    273  N1    A A   9      -0.313  -0.707  26.989  1.00  0.00           \n"
"ATOM    274  C2    A A   9      -0.426  -2.033  26.935  1.00  0.00           \n"
"ATOM    275  H2    A A   9       0.457  -2.653  26.978  1.00  0.00           \n"
"ATOM    276  N3    A A   9      -1.510  -2.770  26.831  1.00  0.00           \n"
"ATOM    277  C4    A A   9      -2.613  -1.996  26.778  1.00  0.00           \n"
"ATOM    278  C3*   A A   9      -6.323  -5.157  26.375  1.00  0.00           \n"
"ATOM    279  H3*   A A   9      -7.407  -5.127  26.486  1.00  0.00           \n"
"ATOM    280  C2*   A A   9      -5.688  -4.107  27.287  1.00  0.00           \n"
"ATOM    281 1H2*   A A   9      -6.471  -3.516  27.762  1.00  0.00           \n"
"ATOM    282 2H2*   A A   9      -5.093  -4.603  28.054  1.00  0.00           \n"
"ATOM    283  O3*   A A   9      -5.839  -6.444  26.733  1.00  0.00           \n"
"ATOM    284  P     C A  10      -5.792  -6.854  28.279  1.00  0.00           \n"
"ATOM    285  O1P   C A  10      -6.090  -8.297  28.420  1.00  0.00           \n"
"ATOM    286  O2P   C A  10      -6.654  -5.939  29.061  1.00  0.00           \n"
"ATOM    287  O5*   C A  10      -4.255  -6.582  28.632  1.00  0.00           \n"
"ATOM    288  C5*   C A  10      -3.246  -6.924  27.664  1.00  0.00           \n"
"ATOM    289 1H5*   C A  10      -3.073  -6.103  26.968  1.00  0.00           \n"
"ATOM    290 2H5*   C A  10      -3.595  -7.798  27.115  1.00  0.00           \n"
"ATOM    291  C4*   C A  10      -1.946  -7.263  28.367  1.00  0.00           \n"
"ATOM    292  H4*   C A  10      -1.256  -7.809  27.723  1.00  0.00           \n"
"ATOM    293  O4*   C A  10      -1.305  -6.011  28.597  1.00  0.00           \n"
"ATOM    294  C1*   C A  10      -1.268  -5.671  29.968  1.00  0.00           \n"
"ATOM    295  H1*   C A  10      -0.237  -5.620  30.318  1.00  0.00           \n"
"ATOM    296  N1    C A  10      -1.738  -4.260  30.050  1.00  0.00           \n"
"ATOM    297  C6    C A  10      -3.064  -3.960  30.021  1.00  0.00           \n"
"ATOM    298  H6    C A  10      -3.794  -4.752  29.938  1.00  0.00           \n"
"ATOM    299  C5    C A  10      -3.493  -2.674  30.095  1.00  0.00           \n"
"ATOM    300  H5    C A  10      -4.560  -2.452  30.070  1.00  0.00           \n"
"ATOM    301  C4    C A  10      -2.486  -1.661  30.205  1.00  0.00           \n"
"ATOM    302  N4    C A  10      -2.833  -0.385  30.282  1.00  0.00           \n"
"ATOM    303 1H4    C A  10      -2.094   0.299  30.359  1.00  0.00           \n"
"ATOM    304 2H4    C A  10      -3.808  -0.120  30.264  1.00  0.00           \n"
"ATOM    305  N3    C A  10      -1.183  -1.968  30.233  1.00  0.00           \n"
"ATOM    306  C2    C A  10      -0.775  -3.260  30.157  1.00  0.00           \n"
"ATOM    307  O2    C A  10       0.419  -3.582  30.179  1.00  0.00           \n"
"ATOM    308  C3*   C A  10      -2.084  -7.888  29.755  1.00  0.00           \n"
"ATOM    309  H3*   C A  10      -2.978  -8.501  29.866  1.00  0.00           \n"
"ATOM    310  C2*   C A  10      -2.188  -6.666  30.667  1.00  0.00           \n"
"ATOM    311 1H2*   C A  10      -3.169  -6.649  31.142  1.00  0.00           \n"
"ATOM    312 2H2*   C A  10      -1.415  -6.717  31.433  1.00  0.00           \n"
"ATOM    313  O3*   C A  10      -0.936  -8.645  30.113  1.00  0.00           \n"
"ATOM    314  P     G A  11      -0.657  -8.949  31.659  1.00  0.00           \n"
"ATOM    315  O1P   G A  11      -0.050 -10.292  31.800  1.00  0.00           \n"
"ATOM    316  O2P   G A  11      -1.892  -8.716  32.441  1.00  0.00           \n"
"ATOM    317  O5*   G A  11       0.427  -7.826  32.012  1.00  0.00           \n"
"ATOM    318  C5*   G A  11       1.443  -7.510  31.044  1.00  0.00           \n"
"ATOM    319 1H5*   G A  11       1.100  -6.745  30.347  1.00  0.00           \n"
"ATOM    320 2H5*   G A  11       1.674  -8.423  30.495  1.00  0.00           \n"
"ATOM    321  C4*   G A  11       2.695  -7.020  31.747  1.00  0.00           \n"
"ATOM    322  H4*   G A  11       3.574  -7.056  31.104  1.00  0.00           \n"
"ATOM    323  O4*   G A  11       2.477  -5.630  31.977  1.00  0.00           \n"
"ATOM    324  C1*   G A  11       2.308  -5.333  33.348  1.00  0.00           \n"
"ATOM    325  H1*   G A  11       3.112  -4.686  33.699  1.00  0.00           \n"
"ATOM    326  N9    G A  11       1.098  -4.468  33.430  1.00  0.00           \n"
"ATOM    327  C8    G A  11      -0.232  -4.827  33.416  1.00  0.00           \n"
"ATOM    328  H8    G A  11      -0.509  -5.868  33.335  1.00  0.00           \n"
"ATOM    329  N7    G A  11      -1.059  -3.814  33.505  1.00  0.00           \n"
"ATOM    330  C5    G A  11      -0.218  -2.705  33.584  1.00  0.00           \n"
"ATOM    331  C6    G A  11      -0.532  -1.326  33.697  1.00  0.00           \n"
"ATOM    332  O6    G A  11      -1.635  -0.790  33.753  1.00  0.00           \n"
"ATOM    333  N1    G A  11       0.628  -0.540  33.747  1.00  0.00           \n"
"ATOM    334  H1    G A  11       0.509   0.450  33.829  1.00  0.00           \n"
"ATOM    335  C2    G A  11       1.921  -1.023  33.694  1.00  0.00           \n"
"ATOM    336  N2    G A  11       2.891  -0.108  33.757  1.00  0.00           \n"
"ATOM    337 1H2    G A  11       2.778   0.893  33.827  1.00  0.00           \n"
"ATOM    338 2H2    G A  11       3.839  -0.452  33.710  1.00  0.00           \n"
"ATOM    339  N3    G A  11       2.214  -2.317  33.588  1.00  0.00           \n"
"ATOM    340  C4    G A  11       1.101  -3.094  33.539  1.00  0.00           \n"
"ATOM    341  C3*   G A  11       2.950  -7.607  33.135  1.00  0.00           \n"
"ATOM    342  H3*   G A  11       2.586  -8.629  33.246  1.00  0.00           \n"
"ATOM    343  C2*   G A  11       2.148  -6.679  34.047  1.00  0.00           \n"
"ATOM    344 1H2*   G A  11       1.344  -7.241  34.522  1.00  0.00           \n"
"ATOM    345 2H2*   G A  11       2.804  -6.267  34.814  1.00  0.00           \n"
"ATOM    346  O3*   G A  11       4.324  -7.545  33.493  1.00  0.00           \n"
"ATOM    347  P     A A  12       4.728  -7.626  35.039  1.00  0.00           \n"
"ATOM    348  O1P   A A  12       6.009  -8.356  35.180  1.00  0.00           \n"
"ATOM    349  O2P   A A  12       3.592  -8.164  35.821  1.00  0.00           \n"
"ATOM    350  O5*   A A  12       4.945  -6.081  35.392  1.00  0.00           \n"
"ATOM    351  C5*   A A  12       5.582  -5.227  34.424  1.00  0.00           \n"
"ATOM    352 1H5*   A A  12       4.855  -4.809  33.728  1.00  0.00           \n"
"ATOM    353 2H5*   A A  12       6.306  -5.829  33.875  1.00  0.00           \n"
"ATOM    354  C4*   A A  12       6.306  -4.095  35.127  1.00  0.00           \n"
"ATOM    355  H4*   A A  12       7.038  -3.607  34.484  1.00  0.00           \n"
"ATOM    356  O4*   A A  12       5.313  -3.099  35.357  1.00  0.00           \n"
"ATOM    357  C1*   A A  12       5.002  -2.958  36.728  1.00  0.00           \n"
"ATOM    358  H1*   A A  12       5.264  -1.986  37.146  1.00  0.00           \n"
"ATOM    359  N9    A A  12       3.515  -2.969  36.810  1.00  0.00           \n"
"ATOM    360  C8    A A  12       2.654  -4.043  36.795  1.00  0.00           \n"
"ATOM    361  H8    A A  12       3.046  -5.049  36.763  1.00  0.00           \n"
"ATOM    362  N7    A A  12       1.402  -3.715  36.884  1.00  0.00           \n"
"ATOM    363  C5    A A  12       1.422  -2.330  36.963  1.00  0.00           \n"
"ATOM    364  C6    A A  12       0.401  -1.373  37.075  1.00  0.00           \n"
"ATOM    365  N6    A A  12      -0.902  -1.684  37.129  1.00  0.00           \n"
"ATOM    366 1H6    A A  12      -1.593  -0.952  37.210  1.00  0.00           \n"
"ATOM    367 2H6    A A  12      -1.193  -2.650  37.089  1.00  0.00           \n"
"ATOM    368  N1    A A  12       0.769  -0.079  37.129  1.00  0.00           \n"
"ATOM    369  C2    A A  12       2.065   0.223  37.075  1.00  0.00           \n"
"ATOM    370  H2    A A  12       2.382   1.255  37.118  1.00  0.00           \n"
"ATOM    371  N3    A A  12       3.101  -0.580  36.971  1.00  0.00           \n"
"ATOM    372  C4    A A  12       2.705  -1.868  36.918  1.00  0.00           \n"
"ATOM    373  C3*   A A  12       6.858  -4.420  36.515  1.00  0.00           \n"
"ATOM    374  H3*   A A  12       7.164  -5.460  36.626  1.00  0.00           \n"
"ATOM    375  C2*   A A  12       5.664  -4.141  37.427  1.00  0.00           \n"
"ATOM    376 1H2*   A A  12       5.344  -5.069  37.902  1.00  0.00           \n"
"ATOM    377 2H2*   A A  12       5.952  -3.422  38.194  1.00  0.00           \n"
"ATOM    378  O3*   A A  12       7.933  -3.562  36.873  1.00  0.00           \n"
"ATOM    379  P     T A  13       8.308  -3.390  38.419  1.00  0.00           \n"
"ATOM    380  O1P   T A  13       9.773  -3.228  38.560  1.00  0.00           \n"
"ATOM    381  O2P   T A  13       7.705  -4.493  39.201  1.00  0.00           \n"
"ATOM    382  O5*   T A  13       7.575  -2.013  38.772  1.00  0.00           \n"
"ATOM    383  C5*   T A  13       7.588  -0.948  37.804  1.00  0.00           \n"
"ATOM    384 1H5*   T A  13       6.754  -1.037  37.108  1.00  0.00           \n"
"ATOM    385 2H5*   T A  13       8.528  -1.010  37.255  1.00  0.00           \n"
"ATOM    386  C4*   T A  13       7.509   0.394  38.507  1.00  0.00           \n"
"ATOM    387  H4*   T A  13       7.815   1.219  37.864  1.00  0.00           \n"
"ATOM    388  O4*   T A  13       6.120   0.616  38.737  1.00  0.00           \n"
"ATOM    389  C1*   T A  13       5.785   0.547  40.108  1.00  0.00           \n"
"ATOM    390  H1*   T A  13       5.420   1.493  40.509  1.00  0.00           \n"
"ATOM    391  N1    T A  13       4.589  -0.336  40.190  1.00  0.00           \n"
"ATOM    392  C6    T A  13       4.718  -1.700  40.160  1.00  0.00           \n"
"ATOM    393  H6    T A  13       5.700  -2.147  40.121  1.00  0.00           \n"
"ATOM    394  C5    T A  13       3.645  -2.504  40.233  1.00  0.00           \n"
"ATOM    395  C7    T A  13       3.776  -3.998  40.202  1.00  0.00           \n"
"ATOM    396 1H7    T A  13       4.388  -4.327  41.042  1.00  0.00           \n"
"ATOM    397 2H7    T A  13       2.787  -4.451  40.273  1.00  0.00           \n"
"ATOM    398 3H7    T A  13       4.248  -4.303  39.268  1.00  0.00           \n"
"ATOM    399  C4    T A  13       2.316  -1.952  40.346  1.00  0.00           \n"
"ATOM    400  O4    T A  13       1.274  -2.608  40.419  1.00  0.00           \n"
"ATOM    401  N3    T A  13       2.278  -0.573  40.369  1.00  0.00           \n"
"ATOM    402  H3    T A  13       1.295  -0.109  40.455  1.00  0.00           \n"
"ATOM    403  C2    T A  13       3.364   0.278  40.295  1.00  0.00           \n"
"ATOM    404  O2    T A  13       3.230   1.489  40.323  1.00  0.00           \n"
"ATOM    405  C3*   T A  13       8.146   0.455  39.895  1.00  0.00           \n"
"ATOM    406  H3*   T A  13       9.005  -0.207  40.005  1.00  0.00           \n"
"ATOM    407  C2*   T A  13       7.016  -0.021  40.807  1.00  0.00           \n"
"ATOM    408 1H2*   T A  13       7.302  -0.959  41.282  1.00  0.00           \n"
"ATOM    409 2H2*   T A  13       6.826   0.730  41.574  1.00  0.00           \n"
"ATOM    410  O3*   T A  13       8.512   1.781  40.253  1.00  0.00           \n"
"ATOM    411  P     A A  14       8.714   2.140  41.799  1.00  0.00           \n"
"ATOM    412  O1P   A A  14       9.804   3.133  41.940  1.00  0.00           \n"
"ATOM    413  O2P   A A  14       8.874   0.894  42.581  1.00  0.00           \n"
"ATOM    414  O5*   A A  14       7.312   2.824  42.152  1.00  0.00           \n"
"ATOM    415  C5*   A A  14       6.696   3.693  41.184  1.00  0.00           \n"
"ATOM    416 1H5*   A A  14       6.074   3.130  40.489  1.00  0.00           \n"
"ATOM    417 2H5*   A A  14       7.492   4.195  40.634  1.00  0.00           \n"
"ATOM    418  C4*   A A  14       5.843   4.732  41.887  1.00  0.00           \n"
"ATOM    419  H4*   A A  14       5.606   5.579  41.243  1.00  0.00           \n"
"ATOM    420  O4*   A A  14       4.589   4.096  42.117  1.00  0.00           \n"
"ATOM    421  C1*   A A  14       4.359   3.843  43.488  1.00  0.00           \n"
"ATOM    422  H1*   A A  14       3.515   4.392  43.906  1.00  0.00           \n"
"ATOM    423  N9    A A  14       3.910   2.425  43.570  1.00  0.00           \n"
"ATOM    424  C8    A A  14       4.665   1.274  43.555  1.00  0.00           \n"
"ATOM    425  H8    A A  14       5.743   1.336  43.523  1.00  0.00           \n"
"ATOM    426  N7    A A  14       3.967   0.185  43.644  1.00  0.00           \n"
"ATOM    427  C5    A A  14       2.656   0.632  43.723  1.00  0.00           \n"
"ATOM    428  C6    A A  14       1.429  -0.043  43.835  1.00  0.00           \n"
"ATOM    429  N6    A A  14       1.322  -1.378  43.889  1.00  0.00           \n"
"ATOM    430 1H6    A A  14       0.412  -1.809  43.970  1.00  0.00           \n"
"ATOM    431 2H6    A A  14       2.151  -1.953  43.849  1.00  0.00           \n"
"ATOM    432  N1    A A  14       0.313   0.707  43.889  1.00  0.00           \n"
"ATOM    433  C2    A A  14       0.426   2.033  43.835  1.00  0.00           \n"
"ATOM    434  H2    A A  14      -0.457   2.653  43.878  1.00  0.00           \n"
"ATOM    435  N3    A A  14       1.510   2.770  43.731  1.00  0.00           \n"
"ATOM    436  C4    A A  14       2.613   1.996  43.678  1.00  0.00           \n"
"ATOM    437  C3*   A A  14       6.323   5.157  43.275  1.00  0.00           \n"
"ATOM    438  H3*   A A  14       7.407   5.127  43.386  1.00  0.00           \n"
"ATOM    439  C2*   A A  14       5.688   4.107  44.187  1.00  0.00           \n"
"ATOM    440 1H2*   A A  14       6.471   3.516  44.662  1.00  0.00           \n"
"ATOM    441 2H2*   A A  14       5.093   4.603  44.954  1.00  0.00           \n"
"ATOM    442  O3*   A A  14       5.839   6.444  43.633  1.00  0.00           \n"
"ATOM    443  P     G A  15       5.792   6.854  45.179  1.00  0.00           \n"
"ATOM    444  O1P   G A  15       6.090   8.297  45.320  1.00  0.00           \n"
"ATOM    445  O2P   G A  15       6.654   5.939  45.961  1.00  0.00           \n"
"ATOM    446  O5*   G A  15       4.255   6.582  45.532  1.00  0.00           \n"
"ATOM    447  C5*   G A  15       3.246   6.924  44.564  1.00  0.00           \n"
"ATOM    448 1H5*   G A  15       3.073   6.103  43.868  1.00  0.00           \n"
"ATOM    449 2H5*   G A  15       3.595   7.798  44.015  1.00  0.00           \n"
"ATOM    450  C4*   G A  15       1.946   7.263  45.267  1.00  0.00           \n"
"ATOM    451  H4*   G A  15       1.256   7.809  44.623  1.00  0.00           \n"
"ATOM    452  O4*   G A  15       1.305   6.011  45.497  1.00  0.00           \n"
"ATOM    453  C1*   G A  15       1.268   5.671  46.868  1.00  0.00           \n"
"ATOM    454  H1*   G A  15       0.237   5.620  47.220  1.00  0.00           \n"
"ATOM    455  N9    G A  15       1.738   4.260  46.950  1.00  0.00           \n"
"ATOM    456  C8    G A  15       3.025   3.769  46.936  1.00  0.00           \n"
"ATOM    457  H8    G A  15       3.861   4.448  46.855  1.00  0.00           \n"
"ATOM    458  N7    G A  15       3.098   2.463  47.025  1.00  0.00           \n"
"ATOM    459  C5    G A  15       1.766   2.061  47.104  1.00  0.00           \n"
"ATOM    460  C6    G A  15       1.210   0.760  47.217  1.00  0.00           \n"
"ATOM    461  O6    G A  15       1.787  -0.322  47.273  1.00  0.00           \n"
"ATOM    462  N1    G A  15      -0.191   0.806  47.267  1.00  0.00           \n"
"ATOM    463  H1    G A  15      -0.677  -0.064  47.349  1.00  0.00           \n"
"ATOM    464  C2    G A  15      -0.953   1.956  47.214  1.00  0.00           \n"
"ATOM    465  N2    G A  15      -2.275   1.787  47.277  1.00  0.00           \n"
"ATOM    466 1H2    G A  15      -2.773   0.911  47.347  1.00  0.00           \n"
"ATOM    467 2H2    G A  15      -2.839   2.623  47.230  1.00  0.00           \n"
"ATOM    468  N3    G A  15      -0.429   3.176  47.108  1.00  0.00           \n"
"ATOM    469  C4    G A  15       0.928   3.150  47.059  1.00  0.00           \n"
"ATOM    470  C3*   G A  15       2.084   7.888  46.655  1.00  0.00           \n"
"ATOM    471  H3*   G A  15       2.978   8.501  46.766  1.00  0.00           \n"
"ATOM    472  C2*   G A  15       2.188   6.666  47.567  1.00  0.00           \n"
"ATOM    473 1H2*   G A  15       3.169   6.649  48.042  1.00  0.00           \n"
"ATOM    474 2H2*   G A  15       1.415   6.717  48.333  1.00  0.00           \n"
"ATOM    475  O3*   G A  15       0.936   8.645  47.013  1.00  0.00           \n"
"ATOM    476  P     A A  16       0.657   8.949  48.559  1.00  0.00           \n"
"ATOM    477  O1P   A A  16       0.050  10.292  48.700  1.00  0.00           \n"
"ATOM    478  O2P   A A  16       1.892   8.716  49.341  1.00  0.00           \n"
"ATOM    479  O5*   A A  16      -0.427   7.826  48.912  1.00  0.00           \n"
"ATOM    480  C5*   A A  16      -1.443   7.510  47.944  1.00  0.00           \n"
"ATOM    481 1H5*   A A  16      -1.100   6.745  47.247  1.00  0.00           \n"
"ATOM    482 2H5*   A A  16      -1.674   8.423  47.395  1.00  0.00           \n"
"ATOM    483  C4*   A A  16      -2.695   7.020  48.647  1.00  0.00           \n"
"ATOM    484  H4*   A A  16      -3.574   7.056  48.004  1.00  0.00           \n"
"ATOM    485  O4*   A A  16      -2.477   5.630  48.877  1.00  0.00           \n"
"ATOM    486  C1*   A A  16      -2.308   5.333  50.248  1.00  0.00           \n"
"ATOM    487  H1*   A A  16      -3.091   4.700  50.665  1.00  0.00           \n"
"ATOM    488  N9    A A  16      -1.098   4.468  50.330  1.00  0.00           \n"
"ATOM    489  C8    A A  16       0.229   4.831  50.315  1.00  0.00           \n"
"ATOM    490  H8    A A  16       0.503   5.875  50.283  1.00  0.00           \n"
"ATOM    491  N7    A A  16       1.050   3.830  50.404  1.00  0.00           \n"
"ATOM    492  C5    A A  16       0.219   2.721  50.483  1.00  0.00           \n"
"ATOM    493  C6    A A  16       0.482   1.346  50.595  1.00  0.00           \n"
"ATOM    494  N6    A A  16       1.719   0.832  50.649  1.00  0.00           \n"
"ATOM    495 1H6    A A  16       1.848  -0.166  50.730  1.00  0.00           \n"
"ATOM    496 2H6    A A  16       2.522   1.443  50.609  1.00  0.00           \n"
"ATOM    497  N1    A A  16      -0.575   0.516  50.649  1.00  0.00           \n"
"ATOM    498  C2    A A  16      -1.802   1.033  50.595  1.00  0.00           \n"
"ATOM    499  H2    A A  16      -2.665   0.385  50.638  1.00  0.00           \n"
"ATOM    500  N3    A A  16      -2.168   2.292  50.491  1.00  0.00           \n"
"ATOM    501  C4    A A  16      -1.091   3.102  50.438  1.00  0.00           \n"
"ATOM    502  C3*   A A  16      -2.950   7.607  50.035  1.00  0.00           \n"
"ATOM    503  H3*   A A  16      -2.586   8.629  50.146  1.00  0.00           \n"
"ATOM    504  C2*   A A  16      -2.148   6.679  50.947  1.00  0.00           \n"
"ATOM    505 1H2*   A A  16      -1.344   7.241  51.422  1.00  0.00           \n"
"ATOM    506 2H2*   A A  16      -2.804   6.267  51.714  1.00  0.00           \n"
"ATOM    507  O3*   A A  16      -4.324   7.545  50.393  1.00  0.00           \n"
"ATOM    508  P     C A  17      -4.728   7.626  51.939  1.00  0.00           \n"
"ATOM    509  O1P   C A  17      -6.009   8.356  52.080  1.00  0.00           \n"
"ATOM    510  O2P   C A  17      -3.592   8.164  52.721  1.00  0.00           \n"
"ATOM    511  O5*   C A  17      -4.945   6.081  52.292  1.00  0.00           \n"
"ATOM    512  C5*   C A  17      -5.582   5.227  51.324  1.00  0.00           \n"
"ATOM    513 1H5*   C A  17      -4.855   4.809  50.628  1.00  0.00           \n"
"ATOM    514 2H5*   C A  17      -6.306   5.829  50.775  1.00  0.00           \n"
"ATOM    515  C4*   C A  17      -6.306   4.095  52.027  1.00  0.00           \n"
"ATOM    516  H4*   C A  17      -7.038   3.607  51.384  1.00  0.00           \n"
"ATOM    517  O4*   C A  17      -5.313   3.099  52.257  1.00  0.00           \n"
"ATOM    518  C1*   C A  17      -5.002   2.958  53.628  1.00  0.00           \n"
"ATOM    519  H1*   C A  17      -5.272   1.961  53.978  1.00  0.00           \n"
"ATOM    520  N1    C A  17      -3.515   2.969  53.710  1.00  0.00           \n"
"ATOM    521  C6    C A  17      -2.819   4.138  53.681  1.00  0.00           \n"
"ATOM    522  H6    C A  17      -3.346   5.077  53.598  1.00  0.00           \n"
"ATOM    523  C5    C A  17      -1.464   4.148  53.755  1.00  0.00           \n"
"ATOM    524  H5    C A  17      -0.924   5.095  53.730  1.00  0.00           \n"
"ATOM    525  C4    C A  17      -0.811   2.878  53.865  1.00  0.00           \n"
"ATOM    526  N4    C A  17       0.509   2.813  53.942  1.00  0.00           \n"
"ATOM    527 1H4    C A  17       0.930   1.898  54.019  1.00  0.00           \n"
"ATOM    528 2H4    C A  17       1.062   3.658  53.924  1.00  0.00           \n"
"ATOM    529  N3    C A  17      -1.506   1.733  53.893  1.00  0.00           \n"
"ATOM    530  C2    C A  17      -2.861   1.744  53.817  1.00  0.00           \n"
"ATOM    531  O2    C A  17      -3.536   0.708  53.839  1.00  0.00           \n"
"ATOM    532  C3*   C A  17      -6.858   4.420  53.415  1.00  0.00           \n"
"ATOM    533  H3*   C A  17      -7.164   5.460  53.526  1.00  0.00           \n"
"ATOM    534  C2*   C A  17      -5.664   4.141  54.327  1.00  0.00           \n"
"ATOM    535 1H2*   C A  17      -5.344   5.069  54.802  1.00  0.00           \n"
"ATOM    536 2H2*   C A  17      -5.952   3.422  55.094  1.00  0.00           \n"
"ATOM    537  O3*   C A  17      -7.933   3.562  53.773  1.00  0.00           \n"
"ATOM    538  P     A A  18      -8.308   3.390  55.319  1.00  0.00           \n"
"ATOM    539  O1P   A A  18      -9.773   3.228  55.460  1.00  0.00           \n"
"ATOM    540  O2P   A A  18      -7.705   4.493  56.101  1.00  0.00           \n"
"ATOM    541  O5*   A A  18      -7.575   2.013  55.672  1.00  0.00           \n"
"ATOM    542  C5*   A A  18      -7.588   0.948  54.704  1.00  0.00           \n"
"ATOM    543 1H5*   A A  18      -6.754   1.037  54.008  1.00  0.00           \n"
"ATOM    544 2H5*   A A  18      -8.528   1.010  54.155  1.00  0.00           \n"
"ATOM    545  C4*   A A  18      -7.509  -0.394  55.407  1.00  0.00           \n"
"ATOM    546  H4*   A A  18      -7.815  -1.219  54.764  1.00  0.00           \n"
"ATOM    547  O4*   A A  18      -6.120  -0.616  55.637  1.00  0.00           \n"
"ATOM    548  C1*   A A  18      -5.785  -0.547  57.008  1.00  0.00           \n"
"ATOM    549  H1*   A A  18      -5.425  -1.487  57.425  1.00  0.00           \n"
"ATOM    550  N9    A A  18      -4.589   0.336  57.090  1.00  0.00           \n"
"ATOM    551  C8    A A  18      -4.523   1.711  57.075  1.00  0.00           \n"
"ATOM    552  H8    A A  18      -5.432   2.294  57.043  1.00  0.00           \n"
"ATOM    553  N7    A A  18      -3.318   2.182  57.164  1.00  0.00           \n"
"ATOM    554  C5    A A  18      -2.520   1.050  57.243  1.00  0.00           \n"
"ATOM    555  C6    A A  18      -1.131   0.875  57.355  1.00  0.00           \n"
"ATOM    556  N6    A A  18      -0.260   1.892  57.409  1.00  0.00           \n"
"ATOM    557 1H6    A A  18       0.729   1.706  57.490  1.00  0.00           \n"
"ATOM    558 2H6    A A  18      -0.593   2.845  57.369  1.00  0.00           \n"
"ATOM    559  N1    A A  18      -0.669  -0.388  57.409  1.00  0.00           \n"
"ATOM    560  C2    A A  18      -1.540  -1.394  57.355  1.00  0.00           \n"
"ATOM    561  H2    A A  18      -1.189  -2.415  57.398  1.00  0.00           \n"
"ATOM    562  N3    A A  18      -2.850  -1.354  57.251  1.00  0.00           \n"
"ATOM    563  C4    A A  18      -3.287  -0.079  57.198  1.00  0.00           \n"
"ATOM    564  C3*   A A  18      -8.146  -0.455  56.795  1.00  0.00           \n"
"ATOM    565  H3*   A A  18      -9.005   0.207  56.905  1.00  0.00           \n"
"ATOM    566  C2*   A A  18      -7.016   0.021  57.707  1.00  0.00           \n"
"ATOM    567 1H2*   A A  18      -7.302   0.959  58.182  1.00  0.00           \n"
"ATOM    568 2H2*   A A  18      -6.826  -0.730  58.474  1.00  0.00           \n"
"ATOM    569  O3*   A A  18      -8.512  -1.781  57.153  1.00  0.00           \n"
"ATOM    570  P     T A  19      -8.714  -2.140  58.699  1.00  0.00           \n"
"ATOM    571  O1P   T A  19      -9.804  -3.133  58.840  1.00  0.00           \n"
"ATOM    572  O2P   T A  19      -8.874  -0.894  59.481  1.00  0.00           \n"
"ATOM    573  O5*   T A  19      -7.312  -2.824  59.052  1.00  0.00           \n"
"ATOM    574  C5*   T A  19      -6.696  -3.693  58.084  1.00  0.00           \n"
"ATOM    575 1H5*   T A  19      -6.074  -3.130  57.389  1.00  0.00           \n"
"ATOM    576 2H5*   T A  19      -7.492  -4.195  57.534  1.00  0.00           \n"
"ATOM    577  C4*   T A  19      -5.843  -4.732  58.787  1.00  0.00           \n"
"ATOM    578  H4*   T A  19      -5.606  -5.579  58.143  1.00  0.00           \n"
"ATOM    579  O4*   T A  19      -4.589  -4.096  59.017  1.00  0.00           \n"
"ATOM    580  C1*   T A  19      -4.359  -3.843  60.388  1.00  0.00           \n"
"ATOM    581  H1*   T A  19      -3.508  -4.393  60.789  1.00  0.00           \n"
"ATOM    582  N1    T A  19      -3.910  -2.425  60.470  1.00  0.00           \n"
"ATOM    583  C6    T A  19      -4.816  -1.398  60.440  1.00  0.00           \n"
"ATOM    584  H6    T A  19      -5.874  -1.613  60.401  1.00  0.00           \n"
"ATOM    585  C5    T A  19      -4.420  -0.117  60.513  1.00  0.00           \n"
"ATOM    586  C7    T A  19      -5.404   1.015  60.482  1.00  0.00           \n"
"ATOM    587 1H7    T A  19      -6.092   0.922  61.322  1.00  0.00           \n"
"ATOM    588 2H7    T A  19      -4.871   1.963  60.553  1.00  0.00           \n"
"ATOM    589 3H7    T A  19      -5.965   0.984  59.548  1.00  0.00           \n"
"ATOM    590  C4    T A  19      -3.021   0.218  60.626  1.00  0.00           \n"
"ATOM    591  O4    T A  19      -2.564   1.361  60.699  1.00  0.00           \n"
"ATOM    592  N3    T A  19      -2.180  -0.876  60.649  1.00  0.00           \n"
"ATOM    593  H3    T A  19      -1.112  -0.674  60.735  1.00  0.00           \n"
"ATOM    594  C2    T A  19      -2.558  -2.202  60.575  1.00  0.00           \n"
"ATOM    595  O2    T A  19      -1.738  -3.103  60.603  1.00  0.00           \n"
"ATOM    596  C3*   T A  19      -6.323  -5.157  60.175  1.00  0.00           \n"
"ATOM    597  H3*   T A  19      -7.407  -5.127  60.286  1.00  0.00           \n"
"ATOM    598  C2*   T A  19      -5.688  -4.107  61.087  1.00  0.00           \n"
"ATOM    599 1H2*   T A  19      -6.471  -3.516  61.562  1.00  0.00           \n"
"ATOM    600 2H2*   T A  19      -5.093  -4.603  61.854  1.00  0.00           \n"
"ATOM    601  O3*   T A  19      -5.839  -6.444  60.533  1.00  0.00           \n"
"ATOM    602  P     G A  20      -5.792  -6.854  62.079  1.00  0.00           \n"
"ATOM    603  O1P   G A  20      -6.090  -8.297  62.220  1.00  0.00           \n"
"ATOM    604  O2P   G A  20      -6.654  -5.939  62.861  1.00  0.00           \n"
"ATOM    605  O5*   G A  20      -4.255  -6.582  62.432  1.00  0.00           \n"
"ATOM    606  C5*   G A  20      -3.246  -6.924  61.464  1.00  0.00           \n"
"ATOM    607 1H5*   G A  20      -3.073  -6.103  60.768  1.00  0.00           \n"
"ATOM    608 2H5*   G A  20      -3.595  -7.798  60.915  1.00  0.00           \n"
"ATOM    609  C4*   G A  20      -1.946  -7.263  62.167  1.00  0.00           \n"
"ATOM    610  H4*   G A  20      -1.256  -7.809  61.523  1.00  0.00           \n"
"ATOM    611  O4*   G A  20      -1.305  -6.011  62.397  1.00  0.00           \n"
"ATOM    612  C1*   G A  20      -1.268  -5.671  63.768  1.00  0.00           \n"
"ATOM    613  H1*   G A  20      -0.237  -5.620  64.120  1.00  0.00           \n"
"ATOM    614  N9    G A  20      -1.738  -4.260  63.850  1.00  0.00           \n"
"ATOM    615  C8    G A  20      -3.025  -3.769  63.836  1.00  0.00           \n"
"ATOM    616  H8    G A  20      -3.861  -4.448  63.755  1.00  0.00           \n"
"ATOM    617  N7    G A  20      -3.098  -2.463  63.925  1.00  0.00           \n"
"ATOM    618  C5    G A  20      -1.766  -2.061  64.004  1.00  0.00           \n"
"ATOM    619  C6    G A  20      -1.210  -0.760  64.117  1.00  0.00           \n"
"ATOM    620  O6    G A  20      -1.787   0.322  64.173  1.00  0.00           \n"
"ATOM    621  N1    G A  20       0.191  -0.806  64.167  1.00  0.00           \n"
"ATOM    622  H1    G A  20       0.677   0.064  64.249  1.00  0.00           \n"
"ATOM    623  C2    G A  20       0.953  -1.956  64.114  1.00  0.00           \n"
"ATOM    624  N2    G A  20       2.275  -1.787  64.177  1.00  0.00           \n"
"ATOM    625 1H2    G A  20       2.773  -0.911  64.247  1.00  0.00           \n"
"ATOM    626 2H2    G A  20       2.839  -2.623  64.130  1.00  0.00           \n"
"ATOM    627  N3    G A  20       0.429  -3.176  64.008  1.00  0.00           \n"
"ATOM    628  C4    G A  20      -0.928  -3.150  63.959  1.00  0.00           \n"
"ATOM    629  C3*   G A  20      -2.084  -7.888  63.555  1.00  0.00           \n"
"ATOM    630  H3*   G A  20      -2.978  -8.501  63.666  1.00  0.00           \n"
"ATOM    631  C2*   G A  20      -2.188  -6.666  64.467  1.00  0.00           \n"
"ATOM    632 1H2*   G A  20      -3.169  -6.649  64.942  1.00  0.00           \n"
"ATOM    633 2H2*   G A  20      -1.415  -6.717  65.233  1.00  0.00           \n"
"ATOM    634  O3*   G A  20      -0.936  -8.645  63.913  1.00  0.00           \n"
"ATOM    635  H3    G A  20      -0.977  -9.058  64.779  1.00  0.00           \n"
"TER\n";

	const char* strand2String = 
"ATOM    636  H5    C B   1       4.448   6.843  66.315  1.00  0.00           \n"
"ATOM    637  O5*   C B   1       4.945   6.081  66.008  1.00  0.00           \n"
"ATOM    638  C5*   C B   1       5.582   5.227  66.976  1.00  0.00           \n"
"ATOM    639 1H5*   C B   1       4.855   4.809  67.672  1.00  0.00           \n"
"ATOM    640 2H5*   C B   1       6.306   5.829  67.525  1.00  0.00           \n"
"ATOM    641  C4*   C B   1       6.306   4.095  66.273  1.00  0.00           \n"
"ATOM    642  H4*   C B   1       7.038   3.607  66.916  1.00  0.00           \n"
"ATOM    643  O4*   C B   1       5.313   3.099  66.043  1.00  0.00           \n"
"ATOM    644  C1*   C B   1       5.002   2.958  64.672  1.00  0.00           \n"
"ATOM    645  H1*   C B   1       5.272   1.961  64.322  1.00  0.00           \n"
"ATOM    646  N1    C B   1       3.515   2.969  64.590  1.00  0.00           \n"
"ATOM    647  C6    C B   1       2.819   4.138  64.619  1.00  0.00           \n"
"ATOM    648  H6    C B   1       3.346   5.077  64.702  1.00  0.00           \n"
"ATOM    649  C5    C B   1       1.464   4.148  64.545  1.00  0.00           \n"
"ATOM    650  H5    C B   1       0.924   5.095  64.570  1.00  0.00           \n"
"ATOM    651  C4    C B   1       0.811   2.878  64.435  1.00  0.00           \n"
"ATOM    652  N4    C B   1      -0.509   2.813  64.358  1.00  0.00           \n"
"ATOM    653 1H4    C B   1      -0.930   1.898  64.281  1.00  0.00           \n"
"ATOM    654 2H4    C B   1      -1.062   3.658  64.376  1.00  0.00           \n"
"ATOM    655  N3    C B   1       1.506   1.733  64.407  1.00  0.00           \n"
"ATOM    656  C2    C B   1       2.861   1.744  64.483  1.00  0.00           \n"
"ATOM    657  O2    C B   1       3.536   0.708  64.461  1.00  0.00           \n"
"ATOM    658  C3*   C B   1       6.858   4.420  64.885  1.00  0.00           \n"
"ATOM    659  H3*   C B   1       7.164   5.460  64.774  1.00  0.00           \n"
"ATOM    660  C2*   C B   1       5.664   4.141  63.973  1.00  0.00           \n"
"ATOM    661 1H2*   C B   1       5.344   5.069  63.498  1.00  0.00           \n"
"ATOM    662 2H2*   C B   1       5.952   3.422  63.206  1.00  0.00           \n"
"ATOM    663  O3*   C B   1       7.933   3.562  64.527  1.00  0.00           \n"
"ATOM    664  P     A B   2       8.308   3.390  62.981  1.00  0.00           \n"
"ATOM    665  O1P   A B   2       9.773   3.228  62.840  1.00  0.00           \n"
"ATOM    666  O2P   A B   2       7.705   4.493  62.199  1.00  0.00           \n"
"ATOM    667  O5*   A B   2       7.575   2.013  62.628  1.00  0.00           \n"
"ATOM    668  C5*   A B   2       7.588   0.948  63.596  1.00  0.00           \n"
"ATOM    669 1H5*   A B   2       6.754   1.037  64.292  1.00  0.00           \n"
"ATOM    670 2H5*   A B   2       8.528   1.010  64.145  1.00  0.00           \n"
"ATOM    671  C4*   A B   2       7.509  -0.394  62.893  1.00  0.00           \n"
"ATOM    672  H4*   A B   2       7.815  -1.219  63.536  1.00  0.00           \n"
"ATOM    673  O4*   A B   2       6.120  -0.616  62.663  1.00  0.00           \n"
"ATOM    674  C1*   A B   2       5.785  -0.547  61.292  1.00  0.00           \n"
"ATOM    675  H1*   A B   2       5.425  -1.487  60.875  1.00  0.00           \n"
"ATOM    676  N9    A B   2       4.589   0.336  61.210  1.00  0.00           \n"
"ATOM    677  C8    A B   2       4.523   1.711  61.225  1.00  0.00           \n"
"ATOM    678  H8    A B   2       5.432   2.294  61.257  1.00  0.00           \n"
"ATOM    679  N7    A B   2       3.318   2.182  61.136  1.00  0.00           \n"
"ATOM    680  C5    A B   2       2.520   1.050  61.057  1.00  0.00           \n"
"ATOM    681  C6    A B   2       1.131   0.875  60.945  1.00  0.00           \n"
"ATOM    682  N6    A B   2       0.260   1.892  60.891  1.00  0.00           \n"
"ATOM    683 1H6    A B   2      -0.729   1.706  60.810  1.00  0.00           \n"
"ATOM    684 2H6    A B   2       0.593   2.845  60.931  1.00  0.00           \n"
"ATOM    685  N1    A B   2       0.669  -0.388  60.891  1.00  0.00           \n"
"ATOM    686  C2    A B   2       1.540  -1.394  60.945  1.00  0.00           \n"
"ATOM    687  H2    A B   2       1.189  -2.415  60.902  1.00  0.00           \n"
"ATOM    688  N3    A B   2       2.850  -1.354  61.049  1.00  0.00           \n"
"ATOM    689  C4    A B   2       3.287  -0.079  61.102  1.00  0.00           \n"
"ATOM    690  C3*   A B   2       8.146  -0.455  61.505  1.00  0.00           \n"
"ATOM    691  H3*   A B   2       9.005   0.207  61.395  1.00  0.00           \n"
"ATOM    692  C2*   A B   2       7.016   0.021  60.593  1.00  0.00           \n"
"ATOM    693 1H2*   A B   2       7.302   0.959  60.118  1.00  0.00           \n"
"ATOM    694 2H2*   A B   2       6.826  -0.730  59.826  1.00  0.00           \n"
"ATOM    695  O3*   A B   2       8.512  -1.781  61.147  1.00  0.00           \n"
"ATOM    696  P     T B   3       8.714  -2.140  59.601  1.00  0.00           \n"
"ATOM    697  O1P   T B   3       9.804  -3.133  59.460  1.00  0.00           \n"
"ATOM    698  O2P   T B   3       8.874  -0.894  58.819  1.00  0.00           \n"
"ATOM    699  O5*   T B   3       7.312  -2.824  59.248  1.00  0.00           \n"
"ATOM    700  C5*   T B   3       6.696  -3.693  60.216  1.00  0.00           \n"
"ATOM    701 1H5*   T B   3       6.074  -3.130  60.911  1.00  0.00           \n"
"ATOM    702 2H5*   T B   3       7.492  -4.195  60.766  1.00  0.00           \n"
"ATOM    703  C4*   T B   3       5.843  -4.732  59.513  1.00  0.00           \n"
"ATOM    704  H4*   T B   3       5.606  -5.579  60.157  1.00  0.00           \n"
"ATOM    705  O4*   T B   3       4.589  -4.096  59.283  1.00  0.00           \n"
"ATOM    706  C1*   T B   3       4.359  -3.843  57.912  1.00  0.00           \n"
"ATOM    707  H1*   T B   3       3.508  -4.393  57.511  1.00  0.00           \n"
"ATOM    708  N1    T B   3       3.910  -2.425  57.830  1.00  0.00           \n"
"ATOM    709  C6    T B   3       4.816  -1.398  57.860  1.00  0.00           \n"
"ATOM    710  H6    T B   3       5.874  -1.613  57.899  1.00  0.00           \n"
"ATOM    711  C5    T B   3       4.420  -0.117  57.787  1.00  0.00           \n"
"ATOM    712  C7    T B   3       5.404   1.015  57.818  1.00  0.00           \n"
"ATOM    713 1H7    T B   3       6.092   0.922  56.978  1.00  0.00           \n"
"ATOM    714 2H7    T B   3       4.871   1.963  57.747  1.00  0.00           \n"
"ATOM    715 3H7    T B   3       5.965   0.984  58.752  1.00  0.00           \n"
"ATOM    716  C4    T B   3       3.021   0.218  57.674  1.00  0.00           \n"
"ATOM    717  O4    T B   3       2.564   1.361  57.601  1.00  0.00           \n"
"ATOM    718  N3    T B   3       2.180  -0.876  57.651  1.00  0.00           \n"
"ATOM    719  H3    T B   3       1.112  -0.674  57.565  1.00  0.00           \n"
"ATOM    720  C2    T B   3       2.558  -2.202  57.725  1.00  0.00           \n"
"ATOM    721  O2    T B   3       1.738  -3.103  57.697  1.00  0.00           \n"
"ATOM    722  C3*   T B   3       6.323  -5.157  58.125  1.00  0.00           \n"
"ATOM    723  H3*   T B   3       7.407  -5.127  58.014  1.00  0.00           \n"
"ATOM    724  C2*   T B   3       5.688  -4.107  57.213  1.00  0.00           \n"
"ATOM    725 1H2*   T B   3       6.471  -3.516  56.738  1.00  0.00           \n"
"ATOM    726 2H2*   T B   3       5.093  -4.603  56.446  1.00  0.00           \n"
"ATOM    727  O3*   T B   3       5.839  -6.444  57.767  1.00  0.00           \n"
"ATOM    728  P     G B   4       5.792  -6.854  56.221  1.00  0.00           \n"
"ATOM    729  O1P   G B   4       6.090  -8.297  56.080  1.00  0.00           \n"
"ATOM    730  O2P   G B   4       6.654  -5.939  55.439  1.00  0.00           \n"
"ATOM    731  O5*   G B   4       4.255  -6.582  55.868  1.00  0.00           \n"
"ATOM    732  C5*   G B   4       3.246  -6.924  56.836  1.00  0.00           \n"
"ATOM    733 1H5*   G B   4       3.073  -6.103  57.532  1.00  0.00           \n"
"ATOM    734 2H5*   G B   4       3.595  -7.798  57.385  1.00  0.00           \n"
"ATOM    735  C4*   G B   4       1.946  -7.263  56.133  1.00  0.00           \n"
"ATOM    736  H4*   G B   4       1.256  -7.809  56.777  1.00  0.00           \n"
"ATOM    737  O4*   G B   4       1.305  -6.011  55.903  1.00  0.00           \n"
"ATOM    738  C1*   G B   4       1.268  -5.671  54.532  1.00  0.00           \n"
"ATOM    739  H1*   G B   4       0.237  -5.620  54.180  1.00  0.00           \n"
"ATOM    740  N9    G B   4       1.738  -4.260  54.450  1.00  0.00           \n"
"ATOM    741  C8    G B   4       3.025  -3.769  54.464  1.00  0.00           \n"
"ATOM    742  H8    G B   4       3.861  -4.448  54.545  1.00  0.00           \n"
"ATOM    743  N7    G B   4       3.098  -2.463  54.375  1.00  0.00           \n"
"ATOM    744  C5    G B   4       1.766  -2.061  54.296  1.00  0.00           \n"
"ATOM    745  C6    G B   4       1.210  -0.760  54.183  1.00  0.00           \n"
"ATOM    746  O6    G B   4       1.787   0.322  54.127  1.00  0.00           \n"
"ATOM    747  N1    G B   4      -0.191  -0.806  54.133  1.00  0.00           \n"
"ATOM    748  H1    G B   4      -0.677   0.064  54.051  1.00  0.00           \n"
"ATOM    749  C2    G B   4      -0.953  -1.956  54.186  1.00  0.00           \n"
"ATOM    750  N2    G B   4      -2.275  -1.787  54.123  1.00  0.00           \n"
"ATOM    751 1H2    G B   4      -2.773  -0.911  54.053  1.00  0.00           \n"
"ATOM    752 2H2    G B   4      -2.839  -2.623  54.170  1.00  0.00           \n"
"ATOM    753  N3    G B   4      -0.429  -3.176  54.292  1.00  0.00           \n"
"ATOM    754  C4    G B   4       0.928  -3.150  54.341  1.00  0.00           \n"
"ATOM    755  C3*   G B   4       2.084  -7.888  54.745  1.00  0.00           \n"
"ATOM    756  H3*   G B   4       2.978  -8.501  54.634  1.00  0.00           \n"
"ATOM    757  C2*   G B   4       2.188  -6.666  53.833  1.00  0.00           \n"
"ATOM    758 1H2*   G B   4       3.169  -6.649  53.358  1.00  0.00           \n"
"ATOM    759 2H2*   G B   4       1.415  -6.717  53.067  1.00  0.00           \n"
"ATOM    760  O3*   G B   4       0.936  -8.645  54.387  1.00  0.00           \n"
"ATOM    761  P     T B   5       0.657  -8.949  52.841  1.00  0.00           \n"
"ATOM    762  O1P   T B   5       0.050 -10.292  52.700  1.00  0.00           \n"
"ATOM    763  O2P   T B   5       1.892  -8.716  52.059  1.00  0.00           \n"
"ATOM    764  O5*   T B   5      -0.427  -7.826  52.488  1.00  0.00           \n"
"ATOM    765  C5*   T B   5      -1.443  -7.510  53.456  1.00  0.00           \n"
"ATOM    766 1H5*   T B   5      -1.100  -6.745  54.153  1.00  0.00           \n"
"ATOM    767 2H5*   T B   5      -1.674  -8.423  54.005  1.00  0.00           \n"
"ATOM    768  C4*   T B   5      -2.695  -7.020  52.753  1.00  0.00           \n"
"ATOM    769  H4*   T B   5      -3.574  -7.056  53.396  1.00  0.00           \n"
"ATOM    770  O4*   T B   5      -2.477  -5.630  52.523  1.00  0.00           \n"
"ATOM    771  C1*   T B   5      -2.308  -5.333  51.152  1.00  0.00           \n"
"ATOM    772  H1*   T B   5      -3.094  -4.693  50.751  1.00  0.00           \n"
"ATOM    773  N1    T B   5      -1.098  -4.468  51.070  1.00  0.00           \n"
"ATOM    774  C6    T B   5       0.158  -5.012  51.100  1.00  0.00           \n"
"ATOM    775  H6    T B   5       0.280  -6.084  51.139  1.00  0.00           \n"
"ATOM    776  C5    T B   5       1.255  -4.240  51.027  1.00  0.00           \n"
"ATOM    777  C7    T B   5       2.636  -4.826  51.058  1.00  0.00           \n"
"ATOM    778 1H7    T B   5       2.760  -5.509  50.218  1.00  0.00           \n"
"ATOM    779 2H7    T B   5       3.373  -4.026  50.987  1.00  0.00           \n"
"ATOM    780 3H7    T B   5       2.780  -5.369  51.992  1.00  0.00           \n"
"ATOM    781  C4    T B   5       1.141  -2.806  50.914  1.00  0.00           \n"
"ATOM    782  O4    T B   5       2.087  -2.018  50.841  1.00  0.00           \n"
"ATOM    783  N3    T B   5      -0.159  -2.344  50.891  1.00  0.00           \n"
"ATOM    784  H3    T B   5      -0.296  -1.266  50.805  1.00  0.00           \n"
"ATOM    785  C2    T B   5      -1.304  -3.113  50.965  1.00  0.00           \n"
"ATOM    786  O2    T B   5      -2.414  -2.612  50.937  1.00  0.00           \n"
"ATOM    787  C3*   T B   5      -2.950  -7.607  51.365  1.00  0.00           \n"
"ATOM    788  H3*   T B   5      -2.586  -8.629  51.254  1.00  0.00           \n"
"ATOM    789  C2*   T B   5      -2.148  -6.679  50.453  1.00  0.00           \n"
"ATOM    790 1H2*   T B   5      -1.344  -7.241  49.978  1.00  0.00           \n"
"ATOM    791 2H2*   T B   5      -2.804  -6.267  49.686  1.00  0.00           \n"
"ATOM    792  O3*   T B   5      -4.324  -7.545  51.007  1.00  0.00           \n"
"ATOM    793  P     C B   6      -4.728  -7.626  49.461  1.00  0.00           \n"
"ATOM    794  O1P   C B   6      -6.009  -8.356  49.320  1.00  0.00           \n"
"ATOM    795  O2P   C B   6      -3.592  -8.164  48.679  1.00  0.00           \n"
"ATOM    796  O5*   C B   6      -4.945  -6.081  49.108  1.00  0.00           \n"
"ATOM    797  C5*   C B   6      -5.582  -5.227  50.076  1.00  0.00           \n"
"ATOM    798 1H5*   C B   6      -4.855  -4.809  50.772  1.00  0.00           \n"
"ATOM    799 2H5*   C B   6      -6.306  -5.829  50.625  1.00  0.00           \n"
"ATOM    800  C4*   C B   6      -6.306  -4.095  49.373  1.00  0.00           \n"
"ATOM    801  H4*   C B   6      -7.038  -3.607  50.016  1.00  0.00           \n"
"ATOM    802  O4*   C B   6      -5.313  -3.099  49.143  1.00  0.00           \n"
"ATOM    803  C1*   C B   6      -5.002  -2.958  47.772  1.00  0.00           \n"
"ATOM    804  H1*   C B   6      -5.272  -1.961  47.422  1.00  0.00           \n"
"ATOM    805  N1    C B   6      -3.515  -2.969  47.690  1.00  0.00           \n"
"ATOM    806  C6    C B   6      -2.819  -4.138  47.719  1.00  0.00           \n"
"ATOM    807  H6    C B   6      -3.346  -5.077  47.802  1.00  0.00           \n"
"ATOM    808  C5    C B   6      -1.464  -4.148  47.645  1.00  0.00           \n"
"ATOM    809  H5    C B   6      -0.924  -5.095  47.670  1.00  0.00           \n"
"ATOM    810  C4    C B   6      -0.811  -2.878  47.535  1.00  0.00           \n"
"ATOM    811  N4    C B   6       0.509  -2.813  47.458  1.00  0.00           \n"
"ATOM    812 1H4    C B   6       0.930  -1.898  47.381  1.00  0.00           \n"
"ATOM    813 2H4    C B   6       1.062  -3.658  47.476  1.00  0.00           \n"
"ATOM    814  N3    C B   6      -1.506  -1.733  47.507  1.00  0.00           \n"
"ATOM    815  C2    C B   6      -2.861  -1.744  47.583  1.00  0.00           \n"
"ATOM    816  O2    C B   6      -3.536  -0.708  47.561  1.00  0.00           \n"
"ATOM    817  C3*   C B   6      -6.858  -4.420  47.985  1.00  0.00           \n"
"ATOM    818  H3*   C B   6      -7.164  -5.460  47.874  1.00  0.00           \n"
"ATOM    819  C2*   C B   6      -5.664  -4.141  47.073  1.00  0.00           \n"
"ATOM    820 1H2*   C B   6      -5.344  -5.069  46.598  1.00  0.00           \n"
"ATOM    821 2H2*   C B   6      -5.952  -3.422  46.306  1.00  0.00           \n"
"ATOM    822  O3*   C B   6      -7.933  -3.562  47.627  1.00  0.00           \n"
"ATOM    823  P     T B   7      -8.308  -3.390  46.081  1.00  0.00           \n"
"ATOM    824  O1P   T B   7      -9.773  -3.228  45.940  1.00  0.00           \n"
"ATOM    825  O2P   T B   7      -7.705  -4.493  45.299  1.00  0.00           \n"
"ATOM    826  O5*   T B   7      -7.575  -2.013  45.728  1.00  0.00           \n"
"ATOM    827  C5*   T B   7      -7.588  -0.948  46.696  1.00  0.00           \n"
"ATOM    828 1H5*   T B   7      -6.754  -1.037  47.392  1.00  0.00           \n"
"ATOM    829 2H5*   T B   7      -8.528  -1.010  47.245  1.00  0.00           \n"
"ATOM    830  C4*   T B   7      -7.509   0.394  45.993  1.00  0.00           \n"
"ATOM    831  H4*   T B   7      -7.815   1.219  46.636  1.00  0.00           \n"
"ATOM    832  O4*   T B   7      -6.120   0.616  45.763  1.00  0.00           \n"
"ATOM    833  C1*   T B   7      -5.785   0.547  44.392  1.00  0.00           \n"
"ATOM    834  H1*   T B   7      -5.420   1.493  43.991  1.00  0.00           \n"
"ATOM    835  N1    T B   7      -4.589  -0.336  44.310  1.00  0.00           \n"
"ATOM    836  C6    T B   7      -4.718  -1.700  44.340  1.00  0.00           \n"
"ATOM    837  H6    T B   7      -5.700  -2.147  44.379  1.00  0.00           \n"
"ATOM    838  C5    T B   7      -3.645  -2.504  44.267  1.00  0.00           \n"
"ATOM    839  C7    T B   7      -3.776  -3.998  44.298  1.00  0.00           \n"
"ATOM    840 1H7    T B   7      -4.388  -4.327  43.458  1.00  0.00           \n"
"ATOM    841 2H7    T B   7      -2.787  -4.451  44.227  1.00  0.00           \n"
"ATOM    842 3H7    T B   7      -4.248  -4.303  45.232  1.00  0.00           \n"
"ATOM    843  C4    T B   7      -2.316  -1.952  44.154  1.00  0.00           \n"
"ATOM    844  O4    T B   7      -1.274  -2.608  44.081  1.00  0.00           \n"
"ATOM    845  N3    T B   7      -2.278  -0.573  44.131  1.00  0.00           \n"
"ATOM    846  H3    T B   7      -1.295  -0.109  44.045  1.00  0.00           \n"
"ATOM    847  C2    T B   7      -3.364   0.278  44.205  1.00  0.00           \n"
"ATOM    848  O2    T B   7      -3.230   1.489  44.177  1.00  0.00           \n"
"ATOM    849  C3*   T B   7      -8.146   0.455  44.605  1.00  0.00           \n"
"ATOM    850  H3*   T B   7      -9.005  -0.207  44.495  1.00  0.00           \n"
"ATOM    851  C2*   T B   7      -7.016  -0.021  43.693  1.00  0.00           \n"
"ATOM    852 1H2*   T B   7      -7.302  -0.959  43.218  1.00  0.00           \n"
"ATOM    853 2H2*   T B   7      -6.826   0.730  42.926  1.00  0.00           \n"
"ATOM    854  O3*   T B   7      -8.512   1.781  44.247  1.00  0.00           \n"
"ATOM    855  P     A B   8      -8.714   2.140  42.701  1.00  0.00           \n"
"ATOM    856  O1P   A B   8      -9.804   3.133  42.560  1.00  0.00           \n"
"ATOM    857  O2P   A B   8      -8.874   0.894  41.919  1.00  0.00           \n"
"ATOM    858  O5*   A B   8      -7.312   2.824  42.348  1.00  0.00           \n"
"ATOM    859  C5*   A B   8      -6.696   3.693  43.316  1.00  0.00           \n"
"ATOM    860 1H5*   A B   8      -6.074   3.130  44.011  1.00  0.00           \n"
"ATOM    861 2H5*   A B   8      -7.492   4.195  43.866  1.00  0.00           \n"
"ATOM    862  C4*   A B   8      -5.843   4.732  42.613  1.00  0.00           \n"
"ATOM    863  H4*   A B   8      -5.606   5.579  43.257  1.00  0.00           \n"
"ATOM    864  O4*   A B   8      -4.589   4.096  42.383  1.00  0.00           \n"
"ATOM    865  C1*   A B   8      -4.359   3.843  41.012  1.00  0.00           \n"
"ATOM    866  H1*   A B   8      -3.515   4.392  40.594  1.00  0.00           \n"
"ATOM    867  N9    A B   8      -3.910   2.425  40.930  1.00  0.00           \n"
"ATOM    868  C8    A B   8      -4.665   1.274  40.945  1.00  0.00           \n"
"ATOM    869  H8    A B   8      -5.743   1.336  40.977  1.00  0.00           \n"
"ATOM    870  N7    A B   8      -3.967   0.185  40.856  1.00  0.00           \n"
"ATOM    871  C5    A B   8      -2.656   0.632  40.777  1.00  0.00           \n"
"ATOM    872  C6    A B   8      -1.429  -0.043  40.665  1.00  0.00           \n"
"ATOM    873  N6    A B   8      -1.322  -1.378  40.611  1.00  0.00           \n"
"ATOM    874 1H6    A B   8      -0.412  -1.809  40.530  1.00  0.00           \n"
"ATOM    875 2H6    A B   8      -2.151  -1.953  40.651  1.00  0.00           \n"
"ATOM    876  N1    A B   8      -0.313   0.707  40.611  1.00  0.00           \n"
"ATOM    877  C2    A B   8      -0.426   2.033  40.665  1.00  0.00           \n"
"ATOM    878  H2    A B   8       0.457   2.653  40.622  1.00  0.00           \n"
"ATOM    879  N3    A B   8      -1.510   2.770  40.769  1.00  0.00           \n"
"ATOM    880  C4    A B   8      -2.613   1.996  40.822  1.00  0.00           \n"
"ATOM    881  C3*   A B   8      -6.323   5.157  41.225  1.00  0.00           \n"
"ATOM    882  H3*   A B   8      -7.407   5.127  41.114  1.00  0.00           \n"
"ATOM    883  C2*   A B   8      -5.688   4.107  40.313  1.00  0.00           \n"
"ATOM    884 1H2*   A B   8      -6.471   3.516  39.838  1.00  0.00           \n"
"ATOM    885 2H2*   A B   8      -5.093   4.603  39.546  1.00  0.00           \n"
"ATOM    886  O3*   A B   8      -5.839   6.444  40.867  1.00  0.00           \n"
"ATOM    887  P     T B   9      -5.792   6.854  39.321  1.00  0.00           \n"
"ATOM    888  O1P   T B   9      -6.090   8.297  39.180  1.00  0.00           \n"
"ATOM    889  O2P   T B   9      -6.654   5.939  38.539  1.00  0.00           \n"
"ATOM    890  O5*   T B   9      -4.255   6.582  38.968  1.00  0.00           \n"
"ATOM    891  C5*   T B   9      -3.246   6.924  39.936  1.00  0.00           \n"
"ATOM    892 1H5*   T B   9      -3.073   6.103  40.632  1.00  0.00           \n"
"ATOM    893 2H5*   T B   9      -3.595   7.798  40.485  1.00  0.00           \n"
"ATOM    894  C4*   T B   9      -1.946   7.263  39.233  1.00  0.00           \n"
"ATOM    895  H4*   T B   9      -1.256   7.809  39.877  1.00  0.00           \n"
"ATOM    896  O4*   T B   9      -1.305   6.011  39.003  1.00  0.00           \n"
"ATOM    897  C1*   T B   9      -1.268   5.671  37.632  1.00  0.00           \n"
"ATOM    898  H1*   T B   9      -0.256   5.616  37.231  1.00  0.00           \n"
"ATOM    899  N1    T B   9      -1.738   4.260  37.550  1.00  0.00           \n"
"ATOM    900  C6    T B   9      -3.074   3.962  37.580  1.00  0.00           \n"
"ATOM    901  H6    T B   9      -3.803   4.758  37.619  1.00  0.00           \n"
"ATOM    902  C5    T B   9      -3.508   2.693  37.507  1.00  0.00           \n"
"ATOM    903  C7    T B   9      -4.969   2.355  37.538  1.00  0.00           \n"
"ATOM    904 1H7    T B   9      -5.471   2.835  36.698  1.00  0.00           \n"
"ATOM    905 2H7    T B   9      -5.094   1.275  37.467  1.00  0.00           \n"
"ATOM    906 3H7    T B   9      -5.405   2.710  38.472  1.00  0.00           \n"
"ATOM    907  C4    T B   9      -2.572   1.599  37.394  1.00  0.00           \n"
"ATOM    908  O4    T B   9      -2.874   0.406  37.321  1.00  0.00           \n"
"ATOM    909  N3    T B   9      -1.249   1.990  37.371  1.00  0.00           \n"
"ATOM    910  H3    T B   9      -0.504   1.199  37.285  1.00  0.00           \n"
"ATOM    911  C2    T B   9      -0.775   3.285  37.445  1.00  0.00           \n"
"ATOM    912  O2    T B   9       0.418   3.532  37.417  1.00  0.00           \n"
"ATOM    913  C3*   T B   9      -2.084   7.888  37.845  1.00  0.00           \n"
"ATOM    914  H3*   T B   9      -2.978   8.501  37.734  1.00  0.00           \n"
"ATOM    915  C2*   T B   9      -2.188   6.666  36.933  1.00  0.00           \n"
"ATOM    916 1H2*   T B   9      -3.169   6.649  36.458  1.00  0.00           \n"
"ATOM    917 2H2*   T B   9      -1.415   6.717  36.167  1.00  0.00           \n"
"ATOM    918  O3*   T B   9      -0.936   8.645  37.487  1.00  0.00           \n"
"ATOM    919  P     C B  10      -0.657   8.949  35.941  1.00  0.00           \n"
"ATOM    920  O1P   C B  10      -0.050  10.292  35.800  1.00  0.00           \n"
"ATOM    921  O2P   C B  10      -1.892   8.716  35.159  1.00  0.00           \n"
"ATOM    922  O5*   C B  10       0.427   7.826  35.588  1.00  0.00           \n"
"ATOM    923  C5*   C B  10       1.443   7.510  36.556  1.00  0.00           \n"
"ATOM    924 1H5*   C B  10       1.100   6.745  37.253  1.00  0.00           \n"
"ATOM    925 2H5*   C B  10       1.674   8.423  37.105  1.00  0.00           \n"
"ATOM    926  C4*   C B  10       2.695   7.020  35.853  1.00  0.00           \n"
"ATOM    927  H4*   C B  10       3.574   7.056  36.496  1.00  0.00           \n"
"ATOM    928  O4*   C B  10       2.477   5.630  35.623  1.00  0.00           \n"
"ATOM    929  C1*   C B  10       2.308   5.333  34.252  1.00  0.00           \n"
"ATOM    930  H1*   C B  10       3.112   4.685  33.902  1.00  0.00           \n"
"ATOM    931  N1    C B  10       1.098   4.468  34.170  1.00  0.00           \n"
"ATOM    932  C6    C B  10      -0.151   5.005  34.199  1.00  0.00           \n"
"ATOM    933  H6    C B  10      -0.276   6.074  34.282  1.00  0.00           \n"
"ATOM    934  C5    C B  10      -1.254   4.217  34.125  1.00  0.00           \n"
"ATOM    935  H5    C B  10      -2.247   4.665  34.150  1.00  0.00           \n"
"ATOM    936  C4    C B  10      -1.035   2.805  34.015  1.00  0.00           \n"
"ATOM    937  N4    C B  10      -2.066   1.977  33.938  1.00  0.00           \n"
"ATOM    938 1H4    C B  10      -1.870   0.989  33.862  1.00  0.00           \n"
"ATOM    939 2H4    C B  10      -3.010   2.336  33.956  1.00  0.00           \n"
"ATOM    940  N3    C B  10       0.199   2.287  33.987  1.00  0.00           \n"
"ATOM    941  C2    C B  10       1.289   3.093  34.063  1.00  0.00           \n"
"ATOM    942  O2    C B  10       2.445   2.651  34.041  1.00  0.00           \n"
"ATOM    943  C3*   C B  10       2.950   7.607  34.465  1.00  0.00           \n"
"ATOM    944  H3*   C B  10       2.586   8.629  34.354  1.00  0.00           \n"
"ATOM    945  C2*   C B  10       2.148   6.679  33.553  1.00  0.00           \n"
"ATOM    946 1H2*   C B  10       1.344   7.241  33.078  1.00  0.00           \n"
"ATOM    947 2H2*   C B  10       2.804   6.267  32.786  1.00  0.00           \n"
"ATOM    948  O3*   C B  10       4.324   7.545  34.107  1.00  0.00           \n"
"ATOM    949  P     G B  11       4.728   7.626  32.561  1.00  0.00           \n"
"ATOM    950  O1P   G B  11       6.009   8.356  32.420  1.00  0.00           \n"
"ATOM    951  O2P   G B  11       3.592   8.164  31.779  1.00  0.00           \n"
"ATOM    952  O5*   G B  11       4.945   6.081  32.208  1.00  0.00           \n"
"ATOM    953  C5*   G B  11       5.582   5.227  33.176  1.00  0.00           \n"
"ATOM    954 1H5*   G B  11       4.855   4.809  33.872  1.00  0.00           \n"
"ATOM    955 2H5*   G B  11       6.306   5.829  33.725  1.00  0.00           \n"
"ATOM    956  C4*   G B  11       6.306   4.095  32.473  1.00  0.00           \n"
"ATOM    957  H4*   G B  11       7.038   3.607  33.116  1.00  0.00           \n"
"ATOM    958  O4*   G B  11       5.313   3.099  32.243  1.00  0.00           \n"
"ATOM    959  C1*   G B  11       5.002   2.958  30.872  1.00  0.00           \n"
"ATOM    960  H1*   G B  11       5.272   1.962  30.521  1.00  0.00           \n"
"ATOM    961  N9    G B  11       3.515   2.969  30.790  1.00  0.00           \n"
"ATOM    962  C8    G B  11       2.650   4.042  30.804  1.00  0.00           \n"
"ATOM    963  H8    G B  11       3.037   5.047  30.885  1.00  0.00           \n"
"ATOM    964  N7    G B  11       1.385   3.708  30.715  1.00  0.00           \n"
"ATOM    965  C5    G B  11       1.414   2.317  30.636  1.00  0.00           \n"
"ATOM    966  C6    G B  11       0.349   1.386  30.523  1.00  0.00           \n"
"ATOM    967  O6    G B  11      -0.858   1.600  30.467  1.00  0.00           \n"
"ATOM    968  N1    G B  11       0.825   0.068  30.473  1.00  0.00           \n"
"ATOM    969  H1    G B  11       0.148  -0.663  30.392  1.00  0.00           \n"
"ATOM    970  C2    G B  11       2.155  -0.301  30.526  1.00  0.00           \n"
"ATOM    971  N2    G B  11       2.402  -1.612  30.463  1.00  0.00           \n"
"ATOM    972 1H2    G B  11       1.722  -2.356  30.393  1.00  0.00           \n"
"ATOM    973 2H2    G B  11       3.372  -1.891  30.510  1.00  0.00           \n"
"ATOM    974  N3    G B  11       3.153   0.573  30.632  1.00  0.00           \n"
"ATOM    975  C4    G B  11       2.709   1.856  30.681  1.00  0.00           \n"
"ATOM    976  C3*   G B  11       6.858   4.420  31.085  1.00  0.00           \n"
"ATOM    977  H3*   G B  11       7.164   5.460  30.974  1.00  0.00           \n"
"ATOM    978  C2*   G B  11       5.664   4.141  30.173  1.00  0.00           \n"
"ATOM    979 1H2*   G B  11       5.344   5.069  29.698  1.00  0.00           \n"
"ATOM    980 2H2*   G B  11       5.952   3.422  29.406  1.00  0.00           \n"
"ATOM    981  O3*   G B  11       7.933   3.562  30.727  1.00  0.00           \n"
"ATOM    982  P     T B  12       8.308   3.390  29.181  1.00  0.00           \n"
"ATOM    983  O1P   T B  12       9.773   3.228  29.040  1.00  0.00           \n"
"ATOM    984  O2P   T B  12       7.705   4.493  28.399  1.00  0.00           \n"
"ATOM    985  O5*   T B  12       7.575   2.013  28.828  1.00  0.00           \n"
"ATOM    986  C5*   T B  12       7.588   0.948  29.796  1.00  0.00           \n"
"ATOM    987 1H5*   T B  12       6.754   1.037  30.492  1.00  0.00           \n"
"ATOM    988 2H5*   T B  12       8.528   1.010  30.345  1.00  0.00           \n"
"ATOM    989  C4*   T B  12       7.509  -0.394  29.093  1.00  0.00           \n"
"ATOM    990  H4*   T B  12       7.815  -1.219  29.736  1.00  0.00           \n"
"ATOM    991  O4*   T B  12       6.120  -0.616  28.863  1.00  0.00           \n"
"ATOM    992  C1*   T B  12       5.785  -0.547  27.492  1.00  0.00           \n"
"ATOM    993  H1*   T B  12       5.420  -1.493  27.091  1.00  0.00           \n"
"ATOM    994  N1    T B  12       4.589   0.336  27.410  1.00  0.00           \n"
"ATOM    995  C6    T B  12       4.718   1.700  27.440  1.00  0.00           \n"
"ATOM    996  H6    T B  12       5.700   2.147  27.479  1.00  0.00           \n"
"ATOM    997  C5    T B  12       3.645   2.504  27.367  1.00  0.00           \n"
"ATOM    998  C7    T B  12       3.776   3.998  27.398  1.00  0.00           \n"
"ATOM    999 1H7    T B  12       4.388   4.327  26.558  1.00  0.00           \n"
"ATOM   1000 2H7    T B  12       2.787   4.451  27.327  1.00  0.00           \n"
"ATOM   1001 3H7    T B  12       4.248   4.303  28.332  1.00  0.00           \n"
"ATOM   1002  C4    T B  12       2.316   1.952  27.254  1.00  0.00           \n"
"ATOM   1003  O4    T B  12       1.274   2.608  27.181  1.00  0.00           \n"
"ATOM   1004  N3    T B  12       2.278   0.573  27.231  1.00  0.00           \n"
"ATOM   1005  H3    T B  12       1.295   0.109  27.145  1.00  0.00           \n"
"ATOM   1006  C2    T B  12       3.364  -0.278  27.305  1.00  0.00           \n"
"ATOM   1007  O2    T B  12       3.230  -1.489  27.277  1.00  0.00           \n"
"ATOM   1008  C3*   T B  12       8.146  -0.455  27.705  1.00  0.00           \n"
"ATOM   1009  H3*   T B  12       9.005   0.207  27.595  1.00  0.00           \n"
"ATOM   1010  C2*   T B  12       7.016   0.021  26.793  1.00  0.00           \n"
"ATOM   1011 1H2*   T B  12       7.302   0.959  26.318  1.00  0.00           \n"
"ATOM   1012 2H2*   T B  12       6.826  -0.730  26.026  1.00  0.00           \n"
"ATOM   1013  O3*   T B  12       8.512  -1.781  27.347  1.00  0.00           \n"
"ATOM   1014  P     A B  13       8.714  -2.140  25.801  1.00  0.00           \n"
"ATOM   1015  O1P   A B  13       9.804  -3.133  25.660  1.00  0.00           \n"
"ATOM   1016  O2P   A B  13       8.874  -0.894  25.019  1.00  0.00           \n"
"ATOM   1017  O5*   A B  13       7.312  -2.824  25.448  1.00  0.00           \n"
"ATOM   1018  C5*   A B  13       6.696  -3.693  26.416  1.00  0.00           \n"
"ATOM   1019 1H5*   A B  13       6.074  -3.130  27.111  1.00  0.00           \n"
"ATOM   1020 2H5*   A B  13       7.492  -4.195  26.966  1.00  0.00           \n"
"ATOM   1021  C4*   A B  13       5.843  -4.732  25.713  1.00  0.00           \n"
"ATOM   1022  H4*   A B  13       5.606  -5.579  26.357  1.00  0.00           \n"
"ATOM   1023  O4*   A B  13       4.589  -4.096  25.483  1.00  0.00           \n"
"ATOM   1024  C1*   A B  13       4.359  -3.843  24.112  1.00  0.00           \n"
"ATOM   1025  H1*   A B  13       3.515  -4.392  23.694  1.00  0.00           \n"
"ATOM   1026  N9    A B  13       3.910  -2.425  24.030  1.00  0.00           \n"
"ATOM   1027  C8    A B  13       4.665  -1.274  24.045  1.00  0.00           \n"
"ATOM   1028  H8    A B  13       5.743  -1.336  24.077  1.00  0.00           \n"
"ATOM   1029  N7    A B  13       3.967  -0.185  23.956  1.00  0.00           \n"
"ATOM   1030  C5    A B  13       2.656  -0.632  23.877  1.00  0.00           \n"
"ATOM   1031  C6    A B  13       1.429   0.043  23.765  1.00  0.00           \n"
"ATOM   1032  N6    A B  13       1.322   1.378  23.711  1.00  0.00           \n"
"ATOM   1033 1H6    A B  13       0.412   1.809  23.630  1.00  0.00           \n"
"ATOM   1034 2H6    A B  13       2.151   1.953  23.751  1.00  0.00           \n"
"ATOM   1035  N1    A B  13       0.313  -0.707  23.711  1.00  0.00           \n"
"ATOM   1036  C2    A B  13       0.426  -2.033  23.765  1.00  0.00           \n"
"ATOM   1037  H2    A B  13      -0.457  -2.653  23.722  1.00  0.00           \n"
"ATOM   1038  N3    A B  13       1.510  -2.770  23.869  1.00  0.00           \n"
"ATOM   1039  C4    A B  13       2.613  -1.996  23.922  1.00  0.00           \n"
"ATOM   1040  C3*   A B  13       6.323  -5.157  24.325  1.00  0.00           \n"
"ATOM   1041  H3*   A B  13       7.407  -5.127  24.214  1.00  0.00           \n"
"ATOM   1042  C2*   A B  13       5.688  -4.107  23.413  1.00  0.00           \n"
"ATOM   1043 1H2*   A B  13       6.471  -3.516  22.938  1.00  0.00           \n"
"ATOM   1044 2H2*   A B  13       5.093  -4.603  22.646  1.00  0.00           \n"
"ATOM   1045  O3*   A B  13       5.839  -6.444  23.967  1.00  0.00           \n"
"ATOM   1046  P     T B  14       5.792  -6.854  22.421  1.00  0.00           \n"
"ATOM   1047  O1P   T B  14       6.090  -8.297  22.280  1.00  0.00           \n"
"ATOM   1048  O2P   T B  14       6.654  -5.939  21.639  1.00  0.00           \n"
"ATOM   1049  O5*   T B  14       4.255  -6.582  22.068  1.00  0.00           \n"
"ATOM   1050  C5*   T B  14       3.246  -6.924  23.036  1.00  0.00           \n"
"ATOM   1051 1H5*   T B  14       3.073  -6.103  23.732  1.00  0.00           \n"
"ATOM   1052 2H5*   T B  14       3.595  -7.798  23.585  1.00  0.00           \n"
"ATOM   1053  C4*   T B  14       1.946  -7.263  22.333  1.00  0.00           \n"
"ATOM   1054  H4*   T B  14       1.256  -7.809  22.977  1.00  0.00           \n"
"ATOM   1055  O4*   T B  14       1.305  -6.011  22.103  1.00  0.00           \n"
"ATOM   1056  C1*   T B  14       1.268  -5.671  20.732  1.00  0.00           \n"
"ATOM   1057  H1*   T B  14       0.256  -5.616  20.331  1.00  0.00           \n"
"ATOM   1058  N1    T B  14       1.738  -4.260  20.650  1.00  0.00           \n"
"ATOM   1059  C6    T B  14       3.074  -3.962  20.680  1.00  0.00           \n"
"ATOM   1060  H6    T B  14       3.803  -4.758  20.719  1.00  0.00           \n"
"ATOM   1061  C5    T B  14       3.508  -2.693  20.607  1.00  0.00           \n"
"ATOM   1062  C7    T B  14       4.969  -2.355  20.638  1.00  0.00           \n"
"ATOM   1063 1H7    T B  14       5.471  -2.835  19.798  1.00  0.00           \n"
"ATOM   1064 2H7    T B  14       5.094  -1.275  20.567  1.00  0.00           \n"
"ATOM   1065 3H7    T B  14       5.405  -2.710  21.572  1.00  0.00           \n"
"ATOM   1066  C4    T B  14       2.572  -1.599  20.494  1.00  0.00           \n"
"ATOM   1067  O4    T B  14       2.874  -0.406  20.421  1.00  0.00           \n"
"ATOM   1068  N3    T B  14       1.249  -1.990  20.471  1.00  0.00           \n"
"ATOM   1069  H3    T B  14       0.504  -1.199  20.385  1.00  0.00           \n"
"ATOM   1070  C2    T B  14       0.775  -3.285  20.545  1.00  0.00           \n"
"ATOM   1071  O2    T B  14      -0.418  -3.532  20.517  1.00  0.00           \n"
"ATOM   1072  C3*   T B  14       2.084  -7.888  20.945  1.00  0.00           \n"
"ATOM   1073  H3*   T B  14       2.978  -8.501  20.834  1.00  0.00           \n"
"ATOM   1074  C2*   T B  14       2.188  -6.666  20.033  1.00  0.00           \n"
"ATOM   1075 1H2*   T B  14       3.169  -6.649  19.558  1.00  0.00           \n"
"ATOM   1076 2H2*   T B  14       1.415  -6.717  19.267  1.00  0.00           \n"
"ATOM   1077  O3*   T B  14       0.936  -8.645  20.587  1.00  0.00           \n"
"ATOM   1078  P     C B  15       0.657  -8.949  19.041  1.00  0.00           \n"
"ATOM   1079  O1P   C B  15       0.050 -10.292  18.900  1.00  0.00           \n"
"ATOM   1080  O2P   C B  15       1.892  -8.716  18.259  1.00  0.00           \n"
"ATOM   1081  O5*   C B  15      -0.427  -7.826  18.688  1.00  0.00           \n"
"ATOM   1082  C5*   C B  15      -1.443  -7.510  19.656  1.00  0.00           \n"
"ATOM   1083 1H5*   C B  15      -1.100  -6.745  20.353  1.00  0.00           \n"
"ATOM   1084 2H5*   C B  15      -1.674  -8.423  20.205  1.00  0.00           \n"
"ATOM   1085  C4*   C B  15      -2.695  -7.020  18.953  1.00  0.00           \n"
"ATOM   1086  H4*   C B  15      -3.574  -7.056  19.596  1.00  0.00           \n"
"ATOM   1087  O4*   C B  15      -2.477  -5.630  18.723  1.00  0.00           \n"
"ATOM   1088  C1*   C B  15      -2.308  -5.333  17.352  1.00  0.00           \n"
"ATOM   1089  H1*   C B  15      -3.112  -4.685  17.002  1.00  0.00           \n"
"ATOM   1090  N1    C B  15      -1.098  -4.468  17.270  1.00  0.00           \n"
"ATOM   1091  C6    C B  15       0.151  -5.005  17.299  1.00  0.00           \n"
"ATOM   1092  H6    C B  15       0.276  -6.074  17.382  1.00  0.00           \n"
"ATOM   1093  C5    C B  15       1.254  -4.217  17.225  1.00  0.00           \n"
"ATOM   1094  H5    C B  15       2.247  -4.665  17.250  1.00  0.00           \n"
"ATOM   1095  C4    C B  15       1.035  -2.805  17.115  1.00  0.00           \n"
"ATOM   1096  N4    C B  15       2.066  -1.977  17.038  1.00  0.00           \n"
"ATOM   1097 1H4    C B  15       1.870  -0.989  16.962  1.00  0.00           \n"
"ATOM   1098 2H4    C B  15       3.010  -2.336  17.056  1.00  0.00           \n"
"ATOM   1099  N3    C B  15      -0.199  -2.287  17.087  1.00  0.00           \n"
"ATOM   1100  C2    C B  15      -1.289  -3.093  17.163  1.00  0.00           \n"
"ATOM   1101  O2    C B  15      -2.445  -2.651  17.141  1.00  0.00           \n"
"ATOM   1102  C3*   C B  15      -2.950  -7.607  17.565  1.00  0.00           \n"
"ATOM   1103  H3*   C B  15      -2.586  -8.629  17.454  1.00  0.00           \n"
"ATOM   1104  C2*   C B  15      -2.148  -6.679  16.653  1.00  0.00           \n"
"ATOM   1105 1H2*   C B  15      -1.344  -7.241  16.178  1.00  0.00           \n"
"ATOM   1106 2H2*   C B  15      -2.804  -6.267  15.886  1.00  0.00           \n"
"ATOM   1107  O3*   C B  15      -4.324  -7.545  17.207  1.00  0.00           \n"
"ATOM   1108  P     G B  16      -4.728  -7.626  15.661  1.00  0.00           \n"
"ATOM   1109  O1P   G B  16      -6.009  -8.356  15.520  1.00  0.00           \n"
"ATOM   1110  O2P   G B  16      -3.592  -8.164  14.879  1.00  0.00           \n"
"ATOM   1111  O5*   G B  16      -4.945  -6.081  15.308  1.00  0.00           \n"
"ATOM   1112  C5*   G B  16      -5.582  -5.227  16.276  1.00  0.00           \n"
"ATOM   1113 1H5*   G B  16      -4.855  -4.809  16.972  1.00  0.00           \n"
"ATOM   1114 2H5*   G B  16      -6.306  -5.829  16.825  1.00  0.00           \n"
"ATOM   1115  C4*   G B  16      -6.306  -4.095  15.573  1.00  0.00           \n"
"ATOM   1116  H4*   G B  16      -7.038  -3.607  16.216  1.00  0.00           \n"
"ATOM   1117  O4*   G B  16      -5.313  -3.099  15.343  1.00  0.00           \n"
"ATOM   1118  C1*   G B  16      -5.002  -2.958  13.972  1.00  0.00           \n"
"ATOM   1119  H1*   G B  16      -5.272  -1.962  13.621  1.00  0.00           \n"
"ATOM   1120  N9    G B  16      -3.515  -2.969  13.890  1.00  0.00           \n"
"ATOM   1121  C8    G B  16      -2.650  -4.042  13.904  1.00  0.00           \n"
"ATOM   1122  H8    G B  16      -3.037  -5.047  13.985  1.00  0.00           \n"
"ATOM   1123  N7    G B  16      -1.385  -3.708  13.815  1.00  0.00           \n"
"ATOM   1124  C5    G B  16      -1.414  -2.317  13.736  1.00  0.00           \n"
"ATOM   1125  C6    G B  16      -0.349  -1.386  13.623  1.00  0.00           \n"
"ATOM   1126  O6    G B  16       0.858  -1.600  13.567  1.00  0.00           \n"
"ATOM   1127  N1    G B  16      -0.825  -0.068  13.573  1.00  0.00           \n"
"ATOM   1128  H1    G B  16      -0.148   0.663  13.492  1.00  0.00           \n"
"ATOM   1129  C2    G B  16      -2.155   0.301  13.626  1.00  0.00           \n"
"ATOM   1130  N2    G B  16      -2.402   1.612  13.563  1.00  0.00           \n"
"ATOM   1131 1H2    G B  16      -1.722   2.356  13.493  1.00  0.00           \n"
"ATOM   1132 2H2    G B  16      -3.372   1.891  13.610  1.00  0.00           \n"
"ATOM   1133  N3    G B  16      -3.153  -0.573  13.732  1.00  0.00           \n"
"ATOM   1134  C4    G B  16      -2.709  -1.856  13.781  1.00  0.00           \n"
"ATOM   1135  C3*   G B  16      -6.858  -4.420  14.185  1.00  0.00           \n"
"ATOM   1136  H3*   G B  16      -7.164  -5.460  14.074  1.00  0.00           \n"
"ATOM   1137  C2*   G B  16      -5.664  -4.141  13.273  1.00  0.00           \n"
"ATOM   1138 1H2*   G B  16      -5.344  -5.069  12.798  1.00  0.00           \n"
"ATOM   1139 2H2*   G B  16      -5.952  -3.422  12.506  1.00  0.00           \n"
"ATOM   1140  O3*   G B  16      -7.933  -3.562  13.827  1.00  0.00           \n"
"ATOM   1141  P     G B  17      -8.308  -3.390  12.281  1.00  0.00           \n"
"ATOM   1142  O1P   G B  17      -9.773  -3.228  12.140  1.00  0.00           \n"
"ATOM   1143  O2P   G B  17      -7.705  -4.493  11.499  1.00  0.00           \n"
"ATOM   1144  O5*   G B  17      -7.575  -2.013  11.928  1.00  0.00           \n"
"ATOM   1145  C5*   G B  17      -7.588  -0.948  12.896  1.00  0.00           \n"
"ATOM   1146 1H5*   G B  17      -6.754  -1.037  13.592  1.00  0.00           \n"
"ATOM   1147 2H5*   G B  17      -8.528  -1.010  13.445  1.00  0.00           \n"
"ATOM   1148  C4*   G B  17      -7.509   0.394  12.193  1.00  0.00           \n"
"ATOM   1149  H4*   G B  17      -7.815   1.219  12.836  1.00  0.00           \n"
"ATOM   1150  O4*   G B  17      -6.120   0.616  11.963  1.00  0.00           \n"
"ATOM   1151  C1*   G B  17      -5.785   0.547  10.592  1.00  0.00           \n"
"ATOM   1152  H1*   G B  17      -5.418   1.511  10.241  1.00  0.00           \n"
"ATOM   1153  N9    G B  17      -4.589  -0.336  10.510  1.00  0.00           \n"
"ATOM   1154  C8    G B  17      -4.520  -1.712  10.524  1.00  0.00           \n"
"ATOM   1155  H8    G B  17      -5.424  -2.297  10.605  1.00  0.00           \n"
"ATOM   1156  N7    G B  17      -3.300  -2.186  10.435  1.00  0.00           \n"
"ATOM   1157  C5    G B  17      -2.506  -1.043  10.356  1.00  0.00           \n"
"ATOM   1158  C6    G B  17      -1.097  -0.916  10.243  1.00  0.00           \n"
"ATOM   1159  O6    G B  17      -0.246  -1.799  10.187  1.00  0.00           \n"
"ATOM   1160  N1    G B  17      -0.707   0.430  10.193  1.00  0.00           \n"
"ATOM   1161  H1    G B  17       0.271   0.623  10.111  1.00  0.00           \n"
"ATOM   1162  C2    G B  17      -1.566   1.510  10.246  1.00  0.00           \n"
"ATOM   1163  N2    G B  17      -0.996   2.716  10.183  1.00  0.00           \n"
"ATOM   1164 1H2    G B  17      -0.009   2.919  10.113  1.00  0.00           \n"
"ATOM   1165 2H2    G B  17      -1.617   3.511  10.230  1.00  0.00           \n"
"ATOM   1166  N3    G B  17      -2.888   1.390  10.352  1.00  0.00           \n"
"ATOM   1167  C4    G B  17      -3.283   0.091  10.401  1.00  0.00           \n"
"ATOM   1168  C3*   G B  17      -8.146   0.455  10.805  1.00  0.00           \n"
"ATOM   1169  H3*   G B  17      -9.005  -0.207  10.695  1.00  0.00           \n"
"ATOM   1170  C2*   G B  17      -7.016  -0.021   9.893  1.00  0.00           \n"
"ATOM   1171 1H2*   G B  17      -7.302  -0.959   9.418  1.00  0.00           \n"
"ATOM   1172 2H2*   G B  17      -6.826   0.730   9.126  1.00  0.00           \n"
"ATOM   1173  O3*   G B  17      -8.512   1.781  10.447  1.00  0.00           \n"
"ATOM   1174  P     T B  18      -8.714   2.140   8.901  1.00  0.00           \n"
"ATOM   1175  O1P   T B  18      -9.804   3.133   8.760  1.00  0.00           \n"
"ATOM   1176  O2P   T B  18      -8.874   0.894   8.119  1.00  0.00           \n"
"ATOM   1177  O5*   T B  18      -7.312   2.824   8.548  1.00  0.00           \n"
"ATOM   1178  C5*   T B  18      -6.696   3.693   9.516  1.00  0.00           \n"
"ATOM   1179 1H5*   T B  18      -6.074   3.130  10.211  1.00  0.00           \n"
"ATOM   1180 2H5*   T B  18      -7.492   4.195  10.066  1.00  0.00           \n"
"ATOM   1181  C4*   T B  18      -5.843   4.732   8.813  1.00  0.00           \n"
"ATOM   1182  H4*   T B  18      -5.606   5.579   9.457  1.00  0.00           \n"
"ATOM   1183  O4*   T B  18      -4.589   4.096   8.583  1.00  0.00           \n"
"ATOM   1184  C1*   T B  18      -4.359   3.843   7.212  1.00  0.00           \n"
"ATOM   1185  H1*   T B  18      -3.508   4.393   6.811  1.00  0.00           \n"
"ATOM   1186  N1    T B  18      -3.910   2.425   7.130  1.00  0.00           \n"
"ATOM   1187  C6    T B  18      -4.816   1.398   7.160  1.00  0.00           \n"
"ATOM   1188  H6    T B  18      -5.874   1.613   7.199  1.00  0.00           \n"
"ATOM   1189  C5    T B  18      -4.420   0.117   7.087  1.00  0.00           \n"
"ATOM   1190  C7    T B  18      -5.404  -1.015   7.118  1.00  0.00           \n"
"ATOM   1191 1H7    T B  18      -6.092  -0.922   6.278  1.00  0.00           \n"
"ATOM   1192 2H7    T B  18      -4.871  -1.963   7.047  1.00  0.00           \n"
"ATOM   1193 3H7    T B  18      -5.965  -0.984   8.052  1.00  0.00           \n"
"ATOM   1194  C4    T B  18      -3.021  -0.218   6.974  1.00  0.00           \n"
"ATOM   1195  O4    T B  18      -2.564  -1.361   6.901  1.00  0.00           \n"
"ATOM   1196  N3    T B  18      -2.180   0.876   6.951  1.00  0.00           \n"
"ATOM   1197  H3    T B  18      -1.112   0.674   6.865  1.00  0.00           \n"
"ATOM   1198  C2    T B  18      -2.558   2.202   7.025  1.00  0.00           \n"
"ATOM   1199  O2    T B  18      -1.738   3.103   6.997  1.00  0.00           \n"
"ATOM   1200  C3*   T B  18      -6.323   5.157   7.425  1.00  0.00           \n"
"ATOM   1201  H3*   T B  18      -7.407   5.127   7.314  1.00  0.00           \n"
"ATOM   1202  C2*   T B  18      -5.688   4.107   6.513  1.00  0.00           \n"
"ATOM   1203 1H2*   T B  18      -6.471   3.516   6.038  1.00  0.00           \n"
"ATOM   1204 2H2*   T B  18      -5.093   4.603   5.746  1.00  0.00           \n"
"ATOM   1205  O3*   T B  18      -5.839   6.444   7.067  1.00  0.00           \n"
"ATOM   1206  P     A B  19      -5.792   6.854   5.521  1.00  0.00           \n"
"ATOM   1207  O1P   A B  19      -6.090   8.297   5.380  1.00  0.00           \n"
"ATOM   1208  O2P   A B  19      -6.654   5.939   4.739  1.00  0.00           \n"
"ATOM   1209  O5*   A B  19      -4.255   6.582   5.168  1.00  0.00           \n"
"ATOM   1210  C5*   A B  19      -3.246   6.924   6.136  1.00  0.00           \n"
"ATOM   1211 1H5*   A B  19      -3.073   6.103   6.832  1.00  0.00           \n"
"ATOM   1212 2H5*   A B  19      -3.595   7.798   6.685  1.00  0.00           \n"
"ATOM   1213  C4*   A B  19      -1.946   7.263   5.433  1.00  0.00           \n"
"ATOM   1214  H4*   A B  19      -1.256   7.809   6.077  1.00  0.00           \n"
"ATOM   1215  O4*   A B  19      -1.305   6.011   5.203  1.00  0.00           \n"
"ATOM   1216  C1*   A B  19      -1.268   5.671   3.832  1.00  0.00           \n"
"ATOM   1217  H1*   A B  19      -0.263   5.619   3.414  1.00  0.00           \n"
"ATOM   1218  N9    A B  19      -1.738   4.260   3.750  1.00  0.00           \n"
"ATOM   1219  C8    A B  19      -3.025   3.773   3.765  1.00  0.00           \n"
"ATOM   1220  H8    A B  19      -3.861   4.457   3.797  1.00  0.00           \n"
"ATOM   1221  N7    A B  19      -3.100   2.481   3.676  1.00  0.00           \n"
"ATOM   1222  C5    A B  19      -1.777   2.073   3.597  1.00  0.00           \n"
"ATOM   1223  C6    A B  19      -1.181   0.806   3.485  1.00  0.00           \n"
"ATOM   1224  N6    A B  19      -1.880  -0.338   3.431  1.00  0.00           \n"
"ATOM   1225 1H6    A B  19      -1.397  -1.221   3.350  1.00  0.00           \n"
"ATOM   1226 2H6    A B  19      -2.889  -0.316   3.471  1.00  0.00           \n"
"ATOM   1227  N1    A B  19       0.162   0.756   3.431  1.00  0.00           \n"
"ATOM   1228  C2    A B  19       0.850   1.895   3.485  1.00  0.00           \n"
"ATOM   1229  H2    A B  19       1.929   1.877   3.442  1.00  0.00           \n"
"ATOM   1230  N3    A B  19       0.407   3.129   3.589  1.00  0.00           \n"
"ATOM   1231  C4    A B  19      -0.941   3.150   3.642  1.00  0.00           \n"
"ATOM   1232  C3*   A B  19      -2.084   7.888   4.045  1.00  0.00           \n"
"ATOM   1233  H3*   A B  19      -2.978   8.501   3.934  1.00  0.00           \n"
"ATOM   1234  C2*   A B  19      -2.188   6.666   3.133  1.00  0.00           \n"
"ATOM   1235 1H2*   A B  19      -3.169   6.649   2.658  1.00  0.00           \n"
"ATOM   1236 2H2*   A B  19      -1.415   6.717   2.367  1.00  0.00           \n"
"ATOM   1237  O3*   A B  19      -0.936   8.645   3.687  1.00  0.00           \n"
"ATOM   1238  P     T B  20      -0.657   8.949   2.141  1.00  0.00           \n"
"ATOM   1239  O1P   T B  20      -0.050  10.292   2.000  1.00  0.00           \n"
"ATOM   1240  O2P   T B  20      -1.892   8.716   1.359  1.00  0.00           \n"
"ATOM   1241  O5*   T B  20       0.427   7.826   1.788  1.00  0.00           \n"
"ATOM   1242  C5*   T B  20       1.443   7.510   2.756  1.00  0.00           \n"
"ATOM   1243 1H5*   T B  20       1.100   6.745   3.453  1.00  0.00           \n"
"ATOM   1244 2H5*   T B  20       1.674   8.423   3.305  1.00  0.00           \n"
"ATOM   1245  C4*   T B  20       2.695   7.020   2.053  1.00  0.00           \n"
"ATOM   1246  H4*   T B  20       3.574   7.056   2.696  1.00  0.00           \n"
"ATOM   1247  O4*   T B  20       2.477   5.630   1.823  1.00  0.00           \n"
"ATOM   1248  C1*   T B  20       2.308   5.333   0.452  1.00  0.00           \n"
"ATOM   1249  H1*   T B  20       3.094   4.693   0.051  1.00  0.00           \n"
"ATOM   1250  N1    T B  20       1.098   4.468   0.370  1.00  0.00           \n"
"ATOM   1251  C6    T B  20      -0.158   5.012   0.400  1.00  0.00           \n"
"ATOM   1252  H6    T B  20      -0.280   6.084   0.439  1.00  0.00           \n"
"ATOM   1253  C5    T B  20      -1.255   4.240   0.327  1.00  0.00           \n"
"ATOM   1254  C7    T B  20      -2.636   4.826   0.358  1.00  0.00           \n"
"ATOM   1255 1H7    T B  20      -2.760   5.509  -0.482  1.00  0.00           \n"
"ATOM   1256 2H7    T B  20      -3.373   4.026   0.287  1.00  0.00           \n"
"ATOM   1257 3H7    T B  20      -2.780   5.369   1.292  1.00  0.00           \n"
"ATOM   1258  C4    T B  20      -1.141   2.806   0.214  1.00  0.00           \n"
"ATOM   1259  O4    T B  20      -2.087   2.018   0.141  1.00  0.00           \n"
"ATOM   1260  N3    T B  20       0.159   2.344   0.191  1.00  0.00           \n"
"ATOM   1261  H3    T B  20       0.296   1.266   0.105  1.00  0.00           \n"
"ATOM   1262  C2    T B  20       1.304   3.113   0.265  1.00  0.00           \n"
"ATOM   1263  O2    T B  20       2.414   2.612   0.237  1.00  0.00           \n"
"ATOM   1264  C3*   T B  20       2.950   7.607   0.665  1.00  0.00           \n"
"ATOM   1265  H3*   T B  20       2.586   8.629   0.554  1.00  0.00           \n"
"ATOM   1266  C2*   T B  20       2.148   6.679  -0.247  1.00  0.00           \n"
"ATOM   1267 1H2*   T B  20       1.344   7.241  -0.722  1.00  0.00           \n"
"ATOM   1268 2H2*   T B  20       2.804   6.267  -1.014  1.00  0.00           \n"
"ATOM   1269  O3*   T B  20       4.324   7.545   0.307  1.00  0.00           \n"
"ATOM   1270  H3    T B  20       4.534   7.903  -0.559  1.00  0.00           \n"
"TER\n";

		std::istringstream strand1Stream(strand1String);
		std::istringstream strand2Stream(strand2String);

		const PdbStructure strand1Structure(strand1Stream);
		const PdbStructure strand2Structure(strand2Stream);

		const PdbChain& strand1Chain = 
			strand1Structure.getModel(Pdb::ModelIndex(0)).getChain('A');
		const PdbChain& strand2Chain = 
			strand2Structure.getModel(Pdb::ModelIndex(0)).getChain('B');

		for (Pdb::ResidueIndex r(0); r < strand1Chain.getNResidues(); ++r) {
			strand1Chain.getResidue(r).
		}

	}

	Transform get5PrimeEndInBodyFrame() {
		return Transform(Vec3(0, -1.7, 0));
	}

	Transform get3PrimeEndInBodyFrame() {
		return Transform(Vec3(0, 1.7, 0));
	}

private:
};

int main() {

    MultibodySystem system;
    SimbodyMatterSubsystem matter(system);

    GeneralForceSubsystem forces(system);

	// This is for testing; not used yet
	DNADuplexTurnBody testTurnBody;

	// One turn of DNA contains ten base pairs
	// This body will be used as a template to stamp out DNA turns later
	Body::Rigid dnaDuplexTurnBody(MassProperties(
			6600, // total mass
			Vec3(0), // center of mass
			// TODO - compute correct inertia matrix
			Inertia(1))); // mass distribution

	// Denote each turn (visually) with a cylinder
	// TODO - create fancier visualizations:
	//   * all atoms
	//   * schematic base pairs
	dnaDuplexTurnBody.addDecoration(
		Transform(), 
		DecorativeCylinder(1.0, 1.7)
		);
	// Add a bump to observe twist
	dnaDuplexTurnBody.addDecoration(
		Transform(Vec3(1.0, 0, 0)),
		DecorativeSphere(0.5)
		);

	// Place first DNA turn (ten base pairs) free floating at origin
	// TODO - make first body Free, when not using gravity
	MobilizedBody::Free firstTurn(
		matter.updGround(), 
		Transform(Vec3(0)), 
		dnaDuplexTurnBody,
		Transform(Vec3(0)));

	// Add ten more turns
    Random::Gaussian randomAngle(0, 0.1);
	MobilizedBodyIndex previousBodyIx = firstTurn.getMobilizedBodyIndex();
	for (int i = 0; i < 10; ++i) 
	{
		// LineOrientation needs special axis along Z
		// But our DNA model has special axis along Y
		Transform parent_X_fixed(Vec3(0, 1.7, 0));
		Transform child_X_mobile(Vec3(0, -1.7, 0));

		// TODO - Compound a LineOrientation and Pin to model bend and twist
		MobilizedBody::Pin nextTurn(
			matter.updMobilizedBody(previousBodyIx),
			parent_X_fixed, 
			dnaDuplexTurnBody,
			// top of new turn
			child_X_mobile
			);

		// Use a spring to inhibit bending
		// TODO - get true spring constant
		Force::MobilityLinearSpring(forces, nextTurn, 0, 100.0, 0.0);

		nextTurn.setDefaultAngle(randomAngle.getValue());

		previousBodyIx = nextTurn.getMobilizedBodyIndex();
	}

	system.updDefaultSubsystem().addEventReporter(new VTKEventReporter(system, 0.5));

	system.realizeTopology();
    State state = system.getDefaultState();
    
    // Simulate it.

    RungeKuttaMersonIntegrator integ(system);
    TimeStepper ts(system, integ);
    ts.initialize(state);
    ts.stepTo(1000.0);

	return 0;
}
