﻿namespace MonoDevelop.FSharp

open FSharp.CompilerBinding
open ICSharpCode.NRefactory.TypeSystem
open Microsoft.FSharp.Compiler
open MonoDevelop.Core
open MonoDevelop.Ide
open MonoDevelop.Ide.Gui
open MonoDevelop.Ide.TypeSystem
open System 
open System.Diagnostics
open System.IO
open System.Text
open System.Threading

type FSharpParsedDocument(fileName) = 
    inherit DefaultParsedDocument(fileName)

// An instance of this type is created by MonoDevelop (as defined in the .xml for the AddIn) 
type FSharpParser() = 
    inherit TypeSystemParser()

    let languageService = MDLanguageService.Instance 

    /// Format errors for the given line (if there are multiple, we collapse them into a single one)
    let formatError (error : FSharpErrorInfo) = 
        // Single error for this line
        let errorType = 
            if error.Severity = FSharpErrorSeverity.Error then ErrorType.Error
            else ErrorType.Warning
        Error(errorType, String.wrapText error.Message 80, DomRegion(error.StartLineAlternate, error.StartColumn + 1, error.EndLineAlternate, error.EndColumn + 1))

    override x.Parse(storeAst : bool, fileName : string, content : System.IO.TextReader, proj : MonoDevelop.Projects.Project) = 
        if fileName = null || not (MDLanguageService.SupportedFileName (fileName)) then null
        else 
            let fileContent = content.ReadToEnd()
            let fileHash = hash fileContent 
            let shortFilename = Path.GetFileName fileName

            let doc = new FSharpParsedDocument(fileName, Flags = ParsedDocumentFlags.NonSerializable)
            // Not sure if these are needed yet. 
            doc.CreateRefactoringContext <- Func<_, _, _>(fun doc token -> FSharpRefactoringContext() :> _)
            doc.CreateRefactoringContextWithEditor <- Func<_, _, _, _> (fun data resolver token -> FSharpRefactoringContext() :> _)
            LoggingService.LogInfo ("FSharpParser: [Thread {0}] Parse {1}, hash {2}", Thread.CurrentThread.ManagedThreadId, shortFilename, fileHash)

            let filePathOpt = 
                // TriggerParse will work only for full paths
                
                
                if IO.Path.IsPathRooted(fileName) then Some(fileName)
                else 
                    let workBench = IdeApp.Workbench
                    if (workBench <> null) then
                        
                        let doc = workBench.ActiveDocument
                        if doc <> null then 
                            let file = doc.FileName.FullPath.ToString()
                            if file = "" then None else Some file
                        else None
                    else 
                        let filePaths = proj.GetItemFiles(true)
                        let res = filePaths |> Seq.find(fun t -> t.FileName = fileName)
                        Some(res.FullPath.ToString())
                        

            match filePathOpt with
            | None -> ()
            | Some filePath -> 
                let projFile, files, args = MonoDevelop.getCheckerArgs (proj, filePath)

                let results =
                    try
                        LoggingService.LogInfo ("FSharpParser: [Thread {0}] Running ParseAndCheckFileInProject for {1}, hash {2}", Thread.CurrentThread.ManagedThreadId, shortFilename, fileHash)
                        Async.RunSynchronously (
                            computation = languageService.ParseAndCheckFileInProject(projFile, filePath, fileContent, files, args, storeAst, false), 
                            timeout = ServiceSettings.maximumTimeout)
                    with
                    | :? TimeoutException ->
                        doc.IsInvalid <- true
                        LoggingService.LogWarning ("FSharpParser: [Thread {0}] ParseAndCheckFileInProject timed out for {1}, hash {2}", Thread.CurrentThread.ManagedThreadId, shortFilename, fileHash)
                        ParseAndCheckResults.Empty
                    | ex -> doc.IsInvalid <- true
                            LoggingService.LogError("FSharpParser: [Thread {0}] Error processing ParseAndCheckFileResults for {1}, hash {2}", Thread.CurrentThread.ManagedThreadId, shortFilename, fileHash, ex)
                            ParseAndCheckResults.Empty
                                                                                     
                results.GetErrors() |> Option.iter (Array.map formatError >> doc.Add)

                //Set code folding regions, GetNavigationItems may throw in some situations
                try 
                    let regions = 
                        let processDecl (decl : SourceCodeServices.FSharpNavigationDeclarationItem) = 
                            let m = decl.Range
                            FoldingRegion(decl.Name, DomRegion(m.StartLine, m.StartColumn + 1, m.EndLine, m.EndColumn + 1))
                        seq {for toplevel in results.GetNavigationItems() do
                                yield processDecl toplevel.Declaration
                                for next in toplevel.Nested do
                                    yield processDecl next }
                    doc.Add(regions)
                with ex -> LoggingService.LogWarning ("FSharpParser: Couldn't update navigation items.", ex)
                //also store the AST of active results if applicable 
                //Is there any reason not to store the AST? The navigation extension depends on it
                if storeAst then doc.Ast <- results
            doc.LastWriteTimeUtc <- try File.GetLastWriteTimeUtc(fileName) with _ -> DateTime.UtcNow
            doc :> ParsedDocument
