/*
 * Copyright (C) 2010 Google Inc. All rights reserved.
 * Copyright (C) 2013, 2014, 2016 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#include <functional>
#include <wtf/Forward.h>
#include <wtf/Function.h>

namespace WebCore {

class BlobDataFileReference;
class BlobPart;
class BlobRegistry;
class URL;

WEBCORE_EXPORT BlobRegistry& blobRegistry();

// BlobRegistry is not thread-safe. It should only be called from main thread.
class WEBCORE_EXPORT BlobRegistry {
public:

    // Registers a blob URL referring to the specified file.
    virtual void registerFileBlobURL(const URL&, Ref<BlobDataFileReference>&&, const String& contentType) = 0;

    // Registers a blob URL referring to the specified blob data.
    virtual void registerBlobURL(const URL&, Vector<BlobPart>&&, const String& contentType) = 0;

    // Registers a new blob URL referring to the blob data identified by the specified srcURL.
    virtual void registerBlobURL(const URL&, const URL& srcURL) = 0;

    // Registers a new blob URL referring to the blob data identified by the specified srcURL or, if none found, referring to the file found at the given path.
    virtual void registerBlobURLOptionallyFileBacked(const URL&, const URL& srcURL, RefPtr<BlobDataFileReference>&&, const String& contentType) = 0;

    // Negative start and end values select from the end.
    virtual void registerBlobURLForSlice(const URL&, const URL& srcURL, long long start, long long end) = 0;

    virtual void unregisterBlobURL(const URL&) = 0;

    virtual unsigned long long blobSize(const URL&) = 0;

    virtual void writeBlobsToTemporaryFiles(const Vector<String>& blobURLs, WTF::Function<void (const Vector<String>& filePaths)>&& completionHandler) = 0;

    virtual bool isBlobRegistryImpl() const { return false; }

protected:
    virtual ~BlobRegistry();
};

} // namespace WebCore
