///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file DataChannelComboBox.h
 * \brief Contains the definition of the AtomViz::DataChannelComboBox class.
 */

#ifndef __DATA_CHANNEL_COMBO_BOX_H
#define __DATA_CHANNEL_COMBO_BOX_H

#include <core/Core.h>
#include <atomviz/AtomViz.h>
#include <atomviz/atoms/datachannels/DataChannel.h>

namespace AtomViz {

/**
 * \brief Widget that allows the user to select a DataChannel from a list.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT DataChannelComboBox : public QComboBox
{
	Q_OBJECT

public:

	/// \brief Default constructor.
	/// \param parent The parent widget of the combo box.
	DataChannelComboBox(QWidget* parent = NULL) : QComboBox(parent) {}

	/// \brief Adds a data channel to the end of the list.
	/// \param channel The channel to be added.
	void addItem(const DataChannelReference& channel) {
		QComboBox::addItem(channel.name(), qVariantFromValue((int)channel.id()));
	}

	/// \brief Adds a data channel to the end of the list.
	/// \param channel The channel to be added.
	void addItem(DataChannel* channel) {
		addItem(DataChannelReference(channel));
	}

	/// \brief Adds multiple data channels to the combo box.
	/// \param list The list of channels to be added.
	void addItems(const DataChannelList& list) {
		Q_FOREACH(DataChannel* c, list)
			addItem(c);
	}

	/// \brief Returns the data channel that is currently selected in the
	///        combo box.
	/// \return The selected item. The returned reference can be null if
	///         no item is currently selected.
	DataChannelReference currentChannel() const {
		int index = currentIndex();
		if(index < 0) return DataChannelReference();
		QString channelName = itemText(index);
		DataChannel::DataChannelIdentifier id = (DataChannel::DataChannelIdentifier)itemData(index).toInt();
		return DataChannelReference(id, channelName);
	}

	/// \brief Sets the selection of the combo box to the fiven data channel.
	/// \param channel The channel to be selected.
	void setCurrentChannel(const DataChannelReference& channel) {
		int index;
		if(channel.id() == DataChannel::UserDataChannel)
			index = findText(channel.name());
		else
			index = findData((int)channel.id());
		setCurrentIndex(index);
	}
};

};	// End of namespace AtomViz

#endif // __DATA_CHANNEL_COMBO_BOX_H
