///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/gui/properties/FloatControllerUI.h>
#include "RectangleObject.h"
#include "RectangleCreationMode.h"

namespace StdObjects {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(RectangleObject, SimpleShapeObject)
DEFINE_REFERENCE_FIELD(RectangleObject, FloatController, "Width", _width)
DEFINE_REFERENCE_FIELD(RectangleObject, FloatController, "Length", _length)
SET_PROPERTY_FIELD_LABEL(RectangleObject, _width, "Width")
SET_PROPERTY_FIELD_LABEL(RectangleObject, _length, "Length")
SET_PROPERTY_FIELD_UNITS(RectangleObject, _width, WorldParameterUnit)
SET_PROPERTY_FIELD_UNITS(RectangleObject, _length, WorldParameterUnit)

IMPLEMENT_PLUGIN_CLASS(RectangleObjectEditor, PropertiesEditor)
IMPLEMENT_PLUGIN_CLASS(RectangleCreationMode, SimpleCreationMode)

/******************************************************************************
* Constructs a new rectangle object.
******************************************************************************/
RectangleObject::RectangleObject(bool isLoading) : SimpleShapeObject(isLoading)
{
	INIT_PROPERTY_FIELD(RectangleObject, _width);
	INIT_PROPERTY_FIELD(RectangleObject, _length);
	if(!isLoading) {
		_width = CONTROLLER_MANAGER.createDefaultController<FloatController>();
		_length = CONTROLLER_MANAGER.createDefaultController<FloatController>();
	}
}

/******************************************************************************
* Asks the object for its validity interval at the given time.
******************************************************************************/
TimeInterval RectangleObject::objectValidity(TimeTicks time)
{
	TimeInterval interval = TimeForever;
	_width->validityInterval(time, interval);
	_length->validityInterval(time, interval);
	return interval;
}

/******************************************************************************
* Builds the shape representation of this procedural object.
******************************************************************************/
void RectangleObject::buildShape(TimeTicks time, BezierShape& shape, TimeInterval& shapeValidity)
{
	// Reset shape validity interval.
	shapeValidity.setInfinite();

	// Query parameters
	FloatType width=0, length=0;
	_width->getValue(time, width, shapeValidity);
	_length->getValue(time, length, shapeValidity);

	width *= 0.5;
	length *= 0.5;

	shape.curves().resize(1);
	BezierCurve& curve = shape.curves()[0];
	curve.setVertexCount(0);
	curve.vertices().push_back(BezierVertex(Point3(-width, -length, 0.0)));
	curve.vertices().push_back(BezierVertex(Point3( width, -length, 0.0)));
	curve.vertices().push_back(BezierVertex(Point3( width,  length, 0.0)));
	curve.vertices().push_back(BezierVertex(Point3(-width,  length, 0.0)));
	curve.setClosed(true);
	curve.invalidate();
}

/******************************************************************************
* Creates the UI controls for the editor.
******************************************************************************/
void RectangleObjectEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create the rollout.
	QWidget* rollout = createRollout(tr("Rectangle"), rolloutParams);

	QGridLayout* layout = new QGridLayout(rollout);
	layout->setContentsMargins(4,4,4,4);
	layout->setSpacing(0);
	layout->setColumnStretch(1, 1);

	// Width parameter.
	FloatControllerUI* widthPUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(RectangleObject, _width));
	layout->addWidget(widthPUI->label(), 0, 0);
	layout->addWidget(widthPUI->textBox(), 0, 1);
	layout->addWidget(widthPUI->spinner(), 0, 2);
	widthPUI->setMinValue(0);

	// Length parameter.
	FloatControllerUI* lengthPUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(RectangleObject, _length));
	layout->addWidget(lengthPUI->label(), 1, 0);
	layout->addWidget(lengthPUI->textBox(), 1, 1);
	layout->addWidget(lengthPUI->spinner(), 1, 2);
	lengthPUI->setMinValue(0);
}

};
