#!/usr/bin/env python
# coding=utf8
# This file is part of Xpra.
# Copyright (C) 2011-2013 Antoine Martin <antoine@devloop.org.uk>
# Xpra is released under the terms of the GNU GPL v2, or, at your option, any
# later version. See the file COPYING for details.

import time
from xpra.util import nonl
encode, decode = None, None

from xpra.net.bencode.bencode import bencode as p_bencode, bdecode as p_bdecode
from xpra.net.bencode.cython_bencode import bencode as c_bencode, bdecode as c_bdecode  #@UnresolvedImport

def use_cython():
    global encode, decode

    encode = c_bencode
    decode = c_bdecode

def use_python():
    global encode, decode
    encode = p_bencode
    decode = p_bdecode

def plog(x):
    print(x)
def noop(x):
    pass
log = plog

def use_print():
    global log
    log = plog
def use_noop():
    global log
    log = noop
    


def i(s):
    v = nonl(str(s))
    if len(v)<48:
        return v
    return v[:46]+".."


def test_decoding():

    def t(s, value, remainder):
        #log(" "+i(s))
        # Test "one-shot":
        rv, rr = decode(s)
        assert rv == value, "expected value %s but got %s" % (rv, value)
        rrstr = s[rr:]
        assert rrstr == remainder, "expected remainder value %s but got %s" % (remainder, rrstr)
        # With gibberish added:
        g_str = s + "asdf"
        rv, rr = decode(g_str)
        assert rv == value, "expected value %s but got %s" % (rv, value)
        rrstr = g_str[rr:]
        assert rrstr.endswith("asdf")

    #t("l16:configure-windowi2ei555ei340ei649ei381ed9:maximizedi0e6:screeni0e9:maximizedi0eee", [], "")

    t("i12345e", 12345, "")
    t("i-12345e", -12345, "")
    t("i12345eQQQ", 12345, "QQQ")
    t("3:foo", "foo", "")
    t("3:fooQQQ", "foo", "QQQ")
    t("li12e4:asdfi34ee", [12, "asdf", 34], "")
    t("d4:asdf3:foo4:bsdfi1234ee", {"asdf": "foo", "bsdf": 1234}, "")

    t("d4:asdfli1ei2ei3ei4ee5:otheri-55e2:qqd2:qql2:hieee",
      {"asdf": [1, 2, 3, 4], "qq": {"qq": ["hi"]}, "other": -55},
      "")

    t("l0:e", [""], "")

    # Keys do not have to be strings:
    t("di0ei0ee", {0 : 0}, "")

    def te(s, exc):
        #log(" "+s)
        v = None
        try:
            v = decode(s)
        except exc:
            pass
        else:
            assert False, "didn't raise exception, returned: %s for %s" % (v, s)

    te("iie", ValueError)
    te("i0x0e", ValueError)
    t("i0e", 0, "")
    te("i00e", ValueError)

    te("0x2:aa", ValueError)
    te("-1:aa", ValueError)
    te("02:aa", ValueError)


def t(v, encstr=None):
    be = encode(v)
    log("bencode(%s)=%s" % (i(v), i(be)))
    if encstr:
        p = 0
        while p<min(len(encstr), len(be)):
            if encstr[p]!=be[p]:
                break
            p += 1
        #bec = be[p:p+20]
        #encstrc = encstr[p:p+20]
        #print("context: expected '%s' but got '%s'" % (nonl(encstrc), nonl(bec)))
        #print("context: expected '%s' but got '%s'" % ([ord(c) for c in encstrc], [ord(c) for c in bec]))
        assert be==encstr, "expected '%s' but got '%s', strings differ at position %s: expected '%s' but found '%s' (lengths: %s vs %s)" % (nonl(encstr), nonl(be), p, encstr[p], be[p], len(encstr), len(be))
    restored = decode(be)
    log("decode(%s)=%s" % (i(be), i(restored)))
    rlist = restored[0]
    if len(rlist)!=len(v):
        log("MISMATCH!")
        log("v=%s" % v)
        log("l=%s" % rlist)
    assert len(rlist)==2
    assert rlist[0]==v[0]
    for ok,ov in v[1].items():
        d = rlist[1]
        if ok not in d:
            log("restored dict is missing %s" % ok)
            return rlist
        rv = d.get(ok)
        if rv!=ov:
            log("value for %s does not match: %s vs %s" % (ok, ov, rv))
            return rlist
    return rlist



hello = ["hello", {
                   "__prerelease_version"   : "0.0.7.26",
                   "desktop_size"           : [480,800],
                   "jpeg"                   : 4,
                   "challenge"              : "ba59e4110119264f4a6eaf3adc075ea2c5408550",
                   "challenge_response"     : "ba59e4110119264f4a6eaf3adc075ea2c5408550",
                   }]
hello_output = "l5:hellod12:desktop_sizeli480ei800ee9:challenge40:ba59e4110119264f4a6eaf3adc075ea2c540855020:__prerelease_version8:0.0.7.264:jpegi4e18:challenge_response40:ba59e4110119264f4a6eaf3adc075ea2c5408550ee"
def test_encoding_hello():
    log("test_hello()")
    t(hello, hello_output)
    log("")

large_hello = ["hello",
               {'pycrypto.version': '2.6.1', 'named_cursors': True, 'bell': True, 'encoding.cython.version': (0, 1),
                'platform.release': '3.12.5-200.fc19.x86_64', 'lz4': True, 'clipboard.greedy': False,
                'encoding.vpx.version': 'v1.2.0', 'xkbmap_print': u'xkb_keymap {\n\txkb_keycodes  { include "evdev+aliases(qwerty)"\t};\n\txkb_types     { include "complete"\t};\n\txkb_compat    { include "complete"\t};\n\txkb_symbols   { include "pc+gb+us:2+inet(evdev)"\t};\n\txkb_geometry  { include "pc(pc104)"\t};\n};\n',
                'sound.receive': True, 'digest': ('hmac', 'xor'),
                'aliases': {'lost-window': 6, 'bell': 7, 'desktop_size': 8, 'new-override-redirect': 9, 'ping_echo': 5, 'new-window': 10, 'connection-lost': 11, 'startup-complete': 12, 'info-response': 2, 'disconnect': 13, 'ping': 3, 'window-resized': 14, 'set_deflate': 15, 'rpc-reply': 16, 'window-icon': 17, 'draw': 18, 'notify_close': 19, 'sound-data': 1, 'raise-window': 20, 'window-metadata': 21, 'set-clipboard-enabled': 22, 'configure-override-redirect': 23, 'challenge': 24, 'cursor': 25, 'notify_show': 26, 'gibberish': 27, 'new-tray': 28, 'hello': 29},
                'platform.platform': 'Linux-3.12.5-200.fc19.x86_64-x86_64-with-fedora-19-Schr\xc3\xb6dinger\xe2\x80\x99s_Cat',
                'mmap_file': '/tmp/xpra.AFrOuc.mmap', 'uuid': '8643124ce701ee68dbb6b7a8c4eb13a5f6409494', 'encoding.opencl.version': '2013.1', 'bencode.version': ('Cython', 0, 11), 'xkbmap_layout': '',
                'xkbmap_mod_meanings': {'ISO_Level3_Shift': 'mod5', 'Meta_L': 'mod1', 'Control_R': 'control', 'Super_R': 'mod4', 'Mode_switch': 'mod5', 'Hyper_L': 'mod4', 'Caps_Lock': 'lock', 'Alt_L': 'mod1', 'Num_Lock': 'mod2', 'Super_L': 'mod4', 'Shift_R': 'shift', 'Shift_L': 'shift', 'Control_L': 'control'},
                'encoding.PIL.version': '1.1.7', 'platform': 'linux2', 'sound.server_driven': True, 'clipboard': True, 'encodings.rgb_formats': ['RGB', 'RGBA'], 'chunked_compression': True, 'keyboard_sync': True, 'sound.pygst.version': (0, 10, 22), 'sound.send': True, 'screen_sizes': [(':0.0', 1920, 1080, 508, 286, [('DVI-I-1', 0, 0, 1920, 1080, 531, 299)], 0, 0, 1920, 1055)],
                'username': 'antoine', 'auto_refresh_delay': 250, 'mmap_token': 215666214940457138203759294163634184205L, 'encoding.h264.YUV420P.profile': 'high10', 'encoding.transparency': True, 'build.cpu': 'x86_64', 'pycrypto.fastmath': True,
                'xkbmap_query': u'rules:      evdev\nmodel:      pc104\nlayout:     gb,us\nvariant:    ,\n',
                'encoding.rgb24zlib': True, 'platform.machine': 'x86_64', 'encoding.csc_atoms': True, 'encoding.x264.YUV420P.profile': 'high10', 'build.on': 'desktop', 'rencode': True, 'generic_window_types': True, 'gtk.version': (2, 24, 22), 'window.raise': True, 'modifiers': [],
                'name': 'Antoine Martin', 'encoding.client_options': True, 'encoding.supports_delta': ['png', 'rgb24', 'rgb32'], 'platform.name': 'Linux', 'zlib': True, 'build.revision': '5071', 'client_type': 'Python/Gtk2', 'sound.pulseaudio.server': '{7725dfc225d14958a625ddaaaea5962b}unix:/run/user/1000/pulse/native', 'encoding_client_options': True,
                'build.by': 'root', 'machine_id': '7725dfc225d14958a625ddaaaea5962b', 'display': ':10', 'python.version': (2, 7, 5), 'encoding.video_scaling': True, 'encoding.x264.version': 130L, 'encoding.uses_swscale': True, 'server_uuid': '', 'desktop_size': [1920, 1080], 'encodings': ['h264', 'vp8', 'webp', 'png', 'png/P', 'png/L', 'rgb', 'jpeg'], 'share': False, 'xkbmap_variant': '', 'sound.pulseaudio.id': '1000@7725dfc225d14958a625ddaaaea5962b/2073', 'cursors': True, 'randr_notify': True,
                'sound.decoders': ['mp3', 'wavpack', 'wav', 'flac', 'speex'], 'rencode.version': '1.0.2', 'encoding.csc_modes': ('YUV420P', 'YUV422P', 'YUV444P', 'BGRA', 'BGRX'), 'generic-rgb-encodings': True,
                'xkbmap_keycodes': [(65307, 'Escape', 9, 0, 0), (49, '1', 10, 0, 0), (33, 'exclam', 10, 0, 1), (185, 'onesuperior', 10, 0, 2), (161, 'exclamdown', 10, 0, 3), (49, '1', 10, 1, 0), (33, 'exclam', 10, 1, 1), (50, '2', 11, 0, 0), (34, 'quotedbl', 11, 0, 1), (178, 'twosuperior', 11, 0, 2), (2755, 'oneeighth', 11, 0, 3), (50, '2', 11, 1, 0), (64, 'at', 11, 1, 1), (51, '3', 12, 0, 0), (163, 'sterling', 12, 0, 1), (179, 'threesuperior', 12, 0, 2), (163, 'sterling', 12, 0, 3), (51, '3', 12, 1, 0), (35, 'numbersign', 12, 1, 1), (52, '4', 13, 0, 0), (36, 'dollar', 13, 0, 1), (8364, 'EuroSign', 13, 0, 2), (188, 'onequarter', 13, 0, 3), (52, '4', 13, 1, 0), (36, 'dollar', 13, 1, 1), (53, '5', 14, 0, 0),
                                    (37, 'percent', 14, 0, 1), (189, 'onehalf', 14, 0, 2), (2756, 'threeeighths', 14, 0, 3), (53, '5', 14, 1, 0), (37, 'percent', 14, 1, 1), (54, '6', 15, 0, 0), (94, 'asciicircum', 15, 0, 1), (190, 'threequarters', 15, 0, 2), (2757, 'fiveeighths', 15, 0, 3), (54, '6', 15, 1, 0), (94, 'asciicircum', 15, 1, 1), (55, '7', 16, 0, 0), (38, 'ampersand', 16, 0, 1), (123, 'braceleft', 16, 0, 2), (2758, 'seveneighths', 16, 0, 3), (55, '7', 16, 1, 0), (38, 'ampersand', 16, 1, 1), (56, '8', 17, 0, 0), (42, 'asterisk', 17, 0, 1), (91, 'bracketleft', 17, 0, 2), (2761, 'trademark', 17, 0, 3), (56, '8', 17, 1, 0), (42, 'asterisk', 17, 1, 1), (57, '9', 18, 0, 0), (40, 'parenleft', 18, 0, 1),
                                    (93, 'bracketright', 18, 0, 2), (177, 'plusminus', 18, 0, 3), (57, '9', 18, 1, 0), (40, 'parenleft', 18, 1, 1), (48, '0', 19, 0, 0), (41, 'parenright', 19, 0, 1), (125, 'braceright', 19, 0, 2), (176, 'degree', 19, 0, 3), (48, '0', 19, 1, 0), (41, 'parenright', 19, 1, 1), (45, 'minus', 20, 0, 0), (95, 'underscore', 20, 0, 1), (92, 'backslash', 20, 0, 2), (191, 'questiondown', 20, 0, 3), (45, 'minus', 20, 1, 0), (95, 'underscore', 20, 1, 1), (61, 'equal', 21, 0, 0), (43, 'plus', 21, 0, 1), (65115, 'dead_cedilla', 21, 0, 2), (65116, 'dead_ogonek', 21, 0, 3), (61, 'equal', 21, 1, 0), (43, 'plus', 21, 1, 1), (65288, 'BackSpace', 22, 0, 0), (65288, 'BackSpace', 22, 0, 1),
                                    (65289, 'Tab', 23, 0, 0), (65056, 'ISO_Left_Tab', 23, 0, 1), (113, 'q', 24, 0, 0), (81, 'Q', 24, 0, 1), (64, 'at', 24, 0, 2), (2009, 'Greek_OMEGA', 24, 0, 3), (113, 'q', 24, 1, 0), (81, 'Q', 24, 1, 1), (119, 'w', 25, 0, 0), (87, 'W', 25, 0, 1), (435, 'lstroke', 25, 0, 2), (419, 'Lstroke', 25, 0, 3), (119, 'w', 25, 1, 0), (87, 'W', 25, 1, 1), (101, 'e', 26, 0, 0), (69, 'E', 26, 0, 1), (101, 'e', 26, 0, 2), (69, 'E', 26, 0, 3), (101, 'e', 26, 1, 0), (69, 'E', 26, 1, 1), (114, 'r', 27, 0, 0), (82, 'R', 27, 0, 1), (182, 'paragraph', 27, 0, 2), (174, 'registered', 27, 0, 3), (114, 'r', 27, 1, 0), (82, 'R', 27, 1, 1), (116, 't', 28, 0, 0), (84, 'T', 28, 0, 1), (956, 'tslash', 28, 0, 2),
                                    (940, 'Tslash', 28, 0, 3), (116, 't', 28, 1, 0), (84, 'T', 28, 1, 1), (121, 'y', 29, 0, 0), (89, 'Y', 29, 0, 1), (2299, 'leftarrow', 29, 0, 2), (165, 'yen', 29, 0, 3), (121, 'y', 29, 1, 0), (89, 'Y', 29, 1, 1), (117, 'u', 30, 0, 0), (85, 'U', 30, 0, 1), (2302, 'downarrow', 30, 0, 2), (2300, 'uparrow', 30, 0, 3), (117, 'u', 30, 1, 0), (85, 'U', 30, 1, 1), (105, 'i', 31, 0, 0), (73, 'I', 31, 0, 1), (2301, 'rightarrow', 31, 0, 2), (697, 'idotless', 31, 0, 3), (105, 'i', 31, 1, 0), (73, 'I', 31, 1, 1), (111, 'o', 32, 0, 0), (79, 'O', 32, 0, 1), (248, 'oslash', 32, 0, 2), (216, 'Oslash', 32, 0, 3), (111, 'o', 32, 1, 0), (79, 'O', 32, 1, 1), (112, 'p', 33, 0, 0), (80, 'P', 33, 0, 1),
                                    (254, 'thorn', 33, 0, 2), (222, 'THORN', 33, 0, 3), (112, 'p', 33, 1, 0), (80, 'P', 33, 1, 1), (91, 'bracketleft', 34, 0, 0), (123, 'braceleft', 34, 0, 1), (65111, 'dead_diaeresis', 34, 0, 2), (65112, 'dead_abovering', 34, 0, 3), (91, 'bracketleft', 34, 1, 0), (123, 'braceleft', 34, 1, 1), (93, 'bracketright', 35, 0, 0), (125, 'braceright', 35, 0, 1), (65107, 'dead_tilde', 35, 0, 2), (65108, 'dead_macron', 35, 0, 3), (93, 'bracketright', 35, 1, 0), (125, 'braceright', 35, 1, 1), (65293, 'Return', 36, 0, 0), (65507, 'Control_L', 37, 0, 0), (97, 'a', 38, 0, 0), (65, 'A', 38, 0, 1), (230, 'ae', 38, 0, 2), (198, 'AE', 38, 0, 3), (97, 'a', 38, 1, 0), (65, 'A', 38, 1, 1),
                                    (115, 's', 39, 0, 0), (83, 'S', 39, 0, 1), (223, 'ssharp', 39, 0, 2), (167, 'section', 39, 0, 3), (115, 's', 39, 1, 0), (83, 'S', 39, 1, 1), (100, 'd', 40, 0, 0), (68, 'D', 40, 0, 1), (240, 'eth', 40, 0, 2), (208, 'ETH', 40, 0, 3), (100, 'd', 40, 1, 0), (68, 'D', 40, 1, 1), (102, 'f', 41, 0, 0), (70, 'F', 41, 0, 1), (496, 'dstroke', 41, 0, 2), (170, 'ordfeminine', 41, 0, 3), (102, 'f', 41, 1, 0), (70, 'F', 41, 1, 1), (103, 'g', 42, 0, 0), (71, 'G', 42, 0, 1), (959, 'eng', 42, 0, 2), (957, 'ENG', 42, 0, 3), (103, 'g', 42, 1, 0), (71, 'G', 42, 1, 1), (104, 'h', 43, 0, 0), (72, 'H', 43, 0, 1), (689, 'hstroke', 43, 0, 2), (673, 'Hstroke', 43, 0, 3), (104, 'h', 43, 1, 0),
                                    (72, 'H', 43, 1, 1), (106, 'j', 44, 0, 0), (74, 'J', 44, 0, 1), (65121, 'dead_hook', 44, 0, 2), (65122, 'dead_horn', 44, 0, 3), (106, 'j', 44, 1, 0), (74, 'J', 44, 1, 1), (107, 'k', 45, 0, 0), (75, 'K', 45, 0, 1), (930, 'kra', 45, 0, 2), (38, 'ampersand', 45, 0, 3), (107, 'k', 45, 1, 0), (75, 'K', 45, 1, 1), (108, 'l', 46, 0, 0), (76, 'L', 46, 0, 1), (435, 'lstroke', 46, 0, 2), (419, 'Lstroke', 46, 0, 3), (108, 'l', 46, 1, 0), (76, 'L', 46, 1, 1), (59, 'semicolon', 47, 0, 0), (58, 'colon', 47, 0, 1), (65105, 'dead_acute', 47, 0, 2), (65113, 'dead_doubleacute', 47, 0, 3), (59, 'semicolon', 47, 1, 0), (58, 'colon', 47, 1, 1), (39, 'apostrophe', 48, 0, 0),
                                    (64, 'at', 48, 0, 1), (65106, 'dead_circumflex', 48, 0, 2), (65114, 'dead_caron', 48, 0, 3), (39, 'apostrophe', 48, 1, 0), (34, 'quotedbl', 48, 1, 1), (96, 'grave', 49, 0, 0), (172, 'notsign', 49, 0, 1), (124, 'bar', 49, 0, 2), (124, 'bar', 49, 0, 3), (96, 'grave', 49, 1, 0), (126, 'asciitilde', 49, 1, 1), (65505, 'Shift_L', 50, 0, 0), (35, 'numbersign', 51, 0, 0), (126, 'asciitilde', 51, 0, 1), (65104, 'dead_grave', 51, 0, 2), (65109, 'dead_breve', 51, 0, 3), (92, 'backslash', 51, 1, 0), (124, 'bar', 51, 1, 1), (122, 'z', 52, 0, 0), (90, 'Z', 52, 0, 1), (171, 'guillemotleft', 52, 0, 2), (60, 'less', 52, 0, 3), (122, 'z', 52, 1, 0), (90, 'Z', 52, 1, 1),
                                    (120, 'x', 53, 0, 0), (88, 'X', 53, 0, 1), (187, 'guillemotright', 53, 0, 2), (62, 'greater', 53, 0, 3), (120, 'x', 53, 1, 0), (88, 'X', 53, 1, 1), (99, 'c', 54, 0, 0), (67, 'C', 54, 0, 1), (162, 'cent', 54, 0, 2), (169, 'copyright', 54, 0, 3), (99, 'c', 54, 1, 0), (67, 'C', 54, 1, 1), (118, 'v', 55, 0, 0), (86, 'V', 55, 0, 1), (2770, 'leftdoublequotemark', 55, 0, 2), (2768, 'leftsinglequotemark', 55, 0, 3), (118, 'v', 55, 1, 0), (86, 'V', 55, 1, 1), (98, 'b', 56, 0, 0), (66, 'B', 56, 0, 1), (2771, 'rightdoublequotemark', 56, 0, 2), (2769, 'rightsinglequotemark', 56, 0, 3), (98, 'b', 56, 1, 0), (66, 'B', 56, 1, 1), (110, 'n', 57, 0, 0), (78, 'N', 57, 0, 1),
                                    (110, 'n', 57, 0, 2), (78, 'N', 57, 0, 3), (110, 'n', 57, 1, 0), (78, 'N', 57, 1, 1), (109, 'm', 58, 0, 0), (77, 'M', 58, 0, 1), (181, 'mu', 58, 0, 2), (186, 'masculine', 58, 0, 3), (109, 'm', 58, 1, 0), (77, 'M', 58, 1, 1), (44, 'comma', 59, 0, 0), (60, 'less', 59, 0, 1), (2211, 'horizconnector', 59, 0, 2), (215, 'multiply', 59, 0, 3), (44, 'comma', 59, 1, 0), (60, 'less', 59, 1, 1), (46, 'period', 60, 0, 0), (62, 'greater', 60, 0, 1), (183, 'periodcentered', 60, 0, 2), (247, 'division', 60, 0, 3), (46, 'period', 60, 1, 0), (62, 'greater', 60, 1, 1), (47, 'slash', 61, 0, 0), (63, 'question', 61, 0, 1), (65120, 'dead_belowdot', 61, 0, 2), (65110, 'dead_abovedot', 61, 0, 3),
                                    (47, 'slash', 61, 1, 0), (63, 'question', 61, 1, 1), (65506, 'Shift_R', 62, 0, 0), (65450, 'KP_Multiply', 63, 0, 0), (65450, 'KP_Multiply', 63, 0, 1), (65450, 'KP_Multiply', 63, 0, 2), (65450, 'KP_Multiply', 63, 0, 3), (269024801, 'XF86ClearGrab', 63, 0, 4), (65513, 'Alt_L', 64, 0, 0), (65511, 'Meta_L', 64, 0, 1), (32, 'space', 65, 0, 0), (65509, 'Caps_Lock', 66, 0, 0), (65470, 'F1', 67, 0, 0), (65470, 'F1', 67, 0, 1), (65470, 'F1', 67, 0, 2), (65470, 'F1', 67, 0, 3), (269024769, 'XF86Switch_VT_1', 67, 0, 4), (65471, 'F2', 68, 0, 0), (65471, 'F2', 68, 0, 1), (65471, 'F2', 68, 0, 2), (65471, 'F2', 68, 0, 3), (269024770, 'XF86Switch_VT_2', 68, 0, 4), (65472, 'F3', 69, 0, 0),
                                    (65472, 'F3', 69, 0, 1), (65472, 'F3', 69, 0, 2), (65472, 'F3', 69, 0, 3), (269024771, 'XF86Switch_VT_3', 69, 0, 4), (65473, 'F4', 70, 0, 0), (65473, 'F4', 70, 0, 1), (65473, 'F4', 70, 0, 2), (65473, 'F4', 70, 0, 3), (269024772, 'XF86Switch_VT_4', 70, 0, 4), (65474, 'F5', 71, 0, 0), (65474, 'F5', 71, 0, 1), (65474, 'F5', 71, 0, 2), (65474, 'F5', 71, 0, 3), (269024773, 'XF86Switch_VT_5', 71, 0, 4), (65475, 'F6', 72, 0, 0), (65475, 'F6', 72, 0, 1), (65475, 'F6', 72, 0, 2), (65475, 'F6', 72, 0, 3), (269024774, 'XF86Switch_VT_6', 72, 0, 4), (65476, 'F7', 73, 0, 0), (65476, 'F7', 73, 0, 1), (65476, 'F7', 73, 0, 2), (65476, 'F7', 73, 0, 3), (269024775, 'XF86Switch_VT_7', 73, 0, 4),
                                    (65477, 'F8', 74, 0, 0), (65477, 'F8', 74, 0, 1), (65477, 'F8', 74, 0, 2), (65477, 'F8', 74, 0, 3), (269024776, 'XF86Switch_VT_8', 74, 0, 4), (65478, 'F9', 75, 0, 0), (65478, 'F9', 75, 0, 1), (65478, 'F9', 75, 0, 2), (65478, 'F9', 75, 0, 3), (269024777, 'XF86Switch_VT_9', 75, 0, 4), (65479, 'F10', 76, 0, 0), (65479, 'F10', 76, 0, 1), (65479, 'F10', 76, 0, 2), (65479, 'F10', 76, 0, 3), (269024778, 'XF86Switch_VT_10', 76, 0, 4), (65407, 'Num_Lock', 77, 0, 0), (65300, 'Scroll_Lock', 78, 0, 0), (65429, 'KP_Home', 79, 0, 0), (65463, 'KP_7', 79, 0, 1), (65431, 'KP_Up', 80, 0, 0), (65464, 'KP_8', 80, 0, 1), (65434, 'KP_Page_Up', 81, 0, 0), (65465, 'KP_9', 81, 0, 1),
                                    (65453, 'KP_Subtract', 82, 0, 0), (65453, 'KP_Subtract', 82, 0, 1), (65453, 'KP_Subtract', 82, 0, 2), (65453, 'KP_Subtract', 82, 0, 3), (269024803, 'XF86Prev_VMode', 82, 0, 4), (65430, 'KP_Left', 83, 0, 0), (65460, 'KP_4', 83, 0, 1), (65437, 'KP_Begin', 84, 0, 0), (65461, 'KP_5', 84, 0, 1), (65432, 'KP_Right', 85, 0, 0), (65462, 'KP_6', 85, 0, 1), (65451, 'KP_Add', 86, 0, 0), (65454, 'KP_Decimal', 129, 0, 0), (65454, 'KP_Decimal', 129, 0, 1), (65329, 'Hangul', 130, 0, 0), (65332, 'Hangul_Hanja', 131, 0, 0), (65515, 'Super_L', 133, 0, 0), (65516, 'Super_R', 134, 0, 0), (65383, 'Menu', 135, 0, 0), (65385, 'Cancel', 136, 0, 0), (65382, 'Redo', 137, 0, 0),
                                    (268828528, 'SunProps', 138, 0, 0), (65381, 'Undo', 139, 0, 0), (268828529, 'SunFront', 140, 0, 0), (269025111, 'XF86Copy', 141, 0, 0), (269025131, 'XF86Open', 142, 0, 0), (269025133, 'XF86Paste', 143, 0, 0), (65384, 'Find', 144, 0, 0), (269025112, 'XF86Cut', 145, 0, 0), (65386, 'Help', 146, 0, 0), (269025125, 'XF86MenuKB', 147, 0, 0), (269025053, 'XF86Calculator', 148, 0, 0), (269025071, 'XF86Sleep', 150, 0, 0), (269025067, 'XF86WakeUp', 151, 0, 0), (269025117, 'XF86Explorer', 152, 0, 0), (269025147, 'XF86Send', 153, 0, 0), (269025162, 'XF86Xfer', 155, 0, 0), (269025089, 'XF86Launch1', 156, 0, 0), (269025090, 'XF86Launch2', 157, 0, 0), (269025070, 'XF86WWW', 158, 0, 0),
                                    (269025114, 'XF86DOS', 159, 0, 0), (269025069, 'XF86ScreenSaver', 160, 0, 0), (269025140, 'XF86RotateWindows', 162, 0, 0), (269025049, 'XF86Mail', 163, 0, 0), (269025072, 'XF86Favorites', 164, 0, 0), (269025075, 'XF86MyComputer', 165, 0, 0), (269025062, 'XF86Back', 166, 0, 0), (269025063, 'XF86Forward', 167, 0, 0), (269025068, 'XF86Eject', 169, 0, 0), (269025068, 'XF86Eject', 170, 0, 0), (269025068, 'XF86Eject', 170, 0, 1), (269025047, 'XF86AudioNext', 171, 0, 0), (269025044, 'XF86AudioPlay', 172, 0, 0), (269025073, 'XF86AudioPause', 172, 0, 1), (269025046, 'XF86AudioPrev', 173, 0, 0), (269025045, 'XF86AudioStop', 174, 0, 0), (269025068, 'XF86Eject', 174, 0, 1),
                                    (269025052, 'XF86AudioRecord', 175, 0, 0), (269025086, 'XF86AudioRewind', 176, 0, 0), (269025134, 'XF86Phone', 177, 0, 0), (269025153, 'XF86Tools', 179, 0, 0), (269025048, 'XF86HomePage', 180, 0, 0), (269025139, 'XF86Reload', 181, 0, 0), (269025110, 'XF86Close', 182, 0, 0), (269025144, 'XF86ScrollUp', 185, 0, 0), (269025145, 'XF86ScrollDown', 186, 0, 0), (40, 'parenleft', 187, 0, 0), (41, 'parenright', 188, 0, 0), (269025128, 'XF86New', 189, 0, 0), (65382, 'Redo', 190, 0, 0), (269025153, 'XF86Tools', 191, 0, 0), (269025093, 'XF86Launch5', 192, 0, 0), (269025094, 'XF86Launch6', 193, 0, 0), (269025095, 'XF86Launch7', 194, 0, 0), (269025096, 'XF86Launch8', 195, 0, 0),
                                    (269025097, 'XF86Launch9', 196, 0, 0), (269025193, 'XF86TouchpadToggle', 199, 0, 0), (269025200, 'XF86TouchpadOn', 200, 0, 0), (269025201, 'XF86TouchpadOff', 201, 0, 0), (65406, 'Mode_switch', 203, 0, 0), (65513, 'Alt_L', 204, 0, 1), (65511, 'Meta_L', 205, 0, 1), (65515, 'Super_L', 206, 0, 1), (65517, 'Hyper_L', 207, 0, 1), (269025044, 'XF86AudioPlay', 208, 0, 0), (269025073, 'XF86AudioPause', 209, 0, 0), (269025091, 'XF86Launch3', 210, 0, 0), (269025092, 'XF86Launch4', 211, 0, 0), (269025099, 'XF86LaunchB', 212, 0, 0), (269025191, 'XF86Suspend', 213, 0, 0), (269025110, 'XF86Close', 214, 0, 0), (269025044, 'XF86AudioPlay', 215, 0, 0), (269025175, 'XF86AudioForward', 216, 0, 0),
                                    (65377, 'Print', 218, 0, 0), (269025167, 'XF86WebCam', 220, 0, 0), (269025049, 'XF86Mail', 223, 0, 0), (269025166, 'XF86Messenger', 224, 0, 0), (269025051, 'XF86Search', 225, 0, 0), (269025119, 'XF86Go', 226, 0, 0), (269025084, 'XF86Finance', 227, 0, 0), (269025118, 'XF86Game', 228, 0, 0), (269025078, 'XF86Shop', 229, 0, 0), (65385, 'Cancel', 231, 0, 0), (269025027, 'XF86MonBrightnessDown', 232, 0, 0), (269025026, 'XF86MonBrightnessUp', 233, 0, 0), (269025074, 'XF86AudioMedia', 234, 0, 0), (269025113, 'XF86Display', 235, 0, 0), (269025028, 'XF86KbdLightOnOff', 236, 0, 0), (269025030, 'XF86KbdBrightnessDown', 237, 0, 0), (269025029, 'XF86KbdBrightnessUp', 238, 0, 0),
                                    (269025147, 'XF86Send', 239, 0, 0), (269025138, 'XF86Reply', 240, 0, 0), (269025168, 'XF86MailForward', 241, 0, 0), (269025143, 'XF86Save', 242, 0, 0), (269025115, 'XF86Documents', 243, 0, 0), (269025171, 'XF86Battery', 244, 0, 0), (269025172, 'XF86Bluetooth', 245, 0, 0), (269025173, 'XF86WLAN', 246, 0, 0)],
                'rgb24zlib': True, 'raw_window_icons': True, 'clipboard.set_enabled': True, 'system_tray': True, 'hostname': 'desktop', 'namespace': True, 'key_repeat': (500L, 30L),
                'encoding.generic': True, 'version': '0.11.0', 'build.bit': '64bit', 'compressible_cursors': True, 'platform.linux_distribution': ('Fedora', '19', 'Schr\xc3\xb6dinger\xe2\x80\x99s Cat'),
                'encoding.rgb_lz4': True, 'clipboard.notifications': True, 'sound.encoders': ['mp3', 'wavpack', 'wav', 'flac', 'speex'], 'encoding.avcodec.version': (54, 92, 100), 'encoding.x264.I420.profile': 'high10', 'notify-startup-complete': True, 'bencode': True, 'xkbmap_mod_pointermissing': [], 'encoding.webp.version': '0.2.2', 'server-window-resize': True, 'xsettings-tuple': True, 'encoding.h264.I420.profile': 'high10',
                'clipboard.selections': ['CLIPBOARD', 'PRIMARY', 'SECONDARY'], 'pygtk.version': (2, 24, 0), 'encoding.video_reinit': True, 'build.date': '2014-01-01',
                'xkbmap_x11_keycodes': {9: ['Escape', '', 'Escape'], 10: ['1', 'exclam', '1', 'exclam', 'onesuperior', 'exclamdown'], 11: ['2', 'quotedbl', '2', 'at', 'twosuperior', 'oneeighth'], 12: ['3', 'sterling', '3', 'numbersign', 'threesuperior', 'sterling'], 13: ['4', 'dollar', '4', 'dollar', 'EuroSign', 'onequarter'], 14: ['5', 'percent', '5', 'percent', 'onehalf', 'threeeighths'], 15: ['6', 'asciicircum', '6', 'asciicircum', 'threequarters', 'fiveeighths'], 16: ['7', 'ampersand', '7', 'ampersand', 'braceleft', 'seveneighths'], 17: ['8', 'asterisk', '8', 'asterisk', 'bracketleft', 'trademark'],
                                        18: ['9', 'parenleft', '9', 'parenleft', 'bracketright', 'plusminus'], 19: ['0', 'parenright', '0', 'parenright', 'braceright', 'degree'], 20: ['minus', 'underscore', 'minus', 'underscore', 'backslash', 'questiondown'], 21: ['equal', 'plus', 'equal', 'plus', 'dead_cedilla', 'dead_ogonek'], 22: ['BackSpace', 'BackSpace', 'BackSpace', 'BackSpace'], 23: ['Tab', 'ISO_Left_Tab', 'Tab', 'ISO_Left_Tab'], 24: ['q', 'Q', 'q', 'Q', 'at', 'Greek_OMEGA'], 25: ['w', 'W', 'w', 'W', 'lstroke', 'Lstroke'], 26: ['e', 'E', 'e', 'E', 'e', 'E'], 27: ['r', 'R', 'r', 'R', 'paragraph', 'registered'],
                                        28: ['t', 'T', 't', 'T', 'tslash', 'Tslash'], 29: ['y', 'Y', 'y', 'Y', 'leftarrow', 'yen'], 30: ['u', 'U', 'u', 'U', 'downarrow', 'uparrow'], 31: ['i', 'I', 'i', 'I', 'rightarrow', 'idotless'],
                                        32: ['o', 'O', 'o', 'O', 'oslash', 'Oslash'], 33: ['p', 'P', 'p', 'P', 'thorn', 'THORN'], 34: ['bracketleft', 'braceleft', 'bracketleft', 'braceleft', 'dead_diaeresis', 'dead_abovering'], 35: ['bracketright', 'braceright', 'bracketright', 'braceright', 'dead_tilde', 'dead_macron'], 36: ['Return', '', 'Return'], 37: ['Control_L', '', 'Control_L'], 38: ['a', 'A', 'a', 'A', 'ae', 'AE'], 39: ['s', 'S', 's', 'S', 'ssharp', 'section'], 40: ['d', 'D', 'd', 'D', 'eth', 'ETH'], 41: ['f', 'F', 'f', 'F', 'dstroke', 'ordfeminine'], 42: ['g', 'G', 'g', 'G', 'eng', 'ENG'], 43: ['h', 'H', 'h', 'H', 'hstroke', 'Hstroke'], 44: ['j', 'J', 'j', 'J', 'dead_hook', 'dead_horn'], 45: ['k', 'K', 'k', 'K', 'kra', 'ampersand'], 46: ['l', 'L', 'l', 'L', 'lstroke', 'Lstroke'],
                                        47: ['semicolon', 'colon', 'semicolon', 'colon', 'dead_acute', 'dead_doubleacute'], 48: ['apostrophe', 'at', 'apostrophe', 'quotedbl', 'dead_circumflex', 'dead_caron'],
                                        49: ['grave', 'notsign', 'grave', 'asciitilde', 'bar', 'bar'], 50: ['Shift_L', '', 'Shift_L'], 51: ['numbersign', 'asciitilde', 'backslash', 'bar', 'dead_grave', 'dead_breve'], 52: ['z', 'Z', 'z', 'Z', 'guillemotleft', 'less'], 53: ['x', 'X', 'x', 'X', 'guillemotright', 'greater'], 54: ['c', 'C', 'c', 'C', 'cent', 'copyright'], 55: ['v', 'V', 'v', 'V', 'leftdoublequotemark', 'leftsinglequotemark'], 56: ['b', 'B', 'b', 'B', 'rightdoublequotemark', 'rightsinglequotemark'], 57: ['n', 'N', 'n', 'N', 'n', 'N'], 58: ['m', 'M', 'm', 'M', 'mu', 'masculine'], 59: ['comma', 'less', 'comma', 'less', 'horizconnector', 'multiply'], 60: ['period', 'greater', 'period', 'greater', 'periodcentered', 'division'], 61: ['slash', 'question', 'slash', 'question', 'dead_belowdot', 'dead_abovedot'],
                                        62: ['Shift_R', '', 'Shift_R'], 63: ['KP_Multiply', 'KP_Multiply', 'KP_Multiply', 'KP_Multiply', 'KP_Multiply', 'KP_Multiply', 'XF86ClearGrab', 'KP_Multiply', 'KP_Multiply', 'XF86ClearGrab'], 64: ['Alt_L', 'Meta_L', 'Alt_L', 'Meta_L'], 65: ['space', '', 'space'], 66: ['Caps_Lock', '', 'Caps_Lock'], 67: ['F1', 'F1', 'F1', 'F1', 'F1', 'F1', 'XF86Switch_VT_1', 'F1', 'F1', 'XF86Switch_VT_1'], 68: ['F2', 'F2', 'F2', 'F2', 'F2', 'F2', 'XF86Switch_VT_2', 'F2', 'F2', 'XF86Switch_VT_2'], 69: ['F3', 'F3', 'F3', 'F3', 'F3', 'F3', 'XF86Switch_VT_3', 'F3', 'F3', 'XF86Switch_VT_3'], 70: ['F4', 'F4', 'F4', 'F4', 'F4', 'F4', 'XF86Switch_VT_4', 'F4', 'F4', 'XF86Switch_VT_4'], 71: ['F5', 'F5', 'F5', 'F5', 'F5', 'F5', 'XF86Switch_VT_5', 'F5', 'F5', 'XF86Switch_VT_5'],
                                        72: ['F6', 'F6', 'F6', 'F6', 'F6', 'F6', 'XF86Switch_VT_6', 'F6', 'F6', 'XF86Switch_VT_6'], 73: ['F7', 'F7', 'F7', 'F7', 'F7', 'F7', 'XF86Switch_VT_7', 'F7', 'F7', 'XF86Switch_VT_7'], 74: ['F8', 'F8', 'F8', 'F8', 'F8', 'F8', 'XF86Switch_VT_8', 'F8', 'F8', 'XF86Switch_VT_8'], 75: ['F9', 'F9', 'F9', 'F9', 'F9', 'F9', 'XF86Switch_VT_9', 'F9', 'F9', 'XF86Switch_VT_9'], 76: ['F10', 'F10', 'F10', 'F10', 'F10', 'F10', 'XF86Switch_VT_10', 'F10', 'F10', 'XF86Switch_VT_10'], 77: ['Num_Lock', '', 'Num_Lock'], 78: ['Scroll_Lock', '', 'Scroll_Lock'], 79: ['KP_Home', 'KP_7', 'KP_Home', 'KP_7'], 80: ['KP_Up', 'KP_8', 'KP_Up', 'KP_8'], 81: ['KP_Prior', 'KP_9', 'KP_Prior', 'KP_9'],
                                        82: ['KP_Subtract', 'KP_Subtract', 'KP_Subtract', 'KP_Subtract', 'KP_Subtract', 'KP_Subtract', 'XF86Prev_VMode', 'KP_Subtract', 'KP_Subtract', 'XF86Prev_VMode'], 83: ['KP_Left', 'KP_4', 'KP_Left', 'KP_4'], 84: ['KP_Begin', 'KP_5', 'KP_Begin', 'KP_5'], 85: ['KP_Right', 'KP_6', 'KP_Right', 'KP_6'], 86: ['KP_Add', 'KP_Add', 'KP_Add', 'KP_Add', 'KP_Add', 'KP_Add', 'XF86Next_VMode', 'KP_Add', 'KP_Add', 'XF86Next_VMode'], 87: ['KP_End', 'KP_1', 'KP_End', 'KP_1'], 88: ['KP_Down', 'KP_2', 'KP_Down', 'KP_2'], 89: ['KP_Next', 'KP_3', 'KP_Next', 'KP_3'], 90: ['KP_Insert', 'KP_0', 'KP_Insert', 'KP_0'], 91: ['KP_Delete', 'KP_Decimal', 'KP_Delete', 'KP_Decimal'], 92: ['ISO_Level3_Shift', '', 'ISO_Level3_Shift'], 94: ['backslash', 'bar', 'backslash', 'bar', 'bar', 'brokenbar', 'bar', 'brokenbar'],
                                        95: ['F11', 'F11', 'F11', 'F11', 'F11', 'F11', 'XF86Switch_VT_11', 'F11', 'F11', 'XF86Switch_VT_11'], 96: ['F12', 'F12', 'F12', 'F12', 'F12', 'F12', 'XF86Switch_VT_12', 'F12', 'F12', 'XF86Switch_VT_12'], 98: ['Katakana', '', 'Katakana'], 99: ['Hiragana', '', 'Hiragana'], 100: ['Henkan_Mode', '', 'Henkan_Mode'], 101: ['Hiragana_Katakana', '', 'Hiragana_Katakana'], 102: ['Muhenkan', '', 'Muhenkan'], 104: ['KP_Enter', '', 'KP_Enter'], 105: ['Control_R', '', 'Control_R'], 106: ['KP_Divide', 'KP_Divide', 'KP_Divide', 'KP_Divide', 'KP_Divide', 'KP_Divide', 'XF86Ungrab', 'KP_Divide', 'KP_Divide', 'XF86Ungrab'], 107: ['Print', 'Sys_Req', 'Print', 'Sys_Req'], 108: ['ISO_Level3_Shift', 'Multi_key', 'ISO_Level3_Shift', 'Multi_key'], 109: ['Linefeed', '', 'Linefeed'], 110: ['Home', '', 'Home'],
                                        111: ['Up', '', 'Up'], 112: ['Prior', '', 'Prior'], 113: ['Left', '', 'Left'], 114: ['Right', '', 'Right'], 115: ['End', '', 'End'], 116: ['Down', '', 'Down'], 117: ['Next', '', 'Next'], 118: ['Insert', '', 'Insert'], 119: ['Delete', '', 'Delete'], 121: ['XF86AudioMute', '', 'XF86AudioMute'], 122: ['XF86AudioLowerVolume', '', 'XF86AudioLowerVolume'], 123: ['XF86AudioRaiseVolume', '', 'XF86AudioRaiseVolume'], 124: ['XF86PowerOff', '', 'XF86PowerOff'], 125: ['KP_Equal', '', 'KP_Equal'], 126: ['plusminus', '', 'plusminus'], 127: ['Pause', 'Break', 'Pause', 'Break'], 128: ['XF86LaunchA', '', 'XF86LaunchA'], 129: ['KP_Decimal', 'KP_Decimal', 'KP_Decimal', 'KP_Decimal'], 130: ['Hangul', '', 'Hangul'], 131: ['Hangul_Hanja', '', 'Hangul_Hanja'], 133: ['Super_L', '', 'Super_L'], 134: ['Super_R', '', 'Super_R'],
                                        135: ['Menu', '', 'Menu'], 136: ['Cancel', '', 'Cancel'], 137: ['Redo', '', 'Redo'], 138: ['SunProps', '', 'SunProps'], 139: ['Undo', '', 'Undo'], 140: ['SunFront', '', 'SunFront'], 141: ['XF86Copy', '', 'XF86Copy'], 142: ['XF86Open', '', 'XF86Open'], 143: ['XF86Paste', '', 'XF86Paste'], 144: ['Find', '', 'Find'], 145: ['XF86Cut', '', 'XF86Cut'], 146: ['Help', '', 'Help'], 147: ['XF86MenuKB', '', 'XF86MenuKB'], 148: ['XF86Calculator', '', 'XF86Calculator'], 150: ['XF86Sleep', '', 'XF86Sleep'], 151: ['XF86WakeUp', '', 'XF86WakeUp'], 152: ['XF86Explorer', '', 'XF86Explorer'], 153: ['XF86Send', '', 'XF86Send'], 155: ['XF86Xfer', '', 'XF86Xfer'], 156: ['XF86Launch1', '', 'XF86Launch1'],
                                        157: ['XF86Launch2', '', 'XF86Launch2'], 158: ['XF86WWW', '', 'XF86WWW'], 159: ['XF86DOS', '', 'XF86DOS'], 160: ['XF86ScreenSaver', '', 'XF86ScreenSaver'], 162: ['XF86RotateWindows', '', 'XF86RotateWindows'], 163: ['XF86Mail', '', 'XF86Mail'], 164: ['XF86Favorites', '', 'XF86Favorites'], 165: ['XF86MyComputer', '', 'XF86MyComputer'], 166: ['XF86Back', '', 'XF86Back'], 167: ['XF86Forward', '', 'XF86Forward'], 169: ['XF86Eject', '', 'XF86Eject'], 170: ['XF86Eject', 'XF86Eject', 'XF86Eject', 'XF86Eject'], 171: ['XF86AudioNext', '', 'XF86AudioNext'], 172: ['XF86AudioPlay', 'XF86AudioPause', 'XF86AudioPlay', 'XF86AudioPause'], 173: ['XF86AudioPrev', '', 'XF86AudioPrev'], 174: ['XF86AudioStop', 'XF86Eject', 'XF86AudioStop', 'XF86Eject'], 175: ['XF86AudioRecord', '', 'XF86AudioRecord'],
                                        176: ['XF86AudioRewind', '', 'XF86AudioRewind'], 177: ['XF86Phone', '', 'XF86Phone'], 179: ['XF86Tools', '', 'XF86Tools'], 180: ['XF86HomePage', '', 'XF86HomePage'], 181: ['XF86Reload', '', 'XF86Reload'], 182: ['XF86Close', '', 'XF86Close'], 185: ['XF86ScrollUp', '', 'XF86ScrollUp'], 186: ['XF86ScrollDown', '', 'XF86ScrollDown'], 187: ['parenleft', '', 'parenleft'], 188: ['parenright', '', 'parenright'], 189: ['XF86New', '', 'XF86New'], 190: ['Redo', '', 'Redo'], 191: ['XF86Tools', '', 'XF86Tools'], 192: ['XF86Launch5', '', 'XF86Launch5'], 193: ['XF86Launch6', '', 'XF86Launch6'], 194: ['XF86Launch7', '', 'XF86Launch7'], 195: ['XF86Launch8', '', 'XF86Launch8'], 196: ['XF86Launch9', '', 'XF86Launch9'], 199: ['XF86TouchpadToggle', '', 'XF86TouchpadToggle'], 200: ['XF86TouchpadOn', '', 'XF86TouchpadOn'],
                                        201: ['XF86TouchpadOff', '', 'XF86TouchpadOff'], 203: ['Mode_switch', '', 'Mode_switch'], 204: ['', 'Alt_L', '', 'Alt_L'], 205: ['', 'Meta_L', '', 'Meta_L'], 206: ['', 'Super_L', '', 'Super_L'], 207: ['', 'Hyper_L', '', 'Hyper_L'], 208: ['XF86AudioPlay', '', 'XF86AudioPlay'], 209: ['XF86AudioPause', '', 'XF86AudioPause'], 210: ['XF86Launch3', '', 'XF86Launch3'], 211: ['XF86Launch4', '', 'XF86Launch4'], 212: ['XF86LaunchB', '', 'XF86LaunchB'], 213: ['XF86Suspend', '', 'XF86Suspend'], 214: ['XF86Close', '', 'XF86Close'], 215: ['XF86AudioPlay', '', 'XF86AudioPlay'], 216: ['XF86AudioForward', '', 'XF86AudioForward'], 218: ['Print', '', 'Print'], 220: ['XF86WebCam', '', 'XF86WebCam'], 223: ['XF86Mail', '', 'XF86Mail'], 224: ['XF86Messenger', '', 'XF86Messenger'], 225: ['XF86Search', '', 'XF86Search'],
                                        226: ['XF86Go', '', 'XF86Go'], 227: ['XF86Finance', '', 'XF86Finance'], 228: ['XF86Game', '', 'XF86Game'], 229: ['XF86Shop', '', 'XF86Shop'], 231: ['Cancel', '', 'Cancel'], 232: ['XF86MonBrightnessDown', '', 'XF86MonBrightnessDown'], 233: ['XF86MonBrightnessUp', '', 'XF86MonBrightnessUp'], 234: ['XF86AudioMedia', '', 'XF86AudioMedia'], 235: ['XF86Display', '', 'XF86Display'], 236: ['XF86KbdLightOnOff', '', 'XF86KbdLightOnOff'], 237: ['XF86KbdBrightnessDown', '', 'XF86KbdBrightnessDown'], 238: ['XF86KbdBrightnessUp', '', 'XF86KbdBrightnessUp'], 239: ['XF86Send', '', 'XF86Send'], 240: ['XF86Reply', '', 'XF86Reply'], 241: ['XF86MailForward', '', 'XF86MailForward'], 242: ['XF86Save', '', 'XF86Save'], 243: ['XF86Documents', '', 'XF86Documents'], 244: ['XF86Battery', '', 'XF86Battery'], 245: ['XF86Bluetooth', '', 'XF86Bluetooth'], 246: ['XF86WLAN', '', 'XF86WLAN']
                                        },
                'encoding.initial_quality': 50, 'xkbmap_mod_managed': [], 'notifications': True, 'windows': True, 'encoding.min-quality': 50, 'clipboard.want_targets': False,
                'encodings.core': ['h264', 'vp8', 'webp', 'png', 'png/P', 'png/L', 'rgb24', 'jpeg', 'rgb32'], 'raw_packets': True, 'compression_level': 1, 'dpi': 96, 'build.local_modifications': '0', 'platform.processor': 'x86_64', 'sound.gst.version': (0, 10, 36), 'encoding.swscale.version': (2, 2, 100), 'encoding.min-speed': 0
               }]

large_hello_output = "l5:hellod13:named_cursorsi1e3:lz4i1e20:encoding.vpx.version6:v1.2.013:sound.receivei1e6:digestl4:hmac3:xore7:aliasesd11:lost-windowi6e4:belli7e12:desktop_sizei8e21:new-override-redirecti9e9:ping_echoi5e10:new-windowi10e15:connection-losti11e16:startup-completei12e13:info-responsei2e10:disconnecti13e4:pingi3e14:window-resizedi14e11:set_deflatei15e9:rpc-replyi16e11:window-iconi17e4:drawi18e12:notify_closei19e10:sound-datai1e12:raise-windowi20e15:window-metadatai21e21:set-clipboard-enabledi22e27:configure-override-redirecti23e9:challengei24e6:cursori25e11:notify_showi26e9:gibberishi27e8:new-trayi28e5:helloi29ee17:platform.platform71:Linux-3.12.5-200.fc19.x86_64-x86_64-with-fedora-19-Schrödinger’s_Cat22:clipboard.want_targetsi0e4:uuid40:8643124ce701ee68dbb6b7a8c4eb13a5f640949423:encoding.opencl.version6:2013.115:bencode.versionl6:Cythoni0ei11ee8:platform6:linux219:sound.server_driveni1e"+ \
"7:version6:0.11.021:encodings.rgb_formatsl3:RGB4:RGBAe19:chunked_compressioni1e13:keyboard_synci1e10:sound.sendi1e12:screen_sizesll4::0.0i1920ei1080ei508ei286ell7:DVI-I-1i0ei0ei1920ei1080ei531ei299eeei0ei0ei1920ei1055eee14:xkbmap_variant0:29:encoding.h264.YUV420P.profile6:high1021:encoding.transparencyi1e13:xkbmap_layout0:18:encoding.csc_atomsi1e26:encoding.h264.I420.profile6:high1012:window.raisei1e4:name14:Antoine Martin23:encoding.client_optionsi1e23:encoding.supports_deltal3:png5:rgb245:rgb32e11:client_type11:Python/Gtk223:sound.pulseaudio.server66:{7725dfc225d14958a625ddaaaea5962b}unix:/run/user/1000/pulse/native18:encoding.csc_modesl7:YUV420P7:YUV422P7:YUV444P4:BGRA4:BGRXe8:build.by4:root10:machine_id32:7725dfc225d14958a625ddaaaea5962b14:python.versionli2ei7ei5ee11:server_uuid0:9:encodingsl4:h2643:vp84:webp3:png5:png/P5:png/L3:rgb4:jpege9:rgb24zlibi1e10:key_repeatli500ei30ee16:encoding.generici1e"+ \
"9:clipboardi1e7:bencodei1e26:encoding.x264.I420.profile6:high1023:notify-startup-completei1e25:xkbmap_mod_pointermissingle21:encoding.webp.version5:0.2.220:server-window-resizei1e13:pygtk.versionli2ei24ei0ee19:xkbmap_x11_keycodesdi9el6:Escape0:6:Escapeei10el1:16:exclam1:16:exclam11:onesuperior10:exclamdownei11el1:28:quotedbl1:22:at11:twosuperior9:oneeighthei12el1:38:sterling1:310:numbersign13:threesuperior8:sterlingei13el1:46:dollar1:46:dollar8:EuroSign10:onequarterei14el1:57:percent1:57:percent7:onehalf12:threeeighthsei15el1:611:asciicircum1:611:asciicircum13:threequarters11:fiveeighthsei16el1:79:ampersand1:79:ampersand9:braceleft12:seveneighthsei17el1:88:asterisk1:88:asterisk11:bracketleft9:trademarkei18el1:99:parenleft1:99:parenleft12:bracketright9:plusminusei19el1:010:parenright1:010:parenright10:braceright6:degreeei20el5:minus10:underscore5:minus10:underscore9:backslash12:questiondownei21e"+\
"l5:equal4:plus5:equal4:plus12:dead_cedilla11:dead_ogonekei22el9:BackSpace9:BackSpace9:BackSpace9:BackSpaceei23el3:Tab12:ISO_Left_Tab3:Tab12:ISO_Left_Tabei24el1:q1:Q1:q1:Q2:at11:Greek_OMEGAei25el1:w1:W1:w1:W7:lstroke7:Lstrokeei26el1:e1:E1:e1:E1:e1:Eei27el1:r1:R1:r1:R9:paragraph10:registeredei28el1:t1:T1:t1:T6:tslash6:Tslashei29el1:y1:Y1:y1:Y9:leftarrow3:yenei30el1:u1:U1:u1:U9:downarrow7:uparrowei31el1:i1:I1:i1:I10:rightarrow8:idotlessei32el1:o1:O1:o1:O6:oslash6:Oslashei33el1:p1:P1:p1:P5:thorn5:THORNei34el11:bracketleft9:braceleft11:bracketleft9:braceleft14:dead_diaeresis14:dead_aboveringei35el12:bracketright10:braceright12:bracketright10:braceright10:dead_tilde11:dead_macronei36el6:Return0:6:Returnei37el9:Control_L0:9:Control_Lei38el1:a1:A1:a1:A2:ae2:AEei39el1:s1:S1:s1:S6:ssharp7:sectionei40el1:d1:D1:d1:D3:eth3:ETHei41el1:f1:F1:f1:F7:dstroke11:ordfeminineei42e"+\
"l1:g1:G1:g1:G3:eng3:ENGei43el1:h1:H1:h1:H7:hstroke7:Hstrokeei44el1:j1:J1:j1:J9:dead_hook9:dead_hornei45el1:k1:K1:k1:K3:kra9:ampersandei46el1:l1:L1:l1:L7:lstroke7:Lstrokeei47el9:semicolon5:colon9:semicolon5:colon10:dead_acute16:dead_doubleacuteei48el10:apostrophe2:at10:apostrophe8:quotedbl15:dead_circumflex10:dead_caronei49el5:grave7:notsign5:grave10:asciitilde3:bar3:barei50el7:Shift_L0:7:Shift_Lei51el10:numbersign10:asciitilde9:backslash3:bar10:dead_grave10:dead_breveei52el1:z1:Z1:z1:Z13:guillemotleft4:lessei53el1:x1:X1:x1:X14:guillemotright7:greaterei54el1:c1:C1:c1:C4:cent9:copyrightei55el1:v1:V1:v1:V19:leftdoublequotemark19:leftsinglequotemarkei56el1:b1:B1:b1:B20:rightdoublequotemark20:rightsinglequotemarkei57el1:n1:N1:n1:N1:n1:Nei58el1:m1:M1:m1:M2:mu9:masculineei59el5:comma4:less5:comma4:less14:horizconnector8:multiplyei60el6:period7:greater6:period7:greater"+\
"14:periodcentered8:divisionei61el5:slash8:question5:slash8:question13:dead_belowdot13:dead_abovedotei62el7:Shift_R0:7:Shift_Rei63el11:KP_Multiply11:KP_Multiply11:KP_Multiply11:KP_Multiply11:KP_Multiply11:KP_Multiply13:XF86ClearGrab11:KP_Multiply11:KP_Multiply13:XF86ClearGrabei64el5:Alt_L6:Meta_L5:Alt_L6:Meta_Lei65el5:space0:5:spaceei66el9:Caps_Lock0:9:Caps_Lockei67el2:F12:F12:F12:F12:F12:F115:XF86Switch_VT_12:F12:F115:XF86Switch_VT_1ei68el2:F22:F22:F22:F22:F22:F215:XF86Switch_VT_22:F22:F215:XF86Switch_VT_2ei69el2:F32:F32:F32:F32:F32:F315:XF86Switch_VT_32:F32:F315:XF86Switch_VT_3ei70el2:F42:F42:F42:F42:F42:F415:XF86Switch_VT_42:F42:F415:XF86Switch_VT_4ei71el2:F52:F52:F52:F52:F52:F515:XF86Switch_VT_52:F52:F515:XF86Switch_VT_5ei72el2:F62:F62:F62:F62:F62:F615:XF86Switch_VT_62:F62:F615:XF86Switch_VT_6ei73el2:F72:F72:F72:F72:F72:F715:XF86Switch_VT_72:F72:F7"+\
"15:XF86Switch_VT_7ei74el2:F82:F82:F82:F82:F82:F815:XF86Switch_VT_82:F82:F815:XF86Switch_VT_8ei75el2:F92:F92:F92:F92:F92:F915:XF86Switch_VT_92:F92:F915:XF86Switch_VT_9ei76el3:F103:F103:F103:F103:F103:F1016:XF86Switch_VT_103:F103:F1016:XF86Switch_VT_10ei77el8:Num_Lock0:8:Num_Lockei78el11:Scroll_Lock0:11:Scroll_Lockei79el7:KP_Home4:KP_77:KP_Home4:KP_7ei80el5:KP_Up4:KP_85:KP_Up4:KP_8ei81el8:KP_Prior4:KP_98:KP_Prior4:KP_9ei82el11:KP_Subtract11:KP_Subtract11:KP_Subtract11:KP_Subtract11:KP_Subtract11:KP_Subtract14:XF86Prev_VMode11:KP_Subtract11:KP_Subtract14:XF86Prev_VModeei83el7:KP_Left4:KP_47:KP_Left4:KP_4ei84el8:KP_Begin4:KP_58:KP_Begin4:KP_5ei85el8:KP_Right4:KP_68:KP_Right4:KP_6ei86el6:KP_Add6:KP_Add6:KP_Add6:KP_Add6:KP_Add6:KP_Add14:XF86Next_VMode6:KP_Add6:KP_Add14:XF86Next_VModeei87el6:KP_End4:KP_16:KP_End4:KP_1ei88el7:KP_Down4:KP_27:KP_Down4:KP_2ei89el7:KP_Next"+\
"4:KP_37:KP_Next4:KP_3ei90el9:KP_Insert4:KP_09:KP_Insert4:KP_0ei91el9:KP_Delete10:KP_Decimal9:KP_Delete10:KP_Decimalei92el16:ISO_Level3_Shift0:16:ISO_Level3_Shiftei94el9:backslash3:bar9:backslash3:bar3:bar9:brokenbar3:bar9:brokenbarei95el3:F113:F113:F113:F113:F113:F1116:XF86Switch_VT_113:F113:F1116:XF86Switch_VT_11ei96el3:F123:F123:F123:F123:F123:F1216:XF86Switch_VT_123:F123:F1216:XF86Switch_VT_12ei98el8:Katakana0:8:Katakanaei99el8:Hiragana0:8:Hiraganaei100el11:Henkan_Mode0:11:Henkan_Modeei101el17:Hiragana_Katakana0:17:Hiragana_Katakanaei102el8:Muhenkan0:8:Muhenkanei104el8:KP_Enter0:8:KP_Enterei105el9:Control_R0:9:Control_Rei106el9:KP_Divide9:KP_Divide9:KP_Divide9:KP_Divide9:KP_Divide9:KP_Divide10:XF86Ungrab9:KP_Divide9:KP_Divide10:XF86Ungrabei107el5:Print7:Sys_Req5:Print7:Sys_Reqei108el16:ISO_Level3_Shift9:Multi_key16:ISO_Level3_Shift9:Multi_keyei109e"+\
"l8:Linefeed0:8:Linefeedei110el4:Home0:4:Homeei111el2:Up0:2:Upei112el5:Prior0:5:Priorei113el4:Left0:4:Leftei114el5:Right0:5:Rightei115el3:End0:3:Endei116el4:Down0:4:Downei117el4:Next0:4:Nextei118el6:Insert0:6:Insertei119el6:Delete0:6:Deleteei121el13:XF86AudioMute0:13:XF86AudioMuteei122el20:XF86AudioLowerVolume0:20:XF86AudioLowerVolumeei123el20:XF86AudioRaiseVolume0:20:XF86AudioRaiseVolumeei124el12:XF86PowerOff0:12:XF86PowerOffei125el8:KP_Equal0:8:KP_Equalei126el9:plusminus0:9:plusminusei127el5:Pause5:Break5:Pause5:Breakei128el11:XF86LaunchA0:11:XF86LaunchAei129el10:KP_Decimal10:KP_Decimal10:KP_Decimal10:KP_Decimalei130el6:Hangul0:6:Hangulei131el12:Hangul_Hanja0:12:Hangul_Hanjaei133el7:Super_L0:7:Super_Lei134el7:Super_R0:7:Super_Rei135el4:Menu0:4:Menuei136el6:Cancel0:6:Cancelei137el4:Redo0:4:Redoei138el8:SunProps0:8:SunPropsei139el4:Undo0:4:Undoei140e"+\
"l8:SunFront0:8:SunFrontei141el8:XF86Copy0:8:XF86Copyei142el8:XF86Open0:8:XF86Openei143el9:XF86Paste0:9:XF86Pasteei144el4:Find0:4:Findei145el7:XF86Cut0:7:XF86Cutei146el4:Help0:4:Helpei147el10:XF86MenuKB0:10:XF86MenuKBei148el14:XF86Calculator0:14:XF86Calculatorei150el9:XF86Sleep0:9:XF86Sleepei151el10:XF86WakeUp0:10:XF86WakeUpei152el12:XF86Explorer0:12:XF86Explorerei153el8:XF86Send0:8:XF86Sendei155el8:XF86Xfer0:8:XF86Xferei156el11:XF86Launch10:11:XF86Launch1ei157el11:XF86Launch20:11:XF86Launch2ei158el7:XF86WWW0:7:XF86WWWei159el7:XF86DOS0:7:XF86DOSei160el15:XF86ScreenSaver0:15:XF86ScreenSaverei162el17:XF86RotateWindows0:17:XF86RotateWindowsei163el8:XF86Mail0:8:XF86Mailei164el13:XF86Favorites0:13:XF86Favoritesei165el14:XF86MyComputer0:14:XF86MyComputerei166el8:XF86Back0:8:XF86Backei167el11:XF86Forward0:11:XF86Forwardei169el9:XF86Eject0:9:XF86Ejectei170e"+\
"l9:XF86Eject9:XF86Eject9:XF86Eject9:XF86Ejectei171el13:XF86AudioNext0:13:XF86AudioNextei172el13:XF86AudioPlay14:XF86AudioPause13:XF86AudioPlay14:XF86AudioPauseei173el13:XF86AudioPrev0:13:XF86AudioPrevei174el13:XF86AudioStop9:XF86Eject13:XF86AudioStop9:XF86Ejectei175el15:XF86AudioRecord0:15:XF86AudioRecordei176el15:XF86AudioRewind0:15:XF86AudioRewindei177el9:XF86Phone0:9:XF86Phoneei179el9:XF86Tools0:9:XF86Toolsei180el12:XF86HomePage0:12:XF86HomePageei181el10:XF86Reload0:10:XF86Reloadei182el9:XF86Close0:9:XF86Closeei185el12:XF86ScrollUp0:12:XF86ScrollUpei186el14:XF86ScrollDown0:14:XF86ScrollDownei187el9:parenleft0:9:parenleftei188el10:parenright0:10:parenrightei189el7:XF86New0:7:XF86Newei190el4:Redo0:4:Redoei191el9:XF86Tools0:9:XF86Toolsei192el11:XF86Launch50:11:XF86Launch5ei193el11:XF86Launch60:11:XF86Launch6ei194el11:XF86Launch70:11:XF86Launch7e"+\
"i195el11:XF86Launch80:11:XF86Launch8ei196el11:XF86Launch90:11:XF86Launch9ei199el18:XF86TouchpadToggle0:18:XF86TouchpadToggleei200el14:XF86TouchpadOn0:14:XF86TouchpadOnei201el15:XF86TouchpadOff0:15:XF86TouchpadOffei203el11:Mode_switch0:11:Mode_switchei204el0:5:Alt_L0:5:Alt_Lei205el0:6:Meta_L0:6:Meta_Lei206el0:7:Super_L0:7:Super_Lei207el0:7:Hyper_L0:7:Hyper_Lei208el13:XF86AudioPlay0:13:XF86AudioPlayei209el14:XF86AudioPause0:14:XF86AudioPauseei210el11:XF86Launch30:11:XF86Launch3ei211el11:XF86Launch40:11:XF86Launch4ei212el11:XF86LaunchB0:11:XF86LaunchBei213el11:XF86Suspend0:11:XF86Suspendei214el9:XF86Close0:9:XF86Closeei215el13:XF86AudioPlay0:13:XF86AudioPlayei216el16:XF86AudioForward0:16:XF86AudioForwardei218el5:Print0:5:Printei220el10:XF86WebCam0:10:XF86WebCamei223el8:XF86Mail0:8:XF86Mailei224el13:XF86Messenger0:13:XF86Messengerei225el10:XF86Search0:10:XF86Search"+\
"ei226el6:XF86Go0:6:XF86Goei227el11:XF86Finance0:11:XF86Financeei228el8:XF86Game0:8:XF86Gameei229el8:XF86Shop0:8:XF86Shopei231el6:Cancel0:6:Cancelei232el21:XF86MonBrightnessDown0:21:XF86MonBrightnessDownei233el19:XF86MonBrightnessUp0:19:XF86MonBrightnessUpei234el14:XF86AudioMedia0:14:XF86AudioMediaei235el11:XF86Display0:11:XF86Displayei236el17:XF86KbdLightOnOff0:17:XF86KbdLightOnOffei237el21:XF86KbdBrightnessDown0:21:XF86KbdBrightnessDownei238el19:XF86KbdBrightnessUp0:19:XF86KbdBrightnessUpei239el8:XF86Send0:8:XF86Sendei240el9:XF86Reply0:9:XF86Replyei241el15:XF86MailForward0:15:XF86MailForwardei242el8:XF86Save0:8:XF86Saveei243el13:XF86Documents0:13:XF86Documentsei244el11:XF86Battery0:11:XF86Batteryei245el13:XF86Bluetooth0:13:XF86Bluetoothei246el8:XF86WLAN0:8:XF86WLANee18:xkbmap_mod_managedle14:encodings.corel4:h2643:vp84:webp3:png5:png/P5:png/L5:rgb244:jpeg5:rgb32e"+\
"11:raw_packetsi1e3:dpii96e18:platform.processor6:x86_6424:encoding.initial_qualityi50e18:encoding.min-speedi0e16:pycrypto.version5:2.6.14:belli1e23:encoding.cython.versionli0ei1ee16:platform.release22:3.12.5-200.fc19.x86_649:build.cpu6:x86_6412:xkbmap_print239:xkb_keymap {\n\txkb_keycodes  { include \"evdev+aliases(qwerty)\"\t};\n\txkb_types     { include \"complete\"\t};\n\txkb_compat    { include \"complete\"\t};\n\txkb_symbols   { include \"pc+gb+us:2+inet(evdev)\"\t};\n\txkb_geometry  { include \"pc(pc104)\"\t};\n};\n9:build.bit5:64bit9:mmap_file21:/tmp/xpra.AFrOuc.mmap19:xkbmap_mod_meaningsd7:Super_R4:mod47:Hyper_L4:mod49:Caps_Lock4:lock5:Alt_L4:mod17:Super_L4:mod46:Meta_L4:mod116:ISO_Level3_Shift4:mod57:Shift_R5:shift9:Control_R7:control11:Mode_switch4:mod58:Num_Lock4:mod27:Shift_L5:shift9:Control_L7:controle20:encoding.PIL.version5:1.1.7"+\
"19:sound.pygst.versionli0ei10ei22ee21:encoding.x264.versioni130e18:auto_refresh_delayi250e10:mmap_tokeni215666214940457138203759294163634184205e17:pycrypto.fastmathi1e12:xkbmap_query68:rules:      evdev\nmodel:      pc104\nlayout:     gb,us\nvariant:    ,\n18:encoding.rgb24zlibi1e16:platform.machine6:x86_6429:encoding.x264.YUV420P.profile6:high108:build.on7:desktop7:rencodei1e20:generic_window_typesi1e11:gtk.versionli2ei24ei22ee9:modifiersle21:encoding.uses_swscalei1e13:platform.name5:Linux4:zlibi1e14:build.revision4:507123:encoding_client_optionsi1e7:display3::1022:encoding.video_scalingi1e8:username7:antoine12:desktop_sizeli1920ei1080ee12:randr_notifyi1e5:sharei0e21:encoding.video_reiniti1e19:sound.pulseaudio.id42:1000@7725dfc225d14958a625ddaaaea5962b/20737:cursorsi1e14:sound.decodersl3:mp37:wavpack3:wav4:flac5:speexe15:rencode.version5:1.0.221:generic-rgb-encodingsi1e"+\
"15:xkbmap_keycodeslli65307e6:Escapei9ei0ei0eeli49e1:1i10ei0ei0eeli33e6:exclami10ei0ei1eeli185e11:onesuperiori10ei0ei2eeli161e10:exclamdowni10ei0ei3eeli49e1:1i10ei1ei0eeli33e6:exclami10ei1ei1eeli50e1:2i11ei0ei0eeli34e8:quotedbli11ei0ei1eeli178e11:twosuperiori11ei0ei2eeli2755e9:oneeighthi11ei0ei3eeli50e1:2i11ei1ei0eeli64e2:ati11ei1ei1eeli51e1:3i12ei0ei0eeli163e8:sterlingi12ei0ei1eeli179e13:threesuperiori12ei0ei2eeli163e8:sterlingi12ei0ei3eeli51e1:3i12ei1ei0eeli35e10:numbersigni12ei1ei1eeli52e1:4i13ei0ei0eeli36e6:dollari13ei0ei1eeli8364e8:EuroSigni13ei0ei2eeli188e10:onequarteri13ei0ei3eeli52e1:4i13ei1ei0eeli36e6:dollari13ei1ei1eeli53e1:5i14ei0ei0eeli37e7:percenti14ei0ei1eeli189e7:onehalfi14ei0ei2eeli2756e12:threeeighthsi14ei0ei3eeli53e1:5i14ei1ei0eeli37e7:percenti14ei1ei1eeli54e1:6i15ei0ei0eeli94e11:asciicircumi15ei0ei1eeli190e13:threequartersi15ei0ei2eeli2757e11:fiveeighthsi15e"+\
"i0ei3eeli54e1:6i15ei1ei0eeli94e11:asciicircumi15ei1ei1eeli55e1:7i16ei0ei0eeli38e9:ampersandi16ei0ei1eeli123e9:bracelefti16ei0ei2eeli2758e12:seveneighthsi16ei0ei3eeli55e1:7i16ei1ei0eeli38e9:ampersandi16ei1ei1eeli56e1:8i17ei0ei0eeli42e8:asteriski17ei0ei1eeli91e11:bracketlefti17ei0ei2eeli2761e9:trademarki17ei0ei3eeli56e1:8i17ei1ei0eeli42e8:asteriski17ei1ei1eeli57e1:9i18ei0ei0eeli40e9:parenlefti18ei0ei1eeli93e12:bracketrighti18ei0ei2eeli177e9:plusminusi18ei0ei3eeli57e1:9i18ei1ei0eeli40e9:parenlefti18ei1ei1eeli48e1:0i19ei0ei0eeli41e10:parenrighti19ei0ei1eeli125e10:bracerighti19ei0ei2eeli176e6:degreei19ei0ei3eeli48e1:0i19ei1ei0eeli41e10:parenrighti19ei1ei1eeli45e5:minusi20ei0ei0eeli95e10:underscorei20ei0ei1eeli92e9:backslashi20ei0ei2eeli191e12:questiondowni20ei0ei3eeli45e5:minusi20ei1ei0eeli95e10:underscorei20ei1ei1eeli61e5:equali21ei0ei0eeli43e4:plusi21ei0ei1eeli65115e12:dead_cedilla"+\
"i21ei0ei2eeli65116e11:dead_ogoneki21ei0ei3eeli61e5:equali21ei1ei0eeli43e4:plusi21ei1ei1eeli65288e9:BackSpacei22ei0ei0eeli65288e9:BackSpacei22ei0ei1eeli65289e3:Tabi23ei0ei0eeli65056e12:ISO_Left_Tabi23ei0ei1eeli113e1:qi24ei0ei0eeli81e1:Qi24ei0ei1eeli64e2:ati24ei0ei2eeli2009e11:Greek_OMEGAi24ei0ei3eeli113e1:qi24ei1ei0eeli81e1:Qi24ei1ei1eeli119e1:wi25ei0ei0eeli87e1:Wi25ei0ei1eeli435e7:lstrokei25ei0ei2eeli419e7:Lstrokei25ei0ei3eeli119e1:wi25ei1ei0eeli87e1:Wi25ei1ei1eeli101e1:ei26ei0ei0eeli69e1:Ei26ei0ei1eeli101e1:ei26ei0ei2eeli69e1:Ei26ei0ei3eeli101e1:ei26ei1ei0eeli69e1:Ei26ei1ei1eeli114e1:ri27ei0ei0eeli82e1:Ri27ei0ei1eeli182e9:paragraphi27ei0ei2eeli174e10:registeredi27ei0ei3eeli114e1:ri27ei1ei0eeli82e1:Ri27ei1ei1eeli116e1:ti28ei0ei0eeli84e1:Ti28ei0ei1eeli956e6:tslashi28ei0ei2eeli940e6:Tslashi28ei0ei3eeli116e1:ti28ei1ei0eeli84e1:Ti28ei1ei1eeli121e1:yi29ei0ei0eeli89e1:Yi29ei0ei1eeli2299e"+\
"9:leftarrowi29ei0ei2eeli165e3:yeni29ei0ei3eeli121e1:yi29ei1ei0eeli89e1:Yi29ei1ei1eeli117e1:ui30ei0ei0eeli85e1:Ui30ei0ei1eeli2302e9:downarrowi30ei0ei2eeli2300e7:uparrowi30ei0ei3eeli117e1:ui30ei1ei0eeli85e1:Ui30ei1ei1eeli105e1:ii31ei0ei0eeli73e1:Ii31ei0ei1eeli2301e10:rightarrowi31ei0ei2eeli697e8:idotlessi31ei0ei3eeli105e1:ii31ei1ei0eeli73e1:Ii31ei1ei1eeli111e1:oi32ei0ei0eeli79e1:Oi32ei0ei1eeli248e6:oslashi32ei0ei2eeli216e6:Oslashi32ei0ei3eeli111e1:oi32ei1ei0eeli79e1:Oi32ei1ei1eeli112e1:pi33ei0ei0eeli80e1:Pi33ei0ei1eeli254e5:thorni33ei0ei2eeli222e5:THORNi33ei0ei3eeli112e1:pi33ei1ei0eeli80e1:Pi33ei1ei1eeli91e11:bracketlefti34ei0ei0eeli123e9:bracelefti34ei0ei1eeli65111e14:dead_diaeresisi34ei0ei2eeli65112e14:dead_aboveringi34ei0ei3eeli91e11:bracketlefti34ei1ei0eeli123e9:bracelefti34ei1ei1eeli93e12:bracketrighti35ei0ei0eeli125e10:bracerighti35ei0ei1eeli65107e10:dead_tildei35ei0ei2eeli65108e"+\
"11:dead_macroni35ei0ei3eeli93e12:bracketrighti35ei1ei0eeli125e10:bracerighti35ei1ei1eeli65293e6:Returni36ei0ei0eeli65507e9:Control_Li37ei0ei0eeli97e1:ai38ei0ei0eeli65e1:Ai38ei0ei1eeli230e2:aei38ei0ei2eeli198e2:AEi38ei0ei3eeli97e1:ai38ei1ei0eeli65e1:Ai38ei1ei1eeli115e1:si39ei0ei0eeli83e1:Si39ei0ei1eeli223e6:ssharpi39ei0ei2eeli167e7:sectioni39ei0ei3eeli115e1:si39ei1ei0eeli83e1:Si39ei1ei1eeli100e1:di40ei0ei0eeli68e1:Di40ei0ei1eeli240e3:ethi40ei0ei2eeli208e3:ETHi40ei0ei3eeli100e1:di40ei1ei0eeli68e1:Di40ei1ei1eeli102e1:fi41ei0ei0eeli70e1:Fi41ei0ei1eeli496e7:dstrokei41ei0ei2eeli170e11:ordfemininei41ei0ei3eeli102e1:fi41ei1ei0eeli70e1:Fi41ei1ei1eeli103e1:gi42ei0ei0eeli71e1:Gi42ei0ei1eeli959e3:engi42ei0ei2eeli957e3:ENGi42ei0ei3eeli103e1:gi42ei1ei0eeli71e1:Gi42ei1ei1eeli104e1:hi43ei0ei0eeli72e1:Hi43ei0ei1eeli689e7:hstrokei43ei0ei2eeli673e7:Hstrokei43ei0ei3eeli104e1:hi43ei1ei0eeli72e1:Hi43ei1ei1ee"+\
"li106e1:ji44ei0ei0eeli74e1:Ji44ei0ei1eeli65121e9:dead_hooki44ei0ei2eeli65122e9:dead_horni44ei0ei3eeli106e1:ji44ei1ei0eeli74e1:Ji44ei1ei1eeli107e1:ki45ei0ei0eeli75e1:Ki45ei0ei1eeli930e3:krai45ei0ei2eeli38e9:ampersandi45ei0ei3eeli107e1:ki45ei1ei0eeli75e1:Ki45ei1ei1eeli108e1:li46ei0ei0eeli76e1:Li46ei0ei1eeli435e7:lstrokei46ei0ei2eeli419e7:Lstrokei46ei0ei3eeli108e1:li46ei1ei0eeli76e1:Li46ei1ei1eeli59e9:semicoloni47ei0ei0eeli58e5:coloni47ei0ei1eeli65105e10:dead_acutei47ei0ei2eeli65113e16:dead_doubleacutei47ei0ei3eeli59e9:semicoloni47ei1ei0eeli58e5:coloni47ei1ei1eeli39e10:apostrophei48ei0ei0eeli64e2:ati48ei0ei1eeli65106e15:dead_circumflexi48ei0ei2eeli65114e10:dead_caroni48ei0ei3eeli39e10:apostrophei48ei1ei0eeli34e8:quotedbli48ei1ei1eeli96e5:gravei49ei0ei0eeli172e7:notsigni49ei0ei1eeli124e3:bari49ei0ei2eeli124e3:bari49ei0ei3eeli96e5:gravei49ei1ei0eeli126e10:asciitildei49ei1ei1eeli65505e"+\
"7:Shift_Li50ei0ei0eeli35e10:numbersigni51ei0ei0eeli126e10:asciitildei51ei0ei1eeli65104e10:dead_gravei51ei0ei2eeli65109e10:dead_brevei51ei0ei3eeli92e9:backslashi51ei1ei0eeli124e3:bari51ei1ei1eeli122e1:zi52ei0ei0eeli90e1:Zi52ei0ei1eeli171e13:guillemotlefti52ei0ei2eeli60e4:lessi52ei0ei3eeli122e1:zi52ei1ei0eeli90e1:Zi52ei1ei1eeli120e1:xi53ei0ei0eeli88e1:Xi53ei0ei1eeli187e14:guillemotrighti53ei0ei2eeli62e7:greateri53ei0ei3eeli120e1:xi53ei1ei0eeli88e1:Xi53ei1ei1eeli99e1:ci54ei0ei0eeli67e1:Ci54ei0ei1eeli162e4:centi54ei0ei2eeli169e9:copyrighti54ei0ei3eeli99e1:ci54ei1ei0eeli67e1:Ci54ei1ei1eeli118e1:vi55ei0ei0eeli86e1:Vi55ei0ei1eeli2770e19:leftdoublequotemarki55ei0ei2eeli2768e19:leftsinglequotemarki55ei0ei3eeli118e1:vi55ei1ei0eeli86e1:Vi55ei1ei1eeli98e1:bi56ei0ei0eeli66e1:Bi56ei0ei1eeli2771e20:rightdoublequotemarki56ei0ei2eeli2769e20:rightsinglequotemarki56ei0ei3eeli98e1:bi56ei1ei0eeli66e"+\
"1:Bi56ei1ei1eeli110e1:ni57ei0ei0eeli78e1:Ni57ei0ei1eeli110e1:ni57ei0ei2eeli78e1:Ni57ei0ei3eeli110e1:ni57ei1ei0eeli78e1:Ni57ei1ei1eeli109e1:mi58ei0ei0eeli77e1:Mi58ei0ei1eeli181e2:mui58ei0ei2eeli186e9:masculinei58ei0ei3eeli109e1:mi58ei1ei0eeli77e1:Mi58ei1ei1eeli44e5:commai59ei0ei0eeli60e4:lessi59ei0ei1eeli2211e14:horizconnectori59ei0ei2eeli215e8:multiplyi59ei0ei3eeli44e5:commai59ei1ei0eeli60e4:lessi59ei1ei1eeli46e6:periodi60ei0ei0eeli62e7:greateri60ei0ei1eeli183e14:periodcenteredi60ei0ei2eeli247e8:divisioni60ei0ei3eeli46e6:periodi60ei1ei0eeli62e7:greateri60ei1ei1eeli47e5:slashi61ei0ei0eeli63e8:questioni61ei0ei1eeli65120e13:dead_belowdoti61ei0ei2eeli65110e13:dead_abovedoti61ei0ei3eeli47e5:slashi61ei1ei0eeli63e8:questioni61ei1ei1eeli65506e7:Shift_Ri62ei0ei0eeli65450e11:KP_Multiplyi63ei0ei0eeli65450e11:KP_Multiplyi63ei0ei1eeli65450e11:KP_Multiplyi63ei0ei2eeli65450e11:KP_Multiplyi63ei0e"+\
"i3eeli269024801e13:XF86ClearGrabi63ei0ei4eeli65513e5:Alt_Li64ei0ei0eeli65511e6:Meta_Li64ei0ei1eeli32e5:spacei65ei0ei0eeli65509e9:Caps_Locki66ei0ei0eeli65470e2:F1i67ei0ei0eeli65470e2:F1i67ei0ei1eeli65470e2:F1i67ei0ei2eeli65470e2:F1i67ei0ei3eeli269024769e15:XF86Switch_VT_1i67ei0ei4eeli65471e2:F2i68ei0ei0eeli65471e2:F2i68ei0ei1eeli65471e2:F2i68ei0ei2eeli65471e2:F2i68ei0ei3eeli269024770e15:XF86Switch_VT_2i68ei0ei4eeli65472e2:F3i69ei0ei0eeli65472e2:F3i69ei0ei1eeli65472e2:F3i69ei0ei2eeli65472e2:F3i69ei0ei3eeli269024771e15:XF86Switch_VT_3i69ei0ei4eeli65473e2:F4i70ei0ei0eeli65473e2:F4i70ei0ei1eeli65473e2:F4i70ei0ei2eeli65473e2:F4i70ei0ei3eeli269024772e15:XF86Switch_VT_4i70ei0ei4eeli65474e2:F5i71ei0ei0eeli65474e2:F5i71ei0ei1eeli65474e2:F5i71ei0ei2eeli65474e2:F5i71ei0ei3eeli269024773e15:XF86Switch_VT_5i71ei0ei4eeli65475e2:F6i72ei0ei0eeli65475e2:F6i72ei0ei1eeli65475e2:F6i72ei0ei2eeli65475e"+\
"2:F6i72ei0ei3eeli269024774e15:XF86Switch_VT_6i72ei0ei4eeli65476e2:F7i73ei0ei0eeli65476e2:F7i73ei0ei1eeli65476e2:F7i73ei0ei2eeli65476e2:F7i73ei0ei3eeli269024775e15:XF86Switch_VT_7i73ei0ei4eeli65477e2:F8i74ei0ei0eeli65477e2:F8i74ei0ei1eeli65477e2:F8i74ei0ei2eeli65477e2:F8i74ei0ei3eeli269024776e15:XF86Switch_VT_8i74ei0ei4eeli65478e2:F9i75ei0ei0eeli65478e2:F9i75ei0ei1eeli65478e2:F9i75ei0ei2eeli65478e2:F9i75ei0ei3eeli269024777e15:XF86Switch_VT_9i75ei0ei4eeli65479e3:F10i76ei0ei0eeli65479e3:F10i76ei0ei1eeli65479e3:F10i76ei0ei2eeli65479e3:F10i76ei0ei3eeli269024778e16:XF86Switch_VT_10i76ei0ei4eeli65407e8:Num_Locki77ei0ei0eeli65300e11:Scroll_Locki78ei0ei0eeli65429e7:KP_Homei79ei0ei0eeli65463e4:KP_7i79ei0ei1eeli65431e5:KP_Upi80ei0ei0eeli65464e4:KP_8i80ei0ei1eeli65434e10:KP_Page_Upi81ei0ei0eeli65465e4:KP_9i81ei0ei1eeli65453e11:KP_Subtracti82ei0ei0eeli65453e11:KP_Subtracti82ei0ei1eeli65453e"+\
"11:KP_Subtracti82ei0ei2eeli65453e11:KP_Subtracti82ei0ei3eeli269024803e14:XF86Prev_VModei82ei0ei4eeli65430e7:KP_Lefti83ei0ei0eeli65460e4:KP_4i83ei0ei1eeli65437e8:KP_Begini84ei0ei0eeli65461e4:KP_5i84ei0ei1eeli65432e8:KP_Righti85ei0ei0eeli65462e4:KP_6i85ei0ei1eeli65451e6:KP_Addi86ei0ei0eeli65454e10:KP_Decimali129ei0ei0eeli65454e10:KP_Decimali129ei0ei1eeli65329e6:Hanguli130ei0ei0eeli65332e12:Hangul_Hanjai131ei0ei0eeli65515e7:Super_Li133ei0ei0eeli65516e7:Super_Ri134ei0ei0eeli65383e4:Menui135ei0ei0eeli65385e6:Canceli136ei0ei0eeli65382e4:Redoi137ei0ei0eeli268828528e8:SunPropsi138ei0ei0eeli65381e4:Undoi139ei0ei0eeli268828529e8:SunFronti140ei0ei0eeli269025111e8:XF86Copyi141ei0ei0eeli269025131e8:XF86Openi142ei0ei0eeli269025133e9:XF86Pastei143ei0ei0eeli65384e4:Findi144ei0ei0eeli269025112e7:XF86Cuti145ei0ei0eeli65386e4:Helpi146ei0ei0eeli269025125e10:XF86MenuKBi147ei0ei0eeli269025053e"+\
"14:XF86Calculatori148ei0ei0eeli269025071e9:XF86Sleepi150ei0ei0eeli269025067e10:XF86WakeUpi151ei0ei0eeli269025117e12:XF86Exploreri152ei0ei0eeli269025147e8:XF86Sendi153ei0ei0eeli269025162e8:XF86Xferi155ei0ei0eeli269025089e11:XF86Launch1i156ei0ei0eeli269025090e11:XF86Launch2i157ei0ei0eeli269025070e7:XF86WWWi158ei0ei0eeli269025114e7:XF86DOSi159ei0ei0eeli269025069e15:XF86ScreenSaveri160ei0ei0eeli269025140e17:XF86RotateWindowsi162ei0ei0eeli269025049e8:XF86Maili163ei0ei0eeli269025072e13:XF86Favoritesi164ei0ei0eeli269025075e14:XF86MyComputeri165ei0ei0eeli269025062e8:XF86Backi166ei0ei0eeli269025063e11:XF86Forwardi167ei0ei0eeli269025068e9:XF86Ejecti169ei0ei0eeli269025068e9:XF86Ejecti170ei0ei0eeli269025068e9:XF86Ejecti170ei0ei1eeli269025047e13:XF86AudioNexti171ei0ei0eeli269025044e13:XF86AudioPlayi172ei0ei0eeli269025073e14:XF86AudioPausei172ei0ei1eeli269025046e13:XF86AudioPrevi173ei0e"+\
"i0eeli269025045e13:XF86AudioStopi174ei0ei0eeli269025068e9:XF86Ejecti174ei0ei1eeli269025052e15:XF86AudioRecordi175ei0ei0eeli269025086e15:XF86AudioRewindi176ei0ei0eeli269025134e9:XF86Phonei177ei0ei0eeli269025153e9:XF86Toolsi179ei0ei0eeli269025048e12:XF86HomePagei180ei0ei0eeli269025139e10:XF86Reloadi181ei0ei0eeli269025110e9:XF86Closei182ei0ei0eeli269025144e12:XF86ScrollUpi185ei0ei0eeli269025145e14:XF86ScrollDowni186ei0ei0eeli40e9:parenlefti187ei0ei0eeli41e10:parenrighti188ei0ei0eeli269025128e7:XF86Newi189ei0ei0eeli65382e4:Redoi190ei0ei0eeli269025153e9:XF86Toolsi191ei0ei0eeli269025093e11:XF86Launch5i192ei0ei0eeli269025094e11:XF86Launch6i193ei0ei0eeli269025095e11:XF86Launch7i194ei0ei0eeli269025096e11:XF86Launch8i195ei0ei0eeli269025097e11:XF86Launch9i196ei0ei0eeli269025193e18:XF86TouchpadTogglei199ei0ei0eeli269025200e14:XF86TouchpadOni200ei0ei0eeli269025201e15:XF86TouchpadOffi201e"+\
"i0ei0eeli65406e11:Mode_switchi203ei0ei0eeli65513e5:Alt_Li204ei0ei1eeli65511e6:Meta_Li205ei0ei1eeli65515e7:Super_Li206ei0ei1eeli65517e7:Hyper_Li207ei0ei1eeli269025044e13:XF86AudioPlayi208ei0ei0eeli269025073e14:XF86AudioPausei209ei0ei0eeli269025091e11:XF86Launch3i210ei0ei0eeli269025092e11:XF86Launch4i211ei0ei0eeli269025099e11:XF86LaunchBi212ei0ei0eeli269025191e11:XF86Suspendi213ei0ei0eeli269025110e9:XF86Closei214ei0ei0eeli269025044e13:XF86AudioPlayi215ei0ei0eeli269025175e16:XF86AudioForwardi216ei0ei0eeli65377e5:Printi218ei0ei0eeli269025167e10:XF86WebCami220ei0ei0eeli269025049e8:XF86Maili223ei0ei0eeli269025166e13:XF86Messengeri224ei0ei0eeli269025051e10:XF86Searchi225ei0ei0eeli269025119e6:XF86Goi226ei0ei0eeli269025084e11:XF86Financei227ei0ei0eeli269025118e8:XF86Gamei228ei0ei0eeli269025078e8:XF86Shopi229ei0ei0eeli65385e6:Canceli231ei0ei0eeli269025027e21:XF86MonBrightnessDowni232ei0e"+\
"i0eeli269025026e19:XF86MonBrightnessUpi233ei0ei0eeli269025074e14:XF86AudioMediai234ei0ei0eeli269025113e11:XF86Displayi235ei0ei0eeli269025028e17:XF86KbdLightOnOffi236ei0ei0eeli269025030e21:XF86KbdBrightnessDowni237ei0ei0eeli269025029e19:XF86KbdBrightnessUpi238ei0ei0eeli269025147e8:XF86Sendi239ei0ei0eeli269025138e9:XF86Replyi240ei0ei0eeli269025168e15:XF86MailForwardi241ei0ei0eeli269025143e8:XF86Savei242ei0ei0eeli269025115e13:XF86Documentsi243ei0ei0eeli269025171e11:XF86Batteryi244ei0ei0eeli269025172e13:XF86Bluetoothi245ei0ei0eeli269025173e8:XF86WLANi246ei0ei0eee16:raw_window_iconsi1e21:clipboard.set_enabledi1e11:system_trayi1e8:hostname7:desktop9:namespacei1e17:compression_leveli1e27:platform.linux_distributionl6:Fedora2:1920:Schrödinger’s Cate16:encoding.rgb_lz4i1e14:sound.encodersl3:mp37:wavpack3:wav4:flac5:speexe24:encoding.avcodec.versionli54ei92ei100ee23:clipboard.notificationsi1e"+\
"15:xsettings-tuplei1e16:clipboard.greedyi0e10:build.date10:2014-01-0113:notificationsi1e7:windowsi1e20:encoding.min-qualityi50e20:compressible_cursorsi1e25:build.local_modifications1:020:clipboard.selectionsl9:CLIPBOARD7:PRIMARY9:SECONDARYe17:sound.gst.versionli0ei10ei36ee24:encoding.swscale.versionli2ei2ei100eeee"
def test_encoding_large_hello():
    log("test_large_hello()")
    t(large_hello, large_hello_output)

nested_dicts = ['some_new_feature_we_may_add', {"with_a_nested_dict" : {"containing_another_dict" : ["with", "nested", "arrays", ["in", ["it", "going", [["deep", 0, -1]]]]]}}]
nested_dicts_output = "l27:some_new_feature_we_may_addd18:with_a_nested_dictd23:containing_another_dictl4:with6:nested6:arraysl2:inl2:it5:goingll4:deepi0ei-1eeeeeeeee"
def test_nested_dicts():
    t(nested_dicts, nested_dicts_output)
    log("")


def test_random():
    log("test_random()")
    import binascii
    u = "6c343a6472617769316569333731656931356569366569313365353a726762333231343a7801fbff7f1490130200099e36d8693265693234656431303a7267625f666f726d6174343a52474258343a7a6c69626931656565"
    s = binascii.unhexlify(u)
    decode(s)
    log("")

def get_test_data_dict():
    try:
        from xpra.x11.gtk_x11 import gdk_display_source             #@UnusedImport
        from xpra.x11.bindings.keyboard_bindings import X11KeyboardBindings        #@UnresolvedImport
        keyboard_bindings = X11KeyboardBindings()
        return keyboard_bindings.get_keycode_mappings()
    except ImportError, e:
        log("cannot use X11 keyboard data as test data: %s" % e);
        return {"foo" : "bar"}
test_dict = get_test_data_dict()

def test_large_dict():
    log("test_large_dict()")
    b = encode(test_dict)
    log("bencode(%s)=%s" % (i(test_dict), i(b)))
    d = decode(b)
    log("bdecode(%s)=%s" % (i(b), i(d)))
    log("")

def test_compare_cython():
    log("test_compare_cython()")
    #suspend logging:
    use_noop()
    results = {}
    for n,x in {"python" : use_python, "cython" : use_cython}.items():
        x()
        start = time.time()
        for i in xrange(50):
            test_large_dict()
            test_random()
            test_decoding()
            test_encoding_hello()
            test_encoding_large_hello()
            test_nested_dicts()
        end = time.time()
        results[n] = int(1000.0*(end-start))
    use_print()
    log("results: %s (in milliseconds)" % results)


def main():
    for x in (use_python, use_cython):
        log("main: testing with %s" % x)
        x()
        test_random()
        test_decoding()
        test_encoding_hello()
        test_encoding_large_hello()
        test_nested_dicts()
        test_large_dict()

    test_compare_cython()


if __name__ == "__main__":
    main()
