/*
 * @(#)xhexagons.c
 *
 * Copyright 1993 - 2024  David A. Bagley, bagleyd AT verizon.net
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/* Driver file for Hexagons */

#ifndef WINVER
#include "version.h"
static const char aboutHelp[] = {
"Hexagons Version " VERSION "\n"
"Send bugs (reports or fixes) to the author: "
"David Bagley <bagleyd AT verizon.net>\n"
"The latest version is at: "
"https://www.sillycycle.com/puzzles.html"
};

static const char synopsisHelp[] = {
"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n"
"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
"[-{foreground|fg} {color}] [-{background|bg} {color}]\n"
"[-tile {color}] [-text {color}] [-frame {color}]\n"
"[-[no]stippleFrame] [-delay msecs] [-[no]sound]\n"
"[-bumpSound {filename}] [-{font|fn} {fontname}]\n"
"[-size{X|Y} {int}] [-[no]corners] [-base {int}]\n"
"[-userName {string}] [-scoreFile {filename}] [-scores]\n"
"[-version]"
};
#endif

#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA) || defined(WINVER)
static const char descriptionHelp[] = {
"A sliding tile puzzle.  A hexagonal version of the "
"15 puzzle.  The 17 puzzle was conceived of by Alwin Vietor\n"
"and marketed in 1885 in Germany as \"Trilemma\" with "
"round pieces.  Sam Cornwell rediscovered this in 1989\n"
"with \"Clockblocks\" of which 12 were numbered using "
"hexagonal pieces.  Also he produced a smaller 12 block\n"
"puzzle with 9 that were numbered called \"Ninefield\".  Also "
"Bernhard Wiezorke came up with the similar 17\n"
"puzzle, \"Hexafax, the New Fifteen Puzzle\", in 1991 and a "
"smaller 11 piece \"Counterfax\" of which 8 were\n"
"numbered."
};

static const char featuresHelp[] = {
"If corners mode on: Press \"mouse-left\" button to move "
"a tile, then release \"mouse-left\" button on a space.\n"
"It will not move if blocked.  One can only move one "
"tile at a time. The only tiles that can be moved are\n"
"the tiles next to both spaces.\n"
"If corners mode off: Click \"mouse-left\" button to "
"move a tile.  Notice that one can move more than one tile\n"
"at a time.  by clicking on the tile farthest from the "
"space but in line with the space.  Clicks on the space,\n"
"or clicks on tiles that are not in line with the space, "
"will not move tiles.\n"
"\n"
"Click \"mouse-right\" button, or press \"C\" or \"c\" "
"keys, to clear the puzzle.\n"
"\n"
"Press \"G\" or \"g\" keys to read (get) a saved puzzle.\n"
"\n"
"Press \"W\" or \"w\" keys to save (write) a puzzle.\n"
"\n"
"Press \"U\" or \"u\" keys to undo a move.\n"
"\n"
"Press \"R\" or \"r\" keys to redo a move.\n"
"\n"
"Press \"Z\" or \"z\" keys to randomize the puzzle.  New records "
"can only be set from a cleared state.\n"
"\n"
"\"S\" or \"s\" keys reserved for the auto-solver "
"(not implemented).\n"
"\n"
"Press \"M\" or \"m\" keys to toggle corners mode.  "
"(Default is on).\n"
"\n"
"Press \">\" or \".\" keys to speed up the movement of tiles.\n"
"\n"
"Press \"<\" or \",\" keys to slow down the movement of tiles.\n"
"\n"
"Press \"@\" key to toggle the sound.\n"
"\n"
"Press \"Esc\" key to hide program.\n"
"\n"
"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
"\n"
"Use the key pad or arrow keys to move without the mouse.\n"
"Key pad is defined for Hexagons as:\n"
"7   9  Upper Left, Upper Right\n"
"  ^\n"
"4< >6  Left, Right\n"
"  v\n"
"1   3  Lower Left, Lower Right\n"
"\n"
"The control key allows you to change the complexity "
"of xhexagons.  Hold down control key and use the\n"
"keypad or arrow keys to reduce or enlarge puzzle complexity."
};

static const char referencesHelp[] = {
"Dieter Gebhardt, \"From Trilemma to Counterfax\", CFF 53, "
"October 2000, pp 8-10.\n"
"Sam Cornwell, \"Ninefield: a Sliding Block Puzzle\", "
"CFF 57, March 2002, pp 4-5.\n"
"Dieter Gebhardt, \"Ninefield Revisited\", CFF 58, "
"July 2002, p 21.\n"
"Bernhard Wiezorke, \"Hexafax, the New Fifteen Puzzle\" "
"brochure."
};
#endif

static const char solveHelp[] = {
"Auto-solver: sorry, not implemented."
};

#include "file.h"
#ifdef WINVER
#include "HexagonsP.h"
#define TITLE "whexagons"

static HexagonsRec widget;

#ifndef SCOREPATH
#ifdef UNIXDELIM
#define SCOREPATH "c:/WINDOWS"
#else
#define SCOREPATH "c:\\WINDOWS"
#endif
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(widget.core.hWnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->hexagons.started = b
#else
#include "xwin.h"
#include "xgui.h"
#define SET_STARTED(w,b) XtVaSetValues(w, XtNstart, b, NULL)
#include "Hexagons.h"
#ifdef HAVE_XPM
#include <X11/xpm.h>
#ifdef CONSTPIXMAPS
#include "icons/16x16/hexagons.xpm"
#include "icons/22x22/hexagons.xpm"
#include "icons/24x24/hexagons.xpm"
#include "icons/32x32/hexagons.xpm"
#include "icons/48x48/hexagons.xpm"
#include "icons/64x64/hexagons.xpm"
#else
#include "pixmaps/16x16/hexagons.xpm"
#include "pixmaps/22x22/hexagons.xpm"
#include "pixmaps/24x24/hexagons.xpm"
#include "pixmaps/32x32/hexagons.xpm"
#include "pixmaps/48x48/hexagons.xpm"
#include "pixmaps/64x64/hexagons.xpm"
#endif
#define RESIZE_XPM(s) ((char **) (((s)<=32)?\
(((s)<=22)?(((s)<=16)?hexagons_16x16:hexagons_22x22):\
(((s)<=24)?hexagons_24x24:hexagons_32x32)):\
(((s)<=48)?hexagons_48x48:hexagons_64x64)))
#endif
#include "pixmaps/64x64/hexagons.xbm"
#define DEFINE_XBM (char *) hexagons_64x64_bits, hexagons_64x64_width, hexagons_64x64_height
#ifndef SCOREPATH
#ifdef VMS
#define SCOREPATH "SYS$LOGIN:"
#else
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif
#endif

#ifndef SCOREFILE
#define SCOREFILE "hexagons.scores"
#endif

#define MAX_TILES 10
#define NEVER (-1)
#define USER_NAME_LENGTH 120
#define MESSAGE_LENGTH (USER_NAME_LENGTH+64)
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USER_NAME_LENGTH];
} PuzzleRecord;

static PuzzleRecord puzzleRecord[MAX_ORIENT]
[MAX_TILES - MIN_TILES + 1][MAX_TILES - MIN_TILES + 1];
static int movesDsp = 0;
static char messageDsp[MESSAGE_LENGTH] = "Welcome";
static char recordDsp[MESSAGE_LENGTH] = "NOT RECORDED";

#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
static Widget movesText, recordText, message;
static Widget cornersToggle;
static Widget soundMenuItem;
static Widget clearDialog, solveDialog;
static Widget descriptionDialog, featuresDialog;
static Widget synopsisDialog, referencesDialog, aboutDialog;
static char buff[MESSAGE_LENGTH];
static const char *tileLabels[] =
{
	"Tiles X:",
	"Tiles Y:"
};
static const char *speedLabel = "Animation Speed:";
#define PRINT_MESSAGE(b) printState(message, b)
#define MIN_SPEED 1
#define MAX_SPEED 50
#elif !defined(WINVER)
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#ifdef HAVE_MOTIF
static Widget tileXChanger, tileYChanger;
#ifdef ANIMATE
static Widget speedChanger;
#endif
#elif defined(HAVE_ATHENA)
static Widget tileXSlider, tileYSlider, speedSlider;
static Widget tileXSliderLabel, tileYSliderLabel, speedSliderLabel;
static const char *fileTypes[] =
{
	"Get",
	"Write",
	"Exit"
};
#define numFileTypes (sizeof(fileTypes)/sizeof(fileTypes[0]))
static const char *playTypes[] =
{
	"Undo",
	"Redo",
	"Clear",
	"Randomize",
	"(Auto-solve)",
#ifdef EXTRA
	"Corners",
	"Speed >",
	"Slow <",
#endif
	"Sound @"
};
#define numPlayTypes (sizeof(playTypes)/sizeof(playTypes[0]))
static const char *helpTypes[] =
{
	"Description...",
	"Features...",
	"Synopsis...",
	"References...",
	"About..."
};
#define numHelpTypes (sizeof(helpTypes)/sizeof(helpTypes[0]))
#endif
static char scoreFileName[FILE_NAME_LENGTH] = SCOREFILE;
static char fileName[FILE_NAME_LENGTH];
static Boolean randomized = False;
#ifdef WINVER
#define PROGRAM_NAME_LENGTH 80
static char progDsp[PROGRAM_NAME_LENGTH] = TITLE;
static char userNameDsp[USER_NAME_LENGTH] = "guest";
#else
static Pixmap pixmap = None;
static Widget topLevel, puzzle;
static char userNameDsp[USER_NAME_LENGTH] = "";

#ifdef HAVE_MOTIF
static void
printState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("printState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtVaSetValues(w,
		XmNlabelString, xmstr, NULL);
	XmStringFree(xmstr);
}
#elif defined(HAVE_ATHENA)
static void
printState(Widget w, char *msg)
{
	XtVaSetValues(w,
		XtNlabel, msg, NULL);
	if (w == movesText)
		XtVaSetValues(w,
			XtNwidth, 64, NULL);
}
#endif

static void
printRecords(void)
{
	int i, j, k;

	(void) printf("     HEXAGONS  HALL OF FAME\n\n");
	(void) printf("CORNER   X   Y USER            MOVES\n");
	for (i = 0; i < MAX_ORIENT; i++)
		for (j = 0; j < MAX_TILES - MIN_TILES + 1; j++)
			for (k = 0; k < MAX_TILES - MIN_TILES + 1; k++) {
				if (puzzleRecord[i][j][k].score > 0)
					(void) printf("%6d%4d%4d %-12s%9d\n",
						i, j + 1, k + 1,
						puzzleRecord[i][j][k].name,
						puzzleRecord[i][j][k].score);
			}
}
#endif

static void
initRecords(void)
{
	int i, j, k;

	for (i = 0; i < MAX_ORIENT; i++)
		for (j = 0; j < MAX_TILES - MIN_TILES + 1; j++)
			for (k = 0; k < MAX_TILES - MIN_TILES + 1; k++) {
				puzzleRecord[i][j][k].score = NEVER;
				(void) strncpy(puzzleRecord[i][j][k].name,
					NOACCESS, USER_NAME_LENGTH);
			}
}

static void
readRecords(void)
{
	FILE *fp;
	int i, j, k, n;
	char userName[USER_NAME_LENGTH];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname;

	stringCat(&buf1, CURRENTDELIM, scoreFileName);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	(void) strncpy(fileName, lname, USER_NAME_LENGTH);
	if ((fp = fopen(fileName, "r")) == NULL) {
		(void) strncpy(fileName, fname, USER_NAME_LENGTH);
		/* Try installed directory. */
		if ((fp = fopen(fileName, "r")) == NULL) {
			stringCat(&buf1, "Cannot read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Cannot read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (i = 0; i < MAX_ORIENT; i++)
		for (j = 0; j < MAX_TILES - MIN_TILES + 1; j++)
			for (k = 0; k < MAX_TILES - MIN_TILES + 1; k++) {
				if (fscanf(fp, "%d %s\n", &n, userName) != 2) {
					(void) fprintf(stderr,
						"corrupt record, expecting integer and name for %d %d %d\n",
						i, j, k);
					(void) fclose(fp);
					return;
				}
				if (n <= puzzleRecord[i][j][k].score ||
						puzzleRecord[i][j][k].score <= NEVER) {
					puzzleRecord[i][j][k].score = n;
					(void) strncpy(puzzleRecord[i][j][k].name,
						userName, USER_NAME_LENGTH);
				}
			}
	(void) fclose(fp);
}

static void
writeRecords(void)
{
	FILE *fp;
	int i, j, k;
	char *buf1 = NULL;

	if ((fp = fopen(fileName, "w")) == NULL) {
		stringCat(&buf1, "Cannot write to ", fileName);
		PRINT_MESSAGE(buf1);
		free(buf1);
		return;
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockFile[FILE_NAME_LENGTH];

		(void) strncpy(lockFile, fileName, FILE_NAME_LENGTH - 6);
		lockFile[FILE_NAME_LENGTH - 6] = '\0';
		(void) strncat(lockFile, ".lock", 6);
		while (((lfd = open(lockFile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
				"Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
				"Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (i = 0; i < MAX_ORIENT; i++) {
			for (j = 0; j < MAX_TILES - MIN_TILES + 1; j++) {
				for (k = 0; k < MAX_TILES - MIN_TILES + 1; k++)
					(void) fprintf(fp, "%d %s\n",
						puzzleRecord[i][j][k].score,
						puzzleRecord[i][j][k].name);
				if (j < MAX_TILES - MIN_TILES ||
						i < MAX_ORIENT - 1)
					(void) fprintf(fp, "\n");
			}
			if (i < MAX_ORIENT - 1)
				(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockFile);
#endif
		(void) fclose(fp);
	}
}

static void
printRecord(int sizeX, int sizeY, Boolean corners)
{
	int i = (corners) ? 1 : 0;
	int j = sizeX - MIN_TILES, k = sizeY - MIN_TILES;

	if (sizeX > MAX_TILES || sizeY > MAX_TILES) {
		(void) strncpy(recordDsp, "NOT RECORDED", 13);
	} else if (puzzleRecord[i][j][k].score <= NEVER) {
		(void) strncpy(recordDsp, "NEVER ", 7);
		(void) strncat(recordDsp, NOACCESS, strlen(NOACCESS) + 1);
	} else {
#ifdef HAVE_SNPRINTF
		(void) snprintf(recordDsp, MESSAGE_LENGTH, "%d %s",
			puzzleRecord[i][j][k].score,
			puzzleRecord[i][j][k].name);
#else
		(void) sprintf(recordDsp, "%d %s",
			puzzleRecord[i][j][k].score,
			puzzleRecord[i][j][k].name);
#endif
	}
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	printState(recordText, recordDsp);
#endif
}

static void
printStatus(char *msg, int nMoves
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
		, int sizeX, int sizeY
#endif
		)
{
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	printState(message, msg);
	intCpy(buff, nMoves);
	printState(movesText, buff);
#else
	char titleDsp[TITLE_LENGTH] = "";
#ifdef HAVE_SNPRINTF
	(void) snprintf(titleDsp, TITLE_LENGTH,
		"%s: %dx%d@ (%d/%s) - %s",
		progDsp, sizeX, sizeY, nMoves, recordDsp, msg);
#else
	(void) sprintf(titleDsp,
		"%s: %dx%d@ (%d/%s) - %s",
		progDsp, sizeX, sizeY, nMoves, recordDsp, msg);
#endif
#ifdef WINVER
	SetWindowText(widget.core.hWnd, (LPSTR) titleDsp);
#else
	XtVaSetValues(XtParent(puzzle),
		XtNtitle, titleDsp, NULL);
#endif
#endif
}

static Boolean
handleSolved(int counter, int sizeX, int sizeY, int corners)
{
	int i = (corners) ? 1 : 0;
	int j = sizeX - MIN_TILES, k = sizeY - MIN_TILES;

	if (sizeX <= MAX_TILES && sizeY <= MAX_TILES &&
			(counter < puzzleRecord[i][j][k].score ||
			puzzleRecord[i][j][k].score <= NEVER)) {
		readRecords();	/* Maybe its been updated by another */
		puzzleRecord[i][j][k].score = counter;
		(void) strncpy(puzzleRecord[i][j][k].name, userNameDsp,
			USER_NAME_LENGTH);
		writeRecords();
		printRecord(sizeX, sizeY, corners);
		return True;
	}
	return False;
}

#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
static int
getSpeed(int delay)
{
	return MAX_SPEED + MIN_SPEED - delay - 1;
}

static int
getDelay(int speed)
{
	return MAX_SPEED + MIN_SPEED - speed - 1;
}

static void
cornersCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmToggleButtonCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		int state
#endif
		)
{
	int sizeX, sizeY;
	Boolean corners, old;

#ifdef HAVE_MOTIF
	corners = cbs->set;
#elif defined(HAVE_ATHENA)
	corners = state;
#endif
	XtVaGetValues(puzzle,
		XtNcorners, &old,
		XtNsizeX, &sizeX,
		XtNsizeY, &sizeY, NULL);
	if (old == corners)
		return;
	if (corners && sizeY == 1) {
#ifdef HAVE_MOTIF
		XmToggleButtonSetState(cornersToggle, !corners, True);
#elif defined(HAVE_ATHENA)
		XtVaSetValues(cornersToggle,
			XtNstate, !corners, NULL);
#endif
		return;
	}
	XtVaSetValues(puzzle,
		XtNcorners, corners, NULL);
	movesDsp = 0;
	intCpy(buff, movesDsp);
	printState(movesText, buff);
	printRecord(sizeX, sizeY, corners);
	messageDsp[0] = '\0';
	printState(message, messageDsp);
}

#if defined(HAVE_ATHENA) && defined(USE_SPIN)
static void
tileXUpCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int sizeX;

	XtVaGetValues(puzzle,
		XtNsizeX, &sizeX, NULL);
	sizeX++;
	setScale(tileXSlider, tileXSliderLabel, sizeX,
		MIN_TILES, MAX_TILES, True);
	XtVaSetValues(puzzle,
		XtNsizeX, sizeX, NULL);
}

static void
tileXDownCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int sizeX;

	XtVaGetValues(puzzle,
		XtNsizeX, &sizeX, NULL);
	sizeX--;
	if (sizeX < MIN_TILES) {
		return;
	}
	setScale(tileXSlider, tileXSliderLabel, sizeX,
		MIN_TILES, MAX_TILES, True);
	XtVaSetValues(puzzle,
		XtNsizeX, sizeX, NULL);
}

#else

static void
tileXChangeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XmSpinBoxCallbackStruct *cbs
#else
		XmScaleCallbackStruct *cbs
#endif
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
	int sizeX, sizeY, old;
	Boolean corners;

#ifdef HAVE_MOTIF
#ifdef USE_SPIN
	int limit;
	sizeX = cbs->position;
#else
	sizeX = cbs->value;
#endif
#elif defined(HAVE_ATHENA)
	Widget label = (Widget) clientData;
	if (!thumbScroll(callData, &sizeX,
			MIN_TILES, MAX_TILES, SCROLL_SIZE)) {
		return;
	}
#endif
	XtVaGetValues(puzzle,
		XtNsizeX, &old,
		XtNsizeY, &sizeY,
		XtNcorners, &corners, NULL);
	if (old == sizeX)
		return;
#if defined(HAVE_MOTIF) && defined(USE_SPIN)
	XtVaGetValues(tileXChanger,
		XmNmaximumValue, &limit, NULL);
	if (sizeX >= limit)
		XtVaSetValues(tileXChanger,
			XmNmaximumValue, sizeX + 1,
			XmNposition, sizeX, NULL);
#elif defined(HAVE_ATHENA)
	setScale(w, label, sizeX, MIN_TILES, MAX_TILES, False);
#endif
	XtVaSetValues(puzzle,
		XtNsizeX, sizeX, NULL);
	movesDsp = 0;
	intCpy(buff, movesDsp);
	printState(movesText, buff);
	printRecord(sizeX, sizeY, corners);
	messageDsp[0] = '\0';
	printState(message, messageDsp);
}
#endif

#if defined(HAVE_ATHENA) && defined(USE_SPIN)
static void
tileYUpCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int sizeY;

	XtVaGetValues(puzzle,
		XtNsizeY, &sizeY, NULL);
	sizeY++;
	setScale(tileYSlider, tileYSliderLabel, sizeY,
		MIN_TILES, MAX_TILES, True);
	XtVaSetValues(puzzle,
		XtNsizeY, sizeY, NULL);
}

static void
tileYDownCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int sizeY;

	XtVaGetValues(puzzle,
		XtNsizeY, &sizeY, NULL);
	sizeY--;
	if (sizeY < MIN_TILES) {
		return;
	}
	setScale(tileYSlider, tileYSliderLabel, sizeY,
		MIN_TILES, MAX_TILES, True);
	XtVaSetValues(puzzle,
		XtNsizeY, sizeY, NULL);
}

#else

static void
tileYChangeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XmSpinBoxCallbackStruct *cbs
#else
		XmScaleCallbackStruct *cbs
#endif
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
	int sizeX, sizeY, old;
	Boolean corners;

#ifdef HAVE_MOTIF
#ifdef USE_SPIN
	int limit;
	sizeY = cbs->position;
#else
	sizeY = cbs->value;
#endif
#elif defined(HAVE_ATHENA)
	Widget label = (Widget) clientData;
	if (!thumbScroll(callData, &sizeY,
			MIN_TILES, MAX_TILES, SCROLL_SIZE)) {
		return;
	}
#endif
	XtVaGetValues(puzzle,
		XtNsizeX, &sizeX,
		XtNsizeY, &old,
		XtNcorners, &corners, NULL);
	if (old == sizeY)
		return;
	if (corners && sizeY == 1) {
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XtVaSetValues(tileYChanger,
			XmNposition, old, NULL);
#else
		XmScaleSetValue(tileYChanger, old);
#endif
#endif
		return;
	}
#if defined(HAVE_MOTIF) && defined(USE_SPIN)
	XtVaGetValues(tileYChanger,
		XmNmaximumValue, &limit, NULL);
	if (sizeY >= limit)
		XtVaSetValues(tileYChanger,
			XmNmaximumValue, sizeY + 1,
			XmNposition, sizeY, NULL);
#elif defined(HAVE_ATHENA)
	setScale(w, label, sizeY, MIN_TILES, MAX_TILES, False);
#endif
	XtVaSetValues(puzzle,
		XtNsizeY, sizeY, NULL);
	movesDsp = 0;
	intCpy(buff, movesDsp);
	printState(movesText, buff);
	printRecord(sizeX, sizeY, corners);
	messageDsp[0] = '\0';
	printState(message, messageDsp);
}
#endif

#ifdef ANIMATE
#if defined(HAVE_ATHENA) && defined(USE_SPIN)
static void
speedUpCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int delay, speed;

	XtVaGetValues(puzzle,
		XtNdelay, &delay, NULL);
	speed = getSpeed(delay);
	speed++;
	if (speed > MAX_SPEED) {
		return;
	}
	setScale(speedSlider, speedSliderLabel, speed,
		MIN_SPEED, MAX_SPEED, True);
	delay = getDelay(speed);
	XtVaSetValues(puzzle,
		XtNdelay, delay, NULL);
}

static void
speedDownCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int delay, speed;

	XtVaGetValues(puzzle,
		XtNdelay, &delay, NULL);
	speed = getSpeed(delay);
	speed--;
	if (speed < MIN_SPEED) {
		return;
	}
	setScale(speedSlider, speedSliderLabel, speed,
		MIN_SPEED, MAX_SPEED, True);
	delay = getDelay(speed);
	XtVaSetValues(puzzle,
		XtNdelay, delay, NULL);
}

#else

static void
speedChangeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XmSpinBoxCallbackStruct *cbs
#else
		XmScaleCallbackStruct *cbs
#endif
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
	int delay, speed, old;

#ifdef HAVE_MOTIF
#ifdef USE_SPIN
	speed = cbs->position;
#else
	speed = cbs->value;
#endif
#elif defined(HAVE_ATHENA)
	Widget label = (Widget) clientData;
	if (!thumbScroll(callData, &speed,
			MIN_SPEED, MAX_SPEED, SCROLL_SIZE)) {
		return;
	}
#endif
	delay = getDelay(speed);
	XtVaGetValues(puzzle,
		XtNdelay, &old, NULL);
	if (old == delay)
		return;
#ifdef HAVE_ATHENA
	setScale(w, label, speed, MIN_SPEED, MAX_SPEED, True);
#endif
	XtVaSetValues(puzzle,
		XtNdelay, delay, NULL);
}
#endif
#endif

static void
puzzleClearCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmAnyCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
#ifdef HAVE_MOTIF
	if (cbs->reason == XmCR_OK)
#elif defined(HAVE_ATHENA)
	XtPopdown(clearDialog);
#endif
		XtVaSetValues(puzzle,
			XtNmenu, ACTION_CLEAR, NULL);
}

static void
fileMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData) + ACTION_GET;

	if (value == ACTION_EXIT)
		exit(0);
	XtVaSetValues(puzzle,
		XtNmenu, value, NULL);
}

static void
setSoundCheck(Widget w)
{
	Boolean sound;
	XtVaGetValues(puzzle,
		XtNsound, &sound, NULL);
#ifdef HAVE_MOTIF
	XmToggleButtonSetState(w, sound, False);
#elif defined(HAVE_ATHENA)
	menuCheck(topLevel, w, sound);
#endif
}

static void
playMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData) + ACTION_UNDO;

#ifndef EXTRA
	if (value >= ACTION_CORNERS)
		value += ACTION_SOUND - ACTION_CORNERS;
#endif
	XtVaSetValues(puzzle,
		XtNmenu, value, NULL);
	if (value == ACTION_SOUND)
		setSoundCheck(w);
}

static void
helpMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);
	Widget dialog;

#ifdef HAVE_ATHENA
	if (wmDeleteWindow == None)
		wmDeleteWindow = XInternAtom(XtDisplay(topLevel),
			"WM_DELETE_WINDOW", FALSE);
#endif
	switch (value) {
	case 0:
		dialog = descriptionDialog;
		break;
	case 1:
		dialog = featuresDialog;
		break;
	case 2:
		dialog = synopsisDialog;
		break;
	case 3:
		dialog = referencesDialog;
		break;
	case 4:
		dialog = aboutDialog;
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpMenuCallback: %d", value);
			XtWarning(buf);
			free(buf);
			return;
		}
	}
#ifdef HAVE_MOTIF
	XtManageChild(dialog);
#elif defined(HAVE_ATHENA)
	XtPopup(dialog, XtGrabNone);
	XSetWMProtocols(XtDisplay(topLevel),
		XtWindow(dialog), &wmDeleteWindow, 1);
#endif
}
#endif

static void
initialize(
#ifdef WINVER
HexagonsWidget w, HBRUSH brush
#else
Widget w
#endif
)
{
	int sizeX, sizeY;
	Boolean corners;
	char *userName, *scoreFile;

#ifdef WINVER
	initializePuzzle(w, brush);

	sizeX = w->hexagons.sizeX;
	sizeY = w->hexagons.sizeY;
	corners = w->hexagons.corners;
	userName = w->hexagons.userName;
	scoreFile = w->hexagons.scoreFile;
	if (strcmp(scoreFile, "")) {
		(void) strncpy(scoreFileName, scoreFile, FILE_NAME_LENGTH - 1);
		scoreFileName[FILE_NAME_LENGTH - 1] = '\0';
	}
#else
	Boolean scoreOnly, versionOnly;

	XtVaGetValues(w,
		XtNuserName, &userName,
		XtNscoreFile, &scoreFile,
		XtNsizeX, &sizeX,
		XtNsizeY, &sizeY,
		XtNcorners, &corners,
		XtNscoreOnly, &scoreOnly,
		XtNversionOnly, &versionOnly, NULL);
	if (versionOnly) {
		(void) printf("%s\n", aboutHelp);
		exit(0);
	}
	if (strcmp(scoreFile, "")) {
		(void) strncpy(scoreFileName, scoreFile, FILE_NAME_LENGTH - 1);
		scoreFileName[FILE_NAME_LENGTH - 1] = '\0';
	}
	if (scoreOnly) {
		initRecords();
		readRecords();
		printRecords();
		exit(0);
	}
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
	{
		int limit;
		XtVaGetValues(tileXChanger,
			XmNmaximumValue, &limit, NULL);
		if (sizeX >= limit)
			XtVaSetValues(tileXChanger,
				XmNposition, sizeX,
				XmNmaximumValue, sizeX + 1, NULL);
		else
			XtVaSetValues(tileXChanger,
				XmNposition, sizeX, NULL);
		XtVaGetValues(tileYChanger,
			XmNmaximumValue, &limit, NULL);
		if (sizeY >= limit)
			XtVaSetValues(tileYChanger,
				XmNposition, sizeY,
				XmNmaximumValue, sizeY + 1, NULL);
		else
			XtVaSetValues(tileYChanger,
				XmNposition, sizeY, NULL);
	}
#else
	if (sizeX > MAX_TILES)
		XtVaSetValues(tileXChanger,
			XmNmaximum, sizeX, NULL);
	XmScaleSetValue(tileXChanger, sizeX);
	if (sizeY > MAX_TILES)
		XtVaSetValues(tileYChanger,
			XmNmaximum, sizeY, NULL);
	XmScaleSetValue(tileYChanger, sizeY);
#endif
	XmToggleButtonSetState(cornersToggle, corners, True);
#ifdef ANIMATE
	{
		int delay, speed;
		XtVaGetValues(w,
			XtNdelay, &delay, NULL);
		speed = getSpeed(delay);
#ifdef USE_SPIN
		XtVaSetValues(speedChanger,
			XmNposition, speed, NULL);
#else
		XmScaleSetValue(speedChanger, speed);
#endif
	}
#endif
	setSoundCheck(soundMenuItem);
#elif defined(HAVE_ATHENA)
	XtVaSetValues(cornersToggle,
		XtNstate, corners, NULL);
	setScale(tileXSlider, tileXSliderLabel, sizeX,
		MIN_TILES, MAX_TILES, False);
	setScale(tileYSlider, tileYSliderLabel, sizeY,
		MIN_TILES, MAX_TILES, False);
#ifdef ANIMATE
	{
		int delay, speed;
		XtVaGetValues(w,
			XtNdelay, &delay, NULL);
		speed = getSpeed(delay);
		setScale(speedSlider, speedSliderLabel, speed,
			MIN_SPEED, MAX_SPEED, True);
	}
#endif
	setSoundCheck(soundMenuItem);
#endif
#endif
	SET_STARTED(w, False);
	initRecords();
	readRecords();
#ifndef WINVER
	(void) strncpy(userNameDsp, userName, USER_NAME_LENGTH - 1);
	userNameDsp[USER_NAME_LENGTH - 1] = '\0';
#endif
	if (!strcmp(userName, "") || !strcmp(userName, "(null)") ||
			!strcmp(userName, NOACCESS) ||
			!strcmp(userName, NOBODY)) {
#ifdef WINVER
		(void) strncpy(userNameDsp, userName, USER_NAME_LENGTH - 1);
		userNameDsp[USER_NAME_LENGTH - 1] = '\0';
#else
		char *login;

#ifdef HAVE_CUSERID
		login = cuserid(NULL);
#else
		login = getlogin();
#endif
		if (login == NULL) {
			userNameDsp[0] = '\0';
		} else {
			(void) strncpy(userNameDsp, login, USER_NAME_LENGTH - 1);
			userNameDsp[USER_NAME_LENGTH - 1] = '\0';
		}
		if (!strcmp(userNameDsp, "") ||
				!strcmp(userNameDsp, "(null)") ||
				!strcmp(userNameDsp, NOACCESS) ||
				!strcmp(userNameDsp, NOBODY))
			/* It really IS nobody */
			(void) strncpy(userNameDsp, "guest", 6);
#endif
	}
	printRecord(sizeX, sizeY, corners);
	printStatus(messageDsp, movesDsp
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
		, sizeX, sizeY
#endif
		);
}

#ifdef WINVER
void
setPuzzle(HexagonsWidget w, int reason)
#else
static void
puzzleCallback(Widget w, caddr_t clientData,
		hexagonsCallbackStruct *callData)
#endif
{
#ifndef WINVER
	int reason = callData->reason;
#endif
#if defined(HAVE_MOTIF) && defined(USE_SPIN)
	int limit;
#endif
	int sizeX, sizeY;
	Boolean corners, cheat;

	messageDsp[0] = '\0';
#ifdef WINVER
	sizeX = w->hexagons.sizeX;
	sizeY = w->hexagons.sizeY;
	corners = w->hexagons.corners;
	cheat = w->hexagons.cheat;
#else
	XtVaGetValues(w,
		XtNsizeX, &sizeX,
		XtNsizeY, &sizeY,
		XtNcorners, &corners,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
	case ACTION_HIDE:
#ifdef WINVER
		ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
		(void) XIconifyWindow(XtDisplay(topLevel),
			XtWindow(topLevel),
			XScreenNumberOfScreen(XtScreen(topLevel)));
#endif
		break;
#if !defined(WINVER) && defined(CLEAR_QUERY)
	case ACTION_CLEAR_QUERY:
#ifdef HAVE_MOTIF
		XtManageChild(clearDialog);
#elif defined(HAVE_ATHENA)
		XtPopup(clearDialog, XtGrabNone);
		XSetWMProtocols(XtDisplay(topLevel),
			XtWindow(clearDialog), &wmDeleteWindow, 1);
#else
		XtVaSetValues(puzzle,
			XtNmenu, ACTION_CLEAR, NULL);
#endif
		break;
#endif
	case ACTION_SOLVE_MESSAGE:
#ifdef WINVER
		(void) MessageBox(w->core.hWnd, solveHelp,
			"Auto-solve", MB_OK);
#else
#ifdef HAVE_MOTIF
		XtManageChild(solveDialog);
#elif defined(HAVE_ATHENA)
		XtPopup(solveDialog, XtGrabNone);
#else
		(void) strncpy(messageDsp, solveHelp, MESSAGE_LENGTH);
#endif
#endif
		break;
	case ACTION_RESTORE:
	case ACTION_RESET:
		movesDsp = 0;
		randomized = False;
		break;
	case ACTION_AMBIGUOUS:
		(void) strncpy(messageDsp, "Ambiguous move", MESSAGE_LENGTH);
		break;
	case ACTION_BLOCKED:
		(void) strncpy(messageDsp, "Blocked", MESSAGE_LENGTH);
		break;
	case ACTION_SPACE:
#if 0
		/* Too annoying */
		if (corners)
			(void) strncpy(messageDsp, "Spaces cannot slide",
				MESSAGE_LENGTH);
		else
			(void) strncpy(messageDsp, "A space cannot slide",
				MESSAGE_LENGTH);
#endif
		break;
	case ACTION_IGNORE:
		(void) strncpy(messageDsp, "Clear to start",
			MESSAGE_LENGTH);
		break;
	case ACTION_MOVED:
		movesDsp++;
		SET_STARTED(w, True);
		break;
	case ACTION_SOLVED:
		if (cheat) {
			(void) strncpy(messageDsp, "No cheating ", 13);
			(void) strncat(messageDsp, userNameDsp, USER_NAME_LENGTH);
			(void) strncat(messageDsp, "!!", 3);
		} else if (!randomized && handleSolved(movesDsp,
				sizeX, sizeY, corners)) {
			(void) strncpy(messageDsp, "Congratulations ", 17);
			(void) strncat(messageDsp, userNameDsp, USER_NAME_LENGTH);
			(void) strncat(messageDsp, "!!", 3);
		} else {
			(void) strncpy(messageDsp, "Solved!", 8);
		}
		SET_STARTED(w, False);
		break;
	case ACTION_COMPUTED:
		SET_STARTED(w, False);
		break;
	case ACTION_RANDOMIZE:
		movesDsp = 0;
		randomized = True;
		SET_STARTED(w, False);
		break;
	case ACTION_INCX:
		movesDsp = 0;
		sizeX++;
		printRecord(sizeX, sizeY, corners);
#ifdef WINVER
		w->hexagons.sizeX = sizeX;
#else
		XtVaSetValues(w,
			XtNsizeX, sizeX, NULL);
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XtVaGetValues(tileXChanger,
			XmNmaximumValue, &limit, NULL);
		if (sizeX >= limit)
			XtVaSetValues(tileXChanger,
				XmNmaximumValue, sizeX + 1,
				XmNposition, sizeX, NULL);
		else
			XtVaSetValues(tileXChanger,
				XmNposition, sizeX, NULL);
#else
		if (sizeX > MAX_TILES)
			XtVaSetValues(tileXChanger,
				XmNmaximum, sizeX, NULL);
		XmScaleSetValue(tileXChanger, sizeX);
#endif
#elif defined(HAVE_ATHENA)
		setScale(tileXSlider, tileXSliderLabel, sizeX,
			MIN_TILES, MAX_TILES, False);
#endif
#endif
		break;
	case ACTION_DECX:
		movesDsp = 0;
		sizeX--;
		printRecord(sizeX, sizeY, corners);
#ifdef WINVER
		w->hexagons.sizeX = sizeX;
#else
		XtVaSetValues(w,
			XtNsizeX, sizeX, NULL);
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		if (sizeX > MAX_TILES)
			XtVaSetValues(tileXChanger,
				XmNmaximumValue, sizeX,
				XmNposition, sizeX, NULL);
		else
			XtVaSetValues(tileXChanger,
				XmNposition, sizeX, NULL);
#else
		if (sizeX > MAX_TILES)
			XtVaSetValues(tileXChanger,
				XmNmaximum, sizeX, NULL);
		XmScaleSetValue(tileXChanger, sizeX);
#endif
#elif defined(HAVE_ATHENA)
		setScale(tileXSlider, tileXSliderLabel, sizeX,
			MIN_TILES, MAX_TILES, False);
#endif
#endif
		break;
	case ACTION_INCY:
		movesDsp = 0;
		sizeY++;
		printRecord(sizeX, sizeY, corners);
#ifdef WINVER
		w->hexagons.sizeY = sizeY;
#else
		XtVaSetValues(w,
			XtNsizeY, sizeY, NULL);
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XtVaGetValues(tileYChanger,
			XmNmaximumValue, &limit, NULL);
		if (sizeY >= limit)
			XtVaSetValues(tileYChanger,
				XmNmaximumValue, sizeY + 1,
				XmNposition, sizeY, NULL);
		else
			XtVaSetValues(tileYChanger,
				XmNposition, sizeY, NULL);
#else
		if (sizeY > MAX_TILES)
			XtVaSetValues(tileYChanger,
				XmNmaximum, sizeY, NULL);
		XmScaleSetValue(tileYChanger, sizeY);
#endif
#elif defined(HAVE_ATHENA)
		setScale(tileYSlider, tileYSliderLabel, sizeY,
			MIN_TILES, MAX_TILES, False);
#endif
#endif
		break;
	case ACTION_DECY:
		movesDsp = 0;
		sizeY--;
		printRecord(sizeX, sizeY, corners);
#ifdef WINVER
		w->hexagons.sizeY = sizeY;
#else
		XtVaSetValues(w,
			XtNsizeY, sizeY, NULL);
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		if (sizeY > MAX_TILES)
			XtVaSetValues(tileYChanger,
				XmNmaximumValue, sizeY,
				XmNposition, sizeY, NULL);
		else
			XtVaSetValues(tileYChanger,
				XmNposition, sizeY, NULL);
#else
		if (sizeY > MAX_TILES)
			XtVaSetValues(tileYChanger,
				XmNmaximum, sizeY, NULL);
		XmScaleSetValue(tileYChanger, sizeY);
#endif
#elif defined(HAVE_ATHENA)
		setScale(tileYSlider, tileYSliderLabel, sizeY,
			MIN_TILES, MAX_TILES, False);
#endif
#endif
		break;
	case ACTION_CORNERS:
		movesDsp = 0;
		corners = !corners;
		printRecord(sizeX, sizeY, corners);
#ifdef WINVER
		w->hexagons.corners = corners;
#else
		XtVaSetValues(w,
			XtNcorners, corners, NULL);
#ifdef HAVE_MOTIF
		XmToggleButtonSetState(cornersToggle, corners, True);
#elif defined(HAVE_ATHENA)
		XtVaSetValues(cornersToggle,
			XtNstate, corners, NULL);
#endif
#endif
		break;
	case ACTION_UNDO:
		movesDsp--;
		SET_STARTED(w, True);
		break;
	case ACTION_REDO:
		movesDsp++;
		SET_STARTED(w, True);
		break;
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
#ifdef ANIMATE
	case ACTION_SPEED:
		{
			int oldDelay, delay, speed;

			XtVaGetValues(puzzle,
				XtNdelay, &delay, NULL);
			oldDelay = delay;
			if (delay > MAX_SPEED - MIN_SPEED)
				delay = MAX_SPEED - MIN_SPEED;
			if (delay < 0)
				delay = 0;
			if (delay != oldDelay) {
				XtVaSetValues(w,
					XtNdelay, delay, NULL);
			}
			speed = getSpeed(delay);
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
			XtVaSetValues(speedChanger,
				XmNposition, speed, NULL);
#else
			XmScaleSetValue(speedChanger, speed);
#endif
#elif defined(HAVE_ATHENA)
			setScale(speedSlider, speedSliderLabel, speed,
				MIN_SPEED, MAX_SPEED, True);
#endif
		}
		return;
#endif
	case ACTION_SOUND:
		setSoundCheck(soundMenuItem);
		break;
#endif
	}
	printStatus(messageDsp, movesDsp
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
		, sizeX, sizeY
#endif
		);
}

#ifdef WINVER
static LRESULT CALLBACK
about(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	if (message == WM_COMMAND && LOWORD(wParam) == IDOK) {
		(void) EndDialog(hDlg, TRUE);
		return TRUE;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;

	widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.hexagons.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			enterPuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.hexagons.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			leavePuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
	case WM_CREATE:
		initialize(&widget, brush);
		break;
	case WM_DESTROY:
		destroyPuzzle(&widget, brush);
		break;
	case WM_SIZE:
		resizePuzzle(&widget);
		(void) InvalidateRect(hWnd, NULL, TRUE);
		break;
	case WM_PAINT:
		widget.core.hDC = BeginPaint(hWnd, &paint);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		exposePuzzle(&widget);
		(void) EndPaint(hWnd, &paint);
		break;
	case WM_RBUTTONDOWN:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		clearPuzzle(&widget);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONDOWN:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		selectPuzzle(&widget, LOWORD(lParam), HIWORD(lParam));
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONUP:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		releasePuzzle(&widget, LOWORD(lParam), HIWORD(lParam));
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
#if (_WIN32_WINNT >= 0x0400) || (_WIN32_WINDOWS > 0x0400)
	case WM_MOUSEWHEEL:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		{
			int zDelta = ((short) HIWORD(wParam));
			POINT cursor, origin;

			origin.x = 0, origin.y = 0;
			ClientToScreen(hWnd, &origin);
			(void) GetCursorPos(&cursor);
			if (zDelta > (WHEEL_DELTA >> 1)) {
				movePuzzle(&widget, TOP,
			(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
				if (GetKeyState(VK_CONTROL) >> 1) {
					sizePuzzle(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
				}
			} else if (zDelta < -(WHEEL_DELTA >> 1)) {
				movePuzzle(&widget, BOTTOM,
			(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
				if (GetKeyState(VK_CONTROL) >> 1) {
					sizePuzzle(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
				}
			}
		}
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
#endif
	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case ACTION_GET:
			getPuzzle(&widget);
			resizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_WRITE:
			writePuzzle(&widget);
			break;
		case ACTION_EXIT:
			destroyPuzzle(&widget, brush);
			break;
		case ACTION_HIDE:
			hidePuzzle(&widget);
			break;
		case ACTION_UNDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			undoPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_REDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			redoPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_CLEAR:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			clearPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_RANDOMIZE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			randomizePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_SOLVE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			solvePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_CORNERS:
			cornersPuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_TR:
		case ACTION_RIGHT:
		case ACTION_BR:
		case ACTION_BL:
		case ACTION_LEFT:
		case ACTION_TL:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			(void) movePuzzle(&widget,
				(int) LOWORD(wParam) - ACTION_TR, FALSE);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_DECY:
		case ACTION_INCX:
		case ACTION_INCY:
		case ACTION_DECX:
			if (movePuzzle(&widget,
					(int) LOWORD(wParam) - ACTION_INCY,
					TRUE)) {
				sizePuzzle(&widget);
				(void) InvalidateRect(hWnd, NULL, TRUE);
			}
			break;
		case ACTION_SPEED:
			speedUpPuzzle(&widget);
			break;
		case ACTION_SLOW:
			slowDownPuzzle(&widget);
			break;
		case ACTION_SOUND:
			toggleSoundPuzzle(&widget);
			break;
		case ACTION_DESCRIPTION:
			(void) MessageBox(hWnd, descriptionHelp,
				"Description", MB_OK | MB_ICONQUESTION);
			break;
		case ACTION_FEATURES:
			(void) MessageBox(hWnd, featuresHelp,
				"Features", MB_OK | MB_ICONEXCLAMATION);
			break;
		case ACTION_REFERENCES:
			(void) MessageBox(hWnd, referencesHelp,
				"References", MB_OK | MB_ICONINFORMATION);
			break;
		case ACTION_ABOUT:
			(void) DialogBox(widget.core.hInstance,
				"About", hWnd, (DLGPROC) about);
			break;
		}
		break;
	default:
		return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(XtPointer) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (int) msg.wParam;
}

#else

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*hexagons.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*hexagons.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*hexagons.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*hexagons.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*hexagons.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*hexagons.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*hexagons.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*hexagons.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-tile", (char *) "*hexagons.tileColor", XrmoptionSepArg, NULL},
	{(char *) "-text", (char *) "*hexagons.textColor", XrmoptionSepArg, NULL},
	{(char *) "-frame", (char *) "*hexagons.frameColor", XrmoptionSepArg, NULL},
	{(char *) "-stippleFrame", (char *) "*hexagons.stippleFrame", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nostippleFrame", (char *) "*hexagons.stippleFrame", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-install", (char *) "*hexagons.install", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noinstall", (char *) "*hexagons.install", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-picture", (char *) "*hexagons.picture", XrmoptionSepArg, NULL},
	{(char *) "-delay", (char *) "*hexagons.delay", XrmoptionSepArg, NULL},
	{(char *) "-sound", (char *) "*hexagons.sound", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosound", (char *) "*hexagons.sound", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-bumpSound", (char *) "*hexagons.bumpSound", XrmoptionSepArg, NULL},
	{(char *) "-base", (char *) "*hexagons.base", XrmoptionSepArg, NULL},
	{(char *) "-fn", (char *) "*hexagons.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*hexagons.font", XrmoptionSepArg, NULL},
	{(char *) "-sizeX", (char *) "*hexagons.sizeX", XrmoptionSepArg, NULL},
	{(char *) "-sizeY", (char *) "*hexagons.sizeY", XrmoptionSepArg, NULL},
	{(char *) "-corners", (char *) "*hexagons.corners", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nocorners", (char *) "*hexagons.corners", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-userName", (char *) "*hexagons.userName", XrmoptionSepArg, NULL},
	{(char *) "-scoreFile", (char *) "*hexagons.scoreFile", XrmoptionSepArg, NULL},
	{(char *) "-scores", (char *) "*hexagons.scoreOnly", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-version", (char *) "*hexagons.versionOnly", XrmoptionNoArg, (char *) "TRUE"}
};

#ifdef HAVE_MOTIF
static void
soundCallback(Widget w, XtPointer clientData,
		XmToggleButtonCallbackStruct *cbs)
{
	Boolean value = cbs->set;

	XtVaSetValues(puzzle,
		XtNsound, value, NULL);
}

#elif defined(HAVE_ATHENA)
static void
puzzleClearCancelCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	XtPopdown(clearDialog);
}

static void
createClearQuery(char *title, char *text)
{
	Widget dialog, okDialog, cancelDialog;

	clearDialog = XtCreatePopupShell(title,
		transientShellWidgetClass, topLevel, NULL, 0);
	dialog = XtVaCreateManagedWidget("dialog",
		dialogWidgetClass, clearDialog,
		XtNlabel, text, NULL);
	okDialog = XtVaCreateManagedWidget("OK",
		commandWidgetClass, dialog, NULL);
	XtAddCallback(okDialog, XtNcallback,
		puzzleClearCallback, dialog);
	cancelDialog = XtVaCreateManagedWidget("Cancel",
		commandWidgetClass, dialog, NULL);
	XtAddCallback(cancelDialog, XtNcallback,
		puzzleClearCancelCallback, dialog);
}
#endif

int
main(int argc, char **argv)
{
	int pixmapSize;
	XtAppContext appCon;
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	Widget mainPanel, controlPanel, menuBar;
	Widget movesRowCol, changerRowCol;
	Widget playMenu;
#endif
#ifdef HAVE_MOTIF
	Widget pullDownMenu, widget;
	Widget menuBarPanel;
	Widget tileXChangerRowCol, tileYChangerRowCol;
#ifdef ANIMATE
	Widget speedChangerRowCol;
#endif
	Widget speedRowCol, messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString undoString, redoString;
	XmString clearString, randomizeString, solveString;
#ifdef EXTRA
	XmString modeString;
	XmString speedString, slowString;
#endif
	XmString soundString;
#elif defined(HAVE_ATHENA)
	XtActionsRec actions[] = {
		{(char *) "DeleteWindowProc", deleteWindowProc},
		{(char *) "ClosePopupPanel", (XtActionProc) closePopupPanel}
	};
	String fallbackResources[] = {
		(char *) "?.translations: #override <Message>WM_PROTOCOLS: DeleteWindowProc()",
		(char *) "?.TransientShell.translations: #override <Message>WM_PROTOCOLS: ClosePopupPanel()",
		NULL
	};
	Widget tileXBox, tileYBox, speedBox;
	Widget cornersBox;
	Widget movesLabel, recordLabel, toggleRowCol;
	Widget w;
	Widget fileLabel, playLabel, helpLabel;
	int sizeX, sizeY, delay;
	Boolean corners;
	unsigned int i;
#endif

	progDsp = strrchr(argv[0], '/');
	if (progDsp != NULL)
		progDsp++;
	if (progDsp == NULL)
		progDsp = argv[0];
	topLevel = XtVaAppInitialize(NULL, "XHexagons",
		options, XtNumber(options), &argc, argv,
#ifdef HAVE_ATHENA
		fallbackResources,
#else
		NULL,
#endif
		NULL);
	appCon = XtWidgetToApplicationContext(topLevel);
	if (argc != 1)
		usage(argv[0], synopsisHelp);
#ifdef HAVE_MOTIF
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P', NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Exit");
	(void) XmVaCreateSimplePulldownMenu(menuBar, (char *) "fileMenu",
		0, fileMenuCallback,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'x', NULL, NULL, NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	undoString = XmStringCreateSimple((char *) "Undo");
	redoString = XmStringCreateSimple((char *) "Redo");
	clearString = XmStringCreateSimple((char *) "Clear");
	randomizeString = XmStringCreateSimple((char *) "Randomize");
	solveString = XmStringCreateSimple((char *) "(Auto-solve)");
#ifdef EXTRA
	modeString = XmStringCreateSimple((char *) "Corners");
	speedString = XmStringCreateSimple((char *) "Speed >");
	slowString = XmStringCreateSimple((char *) "Slow <");
#endif
	soundString = XmStringCreateSimple((char *) "Sound @");
	playMenu = XmVaCreateSimplePulldownMenu(menuBar, (char *) "playMenu",
		1, playMenuCallback,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, redoString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, randomizeString, 'z', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 's', NULL, NULL,
#ifdef EXTRA
		XmVaPUSHBUTTON, modeString, 'o', NULL, NULL,
		XmVaPUSHBUTTON, speedString, '>', NULL, NULL,
		XmVaPUSHBUTTON, slowString, '<', NULL, NULL,
#endif
		XmVaTOGGLEBUTTON, soundString, '@', NULL, NULL, NULL);
	XmStringFree(undoString);
	XmStringFree(redoString);
	XmStringFree(clearString);
	XmStringFree(randomizeString);
	XmStringFree(solveString);
#ifdef EXTRA
	XmStringFree(modeString);
	XmStringFree(speedString);
	XmStringFree(slowString);
#endif
	XmStringFree(soundString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL); /* mnemonic warning on Cygwin */
	XtVaSetValues(menuBar,
		XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 0);
	widget = XtVaCreateManagedWidget("Features...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 1);
	widget = XtVaCreateManagedWidget("Synopsis...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'S', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 2);
	widget = XtVaCreateManagedWidget("References...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 3);
	widget = XtVaCreateManagedWidget("About...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 4);
	XtManageChild(menuBar);
	solveDialog = createHelp(menuBar, "Solve", (char *) solveHelp);
	clearDialog = createQuery(topLevel, "Clear Query",
		(char *) "Are you sure you want to start over?",
		(XtCallbackProc) puzzleClearCallback);
	mainPanel = XtVaCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel, NULL);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate and out of date. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move tile", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Clear", 6, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	movesText = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("nullText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, " ", 2, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	recordText = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	changerRowCol = XtVaCreateManagedWidget("changerRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
	tileXChangerRowCol = XtVaCreateManagedWidget("tileXChangerRowCol",
		xmRowColumnWidgetClass, changerRowCol,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
#ifdef USE_SPIN
	createSpinner(tileXChangerRowCol, &tileXChanger,
		(char *) tileLabels[0], DEFAULT_TILESX,
		MIN_TILES, MAX_TILES, 2,
		(XtCallbackProc) tileXChangeCallback);
#else
	createSlider(tileXChangerRowCol, &tileXChanger,
		(char *) tileLabels[0], DEFAULT_TILESX,
		MIN_TILES, MAX_TILES, 2, SCROLL_SIZE,
		(XtCallbackProc) tileXChangeCallback);
#endif
	tileYChangerRowCol = XtVaCreateManagedWidget("tileYChangerRowCol",
		xmRowColumnWidgetClass, changerRowCol,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
#ifdef USE_SPIN
	createSpinner(tileYChangerRowCol, &tileYChanger,
		(char *) tileLabels[1], DEFAULT_TILESY,
		MIN_TILES, MAX_TILES, 2,
		(XtCallbackProc) tileYChangeCallback);
#else
	createSlider(tileYChangerRowCol, &tileYChanger,
		(char *) tileLabels[0], DEFAULT_TILESY,
		MIN_TILES, MAX_TILES, 2, SCROLL_SIZE,
		(XtCallbackProc) tileYChangeCallback);
#endif
	speedRowCol = XtVaCreateManagedWidget("speedRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
	cornersToggle = XtVaCreateManagedWidget("Corners",
		xmToggleButtonWidgetClass, speedRowCol,
		XmNset, DEFAULT_CORNERS, NULL);
	XtAddCallback(cornersToggle, XmNvalueChangedCallback,
		(XtCallbackProc) cornersCallback, (XtPointer) NULL);
#ifdef ANIMATE
	speedChangerRowCol = XtVaCreateManagedWidget("speedChangerRowCol",
		xmRowColumnWidgetClass, speedRowCol,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
#ifdef USE_SPIN
	createSpinner(speedChangerRowCol, &speedChanger,
		(char *) speedLabel, MAX_SPEED - DEFAULT_DELAY,
		MIN_SPEED, MAX_SPEED, 2,
		(XtCallbackProc) speedChangeCallback);
#else
	createSlider(speedChangerRowCol, &speedChanger,
		(char *) speedLabel, MAX_SPEED - DEFAULT_DELAY,
		MIN_SPEED, MAX_SPEED, 2, SCROLL_SIZE,
		(XtCallbackProc) speedChangeCallback);
#endif
#endif
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Hexagons! (use mouse or keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);
	puzzle = XtVaCreateManagedWidget("hexagons",
		hexagonsWidgetClass, mainPanel, NULL);
#elif defined(HAVE_ATHENA)
	XtAppAddActions(appCon, actions, XtNumber(actions));
	createClearQuery((char *) "Clear Query",
		(char *) "Are you sure you want to start over?");
	createHelp(topLevel, &solveDialog, (char *) "Auto-solve",
		(char *) solveHelp, (XtCallbackProc) closePopupPanel2);
	createHelp(topLevel, &descriptionDialog, (char *) "Description",
		(char *) descriptionHelp, (XtCallbackProc) closePopupPanel2);
	createScrollHelp(topLevel, &featuresDialog, (char *) "Features",
		(char *) featuresHelp, (XtCallbackProc) closePopupPanel2);
	createHelp(topLevel, &synopsisDialog, (char *) "Synopsis",
		(char *) synopsisHelp, (XtCallbackProc) closePopupPanel2);
	createHelp(topLevel, &referencesDialog, (char *) "References",
		(char *) referencesHelp, (XtCallbackProc) closePopupPanel2);
	createHelp(topLevel, &aboutDialog, (char *) "About",
		(char *) aboutHelp, (XtCallbackProc) closePopupPanel2);
	mainPanel = XtVaCreateManagedWidget("form",
		panedWidgetClass, topLevel, NULL);
	menuBar = XtVaCreateManagedWidget("MenuBar",
		formWidgetClass, mainPanel,
		XtNborderWidth, 1, NULL);
	createMenu(menuBar, &fileLabel, NULL,
		fileTypes, "File", numFileTypes,
		0, False, fileMenuCallback);
	playLabel = XtVaCreateManagedWidget("Play",
		menuButtonWidgetClass, menuBar,
		XtNborderWidth, 0,
		XtNfromHoriz, fileLabel, NULL);
	playMenu = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, playLabel, NULL);
	for (i = 0; i < numPlayTypes; i++) {
		if (i == numPlayTypes - 1) {
			w = XtVaCreateManagedWidget(playTypes[i],
				smeBSBObjectClass, playMenu,
				XtNleftMargin, 20, NULL); /* for check */
			soundMenuItem = w;
		} else
			w = XtVaCreateManagedWidget(playTypes[i],
				smeBSBObjectClass, playMenu, NULL);
		XtAddCallback(w,
			XtNcallback, (XtCallbackProc) playMenuCallback,
			(XtPointer) (size_t) i);
	}
	createMenu(menuBar, &helpLabel, playLabel,
		helpTypes, "Help", numHelpTypes,
		0, False, helpMenuCallback);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		formWidgetClass, mainPanel,
		XtNborderWidth, 0, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		formWidgetClass, controlPanel,
		XtNborderWidth, 0, NULL);
	movesLabel = XtVaCreateManagedWidget("Moves",
		labelWidgetClass, movesRowCol,
		XtNborderWidth, 0, NULL);
	movesText = XtVaCreateManagedWidget("0",
		labelWidgetClass, movesRowCol,
		XtNwidth, 32,
		XtNfromHoriz, movesLabel, NULL);
	recordLabel = XtVaCreateManagedWidget("Record",
		labelWidgetClass, movesRowCol,
		XtNborderWidth, 0,
		XtNfromHoriz, movesText, NULL);
	recordText = XtVaCreateManagedWidget("0",
		labelWidgetClass, movesRowCol,
		XtNwidth, 108,
		XtNfromHoriz, recordLabel, NULL);
	changerRowCol = XtVaCreateManagedWidget("changerRowCol",
		formWidgetClass, controlPanel,
		XtNborderWidth, 0,
		XtNfromVert, movesRowCol, NULL);
	tileXBox = XtVaCreateManagedWidget("tileXBox",
		boxWidgetClass, changerRowCol,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0, NULL);
	tileYBox = XtVaCreateManagedWidget("tileYBox",
		boxWidgetClass, changerRowCol,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromHoriz, tileXBox, NULL);
	toggleRowCol = XtVaCreateManagedWidget("toggleRowCol",
		formWidgetClass, controlPanel,
		XtNborderWidth, 0,
		XtNfromVert, changerRowCol, NULL);
	cornersBox = XtVaCreateManagedWidget("Corners",
		boxWidgetClass, toggleRowCol,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNstate, DEFAULT_CORNERS, NULL);
#ifdef ANIMATE
	speedBox = XtVaCreateManagedWidget("speedBox",
		boxWidgetClass, toggleRowCol,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromHoriz, cornersBox, NULL);
#endif
	message = XtVaCreateManagedWidget("Welcome",
		labelWidgetClass, controlPanel,
		XtNjustify,  XtJustifyLeft,
		XtNborderWidth, 0,
		XtNwidth, 180,
		XtNfromVert, toggleRowCol, NULL);
	puzzle = XtVaCreateManagedWidget("hexagons",
		hexagonsWidgetClass, mainPanel,
		XtNfromVert, controlPanel, NULL);
	XtVaGetValues(puzzle,
		XtNsizeX, &sizeX,
		XtNsizeY, &sizeY,
		XtNcorners, &corners,
		XtNdelay, &delay, NULL);
#ifdef USE_SPIN
	createSpinner(tileXBox, &tileXSliderLabel,
		tileLabels[0], 0, sizeX, MIN_TILES, MAX_TILES, False,
		tileXUpCallback, tileXDownCallback);
	createSpinner(tileYBox, &tileYSliderLabel,
		tileLabels[1], 0, sizeY, MIN_TILES, MAX_TILES, False,
		tileYUpCallback, tileYDownCallback);
	createSpinner(speedBox, &speedSliderLabel,
		speedLabel, 0, getSpeed(delay), MIN_SPEED, MAX_SPEED, True,
		speedUpCallback, speedDownCallback);
#else
	createSlider(tileXBox, &tileXSliderLabel, &tileXSlider,
		tileLabels[0], 0, sizeX, MIN_TILES, MAX_TILES, False,
		SCROLL_SIZE, tileXChangeCallback, tileXChangeCallback);
	createSlider(tileYBox, &tileYSliderLabel, &tileYSlider,
		tileLabels[1], 0, sizeY, MIN_TILES, MAX_TILES, False,
		SCROLL_SIZE, tileYChangeCallback, tileYChangeCallback);
	createSlider(speedBox, &speedSliderLabel, &speedSlider,
		speedLabel, 0, getSpeed(delay), MIN_SPEED, MAX_SPEED, True,
		SCROLL_SIZE, speedChangeCallback, speedChangeCallback);
#endif
	createToggle(cornersBox, &cornersToggle, "Corners:", 0,
		corners, (XtCallbackProc) cornersCallback);
	XtRealizeWidget(topLevel);
	if (wmDeleteWindow == None)
		wmDeleteWindow = XInternAtom(XtDisplay(topLevel),
			"WM_DELETE_WINDOW", FALSE);
	XSetWMProtocols(XtDisplay(topLevel), XtWindow(topLevel),
		&wmDeleteWindow, 1);
#else
	puzzle = XtVaCreateManagedWidget("hexagons",
		hexagonsWidgetClass, topLevel, NULL);
#endif
	XtVaGetValues(puzzle,
		XtNpixmapSize, &pixmapSize, NULL);
#ifdef HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel,
			XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			RESIZE_XPM(pixmapSize), &pixmap, NULL,
			&xpmAttributes);
	}
	if (pixmap == (Pixmap) NULL)
#endif
		pixmap = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			DEFINE_XBM);
	XtVaSetValues(topLevel,
#ifdef HAVE_MOTIF
		XmNkeyboardFocusPolicy, XmPOINTER, /* not XmEXPLICIT */
#else
		XtNinput, True,
#endif
		XtNiconPixmap, pixmap, NULL);
	XtAddCallback(puzzle, XtNselectCallback,
		(XtCallbackProc) puzzleCallback, (XtPointer) NULL);
#ifdef HAVE_MOTIF
	updateToggle(playMenu, &soundMenuItem,
		False,
#ifdef EXTRA
		8,
#else
		5,
#endif
		(XtCallbackProc) soundCallback);
	descriptionDialog = createHelp(menuBar, (char *) "Description",
		(char *) descriptionHelp);
	featuresDialog = createScrollHelp(menuBar, (char *) "Features",
		(char *) featuresHelp, pixmap);
	synopsisDialog = createHelp(menuBar, (char *) "Synopsis",
		(char *) synopsisHelp);
	referencesDialog = createHelp(menuBar, (char *) "References",
		(char *) referencesHelp);
	aboutDialog = createHelp(menuBar, (char *) "About",
		(char *) aboutHelp);
#endif
	initialize(puzzle);
	XtRealizeWidget(topLevel);
	XGrabButton(XtDisplay(puzzle), (unsigned int) AnyButton, AnyModifier,
		XtWindow(puzzle), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(puzzle),
		XCreateFontCursor(XtDisplay(puzzle), XC_hand2));
	XtAppMainLoop(appCon);

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
