#!/usr/bin/python3 -su

# Copyright (C) 2025 - 2025 ENCRYPTED SUPPORT LLC <adrelanos@whonix.org>
# See the file COPYING for copying conditions.

# pylint: disable=invalid-name

"""
changescompletepage.py - Notifies the user that software has been installed or
removed and allows the user to launch a newly installed application.
"""

from PyQt5.QtCore import pyqtSignal
from PyQt5.QtWidgets import QWidget

## NOTE: _ui modules are autogenerated by build-ui.sh.
from browser_choice.changescompletepage_ui import Ui_ChangesCompletePage


class ChangesCompletePage(QWidget):
    """
    A wizard screen widget that informs the user that software changes have
    been made, and offers to let the user launch the app if desired.
    """

    doneClicked: pyqtSignal = pyqtSignal()

    # pylint: disable=too-many-arguments
    def __init__(
        self,
        app_name: str,
        repository_name: str,
        app_script: str,
        change_str: str,
        did_succeed: bool,
        allow_launch: bool,
        in_sysmaint_session: bool,
        user_sysmaint_split_installed: bool,
        parent: QWidget | None = None,
    ):
        super().__init__(parent)
        self.ui = Ui_ChangesCompletePage()
        self.ui.setupUi(self)

        self.ui.doneButton.clicked.connect(self.doneClicked)

        exit_wizard_str = "Click 'Done' to exit the wizard. You can restart \
this wizard at any time by "
        if in_sysmaint_session and user_sysmaint_split_installed:
            exit_wizard_str += "clicking 'Install a Browser' in the System \
Maintenance Panel."
        else:
            exit_wizard_str += "opening 'Browser Choice' from the Start Menu."

        if did_succeed:
            if change_str == "installed":
                self.ui.actionCompleteLabel.setText(
                    f"<p>The application '{app_name}' from "
                    f"'{repository_name}' has been {change_str}. You may "
                    "launch the browser from the Start Menu in a user "
                    f"session, or by running <code>{app_script}</code> in a "
                    "terminal.</p>"
                    f"<p>{exit_wizard_str}</p>"
                )
            else:
                self.ui.actionCompleteLabel.setText(
                    f"<p>The application '{app_name}' from "
                    f"'{repository_name}' has been {change_str}.</p>"
                    f"<p>{exit_wizard_str}</p>"
                )
            if allow_launch:
                self.ui.launchAppCheckbox.setText(f"Launch '{app_name}'")
            else:
                self.ui.launchAppCheckbox.setVisible(False)
        else:
            self.ui.actionCompleteLabel.setText(
                f"<p>The application '{app_name}' from '{repository_name} "
                f"could not be {change_str}.</p>"
                f"<p>{exit_wizard_str}</p>"
            )
            self.ui.launchAppCheckbox.setVisible(False)

    def launchAppChecked(self) -> bool:
        """
        Indicates whether the user checked the "launch app" checkbox.
        """

        return self.ui.launchAppCheckbox.isChecked()
